 //
//  WCBaseTableViewController.m
//  WorldCardMobile7
//
//  Created by sanhue on 2015/7/8.
//  Copyright (c) 2015年 penpower. All rights reserved.
//

#import "WCTableViewController.h"

// defines
#import "WCAppearanceDefine.h"
#import "PPSectionIndexDefine.h"

// caterories
#import "NSString+Additions.h"
#import "UIView+Toast.h"
#import "UIView+Appearance.h"
#import "NSDate+Format.h"

// models
#import "WCCardSectionModel.h"
#import "WCCardModel.h"

// views
#import "PPBarView.h"
#import "PPSectionHeaderView.h"

// controllers
#import "PPIndexSectionController.h"
#import "WCToastController.h"


#define Common_NoName [@"MLS_NoName" localized]

////////////////////////////////////////////////////////////////////////////////////////////////////
static NSInteger const DefaultCellHeight = 72;
static NSInteger const SectionIndexViewWidth = 20;

static NSString *const WCBaseTableViewCellIdentifier = @"WCBaseTableViewCellIdentifier";

////////////////////////////////////////////////////////////////////////////////////////////////////
@interface WCTableViewController ()
<
UITableViewDataSource,
UITableViewDelegate,
PPSectionIndexViewDelegate,
WCCardHolderCellDelegate
>

#pragma mark - views
@property (nonatomic, retain, readwrite) UILabel *noResultLabel;
@property (atomic, retain, readwrite) PPTableView *tableView;
@property (atomic, retain, readwrite) PPSectionIndexView *sectionIndexView;



#pragma mark - datas

/// 處理section index 的資料
@property (nonatomic, retain) PPIndexSectionController *indexSectionController;

/// layout constraints for main ui
@property (nonatomic, retain) NSArray *layoutConstraintsForTableView;

/// 預設的小名片圖
@property (nonatomic, retain) UIImage *defaultCardImage;

/// 記錄目前被長按的cell
@property (nonatomic, retain) WCCardHolderCell *currentLongPressCell;

@property (nonatomic, retain, readwrite) NSMutableDictionary *multiSelectCardModelMapping;




#pragma mark - view controller settings

/**
 * 是否允許跨section的排序, default NO;
 */
@property (nonatomic, assign) BOOL enableReorderBetweenSection;

/**
 * 目前view controller使用的index mode
 */
@property (nonatomic, assign, readwrite) PPSIC_Mode mode;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////
@implementation WCTableViewController





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - init/dealloc methods


//==============================================================================
//
//==============================================================================
+(void)initialize
{
    [[WCTableViewController appearance] setSeparatorColor:[UIColor grayColor]];
}


//==============================================================================
//
//==============================================================================
- (instancetype)initWithMode:(PPSIC_Mode)mode
{
    self = [super init];
    if (self)
    {
        [[WCTableViewController appearance] applyInvocationTo:self];
        //////////////////////////////////////////////////
        
        _defaultCardImage = [[UIImage alloc] initWithContentsOfFile:[[NSBundle mainBundle] pathForResource:@"CardHolderCell_defaultcardimage" ofType:@"png"]];
        
        _indexSectionController = [[PPIndexSectionController alloc] init];

        _multiSelectCardIDSet = [[NSMutableSet alloc] init];
        
        _multiSelectCardModelMapping = [[NSMutableDictionary alloc] init];
        
        // !! 要在_indexSectionController建立後設定
        [self setIndexMode:mode];

        //////////////////////////////////////////////////
        _defaultRowHeight = DefaultCellHeight;
        _cellType = WCCardHolderCellType_Normal;
        _showSectionTitle = YES;
        _showSectionIndexView = YES;
        _tableViewContentOffset = CGPointZero;
        _monitorKeyboardState = YES;
    }
    return self;
}


//==============================================================================
//
//==============================================================================
- (void)dealloc
{
    [self _removeTableView];
    
    //將所有cell的delegate都設定成nil
    NSArray *visibleCellArray=[self.tableView visibleCells];
    [visibleCellArray makeObjectsPerformSelector:@selector(setDelegate:) withObject:nil];

    self.noResultString = nil;
    self.separatorColor = nil;
    
    self.currentSelectedIndexPath = nil;
    self.allCardSectionArray = nil;
    self.currentLongPressCell = nil;
    self.multiSelectCardIDSet = nil;
    self.multiSelectCardModelMapping = nil;
    self.defaultCardImage = nil;
    self.indexSectionController = nil;
    self.dataController = nil;
    //////////////////////////////////////////////////
    [super dealloc];
}







////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - view controller life cycle


//==============================================================================
//
//==============================================================================
- (void)viewWillAppear:(BOOL)animated
{
    [super viewWillAppear:animated];
    
    //////////////////////////////////////////////////

    //--------------------------------
    // reset flag
    //--------------------------------
    [WCCardHolderCell tellThreadDecelerating:NO];

    if (self.monitorKeyboardState)
    {
        [self registerForKeyboardNotifications];
    }

    [self _prepareTableView];
    
    //////////////////////////////////////////////////
    // 還原離開前的offset
    if(self.tableViewContentOffset.y+self.tableView.frame.size.height>=self.tableView.contentSize.height)
    {
        CGFloat contentOffset = MIN(self.tableViewContentOffset.y,(self.tableView.contentSize.height-self.tableView.frame.size.height));
        contentOffset = MAX(0,contentOffset);
        self.tableViewContentOffset = CGPointMake(0, contentOffset);
    }
    
    self.tableView.contentOffset = self.tableViewContentOffset;
}


//==============================================================================
//
//==============================================================================
- (void)viewDidAppear:(BOOL)animated
{
    [super viewDidAppear:animated];
    
    //////////////////////////////////////////////////
}


//==============================================================================
//
//==============================================================================
- (void)viewWillDisappear:(BOOL)animated
{
    // !! 記錄離開前的contentOffset
    self.tableViewContentOffset = self.tableView.contentOffset;

    [self unRegisterForKeyboardNotifications];
    //////////////////////////////////////////////////
    
    [super viewWillDisappear:animated];
}


//==============================================================================
//
//==============================================================================
- (void)viewDidDisappear:(BOOL)animated
{
    // avoid app crash when push "HOME" button.
    [WCCardHolderCell tellThreadTernimate];

    [self _removeTableView];
    //////////////////////////////////////////////////
    
    [super viewDidDisappear:animated];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - prepare table


//==============================================================================
//
//==============================================================================
- (void)_prepareTableView
{
    _tableView = [[PPTableView alloc] init];
    
    if (self.tableView)
    {
        [self.tableView setTranslatesAutoresizingMaskIntoConstraints:NO];
        //        self.cardHolderView.backgroundColor = [UIColor yellowColor];
        
        self.tableView.delegate = self;
        self.tableView.dataSource = self;
        
        self.tableView.separatorStyle = UITableViewCellSeparatorStyleSingleLine;
        [self.tableView setSeparatorColor:self.separatorColor];
        self.tableView.rowHeight = self.defaultRowHeight;
        self.tableView.estimatedRowHeight = self.defaultRowHeight;
        
        // !! ios9 以上ipad介面會把table的內容左右內縮，造成cell的layout異常，
        // 設定這個可以讓介面看起來跟ios9以前一樣
        if(@available(iOS 9.0, *))
        {
            [self.tableView setCellLayoutMarginsFollowReadableWidth:NO];
        }
        
        //////////////////////////////////////////////////
        UIView *footerView = [[UIView alloc] initWithFrame:CGRectMake(0, 0, self.view.bounds.size.width, self.defaultFootViewHeight)];
        
        if (footerView)
        {
            [self.tableView setTableFooterView:footerView];
            [footerView release];
        }
        
        //////////////////////////////////////////////////
        
        [self.view addSubview:self.tableView];
    }
    
   
    //////////////////////////////////////////////////
    
    _sectionIndexView = [[PPSectionIndexView alloc] init];
    
    if (self.sectionIndexView)
    {
        [self.sectionIndexView setTranslatesAutoresizingMaskIntoConstraints:NO];
        self.sectionIndexView.delegate = self;
        self.sectionIndexView.backgroundColor = [UIColor clearColor];
        [self.view addSubview:self.sectionIndexView];
        
        [self.sectionIndexView setHidden:(self.showSectionIndexView==NO || self.indexSectionController.style==PPIndexSectionControllerStyle_None)];
    }
    //////////////////////////////////////////////////
    _noResultLabel = [[UILabel alloc] init];
    if (self.noResultLabel)
    {
        [self.noResultLabel setTranslatesAutoresizingMaskIntoConstraints:NO];
        self.noResultLabel.font = [UIFont boldSystemFontOfSize:20];
        self.noResultLabel.textColor = [UIColor grayColor];
        self.noResultLabel.backgroundColor = [UIColor clearColor];
        self.noResultLabel.textAlignment = NSTextAlignmentCenter;
        self.noResultLabel.text = self.noResultString;
        [self.view addSubview:self.noResultLabel];
        [self.noResultLabel setHidden:YES];
    }
    
    //////////////////////////////////////////////////
    [self _resetTableViewLayoutConstraintsWithKeyboardHeight:0];
}


//==============================================================================
//
//==============================================================================
- (void)_removeTableView
{
    [self _removeTableViewLayoutConstraints];
    
    [self.noResultLabel removeFromSuperview];
    self.noResultLabel = nil;

    [self.sectionIndexView removeFromSuperview];
    self.sectionIndexView = nil;
    
    [self.tableView setDelegate:nil];
    [self.tableView setDataSource:nil];
    [self.tableView removeFromSuperview];
    self.tableView = nil;
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - layout constraints methods


//==============================================================================
// 加底線是為了與繼承者的api區分
//==============================================================================
- (void)_resetTableViewLayoutConstraintsWithKeyboardHeight:(CGFloat)keyboardHeight
{
    if (self.tableView==nil)
    {
        return ;
    }
    
    [self _removeTableViewLayoutConstraints];
    
    if (self.layoutConstraintsForTableView == nil)
    {
        self.layoutConstraintsForTableView = [NSArray array];
    }
    
    //////////////////////////////////////////////////
    NSDictionary *views = @{@"topLayoutGuide":self.topLayoutGuide,
                            @"tableView":self.tableView,
                            @"sectionIndexView":self.sectionIndexView,
                            @"noResultLabel":self.noResultLabel};
    
    CGFloat navigationBarHeight = 0;
    // 計算table view 要從哪要開始長
    // 如果不是被navigation controller, 或是tabbar controller包起來的，都從(0，0)長
    // !! 改用topLayoutGuide，可以由系統決定
//    if (self.parentViewController)
//    {
//        if ([self.parentViewController isKindOfClass:[UINavigationController class]]==YES ||
//            [self.parentViewController isKindOfClass:[UITabBarController class]]==YES)
//        {
//            if ([self.navigationController.navigationBar isTranslucent]==YES)
//            {
//                if (self.navigationController.navigationBar &&
//                    [self.navigationController isNavigationBarHidden] == NO)
//                {
//                    navigationBarHeight = PPBarViewDefaultHeightForNavigationBarNormal;
//                }
//                
//                if ([[UIApplication sharedApplication] isStatusBarHidden] == NO)
//                {
//                    CGRect statusBarFrame =[[UIApplication sharedApplication] statusBarFrame];
//                    navigationBarHeight += MIN(statusBarFrame.size.width, statusBarFrame.size.height);
//                }
//            }
//        }
//    }
    
    NSInteger sectionIndexWidth = self.sectionIndexView.hidden ? 0 : SectionIndexViewWidth;
    NSInteger bottomSpace = keyboardHeight + self.reserveBottomToolbarSpace;
    NSDictionary *metrics = @{@"SectionIndexViewWidth":@(sectionIndexWidth),
                              @"NavigationBarHeight":@(navigationBarHeight),
                              @"BottomSpace":@(bottomSpace),
                              @"DefaultCellHeight":@(DefaultCellHeight)};

    NSMutableArray *layoutConstraints = [NSMutableArray array];
    
    //////////////////////////////////////////////////
    // cardHolderView
    // horizontal
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"|[tableView]|"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:nil
                                                                                     views:views]];
    
    //vertical
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|[topLayoutGuide][tableView]-(BottomSpace)-|"
                                                                                   options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                   metrics:metrics
                                                                                     views:views]];

    //////////////////////////////////////////////////
    // sectionIndexView
    // horizontal
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"[sectionIndexView(SectionIndexViewWidth)]|"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:metrics
                                                                                     views:views]];
    
    //vertical
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|[topLayoutGuide][sectionIndexView]-(BottomSpace)-|"
                                                                                   options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                   metrics:metrics
                                                                                     views:views]];
    
    //////////////////////////////////////////////////
    // noCardLabel
    
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"|[noResultLabel][sectionIndexView(SectionIndexViewWidth)]|"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:metrics
                                                                                     views:views]];
    
    //vertical
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|[topLayoutGuide][noResultLabel(DefaultCellHeight)]"
                                                                                   options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                   metrics:metrics
                                                                                     views:views]];
    //////////////////////////////////////////////////
    if ([layoutConstraints count] >0)
    {
        self.layoutConstraintsForTableView = [NSArray arrayWithArray:layoutConstraints];
        [self.view addConstraints:self.layoutConstraintsForTableView];
        [self.view layoutIfNeeded];
    }
}


//==============================================================================
//
//==============================================================================
- (void)_removeTableViewLayoutConstraints
{
    if (self.layoutConstraintsForTableView)
    {
        [self.view removeConstraints:self.layoutConstraintsForTableView];
        self.layoutConstraintsForTableView = nil;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Override property


//==============================================================================
//
//==============================================================================
- (void)setSeparatorColor:(UIColor *)separatorColor
{
    [separatorColor retain];
    [_separatorColor release];
    _separatorColor = separatorColor;
}


//==============================================================================
//
//==============================================================================
- (void)setAutoFocusOneCell:(BOOL)autoFocusOneCell
{
    _autoFocusOneCell = autoFocusOneCell;
}


//==============================================================================
//
//==============================================================================
- (void)setMonitorKeyboardState:(BOOL)monitorKeyboardState
{
    _monitorKeyboardState = monitorKeyboardState;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - mode and style methods


//==============================================================================
//
//==============================================================================
- (PPIndexSectionControllerStyle)styleFromMode:(PPSIC_Mode)mode
{
    PPIndexSectionControllerStyle style = PPIndexSectionControllerStyle_None;
    switch (mode)
    {
            
        case PPSIC_M_Stroke:
        {
            style = PPIndexSectionControllerStyle_StrokeAndEnglish;
            
            break;
        }
        case PPSIC_M_Hanpin:
        {
            style = PPIndexSectionControllerStyle_Hanpin;
            break;
        }
        case PPSIC_M_Zuyin:
        {
            style = PPIndexSectionControllerStyle_ZhuyinAndEnglish;
            break;
        }
        case PPSIC_M_Hiragana:
        {
            style = PPIndexSectionControllerStyle_HiraganaAndEnglish;
            break;
        }
        case PPSIC_M_Hangul:
        {
            style = PPIndexSectionControllerStyle_HangulAndEnglish;
            break;
        }
        case PPSIC_M_Thai:
        {
            style = PPIndexSectionControllerStyle_ThaiAndEnglish;
            break;
        }
        case PPSIC_M_Swedish:
        {
            style = PPIndexSectionControllerStyle_Swedish;
            break;
        }
        case PPSIC_M_English:
        {
            style = PPIndexSectionControllerStyle_English;
            break;
        }
        default:
        {
            break;
        }
    }
    return style;
}


//==============================================================================
//
//==============================================================================
- (PPSIC_Mode)modeFromStyle:(PPIndexSectionControllerStyle)style
{
    PPSIC_Mode mode = PPSIC_M_None;
    switch (style)
    {
        case PPIndexSectionControllerStyle_English:
        {
            mode = PPSIC_M_English;
            break;
        }
        case PPIndexSectionControllerStyle_ZhuyinAndEnglish:
        case PPIndexSectionControllerStyle_Zhuyin:
        {
            mode = PPSIC_M_Zuyin;
            break;
        }
        case PPIndexSectionControllerStyle_Hanpin:
        {
            mode = PPSIC_M_Hanpin;
            break;
        }
        case PPIndexSectionControllerStyle_StrokeAndEnglish:
        case PPIndexSectionControllerStyle_Stroke:
        {
            mode = PPSIC_M_Stroke;
            break;
        }
        case PPIndexSectionControllerStyle_HiraganaAndEnglish:
        case PPIndexSectionControllerStyle_Hiragana:
        {
            mode = PPSIC_M_Hiragana;
            break;
        }
        case PPIndexSectionControllerStyle_HangulAndEnglish:
        case PPIndexSectionControllerStyle_Hangul:
        {
            mode = PPSIC_M_Hangul;
            break;
        }
        case PPIndexSectionControllerStyle_ThaiAndEnglish:
        {
            mode = PPSIC_M_Thai;
            break;
        }
        case PPIndexSectionControllerStyle_Swedish:
        {
            mode = PPSIC_M_Swedish;
            break;
        }
        case PPIndexSectionControllerStyle_Time:
        case PPIndexSectionControllerStyle_None:
        {
            mode = PPSIC_M_None;
            break;
        }
    }
    return mode;
}








////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - (Private) section title methods


//===============================================================================
// return -1 if not found
//===============================================================================
- (NSInteger)sectionIndexByTitle:(NSString *)title
{
    WCCardSectionModel *sectionModel;
    NSInteger sectionCount = [self.allCardSectionArray count];
    NSInteger	i, j, k;
    
    
    //---------------------------------------------------
    // special conditions
    //---------------------------------------------------
    // direct return if only one section
    if(sectionCount == 1)
        return 0;
    
    // '#' is always put as last section
    if([title isEqualToString:@"#"]) // "#"
        return sectionCount - 1;
    else if([title isEqualToString:@"•"])	// "•"
        return -1;
    
    
    //---------------------------------------------------
    // find section with index title
    //---------------------------------------------------
    // get current index title order
    NSArray *allIndexArray = [self.indexSectionController indexs];
    
    for(i=0; i<[allIndexArray count]; i++)
    {
        if([title isEqualToString:[allIndexArray objectAtIndex:i]])
            break;
    }
    
    if(i >= [allIndexArray count])
        i = [allIndexArray count]-1;
    
    // reverse search from current order
    for(j=i; j>=0; j--)
    {
        // check if index title in section index
        for(k=0; k<sectionCount; k++)
        {
            sectionModel = [self.allCardSectionArray objectAtIndex:k];
            
            // input letter is in section title
            if([sectionModel.title isEqualToString:[allIndexArray objectAtIndex:j]])
                return k;
        }
    }
    
    
    // return -1 if not found
    return -1;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - (Private) no result handler


//==============================================================================
//
//==============================================================================
- (void)showNoResultIfNeeded
{
    BOOL isEmpty = ([self.allCardSectionArray count] == 0);
    
    [self.noResultLabel setHidden:!isEmpty];
//    if (isEmpty)
//    {
//        self.noResultLabel.frame = CGRectMake(0, 0, self.tableView.bounds.size.width, self.defaultRowHeight);
//        [self.tableView addSubview:self.noResultLabel];
//    }
//    else
//    {
//        [self.noResultLabel removeFromSuperview];
//    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - (Public) CardSectionData handler


//==============================================================================
//
//==============================================================================
- (void)removeCardFromCardSectionArrayWithCardID:(NSString *)cardID
{
    for (WCCardSectionModel *sectionModel in self.allCardSectionArray)
    {
        NSInteger cardIndex = 0;
        for (WCCardModel * sectionCardModel in sectionModel.cardArray)
        {
            if ([cardID isEqualToString:sectionCardModel.ID])
            {
                // !!不能用這行，因為這個會用 isEqual去判斷，object是不是一樣，但WCCardModel沒有實作，所以會全刪
                //[sectionModel.cardArray removeObject:sectionCardModel];
                
                [sectionModel.cardArray removeObjectAtIndex:cardIndex];
                
                if ([sectionModel.cardArray count]==0)
                {
                    [self.allCardSectionArray removeObject:sectionModel];
                }
                return ;
            }
            cardIndex ++;
        }
    }
}


//================================================================================
//
//================================================================================
- (void)setIndexMode:(PPSIC_Mode)mode
{
    _mode = mode;
    
    //////////////////////////////////////////////////
    PPIndexSectionControllerStyle style = [self styleFromMode:mode];
    
    if (self.indexSectionController.style!=style)
    {
        self.indexSectionController.style = style;
    }
    
    //////////////////////////////////////////////////
    if (self.sectionIndexView)
    {
        [self.sectionIndexView setHidden:(self.showSectionIndexView==NO || self.indexSectionController.style==PPIndexSectionControllerStyle_None)];
    }
}


//================================================================================
//
//================================================================================
- (NSMutableArray *)cardIDArray
{
    NSMutableArray *result = nil;
    for (WCCardSectionModel *sectionModel in self.allCardSectionArray)
    {
        for (WCCardModel *cardModel in sectionModel.cardArray)
        {
            if (result==nil)
            {
                result = [NSMutableArray array];
            }
            
            [result addObject:cardModel.ID];
        }
    }
    
    return result;
}


//==============================================================================
//
//==============================================================================
- (BOOL)validIndexPath:(NSIndexPath *)indexPath
{
    NSInteger section = [indexPath section];
    NSInteger row = [indexPath row];

    if ([self.allCardSectionArray count]<section)
    {
        return NO;
    }
    
    WCCardSectionModel *sectionModel = [self.allCardSectionArray objectAtIndex:[indexPath section]];
    if ([sectionModel.cardArray count]<row)
    {
        return NO;
    }
    
    return YES;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - override


//==============================================================================
//
//==============================================================================
- (void)setReserveBottomToolbarSpace:(CGFloat)reserveBottomToolbarSpace
{
    _reserveBottomToolbarSpace = reserveBottomToolbarSpace;
    
    //////////////////////////////////////////////////
    [self _resetTableViewLayoutConstraintsWithKeyboardHeight:0.0];
}







////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Keyboard show/hide


//================================================================================
//
//================================================================================
- (void)registerForKeyboardNotifications
{
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(recvKeyboardWillShow:)
                                                 name:UIKeyboardWillShowNotification object:nil];
    
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(recvKeyboardWillHidden:)
                                                 name:UIKeyboardWillHideNotification object:nil];
    
}


//================================================================================
//
//================================================================================
- (void)unRegisterForKeyboardNotifications
{
    
    [[NSNotificationCenter defaultCenter] removeObserver:self name:UIKeyboardWillShowNotification object:nil];
    
    [[NSNotificationCenter defaultCenter] removeObserver:self name:UIKeyboardWillHideNotification object:nil];
    
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - recv Keyboard notifivation


//==============================================================================
//
//==============================================================================
- (void)recvKeyboardWillShow:(NSNotification*)notification
{
    NSValue * rectValue = [notification.userInfo objectForKey:UIKeyboardFrameEndUserInfoKey];
    
    if(rectValue!=nil)
    {
        CGRect keyboardRect = [rectValue CGRectValue];
        [self _resetTableViewLayoutConstraintsWithKeyboardHeight:MIN(keyboardRect.size.width,keyboardRect.size.height)];
    }
}


//==============================================================================
//
//==============================================================================
- (void)recvKeyboardWillHidden:(NSNotification*)notification
{
    [self _resetTableViewLayoutConstraintsWithKeyboardHeight:0];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - multi select methods


//==============================================================================
//
//==============================================================================
- (void)selectCellWithCardID:(NSString *)cardID
{
    if (self.enableMultiSelectMode == NO)
    {
        return ;
    }
    
    if ([cardID length]==0)
    {
        return ;
    }
    
    
    // 設定在|allCardSectionArray|中的資料的狀態
    for (WCCardSectionModel *cardSectionModel in self.allCardSectionArray)
    {
        for (WCCardModel *cardModel in cardSectionModel.cardArray)
        {
            if ([cardModel.ID isEqualToString:cardID])
            {
                cardModel.isSelected = YES;
                if ([self.multiSelectCardIDSet containsObject:cardID]==NO)
                {
                    [self.multiSelectCardIDSet addObject:cardModel.ID];
                    [self.multiSelectCardModelMapping setObject:cardModel forKey:cardModel.ID];
                }
            }
        }
    }
}


//===============================================================================
//
//===============================================================================
- (void)onSelectAll
{
    if (self.enableMultiSelectMode == NO)
    {
        return ;
    }
    
    
    BOOL curSelectedAll = YES;
    
    //////////////////////////////////////////////////
    // 檢查目前是否是全選，只要有一個沒有選就不是
    for (WCCardSectionModel *cardSectionModel in self.allCardSectionArray)
    {
        for (WCCardModel *cardModel in cardSectionModel.cardArray)
        {
            if(!cardModel.isSelected)
            {
                curSelectedAll = NO;
            }
        }
    }

    
    //////////////////////////////////////////////////
    [self.multiSelectCardIDSet removeAllObjects];
    [self.multiSelectCardModelMapping removeAllObjects];

    // 設定在|allCardSectionArray|中的資料的狀態
    for (WCCardSectionModel *cardSectionModel in self.allCardSectionArray)
    {
        for (WCCardModel *cardModel in cardSectionModel.cardArray)
        {
            // if current select all, change to unselect all
            // else change to select all
            cardModel.isSelected = !curSelectedAll;

            // 要切到select all的話，要順便加到|multiSelectCardIDSet|
            if(curSelectedAll == NO)
            {
                [self.multiSelectCardIDSet addObject:cardModel.ID];
                [self.multiSelectCardModelMapping setObject:cardModel forKey:cardModel.ID];
            }
        }
    }
    
    // 設定目前看得到的cell的狀態
    for (WCCardHolderCell *cell in [self.tableView visibleCells])
    {
        cell.checked = !curSelectedAll;
    }
}


//==============================================================================
//
//==============================================================================
- (void)onDidSelectCellWithIndexPath:(NSIndexPath *)indexPath
{
    // 記錄目前點選
    self.currentSelectedIndexPath = indexPath;
    
    if (self.enableMultiSelectMode)
    {
        NSUInteger          section = [indexPath section];
        NSUInteger          row = [indexPath row];
        NSMutableArray      *cardSectionArray = self.allCardSectionArray;
        WCCardSectionModel  *cardSectionModel = [cardSectionArray objectAtIndex:section];
        WCCardModel         *cardModel = [cardSectionModel.cardArray objectAtIndex:row];
        
        
        WCCardHolderCell *cell = (WCCardHolderCell *)[self.tableView cellForRowAtIndexPath:indexPath];
        cardModel.isSelected = !cardModel.isSelected;
        cell.checked = cardModel.isSelected;
        
        if(cardModel.isSelected)
        {
            [self.multiSelectCardIDSet addObject:cardModel.ID];
            [self.multiSelectCardModelMapping setObject:cardModel forKey:cardModel.ID];
        }
        else
        {
            [self.multiSelectCardIDSet removeObject:cardModel.ID];
            [self.multiSelectCardModelMapping removeObjectForKey:cardModel.ID];

        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - fous on cell


//===============================================================================
//
//===============================================================================
- (void)selectCurrentCard
{
    if (self.autoFocusOneCell==NO)
    {
        return ;
    }
    
    if(self.currentSelectedIndexPath==nil)
    {
        return;
    }
    
    if (self.tableView)
    {
        if (self.currentSelectedIndexPath.row<[self.tableView numberOfRowsInSection:self.currentSelectedIndexPath.section])
        {
            [self.tableView selectRowAtIndexPath:self.currentSelectedIndexPath animated:YES scrollPosition:UITableViewScrollPositionMiddle];
        }
        else
        {
            if ([self.tableView numberOfRowsInSection:self.currentSelectedIndexPath.section+1]>0)
            {
                self.currentSelectedIndexPath = [NSIndexPath indexPathForRow:0 inSection:self.currentSelectedIndexPath.section+1];
                [self.tableView selectRowAtIndexPath:self.currentSelectedIndexPath animated:YES scrollPosition:UITableViewScrollPositionMiddle];
            }
            else if ([self.tableView numberOfRowsInSection:0]>0)
            {
                self.currentSelectedIndexPath = [NSIndexPath indexPathForRow:0 inSection:0];
                [self.tableView selectRowAtIndexPath:self.currentSelectedIndexPath animated:YES scrollPosition:UITableViewScrollPositionMiddle];
            }
            else
            {
                self.currentSelectedIndexPath = nil;
            }
        }
    }
}


//==============================================================================
//
//==============================================================================
- (void)activeCurrentSelectCell
{
    if (self.autoFocusOneCell==NO)
    {
        return ;
    }
    
    
    if (self.tableView)
    {
        //            if([self.tableView cellForRowAtIndexPath:self.currentSelectedIndexPath])
        {
            [self tableView:self.tableView didSelectRowAtIndexPath:self.currentSelectedIndexPath];
        }
        //            else
        //            {
        //                self.currentSelectedIndexPath = nil;
        //            }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - UIScrollViewDelegate


//==============================================================================
//
//==============================================================================
- (void)scrollViewDidScroll:(UIScrollView *)scrollView
{
    self.tableViewContentOffset = self.tableView.contentOffset;
}









////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - UITableView DataSource Methods


//===============================================================================
//
//===============================================================================
- (NSInteger)numberOfSectionsInTableView:(UITableView *)tableView
{
    NSInteger count = [self.allCardSectionArray count];
    [self showNoResultIfNeeded];
    
    return (count == 0) ? 0 : count;
}


//===============================================================================
//
//===============================================================================
- (NSInteger)tableView:(UITableView *)tableView numberOfRowsInSection:(NSInteger)section
{
    NSMutableArray *cardSectionArray = self.allCardSectionArray;
    
    WCCardSectionModel *cardSectionModel = [cardSectionArray count]>section ? [cardSectionArray objectAtIndex:section] : nil;
    NSInteger count = 0;
    
    if(cardSectionModel)
        count = [cardSectionModel.cardArray count];
    
    return count;
}


//===============================================================================
//
//===============================================================================
- (UITableViewCell *)tableView:(UITableView *)tableView cellForRowAtIndexPath:(NSIndexPath *)indexPath
{
    @synchronized(self)
    {
        NSUInteger section = [indexPath section];
        NSUInteger row = [indexPath row];
        
        //  取得card model
        NSMutableArray *cardSectionArray = self.allCardSectionArray;
        WCCardSectionModel *cardSectionModel = [cardSectionArray objectAtIndex:section];
        
        WCCardModel *cardModel = [cardSectionModel.cardArray objectAtIndex:row];
        
        //  建立cell
        WCCardHolderCell* cell = (WCCardHolderCell*)[tableView dequeueReusableCellWithIdentifier:WCBaseTableViewCellIdentifier];
        
        if(cell == nil)
        {
            cell = [[[WCCardHolderCell alloc] initWithDelegate:self
                                               reuseIdentifier:WCBaseTableViewCellIdentifier] autorelease];
        }
        
        cell.enableLongPress = self.enableLongPress;
        
        UIImage *thumbImage = nil;
        if ([self respondsToSelector:@selector(thumbnailImageWithCardID:)])
        {
            thumbImage = [self cachedThumbnailImageWithCardID:cardModel.ID];
        }
        
        cell.selectedBackgroundView.backgroundColor = WCAppearanceDefine_CellHighlightedBGColor;
        
        cell.cardID = cardModel.ID;
        cell.cardImageView.image = thumbImage ? thumbImage : self.defaultCardImage;
        cell.nameLabel.text = [cardModel.displayName length] ? cardModel.displayName : Common_NoName;
        cell.companyLabel.text = cardModel.displayCompany;
    
        
        // 重複聯絡人要塞creator
        if (self.cellType==WCCardHolderCellType_DuplicateWithOwner ||
            self.cellType==WCCardHolderCellType_CompanyContactsWithOwner)
        {
            if ([self respondsToSelector:@selector(accountNameWithGuid:)])
            {
                NSString *owner = [self accountNameWithGuid:cardModel.owner];
                cell.additionLabel.text = owner;
            }
        }
        
        if (self.cellShowModifiedTime)
        {
            cell.jobTitleLabel.text = cardModel.modifiedTime?[cardModel.modifiedTime stringWithFormat:NSDateFormat_Minute]:nil;
        }
        else
        {
            cell.jobTitleLabel.text = cardModel.displayJobTitle;
        }
        
        cell.selectMode = self.enableMultiSelectMode;
        
        cell.checked = [self.multiSelectCardIDSet containsObject:cardModel.ID];
        cell.verified = !((cardModel.tagMask & WC_TagMask_Unverified)==WC_TagMask_Unverified);
        
        cell.cellType = self.cellType;
        // !! currentActionType要在cell.cellType設定後才能設，因為裡面會參考到cellType
        cell.currentActionType = ContactActionType_Tel;

        if (cell.cellType==WCCardHolderCellType_ActionType)
        {
            // !! 只有ActionType才要去取得目前的contactActionType
            __block ContactActionType actionType = ContactActionType_Tel;
            
            __block typeof(self) blockSelf = self;
            [cell retain];
            dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
                
                if ([blockSelf respondsToSelector:@selector(contactActionTypeWithCardID:)])
                {
                    actionType = [blockSelf contactActionTypeWithCardID:cardModel.ID];
                }
                
                //////////////////////////////////////////////////
                dispatch_async(dispatch_get_main_queue(), ^{
                    cell.currentActionType = actionType;
                    [cell release];
                });
            });
        }
        
        // !! 從WCCardHolderCellType_CompanyContacts開始是不顯示名片圖的
        if (cell.cellType<WCCardHolderCellType_CompanyContacts)
        {
            if(!thumbImage)
                [cell startLoadCardImage];
        }
        
        return cell;
    }
}


//===============================================================================
//
//===============================================================================
- (NSString *)tableView:(UITableView *)tableView titleForHeaderInSection:(NSInteger)section
{
    if (self.showSectionTitle==NO)
    {
        return nil;
    }
    
    NSMutableArray *cardSectionArray = self.allCardSectionArray;
    WCCardSectionModel *cardSectionModel = [cardSectionArray count]>section ? [cardSectionArray objectAtIndex:section] : nil;
    
    return [cardSectionModel title];
}



//==============================================================================
//
//==============================================================================
- (UIView *)tableView:(UITableView *)tableView viewForHeaderInSection:(NSInteger)section
{
    if (self.showSectionTitle==NO)
    {
        return nil;
    }
    
    NSMutableArray *cardSectionArray = self.allCardSectionArray;
    WCCardSectionModel *cardSectionModel = [cardSectionArray count]>section ? [cardSectionArray objectAtIndex:section] : nil;

    PPSectionHeaderView *sectionHeaderView = [[[PPSectionHeaderView alloc] init] autorelease];
    if(sectionHeaderView!=nil)
    {
        sectionHeaderView.textLabel.text = [cardSectionModel title];
    }
    
    sectionHeaderView.gap = 15;
    [sectionHeaderView setBackgroundColor:WCAppearanceDefine_SectionHeaderBGColor];
    [sectionHeaderView.textLabel setTextColor:WCAppearanceDefine_SectionHeaderTitleColor];
    [sectionHeaderView.textLabel setFont:[UIFont systemFontOfSize:WCAppearanceDefine_SectionHeaderFontSize]];
    return sectionHeaderView;
}


//===============================================================================
//
//===============================================================================
- (NSArray*)sectionIndexTitlesForTableView:(UITableView *)tableView
{
    if (self.showSectionIndexView==NO)
    {
        return nil;
    }
    
    //有開啟SectionIndex
    if(self.sectionIndexView!=nil)
    {
        //因為系統SectionIndexBar不能調整字的顏色,所以要改用自訂的PPSectionIndexView
        [self.sectionIndexView setDisplayIndexTitles:[self.indexSectionController sectionIndexTitles]
                                     fullIndexTitles:[self.indexSectionController indexs]
                                      withExtraTitle:PPSectionIndexViewExtraTitleNone];

        //回傳一個空的SectionIndexBar,這樣search bar的寬度才會內縮
        return [NSArray arrayWithObject:@" "];
    }
    
    return nil;
}


#pragma mark reorder

//===============================================================================
//
//===============================================================================
- (BOOL)tableView:(UITableView *)tableView canMoveRowAtIndexPath:(NSIndexPath *)indexPath
{
    return self.enableReorderControl;
}


//===============================================================================
//
//===============================================================================
- (NSIndexPath *)tableView:(UITableView *)tableView targetIndexPathForMoveFromRowAtIndexPath:(NSIndexPath *)sourceIndexPath toProposedIndexPath:(NSIndexPath *)proposedDestinationIndexPath
{    
    // 只有同一個section才可以交換
    if (self.enableReorderBetweenSection==NO)
    {
        if (sourceIndexPath.section == proposedDestinationIndexPath.section)
        {
            return proposedDestinationIndexPath;
        }
    }
    
    NSIndexPath *indexPath = [NSIndexPath indexPathForRow:sourceIndexPath.row inSection:sourceIndexPath.section];
    return indexPath;
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - WCCardHolderCellDelegate Methods


//===============================================================================
//
//===============================================================================
- (UIImage *)copyCardListCellThumbImage:(WCCardHolderCell *)cell
{
    UIImage *image = nil;
    
    if ([self respondsToSelector:@selector(thumbnailImageWithCardID:)])
    {
        NSString *cardID = [cell.cardID copy];
        image = [[self thumbnailImageWithCardID:cardID] copy];
        [cardID release];
    }
    
    return image;
}


//===============================================================================
//
//===============================================================================
- (void)wcCardHolderCell:(WCCardHolderCell *)cell didClickActionButtonWithType:(ContactActionType)actionType
{
    if ([self respondsToSelector:@selector(didClickCellActionType:withCardID:)])
    {
        [self didClickCellActionType:actionType withCardID:cell.cardID];
    }
}


//===============================================================================
//
//===============================================================================
- (void)wcCardHodlerCellDidLongPressActionButton:(WCCardHolderCell *)cell
{
    if ([self respondsToSelector:@selector(longPressActionButtonWithCell:)])
    {
        [self longPressActionButtonWithCell:cell];
    }
}



//===============================================================================
//
//===============================================================================
- (void)wcCardHodlerCellDidLongPress:(WCCardHolderCell *)cell
{
    if ([self respondsToSelector:@selector(longPressWithCell:)])
    {
        [self longPressWithCell:cell];
    }
}


//===============================================================================
//
//===============================================================================
- (void)didClickCheckButtonWithCell:(WCCardHolderCell *)cell
{
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_main_queue(), ^{
        NSIndexPath *indexPath = [blockSelf.tableView indexPathForCell:cell];
        
        [blockSelf.tableView selectRowAtIndexPath:indexPath animated:YES scrollPosition:UITableViewScrollPositionNone];
        if ([blockSelf respondsToSelector:@selector(tableViewdidClickCheckButtonWithCell:)])
        {
            [blockSelf tableViewdidClickCheckButtonWithCell:cell];
        }
    });
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - WCSectionIndexViewDelegate


//==============================================================================
//
//==============================================================================
- (void)sectionIndexView:(PPSectionIndexView *)sectionIndexView didSelectedIndexTitle:(NSString *)indexTitle
{
    [WCToastController showSectionIndexToastFromSuperView:self.view withTitle:indexTitle];
 
    //////////////////////////////////////////////////
    if([indexTitle isEqualToString:PPSectionIndexTitleForSearch])
    {
        [self.tableView setContentOffset:CGPointMake(0, 0)];
    }
    else
    {
        NSInteger sectionIndex = [self sectionIndexByTitle:indexTitle];
        
        if(sectionIndex >= 0)
        {
            [self.tableView scrollToRowAtIndexPath:[NSIndexPath indexPathForRow:0 inSection:sectionIndex]
                                  atScrollPosition:UITableViewScrollPositionTop
                                          animated:NO];
        }
        else
        {
            [self.tableView setContentOffset:CGPointMake(0, 0) animated:NO];
        }
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - WCTableViewControllerProtocol


//==============================================================================
//
//==============================================================================
- (UIImage *)cachedThumbnailImageWithCardID:(NSString *)cardID
{
    return nil;
}


//==============================================================================
//
//==============================================================================
- (UIImage *)thumbnailImageWithCardID:(NSString *)cardID
{
    return nil;
}


//==============================================================================
//
//==============================================================================
- (NSString *)accountNameWithGuid:(NSString *)guid
{
    return nil;
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPAppearance


//==============================================================================
//
//==============================================================================
+ (instancetype)appearance
{
    return [PPAppearance appearanceForClass:[self class]];
}


@end
