//
//  WCTRestClientController.m
//
//
//  Created by Howard on 2016/3/9.
//  Copyright © 2016年 penpower. All rights reserved.
//

#import "WCTRestClientController.h"

// Deifne
#import "CrossPlatformDefine.h"
#import "WCTRestClientController+ParameterDefine.h"
#import "WCTRestClientController+ResourceDefine.h"
#import "WCTRestClientController+SettingsKey.h"

// Controller
#import "PPEncodeController.h"
#import "PPSettingsController.h"
#import "PPLogController.h"
#import "WCToolController.h"
#import "PPSystemInfoController.h"

// Category
#import "NSError+Custom.h"
#import "WCTRestClientController+Batch.h"
#import "WCTRestClientController+Common.h"
#import "WCTRestClientController+LocalMode.h"
#import "WCTRestClientController+Password.h"
#import "WCTRestClientController+Version.h"
#import "NSTimer+Additions.h"
#import "NSDate+Format.h"
#import "NSString+Additions.h"
#import "PPLogControllerExtension.h"


#if TARGET_OS_IPHONE
#import "UIDevice+DeviceModel.h"
#endif

static NSString * const WCTRestClientController_LogDir = @"WCTRestClientLog";

NSString *const WCTRestClientController_LogoutNotification = @"WCTRestClientController_LogoutNotification";
NSString *const WCTRestClientController_LimitedAccountStateChangedNotification = @"WCTRestClientController_LimitedAccountStateChangedNotification";

NSInteger const WCTRestClientController_KeepTokenTimeInterval = 30*60;
////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Interface

@interface WCTRestClientController()

@property (nonatomic,retain) WCTRestClient *restClient;
@property (atomic,retain) WCTRCTokenInfo *loginTokenInfo;   //要用atomic
@property (nonatomic,retain) WCTRCAccountInfo *accountInfo;
@property (nonatomic,retain) WCTRCLoginRequestObject *loginRequestObject;
@property (nonatomic,retain) RKObjectMapping *rkObjectMapping;
@property (nonatomic,retain) PPLogController *logController;
@property (nonatomic,assign) dispatch_queue_t callbackQueue;
@property (nonatomic,retain) NSTimer *tokenKeepTimer;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Implementation

@implementation WCTRestClientController

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Dellocating Objects

//================================================================================
//
//================================================================================
- (id)init
{
    id object = nil;
    
    //////////////////////////////////////////////////
    
    if((self=[super init]))
    {
        do
        {
            NSAssert([NSThread currentThread]!=[NSThread mainThread], @"只能在bg thread使用，不然有機會 deadLock");
            
            //////////////////////////////////////////////////
            
            NSString *urlString = [PPSettingsController stringValueWithKey:WCTRestClientController_SettingsKey_URLAddress];
            
            NSAssert([urlString length]>0, @"請指定WCTRestClientController連線設定");
            
            NSURL *url = [NSURL URLWithString:urlString];
            
            NSString *hostString = [[url host] lowercaseString];
            
            if([hostString isEqualToString:[url host]]==NO)
            {
                urlString = [urlString stringByReplacingOccurrencesOfString:[url host] withString:hostString];
                
                url = [NSURL URLWithString:urlString];
            }
            
            //////////////////////////////////////////////////
            
            _restClient = [[WCTRestClient alloc] initWithBaseURL:url];
            if(_restClient==nil)
            {
                break;
            }
            
            _restClient.timeOutInterval = WCTRestClientController_TimeOutInterval;
            
            //////////////////////////////////////////////////
            
            // 指定callback使用的queue
            self.callbackQueue = dispatch_queue_create("WCTRestClientController", DISPATCH_QUEUE_CONCURRENT);
            _restClient.successCallbackQueue = self.callbackQueue;
            _restClient.failureCallbackQueue = self.callbackQueue;
            
            //////////////////////////////////////////////////
            
            // 初始化 RKObjectMapping,　才能指定 NSDate to NSString
            _rkObjectMapping = [RKObjectMapping mappingForClass:[NSDate class]];
            
            //////////////////////////////////////////////////
            
            // 要在RKObjectMapping第一次initilize之後再加，這一個才會變成預設的第一個dateFormatter
            NSDateFormatter *dateFormatter = [[NSDateFormatter alloc] init];
            [dateFormatter setDateFormat:WCTRestClient_ServerDateFormat];
            dateFormatter.timeZone = [NSTimeZone timeZoneWithAbbreviation:@"UTC"];
            dateFormatter.locale = [[[NSLocale alloc] initWithLocaleIdentifier:@"en_US_POSIX"] autorelease];
            
            [[RKValueTransformer defaultValueTransformer] insertValueTransformer:dateFormatter atIndex:0];
            [dateFormatter release];
            
            //////////////////////////////////////////////////
            // 建立rest client log instance
            
            self.logController = [[[PPLogController alloc] init] autorelease];
            
            if(self.logController != nil)
            {
                NSString *logDirPath = [WCTRestClientController logDirPath];
                
                if([[NSFileManager defaultManager] fileExistsAtPath:logDirPath] == NO)
                {
                    [[NSFileManager defaultManager] createDirectoryAtPath:logDirPath
                                              withIntermediateDirectories:YES
                                                               attributes:nil
                                                                    error:nil];
                }
                
                [self.logController setFileName:WCTRestClientController_LogDir atPath:logDirPath];
                [self.logController setMask:PPLogControllerMask_Normal];
                
                NSLog(@"%@",logDirPath);
            }
            
            //////////////////////////////////////////////////
            // 預設都是1.0
            NSString *macBackwardCompatibleVesion = [PPSettingsController stringValueWithKey:WCTRestClientController_SettingsKey_MinSupportedServerVersion];
            
            if([macBackwardCompatibleVesion length]>0)
            {
                self.maxBackwardCompatibleVersionInfo = [PPVersionInfo versionInfoWithVersionString:macBackwardCompatibleVesion];
            }
            else
            {
                self.maxBackwardCompatibleVersionInfo = [PPVersionInfo versionInfoWithVersionString:@"1.0"];
            }
            
            NSString *supportedClientVersion = [PPSettingsController stringValueWithKey:WCTRestClientController_SettingsKey_MaxSupportedServerVersion];
            
            if([supportedClientVersion length]>0)
            {
                self.clientSupportedVersionInfo = [PPVersionInfo versionInfoWithVersionString:supportedClientVersion];
            }
            else
            {
                self.clientSupportedVersionInfo = [PPVersionInfo versionInfoWithVersionString:@"1.0"];
            }
            
            self.serverVersionInfo = [PPVersionInfo versionInfoWithVersionString:@"1.0"];
            
            object = self;
            
        }while(0);
    }
    
    //////////////////////////////////////////////////
    
    if(object!=self)
    {
        [self release];
    }
    
    //////////////////////////////////////////////////
    
    return object;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    self.serverVersionInfo = nil;
    self.clientSupportedVersionInfo = nil;
    self.maxBackwardCompatibleVersionInfo = nil;
    
    self.logController = nil;
    
    [_restClient release];
    _restClient = nil;
    
    [_loginTokenInfo release];
    _loginTokenInfo = nil;
    
    [_loginRequestObject release];
    _loginRequestObject = nil;
    
    [_accountInfo release];
    _accountInfo = nil;
    
    [_rkObjectMapping release];
    _rkObjectMapping = nil;
    
    if(self.callbackQueue != nil)
    {
        dispatch_release(self.callbackQueue);
        self.callbackQueue = nil;
    }
    
    [self stopTokenKeeper];
    
    //////////////////////////////////////////////////
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - log methods


//===============================================================================
//
//===============================================================================
+ (NSString *)logDirPath
{
    return [WCToolController baseStorePathWithDirName:WCTRestClientController_LogDir isCreatDirPath:NO];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - keep token alive


//==============================================================================
//
//==============================================================================
- (void)startTokenKeeper
{
    __block typeof(self) blockSelf = self;
    
    self.tokenKeepTimer = [NSTimer pp_scheduledTimerWithTimeInterval:WCTRestClientController_KeepTokenTimeInterval
                                                             repeats:YES
                                                               block:^(NSTimer * _Nonnull timer) {
        
        dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
            
            NSError *error = nil;
            [blockSelf refreshTokenWithError:&error];
        });
        
    }];
    
    [[NSRunLoop mainRunLoop] addTimer:self.tokenKeepTimer forMode:NSRunLoopCommonModes];
}


//==============================================================================
//
//==============================================================================
- (void)stopTokenKeeper
{
    [self.tokenKeepTimer invalidate];
    self.tokenKeepTimer = nil;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Method

//================================================================================
//
//================================================================================
- (BOOL)loginExpired
{
    return ([[NSDate date] timeIntervalSince1970]-[self.restClient.lastOperationDate timeIntervalSince1970]>WCTClientControllerLoginExpiredTimeInterval);
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Method

//================================================================================
//
//================================================================================
+ (WCTRestClientController *)shareRestClientController;
{
    NSAssert([NSThread currentThread]!=[NSThread mainThread], @"只能在bg thread使用，不然有機會 deadLock");
    
    static id sharedInstance = nil;
    static dispatch_once_t onceToken;
    
    dispatch_once(&onceToken, ^{
        sharedInstance = [[WCTRestClientController alloc] init];
    });
    
    return sharedInstance;
}


//================================================================================
//
//================================================================================
+ (NSString *)curLoginToken
{
    return [WCTRestClientController shareRestClientController].loginTokenInfo.token;
}


//================================================================================
//
//================================================================================
+ (NSString *)curAccountGuid
{
    return [WCTRestClientController shareRestClientController].accountInfo.guid;
}


//================================================================================
//
//================================================================================
+ (WCTRestClient *)curRestClient
{
    return [WCTRestClientController shareRestClientController].restClient;
}


//==============================================================================
//
//==============================================================================
+ (NSHTTPURLResponse *)responseFromAFRKNetworkingError:(NSError *)error
{
    if(error==nil) return nil;
    
    if(error.userInfo != nil)
    {
        NSArray *infoValues = [error.userInfo allValues];
        
        for(id value in infoValues)
        {
            if([value isKindOfClass:[NSError class]])
            {
                NSError *subError = (NSError *)value;
                
                return [self responseFromAFRKNetworkingError:subError];
            }
        }
        
    }
    
    return [[error userInfo] objectForKey:AFRKNetworkingOperationFailingURLResponseErrorKey];
}


//==============================================================================
//
//==============================================================================
+ (NSInteger)statusCodeFromAFRKNetworkingError:(NSError *)error
{
    if(error==nil) return NSNotFound;
    
    if(error.userInfo != nil)
    {
        NSArray *infoValues = [error.userInfo allValues];
        
        for(id value in infoValues)
        {
            if([value isKindOfClass:[NSError class]])
            {
                NSError *subError = (NSError *)value;
                
                return [self statusCodeFromAFRKNetworkingError:subError];
            }
        }
        
    }
    else
    {
        return [error code];
    }
    
    NSHTTPURLResponse *response = [[error userInfo] objectForKey:AFRKNetworkingOperationFailingURLResponseErrorKey];
    if (response==nil)
    {
        return NSNotFound;
    }
    
    return [response statusCode];
}


//==============================================================================
//
//==============================================================================
+ (NSString *)domainFromAFRKNetworkingError:(NSError *)error
{
    if(error==nil)
    {
        return nil;
    }
    
    if(error.userInfo != nil)
    {
        NSArray *infoValues = [error.userInfo allValues];
        
        for(id value in infoValues)
        {
            if([value isKindOfClass:[NSError class]])
            {
                NSError *subError = (NSError *)value;
                
                return [self domainFromAFRKNetworkingError:subError];
            }
        }
        
    }
    else
    {
        return [error domain];
    }
    
    return [error domain];
}


//==============================================================================
//
//==============================================================================
+ (NSError *)errorFromLogoutNotification:(NSNotification *)notification
{
    return [[notification userInfo] objectForKey:WCTRestClientController_LogoutErrorKey];
}


//==============================================================================
//
//==============================================================================
+ (void)sendLogoutNotification
{
    NSError *erorr = PPErrorMake(WCTServer_Common_ErrorCode_AuthenticationFailed, [@"MLS_AuthenticationFailed" localized], nil);
    [[NSNotificationCenter defaultCenter] postNotificationName:WCTRestClientController_LogoutNotification
                                                        object:self
                                                      userInfo:@{WCTRestClientController_LogoutErrorKey:erorr}];
}




////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Method (Server restore Related)


//==============================================================================
//
//==============================================================================
+ (BOOL)isServerRestoredWithDatabaseCreateTime:(NSDate *)databaseCreateTime
{
    PPLogFuncIn(@" databaseCreateTime:%@", databaseCreateTime);
    BOOL result = NO;
    
    do
    {
        if(databaseCreateTime==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSDate *clientDatabaseCreateTime = [PPSettingsController dateValueWithKey:WCTRestClientController_SettingsKey_DatabaseCreateTime];
        PPLogFuncInfo(@" clientDatabaseCreateTime:%@", clientDatabaseCreateTime);
        if(clientDatabaseCreateTime==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSCalendarUnit calendarUnit = NSCalendarUnitYear|NSCalendarUnitMonth|NSCalendarUnitDay|NSCalendarUnitHour|NSCalendarUnitMinute|NSCalendarUnitSecond;
        
        NSDateComponents *curDateComponents = [[NSCalendar currentCalendar] components:calendarUnit
                                                                              fromDate:databaseCreateTime];
        PPLogFuncInfo(@" curDateComponents:%@", curDateComponents);
        if(curDateComponents==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSDateComponents *localDateComponents = [[NSCalendar currentCalendar] components:calendarUnit
                                                                                fromDate:clientDatabaseCreateTime];
        PPLogFuncInfo(@" localDateComponents:%@", localDateComponents);
        if(localDateComponents==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        // 伺服器的DB建立時間，和本機端紀錄不一樣，代表伺服器有還原
        if([curDateComponents second]!=[localDateComponents second] ||
           [curDateComponents minute]!=[localDateComponents minute] ||
           [curDateComponents hour]!=[localDateComponents hour] ||
           [curDateComponents day]!=[localDateComponents day] ||
           [curDateComponents month]!=[localDateComponents month] ||
           [curDateComponents year]!=[localDateComponents year])
        {
            result = YES;
        }
    }
    while (0);
    PPLogFuncOut(@" result:%@", @(result));
    return result;
}


//==============================================================================
//
//==============================================================================
+ (void)recordDatabaseCreateTime:(NSDate *)databaseCreateTime
{
    PPLogFuncIn(@" databaseCreateTime:%@", databaseCreateTime);
    [PPSettingsController setDateValue:databaseCreateTime withKey:WCTRestClientController_SettingsKey_DatabaseCreateTime];
    PPLogFuncOut();
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class Method (Account subscription status)


//==============================================================================
//
//==============================================================================
+ (BOOL)isLimitedAccount
{
    NSNumber *isLimitedAcccountNumber  = [PPSettingsController numberValueWithKey:WCTRestClientController_SettingsKey_IsLimitedAccount];
    if(isLimitedAcccountNumber!=nil)
    {
        return [isLimitedAcccountNumber boolValue];
    }
    return NO;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Method (Network Related)

//================================================================================
//
//================================================================================
+ (void)cleanCachedResponses
{
    [[NSURLCache sharedURLCache] removeAllCachedResponses];
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance For Category Method

//================================================================================
//
//================================================================================
- (BOOL)checkTimeExpiredAndKeepConnectionWithError:(NSError **)error
{
    BOOL result = YES;
    NSError *returnError = nil;
    
    do
    {
        //超過 WCTClientControllerLoginExpiredTimeInterval 未連線，更新 token
        
        if([self loginExpired]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        WCTRCTokenInfoResponseResult *tokeInfoResponseResult = [self refreshTokenWithError:&returnError];
        
        if(tokeInfoResponseResult!=nil)
        {
            self.loginTokenInfo = tokeInfoResponseResult.data;
        }
        // 更新 token，發現沒有權限，強制登出
        else if(returnError.code==WCTServer_RefreshToken_ErrorCode_AuthenticationFailed)
        {
            result = NO;
        }
        else if([self autoLoginWithError:&returnError]==NO)
        {
            result = NO;
        }
        
    }
    while (0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    return result;
}


//===============================================================================
//
//===============================================================================
- (void)logMessageWithStartTime:(NSDate *)startTime format:(NSString *)format, ...
{
    va_list arguments;
    va_start(arguments, format);
    
    NSTimeInterval exeTime = [[NSDate date] timeIntervalSinceReferenceDate] - [startTime timeIntervalSinceReferenceDate];
    format = [NSString stringWithFormat:@"[Start time: %@ (cost time:%lf)] %@",
                  [startTime stringWithFormat:@"yyyy/MM/dd HH:mm:ss.SSS"], exeTime, format];
        
    [self.logController logWithMask:PPLogControllerMask_Normal format:format arguments:arguments];
    va_end(arguments);
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Method


//================================================================================
// MARK: 背景登入
//================================================================================
- (BOOL)autoLoginWithError:(NSError **)error
{
    __block NSError *returnError = nil;
    
    BOOL result = NO;
    
    do
    {
        // 當離線登入時，此值會是空的，重新建立
        if(self.loginRequestObject==nil)
        {
            //判斷版本
            WCTServerCanLoginResult result = [self canLoginWithError:&returnError];
            
            if (returnError==nil &&
                result!=WCTServerCanLoginResult_YES)
            {
                if(result==WCTServerCanLoginResult_NO_ServerIsNewer)
                {
                    returnError = PPErrorMake(WCTRestClientController_ErrorCode_ServerVersionIsNewer,@"ServerIsNewer",nil);
                }
                else if(result==WCTServerCanLoginResult_NO_UnsupportedServerVersion)
                {
                    returnError = PPErrorMake(WCTRestClientController_ErrorCode_UnsupportedServerVersion,@"Unsupported server version",nil);
                }
                break;
            }
            else if(returnError!=nil)
            {
                break;
            }
            
            [WCTRestClientController setCurLastMininumSupportedVersion:[self minSupportedVersion]];
            
            //////////////////////////////////////////////////
            
            self.loginRequestObject = [[[WCTRCLoginRequestObject alloc] init] autorelease];
            
            if(self.loginRequestObject==nil)
            {
                break;
            }
            
            // !! 4.0 以上才支援 deviceID/ deviceName 登入
            if([[WCTRestClientController curLastMinumSupportedVersion] floatValue]<4.0)
            {
                self.loginRequestObject.deviceId = nil;
                self.loginRequestObject.deviceName = nil;
            }
            
            //////////////////////////////////////////////////
            
            self.loginRequestObject.email = [PPSettingsController stringValueWithKey:WCTRestClientController_SettingsKey_AccountName];
        }
        
        //////////////////////////////////////////////////
        
        //AP 使用過程中可能變動密碼，更新密碼
        self.loginRequestObject.password = [WCTRestClientController decryptAccountPassword];
        
        // 時間要以目前時間傳入
        self.loginRequestObject.clientLocalTime = [NSDate date];
        
        //////////////////////////////////////////////////
        // !! login成功後就會變動了，所以login前要先取得原本的暫停用戶狀態
        BOOL oldLimitedAcountState = [[self class] isLimitedAccount];
        
        WCTRCLoginResponseResult *loginResponseResult = [self loginWithLoginRequest:[[self.loginRequestObject copy] autorelease]
                                                                          withError:&returnError];
        
        if(returnError!=nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        // !!檢查是否因版本不符背景登入
        // 正常來說這邊的self.serverVersionInfo應該已經更新
        NSString *lastServerVersionString = [[self class] lastLoginServerVersion];
        PPVersionInfo *lastServerVersion = [PPVersionInfo versionInfoWithVersionString:lastServerVersionString];
        
        if(lastServerVersion.majorVersion!=self.serverVersionInfo.majorVersion)
        {
            returnError = PPErrorMake(WCTRestClientController_ErrorCode_LoginAfterServerUpdated,@"Server had updated", nil);
            
            // !! 這個設為nil才會重新跑自動登入
            self.loginTokenInfo = nil;
            self.restClient.lastOperationDate = nil;
            //////////////////////////////////////////////////
            // 送出登出通知
            NSError *copyError = [returnError copy];    // 因為要post又要回傳，所以copy一份，以免相互干擾
            dispatch_async(dispatch_get_main_queue(), ^{
                
                [[NSNotificationCenter defaultCenter] postNotificationName:WCTRestClientController_LogoutNotification
                                                                    object:self
                                                                  userInfo:@{WCTRestClientController_LogoutErrorKey:copyError}];
                [copyError release];
            });
            break;
        }
        
        //////////////////////////////////////////////////
        
        //檢查伺服器是否進行還原
        BOOL serverRestore = [[self class] isServerRestoredWithDatabaseCreateTime:loginResponseResult.data.databaseCreateTime];
        
        if(serverRestore==YES)
        {
            returnError = PPErrorMake(WCTRestClientController_ErrorCode_ServerRestored,@"Server had Restored", nil);
            
            // !! 這個設為nil才會重新跑自動登入
            self.loginTokenInfo = nil;
            self.restClient.lastOperationDate = nil;
            //////////////////////////////////////////////////
            // 送出登出通知
            NSError *copyError = [returnError copy];    // 因為要post又要回傳，所以copy一份，以免相互干擾
            dispatch_async(dispatch_get_main_queue(), ^{
                
                [[NSNotificationCenter defaultCenter] postNotificationName:WCTRestClientController_LogoutNotification
                                                                    object:self
                                                                  userInfo:@{WCTRestClientController_LogoutErrorKey:copyError}];
                [copyError release];
            });
            break;
        }
        
        // 不用還原才紀錄時間
        [[self class] recordDatabaseCreateTime:loginResponseResult.data.databaseCreateTime];
        
        //!! 狀態變更時要通知
        BOOL currentLimitedAccountState = [[self class] isLimitedAccount];
        
        if (oldLimitedAcountState!=currentLimitedAccountState)
        {
            dispatch_async(dispatch_get_main_queue(), ^{
                
                [[NSNotificationCenter defaultCenter] postNotificationName:WCTRestClientController_LimitedAccountStateChangedNotification
                                                                    object:self
                                                                  userInfo:nil];
            });
        }
        result = YES;
    }
    while (0);
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    return result;
}


//================================================================================
//
//================================================================================
- (BOOL)changeURLAddress:(NSString *)urlAddress withError:(NSError **)error
{
    NSError *returnError = nil;
    BOOL result = NO;
    
    do
    {
        if(urlAddress==nil ||
           [urlAddress length]<=0)
        {
            returnError = WCTRestClientErrorMake(@"urlAddress 為空",returnError);
            
            break;
        }
        
        //////////////////////////////////////////////////
        RKObjectManager *objectManager = [self.restClient objectManagerForType:PPRCObjectManagerType_JSON];
        
        if([urlAddress compare:[objectManager.HTTPClient.baseURL absoluteString] options:NSCaseInsensitiveSearch]==NSOrderedSame)
        {
            result = YES;
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSURL *url = [NSURL URLWithString:urlAddress];
        
        if(url!=nil)
        {
            urlAddress = [urlAddress stringByReplacingOccurrencesOfString:[url host] withString:[[url host] lowercaseString]];
        }
        
        //////////////////////////////////////////////////
        
        [self.restClient cancellAllRestOperation];
        [self.restClient waitAllRestOperaitonAreFinished];
        
        self.restClient = [[[WCTRestClient alloc] initWithBaseURL:[NSURL URLWithString:urlAddress]] autorelease];
        
        if(self.restClient==nil)
        {
            returnError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",returnError);
            break;
        }
        
        self.restClient.timeOutInterval = WCTRestClientController_TimeOutInterval;
        
        result = YES;
    }
    while (0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return result;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance RestClient Operaiton Method

//================================================================================
//
//================================================================================
- (void)cancellAllRestOperation
{
    [self.restClient cancellAllRestOperation];
}


//================================================================================
//
//================================================================================
- (void)waitAllRestOperaitonAreFinished
{
    [self.restClient waitAllRestOperaitonAreFinished];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Login Method

//================================================================================
// MARK: 登入
//================================================================================
- (WCTRCLoginResponseResult *)loginWithLoginRequest:(WCTRCLoginRequestObject *)loginRequest withError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCLoginResponseResult  *blockLoginResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            // set platform, 2.0後新增，但傳給2.0以下也不會出錯，所以這邊直接用
#if TARGET_OS_IPHONE
            NSString *deviceModel = [UIDevice deviceModel];
            
            if ([deviceModel hasPrefix:@"iPad"] )
            {
                loginRequest.platform = WCTRC_Platform_IPAD;
            }
            else
            {
                loginRequest.platform = WCTRC_Platform_IOS;
            }
#elif TARGET_OS_OSX
            loginRequest.platform = WCTRC_Platform_MACOS;
#endif
            
            //////////////////////////////////////////////////
            
            __block typeof(self) blockself = self;
            
            RKObjectRequestOperation *requestOperation = [self.restClient loginWithLoginRequest:loginRequest
                                                                        minimumSupportedVersion:[WCTRestClientController curLastMinumSupportedVersion]        success:^(WCTRCLoginResponseResult *loginResponseResult)
                                                          {
                blockLoginResponseResult = [loginResponseResult retain];
                
                //////////////////////////////////////////////////
                
                // 紀錄登入Token
                blockself.loginTokenInfo = blockLoginResponseResult.data.tokenInfo;
                
                // 紀錄登入帳密
                blockself.loginRequestObject = loginRequest;
                
                blockself.accountInfo = blockLoginResponseResult.data.accountInfo;
                
                //////////////////////////////////////////////////
                
                [WCTRestClientController encryptAccountPasswordToUserDefault:loginRequest.password];
                
                // 記錄暫停使用狀態
                BOOL isLimitedAccount = [blockLoginResponseResult.data.accountInfo.accountSubscriptionStatus isEqualToString:WCTRC_AccountSubscriptionStatus_TemplateInvalidate];
                [PPSettingsController setNumberValue:@(isLimitedAccount) withKey:WCTRestClientController_SettingsKey_IsLimitedAccount];
                
                //////////////////////////////////////////////////
                // 開始維持token的生命周期
                [self startTokenKeeper];
                
                dispatch_group_leave(group);
            }
                                                                                        failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockLoginResponseResult!=nil)
            {
                [blockLoginResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, loginRequest];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    //////////////////////////////////////////////////
    
    return blockLoginResponseResult;
}


//================================================================================
// MARK: 登出
//================================================================================
- (WCTRCLogoutResponseResult *)logoutWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCLogoutResponseResult  *blockLogoutResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient logoutWithLoginToken:self.loginTokenInfo.token
                                                                                       success:^(WCTRCLogoutResponseResult *logoutResponseObject)
                                                          {
                blockLogoutResponseResult = [logoutResponseObject retain];
                dispatch_group_leave(group);
            }
                                                                                       failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockLogoutResponseResult!=nil)
            {
                [blockLogoutResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
            //////////////////////////////////////////////////
            // 登出後上次執行operation的時間要重設，這樣下次登入時，才會有機會refresh token
            self.restClient.lastOperationDate = nil;
            
            // 登出後，loginTokenInfo要清掉
            self.loginTokenInfo = nil;
            self.loginRequestObject = nil;
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    //////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    // 停止tokenKeeperTimer，!! 失敗也要call
    [self stopTokenKeeper];
    
    //////////////////////////////////////////////////
    
    return blockLogoutResponseResult;
}


//==============================================================================
// 離線登入要call這個清除MEM中的登入資料，以免異常
//==============================================================================
- (void)offlineLogin
{
    self.loginRequestObject = nil;
}


//================================================================================
//
//================================================================================
- (WCTRCTokenInfoResponseResult *)refreshTokenWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCTokenInfoResponseResult  *blockTokenInfoResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient refreshLoginToken:self.loginTokenInfo.token
                                                                                    success:^(WCTRCTokenInfoResponseResult *tokenInfoResponseResult)
                                                          {
                blockTokenInfoResponseResult = [tokenInfoResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                    failure:^(NSError *failureError)
                                                          {
                
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockTokenInfoResponseResult!=nil)
            {
                [blockTokenInfoResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    return blockTokenInfoResponseResult;
}


//================================================================================
// MARK: 檢查伺服器合法性
//================================================================================
- (BOOL)checkServerIslegalWithLoginCode:(WCTRCLoginCode *)loginCode
                                  error:(NSError **)error
{
    BOOL result = NO;
    
    Byte *hexBytes = NULL;
    Byte *keyBytes = NULL;
    Byte *decodeBytes = NULL;
    
    do
    {
        if(loginCode==nil ||
           [loginCode.code length]<=0)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        char *dataByte = (char *)[loginCode.code UTF8String];
        
        int dataByteLen = sizeof(char)* (int)strlen([loginCode.code UTF8String]);
        
        int hexByteLen = 0;
        
        //轉成16進制 Byte Value
        hexBytes = [PPEncodeController allocHexBytesFromHexCharData:dataByte
                                                            dataLen:dataByteLen
                                                             retLen:&hexByteLen];
        
        if (hexBytes==NULL)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        
        int keyLen = sizeof(unichar) * (int)[EasyEncode length];
        NSRange range;
        
        
        range.location = 0;
        range.length = [EasyEncode length];
        
        
        if((keyBytes = malloc(keyLen)))
        {
            memset(keyBytes, 0, keyLen);
            [EasyEncode getCharacters:(unichar *)keyBytes range:range];
        }
        
        //////////////////////////////////////////////////
        
        int decodeLen = 0;
        
        decodeBytes = [PPEncodeController allocDecodeDataWithSource:hexBytes
                                                          sourceLen:hexByteLen
                                                            keyByte:keyBytes
                                                             keyLen:keyLen
                                                             retLen:&decodeLen];
        
        
        if (decodeBytes==NULL)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        RC4_KEY key1;
        RC4_KEY key2;
        
        char* rc4key1 = PPREG_RC4KEY_1;
        char* rc4key2 = PPREG_RC4KEY_2;
        
        if(([PPEncodeController RC4Init:rc4key1 keyLen:(int)strlen(rc4key1) key:&key1]==NO) ||
           ([PPEncodeController RC4Init:rc4key2 keyLen:(int)strlen(rc4key2) key:&key2]==NO) ||
           ([PPEncodeController RC4Works:decodeBytes dataLen:decodeLen key:&key2]==NO) ||
           ([PPEncodeController RC4Works:decodeBytes dataLen:decodeLen key:&key1]==NO) )
        {
            break;
        }
        
        decodeBytes[decodeLen] = 0;
        decodeBytes[decodeLen+1] = 0;
        
        NSString *decodingString= [NSString stringWithCharacters:(const unichar *)decodeBytes
                                                          length:decodeLen/2];
        
        
        
        //////////////////////////////////////////////////
        
        if([[decodingString lowercaseString] containsString:[@"hwid1:" stringByAppendingString:[loginCode.hwid1 lowercaseString]]]==NO ||
           [[decodingString lowercaseString] containsString:[@"hwid2:" stringByAppendingString:[loginCode.hwid2 lowercaseString]]]==NO)
        {
            break;
        }
        
        result = YES;
    }
    while (0);
    
    //////////////////////////////////////////////////
    
    if (hexBytes!=NULL)
    {
        free(hexBytes);
    }
    
    if (keyBytes!=NULL)
    {
        free(keyBytes);
    }
    
    if (decodeBytes!=NULL)
    {
        free(decodeBytes);
    }
    
    if(result==NO &&
       error!=nil)
    {
        *error = PPErrorMake(WCTRestErrorCustom_Code_ServerIsIllegal,WCTRCC_MLS_illegalServer, nil);
    }
    
    return result;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Account Method

//================================================================================
//
//================================================================================
- (WCTRCInLineModel *)accountLogoWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCInLineModel  *blockInLineModel = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient accountLogoWithLoginToken:self.loginTokenInfo.token
                                                                                            success:^(WCTRCInLineModel *inLineModel)
                                                          {
                blockInLineModel = [inLineModel retain];
                dispatch_group_leave(group);
            }
                                                                                            failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockInLineModel!=nil)
            {
                [blockInLineModel autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockInLineModel;
}


//================================================================================
//
//================================================================================
- (WCTRCResetPasswordResponseResult *)resetPasswordWithPassword:(NSString *)password
                                                          error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCResetPasswordResponseResult  *blockResetPasswordResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient resetPasswordWithLoginToken:self.loginTokenInfo.token
                                                                                             password:password
                                                                                              success:^(WCTRCResetPasswordResponseResult *resetPasswordResponseResult)
                                                          {
                blockResetPasswordResponseResult = [resetPasswordResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                              failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResetPasswordResponseResult!=nil)
            {
                [blockResetPasswordResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, password];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResetPasswordResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCChangePasswordResponseResult *)changePasswordWithOldPassword:(NSString *)oldPassword
                                                         newPassword:(NSString *)newPassword
                                                               error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCChangePasswordResponseResult  *blockChangePasswordResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient changePasswordWithLoginToken:self.loginTokenInfo.token
                                                                                           oldPassword:oldPassword
                                                                                           newPassword:newPassword
                                                          
                                                                                               success:^(WCTRCChangePasswordResponseResult *changePasswordResponseResult)
                                                          {
                blockChangePasswordResponseResult = [changePasswordResponseResult retain];
                
                [WCTRestClientController encryptAccountPasswordToUserDefault:newPassword];
                
                dispatch_group_leave(group);
            }
                                                                                               failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockChangePasswordResponseResult!=nil)
            {
                [blockChangePasswordResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@ /%@", __PRETTY_FUNCTION__, oldPassword, newPassword];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockChangePasswordResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCAccountUploadLogoResponseResult *)uploadAccountLogoWithImage:(CPImage *)image
                                                            fileName:(NSString *)fileName
                                                            mineType:(NSString *)mineType
                                                               error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCAccountUploadLogoResponseResult  *blockAccountUploadLogoResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient uploadAccountLogoWithLoginToken:self.loginTokenInfo.token
                                                                                                     image:image
                                                                                                  fileName:fileName
                                                                                                  mineType:mineType
                                                                                                   success:^(WCTRCAccountUploadLogoResponseResult *accountLogoReponseResult)
                                                           {
                blockAccountUploadLogoResponseResult = [accountLogoReponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                   failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockAccountUploadLogoResponseResult!=nil)
            {
                [blockAccountUploadLogoResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@ /%@", __PRETTY_FUNCTION__, fileName, fileName];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockAccountUploadLogoResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCAccountInfoResponseResult *)accountInfoWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCAccountInfoResponseResult  *blockAccountInfoResponseResult = nil;
    
    //////////////////////////////////////////////////
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            
            //////////////////////////////////////////////////
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            RKObjectRequestOperation *requestOperation =  [self.restClient accountInfoWithLoginToken:self.loginTokenInfo.token
                                                                                             success:^(WCTRCAccountInfoResponseResult *accountInfoResponseResult)
                                                           {
                blockAccountInfoResponseResult = [accountInfoResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                             failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            if(blockAccountInfoResponseResult!=nil)
            {
                [blockAccountInfoResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            dispatch_release(group);
            
        } while (0);
        
        //////////////////////////////////////////////////
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    //////////////////////////////////////////////////
    return blockAccountInfoResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCOtherAccountListResponseResult *)otherAccountsInfoWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCOtherAccountListResponseResult  *blockAccountInfoResponseResult = nil;
    
    //////////////////////////////////////////////////
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            
            //////////////////////////////////////////////////
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            RKObjectRequestOperation *requestOperation =  [self.restClient otherAccountsInfoWithLoginToken:self.loginTokenInfo.token
                                                                                                   success:^(WCTRCOtherAccountListResponseResult *accountInfoResponseResult)
                                                           {
                blockAccountInfoResponseResult = [accountInfoResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                   failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            if(blockAccountInfoResponseResult!=nil)
            {
                [blockAccountInfoResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            dispatch_release(group);
            
        } while (0);
        
        //////////////////////////////////////////////////
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    //////////////////////////////////////////////////
    return blockAccountInfoResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCAccountInfoResponseResult *)accountBossWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCAccountInfoResponseResult  *blockAccountInfoResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient accountBossWithLoginToken:self.loginTokenInfo.token
                                                                                             success:^(WCTRCAccountInfoResponseResult *accountInfoResponseResult)
                                                           {
                blockAccountInfoResponseResult = [accountInfoResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                             failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockAccountInfoResponseResult!=nil)
            {
                [blockAccountInfoResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        // !!特殊處理, 沒有boss是正常情況，不用log
        if ([WCTRestClientController statusCodeFromAFRKNetworkingError:blockError]!=WCTServer_Common_ErrorCode_DataNotFound)
        {
            [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
        }
    }
    
    /////////////////////////////////////////////////
    
    return blockAccountInfoResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCAccountInfoResponseResult *)accountSubordinatesWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCAccountInfoResponseResult  *blockAccountInfoResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient accountSubordinatesWithLoginToken:self.loginTokenInfo.token
                                                                                                     success:^(WCTRCAccountInfoResponseResult *accountInfoResponseResult)
                                                           {
                blockAccountInfoResponseResult = [accountInfoResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                     failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockAccountInfoResponseResult!=nil)
            {
                [blockAccountInfoResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        // !!特殊處理, 沒有下屬是正常情況，不用log
        if ([WCTRestClientController statusCodeFromAFRKNetworkingError:blockError]!=WCTServer_Common_ErrorCode_DataNotFound)
        {
            [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
        }
    }
    
    /////////////////////////////////////////////////
    
    return blockAccountInfoResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCMessageResponseResult *)changeDisplayName:(NSString *)displayName withError:(NSError **)error;
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient changeDisplayNameWithLoginToken:self.loginTokenInfo.token
                                                                                               displayName:displayName
                                                                                                   success:^(WCTRCMessageResponseResult *changeDisplayNameResponseResult)
                                                           {
                blockResponseResult = [changeDisplayNameResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                   failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, displayName];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}




////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Category Method

//================================================================================
//
//================================================================================
- (WCTRCCategoryCreateResponseResult *)createCategoryWithCategoryName:(NSString *)categoryName
                                            clientCreatedCategoryGuid:(NSString *)clientCreatedCategoryGuid
                                                                error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCCategoryCreateResponseResult  *blockCategoryCreateResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient createCategoryWithLoginToken:self.loginTokenInfo.token
                                                                                          categoryName:categoryName
                                                                             clientCreatedCategoryGuid:clientCreatedCategoryGuid
                                                                                               success:^(WCTRCCategoryCreateResponseResult *categoryCreateResponseResult)
                                                          {
                blockCategoryCreateResponseResult = [categoryCreateResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                               failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockCategoryCreateResponseResult!=nil)
            {
                [blockCategoryCreateResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@ /%@", __PRETTY_FUNCTION__, categoryName, clientCreatedCategoryGuid];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockCategoryCreateResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCCategoryDeleteResponseResult *)deleteCategoryWithCategoryGuids:(NSArray *)categoryGuids
                                                                 error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCCategoryDeleteResponseResult  *blockCategoryDeleteResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient deleteCategoryWithLoginToken:self.loginTokenInfo.token
                                                                                         categoryGuids:categoryGuids
                                                                                               success:^(WCTRCCategoryDeleteResponseResult *categoryDeleteResponseResult)
                                                          {
                blockCategoryDeleteResponseResult = [categoryDeleteResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                               failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockCategoryDeleteResponseResult!=nil)
            {
                [blockCategoryDeleteResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, categoryGuids];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockCategoryDeleteResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCCategoryUpdateResponseResult *)updateCategoryWithNewCategoryName:(NSString *)newCategoryName
                                                            categoryGuid:(NSString *)categoryGuid
                                                                   error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCCategoryUpdateResponseResult  *blockCategoryUpdateResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient updateCategoryWithLoginToken:self.loginTokenInfo.token
                                                                                       newCategoryName:newCategoryName
                                                                                          categoryGuid:categoryGuid
                                                                                               success:^(WCTRCCategoryUpdateResponseResult *categoryUpdateResponseResult)
                                                          {
                blockCategoryUpdateResponseResult = [categoryUpdateResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                               failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockCategoryUpdateResponseResult!=nil)
            {
                [blockCategoryUpdateResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@ /%@", __PRETTY_FUNCTION__, newCategoryName,categoryGuid];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockCategoryUpdateResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCCategoryInfoListResponseResult *)categoryListWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCCategoryInfoListResponseResult  *blockCategoryInfoListResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient categoryListWithLoginToken:self.loginTokenInfo.token
                                                                                         accountGuid:self.accountInfo.guid
                                                                                             success:^(WCTRCCategoryInfoListResponseResult *categoryInfoListResponseResult)
                                                          {
                blockCategoryInfoListResponseResult = [categoryInfoListResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                             failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockCategoryInfoListResponseResult!=nil)
            {
                [blockCategoryInfoListResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockCategoryInfoListResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCDateTimeResponseResult *)categorySetOrderWithRequest:(WCTRCCategoryOrderInfoRequest *)request
                                                       error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCDateTimeResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient categorySetOrderWithLoginToken:self.loginTokenInfo.token
                                                                                                 request:request
                                                                                                 success:^(WCTRCDateTimeResponseResult *categoryOrderResponseResult)
                                                          {
                blockResponseResult = [categoryOrderResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                 failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, request];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCCategoryOrderInfoResponseResult *)categoryGetOrderWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCCategoryOrderInfoResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient categoryGetOrderWithLoginToken:self.loginTokenInfo.token
                                                                                                 success:^(WCTRCCategoryOrderInfoResponseResult *categoryGetOrderResponseResult)
                                                          {
                blockResponseResult = [categoryGetOrderResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                 failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Contact Method

//================================================================================
//
//================================================================================
- (WCTRCMessageResponseResult *)exportToSalesforceWithContactList:(WCTRCListRequest *)contactList
                                                           target:(NSString *)target
                                              salesforceAccountId:(NSString *)salesforceAccountId
                                                            error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient exportToSalesforceWithLoginToken:self.loginTokenInfo.token
                                                                                                contactList:contactList
                                                                                                     target:target
                                                                                        salesforceAccountId:salesforceAccountId              success:^(WCTRCMessageResponseResult *exportToCRMResponseResult)
                                                           {
                blockResponseResult = [exportToCRMResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                    failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@ /%@", __PRETTY_FUNCTION__, contactList, target];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCMessageResponseResult *)exportToExchangeServerWithContactList:(WCTRCListRequest *)contactList
                                                                error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient exportToExchangeServerWithLoginToken:self.loginTokenInfo.token
                                                                                                    contactList:contactList
                                                                                                        success:^(WCTRCMessageResponseResult *exportToContactServerResponseResult)
                                                           {
                blockResponseResult = [exportToContactServerResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                        failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, contactList];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Private Contact Method


//==============================================================================
//
//==============================================================================
- (WCTRCPrivateContactOperateCategoryResponseResult *)privateContactOperateCategoryWithContactSetCategoriesRequest:(WCTRCContactSetCategoriesRequest *)contactSetCategoriesRequest
                                                                                                             error:(NSError **)error
{
    return [self batchPrivateContactOperateCategoryWithContactSetCategoriesRequest:contactSetCategoriesRequest
                                                                   progressHandler:nil
                                                                             error:error];
}


//==============================================================================
//
//==============================================================================
- (WCTRCPrivateContactOperateCategoryResponseResult *)privateContactOperateCategoryWithContactSetCategoriesRequest:(WCTRCContactSetCategoriesRequest *)contactSetCategoriesRequest
                                                                                                   progressHandler:(void(^)(CGFloat progress))progressHandler
                                                                                                             error:(NSError **)error
{
    return [self batchPrivateContactOperateCategoryWithContactSetCategoriesRequest:contactSetCategoriesRequest
                                                                   progressHandler:progressHandler
                                                                             error:error];
}


//================================================================================
//
//================================================================================
- (WCTRCPrivateContactUploadImageResponseResult *)privateContactUploadImage:(CPImage *)image
                                                                   fileName:(NSString *)fileName
                                                                   mineType:(NSString *)mineType
                                                                      error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCPrivateContactUploadImageResponseResult  *blockPrivateContactUploadImageResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient uploadPrivateContactImageWithLoginToken:self.loginTokenInfo.token
                                                                                                             image:image
                                                                                                          fileName:fileName
                                                                                                          mineType:mineType
                                                                                                           success:^(WCTRCPrivateContactUploadImageResponseResult *privateContactUploadImageResponseResult)
                                                           {
                blockPrivateContactUploadImageResponseResult = [privateContactUploadImageResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                           failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockPrivateContactUploadImageResponseResult!=nil)
            {
                [blockPrivateContactUploadImageResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@ /%@", __PRETTY_FUNCTION__, fileName, mineType];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockPrivateContactUploadImageResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCPrivateContactUpdateResponseResult *)updatePrivateContactWithContactUpdateInfoRequest:(WCTRCContactUpdateInfoRequest *)contactUpdateInfoRequest
                                                                                        error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCPrivateContactUpdateResponseResult  *blockPrivateContactUpdateResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient updatePrivateContactWithLoginToken:self.loginTokenInfo.token
                                                                                     contactUpdateInfoRequest:contactUpdateInfoRequest
                                                                                                      success:^(WCTRCPrivateContactUpdateResponseResult *privateContactUpdateResponseResult)
                                                           {
                blockPrivateContactUpdateResponseResult = [privateContactUpdateResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                      failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockPrivateContactUpdateResponseResult!=nil)
            {
                [blockPrivateContactUpdateResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, contactUpdateInfoRequest];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockPrivateContactUpdateResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCPrivateContactManualUploadResponseResult *)manualUploadPrivateContactWithContactCreationWithImageRequest:(WCTRCContactCreationWithImageRequest *)contactCreationWithImageRequest
                                                                                                           error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCPrivateContactManualUploadResponseResult  *blockPrivateContactManualUploadResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient manualUploadPrivateContactWithLoginToken:self.loginTokenInfo.token
                                                                                    contactCreationWithImageRequest:contactCreationWithImageRequest
                                                                                                            success:^(WCTRCPrivateContactManualUploadResponseResult *privateContactManualUploadResponseResult)
                                                           {
                blockPrivateContactManualUploadResponseResult = [privateContactManualUploadResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                            failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockPrivateContactManualUploadResponseResult!=nil)
            {
                [blockPrivateContactManualUploadResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, contactCreationWithImageRequest];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockPrivateContactManualUploadResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCPrivateContactScanUploadResponseResult *)scanUploadPrivateContactWithContactCreationWithImageRequest:(WCTRCContactCreationWithImageRequest *)contactCreationWithImageRequest
                                                                                  contactReceiverAccountGuid:(NSString *)contactReceiverAccountGuid
                                                                                                       error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCPrivateContactScanUploadResponseResult  *blockPrivateContactScanUploadResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient scanUploadPrivateContactWithLoginToken:self.loginTokenInfo.token
                                                                                  contactCreationWithImageRequest:contactCreationWithImageRequest
                                                                                       contactReceiverAccountGuid:contactReceiverAccountGuid
                                                                                                          success:^(WCTRCPrivateContactScanUploadResponseResult *privateContactManualUploadResponseResult)
                                                           {
                blockPrivateContactScanUploadResponseResult = [privateContactManualUploadResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                          failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockPrivateContactScanUploadResponseResult!=nil)
            {
                [blockPrivateContactScanUploadResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, contactCreationWithImageRequest];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockPrivateContactScanUploadResponseResult;
}



//================================================================================
//
//================================================================================
- (WCTRCPrivateContactDeleteResponseResult *)deletePrivateContactWithContactGuids:(NSArray *)contactGuids
                                                                            error:(NSError **)error
{
    return [self batchDeletePrivateContactWithContactGuids:contactGuids
                                           progressHandler:nil
                                                     error:error];
}



//================================================================================
//
//================================================================================
- (WCTRCPrivateContactDeleteResponseResult *)deletePrivateContactWithContactGuids:(NSArray *)contactGuids
                                                                  progressHandler:(void(^)(CGFloat progress))progressHandler
                                                                            error:(NSError **)error
{
    return [self batchDeletePrivateContactWithContactGuids:contactGuids
                                           progressHandler:progressHandler
                                                     error:error];
}

//================================================================================
//
//================================================================================
- (WCTRCPrivateContactListResponseResult *)privateContactListWithContactGetByCategoryRequest:(WCTRCContactGetByCategoryRequest *)contactGetByCategoryRequest
                                                                                       error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCPrivateContactListResponseResult  *blockPrivateContactListResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient privateContactListUnderCategoryWithLoginToken:self.loginTokenInfo.token
                                                                                             contactGetByCategoryRequest:contactGetByCategoryRequest
                                                                                                                 success:^(WCTRCPrivateContactListResponseResult *privateContactListResponseResult)
                                                           {
                blockPrivateContactListResponseResult = [privateContactListResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                                 failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockPrivateContactListResponseResult!=nil)
            {
                [blockPrivateContactListResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, contactGetByCategoryRequest];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockPrivateContactListResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCPrivateContactListResponseResult *)privateContactListWithContactGetByAccountRequest:(WCTRCContactGetByAccountRequest *)contactGetByAccountRequest
                                                                                      error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCPrivateContactListResponseResult  *blockPrivateContactListResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient privateContactListUnderAccountWithLoginToken:self.loginTokenInfo.token
                                                                                             contactGetByAccountRequest:contactGetByAccountRequest
                                                                                                                success:^(WCTRCPrivateContactListResponseResult *privateContactListResponseResult)
                                                           {
                blockPrivateContactListResponseResult = [privateContactListResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                                failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockPrivateContactListResponseResult!=nil)
            {
                [blockPrivateContactListResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, contactGetByAccountRequest];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockPrivateContactListResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCPrivateContactNumberResponseResult *)privateContactNumberUnderAccountWithAccountGuid:(NSString *)accountGuid
                                                                                       error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCPrivateContactNumberResponseResult  *blockPrivateContactNumberResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient privateContactNumberUnderAccountWithLoginToken:self.loginTokenInfo.token
                                                                                                              accountGuid:accountGuid
                                                                                                                  success:^(WCTRCPrivateContactNumberResponseResult *privateContactNumberResponseResult)
                                                           {
                blockPrivateContactNumberResponseResult = [privateContactNumberResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                                  failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockPrivateContactNumberResponseResult!=nil)
            {
                [blockPrivateContactNumberResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, accountGuid];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockPrivateContactNumberResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCPrivateContactToggleFavoriteResponseResult *)toggleFavoriteContactWithContactsSetFavoriteRequest:(WCTRCContactsSetFavoriteRequest *)contactsSetFavoriteRequest
                                                                                                   error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCPrivateContactToggleFavoriteResponseResult  *blockPrivateContactToggleFavoriteResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient toggleFavoriteContactWithLoginToken:self.loginTokenInfo.token
                                                                                    contactsSetFavoriteRequest:contactsSetFavoriteRequest
                                                                                                       success:^(WCTRCPrivateContactToggleFavoriteResponseResult *privateContactToggleFavoriteResponseResult)
                                                           {
                blockPrivateContactToggleFavoriteResponseResult = [privateContactToggleFavoriteResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                       failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockPrivateContactToggleFavoriteResponseResult!=nil)
            {
                [blockPrivateContactToggleFavoriteResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, contactsSetFavoriteRequest];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockPrivateContactToggleFavoriteResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCPrivateContactListSetCorrectedResponseResult *)privateContactListSetCorrectedWithContactGuids:(NSArray *)contactGuids
                                                                                                error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCPrivateContactListSetCorrectedResponseResult  *blockPrivateContactListSetCorrectedResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient privateContactListSetCorrectedWithLoginToken:self.loginTokenInfo.token
                                                                                                           contactGuids:contactGuids
                                                                                                                success:^(WCTRCPrivateContactListSetCorrectedResponseResult *privateContactListSetCorrectedResponseResult)
                                                           {
                blockPrivateContactListSetCorrectedResponseResult = [privateContactListSetCorrectedResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                                failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockPrivateContactListSetCorrectedResponseResult!=nil)
            {
                [blockPrivateContactListSetCorrectedResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, contactGuids];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockPrivateContactListSetCorrectedResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCContactsRemoveFromCategoryResponseResult *)privateContactsRemoveCategoryWithContactsRemoveFromCategoryRequest:(WCTRCContactsRemoveFromCategoryRequest *)contactsRemoveFromCategoryRequest
                                                                                                                error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCContactsRemoveFromCategoryResponseResult  *blockContactsRemoveFromCategoryResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient privateContactsRemoveCategoryWithLoginToken:self.loginTokenInfo.token
                                                                                     contactsRemoveFromCategoryRequest:contactsRemoveFromCategoryRequest
                                                                                                               success:^(WCTRCContactsRemoveFromCategoryResponseResult *contactsRemoveFromCategoryResponseResult)
                                                           {
                blockContactsRemoveFromCategoryResponseResult = [contactsRemoveFromCategoryResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                               failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockContactsRemoveFromCategoryResponseResult!=nil)
            {
                [blockContactsRemoveFromCategoryResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, contactsRemoveFromCategoryRequest];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockContactsRemoveFromCategoryResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCPrivateContactResponseResult *)privateContactWithContactGuid:(NSString *)contactGuid
                                                               error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCPrivateContactResponseResult  *blockPrivateContactResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient privateContactWithLoginToken:self.loginTokenInfo.token
                                                                                            contactGuid:contactGuid
                                                                                                success:^(WCTRCPrivateContactResponseResult *privateContactResponseResult)
                                                           {
                blockPrivateContactResponseResult = [privateContactResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockPrivateContactResponseResult!=nil)
            {
                [blockPrivateContactResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, contactGuid];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockPrivateContactResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCPrivateContactImageBase64ResponseResult *)privateContactImageBase64WithContactGuid:(NSString *)contactGuid
                                                                                 imageType:(NSString *)imageType
                                                                                     error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCPrivateContactImageBase64ResponseResult  *blockPrivateContactImageBase64ResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient privateContactImageBase64WithLoginToken:self.loginTokenInfo.token
                                                                                                       contactGuid:contactGuid
                                                                                                         imageType:imageType
                                                                                                           success:^(WCTRCPrivateContactImageBase64ResponseResult *privateContactImageBase64ResponseResult)
                                                           {
                blockPrivateContactImageBase64ResponseResult = [privateContactImageBase64ResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                           failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockPrivateContactImageBase64ResponseResult!=nil)
            {
                [blockPrivateContactImageBase64ResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@ /%@", __PRETTY_FUNCTION__, contactGuid, imageType];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockPrivateContactImageBase64ResponseResult;
}


//================================================================================
//
//================================================================================
- (CPImage *)privateContactImageWithContactGuid:(NSString *)contactGuid
                                      imageType:(NSString *)imageType
                                          error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block CPImage *blockImage = nil;
    __block NSMutableURLRequest *request = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            AFRKImageRequestOperation *operation = [self.restClient privateContactImageWithLoginToken:self.loginTokenInfo.token
                                                                                          contactGuid:contactGuid
                                                                                            imageType:imageType
                                                                                              success:^(CPImage *image)
                                                    {
                blockImage = [image retain];
                
                dispatch_group_leave(group);
            }
                                                                                              failure:^(NSError *error)
                                                    {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                dispatch_group_leave(group);
            }];
            
            if(operation==nil)
            {
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockImage!=nil)
            {
                [blockImage autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
        
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, request];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockImage;
}


//================================================================================
//
//================================================================================
- (NSData *)privateContactImageDataWithContactGuid:(NSString *)contactGuid
                                         imageType:(NSString *)imageType
                                             error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block NSData *blockImageData = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            AFRKHTTPRequestOperation *operation = [self.restClient privateContactImageDataWithLoginToken:self.loginTokenInfo.token
                                                                                             contactGuid:contactGuid
                                                                                               imageType:imageType
                                                                                                 success:^(NSData *data)
                                                   {
                blockImageData = [data retain];
                
                dispatch_group_leave(group);
            }
                                                                                                 failure:^(NSError *error)
                                                   {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                dispatch_group_leave(group);
            }];
            
            if(operation==nil)
            {
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockImageData!=nil)
            {
                [blockImageData autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
        
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s contactGuid:%@ imageType:%@", __PRETTY_FUNCTION__, contactGuid,imageType];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockImageData;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instace Private Contact Method (V 1.3)

//================================================================================
//
//================================================================================
- (WCTRCPrivateContactOperateCategoryResponseResult *)privateContactOperateMultiLayerCategoryWithContactSetCategoriesRequest:(WCTRCContactSetCategoriesRequest *)contactSetCategoriesRequest
                                                                                                                       error:(NSError **)error
{
    return [self batchPrivateContactOperateMultiLayerCategoryWithContactSetCategoriesRequest:contactSetCategoriesRequest
                                                                             progressHandler:nil
                                                                                       error:error];
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Public Contact Method

//================================================================================
//
//================================================================================
- (WCTRCPublicContactsDownloadByContactGuidReponseResult *)publicContactsDownloadWithContactGuids:(NSArray *)contactGuids
                                                                                            error:(NSError **)error;
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCPublicContactsDownloadByContactGuidReponseResult  *blockPublicContactsDownloadByContactGuidReponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient downloadPublicContactsWithLoginToken:self.loginTokenInfo.token
                                                                                                   contactGuids:contactGuids
                                                                                                        success:^(WCTRCPublicContactsDownloadByContactGuidReponseResult *publicContactsDownloadByContactGuidReponseResult)
                                                           {
                blockPublicContactsDownloadByContactGuidReponseResult = [publicContactsDownloadByContactGuidReponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                        failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockPublicContactsDownloadByContactGuidReponseResult!=nil)
            {
                [blockPublicContactsDownloadByContactGuidReponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
    }
    else
    {
        blockPublicContactsDownloadByContactGuidReponseResult = [self localPublicContactsDownloadByContactGuids:contactGuids error:&blockError];
    }
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, contactGuids];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockPublicContactsDownloadByContactGuidReponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCSharedAccountsResponseResult *)shareAccountsWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSharedAccountsResponseResult  *blockSharedAccountsResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient shareAccountsWithLoginToken:self.loginTokenInfo.token
                                                                                               success:^(WCTRCSharedAccountsResponseResult *sharedAccountsResponseResult)
                                                           {
                blockSharedAccountsResponseResult = [sharedAccountsResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                               failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSharedAccountsResponseResult!=nil)
            {
                [blockSharedAccountsResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    
    /////////////////////////////////////////////////
    
    return blockSharedAccountsResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCContactNubmerByAccountGuidsResponseResult *)contactNumberWithAccountGuids:(NSArray *)accountGuids
                                                                            error:(NSError **)error;
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCContactNubmerByAccountGuidsResponseResult  *blockContactNubmerByAccountGuidsResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient contactNumberWithLoginToken:self.loginTokenInfo.token
                                                                                          accountGuids:accountGuids
                                                                                               success:^(WCTRCContactNubmerByAccountGuidsResponseResult *contactNubmerByAccountGuidsResponseResult)
                                                           {
                blockContactNubmerByAccountGuidsResponseResult = [contactNubmerByAccountGuidsResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                               failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockContactNubmerByAccountGuidsResponseResult!=nil)
            {
                [blockContactNubmerByAccountGuidsResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, accountGuids];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockContactNubmerByAccountGuidsResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCAllContactNumberResponseResult *)allContactNumberWithError:(NSError **)error;
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCAllContactNumberResponseResult  *blockAllContactNumberResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient allContactNumberWithLoginToken:self.loginTokenInfo.token
                                                                                                  success:^(WCTRCAllContactNumberResponseResult *allContactNumberResponseResult)
                                                           {
                blockAllContactNumberResponseResult = [allContactNumberResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                  failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockAllContactNumberResponseResult!=nil)
            {
                [blockAllContactNumberResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockAllContactNumberResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCContactNumberBySelfResponseResult *)contactNumberWithSelfByError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCContactNumberBySelfResponseResult  *blockContactNumberBySelfResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient contactNumberBySelfWithLoginToken:self.loginTokenInfo.token
                                                                                                     success:^(WCTRCContactNumberBySelfResponseResult *contactNumberBySelfResponseResult)
                                                           {
                blockContactNumberBySelfResponseResult = [contactNumberBySelfResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                     failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockContactNumberBySelfResponseResult!=nil)
            {
                [blockContactNumberBySelfResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockContactNumberBySelfResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCSharedContactStatusResponseResult *)shareContactStatusWithContactGuids:(NSArray *)contactGuids
                                                                         error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSharedContactStatusResponseResult  *blockSharedContactStatusResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient shareContactStatusWithLoginToken:self.loginTokenInfo.token
                                                                                               contactGuids:contactGuids
                                                                                                    success:^(WCTRCSharedContactStatusResponseResult *sharedContactStatusResponseResult)
                                                           {
                blockSharedContactStatusResponseResult = [sharedContactStatusResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                    failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSharedContactStatusResponseResult!=nil)
            {
                [blockSharedContactStatusResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
    }
    else
    {
        blockSharedContactStatusResponseResult = [self localShareContactStatusWithContactGuids:contactGuids
                                                                                         error:&blockError];
    }
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, contactGuids];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockSharedContactStatusResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCContactListPageResponseResult *)shareContactListWithContactGetByAccountRequest:(WCTRCContactGetByAccountRequest *)contactGetByAccountRequest
                                                                             withError:(NSError **)error;
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCContactListPageResponseResult  *blockSharedContactsByAccountResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient shareContactListWithLoginToken:self.loginTokenInfo.token
                                                                               contactGetByAccountRequest:contactGetByAccountRequest
                                                                                                  success:^(WCTRCContactListPageResponseResult *sharedContactsByAccountResponseResult)
                                                           {
                blockSharedContactsByAccountResponseResult = [sharedContactsByAccountResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                  failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSharedContactsByAccountResponseResult!=nil)
            {
                [blockSharedContactsByAccountResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
    }
    else
    {
        blockSharedContactsByAccountResponseResult = [self localShareContactListUnderAccountWithError:&blockError];
    }
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, contactGetByAccountRequest];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockSharedContactsByAccountResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCShareStatusForOneContactResponseResult *)shareStatusForOneContactWithContactGuid:(NSString *)contactGuid
                                                                                   error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCShareStatusForOneContactResponseResult  *blockShareStatusForOneContactResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient shareStatusForOneContactWithLoginToken:self.loginTokenInfo.token
                                                                                                      contactGuid:contactGuid
                                                                                                          success:^(WCTRCShareStatusForOneContactResponseResult *shareStatusForOneContactResponseResult)
                                                           {
                blockShareStatusForOneContactResponseResult = [shareStatusForOneContactResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                          failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockShareStatusForOneContactResponseResult!=nil)
            {
                [blockShareStatusForOneContactResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, contactGuid];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockShareStatusForOneContactResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCContactShareAccountResponseResult *)contactShareOperationWithContactShareAccountRequest:(WCTRCContactShareAccountRequest *)contactShareAccountRequest
                                                                                          error:(NSError **)error
{
    return [self batchContactShareOperationWithContactShareAccountRequest:contactShareAccountRequest
                                                          progressHandler:nil
                                                                    error:error];
}


//================================================================================
//
//================================================================================
- (WCTRCContactShareAccountResponseResult *)contactShareOperationWithContactShareAccountRequest:(WCTRCContactShareAccountRequest *)contactShareAccountRequest
                                                                                progressHandler:(void(^)(CGFloat progress))progressHandler
                                                                                          error:(NSError **)error
{
    return [self batchContactShareOperationWithContactShareAccountRequest:contactShareAccountRequest
                                                          progressHandler:progressHandler
                                                                    error:error];
}


//================================================================================
//
//================================================================================
- (WCTRCSetOwnerOfContactResponseResult *)setContactOwnerWithContactGuids:(NSArray *)contactGuids
                                                      newOwnerAccountGuid:(NSString *)newOwnerAccountGuid
                                                                    error:(NSError **)error
{
    return [self batchSetContactOwnerWithContactGuids:contactGuids
                                  newOwnerAccountGuid:newOwnerAccountGuid
                                      progressHandler:nil
                                                error:error];
}


//================================================================================
//
//================================================================================
- (WCTRCSetOwnerOfContactResponseResult *)setContactOwnerWithContactGuids:(NSArray *)contactGuids
                                                      newOwnerAccountGuid:(NSString *)newOwnerAccountGuid
                                                          progressHandler:(void(^)(CGFloat progress))progressHandler
                                                                    error:(NSError **)error
{
    return [self batchSetContactOwnerWithContactGuids:contactGuids
                                  newOwnerAccountGuid:newOwnerAccountGuid
                                      progressHandler:progressHandler
                                                error:error];
}

//==============================================================================
//
//==============================================================================
- (WCTRCContactListPageResponseResult *)contactListWithRequest:(WCTRCContactGetByGuidListRequest *)request
                                                         error:(NSError **)error;
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCContactListPageResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient contactsWithLoginToken:self.loginTokenInfo.token
                                                                                          request:request
                                                                                          success:^(WCTRCContactListPageResponseResult *contactListPageResponseResult)
                                                           {
                blockResponseResult = [contactListPageResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                          failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, request];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Public Contact Method (V 1.3)


//================================================================================
//
//================================================================================
- (WCTRCSharedContactNumberIsntAccountCategoryResponseResult *)sharedContactNumberIsntAccountCategoryWithAccountGuid:(NSString *)accountGuid
                                                                                                               error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSharedContactNumberIsntAccountCategoryResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient sharedContactNumberIsntAccountCategoryWithLoginToken:self.loginTokenInfo.token accountGuid:accountGuid success:^(WCTRCSharedContactNumberIsntAccountCategoryResponseResult *multiLayerCategorySharedContactNumberIsntAccountCategoryResponseResult
                                                                                                                                                                                           )
                                                          {
                blockResponseResult = [multiLayerCategorySharedContactNumberIsntAccountCategoryResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                                       failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCMLCategorySharedContactNumberWithCategoryGuidsResponseResult *)sharedContactNumberWithCategoryGuids:(NSArray *)categoryGuids
                                                                                          categoryOwnerGuid:(NSString *)categoryOwnerGuid
                                                                                                      error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMLCategorySharedContactNumberWithCategoryGuidsResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient sharedContactNumberWithLoginToken:self.loginTokenInfo.token
                                                                                              categoryGuids:categoryGuids categoryOwnerGuid:categoryOwnerGuid success:^(WCTRCMLCategorySharedContactNumberWithCategoryGuidsResponseResult *multiLayerSharedContactNumberWithCategoryGuidsResponseResult)
                                                          {
                blockResponseResult = [multiLayerSharedContactNumberWithCategoryGuidsResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                    failure:^(NSError *failureError) {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s categoryGuids: %@, categoryOwnerGuid: %@", __PRETTY_FUNCTION__, categoryGuids, categoryOwnerGuid];
        
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCContactListPageResponseResult *)sharedContactListIsntAccountCategoryWithRequest:(WCTRCContactGetByAccountRequest *)request
                                                                                  error:(NSError **)error;
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCContactListPageResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient sharedContactListIsntAccountCategoryWithLoginToken:self.loginTokenInfo.token contactGetByAccountRequest:request
                                                                                                                     success:^(WCTRCContactListPageResponseResult *listPageResponseResult) {
                blockResponseResult = [listPageResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                                     failure:^(NSError *failureError) {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s accountGuid: %@", __PRETTY_FUNCTION__, self.accountInfo.guid];
        
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCContactListPageResponseResult *)sharedContactListWithRequest:(WCTRCContactGetByCategoryAndOwnerVoRequest *)request
                                                               error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCContactListPageResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient sharedContactListWithLoginToken:self.loginTokenInfo.token
                                                                    contactGetByCategoryAndOwnerVoRequest:request
                                                                                                  success:^(WCTRCContactListPageResponseResult *contactListPageResponseResult) {
                blockResponseResult = [contactListPageResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                  failure:^(NSError *failureError) {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request: %@", __PRETTY_FUNCTION__, request];
        
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instace Multi Layer Category Method


//================================================================================
//
//================================================================================
- (WCTRCMLCategoryCreationResponseResult *)createMultiLayerCategoryWithCreationRequest:(WCTRCMLCategoryCreationRequest *)multiLayerCategoryCreationRequest

                                                                                 error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMLCategoryCreationResponseResult  *blockMultiLayerCategoryCreateResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            if(multiLayerCategoryCreationRequest==nil)
            {
                blockError = WCTRestClientErrorMake(@"WCTRCMLCategoryCreationRequest 不存在",blockError);
                
                break;
            }
            
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient
                                                          createMultiLayerCategoryWithLoginToken:self.loginTokenInfo.token
                                                          multiLayerCategoryCreationRequest:multiLayerCategoryCreationRequest
                                                          success:^(WCTRCMLCategoryCreationResponseResult *multiLayerCategoryCreationResponseResult) {
                blockMultiLayerCategoryCreateResponseResult = [multiLayerCategoryCreationResponseResult retain];
                dispatch_group_leave(group);
            }
                                                          failure:^(NSError *failureError) {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockMultiLayerCategoryCreateResponseResult!=nil)
            {
                [blockMultiLayerCategoryCreateResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, multiLayerCategoryCreationRequest];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockMultiLayerCategoryCreateResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCMLCategoryPathCreationResponseResult *)createMultiLayerCategoryPathWithCreationRequest:(WCTRCMLCategoryPathCreationRequest *)categoryPathCreationRequest
                                                                                         error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMLCategoryPathCreationResponseResult  *blockCategoryPathCreationResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            if(categoryPathCreationRequest==nil)
            {
                blockError = WCTRestClientErrorMake(@"WCTRCMLCategoryCreationRequest 不存在",blockError);
                
                break;
            }
            
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient createMultiLayerCategoryPathWithLoginToken:self.loginTokenInfo.token categoryPathCreationRequest:categoryPathCreationRequest success:^(WCTRCMLCategoryPathCreationResponseResult *categoryPathCreationResponseResult) {
                blockCategoryPathCreationResponseResult = [categoryPathCreationResponseResult retain];
                dispatch_group_leave(group);
            } failure:^(NSError *failureError) {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
                
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockCategoryPathCreationResponseResult!=nil)
            {
                [blockCategoryPathCreationResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, categoryPathCreationRequest];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockCategoryPathCreationResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCMLCategoryDeleteResponseResult *)deleteMultiLayerCategoryWithCategoryGuid:(NSString *)categoryGuid
                                                                            error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMLCategoryDeleteResponseResult  *blockMultiLayerCategoryDeleteResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient deleteMultiLayerCategoryWithLoginToken:self.loginTokenInfo.token
                                                                                                    categoryGuid:categoryGuid success:^(WCTRCMLCategoryDeleteResponseResult *multiLayerCategoryDeleteResponseResult) {
                blockMultiLayerCategoryDeleteResponseResult = [multiLayerCategoryDeleteResponseResult retain];
                dispatch_group_leave(group);
                
            } failure:^(NSError *failureError) {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockMultiLayerCategoryDeleteResponseResult!=nil)
            {
                [blockMultiLayerCategoryDeleteResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, categoryGuid];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockMultiLayerCategoryDeleteResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCMLCategoryInfoListResponseResult *)multiLayerCategoryListWithAccountGuid:(NSString *)accountGuid
                                                                           error:(NSError **)error;
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMLCategoryInfoListResponseResult  *blockMultiLayerCategoryInfoListResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient multiLayerCategoryListWithLoginToken:self.loginTokenInfo.token
                                                                                                   accountGuid:accountGuid success:^(WCTRCMLCategoryInfoListResponseResult *multiLayerCategoryInfoListResponseResult) {
                blockMultiLayerCategoryInfoListResponseResult = [multiLayerCategoryInfoListResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                       failure:^(NSError *failureError) {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockMultiLayerCategoryInfoListResponseResult!=nil)
            {
                [blockMultiLayerCategoryInfoListResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s accountGuid: %@", __PRETTY_FUNCTION__, accountGuid];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockMultiLayerCategoryInfoListResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCMLCategoryInfoListResponseResult *)movableMultiLayerCategoryListWithAccountGuid:(NSString *)accountGuid movedCategoryGuid:(NSString *)movedCategoryGuid
                                                                                  error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMLCategoryInfoListResponseResult  *blockMultiLayerCategoryInfoListResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient movableMultiLayerCategoryListWithLoginToken:self.loginTokenInfo.token accountGuid:accountGuid movedCategoryGuid:movedCategoryGuid success:^(WCTRCMLCategoryInfoListResponseResult *multiLayerCategoryInfoListResponseResult) {
                blockMultiLayerCategoryInfoListResponseResult = [multiLayerCategoryInfoListResponseResult retain];
                dispatch_group_leave(group);
            } failure:^(NSError *failureError) {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockMultiLayerCategoryInfoListResponseResult!=nil)
            {
                [blockMultiLayerCategoryInfoListResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s movedCategoryGuid: %@", __PRETTY_FUNCTION__, movedCategoryGuid];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockMultiLayerCategoryInfoListResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCMLCategoryInfoListResponseResult *)multiLayerCategoryOneLayerListWithParentCategoryGuid:(NSString *)parentCategoryGuid
                                                                               needMarkHasChild:(BOOL)needMarkHasChild
                                                                                          error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMLCategoryInfoListResponseResult  *blockMultiLayerCategoryInfoListResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient multiLayerCategoryOneLayerListWithLoginToken:self.loginTokenInfo.token
                                                                                                           accountGuid:self.accountInfo.guid
                                                                                                    parentCategoryGuid:parentCategoryGuid
                                                                                                      needMarkHasChild:needMarkHasChild
                                                                                                               success:^(WCTRCMLCategoryInfoListResponseResult *multiLayerCategoryInfoListResponseResult) {
                blockMultiLayerCategoryInfoListResponseResult = [multiLayerCategoryInfoListResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                               failure:^(NSError *failureError) {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockMultiLayerCategoryInfoListResponseResult!=nil)
            {
                [blockMultiLayerCategoryInfoListResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s parentCategoryGuid: %@, needMarkHasChild: %d", __PRETTY_FUNCTION__, parentCategoryGuid, needMarkHasChild];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockMultiLayerCategoryInfoListResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCMLCategoryInfoListResponseResult *)toggledMultiLayerCategoryListWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMLCategoryInfoListResponseResult  *blockMultiLayerCategoryInfoListResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient toggledMultiLayerCategoryListWithLoginToken:self.loginTokenInfo.token success:^(WCTRCMLCategoryInfoListResponseResult *multiLayerCategoryInfoListResponseResult) {
                blockMultiLayerCategoryInfoListResponseResult = [multiLayerCategoryInfoListResponseResult retain];
                dispatch_group_leave(group);
            } failure:^(NSError *failureError) {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockMultiLayerCategoryInfoListResponseResult!=nil)
            {
                [blockMultiLayerCategoryInfoListResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockMultiLayerCategoryInfoListResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCMLCategorySetOrderForToggledCategoriesResponseResult *)setOrderToggledMultiLayerCategoryListWithRequest:(WCTRCMLSetOrderListRequest *)request
                                                                                                          error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMLCategorySetOrderForToggledCategoriesResponseResult  *blockSetOrderForToggledCategoriesResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient setOrderToggledMultiLayerCategoryListWithRequest:request loginToken:self.loginTokenInfo.token success:^(WCTRCMLCategorySetOrderForToggledCategoriesResponseResult *toggledCategoriesResponseResult) {
                blockSetOrderForToggledCategoriesResponseResult = [toggledCategoriesResponseResult retain];
                dispatch_group_leave(group);
                
            } failure:^(NSError *failureError) {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                dispatch_group_leave(group);
                
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSetOrderForToggledCategoriesResponseResult!=nil)
            {
                [blockSetOrderForToggledCategoriesResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request: %@", __PRETTY_FUNCTION__, request];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockSetOrderForToggledCategoriesResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCMLCategoryToggledCategoriesResponseResult *)setToggledMultiLayerWithCategoryToggleUsualVo:(WCTRCCategoryToggleUsualVo *)categoryToggleUsualVo
                                                                                            error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMLCategoryToggledCategoriesResponseResult  *blockToggledCategoriesResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient setToggledMultiLayerWithCategoryToggleUsualVo:categoryToggleUsualVo
                                                                                                             loginToken:self.loginTokenInfo.token success:^(WCTRCMLCategoryToggledCategoriesResponseResult *toggledCategoriesResponseResult) {
                blockToggledCategoriesResponseResult = [toggledCategoriesResponseResult retain];
                dispatch_group_leave(group);
            } failure:^(NSError *failureError) {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockToggledCategoriesResponseResult!=nil)
            {
                [blockToggledCategoriesResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s categoryToggleUsualVo: %@", __PRETTY_FUNCTION__, categoryToggleUsualVo];
        
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockToggledCategoriesResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCMLCategoryUpdateResponseResult *)updateMultiLayerCategoryWithGuid:(NSString *)guid
                                                             categoryName:(NSString *)categoryName
                                                                    error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMLCategoryUpdateResponseResult  *blockCategoryUpdateResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient updateMultiLayerCategoryWithGuid:guid
                                                                                              categoryName:categoryName loginToken:self.loginTokenInfo.token success:^(WCTRCMLCategoryUpdateResponseResult *categoryUpdateResponseResult) {
                blockCategoryUpdateResponseResult = [categoryUpdateResponseResult retain];
                dispatch_group_leave(group);
                
            } failure:^(NSError *failureError) {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockCategoryUpdateResponseResult!=nil)
            {
                [blockCategoryUpdateResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s categoryGuid%@, categoryName: %@", __PRETTY_FUNCTION__, guid,categoryName];
        
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockCategoryUpdateResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCMLCategoryMoveResponseResult *)moveMultiLayerCategoryWithGuid:(NSString *)guid
                                                 toParentCategoryGuid:(NSString *)parentCategoryGuid
                                                    categoryOwnerGuid:(NSString *)categoryOwnerGuid
                                                                error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMLCategoryMoveResponseResult  *blockCategoryMoveResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient moveMultiLayerCategoryWithGuid:guid
                                                                                    toParentCategoryGuid:parentCategoryGuid
                                                                                       categoryOwnerGuid:categoryOwnerGuid
                                                                                              loginToken:self.loginTokenInfo.token success:^(WCTRCMLCategoryMoveResponseResult *categoryMoveResponseResult) {
                blockCategoryMoveResponseResult = [categoryMoveResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                 failure:^(NSError *failureError) {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockCategoryMoveResponseResult!=nil)
            {
                [blockCategoryMoveResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s categoryGuid%@, parentCategoryGuid: %@, categoryOwnerGuid: %@", __PRETTY_FUNCTION__, guid,parentCategoryGuid,categoryOwnerGuid];
        
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockCategoryMoveResponseResult;
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Search Method

//================================================================================
//
//================================================================================
- (WCTRCSearchPublicContactsByKeywordResponseResult *)searchPublicContactsByKeywordWithKeyword:(NSString *)keyword
                                                                                         error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSearchPublicContactsByKeywordResponseResult  *blockSearchContactsByKeywordResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient searchPublicContactsByKeywordWithLoginToken:self.loginTokenInfo.token
                                                                                                               keyword:keyword
                                                                                                               success:^(WCTRCSearchPublicContactsByKeywordResponseResult *searchPublicContactsByKeywordResponseResult)
                                                           {
                blockSearchContactsByKeywordResponseResult = [searchPublicContactsByKeywordResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                               failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSearchContactsByKeywordResponseResult!=nil)
            {
                [blockSearchContactsByKeywordResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
    }
    else
    {
        blockSearchContactsByKeywordResponseResult = [self localSearchContactsByKeywordWithKeyword:keyword error:&blockError];
    }
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, keyword];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockSearchContactsByKeywordResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCSearchPublicContactsByKeywordSortedResponseResult *)searchPublicContactsByKeywordSortedWithKeyword:(NSString *)keyword
                                                                                                      page:(NSUInteger)page
                                                                                                  pageSize:(NSUInteger)pageSize
                                                                                contactsimpleinfoSortField:(NSString *)contactsimpleinfoSortField
                                                                                             sortDirection:(NSString *)sortDirection
                                                                                                     error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSearchPublicContactsByKeywordSortedResponseResult  *blockSearchPublicContactsByKeywordSortedResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient searchPublicContactsByKeywordSortedWithLoginToken:self.loginTokenInfo.token
                                                                                                                     keyword:keyword
                                                                                                                        page:page
                                                                                                                    pageSize:pageSize
                                                                                                  contactsimpleinfoSortField:contactsimpleinfoSortField
                                                                                                               sortDirection:sortDirection
                                                                                                                     success:^(WCTRCSearchPublicContactsByKeywordSortedResponseResult *searchPublicContactsByKeywordSortedResponseResult)
                                                           
                                                           {
                blockSearchPublicContactsByKeywordSortedResponseResult = [searchPublicContactsByKeywordSortedResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                                     failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSearchPublicContactsByKeywordSortedResponseResult!=nil)
            {
                [blockSearchPublicContactsByKeywordSortedResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
    }
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@ /%td /%td /%@ /%@", __PRETTY_FUNCTION__, keyword, page, pageSize, contactsimpleinfoSortField, sortDirection];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockSearchPublicContactsByKeywordSortedResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCSearchPrivateContactsByKeywordResponseResult *)searchPrivateContactsByKeywordWithKeyword:(NSString *)keyword
                                                                                           error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSearchPrivateContactsByKeywordResponseResult  *blockSearchPrivateContactsByKeywordResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient searchPrivateContactsByKeywordWithLoginToken:self.loginTokenInfo.token
                                                                                                                keyword:keyword
                                                                                                                success:^(WCTRCSearchPrivateContactsByKeywordResponseResult *searchPrivateContactsByKeywordResponseResult)
                                                           {
                blockSearchPrivateContactsByKeywordResponseResult = [searchPrivateContactsByKeywordResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                                failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSearchPrivateContactsByKeywordResponseResult!=nil)
            {
                [blockSearchPrivateContactsByKeywordResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, keyword];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockSearchPrivateContactsByKeywordResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCAdvancedSearchContactListResposeResult *)advancedSearchWithRequest:(WCTRCAdvancedSearchVoRequest *)request
                                                                     error:(NSError **)error;
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCAdvancedSearchContactListResposeResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation =  [self.restClient advancedSearchWithLoginToken:self.loginTokenInfo.token
                                                                                                request:request
                                                                                                success:^(WCTRCAdvancedSearchContactListResposeResult *searchResponseResult)
                                                           {
                blockResponseResult = [searchResponseResult retain];
                dispatch_group_leave(group);
                
            }
                                                                                                failure:^(NSError *failureError)
                                                           {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, request];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Notice methods


//==============================================================================
//
//==============================================================================
- (WCTRCMessageResponseResult *)setLastNoticeReadWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult  *blockResponceResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient setLastNoticeReadWithLoginToken:self.loginTokenInfo.token
                                                                                                  success:^(WCTRCMessageResponseResult *lastNoticeReadResponseResult)
                                                          {
                blockResponceResult = [lastNoticeReadResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                  failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponceResult!=nil)
            {
                [blockResponceResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponceResult;
    
}


//==============================================================================
//
//==============================================================================
- (WCTRCLongResponseResult *)unreadNoticeWithNoticeCategory:(NSString *)noticeCategory
                                                      error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCLongResponseResult  *blockUnreadNoticeResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient unreadNoticeWithLoginToken:self.loginTokenInfo.token
                                                                                      noticeCategory:noticeCategory
                                                                                             success:^(WCTRCLongResponseResult *unreadNoticeResponseResult)
                                                          {
                blockUnreadNoticeResult = [unreadNoticeResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                             failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockUnreadNoticeResult!=nil)
            {
                [blockUnreadNoticeResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, noticeCategory];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockUnreadNoticeResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCNoticeListResponseResult *)noticeListWithNoticeCategory:(NSString *)noticeCategory
                                                          error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCNoticeListResponseResult  *blockNoticeListResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient noticeListWithLoginToken:self.loginTokenInfo.token
                                                                                    noticeCategory:noticeCategory
                                                                                           success:^(WCTRCNoticeListResponseResult *noticeListResponseResult)
                                                          {
                blockNoticeListResponseResult = [noticeListResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                           failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockNoticeListResponseResult!=nil)
            {
                [blockNoticeListResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, noticeCategory];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockNoticeListResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCNoticeListPageResponseResult *)noticeListPageWithNoticeCategory:(NSString *)noticeCategory
                                                       noticeReadStatus:(NSString *)noticeReadStatus
                                                              pageIndex:(NSInteger)pageIndex
                                                               pageSize:(NSInteger)pageSize
                                                              sortField:(NSString *)sortField
                                                          sortDirection:(NSString *)sortDirection
                                                                  error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCNoticeListPageResponseResult  *blockNoticeListPageResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient noticeListPageWithLoginToken:self.loginTokenInfo.token
                                                                                        noticeCategory:noticeCategory
                                                                                      noticeReadStatus:noticeReadStatus
                                                                                             pageIndex:pageIndex
                                                                                              pageSize:pageSize
                                                                                             sortField:sortField
                                                                                         sortDirection:sortDirection
                                                                                               success:^(WCTRCNoticeListPageResponseResult *noticeListPageResponseResult)
                                                          {
                blockNoticeListPageResponseResult = [noticeListPageResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                               failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockNoticeListPageResponseResult!=nil)
            {
                [blockNoticeListPageResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@ /%@ /%ld /%ld /%@", __PRETTY_FUNCTION__, noticeCategory, noticeReadStatus, (long)pageIndex, (long)pageSize, sortField];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockNoticeListPageResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCMessageResponseResult *)noticeContentWithGuid:(NSString *)noticeGuid
                                                error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient noticeContentWithLoginToken:self.loginTokenInfo.token
                                                                                           noticeGuid:noticeGuid
                                                                                              success:^(WCTRCMessageResponseResult *noticeContentResponseResult)
                                                          {
                blockResponseResult = [noticeContentResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                              failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, noticeGuid];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCMessageResponseResult *)noticeSendShareContacts:(NSArray *)contacts
                                      toReceiveAccounts:(NSArray *)accounts
                                                  error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient noticeShareContactsToAccountsWithLoginToken:self.loginTokenInfo.token
                                                                                                             contacts:contacts
                                                                                                             accounts:accounts                                                                                               success:^(WCTRCMessageResponseResult *noticeContentResponseResult)
                                                          {
                blockResponseResult = [noticeContentResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                              failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s contacts:%@, accounts:%@", __PRETTY_FUNCTION__, contacts,accounts];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance NormalUserSetting Method


//==============================================================================
//
//==============================================================================
- (WCTRCAllUserSettingsResponseResult *)allUserSettingsResponseWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCAllUserSettingsResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient allUserSettingsWithLoginToken:self.loginTokenInfo.token
                                                                                                success:^(WCTRCAllUserSettingsResponseResult *allUserSettingsResponseResult)
                                                          {
                blockResponseResult = [allUserSettingsResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
    
}


//================================================================================
//
//================================================================================
- (WCTRCSupportMapEngineResponseResult *)supportMapEngineResponseWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSupportMapEngineResponseResult  *blockSupportMapEngineResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient supportMapEnginesWithLoginToken:self.loginTokenInfo.token
                                                                                                  success:^(WCTRCSupportMapEngineResponseResult *supportMapEngineResponseResult)
                                                          {
                blockSupportMapEngineResponseResult = [supportMapEngineResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                  failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSupportMapEngineResponseResult!=nil)
            {
                [blockSupportMapEngineResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockSupportMapEngineResponseResult;
}



//================================================================================
//
//================================================================================
- (WCTRCNowMapEngineResponseResult *)nowMapEngineResponseWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCNowMapEngineResponseResult  *blockNowMapEngineResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient nowMapEnginesWithLoginToken:self.loginTokenInfo.token
                                                                                              success:^(WCTRCNowMapEngineResponseResult *nowMapEngineResponseResult)
                                                          {
                blockNowMapEngineResponseResult = [nowMapEngineResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                              failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockNowMapEngineResponseResult!=nil)
            {
                [blockNowMapEngineResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockNowMapEngineResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCAsianSurnameOrderResult *)asianSurnameOrderWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCAsianSurnameOrderResult  *blockAsianSurnameOrderResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient asianSurnameOrderWithLoginToken:self.loginTokenInfo.token
                                                                                                  success:^(WCTRCAsianSurnameOrderResult *asianSurnameOrderResult)
                                                          {
                blockAsianSurnameOrderResult = [asianSurnameOrderResult retain];
                dispatch_group_leave(group);
            }
                                                                                                  failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockAsianSurnameOrderResult!=nil)
            {
                [blockAsianSurnameOrderResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockAsianSurnameOrderResult;
}


//================================================================================
//
//================================================================================
- (WCTRCWesternSurnameOrderSettingResult *)westernSurnameOrderWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCWesternSurnameOrderSettingResult  *blockWesternSurnameOrderSettingResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient westernSurnameOrderWithLoginToken:self.loginTokenInfo.token
                                                                                                    success:^(WCTRCWesternSurnameOrderSettingResult *westernSurnameOrderSettingResult)
                                                          {
                blockWesternSurnameOrderSettingResult = [westernSurnameOrderSettingResult retain];
                dispatch_group_leave(group);
            }
                                                                                                    failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockWesternSurnameOrderSettingResult!=nil)
            {
                [blockWesternSurnameOrderSettingResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockWesternSurnameOrderSettingResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCAutoShareResponseResult *)autoShareSettingWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCAutoShareResponseResult  *blockAutoShareResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient autoShareSettingWithLoginToken:self.loginTokenInfo.token
                                                                                                 success:^(WCTRCAutoShareResponseResult *autoShareResponseResult)
                                                          {
                blockAutoShareResponseResult = [autoShareResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                 failure:^(NSError *error)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                
                dispatch_group_leave(group);
            } ];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockAutoShareResponseResult!=nil)
            {
                [blockAutoShareResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockAutoShareResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCAccountSharedTargetsResponseResult *)accountSharedTargetsWithShareItem:(NSString *)shareItem
                                                                         error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCAccountSharedTargetsResponseResult  *blockAccountSharedTargetsResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient accountSharedTargetsWithLoginToken:self.loginTokenInfo.token
                                                                                                   shareItem:shareItem
                                                                                                     success:^(WCTRCAccountSharedTargetsResponseResult *accountSharedTargetsResponseResult)
                                                          {
                blockAccountSharedTargetsResponseResult = [accountSharedTargetsResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                     failure:^(NSError *error)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                
                dispatch_group_leave(group);
            } ];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockAccountSharedTargetsResponseResult!=nil)
            {
                [blockAccountSharedTargetsResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, shareItem];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockAccountSharedTargetsResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCShareRoleResponseResult *)globalShareRoleWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCShareRoleResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient globalShareRoleWithLoginToken:self.loginTokenInfo.token
                                                                                                success:^(WCTRCShareRoleResponseResult *shareRoleResponseResult)
                                                          {
                blockResponseResult = [shareRoleResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                failure:^(NSError *error)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                
                dispatch_group_leave(group);
            } ];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCChineseRecognizeResultOutputResponseResult*)chineseRecognizeResultOutputWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCChineseRecognizeResultOutputResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient chineseRecognizeResultOutputWithLoginToken:self.loginTokenInfo.token
                                                                                                             success:^(WCTRCChineseRecognizeResultOutputResponseResult *chineseRecognizeResultOutputResponseResult)
                                                          {
                blockResponseResult = [chineseRecognizeResultOutputResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                             failure:^(NSError *error)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                
                dispatch_group_leave(group);
            } ];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCSortByFieldResponseResult*)sortByFieldWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSortByFieldResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient sortByFieldWithLoginToken:self.loginTokenInfo.token
                                                                                            success:^(WCTRCSortByFieldResponseResult *sortByFieldResponseResult)
                                                          {
                blockResponseResult = [sortByFieldResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                            failure:^(NSError *error)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                
                dispatch_group_leave(group);
            } ];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCCustomFieldListResponseResult *)customFieldsWithCategory:(NSString *)category
                                                           error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCCustomFieldListResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient customFieldInfosWithLoginToken:self.loginTokenInfo.token
                                                                                     customFieldCategory:category
                                                                                                 success:^(WCTRCCustomFieldListResponseResult *customFieldListResponseResult)
                                                          {
                blockResponseResult = [customFieldListResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                 failure:^(NSError *error)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                
                dispatch_group_leave(group);
            } ];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, category];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//==============================================================================
//
//==============================================================================
- (NSInteger)maxPrivateContactsCountWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient propertyWithLoginToken:self.loginTokenInfo.token
                                                                                    propertyType:WCTRC_SystemProperty_MaxPrivateContactsCount
                                                                                         success:^(WCTRCMessageResponseResult *maxContactsCountResponseResult) {
                blockResponseResult = [maxContactsCountResponseResult retain];
                dispatch_group_leave(group);
            }failure:^(NSError *error){
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    if (blockResponseResult)
    {
        return [blockResponseResult.data integerValue];
    }
    else
    {
        return 0;
    }
}


//==============================================================================
//
//==============================================================================
- (NSInteger)maxPublicContactsCountWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient propertyWithLoginToken:self.loginTokenInfo.token
                                                                                    propertyType:WCTRC_SystemProperty_MaxPublicContactsCount
                                                                                         success:^(WCTRCMessageResponseResult *maxContactsCountResponseResult) {
                blockResponseResult = [maxContactsCountResponseResult retain];
                dispatch_group_leave(group);
            }failure:^(NSError *error){
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    if (blockResponseResult)
    {
        return [blockResponseResult.data integerValue];
    }
    else
    {
        return 0;
    }
}



//==============================================================================
//
//==============================================================================
- (NSInteger)maxServerContactsCountWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient propertyWithLoginToken:self.loginTokenInfo.token
                                                                                    propertyType:WCTRC_SystemProperty_MaxServerContactsCount
                                                                                         success:^(WCTRCMessageResponseResult *maxContactsCountResponseResult) {
                blockResponseResult = [maxContactsCountResponseResult retain];
                dispatch_group_leave(group);
            }failure:^(NSError *error){
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    if (blockResponseResult)
    {
        return [blockResponseResult.data integerValue];
    }
    else
    {
        return 0;
    }
}


//================================================================================
//
//================================================================================
- (WCTRCSetSurnameOrderResponseResult *)setAsianSurnameOrderWithNameOrder:(NSString *)nameOrder
                                                                    error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSetSurnameOrderResponseResult  *blockSetSurnameOrderResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient setAsianSurnameOrderWithLoginToken:self.loginTokenInfo.token
                                                                                                   nameOrder:nameOrder
                                                                                                     success:^(WCTRCSetSurnameOrderResponseResult *setSurnameOrderResponseResult)
                                                          {
                blockSetSurnameOrderResponseResult = [setSurnameOrderResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                     failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSetSurnameOrderResponseResult!=nil)
            {
                [blockSetSurnameOrderResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, nameOrder];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockSetSurnameOrderResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCSetSurnameOrderResponseResult *)setWesternSurnameOrderWithNameOrder:(NSString *)nameOrder
                                                                      error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSetSurnameOrderResponseResult  *blockSetSurnameOrderResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient setWesternSurnameOrderWithLoginToken:self.loginTokenInfo.token
                                                                                                     nameOrder:nameOrder
                                                                                                       success:^(WCTRCSetSurnameOrderResponseResult *setSurnameOrderResponseResult)
                                                          {
                blockSetSurnameOrderResponseResult = [setSurnameOrderResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                       failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSetSurnameOrderResponseResult!=nil)
            {
                [blockSetSurnameOrderResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, nameOrder];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockSetSurnameOrderResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCSetUserMapEngineResponseResult *)setMapEngineWithMapEngine:(NSString *)mapEngine
                                                             error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSetUserMapEngineResponseResult  *blockSetUserMapEngineResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient setMapEngineWithLoginToken:self.loginTokenInfo.token
                                                                                           mapEngine:mapEngine
                                                                                             success:^(WCTRCSetUserMapEngineResponseResult *setUserMapEngineResponseResult)
                                                          {
                blockSetUserMapEngineResponseResult = [setUserMapEngineResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                             failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSetUserMapEngineResponseResult!=nil)
            {
                [blockSetUserMapEngineResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, mapEngine];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockSetUserMapEngineResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCSetAutoShareSettingResponseResult *)setAutoShareSettingWithEnable:(BOOL)enable
                                                                    error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSetAutoShareSettingResponseResult  *blockSetAutoShareSettingResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient setAutoShareSettingWithLoginToken:self.loginTokenInfo.token
                                                                                                     enable:enable                                                                                             success:^(WCTRCSetAutoShareSettingResponseResult *setAutoShareSettingResponseResult)
                                                          {
                blockSetAutoShareSettingResponseResult = [setAutoShareSettingResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                    failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSetAutoShareSettingResponseResult!=nil)
            {
                [blockSetAutoShareSettingResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, enable?@"YES":@"NO"];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockSetAutoShareSettingResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCMessageResponseResult *)addAccountSharedTargetWithRequest:(WCTRCAccountSharedTargetRequest *)request
                                                            error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult  *blockAddAccountSharedTargetResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient addAccountShareTargetWithLoginToken:self.loginTokenInfo.token
                                                                                                      request:request
                                                                                                      success:^(WCTRCMessageResponseResult *addAccountSahreTargetResponseResult)
                                                          {
                blockAddAccountSharedTargetResponseResult = [addAccountSahreTargetResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                      failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockAddAccountSharedTargetResponseResult!=nil)
            {
                [blockAddAccountSharedTargetResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, request];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockAddAccountSharedTargetResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCMessageResponseResult *)removeAccountSharedTargetWithRequest:(WCTRCAccountSharedTargetRequest *)request
                                                               error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult  *blockRemoveAccountSharedTargetResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient removeAccountShareTargetWithLoginToken:self.loginTokenInfo.token
                                                                                                         request:request
                                                                                                         success:^(WCTRCMessageResponseResult *removeAccountSahreTargetResponseResult)
                                                          {
                blockRemoveAccountSharedTargetResponseResult = [removeAccountSahreTargetResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                         failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockRemoveAccountSharedTargetResponseResult!=nil)
            {
                [blockRemoveAccountSharedTargetResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, request];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockRemoveAccountSharedTargetResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCMessageResponseResult *)updateAccountSharedTargetListWithRequest:(WCTRCAccountSharedTargetListRequest *)request
                                                                   error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult  *blockUpdateAccountSharedTargetResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient updateAccountShareTargetWithLoginToken:self.loginTokenInfo.token
                                                                                                         request:request
                                                                                                         success:^(WCTRCMessageResponseResult *updateAccountSahreTargetResponseResult)
                                                          {
                blockUpdateAccountSharedTargetResponseResult = [updateAccountSahreTargetResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                         failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockUpdateAccountSharedTargetResponseResult!=nil)
            {
                [blockUpdateAccountSharedTargetResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, request];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockUpdateAccountSharedTargetResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCSetChineseRecognizeOutputResponseResult *)setChineseRecognizeWithOutput:(NSString *)output
                                                                          error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSetChineseRecognizeOutputResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient setChineseRecognizeOutputWithLoginToken:self.loginTokenInfo.token
                                                                                                           output:output
                                                                                                          success:^(WCTRCSetChineseRecognizeOutputResponseResult *setChineseRecognizeOutputResponseResult)
                                                          {
                blockResponseResult = [setChineseRecognizeOutputResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                          failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, output];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCSetSortByFieldResponseResult *)setSortByFieldWithField:(NSString *)field
                                                         error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSetSortByFieldResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient setSortByFieldWithLoginToken:self.loginTokenInfo.token
                                                                                                 field:field
                                                                                               success:^(WCTRCSetSortByFieldResponseResult *setSortByFieldResponseResult)
                                                          {
                blockResponseResult = [setSortByFieldResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                               failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, field];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCSetEditAfterRecogResponseResult *)setEditAfterRecogSettingWithEnable:(BOOL)enable
                                                                       error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSetEditAfterRecogResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient setEditAfterRecogWithLoginToken:self.loginTokenInfo.token
                                                                                                   enable:enable
                                                                                                  success:^(WCTRCSetEditAfterRecogResponseResult *setEditAfterRecogResponseResult)
                                                          {
                blockResponseResult = [setEditAfterRecogResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                  failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, @(enable)];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCSetAddNoteInfoAfterRecogResponseResult *)setAddNoteInfoAfterRecogWithEnable:(BOOL)enable
                                                                              error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSetAddNoteInfoAfterRecogResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient setAddNoteInfoAfterRecogWithLoginToken:self.loginTokenInfo.token
                                                                                                          enable:enable
                                                                                                         success:^(WCTRCSetAddNoteInfoAfterRecogResponseResult *setAddNoteInfoAfterRecogResponseResult)
                                                          {
                blockResponseResult = [setAddNoteInfoAfterRecogResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                         failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, @(enable)];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - CRM settings methods


//==============================================================================
//
//==============================================================================
- (WCTRCMessageResponseResult *)setCRMSettingsWithSettings:(WCTRCCRMSettings *)settings
                                                     error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient setCRMSettingsWithLoginToken:self.loginTokenInfo.token
                                                                                           crmSettings:settings
                                                                                               success:^(WCTRCMessageResponseResult *setCRMSettingsResponseResult)
                                                          {
                blockResponseResult = [setCRMSettingsResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                               failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, settings];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCCRMSettingsResponseResult *)CRMSettingsWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCCRMSettingsResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient CRMSettingsWithLoginToken:self.loginTokenInfo.token
                                                                                            success:^(WCTRCCRMSettingsResponseResult *CRMSettingsResponseResult)
                                                          {
                blockResponseResult = [CRMSettingsResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                            failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCCRMSystemResponseResult *)CRMSystemWithError:(NSError **)error
{
    NSString *supportedVersion = [WCTRestClientController curLastMinumSupportedVersion];
    
    //////////////////////////////////////////////////
    
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCCRMSystemResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            RKObjectRequestOperation *requestOperation = nil;
            
            if (supportedVersion==nil ||
                [supportedVersion floatValue]<7.0)
            {
                requestOperation = [self.restClient CRMSystemWithLoginToken:self.loginTokenInfo.token
                                                                    success:^(WCTRCMessageResponseResult *CRMSystemResponseResult)
                                    {
                    blockResponseResult = [[WCTRCCRMSystemResponseResult alloc] init];
                    blockResponseResult.errorData = [[CRMSystemResponseResult.errorData copy] autorelease];
                    blockResponseResult.message = [[CRMSystemResponseResult.message copy] autorelease];
                    blockResponseResult.data = [[[WCTRCCRMSystem alloc] init] autorelease];
                    blockResponseResult.data.supportCRM = [[CRMSystemResponseResult.data copy] autorelease];
                    dispatch_group_leave(group);
                }
                                                                    failure:^(NSError *failureError)
                                    {
                    blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                      code:failureError.code
                                                lineNumber:(__LINE__)
                                    localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                    object:failureError] retain];
                    
                    dispatch_group_leave(group);
                }];
            }
            else
            {
                requestOperation = [self.restClient CRMSystemV7WithLoginToken:self.loginTokenInfo.token
                                                                      success:^(WCTRCCRMSystemResponseResult *CRMSystemResponseResult)
                                    {
                    blockResponseResult = [CRMSystemResponseResult retain];
                    dispatch_group_leave(group);
                }
                                                                      failure:^(NSError *failureError)
                                    {
                    blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                      code:failureError.code
                                                lineNumber:(__LINE__)
                                    localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                    object:failureError] retain];
                    
                    dispatch_group_leave(group);
                }];
            }
            
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}

//==============================================================================
//
//==============================================================================
- (WCTRCMessageResponseResult *)logoutCRMWithType:(NSString *)CRMType
                                            error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient logoutCRMWithLoginToken:self.loginTokenInfo.token
                                                                                          CRMType:CRMType
                                                                                          success:^(WCTRCMessageResponseResult *logoutCRMResponseResult)
                                                          {
                blockResponseResult = [logoutCRMResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                          failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, CRMType];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCUserExportInfoResponseResult *)lastCRMExportInfoWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCUserExportInfoResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient lastCRMExportInfoWithLoginToken:self.loginTokenInfo.token
                                                                                                  success:^(WCTRCUserExportInfoResponseResult *lastCRMExportInfoResponseResult)
                                                          {
                blockResponseResult = [lastCRMExportInfoResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                  failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCMessageResponseResult *)setCRMExportModeWithCRMType:(NSString *)CRMType
                                              CTMExportMode:(NSString *)CRMExportMode
                                 CRMCompanyAssignmentOption:(NSString *)CRMCompanyAssignmentOption
                                                      error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient setCRMExportModeWithLoginToken:self.loginTokenInfo.token
                                                                                                 CTMType:CRMType
                                                                                           CRMExportMode:CRMExportMode
                                                                              CRMCompanyAssignmentOption:CRMCompanyAssignmentOption                                   success:^(WCTRCMessageResponseResult *setCRMExportModeResponseResult)
                                                          {
                
                blockResponseResult = [setCRMExportModeResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                 failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@ /%@", __PRETTY_FUNCTION__, CRMType, CRMExportMode];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCMessageResponseResult *)setCRMTokenWithTokenRequest:(WCTRCUpdateCRMTokenRequest *)tokenRequest
                                                      error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient setCRMAccessTokenWithLoginToken:self.loginTokenInfo.token
                                                                                          CRMTokenRequest:tokenRequest
                                                                                                  success:^(WCTRCMessageResponseResult *setCRMAccessTokenResponseResult)
                                                          {
                
                
                blockResponseResult = [setCRMAccessTokenResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                  failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, tokenRequest];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCCRMUserAgentOAuthTokenResponseResult *)salesforceOAuthTokenWithCode:(NSString *)code
                                                                      error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCCRMUserAgentOAuthTokenResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient salesforceOAuthTokenWithLoginToken:self.loginTokenInfo.token code:code success:^(WCTRCCRMUserAgentOAuthTokenResponseResult *userAgentOAuthTokenResponseResult) {
                
                blockResponseResult = [userAgentOAuthTokenResponseResult retain];
                dispatch_group_leave(group);
            } failure:^(NSError *failureError) {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s code:%@", __PRETTY_FUNCTION__, code];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Exchange server settings methods


//==============================================================================
//
//==============================================================================
- (BOOL)isExchangeServerSupported
{
    // !! 這個API要6.1以上才支援
    if (self.serverVersionInfo==nil ||
        self.serverVersionInfo.majorVersion<6||
        (self.serverVersionInfo.majorVersion==6&&self.serverVersionInfo.minorVersion<1))
    {
        return NO;
    }
    return YES;
}


//==============================================================================
//
//==============================================================================
- (WCTRCMessageResponseResult *)setExchangeServerSettingsWithSettings:(WCTRCExchangeServerSettings *)settings
                                                                error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient setExchangeServerSettingsWithLoginToken:self.loginTokenInfo.token
                                                                                                         settings:settings
                                                                                                          success:^(WCTRCMessageResponseResult *setCRMSettingsResponseResult)
                                                          {
                blockResponseResult = [setCRMSettingsResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                          failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, settings];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCExchangeServerSettingsResponseResult *)exchangeServerSettingsWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCExchangeServerSettingsResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient exchangeServerSettingsWithLoginToken:self.loginTokenInfo.token
                                                                                                       success:^(WCTRCExchangeServerSettingsResponseResult *contactServerSettingsResponseResult)
                                                          {
                blockResponseResult = [contactServerSettingsResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                       failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCExchangeServerSystemResponseResult *)exchangeServerInfoWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCExchangeServerSystemResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient exchangeServerInfoWithLoginToken:self.loginTokenInfo.token
                                                                                                   success:^(WCTRCExchangeServerSystemResponseResult *contactServerInfoResponseResult)
                                                          {
                blockResponseResult = [contactServerInfoResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                   failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCMessageResponseResult *)logoutExchangeServerWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient logoutExchangeServerWithLoginToken:self.loginTokenInfo.token
                                                                                                     success:^(WCTRCMessageResponseResult *logoutCRMResponseResult)
                                                          {
                blockResponseResult = [logoutCRMResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                     failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s", __PRETTY_FUNCTION__];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCUserExportInfoResponseResult *)lastExchangeServerExportInfoWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCUserExportInfoResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient lastCRMExportInfoWithLoginToken:self.loginTokenInfo.token
                                                                                                  success:^(WCTRCUserExportInfoResponseResult *lastCRMExportInfoResponseResult)
                                                          {
                blockResponseResult = [lastCRMExportInfoResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                  failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCMessageResponseResult *)setExchangeServerExportModeWithMode:(NSString *)mode
                                                              error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient setExchangeServerExportModeWithLoginToken:self.loginTokenInfo.token
                                                                                                               mode:mode
                                                                                                            success:^(WCTRCMessageResponseResult *setCRMExportModeResponseResult)
                                                          {
                
                blockResponseResult = [setCRMExportModeResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                            failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, mode];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCMessageResponseResult *)setExchangeServerTokenWithTokenRequest:(WCTRCExchangeServerTokenSettings *)tokenSettings
                                                                 error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient setExchangeServerAccessTokenWithLoginToken:self.loginTokenInfo.token
                                                                                                       tokenSettings:tokenSettings
                                                                                                             success:^(WCTRCMessageResponseResult *setExchangeServerAccessTokenResponseResult)
                                                          {
                
                
                blockResponseResult = [setExchangeServerAccessTokenResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                             failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, tokenSettings];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}




////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Statistic Method

//================================================================================
//
//================================================================================
- (WCTRCTotalActivityInfosResponseResult *)totalActivityInfosWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCTotalActivityInfosResponseResult  *blockActivityInfoResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient totalActivityInfosWithLoginToken:self.loginTokenInfo.token
                                                                                                   success:^(WCTRCTotalActivityInfosResponseResult *activityInfoResponseResult)
                                                          {
                blockActivityInfoResponseResult = [activityInfoResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                   failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockActivityInfoResponseResult!=nil)
            {
                [blockActivityInfoResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockActivityInfoResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCAuthorizedInfoResponseResult *)authorizedInfoWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCAuthorizedInfoResponseResult  *blockAuthorizedInfoResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient authorizedInfoWithLoginToken:self.loginTokenInfo.token
                                                                                               success:^(WCTRCAuthorizedInfoResponseResult *authorizedInfoResponseResult)
                                                          {
                blockAuthorizedInfoResponseResult = [authorizedInfoResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                               failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockAuthorizedInfoResponseResult!=nil)
            {
                [blockAuthorizedInfoResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockAuthorizedInfoResponseResult;
}



//================================================================================
//
//================================================================================
- (WCTRCTotalContactInfosResponseResult *)totalContactInfosWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCTotalContactInfosResponseResult  *blockTotalContactInfosResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient totalContactInfosWithLoginToken:self.loginTokenInfo.token
                                                                                                  success:^(WCTRCTotalContactInfosResponseResult *totalContactInfosResponseResult)
                                                          {
                blockTotalContactInfosResponseResult = [totalContactInfosResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                  failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockTotalContactInfosResponseResult!=nil)
            {
                [blockTotalContactInfosResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockTotalContactInfosResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCStorageInfoResponseResult *)storageInfoWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCStorageInfoResponseResult  *blockStorageInfoResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient storageInfoWithLoginToken:self.loginTokenInfo.token
                                                                                            success:^(WCTRCStorageInfoResponseResult *storageInfoResponseResult)
                                                          {
                blockStorageInfoResponseResult = [storageInfoResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                            failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockStorageInfoResponseResult!=nil)
            {
                [blockStorageInfoResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockStorageInfoResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCTotalTaskInfosResponseResult *)totalTaskInfosWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCTotalTaskInfosResponseResult  *blockTotalTaskInfosResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient totalTaskInfosWithLoginToken:self.loginTokenInfo.token
                                                                                               success:^(WCTRCTotalTaskInfosResponseResult *totalTaskInfosResponseResult)
                                                          {
                blockTotalTaskInfosResponseResult = [totalTaskInfosResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                               failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockTotalTaskInfosResponseResult!=nil)
            {
                [blockTotalTaskInfosResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockTotalTaskInfosResponseResult;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Test Method

//================================================================================
//
//================================================================================
- (TestResponseObject *)textResponseObjectByGETMethodWithError:(NSError **)error
{
    __block NSError *blockError = nil;
    __block TestResponseObject  *blockTestResponseObject = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient testGETAPIWithSuccess:^(TestResponseObject *responseObject)
                                                          {
                blockTestResponseObject = [responseObject retain];
                dispatch_group_leave(group);
            }
                                                                                        failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockTestResponseObject!=nil)
            {
                [blockTestResponseObject autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    
    return blockTestResponseObject;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance sync Method


//==============================================================================
//
//==============================================================================
- (WCTRCSyncCategoryCreateResponseResult *)syncCreateCategoryWithCategoryRequest:(WCTRCCategoryInfo *)categoryRequest
                                                                           error:(NSError **)error

{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSyncCategoryCreateResponseResult  *blockSyncCategoryCreateResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            RKObjectRequestOperation *requestOperation =
            [self.restClient syncCreateCategoryWithLoginToken:self.loginTokenInfo.token
                                              categoryRequest:categoryRequest
                                                      success:^(WCTRCSyncCategoryCreateResponseResult *syncCategoryCreateResponseResult)
             {
                blockSyncCategoryCreateResponseResult = [syncCategoryCreateResponseResult retain];
                dispatch_group_leave(group);
            }
                                                      failure:^(NSError *error)
             {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSyncCategoryCreateResponseResult!=nil)
            {
                [blockSyncCategoryCreateResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, categoryRequest];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockSyncCategoryCreateResponseResult;
}



//==============================================================================
//
//==============================================================================
- (WCTRCSyncCategoryDeleteResponseResult *)syncDeleteCategoryWithCategoryGuid:(NSString *)categoryGuid
                                                                        error:(NSError **)error

{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSyncCategoryDeleteResponseResult  *blockSyncCategoryDeleteResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            RKObjectRequestOperation *requestOperation =
            [self.restClient syncDeleteCategoryWithLoginToken:self.loginTokenInfo.token
                                                 categoryGuid:categoryGuid
                                                      success:^(WCTRCSyncCategoryDeleteResponseResult *syncCategoryDeleteResponseResult)
             {
                blockSyncCategoryDeleteResponseResult = [syncCategoryDeleteResponseResult retain];
                dispatch_group_leave(group);
            }
                                                      failure:^(NSError *error)
             {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSyncCategoryDeleteResponseResult!=nil)
            {
                [blockSyncCategoryDeleteResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    //////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, categoryGuid];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockSyncCategoryDeleteResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCDetailCategoryInfoResponseResult *)syncGetCategoryWithCategoryGuid:(NSString *)categoryGuid
                                                                     error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCDetailCategoryInfoResponseResult  *blockSyncCategoryGetResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            RKObjectRequestOperation *requestOperation =
            [self.restClient syncGetCategoryWithLoginToken:self.loginTokenInfo.token
                                              categoryGuid:categoryGuid
                                                   success:^(WCTRCDetailCategoryInfoResponseResult *syncCategoryInfoResponseReuslt)
             {
                blockSyncCategoryGetResponseResult = [syncCategoryInfoResponseReuslt retain];
                dispatch_group_leave(group);
                
            }
                                                   failure:^(NSError *error)
             {
                
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSyncCategoryGetResponseResult!=nil)
            {
                [blockSyncCategoryGetResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    //////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, categoryGuid];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockSyncCategoryGetResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCSyncCategoryUpdateResponseResult *)syncUpdateCategoryWithCategoryRequest:(WCTRCCategoryInfo *)categoryRequest
                                                                           error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSyncCategoryUpdateResponseResult  *blockSyncCategoryUpdateResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            RKObjectRequestOperation *requestOperation =
            [self.restClient syncUpdateCategoryWithLoginToken:self.loginTokenInfo.token
                                              categoryRequest:categoryRequest
                                                      success:^(WCTRCSyncCategoryUpdateResponseResult *syncCategoryUpdateResponseResult)
             {
                blockSyncCategoryUpdateResponseResult = [syncCategoryUpdateResponseResult retain];
                dispatch_group_leave(group);
            }
                                                      failure:^(NSError *error)
             {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                dispatch_group_leave(group);
                dispatch_release(group);
                
                blockError = WCTRestClientErrorMake(@"requestOperation = nil",blockError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSyncCategoryUpdateResponseResult!=nil)
            {
                [blockSyncCategoryUpdateResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    //////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, categoryRequest];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockSyncCategoryUpdateResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCContactProfileResponseResult *)syncCreateContactWithRequest:(WCTRCContactCreationWithImageRequest *)request
                                                              error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCContactProfileResponseResult  *blockSyncContactCreateResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            RKObjectRequestOperation *requestOperation =
            [self.restClient syncCreateContactWithLoginToken:self.loginTokenInfo.token
                                                     request:request
                                                     success:^(WCTRCContactProfileResponseResult *syncContactResponseResult)
             {
                blockSyncContactCreateResponseResult = [syncContactResponseResult retain];
                dispatch_group_leave(group);
            }
                                                     failure:^(NSError *error)
             {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSyncContactCreateResponseResult!=nil)
            {
                [blockSyncContactCreateResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    //////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, request];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockSyncContactCreateResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCBooleanResponseResult *)syncIsExistContactWithContactGuid:(NSString *)contactGuid
                                                      accountGuid:(NSString *)accountGuid
                                                            error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCBooleanResponseResult  *blockSyncIsExistContactResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            RKObjectRequestOperation *requestOperation =
            [self.restClient syncIsExistContactWithLoginToken:self.loginTokenInfo.token
                                                  contactGuid:contactGuid
                                                  accountGuid:accountGuid
                                                      success:^(WCTRCBooleanResponseResult *syncIsExistResponseReuslt)
             {
                blockSyncIsExistContactResponseResult = [syncIsExistResponseReuslt retain];
                dispatch_group_leave(group);
                
            }
                                                      failure:^(NSError *error)
             {
                
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSyncIsExistContactResponseResult!=nil)
            {
                [blockSyncIsExistContactResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    //////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@ / %@", __PRETTY_FUNCTION__, contactGuid, accountGuid];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockSyncIsExistContactResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCSyncContactDeleteResponseResult *)syncDeleteContactWithContactGuid:(NSString *)contactGuid
                                                               accountGuid:(NSString *)accountGuid
                                                                     error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSyncContactDeleteResponseResult  *blockSyncContactDeleteResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            RKObjectRequestOperation *requestOperation =
            [self.restClient syncDeleteContactWithLoginToken:self.loginTokenInfo.token
                                                 contactGuid:contactGuid
                                                 accountGuid:accountGuid
                                                     success:^(WCTRCSyncContactDeleteResponseResult *syncContactDeleteResponseReuslt)
             {
                blockSyncContactDeleteResponseResult = [syncContactDeleteResponseReuslt retain];
                dispatch_group_leave(group);
                
            }
                                                     failure:^(NSError *error)
             {
                
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSyncContactDeleteResponseResult!=nil)
            {
                [blockSyncContactDeleteResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    //////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@ / %@", __PRETTY_FUNCTION__, contactGuid, accountGuid];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockSyncContactDeleteResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCDetailContactInfoResponseResult *)syncGetContactWithContactGuid:(NSString *)contactGuid
                                                                  error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCDetailContactInfoResponseResult  *blockSyncContactGetResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            RKObjectRequestOperation *requestOperation =
            [self.restClient syncGetContactWithLoginToken:self.loginTokenInfo.token
                                              contactGuid:contactGuid
                                                  success:^(WCTRCDetailContactInfoResponseResult *syncContactInfoResponseReuslt)
             {
                blockSyncContactGetResponseResult = [syncContactInfoResponseReuslt retain];
                dispatch_group_leave(group);
                
            }
                                                  failure:^(NSError *error)
             {
                
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSyncContactGetResponseResult!=nil)
            {
                [blockSyncContactGetResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    //////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, contactGuid];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockSyncContactGetResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCContactProfileResponseResult *)syncUpdateContactWithRequest:(WCTRCContactUpdateInfoRequest *)request
                                                              error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCContactProfileResponseResult  *blockSyncContactUpdateResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            RKObjectRequestOperation *requestOperation =
            [self.restClient syncUpdateContactWithLoginToken:self.loginTokenInfo.token
                                                     request:request
                                                     success:^(WCTRCContactProfileResponseResult *syncContactResponseResult)
             {
                blockSyncContactUpdateResponseResult = [syncContactResponseResult retain];
                dispatch_group_leave(group);
            }
                                                     failure:^(NSError *error)
             {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSyncContactUpdateResponseResult!=nil)
            {
                [blockSyncContactUpdateResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    //////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, request];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockSyncContactUpdateResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCSyncContactUpdateCheckResponseResult *)syncContactUpdateCheckWithRequest:(WCTRCContactUpdateCheckRequest *)request
                                                                           error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSyncContactUpdateCheckResponseResult  *blockSyncContactUpdateCheckResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            RKObjectRequestOperation *requestOperation =
            [self.restClient syncContactUpdateCheckWithLoginToken:self.loginTokenInfo.token
                                                          request:request
                                                          success:^(WCTRCSyncContactUpdateCheckResponseResult *syncContactUpdateCheckResponseResult)
             {
                blockSyncContactUpdateCheckResult = [syncContactUpdateCheckResponseResult retain];
                dispatch_group_leave(group);
            }
                                                          failure:^(NSError *error)
             {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSyncContactUpdateCheckResult!=nil)
            {
                [blockSyncContactUpdateCheckResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    //////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, request];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockSyncContactUpdateCheckResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCSyncUpdateActionResponseResult *)syncContactCheckStatusWithRequest:(WCTRCSyncContactCheckRequest *)request
                                                                     error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSyncUpdateActionResponseResult  *blockResponceResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            RKObjectRequestOperation *requestOperation =
            [self.restClient syncContactCheckStatusWithLoginToken:self.loginTokenInfo.token
                                                          request:request
                                                          success:^(WCTRCSyncUpdateActionResponseResult *syncContactUpdateCheckResponseResult)
             {
                blockResponceResult = [syncContactUpdateCheckResponseResult retain];
                dispatch_group_leave(group);
            }
                                                          failure:^(NSError *error)
             {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponceResult!=nil)
            {
                [blockResponceResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    //////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, request];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponceResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCSyncNeedForceCompareResponseResult *)syncNeedForceCompareWithContactLastRecTime:(NSDate *)contactLastRecTime
                                                                                  error:(NSError **)error;
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSyncNeedForceCompareResponseResult  *blockResponseReuslt = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            RKObjectRequestOperation *requestOperation =
            [self.restClient needForceCompareWithLoginToken:self.loginTokenInfo.token
                                         contactLastRecTime:contactLastRecTime
                                                    success:^(WCTRCSyncNeedForceCompareResponseResult *needForceCompareResponseResult) {
                blockResponseReuslt = [needForceCompareResponseResult retain];
                dispatch_group_leave(group);
            } failure:^(NSError *error) {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseReuslt!=nil)
            {
                [blockResponseReuslt autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    //////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, contactLastRecTime];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseReuslt;
}


//==============================================================================
//
//==============================================================================
- (WCTRCStartSyncResponseResult *)syncStartWithSyncInfo:(WCTRCSyncInfo *)syncInfo
                                                  error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCStartSyncResponseResult  *blockStartSyncResponseReuslt = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            RKObjectRequestOperation *requestOperation =
            [self.restClient syncStartWithLoginToken:self.loginTokenInfo.token
                                            syncInfo:syncInfo
                                             success:^(WCTRCStartSyncResponseResult *startSyncResponseResult)
             {
                blockStartSyncResponseReuslt = [startSyncResponseResult retain];
                dispatch_group_leave(group);
            }
                                             failure:^(NSError *error)
             {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockStartSyncResponseReuslt!=nil)
            {
                [blockStartSyncResponseReuslt autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    //////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s request:%@", __PRETTY_FUNCTION__, syncInfo];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockStartSyncResponseReuslt;
}


//==============================================================================
//
//==============================================================================
- (WCTRCMessageResponseResult *)syncDoneWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult  *blockSyncDoneResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            RKObjectRequestOperation *requestOperation =
            [self.restClient syncDoneWithLoginToken:self.loginTokenInfo.token
                                            success:^(WCTRCMessageResponseResult *syncDoneMessageResponseResult)
             {
                blockSyncDoneResponseResult = [syncDoneMessageResponseResult retain];
                dispatch_group_leave(group);
            }
                                            failure:^(NSError *error)
             {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                              userInfo:error.userInfo] retain];
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockSyncDoneResponseResult!=nil)
            {
                [blockSyncDoneResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    else
    {
        // Local Model
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockSyncDoneResponseResult;
}







////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance System Method

//================================================================================
//
//================================================================================
- (WCTRCServerIsAliveResponseResult *)serverIsAliveWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCServerIsAliveResponseResult  *blockServerIsAliveResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient serverIsAliveWithSuccess:^(WCTRCServerIsAliveResponseResult *serverIsAliveResponseResult)
                                                          {
                blockServerIsAliveResponseResult = [serverIsAliveResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                           failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockServerIsAliveResponseResult!=nil)
            {
                [blockServerIsAliveResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockServerIsAliveResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCVersionInfoResponseResult *)serverVersionWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCVersionInfoResponseResult  *blockVersionInfoResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient serverVersionWithSuccess:^(WCTRCVersionInfoResponseResult *versionInfoResponseResult)
                                                          {
                blockVersionInfoResponseResult = [versionInfoResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                           failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockVersionInfoResponseResult!=nil)
            {
                [blockVersionInfoResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockVersionInfoResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCServiceStatusResponseResult *)serverServiceStatusWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCServiceStatusResponseResult  *blockServiceStatusResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient serviceStatusWithSuccess:^(WCTRCServiceStatusResponseResult *serviceStatusResponseResult)
                                                          {
                blockServiceStatusResponseResult = [serviceStatusResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                           failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockServiceStatusResponseResult!=nil)
            {
                [blockServiceStatusResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockServiceStatusResponseResult;
    
}


//================================================================================
//
//================================================================================
- (WCTRCServerIsActivedResult *)serverIsActivedWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCServerIsActivedResult  *blockServerIsActivedResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient serviceIsActivedWithSuccess:^(WCTRCServerIsActivedResult *serverIsActivedResult)
                                                          {
                blockServerIsActivedResult = [serverIsActivedResult retain];
                dispatch_group_leave(group);
            }
                                                                                              failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockServerIsActivedResult!=nil)
            {
                [blockServerIsActivedResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockServerIsActivedResult;
}


//================================================================================
//
//================================================================================
- (WCTRCServerNeedBeActivedResult *)serverNeedBeActivedWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCServerNeedBeActivedResult  *blockServerNeedBeActivedResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient serviceNeedBeActivedWithSuccess:^(WCTRCServerNeedBeActivedResult *serverNeedBeActivedResult)
                                                          {
                blockServerNeedBeActivedResult = [serverNeedBeActivedResult retain];
                dispatch_group_leave(group);
            }
                                                                                                  failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockServerNeedBeActivedResult!=nil)
            {
                [blockServerNeedBeActivedResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockServerNeedBeActivedResult;
}



//================================================================================
//
//================================================================================
- (WCTRCDNSNameResponseResult *)serverDNSNameWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCDNSNameResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient serverDnsNameWithSuccess:^(WCTRCDNSNameResponseResult *serverDnsNameResponseResult)
                                                          {
                blockResponseResult = [serverDnsNameResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                           failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}



//================================================================================
//
//================================================================================
- (WCTRCServerRegisterStatusResponseResult *)serverRegisterStatusWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCServerRegisterStatusResponseResult  *blockServerRegisterStatusWithError = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient serverRegisterStatusWithSuccess:^(WCTRCServerRegisterStatusResponseResult *serverRegisterStatusResponseResult)
                                                          {
                blockServerRegisterStatusWithError = [serverRegisterStatusResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                  failure:^(NSError *failureError)
                                                          {
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockServerRegisterStatusWithError!=nil)
            {
                [blockServerRegisterStatusWithError autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockServerRegisterStatusWithError;
}



//================================================================================
//
//================================================================================
- (WCTRCServerLicenseResponseResult *)serverLicenseWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCServerLicenseResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient serverLicenseWithSuccess:^(WCTRCServerLicenseResponseResult *serverLicenseResponseResult) {
                
                blockResponseResult = [serverLicenseResponseResult retain];
                dispatch_group_leave(group);
            } failure:^(NSError *failureError) {
                
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCDateTimeResponseResult *)serverDBBuildTimeWithError:(NSError **)error
{
    NSString *supportedVersion = [WCTRestClientController curLastMinumSupportedVersion];
    
    // !! 這個API要2.0以上才支援
    if (supportedVersion==nil ||
        [supportedVersion floatValue]<2.0)
    {
        return nil;
    }
    
    //////////////////////////////////////////////////
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCDateTimeResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient serverDBBuildTimeWithSuccess:^(WCTRCDateTimeResponseResult *serverDBBuildTimeResponseResult) {
                
                blockResponseResult = [serverDBBuildTimeResponseResult retain];
                dispatch_group_leave(group);
            } failure:^(NSError *failureError) {
                
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}



//================================================================================
//
//================================================================================
- (BOOL)isMultiCompanyWithError:(NSError **)error
{
    // !! 這個API要6.0以上才支援
    if (self.serverVersionInfo==nil ||
        self.serverVersionInfo.majorVersion<6)
    {
        return NO;
    }
    
    //////////////////////////////////////////////////
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCBooleanResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient getIsMultiCompanyWithSuccess:^(WCTRCBooleanResponseResult *isMultiCompanyResponseResult) {
                
                blockResponseResult = [isMultiCompanyResponseResult retain];
                dispatch_group_leave(group);
            } failure:^(NSError *failureError) {
                
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult.data;
}



//================================================================================
//
//================================================================================
- (BOOL)isSimplifyUIWithError:(NSError **)error
{
    // !! 這個API要6.0以上才支援
    if (self.serverVersionInfo==nil ||
        self.serverVersionInfo.majorVersion<6)
    {
        return NO;
    }
    
    //////////////////////////////////////////////////
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCBooleanResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient getIsSimplifyUIWithSuccess:^(WCTRCBooleanResponseResult *isSimplifyUIResponseResult) {
                
                blockResponseResult = [isSimplifyUIResponseResult retain];
                dispatch_group_leave(group);
            } failure:^(NSError *failureError) {
                
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult.data;
}



//================================================================================
//
//================================================================================
- (WCTRCUIDisplayOptionResponseResult *)displayOptionWithError:(NSError **)error
{
    // !! 這個API要6.0以上才支援
    if (self.serverVersionInfo==nil ||
        self.serverVersionInfo.majorVersion<8)
    {
        return nil;
    }
    
    //////////////////////////////////////////////////
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCUIDisplayOptionResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient displayOptionWithSuccess:^(WCTRCUIDisplayOptionResponseResult *responseResult) {
                
                blockResponseResult = [responseResult retain];
                dispatch_group_leave(group);
            } failure:^(NSError *failureError) {
                
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Register Method


//================================================================================
//
//================================================================================
- (WCTRCAllRegistryListResponseResult *)serverAllRegistryWithError:(NSError **)error
{
    NSString *supportedVersion = [WCTRestClientController curLastMinumSupportedVersion];
    
    // !! 這個API要2.0以上才支援
    if (supportedVersion==nil ||
        [supportedVersion floatValue]<2.0)
    {
        return nil;
    }
    
    //////////////////////////////////////////////////
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCAllRegistryListResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient serverAllRegistryWithLoginToken:self.loginTokenInfo.token
                                                                                                  success:^(WCTRCAllRegistryListResponseResult *allRegistryListResponseResult) {
                blockResponseResult = [allRegistryListResponseResult retain];
                dispatch_group_leave(group);
            }
                                                                                                  failure:^(NSError *failureError) {
                
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Subscribe Method

//================================================================================
//
//================================================================================
- (WCTRCSubscriptionInfoVoResponseResult *)subscriptionInfoVoResponseResultWithError:(NSError **)error
{
    NSString *supportedVersion = [WCTRestClientController curLastMinumSupportedVersion];
    
    //////////////////////////////////////////////////
    
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCSubscriptionInfoVoResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    do
    {
        // !! 這個API要2.0以上才支援
        if (supportedVersion==nil ||
            [supportedVersion floatValue]<2.0)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(self.restClient==nil)
        {
            blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
            break;
        }
        //////////////////////////////////////////////////
        
        //檢查連線狀態是否更新
        if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_t group = dispatch_group_create();
        if(group==NULL)
        {
            blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
            break;
        }
        
        dispatch_group_enter(group);
        
        //////////////////////////////////////////////////
        
        RKObjectRequestOperation *requestOperation =  [self.restClient subscriptionResponseResultWithLoginToken:self.loginTokenInfo.token success:^(WCTRCSubscriptionInfoVoResponseResult *subscriptionInfoVoResponseResult) {
            
            blockResponseResult = [subscriptionInfoVoResponseResult retain];
            
            dispatch_group_leave(group);
            
        } failure:^(NSError *error) {
            blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                              code:error.code
                                        lineNumber:(__LINE__)
                            localizedFailureReason:([error.localizedFailureReason length]>0)?error.localizedFailureReason:error.localizedDescription
                                            object:error] retain];
            
            dispatch_group_leave(group);
        }];
        
        if(requestOperation==nil)
        {
            // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
            if(blockError!=nil)
            {
                [blockError autorelease];
            }
            dispatch_release(group);
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
        
        //////////////////////////////////////////////////
        
        if(blockResponseResult!=nil)
        {
            [blockResponseResult autorelease];
        }
        else if(blockError!=nil)
        {
            [self checkNeedLogoutAndSendNotificationWithError:blockError];
            
            [blockError autorelease];
        }
        
        //////////////////////////////////////////////////
        
        dispatch_release(group);
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCRefreshSubscriptionInfoToServerResult *)refreshSubscriptionInfoVoWithError:(NSError **)error
{
    NSString *supportedVersion = [WCTRestClientController curLastMinumSupportedVersion];
    
    //////////////////////////////////////////////////
    
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCRefreshSubscriptionInfoToServerResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    do
    {
        // !! 這個API要2.0以上才支援
        if (supportedVersion==nil ||
            [supportedVersion floatValue]<2.0)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(self.restClient==nil)
        {
            blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
            break;
        }
        //////////////////////////////////////////////////
        
        //檢查連線狀態是否更新
        if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_t group = dispatch_group_create();
        if(group==NULL)
        {
            blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
            break;
        }
        
        dispatch_group_enter(group);
        
        //////////////////////////////////////////////////
        
        RKObjectRequestOperation *requestOperation =  [self.restClient refreshSubscriptionInfoVoWithLoginToken:self.loginTokenInfo.token
                                                                                                       success:^(WCTRCRefreshSubscriptionInfoToServerResult *refreshSubscriptionInfoToServerResult) {
            blockResponseResult = [refreshSubscriptionInfoToServerResult retain];
            
            dispatch_group_leave(group);
        }
                                                                                                       failure:^(NSError *error) {
            blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                              code:error.code
                                        lineNumber:(__LINE__)
                            localizedFailureReason:([error.localizedFailureReason length]>0)?error.localizedFailureReason:error.localizedDescription
                                            object:error] retain];
            
            dispatch_group_leave(group);
        }];
        
        if(requestOperation==nil)
        {
            // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
            if(blockError!=nil)
            {
                [blockError autorelease];
            }
            dispatch_release(group);
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
        
        //////////////////////////////////////////////////
        
        if(blockResponseResult!=nil)
        {
            [blockResponseResult autorelease];
        }
        else if(blockError!=nil)
        {
            [self checkNeedLogoutAndSendNotificationWithError:blockError];
            
            [blockError autorelease];
        }
        
        //////////////////////////////////////////////////
        
        dispatch_release(group);
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCGetAccountLimitResult *)subscribedAccountLimitWithError:(NSError **)error
{
    NSString *supportedVersion = [WCTRestClientController curLastMinumSupportedVersion];
    
    //////////////////////////////////////////////////
    
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCGetAccountLimitResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    do
    {
        // !! 這個API要2.0以上才支援
        if (supportedVersion==nil ||
            [supportedVersion floatValue]<2.0)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(self.restClient==nil)
        {
            blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        //檢查連線狀態是否更新
        if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_t group = dispatch_group_create();
        if(group==NULL)
        {
            blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
            break;
        }
        
        dispatch_group_enter(group);
        
        //////////////////////////////////////////////////
        
        RKObjectRequestOperation *requestOperation = [self.restClient subscribedAccountLimitWithLoginToken:self.loginTokenInfo.token success:^(WCTRCGetAccountLimitResult *accountLimitResult) {
            
            
            blockResponseResult = [accountLimitResult retain];
            
            dispatch_group_leave(group);
        }
                                                                                                   failure:^(NSError *error) {
            blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                              code:error.code
                                        lineNumber:(__LINE__)
                            localizedFailureReason:([error.localizedFailureReason length]>0)?error.localizedFailureReason:error.localizedDescription
                                            object:error] retain];
            
            dispatch_group_leave(group);
        }];
        
        if(requestOperation==nil)
        {
            // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
            if(blockError!=nil)
            {
                [blockError autorelease];
            }
            dispatch_release(group);
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
        
        //////////////////////////////////////////////////
        
        if(blockResponseResult!=nil)
        {
            [blockResponseResult autorelease];
        }
        else if(blockError!=nil)
        {
            [self checkNeedLogoutAndSendNotificationWithError:blockError];
            
            [blockError autorelease];
        }
        
        //////////////////////////////////////////////////
        
        dispatch_release(group);
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCGetSubscriptionURLResponseResult *)subscriptionURLWithError:(NSError **)error
{
    NSString *supportedVersion = [WCTRestClientController curLastMinumSupportedVersion];
    
    //////////////////////////////////////////////////
    
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCGetSubscriptionURLResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    do
    {
        // !! 這個API要2.0以上才支援
        if (supportedVersion==nil ||
            [supportedVersion floatValue]<2.0)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(self.restClient==nil)
        {
            blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        //檢查連線狀態是否更新
        if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_t group = dispatch_group_create();
        if(group==NULL)
        {
            blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
            break;
        }
        
        dispatch_group_enter(group);
        
        //////////////////////////////////////////////////
        
        RKObjectRequestOperation *requestOperation = [self.restClient subscriptionURLWithLoginToken:self.loginTokenInfo.token success:^(WCTRCGetSubscriptionURLResponseResult *subscriptionURLResponseResult) {
            blockResponseResult = [subscriptionURLResponseResult retain];
            
            dispatch_group_leave(group);
        }
                                                                                            failure:^(NSError *error) {
            blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                              code:error.code
                                        lineNumber:(__LINE__)
                            localizedFailureReason:([error.localizedFailureReason length]>0)?error.localizedFailureReason:error.localizedDescription
                                            object:error] retain];
            
            dispatch_group_leave(group);
        }];
        
        if(requestOperation==nil)
        {
            // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
            if(blockError!=nil)
            {
                [blockError autorelease];
            }
            dispatch_release(group);
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
        
        //////////////////////////////////////////////////
        
        if(blockResponseResult!=nil)
        {
            [blockResponseResult autorelease];
        }
        else if(blockError!=nil)
        {
            [self checkNeedLogoutAndSendNotificationWithError:blockError];
            
            [blockError autorelease];
        }
        
        //////////////////////////////////////////////////
        
        dispatch_release(group);
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCBooleanResponseResult *)upgradeFromRetailWithError:(NSError **)error
{
    //////////////////////////////////////////////////
    
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCBooleanResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    do
    {

        //////////////////////////////////////////////////
        
        if(self.restClient==nil)
        {
            blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        //檢查連線狀態是否更新
        if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_t group = dispatch_group_create();
        if(group==NULL)
        {
            blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
            break;
        }
        
        dispatch_group_enter(group);
        
        //////////////////////////////////////////////////
        
        RKObjectRequestOperation *requestOperation = [self.restClient upgradeFromRetailWithLoginToken:self.loginTokenInfo.token success:^(WCTRCBooleanResponseResult *responseResult) {
            blockResponseResult = [responseResult retain];
            
            dispatch_group_leave(group);
        }
                                                                                            failure:^(NSError *error) {
            blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                              code:error.code
                                        lineNumber:(__LINE__)
                            localizedFailureReason:([error.localizedFailureReason length]>0)?error.localizedFailureReason:error.localizedDescription
                                            object:error] retain];
            
            dispatch_group_leave(group);
        }];
        
        if(requestOperation==nil)
        {
            // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
            if(blockError!=nil)
            {
                [blockError autorelease];
            }
            dispatch_release(group);
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
        
        //////////////////////////////////////////////////
        
        if(blockResponseResult!=nil)
        {
            [blockResponseResult autorelease];
        }
        else if(blockError!=nil)
        {
            [self checkNeedLogoutAndSendNotificationWithError:blockError];
            
            [blockError autorelease];
        }
        
        //////////////////////////////////////////////////
        
        dispatch_release(group);
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}




////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance CRM (Salesforce) Method

//================================================================================
//
//================================================================================
- (WCTSalesForceSearchRecordResponseResult *)salesforceRecentViewAccountsWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTSalesForceSearchRecordResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    do
    {
        if(self.restClient==nil)
        {
            blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
            break;
        }
        //////////////////////////////////////////////////
        
        //檢查連線狀態是否更新
        if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_t group = dispatch_group_create();
        if(group==NULL)
        {
            blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
            break;
        }
        
        dispatch_group_enter(group);
        
        //////////////////////////////////////////////////
        
        RKObjectRequestOperation *requestOperation = [self.restClient salesforceRecentViewAccountsWithLoginToken:self.loginTokenInfo.token success:^(WCTSalesForceSearchRecordResponseResult *result) {
            
            blockResponseResult = [result retain];
            
            dispatch_group_leave(group);
        }
                                                                                            failure:^(NSError *error) {
            blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                              code:error.code
                                        lineNumber:(__LINE__)
                            localizedFailureReason:([error.localizedFailureReason length]>0)?error.localizedFailureReason:error.localizedDescription
                                            object:error] retain];
            
            dispatch_group_leave(group);
        }];
        
        if(requestOperation==nil)
        {
            // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
            if(blockError!=nil)
            {
                [blockError autorelease];
            }
            dispatch_release(group);
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
        
        //////////////////////////////////////////////////
        
        if(blockResponseResult!=nil)
        {
            [blockResponseResult autorelease];
        }
        else if(blockError!=nil)
        {
            [self checkNeedLogoutAndSendNotificationWithError:blockError];
            
            [blockError autorelease];
        }
        
        //////////////////////////////////////////////////
        
        dispatch_release(group);
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTSalesForceSearchRecordResponseResult *)salesforceAccountsWithKeyword:(NSString *)keyword
                              error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTSalesForceSearchRecordResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    do
    {
        if(keyword.length<=0)
        {
            blockError = WCTRestClientErrorMake(@"關鍵字不得為空字串",blockError);
            break;
        }
        
        //////////////////////////////////////////////////

        if(self.restClient==nil)
        {
            blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
            break;
        }
        //////////////////////////////////////////////////
        
        //檢查連線狀態是否更新
        if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_t group = dispatch_group_create();
        if(group==NULL)
        {
            blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
            break;
        }
        
        dispatch_group_enter(group);
        
        //////////////////////////////////////////////////
        
        RKObjectRequestOperation *requestOperation = [self.restClient salesforceAccountsWithLoginToken:self.loginTokenInfo.token
                                                                                               keyword:keyword                                    success:^(WCTSalesForceSearchRecordResponseResult *result) {
            
            blockResponseResult = [result retain];
            
            dispatch_group_leave(group);
        }
                                                                                            failure:^(NSError *error) {
            blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                              code:error.code
                                        lineNumber:(__LINE__)
                            localizedFailureReason:([error.localizedFailureReason length]>0)?error.localizedFailureReason:error.localizedDescription
                                            object:error] retain];
            
            dispatch_group_leave(group);
        }];
        
        if(requestOperation==nil)
        {
            // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
            if(blockError!=nil)
            {
                [blockError autorelease];
            }
            dispatch_release(group);
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
        
        //////////////////////////////////////////////////
        
        if(blockResponseResult!=nil)
        {
            [blockResponseResult autorelease];
        }
        else if(blockError!=nil)
        {
            [self checkNeedLogoutAndSendNotificationWithError:blockError];
            
            [blockError autorelease];
        }
        
        //////////////////////////////////////////////////
        
        dispatch_release(group);
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance QContact Method


//================================================================================
//
//================================================================================
- (WCTRCQContactzAuthInfoResponseResult *)qContactzLoginWithTokenVo:(WCTRCQContactzAuthTokenVo *)tokenVo
                        error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCQContactzAuthInfoResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    do
    {
        if(tokenVo==nil)
        {
            blockError = WCTRestClientErrorMake(@"關鍵字不得為空字串",blockError);
            break;
        }
        
        //////////////////////////////////////////////////

        if(self.restClient==nil)
        {
            blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
            break;
        }
        //////////////////////////////////////////////////
        
        //檢查連線狀態是否更新
        if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_t group = dispatch_group_create();
        if(group==NULL)
        {
            blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
            break;
        }
        
        dispatch_group_enter(group);
        
        //////////////////////////////////////////////////
        
        RKObjectRequestOperation *requestOperation = [self.restClient qContactzLoginWithLoginToken:self.loginTokenInfo.token tokenVo:tokenVo success:^(WCTRCQContactzAuthInfoResponseResult *qContactzAuthInfoResponseResult) {
      
            blockResponseResult = [qContactzAuthInfoResponseResult retain];
            
            dispatch_group_leave(group);
        }
                                                                                            failure:^(NSError *error) {
            blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                              code:error.code
                                        lineNumber:(__LINE__)
                            localizedFailureReason:([error.localizedFailureReason length]>0)?error.localizedFailureReason:error.localizedDescription
                                            object:error] retain];
            
            dispatch_group_leave(group);
        }];
        
        if(requestOperation==nil)
        {
            // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
            if(blockError!=nil)
            {
                [blockError autorelease];
            }
            dispatch_release(group);
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
        
        //////////////////////////////////////////////////
        
        if(blockResponseResult!=nil)
        {
            [blockResponseResult autorelease];
        }
        else if(blockError!=nil)
        {
            [self checkNeedLogoutAndSendNotificationWithError:blockError];
            
            [blockError autorelease];
        }
        
        //////////////////////////////////////////////////
        
        dispatch_release(group);
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCMessageResponseResult *)setQContactzExportMode:(NSString *)exportMode
                                                error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    do
    {
        if([exportMode length]<=0)
        {
            blockError = WCTRestClientErrorMake(@"關鍵字不得為空字串",blockError);
            break;
        }
        //////////////////////////////////////////////////
        
        //檢查連線狀態是否更新
        if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        if(self.restClient==nil)
        {
            blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_t group = dispatch_group_create();
        if(group==NULL)
        {
            blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
            break;
        }
        
        dispatch_group_enter(group);
        
        //////////////////////////////////////////////////
        
        RKObjectRequestOperation *requestOperation = [self.restClient setQContactzExportModeWithLoginToken:self.loginTokenInfo.token exportMode:exportMode success:^(WCTRCMessageResponseResult *messageResponseResult)
        {
            blockResponseResult = [messageResponseResult retain];
            
            dispatch_group_leave(group);
        }
                                                                                            failure:^(NSError *error) {
            blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                              code:error.code
                                        lineNumber:(__LINE__)
                            localizedFailureReason:([error.localizedFailureReason length]>0)?error.localizedFailureReason:error.localizedDescription
                                            object:error] retain];
            
            dispatch_group_leave(group);
        }];
        
        if(requestOperation==nil)
        {
            // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
            if(blockError!=nil)
            {
                [blockError autorelease];
            }
            dispatch_release(group);
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
        
        //////////////////////////////////////////////////
        
        if(blockResponseResult!=nil)
        {
            [blockResponseResult autorelease];
        }
        else if(blockError!=nil)
        {
            [self checkNeedLogoutAndSendNotificationWithError:blockError];
            
            [blockError autorelease];
        }
        
        //////////////////////////////////////////////////
        
        dispatch_release(group);
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//================================================================================
//
//================================================================================
- (WCTRCQContactzEmailResponseResult *)setQContactzEmailSecurityCodeWithBasicInfo:(WCTRCAuthInfo *)basicInfo
                                          error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCQContactzEmailResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    do
    {
        if(basicInfo==nil)
        {
            blockError = WCTRestClientErrorMake(@"basicInfo=nil",blockError);
            break;
        }
        
        //////////////////////////////////////////////////

        if(self.restClient==nil)
        {
            blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
            break;
        }
        //////////////////////////////////////////////////
        
        //檢查連線狀態是否更新
        if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_t group = dispatch_group_create();
        if(group==NULL)
        {
            blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
            break;
        }
        
        dispatch_group_enter(group);
        
        //////////////////////////////////////////////////
        
        RKObjectRequestOperation *requestOperation = [self.restClient setQContactzEmailSecurityCodeWithLoginToken:self.loginTokenInfo.token basicInfo:basicInfo success:^(WCTRCQContactzEmailResponseResult *qContactzEmailResponseResult){
                                                      
            blockResponseResult = [qContactzEmailResponseResult retain];
            
            dispatch_group_leave(group);
        }
                                                                                            failure:^(NSError *error) {
            blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                              code:error.code
                                        lineNumber:(__LINE__)
                            localizedFailureReason:([error.localizedFailureReason length]>0)?error.localizedFailureReason:error.localizedDescription
                                            object:error] retain];
            
            dispatch_group_leave(group);
        }];
        
        if(requestOperation==nil)
        {
            // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
            if(blockError!=nil)
            {
                [blockError autorelease];
            }
            dispatch_release(group);
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
        
        //////////////////////////////////////////////////
        
        if(blockResponseResult!=nil)
        {
            [blockResponseResult autorelease];
        }
        else if(blockError!=nil)
        {
            [self checkNeedLogoutAndSendNotificationWithError:blockError];
            
            [blockError autorelease];
        }
        
        //////////////////////////////////////////////////
        
        dispatch_release(group);
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance MyCard Method



//==============================================================================
//
//==============================================================================
- (WCTRCMyCardInfoResponseResult *)myCardInfoWithID:(NSString *)ID error:(NSError **)error;
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMyCardInfoResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    do
    {
        //////////////////////////////////////////////////
        
        if(self.restClient==nil)
        {
            blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
            break;
        }
        //////////////////////////////////////////////////
        
        //檢查連線狀態是否更新
        if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_t group = dispatch_group_create();
        if(group==NULL)
        {
            blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
            break;
        }
        
        dispatch_group_enter(group);
        
        //////////////////////////////////////////////////
        
        RKObjectRequestOperation *requestOperation = [self.restClient getMyCardWithLoginToken:self.loginTokenInfo.token
                                                                                     myCardID:ID
                                                                                      success:^(WCTRCMyCardInfoResponseResult *myCardResponseResult) {
            
            blockResponseResult = [myCardResponseResult retain];
            
            dispatch_group_leave(group);
        } failure:^(NSError *error) {
            
            blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                              code:error.code
                                        lineNumber:(__LINE__)
                            localizedFailureReason:([error.localizedFailureReason length]>0)?error.localizedFailureReason:error.localizedDescription
                                            object:error] retain];
            
            dispatch_group_leave(group);
        }];
        
        if(requestOperation==nil)
        {
            // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
            if(blockError!=nil)
            {
                [blockError autorelease];
            }
            dispatch_release(group);
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
        
        //////////////////////////////////////////////////
        
        if(blockResponseResult!=nil)
        {
            [blockResponseResult autorelease];
        }
        else if(blockError!=nil)
        {
            [self checkNeedLogoutAndSendNotificationWithError:blockError];
            
            [blockError autorelease];
        }
        
        //////////////////////////////////////////////////
        
        dispatch_release(group);
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}



//================================================================================
//
//================================================================================
- (CPImage *)myCardImageWithID:(NSString *)ID
                     imageType:(NSString *)imageType
                         error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block CPImage *blockResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            //檢查連線狀態是否更新
            if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *operation = [self.restClient getMyCardImageWithLoginToken:self.loginTokenInfo.token
                                                                                       myCardID:ID
                                                                                      imageType:imageType
                                                                                        success:^(CPImage *image) {
                
                blockResult = [image retain];
                
                dispatch_group_leave(group);
            } failure:^(NSError *error) {
                
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                                  code:error.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([error.localizedFailureReason length]>0)?error.localizedFailureReason:error.localizedDescription
                                                object:error] retain];
                dispatch_group_leave(group);

            }];
            
            if(operation==nil)
            {
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResult!=nil)
            {
                [blockResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
        
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"%s imageType:%@", __PRETTY_FUNCTION__,imageType];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResult;
}



//==============================================================================
//
//==============================================================================
- (WCTRCMessageResponseResult *)upsertMyCardWithID:(NSString *)ID
                                          cardInfo:(WCTRCUpdateMyCardInfoRequest *)cardInfoRequest
                                             error:(NSError **)error;
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    do
    {
        //////////////////////////////////////////////////
        
        if(self.restClient==nil)
        {
            blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
            break;
        }
        //////////////////////////////////////////////////
        
        //檢查連線狀態是否更新
        if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_t group = dispatch_group_create();
        if(group==NULL)
        {
            blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
            break;
        }
        
        dispatch_group_enter(group);
        
        //////////////////////////////////////////////////
        
        RKObjectRequestOperation *requestOperation = [self.restClient upsertMyCardWithLoginToken:self.loginTokenInfo.token
                                                                                        myCardID:ID
                                                                                        cardInfo:cardInfoRequest success:^(WCTRCMessageResponseResult *responseResult) {
             
            blockResponseResult = [responseResult retain];
            
            dispatch_group_leave(group);
        } failure:^(NSError *error) {
            
            blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                              code:error.code
                                        lineNumber:(__LINE__)
                            localizedFailureReason:([error.localizedFailureReason length]>0)?error.localizedFailureReason:error.localizedDescription
                                            object:error] retain];
            
            dispatch_group_leave(group);
        }];
        
        if(requestOperation==nil)
        {
            // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
            if(blockError!=nil)
            {
                [blockError autorelease];
            }
            dispatch_release(group);
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
        
        //////////////////////////////////////////////////
        
        if(blockResponseResult!=nil)
        {
            [blockResponseResult autorelease];
        }
        else if(blockError!=nil)
        {
            [self checkNeedLogoutAndSendNotificationWithError:blockError];
            
            [blockError autorelease];
        }
        
        //////////////////////////////////////////////////
        
        dispatch_release(group);
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}



//==============================================================================
//
//==============================================================================
- (WCTRCMyCardSettingResponseResult *)myCardSettingWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMyCardSettingResponseResult *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    do
    {
        //////////////////////////////////////////////////
        
        if(self.restClient==nil)
        {
            blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
            break;
        }
        //////////////////////////////////////////////////
        
        //檢查連線狀態是否更新
        if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_t group = dispatch_group_create();
        if(group==NULL)
        {
            blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
            break;
        }
        
        dispatch_group_enter(group);
        
        //////////////////////////////////////////////////
        
        RKObjectRequestOperation *requestOperation = [self.restClient myCardSettingWithLoginToken:self.loginTokenInfo.token
                                                                                          success:^(WCTRCMyCardSettingResponseResult *responseResult) {
            
            blockResponseResult = [responseResult retain];
            
            dispatch_group_leave(group);
        } failure:^(NSError *error) {
            
            blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                              code:error.code
                                        lineNumber:(__LINE__)
                            localizedFailureReason:([error.localizedFailureReason length]>0)?error.localizedFailureReason:error.localizedDescription
                                            object:error] retain];
            
            dispatch_group_leave(group);
        }];
        
        if(requestOperation==nil)
        {
            // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
            if(blockError!=nil)
            {
                [blockError autorelease];
            }
            dispatch_release(group);
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
        
        //////////////////////////////////////////////////
        
        if(blockResponseResult!=nil)
        {
            [blockResponseResult autorelease];
        }
        else if(blockError!=nil)
        {
            [self checkNeedLogoutAndSendNotificationWithError:blockError];
            
            [blockError autorelease];
        }
        
        //////////////////////////////////////////////////
        
        dispatch_release(group);
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCBooleanResponseResult *)myCardEnable:(BOOL)enable withID:(NSString *)ID error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCBooleanResponseResult *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    do
    {
        //////////////////////////////////////////////////
        
        if(self.restClient==nil)
        {
            blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
            break;
        }
        //////////////////////////////////////////////////
        
        //檢查連線狀態是否更新
        if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_t group = dispatch_group_create();
        if(group==NULL)
        {
            blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
            break;
        }
        
        dispatch_group_enter(group);
        
        //////////////////////////////////////////////////
        
        RKObjectRequestOperation *requestOperation = [self.restClient enableMyCardWithLoginToken:self.loginTokenInfo.token
                                                                                        myCardID:ID
                                                                                          enable:enable
                                                                                         success:^(WCTRCBooleanResponseResult *responseResult) {
            blockResponseResult = [responseResult retain];
            
            dispatch_group_leave(group);
        } failure:^(NSError *error) {
            
            blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                              code:error.code
                                        lineNumber:(__LINE__)
                            localizedFailureReason:([error.localizedFailureReason length]>0)?error.localizedFailureReason:error.localizedDescription
                                            object:error] retain];
            
            dispatch_group_leave(group);
        }];
        
        if(requestOperation==nil)
        {
            // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
            if(blockError!=nil)
            {
                [blockError autorelease];
            }
            dispatch_release(group);
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
        
        //////////////////////////////////////////////////
        
        if(blockResponseResult!=nil)
        {
            [blockResponseResult autorelease];
        }
        else if(blockError!=nil)
        {
            [self checkNeedLogoutAndSendNotificationWithError:blockError];
            
            [blockError autorelease];
        }
        
        //////////////////////////////////////////////////
        
        dispatch_release(group);
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//==============================================================================
//
//==============================================================================
- (NSData *)myCardVCFWithID:(NSString *)ID error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block NSData *downloadedData = nil;
    //////////////////////////////////////////////////
    
    do
    {
        if([ID length]==0)
        {
            blockError = PPErrorParameterInvalidity(nil);
            break;
        }
        
        NSString *baseURL = [PPSettingsController stringValueWithKey:WCTRestClientController_SettingsKey_URLAddress];
        if([baseURL length]==0)
        {
            blockError = WCTRestClientErrorMake(@"baseURL not found",nil);
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSURLResponse *response = nil;

        NSString *vcfURLString = [baseURL stringByAppendingPathComponent:[NSString stringWithFormat:@"api/mycard_unauth/export_to_vcard?mycardId=%@", ID]];
        NSURL *vcfURL = [NSURL URLWithString:vcfURLString];
        NSMutableURLRequest *urlRequest = [NSMutableURLRequest requestWithURL:vcfURL cachePolicy:NSURLRequestReloadIgnoringCacheData timeoutInterval:10.0];
//        urlRequest.HTTPMethod = @"POST";
        
        downloadedData = [NSURLConnection sendSynchronousRequest:urlRequest returningResponse:&response error:&blockError];

        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return downloadedData;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance account manager methods


//==============================================================================
//
//==============================================================================
- (BOOL)isAccountManagerSupported
{
    // !! 這個API要8.1以上才支援
    if (self.serverVersionInfo==nil ||
        self.serverVersionInfo.majorVersion<8||
        (self.serverVersionInfo.majorVersion==8&&self.serverVersionInfo.minorVersion<1))
    {
        return NO;
    }
    return YES;
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PCS Methods (精準校正)


//==============================================================================
//
//==============================================================================
- (WCTRCPCSSettingResponseResult *)pcsSettingWithError:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCPCSSettingResponseResult *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    do
    {
        //////////////////////////////////////////////////
        
        if(self.restClient==nil)
        {
            blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
            break;
        }
        //////////////////////////////////////////////////
        
        //檢查連線狀態是否更新
        if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_t group = dispatch_group_create();
        if(group==NULL)
        {
            blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
            break;
        }
        
        dispatch_group_enter(group);
        
        //////////////////////////////////////////////////
        
        RKObjectRequestOperation *requestOperation = [self.restClient getPCSSettingWithLoginToken:self.loginTokenInfo.token
                                                                                          success:^(WCTRCPCSSettingResponseResult *responseResult) {
            
            blockResponseResult = [responseResult retain];
            
            dispatch_group_leave(group);
        } failure:^(NSError *error) {
            
            blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                              code:error.code
                                        lineNumber:(__LINE__)
                            localizedFailureReason:([error.localizedFailureReason length]>0)?error.localizedFailureReason:error.localizedDescription
                                            object:error] retain];
            
            dispatch_group_leave(group);
        }];
        
        if(requestOperation==nil)
        {
            // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
            if(blockError!=nil)
            {
                [blockError autorelease];
            }
            dispatch_release(group);
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
        
        //////////////////////////////////////////////////
        
        if(blockResponseResult!=nil)
        {
            [blockResponseResult autorelease];
        }
        else if(blockError!=nil)
        {
            [self checkNeedLogoutAndSendNotificationWithError:blockError];
            
            [blockError autorelease];
        }
        
        //////////////////////////////////////////////////
        
        dispatch_release(group);
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCMessageResponseResult *)transferPCSQuotaWithRequest:(WCTRCTransferQuotaRequest *)request error:(NSError **)error
{
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    do
    {
        //////////////////////////////////////////////////
        
        if(self.restClient==nil)
        {
            blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
            break;
        }
        //////////////////////////////////////////////////
        
        //檢查連線狀態是否更新
        if([self checkTimeExpiredAndKeepConnectionWithError:&blockError]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_t group = dispatch_group_create();
        if(group==NULL)
        {
            blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
            break;
        }
        
        dispatch_group_enter(group);
        
        //////////////////////////////////////////////////
        
        RKObjectRequestOperation *requestOperation = [self.restClient transferPCSQuotaWithLoginToken:self.loginTokenInfo.token request:request success:^(WCTRCMessageResponseResult *responseResult) {
            
            blockResponseResult = [responseResult retain];
            
            dispatch_group_leave(group);
        } failure:^(NSError *error) {
            
            blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:error]
                                              code:error.code
                                        lineNumber:(__LINE__)
                            localizedFailureReason:([error.localizedFailureReason length]>0)?error.localizedFailureReason:error.localizedDescription
                                            object:error] retain];
            
            dispatch_group_leave(group);
        }];
        
        if(requestOperation==nil)
        {
            // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
            if(blockError!=nil)
            {
                [blockError autorelease];
            }
            dispatch_release(group);
            break;
        }
        
        //////////////////////////////////////////////////
        
        dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
        
        //////////////////////////////////////////////////
        
        if(blockResponseResult!=nil)
        {
            [blockResponseResult autorelease];
        }
        else if(blockError!=nil)
        {
            [self checkNeedLogoutAndSendNotificationWithError:blockError];
            
            [blockError autorelease];
        }
        
        //////////////////////////////////////////////////
        
        dispatch_release(group);
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - TSMC mthods (只有台積版有)


//==============================================================================
//
//==============================================================================
- (WCTRCPrivateContactListResponseResult *)noAccountSFContactWithPageRequest:(WCTRCContactPageRequest *)pageRequest error:(NSError **)error
{
    //////////////////////////////////////////////////
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCPrivateContactListResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient privateNoAccountSFContactWithLoginToken:self.loginTokenInfo.token
                                                                                               contactPageRequest:pageRequest
                                                                                                          success:^(WCTRCPrivateContactListResponseResult *contactListResponseResult) {
                
                blockResponseResult = [contactListResponseResult retain];
                dispatch_group_leave(group);
                
            } failure:^(NSError *failureError) {
                
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCMessageResponseResult *)applyAccountWithAccountInfo:(WCTRCApplyAccountInfo *)accountInfo error:(NSError **)error
{
    //////////////////////////////////////////////////
    NSDate *startTime = [NSDate date];
    
    __block NSError *blockError = nil;
    __block WCTRCMessageResponseResult  *blockResponseResult = nil;
    
    //////////////////////////////////////////////////
    
    if([self localModel]==NO)
    {
        do
        {
            if(self.restClient==nil)
            {
                blockError = WCTRestClientErrorMake(@"網路傳輸架構 RestClient 不存在",blockError);
                break;
            }
            //////////////////////////////////////////////////
            
            dispatch_group_t group = dispatch_group_create();
            if(group==NULL)
            {
                blockError = WCTRestClientErrorMake(@"dispatch_group_create 失敗",blockError);
                break;
            }
            
            dispatch_group_enter(group);
            
            //////////////////////////////////////////////////
            
            RKObjectRequestOperation *requestOperation = [self.restClient applyAccountWithAccountInfo:accountInfo success:^(WCTRCMessageResponseResult *responseResult) {
                blockResponseResult = [responseResult retain];
                dispatch_group_leave(group);
                
            } failure:^(NSError *failureError) {
                
                blockError = [[NSError errorWithDomain:[self errorDomainMessageForError:failureError]
                                                  code:failureError.code
                                            lineNumber:(__LINE__)
                                localizedFailureReason:([failureError.localizedFailureReason length]>0)?failureError.localizedFailureReason:failureError.localizedDescription
                                                object:failureError] retain];
                
                dispatch_group_leave(group);
            }];
            
            if(requestOperation==nil)
            {
                // !! openation是空的時，API內還是會有failure block call回來，所以這邊也要把blockError autorelease
                if(blockError!=nil)
                {
                    [blockError autorelease];
                }
                dispatch_release(group);
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
            
            //////////////////////////////////////////////////
            
            if(blockResponseResult!=nil)
            {
                [blockResponseResult autorelease];
            }
            else if(blockError!=nil)
            {
                [self checkNeedLogoutAndSendNotificationWithError:blockError];
                
                [blockError autorelease];
            }
            
            //////////////////////////////////////////////////
            
            dispatch_release(group);
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            *error = blockError;
        }
    }
    
    /////////////////////////////////////////////////
    // !! 將錯誤的訊息log下來
    if(blockError)
    {
        [self logMessageWithStartTime:startTime format:@"\tDevice IP:%@", [self.restClient ipAddress]];
        [self logMessageWithStartTime:startTime format:@"%s error: %@", __PRETTY_FUNCTION__, blockError];
    }
    
    /////////////////////////////////////////////////
    
    return blockResponseResult;
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class path methods


//==============================================================================
//
//==============================================================================
+ (NSURL *)accountManagerURL
{
    NSString *langString = @"en";
    PPSystemInfoLanguage language = [PPSystemInfoController language];
    switch (language) {
        case PPSystemInfoLanguage_ChineseTraditional:
        {
            langString = @"zh-tw";
            break;
        }
        case PPSystemInfoLanguage_ChineseSimplified:
        {
            langString = @"zh-cn";
            break;
        }
        case PPSystemInfoLanguage_Japanese:
        {
            langString = @"ja";
            break;
        }
        case PPSystemInfoLanguage_Korean:
        {
            langString = @"ko";
            break;
        }
        default:
            break;
    }
    // 這邊的URL會包含@"/WorldCardTeam"
    NSString *domainString = [PPSettingsController stringValueWithKey:WCTRestClientController_SettingsKey_URLAddress];
    if ([domainString length]==0)
    {
        return nil;
    }
    
    NSString *token = [WCTRestClientController curLoginToken];
    if([token length]==0)
    {
        return nil;
    }
    
    NSString *urlString = [NSString stringWithFormat:@"%@/admin-setting/index.html?token=%@&lang=%@", [domainString stringByDeletingLastPathComponent], token, langString];
    
    return [NSURL URLWithString:urlString];
}




@end
