//
//  WCTRestClient.m
//  
//
//  Created by Howard on 2016/3/9.
//  Copyright © 2016年 penpower. All rights reserved.
//

#import "WCTRestClient.h"
#import "WCTRestClientController+ErrorCodeDefine.h"
#import "WCTRCBasicResponseResult.h"
#import "WCTRCMessageResponseResult.h"
#import "WCTRCLoginResponseResult.h"

#if TARGET_OS_IPHONE
#elif TARGET_OS_MAC
#import "NSImage+Additions.h"
#endif


@implementation WCTRestClient





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - result handler


//==============================================================================
// MARK: 轉換mappingResult看是成功還是失敗，成功回傳nil, 失敗回傳error
//==============================================================================
- (NSError *)errorWithOperation:(AFRKHTTPRequestOperation *)operation
                  mappingResult:(RKMappingResult *)mappingResult
{
    // !! 所有response都繼承WCTRCBasicResponseResult
    //////////////////////////////////////////////////
    NSInteger errorCode = 200;
    if ([mappingResult.firstObject respondsToSelector:@selector(errorCode)])
    {
        errorCode = [(WCTRCBasicResponseResult *)mappingResult.firstObject errorCode];
    }
    
    NSString *errorMessage = nil;
    
    if ([[mappingResult firstObject] isKindOfClass:[WCTRCLoginResponseResult class]])
    {
        errorMessage = [(WCTRCBasicResponseResult *)mappingResult.firstObject exceptionMessage];
        
        if(errorCode==WCTServer_Login_ErrorCode_LoginTimeIsLarge)
        {
            NSDictionary *dictionary = [[mappingResult firstObject] performSelector:@selector(errorData)];
            
            if(dictionary!=nil)
            {
                NSData *jsonData = [NSJSONSerialization dataWithJSONObject:dictionary
                                                                   options:0
                                                                     error:nil];
                
                if(jsonData!=nil)
                {
                    errorMessage = [[[NSString alloc] initWithData:jsonData
                                                          encoding:NSUTF8StringEncoding] autorelease];
                }
            }
        }
    }
    else if ([mappingResult.firstObject respondsToSelector:@selector(exceptionMessage)])
    {
        errorMessage = [(WCTRCBasicResponseResult *)mappingResult.firstObject exceptionMessage];
    }
    
    // !! server有時沒有回exception Message, 所以取message
    if([errorMessage length]==0)
    {
        if ([mappingResult.firstObject respondsToSelector:@selector(message)])
        {
            errorMessage = [mappingResult.firstObject message];
        }
    }
    
    // 舊版的error code會是0，所以加入errorMessage判斷
    if (errorCode!=200 && [errorMessage length]>0)
    {
        // failed，產生一個error
        NSHTTPURLResponse *response = [[[NSHTTPURLResponse alloc] initWithURL:operation.response.URL
                                                                   statusCode:errorCode
                                                                  HTTPVersion:nil
                                                                 headerFields:operation.response.allHeaderFields] autorelease];
        
        return [NSError errorWithDomain:AFRKNetworkingErrorDomain
                                   code:NSURLErrorBadServerResponse
                               userInfo:@{AFRKNetworkingOperationFailingURLResponseErrorKey:response,
                                          NSURLErrorFailingURLErrorKey:operation.response.URL,
                                          NSLocalizedDescriptionKey:errorMessage}];
        
    }
    else
    {
        // success
        return nil;
    }
}


//==============================================================================
//
//==============================================================================
- (void)handleOperation:(RKObjectRequestOperation *)operation
          mappingResult:(RKMappingResult *)mappingResult
                success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    NSError *error = [self errorWithOperation:operation.HTTPRequestOperation
                                mappingResult:mappingResult];
    
    if (error!=nil)
    {
        if(failure!=NULL)
        {
            failure(operation, error);
        }
    }
    else
    {
        if(success!=NULL)
        {
            success(operation, mappingResult);
        }
    }

}



//==============================================================================
//
//==============================================================================
- (void)handleOperation:(RKObjectRequestOperation *)operation
                  error:(NSError *)error
                failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    if(failure!=NULL)
    {
        failure(operation, error);
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - AFRKHTTPRequestOperation error handler


//==============================================================================
//
//==============================================================================
- (void)handleHttpOperation:(AFRKHTTPRequestOperation *)operation
               responseData:(NSData *)responseData
                    success:(void (^)(AFRKHTTPRequestOperation *operation, NSData *responseData))success
                    failure:(void (^)(AFRKHTTPRequestOperation *operation, NSError *error))failure
{
    // 沒有回傳值，表示處理成功，但沒有圖片
    if (responseData==nil)
    {
        success(operation,responseData);
        return;
    }
    
    //////////////////////////////////////////////////
    NSError *returnError = nil;
    id object = [NSJSONSerialization JSONObjectWithData:responseData
                                                options:NSJSONReadingMutableLeaves
                                                  error:&returnError];
    if(returnError==nil && [object isKindOfClass:[NSDictionary class]]==YES)
    {
        NSString *jsonString = [[NSString alloc] initWithData:responseData encoding:NSUTF8StringEncoding];
        WCTRCMessageResponseResult *responseResult = [[WCTRCMessageResponseResult alloc] initWithJsonString:jsonString error:&returnError];
        [jsonString release];

        if (responseResult!=nil)
        {
            RKMappingResult *mappingResult = [[[RKMappingResult alloc] initWithDictionary:@{[NSNull null]: responseResult}] autorelease];
            [responseResult release];
            
            NSError *error = [self errorWithOperation:operation
                                        mappingResult:mappingResult];
            
            if (error!=nil)
            {
                // !! 伺服器回傳錯誤
                if(failure!=NULL)
                {
                    failure(operation, error);
                }
            }
            else
            {
                if(success!=NULL)
                {
                    success(operation, responseData);
                }
            }
        }
        else
        {
            // !!jsonString如果不是WCTRCMessageResponseResult， responseResult 會轉不出來會回傳 nil;
            // 這時當成功，且沒有圖
            if(success!=NULL)
            {
                success(operation, nil);
            }
        }

    }
    else
    {
        // !! 無法轉為json都當成是有圖片
        if(success!=NULL)
        {
            success(operation, responseData);
        }
    }
}

//==============================================================================
//
//==============================================================================
- (void)handleHttpOperation:(AFRKHTTPRequestOperation *)operation
                      error:(NSError *)error
                    failure:(void (^)(AFRKHTTPRequestOperation *operation, NSError *error))failure
{
    if(failure!=NULL)
    {
        failure(operation, error);
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Overide methods

//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)getObject:(id)object
                        byObjectManager:(RKObjectManager *)objectManager
                                   path:(NSString *)path
                             parameters:(NSDictionary *)parameters
                                success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    __block typeof(self) blockself = self;
    
    return [super getObject:object
            byObjectManager:objectManager
                       path:path
                 parameters:parameters
                    success:^(RKObjectRequestOperation *operation, RKMappingResult *mappingResult) {
                        
                        [blockself handleOperation:operation
                                     mappingResult:mappingResult
                                           success:success
                                           failure:failure];
                        
                    } failure:^(RKObjectRequestOperation *operation, NSError *error) {
                        
                        [blockself handleOperation:operation error:error failure:failure];
                    }];
}


//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)getObject:(id)object
                        byObjectManager:(RKObjectManager *)objectManager
                                   path:(NSString *)path
                             parameters:(NSDictionary *)parameters
                   httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    __block typeof(self) blockself = self;
    
    return [super getObject:object
            byObjectManager:objectManager
                       path:path
                 parameters:parameters
       httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                    success:^(RKObjectRequestOperation *operation, RKMappingResult *mappingResult) {
                        
                        [blockself handleOperation:operation
                                     mappingResult:mappingResult
                                           success:success
                                           failure:failure];
                        
                    } failure:^(RKObjectRequestOperation *operation, NSError *error) {
                        
                        [blockself handleOperation:operation error:error failure:failure];
                    }];
}

//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)postObject:(id)object
                         byObjectManager:(RKObjectManager *)objectManager
                                    path:(NSString *)path
                              parameters:(NSDictionary *)parameters
                    httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                 success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                 failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    __block typeof(self) blockself = self;
    
    return [super postObject:object
             byObjectManager:objectManager
                        path:path
                  parameters:parameters
        httpHeaderParameters:httpHeaderParameters
                     success:^(RKObjectRequestOperation *operation, RKMappingResult *mappingResult) {
                         
                         [blockself handleOperation:operation
                                      mappingResult:mappingResult
                                            success:success
                                            failure:failure];
                         
                     } failure:^(RKObjectRequestOperation *operation, NSError *error) {
                         
                         [blockself handleOperation:operation error:error failure:failure];
                     }];
}


//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)postWithPath:(NSString *)path
                           byObjectManager:(RKObjectManager *)objectManager
                            parameterArray:(NSArray *)parameterArray
                      httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                   success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                   failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    __block typeof(self) blockself = self;

    return [super postWithPath:path
               byObjectManager:objectManager
                parameterArray:parameterArray
          httpHeaderParameters:httpHeaderParameters
                       success:^(RKObjectRequestOperation *operation, RKMappingResult *mappingResult) {
                           
                           [blockself handleOperation:operation
                                        mappingResult:mappingResult
                                              success:success
                                              failure:failure];
                           
                       } failure:^(RKObjectRequestOperation *operation, NSError *error) {
                           
                           [blockself handleOperation:operation error:error failure:failure];
                       }];
}


//================================================================================
//
//================================================================================
- (RKObjectRequestOperation *)multipartFormRequestWithObject:(id)object
                                             byObjectManager:(RKObjectManager *)objectManager
                                                      method:(RKRequestMethod)method
                                                        path:(NSString *)path
                                                        data:(NSData *)data
                                                        name:(NSString *)name
                                                    fileName:(NSString *)fileName
                                                    mimeType:(NSString *)mimeType
                                                  parameters:(NSDictionary *)parameters
                                        httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                                     success:(void (^)(RKObjectRequestOperation *operation, RKMappingResult *mappingResult))success
                                                     failure:(void (^)(RKObjectRequestOperation *operation, NSError *error))failure
{
    __block typeof(self) blockself = self;
    
    return [super multipartFormRequestWithObject:object
                                 byObjectManager:objectManager
                                          method:method
                                            path:path
                                            data:data
                                            name:name
                                        fileName:fileName
                                        mimeType:mimeType
                                      parameters:parameters
                            httpHeaderParameters:httpHeaderParameters
                                         success:^(RKObjectRequestOperation *operation, RKMappingResult *mappingResult) {
                                             
                                             [blockself handleOperation:operation
                                                          mappingResult:mappingResult
                                                                success:success
                                                                failure:failure];
                                             
                                         } failure:^(RKObjectRequestOperation *operation, NSError *error) {
                                             
                                             [blockself handleOperation:operation error:error failure:failure];
                                         }];
}



//================================================================================
//
//================================================================================
- (AFRKHTTPRequestOperation *)getDataWithObject:(id)object
                                  objectManager:(RKObjectManager *)objectManager
                                           path:(NSString *)path
                                     parameters:(NSDictionary *)parameters
                           httpHeaderParameters:(NSDictionary *)httpHeaderParameters
                                        success:(void (^)(AFRKHTTPRequestOperation *operation, NSData *responseData))success
                                        failure:(void (^)(AFRKHTTPRequestOperation *operation, NSError *error))failure
{
    __block typeof(self) blockself = self;

    return [super getDataWithObject:object
                      objectManager:objectManager
                               path:path
                         parameters:parameters
               httpHeaderParameters:httpHeaderParameters
                            success:^(AFRKHTTPRequestOperation *operation, NSData *responseData) {
                                
                                [blockself handleHttpOperation:operation
                                                  responseData:responseData
                                                       success:success
                                                       failure:failure];
                            } failure:^(AFRKHTTPRequestOperation *operation, NSError *error) {
                                
                                [blockself handleHttpOperation:operation error:error failure:failure];
                            }];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - handle success & failure


//==============================================================================
//
//==============================================================================
- (void)handleWCTMappingResult:(RKMappingResult *)mappingResult
                       success:(void (^)(id object))success
                       failure:(void (^)(NSError *error))failure
{
    //////////////////////////////////////////////////
    NSInteger errorCode = 200;
    if ([mappingResult.firstObject respondsToSelector:@selector(errorCode)])
    {
        errorCode = [mappingResult.firstObject errorCode];
    }
    
    if (errorCode==200)
    {
        if(success!=NULL)
        {
            success(mappingResult.firstObject);
        }
    }
    else
    {
        NSString *errorMessage = nil;
        
        if ([mappingResult.firstObject respondsToSelector:@selector(exceptionMessage)])
        {
            errorMessage = [mappingResult.firstObject exceptionMessage];
        }
        
        NSError *error = PPErrorMake(errorCode, errorMessage, nil);
        if(failure!=NULL)
        {
            failure(error);
        }
    }
}


//==============================================================================
//
//==============================================================================
- (void)handleError:(NSError *)error
            failure:(void (^)(NSError *error))failure
{
    if(failure!=NULL)
    {
        failure(error);
    }
}


@end
