//
//  WCTViewGroupSelectViewController.m
//  ADAL
//
//  Created by sanhue on 2018/3/28.
//

#import "WCTCompanyGroupSelectViewController.h"

//define
#import "WCAppearanceDefine.h"
#import "WCTCompanyContactViewController+ResourceDefine.h"
#import "WCTCompanyContactViewController+ParameterDefine.h"
#import "WCGroupDefine.h"

//category
#import "PPButton+Factory.h"
#import "NSObject+PPBusyView.h"
#import "NSError+Custom.h"
#import "WCGroupModel+Tree.h"

//view
#import "WCGroupView.h"
#import "PPNavigationBarView.h"

// controller
#import "PPNavigationController.h"
#import "WCToastController.h"
#import "WCAddOrEditGroupViewController.h"
#import "PPAlertController.h"

////////////////////////////////////////////////////////////////////////////////////////////////////
typedef NS_ENUM(NSInteger, WCTCGSVC_ButtonTag)
{
    WCTCGSVC_ButtonTag_Back = 0,
    WCTCGSVC_ButtonTag_Done,
    WCTCGSVC_ButtonTag_Add,
    WCTCGSVC_ButtonTag_Rename,
    WCTCGSVC_ButtonTag_Delete,
};


////////////////////////////////////////////////////////////////////////////////////////////////////
typedef NS_ENUM(NSInteger, WCTCGSVC_TreeLevel)
{
    WCTCGSVC_TreeLevel_Root = 0,
    WCTCGSVC_TreeLevel_Account,
};
////////////////////////////////////////////////////////////////////////////////////////////////////

@interface WCTCompanyGroupSelectViewController() <PPButtonDelegate, WCGroupViewDataSource, WCGroupViewDelegate>

@property (nonatomic, retain) NSArray *layoutConstraints;
@property (nonatomic, retain) WCGroupView *groupView;

@property (nonatomic, retain) PPNavigationBarView *navigationBarView;
@property (nonatomic, retain) PPButton *ppButtonBack;
@property (nonatomic, retain) PPButton *ppButtonDone;

@property (nonatomic, retain) PPBarView *toolBarView;
@property (nonatomic, retain) PPButton *ppButtonAdd;
@property (nonatomic, retain) PPButton *ppButtonRename;
@property (nonatomic, retain) PPButton *ppButtonDelete;


@property (nonatomic, retain) NSArray *accountList;
@property (nonatomic, retain) NSArray *groupsOfCurrentAccount;
@property (nonatomic, retain) NSString *currentAccountName;
@property (nonatomic, assign) BOOL shouldShowSimplifyUI;

@property (nonatomic, retain) NSOperationQueue *operationQueue;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////
@implementation WCTCompanyGroupSelectViewController





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - init/dealloc methods


//==============================================================================
//
//==============================================================================
- (instancetype)init
{
    self = [super init];
    if (self)
    {
        self.operationQueue = [[[NSOperationQueue alloc] init] autorelease];
        [self.operationQueue setMaxConcurrentOperationCount:1];
    }
    return self;
}


//==============================================================================
//
//==============================================================================
- (void)dealloc
{
    [self.operationQueue cancelAllOperations];
    self.operationQueue = nil;
    
    self.accountList = nil;
    self.groupsOfCurrentAccount = nil;
    self.currentGroupGuid = nil;
    self.currentAccountGuid = nil;
    self.currentAccountName = nil;
    
    [self prepareMainUI];
    //////////////////////////////////////////////////
    [super dealloc];
}







////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - view controller life cycle


//==============================================================================
//
//==============================================================================
- (void)viewWillAppear:(BOOL)animated
{
    [super viewWillAppear:animated];
    
    //////////////////////////////////////////////////
    
    [self prepareMainUI];
}


//==============================================================================
//
//==============================================================================
- (void)viewDidAppear:(BOOL)animated
{
    [super viewDidAppear:animated];
    
    //////////////////////////////////////////////////
    [self prepareDataWithCompletion:^{
        
        if(self.enableEditCategoryUI)
        {
            WCGroupModel *rootModel = [self.accountList firstObject];
            
            if ([rootModel.subGroups count]==0)
            {
                UIAlertAction *alertAction = [UIAlertAction actionWithTitle:WCTCCV_MLS_OK
                                                                      style:UIAlertActionStyleDefault
                                                                    handler:^(UIAlertAction * _Nonnull action) {
                                                                        [self goBackAnimated:YES];
                                                                    }];
                [PPAlertController showWithAlertControllerStyle:UIAlertControllerStyleAlert
                                                          title:@""
                                                        message:WCTCCV_MLS_NoUsableAccount
                                         showFromViewController:self
                                                       animated:YES
                                                    alertAction:alertAction, nil];
                return ;
            }
        }
    }];
}


//==============================================================================
//
//==============================================================================
- (void)viewWillDisappear:(BOOL)animated
{
    //////////////////////////////////////////////////
    
    [super viewWillDisappear:animated];
}


//==============================================================================
//
//==============================================================================
- (void)viewDidDisappear:(BOOL)animated
{
    WCGroupModel *groupModel = [self.groupView.treeView itemForSelectedRow];
    
    self.currentGroupGuid = groupModel.guid;
    
    [self.operationQueue cancelAllOperations];
    [self removeMainUI];
    //////////////////////////////////////////////////
    
    [super viewDidDisappear:animated];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - prepare ui


//==============================================================================
//
//==============================================================================
- (void)prepareMainUI
{
    _navigationBarView = [[PPNavigationBarView alloc] initWithFrame:self.navigationController.navigationBar.bounds];
    if (self.navigationBarView!=nil)
    {
        self.navigationBarView.style = PPBarViewStyle_CenteredAbsolute;
        self.navigationBarView.titleLabel.text = self.titleString;
        
        self.ppButtonBack = [PPButton ppButtonWithIconImageName:ImageNamePPButtonIconForSystemBackImitation
                                                            tag:WCTCGSVC_ButtonTag_Back
                                                       delegate:self
                                          normalBackgroundColor:nil
                                     highlightedBackgroundColor:WCAppearanceDefine_ButtonBackgroundColor
                                                imageEdgeInsets:WCAppearanceDefine_ButtonImageEdgeInset];
        if(self.ppButtonBack!=nil)
        {
            //////////////////////////////////////////////////
            
            [self.navigationBarView setViews:@[self.ppButtonBack] forBlockType:PPBarViewBlockType_Left];
        }
        
        if(self.enableEditCategoryUI)
        {
            NSMutableArray *rightButtons = [NSMutableArray array];
            
            //////////////////////////////////////////////////
            self.ppButtonDone = [PPButton ppButtonWithIconImageName:@"GroupDone"
                                                                tag:WCTCGSVC_ButtonTag_Done
                                                           delegate:self
                                              normalBackgroundColor:nil
                                         highlightedBackgroundColor:WCAppearanceDefine_ButtonBackgroundColor
                                                    imageEdgeInsets:WCAppearanceDefine_ButtonImageEdgeInset];
            //        self.ppButtonAdd.ppGuideIndex = WCGroupViewControllerGuideIndex_Add;
            [rightButtons addObject:self.ppButtonDone];
            
            if([rightButtons count]>0)
            {
                [self.navigationBarView setViews:rightButtons forBlockType:PPBarViewBlockType_Right];
            }
        }
        
        self.navigationItem.leftBarButtonItem = [[[UIBarButtonItem alloc] initWithCustomView:self.navigationBarView] autorelease];
    }
    
    
    //////////////////////////////////////////////////
    if(self.enableEditCategoryUI==YES)
    {
        
        _toolBarView = [[PPBarView alloc] initWithFrame:CGRectZero];
        [self.toolBarView setTranslatesAutoresizingMaskIntoConstraints:NO];
        self.toolBarView.style = PPBarViewStyle_Average;
        
        
        [self.view addSubview:self.toolBarView];
        
        //////////////////////////////////////////////////
        NSMutableArray *bottons = [NSMutableArray array];
        
        
        self.ppButtonAdd = [PPButton ppButtonWithIconImageName:nil
                                                            tag:WCTCGSVC_ButtonTag_Add
                                                       delegate:self
                                          normalBackgroundColor:nil
                                     highlightedBackgroundColor:WCAppearanceDefine_ButtonBackgroundColor
                                                imageEdgeInsets:WCAppearanceDefine_ButtonImageEdgeInset];
        if(self.ppButtonAdd!=nil)
        {
            [self.ppButtonAdd setTitle:WCTCCV_MLS_AddButton maxWidth:0.0];
            [bottons addObject:self.ppButtonAdd];
        }
        
        self.ppButtonRename = [PPButton ppButtonWithIconImageName:nil
                                                              tag:WCTCGSVC_ButtonTag_Rename
                                                         delegate:self
                                            normalBackgroundColor:nil
                                       highlightedBackgroundColor:WCAppearanceDefine_ButtonBackgroundColor
                                                  imageEdgeInsets:WCAppearanceDefine_ButtonImageEdgeInset];
        if(self.ppButtonRename!=nil)
        {
            [self.ppButtonRename setTitle:WCTCCV_MLS_RenameButton maxWidth:0.0];
            [bottons addObject:self.ppButtonRename];
        }

        
        self.ppButtonDelete = [PPButton ppButtonWithIconImageName:nil
                                                              tag:WCTCGSVC_ButtonTag_Delete
                                                         delegate:self
                                            normalBackgroundColor:nil
                                       highlightedBackgroundColor:WCAppearanceDefine_ButtonBackgroundColor
                                                  imageEdgeInsets:WCAppearanceDefine_ButtonImageEdgeInset];
        if(self.ppButtonDelete!=nil)
        {
            [self.ppButtonDelete setTitle:WCTCCV_MLS_DeleteButton maxWidth:0.0];
            [bottons addObject:self.ppButtonDelete];
        }
        
        //////////////////////////////////////////////////
        if([bottons count])
        {
            [self.toolBarView setViews:bottons forBlockType:PPBarViewBlockType_Left];
        }

    }
    //////////////////////////////////////////////////
    if (self.groupView==nil)
    {
        self.groupView = [[[WCGroupView alloc] initWithFrame:CGRectZero] autorelease];
    }
    
    [self.groupView setTranslatesAutoresizingMaskIntoConstraints:NO];
    self.groupView.delegate = self;
    self.groupView.dataSource = self;
    [self.groupView setReserveSpaceWhenExpandButtonHidden:YES];
    self.groupView.style = (self.enableEditCategoryUI)?WCGroupViewStyle_Name:WCGroupViewStyle_NameWithCount;
    [self.groupView setBackgroundColor:[UIColor whiteColor]];
    [self.view addSubview:self.groupView];
    //////////////////////////////////////////////////
    
    [self resetLayoutConstraints];
    
}


//==============================================================================
//
//==============================================================================
- (void)removeMainUI
{
    self.navigationItem.leftBarButtonItem = nil;
    
    self.ppButtonBack = nil;
    self.ppButtonDone = nil;
    self.navigationBarView = nil;

    [self.toolBarView removeFromSuperview];
    self.toolBarView = nil;
    
    [self.ppButtonAdd removeFromSuperview];
    self.ppButtonAdd = nil;

    [self.ppButtonRename removeFromSuperview];
    self.ppButtonAdd = nil;

    [self.ppButtonDelete removeFromSuperview];
    self.ppButtonAdd = nil;

    [self.groupView removeFromSuperview];
    self.groupView = nil;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - layout constraints


//==============================================================================
//
//==============================================================================
- (void)resetLayoutConstraints
{
    //////////////////////////////////////////////////
    
    NSMutableArray *layoutConstraints = [NSMutableArray array];

    if(self.enableEditCategoryUI==YES)
    {
        if(self.groupView==nil||
           self.toolBarView==nil)
        {
            return;
        }
        
        
        [self removeLayoutConstraints];
        
        NSDictionary *views = @{@"groupView":self.groupView,
                                @"toolBarView":self.toolBarView};
        NSDictionary *metrics = @{@"toolBarHeight":@(PPBarViewDefaultHeightForToolBar)};
                
        [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[groupView]|"
                                                                                       options:NSLayoutFormatDirectionLeftToRight
                                                                                       metrics:metrics
                                                                                         views:views]];
        [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[toolBarView]|"
                                                                                       options:NSLayoutFormatDirectionLeftToRight
                                                                                       metrics:metrics
                                                                                         views:views]];

        [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|[groupView][toolBarView(toolBarHeight)]|"
                                                                                       options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                       metrics:metrics
                                                                                         views:views]];
        
    }
    else
    {
        if(self.groupView==nil)
        {
            return;
        }
        
        
        [self removeLayoutConstraints];
        
        NSDictionary *views = @{@"groupView":self.groupView};
        NSDictionary *metrics = nil;//@{@"<#viewName#>":<#viewInstance#>};
        
        
        [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[groupView]|"
                                                                                       options:NSLayoutFormatDirectionLeftToRight
                                                                                       metrics:metrics
                                                                                         views:views]];
        
        [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|[groupView]|"
                                                                                       options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                       metrics:metrics
                                                                                         views:views]];
        
    }

    //////////////////////////////////////////////////
    if ([layoutConstraints count])
    {
        self.layoutConstraints = [NSArray arrayWithArray:layoutConstraints];
        [self.view addConstraints:self.layoutConstraints];
        [self.view layoutIfNeeded];
    }
}


//==============================================================================
//
//==============================================================================
- (void)removeLayoutConstraints
{
    if (self.layoutConstraints)
    {
        [self.view removeConstraints:self.layoutConstraints];
        self.layoutConstraints = nil;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Override


//==============================================================================
//
//==============================================================================
- (void)setBusy:(NSNumber *)busy
{
    [super setBusy:busy];
    
    if ([busy boolValue])
    {
        [self.busyView setStyle:PPBusyViewStyle_Border];
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Prepare data


//==============================================================================
//
//==============================================================================
- (void)assignCardCount:(NSDictionary*)cardCount toGroups:(NSArray*)groups
{
    if(cardCount==nil)
    {
        return ;
    }
    
    for (WCGroupModel *groupModel in groups)
    {
        NSNumber *countNumber = [cardCount valueForKey:groupModel.guid];
        if(countNumber!=nil)
        {
            groupModel.cardCount = [countNumber integerValue];
        }
        
        if([groupModel.subGroups count])
        {
            [self assignCardCount:cardCount toGroups:groupModel.subGroups];
        }
    }
}

//==============================================================================
//
//==============================================================================
- (NSArray *)allGuidsFromGroups:(NSArray*)groups
{
    NSMutableArray *allGuids = [NSMutableArray array];
    for (WCGroupModel *groupModel in groups)
    {
        if ([groupModel.guid isEqualToString:@"MLS_Undownload"]==NO)
        {
            [allGuids addObject:groupModel.guid];
        }
        
        if([groupModel.subGroups count])
        {
            [allGuids addObjectsFromArray:[self allGuidsFromGroups:groupModel.subGroups]];
        }
        
    }
    
    return [NSArray arrayWithArray:allGuids];
}


//==============================================================================
//
//==============================================================================
- (WCGroupModel *)findGroupModelWithOwnerGuid:(NSString *)guid withGroups:(NSArray*)groups
{
    for (WCGroupModel *groupModel in groups)
    {
        if ([groupModel.owner isEqualToString:guid])
        {
            return groupModel;
        }
        else
        {
            WCGroupModel *resultGroupModel = [self findGroupModelWithOwnerGuid:guid withGroups:groupModel.subGroups];
            
            if (resultGroupModel)
            {
                return resultGroupModel;
            }
        }
    }
    
    return nil;
}


//==============================================================================
//
//==============================================================================
- (WCGroupModel *)findGroupModelWithGuid:(NSString *)guid withGroups:(NSArray*)groups
{
    for (WCGroupModel *groupModel in groups)
    {
        if ([groupModel.guid isEqualToString:guid])
        {
            return groupModel;
        }
        else
        {
            WCGroupModel *resultGroupModel = [self findGroupModelWithGuid:guid withGroups:groupModel.subGroups];
            
            if (resultGroupModel)
            {
                return resultGroupModel;
            }
        }
    }
    
    return nil;
}


//==============================================================================
// 展開到某個node為止
//==============================================================================
- (void)expandToNode:(WCGroupModel*)node withGroups:(NSArray *)groups
{
    WCGroupModel *targetModel = node;
    NSMutableArray *parentModels = [NSMutableArray array];
    
    while (targetModel.superGroupGuid)
    {
        if (targetModel)
        {
            [parentModels addObject:targetModel];
        }
        targetModel = [self findGroupModelWithGuid:targetModel.superGroupGuid withGroups:groups];
    }
    
    [parentModels enumerateObjectsWithOptions:NSEnumerationReverse usingBlock:^(id  _Nonnull obj, NSUInteger idx, BOOL * _Nonnull stop) {
        
        [self.groupView expendItem:obj withAnimated:NO];
    }];
}




//==============================================================================
//
//==============================================================================
- (void)prepareDataWithCompletion:(void(^)(void))completion
{
    __block typeof(self) blockSelf = self;
    
    [self setBusy:@(YES)];
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{

        //////////////////////////////////////////////////
        // 取得是否顯示simplify ui
        NSError *error = nil;
        if([blockSelf.delegate respondsToSelector:@selector(shouldShowSimplifyUI:)])
        {
            blockSelf.shouldShowSimplifyUI = [blockSelf.delegate shouldShowSimplifyUI:&error];
        }
        
        //////////////////////////////////////////////////
        NSArray *groupsOfCurrentAccount = nil;
        NSArray *accountList = nil;
        if ([blockSelf.delegate respondsToSelector:@selector(accountListWithError:)])
        {
            accountList = [[self delegate] accountListWithError:&error];
        }
        
        //////////////////////////////////////////////////
        // 預設展開的帳號
        // !!如果要展開的是root就不用再做
        if ([blockSelf.currentAccountGuid length]>0)
        {
            WCGroupModel *groupModel = [blockSelf findGroupModelWithOwnerGuid:blockSelf.currentAccountGuid withGroups:accountList];
            
            groupsOfCurrentAccount = [blockSelf categoryOfAccountGuid:groupModel.owner];
            [groupModel.subGroups removeAllObjects];
            [groupModel.subGroups addObjectsFromArray:groupsOfCurrentAccount];
            
            blockSelf.currentAccountGuid = groupModel.owner;
            blockSelf.currentAccountName = groupModel.name;
        }
        //////////////////////////////////////////////////
        dispatch_async(dispatch_get_main_queue(), ^{
            
            if (error)
            {
                // 錯誤處理
                //////////////////////////////////////////////////
                NSString *errorMessage = nil;
                if ([blockSelf.delegate respondsToSelector:@selector(companyGroupSelectViewController:requestMessageWithError:)])
                {
                    errorMessage = [blockSelf.delegate companyGroupSelectViewController:blockSelf requestMessageWithError:error];
                }
                
                if([errorMessage length]>0)
                {
                    [WCToastController showMessageToastFromSuperView:blockSelf.view
                                                         withMessage:errorMessage
                                                            position:PPToastPositionCenter];
                    dispatch_after(dispatch_time(DISPATCH_TIME_NOW, (int64_t)(0.3 * NSEC_PER_SEC)), dispatch_get_main_queue(), ^{
                        [blockSelf goBackAnimated:YES];
                    });
                }
            }
            else
            {
                blockSelf.groupsOfCurrentAccount = groupsOfCurrentAccount;
                blockSelf.accountList = accountList;
                [blockSelf.groupView reloadData];
            }
            
            if(completion)
            {
                completion();
            }
            [blockSelf setBusy:@(NO)];
        });
    });
}


//==============================================================================
//
//==============================================================================
- (NSArray *)categoryOfAccountGuid:(NSString *)accountGuid
{
    NSError *error = nil;
    if ([self.delegate respondsToSelector:@selector(groupArrayWithAccountGuid:error:)])
    {
        NSArray *groupsOfCurrentAccount = [self.delegate groupArrayWithAccountGuid:accountGuid error:&error];

        
        // 這邊沒有成功的話，就不會顯示每個類別的名片數量，但不需要回傳錯誤
        if ([self.delegate respondsToSelector:@selector(sharedContactNumberWithCategoryGuids:ownerGuid:error:)])
        {
            NSError *returnError = nil;
            NSArray *allGuids = [self allGuidsFromGroups:groupsOfCurrentAccount];
            if(allGuids)
            {
                NSDictionary *sharedCardCount = [self.delegate sharedContactNumberWithCategoryGuids:allGuids
                                                                                          ownerGuid:accountGuid
                                                                                              error:&returnError];
                
                if(returnError==nil)
                {
                    [self assignCardCount:sharedCardCount toGroups:groupsOfCurrentAccount];
                }
            }
        }
        return groupsOfCurrentAccount;
    }
    
    return nil;
}


//==============================================================================
// 展開某個聯絡人下的群組
//==============================================================================
- (void)prepareCategoryListWithGroupModel:(WCGroupModel *)groupModel
{
    if ([self.delegate respondsToSelector:@selector(groupArrayWithAccountGuid:error:)])
    {
        __block typeof(self) blockSelf = self;
        [self setBusy:@(YES)];
        dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
            
            NSError *error = nil;
            NSArray *groupsOfCurrentAccount = [self categoryOfAccountGuid:groupModel.owner];
            
            // !! 不顯示簡易介面才要清掉與重算account cardCount
            if(blockSelf.shouldShowSimplifyUI==YES)
            {
                // reset All account cardCount
                WCGroupModel *rootModel = [self.accountList firstObject];
                for (WCGroupModel *accountModel in rootModel.subGroups)
                {
                    accountModel.cardCount = -1;
                }
                
                if ([blockSelf.delegate respondsToSelector:@selector(cardCountWithAccountGuid:error:)])
                {
                    groupModel.cardCount = [blockSelf.delegate cardCountWithAccountGuid:groupModel.owner error:&error];
                }
                
            }
            
            //////////////////////////////////////////////////
            
            [groupModel.subGroups removeAllObjects];
            [groupModel.subGroups addObjectsFromArray:groupsOfCurrentAccount];
            
            dispatch_async(dispatch_get_main_queue(), ^{
                
                if(error)
                {
                    // 錯誤處理
                    [WCToastController showMessageToastFromSuperView:blockSelf.view
                                                         withMessage:[error alertMessage]
                                                            position:PPToastPositionCenter];
                }
                else
                {
                    blockSelf.currentAccountGuid = groupModel.owner;
                    blockSelf.currentAccountName = groupModel.name;
                    blockSelf.currentGroupGuid = groupModel.guid;
                    blockSelf.groupsOfCurrentAccount = groupsOfCurrentAccount;
                    [blockSelf.groupView reloadData];

                    // Next: didReloadDataWithGroupView, 這裡面才會展開目前選取的帳號
                }

                [blockSelf setBusy:@(NO)];
            });
        });
        
        //////////////////////////////////////////////////
        
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - public method


//==============================================================================
//
//==============================================================================
- (void)showFromViewController:(UIViewController *)viewController forcePresent:(BOOL)forcePresent
{
    if (forcePresent ||
        viewController.navigationController==nil)
    {
        PPNavigationController *navigationController = [[PPNavigationController alloc] initWithRootViewController:self];
        [viewController presentViewController:navigationController animated:YES completion:nil];
        [navigationController release];
    }
    else
    {
        [viewController presentViewController:self animated:YES completion:nil];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private edit Category methods


//==============================================================================
// @param name 有值表示是修改，沒有值表示是新增
//==============================================================================
- (void)showGroupEditorWithName:(NSString *)name
{
    __block typeof(self) blockSelf = self;
    
    [WCAddOrEditGroupViewController showAddGroupFromViewController:self
                                                     withMaxLength:40//self.maxGroupLength
                                                      currentGroup:name
                                                groupNameValidator:^BOOL(NSString *groupName)
     {
         // WCT都視為合法
         return YES;
     }
                                                   completeHandler:^void (NSString *groupName)
     {
         
//         __block typeof(self) blockSelf = self;
         [self setBusy:@(YES)];
         
         dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
             
             if([name length]==0)
             {
                 if ([blockSelf.editCategoryDelegate respondsToSelector:@selector(companyGroupSelectViewController:groupModelByNewGroupName:ownerGuid:superGroupGuid:error:)])
                 {
                     NSError *error = nil;
                     WCGroupModel *currentAccoutModel = [self findGroupModelWithOwnerGuid:self.currentAccountGuid withGroups:self.accountList];
                     WCGroupModel *currentGroupModel = [self findGroupModelWithGuid:self.currentGroupGuid withGroups:currentAccoutModel.subGroups];
                     [blockSelf.editCategoryDelegate companyGroupSelectViewController:blockSelf
                                                             groupModelByNewGroupName:groupName
                                                                            ownerGuid:([self.currentAccountGuid length]>0)?self.currentAccountGuid:nil
                                                                       superGroupGuid:currentGroupModel.guid
                                                                                error:&error];
                     
                     //////////////////////////////////////////////////
                     // 加到介面
                     dispatch_async(dispatch_get_main_queue(), ^{
                         
                         if(error!=nil)
                         {
                             NSError *customError = PPErrorMake(WCTCompanyGroupSelectViewControllerErrorCode_FailedToCreateCategory, WCTCCV_MLS_CreateCategoryFailed, error);
                             
                             NSString *errorMessage = [self messageWithError:customError];
                             if([errorMessage length]>0)
                             {
                                 [WCToastController showMessageToastFromSuperView:blockSelf.groupView
                                                                      withMessage:errorMessage
                                                                         position:PPToastPositionCenter];
                             }
                         }
                         else
                         {
                             [blockSelf clearAndLoadDataWithCompletion:^{
                                 
                                 WCGroupModel *currentAccoutModel = [self findGroupModelWithOwnerGuid:self.currentAccountGuid withGroups:self.accountList];
                                 WCGroupModel *currentGroupModel = [self findGroupModelWithGuid:self.currentGroupGuid withGroups:currentAccoutModel.subGroups];
                                 
                                 [blockSelf.groupView expendItem:currentGroupModel withAnimated:NO];
                             }];
                         }
                         [self setBusy:@(NO)];
                     });
                 }
             }
             else
             {
                 //  編輯模式要更新到原本的model上
                 if ([blockSelf.editCategoryDelegate respondsToSelector:@selector(companyGroupSelectViewController:updateGroupModel:)])
                 {
                     WCGroupModel *groupModel = [self findGroupModelWithGuid:self.currentGroupGuid withGroups:self.accountList];
 
                     NSString *oldName = groupModel.name;
                     
                     if (groupModel!=nil)
                     {
                         [groupModel setName:groupName];
                     }
                     
                     NSError *error = [blockSelf.editCategoryDelegate companyGroupSelectViewController:blockSelf updateGroupModel:groupModel];
                     
                     if (error!=nil)
                     {
                         if (groupModel!=nil)
                         {
                             [groupModel setName:oldName];
                         }
                     }
                     
                     if([blockSelf.editCategoryDelegate respondsToSelector:@selector(companyGroupSelectViewController:categoryChangedWithGuid:fullName:)])
                     {
                         WCGroupModel*rootModel = [self.accountList firstObject];
                         NSArray *groupNames = [rootModel superGroupNamesInTreeWithGuid:groupModel.guid];
                         
                         NSString *categoryFullName = [groupNames componentsJoinedByString:@"/"];

                         [blockSelf.editCategoryDelegate companyGroupSelectViewController:blockSelf categoryChangedWithGuid:groupModel.guid fullName:categoryFullName];
                     }
                     //////////////////////////////////////////////////
                     // 加到介面
                     dispatch_async(dispatch_get_main_queue(), ^{
                         
                         if(error)
                         {
                             NSError *customError = PPErrorMake(WCTCompanyGroupSelectViewControllerErrorCode_FailedToRenameCategory, WCTCCV_MLS_RenameCategoryFailed, error);
                             NSString *errorMessage = [self messageWithError:customError];
                             
                             if([errorMessage length]>0)
                             {
                                 [WCToastController showMessageToastFromSuperView:blockSelf.groupView
                                                                      withMessage:errorMessage
                                                                         position:PPToastPositionCenter];
                             }
                         }
                         [blockSelf clearAndLoadDataWithCompletion:^{
                             
                         }];
                         [self setBusy:@(NO)];
                     });
                 }
             }
         });
         
     }];
}


//==============================================================================
//
//==============================================================================
- (void)deleteGroupModel:(WCGroupModel *)groupModel
{
    __block typeof(self) blockSelf = self;
    
    // 防呆用，應該不會出現
    if (groupModel==nil)
    {
        dispatch_async(dispatch_get_main_queue(), ^{
            
            [WCToastController showMessageToastFromSuperView:blockSelf.view
                                                 withMessage:WCTCCV_MLS_DeleteCategoryFailed
                                                    position:PPToastPositionCenter];
        });
        return ;
    }
    
    //////////////////////////////////////////////////
    [self setBusy:@(YES)];
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{

        if ([blockSelf.editCategoryDelegate respondsToSelector:@selector(companyGroupSelectViewController:deleteGroupModel:)])
        {
            WCGroupModel *otherGroupModel = nil;
            

            
            NSError *error = [blockSelf.editCategoryDelegate companyGroupSelectViewController:blockSelf deleteGroupModel:groupModel];

            if(error==nil)
            {
                WCGroupModel *accountModel = [self findGroupModelWithOwnerGuid:self.currentAccountGuid withGroups:self.accountList];
                for (WCGroupModel *groupModel in accountModel.subGroups)
                {
                    if (groupModel.ID==WC_GID_Unfiled)
                    {
                        otherGroupModel = [groupModel retain];
                        break;
                    }
                }
            }
            //////////////////////////////////////////////////
            // 更新介面
            dispatch_async(dispatch_get_main_queue(), ^{

                if(error)
                {
                    NSError *customError = PPErrorMake(WCTCompanyGroupSelectViewControllerErrorCode_FailedToDeleteCategory, WCTCCV_MLS_DeleteCategoryFailed, error);
                    NSString *errorMessage = [self messageWithError:customError];


                    if([errorMessage length]>0)
                    {
                        [WCToastController showMessageToastFromSuperView:blockSelf.groupView
                                                             withMessage:errorMessage
                                                                position:PPToastPositionCenter];

                    }
                }
                else
                {
                    // !!刪除後，預設到這個帳號的"其他聯絡人"類別
                    self.currentGroupGuid = nil;
                    
                    if(otherGroupModel.guid!=nil)
                    {
                        self.currentGroupGuid = otherGroupModel.guid;
                    }
                    
                    [blockSelf clearAndLoadDataWithCompletion:^{
                        
                    }];
                }

                [self setBusy:@(NO)];
            });
        }
    });
}


//==============================================================================
//
//==============================================================================
- (NSString *)messageWithError:(NSError *)error
{
    if ([self.delegate respondsToSelector:@selector(companyGroupSelectViewController:requestErrorMessageForError:)])
    {
        return [self.delegate companyGroupSelectViewController:self requestErrorMessageForError:error];
    }
    
    return nil;
}

//==============================================================================
//
//==============================================================================
- (void)clearAndLoadDataWithCompletion:(void(^)(void))completion
{
    self.accountList = nil;
    [self prepareDataWithCompletion:completion];
}







////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPButtonDelegate


//==============================================================================
//
//==============================================================================
- (void)ppButton:(PPButton *)ppButton controlEvent:(UIControlEvents)controlEvent
{
    switch (ppButton.tag)
    {
            // !! isHelpScan勾選才會選擇
        case WCTCGSVC_ButtonTag_Done:
        {
            if(self.enableEditCategoryUI)
            {
                // !!所有聯絡人，帳號不能選
                // 沒有選取
                if(self.currentAccountGuid==nil||
                   self.currentGroupGuid==nil)
                {
                    [WCToastController showMessageToastFromSuperView:self.view
                                                         withMessage:WCTCCV_MLS_NonCategoryBeSelected
                                                            position:PPToastPositionCenter];
                    return ;
                }
                
                // 所有聯絡人
                if ([self.currentAccountGuid isEqualToString:@""]&&
                    [self.currentGroupGuid isEqualToString:@""])
                {
                    [WCToastController showMessageToastFromSuperView:self.view
                                                         withMessage:WCTCCV_MLS_ThisCategoryCantUseForHelpScan
                                                            position:PPToastPositionCenter];
                    return ;
                }
                
                // 帳號
                if ([self.currentGroupGuid length]==0)
                {
                    [WCToastController showMessageToastFromSuperView:self.view
                                                         withMessage:WCTCCV_MLS_ThisCategoryCantUseForHelpScan
                                                            position:PPToastPositionCenter];
                    return ;
                }
                
                __block typeof(self) blockSelf = self;
                
                dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
                    
                    NSArray *selectedItems = [blockSelf.groupView copySelectedItems];
                    
                    WCGroupModel *selectedGroupModel = [selectedItems firstObject];
                    
                    if ([self.delegate respondsToSelector:@selector(companyGroupSelectViewController:didSelectWithAccountGuid:accountName:categoryGuid:categoryFullName:)])
                    {
                        WCGroupModel *accountGroupTreeModel = [self.accountList firstObject];
                        NSMutableArray *groupNames = [accountGroupTreeModel superGroupNamesInTreeWithGuid:self.currentGroupGuid];
                        //////////////////////////////////////////////////

                        NSString *categoryFullName = [groupNames componentsJoinedByString:@"/"];
                        
                        [self.delegate companyGroupSelectViewController:self
                                               didSelectWithAccountGuid:selectedGroupModel.owner
                                                            accountName:self.currentAccountName
                                                           categoryGuid:selectedGroupModel.guid
                                                       categoryFullName:categoryFullName];
                    }
                    
                    [selectedItems release];
                    //////////////////////////////////////////////////
                    dispatch_async(dispatch_get_main_queue(), ^{
                        
                        [blockSelf goBackAnimated:YES];
                    });
                });
            }

            break;
        }
        case WCTCGSVC_ButtonTag_Add:
        {
            // 沒有選取
            if(self.currentAccountGuid==nil||
               self.currentGroupGuid==nil)
            {
                [WCToastController showMessageToastFromSuperView:self.view
                                                     withMessage:WCTCCV_MLS_NonCategoryBeSelected
                                                        position:PPToastPositionCenter];
                return ;
            }
            
            // 所有聯絡人
            if ([self.currentAccountGuid isEqualToString:@""]&&
                [self.currentGroupGuid isEqualToString:@""])
            {
                [WCToastController showMessageToastFromSuperView:self.view
                                                     withMessage:WCTCCV_MLS_SystemCategoryCantBeAdded
                                                        position:PPToastPositionCenter];
                return ;
            }
            
            // 未下載，不能新增
            if([self.currentGroupGuid isEqualToString:WCTCompanyContactUndownloadCategoryGuid])
            {
                [WCToastController showMessageToastFromSuperView:self.view
                                                     withMessage:WCTCCV_MLS_SystemCategoryCantBeAdded
                                                        position:PPToastPositionCenter];
                return ;
            }

            WCGroupModel *currentGroupModel = [self findGroupModelWithGuid:self.currentGroupGuid withGroups:self.accountList];

            if(currentGroupModel==nil)
            {
                [WCToastController showMessageToastFromSuperView:self.view
                                                     withMessage:WCTCCV_MLS_NonCategoryBeSelected
                                                        position:PPToastPositionCenter];
                return ;
            }

            // 其他聯絡人
            if (currentGroupModel.ID==WC_GID_Unfiled)
            {
                [WCToastController showMessageToastFromSuperView:self.view
                                                     withMessage:WCTCCV_MLS_SystemCategoryCantBeAdded
                                                        position:PPToastPositionCenter];
                return ;
            }
            

            // From XXX不能新增子類別
            if ([currentGroupModel.helper length]>0)
            {
                [WCToastController showMessageToastFromSuperView:self.view
                                                     withMessage:WCTCCV_MLS_CantAddCategoryFromHeler
                                                        position:PPToastPositionCenter];
                return;
            }
            
            [self showGroupEditorWithName:nil];
            break;
        }
        case WCTCGSVC_ButtonTag_Rename:
        {
            // 沒有選取
            if(self.currentAccountGuid==nil||
               self.currentGroupGuid==nil)
            {
                [WCToastController showMessageToastFromSuperView:self.view
                                                     withMessage:WCTCCV_MLS_NonCategoryBeSelected
                                                        position:PPToastPositionCenter];
                return ;
            }
            
            // 所有聯絡人
            if ([self.currentAccountGuid isEqualToString:@""]&&
                [self.currentGroupGuid isEqualToString:@""])
            {
                [WCToastController showMessageToastFromSuperView:self.view
                                                     withMessage:WCTCCV_MLS_SystemCategoryCantBeRename
                                                        position:PPToastPositionCenter];
                return ;
            }
            
            // 帳號
            if ([self.currentGroupGuid length]==0)
            {
                [WCToastController showMessageToastFromSuperView:self.view
                                                     withMessage:WCTCCV_MLS_SystemCategoryCantBeRename
                                                        position:PPToastPositionCenter];
                return ;
            }
            
            // 未下載，不能新增
            if([self.currentGroupGuid isEqualToString:WCTCompanyContactUndownloadCategoryGuid])
            {
                [WCToastController showMessageToastFromSuperView:self.view
                                                     withMessage:WCTCCV_MLS_SystemCategoryCantBeRename
                                                        position:PPToastPositionCenter];
                return ;
            }

            // !! 取得目前選取群組
            // !! account, 其他聯絡人，未下載，不能改名
            WCGroupModel *groupModel = [self findGroupModelWithGuid:self.currentGroupGuid withGroups:self.accountList];
            
            if(groupModel==nil)
            {
                [WCToastController showMessageToastFromSuperView:self.view
                                                     withMessage:WCTCCV_MLS_NonCategoryBeSelected
                                                        position:PPToastPositionCenter];
                return ;
            }
            
            // 其他聯絡人
            if (groupModel.ID==WC_GID_Unfiled)
            {
                [WCToastController showMessageToastFromSuperView:self.view
                                                     withMessage:WCTCCV_MLS_SystemCategoryCantBeRename
                                                        position:PPToastPositionCenter];
                return ;
            }
            
            
            [self showGroupEditorWithName:groupModel.name];
            break;
        }
        case WCTCGSVC_ButtonTag_Delete:
        {
            // 沒有選取
            if(self.currentAccountGuid==nil||
               self.currentGroupGuid==nil)
            {
                [WCToastController showMessageToastFromSuperView:self.view
                                                     withMessage:WCTCCV_MLS_NonCategoryBeSelected
                                                        position:PPToastPositionCenter];
                return ;
            }
            
            // 所有聯絡人
            if ([self.currentAccountGuid isEqualToString:@""]&&
                [self.currentGroupGuid isEqualToString:@""])
            {
                [WCToastController showMessageToastFromSuperView:self.view
                                                     withMessage:WCTCCV_MLS_SystemCategoryCantBeDeleted
                                                        position:PPToastPositionCenter];
                return ;
            }
            
            // 帳號
            if ([self.currentGroupGuid length]==0)
            {
                [WCToastController showMessageToastFromSuperView:self.view
                                                     withMessage:WCTCCV_MLS_SystemCategoryCantBeDeleted
                                                        position:PPToastPositionCenter];
                return ;
            }
            
            // 未下載，不能新增
            if([self.currentGroupGuid isEqualToString:WCTCompanyContactUndownloadCategoryGuid])
            {
                [WCToastController showMessageToastFromSuperView:self.view
                                                     withMessage:WCTCCV_MLS_SystemCategoryCantBeDeleted
                                                        position:PPToastPositionCenter];
                return ;
            }
            
            WCGroupModel *modelToDelete = [self findGroupModelWithGuid:self.currentGroupGuid withGroups:self.accountList];;
            
            // 其他聯絡人
            if (modelToDelete.ID==WC_GID_Unfiled)
            {
                [WCToastController showMessageToastFromSuperView:self.view
                                                     withMessage:WCTCCV_MLS_SystemCategoryCantBeDeleted
                                                        position:PPToastPositionCenter];
                return ;
            }
            
            
            // !! 內容不是空的也不能刪
            if([modelToDelete.subGroups count]>0 ||
               modelToDelete.cardCount>0)
            {
                [WCToastController showMessageToastFromSuperView:self.view
                                                     withMessage:WCTCCV_MLS_RemoveCategroyNotEmpty
                                                        position:PPToastPositionCenter];
                return ;
            }
            
            PPAlertController *alertViewController = [PPAlertController alertControllerWithTitle:@""
                                                                                         message:WCTCCV_MLS_ConfirmDelete
                                                                                  preferredStyle:UIAlertControllerStyleAlert];
            [alertViewController addAction:[UIAlertAction actionWithTitle:WCTCCV_MLS_OK style:UIAlertActionStyleDefault handler:^(UIAlertAction * _Nonnull action) {
                
                [self deleteGroupModel:modelToDelete];
            }]];
            
            [alertViewController addAction:[UIAlertAction actionWithTitle:WCTCCV_MLS_Cancel style:UIAlertActionStyleCancel handler:nil]];
            
            [self presentViewController:alertViewController animated:YES completion:nil];

            break;
        }
        case WCTCGSVC_ButtonTag_Back:
        default:
        {
            [self goBackAnimated:YES];
            
            if([self.delegate respondsToSelector:@selector(didFinishCompanyGroupSelectViewController:)])
            {
                [self.delegate didFinishCompanyGroupSelectViewController:self];
            }
            break;
        }
    }
}







////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - WCGroupViewDataSource


//==============================================================================
//
//==============================================================================
- (NSInteger)groupView:(WCGroupView *)groupView numberOfChildrenOfItem:(nullable id)item
{
    if(item==nil)
    {
        return [self.accountList count];
    }
    else
    {
        return [[(WCGroupModel*)item subGroups] count];
    }
}


//==============================================================================
//
//==============================================================================
- (id)groupView:(WCGroupView *)groupView child:(NSInteger)index ofItem:(nullable id)item;
{
    if(item==nil)
    {
        return [self.accountList objectAtIndex:index];
    }
    else
    {
        return [[(WCGroupModel*)item subGroups] objectAtIndex:index];
    }
}



//==============================================================================
//
//==============================================================================
- (CGFloat)groupView:(nonnull WCGroupView *)groupView heightForRowForItem:(id)item
{
    BOOL isRoot = ([[(WCGroupModel *)item owner] isEqualToString:@""]&&[[(WCGroupModel *)item guid] isEqualToString:@""]);
    if(isRoot)
    {
        if (self.hideRootNode)
        {
            UITableViewCell *cell = [groupView.treeView cellForItem:item];
            cell.hidden = YES;
            return 0;
        }
    }
    
    return 44;
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - WCGroupViewDelegate


//==============================================================================
//
//==============================================================================
- (void)groupView:(nonnull WCGroupView *)groupView didLoadCell:(WCGroupTableViewCell*)cell dataObject:(WCGroupModel *)dataObject
{
//    NSLog(@"didLoadCell:%@ dataObject:%@", cell, dataObject);
    
    NSInteger level = [self.groupView.treeView levelForCellForItem:dataObject];
    if(level==WCTCGSVC_TreeLevel_Root)
    {
        if (self.hideRootNode)
        {
            dataObject.name = @"";
            cell.nameLabel.text = @"";
        }
    }
    if(level<=WCTCGSVC_TreeLevel_Account)
    {
        // root and account
        [cell setExpandButtonHidden:NO];
    }
}

//==============================================================================
//
//==============================================================================
- (void)didReloadDataWithGroupView:(WCGroupView *)groupView
{
    if([[[self.accountList firstObject] subGroups] count]>0)
    {
        [groupView expendRootWithAnimated:NO];
    }
    
    //////////////////////////////////////////////////
    // 預設active一個cell

    WCGroupModel *activedGroupModel = nil;
    if (self.currentAccountGuid)
    {
        WCGroupModel *groupModel = nil;
        groupModel = [self findGroupModelWithOwnerGuid:self.currentAccountGuid withGroups:self.accountList];
        [self.groupView expendItem:groupModel withAnimated:NO];
        if(groupModel)
        {
            activedGroupModel = groupModel;
        }
    }

    //////////////////////////////////////////////////
    // 如果之前是檢視類別，要expand到類別的上一層
    // 如果是編輯，expand到active這一層
    if(self.enableEditCategoryUI)
    {
        if (self.currentGroupGuid)
        {
            WCGroupModel *currentGroupModel = nil;
            WCGroupModel *groupModel = nil;
            groupModel = [self findGroupModelWithGuid:self.currentGroupGuid withGroups:self.groupsOfCurrentAccount];
            
            // 先展開到要active的cell的上一層
            if(groupModel)
            {
                [self expandToNode:groupModel withGroups:self.groupsOfCurrentAccount];
                activedGroupModel = groupModel;
            }
            else
            {
                self.currentGroupGuid = @"";
            }
        }
        else if(self.currentAccountGuid)
        {
            WCGroupModel *groupModel = [self findGroupModelWithGuid:self.currentAccountGuid withGroups:self.accountList];
            if (groupModel)
            {
                [self expandToNode:groupModel withGroups:self.accountList];
                activedGroupModel = groupModel;
            }
            else
            {
                self.currentAccountGuid = nil;
                self.currentGroupGuid = nil;
            }
        }
        else
        {
            
        }
        // !! 如果是有開編輯，要active原本的cell
        if(activedGroupModel!=nil)
        {
            [groupView activeCellForItem:activedGroupModel];
        }
    }
    else
    {
        if (self.currentGroupGuid)
        {
            WCGroupModel *parentGroupModel = nil;
            WCGroupModel *groupModel = nil;
            groupModel = [self findGroupModelWithGuid:self.currentGroupGuid withGroups:self.groupsOfCurrentAccount];
            
            if (groupModel.superGroupGuid)
            {
                parentGroupModel = [self findGroupModelWithGuid:groupModel.superGroupGuid withGroups:self.groupsOfCurrentAccount];
            }
            // 先展開到要active的cell的上一層
            if(parentGroupModel)
            {
                [self expandToNode:parentGroupModel withGroups:self.groupsOfCurrentAccount];
            }
        }
    }
}


//==============================================================================
//
//==============================================================================
- (void)groupView:(WCGroupView *)groupView didExpandCell:(WCGroupTableViewCell *)cell
{
    // 展開account時，要重新讀取
//    NSLog(@"%s:%@", __PRETTY_FUNCTION__,cell);
    NSInteger level = [groupView.treeView levelForCell:cell];
    
    if(self.enableEditCategoryUI)
    {
        if(level==WCTCGSVC_TreeLevel_Root)
        {
            // 重讀帳號列表
            [self prepareDataWithCompletion:nil];
        }
        else if(level==WCTCGSVC_TreeLevel_Account)
        {
            // 先清除原本展開的, 原本在root不能清除
            if([self.currentAccountGuid length]>0)
            {
                WCGroupModel *originalAccountModel = [self findGroupModelWithOwnerGuid:self.currentAccountGuid withGroups:self.accountList];
                [originalAccountModel.subGroups removeAllObjects];
            }
            self.currentAccountGuid = nil;
            self.currentAccountName = nil;
            self.currentGroupGuid = nil;
            self.groupsOfCurrentAccount = nil;

            // 只有account那一層才需要讀取group列表
            WCGroupModel *currentModel = [groupView.treeView itemForCell:cell];
            [self prepareCategoryListWithGroupModel:currentModel];
        }
        else
        {
            WCGroupModel *currentModel = [groupView.treeView itemForCell:cell];

            self.currentAccountGuid = currentModel.owner;
            self.currentGroupGuid = currentModel.guid;
            
            // !! 如果是有開編輯，要active原本的cell
            if(currentModel!=nil)
            {
                [groupView activeCellForItem:currentModel];
            }
        }
    }
    else
    {
        if(level==WCTCGSVC_TreeLevel_Root)
        {
            // 重讀帳號列表
            [self prepareDataWithCompletion:nil];
        }
        else if(level==WCTCGSVC_TreeLevel_Account)
        {
            // 先清除原本展開的
            WCGroupModel *originalAccountModel = [self findGroupModelWithOwnerGuid:self.currentAccountGuid withGroups:self.accountList];
            [originalAccountModel.subGroups removeAllObjects];
            
            self.currentAccountGuid = nil;
            self.currentAccountName = nil;
            self.groupsOfCurrentAccount = nil;
            
            // 只有account那一層才需要讀取group列表
            WCGroupModel *currentModel = [groupView.treeView itemForCell:cell];
            [self prepareCategoryListWithGroupModel:currentModel];
        }
    }
}


//==============================================================================
//
//==============================================================================
- (void)groupView:(WCGroupView *)groupView didCollapeCell:(WCGroupTableViewCell *)cell
{
//    NSLog(@"%s:%@", __PRETTY_FUNCTION__,cell);
    NSInteger level = [groupView.treeView levelForCell:cell];
    
    if (self.enableEditCategoryUI)
    {
        WCGroupModel *activedGroupModel = nil;
        
//        if (level==WCTCGSVC_TreeLevel_Root)
//        {
//            self.currentAccountGuid = nil;
//            self.currentAccountName = nil;
//            self.currentGroupGuid = nil;
//            self.groupsOfCurrentAccount = nil;
//            [[[self.accountList firstObject] subGroups] removeAllObjects];
//        }
//        else
        if(level==WCTCGSVC_TreeLevel_Account)
        {
            WCGroupModel *currentModel = [groupView.treeView itemForCell:cell];
            [[currentModel subGroups] removeAllObjects];

            self.currentAccountGuid = currentModel.owner;
            self.currentGroupGuid = currentModel.guid;
            
            activedGroupModel = currentModel;
        }

        // !! 如果是有開編輯，要active原本的cell
        if(activedGroupModel!=nil)
        {
            [groupView activeCellForItem:activedGroupModel];
        }
    }
    else
    {
        if (level==WCTCGSVC_TreeLevel_Root)
        {
            self.currentAccountGuid = nil;
            self.currentAccountName = nil;
            self.groupsOfCurrentAccount = nil;
            [[[self.accountList firstObject] subGroups] removeAllObjects];
        }
        else if(level==WCTCGSVC_TreeLevel_Account)
        {
            WCGroupModel *currentModel = [groupView.treeView itemForCell:cell];
            [[currentModel subGroups] removeAllObjects];
        }
    }
}



//==============================================================================
//
//==============================================================================
- (void)groupView:(WCGroupView *)groupView didClickCell:(WCGroupTableViewCell *)cell
{
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
        
        NSArray *selectedItems = [blockSelf.groupView copySelectedItems];
        
        WCGroupModel *selectedGroupModel = [selectedItems firstObject];
        
        blockSelf.currentAccountGuid = selectedGroupModel.owner;
        blockSelf.currentGroupGuid = selectedGroupModel.guid;
        
        if(blockSelf.enableEditCategoryUI==NO)
        {
            if ([blockSelf.delegate respondsToSelector:@selector(companyGroupSelectViewController:didSelectWithAccountGuid:accountName:categoryGuid:categoryFullName:)])
            {
                [blockSelf.delegate companyGroupSelectViewController:blockSelf
                                            didSelectWithAccountGuid:selectedGroupModel.owner
                                                         accountName:blockSelf.currentAccountName
                                                        categoryGuid:selectedGroupModel.guid
                                                    categoryFullName:nil];
            }
            
        }
        [selectedItems release];
    });
    
}


//==============================================================================
//
//==============================================================================
- (void)groupView:(WCGroupView *)groupView didCell:(WCGroupTableViewCell *)cell pined:(BOOL)pined
{
    NSLog(@"%s:%@, %d", __PRETTY_FUNCTION__,cell, pined);
}
@end
