//
//  RecognitionOperation.m
//
//
//  Created by Howard.Lin on 13/12/4.
//  Copyright (c) 2013年 Penpower. All rights reserved.
//

////////////////////////////////////////////////////////////////////////////////////////////////////

//controller
#import "PPLogController.h"
#import "PPMemTool.h"
#import <Photos/Photos.h>

//other
#import "RecognitionOperation.h"

//extension
#import "PHAsset+Image.h"
#import "UIImage+Additions.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface RecognitionOperation()

- (NSString *)currentFreeMemory;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation RecognitionOperation

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Synthesize

@synthesize
 delegate                 = delegate_,
 lastError                = lastError_,
 recogSourceArray         = recogSourceArray_,
 ppIndexStyle             = ppIndexStyle_,
 recognitionOpertaionMode = recognitionOpertaionMode_;





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Dellocating Object

//===============================================================================
//
//===============================================================================
- (id)initWithDelegate:(id)delegate recogSources:(NSMutableArray *)recogSourceArray
{
    self = [super init];
    
	if(self)
	{
        delegate_         = delegate;
        recogSourceArray_ = [recogSourceArray retain];
        ppIndexStyle_     = PPIndexingStyle_English;
        
        [self setQueuePriority:NSOperationQueuePriorityLow];
    }

	return self;
}


//===============================================================================
//
//===============================================================================
- (void)dealloc
{
    
    delegate_ = nil;
    
    [lastError_ release];
    lastError_ = nil;
    
    [recogSourceArray_ release];
    recogSourceArray_ = nil;

    //////////////////////////////////////////////////
    
    [super dealloc];
}


////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private methods

//===============================================================================
// !! 辨識用的影像統一在此處理
//===============================================================================
- (UIImage *)imageFromRecogSource:(WCRecogSourceModel *)recogSource
{
    __block UIImage *resultImage = nil;
    
    
    //-------------------------------
    // get image object
    //-------------------------------
    if([recogSource.imageSource isKindOfClass:[UIImage class]])
    {
        resultImage = recogSource.imageSource;
    }
    else if([recogSource.imageSource isKindOfClass:[PHAsset class]])
    {
        PHAsset *asset = (PHAsset *)recogSource.imageSource;
        resultImage = [asset imageWithTagetSize:CGSizeMake(asset.pixelWidth, asset.pixelHeight)];

        // 修正圖片顯析度到1600以下
        if(WC_IMS_Original<MAX(resultImage.size.width, resultImage.size.height) ||
           WC_IMS_Original<MIN(resultImage.size.width, resultImage.size.height))
        {
            resultImage = [resultImage imageScalingMaxLength:WC_IMS_Original];
        }
    }
    else if([recogSource.imageSource isKindOfClass:[NSString class]])
    {
        NSString *filePath = (NSString *)recogSource.imageSource;
        
        NSData *img = [NSData dataWithContentsOfFile:filePath];
        resultImage = [UIImage imageWithData:img];
    }
    
//    resultImage = [resultImage imageScalingMaxLength:WC_IMS_Original];
    
    UIImage *tempImage = nil;
    
    //-------------------------------
    // rotate/resize image
    //-------------------------------
    if(recogSource.rotateDegree != 255) // 255代表核心自動旋轉
    {
        CGFloat maxLength = WC_IMS_Original;
        
        if(maxLength>MAX(resultImage.size.width,resultImage.size.height))
        {
            maxLength = MAX(resultImage.size.width,resultImage.size.height);
        }
        
        tempImage = [resultImage imageRotatedByDegrees:recogSource.rotateDegree scalingMaxLength:maxLength];
        resultImage = tempImage;
        
        recogSource.rotateDegree = 0;
    }
    
    return resultImage;
}


//===============================================================================
//
//===============================================================================
- (NSString *)currentFreeMemory
{
    CGFloat memSize = (float)(PPMT_GetFreeMemorySize()/1024/1024);
    return [NSString stringWithFormat:@" (%.2f MB)", memSize];
}


//===============================================================================
//
//===============================================================================
- (void)mainThreadSendNotify:(NSString *)notifyName
{
    NSDictionary *userInfo = [[NSDictionary alloc] initWithObjectsAndKeys:[NSThread currentThread], WCRO_NOTIFY_UserInfo_kThread, nil];
    
    [[NSNotificationCenter defaultCenter] postNotificationName:notifyName object:nil userInfo:userInfo];
    [userInfo release];
}


//===============================================================================
//
//===============================================================================
- (NSInteger)restRecognitionCountWithIgnoreFirstSource:(BOOL)ignoreFirstSource
{
    int startIndex = 0;
    
    if(ignoreFirstSource)
        startIndex = 1;
    
    NSInteger restCount = [RecognitionOperation cardCountInRecogSourceArray:self.recogSourceArray withStartIndex:startIndex];
    
    return restCount;
}


////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Operation overwrite methods

//===============================================================================
//
//===============================================================================
- (void)cancel
{
    [super cancel];
}


//===============================================================================
//
//===============================================================================
- (void)main
{
    @autoreleasepool
    {
        PPLogFunction(@"start (count:%d) %@",[self.recogSourceArray count],[self currentFreeMemory]);

        //////////////////////////////////////////////////
        
        // !! 通知辨識動作開始，appDeleget收到後會準備背景辨識需要的資料
        [self performSelectorOnMainThread:@selector(mainThreadSendNotify:) withObject:WCRO_NOTIFY_OperationBegin waitUntilDone:YES];
        
        //////////////////////////////////////////////////
        
        WCCardModel *cardModel = nil;
        UIImage *image                    = nil;
        WCRecogSourceModel *recogSource   = nil;
        
        //------------------------------------
        // start recognition
        //------------------------------------
        if([self.delegate respondsToSelector:@selector(wcroWillStartRecogWithMode:totalSource:)]==YES)
        {
            [self.delegate wcroWillStartRecogWithMode:self.recognitionOpertaionMode totalSource:[self restRecognitionCountWithIgnoreFirstSource:NO]];
        }
        
        //////////////////////////////////////////////////
        
        while ([self.recogSourceArray count])
        {
#ifdef PPLogMacro
            NSTimeInterval oneRecogStartTime = [[NSDate date] timeIntervalSince1970];

            NSTimeInterval oneStepStartTime  = [[NSDate date] timeIntervalSince1970];
#endif
            @autoreleasepool
            {
                //////////////////////////////////////////////////
                
                //--------------------------------
                // !! wait until memory is enough
                //--------------------------------
                for(int i=0; i<WCRO_MemoryCheckLoop; i++)
                {
                    if(PPMT_GetFreeMemorySize() < WCRO_MinMemorySize)
                    {
                        [NSThread sleepForTimeInterval:0.1];
                    }
                    else
                    {
                        break;
                    }
                }
                
                self.lastError = nil;
                
                //////////////////////////////////////////////////
                
                //-------------------------------
                // prepare variables
                //-------------------------------
                
                recogSource = [self.recogSourceArray objectAtIndex:0];
                
                //-------------------------------
                // get image and recognize
                //-------------------------------
                
                image = [[self imageFromRecogSource:recogSource] retain];
                
                //////////////////////////////////////////////////
                
                PPLogFunction(@"imageFromRecogSource cost:(%.6f sec) %@",[[NSDate date] timeIntervalSince1970]-oneStepStartTime,[self currentFreeMemory]);
#ifdef PPLogMacro
                oneStepStartTime = [[NSDate date] timeIntervalSince1970];
#endif

                //-------------------------------
                // !! enhance image before recog if needed
                //-------------------------------
                while(self.suspended)
                {
                    [NSThread sleepForTimeInterval:0.05];
                }
                
                //////////////////////////////////////////////////
                
                if([self.delegate respondsToSelector:@selector(wcroResetImageBeforeRecognizing:recogSource:)])
                {
                    UIImage *tempImage = [delegate_ wcroResetImageBeforeRecognizing:image recogSource:recogSource];

                    if(tempImage && image != tempImage)
                    {
                        [image release];
                        image = [tempImage retain];
                    }
                }
                
                //////////////////////////////////////////////////
                
                PPLogFunction(@"wcroResetImageBeforeRecognizing cost:(%.6f sec) %@",[[NSDate date] timeIntervalSince1970]-oneStepStartTime,[self currentFreeMemory]);
#ifdef PPLogMacro
                oneStepStartTime = [[NSDate date] timeIntervalSince1970];
#endif
                //////////////////////////////////////////////////

                //-------------------------------
                // preapre card model for saving result & save recognition result
                //-------------------------------
                while(self.suspended)
                {
                    [NSThread sleepForTimeInterval:0.05];
                }
                
                //////////////////////////////////////////////////
                
                // !! 不管辨識成功或失敗接下來都要儲存
               
                NSError *returnError = nil;
                
                if([self.delegate respondsToSelector:@selector(cardModelWithRecogImage:recogSource:error:)]==YES)
                {
                    cardModel = [self.delegate cardModelWithRecogImage:image recogSource:recogSource error:&returnError];
                }
          
                self.lastError = returnError;
                
                PPLogFunction(@"recognize error:%@",self.lastError);
                
                //////////////////////////////////////////////////
                
                PPLogFunction(@"recognizeImage cost:(%.6f sec) %@",[[NSDate date] timeIntervalSince1970]-oneStepStartTime,[self currentFreeMemory]);
      
                //////////////////////////////////////////////////
                
                // !! enhance image after recog if needed
                if([self.delegate respondsToSelector:@selector(wcroResetImageBeforeSaving:recogSource:)])
                {
                    UIImage *tempImage = [self.delegate wcroResetImageBeforeSaving:image recogSource:recogSource];
                    
                    if(tempImage && image!=tempImage)
                    {
                        [image release];
                        image = [tempImage retain];
                    }
                }

                //////////////////////////////////////////////////

                //-------------------------------
                // finish one recognition
                //-------------------------------
                if([self.delegate respondsToSelector:@selector(wcroRecogOneSource:recogImage:ppRecognitionOperationMode:cardModel:restCount:error:)])
                {
                    // !! 第一筆因為已經辨識過了，接下來就會刪除，所以計算剩下張數時要跳過第一筆。
                    // !! 因流程設計，不要在delegate前就把第一筆刪除。
                    NSInteger restCount = [self restRecognitionCountWithIgnoreFirstSource:YES];
                    
                    [self.delegate wcroRecogOneSource:recogSource
                                           recogImage:image
                           ppRecognitionOperationMode:self.recognitionOpertaionMode
                                            cardModel:cardModel
                                            restCount:restCount
                                                error:self.lastError];
                }
                
                
                // !! remove action must be run after delegating
                if([self.recogSourceArray count])
                {
                    [self.recogSourceArray removeObjectAtIndex:0];
                }
                
                //////////////////////////////////////////////////
                
                //!!take a break
                [NSThread sleepForTimeInterval:0.05];
                
                [image release];
                image = nil;
                
                //////////////////////////////////////////////////

                if(self.isCancelled==YES)
                {
                    break;
                }
            }
            
            //////////////////////////////////////////////////
            
            PPLogFunction(@"recog one source end:(%.6f sec) %@",[[NSDate date] timeIntervalSince1970]-oneRecogStartTime,[self currentFreeMemory]);
        }
    
        //////////////////////////////////////////////////

        if(self.isCancelled==NO &&
           [self delegate]!=nil &&
           [[self delegate] respondsToSelector:@selector(wcroDidFinishRecogWithMode:)])
        {
            [[self delegate] wcroDidFinishRecogWithMode:self.recognitionOpertaionMode];
        }
    
        //////////////////////////////////////////////////
        
        // !! 通知辨識動作結束
        [self performSelectorOnMainThread:@selector(mainThreadSendNotify:) withObject:WCRO_NOTIFY_OperationEnd waitUntilDone:YES];

    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class methods

//===============================================================================
//
//===============================================================================
+ (NSInteger)cardCountInRecogSourceArray:(NSArray *)recogSourceArray withStartIndex:(NSInteger)startIndex
{
    WCRecogSourceModel *sourceModel = nil;
    NSInteger count = 0;
    BOOL isFirstSource = YES;
    
    
    for(NSUInteger i=startIndex; i<[recogSourceArray count]; i++)
    {
        if(isFirstSource)
        {
            // !! 第一筆不管正反都要加1
            count++;
            isFirstSource = NO;
        }
        else
        {
            sourceModel = [recogSourceArray objectAtIndex:i];
            
            if(sourceModel.imageType == WC_IT_FrontSide)
                count++;
        }
    }
    
    return count;
}


@end
