//
//  WCMapViewController.m
//  
//
//  Created by Howard on 2015/12/29.
//
//

#import "WCMapViewController.h"
#import <Contacts/Contacts.h>
#import <AddressBook/AddressBook.h>
#import <CoreLocation/CoreLocation.h>

// Define
#import "WCAppearanceDefine.h"
#import "WCMapViewController+ResourceDefine.h"
#import "WCMapPinAnnotationViewDatasource.h"

// Controller
#import "PPLocationController.h"
#import "AppleLocationManager.h"
#import "GoogleLocationManager.h"
#import "PPLogController.h"
#import "PPSectionController.h"
#import "PPSettingsController.h"
#import "WCToastController.h"
#import "WCToolController.h"
#import "PPCountryCodeConvert.h"

// View
#import "PPActionSheet.h"
#import "PPAlertView.h"
#import "PPButton+Factory.h"
#import "PPBusyView.h"
#import "PPNavigationBarView.h"
#import "PPTableView.h"
#import "PPLabelTableViewCell.h"
#import "WCMapAnnotation.h"
#import "WCMapPinAnnotationView.h"
#import "WCCardHolderCell.h"

// Model
#import "WCMapContactRowModel.h"
#import "WCCacheModel.h"

// Operatoin
#import "RenewAnnotationWhenMapDidZoomOperation.h"
#import "LoadListBlockOperation.h"

// Category
#import "UIViewController+ShareApplication.h"
#import "PPSectionController+MapSection.h"
#import "UIApplication+Idle.h"


static BOOL WCMapViewController_EnagleLog = NO;
NSString * const WCMapViewController_LogDir = @"WCMapViewLog";
////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Interface WCMapViewController

@interface WCMapViewController() <WCCardHolderCellDelegate,MKMapViewDelegate,PPButtonDelegate,UIActionSheetDelegate,UITableViewDataSource,UITableViewDelegate,RenewAnnotationWhenMapDidZoomOperationProtocol,UIAlertViewDelegate>

@property (nonatomic,retain) MKMapView *mkMapView;
@property (atomic,retain) NSMutableDictionary *cardModelDictionarys; // @{cardID:CardModel}
@property (nonatomic,retain) NSMutableDictionary *annotationDictionarys; // @{distance:Annotaiton}
@property (nonatomic,retain) PPSectionController *mapViewSectionController;
@property (nonatomic,retain) PPButton            *backButton;
@property (nonatomic,retain) PPButton            *locatioButton;
@property (nonatomic,retain) PPBusyView          *busyView;
@property (nonatomic,retain) PPTableView         *mapTableView;
@property (nonatomic,retain) PPNavigationBarView *navigationBarView;
@property (nonatomic,retain) UITapGestureRecognizer *tapGestureRecognizer;
@property (atomic,retain) NSOperationQueue *updateMapOperationQueue;
@property (atomic,retain) NSOperationQueue *memoryManageQueue;
@property (atomic,retain) NSOperationQueue *loadListOperationQueue;
@property (nonatomic,assign) double lastScale;
@property (nonatomic,assign) BOOL waitFinishGeoaddressingAndStart;
@property (atomic,assign) BOOL beginScrollViewDecelerating;
@property (atomic,assign) BOOL beginScrollViewDraging;
@property (nonatomic,copy) NSString *myLocationCountryCode;
@property (nonatomic,assign) NSUInteger curFetchLocationTaskIndex;
@property (nonatomic,assign) NSUInteger curContinueTimeoutCount;
@property (nonatomic,retain) UIView *busyCoverView;
@property (atomic,assign) BOOL fetchLocationOnce;
@property (nonatomic, retain) PPLogController *logController;
@end


////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Implementation WCMapViewController

@implementation WCMapViewController

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Dellocation Objects

//================================================================================
//
//================================================================================
- (id)init
{
    if(self=[super init])
    {
        self.view.backgroundColor = [UIColor whiteColor];
        
        _cardModelDictionarys = [[NSMutableDictionary alloc] init];
        
        //////////////////////////////////////////////////

        _mapViewSectionController = [[PPSectionController alloc] init];
        
        //////////////////////////////////////////////////

        _annotationDictionarys = [[NSMutableDictionary alloc] init];
        
        //////////////////////////////////////////////////

        _tapGestureRecognizer = [[UITapGestureRecognizer alloc] init];

        if(_tapGestureRecognizer!=nil)
        {
            [_tapGestureRecognizer addTarget:self action:@selector(handleTapMapViewWithGestureRecognizer:)];
        }
        
        //////////////////////////////////////////////////

        _updateMapOperationQueue = [[NSOperationQueue alloc] init];
        
        if(_updateMapOperationQueue!=nil)
        {
            [_updateMapOperationQueue setMaxConcurrentOperationCount:1];
        }
        
        //////////////////////////////////////////////////

        _memoryManageQueue = [[NSOperationQueue alloc] init];
        
        if(_memoryManageQueue!=nil)
        {
            [_memoryManageQueue setMaxConcurrentOperationCount:1];
        }
        
        //////////////////////////////////////////////////

        _loadListOperationQueue = [[NSOperationQueue alloc] init];
        
        if(_loadListOperationQueue!=nil)
        {
            [_loadListOperationQueue setMaxConcurrentOperationCount:1];
        }
        
        //////////////////////////////////////////////////

        _maximumDistanceKiloMeterThreshold = CGFLOAT_MAX;
        
        _mapCenterDistanceKiloMeterThreshold = DefaultMapCenterDistanceKiloMeterThreshold;
        
        _mapRegionSize = DefaultMapRegionSize;
        
        //////////////////////////////////////////////////

        // IPhone 鎖直式
        if(UI_USER_INTERFACE_IDIOM()==UIUserInterfaceIdiomPhone)
        {
            self.ppInterfaceOrientationController.mask = PPInterfaceOrientationControllerMask_Portrait;
        }
        
        //////////////////////////////////////////////////
        
        //使用PPLocationController 時，清掉所有資訊，防止task數目錯誤
        [PPLocationController cancelAllOperation];
    
        //////////////////////////////////////////////////

        [[PPLocationController sharedInstance] setLocationManagerOrders:@[[AppleLocationManager class]]];
        
        //////////////////////////////////////////////////

        _allowContiueTimeoutCount = WCMapViewController_DefaultAllowTimeOutCount;
        
        // 預設地圖中心，新竹蒙恬
        _myLocation = [[CLLocation alloc] initWithLatitude:24.79372055 longitude:121.00637425];
        
        _fetchLocationOnce = YES;
        
        //////////////////////////////////////////////////
        // log instance
        
        self.logController = [[[PPLogController alloc] init] autorelease];
        
        if(self.logController != nil)
        {
            NSString *logDirPath = [WCMapViewController logDirPath];
            NSLog(@"log path:%@", logDirPath);
            [self.logController setFileName:@"WCMapViewLog" atPath:logDirPath];
            [self.logController setMask:PPLogControllerMask_Normal];
        }
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    self.logController = nil;
    
    [_cardModelDictionarys release];
    _cardModelDictionarys = nil;

    [_annotationDictionarys release];
    _annotationDictionarys = nil;
    
    [_myLocation release];
    _myLocation = nil;
    
    [_mapViewSectionController release];
    _mapViewSectionController = nil;
    
    [_tapGestureRecognizer removeTarget:self action:@selector(handleTapMapViewWithGestureRecognizer:)];
    [_tapGestureRecognizer release];
    _tapGestureRecognizer = nil;
    
    [_focusMapPinAnnotationView release];
    _focusMapPinAnnotationView = nil;
    
    [_updateMapOperationQueue cancelAllOperations];
    [_updateMapOperationQueue release];
    _updateMapOperationQueue = nil;
    
    [_memoryManageQueue cancelAllOperations];
    [_memoryManageQueue release];
    _memoryManageQueue = nil;
    
    [_loadListOperationQueue cancelAllOperations];
    [_loadListOperationQueue release];
    _loadListOperationQueue = nil;
    
    [_focusCardID release];
    _focusCardID = nil;
    
    [_myLocationCountryCode release];
    _myLocationCountryCode = nil;
    //////////////////////////////////////////////////
    
    [self.mkMapView removeAnnotations:[self.mkMapView annotations]];
    
    self.mkMapView.delegate = nil;
    [self.mkMapView removeGestureRecognizer:self.tapGestureRecognizer];
    [self.mkMapView removeFromSuperview];
    self.mkMapView = nil;
    
    //////////////////////////////////////////////////
    
    [self.mapTableView setDataSource:nil];
    [self.mapTableView setDelegate:nil];
    
    [self.mapTableView removeFromSuperview];
    self.mapTableView = nil;
    
    //////////////////////////////////////////////////

    [self.busyCoverView removeFromSuperview];
    self.busyCoverView = nil;
    
    //////////////////////////////////////////////////
    
    [self.backButton setDelegate:nil];
    [self.backButton removeFromSuperview];
    self.backButton = nil;
    
    //////////////////////////////////////////////////
    
    [self.locatioButton setDelegate:nil];
    [self.locatioButton removeFromSuperview];
    self.locatioButton = nil;
    
    
    [self.navigationBarView removeFromSuperview];
    self.navigationBarView = nil;
    
    [self.busyView removeFromSuperview];
    self.busyView = nil;
    
    //////////////////////////////////////////////////

    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Responding To View's Event

//================================================================================
//
//================================================================================
- (void)viewWillAppear:(BOOL)animated
{
    [super viewWillAppear:animated];

    //////////////////////////////////////////////////

    self.backButton = [PPButton ppButtonWithIconImageName:ImageNamePPButtonIconForSystemBackImitation
                                                      tag:WCMapViewControllerButtonTag_Back
                                                 delegate:self
                                    normalBackgroundColor:nil
                               highlightedBackgroundColor:WCAppearanceDefine_ButtonBackgroundColor
                                          imageEdgeInsets:WCAppearanceDefine_ButtonImageEdgeInset];
    
    if(self.backButton!=nil)
    {
        [self.backButton addControlEvents:UIControlEventTouchUpInside];
    }
    
    //////////////////////////////////////////////////
    
    self.locatioButton = [PPButton ppButtonWithIconImageName:ImageNameForWCMapViewControllerButtonLocation
                                                         tag:WCMapViewControllerButtonTag_Location
                                                    delegate:self
                                       normalBackgroundColor:nil
                                  highlightedBackgroundColor:WCAppearanceDefine_ButtonBackgroundColor
                                             imageEdgeInsets:WCAppearanceDefine_ButtonImageEdgeInset];
    
    if(self.locatioButton!=nil)
    {
        [self.locatioButton addControlEvents:UIControlEventTouchUpInside];
    }

    //////////////////////////////////////////////////

    self.navigationBarView = [[[PPNavigationBarView alloc] initWithFrame:self.navigationController.navigationBar.bounds] autorelease];
    
    if(self.navigationBarView!=nil)
    {
        self.navigationBarView.titleLabel.text = WCMV_MLS_NeighborhoodContact;
        
        self.navigationBarView.style = PPBarViewStyle_CenteredAbsolute;
        
        //////////////////////////////////////////////////
        
        if(self.backButton!=nil)
        {
            PPBarViewItemModel *itemModel = [PPBarViewItemModel ppBarViewItemModelWithView:self.backButton];
            
            if(itemModel!=nil)
            {
                itemModel.edgeInsetsForNormalBar = UIEdgeInsetsZero;
                
                [self.navigationBarView setItemModels:@[itemModel] forBlockType:PPBarViewBlockType_Left];
            }
        }

        //////////////////////////////////////////////////

        if(self.locatioButton!=nil)
        {
            PPBarViewItemModel *itemModel = [PPBarViewItemModel ppBarViewItemModelWithView:self.locatioButton];
            
            if(itemModel!=nil)
            {
                itemModel.edgeInsetsForNormalBar = UIEdgeInsetsZero;
                
                [self.navigationBarView setItemModels:@[itemModel] forBlockType:PPBarViewBlockType_Right];
            }
        }
        
        //////////////////////////////////////////////////
        
        self.navigationItem.leftBarButtonItem = [[[UIBarButtonItem alloc] initWithCustomView:self.navigationBarView] autorelease];
    }

    //////////////////////////////////////////////////

    self.mkMapView = [[[MKMapView alloc] initWithFrame:self.view.bounds] autorelease];
    
    if(self.mkMapView!=nil)
    {
        [self.mkMapView setDelegate:self];
        [self.mkMapView setShowsUserLocation:YES];
        
        if([self.mkMapView respondsToSelector:@selector(setShowsScale:)]==YES)
        {
            [self.mkMapView setShowsScale:YES];
        }
        //////////////////////////////////////////////////

        [self.mkMapView setMapType:MKMapTypeStandard];
        [self.mkMapView setRegion:MKCoordinateRegionMake(self.mkMapView.centerCoordinate, MKCoordinateSpanMake(0.01, 0.01))];
        [self.view addSubview:self.mkMapView];
        
        //////////////////////////////////////////////////

        if(self.tapGestureRecognizer!=nil)
        {
            [self.mkMapView addGestureRecognizer:self.tapGestureRecognizer];
        }
    }
    
    //////////////////////////////////////////////////

    self.mapTableView = [[[PPTableView alloc] initWithFrame:CGRectZero style:UITableViewStylePlain] autorelease];
    
    if(self.mapTableView!=nil)
    {
        self.mapTableView.dataSource = self;
        self.mapTableView.delegate = self;
        
        [self.view addSubview:self.mapTableView];
    }
    
    //////////////////////////////////////////////////

    self.busyCoverView = [[[UIView alloc] initWithFrame:CGRectZero] autorelease];
    
    if(self.busyCoverView!=nil)
    {
        self.busyCoverView.backgroundColor = [UIColor clearColor];
        self.busyCoverView.userInteractionEnabled = NO;
        [self.view addSubview:self.busyCoverView];
    }

    //////////////////////////////////////////////////

    self.curContinueTimeoutCount = 0;
    
    //////////////////////////////////////////////////

    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(recvLocationControllerNotificationNotification:)
                                                 name:PPLocationControllerNotification
                                               object:nil];
    
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(recvRenewAnnotationsNotification:)
                                                 name:WCMapViewControllerRenewAnnotationsNotification
                                               object:nil];
}


//================================================================================
//
//================================================================================
- (void)viewDidAppear:(BOOL)animated
{
    [super viewDidAppear:animated];
    
    //////////////////////////////////////////////////
    
    [self decideMoveMapCenter];
    //////////////////////////////////////////////////
    if(self.fetchLocationOnce==YES)
    {
        self.fetchLocationOnce = NO;
        
        //////////////////////////////////////////////////
        // NEXT: recvCurrentLocationUpdateSuccessNotification: or recvCurrentLocationUpdateFailNotification:
        if([PPLocationController updatingCurrentLocation]==NO)
        {
            [PPAlertView showWithStyle:UIAlertViewStyleDefault
                                 title:@""
                               message:WCMV_MLS_LocaitonUnKnowError
                              delegate:self
                     cancelButtonTitle:WCMV_MLS_OK
                     otherButtonTitles:nil];
        }
        else
        {
            self.busyView = [[[PPBusyView alloc] initWithSuperView:self.busyCoverView] autorelease];
        }
    }
    else
    {
        self.busyView = [[[PPBusyView alloc] initWithSuperView:self.busyCoverView] autorelease];
        
        [self performSelectorInBackground:@selector(fetchCardModelsLocation) withObject:nil];
    }
}


//================================================================================
//
//================================================================================
- (void)viewWillDisappear:(BOOL)animated
{
    [super viewWillDisappear:animated];
    
    //////////////////////////////////////////////////

    [[NSNotificationCenter defaultCenter] removeObserver:self];
    
    //////////////////////////////////////////////////

    [PPLocationController cancelAllOperation];

    //////////////////////////////////////////////////

    [WCCardHolderCell tellThreadTernimate];
    
    //////////////////////////////////////////////////

    [self.updateMapOperationQueue cancelAllOperations];

    //////////////////////////////////////////////////

    [self.mkMapView removeAnnotations:[self.mkMapView annotations]];
    
    self.mkMapView.delegate = nil;
    [self.mkMapView removeGestureRecognizer:self.tapGestureRecognizer];
    [self.mkMapView removeFromSuperview];
    self.mkMapView = nil;
    
    //////////////////////////////////////////////////

    [self.mapTableView setDataSource:nil];
    [self.mapTableView setDelegate:nil];
    
    [self.mapTableView removeFromSuperview];
    self.mapTableView = nil;
    
    //////////////////////////////////////////////////

    [self.busyCoverView removeFromSuperview];
    self.busyCoverView = nil;
    
    //////////////////////////////////////////////////

    [self.backButton setDelegate:nil];
    [self.backButton removeFromSuperview];
    self.backButton = nil;
    
    //////////////////////////////////////////////////
    
    [self.locatioButton setDelegate:nil];
    [self.locatioButton removeFromSuperview];
    self.locatioButton = nil;
    
    //////////////////////////////////////////////////
    
    [WCMapPinAnnotationView stopFetchData];
}


//================================================================================
//
//================================================================================
- (void)viewDidDisappear:(BOOL)animated
{
    [super viewDidDisappear:animated];
    
    //////////////////////////////////////////////////

    if([[UIApplication sharedApplication] idleDisableCount]==1)
    {
        [[UIApplication sharedApplication] enableIdle:YES];
    }
    
    //////////////////////////////////////////////////
    
    [self.navigationBarView removeFromSuperview];
    self.navigationBarView = nil;
    
    [self.busyView removeFromSuperview];
    self.busyView = nil;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Layout of subviews

//================================================================================
//
//================================================================================
- (void)layoutSubviews
{
    [super layoutSubviews];
    
    //////////////////////////////////////////////////
  
    CGRect visibleRect = [self layoutFrame];

    if(UIInterfaceOrientationIsPortrait([[UIApplication sharedApplication] statusBarOrientation])==YES)
    {
        self.mkMapView.frame = CGRectMake(visibleRect.origin.x,
                                          visibleRect.origin.y,
                                          visibleRect.size.width,
                                          visibleRect.size.height*WCMapViewController_MapViewPortraitRatio);
        
        self.mapTableView.frame = CGRectMake(visibleRect.origin.x,
                                             self.mkMapView.frame.origin.y+self.mkMapView.frame.size.height,
                                             visibleRect.size.width,
                                             visibleRect.size.height-self.mkMapView.frame.size.height);

    }
    else
    {
        CGFloat mapTableViewWidth = visibleRect.size.width/2;
        
        if(mapTableViewWidth>WCMapViewController_TableViewMaxWidth)
        {
            mapTableViewWidth = WCMapViewController_TableViewMaxWidth;
        }
        
        //////////////////////////////////////////////////

        // left
        self.mapTableView.frame = CGRectMake(visibleRect.origin.x,
                                             visibleRect.origin.y,
                                             mapTableViewWidth,
                                             visibleRect.size.height);
        
        //////////////////////////////////////////////////

        // right
        self.mkMapView.frame = CGRectMake(self.mapTableView.frame.origin.x+self.mapTableView.frame.size.width,
                                          visibleRect.origin.y,
                                          visibleRect.size.width-mapTableViewWidth,
                                          visibleRect.size.height);
    }
    
    //////////////////////////////////////////////////

    self.busyCoverView.frame = self.mapTableView.frame;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Override Method

//================================================================================
//
//================================================================================
- (void)didReceiveMemoryWarning
{
    [super didReceiveMemoryWarning];
    
    //////////////////////////////////////////////////

    [self applyMapViewMemoryHotFix];
    
    //////////////////////////////////////////////////

    [[WCCacheModel defaultCache] clearCachedData];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private SectionController Releated Method

//================================================================================
// MARK: 新增連絡人至連絡人列表，更新連絡人列表顯示
//================================================================================
- (void)updateSectionModelsFromCardID:(NSString *)cardID location:(CLLocation *)location
{
    do
    {
        if(cardID==nil ||
           location==nil ||
           self.myLocation==nil ||
           self.mapViewSectionController==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        WCCardModel *cardModel = [self.cardModelDictionarys objectForKey:cardID];
        
        if(cardModel==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        cardModel.displayGPS = [PPLocationController stringForLocation:location];
        
        WCFieldModel *fieldModel = [cardModel fieldWithType:WC_FT_Address index:0];
        
        if(fieldModel!=nil)
        {
            [fieldModel setSubType2FieldWithValue:cardModel.displayGPS recogRect:CGRectZero subType2:WC_FST2_Address_Location];
        }
        
        //////////////////////////////////////////////////
        
        //通知外部地址更新
        if([self.delegate respondsToSelector:@selector(updateLocation:fromAddressString:cardModel:)]==YES)
        {
            [cardModel retain];
            
            [self.delegate updateLocation:location
                        fromAddressString:cardModel.displayAddress
                                cardModel:cardModel];
            
            [cardModel release];
        }
        
        //////////////////////////////////////////////////
        
        //計算距離
        double distance = [PPLocationController meterDistanceBetweenLocation1:self.myLocation location2:location];
        
        if((distance/1000)>=self.maximumDistanceKiloMeterThreshold)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        BOOL needSortAllRowModels = NO;
        
        PPSectionModel *sectionModel = [[self.mapViewSectionController sectionModelsForSearching:NO] firstObject];
        
        if(sectionModel==nil)
        {
            sectionModel = [[[PPSectionModel alloc] init] autorelease];
            
            needSortAllRowModels = YES;
            
            //////////////////////////////////////////////////
            
            if(sectionModel==nil)
            {
                break;
            }
            
            [[self.mapViewSectionController sectionModelsForSearching:NO] addObject:sectionModel];
        }
        
        //////////////////////////////////////////////////
        
        // 尋找已存在的 rowModel
        WCMapContactRowModel *rowModel = nil;
        
        for(rowModel in sectionModel.rowModels)
        {
            if([[rowModel object] compare:cardID]==NSOrderedSame)
            {
                break;
            }
        }
        
        //////////////////////////////////////////////////
        
        if(rowModel==nil)
        {
            rowModel = [WCMapContactRowModel mapContactRowModelFromCardModel:cardModel
                                                              locationString:[PPLocationController stringForLocation:location]
                                                                    distance:distance];
            
            if(rowModel==nil)
            {
                break;
            }
        }
        else
        {
            rowModel.text = cardModel.displayName;
            
            if([rowModel.text length]<=0)
            {
                rowModel.text = WCMV_MLS_NoName;
            }
            
            //////////////////////////////////////////////////
            
            rowModel.jobTitleText = cardModel.displayJobTitle;
            
            rowModel.companyText = cardModel.displayCompany;
            
            //////////////////////////////////////////////////
            
            rowModel.kiloMeterDistance = @(distance/1000);
            rowModel.detailText = [NSString stringWithFormat:@"%.2f KM",[rowModel.kiloMeterDistance floatValue]];
            
            //////////////////////////////////////////////////
            
            rowModel.locationString = [PPLocationController stringForLocation:location];
            rowModel.object = cardID;
        }
        

        //////////////////////////////////////////////////
        
        //所有資料皆未排序
        if(needSortAllRowModels==YES)
        {
            if([sectionModel.rowModels indexOfObject:rowModel]==NSNotFound)
            {
                [sectionModel.rowModels addObject:rowModel];
            }
            
            //////////////////////////////////////////////////
            
            [self allContactListSortedByDistance];
        }
        //只有新增那筆未依距離排序
        else
        {
            NSInteger rowModelIndex = NSNotFound;
            
            if([sectionModel.rowModels indexOfObject:rowModel]==NSNotFound)
            {
                for(WCMapContactRowModel *curMapContactRowModel in sectionModel.rowModels)
                {
                    if([rowModel.kiloMeterDistance floatValue]<[curMapContactRowModel.kiloMeterDistance floatValue])
                    {
                        rowModelIndex = [sectionModel.rowModels indexOfObject:curMapContactRowModel];

                        break;
                    }
                }

                //////////////////////////////////////////////////

                if(rowModelIndex<sectionModel.rowModels.count)
                {
                    [sectionModel.rowModels insertObject:rowModel atIndex:rowModelIndex];
                }
                else
                {
                    [sectionModel.rowModels addObject:rowModel];
                }
            }
        }
        
        //////////////////////////////////////////////////

        if(self.beginScrollViewDraging==NO &&
           self.beginScrollViewDecelerating==NO)
        {
            [self updateContactDistanceListWithToken:WCMapViewController_ObtainVisibleRange];
        }
        
    }while (0);
}


//================================================================================
//
//================================================================================
- (void)updateContactDistanceListWithToken:(NSString *)token
{
    __block typeof(self) blockself = self;
    
    __block LoadListBlockOperation *loadListOperaiton = [LoadListBlockOperation blockOperationWithBlock:^{
        
        do
        {
            if(loadListOperaiton.isCancelled==YES)
            {
                break;
            }
            
            //////////////////////////////////////////////////

            if(blockself.mapTableView==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            [loadListOperaiton retain];
            
            
            dispatch_async(dispatch_get_main_queue(), ^{
                
                // MARK:計算完距離在插入新的cell時維持目前看到的位置
                WCCardHolderCell *cell = (WCCardHolderCell *)[[blockself.mapTableView visibleCells] firstObject];
                
                [blockself.mapTableView reloadData];

                //////////////////////////////////////////////////
                
                if([loadListOperaiton.token compare:WCMapViewController_ObtainVisibleRange]==NSOrderedSame)
                {
                    NSString *cardID = cell.cardID;
                    
                    NSIndexPath *indexPath = [blockself.mapViewSectionController indexPathForCardID:cardID forSearching:NO];
                    
                    //////////////////////////////////////////////////
                    
                    if(indexPath!=nil)
                    {
                        [blockself.mapTableView scrollToRowAtIndexPath:indexPath
                                                      atScrollPosition:UITableViewScrollPositionTop
                                                              animated:NO];
                    }
                }
                
                //////////////////////////////////////////////////


                [loadListOperaiton release];
            });
        }
        while (0);
    }];
    
    //////////////////////////////////////////////////

    if(loadListOperaiton!=nil)
    {
        if([self.loadListOperationQueue operationCount]<=0)
        {
            loadListOperaiton.token = [NSString stringWithString:token];
            [self.loadListOperationQueue addOperation:loadListOperaiton];
        }
    }
}


//================================================================================
//
//================================================================================
- (void)allContactListSortedByDistance
{
    // 依照距離 sort
    PPSectionModel *sectionModel = [[self.mapViewSectionController sectionModelsForSearching:NO] firstObject];
    
    //所有資料皆未排序
    if(sectionModel!=nil)
    {
        NSArray *sortedRowModels = [sectionModel.rowModels sortedArrayUsingSelector:@selector(displayDistanceSortWithMapContactRowModel:)];
        
        if(sortedRowModels!=nil)
        {
            [sectionModel.rowModels removeAllObjects];
            
            [sectionModel.rowModels addObjectsFromArray:sortedRowModels];
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Fetch CardModel location Method

//================================================================================
// MARK: 更新連絡人列表，開始讀取位置
//================================================================================
- (void)fetchCardModelsLocation
{
    do
    {
        @autoreleasepool
        {
            if([self.datasource respondsToSelector:@selector(queryCardIDsWithMapViewController:)]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            NSArray *cardIDs = [self.datasource queryCardIDsWithMapViewController:self];
            
            if([cardIDs count]<=0)
            {
                //沒有聯絡人資料
                [PPAlertView showWithStyle:UIAlertViewStyleDefault
                                     title:@""
                                   message:WCMV_MLS_NoInformationNearbyContacts
                                  delegate:self
                         cancelButtonTitle:WCMV_MLS_OK
                         otherButtonTitles:nil];
                
                break;
            }
            
            //////////////////////////////////////////////////
            
            if([self.datasource respondsToSelector:@selector(cardModelWithCardID:)]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            for(NSString *cardId in cardIDs)
            {
                WCCardModel *cardModel = [self.datasource cardModelWithCardID:cardId];
                
                if(cardModel!=nil)
                {
                    [self.cardModelDictionarys setObject:cardModel forKey:cardModel.ID];
                }
            }
            
            //////////////////////////////////////////////////
            
            //第一次進來要準備已存在GPS資料，顯示
            
            NSMutableArray *rowModels = [NSMutableArray array];
            
            if(rowModels==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            NSDictionary *cardModelDictionarys = [NSDictionary dictionaryWithDictionary:self.cardModelDictionarys];
            
            if(cardModelDictionarys==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////

            //重新計算時，計數要歸零
            self.curFetchLocationTaskIndex = 0;
            
            //////////////////////////////////////////////////
            
            __block typeof(self) blockself = self;
            
            __block NSBlockOperation *blockOperation = [NSBlockOperation blockOperationWithBlock:^{
                
                NSUInteger hasLocationCount = 0;
    
                // 計算所有已存在Location資料的 RowModel
                NSUInteger couldntFetchLocationCount = 0;
                
                for(id key in cardModelDictionarys)
                {
                    if(blockOperation.isCancelled==YES)
                    {
                        break;
                    }
                    
                    //////////////////////////////////////////////////

                    WCCardModel *cardModel = [cardModelDictionarys objectForKey:key];
                    
                    // log
                    [self logMessageWithFormat:@"%s, cardModel %@ - (%@)", __PRETTY_FUNCTION__,cardModel.displayAddress, cardModel.displayGPS];
                    //////////////////////////////////////////////////
                    
                    CLLocation *location = [[PPLocationController copyLocationFromString:cardModel.displayGPS] autorelease];
                    
                    if(location!=nil)
                    {
                        if([PPLocationController isValidLocation:location]==YES)
                        {
                            //計算距離
                            double distance = [PPLocationController meterDistanceBetweenLocation1:self.myLocation
                                                                                        location2:location];
                            
                            [self logMessageWithFormat:@"%s, self.myLocation: %@ - location:%@ (%lf)", __PRETTY_FUNCTION__,self.myLocation,location, distance];

                            if((distance/1000)<self.maximumDistanceKiloMeterThreshold)
                            {
                                
                                
                                //////////////////////////////////////////////////
                                
                                WCMapContactRowModel *mapContactRowModel = [WCMapContactRowModel mapContactRowModelFromCardModel:cardModel
                                                                                                                  locationString:[PPLocationController stringForLocation:location]
                                                                            
                                                                                                                        distance:distance];
                                
                                
                                if(mapContactRowModel!=nil)
                                {
                                    [rowModels addObject:mapContactRowModel];
                                }
                            }
                            else
                            {
                                couldntFetchLocationCount++;
                            }
                            
                            //////////////////////////////////////////////////

                            // 存在有效的 Location 位置，不用再查詢, 進度增加
                            dispatch_sync(dispatch_get_main_queue(),^{
                                blockself.curFetchLocationTaskIndex++;
                                blockself.navigationBarView.titleLabel.text = [WCMV_MLS_NeighborhoodContact stringByAppendingString:[NSString stringWithFormat:@"(%.02f%%)",(float)blockself.curFetchLocationTaskIndex/(float)cardModelDictionarys.count*100]];
                            });

                        }
                        else if([cardModel.displayAddress length]<=0)
                        {
                            couldntFetchLocationCount++;
                            
                            //////////////////////////////////////////////////
                            
                            // 不存在有效的 Location 位置，也沒有地址能反查 GPS, 進度增加
                            dispatch_sync(dispatch_get_main_queue(),^{
                                blockself.curFetchLocationTaskIndex++;
                                blockself.navigationBarView.titleLabel.text = [WCMV_MLS_NeighborhoodContact stringByAppendingString:[NSString stringWithFormat:@"(%.02f%%)",(float)blockself.curFetchLocationTaskIndex/(float)cardModelDictionarys.count*100]];
                            });
                        }
                    }
                    else if([cardModel.displayAddress length]<=0)
                    {
                        couldntFetchLocationCount++;
                        
                        //////////////////////////////////////////////////
                        
                        // Location 格式錯誤，也沒有地址能反查 GPS, 進度增加
                        dispatch_sync(dispatch_get_main_queue(),^{
                            blockself.curFetchLocationTaskIndex++;
                            blockself.navigationBarView.titleLabel.text = [WCMV_MLS_NeighborhoodContact stringByAppendingString:[NSString stringWithFormat:@"(%.02f%%)",(float)blockself.curFetchLocationTaskIndex/(float)cardModelDictionarys.count*100]];
                        });
                    }
                }
                
                //////////////////////////////////////////////////
                
                do
                {
                    if(blockOperation.isCancelled==YES)
                    {
                        break;
                    }
                    
                    //////////////////////////////////////////////////

                    //有存在GPS的資料，刷新頁面
                    if(rowModels.count>0)
                    {
                        PPSectionModel *sectionModel = [[blockself.mapViewSectionController sectionModelsForSearching:NO] firstObject];
                        
                        if(sectionModel==nil)
                        {
                            sectionModel = [[[PPSectionModel alloc] init] autorelease];
                            
                            if(sectionModel!=nil)
                            {
                                [[blockself.mapViewSectionController sectionModelsForSearching:NO] addObject:sectionModel];
                            }
                        }
                        else
                        {
                            [[sectionModel rowModels] removeAllObjects];
                        }
                        
                        //////////////////////////////////////////////////
                        
                        if(sectionModel!=nil)
                        {
                            [sectionModel.rowModels addObjectsFromArray:rowModels];
                            
                            [blockself allContactListSortedByDistance];
                            
                            //////////////////////////////////////////////////
                            
                            [blockself updateMapAnnotationViewWithRect:blockself.mkMapView.visibleMapRect];
                        }
                    }
                    
                    //////////////////////////////////////////////////
                    
                    blockself.curFetchLocationTaskIndex = couldntFetchLocationCount+rowModels.count;
                    
                    //////////////////////////////////////////////////
                    
                    hasLocationCount = rowModels.count;
                    
                    //////////////////////////////////////////////////
                    
                    //顯示更新進度資訊
                    dispatch_sync(dispatch_get_main_queue(), ^{
                        
                        [blockself.busyView removeFromSuperview];
                        blockself.busyView = nil;
                        
                        //////////////////////////////////////////////////
                        
                        if(hasLocationCount>0)
                        {
                            [blockself.mapTableView reloadData];
                        }
                        
                        //////////////////////////////////////////////////
                        
                        //已存在 GPS 的資料，少於所有名片 GPS 資料，開始索取GPS
                        if(blockself.cardModelDictionarys.count>0 &&
                           hasLocationCount<blockself.cardModelDictionarys.count)
                        {
                            [blockself continueNextUpdatingLocation];
                        }
                        else
                        {
                            blockself.navigationBarView.titleLabel.text = WCMV_MLS_NeighborhoodContact;
                        }
                        
                        // mainqueue end
                    });

                    // while loop end
                } 
                while (0);
                
                // operation end
            }];
            
            //////////////////////////////////////////////////

            [self.loadListOperationQueue cancelAllOperations];
            [self.loadListOperationQueue addOperation:blockOperation];
            
            // autorelease pool end
        }
    }
    while (0);
}


//================================================================================
//
//================================================================================
- (void)continueNextUpdatingLocation
{
    [self logMessageWithFormat:@"%s in", __PRETTY_FUNCTION__];
    if([[UIApplication sharedApplication] idleDisableCount]==0)
    {
        [[UIApplication sharedApplication] enableIdle:NO];
    }
    
    //////////////////////////////////////////////////

    @autoreleasepool
    {
        BOOL result = NO;
        
        NSArray *priorityCardModels = [self priorityCardModels];
        
        if(priorityCardModels!=nil &&
           priorityCardModels.count>0)
        {
            [self logMessageWithFormat:@"\t==============================="];
            [self logMessageWithFormat:@"\tpriorityCardModels"];
            for(WCCardModel *cardModel in priorityCardModels)
            {
                result = [PPLocationController fetchLocationFromAddressString:cardModel.displayAddress
                                                                       region:nil
                                                                     identity:cardModel.ID];
                [self logMessageWithFormat:@"\t\t%@; result:%@", cardModel.displayAddress, @(result)];
                // NEXT: recvAddressLocationUpdateSuccessNotification: or recvAddressLocationUpdateFailNotication:

            }
            [self logMessageWithFormat:@"\t==============================="];
        }
        else
        {
            NSArray *needFetchLocationCardModels = [self needFetchLocationCardModels];
            
            [self logMessageWithFormat:@"\t==============================="];
            [self logMessageWithFormat:@"\tneedFetchLocationCardModels"];
            for(WCCardModel *cardModel in needFetchLocationCardModels)
            {
                result = [PPLocationController fetchLocationFromAddressString:cardModel.displayAddress
                                                                       region:nil
                                                                     identity:cardModel.ID];
                [self logMessageWithFormat:@"\t\t%@; result:%@", cardModel.displayAddress, @(result)];
                // NEXT: recvAddressLocationUpdateSuccessNotification: or recvAddressLocationUpdateFailNotication:
            }
             [self logMessageWithFormat:@"\t==============================="];
        }

        //////////////////////////////////////////////////
        
        //沒有任何有效 GPS資料
        if(result==NO)
        {
            if([[UIApplication sharedApplication] idleDisableCount]==1)
            {
                [[UIApplication sharedApplication] enableIdle:YES];
            }
            
            //////////////////////////////////////////////////

            if([self.mapViewSectionController rowObjectsCountForSearching:NO]<=0)
            {
                //沒有聯絡人資料
                [PPAlertView showWithStyle:UIAlertViewStyleDefault
                                     title:@""
                                   message:WCMV_MLS_NoInformationNearbyContacts
                                  delegate:self
                         cancelButtonTitle:WCMV_MLS_OK
                         otherButtonTitles:nil];
            }
            else
            {
                self.navigationBarView.titleLabel.text = WCMV_MLS_NeighborhoodContact;
            }
        }
    }
    
    [self logMessageWithFormat:@"%s out", __PRETTY_FUNCTION__];
}


//================================================================================
// MARK: 檢查是否繼續更新連絡人位址
//================================================================================
- (BOOL)checkNextUpdatingLocationCycle
{
    BOOL result = NO;
    
    do
    {
        if(self.waitFinishGeoaddressingAndStart==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        //剩最後一筆時開始跑指令
        if([PPLocationController readyOperationCount]>1)
        {
            break;
        }

        //////////////////////////////////////////////////

        result = YES;
    }
    while (0);

    
    return result;
}


//================================================================================
// MARK: 優先查詢名片，目前是同國家
//================================================================================
- (NSMutableArray *)priorityCardModels
{
    NSMutableArray *priorityCardModels = nil;
    
    do
    {
        if([self.cardModelDictionarys allKeys]<=0)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        priorityCardModels = [NSMutableArray array];
        
        if(priorityCardModels==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        CLLocation *location = nil;
       
        @autoreleasepool
        {
            for(id key in self.cardModelDictionarys)
            {
                WCCardModel *cardModel = [self.cardModelDictionarys objectForKey:key];
                
                if(cardModel==nil)
                {
                    break;
                }
                
                //////////////////////////////////////////////////
                
                if([[PPLocationController invalidLocationString] isEqualToString:cardModel.displayGPS]==YES)
                {
                    continue;
                }

                location = [[PPLocationController copyLocationFromString:cardModel.displayGPS] autorelease];
                
                if([PPLocationController isValidLocation:location]==YES)
                {
                    continue;
                }
                
                //////////////////////////////////////////////////
                
                if([cardModel.displayAddress length]<=0)
                {
                    continue;
                }
                
                //////////////////////////////////////////////////
                
                WCFieldModel *fieldModel = [cardModel fieldWithType:WC_FT_Address index:0];
                
                if(fieldModel!=nil)
                {
                    NSString *countryCode = [fieldModel valueWithSubType2:WC_FST2_Address_CountryCode];
                    
                    if (countryCode==nil)
                    {
                        NSString *country = [fieldModel valueWithSubType2:WC_FST2_Address_Country];
                        
                        countryCode = [PPCountryCodeConvert countryCodeFromLocalizeString:country];
                    }
                    
                    if(countryCode!=nil &&
                       [countryCode compare:self.myLocationCountryCode options:NSCaseInsensitiveSearch]==NSOrderedSame)
                    {
                        [priorityCardModels addObject:cardModel];
                    }
                }
            }
        }
    }
    while (0);
    
    return priorityCardModels;
}


//================================================================================
//
//================================================================================
- (NSMutableArray *)needFetchLocationCardModels
{
    NSMutableArray *needFetchLocationCardModels = nil;
    
    do
    {
        if([self.cardModelDictionarys allKeys]<=0)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        needFetchLocationCardModels = [NSMutableArray array];
        
        if(needFetchLocationCardModels==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        CLLocation *location = nil;
        
        @autoreleasepool
        {
            for(id key in self.cardModelDictionarys)
            {
                WCCardModel *cardModel = [self.cardModelDictionarys objectForKey:key];
                
                if(cardModel==nil)
                {
                    break;
                }
                
                //////////////////////////////////////////////////
                
                location = [[PPLocationController copyLocationFromString:cardModel.displayGPS] autorelease];
                
                if([PPLocationController isValidLocation:location]==YES)
                {
                    continue;
                }
                
                //////////////////////////////////////////////////
                
                if([cardModel.displayAddress length]<=0)
                {
                    continue;
                }
                
                //////////////////////////////////////////////////
                
                [needFetchLocationCardModels addObject:cardModel];
            }
        }
    }
    while (0);
    
    return needFetchLocationCardModels;
}


//================================================================================
//
//================================================================================
- (BOOL)hasFetchLocationCardModel
{
    BOOL result = NO;
    
    do
    {
        if([self.cardModelDictionarys allKeys]<=0)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        for(id key in self.cardModelDictionarys)
        {
            WCCardModel *cardModel = [self.cardModelDictionarys objectForKey:key];
            
            if(cardModel==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////

            if([cardModel.displayGPS length]<=0 &&
               [cardModel.displayAddress length]>0)
            {
                result = YES;
                
                break;
            }
        }
    }
    while (0);
    
    return result;
}


//================================================================================
//
//================================================================================
- (void)updateDistanceWithRowModels
{
    do
    {
        PPSectionModel *newSectionModel = [[PPSectionModel alloc] init];
        
        if(newSectionModel==nil)
        {
            break;
        }

        //////////////////////////////////////////////////

        NSArray *sectionModels = [self.mapViewSectionController sectionModelsForSearching:NO];
        
        for(PPSectionModel *sectionModel in sectionModels)
        {
            for(WCMapContactRowModel *rowModel in sectionModel.rowModels)
            {
                CLLocation *rowLocation = [PPLocationController copyLocationFromString:rowModel.locationString];
                
                double distance = [PPLocationController meterDistanceBetweenLocation1:self.myLocation
                                                                            location2:rowLocation];
                
                rowModel.kiloMeterDistance = @(distance);
                
                [rowLocation autorelease];
                
                //////////////////////////////////////////////////
                
                if((distance/1000)>=self.maximumDistanceKiloMeterThreshold)
                {
                    continue;
                }
                
                [newSectionModel.rowModels addObject:rowModel];
            }
        }
        
        //////////////////////////////////////////////////
        
        [self.mapViewSectionController replaceSectionModels:@[newSectionModel] forSearching:NO];
        
        [newSectionModel release];
    }
    while (0);
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Handle Notification Method

//================================================================================
//
//================================================================================
- (void)recvLocationControllerNotificationNotification:(NSNotification *)notification
{
    [self logMessageWithFormat:@"%s in (%@)", __PRETTY_FUNCTION__, notification];
    
    PPLocationControllerAccessResult accessResult = [[notification.userInfo objectForKey:PPLocationControllerUserInfoKey_AccessResult] integerValue];
    
    switch (accessResult)
    {
        case PPLocationControllerAccessResult_GetAddressLocationFailure:
        case PPLocationControllerAccessResult_GetAddressLocationSuccess:
        {
            __block typeof(self) blockself = self;
            
            if(accessResult==PPLocationControllerAccessResult_GetAddressLocationFailure)
            {
                dispatch_sync(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
                    [blockself recvAddressLocationUpdateFailNotication:notification];
                });
             
            }
            else
            {
                dispatch_sync(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
                    [blockself recvAddressLocationUpdateSuccessNotification:notification];
                });
            }
        
            break;
        }
        case PPLocationControllerAccessResult_GetCurrentLocationFailure:
        {
            __block typeof(self) blockself = self;

            dispatch_async(dispatch_get_main_queue(), ^{
                [blockself recvCurrentLocationUpdateFailNotification:notification];
            });

            break;
        }
        case PPLocationControllerAccessResult_GetCurrentLocationSuccess:
        {
            [self recvCurrentLocationUpdateSuccessNotification:notification];
            break;
        }
        case PPLocationControllerAccessResult_GetAddressStringFailure:
        {
            [self recvCurrentAddressStringFailNotification:notification];
            
            break;
        }
        case PPLocationControllerAccessResult_GetAddressStringSuccess:
        {
            [self recvCurrentAddressStringSuccessNotification:notification];
            
            break;
        }
        case PPLocationControllerAccessResult_GetAddressLocationEnd:
        {
            __block typeof(self) blockself = self;
            
            dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
                
                [PPLocationController waitUntilAllLocationOperationsAreFinished];
                [blockself recvAddressLocationEndNotication:notification];
            });
            break;
        }
        case PPLocationControllerAccessResult_None:
        default:
        {
            break;
        }
    }
}


//================================================================================
//
//================================================================================
- (void)recvCurrentLocationUpdateSuccessNotification:(NSNotification *)notification
{
    self.curContinueTimeoutCount = 0;
    
    //////////////////////////////////////////////////

    CLLocation *location = [notification.userInfo objectForKey:PPLocationControllerUserInfoKey_Location];
    
    if(location!=nil)
    {
//        NSLog(@"%s, (%f,%f,%f)",__func__,location.coordinate.longitude,location.coordinate.latitude,location.altitude);
     
        //////////////////////////////////////////////////

        self.myLocation = location;
        
        //////////////////////////////////////////////////

        __block typeof(self) blockself = self;
        
        dispatch_async(dispatch_get_main_queue(), ^{
            
            [blockself decideMoveMapCenter];

            //////////////////////////////////////////////////

            //無法開始透過Location反查地址，也要找名片地址
            if([PPLocationController fetchAddressStringFromLocation:location identity:NSStringFromClass([self class])]==NO)
            {
                [self fetchCardModelsLocation];
            }
        });
    }
}


//================================================================================
//
//================================================================================
- (void)recvCurrentLocationUpdateFailNotification:(NSNotification *)notification
{
    NSError *error = [notification.userInfo objectForKey:PPLocationControllerUserInfoKey_Error];
    
    if(error!=nil)
    {
        NSString *errorString;
        
        PPLogFunction(@"%s, error:%@",__func__,error);

        switch([error code])
        {
            case kCLErrorDenied:
            {
                errorString = WCMV_MLS_LocationDenied;
                break;
            }
            case kCLErrorNetwork:
            {
                errorString = WCMV_MLS_LocationNetworkError;
                break;
            }
            case PPLocationControllerErrorCode_NetworkDisconnected:
            {
                [[NSNotificationCenter defaultCenter] removeObserver:self];
                
                [PPLocationController cancelAllOperation];
                
                //////////////////////////////////////////////////

                errorString = WCMV_MLS_ConnectInternet;
                
                break;
            }
            default:
            {
                errorString = WCMV_MLS_LocaitonUnKnowError;
                break;
            }
        }
        
        //////////////////////////////////////////////////

        [self.busyView removeFromSuperview];
        self.busyView = nil;
        
        //////////////////////////////////////////////////
        
        [PPAlertView showWithStyle:UIAlertViewStyleDefault
                             title:@""
                           message:errorString
                          delegate:self
                 cancelButtonTitle:WCMV_MLS_OK
                 otherButtonTitles:nil];
    }
}


//================================================================================
//
//================================================================================
- (void)recvCurrentAddressStringSuccessNotification:(NSNotification *)notification
{
    CLPlacemark *placemark = [[notification.userInfo objectForKey:PPLocationControllerUserInfoKey_Placemarks] firstObject];
    
    if(placemark!=nil)
    {
        if([placemark respondsToSelector:NSSelectorFromString(@"ISOcountryCode")]==YES)
        {
            self.myLocationCountryCode = [placemark performSelector:NSSelectorFromString(@"ISOcountryCode")];
        }

        //////////////////////////////////////////////////

        [self fetchCardModelsLocation];
    }
    
}


//================================================================================
//
//================================================================================
- (void)recvCurrentAddressStringFailNotification:(NSNotification *)notification
{
    NSError *error = [notification.userInfo objectForKey:PPLocationControllerUserInfoKey_Error];
    
    if(error!=nil)
    {
        PPLogFunction(@"%s, error:%@",__func__,error);
        
        switch([error code])
        {
            case PPLocationControllerErrorCode_NetworkDisconnected:
            {
                [[NSNotificationCenter defaultCenter] removeObserver:self];
                
                [PPLocationController cancelAllOperation];
                
                //////////////////////////////////////////////////

                [PPAlertView showWithStyle:UIAlertViewStyleDefault
                                     title:@""
                                   message:WCMV_MLS_ConnectInternet
                                  delegate:self
                         cancelButtonTitle:WCMV_MLS_OK
                         otherButtonTitles:nil];

                break;
            }
            default:
            {
                //透過Location反查地址失敗，也要找名片地址
                [self fetchCardModelsLocation];
                
                break;
            }
        }
    }
    else
    {
        [self fetchCardModelsLocation];
    }
}



//================================================================================
//
//================================================================================
- (void)recvAddressLocationUpdateSuccessNotification:(NSNotification *)notification
{
    NSString *cardID = [notification.userInfo objectForKey:PPLocationControllerUserInfoKey_Identity];
    NSString *method = [notification.userInfo objectForKey:PPLocationControllerUserInfoKey_Method];
    
    //////////////////////////////////////////////////

    CLLocation *location = nil;
    
    if([method compare:NSStringFromClass([AppleLocationManager class])]==NSOrderedSame)
    {
        NSArray *placemarks = [notification.userInfo objectForKey:PPLocationControllerUserInfoKey_Placemarks];
                
        if([placemarks count]>0)
        {
            CLPlacemark *firstPlacemark = [placemarks firstObject];
            
            // !! 吐回來的值可能是錯的，所以要檢查一下與原地址的國家是不是吻合
            WCCardModel *currentCardModel = [self.cardModelDictionarys objectForKey:cardID];
            WCFieldModel *addressField = [currentCardModel fieldWithType:WC_FT_Address index:0];
            WCFieldModel *countryCodeField = [addressField fieldWithSubType2:WC_FST2_Address_CountryCode];
            NSString *countryCode = [countryCodeField value];

            // !!沒有country code要用country name轉換
            if ([countryCode length]==0)
            {
                WCFieldModel *countryNameField = [addressField fieldWithSubType2:WC_FST2_Address_Country];
                countryCode = [PPCountryCodeConvert countryCodeFromLocalizeString:[countryNameField value]];
            }

            if([[countryCode lowercaseString] isEqualToString:[firstPlacemark.ISOcountryCode lowercaseString]]==NO)
            {
                // 不同表示是錯誤地址, 不處理

                currentCardModel.displayGPS = [PPLocationController invalidLocationString];
                
                WCFieldModel *fieldModel = [currentCardModel fieldWithType:WC_FT_Address index:0];
                
                if(fieldModel!=nil)
                {
                    [fieldModel setSubType2FieldWithValue:currentCardModel.displayGPS recogRect:CGRectZero subType2:WC_FST2_Address_Location];
                }
                //////////////////////////////////////////////////
                __block typeof(self) blockself = self;

                self.curFetchLocationTaskIndex++;
                
                //顯示更新資訊
                dispatch_async(dispatch_get_main_queue(), ^{
                    
                    blockself.navigationBarView.titleLabel.text = [WCMV_MLS_NeighborhoodContact stringByAppendingString:[NSString stringWithFormat:@"(%.02f%%)",(float)self.curFetchLocationTaskIndex/(float)self.cardModelDictionarys.count*100]];
                    
                    //////////////////////////////////////////////////

                    if(self.curFetchLocationTaskIndex>=self.cardModelDictionarys.count)
                    {
                        blockself.navigationBarView.titleLabel.text = WCMV_MLS_NeighborhoodContact;
                    }
                });
            }
            else
            {
                location = firstPlacemark.location;
            }
            
        }

        
        //////////////////////////////////////////////////
   
#if PPLogMacro
        for(CLPlacemark *placemark in placemarks)
        {
            PPLogFunction(@"============================================================");
            PPLogFunction(@"street:%@",placemark.thoroughfare);
            PPLogFunction(@"city:%@",placemark.locality);
            PPLogFunction(@"state:%@",placemark.administrativeArea);
            PPLogFunction(@"postalCode:%@",placemark.postalCode);
            PPLogFunction(@"country:%@",placemark.country);
            PPLogFunction(@"ISOCountryCode:%@",placemark.ISOcountryCode);
            PPLogFunction(@"latitude:%f",placemark.location.coordinate.latitude);
            PPLogFunction(@"longitude:%f",placemark.location.coordinate.longitude);
        }
#endif

    }
    else if([method compare:NSStringFromClass([GoogleLocationManager class])]==NSOrderedSame)
    {
        location = [notification.userInfo objectForKey:PPLocationControllerUserInfoKey_Location];
    }
    
    //////////////////////////////////////////////////

    if(location!=nil)
    {
        [self updateSectionModelsFromCardID:cardID location:location];
        
        //////////////////////////////////////////////////

        __block typeof(self) blockself = self;
        
        //顯示更新資訊
        dispatch_async(dispatch_get_main_queue(), ^{
            
            blockself.curFetchLocationTaskIndex++;
            
            //////////////////////////////////////////////////
            
            blockself.navigationBarView.titleLabel.text = [WCMV_MLS_NeighborhoodContact stringByAppendingString:[NSString stringWithFormat:@"(%.02f%%)",(float)self.curFetchLocationTaskIndex/(float)self.cardModelDictionarys.count*100]];
            
            if(self.curFetchLocationTaskIndex>=self.cardModelDictionarys.count)
            {
                blockself.navigationBarView.titleLabel.text = WCMV_MLS_NeighborhoodContact;
            }
        });

        //////////////////////////////////////////////////

        [self updateMapAnnotationViewWithRect:self.mkMapView.visibleMapRect];
    }
}


//================================================================================
//
//================================================================================
- (void)recvAddressLocationUpdateFailNotication:(NSNotification *)notification
{
    do
    {
        @autoreleasepool
        {
            NSError *error = [notification.userInfo objectForKey:PPLocationControllerUserInfoKey_Error];
            
            if(error.code==PPLocationControllerErrorCode_NetworkDisconnected)
            {
                [[NSNotificationCenter defaultCenter] removeObserver:self];
                
                [PPLocationController cancelAllOperation];
                
                //////////////////////////////////////////////////
              
                dispatch_async(dispatch_get_main_queue(), ^{

                [PPAlertView showWithStyle:UIAlertViewStyleDefault
                                     title:@""
                                   message:WCMV_MLS_ConnectInternet
                                  delegate:self
                         cancelButtonTitle:WCMV_MLS_OK
                         otherButtonTitles:nil];
                });
                
                break;
            }
            else if(error.code==PPLocationControllerErrorCode_RequestTimeOut)
            {
                self.curContinueTimeoutCount++;
                
                if(self.curContinueTimeoutCount>=self.allowContiueTimeoutCount)
                {
                    [[NSNotificationCenter defaultCenter] removeObserver:self];
                    
                    [PPLocationController cancelAllOperation];
                    
                    //////////////////////////////////////////////////
                    
                    dispatch_async(dispatch_get_main_queue(), ^{
                        
                        [PPAlertView showWithStyle:UIAlertViewStyleDefault
                                             title:@""
                                           message:WCMV_MLS_ConnectInternet
                                          delegate:self
                                 cancelButtonTitle:WCMV_MLS_OK
                                 otherButtonTitles:nil];
                    });
                    
                    break;
                }
            }
        
            //////////////////////////////////////////////////
            
            NSString *cardID = [notification.userInfo objectForKey:PPLocationControllerUserInfoKey_Identity];
            
            if(cardID==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            WCCardModel *cardModel = [self.cardModelDictionarys objectForKey:cardID];
            
            if(cardModel==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            NSString *method = [notification.userInfo objectForKey:PPLocationControllerUserInfoKey_Method];
            
            NSString *lastMethod = NSStringFromClass([[PPLocationController sharedInstance].locationManagerOrders lastObject]);
            
            //////////////////////////////////////////////////
            
            cardModel.displayGPS = [PPLocationController invalidLocationString];
            
            WCFieldModel *fieldModel = [cardModel fieldWithType:WC_FT_Address index:0];
            
            if(fieldModel!=nil)
            {
                [fieldModel setSubType2FieldWithValue:cardModel.displayGPS recogRect:CGRectZero subType2:WC_FST2_Address_Location];
            }
            
            //////////////////////////////////////////////////

            // Time out，紀錄，等全部結束時繼續跑
            if(error.code==PPLocationControllerErrorCode_RequestTimeOut)
            {
//                self.waitFinishGeoaddressingAndStart = YES;

                // 特殊做法，雖然 TimeOut 時標註為無效位置，只能視圖重新 alloc 資料才會重取
                if([method compare:lastMethod options:NSCaseInsensitiveSearch]==NSOrderedSame)
                {
                    dispatch_async(dispatch_get_main_queue(), ^{
                        [WCToastController showMessageToastFromSuperView:self.view
                                                             withMessage:WCMV_MLS_QueryLocationBusy
                                                                position:PPToastPositionCenter];
                    });
                }
            }
            else
            {
                self.curContinueTimeoutCount = 0;
                
                //////////////////////////////////////////////////

                //通知外部地址更新
                if([self.delegate respondsToSelector:@selector(updateLocation:fromAddressString:cardModel:)]==YES)
                {
                    [self.delegate updateLocation:[PPLocationController invalidLocation]
                                fromAddressString:cardModel.displayAddress
                                        cardModel:cardModel];
                }
            }
   
            //////////////////////////////////////////////////

            if([method compare:lastMethod options:NSCaseInsensitiveSearch]==NSOrderedSame)
            {
                __block typeof(self) blockself = self;
                
                blockself.curFetchLocationTaskIndex++;
                
                //顯示更新資訊
                dispatch_async(dispatch_get_main_queue(), ^{
                    
                    blockself.navigationBarView.titleLabel.text = [WCMV_MLS_NeighborhoodContact stringByAppendingString:[NSString stringWithFormat:@"(%.02f%%)",(float)self.curFetchLocationTaskIndex/(float)self.cardModelDictionarys.count*100]];
                    
                    //////////////////////////////////////////////////

                    if(self.curFetchLocationTaskIndex>=self.cardModelDictionarys.count)
                    {
                        blockself.navigationBarView.titleLabel.text = WCMV_MLS_NeighborhoodContact;
                    }
                });
            }
        }
    }
    while (0);
}


//================================================================================
//
//================================================================================
- (void)recvAddressLocationEndNotication:(NSNotification *)notification
{
    //名片詢問完畢檢查，是否有 time out 而沒有詢問名片
    if([self checkNextUpdatingLocationCycle]==YES)
    {
        self.waitFinishGeoaddressingAndStart = NO;
        
        //////////////////////////////////////////////////
        
        [self continueNextUpdatingLocation];
    }
    // 檢查是否有名片需要檢查的
    else if([self hasFetchLocationCardModel]==YES)
    {
        [self continueNextUpdatingLocation];
    }
    else
    {
        if([[UIApplication sharedApplication] idleDisableCount]==1)
        {
            [[UIApplication sharedApplication] enableIdle:YES];
        }
        
        //////////////////////////////////////////////////
        
        if([self.mapViewSectionController rowObjectsCountForSearching:NO]<=0)
        {
            dispatch_async(dispatch_get_main_queue(), ^{
                //沒有聯絡人資料
                [PPAlertView showWithStyle:UIAlertViewStyleDefault
                                     title:@""
                                   message:WCMV_MLS_NoInformationNearbyContacts
                                  delegate:self
                         cancelButtonTitle:WCMV_MLS_OK
                         otherButtonTitles:nil];
            });
        }
        else
        {
            //資料更新結束，刷新頁面
            __block typeof(self) blockself = self;
            
            //顯示更新資訊
            dispatch_async(dispatch_get_main_queue(), ^{
                blockself.navigationBarView.titleLabel.text = WCMV_MLS_NeighborhoodContact;
                [blockself.mapTableView reloadData];
            });
        }
    }
}

//================================================================================
//
//================================================================================
- (void)recvRenewAnnotationsNotification:(NSNotification *)notification
{
    __block typeof(self) blockself = self;
    
    NSDictionary *annotations = [notification.userInfo objectForKey:WCMapViewControllerUserInfoKey_Annotations];

    dispatch_async(dispatch_get_main_queue(), ^{
        
        do
        {
            @autoreleasepool
            {
                if(annotations==nil ||
                   annotations.count<=0)
                {
                    break;
                }
                
                //////////////////////////////////////////////////
                
                NSMutableArray *toAdd = [NSMutableArray array];
                
                if(toAdd==nil)
                {
                    break;
                }
                
                //////////////////////////////////////////////////
                
                NSMutableArray *toUpdate = [NSMutableArray array];
                
                if(toUpdate==nil)
                {
                    break;
                }
                
                //////////////////////////////////////////////////
                
                NSMutableArray *toRemove = [NSMutableArray array];
                
                if(toRemove==nil)
                {
                    break;
                }
                
                //////////////////////////////////////////////////

                NSMutableArray *exitAllKey = [NSMutableArray arrayWithArray:[blockself.annotationDictionarys allKeys]];
                
                if(exitAllKey==nil)
                {
                    break;
                }
                
                //////////////////////////////////////////////////

                for(id key in annotations)
                {
                    WCMapAnnotation *mapAnnotation = [annotations objectForKey:key];
                    
                    WCMapAnnotation *existAnnotation = [blockself.annotationDictionarys objectForKey:key];
                    
                    if(existAnnotation==nil)
                    {
                        //調整顯示 ID 順序
                        [mapAnnotation bringFirstWithCardID:self.focusCardID];
                
                        //////////////////////////////////////////////////

                        [toAdd addObject:mapAnnotation];
                        [blockself.annotationDictionarys setObject:mapAnnotation forKey:key];
                        
                        [exitAllKey removeObject:key];
                    }
                    else if([mapAnnotation sameCompare:existAnnotation]==NO)
                    {
                        [existAnnotation updateContentWithAnnotaiton:mapAnnotation];
                        
                        //////////////////////////////////////////////////

                        //調整顯示 ID 順序
                        [existAnnotation bringFirstWithCardID:self.focusCardID];
                
                        //////////////////////////////////////////////////

                        [toUpdate addObject:existAnnotation];
                        
                        [exitAllKey removeObject:key];
                    }
                    else
                    {
                        [exitAllKey removeObject:key];
                    }
                }
                
                //////////////////////////////////////////////////

                for(id exitKey in exitAllKey)
                {
                    WCMapAnnotation *needRemoveAnnotation = [self.annotationDictionarys objectForKey:exitKey];
                    
                    [toRemove addObject:needRemoveAnnotation];
                    
                    [self.annotationDictionarys removeObjectForKey:exitKey];
                }
                
                //////////////////////////////////////////////////

                if(toRemove.count>0)
                {
                    [blockself.mkMapView removeAnnotations:toRemove];
                }
                
                //////////////////////////////////////////////////
                
                if(toUpdate.count>0)
                {
                    [blockself.mkMapView removeAnnotations:toUpdate];
                }
                
                //////////////////////////////////////////////////

                [toAdd addObjectsFromArray:toUpdate];
                
                //////////////////////////////////////////////////

                if(toAdd.count>0)
                {
                    [blockself.mkMapView addAnnotations:toAdd];
                }

                // autorelease end
            }
            
            // while end
        }
        while (0);
    });
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Update AnnotationView Method

//================================================================================
// MARK: 更新地圖上名片位置
//================================================================================
- (void)updateMapAnnotationViewWithRect:(MKMapRect)rect
{
    RenewAnnotationWhenMapDidZoomOperation *operation = [[RenewAnnotationWhenMapDidZoomOperation alloc] init];
    
    if(operation!=nil)
    {
        operation.protocol = self;
        operation.mapRect = rect;
        operation.cluseterMaximumPixel = self.defaultClusterPixel;
        
        NSDictionary *cardDictionary = [NSDictionary dictionaryWithDictionary:self.cardModelDictionarys];
        
        for(id object in cardDictionary)
        {
            [operation.cardModels addObject:[self.cardModelDictionarys objectForKey:object]];
        }
        
        //////////////////////////////////////////////////
        
        [self.updateMapOperationQueue addOperation:operation];
    }
    
    [operation release];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Navagion Method

//================================================================================
// MARK: 判斷是否有多種導航軟體
//================================================================================
- (BOOL)multiNavigationEnable
{
    BOOL result = YES;
    
    do
    {
        if([[UIApplication sharedApplication] canOpenURL:[NSURL URLWithString:@"comgooglemaps-x-callback://"]]==YES)
        {
            break;
        }
        else if([[UIApplication sharedApplication] canOpenURL:[NSURL URLWithString:@"baidumap://"]]==YES)
        {
            break;
        }
        
        result = NO;
    }
    while (0);

    return result;
}



//================================================================================
//
//================================================================================
- (BOOL)navigaitonEnableForMapViewControllerMode:(WCMapViewControllerMode)mapViewControllerMode
{
    BOOL result = NO;
    
    do
    {
        if(mapViewControllerMode==WCMapViewControllerMode_Google &&
           [[UIApplication sharedApplication] canOpenURL:[NSURL URLWithString:@"comgooglemaps-x-callback://"]]==NO)
        {
            break;
        }
        else if(mapViewControllerMode==WCMapViewControllerMode_Baidu &&
                [[UIApplication sharedApplication] canOpenURL:[NSURL URLWithString:@"baidumap://"]]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        result = YES;
        
    }
    while (0);
    
    return result;
}



//================================================================================
// MARK: Apple Map 導航
//================================================================================
- (void)appleNavigaionForCardModel:(WCCardModel *)cardModel
{
    do
    {
        @autoreleasepool
        {
            if(cardModel==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            WCFieldModel *fieldModel = [cardModel fieldWithType:WC_FT_Address index:0];
            
            if(fieldModel==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            NSMutableDictionary *addressDictionary = [[[NSMutableDictionary alloc] init] autorelease];
            
            if(addressDictionary==nil)
            {
                break;
            }

            //////////////////////////////////////////////////

            NSString *street = [fieldModel valueWithSubType2:WC_FST2_Address_Street];
            
            NSString *city = [fieldModel valueWithSubType2:WC_FST2_Address_City];
            
            NSString *country = [fieldModel valueWithSubType2:WC_FST2_Address_Country];

            //////////////////////////////////////////////////

            if([[[UIDevice currentDevice] systemVersion] floatValue]>=9.0)
            {
                if(street!=nil)
                {
                    [addressDictionary setObject:street forKey:CNPostalAddressCountryKey];
                }
                
                if(city!=nil)
                {
                    [addressDictionary setObject:city forKey:CNPostalAddressCityKey];
                }
                
                if(country!=nil)
                {
                    [addressDictionary setObject:country forKey:CNPostalAddressCountryKey];
                }
            }
            else
            {
                if(street!=nil)
                {
                    [addressDictionary setObject:street forKey:(__bridge NSString *)kABPersonAddressStreetKey];
                }
                
                if(city!=nil)
                {
                    [addressDictionary setObject:city forKey:(__bridge NSString *)kABPersonAddressCityKey];
                }
                
                if(country!=nil)
                {
                    [addressDictionary setObject:country forKey:(__bridge NSString *)kABPersonAddressCountryKey];
                }
            }
            
            //////////////////////////////////////////////////
            
            CLLocation *location = [[PPLocationController copyLocationFromString:[fieldModel valueWithSubType2:WC_FST2_Address_Location]] autorelease];
            
            if(location==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            MKPlacemark *endPlacemark = [[[MKPlacemark alloc] initWithCoordinate:location.coordinate addressDictionary:addressDictionary] autorelease];
            
            if(endPlacemark==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            MKMapItem *endItem = [[[MKMapItem alloc] initWithPlacemark:endPlacemark] autorelease];
            
            if(endItem==nil)
            {
                break;
            }
            
            endItem.name = cardModel.displayAddress;
            
            //////////////////////////////////////////////////
            
            MKMapItem *startItem = [MKMapItem mapItemForCurrentLocation];
            
            [MKMapItem openMapsWithItems:@[startItem, endItem]
                           launchOptions:@{
                                           MKLaunchOptionsDirectionsModeKey :
                                               MKLaunchOptionsDirectionsModeDriving
                                           }];

        }
    }
    while (0);
}


//================================================================================
// MARK: Baidu Map 導航
//================================================================================
- (void)baiduNavigationForCardModel:(WCCardModel *)cardModel
{
    do
    {
        @autoreleasepool
        {
            if(cardModel==nil)
            {
                break;
            }
    
            //////////////////////////////////////////////////

            if(self.myLocation==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            WCFieldModel *fieldModel = [cardModel fieldWithType:WC_FT_Address index:0];
            
            if(fieldModel==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////

            CLLocation *location = [[PPLocationController copyLocationFromString:[fieldModel valueWithSubType2:WC_FST2_Address_Location]] autorelease];
            
            if(location==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////

            NSString *direcitonRequeset = [NSString stringWithFormat:@"baidumap://map/direction?origin=%f,%f&destination=%f,%f&mode=driving",self.myLocation.coordinate.latitude,self.myLocation.coordinate.longitude,location.coordinate.latitude,location.coordinate.longitude];
            
            if(direcitonRequeset==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            NSString *webStringURL = [direcitonRequeset stringByAddingPercentEscapesUsingEncoding:NSUTF8StringEncoding];
            
            if(webStringURL==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            NSURL *directionsURL = [NSURL URLWithString:webStringURL];
            
            if(directionsURL==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            if([[UIApplication sharedApplication] openURL:directionsURL]==NO)
            {
                //顯示導航失敗
            }  
        }
    }
    while (0);
}


//================================================================================
// MARK: Google Map 導航
//================================================================================
- (void)googleNavigationForCardModel:(WCCardModel *)cardModel
{
    do
    {
        @autoreleasepool
        {
            if(cardModel==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            WCFieldModel *fieldModel = [cardModel fieldWithType:WC_FT_Address index:0];
            
            if(fieldModel==nil)
            {
                break;
            }
    
            //////////////////////////////////////////////////
            
            NSString *addressStringFormat = nil;

            //////////////////////////////////////////////////

            NSString *country = [fieldModel valueWithSubType2:WC_FST2_Address_Country];
            
            if(country!=nil &&
               country.length>0)
            {
                addressStringFormat = country;
            }
            
            //////////////////////////////////////////////////

            NSString *city = [fieldModel valueWithSubType2:WC_FST2_Address_City];
            
            if(addressStringFormat!=nil)
            {
                if(city!=nil &&
                   city.length>0)
                {
                    addressStringFormat = [addressStringFormat stringByAppendingString:[NSString stringWithFormat:@",+%@",city]];
                }
            }
            else if(city!=nil &&
                    city.length>0)
            {
                addressStringFormat = city;
            }
            
            //////////////////////////////////////////////////

            NSString *street = [fieldModel valueWithSubType2:WC_FST2_Address_Street];
            
            if(addressStringFormat!=nil)
            {
                if(street!=nil &&
                   street.length>0)
                {
                    addressStringFormat = [addressStringFormat stringByAppendingString:[NSString stringWithFormat:@",+%@",street]];
                }
            }
            else if(street!=nil &&
                    street.length>0)
            {
                addressStringFormat = street;
            }
           
            //////////////////////////////////////////////////

            if(addressStringFormat==nil)
            {
                break;
            }
      
            //////////////////////////////////////////////////

            NSString *direcitonRequeset = [NSString stringWithFormat:@"comgooglemaps-x-callback://?daddr=%@&x-success=sourceapp://?resume=true&x-source=AirApp",addressStringFormat];
            
            if(direcitonRequeset==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            NSString *webStringURL = [direcitonRequeset stringByAddingPercentEscapesUsingEncoding:NSUTF8StringEncoding];
            
            if(webStringURL==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////

            NSURL *directionsURL = [NSURL URLWithString:webStringURL];

            if(directionsURL==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////

            if([[UIApplication sharedApplication] openURL:directionsURL]==NO)
            {
                //顯示導航失敗
            }
        }
    }
    while (0);
}


//================================================================================
//　MARK: 使用 URL 顯示目前位置
//================================================================================
- (void)navigationForlatitude:(CGFloat)latitude longitude:(CGFloat)longitude
{
    do
    {
        NSString *parameter = [NSString stringWithFormat:@"%f,%f",latitude,longitude];
        
        if(parameter==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        parameter = [parameter stringByAddingPercentEncodingWithAllowedCharacters:NSCharacterSet.URLQueryAllowedCharacterSet];

        
        NSString *url = [NSString stringWithFormat:@"https://maps.apple.com/?q=%@", parameter];
        
        if(url==nil)
        {
            break;
        }
        
        [[UIApplication sharedApplication] openURL:[NSURL URLWithString:url]];

    }
    while (0);
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Handle TapGesture Method

//================================================================================
//
//================================================================================
- (void)handleTapMapViewWithGestureRecognizer:(UIGestureRecognizer *)gestureRecognizer
{
    // MARK: 地圖手勢處理
    [self performSelectorOnMainThread:@selector(checkingAnnotationViewSelectedFromGestureRecognier:)
                           withObject:gestureRecognizer
                        waitUntilDone:YES];
}


//================================================================================
//
//================================================================================
- (void)checkingAnnotationViewSelectedFromGestureRecognier:(UIGestureRecognizer *)gestureRecognizer
{
    WCMapPinAnnotationView *selectedMapAnnotationView = nil;
    
    for (WCMapAnnotation *mapAnnotation in self.mkMapView.annotations)
    {
        if([mapAnnotation isKindOfClass:[MKUserLocation class]]==YES)
        {
            continue;
        }
        
        if(self.focusCardID!=nil &&
           [mapAnnotation isCardIDExist:self.focusCardID]==NSNotFound)
        {
            continue;
        }
    
        //////////////////////////////////////////////////
        
        WCMapPinAnnotationView *annotationView = (WCMapPinAnnotationView *)[self.mkMapView viewForAnnotation:mapAnnotation];
        
        CGPoint annotationPoint = [self.mkMapView convertCoordinate:mapAnnotation.coordinate toPointToView:self.mkMapView];

        if(annotationPoint.x>=0 &&
           annotationPoint.y>=0 &&
           annotationPoint.x<=self.mkMapView.frame.size.width &&
           annotationPoint.y<=self.mkMapView.frame.size.height)
        {
            CGPoint checkPoint = [gestureRecognizer locationInView:annotationView];
            
            //命中
            if([annotationView tapViewWithPoint:checkPoint])
            {
                selectedMapAnnotationView = annotationView;
            
                //////////////////////////////////////////////////

                //!! 把點選的annotationView移到最上方
                [annotationView.superview bringSubviewToFront:annotationView];
                
                //////////////////////////////////////////////////
                
                if([mapAnnotation multiCardIDs]==YES)
                {
                    // MARK:跳聯絡人選單
                    [self showCardListActionSheetWithMapAnntation:mapAnnotation];
                }
                else
                {
                    if([self multiNavigationEnable]==YES)
                    {
                        [self showMapListActionSheetForCardID:[mapAnnotation currentCardID]];
                    }
                    else
                    {
                        [self appleNavigaionForCardModel:[self.cardModelDictionarys objectForKey:[mapAnnotation currentCardID]]];
                    }
                }
                
                //////////////////////////////////////////////////

                [mapAnnotation bringFirstWithCardID:self.focusCardID];
                
                self.focusCardID = [mapAnnotation currentCardID];
                
                //////////////////////////////////////////////////

                break;
            }
        }
    }
    
    //////////////////////////////////////////////////
    
    if(selectedMapAnnotationView!=nil)
    {
        if(selectedMapAnnotationView!=self.focusMapPinAnnotationView)
        {
            [self.focusMapPinAnnotationView setActive:NO];
        }
        
        //////////////////////////////////////////////////

        [selectedMapAnnotationView setActive:YES];
        self.focusMapPinAnnotationView = selectedMapAnnotationView;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Show Menu Method

//================================================================================
//
//================================================================================
- (void)showCardListActionSheetWithMapAnntation:(WCMapAnnotation *)mapAnnotation
{
    do
    {
        if(mapAnnotation==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        // 準備顯示資料
        NSDictionary *displayTitlesDictionary = [mapAnnotation displayTitlesDictionary];
        
        if(displayTitlesDictionary==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        // 用來紀錄顯示資料的 CardID，防止顯示名稱一樣時，無法判斷是點到哪個連絡人
        NSMutableArray *cardIDArray = [NSMutableArray array];
        
        if(cardIDArray==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        PPActionSheet *actionSheet = [[PPActionSheet alloc] initWithTitle:WCMV_MLS_SelectContact
                                                                 delegate:self
                                                        cancelButtonTitle:WCMV_MLS_Cancel
                                                   destructiveButtonTitle:nil
                                                        otherButtonTitles:nil];
        
        if(actionSheet!=nil)
        {
            actionSheet.tag = WCMapViewControllerActionSheetTag_ContactSelect;
            
            //////////////////////////////////////////////////

            for(id key in displayTitlesDictionary)
            {
                [cardIDArray addObject:key];
                
                //////////////////////////////////////////////////

                NSString *name = [displayTitlesDictionary objectForKey:key];
                
                if(name==nil ||
                   [name length]<=0)
                {
                    name = WCMV_MLS_NoName;
                }
                
                [actionSheet addButtonWithTitle:name];
            }
            
            actionSheet.userInfo = cardIDArray;
            
            [actionSheet showInView:self.view];
        }
        
        [actionSheet release];
    }
    while (0);
}


//================================================================================
//
//================================================================================
- (void)showMapListActionSheetForCardID:(NSString *)cardID
{
    do
    {
        PPActionSheet *actionSheet = [[PPActionSheet alloc] initWithTitle:WCMV_MLS_SelectMap
                                                                 delegate:self
                                                        cancelButtonTitle:WCMV_MLS_Cancel
                                                   destructiveButtonTitle:nil
                                                        otherButtonTitles:nil];
        
        if(actionSheet!=nil)
        {
            actionSheet.tag = WCMapViewControllerActionSheetTag_MapSelect;
            actionSheet.userInfo = cardID;
            
            //////////////////////////////////////////////////

            [actionSheet addButtonWithTitle:WCMV_MLS_AppleMap];
            
            if([self navigaitonEnableForMapViewControllerMode:WCMapViewControllerMode_Baidu]==YES)
            {
                [actionSheet addButtonWithTitle:WCMV_MLS_BaiduMap];
            }
            
            if([self navigaitonEnableForMapViewControllerMode:WCMapViewControllerMode_Google]==YES)
            {
                [actionSheet addButtonWithTitle:WCMV_MLS_GoogleMap];
            }

            [actionSheet showInView:self.view];
        }
        
        [actionSheet release];
    }
    while (0);
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Bring MapAnnotationView to Front Method

//================================================================================
//
//================================================================================
- (void)bringMapAnnotationViewToSuperViewWithCardID:(NSString *)cardID
{
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private MkMapView Memory Management Method

//================================================================================
//
//================================================================================
- (void)applyMapViewMemoryHotFix
{
    //切換地圖型態，會釋放暫存記憶體, trick
    
    switch (self.mkMapView.mapType)
    {
        case MKMapTypeHybrid:
        {
            self.mkMapView.mapType = MKMapTypeStandard;
        }
        case MKMapTypeStandard:
        {
            self.mkMapView.mapType = MKMapTypeHybrid;
        }
        default:
        {
            break;
        }
    }
    
    self.mkMapView.mapType = MKMapTypeStandard;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPButtonDelegate Method

//================================================================================
//
//================================================================================
- (void)ppButton:(PPButton *)ppButton controlEvent:(UIControlEvents)controlEvent
{
    switch (ppButton.tag)
    {
        case WCMapViewControllerButtonTag_Back:
        {
            [self.cardModelDictionarys removeAllObjects];
            [[self.mapViewSectionController sectionModelsForSearching:NO] removeAllObjects];
            
            //////////////////////////////////////////////////

            [self goBackAnimated:YES];
            break;
        }
        case WCMapViewControllerButtonTag_Home:
        {
            
            break;
        }
        case WCMapViewControllerButtonTag_Location:
        {
            // MARK: 點選定位鍵
            if(self.myLocation!=nil)
            {
                [self moveMapToLocation:self.myLocation];
            }
            
            break;
        }
        default:
        {
            break;
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UIActionSheet Method

//================================================================================
//
//================================================================================
- (void)actionSheet:(UIActionSheet *)actionSheet didDismissWithButtonIndex:(NSInteger)buttonIndex
{
    NSString *buttonTitle = [actionSheet buttonTitleAtIndex:buttonIndex];
    
    do
    {
        if([buttonTitle compare:WCMV_MLS_Cancel]==NSOrderedSame)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        if(actionSheet.tag==WCMapViewControllerActionSheetTag_ContactSelect)
        {
            if([actionSheet respondsToSelector:@selector(userInfo)]==NO)
            {
                break;
            }
            
            NSArray *cardArray = [actionSheet performSelector:@selector(userInfo)];
            
            //////////////////////////////////////////////////
            
            // 扣掉Cancel 的Index
            if(buttonIndex>1)
            {
                buttonIndex--;
            }
            
            if(cardArray.count<=buttonIndex)
            {
                break;
            }
     
            //////////////////////////////////////////////////

            NSString *cardID = [cardArray objectAtIndex:buttonIndex];
            
            if(cardID==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////

            if([self multiNavigationEnable]==YES)
            {
                [self showMapListActionSheetForCardID:cardID];
            }
            else
            {
                [self appleNavigaionForCardModel:[self.cardModelDictionarys objectForKey:cardID]];
            }
        }
        else
        {
            if([actionSheet respondsToSelector:@selector(userInfo)]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////

            NSString *cardID = [actionSheet performSelector:@selector(userInfo)];
            
            if(cardID==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////

            if([buttonTitle compare:WCMV_MLS_AppleMap]==NSOrderedSame)
            {
                [self appleNavigaionForCardModel:[self.cardModelDictionarys objectForKey:cardID]];
            }
            else if([buttonTitle compare:WCMV_MLS_BaiduMap]==NSOrderedSame)
            {
                [self baiduNavigationForCardModel:[self.cardModelDictionarys objectForKey:cardID]];
            }
            else if([buttonTitle compare:WCMV_MLS_GoogleMap]==NSOrderedSame)
            {
                [self googleNavigationForCardModel:[self.cardModelDictionarys objectForKey:cardID]];
            }
        }
        
    }
    while (0);

}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UIAlertViewDelegate Method

//================================================================================
//
//================================================================================
- (void)alertView:(UIAlertView *)alertView didDismissWithButtonIndex:(NSInteger)buttonIndex
{
    [self goBackAnimated:YES];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UIScrollViewDelegate Method

//================================================================================
//
//================================================================================
- (void)scrollViewWillBeginDecelerating:(UIScrollView *)scrollView
{
    
    [self.loadListOperationQueue cancelAllOperations];
    
    self.beginScrollViewDecelerating = YES;
}


//================================================================================
//
//================================================================================
- (void)scrollViewDidEndDecelerating:(UIScrollView *)scrollView
{
    self.beginScrollViewDecelerating = NO;
    
    if(self.beginScrollViewDraging==NO)
    {
        __block typeof(self) blockself =self;
        
        dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 00), ^{
            [blockself updateContactDistanceListWithToken:WCMapViewController_ObtainNoVisibleRange];
        });
    }
}


//================================================================================
//
//================================================================================
- (void)scrollViewWillBeginDragging:(UIScrollView *)scrollView
{
    [self.loadListOperationQueue cancelAllOperations];
    
    self.beginScrollViewDraging = YES;
}


//================================================================================
//
//================================================================================
- (void)scrollViewDidEndDragging:(UIScrollView *)scrollView willDecelerate:(BOOL)decelerate;
{
    self.beginScrollViewDraging = NO;
    
    if(self.beginScrollViewDecelerating==NO)
    {
        __block typeof(self) blockself =self;
        
        dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 00), ^{
            [blockself updateContactDistanceListWithToken:WCMapViewController_ObtainNoVisibleRange];
        });
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UITableViewDatasource Method

//================================================================================
//
//================================================================================
- (NSInteger)numberOfSectionsInTableView:(UITableView *)tableView
{
    return [self.mapViewSectionController numberOfSectionsForSearching:NO];
}


//================================================================================
//
//================================================================================
- (NSInteger)tableView:(UITableView *)tableView numberOfRowsInSection:(NSInteger)section
{
    return [self.mapViewSectionController numberOfRowsInSection:section forSearching:NO];
}


//================================================================================
//
//================================================================================
- (CGFloat)tableView:(UITableView *)tableView heightForRowAtIndexPath:(nonnull NSIndexPath *)indexPath
{
    return WCMapViewController_TableViewRowHeight;
}


//================================================================================
//
//================================================================================
- (void)tableView:(UITableView *)tableView didSelectRowAtIndexPath:(NSIndexPath *)indexPath
{
    do
    {
        WCMapContactRowModel *rowModel = (WCMapContactRowModel *)[self.mapViewSectionController rowModelAtIndexPath:indexPath forSearching:NO];
        
        if(rowModel==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        BOOL jumpToTargetCell = YES;
        
        NSIndexPath *lastFocusIndexPath = [self.mapViewSectionController indexPathForCardID:self.focusCardID
                                                                               forSearching:NO];
        
        if(lastFocusIndexPath!=nil)
        {
            if(lastFocusIndexPath.row!=indexPath.row ||
               lastFocusIndexPath.section!=indexPath.section)
            {
                WCCardHolderCell *cardHolderCell = [tableView cellForRowAtIndexPath:indexPath];
                WCCardHolderCell *lastFocusCardHolderCell = [tableView cellForRowAtIndexPath:lastFocusIndexPath];
                
                dispatch_async(dispatch_get_main_queue(), ^{
                    
                    if(cardHolderCell!=nil)
                    {
                        [cardHolderCell setSelected:YES];
                    }
                    
                    //////////////////////////////////////////////////
                    
                    if(lastFocusCardHolderCell!=nil)
                    {
                        [lastFocusCardHolderCell setSelected:NO];
                    }
                });
            }
            //點擊到同一筆
            else
            {
                jumpToTargetCell = NO;
            }
        }
        
        //////////////////////////////////////////////////
        
        __block typeof(self) blockself = self;
        
        dispatch_async(dispatch_get_main_queue(), ^{
            
            CLLocation *location = [PPLocationController copyLocationFromString:rowModel.locationString];
            
            if(location!=nil)
            {
                [blockself moveMapToLocation:location];
            }
            
            //////////////////////////////////////////////////
            
            if(jumpToTargetCell==YES)
            {
                NSString *selectedCardID = [rowModel object];
                
                //////////////////////////////////////////////////
                
                if([selectedCardID compare:blockself.focusCardID]!=NSOrderedSame)
                {
                    if(blockself.focusMapPinAnnotationView!=nil)
                    {
                        [blockself.focusMapPinAnnotationView setActive:NO];
                    }
                    
                    //////////////////////////////////////////////////
                    
                    NSString *key = [PPLocationController stringForLocation:location];
                    
                    WCMapAnnotation *annotation = [blockself.annotationDictionarys
                                                   objectForKey:key];
                    
                    if(annotation!=nil)
                    {
                        [annotation bringFirstWithCardID:selectedCardID];
                    
                        //////////////////////////////////////////////////
                        
                        WCMapPinAnnotationView *annotationView = (WCMapPinAnnotationView *)[blockself.mkMapView viewForAnnotation:annotation];
                        
                        if(annotationView!=nil)
                        {
                            [annotationView setActive:YES];
                            
                            [annotationView fetchDataWithProtocol:self];
                            
                            //////////////////////////////////////////////////
                            
                            self.focusMapPinAnnotationView = annotationView;
                        }
                    }
                    
                    //////////////////////////////////////////////////
                    
                    blockself.focusCardID = [rowModel object];
                }
            }
            
            //////////////////////////////////////////////////
            
            [location release];
        });
    }
    while (0);
}


//================================================================================
//
//================================================================================
- (void)tableView:(UITableView *)tableView willDisplayCell:(UITableViewCell *)cell forRowAtIndexPath:(NSIndexPath *)indexPath
{
    __block typeof(self) blockself = self;
    
    dispatch_async(dispatch_get_main_queue(), ^{
        
        NSIndexPath *focusIndexPath = [blockself.mapViewSectionController indexPathForCardID:blockself.focusCardID
                                                                                forSearching:NO];
        
        if(focusIndexPath!=nil &&
           focusIndexPath.section==indexPath.section &&
           focusIndexPath.row==indexPath.row)
        {
            cell.selected = YES;
        }
    });
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UITableViewDelegate Method

//================================================================================
//
//================================================================================
- (UITableViewCell *)tableView:(UITableView *)tableView cellForRowAtIndexPath:(NSIndexPath *)indexPath
{
    
    WCCardHolderCell *tableViewCell = [tableView dequeueReusableCellWithIdentifier:NSStringFromClass([WCCardHolderCell class])];
    
    WCMapContactRowModel *rowModel = (WCMapContactRowModel *)[self.mapViewSectionController rowModelAtIndexPath:indexPath forSearching:NO];
    
    if(tableViewCell==nil)
    {
        tableViewCell = [[[WCCardHolderCell alloc] initWithDelegate:self
                                                    reuseIdentifier:NSStringFromClass([WCCardHolderCell class])] autorelease];
        
        if(tableViewCell!=nil)
        {
            tableViewCell.selectedBackgroundView.backgroundColor = WCAppearanceDefine_CellHighlightedBGColor;
        }
    }
    
    //////////////////////////////////////////////////
    
    if(rowModel!=nil)
    {
        tableViewCell.cardID = [rowModel object];
        
        //////////////////////////////////////////////////
        
        tableViewCell.nameLabel.text = rowModel.text;
        
        //////////////////////////////////////////////////

        tableViewCell.additionLabel.text = [rowModel distanceTextFormatForMapViewControllerDistanceStyle:self.mapViewControllerDistanceStyle];
        
        //////////////////////////////////////////////////
        
        tableViewCell.companyLabel.text = rowModel.companyText;
        tableViewCell.jobTitleLabel.text = rowModel.jobTitleText;
        
        tableViewCell.selectMode = NO;
        tableViewCell.verified = YES;
        
        tableViewCell.cellType = WCCardHolderCellType_Distance;
    }
    
    //////////////////////////////////////////////////
    
    NSString *key = [tableViewCell.cardID stringByAppendingString:[NSString stringWithFormat:@"%td",WC_IT_FrontSide]];
    
    if(key!=nil)
    {
        UIImage *image = [[WCCacheModel defaultCache] thumbImageWithCardID:key];
        
        if(image==nil)
        {
            [tableViewCell startLoadCardImage];
        }
        else if(tableViewCell.cardImageView.image!=image)
        {
            tableViewCell.cardImageView.image = image;
        }
    }
    
    //////////////////////////////////////////////////
    
    return tableViewCell;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - WCCardHolderCellDelegate Method

//================================================================================
//
//================================================================================
- (UIImage *)copyCardListCellThumbImage:(WCCardHolderCell *)cell
{
    UIImage *image = nil;
    
    do
    {
        NSString *key = [cell.cardID stringByAppendingString:[NSString stringWithFormat:@"%td",WC_IT_FrontSide]];
        
        if(key==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        image = [[WCCacheModel defaultCache] thumbImageWithCardID:key];
        
        if(image!=nil)
        {
            break;
        }
    
        //////////////////////////////////////////////////

        if(self.datasource==nil ||
           [self.datasource respondsToSelector:@selector(imageWithImageType:imageSubType:cardID:)]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        image = [self.datasource imageWithImageType:WC_IT_FrontSide
                                       imageSubType:WC_IST_Thumbnail
                                             cardID:cell.cardID];
        
        //////////////////////////////////////////////////
        
        if(image!=nil)
        {
            [[WCCacheModel defaultCache] addThumbImage:image
                                            withCardID:key];
        }
        else
        {
            [[WCCacheModel defaultCache] removeThumbImageWithCardID:key];
        }
    }
    while (0);
    
    return [image retain];
}


//================================================================================
//
//================================================================================
- (void)didClickAdditionButtonWithCell:(WCCardHolderCell *)cell
{
    do
    {
        if([self.delegate respondsToSelector:@selector(mapViewController:didSelectCardID:)]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        NSIndexPath *indexPath = [self.mapTableView indexPathForCell:cell];
        
        if(indexPath==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        WCMapContactRowModel *rowModel = (WCMapContactRowModel *)[self.mapViewSectionController rowModelAtIndexPath:indexPath
                                                                                                       forSearching:NO];
        if(rowModel==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        [self.delegate mapViewController:self didSelectCardID:rowModel.object];
    }
    while (0);
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - WCMapPinAnnotationViewDatasource method

//================================================================================
//
//================================================================================
- (UIImage *)thumbnailHeadImageWithMapPinAnnotationView:(WCMapPinAnnotationView *)mapPinAnnotationView
{
    UIImage *image = nil;
    
    do
    {

        WCMapAnnotation *mapAnnotation = (WCMapAnnotation *)mapPinAnnotationView.annotation;
        
        if(mapAnnotation==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        NSString *key = [[mapAnnotation currentCardID] stringByAppendingString:[NSString stringWithFormat:@"%td",WC_IT_IDPhoto]];
        
        if(key==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        image = [[WCCacheModel defaultCache] thumbImageWithCardID:key];
        
        if(image!=nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        if(self.datasource==nil ||
           [self.datasource respondsToSelector:@selector(imageWithImageType:imageSubType:cardID:)]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        image = [self.datasource imageWithImageType:WC_IT_IDPhoto imageSubType:WC_IST_Thumbnail cardID:[mapAnnotation currentCardID]];

        //////////////////////////////////////////////////

        //要不到名片圖，改用預設圖
        if(image==nil)
        {
            //從 Cache 取預設圖
            image = [[WCCacheModel defaultCache] thumbImageWithCardID:ImageNameForWCMapViewControllerDefaultHead];
            
            // Cache 圖不存在
            if(image==nil)
            {
                //讀預設圖
                image = [UIImage imageWithName:ImageNameForWCMapViewControllerDefaultHead];
                
                if(image!=nil)
                {
                    //預設圖加入 Cache
                    [[WCCacheModel defaultCache] addThumbImage:image withCardID:ImageNameForWCMapViewControllerDefaultHead];
                }
            }
         
            //////////////////////////////////////////////////

            [[WCCacheModel defaultCache] removeThumbImageWithCardID:key];
        }
        else
        {
            [[WCCacheModel defaultCache] addThumbImage:image withCardID:key];
        }
    }
    while (0);
    
    return image;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - RenewAnnotationWhenMapDidZoomOperationProtocol Method

//================================================================================
//
//================================================================================
- (CGPoint)pointFromCoordinate:(CLLocationCoordinate2D)coordinate
{
    return [self.mkMapView convertCoordinate:coordinate toPointToView:self.mkMapView];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Method

//================================================================================
//
//================================================================================
+ (void)checkAccessPermissionWithCompletion:(void (^)(BOOL authorized))completion
{
    if(completion!=nil)
    {
        completion([PPLocationController checkLocationAccess]);
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Map Center Move Method

//================================================================================
//
//================================================================================
- (void)decideMoveMapCenter
{
    BOOL moveToOurLocation = YES;
    
    if(self.focusCardID.length>0)
    {
        WCCardModel *cardModel = [self.cardModelDictionarys objectForKey:self.focusCardID];
        
        if(cardModel!=nil)
        {
            CLLocation *curLocation = [PPLocationController copyLocationFromString:cardModel.displayGPS];
            
            if(curLocation!=nil)
            {
                [self moveMapToLocation:curLocation];
                
                //////////////////////////////////////////////////
                
                //將列表移動過去
                NSIndexPath *indexPath = [self.mapViewSectionController indexPathForCardID:self.focusCardID forSearching:NO];
                
                if(indexPath!=nil)
                {
                    [self.mapTableView selectRowAtIndexPath:indexPath
                                                   animated:NO
                                             scrollPosition:UITableViewScrollPositionMiddle];
                }
                
                //////////////////////////////////////////////////
                
                moveToOurLocation = NO;
            }
            
            [curLocation release];
        }
    }
    
    //////////////////////////////////////////////////
    
    if(moveToOurLocation==YES)
    {
        [self moveMapToLocation:self.myLocation];
    }
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Method

//================================================================================
//
//================================================================================
- (void)moveMapToLocation:(CLLocation *)location
{
    if(location!=nil)
    {
        MKCoordinateRegion region = MKCoordinateRegionMakeWithDistance(location.coordinate, self.mapRegionSize.width, self.mapRegionSize.height);
        
        self.mkMapView.region = region;
    }
}


//================================================================================
//
//================================================================================
- (void)updateDistanceContactList
{
    [self updateDistanceWithRowModels];
    
    //////////////////////////////////////////////////

    __block typeof(self) blockself = self;
    
    __block LoadListBlockOperation *loadListOperaiton = [LoadListBlockOperation blockOperationWithBlock:^{
        
        do
        {
            if(loadListOperaiton.isCancelled==YES)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            if(blockself.mapTableView==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            dispatch_async(dispatch_get_main_queue(), ^{
                
                [blockself.mapTableView reloadData];
            });
        
        }
        while (0);
    }];
    
    //////////////////////////////////////////////////
    
    if(loadListOperaiton!=nil)
    {
        if([self.loadListOperationQueue operationCount]<=0)
        {
            [self.loadListOperationQueue addOperation:loadListOperaiton];
        }
    }
}


//================================================================================
//
//================================================================================
- (void)moveToDistanceContactListFocusCardID:(NSString *)cardID
{
    do
    {
        if(cardID==nil)
        {
            break;
        }

        //////////////////////////////////////////////////

        NSIndexPath *newIndexPath = [self.mapViewSectionController indexPathForCardID:cardID
                                                                         forSearching:NO];
        
        if(newIndexPath!=nil)
        {
            [self.mapTableView selectRowAtIndexPath:newIndexPath
                                           animated:NO
                                     scrollPosition:UITableViewScrollPositionMiddle];

        }
        
        //////////////////////////////////////////////////

        self.focusCardID = cardID;
    }
    while (0);
}


//================================================================================
//
//================================================================================
- (void)mapViewDidZoomWithinMapRect:(MKMapRect)rect withZoomScale:(double)zoomScale
{
    // 地圖區域改變，清除地圖cache
    if([self.memoryManageQueue operationCount]<=0)
    {
        __block typeof(self) blockself = self;
        
        __block NSBlockOperation *blockOperation = [NSBlockOperation blockOperationWithBlock:^{
            
            if(blockOperation.isCancelled==NO)
            {
                dispatch_async(dispatch_get_main_queue(), ^{
                    [blockself applyMapViewMemoryHotFix];
                });
            }
        }];
        
        //////////////////////////////////////////////////

        [self.memoryManageQueue addOperation:blockOperation];
    }
    
    //////////////////////////////////////////////////

    if(self.defaultClusterPixel>0 &&
       fabs(self.lastScale-zoomScale)>0.01)
    {
        self.lastScale = zoomScale;

        //////////////////////////////////////////////////
        
        [self updateMapAnnotationViewWithRect:rect];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Log methods

//===============================================================================
//
//===============================================================================
- (void)logMessageWithFormat:(NSString *)format, ...
{
    if(WCMapViewController_EnagleLog==NO)
    {
        return ;
    }    
    
    va_list arguments;
    va_start(arguments, format);
    [self.logController logWithMask:PPLogControllerMask_Normal format:format arguments:arguments];
    va_end(arguments);
}

//==============================================================================
//
//==============================================================================
+ (void)logEnable:(BOOL)enable
{
    WCMapViewController_EnagleLog = enable;
}


//===============================================================================
//
//===============================================================================
+ (NSString *)logDirPath
{
    return [WCToolController baseStorePathWithDirName:WCMapViewController_LogDir isCreatDirPath:NO];
}

@end
