//
//  WCGroupView.m
//  ADAL
//
//  Created by sanhue on 2018/2/13.
//

#import "WCGroupView.h"
#import "WCGroupDefine.h"

#import "WCGroupTableViewCell.h"
#import "WCGroupModel+Tree.h"
#import "RATreeView_ClassExtension.h"
#import "RATreeView+Private.h"


#define WCGroupView_NoResultHeight 44

////////////////////////////////////////////////////////////////////////////////////////////////////
@interface WCGroupView() <RATreeViewDataSource, RATreeViewDelegate, WCGroupTableViewCellDelegate>

@property (nonatomic, retain, readwrite) RATreeView *treeView;
@property (nonatomic, retain) NSMutableArray *selectedItems;
@property (nonatomic, retain) UILabel *noResultLabel;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation WCGroupView





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - init/dealloc methods


//==============================================================================
//
//==============================================================================
- (instancetype)initWithFrame:(CGRect)frame
{
    self = [super initWithFrame:frame];
    if (self)
    {
        self.treeView = [[[RATreeView alloc] initWithFrame:self.bounds] autorelease];
        
        self.treeView.delegate = self;
        self.treeView.dataSource = self;
        self.treeView.separatorStyle = RATreeViewCellSeparatorStyleSingleLine;
        
        // !! ios9 以上ipad介面會把table的內容左右內縮，造成cell的layout異常，
        // 設定這個可以讓介面看起來跟ios9以前一樣
        if(@available(iOS 9.0, *))
        {
            [self.treeView setCellLayoutMarginsFollowReadableWidth:NO];
        }
        
        self.treeView.autoresizingMask = UIViewAutoresizingFlexibleWidth | UIViewAutoresizingFlexibleHeight;
        [self.treeView setEditing:NO];
        
        [self addSubview:self.treeView];
        

        self.treeView.treeFooterView = [[[UIView alloc] init] autorelease];
        [self.treeView.treeFooterView setBackgroundColor:[UIColor whiteColor]];
        
        //////////////////////////////////////////////////
        _noResultLabel = [[UILabel alloc] init];
        if (self.noResultLabel)
        {
//            [self.noResultLabel setTranslatesAutoresizingMaskIntoConstraints:NO];
            self.noResultLabel.font = [UIFont boldSystemFontOfSize:16];
            self.noResultLabel.textColor = [UIColor grayColor];
            self.noResultLabel.backgroundColor = [UIColor clearColor];
            self.noResultLabel.textAlignment = NSTextAlignmentCenter;
            [self addSubview:self.noResultLabel];
        }
        
        //////////////////////////////////////////////////
        self.allowMultiSelection = YES;
        self.exclusiveWithOther = YES;
        self.enablePinButton = YES;
        self.selectedItems = [NSMutableArray array];
    }
    return self;
}


//==============================================================================
//
//==============================================================================
- (void)dealloc
{
    self.bottomLayoutGuide = nil;
    
    [self.treeView removeFromSuperview];
    self.treeView = nil;
    
    [self.noResultLabel removeFromSuperview];
    self.noResultLabel = nil;
    
    self.noResultString = nil;
    self.selectedItems = nil;
    //////////////////////////////////////////////////
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - override


//==============================================================================
//
//==============================================================================
- (void)layoutSubviews
{
    [super layoutSubviews];
    //////////////////////////////////////////////////
    if(CGRectEqualToRect(self.bounds, CGRectZero)==NO)
    {
        //////////////////////////////////////////////////
        CGRect treeViewBounds = self.bounds;
        treeViewBounds.size.height -= self.bottomLayoutGuide.length;
        [self.treeView setFrame:treeViewBounds];
    }
    //////////////////////////////////////////////////
    
    [self adjustNoResultLabel];

}







////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - private



//==============================================================================
//
//==============================================================================
- (WCGroupTableViewCellStyleMask)groupTableViewCellStyleMaskFromStyle:(WCGroupViewStyle)style
{
    switch (style)
    {
        case WCGroupViewStyle_NameWithCount:
        {
            return WCGroupTableViewCellStyleMask_CardCount;
        }
        case WCGroupViewStyle_NameWithPin:
        {
            return WCGroupTableViewCellStyleMask_PinButton;
        }
        case WCGroupViewStyle_NameWithCountAndPin:
        {
            return WCGroupTableViewCellStyleMask_CardCount|WCGroupTableViewCellStyleMask_PinButton;
        }
        case WCGroupViewStyle_NameWithCheckBox:
        {
            return WCGroupTableViewCellStyleMask_CheckBox;
        }
            
        default:
        case WCGroupViewStyle_Name:
        {
            return WCGroupTableViewCellStyleMask_None;
        }
            
    }
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - proxy methods


//==============================================================================
//
//==============================================================================
- (void)setEditing:(BOOL)editing
{
    _editing = editing;
    
    [self.treeView setEditing:editing];
}


//==============================================================================
//
//==============================================================================
-(void)reloadData
{
    [self.treeView reloadData];
    
    if ([self.delegate respondsToSelector:@selector(didReloadDataWithGroupView:)])
    {
        [self.delegate didReloadDataWithGroupView:self];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private KVO methods


//==============================================================================
//
//==============================================================================
- (RATreeNodeCollectionController *)treeNodeCollectionController
{
    RATreeNodeCollectionController *controller = [self.treeView valueForKeyPath:@"_treeNodeCollectionController"];
    return controller;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private


//==============================================================================
//
//==============================================================================
- (void)setCell:(WCGroupTableViewCell *)cell checked:(BOOL)checked
{
    if (!checked)
    {
        cell.checkBoxView.status = PPCheckBoxViewStatus_Unselect;
    }
    else
    {
        cell.checkBoxView.status = PPCheckBoxViewStatus_Select;
    }
}


//==============================================================================
//
//==============================================================================
- (void)adjustNoResultLabel
{
    self.noResultLabel.frame = CGRectMake(0,
                                          0,
                                          self.bounds.size.width,
                                          WCGroupView_NoResultHeight);
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Public methods


//==============================================================================
//
//==============================================================================
- (void)setSectionHeaderHeight:(CGFloat)sectionHeaderHeight
{
    [self.treeView.tableView setSectionHeaderHeight:sectionHeaderHeight];
}

//==============================================================================
//
//==============================================================================
- (NSArray *)copySelectedItems
{
    return [self.selectedItems copy];
}


//==============================================================================
//
//==============================================================================
- (WCGroupTableViewCell *)cellForItem:(id)item
{
    return (WCGroupTableViewCell *)[self.treeView cellForItem:item];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark cell control API


//==============================================================================
//
//==============================================================================
- (void)activeCellForItem:(id)item
{
    if(item!=nil)
    {
        [self.treeView selectRowForItem:item animated:NO scrollPosition:RATreeViewScrollPositionNone];
        
        [self.selectedItems addObject:item];
    }
}


//==============================================================================
//
//==============================================================================
- (void)checkCellWithItem:(id)item
{
    // 已選取就不用再選
    if ([self.selectedItems containsObject:item])
    {
        return ;
    }
    
    if (item)
    {
        WCGroupTableViewCell *cell = [self cellForItem:item];
        [self setCell:cell checked:YES];
        [self didCheckCell:cell];
    }
}


//==============================================================================
//
//==============================================================================
- (void)setSrollEnabled:(BOOL)scrollEnabled
{
    [self.treeView.tableView setScrollEnabled:scrollEnabled];
}








////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark expand/collape control API


//==============================================================================
//
//==============================================================================
- (void)expendAllWithAnimated:(BOOL)animated
{
    // 如果有至少一個node
    RATreeNode *treeNode = [self.treeView treeNodeForIndexPath:[NSIndexPath indexPathForRow:0 inSection:0]];
    
    if(treeNode)
    {
        RATreeViewRowAnimation animation = RATreeViewRowAnimationNone;
        if (animated)
        {
            animation = RATreeViewRowAnimationAutomatic;
        }
        [self.treeView expandCellForTreeNode:treeNode expandChildren:YES withRowAnimation:animation];
    }
}


//==============================================================================
//
//==============================================================================
- (void)expendRootWithAnimated:(BOOL)animated
{
    // 如果有至少一個node
    RATreeNode *rootNode= [self.treeView treeNodeForIndexPath:[NSIndexPath indexPathForRow:0 inSection:0]];
    
    if(rootNode)
    {
        RATreeViewRowAnimation animation = RATreeViewRowAnimationNone;
        if (animated)
        {
            animation = RATreeViewRowAnimationAutomatic;
        }
        [self.treeView expandCellForTreeNode:rootNode expandChildren:NO withRowAnimation:animation];
    }
}


//==============================================================================
//
//==============================================================================
- (void)expendItem:(WCGroupModel*)item withAnimated:(BOOL)animated
{
    if (item)
    {
        if ([self.treeView isCellForItemExpanded:item]==NO)
        {
            RATreeViewRowAnimation animation = RATreeViewRowAnimationNone;
            if (animated)
            {
                animation = RATreeViewRowAnimationAutomatic;
            }
            [self.treeView expandRowForItem:item expandChildren:NO withRowAnimation:animation];
            WCGroupTableViewCell *cell = [self cellForItem:item];
            [cell setCellExpanded:YES];
        }
    }
}








////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - TreeView Delegate methods


//==============================================================================
//
//==============================================================================
- (CGFloat)treeView:(RATreeView *)treeView heightForRowForItem:(id)item
{
    if([self.dataSource respondsToSelector:@selector(groupView:heightForRowForItem:)])
    {
        return [self.dataSource groupView:self heightForRowForItem:item];
    }
    
    return 44;
}


//==============================================================================
//
//==============================================================================
- (BOOL)treeView:(RATreeView *)treeView canEditRowForItem:(id)item
{
    return self.editing;
}


//==============================================================================
// 如果點選cell不要展開，這邊要設為NO
//==============================================================================
-(BOOL)treeView:(RATreeView *)treeView shouldExpandRowForItem:(nonnull id)item
{
    return NO;
}


//==============================================================================
// 如果點選cell不要收起，這邊要設為NO
//==============================================================================
-(BOOL)treeView:(RATreeView *)treeView shouldCollapaseRowForItem:(nonnull id)item
{
    return NO;
}


//==============================================================================
//
//==============================================================================
- (void)treeView:(RATreeView *)treeView didSelectRowForItem:(id)item
{
    if (self.style==WCGroupViewStyle_NameWithCheckBox)
    {
        WCGroupTableViewCell *cell = [self cellForItem:item];
        WCGroupModel *dataObject = item;
        
        // 如果是已選取，要改為未選取
        [self setCell:cell checked:!dataObject.isChecked];
        [self didCheckCell:cell];

        [treeView deselectRowForItem:item animated:YES];
    }
    else
    {
        // 如果不是checkbox，都是單選
        [self.selectedItems removeAllObjects];
        [self.selectedItems addObject:item];
        
        if ([self.delegate respondsToSelector:@selector(groupView:didClickCell:)])
        {
            WCGroupTableViewCell *cell = [self cellForItem:item];
            [self.delegate groupView:self didClickCell:cell];
        }
    }
}
//
//
////==============================================================================
////
////==============================================================================
//- (void)treeView:(RATreeView *)treeView didDeselectRowForItem:(id)item
//{
//    NSLog(@"%s", __PRETTY_FUNCTION__);
//}


//==============================================================================
//
//==============================================================================
- (UITableViewCellEditingStyle)treeView:(RATreeView *)treeView editingStyleForRowForItem:(id)item
{
    return UITableViewCellEditingStyleDelete|UITableViewCellEditingStyleInsert;
}


//==============================================================================
//
//==============================================================================
- (void)treeView:(RATreeView *)treeView commitEditingStyle:(UITableViewCellEditingStyle)editingStyle forRowForItem:(id)item
{
    if (editingStyle != UITableViewCellEditingStyleDelete) {
        return;
    }
    
    if ([self.dataSource respondsToSelector:@selector(groupView:commitEditingStyle:forRowForItem:)])
    {
        [self.dataSource groupView:self commitEditingStyle:editingStyle forRowForItem:item];
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark TreeView Data Source


//==============================================================================
//
//==============================================================================
- (UITableViewCell *)treeView:(RATreeView *)treeView cellForItem:(id)item
{
    WCGroupModel *dataObject = item;
    
    NSInteger level = [self.treeView levelForCellForItem:item];
    NSInteger numberOfChildren = [dataObject.subGroups count];
    BOOL expanded = [self.treeView isCellForItemExpanded:item];
    
    WCGroupTableViewCell *cell = [self.treeView dequeueReusableCellWithIdentifier:NSStringFromClass([WCGroupTableViewCell class])];
    
    if (cell==nil)
    {
        cell = [[[WCGroupTableViewCell alloc] initCustomStyleWithReuseIdentifier:NSStringFromClass([WCGroupTableViewCell class])] autorelease];
    }
    
    cell.selectionStyle = UITableViewCellSelectionStyleBlue;
    cell.delegate = self;
    cell.cellStyleMask = [self groupTableViewCellStyleMaskFromStyle:self.style];
    cell.reserveSpaceWhenExpandButtonHidden = self.reserveSpaceWhenExpandButtonHidden;
    cell.enableLongPressCell = self.enableLongPressCell;
    
    [cell setCellExpanded:expanded];
    [cell setExpandButtonHidden:(numberOfChildren==0)];
    
    if (self.style==WCGroupViewStyle_NameWithCountAndPin||
        self.style==WCGroupViewStyle_NameWithPin)
    {
        [cell setPinMark:(dataObject.pinnedOrder!=0)];
    }
    
    if (self.style==WCGroupViewStyle_NameWithCheckBox)
    {
        [cell setCheckBoxViewStatus:dataObject.isChecked];
    }
    
    cell.level = level;
    cell.nameLabel.text = dataObject.name;
    [cell setCardCount:dataObject.cardCount];
    
    [cell.pinMarkButton setEnabled:self.enablePinButton];
    
    //////////////////////////////////////////////////
    // 通知上層，看有沒有需要其他客制動作
    if ([self.delegate respondsToSelector:@selector(groupView:didLoadCell:dataObject:)])
    {
        [self.delegate groupView:self didLoadCell:cell dataObject:dataObject];
    }
    return cell;
}


//==============================================================================
//
//==============================================================================
- (NSInteger)treeView:(RATreeView *)treeView numberOfChildrenOfItem:(id)item
{
    NSInteger numberOfChildren = 0;
    
    if([self.dataSource respondsToSelector:@selector(groupView:numberOfChildrenOfItem:)])
    {
        numberOfChildren = [self.dataSource groupView:self numberOfChildrenOfItem:item];
        /// 如果第一層就是0，那就是沒有資料
        if (item==nil)
        {
            // TODO: 顯示沒有結果
            if ([self.noResultString length]>0)
            {
                self.noResultLabel.text = self.noResultString;
                if (numberOfChildren==0)
                {
                    [self.noResultLabel setHidden:NO];
                }
                else
                {
                    [self.noResultLabel setHidden:YES];
                }
            }
        }
    }
    
    return numberOfChildren;
}


//==============================================================================
//
//==============================================================================
- (id)treeView:(RATreeView *)treeView child:(NSInteger)index ofItem:(id)item
{
    if([self.dataSource respondsToSelector:@selector(groupView:child:ofItem:)])
    {
        WCGroupModel *dataObject =  [self.dataSource groupView:self child:index ofItem:item];
        
        if(self.style==WCGroupViewStyle_NameWithCheckBox)
        {
            if(dataObject.isChecked)
            {
                [self.selectedItems addObject:dataObject];
            }
            else
            {
                [self.selectedItems removeObject:dataObject];
            }
        }
        return dataObject;
    }
    return nil;
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - WCGroupTableViewCellDelegate



//==============================================================================
//
//==============================================================================
- (void)didCheckCell:(WCGroupTableViewCell *)cell
{
    WCGroupModel *dataObject = [self.treeView itemForCell:cell];
    
    //單選時，不能uncheck自已
    if (self.allowMultiSelection==NO)
    {
        if ([self.selectedItems containsObject:dataObject]==YES)
        {
            if(cell.checkBoxView.status==PPCheckBoxViewStatus_Unselect)
            {
                cell.checkBoxView.status = PPCheckBoxViewStatus_Select;
            }
            else
            {
                cell.checkBoxView.status = PPCheckBoxViewStatus_Unselect;
            }
            return ;
        }
    }
    
    //////////////////////////////////////////////////
    dataObject.isChecked = !dataObject.isChecked;
    
    if(dataObject.isChecked)
    {
        [self activeCellForItem:dataObject];
        
        if ([self.delegate respondsToSelector:@selector(groupView:didClickCell:)])
        {
            WCGroupTableViewCell *cell = [self cellForItem:dataObject];
            [self.delegate groupView:self didClickCell:cell];
        }
    }
    else
    {
        [self.selectedItems removeObject:dataObject];
    }
    
    //////////////////////////////////////////////////
    if(self.style==WCGroupViewStyle_NameWithCheckBox)
    {
        if (self.exclusiveWithOther==NO)
        {
            return;
        }
        
        NSMutableArray *removedGroups = [NSMutableArray array];
        if (dataObject.ID==WC_GID_Unfiled)
        {
            // 如果點選到未分類，清除其他選取
            for (WCGroupModel *groupModel in self.selectedItems)
            {
                if (groupModel.ID!=WC_GID_Unfiled)
                {
                    groupModel.isChecked = NO;
                    
                    WCGroupTableViewCell *cell = [self cellForItem:groupModel];
                    [self setCell:cell checked:NO];
                    
                    [removedGroups addObject:groupModel];
                }
            }
        }
        else
        {
            if (self.allowMultiSelection==NO)
            {
                // 不可多選，清除除了dataObject之外的所有類別
                for (WCGroupModel *groupModel in self.selectedItems)
                {
                    if (groupModel!=dataObject)
                    {
                        groupModel.isChecked = NO;
                        
                        WCGroupTableViewCell *cell = [self cellForItem:groupModel];
                        [self setCell:cell checked:NO];
                        
                        [removedGroups addObject:groupModel];
                    }
                }
            }
            else
            {
                // 可多選，如果選到一般類別，清除未分類
                for (WCGroupModel *groupModel in self.selectedItems)
                {
                    if (groupModel.ID==WC_GID_Unfiled)
                    {
                        groupModel.isChecked = NO;
                        
                        WCGroupTableViewCell *cell = [self cellForItem:groupModel];
                        [self setCell:cell checked:NO];
                        
                        [removedGroups addObject:groupModel];
                    }
                }
            }
        }
        [self.selectedItems removeObjectsInArray:removedGroups];

        //多選時，uncheck後如果變空的，要check other
        
        if ([self.selectedItems count]==0)
        {
            if([self.dataSource respondsToSelector:@selector(groupView:child:ofItem:)])
            {
                WCGroupModel *dataObject =  [self.dataSource groupView:self child:0 ofItem:nil];
                WCGroupModel *unfiledObject = [dataObject groupInTreeWithID:WC_GID_Unfiled sourceID:0];
                
                if(unfiledObject)
                {
                    unfiledObject.isChecked = YES;
                    
                    WCGroupTableViewCell *cell = [self cellForItem:unfiledObject];
                    [self setCell:cell checked:YES];
                    
                    [self.selectedItems addObject:unfiledObject];
                }
            }
        }
    }
}


//==============================================================================
//
//==============================================================================
- (void)didClickExpandWithCell:(WCGroupTableViewCell *)cell
{
    WCGroupModel *dataObject = [self.treeView itemForCell:cell];
    BOOL expanded = [self.treeView isCellForItemExpanded:dataObject];
    if(expanded)
    {
        [self.treeView collapseRowForItem:dataObject];
        //////////////////////////////////////////////////
        if ([self.delegate respondsToSelector:@selector(groupView:didCollapeCell:)])
        {
            [self.delegate groupView:self didCollapeCell:cell];
        }
    }
    else
    {
        [self.treeView expandRowForItem:dataObject];
        //////////////////////////////////////////////////
        if ([self.delegate respondsToSelector:@selector(groupView:didExpandCell:)])
        {
            [self.delegate groupView:self didExpandCell:cell];
        }
    }

    //////////////////////////////////////////////////
    if(self.activeWhenExpandCell &&
       self.style!=WCGroupViewStyle_NameWithCheckBox)
    {
        if ([self.selectedItems containsObject:dataObject]==NO)
        {
            [self.selectedItems removeAllObjects];
            [self.selectedItems addObject:dataObject];
            
            for (WCGroupModel *groupModel in self.selectedItems)
            {
                [self.treeView selectRowForItem:groupModel animated:NO scrollPosition:RATreeViewScrollPositionNone];
            }
        }

    }
}


//==============================================================================
//
//==============================================================================
- (void)cell:(WCGroupTableViewCell *)cell isPinned:(BOOL)isPinned;
{
    if ([self.delegate respondsToSelector:@selector(groupView:cell:isPinned:)])
    {
        [self.delegate groupView:self cell:cell isPinned:isPinned];
    }
}

@end
