//
//  CategoryTableViewCell.m
//  WorldNote
//
//  Created by Andrew.Lai on 14/3/20.
//  Copyright (c) 2014年 Eddie. All rights reserved.
//

#import "WCGroupTableViewCell.h"
#import "PPButton+Factory.h"
#import "WCAppearanceDefine.h"



#define WCGroupTableViewCellDefaultEdgeInsets UIEdgeInsetsMake(0, 15, 0, 15)
#define WCGroupTableViewCellButtonEdgeInsets UIEdgeInsetsMake(5, 5, 5, 5)

#define WCGroupTableViewCellCheckBoxWidth 36
#define WCGroupTableViewCellExpandButtonWidth 36
#define WCGroupTableViewCellPinButtonWidth 36
#define WCGroupTableViewCellIndent 18
#define WCGroupTableViewCellDefaultMargin 8

#define WCGroupTableViewCellNameFont 18
#define WCGroupTableViewCellCountFont 14

typedef NS_ENUM(NSInteger, WCGroupTableViewCellTag)
{
    WCGroupTableViewCellTag_None,
    WCGroupTableViewCellTag_Check,
    WCGroupTableViewCellTag_Expanded,
    WCGroupTableViewCellTag_Pinned,
};
////////////////////////////////////////////////////////////////////////////////////////////////////

@interface WCGroupTableViewCell() <PPButtonDelegate>

@property (nonatomic, retain, readwrite) UILabel *nameLabel;
@property (nonatomic, retain, readwrite) UILabel *cardCountLabel;
@property (nonatomic, retain, readwrite) PPButton *pinMarkButton;
@property (nonatomic, retain, readwrite) PPButton *expandButton;
@property (nonatomic, retain, readwrite) PPCheckBoxView *checkBoxView;

@property (nonatomic, retain) NSArray *layoutConstraints;
@property (nonatomic, assign) BOOL isExpanded;
@property (nonatomic, assign) BOOL isPinned;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Implementation CategoryTableViewCell

@implementation WCGroupTableViewCell





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - init/dealloc methods


//==============================================================================
//
//==============================================================================
- (instancetype)initCustomStyleWithReuseIdentifier:(NSString *)reuseIdentifier
{
    self = [super initWithStyle:UITableViewCellStyleValue1 reuseIdentifier:reuseIdentifier];
    if (self)
    {
        self.textLabel.hidden = YES;
        self.detailTextLabel.hidden = YES;
        
        self.contentView.frame = CGRectMake(0, 0, self.frame.size.width, self.frame.size.height);
        
        self.edgeInsets = WCGroupTableViewCellDefaultEdgeInsets;
        
        self.selectedBackgroundView = [[[UIView alloc] init] autorelease];
        self.selectedBackgroundView.backgroundColor = WCAppearanceDefine_CellHighlightedBGColor;
        
        self.cellStyleMask = WCGroupTableViewCellStyleMask_None;
        //////////////////////////////////////////////////
        _checkBoxView = [[PPCheckBoxView alloc] initWithFrame:CGRectZero];
        
        [self.checkBoxView setTranslatesAutoresizingMaskIntoConstraints:NO];
        self.checkBoxView.imageEdgeInsets = UIEdgeInsetsMake(0, 6, 0, 6);
        self.checkBoxView.status = PPCheckBoxViewStatus_Unselect;
        self.checkBoxView.checkBoxImageSelected = [UIImage imageWithName:@"WCGroupTableViewCell_Checked"];
        self.checkBoxView.checkBoxImageUnSelected = [UIImage imageWithName:@"WCGroupTableViewCell_Uncheck"];
        self.checkBoxView.hidden = YES;
        [self.contentView addSubview:self.checkBoxView];

        //////////////////////////////////////////////////
        self.pinMarkButton = [PPButton ppButtonWithIconImageName:nil
                                                             tag:WCGroupTableViewCellTag_Pinned
                                                        delegate:self
                                           normalBackgroundColor:nil
                                      highlightedBackgroundColor:nil
                                                 imageEdgeInsets:WCGroupTableViewCellButtonEdgeInsets];
        [self.pinMarkButton setTranslatesAutoresizingMaskIntoConstraints:NO];
        self.pinMarkButton.hidden = YES;
        [self.contentView addSubview:self.pinMarkButton];
        
        //////////////////////////////////////////////////
        self.expandButton = [PPButton ppButtonWithIconImageName:nil
                                                            tag:WCGroupTableViewCellTag_Expanded
                                                       delegate:self
                                          normalBackgroundColor:nil
                                     highlightedBackgroundColor:nil
                                                imageEdgeInsets:WCGroupTableViewCellButtonEdgeInsets];
        [self.expandButton setTranslatesAutoresizingMaskIntoConstraints:NO];
        [self.contentView addSubview:self.expandButton];

        [self setCellExpanded:NO];
        
        //////////////////////////////////////////////////
        self.nameLabel = [[[UILabel alloc] initWithFrame:CGRectZero] autorelease];
        [self.nameLabel setTranslatesAutoresizingMaskIntoConstraints:NO];
        self.nameLabel.font = [UIFont systemFontOfSize:WCGroupTableViewCellNameFont];
        self.nameLabel.textAlignment = NSTextAlignmentLeft;
        self.nameLabel.textColor = WCAppearanceDefine_CellTitleColor;

        [self.contentView addSubview:self.nameLabel];

        //////////////////////////////////////////////////
        self.cardCountLabel = [[[UILabel alloc] initWithFrame:CGRectZero] autorelease];
        [self.cardCountLabel setTranslatesAutoresizingMaskIntoConstraints:NO];
        self.cardCountLabel.font = [UIFont systemFontOfSize:WCGroupTableViewCellCountFont];
        self.cardCountLabel.textAlignment = NSTextAlignmentRight;
        self.cardCountLabel.textColor = WCAppearanceDefine_CellTitleColor;
        
        [self.contentView addSubview:self.cardCountLabel];

        //////////////////////////////////////////////////
        [self addCheckBoxGesture];
//        [self addLongPressGesture];
        
        [self resetLayoutConstraints];
        //////////////////////////////////////////////////
        // add background color for debug
//        [self.pinMarkButton setBackgroundColor:[UIColor redColor]];
//        [self.checkBoxView setBackgroundColor:[UIColor blueColor]];
//        [self.nameLabel setBackgroundColor:[UIColor yellowColor]];
//        [self.cardCountLabel setBackgroundColor:[UIColor purpleColor]];
//        [self.expandButton setBackgroundColor:[UIColor orangeColor]];
    }
    return self;
}


//==============================================================================
//
//==============================================================================
- (void)dealloc
{
    [self removeLayoutConstraints];
    [self removeCheckBoxGesture];
    [self removeLongPressGesture];
    
    [self.nameLabel removeFromSuperview];
    self.nameLabel = nil;
    
    [self.cardCountLabel removeFromSuperview];
    self.cardCountLabel = nil;
    
    [self.expandButton removeFromSuperview];
    self.expandButton = nil;
    
    [self.checkBoxView removeFromSuperview];
    self.checkBoxView = nil;
    
    [self.pinMarkButton removeFromSuperview];
    self.pinMarkButton = nil;
    //////////////////////////////////////////////////
    [super dealloc];
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Laying out Subviews

//================================================================================
//
//================================================================================
- (void)layoutSubviews
{
    [super layoutSubviews];

    //////////////////////////////////////////////////
    [self resetLayoutConstraints];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - layout


//==============================================================================
//
//==============================================================================
- (void)resetLayoutConstraints
{
    if (self.checkBoxView==nil ||
        self.expandButton==nil ||
        self.nameLabel==nil ||
        self.cardCountLabel==nil ||
        self.pinMarkButton==nil)
    {
        return ;
    }
    
    [self removeLayoutConstraints];
    //////////////////////////////////////////////////
    
    //////////////////////////////////////////////////
    NSString * groupCount = [NSString stringWithFormat:@"(%@)",self.cardCountLabel.text];
    CGSize detailTextLabelSize = [groupCount sizeWithAttributes:@{NSFontAttributeName:self.cardCountLabel.font}];

    NSInteger indentLevel = self.level*WCGroupTableViewCellIndent;
    NSInteger rightMargin = self.edgeInsets.right;
    
    NSDictionary *views = @{@"checkBoxView":self.checkBoxView,
                            @"expandButton":self.expandButton,
                            @"nameLabel":self.nameLabel,
                            @"cardCountLabel":self.cardCountLabel,
                            @"pinMarkButton":self.pinMarkButton
                            };
    
    NSDictionary *metrics = @{@"TopMargin":@(self.edgeInsets.top),
                              @"BottomMargin":@(self.edgeInsets.bottom),
                              @"RightMargin":@(self.edgeInsets.right),
                              @"detailTextLabelWidth":@(detailTextLabelSize.width),
                              @"WCGroupTableViewCellPinButtonWidth":@(WCGroupTableViewCellPinButtonWidth),
                              @"WCGroupTableViewCellCheckBoxWidth":@(WCGroupTableViewCellCheckBoxWidth),
                              @"WCGroupTableViewCellExpandButtonWidth":@(WCGroupTableViewCellExpandButtonWidth)
                              };
    
    NSMutableArray *constraints = [NSMutableArray array];

    self.pinMarkButton.hidden = ((self.cellStyleMask&WCGroupTableViewCellStyleMask_PinButton)!=WCGroupTableViewCellStyleMask_PinButton);
    self.checkBoxView.hidden = ((self.cellStyleMask&WCGroupTableViewCellStyleMask_CheckBox)!=WCGroupTableViewCellStyleMask_CheckBox);
    self.cardCountLabel.hidden = ((self.cellStyleMask&WCGroupTableViewCellStyleMask_CardCount)!=WCGroupTableViewCellStyleMask_CardCount);;

    NSMutableString *horzFormat = [NSMutableString string];
    
    // pin與check
    if ((self.cellStyleMask&WCGroupTableViewCellStyleMask_PinButton)==WCGroupTableViewCellStyleMask_PinButton)
    {
        [horzFormat appendString:@"[pinMarkButton(WCGroupTableViewCellPinButtonWidth)]"];

        [constraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|-(TopMargin)-[pinMarkButton]-(BottomMargin)-|"
                                                                                 options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                 metrics:metrics
                                                                                   views:views]];
        
    }
    
    if ((self.cellStyleMask&WCGroupTableViewCellStyleMask_CheckBox)==WCGroupTableViewCellStyleMask_CheckBox)
    {
        // 如果check box是第一個，margin為self.edgeInsets.left，不然就是0
        if ([horzFormat length]==0)
        {
            [horzFormat appendFormat:@"-(%f)-", self.edgeInsets.left];
        }
        [horzFormat appendString:@"[checkBoxView(WCGroupTableViewCellCheckBoxWidth)]"];
        [constraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|-(TopMargin)-[checkBoxView]-(BottomMargin)-|"
                                                                                 options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                 metrics:metrics
                                                                                   views:views]];
        
    }

    // 名稱一直都會存在
    // 如果Name 是第一個，leftMargin==self.edgeInsets.left，不然就用預設間距
    NSInteger leftMargin = 0;
    if ([horzFormat length]==0)
    {
        leftMargin = self.edgeInsets.left;
    }
    [horzFormat appendFormat:@"-(%td)-", leftMargin+indentLevel];
    [horzFormat appendString:@"[nameLabel]"];
    [constraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|-(TopMargin)-[nameLabel]-(BottomMargin)-|"
                                                                             options:NSLayoutFormatDirectionLeadingToTrailing
                                                                             metrics:metrics
                                                                               views:views]];
    
    
    if ((self.cellStyleMask&WCGroupTableViewCellStyleMask_CardCount)==WCGroupTableViewCellStyleMask_CardCount)
    {
        [horzFormat appendString:@"[cardCountLabel(detailTextLabelWidth)]"];
        [constraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|-(TopMargin)-[cardCountLabel]-(BottomMargin)-|"
                                                                                 options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                 metrics:metrics
                                                                                   views:views]];
        
    }
    
    // 沒有隱藏，或隱藏時保留才要有expandButton
    if (self.expandButton.hidden==NO ||
        self.reserveSpaceWhenExpandButtonHidden==YES)
    {
        
        [horzFormat appendString:@"-[expandButton(WCGroupTableViewCellExpandButtonWidth)]"];
        [constraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|-(TopMargin)-[expandButton]-(BottomMargin)-|"
                                                                                 options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                 metrics:metrics
                                                                                   views:views]];
        rightMargin = 0;
    }
    
    [horzFormat appendFormat:@"-(%ld)-", (long)rightMargin];

    NSString *fullHorzFormat = [NSString stringWithFormat:@"H:|%@|", horzFormat];
    [constraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:fullHorzFormat
                                                                             options:NSLayoutFormatDirectionLeftToRight
                                                                             metrics:metrics
                                                                               views:views]];

    if ([constraints count])
    {
        self.layoutConstraints = [NSArray arrayWithArray:constraints];
        [self.contentView addConstraints:self.layoutConstraints];
        [self.contentView layoutIfNeeded];
    }
}


//==============================================================================
//
//==============================================================================
- (void)removeLayoutConstraints
{
    if (self.layoutConstraints)
    {
        [self.contentView removeConstraints:self.layoutConstraints];
        self.layoutConstraints = nil;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private check gesture


//==============================================================================
//
//==============================================================================
- (void)addCheckBoxGesture
{
    [self removeCheckBoxGesture];
    
    UITapGestureRecognizer *gestureRecognizer = [[UITapGestureRecognizer alloc] initWithTarget:self action:@selector(onClickCheckBox:)];
    [self.checkBoxView addGestureRecognizer:gestureRecognizer];
    [gestureRecognizer release];
}


//==============================================================================
//
//==============================================================================
- (void)removeCheckBoxGesture
{
    for (UIGestureRecognizer *gestureRecognizer in self.checkBoxView.gestureRecognizers)
    {
        [self.checkBoxView removeGestureRecognizer:gestureRecognizer];
    }
}


//==============================================================================
//
//==============================================================================
- (void)onClickCheckBox:(UIGestureRecognizer*)gestureRecognizer
{
    if (self.checkBoxView.status==PPCheckBoxViewStatus_Select)
    {
        self.checkBoxView.status = PPCheckBoxViewStatus_Unselect;
    }
    else if (self.checkBoxView.status==PPCheckBoxViewStatus_Unselect)
    {
        self.checkBoxView.status = PPCheckBoxViewStatus_Select;
    }
    else
    {
        self.checkBoxView.status = PPCheckBoxViewStatus_None;
    }
    
    if(gestureRecognizer.state==UIGestureRecognizerStateEnded)
    {
        if ([self.delegate respondsToSelector:@selector(didCheckCell:)])
        {
            [self.delegate didCheckCell:self];
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private long press gesture


//==============================================================================
//
//==============================================================================
- (void)addLongPressGesture
{
    [self removeLongPressGesture];
    
    UILongPressGestureRecognizer *gestureRecognizer = [[UILongPressGestureRecognizer alloc] initWithTarget:self action:@selector(onLongPressCell:)];
    [self.contentView addGestureRecognizer:gestureRecognizer];
    [gestureRecognizer release];
}


//==============================================================================
//
//==============================================================================
- (void)removeLongPressGesture
{
    for (UIGestureRecognizer *gestureRecognizer in self.contentView.gestureRecognizers)
    {
        if([gestureRecognizer isKindOfClass:[UILongPressGestureRecognizer class]])
        {
            [self.contentView removeGestureRecognizer:gestureRecognizer];
        }
    }
}


//==============================================================================
//
//==============================================================================
- (void)onLongPressCell:(UIGestureRecognizer*)gestureRecognizer
{
    if(self.enableLongPressCell==NO)
    {
        return ;
    }
    
    if(gestureRecognizer.state==UIGestureRecognizerStateBegan)
    {
        if ((self.cellStyleMask&WCGroupTableViewCellStyleMask_PinButton)==WCGroupTableViewCellStyleMask_PinButton)
        {
            [self setPinMark:!self.isPinned];
            
            if ([self.delegate respondsToSelector:@selector(cell:isPinned:)])
            {
                [self.delegate cell:self isPinned:self.isPinned];
            }
        }
    }
        
}







////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Method

//================================================================================
// 避免內外格式不統一的錯誤,讓使用者只需傳入數字
//================================================================================
- (void)setCardCount:(NSUInteger)cardCount
{
    if (cardCount==-1)
    {
        self.cardCountLabel.text = nil;
    }
    else
    {
        self.cardCountLabel.text = [NSString stringWithFormat:@"(%lu)", (long)cardCount];
    }
}


//==============================================================================
//
//==============================================================================
- (void)setPinMark:(BOOL)isMarked
{
    if (isMarked)
    {
        [self.pinMarkButton setImageWithName:@"Pin"];
    }
    else
    {
        [self.pinMarkButton setImageWithName:@"UnPin"];
    }

    self.isPinned = isMarked;
}


//================================================================================
//
//================================================================================
- (void)setCheckBoxViewStatus:(BOOL)isSelected
{
    if (isSelected==YES)
    {
        self.checkBoxView.status = PPCheckBoxViewStatus_Select;
    }
    else
    {
        self.checkBoxView.status = PPCheckBoxViewStatus_Unselect;
    }
}


//================================================================================
//
//================================================================================
- (void)setCellExpanded:(BOOL)expanded
{
    self.isExpanded = expanded;
    
    //////////////////////////////////////////////////
    if (!self.isExpanded)
    {
        [self.expandButton setImageWithName:@"Expand"];
    }
    else
    {
        [self.expandButton setImageWithName:@"Collapse"];
    }
}


//================================================================================
//
//================================================================================
- (void)setExpandButtonHidden:(BOOL)hidden
{
    self.expandButton.hidden = hidden;
}



////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPButtonDelegate


//==============================================================================
//
//==============================================================================
- (void)ppButton:(PPButton *)ppButton controlEvent:(UIControlEvents)controlEvent
{
    switch (ppButton.tag) {
        case WCGroupTableViewCellTag_Expanded:
        {
            [self setCellExpanded:!self.isExpanded];
            
            if ([self.delegate respondsToSelector:@selector(didClickExpandWithCell:)])
            {
                [self.delegate didClickExpandWithCell:self];
            }
            break;
        }
        
        case WCGroupTableViewCellTag_Pinned:
        {
            [self setPinMark:!self.isPinned];
            
            if ([self.delegate respondsToSelector:@selector(cell:isPinned:)])
            {
                [self.delegate cell:self isPinned:self.isPinned];
            }
            break;
        }
        default:
            break;
    }
}



@end
