//
//  WCExportFlowController.m
//  Example
//
//  Created by Andrew on 2015/8/3.
//  Copyright (c) 2015年 Eddie. All rights reserved.
//

#import "WCExportFlowController.h"

// Define
#import "WCCloudFileDefine.h"
#import "UIColor+HexString.h"
#import "UIApplication+Idle.h"
#import "WCExportFlowController+SettingsKey.h"

// View
#import "PPBusyView.h"
#import "PPAlertView.h"

// Controller
#import "EFCExportViewController.h"
#import "PPNavigationController.h"
#import "WCToastController.h"
#import "PPSettingsController.h"
#import "WCToastController.h"

// View/Flow Controller
#import "WCABItemSelectFlowController.h"

// ToolController
#import "WCDisplayNameController.h"
#import "WCToolController.h"
#import "PPBackgroundTaskController.h"
#import "PPCloudController.h"
#import "PPNetworkReachabilityController.h"

// DataController
#import "WCCSVDataController.h"
#import "WCXFDataController.h"
#import "WCVCFDataController.h"
#import "WCXLSXDataController.h"

#if defined (PRODUCTLINE_WCT)
#import "WCTMatomoController.h"
#import "WCTServerContactExportFlowController.h"
#import "PPSettingsController.h"
#import "WCTSettingsKey.h"

#elif defined (PRODUCTLINE_WCSF)
#import "WCSFMatomoController.h"

#endif
////////////////////////////////////////////////////////////////////////////////////////////////////

static NSString *const WCExportFlowController_ContactExportRootFolder  = @"ContactExportFolder";
static NSString *const WCExportFlowController_AppStorageSpaceDefaultFolder = WCCloudFileDefine_ExchangeDirPath;

static double WCExportFlowController_MinimumSystemMemoryRequestBytes = 1024*1024;

typedef NS_ENUM(NSInteger, WCExportFlowControllerAlertTag)
{
    WCExportFlowControllerAlertTag_Default = 0,
    WCExportFlowControllerAlertTag_Salesforce,
};
////////////////////////////////////////////////////////////////////////////////////////////////////

typedef NS_OPTIONS(NSInteger, WCExportFlowController_Error)
{
    WCExportFlowController_Error_Unknown = 0,
    WCExportFlowController_Error_NoCardToWrite
};

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - WCExportFlowController Extension

@interface WCExportFlowController () <EFCExportViewControllerDelegate,
                                      WCABItemSelectFlowControllerDelegate,
                                      WCCSVDataControllerDelegate,
                                      WCXFDataControllerDelegate,
                                      WCXLSXDataControllerDelegate,
                                      PPCloudDelegate_LoadAccountInfo,
                                      PPCloudDelegate_CreateFolder,
                                      PPCloudDelegate_UploadFile,
                                      PPStorageSpaceSelectViewControllerDelegate,
                                      UIAlertViewDelegate>

@property (nonatomic, assign) BOOL                                    completeSuccess; //是否所有匯出的檔案創建成功
@property (nonatomic, assign) BOOL                                    releaseCloudController;
@property (nonatomic, retain) NSArray                                *sourceTempPathArray; //進行多檔上傳時，必須控管數量與進度
@property (nonatomic, retain) NSError                                *exportFlowError;
@property (nonatomic, assign) PPStorageSpaceSelectControllerSpaceType spaceTypeSelected;
@property (nonatomic, assign) WCDisplayNameOrder                      displayEastNameOrder;
@property (nonatomic, assign) WCDisplayNameOrder                      displayWestNameOrder;

@property (nonatomic, retain) PPBusyView                             *busyView;

@property (nonatomic, retain) EFCExportViewController                *exportViewController;
@property (nonatomic, retain) PPStorageSpaceSelectViewController     *storageSpaceSelectViewController;
@property (nonatomic, retain) UIViewController                       *dismissGroupViewController;

@property (atomic, retain) NSNumber *requestResponse;
@property (atomic, retain) NSError *requestResponseError;
@property (nonatomic, assign) long long spaceTotalBytes;
@property (nonatomic, assign) long long spaceFreeBytes;

@end





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - WCExportFlowController Implementation

@implementation WCExportFlowController





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Deallocating Objects

//================================================================================
//
//================================================================================
- (id)init
{
    if ((self=[super init]))
    {
        if ([PPCloudController sharedCloudController]==nil)
        {
            _releaseCloudController = YES;
            _fileTypeOrderSource = nil;
            
            //////////////////////////////////////////////////
            
            PPCloudController *cloudController = [[PPCloudController alloc] init];
            if (cloudController!=nil)
            {
                [PPCloudController setSharedCloudController:cloudController];
                
                [cloudController release];
            }
        }
        
        //////////////////////////////////////////////////
        
        _exportFlowError      = nil;
        _defaultFilePath      = [WCExportFlowController_AppStorageSpaceDefaultFolder retain];
        _displayEastNameOrder = WCDisplayNameOrder_None;
        _displayWestNameOrder = WCDisplayNameOrder_None;
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    _delegate = nil;
    
    [_sourceTempPathArray release];
    _sourceTempPathArray = nil;
    
    [_contactIdentifierArray release];
    _contactIdentifierArray = nil;
    
    _exportViewController.delegate = nil;
    [_exportViewController release];
    _exportViewController = nil;
    
    [_dismissGroupViewController release];
    _dismissGroupViewController = nil;
    
    [_exportFlowError release];
    _exportFlowError = nil;
    
    [_defaultFilePath release];
    _defaultFilePath = nil;
    
    [_fileTypeOrderSource release];
    _fileTypeOrderSource = nil;
    
    self.storageSpaceSelectViewController = nil;
    //////////////////////////////////////////////////
    
    if (_releaseCloudController==YES)
    {
        [PPCloudController setSharedCloudController:nil];
    }
    
    [_busyView removeFromSuperview];
    [_busyView release];
    _busyView = nil;
    
    [_requestResponse release];
    _requestResponse = nil;
    
    [_requestResponseError release];
    _requestResponseError = nil;
    
    [_syncAccountName release];
    _syncAccountName = nil;
    
    [_exchangeName release];
    _exchangeName = nil;
    //////////////////////////////////////////////////
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Method

//================================================================================
//
//================================================================================
- (unsigned long long int)folderSize:(NSString *)folderPath
{
    NSArray *filesArray = [[NSFileManager defaultManager] subpathsOfDirectoryAtPath:folderPath error:nil];
    NSEnumerator *filesEnumerator = [filesArray objectEnumerator];
    NSString *fileName;
    unsigned long long int fileSize = 0;
    
    while (fileName = [filesEnumerator nextObject])
    {
        NSDictionary *fileDictionary = [[NSFileManager defaultManager] attributesOfItemAtPath:[folderPath stringByAppendingPathComponent:fileName] error:nil];
        fileSize += [fileDictionary fileSize];
    }
    
    return fileSize;
}


//================================================================================
//
//================================================================================
- (void)requestSpaceInfo
{
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
        
        blockSelf.requestResponse = nil;
        blockSelf.spaceTotalBytes = 0;
        blockSelf.spaceFreeBytes = 0;
        
        //////////////////////////////////////////////////
        
        switch (blockSelf.spaceTypeSelected)
        {
            case PPStorageSpaceSelectControllerSpaceType_iTune:
            {
                NSArray *paths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
                NSDictionary *dictionary = [[NSFileManager defaultManager] attributesOfFileSystemForPath:[paths lastObject] error:nil];
                
                blockSelf.spaceTotalBytes = [[dictionary objectForKey:NSFileSystemSize] doubleValue];
                blockSelf.spaceFreeBytes = [[dictionary objectForKey:NSFileSystemFreeSize] doubleValue];
                
                blockSelf.requestResponse = @(YES);
                
                break;
            }
            case PPStorageSpaceSelectControllerSpaceType_iCloud:
            {
                // iCloud無法抓取
                blockSelf.requestResponse = @(YES);
                break;
            }
            case PPStorageSpaceSelectControllerSpaceType_Dropbox:
            {
                [PPCloudController loadAccountInfoWithDelegate:blockSelf
                                                      userInfo:nil
                                                    cloudClass:[PPCloud_Dropbox class]];
                break;
            }
            case PPStorageSpaceSelectControllerSpaceType_GoogleDrive:
            {
                [PPCloudController loadAccountInfoWithDelegate:blockSelf
                                                      userInfo:nil
                                                    cloudClass:[PPCloud_GoogleDrive class]];
                break;
            }
            case PPStorageSpaceSelectControllerSpaceType_OneDrive:
            {
                [PPCloudController loadAccountInfoWithDelegate:blockSelf
                                                      userInfo:nil
                                                    cloudClass:[PPCloud_OneDrive class]];
                break;
            }
            case PPStorageSpaceSelectControllerSpaceType_Huawei:
            {
                // Huawei無法抓取
                blockSelf.requestResponse = @(YES);
                break;
            }
            default:
            {
                blockSelf.requestResponse = @(NO);
                break;
            }
        }
        
        //////////////////////////////////////////////////
        
        while (blockSelf.requestResponse == nil)
        {
            [NSThread sleepForTimeInterval:0.1];
        }
    });
}


//================================================================================
//
//================================================================================
- (void)showBusyView:(BOOL)show
{
    if(show==YES)
    {
        if(self.busyView==nil)
        {
            _busyView = [[PPBusyView alloc] initWithSuperView:[UIApplication sharedApplication].keyWindow];
        }
    }
    else
    {
        [self.busyView removeFromSuperview];
        self.busyView = nil;
    }
}


//================================================================================
//
//================================================================================
- (NSString *)singleFileCompleteNameWithFormatString:(NSString *)formatString
{
    NSString *deviceName = [[UIDevice currentDevice] name];
    NSString *dateTime   = nil;
    
    //!! 壓縮檔名中不能有空白
    deviceName = [deviceName stringByReplacingOccurrencesOfString:@" " withString:@"_"];
    
    NSDateFormatter *dateFormatter = [[NSDateFormatter alloc] init];
    if(dateFormatter!=nil)
    {
        [dateFormatter setDateFormat:WCCloudFileDefine_DateTimeFormat];
        
        dateTime = [dateFormatter stringFromDate:[NSDate date]];
        [dateFormatter release];
    }
    
    return [NSString stringWithFormat:WCCloudFileDefine_MultiContactsFileNameFormat, deviceName, dateTime, formatString];
}


//================================================================================
// 因為CSV有兩種樣式，需於時間後面添加_Gmail或_Outlook
//================================================================================
- (NSString *)singleFileCompleteNameForFileTypeCSV:(WCFileTypeSelectView_ContactFileType)fileType
{
    NSString *deviceName = [[UIDevice currentDevice] name];
    NSString *dateTime   = nil;
    
    //!! 壓縮檔名中不能有空白
    deviceName = [deviceName stringByReplacingOccurrencesOfString:@" " withString:@"_"];
    
    NSDateFormatter *dateFormatter = [[NSDateFormatter alloc] init];
    if(dateFormatter!=nil)
    {
        [dateFormatter setDateFormat:WCCloudFileDefine_DateTimeFormat];
        
        dateTime = [dateFormatter stringFromDate:[NSDate date]];
        [dateFormatter release];
    }
    
    switch (fileType)
    {
        case WCFileTypeSelectView_ContactFileType_CSV_iWorks:
        {
            dateTime = [dateTime stringByAppendingString:@"_iWorks"];
            break;
        }
        case WCFileTypeSelectView_ContactFileType_CSV_Gmail:
        {
            dateTime = [dateTime stringByAppendingString:@"_Gmail"];
            break;
        }
        case WCFileTypeSelectView_ContactFileType_CSV_Outlook:
        {
            dateTime = [dateTime stringByAppendingString:@"_Outlook"];
            break;
        }
        default:
            break;

    }
    
    return [NSString stringWithFormat:WCCloudFileDefine_MultiContactsFileNameFormat, deviceName, dateTime, @"csv"];
}


//================================================================================
//
//================================================================================
- (Class)classOfCurrentCloudType
{
    Class activeClass = nil;
    
    //////////////////////////////////////////////////
    
    switch (self.spaceTypeSelected)
    {
        case PPStorageSpaceSelectControllerSpaceType_iCloud:
        {
            activeClass = [PPCloud_iCloud class];
            break;
        }
        case PPStorageSpaceSelectControllerSpaceType_Dropbox:
        {
            activeClass = [PPCloud_Dropbox class];
            break;
        }
        case PPStorageSpaceSelectControllerSpaceType_GoogleDrive:
        {
            activeClass = [PPCloud_GoogleDrive class];
            break;
        }
        case PPStorageSpaceSelectControllerSpaceType_OneDrive:
        {
            activeClass = [PPCloud_OneDrive class];
            break;
        }
//        case PPStorageSpaceSelectControllerSpaceType_Huawei:
//        {
//            activeClass = [PPCloud_Huawei class];
//            break;
//        }
        default:
            break;
    }
    
    return activeClass;
}


//================================================================================
// 判斷是否執行上傳雲端，若為iTunesShare則不進行動作並關閉Loading(下方回傳的BOOL為是否成功加入operation)
//================================================================================
- (void)uploadFilesWithPath:(NSArray *)filePathArray
{
    if (self.spaceTypeSelected!=PPStorageSpaceSelectControllerSpaceType_iTune)
    {
        Class curClass = [self classOfCurrentCloudType];
        
        //////////////////////////////////////////////////
        
        NSString *dstDirPath  = self.defaultFilePath;
        if (self.spaceTypeSelected==PPStorageSpaceSelectControllerSpaceType_Huawei)
        {
            NSMutableArray *components = [NSMutableArray arrayWithArray:[WCCloudFileDefine_ExchangeDirPath pathComponents]];
            
            [components removeObjectAtIndex:1];
            
            dstDirPath = [NSString pathWithComponents:components];
        }
        
        [PPCloudController createFolder:dstDirPath
                               delegate:self
                               userInfo:nil
                             cloudClass:curClass];
        
        //////////////////////////////////////////////////
        //背景處理
        
        self.sourceTempPathArray = filePathArray;
        [PPBackgroundTaskController addTaskWithKey:NSStringFromClass([self class]) terminate:nil];
        
        for (NSString *srcFilePath in filePathArray)
        {
            NSString *dstFileName = [srcFilePath lastPathComponent];
            
            //!! userInfo塞入srcFilePath避免上傳失敗無法刪檔
            [PPCloudController uploadFile:dstFileName
                                   toPath:dstDirPath
                                 fromPath:srcFilePath
                                 delegate:self
                                 userInfo:srcFilePath
                               cloudClass:curClass];
        }
        
        [PPBackgroundTaskController removeTaskWithKey:NSStringFromClass([self class])];
    }
    else
    {
        __block WCExportFlowController *blockSelf = self;
        dispatch_async(dispatch_get_main_queue(), ^{
            [PPBusyView postMessage:[NSString stringWithFormat:@"%@%.2f%@", WCEFC_MLS_Exporting, 100.00, @"%"]];
            [[UIApplication sharedApplication] enableIdle:YES];
            [blockSelf showBusyView:NO];
            
            //////////////////////////////////////////////////
            // 重新定義錯誤紀錄
            
            PPAlertView *alertView = [[PPAlertView alloc] initWithTitle:@""
                                                                message:WCEFC_MLS_ExportSuccess
                                                               delegate:blockSelf
                                                      cancelButtonTitle:WCEFC_MLS_OK
                                                      otherButtonTitles:nil];
            [alertView show];
            [alertView release];
        });
    }
}


//==============================================================================
//
//==============================================================================
- (NSString *)errorMessageWithError:(NSError *)error
{
    NSError *authError = [error findErrorWithConditionDict:@{@"org.openid.appauth.oauth_token":@[@(-10)],
                                                                      @"com.google.HTTPStatus":@[@(401)]
    }];
    
    if(authError)
    {
        [[PPCloudController sharedCloudController] unlinkWithDelegate:nil userInfo:nil cloudClass:[PPCloud_GoogleDrive class]];
        
        return [WCEFC_MLS_ExportFailed stringByAppendingFormat:@"\n(%@)", WCEFC_MLS_AuthenticationFail];
    }
    else
    {
        return WCEFC_MLS_ExportFailed;
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Method

//================================================================================
//
//================================================================================
- (BOOL)showFromSuperViewController:(UIViewController *)superViewController
             withCloudKeyDictionary:(NSDictionary *)cloudKeyDictionary
                     spaceTypeArray:(NSArray *)spaceTypeArray
{
    BOOL isSuccess = NO;
    
    do
    {
        if (cloudKeyDictionary==nil || spaceTypeArray==nil || superViewController==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        PPStorageSpaceSelectViewController *storageSpaceSelectViewController = [[PPStorageSpaceSelectViewController alloc] init];
        if(storageSpaceSelectViewController==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        storageSpaceSelectViewController.delegate = self;
        storageSpaceSelectViewController.controllerTitle = WCEFC_MLS_Export;
        storageSpaceSelectViewController.navigationBarButtonImageInsets = WCExportFlowControllerNavigationBarButtonImageEdgeInsets;
        storageSpaceSelectViewController.navigationViewButtonHighlightedBackgroundColor = [UIColor colorWithHexString:@"#2c538f"];
        storageSpaceSelectViewController.cloudKeyDictionary = cloudKeyDictionary;
        storageSpaceSelectViewController.spaceTypeSelectArray = spaceTypeArray;
        storageSpaceSelectViewController.exchangeServerName = self.exchangeName;
        
        self.storageSpaceSelectViewController = storageSpaceSelectViewController;
        //////////////////////////////////////////////////
        
        PPNavigationController *navigationController = [[PPNavigationController alloc] initWithRootViewController:storageSpaceSelectViewController];
        if(navigationController==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        //MARK: 客制橫式NavigationBar高度44

        [superViewController presentViewController:navigationController
                                          animated:YES
                                        completion:nil];
        
        [navigationController release];
        [storageSpaceSelectViewController release];
        
        //////////////////////////////////////////////////
        
        isSuccess = YES;
    }
    while (0);
    
    return isSuccess;
}


//================================================================================
//
//================================================================================
- (BOOL)exportAddressBookCardsFromSuperViewController:(UIViewController *)superViewController
{
    BOOL isSuccess = NO;
    
    do
    {
        if (superViewController==nil)
        {
            break;
        }
        
        [WCABItemSelectFlowController showGroupViewControllerWithABSelectFlowControllerMode:WCABItemSelectFlowControllerMode_ApToSystemContact
                                                             abSelectFlowControllerDelegate:self
                                                                            syncAccountName:self.syncAccountName
                                                                    fromSuperViewController:superViewController];
    } while (0);
    
    return isSuccess;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPStorageSpaceSelectViewControllerDelegate

//================================================================================
//
//================================================================================
- (BOOL)storageSpaceSelectViewControllerShowAlertWithError:(NSError *)error
{
    if(error.code == PPCloudCommonError_CancelLogin)
    {
        return NO;
    }
    
    return YES;
}


//==============================================================================
//
//==============================================================================
- (NSString *)storageSpaceSelectViewControllerRequestErrorMessageWithError:(NSError *)error
{
    if (error.code ==PPStorageSpaceSelectControllerErrorCode_NetworkDisconnected)
    {
        return [error localizedFailureReason];
    }
    
    return nil;
}


//================================================================================
//
//================================================================================
- (void)storageSpaceSelectViewController:(PPStorageSpaceSelectViewController *)storageSpaceSelectViewController didStorageSpaceSelectControllerSpaceType:(PPStorageSpaceSelectControllerSpaceType)storageSpaceSelectControllerSpaceType
{
//    NSLog(@" %s, storageSpaceSelectControllerSpaceType:%td", __func__, storageSpaceSelectControllerSpaceType);
    
    //////////////////////////////////////////////////
    
    self.spaceTypeSelected = storageSpaceSelectControllerSpaceType;
    
    [self requestSpaceInfo];
    
    //////////////////////////////////////////////////
    
    if (storageSpaceSelectControllerSpaceType==PPStorageSpaceSelectControllerSpaceType_AddressBook)
    {
        [WCABItemSelectFlowController showGroupViewControllerWithABSelectFlowControllerMode:WCABItemSelectFlowControllerMode_ApToSystemContact
                                                             abSelectFlowControllerDelegate:self
                                                                            syncAccountName:self.syncAccountName
                                                                    fromSuperViewController:storageSpaceSelectViewController];
    }
    #if defined (PRODUCTLINE_WCT)
    else if (storageSpaceSelectControllerSpaceType==PPStorageSpaceSelectControllerSpaceType_Salesforce)
    {
        // salesforce匯出流程

        if (self.enableCRM==NO)
        {
            [WCToastController showMessageToastFromSuperView:storageSpaceSelectViewController.view
                                                 withMessage:WCEFC_MLS_CRMNotEnable
                                                    position:PPToastPositionCenter];
            return ;
        }
        
        // TODO: !! 如果有選取到沒有同步過的聯絡人，就不能匯出
        if ([self.delegate respondsToSelector:@selector(exportFlowController:checkIfSyncWithContactGuids:)])
        {
            if ([self.delegate exportFlowController:self checkIfSyncWithContactGuids:self.contactIdentifierArray]==NO)
            {
                [WCToastController showMessageToastFromSuperView:storageSpaceSelectViewController.view
                                                     withMessage:WCEFC_MLS_UnSyncContactsExportToCRM
                                                        position:PPToastPositionCenter];
                return ;
            }
        }
        //////////////////////////////////////////////////
        // CRM 匯出流程
        [WCTServerContactExportFlowController exportToTarget:WCTServerExportTarget_Salesforce
                                                withContacts:self.contactIdentifierArray
                                          fromViewController:storageSpaceSelectViewController
                                             completeHandler:^(NSError *error) {
            NSString *message = nil;
            if(error)
            {
                if ([self.delegate respondsToSelector:@selector(exportFlowController:requestErrorMessageWithError:)])
                {
                    message = [self.delegate exportFlowController:self requestErrorMessageWithError:error];
                }
            }
            else
            {
                message = WCEFC_MLS_ExportSuccess;
            }
            
            if([message length]>0)
            {
                [PPAlertView showWithStyle:UIAlertViewStyleDefault
                                     title:@""
                                   message:message
                                  delegate:self
                                       tag:WCExportFlowControllerAlertTag_Salesforce
                         cancelButtonTitle:WCEFC_MLS_OK
                         otherButtonTitles:nil];
            }
        }];
    }
    else if (storageSpaceSelectControllerSpaceType==PPStorageSpaceSelectControllerSpaceType_ExchangeServer)
    {
        // exchange匯出流程
        NSString *exchangeType = [PPSettingsController stringValueWithKey:WCTSettingsKey_ExchangeType];
        WCTServerExportTarget target = [WCTServerContactExportHelper targetFromServerType:exchangeType];
        
        if (self.enableExchangeServer==NO)
        {
            [WCToastController showMessageToastFromSuperView:storageSpaceSelectViewController.view
                                                 withMessage:[WCTServerContactExportHelper disabledByManagerMessage]
                                                    position:PPToastPositionCenter];
            return ;
        }
        
        // !! 如果有選取到沒有同步過的聯絡人，就不能匯出
        if ([self.delegate respondsToSelector:@selector(exportFlowController:checkIfSyncWithContactGuids:)])
        {
            if ([self.delegate exportFlowController:self checkIfSyncWithContactGuids:self.contactIdentifierArray]==NO)
            {
                [WCToastController showMessageToastFromSuperView:storageSpaceSelectViewController.view
                                                     withMessage:[WCTServerContactExportHelper tryToExportUnSyncDataToTarget:target]
                                                        position:PPToastPositionCenter];
                return ;
            }
        }
        //////////////////////////////////////////////////
        // office365與MS exhcange  是一樣的API，所以只要帶WCTServerExportTarget_Exchange_MS就好

        [WCTServerContactExportFlowController exportToTarget:target
                                                withContacts:self.contactIdentifierArray
                                          fromViewController:storageSpaceSelectViewController
                                             completeHandler:^(NSError *error) {
            NSString *message = nil;
            if(error)
            {
                if ([self.delegate respondsToSelector:@selector(exportFlowController:requestErrorMessageWithError:)])
                {
                    message = [self.delegate exportFlowController:self requestErrorMessageWithError:error];
                }
            }
            else
            {
                message = WCEFC_MLS_ExportSuccess;
            }
            
            if([message length]>0)
            {
                [PPAlertView showWithStyle:UIAlertViewStyleDefault
                                     title:@""
                                   message:message
                                  delegate:self
                                       tag:WCExportFlowControllerAlertTag_Salesforce
                         cancelButtonTitle:WCEFC_MLS_OK
                         otherButtonTitles:nil];
            }
        }];
    }
    #endif
    else
    {
        if (self.exportViewController==nil)
        {
            _exportViewController = [[EFCExportViewController alloc] init];
        }
        
        self.exportViewController.delegate = self;
        
        //////////////////////////////////////////////////
        //!! 如果是icloud, 只允許匯出WCXF, VCF
        // 因為目前匯入只有支援這兩種，其他的檔案，沒有地方可以讀取
        
        NSMutableArray *tempFileTypeOrderSource = [NSMutableArray arrayWithArray:self.fileTypeOrderSource];
        
        if (storageSpaceSelectControllerSpaceType==PPStorageSpaceSelectControllerSpaceType_iCloud)
        {
            [tempFileTypeOrderSource removeObject:@(WCFileTypeSelectView_ContactFileType_XLSX)];
            [tempFileTypeOrderSource removeObject:@(WCFileTypeSelectView_ContactFileType_CSV_Gmail)];
            [tempFileTypeOrderSource removeObject:@(WCFileTypeSelectView_ContactFileType_CSV_Outlook)];
            [tempFileTypeOrderSource removeObject:@(WCFileTypeSelectView_ContactFileType_CSV_iWorks)];
        }
        
        self.exportViewController.fileTypeOrderSource = tempFileTypeOrderSource;
        
        //////////////////////////////////////////////////
        
        if(storageSpaceSelectViewController.navigationController!=nil)
        {
            [storageSpaceSelectViewController.navigationController pushViewController:self.exportViewController animated:YES];
        }
        else
        {
            PPNavigationController *navigationController = [[PPNavigationController alloc] initWithRootViewController:self.exportViewController];
            if (navigationController!=nil)
            {
                [storageSpaceSelectViewController presentViewController:navigationController animated:YES completion:^{}];
                
                [navigationController release];
            }
        }
    }
}


//================================================================================
//
//================================================================================
- (void)storageSpaceSelectViewControllerDidCancel:(PPStorageSpaceSelectViewController *)storageSpaceSelectViewController
{
//    NSLog(@" %s",__func__);
    
    [storageSpaceSelectViewController goBackAnimated:YES];
}


//================================================================================
//
//================================================================================
- (void)storageSpaceSelectViewController:(PPStorageSpaceSelectViewController *)storageSpaceSelectViewController didLoginSuccessWithStorageSpaceSelectControllerSpaceType:(PPStorageSpaceSelectControllerSpaceType)storageSpaceSelectControllerSpaceType
{
    self.spaceTypeSelected = storageSpaceSelectControllerSpaceType;
    
    [self requestSpaceInfo];
    
    //////////////////////////////////////////////////
    
    if (storageSpaceSelectControllerSpaceType==PPStorageSpaceSelectControllerSpaceType_AddressBook)
    {
        [WCABItemSelectFlowController showGroupViewControllerWithABSelectFlowControllerMode:WCABItemSelectFlowControllerMode_ApToSystemContact
                                                             abSelectFlowControllerDelegate:self
                                                                            syncAccountName:self.syncAccountName
                                                                    fromSuperViewController:storageSpaceSelectViewController];
    }
    else
    {
        if (self.exportViewController==nil)
        {
            _exportViewController = [[EFCExportViewController alloc] init];
        }
        
        self.exportViewController.delegate = self;
        
        //////////////////////////////////////////////////
        //!! 如果是icloud, 只允許匯出WCXF, VCF
        // 因為目前匯入只有支援這兩種，其他的檔案，沒有地方可以讀取
        
        NSMutableArray *tempFileTypeOrderSource = [NSMutableArray arrayWithArray:self.fileTypeOrderSource];
        
        if (storageSpaceSelectControllerSpaceType==PPStorageSpaceSelectControllerSpaceType_iCloud)
        {
            [tempFileTypeOrderSource removeObject:@(WCFileTypeSelectView_ContactFileType_XLSX)];
            [tempFileTypeOrderSource removeObject:@(WCFileTypeSelectView_ContactFileType_CSV_Gmail)];
            [tempFileTypeOrderSource removeObject:@(WCFileTypeSelectView_ContactFileType_CSV_Outlook)];
            [tempFileTypeOrderSource removeObject:@(WCFileTypeSelectView_ContactFileType_CSV_iWorks)];
        }
        
        self.exportViewController.fileTypeOrderSource = tempFileTypeOrderSource;
        
        //////////////////////////////////////////////////
        
        if(storageSpaceSelectViewController.navigationController!=nil)
        {
            [storageSpaceSelectViewController.navigationController pushViewController:self.exportViewController animated:YES];
        }
        else
        {
            PPNavigationController *navigationController = [[PPNavigationController alloc] initWithRootViewController:self.exportViewController];
            if (navigationController!=nil)
            {
                [storageSpaceSelectViewController presentViewController:navigationController animated:YES completion:^{}];
                
                [navigationController release];
            }
        }
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)storageSpaceSelectViewController:(PPStorageSpaceSelectViewController *)storageSpaceSelectViewController
                     isLinkWithSpaceType:(PPStorageSpaceSelectControllerSpaceType)spaceType
{
    if(spaceType==PPStorageSpaceSelectControllerSpaceType_Salesforce)
    {
        return YES;
    }
    
    return NO;
}


//==============================================================================
//
//==============================================================================
- (NSString *)storageSpaceSelectViewController:(PPStorageSpaceSelectViewController *)storageSpaceSelectViewController
                      accountNameWithSpaceType:(PPStorageSpaceSelectControllerSpaceType)spaceType
                                         error:(NSError **)error
{
    if(spaceType==PPStorageSpaceSelectControllerSpaceType_Salesforce)
    {
        
    }
    
    return nil;
}



////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - WCABItemSelectFlowController Delegate Method

//================================================================================
//!! 進行聯絡人匯出至系統(目前群組僅可單選)-需WCM紀錄ID，以便重複的處理！香同時進行更新
//================================================================================
- (void)abSelectFlowController:(WCABItemSelectFlowController *)abSelectFlowController
            fromViewController:(UIViewController *)fromViewController
         didSelectABGroupModel:(WCABGroupModel *)abGroupModel
{
    self.dismissGroupViewController = fromViewController;
    
    //////////////////////////////////////////////////
    // !! 一定要先確認權限才能繼續進行操作
    
    __block WCExportFlowController *blockSelf = self;
    [WCABDataController checkAccessPermissionWithCompletion:^(BOOL authorized) {
        if (authorized==YES)
        {
            [[UIApplication sharedApplication] enableIdle:NO];
            [blockSelf showBusyView:YES];
            
            dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
                NSError *error = nil;
                NSUInteger successContactCount = 0;
                
                do
                {
                    WCABDataController *dataController = [[WCABDataController alloc] initWithError:&error];
                    if (dataController==nil)
                    {
                        break;
                    }
                    
                    if (error!=nil)
                    {
                        [dataController release];
                        break;
                    }
                    
                    //////////////////////////////////////////////////
                    // 判斷所有將匯出的聯絡人該使用新增或更新
                    
                    if (blockSelf.delegate!=nil && [blockSelf.delegate respondsToSelector:@selector(exportFlowController:addressBookCardModelWithCardID:)]==YES)
                    {
                        BOOL perABCardModelSuccess = YES;
                        
                        for (NSInteger srcIndex=0; srcIndex<blockSelf.contactIdentifierArray.count; srcIndex++)
                        {
                            @autoreleasepool
                            {
                                dispatch_async(dispatch_get_main_queue(), ^{
                                    CGFloat segmentSize   = 100.0/[blockSelf.contactIdentifierArray count];
                                    CGFloat totalProgress = (srcIndex+1)*segmentSize;
                                    
                                    [PPBusyView postMessage:[NSString stringWithFormat:@"%@%.2f%@", WCEFC_MLS_Exporting, totalProgress, @"%"]];
                                });
                                
                                //////////////////////////////////////////////////
                                
                                NSString *cardID = [self.contactIdentifierArray objectAtIndex:srcIndex];
                                
                                WCABCardModel *abCardModel = [blockSelf.delegate exportFlowController:blockSelf addressBookCardModelWithCardID:cardID];
                                if (abCardModel!=nil)
                                {
                                    if (abCardModel.abPersonID==nil)
                                    {
                                        abCardModel.abSourceID = abGroupModel.abSourceID;
                                        abCardModel.abGroupID  = abGroupModel.abGroupID;
#if defined (PRODUCTLINE_WCSF)
                                        // MARK: !!Matomo Log (匯出到系統聯絡人)
                                        [WCSFMatomoController sendSaveToMobileContactLogWithAction:WCSFMatomoController_SaveCardAction_Create];
#endif
                                        perABCardModelSuccess = [dataController addPersonWithCardModel:abCardModel error:&error];
                                        if (perABCardModelSuccess==YES && error==nil)
                                        {
                                            successContactCount++;
                                        }
                                    }
                                    else
                                    {
                                        perABCardModelSuccess = [dataController updatePersonWithCardModel:abCardModel newSourceID:abGroupModel.abSourceID newGroupID:abGroupModel.abGroupID error:&error];
                                        if (perABCardModelSuccess==NO || error!=nil)
                                        {
                                            if (error.code==WCABDataController_Error_FailedToGetContact || error.code==WCABDataController_Error_FailedToGetSourceID)
                                            {
                                                //!! 可能發生刪除系統聯絡人的問題，故以上兩個Error需重新添加
                                                abCardModel.abSourceID = abGroupModel.abSourceID;
                                                abCardModel.abGroupID  = abGroupModel.abGroupID;
                                                
#if defined (PRODUCTLINE_WCSF)
                                                // MARK: !!Matomo Log (匯出到系統聯絡人)
                                                [WCSFMatomoController sendSaveToMobileContactLogWithAction:WCSFMatomoController_SaveCardAction_Create];
#endif

                                                perABCardModelSuccess = [dataController addPersonWithCardModel:abCardModel error:&error];
                                                if (perABCardModelSuccess==YES && error==nil)
                                                {
                                                    successContactCount++;
                                                }
                                            }
                                        }
                                        else
                                        {
#if defined (PRODUCTLINE_WCSF)
                                            // MARK: !!Matomo Log (匯出到系統聯絡人)
                                            [WCSFMatomoController sendSaveToMobileContactLogWithAction:WCSFMatomoController_SaveCardAction_Update];
#endif

                                            successContactCount++;
                                        }
                                    }
                                    
                                    //////////////////////////////////////////////////
                                    
                                    if (error==nil && blockSelf.delegate!=nil && [blockSelf.delegate respondsToSelector:@selector(exportFlowController:exportOneAddressBookCardModel:withCardID:)]==YES)
                                    {
                                        [blockSelf.delegate exportFlowController:blockSelf exportOneAddressBookCardModel:abCardModel withCardID:cardID];
                                    }
                                }
                            }
                        }
                        
                        //////////////////////////////////////////////////
                        // 判斷匯出聯絡人是否有發生錯誤
                        
                        if (successContactCount>0)
                        {
                            blockSelf.exportFlowError = nil;
                        }
                        else
                        {
                            blockSelf.exportFlowError = PPErrorMake(WCExportFlowController_Error_NoCardToWrite, nil, nil);
                        }
                    }
                    
                    [dataController release];
                } while (0);
                
                //////////////////////////////////////////////////
                
                if (blockSelf.exportFlowError==nil)
                {
                     dispatch_async(dispatch_get_main_queue(), ^{
                         [[UIApplication sharedApplication] enableIdle:YES];
                         [blockSelf showBusyView:NO];
                         
                         //////////////////////////////////////////////////
                         
                         [PPAlertView showWithStyle:UIAlertViewStyleDefault
                                              title:@""
                                            message:WCEFC_MLS_ExportSuccess
                                           delegate:blockSelf
                                  cancelButtonTitle:WCEFC_MLS_OK
                                  otherButtonTitles:nil];
                     });
                }
                else
                {
                    dispatch_async(dispatch_get_main_queue(), ^{
                        [[UIApplication sharedApplication] enableIdle:YES];
                        [blockSelf showBusyView:NO];
                        
                        //////////////////////////////////////////////////
                        
                        if (blockSelf.exportFlowError.code==WCExportFlowController_Error_NoCardToWrite)
                        {
                            [WCToastController showMessageToastFromSuperView:fromViewController.view withMessage:WCEFC_MLS_FailedToLoadDataWithID position:PPToastPositionBottom];
                        }
                        else
                        {
                            [PPAlertView showWithStyle:UIAlertViewStyleDefault
                                                 title:@""
                                               message:WCEFC_MLS_ExportFailed
                                              delegate:blockSelf
                                     cancelButtonTitle:WCEFC_MLS_OK
                                     otherButtonTitles:nil];
                        }
                    });
                }
            });
        }
    }];
}


//================================================================================
//!! 匯出頁面無需系統聯絡人清單
//================================================================================
- (void)abSelectFlowController:(WCABItemSelectFlowController *)abSelectFlowController
            fromViewController:(UIViewController *)fromViewController
         didSelectABCardModels:(NSArray *)abCardModels
{
//    NSLog(@" %s, abCardModels:%@",__func__,abCardModels);
}


//================================================================================
//
//================================================================================
- (void)abSelectFlowControllerFailWithError:(NSError *)error
{
    PPAlertView *alertView = [[PPAlertView alloc] initWithTitle:@""
                                                        message:error.localizedFailureReason
                                                       delegate:nil
                                              cancelButtonTitle:WCEFC_MLS_OK
                                              otherButtonTitles:nil];
    [alertView show];
    [alertView release];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - EFCExportViewControllerDelegate

//================================================================================
// 單選故使用if..else...製作檔案格式對應的檔案放置於路徑中
//================================================================================
- (void)exportViewController:(EFCExportViewController *)exportViewController
    didSelectContactFileType:(WCFileTypeSelectView_ContactFileType)contactFileType
    fileTypeTextInfoSelected:(WCFileTypeSelectView_FileTypeTextInfo)fileTypeTextInfoSelected
{
    if ([PPNetworkReachabilityController checkForInternetConnection]==YES || self.spaceTypeSelected==PPStorageSpaceSelectControllerSpaceType_iTune)
    {
        [[UIApplication sharedApplication] enableIdle:NO];
        [self showBusyView:YES];
        
        //////////////////////////////////////////////////
        
        __block WCExportFlowController *blockSelf = self;
        
        dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
            
            // MARK: !! matomo log (匯出)
#if defined (PRODUCTLINE_WCT)
            switch (contactFileType)
            {
                case WCFileTypeSelectView_ContactFileType_VCF:
                {
                    [WCTMatomoController sendLogWithExportTarget:WCTMatomoController_ExportTarget_VCF];
                    break;
                }
                    case WCFileTypeSelectView_ContactFileType_CSV_Gmail:
                {
                    [WCTMatomoController sendLogWithExportTarget:WCTMatomoController_ExportTarget_CSV];
                    break;
                }
                    case WCFileTypeSelectView_ContactFileType_CSV_Outlook:
                {
                    [WCTMatomoController sendLogWithExportTarget:WCTMatomoController_ExportTarget_Outlook];
                    break;
                }
                    
                default:
                    break;
            }
#endif

            //////////////////////////////////////////////////
            // 需判斷storageSpaceType決定檔案放置路徑(若iTune直接放置於Document)
            
            NSString *tempFolderPath = [[WCToolController tempFolderPath] stringByAppendingPathComponent:WCExportFlowController_ContactExportRootFolder];
            NSError *fileError       = nil;
            
            //////////////////////////////////////////////////
            //!! 避免操作過快，網路過慢，資訊尚未抓取的問題發生
            
            while (blockSelf.requestResponse == nil)
            {
                [NSThread sleepForTimeInterval:0.1];
            }
            
            //////////////////////////////////////////////////
            
            do
            {
                if ([[NSFileManager defaultManager] fileExistsAtPath:tempFolderPath]==YES)
                {
                    [[NSFileManager defaultManager] removeItemAtPath:tempFolderPath error:nil];
                }
                
                [[NSFileManager defaultManager] createDirectoryAtPath:tempFolderPath
                                          withIntermediateDirectories:YES
                                                           attributes:nil
                                                                error:&fileError];
                
                //////////////////////////////////////////////////
                // 判斷手機記憶體容量
                
                NSArray *paths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
                NSDictionary *dictionary = [[NSFileManager defaultManager] attributesOfFileSystemForPath:[paths lastObject] error:nil];
                
                double freeSystemSize = [[dictionary objectForKey:NSFileSystemFreeSize] doubleValue];
                if (freeSystemSize<=WCExportFlowController_MinimumSystemMemoryRequestBytes)
                {
                    dispatch_async(dispatch_get_main_queue(), ^{
                        [[UIApplication sharedApplication] enableIdle:YES];
                        [blockSelf showBusyView:NO];
                        
                        blockSelf.exportFlowError = fileError;
                        
                        PPAlertView *alertView = [[PPAlertView alloc] initWithTitle:@""
                                                                            message:WCEFC_MLS_ExportDeviceStorageNotEnough
                                                                           delegate:blockSelf
                                                                  cancelButtonTitle:WCEFC_MLS_OK
                                                                  otherButtonTitles:nil];
                        [alertView show];
                        [alertView release];
                    });
                    
                    break;
                }
                
                //////////////////////////////////////////////////
                // iTunes改變路徑，直接放置於Documents內
                
                if (blockSelf.spaceTypeSelected==PPStorageSpaceSelectControllerSpaceType_iTune)
                {
                    tempFolderPath = [WCToolController shareFolderPath];
                    tempFolderPath = [tempFolderPath stringByAppendingString:WCCloudFileDefine_FileSharingDirPath];
                }
                
                //////////////////////////////////////////////////
                // 於uploadFilesWithPath裡面會自動判斷spaceTypeSelected
                
                if ((contactFileType&WCFileTypeSelectView_ContactFileType_WCXF)==WCFileTypeSelectView_ContactFileType_WCXF)
                {
                    // MARK: WCSF
                    WCXFDataController *wcxfDataController = [[WCXFDataController alloc] initWithDelegate:blockSelf];
                    if (wcxfDataController!=nil)
                    {
                        NSString *writePath = [NSString stringWithFormat:@"%@/%@", tempFolderPath, [blockSelf singleFileCompleteNameWithFormatString:@"wcxf"]];
                        
                        blockSelf.completeSuccess = [wcxfDataController writeFileWithPath:writePath
                                                                                  cardIDs:blockSelf.contactIdentifierArray
                                                                                    error:&fileError];
                        if (blockSelf.completeSuccess==YES && blockSelf.requestResponse!=nil && [blockSelf.requestResponse boolValue]==YES)
                        {
                            if ((blockSelf.spaceTotalBytes>0) && (blockSelf.spaceFreeBytes<[blockSelf folderSize:tempFolderPath]))
                            {
                                blockSelf.exportFlowError = fileError;
                                
                                dispatch_async(dispatch_get_main_queue(), ^{
                                    [[UIApplication sharedApplication] enableIdle:YES];
                                    [blockSelf showBusyView:NO];
                                    
                                    PPAlertView *alertView = [[PPAlertView alloc] initWithTitle:@""
                                                                                        message:WCEFC_MLS_ExportCloudStorageNotEnough
                                                                                       delegate:blockSelf
                                                                              cancelButtonTitle:WCEFC_MLS_OK
                                                                              otherButtonTitles:nil];
                                    [alertView show];
                                    [alertView release];
                                });
                            }
                            else
                            {
                                [blockSelf uploadFilesWithPath:[NSArray arrayWithObject:writePath]];
                            }
                        }
                        else
                        {
                            // 如果是requestResponseError
                            if(blockSelf.requestResponseError!=nil)
                            {
                                blockSelf.exportFlowError = blockSelf.requestResponseError;
                            }
                            else if(fileError)
                            {
                                blockSelf.exportFlowError = fileError;
                            }
                            
                            dispatch_async(dispatch_get_main_queue(), ^{
                                [[UIApplication sharedApplication] enableIdle:YES];
                                [blockSelf showBusyView:NO];
                                
                                if (blockSelf.exportFlowError.code==WCXFDataController_Error_NoCardToWrite)
                                {
                                    [WCToastController showMessageToastFromSuperView:self.exportViewController.view withMessage:WCEFC_MLS_FailedToLoadDataWithID position:PPToastPositionBottom];
                                }
                                else
                                {
                                    NSString *alertMessage = [self errorMessageWithError:blockSelf.exportFlowError];
                                    
                                    PPAlertView *alertView = [[PPAlertView alloc] initWithTitle:@""
                                                                                        message:alertMessage
                                                                                       delegate:blockSelf
                                                                              cancelButtonTitle:WCEFC_MLS_OK
                                                                              otherButtonTitles:nil];
                                    [alertView show];
                                    [alertView release];
                                }
                            });
                        }
                        
                        [wcxfDataController release];
                    }
                }
                else if ((contactFileType&WCFileTypeSelectView_ContactFileType_XLSX)==WCFileTypeSelectView_ContactFileType_XLSX)
                {
                    // MARK: XLSX
                    WCXLSXDataController *xlsxDataController = [[WCXLSXDataController alloc] initWithDelegate:blockSelf];
                    if (xlsxDataController!=nil)
                    {
                        NSString *writePath = [NSString stringWithFormat:@"%@/%@", tempFolderPath, [blockSelf singleFileCompleteNameWithFormatString:@"xlsx"]];
                        
                        blockSelf.completeSuccess = [xlsxDataController writeFileWithPath:writePath
                                                                                  cardIDs:blockSelf.contactIdentifierArray
                                                                                    error:&fileError];
                        if (blockSelf.completeSuccess==YES && blockSelf.requestResponse!=nil && [blockSelf.requestResponse boolValue]==YES)
                        {
                            if ((blockSelf.spaceTotalBytes>0) && (blockSelf.spaceFreeBytes<[blockSelf folderSize:tempFolderPath]))
                            {
                                blockSelf.exportFlowError = fileError;
                                
                                dispatch_async(dispatch_get_main_queue(), ^{
                                    [[UIApplication sharedApplication] enableIdle:YES];
                                    [blockSelf showBusyView:NO];
                                    
                                    PPAlertView *alertView = [[PPAlertView alloc] initWithTitle:@""
                                                                                        message:WCEFC_MLS_ExportCloudStorageNotEnough
                                                                                       delegate:blockSelf
                                                                              cancelButtonTitle:WCEFC_MLS_OK
                                                                              otherButtonTitles:nil];
                                    [alertView show];
                                    [alertView release];
                                });
                            }
                            else
                            {
                                [blockSelf uploadFilesWithPath:[NSArray arrayWithObject:writePath]];
                            }
                        }
                        else
                        {
                            // 如果是requestResponseError
                            if(blockSelf.requestResponseError!=nil)
                            {
                                blockSelf.exportFlowError = blockSelf.requestResponseError;
                            }
                            else if(fileError)
                            {
                                blockSelf.exportFlowError = fileError;
                            }
                            
                            dispatch_async(dispatch_get_main_queue(), ^{
                                [[UIApplication sharedApplication] enableIdle:YES];
                                [blockSelf showBusyView:NO];
                                
                                if (blockSelf.exportFlowError.code==WCXFDataController_Error_NoCardToWrite)
                                {
                                    [WCToastController showMessageToastFromSuperView:self.exportViewController.view withMessage:WCEFC_MLS_FailedToLoadDataWithID position:PPToastPositionBottom];
                                }
                                else
                                {
                                    NSString *alertMessage = [self errorMessageWithError:blockSelf.exportFlowError];
                                    
                                    PPAlertView *alertView = [[PPAlertView alloc] initWithTitle:@""
                                                                                        message:alertMessage
                                                                                       delegate:blockSelf
                                                                              cancelButtonTitle:WCEFC_MLS_OK
                                                                              otherButtonTitles:nil];
                                    [alertView show];
                                    [alertView release];
                                }
                            });
                        }
                        
                        [xlsxDataController release];
                    }
                }
                else if ((contactFileType&WCFileTypeSelectView_ContactFileType_CSV_iWorks)==WCFileTypeSelectView_ContactFileType_CSV_iWorks ||
                         (contactFileType&WCFileTypeSelectView_ContactFileType_CSV_Gmail)==WCFileTypeSelectView_ContactFileType_CSV_Gmail ||
                         (contactFileType&WCFileTypeSelectView_ContactFileType_CSV_Outlook)==WCFileTypeSelectView_ContactFileType_CSV_Outlook)
                {
                    // MARK: CSV
                    WCCSVDataController *csvDataController = [[WCCSVDataController alloc] initWithDelegate:blockSelf];
                    if (csvDataController!=nil)
                    {
                        NSString *writePath  = [NSString stringWithFormat:@"%@/%@", tempFolderPath, [blockSelf singleFileCompleteNameForFileTypeCSV:contactFileType]];
                        
                        WCCSVDataController_Format format = WCCSVDataController_Format_None;
                        switch (contactFileType)
                        {
                            case WCFileTypeSelectView_ContactFileType_CSV_iWorks:
                            {
                                format = WCCSVDataController_Format_iWorks;
                                break;
                            }
                            case WCFileTypeSelectView_ContactFileType_CSV_Gmail:
                            {
                                format = WCCSVDataController_Format_Gmail;
                                break;
                            }
                            case WCFileTypeSelectView_ContactFileType_CSV_Outlook:
                            {
                                format = WCCSVDataController_Format_Outlook;
                                break;
                            }
                            default:
                                break;
                        }

                        
                        blockSelf.completeSuccess = [csvDataController writeFileWithPath:writePath
                                                                                 cardIDs:blockSelf.contactIdentifierArray
                                                                                  format:format
                                                                         willSendByEmail:NO 
                                                                                   error:&fileError];
                        if (blockSelf.completeSuccess==YES && blockSelf.requestResponse!=nil && [blockSelf.requestResponse boolValue]==YES)
                        {
                            if ((blockSelf.spaceTotalBytes>0) && (blockSelf.spaceFreeBytes<[blockSelf folderSize:tempFolderPath]))
                            {
                                blockSelf.exportFlowError = fileError;
                                
                                dispatch_async(dispatch_get_main_queue(), ^{
                                    [[UIApplication sharedApplication] enableIdle:YES];
                                    [blockSelf showBusyView:NO];

                                    PPAlertView *alertView = [[PPAlertView alloc] initWithTitle:@""
                                                                                        message:WCEFC_MLS_ExportCloudStorageNotEnough
                                                                                       delegate:blockSelf
                                                                              cancelButtonTitle:WCEFC_MLS_OK
                                                                              otherButtonTitles:nil];
                                    [alertView show];
                                    [alertView release];
                                });
                            }
                            else
                            {
                                [blockSelf uploadFilesWithPath:[NSArray arrayWithObject:writePath]];
                            }
                        }
                        else
                        {
                            // 如果是requestResponseError
                            if(blockSelf.requestResponseError!=nil)
                            {
                                blockSelf.exportFlowError = blockSelf.requestResponseError;
                            }
                            else if(fileError)
                            {
                                blockSelf.exportFlowError = fileError;
                            }
                            
                            dispatch_async(dispatch_get_main_queue(), ^{
                                [[UIApplication sharedApplication] enableIdle:YES];
                                [blockSelf showBusyView:NO];
                                
                                if (blockSelf.exportFlowError.code==WCXFDataController_Error_NoCardToWrite)
                                {
                                    [WCToastController showMessageToastFromSuperView:self.exportViewController.view withMessage:WCEFC_MLS_FailedToLoadDataWithID position:PPToastPositionBottom];
                                }
                                else
                                {
                                    NSString *alertMessage = [self errorMessageWithError:blockSelf.exportFlowError];
                                    
                                    PPAlertView *alertView = [[PPAlertView alloc] initWithTitle:@""
                                                                                        message:alertMessage
                                                                                       delegate:blockSelf
                                                                              cancelButtonTitle:WCEFC_MLS_OK
                                                                              otherButtonTitles:nil];
                                    [alertView show];
                                    [alertView release];
                                }
                            });
                        }
                        
                        [csvDataController release];
                    }
                }
                else if ((contactFileType&WCFileTypeSelectView_ContactFileType_VCF)==WCFileTypeSelectView_ContactFileType_VCF)
                {
                    // MARK: VCF
                    //////////////////////////////////////////////////
                    // 建立存放VCF檔案路徑陣列
                    
                    NSMutableArray *vcfFileArray = [NSMutableArray array];
                    
                    //////////////////////////////////////////////////
                    //!! 一定要先確認權限才能繼續進行操作
                    
                    [WCVCFDataController checkAccessPermissionWithCompletion:^(BOOL authorized) {
                        if (authorized==YES)
                        {
                            NSError *error = nil;
                            do
                            {
                                WCVCFDataController *vcfDataController = [[WCVCFDataController alloc] initWithError:&error];
                                if (vcfDataController==nil)
                                {
                                    break;
                                }
                                
                                if (error!=nil)
                                {
                                    [vcfDataController release];
                                    break;
                                }
                                
                                //////////////////////////////////////////////////
                                // 依序創建每個聯絡人各自的VCF檔
                                
                                BOOL perFileConstructSuccess = YES;
                                for (NSInteger index=0; index<[blockSelf.contactIdentifierArray count]; index++)
                                {
                                    dispatch_async(dispatch_get_main_queue(), ^{
                                        CGFloat segmentSize   = 100.0/[blockSelf.contactIdentifierArray count];
                                        CGFloat totalProgress = (index+1)*segmentSize;
                                        
                                        [PPBusyView postMessage:[NSString stringWithFormat:@"%@%.2f%@", WCEFC_MLS_GeneratingData, totalProgress, @"%"]];
                                    });
                                    
                                    
                                    NSString *cardID = [blockSelf.contactIdentifierArray objectAtIndex:index];
                                    
                                    if (blockSelf.delegate!=nil && [blockSelf.delegate respondsToSelector:@selector(exportFlowController:cardModelToWriteWithCardID:)]==YES)
                                    {
                                        //////////////////////////////////////////////////
                                        // 取得檔案名稱(聯絡人前後名稱)
                                        // 104/12/11移除使用聯絡人名稱對檔案命名(分享頁面在壓縮後，英文其他語系會產生亂碼)
                                        
                                        WCCardModel *cardModel = [blockSelf.delegate exportFlowController:blockSelf cardModelToWriteWithCardID:cardID];
                                        
//                                        NSString *fullName         = WCExportFlowController_ContactFileDefaultName;
//                                        NSMutableArray *fieldArray = [cardModel fieldArrayWithType:WC_FT_Name];
//                                        if ([fieldArray count]>0 && fieldArray!=nil)
//                                        {
//                                            WCFieldModel *nameField = [fieldArray objectAtIndex:0];
//                                            NSString *firstName     = [nameField valueWithSubType2:WC_FST2_Name_First];
//                                            NSString *lastName      = [nameField valueWithSubType2:WC_FST2_Name_Last];
//                                            
//                                            //////////////////////////////////////////////////
//                                            // 取得名字排序 & 完全姓名
//                                            
//                                            if (blockSelf.displayEastNameOrder==WCDisplayNameOrder_None && (blockSelf.delegate!=nil && [blockSelf.delegate respondsToSelector:@selector(contactNameEastOrderInExportFlowController:)]==YES))
//                                            {
//                                                blockSelf.displayEastNameOrder = [blockSelf.delegate contactNameEastOrderInExportFlowController:blockSelf];
//                                            }
//                                            
//                                            if (blockSelf.displayWestNameOrder==WCDisplayNameOrder_None && (blockSelf.delegate!=nil && [blockSelf.delegate respondsToSelector:@selector(contactNameWestOrderInExportFlowController:)]==YES))
//                                            {
//                                                blockSelf.displayWestNameOrder = [blockSelf.delegate contactNameWestOrderInExportFlowController:blockSelf];
//                                            }
//                                            
//                                            fullName = [WCDisplayNameController displayNameWithFirstName:firstName
//                                                                                                lastName:lastName
//                                                                                               eastOrder:blockSelf.displayEastNameOrder
//                                                                                               westOrder:blockSelf.displayWestNameOrder];
//                                        }
                                        
                                        //////////////////////////////////////////////////
                                        // 創建VCF檔
                                        
                                        NSString *writePath = [NSString stringWithFormat:@"%@/", tempFolderPath];
                                        NSString *dateTime  = @"";
                                        
                                        NSDateFormatter *dateFormatter = [[NSDateFormatter alloc] init];
                                        if(dateFormatter!=nil)
                                        {
                                            [dateFormatter setDateFormat:WCCloudFileDefine_DateTimeFormat];
                                            
                                            dateTime = [dateFormatter stringFromDate:[NSDate date]];
                                            [dateFormatter release];
                                        }
                                        
                                        dateTime  = [[dateTime stringByAppendingString:@"_"] stringByAppendingFormat:WCCloudFileDefine_SingleContactFileNameFormat, (short)index+1, @"", @"vcf"];
                                        writePath = [writePath stringByAppendingString:dateTime];
                                        
                                        perFileConstructSuccess = [vcfDataController writeFileWithPath:writePath cardModel:cardModel error:&error];
                                        if (perFileConstructSuccess==YES || error==nil)
                                        {
                                            [vcfFileArray addObject:writePath];
                                            
                                            //////////////////////////////////////////////////
                                            // 若無錯誤即進行紀錄創建成功的檔案路徑＆發布通知
                                            
                                            if (blockSelf.delegate!=nil && [blockSelf.delegate respondsToSelector:@selector(exportFlowController:didWriteCardWithIndex:totalCount:)]==YES)
                                            {
                                                [blockSelf.delegate exportFlowController:blockSelf didWriteCardWithIndex:index totalCount:[blockSelf.contactIdentifierArray count]];
                                            }
                                        }
                                    }
                                }
                                
                                [vcfDataController release];
                                
                            } while (0);
                            
                            //////////////////////////////////////////////////
                            
                            if ([vcfFileArray count]>0 && blockSelf.requestResponse!=nil && [blockSelf.requestResponse boolValue]==YES)
                            {
                                blockSelf.completeSuccess = YES;
                                
                                if ((blockSelf.spaceTotalBytes>0) &&  (blockSelf.spaceFreeBytes<[blockSelf folderSize:tempFolderPath]))
                                {
                                    dispatch_async(dispatch_get_main_queue(), ^{
                                        [[UIApplication sharedApplication] enableIdle:YES];
                                        [blockSelf showBusyView:NO];
                                        
                                        blockSelf.exportFlowError = error;
                                        
                                        PPAlertView *alertView = [[PPAlertView alloc] initWithTitle:@""
                                                                                            message:WCEFC_MLS_ExportCloudStorageNotEnough
                                                                                           delegate:blockSelf
                                                                                  cancelButtonTitle:WCEFC_MLS_OK
                                                                                  otherButtonTitles:nil];
                                        [alertView show];
                                        [alertView release];
                                    });
                                }
                                else
                                {
                                    [blockSelf uploadFilesWithPath:[NSArray arrayWithArray:vcfFileArray]];
                                }
                            }
                            else
                            {
                                blockSelf.completeSuccess = NO;
                                if(blockSelf.requestResponseError!=nil)
                                {
                                    blockSelf.exportFlowError = blockSelf.requestResponseError;
                                }
                                else if(fileError)
                                {
                                    blockSelf.exportFlowError = fileError;
                                }
                                
                                
                                //!! 取消Loading頁面
                                dispatch_async(dispatch_get_main_queue(), ^{
                                    [[UIApplication sharedApplication] enableIdle:YES];
                                    [blockSelf showBusyView:NO];
                                    
                                    if (blockSelf.exportFlowError.code==WCXFDataController_Error_NoCardToWrite)
                                    {
                                        [WCToastController showMessageToastFromSuperView:self.exportViewController.view withMessage:WCEFC_MLS_FailedToLoadDataWithID position:PPToastPositionBottom];
                                    }
                                    else
                                    {
                                        NSString *alertMessage = [self errorMessageWithError:blockSelf.exportFlowError];
                                        
                                        PPAlertView *alertView = [[PPAlertView alloc] initWithTitle:@""
                                                                                            message:alertMessage
                                                                                           delegate:blockSelf
                                                                                  cancelButtonTitle:WCEFC_MLS_OK
                                                                                  otherButtonTitles:nil];
                                        [alertView show];
                                        [alertView release];
                                    }
                                });
                            }
                        }
                        else
                        {
                            dispatch_async(dispatch_get_main_queue(), ^{
                                [blockSelf showBusyView:NO];
                                
                                PPAlertView *alertView = [[PPAlertView alloc] initWithTitle:@""
                                                                                    message:WCEFC_MLS_NoAddressBookPermission
                                                                                   delegate:blockSelf
                                                                          cancelButtonTitle:WCEFC_MLS_OK
                                                                          otherButtonTitles:nil];
                                [alertView show];
                                [alertView release];
                            });
                        }
                    }];
                }
                else
                {
                    dispatch_async(dispatch_get_main_queue(), ^{
                        [[UIApplication sharedApplication] enableIdle:YES];
                        [blockSelf showBusyView:NO];
                    });
                    NSLog(@"Error ContactFileType");
                }
                
            } while (0);
        });
    }
    else
    {
        [PPAlertView showWithStyle:UIAlertViewStyleDefault
                             title:@""
                           message:WCEFC_MLS_PleaseConnectNetwork
                 cancelButtonTitle:WCEFC_MLS_OK
                 otherButtonTitles:nil];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - WCCSVDataControllerDelegate

//================================================================================
//
//================================================================================
- (WCCardModel *)csvDataContrller:(WCCSVDataController *)csvDataController
       cardModelToWriteWithCardID:(NSString *)cardID
{
    WCCardModel *cardModel= nil;
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(exportFlowController:cardModelToWriteWithCardID:)]==YES)
    {
        cardModel = [self.delegate exportFlowController:self cardModelToWriteWithCardID:cardID];
    }
    
    return cardModel;
}


//================================================================================
//
//================================================================================
- (BOOL)csvDataContrller:(WCCSVDataController *)csvDataController shouldWriteCardWithIndex:(NSInteger)index totalCount:(NSInteger)totalCount
{
    return YES;
}


//================================================================================
//
//================================================================================
- (void)csvDataContrller:(WCCSVDataController *)csvDataController
   didWriteCardWithIndex:(NSInteger)index
              totalCount:(NSInteger)totalCount
{
    dispatch_async(dispatch_get_main_queue(), ^{
        CGFloat segmentSize = 100.0/totalCount;
        CGFloat totalProgress = (index+1)*segmentSize;
        
        [PPBusyView postMessage:[NSString stringWithFormat:@"%@%.2f%@", WCEFC_MLS_GeneratingData, totalProgress, @"%"]];
    });
    
    //////////////////////////////////////////////////
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(exportFlowController:didWriteCardWithIndex:totalCount:)]==YES)
    {
        [self.delegate exportFlowController:self didWriteCardWithIndex:index totalCount:totalCount];
    }
}


//================================================================================
//
//================================================================================
- (NSString *)csvDataContrller:(WCCSVDataController *)csvDataController
   groupNameToWriteWithGroupID:(WC_GroupID)groupID
{
    NSString *groupName = nil;
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(exportFlowController:groupNameToWriteWithGroupID:)]==YES)
    {
        groupName = [self.delegate exportFlowController:self groupNameToWriteWithGroupID:groupID];
    }
    
    return groupName;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - WCXFDataControllerDelegate


//==============================================================================
//
//==============================================================================
- (BOOL)wcxfDataContrller:(WCXFDataController *)wcxfDataController
 shouldWriteCardWithIndex:(NSInteger)index
               totalCount:(NSInteger)totalCount
{
    return YES;
}


//================================================================================
//
//================================================================================
- (WCCardModel *)wcxfDataContrller:(WCXFDataController *)wcxfDataController
        cardModelToWriteWithCardID:(NSString *)cardID
{
    WCCardModel *cardModel= nil;
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(exportFlowController:cardModelToWriteWithCardID:)]==YES)
    {
        cardModel = [self.delegate exportFlowController:self cardModelToWriteWithCardID:cardID];
    }
    
    return cardModel;
}


//================================================================================
//
//================================================================================
- (NSString *)wcxfDataContrller:(WCXFDataController *)wcxfDataController
    groupNameToWriteWithGroupID:(WC_GroupID)groupID
{
    NSString *groupName = nil;
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(exportFlowController:groupNameToWriteWithGroupID:)]==YES)
    {
        groupName = [self.delegate exportFlowController:self groupNameToWriteWithGroupID:groupID];
    }
    
    return groupName;
}


//================================================================================
//
//================================================================================
- (NSData *)wcxfDataContrller:(WCXFDataController *)wcxfDataController
   imageDataToWriteWithCardID:(NSString *)cardID
                    imageType:(WC_ImageType)imageType
{
    NSData *imageData = nil;
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(exportFlowController:imageDataToWriteWithCardID:imageType:)]==YES)
    {
        imageData = [self.delegate exportFlowController:self imageDataToWriteWithCardID:cardID imageType:imageType];
    }
    
    return imageData;
}


//================================================================================
//
//================================================================================
- (void)wcxfDataContrller:(WCXFDataController *)wcxfDataController
    didWriteCardWithIndex:(NSInteger)index
               totalCount:(NSInteger)totalCount
{
    dispatch_async(dispatch_get_main_queue(), ^{
        CGFloat segmentSize = 100.0/totalCount;
        CGFloat totalProgress = (index+1)*segmentSize;
        
        [PPBusyView postMessage:[NSString stringWithFormat:@"%@%.2f%@", WCEFC_MLS_GeneratingData, totalProgress, @"%"]];
    });
    
    //////////////////////////////////////////////////
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(exportFlowController:didWriteCardWithIndex:totalCount:)]==YES)
    {
        [self.delegate exportFlowController:self didWriteCardWithIndex:index totalCount:totalCount];
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - WCWLSXDataControllerDelegate


//==============================================================================
//
//==============================================================================
- (WCCardModel *)xlsxDataContrller:(WCXLSXDataController *)xlsxDataContrller
        cardModelToWriteWithCardID:(NSString *)cardID;
{
    WCCardModel *cardModel= nil;
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(exportFlowController:cardModelToWriteWithCardID:)]==YES)
    {
        cardModel = [self.delegate exportFlowController:self cardModelToWriteWithCardID:cardID];
    }
    
    return cardModel;
}


//==============================================================================
//
//==============================================================================
- (NSString *)xlsxDataContrller:(WCXLSXDataController *)xlsxDataContrller
    groupNameToWriteWithGroupID:(WC_GroupID)groupID
{
    NSString *groupName = nil;
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(exportFlowController:groupNameToWriteWithGroupID:)]==YES)
    {
        groupName = [self.delegate exportFlowController:self groupNameToWriteWithGroupID:groupID];
    }
    
    return groupName;
}


//================================================================================
//
//================================================================================
- (BOOL)xlsxDataContrller:(WCXLSXDataController *)xlsxDataContrller shouldWriteCardWithIndex:(NSInteger)index totalCount:(NSInteger)totalCount
{
    return YES;
}


//==============================================================================
//
//==============================================================================
- (void)xlsxDataContrller:(WCXLSXDataController *)xlsxDataContrller
    didWriteCardWithIndex:(NSInteger)index
               totalCount:(NSInteger)totalCount
{
    dispatch_async(dispatch_get_main_queue(), ^{
        CGFloat segmentSize = 100.0/totalCount;
        CGFloat totalProgress = (index+1)*segmentSize;
        
        [PPBusyView postMessage:[NSString stringWithFormat:@"%@%.2f%@", WCEFC_MLS_GeneratingData, totalProgress, @"%"]];
    });
    
    //////////////////////////////////////////////////
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(exportFlowController:didWriteCardWithIndex:totalCount:)]==YES)
    {
        [self.delegate exportFlowController:self didWriteCardWithIndex:index totalCount:totalCount];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPCloudDelegate_LoadAccountInfo

//================================================================================
//
//===============================================================================
- (void)ppCloud:(PPCloud *)ppCloud loadAccountInfoFailedWithError:(NSError *)error userInfo:(id)userInfo
{
    self.requestResponseError = error;
    self.requestResponse = @(NO);
}


//================================================================================
//
//================================================================================
- (void)ppCloud:(PPCloud *)ppCloud loadedAccountInfo:(PPCloudAccountInfo *)accountInfo userInfo:(id)userInfo
{
    long long spaceTotalBytes = (accountInfo.totalBytes!=nil) ? [accountInfo.totalBytes longLongValue] : LONG_LONG_MAX;
    long long spaceFreeBytes = (spaceTotalBytes==LONG_LONG_MAX) ? LONG_LONG_MAX : spaceTotalBytes - [accountInfo.usedBytes longLongValue];

    //////////////////////////////////////////////////

    self.spaceTotalBytes = spaceTotalBytes;
    self.spaceFreeBytes = spaceFreeBytes;
    self.requestResponse = @(YES);
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPCloudDelegate_UploadFile

//================================================================================
//
//================================================================================
- (void)ppCloud:(PPCloud *)ppCloud uploadFileFailedWithError:(NSError *)error userInfo:(id)userInfo
{
    NSString *filePath = (NSString *)userInfo;
    
    //////////////////////////////////////////////////
    // 無網路會每個檔案傳送失敗訊息，刪除暫存檔
    
    [[NSFileManager defaultManager] removeItemAtPath:filePath error:nil];
    
    //////////////////////////////////////////////////
    // 最後一個文件上傳完成時，關閉Loading畫面
    
    if ([self.sourceTempPathArray indexOfObject:filePath]==[self.sourceTempPathArray count]-1)
    {
        __block WCExportFlowController *blockSelf = self;
        dispatch_async(dispatch_get_main_queue(), ^{
            [[UIApplication sharedApplication] enableIdle:YES];
            [blockSelf showBusyView:NO];
            
            blockSelf.exportFlowError = error;
            
            NSString *alertMessage = [self errorMessageWithError:blockSelf.exportFlowError];
            
            PPAlertView *alertView = [[PPAlertView alloc] initWithTitle:@""
                                                                message:alertMessage
                                                               delegate:blockSelf
                                                      cancelButtonTitle:WCEFC_MLS_OK
                                                      otherButtonTitles:nil];
            [alertView show];
            [alertView release];
        });
    }
}


//================================================================================
//
//================================================================================
- (void)ppCloud:(PPCloud *)ppCloud uploadedFile:(NSString *)destPath from:(NSString *)srcPath metadata:(PPCloudMetadata *)metadata userInfo:(id)userInfo
{
    //////////////////////////////////////////////////
    // 刪除暫存檔
    
    NSError *fileError = nil;
    [[NSFileManager defaultManager] removeItemAtPath:srcPath error:&fileError];
    
    //////////////////////////////////////////////////
    // 最後一個文件上傳完成時，關閉Loading畫面
    
    if ([self.sourceTempPathArray indexOfObject:srcPath]==[self.sourceTempPathArray count]-1)
    {
        __block WCExportFlowController *blockSelf = self;
        dispatch_async(dispatch_get_main_queue(), ^{
            [PPBusyView postMessage:[NSString stringWithFormat:@"%@%.2f%@", WCEFC_MLS_Exporting, 100.00, @"%"]];
            [[UIApplication sharedApplication] enableIdle:YES];
            [blockSelf showBusyView:NO];
            
            PPAlertView *alertView = [[PPAlertView alloc] initWithTitle:@""
                                                                message:WCEFC_MLS_ExportSuccess
                                                               delegate:blockSelf
                                                      cancelButtonTitle:WCEFC_MLS_OK
                                                      otherButtonTitles:nil];
            [alertView show];
            [alertView release];
        });
    }
}


//================================================================================
//
//================================================================================
- (void)ppCloud:(PPCloud *)ppCloud uploadProgress:(CGFloat)progress forFile:(NSString *)destPath from:(NSString *)srcPath userInfo:(id)userInfo
{
    __block WCExportFlowController *blockSelf = self;
    dispatch_async(dispatch_get_main_queue(), ^{
        NSInteger srcIndex  = [blockSelf.sourceTempPathArray indexOfObject:srcPath];
        CGFloat segmentSize = 100.0/[blockSelf.sourceTempPathArray count];
        
        CGFloat totalProgress = (srcIndex+progress)*segmentSize;
        
        [PPBusyView postMessage:[NSString stringWithFormat:@"%@%.2f%@", WCEFC_MLS_Exporting, totalProgress, @"%"]];
    });
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPCloudDelegate_CreateFolder

//================================================================================
//
//================================================================================
- (void)ppCloud:(PPCloud *)ppCloud createFolderFailedWithError:(NSError *)error userInfo:(id)userInfo
{

}


//================================================================================
// 
//================================================================================
- (void)ppCloud:(PPCloud *)ppCloud createdFolder:(PPCloudMetadata *)folder userInfo:(id)userInfo
{
    
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UIAlertViewDelegate

//================================================================================
//
//================================================================================
- (void)alertView:(UIAlertView *)alertView didDismissWithButtonIndex:(NSInteger)buttonIndex
{
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(exportFlowController:didFinishWithError:)]==YES)
    {
        if ([alertView.message isEqualToString:WCEFC_MLS_ExportSuccess])
        {
            __block WCExportFlowController *blockSelf = self;
            
            if (alertView.tag ==WCExportFlowControllerAlertTag_Salesforce)
            {
                [blockSelf.delegate exportFlowController:blockSelf didFinishWithError:nil];
            }
            
            if (self.exportViewController!=nil)
            {
                [self.exportViewController dismissAnimated:YES completion:^{
                    
                    
                    if(self.storageSpaceSelectViewController)
                    {
                        [self.storageSpaceSelectViewController goBackAnimated:YES];
                        self.storageSpaceSelectViewController = nil;
                    }
                    
                    [blockSelf.delegate exportFlowController:blockSelf didFinishWithError:nil];
                }];
            }
            
            if (self.dismissGroupViewController!=nil)
            {
                if(self.dismissGroupViewController.navigationController)
                {
                    
                    [self.dismissGroupViewController.navigationController dismissAnimated:YES completion:^{
                        
                        
                        if(self.storageSpaceSelectViewController)
                        {
                            [self.storageSpaceSelectViewController goBackAnimated:YES];
                            self.storageSpaceSelectViewController = nil;
                        }
                        [blockSelf.delegate exportFlowController:blockSelf didFinishWithError:nil];
                    }];
                }
                else
                {
                    [self.dismissGroupViewController dismissAnimated:YES completion:^{
                    
                        
                        if(self.storageSpaceSelectViewController)
                        {
                            [self.storageSpaceSelectViewController goBackAnimated:YES];
                            self.storageSpaceSelectViewController = nil;
                        }
                        [blockSelf.delegate exportFlowController:blockSelf didFinishWithError:nil];
                    }];
                }
            }

        }
        else
        {
            [self.delegate exportFlowController:self didFinishWithError:self.exportFlowError];
        }
    }
}

@end
