//
//  ContactInfoViewController.m
//  WCMContactInfoViewController
//
//  Created by Andrew on 2015/5/20.
//  Copyright (c) 2015年 Andrew.Lai. All rights reserved.
//
//  NOTE:
//  -----------------------------------------
//  1. contactNavigationBarView.titleLabel的顯示位置需要當作移動效果的終點，
//     但一開始如果用hidden隱藏，取得的位置會錯誤，所以必須要用alpha值控制。
//

#import "ContactInfoViewController.h"

// Define
#import "WCAppearanceDefine.h"
#import "WCContactViewController+ResourceDefine.h"
#import "WCContactViewController+PrivateParameterDefine.h"
#import "WCContactViewController+PublicParameterDefine.h"
#import "WCFieldDefine.h"
#import "ContactApperanceDefine.h"

// Model
#import "NSString+Additions.h"
#import "PPContactRowObject.h"
#import "PPRowModel+Contact.h"
#import "UIColor+Contact.h"
#import "ContactDataExchangeModel+DisplayPhoneSort.h"

// View
#import "PPCommentButton.h"
#import "PPTopButtonsActionSheet.h"
#import "ContactInfoLeftSideView.h"
#import "ContactInfoSwitchCardHeaderView.h"
#import "ContactTextViewTableViewCell.h"
#import "ContactTranslateNavigationBarView.h"
#import "PPTableView.h"
#import "ContactLabelTableViewCell.h"

// Controller
#import "PPLogController.h"
#import "PPContactInfoSectionController.h"
#import "PPSectionController+Contact.h"
#import "UIColor+HexString.h"
#import "WCDisplayNameController.h"
#import "PPSelectController.h"
#import "WCToastController.h"

////////////////////////////////////////////////////////////////////////////////////////////////////
// 參照ContactTextViewTableViewCell和ContactLabelTableViewCell所制定的Inset

#define ContactInfoViewController_TableViewSeparatorInset UIEdgeInsetsMake(0, 60, 0, 0)
#define ContactInfoViewController_TableViewGroupAccessoryViewWidth 30

////////////////////////////////////////////////////////////////////////////////////////////////////
// 使用Cell的Tag紀錄ShortPress事件(section和一些不顯示設為預設)

typedef NS_ENUM(NSUInteger, ContactInfoViewControllerCellShortPressEvent)
{
    ContactInfoViewControllerCellShortPressEvent_HiddenFirstPriority = 0,
    ContactInfoViewControllerCellShortPressEvent_ShowFirstPriority
};

////////////////////////////////////////////////////////////////////////////////////////////////////

static CGFloat const ContactInfoViewController_TableSectionEdgeHeight   = 1.0f;
static CGFloat const ContactInfoViewController_TableSectionHeaderHeight = 5.0f;
static CGFloat const ContactInfoViewController_MaximumScrollOffsetX     = 20.0f;

static NSUInteger const ContactInfoViewController_TableCellImageViewSize           = 24;
static NSUInteger const ContactInfoViewController_TableCellImageViewMargin         = 15;
static NSUInteger const ContactInfoViewController_TableCellTextLabelFontSize       = 16;
static NSUInteger const ContactInfoViewController_TableCellDetailTextLabelFontSize = 12;
static NSUInteger const ContactInfoViewController_TableTextViewCellHeight          = 100;

#define ContactInfoViewController_DefaultContactName [@"MLS_NoName" localized]
#define ContactInfoViewController_SaveAsFormat WCCV_MLS_SaveAs

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - ContactInfoViewController Extension

@interface ContactInfoViewController () <PPLabelTableViewCellDelegate, PPButtonDelegate, PPTopButtonsActionSheetDelegate, PPSelectControllerDelegate, ContactInfoNavigationBarViewDelegate, ContactTranslateNavigationBarViewDelegate, ContactInfoHeaderViewDelegate, ContactInfoLeftSideViewDelegate, UITableViewDataSource, UITableViewDelegate>
@property (nonatomic, assign) BOOL                               isContactCardSwitchShowed;
@property (nonatomic, assign) BOOL                               haveLongPressCell;
@property (nonatomic, assign) BOOL                               updateOffset; //手指放掉往上滑動時，避免卡在自動縮起的位置
@property (nonatomic, retain) CAGradientLayer                   *gradientLayer;
@property (nonatomic, retain) PPRowModel                        *contactNameRowModel;
@property (nonatomic, retain) NSOperationQueue                  *fetchInfoOperationQueue;
@property (nonatomic, retain) NSOperationQueue                  *fetchHeadOperationQueue;
@property (nonatomic, retain) NSOperationQueue                  *fetchCardOperationQueue;

@property (nonatomic, retain) ContactInfoLeftSideView           *contactInfoLeftSideView;
@property (nonatomic, retain) ContactInfoSwitchCardHeaderView   *contactInfoSwitchCardHeaderView;
@property (nonatomic, retain) PPTableView                       *contactInfoTableView;
@property (nonatomic, retain) PPLabelTableViewCell              *currentPressLabelCell;

@property (nonatomic, retain) PPTopButtonsActionSheet           *ppTopButtonsActionSheet;
@property (nonatomic, retain) PPNavigationBarView               *contactNavigationBarView;
@property (nonatomic, retain) PPBusyView                        *busyView;

@property (nonatomic, retain) NSArray                           *prepareOrders;
@property (nonatomic, retain) NSDictionary                      *prepareFieldDictionay;
@property (nonatomic, retain) PPContactInfoSectionController    *contactInfoSectionController;
@property (nonatomic, retain) PPSelectController                *selectController;

@property (nonatomic, assign) CGPoint nameLabelBeginAnchor;       // nameLabel起始的參考點
@property (nonatomic, assign) CGSize  nameLabelBeginSize;         // nameLabel起始的長寬大小
@property (nonatomic, assign) CGFloat nameLabelFontScaleDiff;     // nameLabel文字大小的變化量
@property (nonatomic, assign) CGPoint nameLabelAnchorMoveDiff;    // nameLabelAnchor位置的變化量
@property (nonatomic, assign) CGSize  nameLabelSizeMoveDiff;      // nameLabel長寬大小的變化量
@property (nonatomic, assign) CGFloat changeNavigationBarOffsetY; // navBar切換為透明/不透明的位移量
@property (nonatomic, assign) CGFloat changeInfoLayoutOffsetY;    // nameLabel/IDPhoto開始變化的位移量(自動縮起)

@property (atomic, assign) BOOL                                    isDeallocing;

@end





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - ContactInfoViewController Implementation

@implementation ContactInfoViewController

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Dellocating Objects

//================================================================================
//
//================================================================================
- (id)init
{
    if(self = [super init])
    {
        _contactInfoSectionController = [[PPContactInfoSectionController alloc] init];
        _interfaceStyle               = ContactInfoViewControllerInterfaceStyle_Browse;
        _interfaceIdom                = ContactInfoViewControllerUserInterfaceIdom_Phone;
        _contactNameRowModel          = nil;
        _prepareOrders                = nil;                
        _prepareFieldDictionay        = nil;
        _isBackButtonShowed           = YES;

        //////////////////////////////////////////////////

        _fetchInfoOperationQueue = [[NSOperationQueue alloc] init];
        if (self.fetchInfoOperationQueue!=nil)
        {
            [self.fetchInfoOperationQueue setMaxConcurrentOperationCount:1];
        }
        
        _fetchCardOperationQueue = [[NSOperationQueue alloc] init];
        if (self.fetchCardOperationQueue!=nil)
        {
            [self.fetchCardOperationQueue setMaxConcurrentOperationCount:1];
        }
        
        _fetchHeadOperationQueue = [[NSOperationQueue alloc] init];
        if (self.fetchHeadOperationQueue!=nil)
        {
            [self.fetchHeadOperationQueue setMaxConcurrentOperationCount:1];
        }
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    @synchronized(self)
    {
        self.isDeallocing = YES;
        self.contactImageProtocol = nil;
        self.contactDataExchangeProtocol = nil;
        self.dataSource = nil;
        self.delegate = nil;
        
        [[NSOperationQueue mainQueue] cancelAllOperations];
        
        //////////////////////////////////////////////////
        [_fetchInfoOperationQueue cancelAllOperations];
        [_fetchInfoOperationQueue release];
        _fetchInfoOperationQueue = nil;
        
        [_fetchCardOperationQueue cancelAllOperations];
        [_fetchCardOperationQueue release];
        _fetchCardOperationQueue = nil;
        
        [_fetchHeadOperationQueue cancelAllOperations];
        [_fetchHeadOperationQueue release];
        _fetchHeadOperationQueue = nil;
        
        [_contactNameRowModel release];
        _contactNameRowModel = nil;
        
        [_selectController release];
        _selectController = nil;
        
        [_contactInfoSectionController release];
        _contactInfoSectionController = nil;
        
        [_prepareFieldDictionay release];
        _prepareFieldDictionay = nil;
        
        [_busyView removeFromSuperview];
        [_busyView release];
        _busyView = nil;
        
        [_contactInfoSwitchCardHeaderView removeFromSuperview];
        _contactInfoSwitchCardHeaderView.delegate = nil;
        _contactInfoSwitchCardHeaderView = nil;
        
        [_contactInfoLeftSideView removeFromSuperview];
        _contactInfoLeftSideView.delegate = nil;
        _contactInfoLeftSideView = nil;
        
        [_gradientLayer removeFromSuperlayer];
        _gradientLayer.colors = nil;
        _gradientLayer = nil;
        
        [_contactNavigationBarView removeFromSuperview];
        _contactNavigationBarView = nil;
        
        [_contactInfoTableView removeFromSuperview];
        _contactInfoTableView.dataSource = nil;
        _contactInfoTableView.delegate = nil;
        _contactInfoTableView = nil;
        
        [_ppTopButtonsActionSheet removeFromSuperview];
        _ppTopButtonsActionSheet.delegate = nil;
        _ppTopButtonsActionSheet = nil;
        
        [_currentPressLabelCell removeFromSuperview];
        [_currentPressLabelCell release];
        _currentPressLabelCell = nil;
        
        [_prepareOrders release];
        _prepareOrders = nil;
        
        //////////////////////////////////////////////////
        
        [super dealloc];
    }
}


//================================================================================
//
//================================================================================
- (void)willAnimateRotationToInterfaceOrientation:(UIInterfaceOrientation)toInterfaceOrientation duration:(NSTimeInterval)duration
{
    [super willAnimateRotationToInterfaceOrientation:toInterfaceOrientation duration:duration];
    
    //////////////////////////////////////////////////
    //!! 因為InfoViewController指定橫式會呈現雙面，直式呈現單面
    //!! 必須在此才能重算出正確的滾動效果參數
    
    [self layoutSubviewsWithOrientation:toInterfaceOrientation];
}


//================================================================================
//
//================================================================================
- (void)didRotateFromInterfaceOrientation:(UIInterfaceOrientation)fromInterfaceOrientation
{
    [super didRotateFromInterfaceOrientation:fromInterfaceOrientation];
    
    //////////////////////////////////////////////////
    [self addTableFooterView];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Responding to View Events

//================================================================================
//
//================================================================================
- (void)viewWillAppear:(BOOL)animated
{
    PPLogFunction();
    [super viewWillAppear:animated];

    //////////////////////////////////////////////////
    // 當MenuController消失時，需將currentPressLabelCell設為nil
    
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(recvUIMenuControllerDidHideMenuNotification:)
                                                 name:UIMenuControllerDidHideMenuNotification
                                               object:nil];

    //////////////////////////////////////////////////
    // 判斷使用的navigationBar的類型
    
    if (self.interfaceStyle==ContactInfoViewControllerInterfaceStyle_Translate)
    {
        ContactTranslateNavigationBarView *contactNavigationBarView = [[ContactTranslateNavigationBarView alloc] initWithFrame:self.navigationController.navigationBar.bounds];
        if (contactNavigationBarView!=nil)
        {
            contactNavigationBarView.backgroundImageView.image  = nil;
            contactNavigationBarView.titleLabel.alpha           = 0.0;
            contactNavigationBarView.titleLabel.textColor       = [UIColor whiteColor];
            contactNavigationBarView.titleLabel.text            = ContactInfoViewController_DefaultContactName;
            contactNavigationBarView.titleLabelAutoresizingGap  = 0;
            
            if (self.dataSource!=nil && [self.dataSource respondsToSelector:@selector(translateLanguageArrayInContactInfoViewController:translateLocation:)]==YES)
            {
                NSArray *translateLanguages = [self.dataSource translateLanguageArrayInContactInfoViewController:self translateLocation:ContactInfoViewControllerTranslateLocation_Source];
                [contactNavigationBarView setSourceLanguageString:[translateLanguages objectAtIndex:0]];
                
                translateLanguages = [self.dataSource translateLanguageArrayInContactInfoViewController:self translateLocation:ContactInfoViewControllerTranslateLocation_Destination];
                [contactNavigationBarView setDestinationLanguageString:[translateLanguages objectAtIndex:0]];
            }
            
            contactNavigationBarView.delegate = self;
            self.navigationItem.leftBarButtonItem  = [[[UIBarButtonItem alloc] initWithCustomView:contactNavigationBarView] autorelease];
            
            self.contactNavigationBarView = contactNavigationBarView;
            [contactNavigationBarView release];
        }
    }
    else
    {
        ContactInfoNavigationBarView *contactNavigationBarView = nil;
        ContactInfoNavigationBarViewMode viewMode = ContactInfoNavigationBarViewMode_Normal;
        switch (self.interfaceStyle)
        {

            case ContactInfoViewControllerInterfaceStyle_DuplicateBrowse:
            {
                viewMode = ContactInfoNavigationBarViewMode_Duplicate;
                break;
            }
            case ContactInfoViewControllerInterfaceStyle_CompanyContacts:
            {
                viewMode = ContactInfoNavigationBarViewMode_Company;
                break;
            }
            case ContactInfoViewControllerInterfaceStyle_NearbyContacts:
            {
                viewMode = ContactInfoNavigationBarViewMode_Nearby;
                break;
            }
            case ContactInfoViewControllerInterfaceStyle_ContactsForSalesforce:
            {
                viewMode = ContactInfoNavigationBarViewMode_Salesforce;
                break;
            }
            case ContactInfoViewControllerInterfaceStyle_Browse:
            default:
                break;
        }
        
        contactNavigationBarView = [[ContactInfoNavigationBarView alloc] initWithFrame:self.navigationController.navigationBar.bounds
                                                                     navigationBarMode:viewMode
                                                                    isBackButtonShowed:self.isBackButtonShowed];

        if (contactNavigationBarView!=nil)
        {
            contactNavigationBarView.backgroundImageView.image  = nil;
            contactNavigationBarView.titleLabel.alpha           = 0.0;
            contactNavigationBarView.titleLabel.textColor       = [UIColor whiteColor];
            contactNavigationBarView.titleLabel.text            = ContactInfoViewController_DefaultContactName;
            contactNavigationBarView.titleLabelAutoresizingGap  = 0;
            
            contactNavigationBarView.delegate = self;
            self.navigationItem.leftBarButtonItem  = [[[UIBarButtonItem alloc] initWithCustomView:contactNavigationBarView] autorelease];
            
            self.contactNavigationBarView = contactNavigationBarView;
            [contactNavigationBarView release];
        }
    }
    
    //////////////////////////////////////////////////
    // 保留呈現效果的NavigationController
    
    [self updateCustomNavigationViewSettings];
    
    //////////////////////////////////////////////////
    // prepare table view
    
    _contactInfoTableView = [[PPTableView alloc] initWithFrame:CGRectZero];
    if (self.contactInfoTableView!=nil)
    {
        self.contactInfoTableView.backgroundColor = [UIColor whiteColor];
        self.contactInfoTableView.dataSource      = self;
        self.contactInfoTableView.delegate        = self;
        self.contactInfoTableView.separatorInset  = ContactInfoViewController_TableViewSeparatorInset;
        self.contactInfoTableView.separatorColor  = ContactInfoViewControllerSeparatorColorColor;
      
#ifdef __IPHONE_11_0
        if (@available(iOS 11.0, *)) {
            self.contactInfoTableView.contentInsetAdjustmentBehavior = UIScrollViewContentInsetAdjustmentNever;
        }
#endif

        //////////////////////////////////////////////////
        // 計算headerView的最大高度並進行設定(裡面包含大頭照和名片區域)
        // 名片圖為顯示畫面的0.35,大頭照區域高為96
        
        do
        {
            _contactInfoSwitchCardHeaderView = [[ContactInfoSwitchCardHeaderView alloc] initWithFrame:CGRectZero];
            if (self.contactInfoSwitchCardHeaderView==nil)
            {
                break;
            }
#if defined (PRODUCTLINE_WCSF)
            self.contactInfoSwitchCardHeaderView.showSaveAsLabel = YES;
#endif
            _contactInfoLeftSideView = [[ContactInfoLeftSideView alloc] initWithFrame:CGRectZero];
            if (self.contactInfoLeftSideView==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            if (self.interfaceStyle!=ContactInfoViewControllerInterfaceStyle_Browse&&
                self.interfaceStyle!=ContactInfoViewControllerInterfaceStyle_ContactsForSalesforce)
            {
                self.contactInfoLeftSideView.contactCardImageButton.enabled         = NO;
                self.contactInfoSwitchCardHeaderView.contactCardImageButton.enabled = NO;
            }
            
            //////////////////////////////////////////////////
            
            if (self.interfaceIdom==ContactInfoViewControllerUserInterfaceIdom_Phone &&
                UIInterfaceOrientationIsLandscape(self.interfaceOrientation)==YES)
            {
                self.contactImageProtocol                               = self.contactInfoLeftSideView;
                self.contactInfoTableView.tableHeaderView               = self.contactInfoSwitchCardHeaderView;
                self.contactInfoTableView.tableHeaderView.clipsToBounds = YES;
                
                [self.view addSubview:self.contactInfoLeftSideView];
            }
            else
            {
                self.contactImageProtocol                               = self.contactInfoSwitchCardHeaderView;
                self.contactInfoTableView.tableHeaderView               = self.contactInfoSwitchCardHeaderView;
                self.contactInfoTableView.tableHeaderView.clipsToBounds = YES;
            }
            
            [self.contactInfoSwitchCardHeaderView setDelegate:self];
            [self.contactInfoLeftSideView setDelegate:self];
            
            //////////////////////////////////////////////////
            
            [self prepareContactFieldDictionary];
            [self fetchContactFieldDictionaryWithReload:YES loadDictionaryCompleted:nil]; //因為改為先呈現介面進行Loading在更新資料內容
            [self fetchContactHeadImagesData];
            [self fetchContactCardImagesData];
            
            [self.contactInfoLeftSideView setNeedsLayout];
            [self.contactInfoSwitchCardHeaderView setNeedsLayout];
        }
        while (0);
        
        [self.view addSubview:self.contactInfoTableView];
    }
    
    [self layoutSubviewsWithOrientation:self.interfaceOrientation];
}


//================================================================================
//
//================================================================================
- (void)layoutSubviewsWithOrientation:(UIInterfaceOrientation)interfaceOrientation
{
    [super layoutSubviews];
    
    //////////////////////////////////////////////////
    //!! 若該View的大小為CGRectZero，則該View的later會是nil
    
    if(self.contactNavigationBarView.backgroundImageView.layer!=nil)
    {
        if (self.gradientLayer==nil)
        {
            self.gradientLayer        = [CAGradientLayer layer];
            self.gradientLayer.colors = [NSArray arrayWithObjects:(id)[[UIColor colorWithWhite:0.0f alpha:0.4f] CGColor],
                                         (id)[[UIColor colorWithWhite:0.0f alpha:0.0f] CGColor], nil]; // 由上到下的漸層顏色
            
            [self.contactNavigationBarView.backgroundImageView.layer insertSublayer:self.gradientLayer atIndex:0];
        }
        
        CGRect frame = self.contactNavigationBarView.backgroundImageView.bounds;
        frame.size.height = self.view.bounds.size.height/3;
        self.gradientLayer.frame = frame;
    }
    
    //////////////////////////////////////////////////
    // 重設畫面大小避免璇轉畫面跳動
    
    [self rotateToInterfaceOrientation:interfaceOrientation];
}


//================================================================================
//
//================================================================================
- (void)viewWillDisappear:(BOOL)animated
{
    PPLogFunction();
    [self.fetchCardOperationQueue cancelAllOperations];
    [self.fetchHeadOperationQueue cancelAllOperations];
    [self.fetchInfoOperationQueue cancelAllOperations];
    [[NSOperationQueue mainQueue] cancelAllOperations];

    //////////////////////////////////////////////////
    
    [super viewWillDisappear:animated];
}


//================================================================================
//
//================================================================================
- (void)viewDidDisappear:(BOOL)animated
{
    PPLogFunction();

    self.contactImageProtocol = nil;
    
    [self.contactInfoSwitchCardHeaderView removeFromSuperview];
    self.contactInfoSwitchCardHeaderView.delegate = nil;
    self.contactInfoSwitchCardHeaderView = nil;
    
    [self.contactInfoLeftSideView removeFromSuperview];
    self.contactInfoLeftSideView.delegate = nil;
    self.contactInfoLeftSideView = nil;
    
    [self.gradientLayer removeFromSuperlayer];
    self.gradientLayer = nil;
    
    [self.contactNavigationBarView removeFromSuperview];
    self.contactNavigationBarView = nil;
    
    [self.contactInfoTableView removeFromSuperview];
    self.contactInfoTableView.dataSource = nil;
    self.contactInfoTableView.delegate = nil;
    self.contactInfoTableView = nil;
    
    [self.ppTopButtonsActionSheet removeFromSuperview];
    self.ppTopButtonsActionSheet.delegate = nil;
    self.ppTopButtonsActionSheet = nil;
    
    //////////////////////////////////////////////////
    
    [super viewDidDisappear:animated];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Override Base Class Instance Method

//================================================================================
//
//================================================================================
- (void)fetchContactCardImagesData
{
    do
    {
//        UIImage *defaultCardImage = [self imageForPPContactMainType:PPContactMainType_FrontCardImage ppContactSubType:PPContactSubType_FrontCardImage_Default];
//        [self.contactInfoDoubleCardHeaderView setContactRearCardImage:defaultCardImage];
        
        //////////////////////////////////////////////////
        
        __block UIImage *rearCardImage = [self imageForPPContactMainType:PPContactMainType_BackCardImage
                                                        ppContactSubType:PPContactSubType_BackCardImage_Thumbnail];
        
        if(rearCardImage!=nil)
        {
            if ([self setImage:rearCardImage forPPContactMainType:PPContactMainType_BackCardImage ppContactSubType:PPContactSubType_BackCardImage_Thumbnail]==YES)
            {
                self.isContactCardSwitchShowed = YES;
                self.contactInfoSwitchCardHeaderView.contactCardSwitchButtonVisible = YES;
            }
        }

        //////////////////////////////////////////////////

        
        __block UIImage *frontCardImage = [self imageForPPContactMainType:PPContactMainType_FrontCardImage
                                                         ppContactSubType:PPContactSubType_FrontCardImage_Thumbnail];

        if(frontCardImage!=nil)
        {
            if ([self setImage:frontCardImage forPPContactMainType:PPContactMainType_FrontCardImage
              ppContactSubType:PPContactSubType_FrontCardImage_Thumbnail]==YES)
            {
                break;
            }
        }

        //////////////////////////////////////////////////
        
        // 無名片圖時，進行抓取的Loading
        if(self.interfaceIdom==ContactInfoViewControllerUserInterfaceIdom_Phone &&
           UIInterfaceOrientationIsLandscape(self.interfaceOrientation)==YES)
        {
            [self.contactInfoLeftSideView cardImageIndicatorViewVisible:YES];
        }
        else
        {
            [self.contactInfoSwitchCardHeaderView cardImageIndicatorViewVisible:YES];
        }
        
        //////////////////////////////////////////////////
        
        // 進行Background Thread抓取影像
        
        if(self.fetchCardOperationQueue==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        if (self.contactDataExchangeProtocol==nil ||
            [self.contactDataExchangeProtocol respondsToSelector:@selector(sender:imageForPPContactMainType:ppContactSubType:)]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        __block NSBlockOperation *blockOperation = [[[NSBlockOperation alloc] init] autorelease];
        
        if(blockOperation==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        __block typeof(self) blockself = self;
        
        [blockOperation addExecutionBlock:^{
            
            do
            {
                // !!讀取前就要重設，不然有可能造成後面被cancel掉，而沒有重設這個值
                blockself.isContactCardSwitchShowed = NO;

                __block BOOL thumbnailRearCardImageExist = YES;
                
                if (blockOperation.isCancelled==YES)
                {
                    break;
                }
                
                //////////////////////////////////////////////////

                //後名片圖
                rearCardImage = [[blockself.contactDataExchangeProtocol sender:blockself
                                                    imageForPPContactMainType:PPContactMainType_BackCardImage
                                                             ppContactSubType:PPContactSubType_BackCardImage_Thumbnail] retain];
                if (rearCardImage==nil)
                {
                    thumbnailRearCardImageExist = NO;
                }
                
                //////////////////////////////////////////////////
                
                BOOL thumbnailFrontCardImageExist = YES;
                BOOL defaultCardImageExist        = YES;
                UIImage *defaultCardImage         = nil;
                
                //前名片圖
                frontCardImage = [[blockself.contactDataExchangeProtocol sender:blockself
                                                      imageForPPContactMainType:PPContactMainType_FrontCardImage
                                                               ppContactSubType:PPContactSubType_FrontCardImage_Thumbnail] retain];
                if(frontCardImage==nil)
                {
                    thumbnailFrontCardImageExist = NO;
                    
                    if (blockself.delegate==nil ||
                        [blockself.delegate respondsToSelector:@selector(defaultCardImageInContactInfoViewController:)]==NO)
                    {
                        break;
                    }
                    
                    //////////////////////////////////////////////////

                    defaultCardImage = [[blockself.delegate defaultCardImageInContactInfoViewController:blockself] retain];
                  
                    if (defaultCardImage==nil)
                    {
                        defaultCardImageExist = NO;
                    }
                }
                
                //////////////////////////////////////////////////
                
                //!! 因為預設先顯示前名片圖，故前名片後放
                
                __block NSBlockOperation *blockMainOperation = [NSBlockOperation blockOperationWithBlock:^{
                    
                    do
                    {
                        //寫入假名片圖
                        if (defaultCardImageExist==YES)
                        {
                            if ([blockself setImage:defaultCardImage forPPContactMainType:PPContactMainType_FrontCardImage ppContactSubType:PPContactSubType_FrontCardImage_Default]==YES)
                            {
                                
                            }
                        }
                        else
                        {
                            if ([blockself setImage:defaultCardImage forPPContactMainType:PPContactMainType_FrontCardImage ppContactSubType:PPContactSubType_FrontCardImage_Default]==YES)
                            {
                                
                            }
                        }
                        
                        //////////////////////////////////////////////////
                        //寫入後名片圖
                        if (blockself.isDeallocing==YES)
                        {
                            break;
                        }
                        
                        if(thumbnailRearCardImageExist==YES &&
                           [blockself setImage:rearCardImage forPPContactMainType:PPContactMainType_BackCardImage ppContactSubType:PPContactSubType_BackCardImage_Thumbnail]==YES)
                        {
                            blockself.isContactCardSwitchShowed = YES;
                            blockself.contactInfoSwitchCardHeaderView.contactCardSwitchButtonVisible = YES;
                        }
                        else
                        {
                            //!! 避免被刪除後名片圖的流程
                            blockself.showFrontCard = YES;
                            blockself.isContactCardSwitchShowed = NO;
                            blockself.contactInfoSwitchCardHeaderView.contactCardSwitchButtonVisible = NO;
                        }
                        
                        //////////////////////////////////////////////////
                        //寫入前名片圖
                        
                        if (blockself.isDeallocing==YES)
                        {
                            break;
                        }
                        
                        if (thumbnailFrontCardImageExist==YES &&
                            [blockself setImage:frontCardImage forPPContactMainType:PPContactMainType_FrontCardImage ppContactSubType:PPContactSubType_FrontCardImage_Thumbnail]==YES)
                        {
                            
                        }
                        
                        //////////////////////////////////////////////////
                        if (blockself.isDeallocing==YES)
                        {
                            break;
                        }
                        
                        if (self.interfaceIdom==ContactInfoViewControllerUserInterfaceIdom_Phone &&
                            UIInterfaceOrientationIsLandscape(blockself.interfaceOrientation)==YES)
                        {
                            [blockself.contactInfoLeftSideView cardImageIndicatorViewVisible:NO];
                        }
                        else
                        {
                            [blockself.contactInfoSwitchCardHeaderView cardImageIndicatorViewVisible:NO];
                        }

                    }
                    while (0);
                    
                    //////////////////////////////////////////////////
                    
                    [defaultCardImage release];
                    [rearCardImage release];
                    [frontCardImage release];
                    
                }];
                
                //////////////////////////////////////////////////
                
                @synchronized (blockOperation)
                {
                    if(blockOperation.isCancelled==NO)
                    {
                        @synchronized([NSOperationQueue mainQueue])
                        {
                            [[NSOperationQueue mainQueue] addOperation:blockMainOperation];
                        }
                    }
                    else
                    {
                        [defaultCardImage release];
                        [rearCardImage release];
                        [frontCardImage release];
                    }
                }
            }
            while (0);

        }];
        
        //////////////////////////////////////////////////
        
        [self.fetchCardOperationQueue addOperation:blockOperation];

    } while (0);
}


//================================================================================
//
//================================================================================
- (void)fetchContactHeadImagesData
{
    do
    {
        __block UIImage *headImage = [self imageForPPContactMainType:PPContactMainType_HeadImage
                                                    ppContactSubType:PPContactSubType_HeadImage_Thumbnail];
        if (headImage!=nil)
        {
            if ([self setImage:headImage
          forPPContactMainType:PPContactMainType_HeadImage
              ppContactSubType:PPContactSubType_HeadImage_Thumbnail]==YES)
            {
                break;
            }
        }
        
        //////////////////////////////////////////////////
        // 無名片圖時，進行抓取的Loading
        
        [self.contactInfoSwitchCardHeaderView showDefaultHeadImage];
        [self.contactInfoSwitchCardHeaderView headImageIndicatorViewVisible:YES];
        
        //////////////////////////////////////////////////
        // 進行Background Thread抓取影像
        
        if(self.fetchHeadOperationQueue==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        if (self.contactDataExchangeProtocol==nil &&
            [self.contactDataExchangeProtocol respondsToSelector:@selector(sender:imageForPPContactMainType:ppContactSubType:)]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        __block NSBlockOperation *blockOperation = [[[NSBlockOperation alloc] init] autorelease];
        
        if(blockOperation==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        __block typeof(self) blockSelf = self;
        
        [blockOperation addExecutionBlock:^{
         
            __block BOOL thumbnailHeadImageExist = YES;
            
            do
            {
                //大頭照
                if (blockOperation.isCancelled==YES)
                {
                    break;
                }
                
                //////////////////////////////////////////////////

                headImage = [[blockSelf.contactDataExchangeProtocol sender:blockSelf
                                                 imageForPPContactMainType:PPContactMainType_HeadImage
                                                          ppContactSubType:PPContactSubType_HeadImage_Thumbnail] retain];
                
                // !!這邊要記錄userInfo, 後面setImage時，如果不同，就不設定，以免切換不同聯絡人時，會設定錯誤
                NSString *currentUserInfoObject = blockSelf.userInfoObject;
                
                if(headImage==nil)
                {
                    thumbnailHeadImageExist = NO;
                }
                
                //////////////////////////////////////////////////
                
                __block NSBlockOperation *blockMainOperation = [NSBlockOperation blockOperationWithBlock:^{
                    
                    do
                    {
                        //寫入大頭照
                        if (blockSelf.isDeallocing==YES)
                        {
                            break;
                        }
                        
                        
                        //////////////////////////////////////////////////

                        if ([currentUserInfoObject isEqual:blockSelf.userInfoObject])
                        {
                            if(thumbnailHeadImageExist==YES &&
                               [blockSelf setImage:headImage forPPContactMainType:PPContactMainType_HeadImage ppContactSubType:PPContactSubType_HeadImage_Thumbnail]==YES)
                            {
                                
                            }
                        }
                        
                        //////////////////////////////////////////////////
                        if (blockSelf.isDeallocing==YES)
                        {
                            break;
                        }
                        
                        [blockSelf.contactInfoSwitchCardHeaderView headImageIndicatorViewVisible:NO];
                    } 
                    while (0);
                    
                    //////////////////////////////////////////////////
                    
                    [headImage release];
                }];
                
                //////////////////////////////////////////////////

                @synchronized (blockOperation)
                {
                    if(blockOperation.isCancelled==NO)
                    {
                        @synchronized([NSOperationQueue mainQueue])
                        {
                            [[NSOperationQueue mainQueue] addOperation:blockMainOperation];
                        }
                    }
                    else
                    {
                        [headImage release];
                    }
                }
            }
            while (0);
        }];

        [self.fetchHeadOperationQueue addOperation:blockOperation];

    } while (0);
}


//================================================================================
//
//================================================================================
- (void)setContactHeadImage:(UIImage *)contactHeadImage
{
    [super setContactHeadImage:contactHeadImage];
    
    //////////////////////////////////////////////////
    
    [self.contactInfoSwitchCardHeaderView setContactHeadImage:contactHeadImage];
}


//================================================================================
//
//================================================================================
- (void)setContactName:(NSString *)contactName
{
    //!! HeaderLabel的sizeToFit於HeaderView內部
    [self.contactInfoSwitchCardHeaderView setContactName:contactName];
    
    self.contactNavigationBarView.titleLabel.text = contactName;
    
    // !! 讓contactInfoHeaderView & contactInfoNavigationBarView進行更新
    [self.view setNeedsLayout];
}


//================================================================================
//
//================================================================================
- (void)setSaveAsValue:(NSString *)saveAsValue
{
    NSString *saveAsString = [ContactInfoViewController_SaveAsFormat stringByAppendingFormat:@" %@", saveAsValue];
    NSMutableAttributedString* attrString = [[NSMutableAttributedString  alloc] initWithString:saveAsString];
    NSMutableParagraphStyle *style = [[NSMutableParagraphStyle alloc] init];
    [style setFirstLineHeadIndent:15];
    [attrString addAttribute:NSParagraphStyleAttributeName
                       value:style
                       range:NSMakeRange(0, [attrString length])];
    [style release];
    
    [self.contactInfoSwitchCardHeaderView.saveAsLabel setAttributedText:attrString];
    [attrString release];
}

//================================================================================
//
//================================================================================
- (void)setDataState:(WCContactViewControllerDataState)dataState
{
    [super setDataState:dataState];
    
    //////////////////////////////////////////////////
    
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_main_queue(), ^{
        switch (dataState)
        {
            case WCContactViewControllerDataState_None:
            case WCContactViewControllerDataState_Exist:
            {
                [blockSelf.contactInfoSwitchCardHeaderView emptyErrorMessageLabelVisible:NO];
                [blockSelf.contactInfoLeftSideView emptyErrorMessageLabelVisible:NO];
                
                break;
            }
            case WCContactViewControllerDataState_NotExist:
            {
                [blockSelf.contactInfoSwitchCardHeaderView emptyErrorMessageLabelVisible:YES];
                [blockSelf.contactInfoLeftSideView emptyErrorMessageLabelVisible:YES];
                
                break;
            }
            default:
            {
                [blockSelf.contactInfoSwitchCardHeaderView emptyErrorMessageLabelVisible:NO];
                [blockSelf.contactInfoLeftSideView emptyErrorMessageLabelVisible:NO];
                
                break;
            }
        }
    });
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Method

//================================================================================
// 為了使上下滾動正常，需針對少量資料額外添加空白footerView
// 這邊不要在codedispatch出去做，不然關閉時還有機會進入，
// 要dispatch到main thread由call的地方控制
//================================================================================
- (void)addTableFooterView
{
    @synchronized(self)
    {
        __block ContactInfoViewController *blockSelf = self;
            do
            {
                self.contactInfoTableView.estimatedRowHeight = 0;
                self.contactInfoTableView.estimatedSectionHeaderHeight = 0;
                self.contactInfoTableView.estimatedSectionFooterHeight = 0;
                
                self.contactInfoTableView.tableFooterView = nil;

                //////////////////////////////////////////////////
                
                CGFloat navBarHeight = self.navigationController.navigationBar.frame.size.height;
                PPLogFunction(@"NavigationBarHeight: %f", navBarHeight);
                
                CGFloat tableContentHeight = self.contactInfoTableView.contentSize.height;
                NSInteger tableInfoHeight    = tableContentHeight-blockSelf.contactInfoTableView.tableHeaderView.bounds.size.height;
                NSInteger tableHeight        = self.contactInfoTableView.bounds.size.height;
                PPLogFunction(@"Contain Table HeaderView ContentSize: %@", NSStringFromCGSize(self.contactInfoTableView.contentSize));
                PPLogFunction(@"Table HeaderView Height: %f", self.contactInfoTableView.tableHeaderView.bounds.size.height);
                PPLogFunction(@"Without Table HeaderView Only Info ContentSize: %d", tableInfoHeight);
                PPLogFunction(@"TableView Height: %d", tableHeight);

                if (tableInfoHeight<tableHeight)
                {
                    NSInteger footerHeight = tableHeight-navBarHeight-tableInfoHeight%tableHeight;
                    PPLogFunction(@"FooterView Height: %d", footerHeight);
                    
                    UIView *footView = [[UIView alloc] initWithFrame:CGRectMake(0, 0, blockSelf.view.bounds.size.width, footerHeight)];
                    if (footView!=nil)
                    {
                        footView.backgroundColor = [UIColor clearColor];
                        blockSelf.contactInfoTableView.tableFooterView = footView;
                        [footView release];
                    }
                }
            } while(0);
    }
}
                                                    

//================================================================================
// 取得lable的參考點 （水平左側, 垂直中間）
//================================================================================
- (CGPoint)anchorPointWithLable:(UILabel *)lable
{
    CGPoint anchorPoint = CGPointMake(lable.frame.origin.x, lable.center.y);

    return anchorPoint;
}


//================================================================================
// 負責處理字串的複製
//================================================================================
- (void)copyActionWithString:(NSString *)copyString
{
    UIPasteboard *pasteboard = [UIPasteboard generalPasteboard];
    
    //////////////////////////////////////////////////
    
    if (copyString!=nil)
    {
        [pasteboard setString:copyString];
    }
    
    //////////////////////////////////////////////////
    
    [WCToastController showMessageToastFromSuperView:self.view
                                         withMessage:[@"MLS_CopyInformationTextToClipboard" localizedFromTable:@"WCContactViewController"]
                                            position:PPToastPositionCenter];
}


//================================================================================
// 移動label到參考點
//================================================================================
- (void)moveLable:(UILabel *)lable toAnchorPoint:(CGPoint)anchorPoint
{
    CGPoint curAnchorPoint = [self anchorPointWithLable:lable];
    CGPoint offset = CGPointMake(anchorPoint.x-curAnchorPoint.x,
                                 anchorPoint.y-curAnchorPoint.y);
    
    CGRect frame = lable.frame;
    frame.origin.x += offset.x;
    frame.origin.y += offset.y;
    lable.frame = frame;
}


//================================================================================
//
//================================================================================
- (void)resetEffectParameter
{
    __block ContactInfoViewController *blockSelf = self;
    
//    __block NSBlockOperation *blockMainOperation = [NSBlockOperation blockOperationWithBlock:^{
        
        //////////////////////////////////////////////////
        // 計算文字大小變化
        
        CGFloat beginFontSize = 0.0;
        CGSize beginLabelSize = CGSizeZero;
        CGPoint beginAnchor   = CGPointZero;
        CGFloat navBarHeight  = blockSelf.navigationController.navigationBar.frame.size.height;
        
        beginFontSize  = blockSelf.contactInfoSwitchCardHeaderView.contactNameLabel.font.pointSize;
        beginLabelSize = blockSelf.contactInfoSwitchCardHeaderView.contactNameLabel.frame.size;
        
        blockSelf.nameLabelBeginAnchor = [blockSelf anchorPointWithLable:blockSelf.contactInfoSwitchCardHeaderView.contactNameLabel];
        
        // !! 因為label的位置轉為window座標後可能是負值，所以這邊直接用status bar來計算
        // 這邊的值應該與第一次進入一樣，不能因為scroll而有不同的結果
        if([[UIApplication sharedApplication] isStatusBarHidden]==NO)
        {
            beginAnchor = blockSelf.nameLabelBeginAnchor;
            beginAnchor.y += MIN([UIApplication sharedApplication].statusBarFrame.size.width,
                                                    [UIApplication sharedApplication].statusBarFrame.size.height);
        }

        //////////////////////////////////////////////////
        // 計算變化效果臨界點
        
        //!! 要變為呈現navigationBar的時機點offset
        blockSelf.changeNavigationBarOffsetY = blockSelf.contactInfoSwitchCardHeaderView.bounds.size.height-navBarHeight;
        //!! 要進行大頭照和名稱的Layout變動的時機點offset
        blockSelf.changeInfoLayoutOffsetY = blockSelf.contactInfoSwitchCardHeaderView.frame.size.height/2;
        
        //////////////////////////////////////////////////
        
        CGFloat endFontSize              = blockSelf.contactNavigationBarView.titleLabel.font.pointSize;
        blockSelf.nameLabelFontScaleDiff = (endFontSize/beginFontSize) - 1.0;
        
        //////////////////////////////////////////////////
        // 紀錄NameLabel初始的寬度，並計算長寬大小變化
        
        CGSize endLabelSize             = blockSelf.contactNavigationBarView.titleLabel.frame.size;
        blockSelf.nameLabelBeginSize    = beginLabelSize;
        blockSelf.nameLabelSizeMoveDiff = CGSizeMake(endLabelSize.width-beginLabelSize.width, endLabelSize.height-beginLabelSize.height);

        //////////////////////////////////////////////////
        // 計算Anchor位移變化（轉換成window座標計算）
        
        CGPoint endAnchor = [blockSelf anchorPointWithLable:blockSelf.contactNavigationBarView.titleLabel];
        
        if([[UIApplication sharedApplication] isStatusBarHidden]==NO)
        {
            endAnchor.y += MIN([UIApplication sharedApplication].statusBarFrame.size.width,
                                                    [UIApplication sharedApplication].statusBarFrame.size.height);
        }

        //!! 因垂直方向offset和座標相反，所以要*(-1)
        blockSelf.nameLabelAnchorMoveDiff = CGPointMake(endAnchor.x-beginAnchor.x, (endAnchor.y-beginAnchor.y)*(-1));
//    }];
//    
//    @synchronized([NSOperationQueue mainQueue])
//    {
//        [[NSOperationQueue mainQueue] addOperation:blockMainOperation];
//    }
}


//================================================================================
//
//================================================================================
- (void)updateHeaderSubviewsWithOffset:(CGPoint)newOffset
{
    CGFloat navBarHeight = self.navigationController.navigationBar.frame.size.height;
    CGFloat statusBarHeight = [UIApplication sharedApplication].statusBarFrame.size.height;
    
    //////////////////////////////////////////////////
    
    if(newOffset.y <= 0)
    {
        //////////////////////////////////////////////////
        // tabelView在最上方時(offset.y == 0)往下drag, 回復初始值
        
        self.contactInfoTableView.contentInset = UIEdgeInsetsMake(0, 0, 0, 0);
        
        if (self.interfaceIdom==ContactInfoViewControllerUserInterfaceIdom_Phone &&
            UIInterfaceOrientationIsLandscape(self.interfaceOrientation)==YES)
        {
            CGRect cardFrame   = self.contactInfoLeftSideView.contactCardImageButton.frame;
            cardFrame.origin.y = 0;
            self.contactInfoLeftSideView.contactCardImageButton.frame = cardFrame;
        }
        else
        {
            //  MARK: scroll時調整cardImageButton的位置，以產生效果
            [[self contactInfoSwitchCardHeaderView] setCardImagePositionY:0];
        }
        
        self.contactInfoSwitchCardHeaderView.contactHeadImageButton.alpha     = 1.0;
        self.contactInfoSwitchCardHeaderView.contactHeadImageButton.transform = CGAffineTransformIdentity;
        
        //contactNameLabel初值
        CGRect frame = self.contactInfoSwitchCardHeaderView.contactNameLabel.frame;
        frame.size   = self.nameLabelBeginSize;
        self.contactInfoSwitchCardHeaderView.contactNameLabel.frame = frame;
        
        self.contactInfoSwitchCardHeaderView.contactNameLabel.font  = [UIFont boldSystemFontOfSize:ContactInfoHeaderView_ContactNameLabelMaxFont];
        [self moveLable:self.contactInfoSwitchCardHeaderView.contactNameLabel toAnchorPoint:self.nameLabelBeginAnchor];
        [self.contactInfoSwitchCardHeaderView setContactCardMaskAlpha:0.0 switchButtonHidden:NO];

        //////////////////////////////////////////////////
        
        [self.contactNavigationBarView.backgroundImageView setAlpha:1.0];
        
        //------------------------------------------------
        // 若為翻譯聯絡人介面，需變更上方翻譯工具列
        
        if (self.interfaceStyle==ContactInfoViewControllerInterfaceStyle_Translate)
        {
            [(ContactTranslateNavigationBarView *)self.contactNavigationBarView showTranslateToolBar:YES];
        }
        
        //////////////////////////////////////////////////
        // 顯示透明背景
        
        if(self.contactNavigationBarView.titleLabel.alpha == 1.0)
        {
            [self.navigationController.navigationBar setBackgroundImage:[[UIImage new] autorelease] forBarMetrics:UIBarMetricsDefault];
            [self.navigationController.navigationBar setShadowImage:[[UIImage new] autorelease]];
            [self.navigationController.navigationBar setBackgroundColor:[UIColor clearColor]];
            [self.navigationController.navigationBar setBarTintColor:[UIColor clearColor]];
           [[[self.navigationController.navigationBar subviews] firstObject] setBackgroundColor:[UIColor clearColor]];
            
            self.contactNavigationBarView.titleLabel.alpha = 0.0;
            self.contactInfoSwitchCardHeaderView.contactHeadImageButton.alpha = 1.0;
            
            // !! navigationBar.translucent會影響到顯示顏色，所以bar透明時需設為YES，有顏色時需設為NO。
            //    translucent設定同時會影響view的起始Y座標，所以要配合調整。
            self.navigationController.navigationBar.translucent = YES;
            self.contactInfoTableView.frame = CGRectMake(0, statusBarHeight, self.view.bounds.size.width, self.view.bounds.size.height-statusBarHeight);
        }
    }
    else
    {
        if(newOffset.y >= self.contactInfoTableView.tableHeaderView.frame.size.height-navBarHeight)
        {
            [self.contactNavigationBarView.backgroundImageView setAlpha:0.0];
            
            //////////////////////////////////////////////////
            // 移動headerView的下緣到‘高於’navigationBar下緣的位置

            self.contactInfoTableView.contentInset = UIEdgeInsetsMake(0, 0, 0, 0);
            
            if(self.contactNavigationBarView.titleLabel.alpha == 0.0)
            {
                [self.navigationController.navigationBar setBackgroundImage:nil forBarMetrics:UIBarMetricsDefault];
                [self.navigationController.navigationBar setShadowImage:nil];
                self.navigationController.navigationBar.barTintColor = WCAppearanceDefine_NavigationBarColor;
                [[[self.navigationController.navigationBar subviews] firstObject] setBackgroundColor:WCAppearanceDefine_NavigationBarColor];
                
                self.contactNavigationBarView.titleLabel.alpha = 1.0;
                
                // !! navigationBar.translucent會影響到顯示顏色，所以bar透明時需設為YES，有顏色時需設為NO。
                //    translucent設定同時會影響view的起始Y座標，所以要配合調整。
                self.navigationController.navigationBar.translucent = NO;
                self.contactInfoTableView.frame = CGRectMake(0, -navBarHeight, self.view.bounds.size.width, self.view.bounds.size.height+navBarHeight);
            }
        }
        else
        {
            //////////////////////////////////////////////////
            // 移動headerView的下緣到‘低於’navigationBar下緣的位置
            
            //------------------------------------------------
            // 重設inset阻止tableView回彈
            
            if (self.contactInfoTableView.isDecelerating==NO)
            {
                if (newOffset.y<=self.contactInfoTableView.tableHeaderView.frame.size.height-navBarHeight-statusBarHeight)
                {
                    self.contactInfoTableView.contentInset = UIEdgeInsetsMake(-newOffset.y, 0, 0, 0);
                }
            }
            
            //------------------------------------------------
            // 處理navigationBar
            
            if(newOffset.y < self.changeNavigationBarOffsetY)
            {
                // 顯示透明背景
                if(self.contactNavigationBarView.titleLabel.alpha == 1.0)
                {
                    [self.navigationController.navigationBar setBackgroundImage:[[UIImage new] autorelease] forBarMetrics:UIBarMetricsDefault];
                    [self.navigationController.navigationBar setShadowImage:[[UIImage new] autorelease]];
                    [self.navigationController.navigationBar setBackgroundColor:[UIColor clearColor]];
                    [self.navigationController.navigationBar setBarTintColor:[UIColor clearColor]];
                 
                    self.contactNavigationBarView.titleLabel.alpha = 0.0;
                    
                    
                    // !! navigationBar.translucent會影響到顯示顏色，所以bar透明時需設為YES，有顏色時需設為NO。
                    //    translucent設定同時會影響view的起始Y座標，所以要配合調整。
                    self.navigationController.navigationBar.translucent = YES;
                    [self.navigationController.navigationBar subviews].firstObject.backgroundColor = [UIColor clearColor];
                    
        
                    self.contactInfoTableView.frame = CGRectMake(0, statusBarHeight, self.view.bounds.size.width, self.view.bounds.size.height-statusBarHeight);
                }
            }
            
            //------------------------------------------------
            // 處理subViews顯示效果
            
            //!! 為了避免HeadImage與NameLabel重疊現象，所以變換距離除以2
            CGFloat alphaChange = (newOffset.y-self.changeInfoLayoutOffsetY)/((self.changeInfoLayoutOffsetY-navBarHeight-statusBarHeight)/2);
            CGFloat changeRate  = newOffset.y/self.nameLabelAnchorMoveDiff.y;
            
            if (self.interfaceIdom==ContactInfoViewControllerUserInterfaceIdom_Phone &&
                UIInterfaceOrientationIsLandscape(self.interfaceOrientation)==YES)
            {
                // update cardImage frame
                CGRect frame   = self.contactInfoLeftSideView.contactCardImageButton.frame;
                frame.origin.y = newOffset.y - changeRate*100;
                self.contactInfoLeftSideView.contactCardImageButton.frame = frame;
            }
            else
            {
                // update cardImage frame
                //  MARK: scroll時調整cardImageButton的位置，以產生效果
                [[self contactInfoSwitchCardHeaderView] setCardImagePositionY:newOffset.y - changeRate*100];
            }
            
            // update mask alpha
            [self.contactInfoSwitchCardHeaderView setContactCardMaskAlpha:changeRate switchButtonHidden:NO];
            
            // update IDPhoto
            self.contactInfoSwitchCardHeaderView.contactHeadImageButton.alpha = 1.0-alphaChange;
            
            // update navigationbar alpha
            [self.contactNavigationBarView.backgroundImageView setAlpha:1.0-changeRate];
            
            // !! name/IDPhoto是位移一段距離後才開始變化位置
            if(newOffset.y > self.changeInfoLayoutOffsetY)
            {
                // update mask alpha
                [self.contactInfoSwitchCardHeaderView setContactCardMaskAlpha:changeRate switchButtonHidden:YES];
                
                //!! reset changeRate
                //!! (目前所在高度量-Layout進行變換的高度量)/(Label位置高度變化量-Layout進行變換的高度量)＝(已進行Layout變換的高度量/需進行Layout變換的高度變化量)
                changeRate = (newOffset.y-self.changeInfoLayoutOffsetY)/(self.nameLabelAnchorMoveDiff.y-self.changeInfoLayoutOffsetY);
                
                // !! 提早將nameLable縮放到目標位置，減少navigationBar顯示title時跳動的感覺。
                changeRate = fminf(1.0, changeRate*1.3);
                
                // update IDPhoto
                CGFloat scale = fmax(0.0, 1.0 - changeRate);
                CGAffineTransform transform = CGAffineTransformScale(CGAffineTransformIdentity, scale, scale);
                
                self.contactInfoSwitchCardHeaderView.contactHeadImageButton.transform = transform;
                
                // update nameLable
                if(newOffset.y < self.nameLabelAnchorMoveDiff.y)
                {
                    //////////////////////////////////////////////////
                    // nameLabel往上拖曳未達navigationTitle位置
                    
                    scale = 1.0 + self.nameLabelFontScaleDiff * changeRate;
                    
                    // 縮放文字大小
                    self.contactInfoSwitchCardHeaderView.contactNameLabel.font = [UIFont boldSystemFontOfSize:scale*ContactInfoHeaderView_ContactNameLabelMaxFont];
                    
                    // 縮放Label長寬大小
                    CGRect nameLabelFrame = self.contactInfoSwitchCardHeaderView.contactNameLabel.frame;
                    CGSize sizeDiff       = CGSizeMake(self.nameLabelSizeMoveDiff.width*changeRate, self.nameLabelSizeMoveDiff.height*changeRate);
                    CGSize nameLabelSize  = CGSizeMake(self.nameLabelBeginSize.width+sizeDiff.width, self.nameLabelBeginSize.height+sizeDiff.height);
                    nameLabelFrame.size   = nameLabelSize;
                    self.contactInfoSwitchCardHeaderView.contactNameLabel.frame = nameLabelFrame;
                    
                    // 水平移動
                    CGPoint newPoint = CGPointMake(self.nameLabelBeginAnchor.x + self.nameLabelAnchorMoveDiff.x*changeRate, self.nameLabelBeginAnchor.y);
                    [self moveLable:self.contactInfoSwitchCardHeaderView.contactNameLabel toAnchorPoint:newPoint];
                }
                else
                {
                    //////////////////////////////////////////////////
                    // nameLabel往上拖曳超過navigationTitle位置，垂直移動
                    
                    // 縮放文字大小
                    self.contactInfoSwitchCardHeaderView.contactNameLabel.font = [UIFont boldSystemFontOfSize:(1.0 + self.nameLabelFontScaleDiff)*ContactInfoHeaderView_ContactNameLabelMaxFont];
                    
                    // 縮放Label長寬大小
                    CGRect nameLabelFrame = self.contactInfoSwitchCardHeaderView.contactNameLabel.frame;
                    CGSize nameLabelSize  = CGSizeMake(self.nameLabelBeginSize.width+self.nameLabelSizeMoveDiff.width, self.nameLabelBeginSize.height+self.nameLabelSizeMoveDiff.height);
                    nameLabelFrame.size   = nameLabelSize;
                    self.contactInfoSwitchCardHeaderView.contactNameLabel.frame = nameLabelFrame;
                    
                    CGPoint newPoint = CGPointMake(self.nameLabelBeginAnchor.x+self.nameLabelAnchorMoveDiff.x, self.nameLabelBeginAnchor.y + (newOffset.y - self.nameLabelAnchorMoveDiff.y));
                    [self moveLable:self.contactInfoSwitchCardHeaderView.contactNameLabel toAnchorPoint:newPoint];
                }
            }
            else
            {
                // 一開始移動值小於changeInfoLayoutOffsetY，不用改變大小，位置
                self.contactInfoSwitchCardHeaderView.contactHeadImageButton.transform = CGAffineTransformIdentity;
                
                CGRect labelFrame = self.contactInfoSwitchCardHeaderView.contactNameLabel.frame;
                labelFrame.size.width = self.nameLabelBeginSize.width;
                self.contactInfoSwitchCardHeaderView.contactNameLabel.frame = labelFrame;
                
                self.contactInfoSwitchCardHeaderView.contactNameLabel.font  = [UIFont boldSystemFontOfSize:ContactInfoHeaderView_ContactNameLabelMaxFont];
                [self moveLable:self.contactInfoSwitchCardHeaderView.contactNameLabel toAnchorPoint:self.nameLabelBeginAnchor];
            }
            
            //------------------------------------------------
            // 若為翻譯聯絡人介面，需變更上方翻譯工具列
            
            if (self.interfaceStyle==ContactInfoViewControllerInterfaceStyle_Translate)
            {
                [(ContactTranslateNavigationBarView *)self.contactNavigationBarView showTranslateToolBar:NO];
            }
        }
        
    }
}


//================================================================================
//
//================================================================================
- (CGFloat)contactInfoTableViewCellWithDefaultHeight:(CGFloat)rowHeight indexPath:(NSIndexPath *)indexPath
{
    CGFloat returnRowHeight = rowHeight;
    PPRowModel *rowModel = nil;
    
    @autoreleasepool
    {
        do
        {
            @synchronized(self.contactInfoSectionController)
            {
                rowModel = [self.contactInfoSectionController copyRowModelAtIndexPath:indexPath forSearching:NO];
            }
         
            if(rowModel==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            PPContactRowObject *contactRowObject = [rowModel ppContactRowObjectOfRowModel];
            
            if (contactRowObject==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            if(contactRowObject.tableViewCellStyle==PPContactTableViewCellStyle_TextView)
            {
                returnRowHeight = ContactInfoViewController_TableTextViewCellHeight;
            }
            else if(contactRowObject.contactMainType==PPContactMainType_Name)
            {
                CGFloat contentWidth = self.contactInfoTableView.bounds.size.width-ContactInfoViewController_TableCellImageViewSize-ContactInfoViewController_TableCellImageViewMargin*2;
                UILabel *tempLabel = [[UILabel alloc] initWithFrame:CGRectMake(0, 0, contentWidth, MAXFLOAT)];
                if (tempLabel!=nil)
                {
                    tempLabel.font = [UIFont systemFontOfSize:ContactInfoViewController_TableCellTextLabelFontSize];
                    tempLabel.numberOfLines = 0;
                    tempLabel.lineBreakMode = NSLineBreakByWordWrapping;
                    
                    //////////////////////////////////////////////////
                    //!! 可能名字會是空字串，僅有讀音！則會導致高度錯誤
                    if ([rowModel.text length]==0)
                    {
                        tempLabel.text = @" ";
                    }
                    else
                    {
                        tempLabel.text = rowModel.text;
                    }
                    
                    [tempLabel sizeToFit];
                
                    //////////////////////////////////////////////////
                    returnRowHeight += tempLabel.frame.size.height; //margin
                    
                    tempLabel.frame = CGRectMake(0, 0, contentWidth, MAXFLOAT);
                    tempLabel.font = [UIFont systemFontOfSize:ContactInfoViewController_TableCellDetailTextLabelFontSize];
                    tempLabel.text = rowModel.detailText;
                    [tempLabel sizeToFit];
                    
                    returnRowHeight += (tempLabel.frame.size.height+10);
                    
                    [tempLabel release];
                }
            }
            //地址欄位高度計算，利用暫時的Label計算高度(內容偏短)
            else if(contactRowObject.contactMainType==PPContactMainType_Company || contactRowObject.contactMainType==PPContactMainType_BirthDay)
            {
                CGFloat contentWidth = self.contactInfoTableView.bounds.size.width-ContactInfoViewController_TableCellImageViewSize-ContactInfoViewController_TableCellImageViewMargin*2;
                UILabel *tempLabel = [[UILabel alloc] initWithFrame:CGRectMake(0, 0, contentWidth, MAXFLOAT)];
                if (tempLabel!=nil)
                {
                    tempLabel.font = [UIFont systemFontOfSize:ContactInfoViewController_TableCellTextLabelFontSize];
                    tempLabel.numberOfLines = 0;
                    tempLabel.lineBreakMode = NSLineBreakByWordWrapping;
                    tempLabel.text = rowModel.text;
                    [tempLabel sizeToFit];
                    
                    returnRowHeight += tempLabel.frame.size.height+10; //margin
                    
                    [tempLabel release];
                }
            }
            else if(contactRowObject.contactMainType==PPContactMainType_Group ||
                    contactRowObject.contactMainType==PPContactMaintype_VisibleAccount)
            {
                CGFloat contentWidth = self.contactInfoTableView.bounds.size.width-ContactInfoViewController_TableCellImageViewSize-ContactInfoViewController_TableCellImageViewMargin*2;
                if (self.interfaceStyle==ContactInfoViewControllerInterfaceStyle_Browse)
                {
                    contentWidth-=ContactInfoViewController_TableViewGroupAccessoryViewWidth;
                }
                
                UILabel *tempLabel = [[UILabel alloc] initWithFrame:CGRectMake(0, 0, contentWidth, MAXFLOAT)];
                if (tempLabel!=nil)
                {
                    tempLabel.font = [UIFont systemFontOfSize:ContactInfoViewController_TableCellTextLabelFontSize];
                    tempLabel.numberOfLines = 0;
                    tempLabel.lineBreakMode = NSLineBreakByWordWrapping;
                    tempLabel.text = rowModel.detailText;
                    [tempLabel sizeToFit];
                    
                    returnRowHeight += tempLabel.frame.size.height+10; //margin
                    
                    [tempLabel release];
                }
            }
            else
            {
                // !! cell右邊的margin要另外扣掉
                CGFloat contentWidth = self.contactInfoTableView.bounds.size.width-ContactInfoViewController_TableCellImageViewSize-ContactInfoViewController_TableCellImageViewMargin*2-ContactInfoViewController_TableCellImageViewMargin;
                if (contactRowObject.contactMainType==PPContactMainType_Phone)
                {
                    switch (contactRowObject.contactLabelType)
                    {
                        case PPContactLabelType_Phone_WorkFax:
                        case PPContactLabelType_Phone_HomeFax:
                        case PPContactLabelType_Phone_Pager:
                        {
                            break;
                        }
                        default:
                        {
                            NSString *phoneString = [contactRowObject visibleTextFromSubType];
                            
                            NSRange extSymbolRange = [phoneString rangeOfCharacterFromSet:[NSCharacterSet characterSetWithCharactersInString:@"#,p"]];
                            if (extSymbolRange.length==0)
                            {
                                contentWidth = contentWidth-ContactInfoViewController_TableCellImageViewSize*2-ContactInfoViewController_TableCellImageViewMargin*2;
                            }
                            
                            break;
                        }
                    }
                }
                
                UILabel *tempLabel = [[UILabel alloc] initWithFrame:CGRectMake(0, 0, contentWidth, MAXFLOAT)];
                if (tempLabel!=nil)
                {
                    tempLabel.numberOfLines = 0;
                    tempLabel.lineBreakMode = NSLineBreakByWordWrapping;
                    
                    tempLabel.font = [UIFont systemFontOfSize:ContactInfoViewController_TableCellTextLabelFontSize];
                    tempLabel.text = rowModel.text;
                    [tempLabel sizeToFit];
                    returnRowHeight+=tempLabel.frame.size.height;
                    returnRowHeight+=(ContactLabelTableViewCellDetailHeight);
                    
                    [tempLabel release];
                }
            }
            
        } while (0);
    }
    
    [rowModel release];
    
    return returnRowHeight;
}


//===============================================================================
//
//===============================================================================
- (void)showBusyView:(BOOL)show
{
    if(show==YES)
    {
        if(self.busyView==nil)
        {
            _busyView = [[PPBusyView alloc] initWithSuperView:self.navigationController.view];
        }
    }
    else
    {
        [self.busyView removeFromSuperview];
        self.busyView = nil;
    }
}


//================================================================================
//
//================================================================================
- (NSArray *)otherCapabilityButtonArray
{
    NSMutableArray *capabilityButtonArray = [NSMutableArray array];
    
    PPCommentButton *editButton = [[PPCommentButton alloc] initWithFrame:CGRectZero
                                                         buttonImageName:ImageNameWCContactInfoViewControllerMoreButtonEdit
                                                           buttonComment:[@"MLS_ContactEdit" localizedFromTable:@"WCContactViewController"]];
    if (editButton!=nil)
    {
        editButton.tag = ContactInfoViewControllerAction_Edit;
        
        [capabilityButtonArray addObject:editButton];
        [editButton release];
    }
    
    //////////////////////////////////////////////////
    // WorldCard salesforce沒有分享
    // TSMC_Feature 沒有分享
#if !defined (PRODUCTLINE_WCSF) &&!defined(PRODUCTLINE_WCO365) &&!defined(FOR_TSMC_VERSION)
    PPCommentButton *shareButton = [[PPCommentButton alloc] initWithFrame:CGRectZero
                                                          buttonImageName:ImageNameWCContactInfoViewControllerMoreButtonShare
                                                            buttonComment:[@"MLS_ContactShare" localizedFromTable:@"WCContactViewController"]];
    if (shareButton!=nil)
    {
        shareButton.tag = ContactInfoViewControllerAction_ShareContact;
        
        [capabilityButtonArray addObject:shareButton];
        [shareButton release];
    }
    
#endif


    //////////////////////////////////////////////////

    PPCommentButton *saveButton = [[PPCommentButton alloc] initWithFrame:CGRectZero
                                                         buttonImageName:ImageNameWCContactInfoViewControllerMoreButtonSave
                                                           buttonComment:[@"MLS_SaveToSystemContactInformation" localizedFromTable:@"WCContactViewController"]];
    if (saveButton!=nil)
    {
        saveButton.tag = ContactInfoViewControllerAction_SaveContact;
        
        [capabilityButtonArray addObject:saveButton];
        [saveButton release];
    }
    
    //////////////////////////////////////////////////
    //!! WCM 7.0(2015/10/14版本)-取消翻譯聯絡人，暫時只不顯示按鈕
    
//    PPCommentButton *translateButton = [[PPCommentButton alloc] initWithFrame:CGRectZero
//                                                              buttonImageName:ImageNameWCContactInfoViewControllerMoreButtonTranslate
//                                                                buttonComment:[@"MLS_Translate" localizedFromTable:@"WCContactViewController"]];
//    if (translateButton!=nil)
//    {
//        translateButton.tag = ContactInfoViewControllerAction_Translate;
//        
//        [capabilityButtonArray addObject:translateButton];
//        [translateButton release];
//    }
    
    //////////////////////////////////////////////////
    // WorldCard salesforce沒有"重新辨識"
    
#if !defined (PRODUCTLINE_WCSF)&&!defined(PRODUCTLINE_WCO365)

    PPCommentButton *recognizeButton = [[PPCommentButton alloc] initWithFrame:CGRectZero
                                                              buttonImageName:ImageNameWCContactInfoViewControllerMoreButtonRecog
                                                                buttonComment:[@"MLS_RecognizeButton" localizedFromTable:@"WCContactViewController"]];
    if (recognizeButton!=nil)
    {
        recognizeButton.tag = ContactInfoViewControllerAction_Recognize;
        
        [capabilityButtonArray addObject:recognizeButton];
        [recognizeButton release];
    }
#endif
    //////////////////////////////////////////////////

    PPCommentButton *searchButton = [[PPCommentButton alloc] initWithFrame:CGRectZero
                                                           buttonImageName:ImageNameWCContactInfoViewControllerMoreButtonCommunitySearch
                                                             buttonComment:[@"MLS_CommunitySearch" localizedFromTable:@"WCContactViewController"]];
    if (searchButton!=nil)
    {
        searchButton.tag = ContactInfoViewControllerAction_CommunitySearch;
        
        [capabilityButtonArray addObject:searchButton];
        [searchButton release];
    }
    
    //////////////////////////////////////////////////
    
    PPCommentButton *deleteButton = [[PPCommentButton alloc] initWithFrame:CGRectZero
                                                           buttonImageName:ImageNameWCContactInfoViewControllerMoreButtonDeleteContact
                                                             buttonComment:[@"MLS_DeleteContacter" localizedFromTable:@"WCContactViewController"]];
    if (deleteButton!=nil)
    {
        deleteButton.tag = ContactInfoViewControllerAction_DeleteContact;
        
        [capabilityButtonArray addObject:deleteButton];
        [deleteButton release];
    }
    
    //////////////////////////////////////////////////

    for(PPCommentButton *button in capabilityButtonArray)
    {
        [button.titleLabel setFont:[UIFont systemFontOfSize:PPTopButtonFontSize]];
        
        //////////////////////////////////////////////////
        
        [button setBackgroundImage:[UIImage imageWithColor:PPTopButtonBackgroundColorHighlighted]
                          forState:UIControlStateHighlighted];
        
        [button setCommentTextColorNormal:PPTopButtonFontColorNormal];
        [button setCommentTextColorHighlighted:PPTopButtonFontColorHighlighted];
    }
    
    //////////////////////////////////////////////////

    return capabilityButtonArray;
}


//===============================================================================
//
//===============================================================================
- (void)launchSelectTranslateLanguageWithLocation:(ContactInfoViewControllerTranslateLocation)location arrayOfLanguage:(NSArray *)arrayOfLanguage
{
    if (self.selectController!=nil)
    {
        //!! iOS 9需先釋放再次創建，否則無法呈現
        self.selectController = nil;
    }
    
    //////////////////////////////////////////////////
    // Lazy load selectItem
    
    if (self.selectController==nil)
    {
        _selectController = [[PPSelectController alloc] init];
        if (self.selectController==nil)
        {
            return;
        }
        
        self.selectController.delegate = self;
    }
    
    //////////////////////////////////////////////////
    
    self.selectController.tableViewCellSelectionStyle = UITableViewCellSelectionStyleNone;
    
    //////////////////////////////////////////////////
    // Data Source
    
    self.selectController.sourceItemStringArray = arrayOfLanguage;
    self.selectController.tag = location;
    [self.selectController showFromViewController:self animated:YES];
    
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Custom Navigation View Method

//================================================================================
//
//================================================================================
- (void)updateCustomNavigationViewSettings
{
    if(self.interfaceIdom==ContactInfoViewControllerUserInterfaceIdom_Phone &&
       UIInterfaceOrientationIsLandscape(self.interfaceOrientation)==YES)
    {
        [self.navigationController.navigationBar setTranslucent:NO];
        [self.navigationController.navigationBar setBackgroundImage:nil forBarMetrics:UIBarMetricsDefault];
        [self.navigationController.navigationBar setShadowImage:nil];
        self.navigationController.navigationBar.barTintColor = WCAppearanceDefine_NavigationBarColor;
    }
    else
    {
        [self.navigationController.navigationBar setBackgroundColor:[UIColor clearColor]];
        [self.navigationController.navigationBar setTranslucent:YES];
        [self.navigationController.navigationBar setBackgroundImage:[[UIImage new] autorelease] forBarMetrics:UIBarMetricsDefault];
        [self.navigationController.navigationBar setShadowImage:[[UIImage new] autorelease]];
        [self.navigationController.navigationBar setBarTintColor:[UIColor clearColor]];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - MenuItem Method

//================================================================================
//
//================================================================================
- (void)handleCopyAction:(UIMenuController *)menuController
{
    NSString *stringToCopy = self.currentPressLabelCell.valueTextLabel.text;
    [self copyActionWithString:stringToCopy];
}


//================================================================================
//
//================================================================================
- (void)handlePriorityAction:(UIMenuController *)menuController
{
    NSIndexPath *currentIndexPath = [self.contactInfoTableView indexPathForCell:self.currentPressLabelCell];
    
    //////////////////////////////////////////////////
    // 移除當前點選的Cell和在sectionController該位置資料
    
    NSMutableArray *sectionModels = [[self.contactInfoSectionController sectionModelsForSearching:NO] retain];
    PPSectionModel *sectionModel  = [sectionModels objectAtIndex:currentIndexPath.section];
    
    PPRowModel *currentRowModel = [self.contactInfoSectionController copyRowModelAtIndexPath:currentIndexPath forSearching:NO];
    [sectionModel.rowModels removeObjectAtIndex:currentIndexPath.row];
    
    //////////////////////////////////////////////////
    //!! 判斷當前設定Priority的類型，插入於第一筆
    
    NSIndexPath *firstIndexPath = [NSIndexPath indexPathForRow:0 inSection:currentIndexPath.section];
    
    PPContactRowObject *contactRowObject = [currentRowModel ppContactRowObjectOfRowModel];
    if (contactRowObject.contactMainType==PPContactMainType_Name)
    {
        [sectionModel.rowModels insertObject:self.contactNameRowModel atIndex:firstIndexPath.row];
        
        self.contactNameRowModel = currentRowModel;
        
        //////////////////////////////////////////////////
        // 如果rowModel的text沒有值，還是顯示 NoName
        NSString *contactName = ContactInfoViewController_DefaultContactName;
        if ([[self.contactNameRowModel text] length]>0)
        {
            contactName = self.contactNameRowModel.text;
        }
        [self setContactName:contactName];
    }
    else
    {
        [sectionModel.rowModels insertObject:currentRowModel atIndex:firstIndexPath.row];
    }
    
    //////////////////////////////////////////////////
    
    [self.contactInfoSectionController replaceSectionModels:sectionModels forSearching:NO];
    [self.contactInfoTableView reloadSections:[NSIndexSet indexSetWithIndex:currentIndexPath.section] withRowAnimation:UITableViewRowAnimationNone];

    //////////////////////////////////////////////////
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(contactInfoViewController:setFieldIndex:toDefaultWithMainType:)]==YES)
    {
        PPContactRowObject *contactRowObject = [currentRowModel ppContactRowObjectOfRowModel];
        NSUInteger contactInfoIndex          = currentIndexPath.row;
        if (contactRowObject.contactMainType==PPContactMainType_Name)
        {
            //!! 因為在抓資訊時，聯絡人的第一筆名字會從SectionController放置於HeaderView
            //!! 所以為了和外部完整資料對應，需要額外加一
            contactInfoIndex++;
        }
        
        [self.delegate contactInfoViewController:self setFieldIndex:contactInfoIndex toDefaultWithMainType:contactRowObject.contactMainType];
    }
    
    [currentRowModel release];
    [sectionModels release];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Notification Method

//================================================================================
// 當MenuController消失時，需將currentPressLabelCell設為nil
//================================================================================
- (void)recvUIMenuControllerDidHideMenuNotification:(NSNotification *)notification
{
    self.currentPressLabelCell = nil;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Method

//================================================================================
// 透過左右滾動的距離，計算按鈕的透明度，以至於隱藏動畫
//================================================================================
- (void)contactInfoNavigationBarButtonsAlphaWithOffset:(CGFloat)offsetX
{
    PPLogFunction();
    
    //////////////////////////////////////////////////
    
    CGFloat viewAlpha;
    
    if (offsetX>=self.view.bounds.size.width/2)
    {
        viewAlpha = 1.0f-(self.view.bounds.size.width-offsetX)/ContactInfoViewController_MaximumScrollOffsetX;
    }
    else
    {
        if (offsetX<0)
        {
            //!! 使得最左側的view往右滑動時，一樣有逐漸消失效果
            offsetX = -offsetX;
        }
        
        viewAlpha = 1.0f-offsetX/ContactInfoViewController_MaximumScrollOffsetX;
    }
    
    ContactInfoNavigationBarView *contactInfoNavigatinBarView = (ContactInfoNavigationBarView *)self.contactNavigationBarView;
    contactInfoNavigatinBarView.backButton.alpha     = viewAlpha;
    contactInfoNavigatinBarView.favoriteButton.alpha = viewAlpha;
    contactInfoNavigatinBarView.noteButton.alpha     = viewAlpha;
    contactInfoNavigatinBarView.otherButton.alpha    = viewAlpha;
}


//================================================================================
//
//================================================================================
- (void)clearImageForAllPPContactMainType
{
    [self.fetchCardOperationQueue cancelAllOperations];
    [self.fetchHeadOperationQueue cancelAllOperations];
    
    //////////////////////////////////////////////////
    
    if (self.interfaceIdom==ContactInfoViewControllerUserInterfaceIdom_Phone &&
        UIInterfaceOrientationIsLandscape(self.interfaceOrientation)==YES)
    {
        [self.contactInfoLeftSideView cardImageIndicatorViewVisible:NO];
    }
    else
    {
        [self.contactInfoSwitchCardHeaderView cardImageIndicatorViewVisible:NO];
    }
    
    [self.contactInfoSwitchCardHeaderView headImageIndicatorViewVisible:NO];
    
    //////////////////////////////////////////////////
    
    [self clearImageForPPContactMainType:PPContactMainType_BackCardImage ppContactSubType:PPContactSubType_BackCardImage_Original];
    [self clearImageForPPContactMainType:PPContactMainType_BackCardImage ppContactSubType:PPContactSubType_BackCardImage_Thumbnail];
    [self clearImageForPPContactMainType:PPContactMainType_FrontCardImage ppContactSubType:PPContactSubType_FrontCardImage_Original];
    [self clearImageForPPContactMainType:PPContactMainType_FrontCardImage ppContactSubType:PPContactSubType_FrontCardImage_Thumbnail];
    [self clearImageForPPContactMainType:PPContactMainType_HeadImage ppContactSubType:PPContactSubType_HeadImage_Original];
    [self clearImageForPPContactMainType:PPContactMainType_HeadImage ppContactSubType:PPContactSubType_HeadImage_Thumbnail];
    
    self.isContactCardSwitchShowed = NO;
    self.haveLongPressCell = NO;
    self.contactInfoSwitchCardHeaderView.contactCardSwitchButtonVisible = NO;
}


//================================================================================
//
//================================================================================
- (void)clearFieldDictionaryForAllAllPPContactMainType
{
    PPLogFunction();
    
    //////////////////////////////////////////////////
    
    [self.fetchInfoOperationQueue cancelAllOperations];
    [self showBusyView:NO];
    
    //////////////////////////////////////////////////
    
    self.contactNameRowModel = nil;
    
    [[self.contactInfoSectionController sectionModelsForSearching:NO] removeAllObjects];
    self.prepareFieldDictionay = nil;
    
    if (self.contactInfoTableView!=nil)
    {
        //!! 資料清空，故需一併移除footerView，避免滾動異常
        self.contactInfoTableView.tableFooterView = nil;
        
        [self updateHeaderSubviewsWithOffset:CGPointZero];
    }
}


//================================================================================
// 
//================================================================================
- (void)recoveryInterfaceInitialSetting
{
    PPLogFunction();
    
    //////////////////////////////////////////////////
    
    [self updateHeaderSubviewsWithOffset:CGPointZero];
    [self.contactInfoTableView setContentOffset:CGPointZero];
    
    [self.ppTopButtonsActionSheet hideWithAnimated:NO];
}


//================================================================================
//
//================================================================================
- (void)prepareContactFieldDictionary
{
    do
    {
        //////////////////////////////////////////////////
        // 無聯絡人資訊，進行抓取的Loading
        
        [self showBusyView:YES];
        
        //////////////////////////////////////////////////
        // 進行Background Thread抓取資訊
        
        if (self.fetchInfoOperationQueue==nil)
        {
            break;
        }
        
        __block NSBlockOperation *blockOperation = [[[NSBlockOperation alloc] init] autorelease];
        if (blockOperation==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if([self.dataSource respondsToSelector:@selector(ppContactMainTypeArrayFetchingDataForContactInfoViewController:)]==YES ||
           [self.contactDataExchangeProtocol respondsToSelector:@selector(sender:fieldDictionaryforPPContactMainTypeArray:)]==YES)
        {
            // MARK: 判斷資料是否被刪除的機制
            self.dataState = WCContactViewControllerDataState_None;
            
            //////////////////////////////////////////////////
            
            __block typeof(self) blockSelf = self;
            [blockOperation addExecutionBlock:^{
                
                blockSelf.prepareOrders = [blockSelf.dataSource ppContactMainTypeArrayFetchingDataForContactInfoViewController:blockSelf];
                
                //////////////////////////////////////////////////
                // 聯絡人抓取所有資訊
                
                blockSelf.prepareFieldDictionay = [blockSelf.contactDataExchangeProtocol sender:blockSelf fieldDictionaryforPPContactMainTypeArray:blockSelf.prepareOrders];
                
                //////////////////////////////////////////////////
                // MARK: 判斷資料是否被刪除的機制
                
                if (blockSelf.isShowDefaultView==YES)
                {
                    blockSelf.dataState = WCContactViewControllerDataState_EmptyData;
                }
                else
                {
                    blockSelf.dataState = (blockSelf.prepareFieldDictionay!=nil) ? WCContactViewControllerDataState_Exist:WCContactViewControllerDataState_NotExist;
                }
            }];
            
            @synchronized(self.fetchInfoOperationQueue)
            {
                if([self.fetchInfoOperationQueue operationCount]>0)
                {
                    [self.fetchInfoOperationQueue cancelAllOperations];
                }
                
                [self.fetchInfoOperationQueue addOperation:blockOperation];
            }
        }
    }
    while (0);
    
}


//================================================================================
//
//================================================================================
- (void)fetchContactFieldDictionaryWithReload:(BOOL)isReload loadDictionaryCompleted:(void (^)(void))loadDictionaryCompleted
{
    do
    {
        [self setContactName:ContactInfoViewController_DefaultContactName];
        [self resetEffectParameter];
        
        ////////////////////////////////////////////////////////////////////////////////////////////////////
        // 若有聯絡人資訊歷史紀錄，即進行介面設定
        
        NSMutableArray *infoSectionModels = [[self.contactInfoSectionController sectionModelsForSearching:NO] retain];
        if (infoSectionModels!=nil)
        {
            if ([infoSectionModels count]>0)
            {
                NSString *saveAsValue = [self.contactInfoSectionController fetchSaveAsStringFromFieldDictionary:self.contactInfoSectionController.currentFieldDictionary];
                [self setSaveAsValue:saveAsValue];
                
                if ([[self.contactNameRowModel text] length]>0)
                {
                    [self setContactName:self.contactNameRowModel.text];
                    [self resetEffectParameter];
                }
                
                if ([self.contactDataExchangeProtocol respondsToSelector:@selector(favoriteWithSender:)]==YES &&
                    self.interfaceStyle!=ContactInfoViewControllerInterfaceStyle_Translate)
                {
                    ContactInfoNavigationBarView *contactInfoNavigationBarView = (ContactInfoNavigationBarView *)self.contactNavigationBarView;
                    [contactInfoNavigationBarView setContactFavorite:[self.contactDataExchangeProtocol favoriteWithSender:self]];
                }
                __block NSBlockOperation *blockMainOperation = [NSBlockOperation blockOperationWithBlock:^{
                    do
                    {
                        //////////////////////////////////////////////////
                        //!! 因為需進行介面的變換，所以丟回Main Thread
                        if (isReload==YES)
                        {
                            [self.contactInfoTableView reloadData];
                            [self addTableFooterView];
                            
                            if (loadDictionaryCompleted!=nil)
                            {
                                self.dataState = WCContactViewControllerDataState_Exist;
                                loadDictionaryCompleted();
                            }
                        }
                        
                        [infoSectionModels release];
                        [self showBusyView:NO];
                        
                    } while (0);
                }];
                
                //////////////////////////////////////////////////
                @synchronized([NSOperationQueue mainQueue])
                {
                    [[NSOperationQueue mainQueue] addOperation:blockMainOperation];
                }
//                __block typeof(self) blockSelf = self;
//
//                dispatch_async(dispatch_get_main_queue(), ^{
//                    if (isReload==YES)
//                    {
//                        [self.contactInfoTableView reloadData];
//                        [self addTableFooterView];
//
//                        if (loadDictionaryCompleted!=nil)
//                        {
//                            self.dataState = WCContactViewControllerDataState_Exist;
//                            loadDictionaryCompleted();
//                        }
//                    }
//
//                    [infoSectionModels release];
//                    [self showBusyView:NO];
//                });

                break;
            }
            
            [infoSectionModels release];
        }
        
        ////////////////////////////////////////////////////////////////////////////////////////////////////
        // 進行Background Thread Loading
        if (self.fetchInfoOperationQueue==nil)
        {
            break;
        }
        
        __block NSBlockOperation *blockOperation = [[[NSBlockOperation alloc] init] autorelease];
        if (blockOperation==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        __block typeof(self) blockSelf = self;
        [blockOperation addExecutionBlock:^{
            
            do
            {
//                if (blockOperation.isCancelled==YES)
//                {
//                    break;
//                }
//                
//                //////////////////////////////////////////////////
//                while (blockSelf.prepareFieldDictionay==nil)
//                {
//                    [NSThread sleepForTimeInterval:0.1];
//                    
//                    if (blockOperation.isCancelled==YES &&
//                        (blockSelf.dataState==WCContactViewControllerDataState_NotExist || blockSelf.dataState==WCContactViewControllerDataState_EmptyData))
//                    {
//                        break;
//                    }
//                }
                
                if (blockOperation.isCancelled==YES)
                {
                    break;
                }
                
                BOOL isFavorite = NO;
                
                if ([blockSelf.contactDataExchangeProtocol respondsToSelector:@selector(favoriteWithSender:)]==YES &&
                    blockSelf.interfaceStyle!=ContactInfoViewControllerInterfaceStyle_Translate)
                {
                    isFavorite = [blockSelf.contactDataExchangeProtocol favoriteWithSender:blockSelf];
                }
                
                //////////////////////////////////////////////////
                // 抓取聯絡人名稱和位置
                
                WCDisplayNameOrder eastOrder = WCDisplayNameOrder_LastFirst;
                WCDisplayNameOrder westOrder = WCDisplayNameOrder_LastFirst;
                
                if (blockSelf.dataSource!=nil && [blockSelf.dataSource respondsToSelector:@selector(contactNameEastOrderInContactInfoViewController:)]==YES)
                {
                    eastOrder = [blockSelf.dataSource contactNameEastOrderInContactInfoViewController:blockSelf];
                }
                
                if (blockOperation.isCancelled==YES)
                {
                    break;
                }

                if (blockSelf.dataSource!=nil && [blockSelf.dataSource respondsToSelector:@selector(contactNameWestOrderInContactInfoViewController:)]==YES)
                {
                    westOrder = [blockSelf.dataSource contactNameWestOrderInContactInfoViewController:blockSelf];
                }
                
                if (blockOperation.isCancelled==YES)
                {
                    break;
                }
                
                blockSelf.contactInfoSectionController.easterNameOrder  = (PPContactNameOrder)eastOrder;
                blockSelf.contactInfoSectionController.westernNameOrder = (PPContactNameOrder)westOrder;
                
                if (blockOperation.isCancelled==YES)
                {
                    break;
                }
                //////////////////////////////////////////////////
                // 設定是否包含拼音
                
                blockSelf.contactInfoSectionController.showPhonetic = NO;
                if (blockSelf.dataSource!=nil && [blockSelf.dataSource respondsToSelector:@selector(shouldShowPhoneticOnContactInfoViewController:)]==YES)
                {
                    blockSelf.contactInfoSectionController.showPhonetic = [blockSelf.dataSource shouldShowPhoneticOnContactInfoViewController:blockSelf];
                }
                
                if (blockOperation.isCancelled==YES)
                {
                    break;
                }
                //////////////////////////////////////////////////
                
                NSMutableArray *sectionModels = [blockSelf.contactInfoSectionController fetchFieldDictionary:blockSelf.prepareFieldDictionay
                                                                                 toSectionModelsForSearching:NO
                                                                             frontCardBCRRecognitionLanguage:blockSelf.frontCardRecognitionLanguage
                                                                              backCardBCRRecognitionLanguage:blockSelf.backCardRecognitionLanguage
                                                                                                      orders:blockSelf.prepareOrders];
                if (blockOperation.isCancelled==YES)
                {
                    break;
                }
                
                //////////////////////////////////////////////////
                // 取得儲存至
                NSString *saveAsValue = [[self.contactInfoSectionController fetchSaveAsStringFromFieldDictionary:blockSelf.prepareFieldDictionay] retain];
                //////////////////////////////////////////////////
                // 取得第一個姓名
                //!! fetchFieldDictionary內部有進行key值的排序，Name
                PPRowModel *rowModel = nil;
                
                if ([sectionModels count]>0)
                {
                    PPSectionModel *sectionModel = [sectionModels firstObject];
                    
                    if ([sectionModel.rowModels count]>0)
                    {
                        rowModel = [sectionModel.rowModels firstObject];
                        
                        if([(PPContactRowObject *)rowModel.object contactMainType]==PPContactMainType_Name)
                        {
                            blockSelf.contactNameRowModel = rowModel;
                            
                            [sectionModel.rowModels removeObject:rowModel];
                            
                            if ([sectionModel.rowModels count]==0)
                            {
                                [sectionModels removeObject:sectionModel];
                            }
                        }
                    }
                }
            
                blockSelf.prepareFieldDictionay = nil;

                if (blockOperation.isCancelled==YES)
                {
                    [saveAsValue release];
                    break;
                }
                
                //////////////////////////////////////////////////
                __block NSBlockOperation *blockMainOperation = [NSBlockOperation blockOperationWithBlock:^{
                    do
                    {
                        //////////////////////////////////////////////////
                        //!! 因為需進行介面的變換，所以丟回Main Thread
                        [blockSelf setSaveAsValue:saveAsValue];
                        [saveAsValue release];
                        
                        if ([[blockSelf.contactNameRowModel text] length]>0)
                        {
                            [blockSelf setContactName:blockSelf.contactNameRowModel.text];
                        }

                        if (blockSelf.isDeallocing==YES)
                        {
                            break;
                        }

                        [blockSelf resetEffectParameter];

                        if (blockSelf.isDeallocing==YES)
                        {
                            break;
                        }
                        
                        if (blockSelf.interfaceStyle!=ContactInfoViewControllerInterfaceStyle_Translate)
                        {
                            ContactInfoNavigationBarView *contactInfoNavigationBarView = (ContactInfoNavigationBarView *)blockSelf.contactNavigationBarView;
                            [contactInfoNavigationBarView setContactFavorite:isFavorite];
                        }
                        
                        if (blockSelf.isDeallocing==YES)
                        {
                            break;
                        }
                        
                        if (isReload==YES)
                        {
                            if (loadDictionaryCompleted!=nil)
                            {
                                loadDictionaryCompleted();
                            }
                        }

                        if (blockSelf.isDeallocing==YES)
                        {
                            break;
                        }
                        
                        if (sectionModels!=nil)
                        {
                            [blockSelf.contactInfoSectionController replaceSectionModels:sectionModels forSearching:NO];
                        }
                        [blockSelf.contactInfoTableView reloadData];
                        
                        if (blockSelf.isDeallocing==YES)
                        {
                            break;
                        }
                        
                        [blockSelf addTableFooterView];
                        
                        if (blockSelf.isDeallocing==YES)
                        {
                            break;
                        }
                        [blockSelf showBusyView:NO];

                    } while (0);
                }];
                
                //////////////////////////////////////////////////
                @synchronized (blockOperation)
                {
                    if(blockOperation.isCancelled==NO)
                    {
                        @synchronized([NSOperationQueue mainQueue])
                        {
                            [[NSOperationQueue mainQueue] addOperation:blockMainOperation];
                        }
                    }
                }

            } while (0);
        }];
        
        [self.fetchInfoOperationQueue addOperation:blockOperation];
    } while (0);
}


//================================================================================
//
//================================================================================
- (void)rotateToInterfaceOrientation:(UIInterfaceOrientation)orientation
{
    PPLogFunction();
    
    //////////////////////////////////////////////////
    
    [self recoveryInterfaceInitialSetting];
    [self.ppTopButtonsActionSheet reLayoutSubViews];
    
    //////////////////////////////////////////////////
    
    CGFloat statusBarHeight = [UIApplication sharedApplication].statusBarFrame.size.height;
    
    //////////////////////////////////////////////////

    if (self.interfaceIdom==ContactInfoViewControllerUserInterfaceIdom_Phone &&
        UIInterfaceOrientationIsLandscape(orientation)==YES)
    {
        [self.view addSubview:self.contactInfoLeftSideView];
        
        //////////////////////////////////////////////////
        
        [self updateCustomNavigationViewSettings];
        
        //////////////////////////////////////////////////
        
        CGRect tableFrame      = self.contactInfoTableView.frame;
        tableFrame.origin.x    = self.view.bounds.size.width*ContactInfoViewController_LandscapeContactCardWidthPercent;
        tableFrame.origin.y    = 0;
        tableFrame.size.width  = self.view.bounds.size.width*(1.0-ContactInfoViewController_LandscapeContactCardWidthPercent);
        tableFrame.size.height = self.view.bounds.size.height-tableFrame.origin.y;
        self.contactInfoTableView.frame = tableFrame;
        
        CGRect leftSideFrame      = self.contactInfoLeftSideView.frame;
        leftSideFrame.origin.x    = 0;
        leftSideFrame.origin.y    = 0;
        leftSideFrame.size.width  = self.view.bounds.size.width*ContactInfoViewController_LandscapeContactCardWidthPercent;
        leftSideFrame.size.height = self.view.bounds.size.height-tableFrame.origin.y;
        self.contactInfoLeftSideView.frame = leftSideFrame;
        
        self.gradientLayer.frame = CGRectZero;
        
        //////////////////////////////////////////////////
        
        CGRect headerFrame      = self.contactInfoSwitchCardHeaderView.frame;
        headerFrame.size.width  = tableFrame.size.width;
        headerFrame.size.height = ContactInfoHeaderView_ContactInfoHeight; //無需顯示名片，故只需大頭照區域高度
        self.contactInfoSwitchCardHeaderView.frame = headerFrame;
        self.contactInfoTableView.tableHeaderView = nil;
        self.contactInfoTableView.tableHeaderView = self.contactInfoSwitchCardHeaderView;
        
        self.contactImageProtocol = self.contactInfoLeftSideView;
        [self fetchContactCardImagesData];
        
        //!! 須重新縮放影像
        [self.contactInfoSwitchCardHeaderView setContactHeadImage:[self imageForPPContactMainType:PPContactMainType_HeadImage ppContactSubType:PPContactSubType_HeadImage_Thumbnail]];
        [self.contactInfoSwitchCardHeaderView layoutInfoHeaderSubViews];
    }
    else
    {
        [self.contactInfoLeftSideView removeFromSuperview];
        
        //////////////////////////////////////////////////
        
        [self updateCustomNavigationViewSettings];
        //////////////////////////////////////////////////
        
        CGRect tableFrame      = self.contactInfoTableView.frame;
        tableFrame.origin.x    = 0;
        tableFrame.origin.y    = statusBarHeight;
        tableFrame.size.width  = self.view.bounds.size.width;
        tableFrame.size.height = self.view.bounds.size.height-tableFrame.origin.y;
        self.contactInfoTableView.frame = tableFrame;
        
        CGRect frame = self.contactNavigationBarView.backgroundImageView.bounds;
        frame.size.height = self.view.bounds.size.height/3;
        frame.size.width  = self.view.bounds.size.width;
        self.gradientLayer.frame = frame;
        
        //////////////////////////////////////////////////
        
        CGRect headerFrame      = self.contactInfoSwitchCardHeaderView.frame;
        headerFrame.size.width  = self.view.bounds.size.width;
        CGFloat saveAsLabelHeight = 0;
        if (self.contactInfoSwitchCardHeaderView.showSaveAsLabel)
        {
            saveAsLabelHeight = ContactInfoHeaderView_SaveAsInfoHeight;
        }
        headerFrame.size.height = (NSInteger)((self.view.bounds.size.height*ContactInfoViewController_PortraitContactCardHeightPercent)+ContactInfoHeaderView_ContactInfoHeight+saveAsLabelHeight);
        self.contactInfoSwitchCardHeaderView.frame = headerFrame;
        self.contactInfoTableView.tableHeaderView = nil;
        self.contactInfoTableView.tableHeaderView = self.contactInfoSwitchCardHeaderView;
        
        self.contactImageProtocol = self.contactInfoSwitchCardHeaderView;
        [self fetchContactCardImagesData];
        
        //!! 須重新縮放影像
        [self.contactInfoSwitchCardHeaderView setContactHeadImage:[self imageForPPContactMainType:PPContactMainType_HeadImage ppContactSubType:PPContactSubType_HeadImage_Thumbnail]];
        [self.contactInfoSwitchCardHeaderView layoutInfoHeaderSubViews];
    }
    
    [self.contactInfoTableView reloadData];
    [self resetEffectParameter];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPTopButtonsActionSheetDelegate

//================================================================================
//
//================================================================================
- (void)ppTopButtonsActionSheet:(PPTopButtonsActionSheet *)ppTopButtonsActionSheet capabilityButtonSelectedTag:(NSInteger)buttonTag
{
    PPLogFunction();
    
    //////////////////////////////////////////////////
    // 判斷按鈕清單點擊
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(contactInfoViewController:responderView:action:)]==YES)
    {
        [self.delegate contactInfoViewController:self responderView:ppTopButtonsActionSheet action:buttonTag];
    }
}


//================================================================================
//
//================================================================================
- (void)ppTopButtonsActionSheetDidDismiss:(PPTopButtonsActionSheet *)ppTopButtonsActionSheet
{
    
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - ContactInfoHeaderViewDelegate

//================================================================================
//
//================================================================================
- (void)contactInfoHeaderView:(ContactInfoHeaderView *)contactInfoHeaderView ppButton:(PPButton *)button didControlEvents:(UIControlEvents)controlEvents
{
    switch (button.tag)
    {
        case ContactInfoHeaderViewButtonTag_CardImage:
        {
            if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(contactInfoViewController:responderView:action:)]==YES)
            {
                [self.delegate contactInfoViewController:self responderView:button action:ContactInfoViewControllerAction_ClickCardImage];
            }
            
            break;
        }
        case ContactInfoHeaderViewButtonTag_HeadImage:
        {
            PPLogFunction(@"ContactHead Button");
            
            break;
        }
        case ContactInfoHeaderViewButtonTag_SwitchCard:
        {
            PPLogFunction(@"SwitchCard Button");
    
            self.showFrontCard = !self.showFrontCard;
            
            if (self.showFrontCard==YES)
            {
                UIImage *frontImage = [self imageForPPContactMainType:PPContactMainType_FrontCardImage
                                                     ppContactSubType:PPContactSubType_FrontCardImage_Thumbnail];
                
                [contactInfoHeaderView setContactFrontCardImage:frontImage];
                [self.contactInfoLeftSideView setContactFrontCardImage:frontImage];
            }
            else
            {
                UIImage *rearImage = [self imageForPPContactMainType:PPContactMainType_BackCardImage
                                                    ppContactSubType:PPContactSubType_BackCardImage_Thumbnail];
                
                [contactInfoHeaderView setContactRearCardImage:rearImage];
                [self.contactInfoLeftSideView setContactFrontCardImage:rearImage];
            }
            
            break;
        }
        default:
        {
            PPLogFunction(@"Error Button Tag");
            break;
        }
    }
}


//================================================================================
//
//================================================================================
- (void)contactInfoHeaderViewDidShortPressContactName:(ContactInfoHeaderView *)contactInfoHeaderView
{
    if (self.interfaceStyle!=ContactInfoViewControllerInterfaceStyle_DuplicateBrowse)
    {
        PPLogFunction();
        
        //////////////////////////////////////////////////
        
        NSString *stringToCopy = contactInfoHeaderView.contactNameLabel.text;
        [self copyActionWithString:stringToCopy];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - ContactInfoLeftSideViewDelegate

//================================================================================
//
//================================================================================
- (void)contactInfoLeftSideViewDidClickedContactCardImage:(ContactInfoLeftSideView *)contactInfoLeftSideView
{
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(contactInfoViewController:responderView:action:)]==YES)
    {
        [self.delegate contactInfoViewController:self responderView:contactInfoLeftSideView action:ContactInfoViewControllerAction_ClickCardImage];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - ContactInfoNavigationBarViewDelegate

//================================================================================
//
//================================================================================
- (void)contactInfoNavigationBarView:(ContactInfoNavigationBarView *)contactInfoNavigationBarView ppButton:(PPButton *)button withControlEvent:(UIControlEvents)event
{
    switch (button.tag)
    {
        case ContactInfoNavigationBarViewButtonTag_Back:
        {
            if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(contactInfoViewController:responderView:action:)]==YES)
            {
                [self.delegate contactInfoViewController:self responderView:button action:ContactInfoViewControllerAction_Back];
            }

            break;
        }
        case ContactInfoNavigationBarViewButtonTag_Favorite:
        {
            PPLogFunction(@"Favorite Button");
            
            if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(contactInfoViewController:responderView:action:)]==YES)
            {
                [self.delegate contactInfoViewController:self responderView:button action:ContactInfoViewControllerAction_Favorite];
            }
            
            break;
        }
        case ContactInfoNavigationBarViewButtonTag_Note:
        {
            if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(contactInfoViewController:responderView:action:)]==YES)
            {
                [self.delegate contactInfoViewController:self responderView:button action:ContactInfoViewControllerAction_Note];
            }
            
            break;
        }
        case ContactInfoNavigationBarViewButtonTag_Other:
        {
            PPLogFunction(@"Other Button");
            
            //////////////////////////////////////////////////
            // 呈現額外其他功能清單
            
            UIView *presentView = nil;
            if (self.interfaceIdom==ContactInfoViewControllerUserInterfaceIdom_Phone)
            {
                presentView = self.navigationController.view;
            }
            else
            {
                UIWindow *keyWindow = [[UIApplication sharedApplication] keyWindow];
                presentView = keyWindow.rootViewController.view;
            }
            
            
            if (self.ppTopButtonsActionSheet==nil)
            {
                _ppTopButtonsActionSheet = [[PPTopButtonsActionSheet alloc] initWithFrame:presentView.bounds];
                if (self.ppTopButtonsActionSheet!=nil)
                {
                    self.ppTopButtonsActionSheet.capabilityButtonArray = [self otherCapabilityButtonArray];
                    self.ppTopButtonsActionSheet.delegate = self;
                    
                    self.ppTopButtonsActionSheet.buttonListViewBackgroundColor = PPTopButtonsActionSheetViewBackgroundColor;
                }
            }
            
            [self.ppTopButtonsActionSheet showFromSuperView:presentView animated:YES];

            break;
        }
        default:
        {
            PPLogFunction(@"Error Button Tag");
            break;
        }
    }
}


//================================================================================
//
//================================================================================
- (void)contactInfoNavigationBarViewDidShortPressContactName:(ContactInfoNavigationBarView *)contactInfoNavigationBarView
{
    if (self.interfaceStyle!=ContactInfoViewControllerInterfaceStyle_DuplicateBrowse)
    {
        PPLogFunction();
        
        //////////////////////////////////////////////////
        
        NSString *stringToCopy = contactInfoNavigationBarView.titleLabel.text;
        [self copyActionWithString:stringToCopy];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - ContactTranslateNavigationBarViewDelegate

//================================================================================
//
//================================================================================
- (void)contactTranslateNavigationBarView:(ContactTranslateNavigationBarView *)contactTranslateNavigationBarView ppButton:(PPButton *)button withControlEvent:(UIControlEvents)event
{
    switch (button.tag)
    {
        case ContactTranslateNavigationBarViewButtonTag_Back:
        {
            if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(contactInfoViewController:responderView:action:)]==YES)
            {
                [self.delegate contactInfoViewController:self responderView:button action:ContactInfoViewControllerAction_Back];
            }
            
            break;
        }
        case ContactTranslateNavigationBarViewButtonTag_SourceLanguage:
        {
            if (self.dataSource!=nil && [self.dataSource respondsToSelector:@selector(translateLanguageArrayInContactInfoViewController:translateLocation:)]==YES)
            {
                NSArray *array = [self.dataSource translateLanguageArrayInContactInfoViewController:self translateLocation:ContactInfoViewControllerTranslateLocation_Source];
                if ([array count]>1)
                {
                    [self launchSelectTranslateLanguageWithLocation:ContactInfoViewControllerTranslateLocation_Source arrayOfLanguage:array];
                }
            }
            
            break;
        }
        case ContactTranslateNavigationBarViewButtonTag_DestinationLanguage:
        {
            if (self.dataSource!=nil && [self.dataSource respondsToSelector:@selector(translateLanguageArrayInContactInfoViewController:translateLocation:)]==YES)
            {
                NSArray *array = [self.dataSource translateLanguageArrayInContactInfoViewController:self translateLocation:ContactInfoViewControllerTranslateLocation_Destination];
                if ([array count]>1)
                {
                    [self launchSelectTranslateLanguageWithLocation:ContactInfoViewControllerTranslateLocation_Destination arrayOfLanguage:array];
                }
            }
            
            break;
        }
        default:
            break;
    }
}




////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPButtonDelegate

//================================================================================
//
//================================================================================
- (void)ppButton:(PPButton *)ppButton controlEvent:(UIControlEvents)controlEvent
{
    NSMutableArray *tempArray = [self.contactInfoSectionController sectionModelsForSearching:NO];
    for (NSInteger index=0; index<tempArray.count; index++)
    {
        PPContactSectionModel *sectionModel = [tempArray objectAtIndex:index];
        if (sectionModel.contactMainType==PPContactMainType_Phone)
        {
            NSIndexPath *indexPath = [NSIndexPath indexPathForRow:ppButton.tag inSection:index];
            PPRowModel *rowModel   = [self.contactInfoSectionController rowModelAtIndexPath:indexPath forSearching:NO];
            PPContactRowObject *contactRowObject = [rowModel ppContactRowObjectOfRowModel];
            
            NSString *phoneNumber = [contactRowObject visibleTextFromSubType];
            
            if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(contactInfoViewController:didSelectPhoneNumber:)]==YES)
            {
                [self.delegate contactInfoViewController:self didSelectPhoneNumber:phoneNumber];
            }
            
            break;
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPLabelTableViewCellDelegate

//================================================================================
//
//================================================================================
- (void)ppLabelTableViewCellDidShortPress:(PPLabelTableViewCell *)labelTableViewCell
{
    if (self.interfaceStyle!=ContactInfoViewControllerInterfaceStyle_DuplicateBrowse)
    {
        self.currentPressLabelCell = labelTableViewCell;
        UIMenuController *selectMenuController = [UIMenuController sharedMenuController];
        
        //////////////////////////////////////////////////
        
        if (self.interfaceStyle==ContactInfoViewControllerInterfaceStyle_NearbyContacts||
            self.interfaceStyle==ContactInfoViewControllerInterfaceStyle_CompanyContacts)
        {
            [self handleCopyAction:selectMenuController];
        }
        else if (labelTableViewCell.tag==ContactInfoViewControllerCellShortPressEvent_ShowFirstPriority)
        {
            if (selectMenuController!=nil)
            {
                NSMutableArray *menuItems = [NSMutableArray array];
                
                UIMenuItem *copyMenuItem = [[UIMenuItem alloc] initWithTitle:[@"MLS_Copy" localizedFromTable:@"WCContactViewController"] action:@selector(handleCopyAction:)];
                if (copyMenuItem!=nil)
                {
                    [menuItems addObject:copyMenuItem];
                    [copyMenuItem release];
                }
                
                
                UIMenuItem *priorityMenuItem = [[UIMenuItem alloc] initWithTitle:[@"MLS_FirstPriority" localizedFromTable:@"WCContactViewController"] action:@selector(handlePriorityAction:)];
                if (priorityMenuItem!=nil)
                {
                    [menuItems addObject:priorityMenuItem];
                    [priorityMenuItem release];
                }
                
                [selectMenuController setMenuItems:menuItems];
                [selectMenuController setTargetRect:labelTableViewCell.bounds inView:labelTableViewCell];
                [selectMenuController setMenuVisible:YES animated:YES];
            }
        }
        else
        {
            [self handleCopyAction:selectMenuController];
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPSelectControllerDelegate

//================================================================================
//
//================================================================================
- (void)ppSelectController:(PPSelectController *)selectController itemSelectedIndex:(NSInteger)itemIndex withString:(NSString *)itemString
{
    switch (selectController.tag)
    {
        case ContactInfoViewControllerTranslateLocation_Source:
        {
            [(ContactTranslateNavigationBarView *)self.contactNavigationBarView setSourceLanguageString:itemString];
            break;
        }
        case ContactInfoViewControllerTranslateLocation_Destination:
        {
            [(ContactTranslateNavigationBarView *)self.contactNavigationBarView setDestinationLanguageString:itemString];
            break;
        }
        default:
        {
            NSLog(@"Error selectController Tag For TranslateLocation!");
            break;
        }
    }
    
    //////////////////////////////////////////////////
    
    if (self.delegate!=nil && [self.delegate respondsToSelector:@selector(contactInfoViewController:translateFromLanguage:toLanguage:)]==YES)
    {
        NSString *sourceLanguage = [(ContactTranslateNavigationBarView *)self.contactNavigationBarView sourceLanguageString];
        NSString *destinationLanguage = [(ContactTranslateNavigationBarView *)self.contactNavigationBarView destinationLanguageString];
        
        [self.delegate contactInfoViewController:self translateFromLanguage:sourceLanguage toLanguage:destinationLanguage];
    }
}


//================================================================================
//
//================================================================================
- (UITableViewCell *)ppSelectController:(PPSelectController *)selectController tableView:(UITableView *)tableView cellForRowAtIndexPath:(NSIndexPath *)indexPath itemString:(NSString *)itemString
{
    static NSString *CellIdentifier = @"PPSelectControllerCellIdentifier";
    UITableViewCell *cell = nil;
    
    cell = [tableView dequeueReusableCellWithIdentifier:CellIdentifier];
    if (cell==nil)
    {
        cell = [[[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault reuseIdentifier:CellIdentifier] autorelease];
        if (cell==nil)
        {
            return nil;
        }
    }
    
    cell.textLabel.text = [[selectController.sourceItemStringArray objectAtIndex:indexPath.row] localized];
    
    cell.backgroundColor     = [UIColor clearColor];
    cell.textLabel.textColor = selectController.cellNormalTextColor;
    
    return cell;
}


//================================================================================
//
//================================================================================
- (void)ppSelectControllerDidDismiss:(PPSelectController *)selectController
{
    self.selectController = nil;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UIScrollViewDelegate

//================================================================================
// 處理自動縮起效果
//================================================================================
- (void)scrollViewWillEndDragging:(UIScrollView *)scrollView withVelocity:(CGPoint)velocity targetContentOffset:(inout CGPoint *)targetContentOffset
{
    PPLogFunction();
    
    //////////////////////////////////////////////////
    
    CGPoint targetOffset = *targetContentOffset;
    
    if(targetOffset.y < self.changeNavigationBarOffsetY && targetOffset.y > self.changeInfoLayoutOffsetY)
    {        
        self.updateOffset = YES;
    }
    else
    {
        self.updateOffset = NO;
    }
}


//================================================================================
//
//================================================================================
- (void)scrollViewDidEndDragging:(UIScrollView *)scrollView willDecelerate:(BOOL)decelerate
{
    PPLogFunction();
    
    //////////////////////////////////////////////////
    // 處理往下滾動，自動縮起效果
    
    if(decelerate == NO &&
       self.contactInfoTableView.contentOffset.y < self.changeNavigationBarOffsetY &&
       self.contactInfoTableView.contentOffset.y > self.changeInfoLayoutOffsetY)
    {
        [self.contactInfoTableView setContentOffset:CGPointMake(0, self.changeNavigationBarOffsetY) animated:YES];
    }
}


//================================================================================
//
//================================================================================
- (void)scrollViewDidEndDecelerating:(UIScrollView *)scrollView
{
    if (self.updateOffset==YES)
    {
        [self.contactInfoTableView setContentOffset:CGPointMake(0, self.changeNavigationBarOffsetY) animated:YES];
        self.updateOffset = NO;
    }
}


//================================================================================
//
//================================================================================
- (void)scrollViewDidScroll:(UIScrollView *)scrollView
{
    if (self.interfaceIdom==ContactInfoViewControllerUserInterfaceIdom_Phone &&
        UIInterfaceOrientationIsLandscape(self.interfaceOrientation)==YES)
    {
        
    }
    else
    {
        [self updateHeaderSubviewsWithOffset:self.contactInfoTableView.contentOffset];
    }
    
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UITableViewDataSource

//================================================================================
//
//================================================================================
- (NSInteger)numberOfSectionsInTableView:(UITableView *)tableView
{
    return [self.contactInfoSectionController numberOfSectionsForSearching:NO];
}


//================================================================================
//
//================================================================================
- (NSInteger)tableView:(UITableView *)tableView numberOfRowsInSection:(NSInteger)section
{
    return [self.contactInfoSectionController numberOfRowsInSection:section forSearching:NO];
}


//================================================================================
//
//================================================================================
- (UITableViewCell *)tableView:(UITableView *)tableView cellForRowAtIndexPath:(NSIndexPath *)indexPath
{
    UITableViewCell *cell = nil;
    
    do
    {
        PPRowModel *rowModel = [self.contactInfoSectionController rowModelAtIndexPath:indexPath forSearching:NO];
        PPContactRowObject *contactRowObject = [rowModel ppContactRowObjectOfRowModel];
        
        //////////////////////////////////////////////////
        
        switch (contactRowObject.tableViewCellStyle)
        {
            case PPContactTableViewCellStyle_TextView:
            {
                NSString *identifier = [TextViewCellIdenttifier stringByAppendingString:[NSString stringWithFormat:@"%td-textView",indexPath.section]];
                cell = [tableView dequeueReusableCellWithIdentifier:identifier];
                if(cell==nil)
                {
                    UITableViewCellStyle style = UITableViewCellStyleDefault;
                    
                    cell = [[[ContactTextViewTableViewCell alloc] initWithStyle:style
                                                                reuseIdentifier:identifier] autorelease];
                }

                //////////////////////////////////////////////////

                ContactTextViewTableViewCell *textViewTableViewCell = (ContactTextViewTableViewCell *)cell;

                //////////////////////////////////////////////////

                textViewTableViewCell.maxLength = 0;
                if([self.dataSource respondsToSelector:@selector(contactInfoTableViewCellTextMaxLengthInContactInfoViewController:ppContactMainType:)]==YES)
                {
                    textViewTableViewCell.maxLength = [self.dataSource contactInfoTableViewCellTextMaxLengthInContactInfoViewController:self ppContactMainType:contactRowObject.contactMainType];
                }

                if(textViewTableViewCell.maxLength<=0)
                {
                    textViewTableViewCell.maxLength = NSUIntegerMax;
                }

                textViewTableViewCell.contactTableViewCellLineView.hidden = YES;
                textViewTableViewCell.selectionStyle = UITableViewCellSelectionStyleNone;

                if(contactRowObject.inputValueDictionary.count>0)
                {
                    textViewTableViewCell.ppTextView.text = [contactRowObject visibleTextFromSubType];
                    textViewTableViewCell.ppTextView.font = [UIFont systemFontOfSize:ContactInfoViewController_TableCellTextLabelFontSize];
                }

                //禁止編輯
                textViewTableViewCell.ppTextView.editable = NO;

                //////////////////////////////////////////////////

                [textViewTableViewCell.ppTextView scrollRangeToVisible:NSMakeRange(0,0)];

                break;
            }
            case PPContactTableViewCellStyle_Label:
            {
                if (contactRowObject.contactMainType==PPContactMainType_Name)
                {
                    if ([rowModel.detailText isEqualToString:@""] || rowModel.detailText==nil)
                    {
                        NSString *identifier = [LabelCellIdentifier stringByAppendingString:[NSString stringWithFormat:@"%td-NoDetail",indexPath.section]];
                        cell = [tableView dequeueReusableCellWithIdentifier:identifier];
                        if (cell==nil)
                        {
                            cell = [[[ContactLabelTableViewCell alloc] initWithStyle:UITableViewCellStyleDefault
                                                                     reuseIdentifier:identifier] autorelease];
                        }
                    }
                    else
                    {
                        NSString *identifier = [LabelCellIdentifier stringByAppendingString:[NSString stringWithFormat:@"%td-Detail",indexPath.section]];
                        cell = [tableView dequeueReusableCellWithIdentifier:identifier];
                        if (cell==nil)
                        {
                            cell = [[[ContactLabelTableViewCell alloc] initWithStyle:UITableViewCellStyleSubtitle
                                                                     reuseIdentifier:identifier] autorelease];
                        }
                        
                    }
                    
                    //////////////////////////////////////////////////
                    
                    ContactLabelTableViewCell *contactLabelTableViewCell = (ContactLabelTableViewCell *)cell;
                    [contactLabelTableViewCell addShortPressGestureRecognizer];
                    contactLabelTableViewCell.selectionStyle = UITableViewCellSelectionStyleNone;
                    contactLabelTableViewCell.delegate = self;
                    
                    contactLabelTableViewCell.valueTextLabel.numberOfLines = 0;
                    contactLabelTableViewCell.valueTextLabel.lineBreakMode = NSLineBreakByWordWrapping;
                    contactLabelTableViewCell.valueTextLabel.textAlignment = NSTextAlignmentLeft;
                    contactLabelTableViewCell.valueTextLabel.font  = [UIFont systemFontOfSize:ContactInfoViewController_TableCellTextLabelFontSize];
                    contactLabelTableViewCell.valueTextLabel.text  = rowModel.text;
                    contactLabelTableViewCell.detailTextLabel.numberOfLines = 0;
                    contactLabelTableViewCell.detailTextLabel.lineBreakMode = NSLineBreakByWordWrapping;
                    contactLabelTableViewCell.detailTextLabel.font = [UIFont systemFontOfSize:ContactInfoViewController_TableCellDetailTextLabelFontSize];
                    contactLabelTableViewCell.detailTextLabel.text = rowModel.detailText;
                }
                else if((contactRowObject.contactMainType!=PPContactMainType_Company && contactRowObject.contactMainType!=PPContactMainType_BirthDay))
                {
                    NSString *identifier = [LabelCellIdentifier stringByAppendingString:[NSString stringWithFormat:@"%td-SubTitle",indexPath.section]];
                    cell = [tableView dequeueReusableCellWithIdentifier:identifier];

                    if(cell==nil)
                    {
                        cell = [[[ContactLabelTableViewCell alloc] initWithStyle:UITableViewCellStyleSubtitle
                                                                 reuseIdentifier:identifier] autorelease];
                    }
                    
                    //////////////////////////////////////////////////
                    
                    ContactLabelTableViewCell *contactLabelTableViewCell = (ContactLabelTableViewCell *)cell;
                    [contactLabelTableViewCell addShortPressGestureRecognizer];
                    contactLabelTableViewCell.selectionStyle = UITableViewCellSelectionStyleNone;
                    contactLabelTableViewCell.delegate = self;
                    
                    contactLabelTableViewCell.detailTextLabel.text = [rowModel.detailText localizedFromTable:WCFL];
                    
                    //多語系字串轉換失敗，使用 WCCV 字串表找一次
                    if([contactLabelTableViewCell.detailTextLabel.text compare:rowModel.detailText]==NSOrderedSame)
                    {
                        contactLabelTableViewCell.detailTextLabel.text = [rowModel.detailText localizedFromTable:WCCV];
                    }
                    
                    //////////////////////////////////////////////////

                    contactLabelTableViewCell.detailTextLabel.font = [UIFont systemFontOfSize:ContactInfoViewController_TableCellDetailTextLabelFontSize];
                    
                    contactLabelTableViewCell.valueTextLabel.numberOfLines = 0;
                    contactLabelTableViewCell.valueTextLabel.lineBreakMode = NSLineBreakByWordWrapping;
                    contactLabelTableViewCell.valueTextLabel.textAlignment = NSTextAlignmentLeft;
                    contactLabelTableViewCell.valueTextLabel.font = [UIFont systemFontOfSize:ContactInfoViewController_TableCellTextLabelFontSize];

                    contactLabelTableViewCell.valueTextLabel.text = rowModel.text;
                    
                    // date要顯示年
                    if((contactRowObject.contactMainType==PPContactMainType_UserDefine &&
                        ([contactRowObject curEditingSubType]==PPContactSubType_UserDefine_DateAndTime ||
                         [contactRowObject curEditingSubType]==PPContactSubType_UserDefine_Date)))
                    {
                        if(contactRowObject.inputValueDictionary.count>0)
                        {
                            NSString *rowModelText = [contactRowObject visibleTextFromSubType];
                            
                            NSArray *rowModelTexts = [rowModelText componentsSeparatedByString:@"/"];
                            
                            //含年份, 小於1901年份為隱藏欄位
                            if(rowModelTexts.count>=3 &&
                               [[rowModelTexts firstObject] integerValue]<=(NSInteger)WC_ValidMinYear)
                            {
                                contactLabelTableViewCell.valueTextLabel.text = [[rowModelTexts objectAtIndex:1] stringByAppendingString:[NSString stringWithFormat:@"/%@",[rowModelTexts lastObject]]];
                            }
                            else
                            {
                                contactLabelTableViewCell.valueTextLabel.text = rowModelText;
                            }
                        }
                        else
                        {
                            contactLabelTableViewCell.valueTextLabel.text = @"";
                        }
                    }
                    else
                    {
                        contactLabelTableViewCell.valueTextLabel.text          = [contactRowObject visibleTextFromSubType];
                    }
                    
                }
                else
                {
                    NSString *identifier = [LabelCellIdentifier stringByAppendingString:[NSString stringWithFormat:@"%td-Default",indexPath.section]];
                    cell = [tableView dequeueReusableCellWithIdentifier:identifier];

                    if(cell==nil)
                    {
                        cell = [[[ContactLabelTableViewCell alloc] initWithStyle:UITableViewCellStyleDefault
                                                                 reuseIdentifier:identifier] autorelease];
                    }
                    
                    ContactLabelTableViewCell *contactLabelTableViewCell = (ContactLabelTableViewCell *)cell;
                    [contactLabelTableViewCell addShortPressGestureRecognizer];
                    contactLabelTableViewCell.selectionStyle = UITableViewCellSelectionStyleNone;
                    contactLabelTableViewCell.delegate = self;
                    
                    contactLabelTableViewCell.valueTextLabel.adjustsFontSizeToFitWidth = YES;
                    contactLabelTableViewCell.valueTextLabel.font          = [UIFont systemFontOfSize:ContactInfoViewController_TableCellTextLabelFontSize];
                    contactLabelTableViewCell.valueTextLabel.textAlignment = NSTextAlignmentLeft;
                    contactLabelTableViewCell.valueTextLabel.numberOfLines = 0;
                    contactLabelTableViewCell.valueTextLabel.lineBreakMode = NSLineBreakByWordWrapping;
                    contactLabelTableViewCell.detailTextLabel.text         = nil;
                    
                    
                    //含年份, 小於1901年份為隱藏欄位
                    if(contactRowObject.contactMainType==PPContactMainType_BirthDay)
                    {
                        if(contactRowObject.inputValueDictionary.count>0)
                        {
                            NSString *rowModelText = [contactRowObject visibleTextFromSubType];
                            
                            NSArray *rowModelTexts = [rowModelText componentsSeparatedByString:@"/"];
                            
                            //含年份, 小於1901年份為隱藏欄位
                            if(rowModelTexts.count>=3 &&
                               [[rowModelTexts firstObject] integerValue]<=(NSInteger)WC_ValidMinYear)
                            {
                                contactLabelTableViewCell.valueTextLabel.text = [[rowModelTexts objectAtIndex:1] stringByAppendingString:[NSString stringWithFormat:@"/%@",[rowModelTexts lastObject]]];
                            }
                            else
                            {
                                contactLabelTableViewCell.valueTextLabel.text = rowModelText;
                            }
                        }
                        else
                        {
                            contactLabelTableViewCell.valueTextLabel.text = @"";
                        }
                    }
                    else
                    {
                        contactLabelTableViewCell.valueTextLabel.text          =  [contactRowObject visibleTextFromSubType];
                    }
      
                }
                
                break;
            }
            case PPContactTableViewCellStyle_Default:
            default:
            {
                NSString *identifier = [DefaultCellIdentifier stringByAppendingString:[NSString stringWithFormat:@"%td-None",indexPath.section]];
                cell = [tableView dequeueReusableCellWithIdentifier:identifier];
                
                if(cell==nil)
                {
                    cell = [[[ContactLabelTableViewCell alloc] initCustomStyleWithReuseIdentifier:identifier] autorelease];
                }
                
                ContactLabelTableViewCell *contactLabelTableViewCell = (ContactLabelTableViewCell *)cell;
                contactLabelTableViewCell.selectionStyle = UITableViewCellSelectionStyleNone;
                contactLabelTableViewCell.delegate = self;
                contactLabelTableViewCell.backgroundImageView.contentMode = UIViewContentModeScaleAspectFit;
                contactLabelTableViewCell.valueTextLabel.adjustsFontSizeToFitWidth = YES;
                contactLabelTableViewCell.valueTextLabel.font          = [UIFont systemFontOfSize:ContactInfoViewController_TableCellTextLabelFontSize];
                contactLabelTableViewCell.valueTextLabel.textAlignment = NSTextAlignmentLeft;
                contactLabelTableViewCell.valueTextLabel.numberOfLines = 0;
                contactLabelTableViewCell.valueTextLabel.lineBreakMode = NSLineBreakByWordWrapping;
                contactLabelTableViewCell.valueTextLabel.text          = [rowModel.detailText localizedFromTable:WCFL];
                contactLabelTableViewCell.detailTextLabel.text         = nil;
                
                if(self.interfaceStyle!=ContactInfoViewControllerInterfaceStyle_DuplicateBrowse&&
                   self.interfaceStyle!=ContactInfoViewControllerInterfaceStyle_CompanyContacts)
                {
                    if (contactRowObject.contactMainType==PPContactMainType_Group ||
                        contactRowObject.contactMainType==PPContactMaintype_VisibleAccount)
                    {
                        contactLabelTableViewCell.accessoryType = UITableViewCellAccessoryDisclosureIndicator;
                    }
                }
                
                //////////////////////////////////////////////////

                //!! Group 類別未同步圖示設定流程
                if(contactRowObject.contactMainType==PPContactMainType_Group)
                {
                    // MARK: 要調整圖示
                    if([rowModel.detailText isEqualToString:WCVC_GroupNameNotSyncedSymbol]==YES)
                    {
                        contactLabelTableViewCell.backgroundImageView.image = [UIImage imageWithName:ImageNameWCContactViewControllerButtonCategoryNotSynced];
                        contactLabelTableViewCell.backgroundImageView.hidden = NO;
                        contactLabelTableViewCell.valueTextLabel.hidden = YES;
                    }
                    else
                    {
                        contactLabelTableViewCell.backgroundImageView.image = nil;
                        contactLabelTableViewCell.backgroundImageView.hidden = YES;
                        contactLabelTableViewCell.valueTextLabel.hidden = NO;
                    }
                }
                else
                {
                    contactLabelTableViewCell.backgroundImageView.image = nil;
                    contactLabelTableViewCell.backgroundImageView.hidden = YES;
                    contactLabelTableViewCell.valueTextLabel.hidden = NO;
                }
                
                break;
            }
        }
        
        //////////////////////////////////////////////////
        
        cell.tag = ContactInfoViewControllerCellShortPressEvent_HiddenFirstPriority;
        cell.contentView.ppGuideIndex = 0;
        
        if ((contactRowObject.contactMainType!=PPContactMainType_BirthDay) &&
            (contactRowObject.contactMainType!=PPContactMainType_ConstructTime) &&
            (contactRowObject.contactMainType!=PPContactMainType_Other)&&
            (contactRowObject.contactMainType!=PPContactMainType_UserDefine)&&
            (contactRowObject.contactMainType!=PPContactMainType_Status))
        {
            if (([self.contactInfoSectionController numberOfRowsInSection:indexPath.section forSearching:NO]>1 && indexPath.row>0)||
                contactRowObject.contactMainType==PPContactMainType_Name)
            {
                //設定Cell是否含有設為預設功能
                cell.tag = ContactInfoViewControllerCellShortPressEvent_ShowFirstPriority;
            }
            
            if (contactRowObject.contactMainType==PPContactMainType_Name && indexPath.row==0)
            {
                self.haveLongPressCell = YES;
                cell.ppGuideIndex = PPGuideDataModel_GuideIndex_InfoLongPressCell;
            }
            else if (contactRowObject.contactMainType==PPContactMainType_Phone && indexPath.row==0 && self.haveLongPressCell==NO)
            {
                self.haveLongPressCell = YES;
                cell.ppGuideIndex = PPGuideDataModel_GuideIndex_InfoLongPressCell;
            }
        }
        
        //////////////////////////////////////////////////
        
        cell.textLabel.textColor       = ContactInfoViewControllerTableViewCellTextFontColor;
        cell.detailTextLabel.textColor = ContactInfoViewControllerTableViewCellDetailTextFontColor;
        
        //////////////////////////////////////////////////
        // 設定AccessoryView的樣式，無需要的cell務必設成nil
        
        if (contactRowObject.contactMainType==PPContactMainType_Phone && self.interfaceIdom==ContactInfoViewControllerUserInterfaceIdom_Phone)
        {
            switch (contactRowObject.contactLabelType)
            {
                case PPContactLabelType_Phone_WorkFax:
                case PPContactLabelType_Phone_HomeFax:
                case PPContactLabelType_Phone_Pager:
                {
                    cell.accessoryView = nil;
                    
                    break;
                }
                default:
                {
                    // 重複聯絡人不能操作action所以也不用簡訊按鈕
                    if (self.interfaceStyle==ContactInfoViewControllerInterfaceStyle_DuplicateBrowse)
                    {
                        cell.accessoryView = nil;
                        break;
                    }

                    //////////////////////////////////////////////////
                    NSString *phoneString = [contactRowObject visibleTextFromSubType];
                    
                    NSRange extSymbolRange = [phoneString rangeOfCharacterFromSet:[NSCharacterSet characterSetWithCharactersInString:@"#,p"]];
                    if (extSymbolRange.length==0)
                    {
                        PPButton *ppButtonCapability = [[[PPButton alloc] initWithFrame:CGRectMake(0, 0, ContactInfoViewController_TableCellImageViewSize, cell.bounds.size.height)] autorelease];
                        if (ppButtonCapability!=nil)
                        {
                            ppButtonCapability.imageEdgeInsets = UIEdgeInsetsMake((cell.bounds.size.height-ContactInfoViewController_TableCellImageViewSize)/2, 0, (cell.bounds.size.height-ContactInfoViewController_TableCellImageViewSize)/2, 0);
                            ppButtonCapability.delegate = self;
                            ppButtonCapability.tag = indexPath.row;
                            
                            [ppButtonCapability setImageWithName:ImageNameWCContactInfoViewControllerCellSMS];
                            [ppButtonCapability addControlEvents:UIControlEventTouchUpInside];
                            
                            cell.accessoryView = ppButtonCapability;
                        }
                    }
                    else
                    {
                        cell.accessoryView = nil;
                    }
                    
                    break;
                }
            }
        }
        else
        {
            cell.accessoryView = nil;
        }
        
        //////////////////////////////////////////////////
        // 設定左側功能影像
        
        UIImage *capabilityIcon = nil;
        
        switch (contactRowObject.contactMainType)
        {
            case PPContactMainType_Name:
            {
                capabilityIcon = [UIImage imageWithName:ImageNameWCContactInfoViewControllerCellContactName forState:UIControlStateNormal];
                break;
            }
            case PPContactMainType_Phone:
            {
                capabilityIcon = [UIImage imageWithName:ImageNameWCContactInfoViewControllerCellPhone forState:UIControlStateNormal];
                break;
            }
            case PPContactMainType_Email:
            {
                capabilityIcon = [UIImage imageWithName:ImageNameWCContactInfoViewControllerCellEmail forState:UIControlStateNormal];
                break;
            }
            case PPContactMainType_Company:
            {
                capabilityIcon = [UIImage imageWithName:ImageNameWCContactInfoViewControllerCellCompany forState:UIControlStateNormal];
                break;
            }
            case PPContactMainType_Address:
            {
                capabilityIcon = [UIImage imageWithName:ImageNameWCContactInfoViewControllerCellAddress forState:UIControlStateNormal];
                break;
            }
            case PPContactMainType_URL:
            {
                capabilityIcon = [UIImage imageWithName:ImageNameWCContactInfoViewControllerCellURL forState:UIControlStateNormal];
                break;
            }
            case PPContactMainType_InstantMessage:
            {
                capabilityIcon = [UIImage imageWithName:ImageNameWCContactInfoViewControllerCellIM forState:UIControlStateNormal];
                break;
            }
            case PPContactMainType_SocialNetwork:
            {
                capabilityIcon = [UIImage imageWithName:ImageNameWCContactInfoViewControllerCellSocialNetwork forState:UIControlStateNormal];
                break;
            }
            case PPContactMainType_BirthDay:
            {
                capabilityIcon = [UIImage imageWithName:ImageNameWCContactInfoViewControllerCellBityhday forState:UIControlStateNormal];
                break;
            }
            case PPContactMainType_Date:
            {
                capabilityIcon = [UIImage imageWithName:ImageNameWCContactInfoViewControllerCellDate forState:UIControlStateNormal];
                break;
            }
            case PPContactMainType_Other:
            {
                capabilityIcon = [UIImage imageWithName:ImageNameWCContactInfoViewControllerCellOthers forState:UIControlStateNormal];
                break;
            }
            case PPContactMainType_Group:
            {
                capabilityIcon = [UIImage imageWithName:ImageNameWCContactInfoViewControllerCellGroup forState:UIControlStateNormal];
                
                break;
            }
            case PPContactMaintype_VisibleAccount:
            {
                capabilityIcon = [UIImage imageWithName:ImageNameWCContactInfoViewControllerCellViewAccount forState:UIControlStateNormal];
                
                break;
            }
            case PPContactMainType_UserDefine:
            {
                capabilityIcon = [UIImage imageWithName:ImageNameWCContactInfoViewControllerCellUserDefine forState:UIControlStateNormal];
                
                break;
            }
            case PPContactMainType_Status:
            {
                capabilityIcon = [UIImage imageWithName:ImageNameWCContactInfoViewControllerCellOthers forState:UIControlStateNormal];
                
                break;
            }
            default:
            {
                capabilityIcon = [UIImage imageWithName:ImageNameWCContactInfoViewControllerCellPhone forState:UIControlStateNormal];
                break;
            }
        }
        
        if (capabilityIcon!=nil)
        {
            if (indexPath.row==0)
            {
                cell.imageView.image = capabilityIcon;
            }
            else
            {
                cell.imageView.image = [UIImage imageWithColor:[UIColor clearColor] size:capabilityIcon.size];
            }
        }
        
    } while (0);
    
    return cell;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UITableView Delegate

//================================================================================
//
//================================================================================
- (void)tableView:(UITableView *)tableView didSelectRowAtIndexPath:(NSIndexPath *)indexPath
{
    if (self.interfaceStyle!=ContactInfoViewControllerInterfaceStyle_DuplicateBrowse)
    {
        do
        {
            if([self.delegate respondsToSelector:@selector(contactInfoViewController:didSelectValueDictionary:)]==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            if([self.contactInfoSectionController sectionModelsForSearching:NO].count<=indexPath.section)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            PPContactSectionModel *sectionModel = [[self.contactInfoSectionController sectionModelsForSearching:NO]objectAtIndex:indexPath.section];
            
            if(sectionModel==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////

            // 通知點選類別資訊欄位，進行變更類別功能
            switch (sectionModel.contactMainType)
            {
                case PPContactMainType_Group:
                case PPContactMaintype_VisibleAccount:
                {
                    // !! 共享聯絡人模式不能改群組及可檢視用戶
                    if (self.interfaceStyle==ContactInfoViewControllerInterfaceStyle_CompanyContacts)
                    {
                        break;
                    }
                    ContactInfoViewControllerAction action = ContactInfoViewControllerAction_ChangeGroup;
                    
                    if(sectionModel.contactMainType==PPContactMaintype_VisibleAccount)
                    {
                        action = ContactInfoViewControllerAction_ChangeVisibleAccount;
                    }
                    
                    //////////////////////////////////////////////////

                    if ([self.delegate respondsToSelector:@selector(contactInfoViewController:responderView:action:)]==YES)
                    {
                        __block typeof(self) blockSelf = self;
                        
                        dispatch_async(dispatch_get_main_queue(), ^{
                            [blockSelf.delegate contactInfoViewController:blockSelf
                                                            responderView:tableView
                                                                   action:action];
                        });
                    }
                    break;
                }
                default:
                {
                    //!!拿contactMainType 所對應的原始資料欄位
                    
                    NSArray *fieldArray = [[self.contactInfoSectionController fieldDictionaryFromSectionModelForSearching:NO] objectForKey:[NSNumber numberWithUnsignedInteger:sectionModel.contactMainType]];
                    if(fieldArray==nil || fieldArray.count<=indexPath.row)
                    {
                        break;
                    }
                    
                    //////////////////////////////////////////////////
                    
                    NSArray *targetArray = [NSArray arrayWithObject:[fieldArray objectAtIndex:indexPath.row]];
                    
                    NSDictionary *targetDictionary = [NSDictionary dictionaryWithObject:targetArray forKey:[NSNumber numberWithUnsignedInteger:sectionModel.contactMainType]];
                    
                    [self.delegate contactInfoViewController:self didSelectValueDictionary:targetDictionary];

                    break;
                }
            }

        } while (0);
    }
}


//================================================================================
//
//================================================================================
- (CGFloat)tableView:(UITableView *)tableView estimatedHeightForRowAtIndexPath:(NSIndexPath *)indexPath
{
    return [self contactInfoTableViewCellWithDefaultHeight:tableView.estimatedRowHeight indexPath:indexPath];
}


//================================================================================
//
//================================================================================
- (CGFloat)tableView:(UITableView *)tableView heightForRowAtIndexPath:(NSIndexPath *)indexPath
{
    CGFloat rowHeight = 0;
    
    if([[[UIDevice currentDevice] systemVersion] floatValue]<7.0)
    {
        rowHeight = [self contactInfoTableViewCellWithDefaultHeight:tableView.rowHeight indexPath:indexPath];
        
    }
    else
    {
        rowHeight = [self contactInfoTableViewCellWithDefaultHeight:tableView.estimatedRowHeight indexPath:indexPath];
    }
    
    return rowHeight;
}


//================================================================================
// 
//================================================================================
- (CGFloat)tableView:(UITableView *)tableView heightForHeaderInSection:(NSInteger)section
{
    if (section==0)
    {
        return ContactInfoViewController_TableSectionEdgeHeight;
    }
    else
    {
        return ContactInfoViewController_TableSectionHeaderHeight;
    }
}


//================================================================================
//
//================================================================================
- (UIView *)tableView:(UITableView *)tableView viewForHeaderInSection:(NSInteger)section
{
    if (section==0)
    {
        UIView *headerView = [[[UIView alloc] initWithFrame:CGRectMake(0, 0, self.view.bounds.size.width, ContactInfoViewController_TableSectionEdgeHeight)] autorelease];
        headerView.backgroundColor = ContactInfoViewControllerTableViewSectionHeaderBackgroundColor;
        
        return headerView;
    }
    else
    {
        UIView *headerView = [[[UIView alloc] initWithFrame:CGRectMake(0, 0, self.view.bounds.size.width, ContactInfoViewController_TableSectionHeaderHeight)] autorelease];
        headerView.backgroundColor = ContactInfoViewControllerTableViewSectionHeaderBackgroundColor;
        
        return headerView;
    }
}


//================================================================================
//
//================================================================================
- (void)updateInfoLayoutWithKeyboardHeight:(CGFloat)keyboardHeight
{
    if (keyboardHeight>0)
    {
        [self updateHeaderSubviewsWithOffset:CGPointZero];
        [self.contactInfoTableView setContentOffset:CGPointZero];
        
        CGRect frame = self.navigationController.view.frame;
        frame.size.height+=keyboardHeight;
        self.navigationController.view.frame = frame;
        
        CGRect tableFrame = self.contactInfoTableView.frame;
        tableFrame.size.height+=keyboardHeight;
        self.contactInfoTableView.frame = tableFrame;
        
        [self updateHeaderSubviewsWithOffset:CGPointZero];
        [self addTableFooterView];
    }
    else
    {
        [self updateHeaderSubviewsWithOffset:CGPointZero];
        [self.contactInfoTableView setContentOffset:CGPointZero];
        
        CGRect frame = self.navigationController.view.frame;
        frame.size.height+=keyboardHeight;
        self.navigationController.view.frame = frame;
        
        CGRect tableFrame = self.contactInfoTableView.frame;
        tableFrame.size.height+=keyboardHeight;
        self.contactInfoTableView.frame = tableFrame;
        
        [self updateHeaderSubviewsWithOffset:CGPointZero];
        [self addTableFooterView];
    }
}

@end
