//
//  WCContactEditSectionController.m
//
//  Created by Howard.Lin on 13/9/27.
//  Copyright (c) 2013年 Penpower. All rights reserved.
//

// define
#import "ContactApperanceDefine.h"
#import "WCContactEditViewController+SettingKey.h"

// controller
#import "PPSettingsController.h"
#import "WCContactEditSectionController.h"
#import "WCABDataController.h"

// model
#import "ContactDataExchangeModel.h"
#import "SalesforceEditDataModel.h"
#import "Office365EditDataModel.h"

// tool
#import "PPCountryCodeConvert.h"

// extension
#import "NSString+Additions.h"
#import "PPSectionController+Contact.h"


////////////////////////////////////////////////////////////////////////////////////////////////////

@interface WCContactEditSectionController()

@property (nonatomic,assign) NSMutableArray *extraSubTypeArray;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation WCContactEditSectionController

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Dellocating Objects


- (id)init
{
    if(self = [super init])
    {
#if defined (PRODUCTLINE_WCSF)
        //!!設定資料來源
        _dataModel = [[SalesforceEditDataModel alloc] init];
#elif defined (PRODUCTLINE_WCO365)
        _dataModel = [[Office365EditDataModel alloc] init];
#else
        _dataModel = [[ContactEditDataModel alloc] init];
#endif
        
        //////////////////////////////////////////////////
        
        _extraSubTypeArray = [[NSMutableArray array] retain];
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    [_dataModel release];
    _dataModel = nil;
    
    [_extraSubTypeArray removeAllObjects];
    [_extraSubTypeArray release];
    _extraSubTypeArray = nil;
    
    [_noteFieldArray release];
    _noteFieldArray = nil;
    
    [_systemCategoryArray release];
    _systemCategoryArray = nil;
    
    //////////////////////////////////////////////////
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Method

//================================================================================
//
//================================================================================
- (NSString *)firstNonOccurStringForPPContactMainType:(PPContactMainType)contactMainType
                                        compareTitles:(NSArray *)titles
{
    NSString *result = WCCV_MLS_Other;
    
    do
    {
        if(titles==nil || titles.count<=0)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSArray *defaultTitles = [[self.dataModel class] totalTitleArrayForPPContactMainType:contactMainType];
        
        if(defaultTitles==nil || defaultTitles.count<=0)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        //預設取第一個當title
        result = [defaultTitles objectAtIndex:0];
        
        //////////////////////////////////////////////////
        
        BOOL found;
        
        for(NSString *candidateString in defaultTitles)
        {
            found = NO;
            
            for(NSString *compareString in titles)
            {
                if([candidateString isEqualToString:compareString]==YES)
                {
                    found = YES;
                    break;
                }
            }
            
            //////////////////////////////////////////////////
            
            if(found==NO)
            {
                result = candidateString;
                break;
            }
        }
        
    } while (0);
    
    return result;
}


//================================================================================
//
//================================================================================
- (NSMutableArray *)existRowModelTitlesInSection:(NSUInteger)sectionIndex
                                    forSearching:(BOOL)searching
{
    NSMutableArray *titles = [NSMutableArray array];
    
    do
    {
        if(self.sectionModelsForDefault.count<sectionIndex)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        PPSectionModel *sectionModel = [self.sectionModelsForDefault objectAtIndex:sectionIndex];
        
        if(sectionModel==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        
        for (PPRowModel *rowModel in sectionModel.rowModels)
        {
            [titles addObject:rowModel.text];
        }
        
    } while (0);
    
    return titles;
}


//================================================================================
//
//================================================================================
- (PPRowModel *)nextRowModelForPPContactMainType:(PPContactMainType)contactMainType
                                             row:(NSUInteger)row
                                        rowTitle:(NSString *)title
                       tableViewCellEditingStyle:(UITableViewCellEditingStyle)tableViewCellEditingStyle
{
    PPRowModel *nextRowModel = nil;
    
    
    do
    {
        PPContactRowObject *contactRowObject = [PPContactRowObject ppContactRowObjectWithPPContactMainType:contactMainType
                                                                                        ppContactLabelType:[PPContactRowObject ppContactLabelTypeForPPContactMainType:contactMainType labelText:title]
                                                                                 tableViewCellEditingStyle:tableViewCellEditingStyle
                                                                               ppContactTableViewCellStyle:[[self.dataModel class] contactTableViewCellStyleForContactMainType:contactMainType]
                                                selected:NO];
        
        if(contactRowObject==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        nextRowModel = [PPRowModel rowModelWithRow:row
                                              text:title
                                        detailText:nil
                                            object:contactRowObject];
        
        if(nextRowModel!=nil)
        {
            [nextRowModel setPPContactFieldSource:PPContactFieldSource_UserCreate];
            
            //////////////////////////////////////////////////
            
            NSMutableArray *placeholderArray = nil;
            
            if(contactMainType==PPContactMainType_Name)
            {
                NSMutableArray *currentSubTypeArray = [[NSMutableArray alloc] initWithArray:self.extraSubTypeArray];
                
                [currentSubTypeArray addObject:@(PPContactSubType_Name_LastName)];
                
                if(self.showNamePhonetic==YES)
                {
                    [currentSubTypeArray addObject:@(PPContactSubType_Name_LastNamePhonetic)];
                }
                
                [currentSubTypeArray addObject:@(PPContactSubType_Name_MiddleName)];
                [currentSubTypeArray addObject:@(PPContactSubType_Name_FirstName)];
                
                if(self.showNamePhonetic==YES)
                {
                    [currentSubTypeArray addObject:@(PPContactSubType_Name_FirstNamePhonetic)];
                }
                
                [currentSubTypeArray addObject:@(PPContactSubType_Name_Prefix)];
                [currentSubTypeArray addObject:@(PPContactSubType_Name_Surffix)];
                
                placeholderArray = [[self.dataModel class] customSectionPlaceholderArrayForPPContactMainType:PPContactMainType_Name
                                                                                       ppContactSubTypeArray:currentSubTypeArray];
                
                [currentSubTypeArray removeAllObjects];
                [currentSubTypeArray release];
            }
            else if(contactMainType==PPContactMainType_Company)
            {
                NSMutableArray *currentSubTypeArray = [[NSMutableArray alloc] initWithArray:self.extraSubTypeArray];
                
                [currentSubTypeArray addObject:@(PPContactSubType_Company_Name)];
                
                if(self.showCompanyPhonetic==YES)
                {
                    [currentSubTypeArray addObject:@(PPContactSubType_Company_NamePhonetic)];
                }
                
                [currentSubTypeArray addObject:@(PPContactSubType_Company_JobTitle)];
                [currentSubTypeArray addObject:@(PPContactSubType_Company_Department)];
                
                placeholderArray = [[self.dataModel class] customSectionPlaceholderArrayForPPContactMainType:PPContactMainType_Company
                                                                                       ppContactSubTypeArray:currentSubTypeArray];
                
                [currentSubTypeArray removeAllObjects];
                [currentSubTypeArray release];
                
            }
            else if(contactMainType==PPContactMainType_Address)
            {
                placeholderArray = [NSMutableArray arrayWithArray:[[self.dataModel class] addressPlaceholderArrayWithCountryCode:[PPCountryCodeConvert defaultCountryCode]]];
            }
            else
            {
                [contactRowObject.inputValueDictionary setObject:@""
                                                          forKey:@(PPContactSubType_None)];
            }
            
            //////////////////////////////////////////////////
            
            if(contactMainType==PPContactMainType_Address ||
               contactMainType==PPContactMainType_Name ||
               contactMainType==PPContactMainType_Company)
            {
                for(NSString *placeholder in placeholderArray)
                {
                    NSString *value = @"";
                    
                    if([placeholder isEqualToString:WCCV_MLS_Address_Country]==YES)
                    {
                        NSString *countryCode = [PPCountryCodeConvert defaultCountryCode];
                        
                        if(countryCode.length>0)
                        {
                            value = [PPCountryCodeConvert localizeStringFromCountryCode:countryCode];
                            
                            [contactRowObject.inputValueDictionary setObject:countryCode
                                                                      forKey:@(PPContactSubType_Address_CountryCode)];
                            
                            //////////////////////////////////////////////////
                            
                            [contactRowObject.inputValueDictionary setObject:@([PPCountryCodeConvert defaultRecogLangWithCountryCode:countryCode])
                                                                      forKey:@(PPContactSubType_Address_RecognitionLanguageCode)];
                        }
                    }
                    
                    //////////////////////////////////////////////////
                    
                    [contactRowObject.inputValueDictionary setObject:value
                                                              forKey:@([contactRowObject contactSubTypeWithLocalizedPlaceholder:placeholder])];
                    
                }
            }
        }
        
    } while (0);
    
    return nextRowModel;
}


//================================================================================
//
//================================================================================
- (void)insertPPContactSectionModel:(PPContactSectionModel *)ppContactSectionModel
                  ppContactMainType:(PPContactMainType)contactMainType
                       forSearching:(BOOL)searching;
{
    if(ppContactSectionModel!=nil)
    {
        NSInteger targetInsertSection = 0;
        NSMutableArray *sectionItemArray = [self sectionModelsForSearching:NO];
        
        if(sectionItemArray!=nil)
        {
            NSUInteger totalSectionNumber  = sectionItemArray.count;
            
            for (; targetInsertSection<totalSectionNumber ; targetInsertSection++)
            {
                PPContactSectionModel *contactSectionModel = [sectionItemArray objectAtIndex:targetInsertSection];
                
                //!!找到該插入的地方
                if(contactSectionModel.contactMainType>contactMainType)
                {
                    break;
                }
            }
            
            if(targetInsertSection-1<0)
            {
                targetInsertSection = 0;
            }
            
            [sectionItemArray insertObject:ppContactSectionModel atIndex:targetInsertSection];
        }
    }
}


//================================================================================
//
//================================================================================
- (void)updateDataFromPPContactSectionModel:(PPContactSectionModel *)ppContactSectionModel
                                     orders:(NSArray *)orders
                               forSearching:(BOOL)searching
{
    @autoreleasepool
    {
        if(ppContactSectionModel!=nil && ppContactSectionModel.rowModels.count>0)
        {
            NSInteger section = [self sectionForPPContactMainType:ppContactSectionModel.contactMainType forSearching:NO];
            
            if(section!=NSNotFound)
            {
                BOOL finalDataIsAdd = NO;
                
                PPContactSectionModel *existSectionModel = [[self sectionModelsForSearching:NO] objectAtIndex:section];
                
                PPRowModel *existLastRowModel = [existSectionModel.rowModels lastObject];
                
                //最後一筆為新增xx資料
                if(existLastRowModel!=nil && [existLastRowModel ppContactRowObjectOfRowModel].editingStyle==UITableViewCellEditingStyleInsert)
                {
                    //!!若原有欄位(將原有的最後一筆新增xx資料先扣掉) +加進來的欄位超過最欄位限制,即移除新增xx資料
                    if(((existSectionModel.rowModels.count-1)+ppContactSectionModel.rowModels.count)>=[[self.dataModel class]maxNumberOfRowForPPContactMainType:ppContactSectionModel.contactMainType])
                    {
                        [existSectionModel.rowModels removeObject:existLastRowModel];
                    }
                    else
                    {
                        finalDataIsAdd = YES;
                    }
                }
                
                ///////////////////////////////////////////////////////
                
                if(finalDataIsAdd==NO)
                {
                    //!! spec 要求,  other 欄位(統一編號，暱稱都是有效欄位)
                    if(existSectionModel.contactMainType==PPContactMainType_Other)
                    {
                        // 檢查是否有統一編號欄位, 暱稱欄位(必出現)
                        BOOL hasUniNumberEntry = NO;
                        BOOL hasNickNameEntry = NO;
                        BOOL hasBirthdayEntry = NO;
                        BOOL hasWorkURLEntry = NO;

                        for(PPRowModel *newRowModel in ppContactSectionModel.rowModels)
                        {
                            if([newRowModel ppContactRowObjectOfRowModel].contactLabelType==PPContactLabelType_Other_NickName)
                            {
                                hasNickNameEntry = YES;
                            }
                            else if([newRowModel ppContactRowObjectOfRowModel].contactLabelType==PPContactLabelType_Other_UnifiedBusinessNumber)
                            {
                                hasUniNumberEntry = YES;
                            }
                            else if([newRowModel ppContactRowObjectOfRowModel].contactLabelType==PPContactLabelType_Other_Birthday)
                            {
                                hasBirthdayEntry = YES;
                            }
                            else if([newRowModel ppContactRowObjectOfRowModel].contactLabelType==PPContactLabelType_Other_URL_Work)
                            {
                                hasWorkURLEntry = YES;
                            }

                            //////////////////////////////////////////////////
                            
                            [existSectionModel.rowModels addObject:newRowModel];
                        }
                        
                        //////////////////////////////////////////////////
                        
                        //踢除無效欄位(沒有初始值)
                        if(hasNickNameEntry==YES ||
                           hasUniNumberEntry==YES||
                           hasBirthdayEntry==YES||
                           hasWorkURLEntry==YES)
                        {
                            for(NSInteger existRowModelIndex=existSectionModel.rowModels.count-1; existRowModelIndex>=0; existRowModelIndex--)
                            {
                                PPRowModel *existRowModel = [existSectionModel.rowModels objectAtIndex:existRowModelIndex];
                                
                                //沒有有效值，移掉預設欄位
                                
                                // 檢查暱稱欄位
                                if(hasNickNameEntry==YES &&
                                   [existRowModel ppContactRowObjectOfRowModel].contactLabelType==PPContactLabelType_Other_NickName &&
                                   [[existRowModel ppContactRowObjectOfRowModel] checkInputValueDictionaryKeyHasValidValue]==NO)
                                {
                                    [existSectionModel.rowModels removeObject:existRowModel];
                                    
                                    existRowModelIndex++;
                                }
                                // 檢查統一編號欄位
                                else if(hasUniNumberEntry==YES &&
                                        [existRowModel ppContactRowObjectOfRowModel].contactLabelType==PPContactLabelType_Other_UnifiedBusinessNumber &&
                                        [[existRowModel ppContactRowObjectOfRowModel] checkInputValueDictionaryKeyHasValidValue]==NO)
                                {
                                    [existSectionModel.rowModels removeObject:existRowModel];
                                    
                                    existRowModelIndex++;
                                }
                                else if(hasBirthdayEntry==YES &&
                                   [existRowModel ppContactRowObjectOfRowModel].contactLabelType==PPContactLabelType_Other_Birthday &&
                                   [[existRowModel ppContactRowObjectOfRowModel] checkInputValueDictionaryKeyHasValidValue]==NO)
                                {
                                    [existSectionModel.rowModels removeObject:existRowModel];
                                    
                                    existRowModelIndex++;
                                }
                                else if(hasWorkURLEntry==YES &&
                                   [existRowModel ppContactRowObjectOfRowModel].contactLabelType==PPContactLabelType_Other_URL_Work &&
                                   [[existRowModel ppContactRowObjectOfRowModel] checkInputValueDictionaryKeyHasValidValue]==NO)
                                {
                                    [existSectionModel.rowModels removeObject:existRowModel];
                                    
                                    existRowModelIndex++;
                                }
                            }
                            
                            //////////////////////////////////////////////////
                            
                            //排序, 統一編號，暱稱

                            NSArray *sortRowModes = [existSectionModel.rowModels sortedArrayUsingSelector:@selector(labelTypeCompare:)];
                            
                            if(sortRowModes!=nil)
                            {
                                [existSectionModel.rowModels removeAllObjects];
                                [existSectionModel.rowModels addObjectsFromArray:sortRowModes];
                            }
                            
                            //////////////////////////////////////////////////
                            
                            //處理欄位是否能編輯的類型 UITableViewCellEditingStyle
                            //目前各允許一個欄位
                            BOOL archiveNickNameMaxCount = NO;
                            BOOL archiveUniNumberMaxCount = NO;
                            
                            
                            for(PPRowModel *rowModel in existSectionModel.rowModels)
                            {
                                if(archiveNickNameMaxCount==NO &&
                                   [rowModel ppContactRowObjectOfRowModel].contactLabelType==PPContactLabelType_Other_NickName)
                                {
                                    archiveNickNameMaxCount = YES;
                                    
                                    //////////////////////////////////////////////////
                                    
                                    [rowModel ppContactRowObjectOfRowModel].editingStyle=UITableViewCellEditingStyleNone;
                                }
                                else if(archiveUniNumberMaxCount==NO &&
                                        [rowModel ppContactRowObjectOfRowModel].contactLabelType==PPContactLabelType_Other_UnifiedBusinessNumber)
                                {
                                    archiveUniNumberMaxCount = YES;
                                    
                                    //////////////////////////////////////////////////
                                    
                                    [rowModel ppContactRowObjectOfRowModel].editingStyle=UITableViewCellEditingStyleNone;
                                }
                                
                                //////////////////////////////////////////////////
                                
                                if(archiveNickNameMaxCount==YES &&
                                   archiveUniNumberMaxCount==YES)
                                {
                                    break;
                                }
                            }
                        }
                    }
                    else
                    {
                        for(NSInteger existRowModelIndex=0; existRowModelIndex<existSectionModel.rowModels.count; existRowModelIndex++)
                        {
                            PPRowModel *existRowModel = [existSectionModel.rowModels objectAtIndex:existRowModelIndex];
                            
                            //沒有有效值，移掉預設欄位
                            if([[existRowModel ppContactRowObjectOfRowModel] checkInputValueDictionaryKeyHasValidValue]==NO)
                            {
                                [existSectionModel.rowModels removeObject:existRowModel];
                                
                                existRowModelIndex--;
                            }
                        }
                        
                        //////////////////////////////////////////////////
                        
                        for(PPRowModel *newRowModel in ppContactSectionModel.rowModels)
                        {
                            [existSectionModel.rowModels addObject:newRowModel];
                        }
                    }
                }
                else
                {
                    for(PPRowModel *newRowModel in ppContactSectionModel.rowModels)
                    {
                        if (existSectionModel.rowModels.count>1)
                        {
                            [existSectionModel.rowModels insertObject:newRowModel atIndex:existSectionModel.rowModels.count-1];
                        }
                        else
                        {
                            [existSectionModel.rowModels insertObject:newRowModel atIndex:0];
                        }
                    }
                }
            }
            else
            {
                //!!自訂欄位不允許新增
                if(ppContactSectionModel.contactMainType!=PPContactMainType_UserDefine)
                {
                    //!!若進來的欄位沒有超過最大欄位的限制，加入"新增xx"的欄位
                    if(ppContactSectionModel.rowModels.count<[[self.dataModel class] maxNumberOfRowForPPContactMainType:ppContactSectionModel.contactMainType])
                    {
                        PPContactSectionModel *defaultSectionModel = [self defaultContactSectionModelForContactMainType:ppContactSectionModel.contactMainType];
                        
                        for (PPRowModel *rowModel in defaultSectionModel.rowModels)
                        {
                            [ppContactSectionModel.rowModels addObject:rowModel];
                        }
                    }
                }
                
                //////////////////////////////////////////////////
                
                // 根據順序開始加入 sectionModel
                NSUInteger insertedIndex = NSNotFound;
                
                if([orders count]>0)
                {
                    NSInteger insertSectionIndex = [orders indexOfObject:@(ppContactSectionModel.contactMainType)];
                    
                    //////////////////////////////////////////////////
                    
                    for(PPContactSectionModel *existSectionModel in self.sectionModelsForDefault)
                    {
                        NSInteger existSectionIndex = [orders indexOfObject:@(existSectionModel.contactMainType)];
                        
                        // 新加入的 sectionModel 要插在已存在的 sectionModel 之前
                        if(insertSectionIndex<=existSectionIndex)
                        {
                            insertedIndex = [self.sectionModelsForDefault indexOfObject:existSectionModel];
                            break;
                        }
                    }
                }
                
                //////////////////////////////////////////////////
                
                if(insertedIndex==NSNotFound)
                {
                    [self.sectionModelsForDefault addObject:ppContactSectionModel];
                }
                else
                {
                    [self.sectionModelsForDefault insertObject:ppContactSectionModel atIndex:insertedIndex];
                }
            }
            
            //////////////////////////////////////////////////
            
            //!!計算點選過的 row 欄位
            for (PPRowModel *newRowModel in ppContactSectionModel.rowModels)
            {
                PPContactRowObject *contactRowObject = [newRowModel ppContactRowObjectOfRowModel];
                
                if(contactRowObject!=nil)
                {
                    for(id key in contactRowObject.inputValueDictionary.allKeys)
                    {
                        if([key unsignedIntegerValue]!=PPContactSubType_None)
                        {
                            //!!紀錄目前cell欄位選擇
                            [self.extraSubTypeArray addObject:key];
                        }
                    }
                }
                
                //!!同個section裡面，欄位長相一致，所以做一次就行了
                break;
            }
            
        }
    }
}


//================================================================================
//
//================================================================================
- (void)addOneNameRowModelIfNeedOrNot
{
    do
    {
        NSInteger section = [self sectionForPPContactMainType:PPContactMainType_Name forSearching:NO];
        
        if(section==NSNotFound ||
           [self.sectionModelsForDefault count]<=section)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        PPContactSectionModel *sectionModel = [[self sectionModelsForDefault] objectAtIndex:section];
        
        //不是姓名欄位，或姓名欄位已有存在的有效資料，則不用加姓名欄位
        if(sectionModel==nil ||
           sectionModel.contactMainType!=PPContactMainType_Name)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(sectionModel.rowModels.count>=2)
        {
            //檢查是否姓名有空的預設欄位
            
            PPRowModel *rowModel = [[sectionModel rowModels] firstObject];
            
            if(rowModel!=nil)
            {
                if ([[rowModel ppContactRowObjectOfRowModel] checkInputValueDictionaryKeyHasValidValue]==NO)
                {
                    [sectionModel.rowModels removeObjectAtIndex:0];
                }
            }
        }
        else
        {
            NSMutableArray *titleArray  = [[self.dataModel class] defaultTitleArrayForPPContactMainType:sectionModel.contactMainType];
            
            if(titleArray==nil || titleArray.count<=0)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            PPRowModel *nextRowModel = [self nextRowModelForPPContactMainType:sectionModel.contactMainType
                                                                          row:sectionModel.rowModels.count
                                                                     rowTitle:[titleArray lastObject]
                                                    tableViewCellEditingStyle:UITableViewCellEditingStyleDelete];
            
            if(nextRowModel==nil)
            {
                break;
            }
            
            [sectionModel.rowModels insertObject:nextRowModel atIndex:0];
        }
    }
    while (0);
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Custom SystemCategoryContactDataExchangeModel Generation Method

//================================================================================
//
//================================================================================
- (ContactDataExchangeModel *)customSystemCategoryContactDataExchangeModel
{
    ContactDataExchangeModel *contactDataExchangeModel = [[ContactDataExchangeModel alloc] init];
    
    if(contactDataExchangeModel!=nil)
    {
        contactDataExchangeModel.labelType = PPContactLabelType_SystemGroup_Save;
        contactDataExchangeModel.labelText = PPContactLabelText_SaveIPhoneGroup;
        
        //////////////////////////////////////////////////
        
        if(contactDataExchangeModel.propertyDictionary!=nil)
        {
            if ([PPSettingsController integerValueWithKey:WCContactViewController_SettingsKey_EnableSystemContactCategory]==1)
            {
                [contactDataExchangeModel.propertyDictionary setObject:@(YES)
                                                                forKey:@(PPContactSubType_SaveToSystemGroup)];
            }
            else
            {
                [contactDataExchangeModel.propertyDictionary setObject:@(NO)
                                                                forKey:@(PPContactSubType_SaveToSystemGroup)];
            }
        }
    }
    
    return [contactDataExchangeModel autorelease];
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Method

//================================================================================
//
//================================================================================
- (BOOL)checkArchivingUpperLimitAndRemoveRowModelForSection:(NSUInteger)section forSearching:(BOOL)searching
{
    BOOL result = YES;
    
    do
    {
        if(section>=[self sectionModelsForSearching:searching].count)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        PPContactSectionModel *sectionModel = [[self sectionModelsForSearching:searching] objectAtIndex:section];
        
        if(sectionModel==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        //!! +1 是新增xx欄位並不算在有效欄位內
        if(sectionModel.rowModels.count>=[[self.dataModel class] maxNumberOfRowForPPContactMainType:[self contactMainTypeInSection:section forSearching:NO]]+1)
        {
            [sectionModel.rowModels removeObjectAtIndex:sectionModel.rowModels.count-1];
            
            break;
        }
        
        //////////////////////////////////////////////////
        
        result = NO;
        
    } while (0);
    
    
    return result;
}


//================================================================================
//
//================================================================================
- (BOOL)changeRowModelText:(NSString *)text atIndexPath:(NSIndexPath *)indexPath
{
    BOOL result = NO;
    
    do
    {
        if(indexPath==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        PPRowModel *rowModel = [self rowModelAtIndexPath:indexPath forSearching:NO];
        
        if(rowModel!=nil && [rowModel.text isEqualToString:text]==NO)
        {
            rowModel.text = text;
            
            //////////////////////////////////////////////////
            
            PPContactRowObject *contactRowObject = [rowModel ppContactRowObjectOfRowModel];
            
            if(contactRowObject!=nil)
            {
                contactRowObject.contactLabelType = [PPContactRowObject ppContactLabelTypeForPPContactMainType:contactRowObject.contactMainType labelText:rowModel.text];
            }
            
            [rowModel setPPContactFieldSource:PPContactFieldSource_UserCreate];
            
            result = YES;
        }
        
    } while (0);
    
    return result;
}


//================================================================================
//
//================================================================================
- (BOOL)checkIllegalAndSaveRowModelInSection:(NSUInteger)section forSearching:(BOOL)searching
{
    BOOL result = NO;
    
    do
    {
        if([self sectionModelsForSearching:searching].count<=section)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        PPContactSectionModel *sectionModel = [[self sectionModelsForSearching:searching]objectAtIndex:section];
        
        if(sectionModel==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        //!!檢查此section裡面是否有"新增xx"欄位的存在,才能判斷是否加入"新增xx"的欄位
        
        //!!此section欄位沒有任何row欄位
        if(sectionModel.rowModels.count<=0)
        {
            result = YES;
        }
        //!!此section仍然超過最大上限
        else if(sectionModel.rowModels.count>=[[self.dataModel class]maxNumberOfRowForPPContactMainType:sectionModel.contactMainType])
        {
            result = NO;
        }
        else
        {
            PPRowModel *rowModel = [sectionModel.rowModels objectAtIndex:(sectionModel.rowModels.count-1)];
            
            if(rowModel==nil || [rowModel ppContactRowObjectOfRowModel]==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            PPContactRowObject *contactRowObject = rowModel.object;
            
            //!!發現最尾一筆不是新增xx的欄位
            if(contactRowObject.editingStyle!=UITableViewCellEditingStyleInsert)
            {
                result = YES;
            }
        }
        
        //////////////////////////////////////////////////
        
        //!!新增"新增xx"欄位
        if(result==YES)
        {
            PPContactSectionModel *newSectionModel =[[self.dataModel class] ppContactSectionModelForPPContactMainType:sectionModel.contactMainType];
            
            if(newSectionModel==nil || newSectionModel.rowModels.count<=0)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            [sectionModel.rowModels addObject:[newSectionModel.rowModels objectAtIndex:newSectionModel.rowModels.count-1]];
        }
        
    } while (0);
    
    return result;
}


//================================================================================
//
//================================================================================
- (BOOL)deleteRowModelAtIndexPath:(NSIndexPath *)indexPath
                     forSearching:(BOOL)searching
{
    BOOL result = NO;
    
    @autoreleasepool
    {
        do
        {
            if(indexPath==nil || [self sectionModelsForSearching:searching].count<=indexPath.section)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            PPContactSectionModel *sectionModel = [[self sectionModelsForSearching:searching]objectAtIndex:indexPath.section];
            
            if(sectionModel==nil || sectionModel.rowModels.count<=indexPath.row)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            [sectionModel.rowModels removeObjectAtIndex:indexPath.row];
            
            //////////////////////////////////////////////////
            
            result = YES;
            
        } while (0);
    }
    
    
    return result;
}


//================================================================================
//
//================================================================================
- (BOOL)deleteSectionModelAtIndex:(NSUInteger)section
                     forSearching:(BOOL)searching
{
    if(section>=[self sectionModelsForSearching:searching].count)
    {
        return NO;
    }
    else
    {
        [[self sectionModelsForSearching:searching]removeObjectAtIndex:section];
        
        return YES;
    }
}


//================================================================================
//
//================================================================================
- (BOOL)expandRowModelFieldInPPContanctSectionWithTitle:(NSString *)title
                                      ppContactMainType:(PPContactMainType)contactMainType
                                           forSearching:(BOOL)searching
{
    BOOL result = NO;
    
    do
    {
        @autoreleasepool
        {
            PPContactRowObject *checkContactRowObject = [[[PPContactRowObject alloc] init] autorelease];
            
            checkContactRowObject.contactMainType = contactMainType;
            
            NSMutableArray *subTypeArray = [NSMutableArray array];
            
            
            if(contactMainType==PPContactMainType_Name && [title isEqualToString:WCCV_MLS_Phonetic]==YES)
            {
                [subTypeArray addObject:@([checkContactRowObject contactSubTypeWithLocalizedPlaceholder:WCCV_MLS_Name_FirstPhonetic])];
                [subTypeArray addObject:@([checkContactRowObject contactSubTypeWithLocalizedPlaceholder:WCCV_MLS_Name_LastPhonetic])];
            }
            else if(contactMainType==PPContactMainType_Company && [title isEqualToString:WCCV_MLS_Phonetic]==YES)
            {
                [subTypeArray addObject:@([checkContactRowObject contactSubTypeWithLocalizedPlaceholder:WCCV_MLS_Company_Phonetic])];
            }
            else
            {
                [subTypeArray addObject:@([checkContactRowObject contactSubTypeWithLocalizedPlaceholder:title])];
            }
            
            
            //////////////////////////////////////////////////
            
            for(NSNumber *subType in subTypeArray)
            {
                //!!紀錄目前cell欄位選擇
                [self.extraSubTypeArray addObject:@([subType unsignedIntegerValue])];
                
                //////////////////////////////////////////////////
                
                //!!此時不存在任何對應資料，則無法新增
                NSInteger targetSection = [self sectionForPPContactMainType:contactMainType forSearching:searching];
                
                if(targetSection==NSNotFound ||
                   targetSection>=[self sectionModelsForSearching:NO].count)
                {
                    break;
                }
                
                //////////////////////////////////////////////////
                
                PPContactSectionModel *contactSectionModel = [[self sectionModelsForSearching:searching]objectAtIndex:targetSection];
                
                //!!contactSectionModel.rowModels.count < 2,為新增姓名欄位，無任何資料
                if(targetSection>=[self sectionModelsForSearching:searching].count || contactSectionModel.rowModels.count<2)
                {
                    continue;
                }
                
                //////////////////////////////////////////////////
                
                NSMutableArray *placeholderArray = [[self.dataModel class] customSectionPlaceholderArrayForPPContactMainType:contactMainType
                                                                                                       ppContactSubTypeArray:[NSArray arrayWithObject:subType]];
                
                if(placeholderArray==nil || placeholderArray.count<=0)
                {
                    break;
                }
                
                
                //////////////////////////////////////////////////
                
                for(NSUInteger rowModelIndex = 0 ; rowModelIndex < contactSectionModel.rowModels.count ; rowModelIndex++)
                {
                    PPRowModel *rowModel = [contactSectionModel.rowModels objectAtIndex:rowModelIndex];
                    
                    if(rowModel!=nil)
                    {
                        PPContactRowObject *contactRowObject = [rowModel ppContactRowObjectOfRowModel];
                        
                        if(contactRowObject==nil)
                        {
                            break;
                        }
                        
                        //////////////////////////////////////////////////
                        
                        if(contactRowObject.editingStyle == UITableViewCellEditingStyleInsert)
                        {
                            continue;
                        }
                        
                        //////////////////////////////////////////////////
                        
                        for(NSString *itemString in placeholderArray)
                        {
                            id object = [contactRowObject.inputValueDictionary objectForKey:@([contactRowObject contactSubTypeWithLocalizedPlaceholder:itemString])];
                            
                            if(object==nil)
                            {
                                [contactRowObject.inputValueDictionary setObject:@"" forKey:@([contactRowObject contactSubTypeWithLocalizedPlaceholder:itemString])];
                            }
                            else
                            {
                                [contactRowObject.inputValueDictionary setObject:object forKey:@([contactRowObject contactSubTypeWithLocalizedPlaceholder:itemString])];
                            }
                        }
                    }
                }
                
            }
            
            result = YES;
        }
        
    } while (0);
    
    return result;
}


//================================================================================
//
//================================================================================
- (BOOL)insertNewSectionModelForPPContactMainType:(PPContactMainType)contactMainType
                                     forSearching:(BOOL)searching;

{
    BOOL result = NO;
    
    do
    {
        if(contactMainType!=PPContactMainType_Name &&
           contactMainType!=PPContactMainType_Other &&
           contactMainType!=PPContactMainType_InstantMessage &&
           contactMainType!=PPContactMainType_Note &&
           contactMainType!=PPContactMainType_PublicNote &&
           contactMainType!=PPContactMainType_BirthDay &&
           contactMainType!=PPContactMainType_Date &&
           contactMainType!=PPContactMainType_Phonetic &&
           contactMainType!=PPContactMainType_UserDefine)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        @autoreleasepool
        {
            NSMutableArray *titleLabelArray = [[self.dataModel class] defaultTitleArrayForPPContactMainType:contactMainType];
            
            if(titleLabelArray==nil || titleLabelArray.count<=0)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            PPContactRowObject *contactRowObject = [PPContactRowObject ppContactRowObjectWithPPContactMainType:contactMainType
                                                                                            ppContactLabelType:[PPContactRowObject ppContactLabelTypeForPPContactMainType:contactMainType labelText:[titleLabelArray objectAtIndex:0]]
                                                                                     tableViewCellEditingStyle:UITableViewCellEditingStyleInsert
                                                                                   ppContactTableViewCellStyle:PPContactTableViewCellStyle_Default
                                                    selected:NO];
            
            if(contactRowObject==nil)
            {
                break;
            }
            
            
            //////////////////////////////////////////////////
            
            PPContactSectionModel *contactSectionModel = [PPContactSectionModel ppContactSectionModelForPPContactMainType:contactMainType];
            
            if(contactSectionModel==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            PPRowModel *newRowModel = [PPRowModel rowModelWithRow:contactSectionModel.rowModels.count text:[titleLabelArray objectAtIndex:0] detailText:nil object:contactRowObject];
            
            [newRowModel setPPContactFieldSource:PPContactFieldSource_UserCreate];
            
            if(newRowModel==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            [contactSectionModel.rowModels addObject:newRowModel];
            
            [self insertPPContactSectionModel:contactSectionModel
                            ppContactMainType:contactMainType
                                 forSearching:NO];
            
            //////////////////////////////////////////////////
            
            result = YES;
            
        }
    } while (0);
    
    return result;
}


//================================================================================
//
//================================================================================
- (PPContactMainType)contactMainTypeInSection:(NSUInteger)section forSearching:(BOOL)searching
{
    PPContactMainType contactMainType = PPContactMainType_None;
    NSMutableArray *sectionModels = [self sectionModelsForSearching:searching];
    
    do
    {
        if(sectionModels==nil || sectionModels.count<=section)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        PPContactSectionModel *contactSectionModel = [sectionModels objectAtIndex:section];
        
        contactMainType = contactSectionModel.contactMainType;
        
    } while (0);
    
    return contactMainType;
}




//================================================================================
// MARK: 預設欄位
//================================================================================
- (PPContactSectionModel *)defaultContactSectionModelForContactMainType:(PPContactMainType)contactMainType
{
    PPContactSectionModel *sectionModel = nil;
    
    do
    {
        if(self.dataModel==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        switch (contactMainType)
        {
            case PPContactMainType_Name:
            {
                NSMutableArray *currentContactSubTypeArray = [[NSMutableArray alloc] initWithArray:self.extraSubTypeArray];
                
                [currentContactSubTypeArray addObject:@(PPContactSubType_Name_FirstName)];
                [currentContactSubTypeArray addObject:@(PPContactSubType_Name_LastName)];
                
                sectionModel = [[self.dataModel class] ppContactSectionModelForPPContactMainType:PPContactMainType_Name];
                
                [currentContactSubTypeArray removeAllObjects];
                [currentContactSubTypeArray release];
                
                break;
            }
            case PPContactMainType_Company:
            {
                NSMutableArray *currentContactSubTypeArray = [[NSMutableArray alloc] initWithArray:self.extraSubTypeArray];
                
                [currentContactSubTypeArray addObject:@(PPContactSubType_Company_Name)];
                [currentContactSubTypeArray addObject:@(PPContactSubType_Company_Department)];
                [currentContactSubTypeArray addObject:@(PPContactSubType_Company_JobTitle)];
                
                sectionModel = [[self.dataModel class] ppContactSectionModelForPPContactMainType:PPContactMainType_Company];
                
                [currentContactSubTypeArray removeAllObjects];
                [currentContactSubTypeArray release];
                
                break;
            }
            case PPContactMainType_UserDefine:
            {
                // !!使用者自訂欄位要額外處理，沒有預設欄位做法
                
                break;
            }
            case PPContactMainType_Other:
            case PPContactMainType_HeadImage:
            case PPContactMainType_Phone:
            case PPContactMainType_InstantMessage:
            case PPContactMainType_Note:
            case PPContactMainType_PublicNote:
            case PPContactMainType_Date:
            case PPContactMainType_Phonetic:
            case PPContactMainType_Email:
            case PPContactMainType_URL:
            case PPContactMainType_Address:
            case PPContactMainType_AddField:
            case PPContactMainType_BirthDay:
            case PPContactMainType_SocialNetwork:
            case PPContactMainType_Salutation:
            case PPContactMainType_Status:
            case PPContactMainType_SalesforceSaveAs:
            {
                sectionModel = [[self.dataModel class] ppContactSectionModelForPPContactMainType:contactMainType];
                
                break;
            }
            default:
            {
                break;
            }
        }
        
    } while (0);
    
    return sectionModel;
}


//================================================================================
//
//================================================================================
- (NSIndexPath *)indexPathForinsertingRowModelAtSection:(NSUInteger)section forSearching:(BOOL)searching;
{
    NSIndexPath *indexPath = nil;
    
    @autoreleasepool
    {
        NSMutableArray *targetSectionArray = [self sectionModelsForSearching:NO];
        
        do
        {
            if(targetSectionArray==nil || targetSectionArray.count<=0 || section>=targetSectionArray.count)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            PPContactSectionModel *contactSectionModel = [targetSectionArray objectAtIndex:section];
            
            if(contactSectionModel==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            //!!取cell的標題集
            NSMutableArray *titleArray  = [[self.dataModel class] defaultTitleArrayForPPContactMainType:contactSectionModel.contactMainType];
            
            if(titleArray==nil || titleArray.count<=0)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            NSString *nextString = nil;
            
            if(contactSectionModel.contactMainType==PPContactMainType_Company ||
               contactSectionModel.contactMainType==PPContactMainType_Name ||
               contactSectionModel.contactMainType==PPContactMainType_UserDefine ||
               contactSectionModel.contactMainType==PPContactMainType_HeadImage)
            {
                nextString = [titleArray objectAtIndex:titleArray.count-1];
            }
            else
            {
                nextString =  [self firstNonOccurStringForPPContactMainType:contactSectionModel.contactMainType
                                                              compareTitles:[self existRowModelTitlesInSection:section forSearching:NO]];
            }
            
            //////////////////////////////////////////////////
            
            //!產生rowModel
            PPRowModel *nextRowModel = [self nextRowModelForPPContactMainType:contactSectionModel.contactMainType
                                                                          row:contactSectionModel.rowModels.count
                                                                     rowTitle:nextString
                                                    tableViewCellEditingStyle:UITableViewCellEditingStyleDelete];
            
            if(nextRowModel==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            if(contactSectionModel.rowModels.count==1)
            {
                [contactSectionModel.rowModels insertObject:nextRowModel atIndex:0];
                
                //////////////////////////////////////////////////
                
                indexPath = [[NSIndexPath indexPathForRow:0 inSection:section] retain];
                
            }
            else
            {
                [contactSectionModel.rowModels insertObject:nextRowModel atIndex:contactSectionModel.rowModels.count-1];
                
                indexPath = [[NSIndexPath indexPathForRow:contactSectionModel.rowModels.count-2 inSection:section] retain];
            }
            
        } while (0);
        
    }
    
    return [indexPath autorelease];
}


//================================================================================
//
//================================================================================
- (void)clearImageRectDataWithContactFieldSource:(PPContactFieldSource)contactFieldSource searching:(BOOL)searching
{
    NSMutableArray *sectionModelArray = [self sectionModelsForSearching:searching];
    
    for(PPContactSectionModel *contactSectionModel in sectionModelArray)
    {
        @autoreleasepool
        {
            for(NSUInteger rowModelIndex=0 ; rowModelIndex<contactSectionModel.rowModels.count ; rowModelIndex++)
            {
                PPRowModel *rowModel = [contactSectionModel.rowModels objectAtIndex:rowModelIndex];
                
                if([rowModel ppContactFieldSource]==contactFieldSource)
                {
                    PPContactRowObject *contactRowObject = [rowModel ppContactRowObjectOfRowModel];
                    [contactRowObject.imageRectDictionary removeAllObjects];
                }
            }
        }
    }
}


//================================================================================
//
//================================================================================
- (void)clearRowModelWithContactFieldSource:(PPContactFieldSource)contactFieldSource searching:(BOOL)searching
{
    NSMutableArray *sectionModelArray = [self sectionModelsForSearching:searching];
    
    for(PPContactSectionModel *contactSectionModel in sectionModelArray)
    {
        @autoreleasepool
        {
            for(NSUInteger rowModelIndex=0 ; rowModelIndex<contactSectionModel.rowModels.count ; rowModelIndex++)
            {
                PPRowModel *rowModel = [contactSectionModel.rowModels objectAtIndex:rowModelIndex];
                
                if([rowModel ppContactFieldSource]==contactFieldSource)
                {
                    [contactSectionModel.rowModels removeObject:rowModel];
                    rowModelIndex--;
                }
            }
        }
    }
}


//================================================================================
//
//================================================================================
- (void)defaultSectionDatasWithDataModel:(ContactEditDataModel *)dataModel
                  ppContactMainTypeArray:(NSArray *)contactMainTypeArray
                            forSearching:(BOOL)searching
{
    if(dataModel!=nil)
    {
        NSMutableArray *targetArray = [self sectionModelsForSearching:searching];
        [targetArray removeAllObjects];
        
        //////////////////////////////////////////////////
        
        self.dataModel = dataModel;
        
        //////////////////////////////////////////////////
        
        //!!給定預設資料
        for (NSNumber *contactMainType in contactMainTypeArray)
        {
            @autoreleasepool
            {
                PPContactSectionModel *sectionModel = [self defaultContactSectionModelForContactMainType:[contactMainType unsignedIntegerValue]];
                
                if(targetArray!=nil && sectionModel!=nil)
                {
                    [targetArray addObject:sectionModel];
                }
            }
        }
    }
}


//================================================================================
//
//================================================================================
- (void)dumpDataForSearching:(BOOL)searching
{
    for (PPContactSectionModel *sectionModel in [self sectionModelsForSearching:searching])
    {
        for (PPRowModel *rowModel in sectionModel.rowModels)
        {
            PPContactRowObject *contactRowObject = rowModel.object;
            
            NSLog(@"rowModel.text:%@,rowModel.object.inputValueDictionary:%@, [rowModel ppContactFieldSource]:%td",rowModel.text,contactRowObject.inputValueDictionary,[rowModel ppContactFieldSource]);
        }
    }
}





//================================================================================
//
//================================================================================
- (void)fetchFieldDictionary:(NSDictionary *)fieldDictionary
                      orders:(NSArray *)orders
 toSectionModelsForSearching:(BOOL)searching
{
    do
    {
        if(fieldDictionary==nil || fieldDictionary.count<=0)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSArray *allkeys = orders;
        
        if(allkeys==nil ||
           allkeys.count<=0)
        {
            allkeys = fieldDictionary.allKeys;
        }
        
        //////////////////////////////////////////////////
        
        for(id key in allkeys)
        {
            @autoreleasepool
            {
                NSArray *targetFields = [fieldDictionary objectForKey:key];
                
                if(targetFields==nil ||
                   targetFields.count<=0)
                {
                    continue;
                }
                
                //////////////////////////////////////////////////
                
                if([key unsignedIntegerValue]==PPContactMainType_Note)
                {
                    self.noteFieldArray = targetFields;
                }
                else
                {
                    //系統群組，特殊解析方式
                    if([key unsignedIntegerValue]==PPContactMainType_SystemGroup)
                    {
                        self.systemCategoryArray = targetFields;
                        
                        //////////////////////////////////////////////////
                      
                        if([PPSettingsController integerValueWithKey:WCContactViewController_SettingsKey_EnableSystemContactCategory]==1)
                        {
                            [self enableSystemCategory:YES orders:orders];
                        }
                        else
                        {
                            [self enableSystemCategory:NO orders:orders];
                        }
                    }
                    else
                    {
                        BOOL showPhonetic = NO;
                        
                        if([key unsignedIntegerValue]==PPContactMainType_Company)
                        {
                            showPhonetic = self.showCompanyPhonetic;
                        }
                        else if([key unsignedIntegerValue]==PPContactMainType_Name)
                        {
                            showPhonetic = self.showNamePhonetic;
                        }
                        
                        //////////////////////////////////////////////////
                        
                        PPContactSectionModel *sectionModel = [[self.dataModel class] ppContactSectionModelForParsingFields:targetFields
                                                                                                          ppContactMainType:[key unsignedIntegerValue]
                                                                                                               showPhonetic:showPhonetic];
                        
                        //////////////////////////////////////////////////
                        
                        [self updateDataFromPPContactSectionModel:sectionModel
                                                           orders:orders
                                                     forSearching:NO];
                    }
                }
            }
        }
        
        //////////////////////////////////////////////////

#if !defined (PRODUCTLINE_WCSF)&&!defined(PRODUCTLINE_WCO365)
        [self addOneNameRowModelIfNeedOrNot];
#endif
    }
    while (0);
    
}


//================================================================================
//
//================================================================================
- (BOOL)validGroup
{
    BOOL result = NO;
    
    NSMutableDictionary *fieldDictionary = [self fieldDictionaryFromSectionModelForSearching:NO];
    
    if(fieldDictionary!=nil && fieldDictionary.count>0)
    {
        NSArray *groupFields = [fieldDictionary objectForKey:@(PPContactMainType_Group)];
        
        PPContactSectionModel *groupSectionModel = [[self.dataModel class] ppContactSectionModelForParsingFields:groupFields
                                                                                               ppContactMainType:PPContactMainType_Group
                                                                                                    showPhonetic:NO];
        
        
        if(groupSectionModel!=nil && groupSectionModel.rowModels.count>0)
        {
            result = YES;
        }
    }
    
    return result;
}


//================================================================================
//
//================================================================================
- (BOOL)emptyNameFieldAtFirst
{
    BOOL result = YES;
    
    NSMutableDictionary *fieldDictionary = [self fieldDictionaryFromSectionModelForSearching:NO];
    
    if(fieldDictionary!=nil && fieldDictionary.count>0)
    {
        NSArray *groupFields = [fieldDictionary objectForKey:@(PPContactMainType_Name)];
        
        PPContactSectionModel *nameSectionModel = [[self.dataModel class] ppContactSectionModelForParsingFields:groupFields
                                                                                              ppContactMainType:PPContactMainType_Group
                                                                                                   showPhonetic:NO];
        
        
        for(PPRowModel *rowModel in nameSectionModel.rowModels)
        {
            NSDictionary *inputValueDictionary = [rowModel ppContactRowObjectOfRowModel].inputValueDictionary;
            
            for(id key in inputValueDictionary.allKeys)
            {
                id value = [inputValueDictionary objectForKey:key];
                
                if([value respondsToSelector:@selector(length)]==YES &&
                   [value length]>0)
                {
                    result = NO;
                    
                    break;
                }
            }
            
            //////////////////////////////////////////////////
            
            if(result==NO)
            {
                break;
            }
        }
    }
    
    return result;
}


//================================================================================
//
//================================================================================
- (void)enableSystemCategory:(BOOL)enable orders:(NSArray *)orders
{
    ContactDataExchangeModel *dataExchangeModel = [self customSystemCategoryContactDataExchangeModel];
    
    //////////////////////////////////////////////////
    
    if(dataExchangeModel!=nil)
    {
        PPContactSectionModel *sectionModel = nil;
        
        if(enable==YES)
        {
            sectionModel = [[self.dataModel class] ppContactSectionModelForParsingFields:@[dataExchangeModel,[self.systemCategoryArray firstObject]]
                                                                       ppContactMainType:PPContactMainType_SystemGroup
                                                                            showPhonetic:NO];
        }
        else
        {
            sectionModel = [[self.dataModel class] ppContactSectionModelForParsingFields:@[dataExchangeModel]
                                                                       ppContactMainType:PPContactMainType_SystemGroup
                                                                            showPhonetic:NO];
        }
        
        //////////////////////////////////////////////////
        
        NSInteger section = [self sectionForPPContactMainType:PPContactMainType_SystemGroup
                                                 forSearching:NO];
        
        if(section!=NSNotFound)
        {
            [self deleteSectionModelAtIndex:section forSearching:NO];
        }
        
        [self updateDataFromPPContactSectionModel:sectionModel orders:orders forSearching:NO];
    }
}


//================================================================================
//
//================================================================================
- (BOOL)legalDataText
{
    return ([self firstIllegalDataForIndexPath]==nil) ? YES : NO;
}


//================================================================================
//
//================================================================================
- (NSIndexPath *)firstIllegalDataForIndexPath
{
    NSIndexPath *indexPath = nil;
    
    PPSectionModel *sectionModel = nil;
    PPRowModel *rowModel = nil;
    
    for(NSInteger section=0; section<self.sectionModelsForDefault.count; section++)
    {
        sectionModel = [self.sectionModelsForDefault objectAtIndex:section];
        
        for(NSInteger row=0; row<sectionModel.rowModels.count; row++)
        {
            rowModel = [sectionModel.rowModels objectAtIndex:row];
            
            if([[rowModel ppContactRowObjectOfRowModel] checkValueIsLegal]==NO)
            {
                indexPath = [NSIndexPath indexPathForRow:row inSection:section];
                break;
            }
        }
        
        //////////////////////////////////////////////////

        if(indexPath!=nil)
        {
            break;
        }
    }
    
    return indexPath;
}
@end
