//
//  WCCaptureViewController.m
//
//

#import "WCCaptureViewController.h"
#import "WCCaptureViewController+ResourceDefine.h"
#import "WCCaptureViewController+SettingsKey.h"
#import "UIApplication+Idle.h"
#import "UIDevice+DeviceModel.h"
#import "PPCameraView+iOS.h"
#import "PPSelectController.h"
#import "PPSettingsController.h"
#import "PPGuideController.h"
#import "PPAlertView.h"
#import "CVCManualCaptureController.h"
#import "CVCDockingCaptureController.h"
#import "CVCQRCodeCaptureController.h"
#import "CVCScrollMenu.h"
#import "AVCaptureDevice+RequestAuthorization.h"
#import "PPLogController.h"
#import "WCToolController.h"
#import "PPAlertController.h"

// 測試超廣角拍照時打開
//#define WCCapture_UltraWideCameraTest

// 顯示放大倍率
//#define WCCapture_ShowZoomFactor

NSString * const WCCaptureViewControllr_SettingsKey_UltraWideCameraTest_CropRatio = @"WCCaptureViewControllr_SettingsKey_UltraWideCameraTest_CropRatio";
NSString * const WCCaptureViewControllr_SettingsKey_UltraWideCameraTest_CropOffsetY = @"WCCaptureViewControllr_SettingsKey_UltraWideCameraTest_CropOffsetY";

////////////////////////////////////////////////////////////////////////////////////////////////////

/// log dir
NSString * const WCCaptureViewController_CaptureLogFolder = @"CaptureLog";

CGFloat const WCCaptureViewController_ViewGap = 0.0;
CGFloat const WCCaptureViewController_AnimateDuration = 0.3;
UIDeviceOrientation const WCCaptureViewController_FaceUpReplaceOrientation = UIDeviceOrientationLandscapeLeft;

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface WCCaptureViewController () < UIScrollViewDelegate,
                                        PPCameraViewDelegate,
                                        PPGuideControllerDelegate,
                                        WCBaseCaptureModeDelegate,
                                        CVCScrollMenuDelegate,
                                        PPSelectControllerDelegate>

/**
 *  主介面
 *  iOS7會直接加到KeyWindow中，iOS8會放到notRotateView上
 */
@property (nonatomic, retain) UIView *mainView;

/**
 *  iOS8專用，由於設定transform的View無法做AutoLayout，故再多一層notRotateView用以設定transform避免動畫旋轉
 *  而主要做autoLayout的View(mainView)則為此view的SubView
 */
@property (nonatomic, retain) UIView *notRotateView;

/**
 *  是否正在照相中，若是則拒絕任何拍照的要求
 */
@property (atomic, assign) BOOL isCapturingPhoto;

/**
 *  是否正在變換拍照模式
 */
@property (atomic, assign) BOOL isChangingMode;


@property (nonatomic, assign) UIDeviceOrientation actualDeviceOrientation;  /// 手機真正的旋轉方向
@property (nonatomic, assign) BOOL enableFlash;
@property (nonatomic, assign) BOOL enableTorch;
@property (nonatomic, assign) BOOL enableImageEnhance;
@property (nonatomic, assign) BOOL enableAutoCrop;
@property (nonatomic, assign) BOOL enableDoubleSide;

@property (nonatomic, retain) UISwipeGestureRecognizer *swipeLeftGestureRecognizer;
@property (nonatomic, retain) UISwipeGestureRecognizer *swipeRightGestureRecognizer;

@property (nonatomic, retain) CVCBaseCaptureController *activeCaptureController;
@property (nonatomic, retain) CVCScrollMenu *scrollMenu;
@property (nonatomic, retain) NSMutableArray *mainViewLayoutConstraints;

@property (nonatomic, retain) PPSelectController *selectController;
@property (nonatomic, retain) PPCameraView *ppCameraView;
@property (nonatomic, retain) UILabel *debugView;

@property (nonatomic, retain) PPLogController *logController;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation WCCaptureViewController

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods

//================================================================================
//
//================================================================================
- (id)init
{
    if(self = [super init])
    {
        //////////////////////////////////////////////////
        // 建立log instance
        
        
#ifdef _DUMP_LOG_
        
        self.logController = [[[PPLogController alloc] init] autorelease];
        
        if(self.logController != nil)
        {
            NSString *logDirPath = [WCCaptureViewController logDirPath];
            
            if([[NSFileManager defaultManager] fileExistsAtPath:logDirPath] == NO)
            {
                [[NSFileManager defaultManager] createDirectoryAtPath:logDirPath
                                          withIntermediateDirectories:YES
                                                           attributes:nil
                                                                error:nil];
            }
            
            [self.logController setFileName:WCCaptureViewController_CaptureLogFolder atPath:logDirPath];
            [self.logController setMask:PPLogControllerMask_Normal];
        }

#endif

#ifdef WCCapture_UltraWideCameraTest
        if([PPSettingsController integerValueWithKey:WCCaptureViewControllr_SettingsKey_UltraWideCameraTest_CropRatio]==PPSettingsController_UnsetIntValue)
        {
            [PPSettingsController setDefaultNumberValue:@(0.5) withKey:WCCaptureViewControllr_SettingsKey_UltraWideCameraTest_CropRatio];
        }
        if([PPSettingsController integerValueWithKey:WCCaptureViewControllr_SettingsKey_UltraWideCameraTest_CropOffsetY]==PPSettingsController_UnsetIntValue)
        {
            [PPSettingsController setDefaultNumberValue:@(0.0) withKey:WCCaptureViewControllr_SettingsKey_UltraWideCameraTest_CropOffsetY];
        }
 #endif


        //////////////////////////////////////////////////
        // Handle settings
        
        if([PPSettingsController integerValueWithKey:WCCaptureViewControllr_SettingsKey_EnableFlash]==PPSettingsController_UnsetIntValue)
        {
            [PPSettingsController setDefaultIntegerValue:0
                                                 withKey:WCCaptureViewControllr_SettingsKey_EnableFlash];
        }
        
        if([PPSettingsController integerValueWithKey:WCCaptureViewControllr_SettingsKey_EnableTorch]==PPSettingsController_UnsetIntValue)
        {
            
            [PPSettingsController setDefaultIntegerValue:0
                                                 withKey:WCCaptureViewControllr_SettingsKey_EnableTorch];
        }
        if([PPSettingsController integerValueWithKey:WCCaptureViewControllr_SettingsKey_EnableImageEnhance]==PPSettingsController_UnsetIntValue)
        {
            
            [PPSettingsController setDefaultIntegerValue:1
                                                 withKey:WCCaptureViewControllr_SettingsKey_EnableImageEnhance];
        }
        
        if([PPSettingsController integerValueWithKey:WCCaptureViewControllr_SettingsKey_EnableAutoCrop]==PPSettingsController_UnsetIntValue)
        {
            [PPSettingsController setDefaultIntegerValue:1
                                                 withKey:WCCaptureViewControllr_SettingsKey_EnableAutoCrop];
        }
        
        if([PPSettingsController integerValueWithKey:WCCaptureViewControllr_SettingsKey_EnableDoubleSide]==PPSettingsController_UnsetIntValue)
        {
            [PPSettingsController setDefaultIntegerValue:1
                                                 withKey:WCCaptureViewControllr_SettingsKey_EnableDoubleSide];
        }
        
        [PPSettingsController setDefaultIntegerValue:1
                                             withKey:WCCaptureViewControllr_SettingsKey_ShowManualCaptureGuide];
        
        [PPSettingsController setDefaultIntegerValue:1
                                             withKey:WCCaptureViewControllr_SettingsKey_ShowDockingCaptureGuide];
        
        
        _enableFlash = [PPSettingsController integerValueWithKey:WCCaptureViewControllr_SettingsKey_EnableFlash];
        _enableTorch = [PPSettingsController integerValueWithKey:WCCaptureViewControllr_SettingsKey_EnableTorch];
        _enableImageEnhance = [PPSettingsController integerValueWithKey:WCCaptureViewControllr_SettingsKey_EnableImageEnhance];
        _enableAutoCrop = [PPSettingsController integerValueWithKey:WCCaptureViewControllr_SettingsKey_EnableAutoCrop];
        _enableDoubleSide = [PPSettingsController integerValueWithKey:WCCaptureViewControllr_SettingsKey_EnableDoubleSide];
        
        
        //////////////////////////////////////////////////
        // Set init value

        self.captureSide = CaptureSide_None;
        self.isCapturingPhoto = NO;
        self.view.backgroundColor = [UIColor blackColor];
        self.ppInterfaceOrientationController.mask = PPInterfaceOrientationControllerMask_All;
        self.actualDeviceOrientation = UIDeviceOrientationUnknown;
        
        
        //////////////////////////////////////////////////
        // !! mainView大小固定，只用transform控制方向
        
        UIWindow *keyWindow = [self keyWindow];
        CGRect mainFrame = CGRectMake(0,
                                      0,
                                      MIN(keyWindow.bounds.size.width, keyWindow.bounds.size.height),
                                      MAX(keyWindow.bounds.size.width, keyWindow.bounds.size.height));
        

        _mainView = [[UIView alloc] initWithFrame:mainFrame];
        
        if (self.mainView != nil)
        {
            // for iOS8
            if ([[UIDevice currentDevice] systemVersion].floatValue >= 8.0)
            {
                // !! 將mainView放在notRotateView中
                _notRotateView = [[UIView alloc] initWithFrame:mainFrame];
                
                if(self.notRotateView != nil)
                {
                    [self.notRotateView addSubview:self.mainView];
                    [self.view addSubview:self.notRotateView];
                }
            }
            else // for iOS7
            {
                //--------------------------------------------------
                // MARK: [iOS7顯示流程說明]
                // 因為mainView開啟/關閉時必須要跟隨目前旋轉方向，但操作時又要限定在portrait方向，所以必須特殊處理。
                //--------------------------------------------------
            }
        }

        //////////////////////////////////////////////////
        // iphone 13 需要用超廣角鏡頭，不然要拿比較遠才能拍
        // 如果是新出的手機deviceID應該是PPDeviceModelID_iPhone，所以也加入超廣角的判斷
        PPDeviceModelID deviceID = [UIDevice deviceModelID];
        if((deviceID>=PPDeviceModelID_iPhone_13Mini||deviceID==PPDeviceModelID_iPhone)&&
           [PPCameraView isSupportTripleCamera]==YES)
        {
            if (@available(iOS 13.0, *)) {
                [PPCameraView setAvailableDeviceTypes:@[AVCaptureDeviceTypeBuiltInUltraWideCamera]];
            }
        }
        
        _ppCameraView = [[PPCameraView alloc] initWithFrame:mainFrame];
        if(self.ppCameraView!=nil)
        {
            self.ppCameraView.adjustOutputImageOrientation = YES;
            self.ppCameraView.translatesAutoresizingMaskIntoConstraints = NO;
            
            [self.mainView addSubview:self.ppCameraView];
            
            
            // MARK: [Debug用] 在UI中顯示對焦點
            // 需要看對焦點是否正確時，請把圖檔命名為"PPCameraView_focus.png"加入boundle中，再打開下面設定。
//            self.ppCameraView.alwaysShowFocusImageWhenSetFoucsPoint = YES;
            
        }
        
        //////////////////////////////////////////////////

        _scrollMenu = [[CVCScrollMenu alloc] init];
        if (self.scrollMenu)
        {
            self.scrollMenu.proxy = self;
            self.scrollMenu.hidden = YES;
            self.scrollMenu.translatesAutoresizingMaskIntoConstraints = NO;
            
            [self.mainView addSubview:self.scrollMenu];
        }

        //////////////////////////////////////////////////

        _swipeLeftGestureRecognizer = [[UISwipeGestureRecognizer alloc] initWithTarget:self action:@selector(onSwipeLeft)];
        if(self.swipeLeftGestureRecognizer)
        {
            self.swipeLeftGestureRecognizer.direction = UISwipeGestureRecognizerDirectionLeft;
            [self.mainView addGestureRecognizer:self.swipeLeftGestureRecognizer];
        }
        
        _swipeRightGestureRecognizer = [[UISwipeGestureRecognizer alloc] initWithTarget:self action:@selector(onSwipeRight)];
        if(self.swipeRightGestureRecognizer)
        {
            self.swipeRightGestureRecognizer.direction = UISwipeGestureRecognizerDirectionRight;
            [self.mainView addGestureRecognizer:self.swipeRightGestureRecognizer];
        }

#ifdef WCCapture_ShowZoomFactor
        _debugView = [[UILabel alloc]initWithFrame:CGRectMake(0, 0, self.view.bounds.size.width, 90)];
        
        if(self.debugView)
        {
            self.debugView.hidden = YES;
            self.debugView.translatesAutoresizingMaskIntoConstraints = NO;
            self.debugView.backgroundColor = [UIColor colorWithRed:0 green:0.2 blue:0.2 alpha:0.2];
            self.debugView.numberOfLines = 0;
            
            [self.debugView setText:@"DEBUG"];
        }
        [self.mainView addSubview:self.debugView];
        
        self.debugView.hidden = NO;
#endif
        //////////////////////////////////////////////////
        // Notifications

        [[NSNotificationCenter defaultCenter] addObserver:self
                                                 selector:@selector(recvUIApplicationWillEnterForegroundNotification:)
                                                     name:UIApplicationWillEnterForegroundNotification
                                                   object:nil];
        
        [[NSNotificationCenter defaultCenter] addObserver:self
                                                 selector:@selector(recvUIApplicationDidEnterBackgroundNotification:)
                                                     name:UIApplicationDidEnterBackgroundNotification
                                                   object:nil];
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    self.logController = nil;
    
    //////////////////////////////////////////////////
    // viewWillDisappear的isBeingDismissed有時不會成立，所以這裡要再檢查一次。
    
    if(self.actualDeviceOrientation != UIDeviceOrientationUnknown)
    {
        [self restoreDeviceOrientation];
        self.actualDeviceOrientation = UIDeviceOrientationUnknown;
    }
    
    //////////////////////////////////////////////////
    
    [[NSNotificationCenter defaultCenter] removeObserver:self];

    //////////////////////////////////////////////////
    
    if(self.swipeLeftGestureRecognizer)
    {
        [self.mainView removeGestureRecognizer:self.swipeLeftGestureRecognizer];
        self.swipeLeftGestureRecognizer = nil;
    }
    
    if(self.swipeRightGestureRecognizer)
    {
        [self.mainView removeGestureRecognizer:self.swipeRightGestureRecognizer];
        self.swipeRightGestureRecognizer = nil;
    }
    
#ifdef WCCapture_ShowZoomFactor
    [self.debugView removeFromSuperview];
    self.debugView = nil;
#endif
    //////////////////////////////////////////////////
    
    self.mainView = nil;
    self.notRotateView = nil;
    self.scrollMenu = nil;
    
    // synchronize mainViewLayoutConstraints
    @synchronized(self)
    {
        self.mainViewLayoutConstraints = nil;
    }
    
    self.ppCameraView.delegate = nil;
    [self.ppCameraView stopCameraPreview];
    self.ppCameraView = nil;

    self.recogLanguages = nil;
    self.supportCaptureModes = nil;
    self.activeCaptureController.delegate = nil;
    self.activeCaptureController = nil;
    
    self.selectController.delegate = nil;
    self.selectController = nil;

    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Override methods

//================================================================================
//
//================================================================================
- (BOOL)prefersStatusBarHidden
{
    if(self.isBeingDismissed == YES)
    {
        return NO;
    }
    else
    {
        return YES;
    }
}


//================================================================================
//
//================================================================================
- (void)viewWillAppear:(BOOL)animated
{
    [super viewWillAppear:animated];
 
    //////////////////////////////////////////////////
    
    [PPGuideController setDelegate:self];

    //////////////////////////////////////////////////

//    NSLog(@"%s", __func__);
//    NSLog(@"orientation %ld", (long)[UIApplication sharedApplication].statusBarOrientation);

//    NSLog(@"self.view.frame %@", NSStringFromCGRect(self.view.frame));
//    NSLog(@"self.view.bounds %@", NSStringFromCGRect(self.view.bounds));
//    NSLog(@"self.view.transform %@", NSStringFromCGAffineTransform(self.view.transform));
//    
//    NSLog(@"self.notRotateView.frame %@", NSStringFromCGRect(self.notRotateView.frame));
//    NSLog(@"self.notRotateView.bounds %@", NSStringFromCGRect(self.notRotateView.bounds));
//    NSLog(@"self.notRotateView.transform %@", NSStringFromCGAffineTransform(self.notRotateView.transform));
    
    
    [[UIApplication sharedApplication] enableIdle:NO];
    
    //////////////////////////////////////////////////
    
    // for iOS8
    if ([[UIDevice currentDevice] systemVersion].floatValue >= 8.0)
    {
        //////////////////////////////////////////////////
        // !! notRotateView是固定大小且只能用transform控制，
        //    預設是以portrait方向設計，所以當viewWillAppear方向不是Portrait時
        //    要調整transform。
        
        if(self.notRotateView != nil)
        {
            [self adjustTransformWithView:self.notRotateView
                     interfaceOrientation:[self curInterfaceOrientation]];
        }
    }
    else // for iOS7
    {
        //--------------------------------------------------
        // MARK: [iOS7顯示流程-1] 開啟時的處理
        
        // 為了讓拍照介面跟隨開啟時的動畫顯示，mainView的superView必須是self.view。
        [self.view addSubview:self.mainView];

        // 旋轉到self.view的方向
        [self adjustTransformWithView:self.mainView
                 interfaceOrientation:[self curInterfaceOrientation]];
        
        // 修正旋轉後的位置偏移
        [self adjustOriginWithView:self.mainView];
        
        //--------------------------------------------------
    }
    
    
    //////////////////////////////////////////////////
    // 設定scroll bar目前模式
    
    self.scrollMenu.currentIndex = MAX(0, [self.supportCaptureModes indexOfObject:@(self.activeCaptureMode)]);
    
    
    //////////////////////////////////////////////////
    // 啟動照相機

    self.ppCameraView.delegate = self;
    [self.ppCameraView startCameraPreviewWithCompletion:^{
        __block typeof(self) blockSelf = self;
        
        [[NSOperationQueue mainQueue] addOperationWithBlock:^{
            [blockSelf adjustVideoZoomFactorWithCaptureMode:self.activeCaptureMode];
        }];
    }];
}


//================================================================================
//
//================================================================================
- (void)viewDidAppear:(BOOL)animated
{
    [super viewDidAppear:animated];
    
    //////////////////////////////////////////////////

    // for iOS7
    if ([[UIDevice currentDevice] systemVersion].floatValue < 8.0)
    {
        //--------------------------------------------------
        // MARK: [iOS7顯示流程-2] 操作時的處理

        // 操作時要固定在portrait方向，所以把superView改為keyWindow。
        [[self keyWindow] addSubview:self.mainView];
        
        // 旋轉到keyWindow的方向
        self.mainView.transform = CGAffineTransformIdentity;
        
        // 修正旋轉後的位置偏移
        [self adjustOriginWithView:self.mainView];
        
        //--------------------------------------------------
        
        // !! 這裏才抓得到targetBoxImageView的大小
        [self.activeCaptureController setupOverlayViewLayoutConstraintWithStyle:LayoutStyle_AnimationStop];
        [self.activeCaptureController.overlayView layoutIfNeeded];
    }
    
    
    //////////////////////////////////////////////////
    
    [self.activeCaptureController actionWhenActive];
    
    // !! 強制顯示Info
    [self.activeCaptureController updateInfoLabelLastingWithText:nil];
    [self.activeCaptureController updateCaptureCountLabelWithNumber:self.captureCount];    
    
    // 7.3.2 之後拿掉tip
//    [self showGuideIfNeeded];

    
    //////////////////////////////////////////////////
    // Device方向監測及強制旋轉 (只有在“開啟時”設定)
    
    if([self isBeingPresented] == YES)
    {
        // 記錄present時的方向
        self.actualDeviceOrientation = [UIDevice currentDevice].orientation;
        
        [[NSNotificationCenter defaultCenter] addObserver:self
                                                 selector:@selector(recvUIDeviceOrientationDidChangeNotification:)
                                                     name:UIDeviceOrientationDidChangeNotification
                                                   object:nil];
    }

    // 調整顯示方向
    [self adjustDeviceOrientationWithCaptureMode:self.activeCaptureMode];
    
    
    //////////////////////////////////////////////////
    
    __block typeof(self) blockSelf = self;

    dispatch_async(dispatch_get_main_queue(), ^{
        
        [blockSelf prepareCamera];
    });
    

    //////////////////////////////////////////////////
    // MARK: [Debug用] 在console中顯示對焦相關log
    
//    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_BACKGROUND, 0), ^{
//        
//        while (true)
//        {
//            NSLog(@"focusing:%d, point:%@", [self.ppCameraView isFocusing], NSStringFromCGPoint([self.ppCameraView focusPoint]));
//            
//            [NSThread sleepForTimeInterval:0.01];
//        }
//    });
}


//================================================================================
//
//================================================================================
- (void)viewWillLayoutSubviews
{
    [super viewWillLayoutSubviews];
    
    //////////////////////////////////////////////////
    // 開啟及旋轉時的介面處理，切換模式時不會進來這裡。
    
    [self setupMainViewLayoutConstraint];
    [self.activeCaptureController addOptionalSubviews];
    [self.activeCaptureController setupOverlayViewLayoutConstraintWithStyle:LayoutStyle_AnimationStop];
    [self.activeCaptureController setupSubviewsInterfaceOrientation:[self curInterfaceOrientation]];
    
    //////////////////////////////////////////////////
    
    // for iOS8
    if ([[UIDevice currentDevice] systemVersion].floatValue >= 8.0)
    {
        [self.notRotateView setCenter:CGPointMake(CGRectGetMidX(self.view.bounds), CGRectGetMidY(self.view.bounds))];
    }
}


//================================================================================
//
//================================================================================
- (void)viewDidLayoutSubviews
{
    [super viewDidLayoutSubviews];
}


//================================================================================
//
//================================================================================
- (void)viewWillDisappear:(BOOL)animated
{
    [self logMessageWithFormat:@"%s in", __func__];
    
    [[UIApplication sharedApplication] enableIdle:YES];
    
    //////////////////////////////////////////////////
    
    [super viewWillDisappear:animated];
    
    self.ppCameraView.delegate = nil;
    [self.activeCaptureController actionWhenDeactive];

    //////////////////////////////////////////////////
    
    if([self isBeingDismissed] == YES)
    {
        [[NSNotificationCenter defaultCenter] removeObserver:self
                                                        name:UIDeviceOrientationDidChangeNotification
                                                      object:nil];
        
        
        [self restoreDeviceOrientation];
        self.actualDeviceOrientation = UIDeviceOrientationUnknown;
    }
    
    //////////////////////////////////////////////////
    
    // for iOS7
    if ([[UIDevice currentDevice] systemVersion].floatValue < 8.0)
    {
        //--------------------------------------------------
        // MARK: [iOS7顯示流程-3] 關閉時的處理
        
        // 為了讓拍照介面跟隨關閉時的動畫顯示，mainView的superView必須是self.view。
        [self.view addSubview:self.mainView];
        
        // 旋轉mainView的方向
        [self adjustTransformWithView:self.mainView interfaceOrientation:self.activeCaptureController.interfaceOrientation];
        
        // 修正旋轉後的位置偏移
        [self adjustOriginWithView:self.mainView];
        
        //--------------------------------------------------
    }
    
    //////////////////////////////////////////////////

    [self logMessageWithFormat:@"%s out", __func__];
}


//================================================================================
//
//================================================================================
- (void)viewDidDisappear:(BOOL)animated
{
    [super viewDidDisappear:animated];
    
    //////////////////////////////////////////////////
    
    [self.ppCameraView stopCameraPreview];
}


//================================================================================
// for iOS8 notRotateView
//================================================================================
- (void)viewWillTransitionToSize:(CGSize)size withTransitionCoordinator:(id<UIViewControllerTransitionCoordinator>)coordinator
{
    [super viewWillTransitionToSize:size withTransitionCoordinator:coordinator];
 
    //////////////////////////////////////////////////
    // iOS 8.3以後在dismiss時，若開始和結束時的orientation不同，
    // 結束時會多收到一個旋轉通知，造成畫面錯亂，所以在這裡跳過旋轉動作。
    
    if([self isBeingDismissed])
    {
        return;
    }
    
    //////////////////////////////////////////////////
    
    [coordinator animateAlongsideTransition:^(id<UIViewControllerTransitionCoordinatorContext> context) {
        
        CGAffineTransform deltaTransform = coordinator.targetTransform;
        CGFloat deltaAngle = atan2f(deltaTransform.b, deltaTransform.a);
    
        CGFloat currentRotation = [[self.notRotateView.layer valueForKeyPath:@"transform.rotation.z"] floatValue];
        // Adding a small value to the rotation angle forces the animation to occur in a the desired direction, preventing an issue where the view would appear to rotate 2PI radians during a rotation from LandscapeRight -> LandscapeLeft.
        currentRotation += -1 * deltaAngle + 0.0001;
        [self.notRotateView.layer setValue:@(currentRotation) forKeyPath:@"transform.rotation.z"];
        
    } completion:^(id<UIViewControllerTransitionCoordinatorContext> context) {
        
        // Integralize the transform to undo the extra 0.0001 added to the rotation angle.
        CGAffineTransform currentTransform = self.notRotateView.transform;
        currentTransform.a = round(currentTransform.a);
        currentTransform.b = round(currentTransform.b);
        currentTransform.c = round(currentTransform.c);
        currentTransform.d = round(currentTransform.d);
        self.notRotateView.transform = currentTransform;
    }];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Property methods

//================================================================================
//
//================================================================================
- (void)setEnableFlash:(BOOL)enableFlash
{
    if(_enableFlash != enableFlash)
    {
        _enableFlash = enableFlash;
        [PPSettingsController setIntegerValue:enableFlash
                                      withKey:WCCaptureViewControllr_SettingsKey_EnableFlash];
    }
}


//================================================================================
//
//================================================================================
- (void)setEnableTorch:(BOOL)enableTorch
{
    if(_enableTorch != enableTorch)
    {
        _enableTorch = enableTorch;
        [PPSettingsController setIntegerValue:enableTorch
                                      withKey:WCCaptureViewControllr_SettingsKey_EnableTorch];
    }
}


//================================================================================
//
//================================================================================
- (void)setEnableImageEnhance:(BOOL)enableImageEnhance
{
    if(_enableImageEnhance != enableImageEnhance)
    {
        _enableImageEnhance = enableImageEnhance;
        [PPSettingsController setIntegerValue:enableImageEnhance
                                      withKey:WCCaptureViewControllr_SettingsKey_EnableImageEnhance];
    }
}


//================================================================================
//
//================================================================================
- (void)setEnableAutoCrop:(BOOL)enableAutoCrop
{
    if(_enableAutoCrop != enableAutoCrop)
    {
        _enableAutoCrop = enableAutoCrop;
        [PPSettingsController setIntegerValue:enableAutoCrop
                                      withKey:WCCaptureViewControllr_SettingsKey_EnableAutoCrop];
    }
}


//================================================================================
//
//================================================================================
- (void)setEnableDoubleSide:(BOOL)enableDoubleSide
{
    if(_enableDoubleSide != enableDoubleSide)
    {
        _enableDoubleSide = enableDoubleSide;
        [PPSettingsController setIntegerValue:enableDoubleSide
                                      withKey:WCCaptureViewControllr_SettingsKey_EnableDoubleSide];
    }
}


//================================================================================
//
//================================================================================
- (void)setCaptureCount:(NSInteger)captureCount
{
    if(_captureCount != captureCount)
    {
        _captureCount = captureCount;
        [self.activeCaptureController setupCaptureCount:captureCount];
    }
}


//================================================================================
//
//================================================================================
- (void)setCaptureSide:(WCCaptureViewController_CaptureSide)captureSide
{
    if(_captureSide != captureSide)
    {
        
        _captureSide = captureSide;
        [self.activeCaptureController setupCaptureSide:captureSide];
        
        //////////////////////////////////////////////////
        // 變更正反面時要更新目前選擇語系及語系列表內容

        if([self.delegate respondsToSelector:@selector(captureViewController:recogLanguagesWithCaptureSide:)])
        {
            self.recogLanguages = [self.delegate captureViewController:self recogLanguagesWithCaptureSide:captureSide];
        }
        
        [self.activeCaptureController setupRecogLanguageState:[self.recogLanguages firstObject]];
    }
}


//================================================================================
// 切換拍照模式
//================================================================================
- (void)setActiveCaptureMode:(WCCaptureViewController_CaptureMode)captureMode
{
    // synchronize activeCaptureController
    @synchronized(self)
    {
        if(_activeCaptureMode != captureMode)
        {
            //////////////////////////////////////////////////
            // close previous mode
            
            if(self.activeCaptureController != nil)
            {
                [self.activeCaptureController.overlayView removeFromSuperview];
                [self.activeCaptureController.captureBar removeFromSuperview];
            }
            
            if (captureMode != CaptureMode_QRCode)
            {
                self.enableTorch = NO;
            }
            
            // !! 先把舊的release
            self.activeCaptureController = nil;
            
            
            //////////////////////////////////////////////////
            // open new mode
            NSArray *deviceTypes = nil;
            if (@available(iOS 13, *))
            {
                if(captureMode==CaptureMode_Manual)
                {
                    deviceTypes = @[AVCaptureDeviceTypeBuiltInTripleCamera,
                                    AVCaptureDeviceTypeBuiltInDualWideCamera,
                                    AVCaptureDeviceTypeBuiltInWideAngleCamera];
                }
                else
                {
                    deviceTypes = @[AVCaptureDeviceTypeBuiltInDualWideCamera];
                }
            }
            else
            {
                deviceTypes = @[AVCaptureDeviceTypeBuiltInWideAngleCamera];
            }
            [PPCameraView setAvailableDeviceTypes:deviceTypes];
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            self.activeCaptureController = [self captureControllerWithCaptureMode:captureMode];
            
            [self.activeCaptureController setupCaptureSide:self.captureSide];
            [self.activeCaptureController setupCaptureCount:self.captureCount];
            [self.activeCaptureController setupFlashState:self.enableFlash];
            [self.activeCaptureController setupTorchState:self.enableTorch];
            [self.activeCaptureController setupImageEnhanceState:self.enableImageEnhance];
            [self.activeCaptureController setupAutoCropState:self.enableAutoCrop];
            [self.activeCaptureController setupDoubleSideState:self.enableDoubleSide];
            [self.activeCaptureController setupRecogLanguageState:[self.recogLanguages firstObject]];
            
            if(captureMode == CaptureMode_Docking)
            {
                BOOL enable = [WCCaptureViewController dockingCaptureAutoDetectionEnabled];
                [self.activeCaptureController setupAutoDetectionState:enable];
            }
            
            [self.mainView addSubview:self.activeCaptureController.overlayView];
            [self.mainView addSubview:self.activeCaptureController.captureBar];
            
            self.ppCameraView.focusMode = self.activeCaptureController.focusMode;            
            [self.ppCameraView turnOnTorch:self.enableTorch];

            __block typeof(self) blockSelf = self;
            
            [[NSOperationQueue mainQueue] addOperationWithBlock:^{
                [blockSelf adjustVideoZoomFactorWithCaptureMode:captureMode];
            }];
            

            _activeCaptureMode = captureMode;
        }
    }
}


//==============================================================================
// 不同手機的放大要設不一樣，目前以雙鏡頭和三鏡頭分，iphone 16是特殊處理
//==============================================================================
- (void)adjustVideoZoomFactorWithCaptureMode:(WCCaptureViewController_CaptureMode)captureMode
{
//    NSLog(@"adjustVideoZoomFactor - (%p)deviceType:%@", self.ppCameraView, self.ppCameraView.deviceType);
//    NSLog(@"adjustVideoZoomFactor - videoZoomFactor:%@", @(self.ppCameraView.videoZoomFactor));
//    
//
    AVCaptureDeviceType oriDeviceType = self.ppCameraView.deviceType;
    CGFloat oriVideoZoomFactor = self.ppCameraView.videoZoomFactor;
    
    // MARK: 對焦處理機制
    if (@available(iOS 13, *))
    {
        if(captureMode==CaptureMode_Manual)
        {
            // 倍率調整到對焦清晰並佔滿preview範圍
            if (oriDeviceType == AVCaptureDeviceTypeBuiltInDualWideCamera)
            {
                PPDeviceModelID deviceID = [UIDevice deviceModelID];
                if(deviceID==PPDeviceModelID_iPhone_16)
                {
                    // iphone 16x 有微距模式, 所以也設定為2倍
                    self.ppCameraView.videoZoomFactor = 2.0;
                }
                else
                {
                    // 2鏡頭機型 (無微距模式，最近對焦距離10cm)
                    self.ppCameraView.videoZoomFactor = 3.0;
                }
            }
            else if (oriDeviceType == AVCaptureDeviceTypeBuiltInTripleCamera)
            {
                // 3鏡頭機型 (有微距模式)
                self.ppCameraView.videoZoomFactor = 2.0;
            }
        }
        else
        {
            self.ppCameraView.videoZoomFactor = 2.0;
        }
    }
    else
    {
        self.ppCameraView.videoZoomFactor = 2.0;
    }
    
#ifdef WCCapture_ShowZoomFactor
    [self.debugView setText:[NSString stringWithFormat:@"deviceType: %@\n(%@) -> (%@)", oriDeviceType, @(oriVideoZoomFactor), @(self.ppCameraView.videoZoomFactor)]];
#endif

//    NSLog(@"adjustVideoZoomFactor - 1 deviceType:%@", self.ppCameraView.deviceType);
//    NSLog(@"adjustVideoZoomFactor - 1 videoZoomFactor:%@", @(self.ppCameraView.videoZoomFactor));
}



//================================================================================
//
//================================================================================
- (void)setSupportCaptureModes:(NSArray *)supportCaptureModes
{
    if(_supportCaptureModes != supportCaptureModes)
    {
        [_supportCaptureModes release];
        _supportCaptureModes = [supportCaptureModes retain];

        //////////////////////////////////////////////////
        NSMutableArray *itemTitles = [NSMutableArray array];
        
        // prepare scroll menu items
        for(NSNumber *modeNumber in supportCaptureModes)
        {
            [itemTitles addObject:[self captureModeStringWithMode:[modeNumber integerValue]]];
        }
        
        self.scrollMenu.hidden = ([supportCaptureModes count] <= 1);
        self.scrollMenu.itemTitles = itemTitles;

        //////////////////////////////////////////////////
        
        if([supportCaptureModes count] <= 1)
        {
            // remove swipe gesture
            if(self.swipeLeftGestureRecognizer)
            {
                [self.mainView removeGestureRecognizer:self.swipeLeftGestureRecognizer];
                self.swipeLeftGestureRecognizer = nil;
            }
            
            if(self.swipeRightGestureRecognizer)
            {
                [self.mainView removeGestureRecognizer:self.swipeRightGestureRecognizer];
                self.swipeRightGestureRecognizer = nil;
            }
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Notification receiver

//==============================================================================
//
//==============================================================================
- (void)recvUIApplicationWillEnterForegroundNotification:(NSNotification *)notify
{
    self.ppCameraView.delegate = self;
    [self.ppCameraView startCameraPreviewWithCompletion:^{
        
        [self.activeCaptureController actionWhenActive];

        __block typeof(self) blockSelf = self;
        
        [[NSOperationQueue mainQueue] addOperationWithBlock:^{
            [blockSelf adjustVideoZoomFactorWithCaptureMode:self.activeCaptureMode];
        }];
    }];
}


//==============================================================================
//
//==============================================================================
- (void)recvUIApplicationDidEnterBackgroundNotification:(NSNotification *)notify
{
    [self.activeCaptureController actionWhenDeactive];
    
    self.ppCameraView.delegate = nil;
    [self.ppCameraView stopCameraPreview];
}


//==============================================================================
//
//==============================================================================
- (void)recvUIDeviceOrientationDidChangeNotification:(NSNotification *)notify
{
    UIDeviceOrientation oldOrientation = self.actualDeviceOrientation;
    self.actualDeviceOrientation = [UIDevice currentDevice].orientation;

    // 只有captureViewController正在顯示時才需要調整旋轉方向
    if (self.isViewLoaded == YES && self.view.window != nil)
    {
        [self adjustDeviceOrientationWithCaptureMode:self.activeCaptureMode];
        
        // !!如果是平板模式，平放，不用重新layout
        if((UI_USER_INTERFACE_IDIOM()==UIUserInterfaceIdiomPad) &&
           (self.actualDeviceOrientation == UIDeviceOrientationUnknown  ||
            self.actualDeviceOrientation == UIDeviceOrientationFaceUp   ||
            self.actualDeviceOrientation == UIDeviceOrientationFaceDown ))
        {
            return ;
        }
        
        // !!方向不一樣就要重新layout
        if (self.actualDeviceOrientation!=oldOrientation)
        {
            [self.view setNeedsLayout];
        }
    }    
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Gesture recognition actions

//================================================================================
//
//================================================================================
- (void)onSwipeLeft
{
    if([PPGuideController isGuideDisplaying] == YES)
    {
        return;
    }
    
    // scrollMenu只要看不到就不能操作
    if([self.scrollMenu isHidden] == YES)
    {
        return;
    }
    
    [self.scrollMenu scrollToRightSideItem];
}


//================================================================================
//
//================================================================================
- (void)onSwipeRight
{
    if([PPGuideController isGuideDisplaying] == YES)
    {
        return;
    }
    
    // scrollMenu只要看不到就不能操作
    if([self.scrollMenu isHidden] == YES)
    {
        return;
    }
    
    [self.scrollMenu scrollToLeftSideItem];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private

//===============================================================================
//
//===============================================================================
- (void)logMessageWithFormat:(NSString *)format, ...
{
    va_list arguments;
    va_start(arguments, format);
    [self.logController logWithMask:PPLogControllerMask_Normal format:format arguments:arguments];
    va_end(arguments);
}


//================================================================================
//
//================================================================================
- (void)restoreDeviceOrientation
{
    // MARK: 平放手機關閉時的特殊處理
    // !! 平板不要特殊處理
    if((UI_USER_INTERFACE_IDIOM()==UIUserInterfaceIdiomPhone) &&
       self.actualDeviceOrientation == UIDeviceOrientationFaceUp)
    {
        // !! 如果原始方向是Faceup，但被強制旋轉到Landscape，關閉時把方向改回Faceup是不夠的，
        //    後續present其他viewController時還是會從landscape出現。
        //    目前唯一可行的方法是先改成Portrait，過一段時間再改Faceup。
        
        [self manuallyRotateDeviceWithOrientation:UIDeviceOrientationPortrait];
        
        // !! delay時間不夠還是會失敗，到時候再看情況調整。
        double delayInSeconds = 1.0;
        dispatch_time_t popTime = dispatch_time(DISPATCH_TIME_NOW, (int64_t)(delayInSeconds*NSEC_PER_SEC));
        
        dispatch_after(popTime, dispatch_get_main_queue(), ^(void){
            
            [[UIDevice currentDevice] setValue:@(UIDeviceOrientationFaceUp) forKey:@"orientation"];
        });
    }
    else
    {
        [self manuallyRotateDeviceWithOrientation:self.actualDeviceOrientation];
    }
}


//================================================================================
//
//================================================================================
- (void)manuallyRotateDeviceWithOrientation:(UIDeviceOrientation)orientation
{
    // !! 為了要區別手機方向是否真的有變動，程式強制旋轉時要停止接收旋轉通知。
    [[NSNotificationCenter defaultCenter] removeObserver:self
                                                    name:UIDeviceOrientationDidChangeNotification
                                                  object:nil];

    [[UIDevice currentDevice] setValue:@(orientation) forKey:@"orientation"];
    
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(recvUIDeviceOrientationDidChangeNotification:)
                                                 name:UIDeviceOrientationDidChangeNotification
                                               object:nil];
}

//================================================================================
//
//================================================================================
- (void)adjustDeviceOrientationWithCaptureMode:(WCCaptureViewController_CaptureMode)captureMode
{
    switch (captureMode)
    {
        case CaptureMode_Manual:
        {
            // 手機水平放置時，manualCapture預設開啟方向為landscape。
            // !! 平板平放，以進入方式為準，不強制修改方向
            if((UI_USER_INTERFACE_IDIOM()==UIUserInterfaceIdiomPhone) &&
               (self.actualDeviceOrientation == UIDeviceOrientationUnknown  ||
                self.actualDeviceOrientation == UIDeviceOrientationFaceUp   ||
                self.actualDeviceOrientation == UIDeviceOrientationFaceDown ))
            {
                [self manuallyRotateDeviceWithOrientation:WCCaptureViewController_FaceUpReplaceOrientation];
            }
            else
            {
                [self manuallyRotateDeviceWithOrientation:self.actualDeviceOrientation];
            }
            
            break;
        }

        case CaptureMode_Docking:
        {
            [self manuallyRotateDeviceWithOrientation:UIDeviceOrientationPortrait];
            break;
        }
            
        default:
            break;
    }
}


//================================================================================
//
//================================================================================
- (UIInterfaceOrientation)curInterfaceOrientation
{
    return [UIApplication sharedApplication].statusBarOrientation;
}


//================================================================================
//
//================================================================================
- (void)adjustOriginWithView:(UIView *)view
{
    CGRect frame = view.frame;
    frame.origin = CGPointZero;
    view.frame = frame;
}


//================================================================================
//
//================================================================================
- (void)adjustTransformWithView:(UIView *)view interfaceOrientation:(UIInterfaceOrientation)interfaceOrientation
{
    switch (interfaceOrientation)
    {
        case UIInterfaceOrientationPortrait:
            view.transform = CGAffineTransformMakeRotation(0);
            break;
            
        case UIInterfaceOrientationPortraitUpsideDown:
            view.transform = CGAffineTransformMakeRotation(M_PI);
            break;
            
        case UIInterfaceOrientationLandscapeLeft:
            view.transform = CGAffineTransformMakeRotation(M_PI/2);
            break;
            
        case UIInterfaceOrientationLandscapeRight:
            view.transform = CGAffineTransformMakeRotation(M_PI*3/2);
            break;
            
        default:
            break;
    }

}


//================================================================================
//
//================================================================================
- (WCCaptureViewController_CaptureMode)captureModeWithString:(NSString *)captureModeString
{
    if([captureModeString isEqualToString:WCCVC_MLS_ManualCapture] == YES)
    {
        return CaptureMode_Manual;
    }
    else if([captureModeString isEqualToString:WCCVC_MLS_DockingCapture] == YES)
    {
        return CaptureMode_Docking;
    }
    else if([captureModeString isEqualToString:WCCVC_MLS_QRCodeCapture] == YES)
    {
        return CaptureMode_QRCode;
    }
    else
    {
        return CaptureMode_None;
    }
}


//================================================================================
//
//================================================================================
- (void)setupMainViewLayoutConstraint
{
    NSInteger safeAreaInsetsTop = 0;
    NSInteger safeAreaInsetsBottom = 0;
    
    // !! safeAreaInsets只有ios11以上可以用
    if(@available(iOS 11.0, *))
    {
        // 只要不是橫式都用直式排序
        if (UIInterfaceOrientationIsPortrait(self.interfaceOrientation))
        {
            safeAreaInsetsTop = self.view.safeAreaInsets.top;
            safeAreaInsetsBottom = self.view.safeAreaInsets.top;
        }
        else
        {
            safeAreaInsetsTop = self.view.safeAreaInsets.left;
            safeAreaInsetsBottom = self.view.safeAreaInsets.left;
        }
    }

    NSDictionary *views = @{@"scrollMenu":self.scrollMenu,
#ifdef WCCapture_ShowZoomFactor
                            @"debugView":self.debugView,
#endif
                            @"ppCameraView":self.ppCameraView,
                            @"overlayView":self.activeCaptureController.overlayView,
                            @"captureBar":self.activeCaptureController.captureBar};
    
    NSDictionary *metrics = @{@"viewGap": @(WCCaptureViewController_ViewGap),
                              @"settingsBarHeight":@(WCCVC_LD_SettingsBarHeight),
                              @"captureBarHeight":@(WCCVC_LD_CaptureBarHeight),
                              @"scrollMenuHeight":@(WCCVC_LD_ScrollMenuBarHeight),
                              @"safeAreaInsetsTop":@(safeAreaInsetsTop),
                              @"safeAreaInsetsBottom":@(safeAreaInsetsBottom)};

    NSMutableArray *layoutConstraints = [NSMutableArray array];
    
    
    // 水平
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[overlayView]|"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:metrics
                                                                                     views:views]];
    
    if(self.activeCaptureMode == CaptureMode_Docking)
    {
        [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[ppCameraView]-(settingsBarHeight)-|"
                                                                                       options:NSLayoutFormatDirectionLeftToRight
                                                                                       metrics:metrics
                                                                                         views:views]];
    }
    else
    {
        [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[ppCameraView]|"
                                                                                       options:NSLayoutFormatDirectionLeftToRight
                                                                                       metrics:metrics
                                                                                         views:views]];
    }

    if([self.scrollMenu isHidden] == NO)
    {
        [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[scrollMenu]|"
                                                                                       options:NSLayoutFormatDirectionLeftToRight
                                                                                       metrics:metrics
                                                                                         views:views]];
    }
    
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[captureBar]|"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:metrics
                                                                                     views:views]];
#ifdef WCCapture_ShowZoomFactor

    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[debugView]|"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:metrics
                                                                                     views:views]];
#endif
    // 垂直
    if([self.scrollMenu isHidden] == NO)
    {
        [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|-(safeAreaInsetsTop)-[overlayView][scrollMenu(scrollMenuHeight)][captureBar(captureBarHeight)]-(safeAreaInsetsBottom)-|"
                                                                                       options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                       metrics:metrics
                                                                                         views:views]];
        
        [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|-(safeAreaInsetsTop)-[ppCameraView][scrollMenu(scrollMenuHeight)][captureBar(captureBarHeight)]-(safeAreaInsetsBottom)-|"
                                                                                       options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                       metrics:metrics
                                                                                         views:views]];
#ifdef WCCapture_ShowZoomFactor
        [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:[debugView(90)][scrollMenu(scrollMenuHeight)][captureBar(captureBarHeight)]-(safeAreaInsetsBottom)-|"
                                                                                       options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                       metrics:metrics
                                                                                         views:views]];
#endif
    }
    else
    {
        [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|-(safeAreaInsetsTop)-[overlayView][captureBar(captureBarHeight)]-(safeAreaInsetsBottom)-|"
                                                                                       options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                       metrics:metrics
                                                                                         views:views]];
        
        [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|-(safeAreaInsetsTop)-[ppCameraView][captureBar(captureBarHeight)]-(safeAreaInsetsBottom)-|"
                                                                                       options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                       metrics:metrics
                                                                                         views:views]];
#ifdef WCCapture_ShowZoomFactor
        [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:[debugView(90)][captureBar(captureBarHeight)]-(safeAreaInsetsBottom)-|"
                                                                                       options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                       metrics:metrics
                                                                                         views:views]];
#endif
    }
    

    //////////////////////////////////////////////////
    // synchronize mainViewLayoutConstraints
    
    @synchronized(self)
    {
        if ([layoutConstraints count] > 0)
        {
            if(self.mainViewLayoutConstraints != nil)
            {
                [self.mainView removeConstraints:self.mainViewLayoutConstraints];
            }
            
            [self.mainView addConstraints:layoutConstraints];
            self.mainViewLayoutConstraints = layoutConstraints;
        }
    }
}


//================================================================================
//
//================================================================================
- (UIWindow *)keyWindow
{
    UIWindow *keyWindow = [UIApplication sharedApplication].keyWindow;
    
    if (!keyWindow)
    {
        keyWindow = [[UIApplication sharedApplication].windows objectAtIndex:0];
    }
    
    return keyWindow;
}


//================================================================================
//
//================================================================================
- (void)animateChangeToNewCaptureModeWithIndex:(NSInteger)newCaptureModeIndex
{
    WCCaptureViewController_CaptureMode newCaptureMode = [[self.supportCaptureModes objectAtIndex:newCaptureModeIndex] integerValue];

    //////////////////////////////////////////////////
    // 切換Mode
    
    self.activeCaptureMode = newCaptureMode;
    
    //////////////////////////////////////////////////
    
    [self setupMainViewLayoutConstraint];
    [self.mainView layoutIfNeeded];
    
    [self.activeCaptureController addOptionalSubviews];
    [self.activeCaptureController setupOverlayViewLayoutConstraintWithStyle:LayoutStyle_AnimationStart];
    [self.activeCaptureController.overlayView layoutIfNeeded];
    
    //////////////////////////////////////////////////
    
    __block typeof(self) blockSelf = self;
    
    [UIView animateWithDuration:WCCaptureViewController_AnimateDuration
                          delay:0.0
                        options:UIViewAnimationOptionBeginFromCurrentState
                     animations:^{
                         
                         [blockSelf.activeCaptureController setupOverlayViewLayoutConstraintWithStyle:LayoutStyle_AnimationStop];
                         [blockSelf.activeCaptureController.overlayView layoutIfNeeded];
                         [blockSelf.scrollMenu moveToItemIndex:newCaptureModeIndex];
                     }
                     completion:^(BOOL finished){
                         
                         [blockSelf adjustDeviceOrientationWithCaptureMode:blockSelf.activeCaptureMode];
                         
                         [blockSelf.activeCaptureController actionWhenActive];
                         
                         // !! 強制顯示Info
                         [blockSelf.activeCaptureController updateInfoLabelLastingWithText:nil];
                       
                         // !! 拿掉tips
//                         [blockSelf showGuideIfNeeded];
                         
                         // for iOS7
                         if ([[UIDevice currentDevice] systemVersion].floatValue < 8.0)
                         {
                             // MARK: [iOS7顯示流程-4] 切換到手持拍照時，需要強制畫面旋轉。
                             [blockSelf.activeCaptureController setupSubviewsInterfaceOrientation:[self curInterfaceOrientation]];
                         }
                    }];
}


//================================================================================
//
//================================================================================
- (void)resetCameraFocusWithPointValue:(NSValue *)focusPointValue
{
    if(focusPointValue == nil)
    {
        return;
    }

    CGPoint newFocusPoint = [focusPointValue CGPointValue];
    CGFloat pointDiff = ABS(self.ppCameraView.focusPoint.x-newFocusPoint.x) + ABS(self.ppCameraView.focusPoint.y-newFocusPoint.y);
    
    // !! 太接近的點略過不設，否則一直重複對焦動作，反而容易干擾使用者操作。
    if(pointDiff > 4.0)
    {
        self.ppCameraView.focusPoint = newFocusPoint;
    }
}


//================================================================================
//
//================================================================================
- (void)showGuideIfNeeded
{
    switch (self.activeCaptureMode)
    {
        case CaptureMode_Manual:
        {
            if([PPSettingsController integerValueWithKey:WCCaptureViewControllr_SettingsKey_ShowManualCaptureGuide] == 1)
            {
//                [self temporarilyHideMenu:YES];
                [self pauseCameraPreview:YES];
                [self.activeCaptureController manuallyShowGuide];
            }
            
            break;
        }            
        default:
            break;
    }
    
}


//================================================================================
//
//================================================================================
- (void)pauseCameraPreview:(BOOL)pause
{
    if(pause == YES)
    {
        [self.ppCameraView pauseCameraPreview];
    }
    else
    {
        [self.ppCameraView resumeCameraPreview];
    }
}


//================================================================================
//
//================================================================================
- (void)prepareCamera
{
    // synchronize activeCaptureController
    @synchronized(self)
    {
        if(self.activeCaptureController != nil)
        {
            [self.ppCameraView setFocusMode:self.activeCaptureController.focusMode];
        }
        
        [self.ppCameraView turnOnFlash:self.enableFlash useAutoMode:NO];
        [self.ppCameraView turnOnTorch:self.enableTorch];
        
    }
}




////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - CVCScrollMenuDelegate methods

//================================================================================
//
//================================================================================
- (void)scrollMenu:(CVCScrollMenu *)scrollMenu willScrollToIndex:(NSInteger)toIndex
{
    WCCaptureViewController_CaptureMode captureMode = [[self.supportCaptureModes objectAtIndex:toIndex] integerValue];
    BOOL shouldChange = YES;
    
    if([self.delegate respondsToSelector:@selector(captureViewController:shouldChangeToCaptureMode:)])
    {
        shouldChange = [self.delegate captureViewController:self shouldChangeToCaptureMode:captureMode];
    }
    
    if(shouldChange == YES)
    {
        __block typeof(self) blockSelf = self;
        
        dispatch_async(dispatch_get_main_queue(), ^{
            [blockSelf animateChangeToNewCaptureModeWithIndex:toIndex];
        });
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - CVCBaseCaptureControllerDelegate


//================================================================================
//
//================================================================================
- (UIView *)captureControllerRequestShowGuide:(CVCBaseCaptureController *)captureController
{
    [self pauseCameraPreview:YES];
    
    // !! 指定要顯示guide的view使用的guideClassName
    self.mainView.ppGuideClassName = NSStringFromClass([self class]);

    return self.mainView;
}


//================================================================================
//
//================================================================================
- (void)captureControllerRequestSelectRecogLanguage:(CVCBaseCaptureController *)captureController
{
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_main_queue(), ^{
    
        blockSelf.selectController = nil;
        blockSelf.selectController = [[[PPSelectController alloc] init] autorelease];
        
        if(blockSelf.selectController != nil)
        {
            blockSelf.selectController.delegate = blockSelf;
            blockSelf.selectController.sourceItemStringArray = blockSelf.recogLanguages;
            [blockSelf.selectController showFromViewController:blockSelf animated:YES];
        }
    });
}


//================================================================================
//
//================================================================================
- (void)captureControllerRequestCapture:(CVCBaseCaptureController *)captureController
{
    if(self.isCapturingPhoto == YES)
    {
        return;
    }
    
    self.isCapturingPhoto = YES;
    
    //////////////////////////////////////////////////
    
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_main_queue(), ^{
        
        if([blockSelf.delegate respondsToSelector:@selector(captureViewController:
                                                            shouldCaptureImageWithCaptureSide:
                                                            isDoubleSideMode:)] == YES)
        {
            if([blockSelf.delegate captureViewController:blockSelf
                    shouldCaptureImageWithCaptureSide:blockSelf.captureSide
                                        isDoubleSideMode:blockSelf.enableDoubleSide] == YES)
            {
                [blockSelf pauseCameraPreview:YES];
                [blockSelf.ppCameraView captureStillImagWithStableCheck:NO];
                return;
            }
        }
        
        //////////////////////////////////////////////////

        blockSelf.isCapturingPhoto = NO;
    });
}


//================================================================================
//
//================================================================================
- (void)captureControllerRequestCancel:(CVCBaseCaptureController *)captureController
{
    [self logMessageWithFormat:@"%s", __func__];
    
    //////////////////////////////////////////////////

    self.ppCameraView.delegate = nil;
    [self.activeCaptureController actionWhenDeactive];
    
    //////////////////////////////////////////////////
    
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_main_queue(), ^{
    
        if([blockSelf.delegate respondsToSelector:@selector(captureViewControllerDidClickCancel:)])
        {
            [blockSelf logMessageWithFormat:@"%s call delegate captureViewControllerDidClickCancel", __func__];

            [blockSelf.delegate captureViewControllerDidClickCancel:blockSelf];
        }    
    });
}


//================================================================================
//
//================================================================================
- (void)captureControllerRequestDone:(CVCBaseCaptureController *)captureController
{
    [self logMessageWithFormat:@"%s", __func__];

    //////////////////////////////////////////////////

    self.ppCameraView.delegate = nil;
    [self.activeCaptureController actionWhenDeactive];

    //////////////////////////////////////////////////
    
    __block typeof(self) blockSelf = self;

    dispatch_async(dispatch_get_main_queue(), ^{
    
        if([blockSelf.delegate respondsToSelector:@selector(captureViewControllerDidClickDone:)])
        {
            [blockSelf logMessageWithFormat:@"%s call delegate captureViewControllerDidClickDone", __func__];

            [blockSelf.delegate captureViewControllerDidClickDone:blockSelf];
        }
    });
}


//================================================================================
//
//================================================================================
- (void)captureControllerRequestSkip:(CVCBaseCaptureController *)captureController
{
    [self.activeCaptureController actionWhenSkipBackSide];
    
    if([self.delegate respondsToSelector:@selector(captureViewControllerSkipBackSide:)])
    {
        [self.delegate captureViewControllerSkipBackSide:self];
    }
}


//================================================================================
//
//================================================================================
- (void)captureController:(CVCBaseCaptureController *)captureController requestUpdateFlashState:(BOOL)state
{
#ifdef WCCapture_UltraWideCameraTest
    [self debugTest];
#else
    self.enableFlash = state;
    [self.ppCameraView turnOnFlash:state useAutoMode:NO];
#endif
    
}


//================================================================================
//
//================================================================================
- (void)captureController:(CVCBaseCaptureController *)captureController requestUpdateTorchState:(BOOL)state
{
    self.enableTorch = state;
    [self.ppCameraView turnOnTorch:state];
}


//================================================================================
//
//================================================================================
- (void)captureController:(CVCBaseCaptureController *)captureController requestUpdateImageEnhanceState:(BOOL)state
{
    self.enableImageEnhance = state;
}


//================================================================================
//
//================================================================================
- (void)captureController:(CVCBaseCaptureController *)captureController requestUpdateAutoCropState:(BOOL)state
{
    self.enableAutoCrop = state;
}


//================================================================================
//
//================================================================================
- (void)captureController:(CVCBaseCaptureController *)captureController requestUpdateDoubleSideState:(BOOL)state
{
    self.enableDoubleSide = state;
}


//================================================================================
//
//================================================================================
- (void)captureController:(CVCBaseCaptureController *)captureController requestUpdateCaptureSideState:(WCCaptureViewController_CaptureSide)captureSide
{
    self.captureSide = captureSide;
}


//================================================================================
//
//================================================================================
- (CGSize)captureControllerRequestGetCameraViewSize:(CVCBaseCaptureController *)captureController
{
    return self.ppCameraView.bounds.size;
}


//================================================================================
//
//================================================================================
- (void)captureController:(CVCBaseCaptureController *)captureController requestSetFocusPoint:(CGPoint)focusPoint
{
    [self resetCameraFocusWithPointValue:[NSValue valueWithCGPoint:focusPoint]];
}


//================================================================================
//
//================================================================================
- (void)captureController:(CVCBaseCaptureController *)captureController requestSendQRCodeString:(NSString *)qrCodeString
{
    if([self.delegate respondsToSelector:@selector(captureViewController:didRecongizeQRCodeString:)])
    {
        [self.delegate captureViewController:self didRecongizeQRCodeString:qrCodeString];
    }
}


//================================================================================
//
//================================================================================
- (void)captureController:(CVCBaseCaptureController*)captureController requestLogMessage:(NSString *)message
{
    [self logMessageWithFormat:@"%@", message];
}


//================================================================================
//
//================================================================================
- (void)captureControllerRequestShowDockPromotion:(CVCBaseCaptureController*)captureController
{
    if([self.delegate respondsToSelector:@selector(captureViewControllerShowDockPromotion:)])
    {
        [self.delegate captureViewControllerShowDockPromotion:self];
    }
}


//================================================================================
//
//================================================================================
- (void)captureController:(CVCBaseCaptureController*)captureController requestSwitchToMode:(WCCaptureViewController_CaptureMode)captureMode
{
    if([self.delegate respondsToSelector:@selector(captureViewController:shouldChangeToCaptureMode:)])
    {
        [self.delegate captureViewController:self shouldChangeToCaptureMode:captureMode];
    }
    
    NSInteger index = [self.supportCaptureModes indexOfObject:@(captureMode)];
    
    [self animateChangeToNewCaptureModeWithIndex:index];
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPCameraViewDelegate

//================================================================================
//
//================================================================================
- (void)ppCameraView:(PPCameraView *)ppCameraView didReceivePreviewImage:(UIImage *)image
{
    if([self.ppCameraView isFocusing]==YES || self.isCapturingPhoto==YES)
    {
        return;
    }
    
    [self.activeCaptureController actionWhenReceivePreviewImage:image];
}


//================================================================================
//
//================================================================================
- (void)ppCameraView:(PPCameraView *)ppCameraView didGetStillImage:(UIImage *)image
{
#if (TARGET_IPHONE_SIMULATOR)
    
    if(self.captureSide == CaptureSide_Front)
    {
        image = [UIImage imageNamed:@"WCMCC_FakeCardFrontSide.jpg"];
    }
    else
    {
        image = [UIImage imageNamed:@"WCMCC_FakeCardBackSide.jpg"];
    }
    
    
#endif
    
    //////////////////////////////////////////////////
    
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_main_queue(), ^{
        
        if(image != nil)
        {
            //////////////////////////////////////////////////
            // 第一次拍照後就禁止變換模式
            
            if(blockSelf.scrollMenu.hidden == NO)
            {
                blockSelf.scrollMenu.hidden = YES;
                [blockSelf setupMainViewLayoutConstraint];
                [blockSelf.mainView layoutIfNeeded];
            }
            
            //////////////////////////////////////////////////
            
            if([blockSelf.delegate respondsToSelector:@selector(captureViewController:
                                                                didCaptureImage:
                                                                recogLanguage:
                                                                isDoubleSideMode:)])
            {
                UIImageOrientation imageOrientation = UIImageOrientationUp;
                
                switch ([blockSelf curInterfaceOrientation])
                {
                    case UIInterfaceOrientationLandscapeLeft: imageOrientation = UIImageOrientationRight; break;
                    case UIInterfaceOrientationLandscapeRight: imageOrientation = UIImageOrientationLeft; break;
                    case UIInterfaceOrientationPortraitUpsideDown: imageOrientation = UIImageOrientationDown; break;
                    default: imageOrientation = UIImageOrientationUp; break;
                }

                UIImage *adjustedImage = [[UIImage alloc] initWithCGImage:image.CGImage
                                                                    scale:1.0
                                                              orientation:imageOrientation];
                
                
                // !!在iphone 13 pro以上的手機，會用超廣角，造成無法裁切，所以這邊要先裁小一點
#ifdef WCCapture_UltraWideCameraTest
                PPDeviceModelID deviceID = [UIDevice deviceModelID];
                if((deviceID>=PPDeviceModelID_iPhone_13Mini||deviceID==PPDeviceModelID_iPhone)&&
                   [PPCameraView isSupportTripleCamera]==YES &&
                   blockSelf.activeCaptureMode==CaptureMode_Docking)
                {
                    NSNumber *cropRatioNumber = [PPSettingsController numberValueWithKey:WCCaptureViewControllr_SettingsKey_UltraWideCameraTest_CropRatio];
                    NSNumber *cropOffsetYNumber = [PPSettingsController numberValueWithKey:WCCaptureViewControllr_SettingsKey_UltraWideCameraTest_CropOffsetY];

                    CGFloat cropRatio = [cropRatioNumber floatValue];
                    CGFloat cropOffsetY = [cropOffsetYNumber floatValue];
                    CGFloat cropedWidth = ceilf(adjustedImage.size.width*cropRatio);
                    CGFloat cropedHeight = ceilf(adjustedImage.size.height*cropRatio);
                    CGRect cropRect = CGRectMake((adjustedImage.size.width-cropedWidth)/2, (adjustedImage.size.height-cropedHeight)/2+cropOffsetY, cropedWidth, cropedHeight);
                    UIImage *cropedImage = [adjustedImage imageCroppedByRect:cropRect];
                    [adjustedImage release];
                    adjustedImage = [cropedImage retain];
                }
#endif

                [blockSelf.delegate captureViewController:blockSelf
                                          didCaptureImage:adjustedImage
                                            recogLanguage:[blockSelf.recogLanguages firstObject]
                                         isDoubleSideMode:blockSelf.enableDoubleSide];
                
                [adjustedImage release];
            }
            
            [blockSelf.activeCaptureController actionWhenGetStillImage];
        }
        
        [blockSelf pauseCameraPreview:NO];
        blockSelf.isCapturingPhoto = NO;
    });
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPSelectController Delegate

//================================================================================
//
//================================================================================
- (void)ppSelectController:(PPSelectController *)selectController itemSelectedIndex:(NSInteger)itemIndex withString:(NSString *)itemString
{
    if(itemIndex >= 0)
    {
        [self.activeCaptureController setupRecogLanguageState:itemString];
        
        //////////////////////////////////////////////////

        //通知語系變動
        if([self.delegate respondsToSelector:@selector(captureViewController:didSelectRecogLangauge:withCaptureSide:)]==YES)
        {
            [self.delegate captureViewController:self
                          didSelectRecogLangauge:itemString
                                 withCaptureSide:self.captureSide];
        }
        
        //////////////////////////////////////////////////

        //交換已知辨識語系順序
        do
        {
            NSMutableArray *recogLanguages = [NSMutableArray array];
            
            if(recogLanguages==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////

            for (NSString *recogLanguage in self.recogLanguages)
            {
                if([recogLanguage compare:itemString]==NSOrderedSame)
                {
                    [recogLanguages insertObject:recogLanguage atIndex:0];
                }
                else
                {
                    [recogLanguages addObject:recogLanguage];
                }
            }
            
            //////////////////////////////////////////////////

            self.recogLanguages = recogLanguages;
        }
        while (0);
    }
}


//================================================================================
//
//================================================================================
- (void)ppSelectControllerDidDismiss:(PPSelectController *)selectController
{
    self.selectController = nil;
}


//================================================================================
//
//================================================================================
- (BOOL)ppSelectControllerShouldHideAfterSelected:(PPSelectController *)selectController
{
    return YES;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPGuideControllerDelegate methods

//================================================================================
//
//================================================================================
- (void)guideControllerDidFinishScript:(NSString *)guideScriptName
{
    [self pauseCameraPreview:NO];
    
    if([guideScriptName isEqualToString:WCCVC_ManualCaptureGuide] == YES)
    {
        [PPSettingsController setIntegerValue:0 withKey:WCCaptureViewControllr_SettingsKey_ShowManualCaptureGuide];
    }
    else if([guideScriptName isEqualToString:WCCVC_DockingCaptureGuide] == YES)
    {
        [PPSettingsController setIntegerValue:0 withKey:WCCaptureViewControllr_SettingsKey_ShowDockingCaptureGuide];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class methods

//================================================================================
//
//================================================================================
+ (void)presentSingleManualModeFromViewController:(UIViewController *)viewController
                                         delegate:(id<WCCaptureViewControllerDelegate>)delegate
                                      captureSide:(WCCaptureViewController_CaptureSide)captureSide
                                presentCompletion:(void (^)(BOOL success))presentCompletion
{
    [AVCaptureDevice requestAccessCameraWithSuccessHandler:^{
        
        WCCaptureViewController *captureViewController = [[WCCaptureViewController alloc] init];
        
        if(captureViewController != nil)
        {
            //////////////////////////////////////////////////
            // 顯示相機介面
            
            captureViewController.delegate = delegate;
            captureViewController.supportCaptureModes = @[@(CaptureMode_Manual)];
            captureViewController.activeCaptureMode = CaptureMode_Manual;
            captureViewController.captureSide = captureSide;
            
            [viewController presentViewController:captureViewController
                                         animated:YES
                                       completion:^{
                                           
                                           if(presentCompletion != nil)
                                           {
                                               presentCompletion(YES);
                                           }
                                       }];
            
            [captureViewController release];
        }
        else
        {
            // 應該不會發生
            [PPAlertView showWithStyle:UIAlertViewStyleDefault
                                 title:@""
                               message:@"WCCaptureViewController init failed"
                     cancelButtonTitle:WCCVC_MLS_Ok
                     otherButtonTitles:nil];
            
            if(presentCompletion != nil)
            {
                presentCompletion(NO);
            }
        }
        
    } failedHandler:^{
        
        [PPAlertView showWithStyle:UIAlertViewStyleDefault
                             title:@""
                           message:WCCVC_MLS_CameraPolicyFailed
                 cancelButtonTitle:WCCVC_MLS_Ok
                 otherButtonTitles:nil];
        
        if(presentCompletion != nil)
        {
            presentCompletion(NO);
        }
    }];
}


//================================================================================
//
//================================================================================
+ (void)presentFromViewController:(UIViewController *)viewController
                         delegate:(id<WCCaptureViewControllerDelegate>)delegate
                      captureSide:(WCCaptureViewController_CaptureSide)captureSide
                     captureCount:(NSInteger)captureCount
              supportCaptureModes:(NSArray *)supportCaptureModes
                activeCaptureMode:(WCCaptureViewController_CaptureMode)activeCaptureMode
                presentCompletion:(void (^)(BOOL success))presentCompletion
{
    [AVCaptureDevice requestAccessCameraWithSuccessHandler:^{
        
        WCCaptureViewController *captureViewController = [[WCCaptureViewController alloc] init];
        
        if(captureViewController != nil)
        {
            //////////////////////////////////////////////////
            // 顯示相機介面
            
            captureViewController.delegate = delegate;
            captureViewController.captureSide = captureSide;
            captureViewController.captureCount = captureCount;
            captureViewController.supportCaptureModes = supportCaptureModes;
            captureViewController.activeCaptureMode = activeCaptureMode;            
            
            [viewController presentViewController:captureViewController
                                         animated:YES
                                       completion:^{
                                       
                                           if(presentCompletion != nil)
                                           {
                                               presentCompletion(YES);
                                           }
                                       }];
            
            [captureViewController release];
        }
        else
        {
            // 應該不會發生
            [PPAlertView showWithStyle:UIAlertViewStyleDefault
                                 title:@""
                               message:@"WCCaptureViewController init failed"
                     cancelButtonTitle:WCCVC_MLS_Ok
                     otherButtonTitles:nil];
            
            if(presentCompletion != nil)
            {
                presentCompletion(NO);
            }
        }

    } failedHandler:^{
        
        [PPAlertView showWithStyle:UIAlertViewStyleDefault
                             title:@""
                           message:WCCVC_MLS_CameraPolicyFailed
                 cancelButtonTitle:WCCVC_MLS_Ok
                 otherButtonTitles:nil];

        if(presentCompletion != nil)
        {
            presentCompletion(NO);
        }
    }];
}


//================================================================================
//
//================================================================================
+ (void)setDockingCaptureAutoDetectionEnabled:(BOOL)enabled
{
    NSString *key = WCCaptureViewControllr_SettingsKey_EnableDockingCaptureAutoDetection;

    [PPSettingsController setIntegerValue:(NSInteger)enabled withKey:key];
}


//================================================================================
//
//================================================================================
+ (BOOL)dockingCaptureAutoDetectionEnabled
{
    NSString *key = WCCaptureViewControllr_SettingsKey_EnableDockingCaptureAutoDetection;
    
    [PPSettingsController setDefaultIntegerValue:1 withKey:key];

    return (BOOL)[PPSettingsController integerValueWithKey:key];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Log methods

//===============================================================================
//
//===============================================================================
+ (NSString *)logDirPath
{
    return [WCToolController baseStorePathWithDirName:WCCaptureViewController_CaptureLogFolder isCreatDirPath:NO];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Inherit Method

//================================================================================
//
//================================================================================
- (NSString *)captureModeStringWithMode:(WCCaptureViewController_CaptureMode)captureMode
{
    NSString *captureModeString = nil;
    
    switch (captureMode)
    {
        case CaptureMode_Manual:
            captureModeString = WCCVC_MLS_ManualCapture;
            break;

        case CaptureMode_Docking:
            captureModeString = WCCVC_MLS_DockingCapture;
            break;

        case CaptureMode_QRCode:
            captureModeString = WCCVC_MLS_QRCodeCapture;
            break;

        default:
            break;
    }
    
    return captureModeString;
}


//================================================================================
//
//================================================================================
- (CVCBaseCaptureController *)captureControllerWithCaptureMode:(WCCaptureViewController_CaptureMode)captureMode
{
    CVCBaseCaptureController *captureController = nil;

    self.ppInterfaceOrientationController.mask = PPInterfaceOrientationControllerMask_All;

    switch (captureMode)
    {
        case CaptureMode_Manual:
            captureController = [[CVCManualCaptureController alloc] init];
            break;
            
        case CaptureMode_Docking:
            captureController = [[CVCDockingCaptureController alloc] init];
            self.ppInterfaceOrientationController.mask = PPInterfaceOrientationControllerMask_Portrait;
            break;
            
        case CaptureMode_QRCode:
            captureController = [[CVCQRCodeCaptureController alloc] init];
            break;
            
        default:
            break;
    }
    
    if(captureController != nil)
    {
        captureController.delegate = self;
    }
    
    return [captureController autorelease];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - DEBUG



//==============================================================================
//
//==============================================================================
- (void)debugTest
{
    PPAlertController *alertController = [PPAlertController alertControllerWithAlertControllerStyle:UIAlertControllerStyleAlert title:@"測試" message:@"參數調整" alertAction: nil];
    [PPAlertController alertController:alertController addTextFieldHandle:^(UITextField *textField) {
        textField.tag = 1;
        textField.placeholder = @"倍率";
        textField.keyboardType = UIKeyboardTypeDecimalPad;
        NSNumber *cropRatio = [PPSettingsController numberValueWithKey:WCCaptureViewControllr_SettingsKey_UltraWideCameraTest_CropRatio];
        [textField setText:[NSString stringWithFormat:@"%@", cropRatio]];
    },^(UITextField *textField) {
        textField.tag = 2;
        textField.placeholder = @"Offset";
        textField.keyboardType = UIKeyboardTypeNumberPad;
        NSNumber *cropOffsetY = [PPSettingsController numberValueWithKey:WCCaptureViewControllr_SettingsKey_UltraWideCameraTest_CropOffsetY];
        [textField setText:[NSString stringWithFormat:@"%@", cropOffsetY]];
    }, nil];
    
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(textFieldTextDidChangeNotification:)
                                                 name:UITextFieldTextDidChangeNotification
                                               object:nil];
    [alertController addAction:[UIAlertAction actionWithTitle:@"OK" style:UIAlertActionStyleCancel handler:^(UIAlertAction * _Nonnull action) {
        [[NSNotificationCenter defaultCenter] removeObserver:self name:UITextFieldTextDidChangeNotification object:nil];
    }]];
    [self presentViewController:alertController animated:YES completion:nil];
    [alertController release];
}


//==============================================================================
//
//==============================================================================
- (void)textFieldTextDidChangeNotification:(NSNotification*)notification
{
    UITextField * textField = [notification object];
    NSInteger textFieldTag = textField.tag;
    if(textFieldTag==1)
    {
        [PPSettingsController setNumberValue:@([textField.text floatValue]) withKey:WCCaptureViewControllr_SettingsKey_UltraWideCameraTest_CropRatio];
    }
    else if(textFieldTag==2)
    {
        [PPSettingsController setNumberValue:@([textField.text floatValue]) withKey:WCCaptureViewControllr_SettingsKey_UltraWideCameraTest_CropOffsetY];
    }

}

@end
