//
//  CVCDockingCaptureController.m
//  Pods
//
//  Created by Duke on 2015/4/7.
//
//

#import <CoreMotion/CoreMotion.h>
#import "WCCaptureViewController+ResourceDefine.h"
#import "WCCaptureViewController+SettingsKey.h"
#import "PPSettingsController.h"
#import "PPGuideController.h"
#import "CVCDockingCaptureController.h"
#import "PPPatternOCRController.h"
#import "UIDevice+DeviceModel.h"
#import "PPAutoResizedLabel+Factory.h"
#import "PPButton+Factory.h"
#import "PPAlertView.h"
#import "PPInterfaceOrientationController.h"
#import "NSThread+Additions.h"
#import "PPCameraView.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

/// 自動偵測時限 (second)
NSTimeInterval const CVCDockingCaptureController_AutoDetectionWarningInterval = 30.0;

/// 穩定偵測回報時間 (second)
NSTimeInterval const CVCDockingCaptureController_MotionDetectionInterval = 0.2;

////////////////////////////////////////////////////////////////////////////////////////////////////

/// Button tag
typedef NS_ENUM(NSUInteger, CVCDockingCaptureController_ButtonTag)
{
    ButtonTag_None  = 2000,
    ButtonTag_Flash,
    ButtonTag_ImageEnhance,
    ButtonTag_AutoCrop,
    ButtonTag_DoubleSide,
    ButtonTag_Guide,
    ButtonTag_RecogLanguage,
    ButtonTag_Capture,
    ButtonTag_Cancel,
    ButtonTag_Done,
    ButtonTag_Skip,
};

/// Alert tag
typedef NS_ENUM(NSUInteger, CVCDockingCaptureController_AlertTag)
{
    AlertTag_None  = 3000,
    AlertTag_NeedToByDock,
};


////////////////////////////////////////////////////////////////////////////////////////////////////

@interface CVCDockingCaptureController() <UIAlertViewDelegate>

// Preview ToolBar
@property (nonatomic, retain) PPBarView *settingsBar;
@property (nonatomic, retain) PPButton *ppButtonFlash;
@property (nonatomic, retain) PPButton *ppButtonImageEnhance;
@property (nonatomic, retain) PPButton *ppButtonAutoCrop;
@property (nonatomic, retain) PPButton *ppButtonDoubleSide;
@property (nonatomic, retain) PPButton *ppButtonGuide;

// Capture ToolBar
@property (nonatomic, retain) PPButton *ppButtonRecogLanguage;
@property (nonatomic, retain) PPButton *ppButtonCapture;
@property (nonatomic, retain) PPButton *ppButtonCancelDone;

// Skip Button
@property (nonatomic, retain) PPButton *ppButtonSkip;

// Subviews Of OverlayView
@property (nonatomic, retain) UIImageView *cardDirectionImageView;

// Others for UI
@property (nonatomic, assign) BOOL isDoubleSideMode;
@property (nonatomic ,assign) NSInteger captureCount;
@property (nonatomic, assign) WCCaptureViewController_CaptureSide captureSide;
@property (nonatomic, retain) NSString *recogLanguage;
@property (nonatomic, assign) CGSize cameraViewSize;
@property (nonatomic, retain) PPAlertView *alertView;

// auto detection
@property (nonatomic, assign) BOOL autoDetectionEnabled; // default is YES
@property (nonatomic, assign) BOOL hasDetectedQRCode;
@property (nonatomic, assign) BOOL hasCardInDock;
@property (nonatomic, retain) NSDate *autoDetectionBeginDate;
@property (nonatomic, retain) NSDate *focusAdjustBeginDate;
@property (nonatomic, retain) NSOperationQueue *recognitionOperationQueue;

// device position
@property (atomic, retain) CMMotionManager *motionManager;
@property (atomic, retain) NSOperationQueue *motionOperationQueue;
@property (atomic, assign) BOOL isDevicePositionConfirmed;

@property (atomic, assign) BOOL isActive;

@end


////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation CVCDockingCaptureController

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods

//================================================================================
//
//================================================================================
- (id)init
{
    if(self = [super init])
    {
        //////////////////////////////////////////////////
        
        self.autoDetectionEnabled = YES;
        self.cameraViewSize = CGSizeZero;
        self.isDoubleSideMode = YES;
        self.captureSide = CaptureSide_Front;
        self.captureCount = 0;
        self.focusMode = AVCaptureFocusModeAutoFocus;

        
        //////////////////////////////////////////////////
        // Capture ToolBar
        
        _ppButtonRecogLanguage = [[PPButton alloc] initWithFrame:UIButtonAdditions_DefaultRect];
        if (self.ppButtonRecogLanguage != nil)
        {
            self.ppButtonRecogLanguage.titleEdgeInsets = WCCVC_LD_CaptureBarTitleEdgeInset;
            self.ppButtonRecogLanguage.tag = ButtonTag_RecogLanguage;
            self.ppButtonRecogLanguage.ppGuideIndex = ButtonTag_RecogLanguage;
            [self.ppButtonRecogLanguage addControlEvents:UIControlEventTouchUpInside];
            [self.ppButtonRecogLanguage setDelegate:self];
        }

        //--------------------------------------------------
        
        self.ppButtonCapture = [PPButton ppButtonWithIconImageName:WCCVC_IFP_CaptureButton
                                                               tag:ButtonTag_Capture
                                                          delegate:self
                                             normalBackgroundColor:nil
                                        highlightedBackgroundColor:nil
                                                   imageEdgeInsets:WCCVC_LD_CaptureBarImageEdgeInset];

        if (self.ppButtonCapture != nil)
        {
            self.ppButtonCapture.ppGuideIndex = self.ppButtonCapture.tag;
            [self.ppButtonCapture addControlEvents:UIControlEventTouchUpInside];
        }
        
        //--------------------------------------------------
        
        self.ppButtonCancelDone = [PPButton ppButtonWithIconImageName:WCCVC_IFP_CancelButton
                                                                  tag:ButtonTag_Cancel
                                                             delegate:self
                                                normalBackgroundColor:nil
                                           highlightedBackgroundColor:nil
                                                      imageEdgeInsets:WCCVC_LD_CaptureBarImageEdgeInset];

        if (self.ppButtonCancelDone != nil)
        {
            self.ppButtonCancelDone.ppGuideIndex = ButtonTag_Cancel;
            [self.ppButtonCancelDone addControlEvents:UIControlEventTouchUpInside];
            
            self.ppButtonCancelDone.clipsToBounds = NO;
            self.ppButtonCancelDone.imageView.clipsToBounds = NO;
        }

        //--------------------------------------------------
        
        if (self.captureBar != nil)
        {
            self.captureBar.ignoreEdgeInset = YES;
            self.captureBar.style = PPBarViewStyle_Average;
            [self.captureBar setViews:@[self.ppButtonRecogLanguage] forBlockType:PPBarViewBlockType_Left];
            [self.captureBar setViews:@[self.ppButtonCapture] forBlockType:PPBarViewBlockType_Center];
            [self.captureBar setViews:@[self.ppButtonCancelDone] forBlockType:PPBarViewBlockType_Right];
        }
        
        
        //////////////////////////////////////////////////
        // Settings ToolBar
        
        self.ppButtonFlash = [PPButton ppButtonWithIconImageName:WCCVC_IFP_FlashButton
                                                             tag:ButtonTag_Flash
                                                        delegate:self
                                           normalBackgroundColor:nil
                                      highlightedBackgroundColor:nil
                                                 imageEdgeInsets:WCCVC_LD_SettingsBarImageEdgeInset];
        
        if (self.ppButtonFlash != nil)
        {
            self.ppButtonFlash.ppGuideIndex = ButtonTag_Flash;
            [self.ppButtonFlash addControlEvents:UIControlEventTouchUpInside];
        }
        
        //--------------------------------------------------
        
        self.ppButtonImageEnhance = [PPButton ppButtonWithIconImageName:WCCVC_IFP_ImageEnhanceButton
                                                                    tag:ButtonTag_ImageEnhance
                                                               delegate:self
                                                  normalBackgroundColor:nil
                                             highlightedBackgroundColor:nil
                                                        imageEdgeInsets:WCCVC_LD_SettingsBarImageEdgeInset];
        
        if (self.ppButtonImageEnhance != nil)
        {
            self.ppButtonImageEnhance.ppGuideIndex = ButtonTag_ImageEnhance;
            [self.ppButtonImageEnhance addControlEvents:UIControlEventTouchUpInside];
        }
        
        //--------------------------------------------------
        
        self.ppButtonAutoCrop = [PPButton ppButtonWithIconImageName:WCCVC_IFP_AutoCropButton
                                                                tag:ButtonTag_AutoCrop
                                                           delegate:self
                                              normalBackgroundColor:nil
                                         highlightedBackgroundColor:nil
                                                    imageEdgeInsets:WCCVC_LD_SettingsBarImageEdgeInset];
        
        if (self.ppButtonAutoCrop != nil)
        {
            self.ppButtonAutoCrop.ppGuideIndex = ButtonTag_AutoCrop;
            [self.ppButtonAutoCrop addControlEvents:UIControlEventTouchUpInside];
        }
        
        //--------------------------------------------------
        
        self.ppButtonDoubleSide = [PPButton ppButtonWithIconImageName:WCCVC_IFP_DoubleSideButton
                                                                  tag:ButtonTag_DoubleSide
                                                             delegate:self
                                                normalBackgroundColor:nil
                                           highlightedBackgroundColor:nil
                                                      imageEdgeInsets:WCCVC_LD_SettingsBarImageEdgeInset];

        if (self.ppButtonDoubleSide != nil)
        {
            self.ppButtonDoubleSide.ppGuideIndex = ButtonTag_DoubleSide;
            [self.ppButtonDoubleSide addControlEvents:UIControlEventTouchUpInside];
            [self.ppButtonDoubleSide setSelected:self.isDoubleSideMode];
        }
        
        //--------------------------------------------------
        
        self.ppButtonGuide = [PPButton ppButtonWithIconImageName:WCCVC_IFP_GuideButton
                                                             tag:ButtonTag_Guide
                                                        delegate:self
                                           normalBackgroundColor:nil
                                      highlightedBackgroundColor:nil
                                                 imageEdgeInsets:WCCVC_LD_SettingsBarImageEdgeInset];
        
        if (self.ppButtonGuide != nil)
        {
            self.ppButtonGuide.ppGuideIndex = ButtonTag_Guide;
            [self.ppButtonGuide addControlEvents:UIControlEventTouchUpInside];
        }
        
        //--------------------------------------------------
        
        _settingsBar = [[PPBarView alloc] init];
        if(self.settingsBar)
        {
            [self.settingsBar setIgnoreEdgeInset:YES];
            [self.settingsBar setStyle:PPBarViewStyle_Average];
            [self.settingsBar setTranslatesAutoresizingMaskIntoConstraints:NO];
            [self.settingsBar setBackgroundColor:WCCVC_LD_DarkBlueColor];
            [self.settingsBar setAlpha:1.0];
            [self.overlayView addSubview:self.settingsBar];
            
            if([PPCameraView canUseFlashForCameraType:PPCameraType_Back] == YES)
            {
                [self.settingsBar setViews:@[self.ppButtonFlash, self.ppButtonImageEnhance, self.ppButtonAutoCrop, self.ppButtonDoubleSide, self.ppButtonGuide] forBlockType:PPBarViewBlockType_Center];
            }
            else
            {
                [self.settingsBar setViews:@[self.ppButtonImageEnhance, self.ppButtonAutoCrop, self.ppButtonDoubleSide, self.ppButtonGuide] forBlockType:PPBarViewBlockType_Center];
            }
        }
        
        
        //////////////////////////////////////////////////
        // target view
        
        if(self.targetView)
        {
            self.targetView.horzBoxInset = WCCVC_LD_UnfixedTargetBoxMargin;
            self.targetView.vertBoxInset = WCCVC_LD_UnfixedTargetBoxMargin;
            [self.overlayView addSubview:self.targetView];
        }
        
        
        //////////////////////////////////////////////////
        // Card Direction Image
        
        _cardDirectionImageView = [[UIImageView alloc] init];
        if (self.cardDirectionImageView)
        {
            [self.cardDirectionImageView setTranslatesAutoresizingMaskIntoConstraints:NO];
            [self updateCardDirectionImageView];
            [self.targetView addSubview:self.cardDirectionImageView];
        }
        
        
        //////////////////////////////////////////////////
        // Skip button
        
        _ppButtonSkip = [[PPButton alloc] initWithFrame:CGRectZero];
        if (self.ppButtonSkip != nil)
        {
            self.ppButtonSkip.hidden = YES;
            self.ppButtonSkip.tag = ButtonTag_Skip;
            self.ppButtonSkip.translatesAutoresizingMaskIntoConstraints = NO;
            
            [self.ppButtonSkip setTitle:WCCVC_MLS_SkipBackSide forState:UIControlStateNormal];
            [self.ppButtonSkip setTitleColor:WCCVC_LD_LightBlueColor forState:UIControlStateNormal];
            
            self.ppButtonSkip.titleLabel.font = [UIFont systemFontOfSize:15];
            self.ppButtonSkip.backgroundColor = [UIColor clearColor];
            self.ppButtonSkip.layer.borderWidth = 1.0;
            self.ppButtonSkip.layer.borderColor = WCCVC_LD_LightBlueColor.CGColor;
            self.ppButtonSkip.layer.masksToBounds = YES;
            self.ppButtonSkip.layer.cornerRadius = 5.0;
            
            [self.ppButtonSkip addControlEvents:UIControlEventTouchUpInside];
            [self.ppButtonSkip setDelegate:self];
            [self.overlayView addSubview:self.ppButtonSkip];
        }            
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    [self logMessageWithFormat:@"%s in", __func__];
    
    //////////////////////////////////////////////////
    
    [self actionWhenDeactive];

    //////////////////////////////////////////////////

    self.cardDirectionImageView = nil;

    self.ppButtonRecogLanguage  = nil;
    self.ppButtonCapture        = nil;
    self.ppButtonCancelDone     = nil;

    self.ppButtonSkip           = nil;

    self.ppButtonFlash          = nil;
    self.ppButtonImageEnhance   = nil;
    self.ppButtonAutoCrop       = nil;
    self.ppButtonDoubleSide     = nil;
    self.ppButtonGuide          = nil;
    self.settingsBar            = nil;
    self.alertView              = nil;
    self.autoDetectionBeginDate = nil;
    self.focusAdjustBeginDate   = nil;

    //////////////////////////////////////////////////

    [self logMessageWithFormat:@"%s out", __func__];
    
    [super dealloc];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance Methods - Guide

//================================================================================
//
//================================================================================
- (NSArray *)guideDataModelsWithClassName:(NSString *)className
{
    NSArray *guideDataModels = @[[PPGuideDataModel modelWithTargetClassName:className
                                                                 guideIndex:ButtonTag_Flash
                                                                      title:WCCVC_MLS_GuideFlashTitle
                                                                    message:WCCVC_MLS_GuideFlashMessage
                                                            layoutDirection:PPGuideDataModel_LayoutDirection_Horz],
                                 
                                 [PPGuideDataModel modelWithTargetClassName:className
                                                                 guideIndex:ButtonTag_ImageEnhance
                                                                      title:WCCVC_MLS_GuideImageEnhanceTitle
                                                                    message:WCCVC_MLS_GuideImageEnhanceMessage
                                                            layoutDirection:PPGuideDataModel_LayoutDirection_Horz],
                                 
                                 [PPGuideDataModel modelWithTargetClassName:className
                                                                 guideIndex:ButtonTag_AutoCrop
                                                                      title:WCCVC_MLS_GuideAutoCropTitle
                                                                    message:WCCVC_MLS_GuideAutoCropMessage
                                                            layoutDirection:PPGuideDataModel_LayoutDirection_Horz],
                                 
                                 [PPGuideDataModel modelWithTargetClassName:className
                                                                 guideIndex:ButtonTag_DoubleSide
                                                                      title:WCCVC_MLS_GuideDoubleSideTitle
                                                                    message:WCCVC_MLS_GuideDoubleSideMessage
                                                            layoutDirection:PPGuideDataModel_LayoutDirection_Horz],
                                 ];
    
    return guideDataModels;
}


//================================================================================
//
//================================================================================
- (void)manuallyShowGuide
{
    UIView *guideView = nil;
    
    if([self.delegate respondsToSelector:@selector(captureControllerRequestShowGuide:)])
    {
        guideView = [self.delegate captureControllerRequestShowGuide:self];
    }
    
    
    [PPGuideController clearAllGuideScripts];
    
    [PPGuideController addGuideScriptWithName:WCCVC_DockingCaptureGuide
                              guideDataModels:[self guideDataModelsWithClassName:guideView.ppGuideClassName]];
    
    [PPGuideController manuallyShowGuideWithView:guideView];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance Methods - subview layout

//================================================================================
//
//================================================================================
- (void)addOptionalSubviews
{
    [self addInfoLabelWithSuperView:self.cardDirectionImageView];    
    [self setupInfoLabelLayoutConstraintsWithMaxSize:WCCVC_LD_InfoLabelMaxSize];

    [self addCaptureCountLabelWithSuperview:self.ppButtonCancelDone.imageView];
}


//================================================================================
//
//================================================================================
- (void)setupOverlayViewLayoutConstraintWithStyle:(CVCBaseCaptureController_LayoutStyle)layoutStyle
{
    CGSize ppButtonSkipSize = [self.ppButtonSkip.titleLabel sizeThatFits:CGSizeMake(self.overlayView.bounds.size.width, 30)];
    ppButtonSkipSize.width += 16;
    ppButtonSkipSize.height += 8;

    
    CGFloat settingsBarHeight = (layoutStyle == LayoutStyle_AnimationStart) ? 1 : WCCVC_LD_SettingsBarHeight;
    NSMutableArray *layoutConstraints = [NSMutableArray array];
    NSDictionary *views = @{@"settingsBar":self.settingsBar,
                            @"targetView":self.targetView,
                            @"ppButtonSkip":self.ppButtonSkip};
    NSDictionary *metrics = @{@"settingsBarHeight": @(settingsBarHeight),
                              @"ppButtonSkipMargin": @(WCCVC_LD_UnfixedTargetBoxMargin*2),
                              @"ppButtonSkipWidth": @(ppButtonSkipSize.width),
                              @"ppButtonSkipHeight": @(ppButtonSkipSize.height)};
    
    
    // 水平佈局
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[targetView][settingsBar(settingsBarHeight)]|"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:metrics
                                                                                     views:views]];

    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:[ppButtonSkip(ppButtonSkipWidth)]-(ppButtonSkipMargin)-[settingsBar(settingsBarHeight)]|"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:metrics
                                                                                     views:views]];
    
    
    // 垂直佈局
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|[settingsBar]|"
                                                                                   options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                   metrics:metrics
                                                                                     views:views]];
    
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|[targetView]|"
                                                                                   options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                   metrics:metrics
                                                                                     views:views]];
    
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:[ppButtonSkip(ppButtonSkipHeight)]-(ppButtonSkipMargin)-|"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:metrics
                                                                                     views:views]];

    
    // 其他
    [layoutConstraints addObject: [NSLayoutConstraint constraintWithItem:self.cardDirectionImageView
                                                               attribute:NSLayoutAttributeCenterX
                                                               relatedBy:NSLayoutRelationEqual
                                                                  toItem:self.targetView
                                                               attribute:NSLayoutAttributeCenterX
                                                              multiplier:1.0f
                                                                constant:0.0f]];
    
    [layoutConstraints addObject: [NSLayoutConstraint constraintWithItem:self.cardDirectionImageView
                                                               attribute:NSLayoutAttributeCenterY
                                                               relatedBy:NSLayoutRelationEqual
                                                                  toItem:self.targetView
                                                               attribute:NSLayoutAttributeCenterY
                                                              multiplier:1.0f
                                                                constant:0.0f]];


    //////////////////////////////////////////////////
    // synchronize overlayViewLayoutConstraints
    
    @synchronized(self)
    {
        if ([layoutConstraints count] > 0)
        {
            if(self.overlayViewLayoutConstraints != nil)
            {
                [self.overlayView removeConstraints:self.overlayViewLayoutConstraints];
            }
            
            [self.overlayView addConstraints:layoutConstraints];
            self.overlayViewLayoutConstraints = layoutConstraints;
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance Methods - subview content/status

//================================================================================
//
//================================================================================
- (void)setupCaptureCount:(NSInteger)captureCount
{
    self.captureCount = captureCount;
    
    if(captureCount > 0)
    {
        [self updateCaptureCountLabelWithNumber:self.captureCount];
        
        //////////////////////////////////////////////////
        // convert cancel button to done
        
        if(self.ppButtonCancelDone.tag == ButtonTag_Cancel)
        {
            self.ppButtonCancelDone.tag = ButtonTag_Done;
            [self.ppButtonCancelDone setImageWithName:WCCVC_IFP_DoneButton];
        }
    }
}


//================================================================================
//
//================================================================================
- (void)setupFlashState:(BOOL)enableFlash
{
    [self.ppButtonFlash setSelected:enableFlash];
}


//================================================================================
//
//================================================================================
- (void)setupImageEnhanceState:(BOOL)enableImageEnhance
{
    [self.ppButtonImageEnhance setSelected:enableImageEnhance];
}


//================================================================================
//
//================================================================================
- (void)setupAutoCropState:(BOOL)enableAutoCrop
{
    [self.ppButtonAutoCrop setSelected:enableAutoCrop];
}


//================================================================================
//
//================================================================================
- (void)setupDoubleSideState:(BOOL)enableDoubleSide
{
    self.isDoubleSideMode = enableDoubleSide;
    [self.ppButtonDoubleSide setSelected:enableDoubleSide];
    [self updateCardDirectionImageView];
    
    if(self.isDoubleSideMode == YES && self.captureSide == CaptureSide_Back)
    {
        [self hideSkipButton:NO];
    }
    else
    {
        [self hideSkipButton:YES];
    }
}


//================================================================================
//
//================================================================================
- (void)setupCaptureSide:(WCCaptureViewController_CaptureSide)captureSide
{
    self.captureSide = captureSide;
    
    if(self.isDoubleSideMode == YES && self.captureSide == CaptureSide_Back)
    {
        [self hideSkipButton:NO];
    }
    else
    {
        [self hideSkipButton:YES];
    }
}


//================================================================================
//
//================================================================================
- (void)setupRecogLanguageState:(NSString *)recogLanguage
{
    if([self.recogLanguage isEqualToString:recogLanguage] == NO)
    {
        self.recogLanguage = recogLanguage;
        [self.ppButtonRecogLanguage setTitle:recogLanguage maxWidth:0.0];
    }
}


//================================================================================
//
//================================================================================
- (void)setupAutoDetectionState:(BOOL)enableAutoDetection
{
    self.autoDetectionEnabled = enableAutoDetection;
    self.autoDetectionBeginDate = nil;
    self.ppButtonCapture.hidden = self.autoDetectionEnabled;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance Methods - Flow action methods

//================================================================================
//
//================================================================================
- (void)actionWhenActive
{
    [self logMessageWithFormat:@"%s in", __func__];
    
    
    //////////////////////////////////////////////////
    // 確認基座是否購買
    
    BOOL hasBoughtDock = ([PPSettingsController integerValueWithKey:WCCaptureViewControllr_SettingsKey_HasBoughtDock] == 1);

    if(hasBoughtDock == YES)
    {
        [self startActive];
    }
    else
    {
        __block typeof(self) blockSelf = self;
        
        dispatch_async(dispatch_get_main_queue(), ^{
            
            blockSelf.alertView = [[[PPAlertView alloc] initWithTitle:@""
                                                              message:WCCVC_MLS_NeedToBuyDock
                                                             delegate:blockSelf
                                                    cancelButtonTitle:nil
                                                    otherButtonTitles:nil] autorelease];
            
            // 都要有cancel button
            [blockSelf.alertView addButtonWithTitle:WCCVC_MLS_HasBoughtDock];
            blockSelf.alertView.cancelButtonIndex = [blockSelf.alertView addButtonWithTitle:WCCVC_MLS_Close];
            // 7.3.5 拿掉這邊的promotion
//            [blockSelf.alertView addButtonWithTitle:WCCVC_MLS_MoreDockInfo];
            
            blockSelf.alertView.tag = AlertTag_NeedToByDock;
            [blockSelf.alertView show];
        });
        
        return;
    }

    //////////////////////////////////////////////////

    [self logMessageWithFormat:@"%s out", __func__];
}


//================================================================================
//
//================================================================================
- (void)actionWhenDeactive
{
    [self logMessageWithFormat:@"%s in (operations=%d, isMainThread=%d)", __func__, self.recognitionOperationQueue.operationCount, [NSThread isMainThread]];

    //////////////////////////////////////////////////

    self.isActive = NO;
    self.autoDetectionBeginDate = nil;
    
    while ([self.recognitionOperationQueue operationCount] > 0)
    {
        [NSThread waitWithTimeInterval:0.05];
    }
    
    self.recognitionOperationQueue = nil;
    
    [PPPatternOCRController releasePatternOCRControler];
    
    [self stopDevicePositionCheck];
    
    //////////////////////////////////////////////////

    [self logMessageWithFormat:@"%s out", __func__];
}


//================================================================================
//
//================================================================================
- (void)actionWhenReceivePreviewImage:(UIImage *)previewImage
{
    if (self.isActive == NO)
    {
        [self logMessageWithFormat:@"%s out (not active)", __func__];
        return;
    }
    
    if (previewImage == nil)
    {
//        [self logMessageWithFormat:@"%s out (no preview image)", __func__];
        return;
    }
    
    if (self.autoDetectionEnabled == NO)
    {
//        [self logMessageWithFormat:@"%s out (auto detection disabled)", __func__];
        return;
    }

    if(self.isDevicePositionConfirmed == NO)
    {
//        [self logMessageWithFormat:@"%s out (wrong device position)", __func__];
        return;
    }
    
    if([self.recognitionOperationQueue operationCount] > 0)
    {
//        [self logMessageWithFormat:@"%s out (has running operation)", __func__];
        return;
    }
    
    if ([self.alertView isVisible] == YES)
    {
//        [self logMessageWithFormat:@"%s out (alert is show)", __func__];
        return;
    }

    //////////////////////////////////////////////////
    
    NSInvocationOperation *operation = [[NSInvocationOperation alloc] initWithTarget:self selector:@selector(recognizePreviewImage:) object:previewImage];
    [self.recognitionOperationQueue addOperation:operation];
    [operation release];
}


//================================================================================
//
//================================================================================
- (void)actionWhenGetStillImage
{
    if (self.isDoubleSideMode)
    {
        //////////////////////////////////////////////////
        // 自動/手動更新訊息時機不同

        if(self.autoDetectionEnabled == YES)
        {
            [self updateLastingCaptureInfo];
        }
        
        if(self.captureSide == CaptureSide_Front)
        {
            self.captureSide = CaptureSide_Back;
        }
        else if(self.captureSide == CaptureSide_Back)
        {
            self.captureSide = CaptureSide_Front;
        }
        
        if(self.autoDetectionEnabled == NO)
        {
            [self updateLastingCaptureInfo];
            [self updateCardDirectionImageView];
        }
        
        //////////////////////////////////////////////////

        if([self.delegate respondsToSelector:@selector(captureController:requestUpdateCaptureSideState:)])
        {
            [self.delegate captureController:self requestUpdateCaptureSideState:self.captureSide];
        }
    }
    else
    {
        [self updateLastingCaptureInfo];
    }
}


//================================================================================
//
//================================================================================
- (void)actionWhenSkipBackSide
{
    self.captureSide = CaptureSide_Front;
    [self updateCardDirectionImageView];

    if(self.hasCardInDock == YES)
    {
        [self updateInfoLabelLastingWithText:WCCVC_MLS_RemoveCard];
    }
    else
    {
        [self updateLastingCaptureInfo];
    }
    
    if([self.delegate respondsToSelector:@selector(captureController:requestUpdateCaptureSideState:)])
    {
        [self.delegate captureController:self requestUpdateCaptureSideState:self.captureSide];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods

//===============================================================================
//
//===============================================================================
- (void)startActive
{
    self.isActive = YES;
    self.hasDetectedQRCode = NO;
    self.hasCardInDock = NO;
    
    if(self.recognitionOperationQueue == nil)
    {
        self.recognitionOperationQueue = [[[NSOperationQueue alloc] init] autorelease];
        [self.recognitionOperationQueue setMaxConcurrentOperationCount:1];
    }
    
    //////////////////////////////////////////////////
    
    if(self.autoDetectionEnabled == YES)
    {
        [self startDevicePositionCheck];
    }
    
    [self updateLastingCaptureInfo];
    
    
    //////////////////////////////////////////////////
    // 取得cameraView大小供調整對焦點使用
    
    if(CGSizeEqualToSize(self.cameraViewSize, CGSizeZero) == YES)
    {
        if ([self.delegate respondsToSelector:@selector(captureControllerRequestGetCameraViewSize:)])
        {
            self.cameraViewSize = [self.delegate captureControllerRequestGetCameraViewSize:self];
        }
    }
    
    
    //////////////////////////////////////////////////
    // MARK: [對焦點調整] 進入時先用記錄的QRCode位置對焦一次
    
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_main_queue(), ^{
        
        NSString *string = [PPSettingsController stringValueWithKey:WCCaptureViewControllr_SettingsKey_LastFocusFactor];
        
        if([string length] > 0)
        {
            CGPoint factor = CGPointFromString(string);
            CGPoint focusPoint = CGPointMake(blockSelf.cameraViewSize.width * factor.x,
                                             blockSelf.cameraViewSize.height * factor.y);
            
            //            NSLog(@"!!!!! init focusPoint:%@ !!!!!", NSStringFromCGPoint(focusPoint));
            
            [blockSelf.delegate captureController:blockSelf requestSetFocusPoint:focusPoint];
        }
    });
}


//===============================================================================
//
//===============================================================================
- (void)logMessageWithFormat:(NSString *)format, ...
{
    
#ifdef _DUMP_LOG_
    
    va_list args;
    va_start(args,format);
    
    if([self.delegate respondsToSelector:@selector(captureController:requestLogMessage:)])
    {
        NSString *message = [[[NSString alloc] initWithFormat:format arguments:args] autorelease];
        [self.delegate captureController:self requestLogMessage:message];
    }

    va_end(args);
    
#endif
    
}


//================================================================================
//
//================================================================================
- (void)recognizePreviewImage:(UIImage *)previewImage
{
    //////////////////////////////////////////////////
    // recognize image in background thread
    
    if (self.isActive == NO)
    {
        return;
    }
    
    __block CaptureState captureState = [PPPatternOCRController captureStateWithImage:previewImage];
    
    
    //////////////////////////////////////////////////
    // handle state in main thread

    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_main_queue(), ^{
        
        if (blockSelf.isActive == NO)
        {
            return;
        }

        //////////////////////////////////////////////////

        switch (captureState)
        {
            case CaptureState_ReadyToCapture:
            {
                [blockSelf resetFocusAdjust];
                
                
                //////////////////////////////////////////////////
                // 雖然有找到但有可能是模糊的，所以要QRCode位置重新對焦，確保影像清晰
                
                CGPoint QRCodePosition = [PPPatternOCRController focusPointForQRCodePosition];
                
                // MARK: [Debug用] 持續在console中顯示QRCode pattern位置
                //            NSLog(@"QRCodePosition:%@", NSStringFromCGPoint(QRCodePosition));
                //            [PPPatternOCRController releasePatternOCRControler];
                
                
                if (CGPointEqualToPoint(CGPointZero, QRCodePosition) == NO &&
                    CGSizeEqualToSize(CGSizeZero, blockSelf.cameraViewSize) == NO)
                {
                    if ([blockSelf.delegate respondsToSelector:@selector(captureController:requestSetFocusPoint:)])
                    {
                        CGPoint focusPoint = CGPointMake(ceilf(blockSelf.cameraViewSize.width*QRCodePosition.x),
                                                         ceilf(blockSelf.cameraViewSize.height*QRCodePosition.y));
                        
                        [blockSelf logMessageWithFormat:@"%s (set focus : %@)", __func__, NSStringFromCGPoint(focusPoint)];
                        [blockSelf.delegate captureController:blockSelf requestSetFocusPoint:focusPoint];
                    }
                    
                    
                    //////////////////////////////////////////////////
                    // MARK: [對焦點調整] 記錄核心判斷的QRCode位置
                    
                    NSString *string = NSStringFromCGPoint(QRCodePosition);
                    [PPSettingsController setStringValue:string withKey:WCCaptureViewControllr_SettingsKey_LastFocusFactor];
                }
                
                // 停止偵測，提示放上名片
                blockSelf.autoDetectionBeginDate = nil;
                blockSelf.hasDetectedQRCode = YES;
                blockSelf.hasCardInDock = NO;
                [blockSelf updateLastingCaptureInfo];
                [blockSelf updateCardDirectionImageView];
                
                break;
            }
                
            case CaptureState_CaptureNow:
            {
                [blockSelf resetFocusAdjust];
                
                if ([blockSelf.delegate respondsToSelector:@selector(captureControllerRequestCapture:)])
                {
                    [blockSelf logMessageWithFormat:@"%s (request capture)", __func__];
                    [blockSelf.delegate captureControllerRequestCapture:blockSelf];
                }
                
                blockSelf.hasCardInDock = YES;
                
                break;
            }
                
            case CaptureState_NoPatternAlarm:
            {
                [blockSelf logMessageWithFormat:@"%s (CaptureState_NoPatternAlarm)", __func__];
                
                //////////////////////////////////////////////////
                // 有偵測過QRCode就不用處理了
                
                if(blockSelf.hasDetectedQRCode == YES)
                {
                    blockSelf.autoDetectionBeginDate = nil;
                    break;
                }
                
                //////////////////////////////////////////////////
                
                if(blockSelf.autoDetectionBeginDate == nil)
                {
                    blockSelf.autoDetectionBeginDate = [NSDate date];
                }
                else
                {
                    if(ABS([blockSelf.autoDetectionBeginDate timeIntervalSinceNow]) > CVCDockingCaptureController_AutoDetectionWarningInterval)
                    {
                        [self logMessageWithFormat:@"%s !! show alert (isActive=%d) !!", __func__, blockSelf.isActive];
                        
                        blockSelf.alertView = [[[PPAlertView alloc] initWithTitle:@""
                                                                          message:WCCVC_MLS_SwitchToManual
                                                                         delegate:blockSelf
                                                                cancelButtonTitle:WCCVC_MLS_Cancel
                                                                otherButtonTitles:WCCVC_MLS_ManualMode, nil] autorelease];
                        [blockSelf.alertView show];
                    }
                    else
                    {
                        [blockSelf proceedFocusAdjust];
                    }
                }
                
                break;
            }
                
            case CaptureState_Nothing:
            {
                break;
            }
                
            default:
            {
                // should not happen
                break;
            }
        }
    });
}


//================================================================================
//
//================================================================================
- (void)resetFocusAdjust
{
    self.focusAdjustBeginDate = nil;
}


//================================================================================
// 變更對焦點（目的是讓核心可以辨識到QRCode位置，若辨識成功會用回傳位置重新對焦一次）
//================================================================================
- (void)proceedFocusAdjust
{
    //////////////////////////////////////////////////
    // MARK: [對焦點調整] 切換對焦點讓核心可以找到QRCode pattern
    
    if(self.hasCardInDock==NO &&
       (self.focusAdjustBeginDate==nil || ABS([self.focusAdjustBeginDate timeIntervalSinceNow])>3.0))
    {
        static int s_focusIndex = 0;
        int candidateCount = 5;
        CGFloat focusPointYFactors[] = {0.5, 0.9, 0.7, 0.8, 0.6};
        CGPoint focusPoint = CGPointMake(self.cameraViewSize.width * 0.5,
                                         self.cameraViewSize.height * focusPointYFactors[s_focusIndex]);
        
//        NSLog(@"***** adjust focusPoint:%@ *****", NSStringFromCGPoint(focusPoint));        
        [self.delegate captureController:self requestSetFocusPoint:focusPoint];
        
        if(++s_focusIndex >= candidateCount)
        {
            s_focusIndex = 0;
        }
        
        self.focusAdjustBeginDate = [NSDate date];
    }
}


//================================================================================
//
//================================================================================
- (void)updateCardDirectionImageView
{
    UIImage *cardDirectionImage = nil;
    
    if (self.isDoubleSideMode)
    {
        if(self.captureSide == CaptureSide_Front)
        {
            cardDirectionImage = [UIImage imageWithName:WCCVC_IFP_CardDirectionPortraitFrontSide];
        }
        else if(self.captureSide == CaptureSide_Back)
        {
            cardDirectionImage = [UIImage imageWithName:WCCVC_IFP_CardDirectionPortraitBackSide];
        }
    }
    else
    {
        cardDirectionImage = [UIImage imageWithName:WCCVC_IFP_CardDirectionPortrait];
    }
    
    //////////////////////////////////////////////////
    
    if(cardDirectionImage != nil)
    {
        cardDirectionImage = [UIImage imageWithCGImage:cardDirectionImage.CGImage
                                                 scale:WCCVC_LD_ImageScale
                                           orientation:cardDirectionImage.imageOrientation];
        
        //////////////////////////////////////////////////
        
        [self.cardDirectionImageView setImage:cardDirectionImage];
    }
}


//================================================================================
//
//================================================================================
- (void)hideSkipButton:(BOOL)hide
{
    self.ppButtonSkip.hidden = hide;
}


//================================================================================
//
//================================================================================
- (void)updateLastingCaptureInfo
{
    NSString *message = nil;
    
    if(self.autoDetectionEnabled == YES && self.hasDetectedQRCode == NO)
    {
        message = WCCVC_MLS_DetectingQRCode;
    }
    else
    {
        if (self.isDoubleSideMode)
        {
            if(self.hasCardInDock == YES)
            {
                if(self.captureSide == CaptureSide_Front)
                {
                    message = WCCVC_MLS_DoubleSideRemoveFront;
                }
                else if(self.captureSide == CaptureSide_Back)
                {
                    message = WCCVC_MLS_DoubleSideRemoveBack;
                }
            }
            else
            {
                if(self.autoDetectionEnabled == YES)
                {
                    if(self.captureSide == CaptureSide_Front)
                    {
                        message = WCCVC_MLS_AutoDoubleSidePutFront;
                    }
                    else if(self.captureSide == CaptureSide_Back)
                    {
                        message = WCCVC_MLS_AutoDoubleSidePutBack;
                    }
                }
                else
                {
                    if(self.captureSide == CaptureSide_Front)
                    {
                        message = WCCVC_MLS_ManualDoubleSidePutFront;
                    }
                    else if(self.captureSide == CaptureSide_Back)
                    {
                        message = WCCVC_MLS_ManualDoubleSidePutBack;
                    }
                }
            }            
        }
        else
        {
            if(self.hasCardInDock == YES)
            {
                message = WCCVC_MLS_SingleSideRemoveFront;
            }
            else
            {
                if(self.autoDetectionEnabled == YES)
                {
                    message = WCCVC_MLS_AutoSingleSidePutFront;
                }
                else
                {
                    message = WCCVC_MLS_ManualSingleSidePutFront;
                }
            }
        }
    }
    
    [self updateInfoLabelLastingWithText:message];
}


//================================================================================
// 確認手機位置是否可以拍照
//================================================================================
- (void)startDevicePositionCheck
{
    // iOS7接收motion通知會卡卡的，所以不做位置判斷。
    if([[[UIDevice currentDevice] systemVersion] floatValue] < 8.0)
    {
        self.isDevicePositionConfirmed = YES;
        return;
    }
    
    self.isDevicePositionConfirmed = NO;

    //////////////////////////////////////////////////
    
    if(self.motionManager == nil)
    {
        self.motionManager = [[[CMMotionManager alloc] init] autorelease];
        [self.motionManager setAccelerometerUpdateInterval:CVCDockingCaptureController_MotionDetectionInterval];
    }
    
    if(self.motionOperationQueue == nil)
    {
        self.motionOperationQueue = [[[NSOperationQueue alloc] init] autorelease];
    }
    
    //////////////////////////////////////////////////
    
    __block typeof(self) blockSelf = self;
    
    [self.motionManager startAccelerometerUpdatesToQueue:self.motionOperationQueue
                                             withHandler:^(CMAccelerometerData *data, NSError *error)
    {
        if(blockSelf.isActive == NO)
        {
            return;
        }
        
        //////////////////////////////////////////////////

        static NSInteger s_stableCounter = 0;
        static CMAcceleration s_preAcceleration;
        float shakingDiff = 0;

        shakingDiff = ABS(s_preAcceleration.x-data.acceleration.x)+ABS(s_preAcceleration.y-data.acceleration.y)+ABS(s_preAcceleration.z-data.acceleration.z);

        // update stable counter
        if(shakingDiff >= 0.3)
        {
            s_stableCounter = 0;
        }
        else
        {
            s_stableCounter++;
        }

        // !! 確認的條件
        // 1. 穩定時間保持1秒
        // 2. z軸位置在-0.2到-0.7之間（放在dock上的z值大概是-0.45左右）
        blockSelf.isDevicePositionConfirmed = (s_stableCounter >= 5 &&
                                               data.acceleration.z < -0.2 &&
                                               data.acceleration.z > -0.7);
        
        //test
//        NSLog(@"%s isDevicePositionConfirmed:%d (%ld, %@)", __func__, blockSelf.isDevicePositionConfirmed, (long)s_stableCounter, data);

        // keep acceleration for next compare
        s_preAcceleration = data.acceleration;
        
     }];
}


//================================================================================
//
//================================================================================
- (void)stopDevicePositionCheck
{
    [self.motionManager stopAccelerometerUpdates];
    
    while ([self.motionOperationQueue operationCount] > 0)
    {
        [NSThread waitWithTimeInterval:0.05];
    }
    
    self.motionOperationQueue = nil;
    self.motionManager = nil;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPButtonDelegate methods

//================================================================================
//
//================================================================================
- (void)ppButton:(PPButton *)ppButton controlEvent:(UIControlEvents)controlEvent
{    
    switch (ppButton.tag)
    {
        case ButtonTag_Flash:
        {
            [self setupFlashState:!ppButton.selected];
            
            if(ppButton.selected == YES)
            {
                [self updateInfoLabelTemporaryWithText:WCCVC_MLS_FlashOn];
            }
            else
            {
                [self updateInfoLabelTemporaryWithText:WCCVC_MLS_FlashOff];
            }
            
            if ([self.delegate respondsToSelector:@selector(captureController:requestUpdateFlashState:)])
            {
                [self.delegate captureController:self requestUpdateFlashState:[ppButton isSelected]];
            }
            break;
        }
            
        case ButtonTag_ImageEnhance:
        {
            [self setupImageEnhanceState:!ppButton.selected];
            
            if(ppButton.selected == YES)
            {
                [self updateInfoLabelTemporaryWithText:WCCVC_MLS_ImageEnhanceOn];
            }
            else
            {
                [self updateInfoLabelTemporaryWithText:WCCVC_MLS_ImageEnhanceOff];
            }
            
            if ([self.delegate respondsToSelector:@selector(captureController:requestUpdateImageEnhanceState:)])
            {
                [self.delegate captureController:self requestUpdateImageEnhanceState:[ppButton isSelected]];
            }
            
            break;
        }
            
        case ButtonTag_AutoCrop:
        {
            [self setupAutoCropState:!ppButton.selected];
            
            if(ppButton.selected == YES)
            {
                [self updateInfoLabelTemporaryWithText:WCCVC_MLS_AutoCropOn];
            }
            else
            {
                [self updateInfoLabelTemporaryWithText:WCCVC_MLS_AutoCropOff];
            }
            
            if ([self.delegate respondsToSelector:@selector(captureController:requestUpdateAutoCropState:)])
            {
                [self.delegate captureController:self requestUpdateAutoCropState:[ppButton isSelected]];
            }
            
            break;
        }
            
        case ButtonTag_DoubleSide:
        {
            // 雙面模式只拍了正面，切換到單面模式拍照前要先送出skip。
            if(self.isDoubleSideMode == YES && self.captureSide == CaptureSide_Back)
            {
                if ([self.delegate respondsToSelector:@selector(captureControllerRequestSkip:)])
                {
                    [self.delegate captureControllerRequestSkip:self];
                }
            }
            
            //////////////////////////////////////////////////
            
            [self setupDoubleSideState:!self.isDoubleSideMode];
            
            [self updateLastingCaptureInfo];
            
            if(self.isDoubleSideMode == YES)
            {
                [self updateInfoLabelTemporaryWithText:WCCVC_MLS_DoubleSide];
            }
            else
            {
                [self updateInfoLabelTemporaryWithText:WCCVC_MLS_SingleSide];
            }
            
            if ([self.delegate respondsToSelector:@selector(captureController:requestUpdateDoubleSideState:)])
            {
                [self.delegate captureController:self requestUpdateDoubleSideState:[ppButton isSelected]];
            }
            
            break;
        }
            
        case ButtonTag_Guide:
        {
            [self manuallyShowGuide];
            break;
        }
            
        case ButtonTag_RecogLanguage:
        {
            if ([self.delegate respondsToSelector:@selector(captureControllerRequestSelectRecogLanguage:)])
            {
                [self.delegate captureControllerRequestSelectRecogLanguage:self];
            }
            
            break;
        }
            
        case ButtonTag_Capture:
        {
            if ([self.delegate respondsToSelector:@selector(captureControllerRequestCapture:)])
            {
                [self.delegate captureControllerRequestCapture:self];
            }
            break;
        }
            
        case ButtonTag_Cancel:
        {
            if ([self.delegate respondsToSelector:@selector(captureControllerRequestCancel:)])
            {
                [self.delegate captureControllerRequestCancel:self];
            }
            break;
        }
            
        case ButtonTag_Done:
        {
            if ([self.delegate respondsToSelector:@selector(captureControllerRequestDone:)])
            {
                [self.delegate captureControllerRequestDone:self];
            }
            break;
        }
            
        case ButtonTag_Skip:
        {
            if ([self.delegate respondsToSelector:@selector(captureControllerRequestSkip:)])
            {
                [self.delegate captureControllerRequestSkip:self];
            }
            break;
        }
            
        default:
        {
            break;
        }
    }
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - UIAlertViewDelegate methods

//================================================================================
//
//================================================================================
- (void)alertView:(UIAlertView *)alertView didDismissWithButtonIndex:(NSInteger)buttonIndex
{
    switch (alertView.tag)
    {
        case AlertTag_NeedToByDock:
        {
            if([[alertView buttonTitleAtIndex:buttonIndex] isEqualToString:WCCVC_MLS_HasBoughtDock] == YES)
            {
                [PPSettingsController setIntegerValue:1 withKey:WCCaptureViewControllr_SettingsKey_HasBoughtDock];
                [self startActive];
            }
            else if([[alertView buttonTitleAtIndex:buttonIndex] isEqualToString:WCCVC_MLS_MoreDockInfo] == YES)
            {
                if([self.delegate respondsToSelector:@selector(captureControllerRequestShowDockPromotion:)])
                {
                    [self.delegate captureControllerRequestShowDockPromotion:self];
                }
                
                if([self.delegate respondsToSelector:@selector(captureController:requestSwitchToMode:)])
                {
                    [self.delegate captureController:self requestSwitchToMode:CaptureMode_Manual];
                }
            }
            else
            {
                if([self.delegate respondsToSelector:@selector(captureController:requestSwitchToMode:)])
                {
                    [self.delegate captureController:self requestSwitchToMode:CaptureMode_Manual];
                }            
            }

            break;
        }
            
        default:
        {
            if([[alertView buttonTitleAtIndex:buttonIndex] isEqualToString:WCCVC_MLS_ManualMode] == YES)
            {
                self.autoDetectionEnabled = NO;
                self.autoDetectionBeginDate = nil;
                self.ppButtonCapture.hidden = NO;
                [self updateLastingCaptureInfo];
                [self stopDevicePositionCheck];
            }
            else
            {
                self.autoDetectionBeginDate = [NSDate date];
            }
            
            break;
        }
    }
}

@end
