//
//  PPWebSocketController.m
//  Example
//
//  Created by AndrewLai on 2016/3/15.
//  Copyright © 2016年 penpower. All rights reserved.
//

#import "PPWebSocketController.h"
#import "PPNetworkReachabilityController.h"
#import "NSError+Custom.h"
#import "NSTimer+Additions.h"

#import <SocketRocket/SRWebSocket.h>

////////////////////////////////////////////////////////////////////////////////////////////////////

static NSString *PPWebSocketController_DefaultURLString = nil;
static NSInteger PPWebSocketKeepAliveTimeInterval = 20;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPWebSocketController Extension

@interface PPWebSocketController () <SRWebSocketDelegate>

@property (atomic, assign) id<PPWebSocketControllerDelegate> delegate;

@property (atomic, retain) SRWebSocket *webSocket;
@property (atomic, retain) PPNetworkReachabilityController *ppNetworkReachabilityController;
@property (atomic, retain) NSTimer *keepAliveTimer;
@end





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPWebSocketController Implementation

@implementation PPWebSocketController





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Singleton Creating Object

//================================================================================
//
//================================================================================
+ (instancetype)sharedInstance
{
//    NSLog(@"%s", __FUNCTION__);
    
    //////////////////////////////////////////////////
    
    static PPWebSocketController *sharedInstance = nil;
    static dispatch_once_t onceToken;
    
    dispatch_once(&onceToken, ^{
        sharedInstance = [[self alloc] init];
        sharedInstance.keepAliveTimeInterval = PPWebSocketKeepAliveTimeInterval;
    });
    
    return sharedInstance;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Deallocating Objects

//================================================================================
//
//================================================================================
- (id)init
{
    if((self = [super init]))
    {
//        NSLog(@"%s", __FUNCTION__);
        
        
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
//    NSLog(@"%s", __FUNCTION__);
    
    _delegate = nil;
    
    //////////////////////////////////////////////////
    
    [self.ppNetworkReachabilityController stopNotifier];
    self.ppNetworkReachabilityController = nil;

    //////////////////////////////////////////////////
    
    [_webSocket release];
    _webSocket = nil;
    
    [_keepAliveTimer invalidate];
    _keepAliveTimer = nil;
    //////////////////////////////////////////////////
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Internet Monitor

//================================================================================
//
//================================================================================
- (void)reachabilityChanged:(NSNotification *)notify
{
//    NSLog(@"%s", __FUNCTION__);
    
    //////////////////////////////////////////////////
    
    PPNetworkReachabilityController *networkReachabilityController = [notify object];
    if ([networkReachabilityController status]==PPNetworkReachabilityControllerStatus_ReachableViaWiFi ||
        [networkReachabilityController status]==PPNetworkReachabilityControllerStatus_ReachableViaWWAN)
    {
        [self connect];
    }
}


//================================================================================
//
//================================================================================
- (void)registerForNetworkReachabilityNotifications
{
//    NSLog(@"%s", __FUNCTION__);
    
    //////////////////////////////////////////////////
    
    //  Register notification for Network status change
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(reachabilityChanged:) name:PPNetworkReachabilityControllerChangedNotification object:nil];
    
    //  Initialize the listener for network status
    self.ppNetworkReachabilityController = [PPNetworkReachabilityController networkReachabilityControllerForInternetConnection];
    [self.ppNetworkReachabilityController startNotifier];
}


//================================================================================
//
//================================================================================
- (void)unsubscribeFromNetworkReachabilityNotifications
{
//    NSLog(@"%s", __FUNCTION__);
    
    //////////////////////////////////////////////////
    
    //  Unregister notification for Network status change
    [[NSNotificationCenter defaultCenter] removeObserver:self name:PPNetworkReachabilityControllerChangedNotification object:nil];
    
    // Remove the listener for network status
    [self.ppNetworkReachabilityController stopNotifier];
    self.ppNetworkReachabilityController = nil;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Method

//================================================================================
//
//================================================================================
- (SRReadyState)readyState
{
//    NSLog(@"%s", __FUNCTION__);
    
    //////////////////////////////////////////////////
    
    return [self.webSocket readyState];
}


//================================================================================
//!! 因為無論是中斷連線或網路中斷皆會使readyState改為SR_CLOSED，而導致無法再次Connect
//!! 目前官方範例即是重新創建！並且不可連線成功前，即將webSocket記錄於property，會導致流程衝突而Crash！
//================================================================================
- (void)connect
{
//    NSLog(@"%s", __FUNCTION__);
    
    //////////////////////////////////////////////////
    
    self.webSocket.delegate = nil;
    self.webSocket = nil;
    
    //////////////////////////////////////////////////
    
    if ([PPWebSocketController_DefaultURLString length]>0)
    {
        SRWebSocket *newWebSocket = [[SRWebSocket alloc] initWithURL:[NSURL URLWithString:PPWebSocketController_DefaultURLString]
                                                           protocols:nil
                                      allowsUntrustedSSLCertificates:YES];
        if (newWebSocket!=nil)
        {
            newWebSocket.delegate = self;
            
            [newWebSocket open];
            [newWebSocket release];
        }
    }
}


//================================================================================
//
//================================================================================
- (void)disconnect
{
//    NSLog(@"%s", __FUNCTION__);
    
    //////////////////////////////////////////////////
    
    [self.webSocket close];
}


//================================================================================
//
//================================================================================
- (void)sendData:(id)data
{
//    NSLog(@"%s", __FUNCTION__);
    
    //////////////////////////////////////////////////
    
    [self.webSocket send:data];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Timer method

//================================================================================
//
//================================================================================
- (void)sendPingMessage
{
//    NSLog(@"%s",__func__);
    if([self readyState]==SR_OPEN)
    {
        [self.webSocket sendPing:nil];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Method

//================================================================================
//
//================================================================================
+ (void)setDefaultURLString:(NSString *)urlString
{
//    NSLog(@"%s", __FUNCTION__);
    
    //////////////////////////////////////////////////
    [urlString retain];
    [PPWebSocketController_DefaultURLString release];
    PPWebSocketController_DefaultURLString = urlString;
}

//================================================================================
//
//================================================================================
+ (void)setDelegate:(id<PPWebSocketControllerDelegate>)delegate
{
//    NSLog(@"%s", __FUNCTION__);
    
    //////////////////////////////////////////////////
    
    [[PPWebSocketController sharedInstance] setDelegate:delegate];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Action Class Method

//================================================================================
//
//================================================================================
+ (void)connect
{
//    NSLog(@"%s", __FUNCTION__);
    
    //////////////////////////////////////////////////
    
    [[PPWebSocketController sharedInstance] connect];
}


//================================================================================
//
//================================================================================
+ (void)disconnect
{
//    NSLog(@"%s", __FUNCTION__);
    
    //////////////////////////////////////////////////

    [[PPWebSocketController sharedInstance].keepAliveTimer invalidate];
    [PPWebSocketController sharedInstance].keepAliveTimer = nil;
    
    //////////////////////////////////////////////////
    
    [[PPWebSocketController sharedInstance] disconnect];
}


//================================================================================
//
//================================================================================
+ (BOOL)sendData:(id)data error:(NSError **)error
{
//    NSLog(@"%s", __FUNCTION__);

    BOOL result = NO;
    //////////////////////////////////////////////////
    
    if ([[PPWebSocketController sharedInstance] readyState]==SR_OPEN)
    {
        [[PPWebSocketController sharedInstance] sendData:data];
        
        result = YES;
    }
    else
    {
        if(error!=nil)
        {
            
            *error = PPErrorMake(1, @"Warning: Invalid State: Cannot call send: Until connection is open!", nil);
        }
    }
    
    return result;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - SRWebSocketDelegate

//================================================================================
//
//================================================================================
- (void)webSocketDidOpen:(SRWebSocket *)webSocket
{
//    NSLog(@"%s", __FUNCTION__);
//    NSLog(@"webSocket: %@", webSocket);
    
    //////////////////////////////////////////////////
    
    [self unsubscribeFromNetworkReachabilityNotifications];
    
    self.webSocket = webSocket;
    
    //////////////////////////////////////////////////
    
    if ([self.delegate respondsToSelector:@selector(webSocketControllerDidConnectedServer:)]==YES)
    {
        [self.delegate webSocketControllerDidConnectedServer:self];
    }
    
    //////////////////////////////////////////////////

    // 連線成功啟動 timer keep alive
    [self.keepAliveTimer invalidate];
    
    __block typeof(self) blockSelf = self;
    
    //元件內部會協助判斷改使用mac 10.12/iOS 10的method 或 該版號以下的
    self.keepAliveTimer = [NSTimer pp_scheduledTimerWithTimeInterval:self.keepAliveTimeInterval
                                                             repeats:YES
                                                               block:^(NSTimer *timer) {
                                                                   [blockSelf sendPingMessage];
                                                               }];
    [self.keepAliveTimer fire];
}


//================================================================================
//
//================================================================================
- (void)webSocket:(SRWebSocket *)webSocket didReceiveMessage:(id)message
{
//    NSLog(@"%s", __FUNCTION__);
//    NSLog(@"webSocket: %@", webSocket);
//    NSLog(@"message: %@", message);
    
    //////////////////////////////////////////////////
    
    if ([self.delegate respondsToSelector:@selector(webSocketController:didReceiveMessage:)]==YES)
    {
        [self.delegate webSocketController:self didReceiveMessage:message];
    }
}


//================================================================================
//
//================================================================================
- (void)webSocket:(SRWebSocket *)webSocket didFailWithError:(NSError *)error
{
//    NSLog(@"%s", __FUNCTION__);
//    NSLog(@"webSocket: %@", webSocket);
//    NSLog(@"error: %@", error);
    
    // !! 連線失敗也要記錄，不然會無法正常disconnect
    self.webSocket = webSocket;

    //////////////////////////////////////////////////
    
    if([self.delegate respondsToSelector:@selector(webSocketController:didFailWithError:)]==YES)
    {
        [self.delegate webSocketController:self didFailWithError:error];
    }
    else
    {
        do
        {
            if([error.domain isEqualToString:SRWebSocketErrorDomain]==YES &&
               error.code==2132)
            {
                break;
            }
            
            //////////////////////////////////////////////////
          
            // 重新連線
            
            if ([PPNetworkReachabilityController checkForInternetConnection]==YES)
            {
                [self connect];
            }
            else
            {
                [self registerForNetworkReachabilityNotifications];
            }
        }
        while (0);
    }
}


//================================================================================
//
//================================================================================
- (void)webSocket:(SRWebSocket *)webSocket didCloseWithCode:(NSInteger)code reason:(NSString *)reason wasClean:(BOOL)wasClean
{
//    NSLog(@"%s", __FUNCTION__);
//    NSLog(@"webSocket: %@", webSocket);
//    NSLog(@"code: %ld", (long)code);
//    NSLog(@"reason: %@", reason);
//    NSLog(@"wasClean: %@", wasClean?@"YES":@"NO");
    
    //////////////////////////////////////////////////

    if([self.delegate respondsToSelector:@selector(webSocketController:didCloseWithCode:reason:wasClean:)]==YES)
    {
        [self.delegate webSocketController:self didCloseWithCode:code reason:reason wasClean:wasClean];
    }
}


//================================================================================
//
//================================================================================
- (void)webSocket:(SRWebSocket *)webSocket didReceivePong:(NSData *)pongPayload
{
//    NSLog(@"%s", __FUNCTION__);
//    NSLog(@"webSocket: %@", webSocket);
//    NSLog(@"pongPayload: %@", pongPayload);
    
    //////////////////////////////////////////////////

    if([self.delegate respondsToSelector:@selector(webSocket:didReceivePong:)]==YES)
    {
        [self.delegate webSocketController:self didReceivePong:pongPayload];
    }
}

@end
