//
//  PPSyncActionController.m
//  
//  MARK:特殊問題紀錄
//  -------------------------------------------------------------------------
//  search #groupID對應錯誤
//         #不知道作用的code
//

#import "PPSyncActionController.h"
#import "NSThread+Additions.h"
#import "NSString+Additions.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

#define PPSyncActionControllerRescheduleLimit                       3

/// 同步流程完成時間
#define PPSyncActionController_Key_LastSyncCompleteStartDate        @"LastSyncCompleteStartDate"
#define PPSyncActionController_Key_LastSyncCompleteStartDateOffset  (15*60)

/// 同步動作完成時間
#define PPSyncActionController_Key_LastSyncActionSuccessDate        @"LastSyncActionSuccessDate"



////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPSyncActionController()

@interface PPSyncActionController()
@property(assign)                       BOOL                cancel;
@property(assign)                       BOOL                executing;
@property(assign)                       BOOL                suspended;
@property(nonatomic,retain)             NSDate              *startDate;
@property(nonatomic,readonly)           NSMutableArray      *syncActionModels;
@property(nonatomic,readonly)           NSMutableArray      *forceUpdateActionModels;
@property(nonatomic,readwrite,assign)   NSUInteger          currentSyncActionIndex;
@property(nonatomic,readwrite,assign)   NSUInteger          localCardCount;
@property(nonatomic,readwrite,assign)   NSUInteger          localGroupCount;
@property(nonatomic,readwrite,assign)   NSUInteger          remoteCardCount;
@property(nonatomic,readwrite,assign)   NSUInteger          remoteGroupCount;
@property(nonatomic,readwrite,assign)   NSUInteger          totalSyncActionCount;
@property(nonatomic,readwrite,retain)   PPSyncActionModel   *currentSyncActionModel;

@property(atomic,readwrite,assign)      PPSyncActionStep    step;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPSyncActionController

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Synthesize

@synthesize
delegate                    = delegate_,
cancel                      = cancel_,
executing                   = executing_,
suspended                   = suspended_,
startDate                   = startDate_,
syncActionModels            = syncActionModels_,
currentSyncActionIndex      = currentSyncActionIndex_,
localCardCount              = localCardCount_,
localGroupCount             = localGroupCount_,
remoteCardCount             = remoteCardCount_,
remoteGroupCount            = remoteGroupCount_,
totalSyncActionCount        = totalSyncActionCount_,
syncCardVersion             = syncCardVersion_,
syncGroupVersion            = syncGroupVersion_,
conflict                    = conflict_,
step                        = step_,
currentSyncActionModel      = currentSyncActionModel_,
localSyncDataController     = localSyncDataController_,
remoteSyncDataController    = remoteSyncDataController_;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Deallocating Objects

//================================================================================
//
//================================================================================
- (id)initWithFileName:(NSString *)fileName atPath:(NSString *)path error:(NSError **)error localSyncDataController:(PPSyncDataController *)localSyncDataController remoteSyncDataController:(PPSyncDataController *)remoteSyncDataController
{
    id object = nil;
    
    //////////////////////////////////////////////////
    
    NSError *originError = ((error!=nil)?*error:nil);
    NSError *returnError = originError;
    
    //////////////////////////////////////////////////
    
    do
    {
        self = [super initWithFileName:fileName atPath:path error:&returnError];
        if(self==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        //預設的Version
        
        self.syncCardVersion    = 1;
        self.syncGroupVersion   = 1;
        
        //////////////////////////////////////////////////
        
        localSyncDataController_ = [localSyncDataController retain];
        if(localSyncDataController_==nil)
        {
            returnError = PPErrorOperationFailed(returnError);
            break;
        }
        
        localSyncDataController_.delegate = self;
        
        //////////////////////////////////////////////////
        
        remoteSyncDataController_ = [remoteSyncDataController retain];
        if(remoteSyncDataController_==nil)
        {
            returnError = PPErrorOperationFailed(returnError);
            break;
        }
        
        remoteSyncDataController_.delegate = self;
        
        //////////////////////////////////////////////////
        
        syncActionModels_ = [[NSMutableArray alloc] init];
        if(syncActionModels_==nil)
        {
            returnError = PPErrorOperationFailed(returnError);
            break;
        }
        
        _forceUpdateActionModels = [[NSMutableArray alloc] init];
        if(_forceUpdateActionModels==nil)
        {
            returnError = PPErrorOperationFailed(returnError);
            break;
        }
        //////////////////////////////////////////////////
        
        object = self;
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    if(object==nil)
    {
        [self release];
    }
    
    //////////////////////////////////////////////////
    
    return object;
}

//================================================================================
//
//================================================================================
- (void)dealloc 
{
    self.delegate = nil;
    
    //////////////////////////////////////////////////
    
    [startDate_ release];
    startDate_ = nil;
    
    //////////////////////////////////////////////////
    
    [currentSyncActionModel_ release];
    currentSyncActionModel_ = nil;

    //////////////////////////////////////////////////
    
    [_forceUpdateActionModels release];
    _forceUpdateActionModels = nil;
    
    //////////////////////////////////////////////////
    
    [syncActionModels_ release];
    syncActionModels_ = nil;
    
    //////////////////////////////////////////////////
    
    [localSyncDataController_ release];
    localSyncDataController_ = nil;

    //////////////////////////////////////////////////
    
    [remoteSyncDataController_ release];
    remoteSyncDataController_ = nil;
    
    //////////////////////////////////////////////////
    
	[super dealloc];
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Methods

//================================================================================
//
//================================================================================
- (BOOL)isExecuting
{
    return self.executing;
}

//================================================================================
//
//================================================================================
- (BOOL)isSuspended
{
    return self.suspended;
}

//================================================================================
//
//================================================================================
- (void)cancelActions
{
    @autoreleasepool
    {
        [self dumpLogMessageWithFormat:@"%s in", __FUNCTION__];
        
        //////////////////////////////////////////////////
        
        self.cancel = YES;
        
        [self.localSyncDataController cancel];
        [self.remoteSyncDataController cancel];
        
        //////////////////////////////////////////////////
        
        [self dumpLogMessageWithFormat:@"%s out", __FUNCTION__];
    }
}

//================================================================================
//
//================================================================================
- (BOOL)cleanActionsWithError:(NSError **)error
{
    NSError     *originError    = ((error!=nil)?*error:nil);
    NSError     *returnError    = originError;
    
    //////////////////////////////////////////////////
    
    @autoreleasepool
    {
        [self dumpLogMessageWithFormat:@"%s in", __FUNCTION__];
        
        //////////////////////////////////////////////////
        
        do
        {
            //目前狀態(沒有在執行),才可以清除
            if(self.executing==YES)
            {
                returnError = PPErrorOperationFailed(returnError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            [self.syncActionModels removeAllObjects];
            
            self.localCardCount         = 0;
            self.localGroupCount        = 0;
            self.remoteCardCount        = 0;
            self.remoteGroupCount       = 0;
            self.currentSyncActionIndex = 0;
            self.totalSyncActionCount   = 0;
            self.suspended              = NO;
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        [self dumpLogMessageWithFormat:@"%s out returnError=%@", __FUNCTION__, returnError];
        
        //////////////////////////////////////////////////
        
        [returnError retain];
    }
    
    [returnError autorelease];
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return (returnError==originError);
}

//================================================================================
//
//================================================================================
- (BOOL)resumeActionsWithError:(NSError **)error
{
    NSError     *originError    = ((error!=nil)?*error:nil);
    NSError     *returnError    = originError;
    
    //////////////////////////////////////////////////
    
    @autoreleasepool
    {
        [self dumpLogMessageWithFormat:@"%s in", __FUNCTION__];
        
        //////////////////////////////////////////////////
        
        do
        {
            //目前狀態(停止執行&&已暫停&&有未做完的syncActionModels),才可以重新執行
            if(self.executing==YES || self.suspended==NO || [self.syncActionModels count]+[self.forceUpdateActionModels count]==0)
            {
                returnError = PPErrorOperationFailed(returnError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            self.executing  = YES;
            self.suspended  = NO;
            self.cancel     = NO;
            
            //////////////////////////////////////////////////

            [self executeActionsWithError:&returnError];
            
            //////////////////////////////////////////////////
            
            if(self.cancel==YES)
            {
                if(returnError==nil || [returnError.domain isEqualToString:NSStringFromClass([self class])]==NO || returnError.code!=NSErrorCustom_Code_OperationCancel)
                {
                    returnError = PPErrorOperationCancel(returnError);
                }
                
                //////////////////////////////////////////////////
                
                [self.syncActionModels removeAllObjects];
                
                self.currentSyncActionIndex = 0;
                self.totalSyncActionCount   = 0;
                self.suspended              = NO;
            }
            
            //////////////////////////////////////////////////
            
            self.executing  = NO;
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        [self dumpLogMessageWithFormat:@"%s out returnError=%@", __FUNCTION__, returnError];
        
        //////////////////////////////////////////////////
        
        [returnError retain];
    }
    
    [returnError autorelease];
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return (returnError==originError);
}

//================================================================================
//
//================================================================================
- (BOOL)suspendActionsWithError:(NSError **)error
{
    NSError     *originError    = ((error!=nil)?*error:nil);
    NSError     *returnError    = originError;
    
    //////////////////////////////////////////////////
    
    @autoreleasepool
    {
        [self dumpLogMessageWithFormat:@"%s in", __FUNCTION__];
        
        //////////////////////////////////////////////////
        
        do
        {
            //目前狀態(正在執行&&未暫停&&有未做完的syncActionModels),才可以暫停執行
            if(self.executing==NO || self.suspended==YES || [self.syncActionModels count]+[self.forceUpdateActionModels count]==0)
            {
                returnError = PPErrorOperationFailed(returnError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            self.suspended = YES;
            
            //等待真的停止運作才離開
            while(self.executing==YES)
            {
                [NSThread waitWithTimeInterval:0.01];
            }
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        [self dumpLogMessageWithFormat:@"%s out returnError=%@", __FUNCTION__, returnError];
        
        //////////////////////////////////////////////////
        
        [returnError retain];
    }
    
    [returnError autorelease];
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return (returnError==originError);
}

//================================================================================
//
//================================================================================
- (BOOL)skipOneOfActionsWithError:(NSError **)error
{
    NSError     *originError    = ((error!=nil)?*error:nil);
    NSError     *returnError    = originError;
    
    //////////////////////////////////////////////////
    
    @autoreleasepool
    {
        [self dumpLogMessageWithFormat:@"%s in", __FUNCTION__];
        
        //////////////////////////////////////////////////
        
        do
        {
            //目前狀態(停止執行&&已暫停&&有未做完的syncActionModels),才可以重新執行
            if(self.executing==YES || self.suspended==NO || [self.syncActionModels count]+[self.forceUpdateActionModels count]==0)
            {
                returnError = PPErrorOperationFailed(returnError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            self.executing  = YES;
            self.suspended  = NO;
            self.cancel     = NO;
            
            //////////////////////////////////////////////////
            
            //移除第一筆syncActionModels
            id firstObject = [self.syncActionModels firstObject];
            if(firstObject!=nil)
            {
                [self.syncActionModels removeObject:firstObject];
                self.currentSyncActionIndex++;
            }
            
            //////////////////////////////////////////////////
            
            [self executeActionsWithError:&returnError];
            
            //////////////////////////////////////////////////
            
            if(self.cancel==YES)
            {
                if(returnError==nil || [returnError.domain isEqualToString:NSStringFromClass([self class])]==NO || returnError.code!=NSErrorCustom_Code_OperationCancel)
                {
                    returnError = PPErrorOperationCancel(returnError);
                }
                
                //////////////////////////////////////////////////
                
                [self.syncActionModels removeAllObjects];
                
                self.currentSyncActionIndex = 0;
                self.totalSyncActionCount   = 0;
                self.suspended              = NO;
            }
            
            //////////////////////////////////////////////////
            
            self.executing = NO;
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        [self dumpLogMessageWithFormat:@"%s out returnError=%@", __FUNCTION__, returnError];
        
        //////////////////////////////////////////////////
        
        [returnError retain];
    }
    
    [returnError autorelease];
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return (returnError==originError);
}

//================================================================================
//
//================================================================================
- (BOOL)syncWithError:(NSError **)error quickSyncVaildTimeInterval:(NSTimeInterval)quickSyncVaildTimeInterval
{
    NSError     *originError    = ((error!=nil)?*error:nil);
    NSError     *returnError    = originError;
    
    //////////////////////////////////////////////////
    
    @autoreleasepool
    {
        [self dumpLogMessageWithFormat:@"%s in", __FUNCTION__];
        
        //////////////////////////////////////////////////
        
        BOOL originExecuting = self.executing;
     
        //////////////////////////////////////////////////
        
        do
        {
            if(self.executing==YES || self.suspended==YES || self.cancel==YES)
            {
                returnError = PPErrorOperationFailed(returnError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            self.executing  = YES;

            //////////////////////////////////////////////////
            // 設定是否要強制更新
            self.remoteSyncDataController.needSpecialCheck = self.needAddForceUpdate;
            self.localSyncDataController.needSpecialCheck = self.needAddForceUpdate;
            
            //////////////////////////////////////////////////
            //檢查版本升級Group
            
            [self reportStep:PPSyncActionStep_UpgradeSyncGroupModels];
            
            NSMutableArray *upgradeGroupSyncRecordModels = [self groupSyncRecordModelsOfLessThanVersion:self.syncGroupVersion error:&returnError];
            CGFloat upgradeGroupSyncRecordIndex = 0.0;
            CGFloat upgradeGroupSyncRecordTotalCount = (CGFloat)[upgradeGroupSyncRecordModels count];
            
            while([upgradeGroupSyncRecordModels count]>0)
            {
                PPSyncRecordModel *upgradeGroupSyncRecordModel = [upgradeGroupSyncRecordModels firstObject];
                if(upgradeGroupSyncRecordModel==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                    break;
                }
                
                ///////////////////////////////////////////////////
                
                PPSyncGroupModel *upgradeSyncGroupModel  = [[[PPSyncGroupModel alloc] init] autorelease];
                if(upgradeSyncGroupModel==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                    break;
                }
                
                upgradeSyncGroupModel.uniqueID  = upgradeGroupSyncRecordModel.localSyncCompareModel.uniqueID;
                
                ///////////////////////////////////////////////////
                
                BOOL result = NO;
                
                do
                {
                    result = [self.localSyncDataController updateSyncGroupModel:upgradeSyncGroupModel
                                                                    fromVersion:upgradeGroupSyncRecordModel.version
                                                                      toVersion:self.syncGroupVersion];
                    
                }while(result==NO && [self shouldRetryActionWithError:self.localSyncDataController.lastError]==YES);
                
                ///////////////////////////////////////////////////
                
                if(result==YES)
                {
                    //更新成功資料要寫回同步記錄
                    upgradeGroupSyncRecordModel.version = self.syncGroupVersion;
                    
                    ///////////////////////////////////////////////////
                    
                    [self updateGroupSyncRecordWithSyncRecordModel:upgradeGroupSyncRecordModel
                                                             error:&returnError];
                }
                else
                {
                    returnError = [[self.localSyncDataController.lastError retain] autorelease];
                    if(returnError==nil)
                    {
                        returnError = PPErrorOperationFailed(returnError);
                    }
                }
                
                [upgradeGroupSyncRecordModels removeObject:upgradeGroupSyncRecordModel];
                
                //////////////////////////////////////////////////
                // 計算進度
                upgradeGroupSyncRecordIndex ++;
                [self reportProgress:upgradeGroupSyncRecordIndex/upgradeGroupSyncRecordTotalCount];
                
                ///////////////////////////////////////////////////
                
                if(self.cancel==YES)
                {
                    break;
                }
            }
            
            if(self.cancel==YES || returnError!=originError)
            {
                break;
            }
            
            // 回報進度
            [self reportProgress:1.0];

            
            //////////////////////////////////////////////////
            //檢查版本升級Card
            
            [self reportStep:PPSyncActionStep_UpgradeSyncCardModels];
            NSMutableArray *upgradeCardSyncRecordModels = [self cardSyncRecordModelsOfLessThanVersion:self.syncCardVersion error:&returnError];
            CGFloat upgradeCardSyncRecordIndex = 0.0;
            CGFloat upgradeCardSyncRecordTotalCount = (CGFloat)[upgradeCardSyncRecordModels count];

            while([upgradeCardSyncRecordModels count]>0)
            {
                PPSyncRecordModel *upgradeCardSyncRecordModel = [upgradeCardSyncRecordModels firstObject];
                if(upgradeCardSyncRecordModel==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                    break;
                }
                
                ///////////////////////////////////////////////////
                
                PPSyncCardModel *upgradeSyncCardModel  = [[[PPSyncCardModel alloc] init] autorelease];
                if(upgradeSyncCardModel==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                    break;
                }
                
                upgradeSyncCardModel.uniqueID  = upgradeCardSyncRecordModel.localSyncCompareModel.uniqueID;
                
                ///////////////////////////////////////////////////
                
                BOOL result = NO;
                
                do
                {
                    result = [self.localSyncDataController updateSyncCardModel:upgradeSyncCardModel
                                                                   fromVersion:upgradeCardSyncRecordModel.version
                                                                     toVersion:self.syncCardVersion];
                    
                }while(result==NO && [self shouldRetryActionWithError:self.localSyncDataController.lastError]==YES);
                
                ///////////////////////////////////////////////////
                
                if(result==YES)
                {
                    //更新成功資料要寫回同步記錄
                    upgradeCardSyncRecordModel.version = self.syncCardVersion;
                    
                    ///////////////////////////////////////////////////
                    
                    [self updateCardSyncRecordWithSyncRecordModel:upgradeCardSyncRecordModel
                                                            error:&returnError];
                }
                else
                {
                    returnError = [[self.localSyncDataController.lastError retain] autorelease];
                    if(returnError==nil)
                    {
                        returnError = PPErrorOperationFailed(returnError);
                    }
                }
                
                [upgradeCardSyncRecordModels removeObject:upgradeCardSyncRecordModel];
                
                //////////////////////////////////////////////////
                // 計算進度
                upgradeCardSyncRecordIndex ++;
                [self reportProgress:upgradeCardSyncRecordIndex/upgradeCardSyncRecordTotalCount];

                ///////////////////////////////////////////////////
                
                if(self.cancel==YES)
                {
                    break;
                }
            }
            
            if(self.cancel==YES || returnError!=originError)
            {
                break;
            }
            
            // 回報進度
            [self reportProgress:1.0];

            //////////////////////////////////////////////////
            //記錄開始時間
            
            self.startDate = [NSDate date];
            
            //////////////////////////////////////////////////
            //檢查上次同步完成時間
            
            NSDate *lastSyncCompleteStartDate = [self lastSyncCompleteStartDate];
            if(lastSyncCompleteStartDate!=nil)
            {
                if(([self.startDate timeIntervalSince1970]-[lastSyncCompleteStartDate timeIntervalSince1970])<quickSyncVaildTimeInterval)
                {
                    lastSyncCompleteStartDate = [NSDate dateWithTimeInterval:-PPSyncActionController_Key_LastSyncCompleteStartDateOffset sinceDate:lastSyncCompleteStartDate];
                }
                else
                {
                    lastSyncCompleteStartDate = nil;
                }
            }
            
            [self dumpLogMessageWithFormat:@"lastSyncCompleteStartDate:%@", lastSyncCompleteStartDate];
            
            //////////////////////////////////////////////////
            //取得本地Group同步資料
            
            [self reportStep:PPSyncActionStep_LocalCopySyncGroupModels];
            
            NSMutableArray *localSyncGroupModels = [[self.localSyncDataController copySyncGroupModelsAfterDate:lastSyncCompleteStartDate] autorelease];
            if(localSyncGroupModels==nil)
            {
                returnError = [[self.localSyncDataController.lastError retain] autorelease];
                if(returnError==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                }
                
                break;
            }
            
            self.localGroupCount = [localSyncGroupModels count];
            
            [self dumpLogMessageWithFormat:@"localGroupCount:%lu", (unsigned long)self.localGroupCount];
            
            if(self.cancel==YES)
            {
                break;
            }
            
            // 回報進度
            [self reportProgress:1.0];
            
            //////////////////////////////////////////////////
            //取得遠端Group同步資料
            
            [self reportStep:PPSyncActionStep_RemoteCopySyncGroupModels];
            
            NSMutableArray *remoteSyncGroupModels = [[self.remoteSyncDataController copySyncGroupModelsAfterDate:lastSyncCompleteStartDate] autorelease];
            if(remoteSyncGroupModels==nil)
            {
                returnError = [[self.remoteSyncDataController.lastError retain] autorelease];
                if(returnError==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                }
                
                break;
            }
            
            self.remoteGroupCount = [remoteSyncGroupModels count];
            
            [self dumpLogMessageWithFormat:@"remoteGroupCount:%lu", (unsigned long)self.remoteGroupCount];
            
            if(self.cancel==YES)
            {
                break;
            }
            
            // 回報進度
            [self reportProgress:1.0];

            //////////////////////////////////////////////////
            // MARK:產生Group同步動作
            
            NSArray *groupSyncActionModels = [self syncActionModelsWithLocalSyncCompareModels:localSyncGroupModels
                                                                      remoteSyncCompareModels:remoteSyncGroupModels
                                                                                         type:PPSyncType_Group
                                                                                    afterDate:lastSyncCompleteStartDate
                                                                      forceUpdateActionModels:nil
                                                                                        error:&returnError];
            
            [self dumpLogMessageWithFormat:@"Group action count:%lu", (unsigned long)[groupSyncActionModels count]];
            
            if(groupSyncActionModels!=nil && [groupSyncActionModels count]>0)
            {
                [self.syncActionModels addObjectsFromArray:groupSyncActionModels];
                
            }

            if(self.cancel==YES || (returnError!=originError && returnError.code!=NSErrorPPSyncActionController_Code_LocalDeleteAllWarnning && returnError.code!=NSErrorPPSyncActionController_Code_RemoteDeleteAllWarnning))
            {
                break;
            }

            //////////////////////////////////////////////////
            //取得本地Card同步資料
            
            [self reportStep:PPSyncActionStep_LocalCopySyncCardModels];
            
            NSMutableArray *localSyncCardModels = [[self.localSyncDataController copySyncCardModelsAfterDate:lastSyncCompleteStartDate] autorelease];
            if(localSyncCardModels==nil)
            {
                returnError = [[self.localSyncDataController.lastError retain] autorelease];
                if(returnError==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                }
                
                break;
            }
            
            self.localCardCount = [localSyncCardModels count];
            
            [self dumpLogMessageWithFormat:@"localCardCount:%lu", (unsigned long)self.localCardCount];
            
            if(self.cancel==YES)
            {
                break;
            }
            
            // 回報進度
            [self reportProgress:1.0];

            //////////////////////////////////////////////////
            //取得遠端Card同步資料
            
            [self reportStep:PPSyncActionStep_RemoteCopySyncCardModels];
            
            NSMutableArray *remoteSyncCardModels = [[self.remoteSyncDataController copySyncCardModelsAfterDate:lastSyncCompleteStartDate] autorelease];
            if(remoteSyncCardModels==nil)
            {
                returnError = [[self.remoteSyncDataController.lastError retain] autorelease];
                if(returnError==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                }
                
                break;
            }
            
            self.remoteCardCount = [remoteSyncCardModels count];
            
            [self dumpLogMessageWithFormat:@"remoteCardCount:%lu", (unsigned long)self.remoteCardCount];
            
            if(self.cancel==YES)
            {
                break;
            }
            
            // 回報進度
            [self reportProgress:1.0];

            
            //////////////////////////////////////////////////
            //產生Card同步動作
            NSMutableArray * forceUpdateActionModels = [[NSMutableArray alloc] init];
            NSArray *cardSyncActionModels = [self syncActionModelsWithLocalSyncCompareModels:localSyncCardModels
                                                                     remoteSyncCompareModels:remoteSyncCardModels
                                                                                        type:PPSyncType_Card
                                                                                   afterDate:lastSyncCompleteStartDate
                                                                     forceUpdateActionModels:forceUpdateActionModels
                                                                                       error:&returnError];
            
            [self dumpLogMessageWithFormat:@"Card action count:%lu", (unsigned long)[cardSyncActionModels count]];
            [self dumpLogMessageWithFormat:@"force update action count:%lu", (unsigned long)[forceUpdateActionModels count]];

            if(cardSyncActionModels!=nil && [cardSyncActionModels count]>0)
            {
                [self.syncActionModels addObjectsFromArray:cardSyncActionModels];
            }
            
            //////////////////////////////////////////////////
            // 加入強制更新的動作
            if ([forceUpdateActionModels count]>0)
            {
                [self.forceUpdateActionModels addObjectsFromArray:forceUpdateActionModels];
            }
            [forceUpdateActionModels release];
            
            //////////////////////////////////////////////////
            
            if(self.cancel==YES || (returnError!=originError && returnError.code!=NSErrorPPSyncActionController_Code_LocalDeleteAllWarnning && returnError.code!=NSErrorPPSyncActionController_Code_RemoteDeleteAllWarnning))
            {
                break;
            }
            
            //////////////////////////////////////////////////
            //排序動作
            
            [self.syncActionModels sortUsingSelector:@selector(compare:)];
            
            //////////////////////////////////////////////////
            //檢查需要同步嗎
            
            self.currentSyncActionIndex = 0;
            self.totalSyncActionCount   = [self.syncActionModels count] + [self.forceUpdateActionModels count];
            
            if(self.totalSyncActionCount>0)
            {
                //兩段式作法,需要同步不回傳YES
                self.suspended = YES;
                
                //有可能會有上面刪除全部的warnning,避免壓掉
                if(returnError==nil)
                {
                    returnError = PPErrorMake(NSErrorPPSyncActionController_Code_ReportSuspend, @"Report Suspend", returnError);
                }
                
                break;
            }
            
            //////////////////////////////////////////////////

            if(self.startDate!=nil)
            {
                [self setLastSyncCompleteStartDate:self.startDate];
                
                // 避免沒有同步動作時，同步完時間沒有異動。
                if([self.startDate compare:[self lastSyncActionSuccessDate]] == kCFCompareGreaterThan)
                {
                    [self setLastSyncActionSuccessDate:self.startDate];
                }

                self.startDate = nil;
            }
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(self.cancel==YES) 
        {
            if(returnError==nil || [returnError.domain isEqualToString:NSStringFromClass([self class])]==NO || returnError.code!=NSErrorCustom_Code_OperationCancel)
            {
                returnError = PPErrorOperationCancel(returnError);
            }
            
            //////////////////////////////////////////////////
            
            [self.syncActionModels removeAllObjects];
            
            self.currentSyncActionIndex = 0;
            self.totalSyncActionCount   = 0;
            self.suspended              = NO;
        }
        
        //////////////////////////////////////////////////
        
        self.executing = originExecuting;
        
        //////////////////////////////////////////////////
        
        [self dumpLogMessageWithFormat:@"%s out returnError=%@", __FUNCTION__, returnError];
        
        //////////////////////////////////////////////////
        
        [returnError retain];
    }
    
    [returnError autorelease];
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return (returnError==originError);
}


//================================================================================
//
//================================================================================
- (NSDate *)lastSyncCompleteStartDate
{
    return [self statusObjectForKey:PPSyncActionController_Key_LastSyncCompleteStartDate];
}


//================================================================================
//
//================================================================================
- (void)setLastSyncCompleteStartDate:(NSDate *)date
{
    return [self setStatusObject:date forKey:PPSyncActionController_Key_LastSyncCompleteStartDate];
}


//================================================================================
//
//================================================================================
- (NSDate *)lastSyncActionSuccessDate
{
    return [self statusObjectForKey:PPSyncActionController_Key_LastSyncActionSuccessDate];
}


//================================================================================
//
//================================================================================
- (void)setLastSyncActionSuccessDate:(NSDate *)date
{
    return [self setStatusObject:date forKey:PPSyncActionController_Key_LastSyncActionSuccessDate];
}


//================================================================================
//
//================================================================================
- (BOOL)cleanSyncData:(NSError **)error
{
    BOOL result = [super deleteAllSyncRecordWithError:error];
    
    if(result==YES)
    {
        [self setLastSyncCompleteStartDate:nil];
        [self setLastSyncActionSuccessDate:nil];
    }
    
    return result;
}


//================================================================================
//
//================================================================================
- (BOOL)hasSyncData
{
    return ([self lastSyncCompleteStartDate] != nil ||
            [self lastSyncActionSuccessDate] != nil );
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Override Methods

//================================================================================
//
//================================================================================
- (void)executeActionSuccess
{
    [self setLastSyncActionSuccessDate:[NSDate date]];
}

//================================================================================
// !! sender目前沒有拿來應用
//================================================================================
- (void)logMessage:(NSString *)logMessage sender:(id)sender
{
    if([self.delegate respondsToSelector:@selector(ppSyncActionController:logMessage:)]==YES)
    {
        [self.delegate ppSyncActionController:self logMessage:logMessage];
    }
}

//================================================================================
//
//================================================================================
- (BOOL)shouldRescheduleActionWithError:(NSError *)error
{
    return  (error!=nil &&
             (([error.domain isEqualToString:NSStringFromClass([self.localSyncDataController class])]==YES && [self.localSyncDataController shouldRescheduleActionWithError:error]==YES) ||
              ([error.domain isEqualToString:NSStringFromClass([self.remoteSyncDataController class])]==YES && [self.remoteSyncDataController shouldRescheduleActionWithError:error]==YES)));
}

//================================================================================
//
//================================================================================
- (BOOL)shouldRetryActionWithError:(NSError *)error
{
    return  (error!=nil &&
             (([error.domain isEqualToString:NSStringFromClass([self.localSyncDataController class])]==YES && [self.localSyncDataController shouldRetryActionWithError:error]==YES) ||
              ([error.domain isEqualToString:NSStringFromClass([self.remoteSyncDataController class])]==YES && [self.remoteSyncDataController shouldRetryActionWithError:error]==YES)));
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private Methods

//==============================================================================
//
//==============================================================================
- (void)dumpLogMessageWithFormat:(NSString *)format, ...
{
   @autoreleasepool
    {
        va_list args;
        va_start(args,format);
        
        NSString *message = [[[NSString alloc] initWithFormat:format arguments:args] autorelease];
        message = [@"\t##1## " stringByAppendingString:message];
        [self logMessage:message sender:self];
        
        va_end(args);
    }
}


//================================================================================
//
//================================================================================
- (BOOL)executeActionsWithError:(NSError **)error
{
    [self dumpLogMessageWithFormat:@"%s in", __FUNCTION__];

    //////////////////////////////////////////////////
    
//    NSUInteger  rescheduleCount = 0;
    NSError     *originError    = ((error!=nil)?*error:nil);
    NSError     *returnError    = originError;
    
    //////////////////////////////////////////////////
    //同步動作開始
    
    [self reportStep:PPSyncActionStep_ExecuteSyncActions];
    
    //////////////////////////////////////////////////
    
    if([self executeActionModels:self.syncActionModels withError:&returnError])
    {
//        NSLog(@"sync finish");
        [self reportProgress:1.0];
    }
    

    
    // 同步成功後
    
    [self reportStep:PPSyncActionStep_ExecuteSpecialSyncActions];
    
    if (returnError==nil &&
        self.needAddForceUpdate)
    {
        // 補齊強制更新的localSyncCompareModel
        [self filloutForceUpdateActionModels];
        
        // 處理強制更新動作
        NSError *forceUpdateError = nil;
        [self executeActionModels:self.forceUpdateActionModels withError:&forceUpdateError];
        [self reportProgress:1.0];
        
        if(forceUpdateError!=nil)
        {
            returnError = forceUpdateError;
        }
    }
    //////////////////////////////////////////////////
    
    if(self.cancel==YES)
    {
        if(returnError==nil || [returnError.domain isEqualToString:NSStringFromClass([self class])]==NO || returnError.code!=NSErrorCustom_Code_OperationCancel)
        {
            returnError = PPErrorOperationCancel(returnError);
        }
    }
    else
    {
        if(self.suspended==YES)
        {
            returnError = PPErrorMake(NSErrorPPSyncActionController_Code_Suspend, @"Suspend", returnError);
        }
        else
        {
            if(returnError==originError)
            {
                if(self.startDate!=nil)
                {
                    [self setLastSyncCompleteStartDate:self.startDate];
                    
                    // 避免沒有同步動作時，同步完時間沒有異動。
                    if([self.startDate compare:[self lastSyncActionSuccessDate]] == kCFCompareGreaterThan)
                    {
                        [self setLastSyncActionSuccessDate:self.startDate];
                    }

                    self.startDate = nil;
                }
                
                [self reportStep:PPSyncActionStep_Idle];
            }
            else
            {
                self.suspended = YES;
            }
        }
    }
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    [self dumpLogMessageWithFormat:@"%s out returnError=%@", __FUNCTION__, returnError];
    
    return (returnError==originError);
}


//==============================================================================
//
//==============================================================================
- (BOOL)executeActionModels:(NSMutableArray *)actionModels withError:(NSError **)error
{
    [self dumpLogMessageWithFormat:@"%s in (count:%lu)", __func__, (unsigned long)[actionModels count]];

    //////////////////////////////////////////////////
    
    NSUInteger  rescheduleCount = 0;
    NSError     *originError    = ((error!=nil)?*error:nil);
    NSError     *returnError    = originError;

    //////////////////////////////////////////////////
    
    while([actionModels count]>0)
    {
        // !! 檢查break。會跳出while的，returnError必須再retain一次，不然會被autoreleasepool釋放。
        @autoreleasepool
        {
            if(self.cancel==YES || self.suspended==YES)
            {
                [returnError retain];
                break;
            }
            
            ///////////////////////////////////////////////////
            
            self.currentSyncActionModel = [actionModels firstObject];
            if(self.currentSyncActionModel==nil)
            {
                returnError = PPErrorOperationFailed(returnError);
                [returnError retain];
                break;
            }
            
            ///////////////////////////////////////////////////
            //先判斷對哪一目標做動作
            
            PPSyncCompareModel      *sourceSyncCompareModel     = nil;
            PPSyncCompareModel      *targetSyncCompareModel     = nil;
            PPSyncDataController    *targetSyncDataController   = nil;
            PPSyncRecordModel       *resultSyncRecordModel      = nil;
            
            switch(self.currentSyncActionModel.target)
            {
                case PPSyncTarget_Local:
                {
                    sourceSyncCompareModel      = self.currentSyncActionModel.remoteSyncCompareModel;
                    targetSyncCompareModel      = self.currentSyncActionModel.localSyncCompareModel;
                    targetSyncDataController    = self.localSyncDataController;
                    
                    break;
                }
                case PPSyncTarget_Remote:
                {
                    sourceSyncCompareModel      = self.currentSyncActionModel.localSyncCompareModel;
                    targetSyncCompareModel      = self.currentSyncActionModel.remoteSyncCompareModel;
                    targetSyncDataController    = self.remoteSyncDataController;
                    
                    break;
                }
                default:
                {
                    returnError = PPErrorOperationFailed(returnError);
                    break;
                }
            }
            
            ///////////////////////////////////////////////////
            
            if(returnError!=originError)
            {
                [returnError retain];
                break;
            }
            
            ///////////////////////////////////////////////////
            
            // MARK: #Dump log
            // I/L (Input/Local)
            // I/R (Input/Remote)
            [self dumpLogMessageWithFormat:@"--------------------------------------------------"];
            [self dumpLogMessageWithFormat:[NSString stringWithFormat:@"Action:%@ Target:%@ Type:%@",
                                            [self nameForAction:self.currentSyncActionModel.action],
                                            [self nameForTarget:self.currentSyncActionModel.target],
                                            [self nameForType:self.currentSyncActionModel.type]]];
            [self dumpLogMessageWithFormat:[NSString stringWithFormat:@"I/L:%@", self.currentSyncActionModel.localSyncCompareModel]];
            [self dumpLogMessageWithFormat:[NSString stringWithFormat:@"I/R:%@", self.currentSyncActionModel.remoteSyncCompareModel]];
            
            ///////////////////////////////////////////////////
            
            BOOL canSync = YES;
            
            switch(self.currentSyncActionModel.action)
            {
                case PPSyncAction_Delete:
                {
                    PPSyncRecordType ppSyncRecordType = (self.currentSyncActionModel.target==PPSyncTarget_Local?PPSyncRecordType_Local:PPSyncRecordType_Remote);
                    
                    switch(self.currentSyncActionModel.type)
                    {
                            // MARK:PPSyncAction_Delete Group

                        case PPSyncType_Group:
                        {
                            BOOL result = NO;
                            
                            do
                            {
                                result = [targetSyncDataController removeSyncGroupModelByUniqueID:targetSyncCompareModel.uniqueID];
                                
                            }while(result==NO && [self shouldRetryActionWithError:targetSyncDataController.lastError]==YES);
                            
                            ///////////////////////////////////////////////////
                            
                            if(result==YES)
                            {
                                [self deleteGroupSyncRecordWithUniqueID:targetSyncCompareModel.uniqueID
                                                         syncRecordType:ppSyncRecordType
                                                                  error:&returnError];
                            }
                            else
                            {
                                returnError = [[targetSyncDataController.lastError retain] autorelease];
                                if(returnError==nil)
                                {
                                    returnError = PPErrorOperationFailed(returnError);
                                }
                            }
                            
                            break;
                        }
                            // MARK:PPSyncAction_Delete Card

                        case PPSyncType_Card:
                        {
                            if(self.conflict == PPSyncActionConflict_DeleteFirst)
                            {
                                // 刪除優先就不用檢查目標有沒有異動了，直接刪除。
                                canSync = YES;
                            }
                            else
                            {
                                canSync = [targetSyncDataController canSyncCardWithUniqueID:targetSyncCompareModel.uniqueID
                                                                           lastModifiedDate:targetSyncCompareModel.lastModifiedDate];
                            }
                            
                            if(canSync==YES)
                            {
                                BOOL result = NO;
                                
                                do
                                {
                                    result = [targetSyncDataController removeSyncCardModelByUniqueID:targetSyncCompareModel.uniqueID];
                                    
                                }while(result==NO && [self shouldRetryActionWithError:targetSyncDataController.lastError]==YES);
                                
                                ///////////////////////////////////////////////////
                                
                                if(result==YES)
                                {
                                    [self deleteCardSyncRecordWithUniqueID:targetSyncCompareModel.uniqueID
                                                            syncRecordType:ppSyncRecordType
                                                                     error:&returnError];
                                }
                                else
                                {
                                    returnError = [[targetSyncDataController.lastError retain] autorelease];
                                    if(returnError==nil)
                                    {
                                        returnError = PPErrorOperationFailed(returnError);
                                    }
                                }
                            }
                            else
                            {
                                self.hasSkipedAction = YES;
                            }
                            break;
                        }
                        default:
                        {
                            returnError = PPErrorOperationFailed(returnError);
                            
                            break;
                        }
                    }
                    
                    break;
                }
                case PPSyncAction_Insert:
                {
                    PPSyncRecordType ppSyncRecordType = (self.currentSyncActionModel.target==PPSyncTarget_Local?PPSyncRecordType_Remote:PPSyncRecordType_Local);
                    
                    switch(self.currentSyncActionModel.type)
                    {
                            // MARK:PPSyncAction_Insert Group

                        case PPSyncType_Group:
                        {
                            PPSyncGroupModel *sourceSyncGroupModel  = (PPSyncGroupModel *)sourceSyncCompareModel;
                            if(sourceSyncGroupModel==nil)
                            {
                                returnError = PPErrorOperationFailed(returnError);
                                break;
                            }
                            
                            ///////////////////////////////////////////////////
                            
                            PPSyncGroupModel *targetSyncGroupModel  = [[[PPSyncGroupModel alloc] init] autorelease];
                            if(targetSyncGroupModel==nil)
                            {
                                returnError = PPErrorOperationFailed(returnError);
                                break;
                            }
                            
                            targetSyncGroupModel.name = sourceSyncGroupModel.name;
                            targetSyncGroupModel.defaultGroup = sourceSyncGroupModel.defaultGroup;
                            
                            ///////////////////////////////////////////////////
                            
                            BOOL result = NO;
                            
                            do
                            {
                                result = [targetSyncDataController addSyncGroupModel:targetSyncGroupModel];
                                
                            }while(result==NO && [self shouldRetryActionWithError:targetSyncDataController.lastError]==YES);
                            
                            ///////////////////////////////////////////////////
                            
                            if(result==YES)
                            {
                                //新增成功資料要寫回同步記錄
                                resultSyncRecordModel = [[[PPSyncRecordModel alloc] init] autorelease];
                                if(resultSyncRecordModel==nil)
                                {
                                    returnError = PPErrorOperationFailed(returnError);
                                    break;
                                }
                                
                                ///////////////////////////////////////////////////
                                
                                switch(self.currentSyncActionModel.target)
                                {
                                    case PPSyncTarget_Local:
                                    {
                                        resultSyncRecordModel.localSyncCompareModel     = targetSyncGroupModel;
                                        resultSyncRecordModel.remoteSyncCompareModel    = sourceSyncGroupModel;
                                        
                                        break;
                                    }
                                    case PPSyncTarget_Remote:
                                    {
                                        resultSyncRecordModel.localSyncCompareModel     = sourceSyncGroupModel;
                                        resultSyncRecordModel.remoteSyncCompareModel    = targetSyncGroupModel;
                                        
                                        break;
                                    }
                                    default:
                                    {
                                        returnError = PPErrorOperationFailed(returnError);
                                        
                                        break;
                                    }
                                }
                                
                                resultSyncRecordModel.version = self.syncGroupVersion;
                                
                                ///////////////////////////////////////////////////
                                //衝突處理的話會已經存在同步記錄,要先檢查是不是衝突處理產生的同步記錄,把已存在的紀錄先砍掉
                                
                                BOOL find = [self isFindGroupSyncRecordModelWithUniqueID:sourceSyncGroupModel.uniqueID
                                                                          syncRecordType:ppSyncRecordType
                                                                                   error:&returnError];
                                if(returnError!=originError)
                                {
                                    break;
                                }
                                
                                ///////////////////////////////////////////////////
                                
                                if(find==YES)
                                {
                                    if([self beginTransactionWithError:&returnError]==YES)
                                    {
                                        if([self deleteGroupSyncRecordWithUniqueID:sourceSyncGroupModel.uniqueID
                                                                    syncRecordType:ppSyncRecordType
                                                                             error:&returnError]==YES
                                           &&
                                           [self insertGroupSyncRecordWithSyncRecordModel:resultSyncRecordModel
                                                                                    error:&returnError]==YES)
                                        {
                                            [self endTransactionWithError:&returnError];
                                        }
                                        
                                        ///////////////////////////////////////////////////
                                        
                                        if(returnError!=originError)
                                        {
                                            [self rollbackTransactionWithError:&returnError];
                                        }
                                    }
                                }
                                else
                                {
                                    [self insertGroupSyncRecordWithSyncRecordModel:resultSyncRecordModel
                                                                             error:&returnError];
                                }
                            }
                            else
                            {
                                returnError = [[targetSyncDataController.lastError retain] autorelease];
                                if(returnError==nil)
                                {
                                    returnError = PPErrorOperationFailed(returnError);
                                }
                            }
                            
                            break;
                        }
                            // MARK:PPSyncAction_Insert Card

                        case PPSyncType_Card:
                        {
                            PPSyncCardModel *sourceSyncCardModel    = (PPSyncCardModel *)sourceSyncCompareModel;
                            if(sourceSyncCardModel==nil)
                            {
                                returnError = PPErrorOperationFailed(returnError);
                                break;
                            }
                            
                            ///////////////////////////////////////////////////
                            
                            PPSyncCardModel *targetSyncCardModel    = [[[PPSyncCardModel alloc] initWithSyncCardModel:sourceSyncCardModel] autorelease];
                            if(targetSyncCardModel==nil)
                            {
                                returnError = PPErrorOperationFailed(returnError);
                                break;
                            }
                            
                            ///////////////////////////////////////////////////
                            
                            BOOL result = NO;
                            
                            do
                            {
                                result = [targetSyncDataController addSyncCardModel:targetSyncCardModel];
                                
                            }while(result==NO && [self shouldRetryActionWithError:targetSyncDataController.lastError]==YES);
                            
                            ///////////////////////////////////////////////////
                            
                            if(result==YES)
                            {
                                //新增成功資料要寫回同步記錄
                                resultSyncRecordModel = [[[PPSyncRecordModel alloc] init] autorelease];
                                if(resultSyncRecordModel==nil)
                                {
                                    returnError = PPErrorOperationFailed(returnError);
                                    break;
                                }
                                
                                ///////////////////////////////////////////////////
                                
                                switch(self.currentSyncActionModel.target)
                                {
                                    case PPSyncTarget_Local:
                                    {
                                        resultSyncRecordModel.localSyncCompareModel     = targetSyncCardModel;
                                        resultSyncRecordModel.remoteSyncCompareModel    = sourceSyncCardModel;
                                        
                                        break;
                                    }
                                    case PPSyncTarget_Remote:
                                    {
                                        resultSyncRecordModel.localSyncCompareModel     = sourceSyncCardModel;
                                        resultSyncRecordModel.remoteSyncCompareModel    = targetSyncCardModel;
                                        
                                        break;
                                    }
                                    default:
                                    {
                                        returnError = PPErrorOperationFailed(returnError);
                                        
                                        break;
                                    }
                                }
                                
                                resultSyncRecordModel.version = self.syncCardVersion;
                                
                                ///////////////////////////////////////////////////
                                //衝突處理的話會已經存在同步記錄,要先檢查是不是衝突處理產生的同步記錄,把已存在的紀錄先砍掉
                                
                                BOOL find = [self isFindCardSyncRecordModelWithUniqueID:sourceSyncCardModel.uniqueID
                                                                         syncRecordType:ppSyncRecordType
                                                                                  error:&returnError];
                                if(returnError!=originError)
                                {
                                    break;
                                }
                                
                                ///////////////////////////////////////////////////
                                
                                if(find==YES)
                                {
                                    if([self beginTransactionWithError:&returnError]==YES)
                                    {
                                        if([self deleteCardSyncRecordWithUniqueID:sourceSyncCardModel.uniqueID
                                                                   syncRecordType:ppSyncRecordType
                                                                            error:&returnError]==YES
                                           &&
                                           [self insertCardSyncRecordWithSyncRecordModel:resultSyncRecordModel
                                                                                   error:&returnError]==YES)
                                        {
                                            [self endTransactionWithError:&returnError];
                                        }
                                        
                                        ///////////////////////////////////////////////////
                                        
                                        if(returnError!=originError)
                                        {
                                            [self rollbackTransactionWithError:&returnError];
                                        }
                                    }
                                }
                                else
                                {
                                    [self insertCardSyncRecordWithSyncRecordModel:resultSyncRecordModel
                                                                            error:&returnError];
                                }
                            }
                            else
                            {
                                returnError = [[targetSyncDataController.lastError retain] autorelease];
                                if(returnError==nil)
                                {
                                    returnError = PPErrorOperationFailed(returnError);
                                }
                            }
                            
                            break;
                        }
                        default:
                        {
                            returnError = PPErrorOperationFailed(returnError);
                            
                            break;
                        }
                    }
                    
                    break;
                }
                case PPSyncAction_Update:
                {
                    switch(self.currentSyncActionModel.type)
                    {
                            // MARK:PPSyncAction_Update Group

                        case PPSyncType_Group:
                        {
                            PPSyncGroupModel *sourceSyncGroupModel  = (PPSyncGroupModel *)sourceSyncCompareModel;
                            if(sourceSyncGroupModel==nil)
                            {
                                returnError = PPErrorOperationFailed(returnError);
                                break;
                            }

                            PPSyncGroupModel *originTargetSyncGroupModel  = (PPSyncGroupModel *)targetSyncCompareModel;
                            if(originTargetSyncGroupModel==nil)
                            {
                                returnError = PPErrorOperationFailed(returnError);
                                break;
                            }

                            ///////////////////////////////////////////////////
                            
                            PPSyncGroupModel *targetSyncGroupModel  = [[[PPSyncGroupModel alloc] init] autorelease];
                            if(targetSyncGroupModel==nil)
                            {
                                returnError = PPErrorOperationFailed(returnError);
                                break;
                            }
                            
                            targetSyncGroupModel.uniqueID       = originTargetSyncGroupModel.uniqueID;
                            targetSyncGroupModel.defaultGroup   = sourceSyncGroupModel.defaultGroup;
                            targetSyncGroupModel.name           = sourceSyncGroupModel.name;
                            
                            ///////////////////////////////////////////////////
                            
                            BOOL result = NO;
                            
                            do
                            {
                                result = [targetSyncDataController updateSyncGroupModel:targetSyncGroupModel
                                                                            fromVersion:self.syncGroupVersion
                                                                              toVersion:self.syncGroupVersion];
                                
                            }while(result==NO && [self shouldRetryActionWithError:targetSyncDataController.lastError]==YES);
                            
                            ///////////////////////////////////////////////////
                            
                            if(result==YES)
                            {
                                //更新成功資料要寫回同步記錄
                                targetSyncCompareModel.uniqueID         = targetSyncGroupModel.uniqueID;
                                targetSyncCompareModel.lastModifiedDate = targetSyncGroupModel.lastModifiedDate;
                                
                                ///////////////////////////////////////////////////
                                
                                resultSyncRecordModel = [[[PPSyncRecordModel alloc] init] autorelease];
                                if(resultSyncRecordModel==nil)
                                {
                                    returnError = PPErrorOperationFailed(returnError);
                                    break;
                                }
                                
                                resultSyncRecordModel.localSyncCompareModel     = self.currentSyncActionModel.localSyncCompareModel;
                                resultSyncRecordModel.remoteSyncCompareModel    = self.currentSyncActionModel.remoteSyncCompareModel;
                                resultSyncRecordModel.version                   = self.syncGroupVersion;
                                
                                ///////////////////////////////////////////////////
                                
                                [self updateGroupSyncRecordWithSyncRecordModel:resultSyncRecordModel
                                                                         error:&returnError];
                            }
                            else
                            {
                                returnError = [[targetSyncDataController.lastError retain] autorelease];
                                if(returnError==nil)
                                {
                                    returnError = PPErrorOperationFailed(returnError);
                                }
                            }
                            
                            break;
                        }
                            // MARK:PPSyncAction_Update Card

                        case PPSyncType_Card:
                        {
                            canSync = [targetSyncDataController canSyncCardWithUniqueID:targetSyncCompareModel.uniqueID
                                                                       lastModifiedDate:targetSyncCompareModel.lastModifiedDate];
                            if(canSync==YES)
                            {
                                PPSyncCardModel *sourceSyncCardModel    = (PPSyncCardModel *)sourceSyncCompareModel;
                                if(sourceSyncCardModel==nil)
                                {
                                    returnError = PPErrorOperationFailed(returnError);
                                    break;
                                }
                                
                                ///////////////////////////////////////////////////
                                
                                PPSyncCardModel *targetSyncCardModel    = [[[PPSyncCardModel alloc] initWithSyncCardModel:sourceSyncCardModel] autorelease];
                                if(sourceSyncCardModel==nil)
                                {
                                    returnError = PPErrorOperationFailed(returnError);
                                    break;
                                }
                                
                                targetSyncCardModel.uniqueID = targetSyncCompareModel.uniqueID;
                                targetSyncCardModel.needForceUpdate = targetSyncCompareModel.needForceUpdate;

                                ///////////////////////////////////////////////////
                                
                                BOOL result = NO;
                                
                                do
                                {
                                    result = [targetSyncDataController updateSyncCardModel:targetSyncCardModel
                                                                               fromVersion:self.syncCardVersion
                                                                                 toVersion:self.syncCardVersion];
                                    
                                }while(result==NO && [self shouldRetryActionWithError:targetSyncDataController.lastError]==YES);
                                
                                ///////////////////////////////////////////////////
                                
                                if(result==YES)
                                {
                                    //更新成功資料要寫回同步記錄
                                    targetSyncCompareModel.uniqueID         = targetSyncCardModel.uniqueID;
                                    targetSyncCompareModel.lastModifiedDate = targetSyncCardModel.lastModifiedDate;
                                    
                                    ///////////////////////////////////////////////////
                                    
                                    resultSyncRecordModel = [[[PPSyncRecordModel alloc] init] autorelease];
                                    if(resultSyncRecordModel==nil)
                                    {
                                        returnError = PPErrorOperationFailed(returnError);
                                        break;
                                    }
                                    
                                    resultSyncRecordModel.localSyncCompareModel     = self.currentSyncActionModel.localSyncCompareModel;
                                    resultSyncRecordModel.remoteSyncCompareModel    = self.currentSyncActionModel.remoteSyncCompareModel;
                                    resultSyncRecordModel.version                   = self.syncCardVersion;
                                    
                                    ///////////////////////////////////////////////////
                                    
                                    [self updateCardSyncRecordWithSyncRecordModel:resultSyncRecordModel
                                                                            error:&returnError];
                                }
                                else
                                {
                                    returnError = [[targetSyncDataController.lastError retain] autorelease];
                                    if(returnError==nil)
                                    {
                                        returnError = PPErrorOperationFailed(returnError);
                                    }
                                }
                            }
                            else
                            {
                                self.hasSkipedAction = YES;
                            }

                            break;
                        }
                        default:
                        {
                            returnError = PPErrorOperationFailed(returnError);
                            
                            break;
                        }
                    }
                    
                    break;
                }
                default:
                {
                    returnError = PPErrorOperationFailed(returnError);
                    
                    break;
                }
            }

            //////////////////////////////////////////////////

            // MARK: #Dump log
            // O/L (Output/Local)
            // O/R (Output/Remote)
            // O/RE (Output/returnError)
            // O/OE (Output/originError)
            // action執行成功才會有ppSyncRecordModel
            [self dumpLogMessageWithFormat:@"O/L:%@", [resultSyncRecordModel localSyncCompareModel]];
            [self dumpLogMessageWithFormat:@"O/R:%@", [resultSyncRecordModel remoteSyncCompareModel]];
            
            if(returnError != nil)
            {
                [self dumpLogMessageWithFormat:@"O/RE:%@", returnError];
                [self dumpLogMessageWithFormat:@"O/OE:%@", originError];
            }
            
            //////////////////////////////////////////////////
            
            if(returnError!=originError)
            {
                //檢查是否可以進行重排
                if(rescheduleCount<PPSyncActionControllerRescheduleLimit && [self shouldRescheduleActionWithError:returnError]==YES)
                {
                    //先從現在的位置移走
                    
                    [actionModels removeObject:self.currentSyncActionModel];
                    
                    //////////////////////////////////////////////////
                    //尋找要插入的位置
                    
                    if(self.currentSyncActionModel.type==PPSyncType_Group)
                    {
                        for(PPSyncActionModel *ppSyncActionModel in self.syncActionModels)
                        {
                            if(ppSyncActionModel.type==PPSyncType_Card)
                            {
                                [actionModels insertObject:self.currentSyncActionModel atIndex:[self.syncActionModels indexOfObject:ppSyncActionModel]];
                                
                                self.currentSyncActionModel = nil;
                                
                                break;
                            }
                        }
                    }
                    
                    //////////////////////////////////////////////////
                    //PPSyncType_Group後面都沒有Card或PPSyncType_Card直接append在最後面
                    
                    if(self.currentSyncActionModel!=nil)
                    {
                        [actionModels addObject:self.currentSyncActionModel];
                        
                        self.currentSyncActionModel = nil;
                    }
                    
                    //////////////////////////////////////////////////
                    
                    //重排計數器增加
                    rescheduleCount++;
                    
                    //清掉錯誤
                    returnError = originError;
                }
                else
                {
                    [returnError retain];
                    break;
                }
            }
            else
            {
                //////////////////////////////////////////////////
                // MARK: #groupID對應錯誤
                
                // 2017/12/15 eddie
                // 在group操作時，local先刪除"Jap"，接著remote又修改"Jap"，然後同步。
                // 結果同步邏輯要求local新增"Jap"(104)，並把對應關係記錄到table中，到這裡沒問題。
                // ------------------------------------------------------------------------
                // ##1## Action:Insert Target:Local Type:Group
                // ##1## I/L:{name:(null), defaultGroup:0}{deleted:YES needForceUpdate:NO uniqueID:101 lastModifiedDate:...}
                // ##1## I/R:{name:Jap, defaultGroup:0}{deleted:NO needForceUpdate:NO uniqueID:EAD0C64F-E15E-4B78-ACE1-8188816B9DF0:ABGroup lastModifiedDate:...}
                // ##2WC## -[ABLocalSyncDataController addSyncGroupModel:]
                // ##1## O/L:{name:Jap, defaultGroup:0}{deleted:NO needForceUpdate:NO uniqueID:104 lastModifiedDate:...}
                // ##1## O/R:{name:Jap, defaultGroup:0}{deleted:NO needForceUpdate:NO uniqueID:EAD0C64F-E15E-4B78-ACE1-8188816B9DF0:ABGroup lastModifiedDate:...}
                
                // 但接下來有問題的code會拿原本輸入時的資料去更新對應關係記錄，變成remote端的
                // groupID又對應到之前local刪除的groupID(101)，後續的group對應就會錯誤。
                // ------------------------------------------------------------------------
                // ##1## O/L 2:{name:(null), defaultGroup:0}{deleted:YES needForceUpdate:NO uniqueID:101 lastModifiedDate:...}
                // ##1## O/R 2:{name:Jap, defaultGroup:0}{deleted:NO needForceUpdate:NO uniqueID:EAD0C64F-E15E-4B78-ACE1-8188816B9DF0:ABGroup lastModifiedDate:...}

                // 結論：目前的對應錯誤不會影嚮到DB中的記錄，暫時先不處理，再觀察。
                //     （下面code更新資料庫會沒有結果，因為沒有這對ID。）
                

                //////////////////////////////////////////////////
                // mike說google同步action處理完後時間還是有可能變動，所以要再更新一次。
                // 不更新頂多下次同步可能有些時間不一致又產生action，其實也沒關係。

                if(canSync==YES)
                {
                    //更新同步記錄
                    PPSyncRecordModel *updateSyncRecordModel = [[[PPSyncRecordModel alloc] init] autorelease];
                    if(updateSyncRecordModel==nil)
                    {
                        returnError = PPErrorOperationFailed(returnError);
                        [returnError retain];
                        break;
                    }
                    
                    updateSyncRecordModel.localSyncCompareModel  = self.currentSyncActionModel.localSyncCompareModel;
                    updateSyncRecordModel.remoteSyncCompareModel = self.currentSyncActionModel.remoteSyncCompareModel;
                    
                    //////////////////////////////////////////////////
                    
                    switch(self.currentSyncActionModel.type)
                    {
                        case PPSyncType_Group:
                        {
                            updateSyncRecordModel.version = self.syncGroupVersion;
                            
                            [self updateGroupSyncRecordWithSyncRecordModel:updateSyncRecordModel error:&returnError];
                            
                            break;
                        }
                        case PPSyncType_Card:
                        {
                            updateSyncRecordModel.version = self.syncCardVersion;
                            
                            [self updateCardSyncRecordWithSyncRecordModel:updateSyncRecordModel error:&returnError];
                            
                            break;
                        }
                        default:
                        {
                            updateSyncRecordModel = nil;
                            returnError = PPErrorOperationFailed(returnError);
                            break;
                        }
                    }
                    
                    
                    //////////////////////////////////////////////////
                    // !! 觀察什麼狀況下有變化
                    
                    // MARK: #Dump log
                    if(resultSyncRecordModel != nil && updateSyncRecordModel != nil &&
                       [resultSyncRecordModel isEqual:updateSyncRecordModel] == NO)
                    {
                        [self dumpLogMessageWithFormat:@"U/L:%@", [updateSyncRecordModel localSyncCompareModel]];
                        [self dumpLogMessageWithFormat:@"U/R:%@", [updateSyncRecordModel remoteSyncCompareModel]];
                        
                        if(returnError != nil)
                        {
                            [self dumpLogMessageWithFormat:@"U/RE:%@", returnError];
                            [self dumpLogMessageWithFormat:@"U/OE:%@", originError];
                        }
                    }

                    //////////////////////////////////////////////////
                    
                    if(returnError==originError)
                    {
                        //成功要把重排計數器歸零
                        rescheduleCount = 0;
                        
                        [self executeActionSuccess];
                    }
                    else
                    {
                        [returnError retain];
                        break;
                    }
                }
                
                //////////////////////////////////////////////////
                
                [actionModels removeObject:self.currentSyncActionModel];
                self.currentSyncActionModel = nil;
                
                //////////////////////////////////////////////////
                
                self.currentSyncActionIndex++;
                [self reportProgress:((CGFloat)(self.currentSyncActionIndex))/self.totalSyncActionCount];
            }
            
        } // @autoreleasepool
        
    } // while
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    [returnError autorelease];
    
    //////////////////////////////////////////////////
    
    // MARK: #Dump log
    [self dumpLogMessageWithFormat:@"=================================================="];
    [self dumpLogMessageWithFormat:@"%s out returnError=%@", __FUNCTION__, returnError];

    return (returnError==originError);
}


//==============================================================================
//
//==============================================================================
- (PPSyncActionModel *)forceLocalUpdateActionWithSyncRecordModel:(PPSyncRecordModel *)syncRecordModel type:(PPSyncType)type
{
    PPSyncActionModel *ppSyncActionModel = [[[PPSyncActionModel alloc] init] autorelease];
    if(ppSyncActionModel==nil)
    {
        return ppSyncActionModel;
    }
    
    ppSyncActionModel.localSyncCompareModel     = syncRecordModel.localSyncCompareModel;
    ppSyncActionModel.remoteSyncCompareModel    = syncRecordModel.remoteSyncCompareModel;
    ppSyncActionModel.action                    = (syncRecordModel.localSyncCompareModel.deleted==YES?PPSyncAction_Delete:PPSyncAction_Update);
    ppSyncActionModel.target                    = PPSyncTarget_Local;
    ppSyncActionModel.type                      = type;
    
    return ppSyncActionModel;
}

//==============================================================================
//
//==============================================================================
- (PPSyncActionModel *)forceRemoteUpdateActionWithSyncRecordModel:(PPSyncRecordModel *)syncRecordModel type:(PPSyncType)type
{
    PPSyncActionModel *ppSyncActionModel = [[[PPSyncActionModel alloc] init] autorelease];
    if(ppSyncActionModel==nil)
    {
        return ppSyncActionModel;
    }
    
    ppSyncActionModel.localSyncCompareModel     = syncRecordModel.localSyncCompareModel;
    ppSyncActionModel.remoteSyncCompareModel    = syncRecordModel.remoteSyncCompareModel;
    ppSyncActionModel.action                    = (syncRecordModel.localSyncCompareModel.deleted==YES?PPSyncAction_Delete:PPSyncAction_Update);
    ppSyncActionModel.target                    = PPSyncTarget_Remote;
    ppSyncActionModel.type                      = type;

    return ppSyncActionModel;
}


//==============================================================================
//
//==============================================================================
- (void)filloutForceUpdateActionModels
{
    // 目前只有名片的remote update,
    for (PPSyncActionModel *syncActionModel in self.forceUpdateActionModels)
    {
        if (syncActionModel.type == PPSyncType_Card &&
            syncActionModel.target == PPSyncTarget_Remote &&
            syncActionModel.remoteSyncCompareModel)
        {
            NSError *error = nil;
            
            PPSyncRecordModel * recordModel = [self cardSyncRecordModelWithUniqueID:syncActionModel.remoteSyncCompareModel.uniqueID syncRecordType:PPSyncRecordType_Remote error:&error];
            
            if (recordModel)
            {
                PPSyncCardModel * syncCardModel = [self.localSyncDataController copySyncCardModelWithUniqueID:recordModel.localSyncCompareModel.uniqueID fromVersion:PPSyncVersion_Ignore toVersion:PPSyncVersion_Ignore];
                syncActionModel.localSyncCompareModel = syncCardModel;
                [syncCardModel release];
            }
        }
        
        if (syncActionModel.type == PPSyncType_Card &&
            syncActionModel.target == PPSyncTarget_Local &&
            syncActionModel.localSyncCompareModel)
        {
            NSError *error = nil;
            
            PPSyncRecordModel * recordModel = [self cardSyncRecordModelWithUniqueID:syncActionModel.localSyncCompareModel.uniqueID syncRecordType:PPSyncRecordType_Local error:&error];
            
            if (recordModel)
            {
                PPSyncCardModel * syncCardModel = [self.remoteSyncDataController copySyncCardModelWithUniqueID:recordModel.remoteSyncCompareModel.uniqueID fromVersion:PPSyncVersion_Ignore toVersion:PPSyncVersion_Ignore];
                syncActionModel.remoteSyncCompareModel = syncCardModel;
                [syncCardModel release];
            }
        }
    }
}


//================================================================================
//
//================================================================================
- (NSString *)nameForAction:(PPSyncAction)action
{
    NSString *name = nil;
    
    switch(action)
    {
        case PPSyncAction_Delete:
        {
            name = @"Delete";
            break;
        }
        case PPSyncAction_Insert:
        {
            name = @"Insert";
            break;
        }
        case PPSyncAction_Update:
        {
            name = @"Update";
            break;
        }
        default:
        {
            name = @"Unknown SyncAction";
            break;
        }
    }
    
    return name;
}

//================================================================================
//
//================================================================================
- (NSString *)nameForTarget:(PPSyncTarget)target
{
    NSString *name = nil;
    
    switch(target)
    {
        case PPSyncTarget_Local:
        {
            name = @"Local";
            break;
        }
        case PPSyncTarget_Remote:
        {
            name = @"Remote";
            break;
        }
        default:
        {
            name = @"Unknown SyncTarget";
            break;
        }
    }
    
    return name;
}

//================================================================================
//
//================================================================================
- (NSString *)nameForType:(PPSyncType)type
{
    NSString *name = nil;
    
    switch(type)
    {
        case PPSyncType_Group:
        {
            name = @"Group";
            break;
        }
        case PPSyncType_Card:
        {
            name = @"Card";
            break;
        }
        default:
        {
            name = @"Unknown SyncType";
            break;
        }
    }
    
    return name;
}

//================================================================================
//
//================================================================================
- (void)reportProgress:(CGFloat)progress
{
    if(self.delegate!=nil && [self.delegate respondsToSelector:@selector(ppSyncActionController:progress:)]==YES)
    {
        [self.delegate ppSyncActionController:self progress:progress];
    }
}

//================================================================================
//
//================================================================================
- (void)reportStep:(PPSyncActionStep)step
{
    self.step = step;
    
    //////////////////////////////////////////////////
    
    if(self.delegate!=nil && [self.delegate respondsToSelector:@selector(ppSyncActionController:step:)]==YES)
    {
        [self.delegate ppSyncActionController:self step:step];
    }
}

//================================================================================
//
//================================================================================
- (NSArray *)syncActionModelsWithLocalSyncCompareModels:(NSMutableArray *)localSyncCompareModels
                                remoteSyncCompareModels:(NSMutableArray *)remoteSyncCompareModels
                                                   type:(PPSyncType)type
                                              afterDate:(NSDate *)afetrDate
                                forceUpdateActionModels:(NSMutableArray *)forceUpdateActionModels
                                                  error:(NSError **)error
{
    NSMutableArray  *syncActionModels           = nil;
    NSError         *originError                = ((error!=nil)?*error:nil);
    NSError         *returnError                = originError;
    
    //////////////////////////////////////////////////
    
    do
    {
        if(localSyncCompareModels==nil || remoteSyncCompareModels==nil)
        {
            returnError = PPErrorParameterInvalidity(returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        syncActionModels = [[[NSMutableArray alloc] init] autorelease];
        if(syncActionModels==nil)
        {
            returnError = PPErrorOperationFailed(returnError);
            break;
        }

        //////////////////////////////////////////////////
        //取得同步記錄

        NSMutableArray *syncRecordModels = nil;
        
        if(type==PPSyncType_Group)
        {
            [self reportStep:PPSyncActionStep_CreateSyncGroupActions];
            
            syncRecordModels = [self groupSyncRecordModelsWithError:&returnError];
            if(returnError!=originError)
            {
                break;
            }
        }
        else if(type==PPSyncType_Card)
        {
            [self reportStep:PPSyncActionStep_CreateSyncCardActions];
            
            syncRecordModels = [self cardSyncRecordModelsWithError:&returnError];
            if(returnError!=originError)
            {
                break;
            }
        }
        else
        {
            returnError = PPErrorParameterInvalidity(returnError);
            break;
        }
        
        // 回報進度
        [self reportProgress:0.1];

        //////////////////////////////////////////////////
        //比對同步記錄產生動作列表
        
//        NSLog(@"remoteSyncCompareModels:%@", remoteSyncCompareModels);

        for(PPSyncRecordModel *syncRecordModel in syncRecordModels)
        {
            BOOL localFind      = NO;
            BOOL localEqual     = NO;
            BOOL remoteFind     = NO;
            BOOL remoteEqual    = NO;
            
            //////////////////////////////////////////////////
            
            for(PPSyncCompareModel *localSyncCompareModel in localSyncCompareModels)
            {
                if([syncRecordModel.localSyncCompareModel.uniqueID isEqualToString:localSyncCompareModel.uniqueID]==YES)
                {
                    syncRecordModel.localSyncCompareModel.needForceUpdate = localSyncCompareModel.needForceUpdate;

                    //////////////////////////////////////////////////
                    // 這裡先處理有在sync record 中的，不管是remote update或是local update, 都要檢查是否強制更新
                    if (localSyncCompareModel.deleted==NO)
                    {
                        if (self.needAddForceUpdate && localSyncCompareModel.needForceUpdate)
                        {
                            PPSyncActionModel *ppSyncActionModel = [self forceLocalUpdateActionWithSyncRecordModel:syncRecordModel type:type];
                            if(ppSyncActionModel==nil)
                            {
                                returnError = PPErrorOperationFailed(returnError);
                                break;
                            }
                            
                            [forceUpdateActionModels addObject:ppSyncActionModel];
                        }
                    }
                    //////////////////////////////////////////////////
                    localFind = YES;
                    
                    //TimeInterval精確度只到milliSecond,小數點後三位
                    if(fabs([syncRecordModel.localSyncCompareModel.lastModifiedDate timeIntervalSince1970]-[localSyncCompareModel.lastModifiedDate timeIntervalSince1970])<0.001)
                    {
                        localEqual = YES;
                    }
                    
                    syncRecordModel.localSyncCompareModel = localSyncCompareModel;
                    
                    [localSyncCompareModels removeObject:localSyncCompareModel];
                    
                    break;
                }
            }
            
            if(localFind==NO)
            {
                if(afetrDate==nil)
                {
                    //全部記錄找不到,代表被刪除
                    syncRecordModel.localSyncCompareModel.deleted = YES;
                }
                else
                {
                    //部分記錄找不到,代表無異動
                    localEqual = YES;
                }
            }
            
            //////////////////////////////////////////////////
            
            for(PPSyncCompareModel *remoteSyncCompareModel in remoteSyncCompareModels)
            {
                if([syncRecordModel.remoteSyncCompareModel.uniqueID isEqualToString:remoteSyncCompareModel.uniqueID]==YES)
                {
                    syncRecordModel.remoteSyncCompareModel.needForceUpdate = remoteSyncCompareModel.needForceUpdate;
                    //////////////////////////////////////////////////
                    // 這裡先處理有在sync record 中的，不管是remote update或是local update, 都要檢查是否強制更新
                    if (remoteSyncCompareModel.deleted==NO)
                    {
                        if (self.needAddForceUpdate && remoteSyncCompareModel.needForceUpdate)
                        {
                            PPSyncActionModel *ppSyncActionModel = [self forceRemoteUpdateActionWithSyncRecordModel:syncRecordModel type:type];
                            if(ppSyncActionModel==nil)
                            {
                                returnError = PPErrorOperationFailed(returnError);
                                break;
                            }
                            
                            [forceUpdateActionModels addObject:ppSyncActionModel];
                        }                        
                    }
                    //////////////////////////////////////////////////
                    remoteFind = YES;
                    
                    //TimeInterval精確度只到milliSecond,小數點後三位
                    if(fabs([syncRecordModel.remoteSyncCompareModel.lastModifiedDate timeIntervalSince1970]-[remoteSyncCompareModel.lastModifiedDate timeIntervalSince1970])<0.001)
                    {
                        remoteEqual = YES;
                    }
                    
                    syncRecordModel.remoteSyncCompareModel = remoteSyncCompareModel;
                    
                    [remoteSyncCompareModels removeObject:remoteSyncCompareModel];
                    
                    break;
                }
            }
            
            if(remoteFind==NO)
            {
                if(afetrDate==nil)
                {
                    //全部記錄找不到,代表被刪除
                    syncRecordModel.remoteSyncCompareModel.deleted = YES;
                }
                else
                {
                    //部分記錄找不到,代表無異動
                    remoteEqual = YES;
                }
            }
            
            //////////////////////////////////////////////////
            
            if(localFind==YES)
            {
                if(localEqual==YES)
                {
                    if(remoteFind==YES)
                    {
                        if(remoteEqual==YES)
                        {
                            //(YES,YES,YES,YES)沒有異動
                        }
                        else
                        {
                            //(YES,YES,YES,NO)遠端異動(更新or刪除)
                            PPSyncActionModel *ppSyncActionModel = [[[PPSyncActionModel alloc] init] autorelease];
                            if(ppSyncActionModel==nil)
                            {
                                returnError = PPErrorOperationFailed(returnError);
                                break;
                            }
                            
                            ppSyncActionModel.localSyncCompareModel     = syncRecordModel.localSyncCompareModel;
                            ppSyncActionModel.remoteSyncCompareModel    = syncRecordModel.remoteSyncCompareModel;
                            ppSyncActionModel.action                    = (syncRecordModel.remoteSyncCompareModel.deleted==YES?PPSyncAction_Delete:PPSyncAction_Update);
                            ppSyncActionModel.target                    = PPSyncTarget_Local;
                            ppSyncActionModel.type                      = type;
                            
                            [syncActionModels addObject:ppSyncActionModel];
                        }
                    }
                    else
                    {
                        if(remoteEqual==YES)
                        {
                            //(YES,YES,NO,YES)沒有異動
                        }
                        else
                        {
                            //(YES,YES,NO,NO)遠端異動(刪除)理論上只有刪除的狀況
                            PPSyncActionModel *ppSyncActionModel = [[[PPSyncActionModel alloc] init] autorelease];
                            if(ppSyncActionModel==nil)
                            {
                                returnError = PPErrorOperationFailed(returnError);
                                break;
                            }
                            
                            ppSyncActionModel.localSyncCompareModel     = syncRecordModel.localSyncCompareModel;
                            ppSyncActionModel.remoteSyncCompareModel    = syncRecordModel.remoteSyncCompareModel;
                            ppSyncActionModel.action                    = (syncRecordModel.remoteSyncCompareModel.deleted==YES?PPSyncAction_Delete:PPSyncAction_Update);
                            ppSyncActionModel.target                    = PPSyncTarget_Local;
                            ppSyncActionModel.type                      = type;
                            
                            [syncActionModels addObject:ppSyncActionModel];
                        }
                    }
                }
                else
                {
                    if(remoteFind==YES)
                    {
                        if(remoteEqual==YES)
                        {
                            //(YES,NO,YES,YES)本地異動(更新or刪除)
                            PPSyncActionModel *ppSyncActionModel = [[[PPSyncActionModel alloc] init] autorelease];
                            if(ppSyncActionModel==nil)
                            {
                                returnError = PPErrorOperationFailed(returnError);
                                break;
                            }
                            
                            ppSyncActionModel.localSyncCompareModel     = syncRecordModel.localSyncCompareModel;
                            ppSyncActionModel.remoteSyncCompareModel    = syncRecordModel.remoteSyncCompareModel;
                            ppSyncActionModel.action                    = (syncRecordModel.localSyncCompareModel.deleted==YES?PPSyncAction_Delete:PPSyncAction_Update);
                            ppSyncActionModel.target                    = PPSyncTarget_Remote;
                            ppSyncActionModel.type                      = type;
                            
                            [syncActionModels addObject:ppSyncActionModel];
                        }
                        else
                        {
                            //(YES,NO,YES,NO)雙邊異動(衝突處理)
                            PPSyncActionModel *ppSyncActionModel = [self syncActionModelWithConflictSyncRecordModel:syncRecordModel type:type error:&returnError];
                            if(returnError!=originError)
                            {
                                break;
                            }
                            
                            if(ppSyncActionModel!=nil)
                            {
                                [syncActionModels addObject:ppSyncActionModel];
                            }
                        }
                    }
                    else
                    {
                        if(remoteEqual==YES)
                        {
                            //(YES,NO,NO,YES)本地異動(更新or刪除)
                            PPSyncActionModel *ppSyncActionModel = [[[PPSyncActionModel alloc] init] autorelease];
                            if(ppSyncActionModel==nil)
                            {
                                returnError = PPErrorOperationFailed(returnError);
                                break;
                            }
                            
                            ppSyncActionModel.localSyncCompareModel     = syncRecordModel.localSyncCompareModel;
                            ppSyncActionModel.remoteSyncCompareModel    = syncRecordModel.remoteSyncCompareModel;
                            ppSyncActionModel.action                    = (syncRecordModel.localSyncCompareModel.deleted==YES?PPSyncAction_Delete:PPSyncAction_Update);
                            ppSyncActionModel.target                    = PPSyncTarget_Remote;
                            ppSyncActionModel.type                      = type;
                            
                            [syncActionModels addObject:ppSyncActionModel];
                        }
                        else
                        {
                            //(YES,NO,NO,NO)雙邊異動(衝突處理)
                            PPSyncActionModel *ppSyncActionModel = [self syncActionModelWithConflictSyncRecordModel:syncRecordModel type:type error:&returnError];
                            if(returnError!=originError)
                            {
                                break;
                            }
                            
                            if(ppSyncActionModel!=nil)
                            {
                                [syncActionModels addObject:ppSyncActionModel];
                            }
                        }
                    }
                }
            }
            else
            {
                if(localEqual==YES)
                {
                    if(remoteFind==YES)
                    {
                        if(remoteEqual==YES)
                        {
                            //(NO,YES,YES,YES)沒有異動
                        }
                        else
                        {
                            //(NO,YES,YES,NO)遠端異動(更新or刪除)
                            PPSyncActionModel *ppSyncActionModel = [[[PPSyncActionModel alloc] init] autorelease];
                            if(ppSyncActionModel==nil)
                            {
                                returnError = PPErrorOperationFailed(returnError);
                                break;
                            }
                            
                            ppSyncActionModel.localSyncCompareModel     = syncRecordModel.localSyncCompareModel;
                            ppSyncActionModel.remoteSyncCompareModel    = syncRecordModel.remoteSyncCompareModel;
                            ppSyncActionModel.action                    = (syncRecordModel.remoteSyncCompareModel.deleted==YES?PPSyncAction_Delete:PPSyncAction_Update);
                            ppSyncActionModel.target                    = PPSyncTarget_Local;
                            ppSyncActionModel.type                      = type;
                            
                            [syncActionModels addObject:ppSyncActionModel];
                        }
                    }
                    else
                    {
                        if(remoteEqual==YES)
                        {
                            //(NO,YES,NO,YES)沒有異動
                        }
                        else
                        {
                            //(NO,YES,NO,NO)遠端異動(刪除)理論上不會有這種狀況
                            PPSyncActionModel *ppSyncActionModel = [[[PPSyncActionModel alloc] init] autorelease];
                            if(ppSyncActionModel==nil)
                            {
                                returnError = PPErrorOperationFailed(returnError);
                                break;
                            }
                            
                            ppSyncActionModel.localSyncCompareModel     = syncRecordModel.localSyncCompareModel;
                            ppSyncActionModel.remoteSyncCompareModel    = syncRecordModel.remoteSyncCompareModel;
                            ppSyncActionModel.action                    = (syncRecordModel.remoteSyncCompareModel.deleted==YES?PPSyncAction_Delete:PPSyncAction_Update);
                            ppSyncActionModel.target                    = PPSyncTarget_Local;
                            ppSyncActionModel.type                      = type;
                            
                            [syncActionModels addObject:ppSyncActionModel];
                        }
                    }
                }
                else
                {
                    if(remoteFind==YES)
                    {
                        if(remoteEqual==YES)
                        {
                            //(NO,NO,YES,YES)本地異動(更新or刪除)理論上只有刪除的狀況
                            PPSyncActionModel *ppSyncActionModel = [[[PPSyncActionModel alloc] init] autorelease];
                            if(ppSyncActionModel==nil)
                            {
                                returnError = PPErrorOperationFailed(returnError);
                                break;
                            }
                            
                            ppSyncActionModel.localSyncCompareModel     = syncRecordModel.localSyncCompareModel;
                            ppSyncActionModel.remoteSyncCompareModel    = syncRecordModel.remoteSyncCompareModel;
                            ppSyncActionModel.action                    = (syncRecordModel.localSyncCompareModel.deleted==YES?PPSyncAction_Delete:PPSyncAction_Update);
                            ppSyncActionModel.target                    = PPSyncTarget_Remote;
                            ppSyncActionModel.type                      = type;
                            
                            [syncActionModels addObject:ppSyncActionModel];
                        }
                        else
                        {
                            //(NO,NO,YES,NO)雙邊異動(衝突處理)
                            PPSyncActionModel *ppSyncActionModel = [self syncActionModelWithConflictSyncRecordModel:syncRecordModel type:type error:&returnError];
                            if(returnError!=originError)
                            {
                                break;
                            }
                            
                            if(ppSyncActionModel!=nil)
                            {
                                [syncActionModels addObject:ppSyncActionModel];
                            }
                        }
                    }
                    else
                    {
                        if(remoteEqual==YES)
                        {
                            //(NO,NO,NO,YES)本地異動(更新or刪除)理論上只有刪除的狀況
                            PPSyncActionModel *ppSyncActionModel = [[[PPSyncActionModel alloc] init] autorelease];
                            if(ppSyncActionModel==nil)
                            {
                                returnError = PPErrorOperationFailed(returnError);
                                break;
                            }
                            
                            ppSyncActionModel.localSyncCompareModel     = syncRecordModel.localSyncCompareModel;
                            ppSyncActionModel.remoteSyncCompareModel    = syncRecordModel.remoteSyncCompareModel;
                            ppSyncActionModel.action                    = (syncRecordModel.localSyncCompareModel.deleted==YES?PPSyncAction_Delete:PPSyncAction_Update);
                            ppSyncActionModel.target                    = PPSyncTarget_Remote;
                            ppSyncActionModel.type                      = type;
                            
                            [syncActionModels addObject:ppSyncActionModel];
                        }
                        else
                        {
                            //(NO,NO,NO,NO)雙邊異動(衝突處理)
                            PPSyncActionModel *ppSyncActionModel = [self syncActionModelWithConflictSyncRecordModel:syncRecordModel type:type error:&returnError];
                            if(returnError!=originError)
                            {
                                break;
                            }
                            
                            if(ppSyncActionModel!=nil)
                            {
                                [syncActionModels addObject:ppSyncActionModel];
                            }
                        }
                    }
                }
            }
            
            //////////////////////////////////////////////////
            
            if(self.cancel==YES || returnError!=originError)
            {
                break;
            }
        }

        // 回報進度
        [self reportProgress:0.6];
        
        //////////////////////////////////////////////////
        
        if(self.cancel==YES || returnError!=originError)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(type==PPSyncType_Group)
        {
            // !! Group要比對名字是否一樣
            
            for(PPSyncGroupModel *remoteSyncGroupModel in remoteSyncCompareModels)
            {
                if(remoteSyncGroupModel.deleted==NO)
                {
                    BOOL find = NO;
                    
                    for(PPSyncGroupModel *localSyncGroupModel in localSyncCompareModels)
                    {
                        if(localSyncGroupModel.deleted==NO && remoteSyncGroupModel.defaultGroup==localSyncGroupModel.defaultGroup && [remoteSyncGroupModel.name isEqual:localSyncGroupModel.name]==YES)
                        {
                            find = YES;
                            
                            //有找到名字一樣的對應,直接寫入同步記錄檔
                            PPSyncRecordModel *ppSyncRecordModel = [[[PPSyncRecordModel alloc] init] autorelease];
                            if(ppSyncRecordModel==nil)
                            {
                                returnError = PPErrorOperationFailed(returnError);
                                break;
                            }
                            
                            ppSyncRecordModel.localSyncCompareModel     = localSyncGroupModel;
                            ppSyncRecordModel.remoteSyncCompareModel    = remoteSyncGroupModel;
                            ppSyncRecordModel.version                   = self.syncGroupVersion;
                            
                            if([self insertGroupSyncRecordWithSyncRecordModel:ppSyncRecordModel error:&returnError]==NO)
                            {
                                break;
                            }
                            
                            [localSyncCompareModels removeObject:localSyncGroupModel];
                            
                            break;
                        }
                        
                        //////////////////////////////////////////////////
                        
                        if(self.cancel==YES)
                        {
                            break;
                        }
                    }
                    
                    //////////////////////////////////////////////////
                    
                    if(self.cancel==YES || returnError!=originError)
                    {
                        break;
                    }
                    
                    //////////////////////////////////////////////////
                    
                    if(find==NO)
                    {
                        //沒找到名字一樣的對應,要產生PPSyncActionModel
                        PPSyncActionModel *ppSyncActionModel = [[[PPSyncActionModel alloc] init] autorelease];
                        if(ppSyncActionModel==nil)
                        {
                            returnError = PPErrorOperationFailed(returnError);
                            break;
                        }
                        
                        ppSyncActionModel.remoteSyncCompareModel    = remoteSyncGroupModel;
                        ppSyncActionModel.action                    = PPSyncAction_Insert;
                        ppSyncActionModel.target                    = PPSyncTarget_Local;
                        ppSyncActionModel.type                      = type;
                        
                        [syncActionModels addObject:ppSyncActionModel];
                    }
                }
                
                //////////////////////////////////////////////////
                
                if(self.cancel==YES || returnError!=originError)
                {
                    break;
                }
            }
            
            //////////////////////////////////////////////////
            
            if(self.cancel==YES || returnError!=originError)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            //沒找到對應的本地Group資料要加入遠端
            
            for(PPSyncCompareModel *localSyncCompareModel in localSyncCompareModels)
            {
                if(localSyncCompareModel.deleted==NO)
                {
                    PPSyncActionModel *ppSyncActionModel = [[[PPSyncActionModel alloc] init] autorelease];
                    if(ppSyncActionModel==nil)
                    {
                        returnError = PPErrorOperationFailed(returnError);
                        break;
                    }
                    
                    ppSyncActionModel.localSyncCompareModel = localSyncCompareModel;
                    ppSyncActionModel.action                = PPSyncAction_Insert;
                    ppSyncActionModel.target                = PPSyncTarget_Remote;
                    ppSyncActionModel.type                  = type;
                    
                    [syncActionModels addObject:ppSyncActionModel];
                }
                
                //////////////////////////////////////////////////
                
                if(self.cancel==YES || returnError!=originError)
                {
                    break;
                }
            }
        }
        else if(type==PPSyncType_Card)
        {
            //遠端Card資料要加入本地
            
            for(PPSyncCompareModel *remoteSyncCompareModel in remoteSyncCompareModels)
            {
                if(remoteSyncCompareModel.deleted==NO)
                {
                    PPSyncActionModel *ppSyncActionModel = [[[PPSyncActionModel alloc] init] autorelease];
                    if(ppSyncActionModel==nil)
                    {
                        returnError = PPErrorOperationFailed(returnError);
                        break;
                    }
                    
                    ppSyncActionModel.remoteSyncCompareModel    = remoteSyncCompareModel;
                    ppSyncActionModel.action                    = PPSyncAction_Insert;
                    ppSyncActionModel.target                    = PPSyncTarget_Local;
                    ppSyncActionModel.type                      = type;
                    
                    [syncActionModels addObject:ppSyncActionModel];
                    
                    //////////////////////////////////////////////////
                    // !!如果是remote新增到local，也要再檢查是否要強制更新回remote,
                    if (self.needAddForceUpdate && remoteSyncCompareModel.needForceUpdate)
                    {
                        PPSyncActionModel *ppSyncActionModel = [[[PPSyncActionModel alloc] init] autorelease];
                        if(ppSyncActionModel==nil)
                        {
                            returnError = PPErrorOperationFailed(returnError);
                            break;
                        }
                        
                        ppSyncActionModel.remoteSyncCompareModel    = remoteSyncCompareModel;
                        ppSyncActionModel.action                    = PPSyncAction_Update;
                        ppSyncActionModel.target                    = PPSyncTarget_Remote;
                        ppSyncActionModel.type                      = type;
                        
                        [forceUpdateActionModels addObject:ppSyncActionModel];
                    }
                }
                
                //////////////////////////////////////////////////
                
                if(self.cancel==YES)
                {
                    break;
                }
            }
            
            //////////////////////////////////////////////////
            
            if(self.cancel==YES || returnError!=originError)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            //本地Card資料要加入遠端
            
            for(PPSyncCompareModel *localSyncCompareModel in localSyncCompareModels)
            {
                if(localSyncCompareModel.deleted==NO)
                {
                    PPSyncActionModel *ppSyncActionModel = [[[PPSyncActionModel alloc] init] autorelease];
                    if(ppSyncActionModel==nil)
                    {
                        returnError = PPErrorOperationFailed(returnError);
                        break;
                    }
                    
                    ppSyncActionModel.localSyncCompareModel = localSyncCompareModel;
                    ppSyncActionModel.action                = PPSyncAction_Insert;
                    ppSyncActionModel.target                = PPSyncTarget_Remote;
                    ppSyncActionModel.type                  = type;
                    
                    [syncActionModels addObject:ppSyncActionModel];
                }
                
                //////////////////////////////////////////////////
                
                if(self.cancel==YES)
                {
                    break;
                }
            }
            
            //////////////////////////////////////////////////
            
            if(self.cancel==YES || returnError!=originError)
            {
                break;
            }
            
            
            //////////////////////////////////////////////////
            // 檢查是否有全部刪除的動作
            
            if(syncRecordModels!=nil)
            {
                NSUInteger localDeleteCount = 0;
                NSUInteger remoteDeleteCount = 0;
                
                for(PPSyncActionModel *syncActionModel in syncActionModels)
                {
                    if(syncActionModel.type == PPSyncType_Card && syncActionModel.action == PPSyncAction_Delete)
                    {
                        if(syncActionModel.target == PPSyncTarget_Local)
                        {
                            localDeleteCount++;
                        }
                        else
                        {
                            remoteDeleteCount++;
                        }
                    }
                }
                
                // !! 原本只要任一邊刪除的名片數超過存在的名片數就要發出通知，改為檢查source名片資料是否為0。
                //    (如果在同步新增名片到target的過程中刪除source名片，因為target會略過被刪掉的名片，再同
                //     步就有可能會出現target刪除數量超過存在數量的狀況，這時發出delete all的通知是錯誤的。)
                if(localDeleteCount > 0)
                {
                    NSNumber *remoteCardCount = [self.remoteSyncDataController totalSyncCardsCount];
                    
                    // 當remote全部刪除才需要發出通知
                    if(remoteCardCount != nil && [remoteCardCount integerValue] == 0)
                    {
                        returnError = PPErrorMake(NSErrorPPSyncActionController_Code_LocalDeleteAllWarnning, @"Local Delete All Warnning", returnError);
                    }

                    // 原做法
//                    NSNumber *localCardCount = [self.localSyncDataController totalSyncCardsCount];
//
//                    // nil是發生錯誤，先跳過
//                    if(localCardCount != nil)
//                    {
//                        if(localDeleteCount >= [localCardCount unsignedIntegerValue])
//                        {
//                            returnError = PPErrorMake(NSErrorPPSyncActionController_Code_LocalDeleteAllWarnning, @"Local Delete All Warnning", returnError);
//                        }
//                    }
                }
                else if(remoteDeleteCount > 0)
                {
                    NSNumber *localCardCount = [self.localSyncDataController totalSyncCardsCount];
                    
                    // 當local全部刪除才需要發出通知
                    if(localCardCount != nil && [localCardCount integerValue] == 0)
                    {
                        returnError = PPErrorMake(NSErrorPPSyncActionController_Code_RemoteDeleteAllWarnning, @"Remote Delete All Warnning", returnError);
                    }

                    // 原做法
//                    NSNumber *remoteCardCount = [self.remoteSyncDataController totalSyncCardsCount];
//
//                    // nil是發生錯誤，先跳過
//                    if(remoteCardCount != nil)
//                    {
//                        if(remoteDeleteCount >= [remoteCardCount unsignedIntegerValue])
//                        {
//                            returnError = PPErrorMake(NSErrorPPSyncActionController_Code_RemoteDeleteAllWarnning, @"Remote Delete All Warnning", returnError);
//                        }
//                    }
                }
            }
        }
        
        // 回報進度
        [self reportProgress:1.0];

//        NSLog(@"forceUpdateActionModels:%@", forceUpdateActionModels);

    }while(0);
    
    //////////////////////////////////////////////////
    
    if(self.cancel==YES)
    {
        if(returnError==nil || [returnError.domain isEqualToString:NSStringFromClass([self class])]==NO || returnError.code!=NSErrorCustom_Code_OperationCancel)
        {
            returnError = PPErrorOperationCancel(returnError);
        }
    }
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return syncActionModels;
}

//================================================================================
//
//================================================================================
- (PPSyncActionModel *)syncActionModelWithConflictSyncRecordModel:(PPSyncRecordModel *)syncRecordModel type:(PPSyncType)type error:(NSError **)error
{
    PPSyncActionModel   *ppSyncActionModel  = nil;
    NSError             *originError        = ((error!=nil)?*error:nil);
    NSError             *returnError        = originError;
    
    //////////////////////////////////////////////////
    
    do
    {
        if(syncRecordModel==nil)
        {
            returnError = PPErrorParameterInvalidity(returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        ppSyncActionModel = [[[PPSyncActionModel alloc] init] autorelease];
        if(ppSyncActionModel==nil)
        {
            returnError = PPErrorOperationFailed(returnError);
            break;
        }
        
        ppSyncActionModel.localSyncCompareModel     = syncRecordModel.localSyncCompareModel;
        ppSyncActionModel.remoteSyncCompareModel    = syncRecordModel.remoteSyncCompareModel;
        ppSyncActionModel.type                      = type;
        
        //////////////////////////////////////////////////
        
        //衝突處理
        //(YES,NO,YES,NO)
        //(YES,NO,NO,NO)
        //(NO,NO,YES,NO)
        //(NO,NO,NO,NO)
        
        switch(self.conflict)
        {
            case PPSyncActionConflict_LocalFirst:
            {
                ppSyncActionModel.target = PPSyncTarget_Remote;
                
                //////////////////////////////////////////////////
                
                if(syncRecordModel.localSyncCompareModel.deleted==YES)
                {
                    if(syncRecordModel.remoteSyncCompareModel.deleted==YES)
                    {
                        //本地資料不存在,遠端資料不存在,直接刪掉同步記錄
                        if(type==PPSyncType_Group)
                        {
                            [self deleteGroupSyncRecordWithUniqueID:syncRecordModel.localSyncCompareModel.uniqueID
                                                     syncRecordType:PPSyncRecordType_Local
                                                              error:&returnError];
                        }
                        else if(type==PPSyncType_Card)
                        {
                            [self deleteCardSyncRecordWithUniqueID:syncRecordModel.localSyncCompareModel.uniqueID
                                                    syncRecordType:PPSyncRecordType_Local
                                                             error:&returnError];
                        }
                        else
                        {
                            returnError = PPErrorOperationFailed(returnError);
                        }
                        
                        ppSyncActionModel = nil;
                    }
                    else
                    {
                        //本地資料不存在,遠端資料存在,刪除遠端資料
                        ppSyncActionModel.action = PPSyncAction_Delete;
                    }
                }
                else
                {
                    if(syncRecordModel.remoteSyncCompareModel.deleted==YES)
                    {
                        //本地資料存在,遠端資料不存在,新增遠端資料
                        ppSyncActionModel.action = PPSyncAction_Insert;
                    }
                    else
                    {
                        //本地資料存在,遠端資料存在,更新遠端資料
                        ppSyncActionModel.action = PPSyncAction_Update;
                    }
                }
                
                break;
            }
            case PPSyncActionConflict_RemoteFirst:
            {
                ppSyncActionModel.target = PPSyncTarget_Local;
                
                //////////////////////////////////////////////////
                
                if(syncRecordModel.remoteSyncCompareModel.deleted==YES)
                {
                    if(syncRecordModel.localSyncCompareModel.deleted==YES)
                    {
                        //遠端資料不存在,本地資料不存在,直接刪掉同步記錄
                        if(type==PPSyncType_Group)
                        {
                            [self deleteGroupSyncRecordWithUniqueID:syncRecordModel.localSyncCompareModel.uniqueID
                                                     syncRecordType:PPSyncRecordType_Local
                                                              error:&returnError];
                        }
                        else if(type==PPSyncType_Card)
                        {
                            [self deleteCardSyncRecordWithUniqueID:syncRecordModel.localSyncCompareModel.uniqueID
                                                    syncRecordType:PPSyncRecordType_Local
                                                             error:&returnError];
                        }
                        else
                        {
                            returnError = PPErrorOperationFailed(returnError);
                        }
                        
                        ppSyncActionModel = nil;
                    }
                    else
                    {
                        //遠端資料不存在,本地資料存在,刪除本地資料
                        ppSyncActionModel.action = PPSyncAction_Delete;
                    }
                }
                else
                {
                    if(syncRecordModel.localSyncCompareModel.deleted==YES)
                    {
                        //遠端資料存在,本地資料不存在,新增本地資料
                        ppSyncActionModel.action = PPSyncAction_Insert;
                    }
                    else
                    {
                        //遠端資料存在,本地資料存在,更新本地資料
                        ppSyncActionModel.action = PPSyncAction_Update;
                    }
                }
                
                break;
            }
                
            case PPSyncActionConflict_DeleteFirst:
            {
                if(syncRecordModel.localSyncCompareModel.deleted==YES)
                {
                    if(syncRecordModel.remoteSyncCompareModel.deleted==YES)
                    {
                        //本地資料不存在,遠端資料不存在,直接刪掉同步記錄
                        if(type==PPSyncType_Group)
                        {
                            [self deleteGroupSyncRecordWithUniqueID:syncRecordModel.localSyncCompareModel.uniqueID
                                                     syncRecordType:PPSyncRecordType_Local
                                                              error:&returnError];
                        }
                        else if(type==PPSyncType_Card)
                        {
                            [self deleteCardSyncRecordWithUniqueID:syncRecordModel.localSyncCompareModel.uniqueID
                                                    syncRecordType:PPSyncRecordType_Local
                                                             error:&returnError];
                        }
                        else
                        {
                            returnError = PPErrorOperationFailed(returnError);
                        }
                        
                        ppSyncActionModel = nil;
                    }
                    else
                    {
                        //本地資料不存在,遠端資料存在,刪除遠端資料
                        ppSyncActionModel.action = PPSyncAction_Delete;
                        ppSyncActionModel.target = PPSyncTarget_Remote;
                    }
                }
                else
                {
                    if(syncRecordModel.remoteSyncCompareModel.deleted==YES)
                    {
                        //本地資料存在,遠端資料不存在,刪除本地資料
                        ppSyncActionModel.action = PPSyncAction_Delete;
                        ppSyncActionModel.target = PPSyncTarget_Local;
                   }
                    else
                    {
                        //本地資料存在,遠端資料存在,比時間
                        switch([syncRecordModel.localSyncCompareModel.lastModifiedDate compare:syncRecordModel.remoteSyncCompareModel.lastModifiedDate])
                        {
                            case NSOrderedAscending:
                            {
                                //遠端比較新,更新本地資料
                                ppSyncActionModel.action = PPSyncAction_Update;
                                ppSyncActionModel.target = PPSyncTarget_Local;
                                
                                break;
                            }
                            case NSOrderedDescending:
                            {
                                //本地比較新,更新遠端資料
                                ppSyncActionModel.action = PPSyncAction_Update;
                                ppSyncActionModel.target = PPSyncTarget_Remote;
                                
                                break;
                            }
                            case NSOrderedSame:
                            default:
                            {
                                //無法區分,沒有任何動作
                                ppSyncActionModel = nil;
                                
                                break;
                            }
                        }
                    }
                }
                
                break;
            }

            case PPSyncActionConflict_ModifiedTimeFirst:
            default:
            {
                if(syncRecordModel.localSyncCompareModel.deleted==YES)
                {
                    if(syncRecordModel.remoteSyncCompareModel.deleted==YES)
                    {
                        //本地資料不存在,遠端資料不存在,直接刪掉同步記錄
                        if(type==PPSyncType_Group)
                        {
                            [self deleteGroupSyncRecordWithUniqueID:syncRecordModel.localSyncCompareModel.uniqueID
                                                     syncRecordType:PPSyncRecordType_Local
                                                              error:&returnError];
                        }
                        else if(type==PPSyncType_Card)
                        {
                            [self deleteCardSyncRecordWithUniqueID:syncRecordModel.localSyncCompareModel.uniqueID
                                                    syncRecordType:PPSyncRecordType_Local
                                                             error:&returnError];
                        }
                        else
                        {
                            returnError = PPErrorOperationFailed(returnError);
                        }
                        
                        ppSyncActionModel = nil;
                    }
                    else
                    {
                        //本地資料不存在,遠端資料存在,比時間
                        switch([syncRecordModel.localSyncCompareModel.lastModifiedDate compare:syncRecordModel.remoteSyncCompareModel.lastModifiedDate])
                        {
                            case NSOrderedAscending:
                            {
                                //遠端比較新,新增本地資料
                                ppSyncActionModel.action    = PPSyncAction_Insert;
                                ppSyncActionModel.target    = PPSyncTarget_Local;
                                
                                break;
                            }
                            case NSOrderedDescending:
                            {
                                //本地比較新,刪除遠端資料
                                ppSyncActionModel.action    = PPSyncAction_Delete;
                                ppSyncActionModel.target    = PPSyncTarget_Remote;
                                
                                break;
                            }
                            case NSOrderedSame:
                            default:
                            {
                                //保留資料為原則,新增本地資料
                                ppSyncActionModel.action    = PPSyncAction_Insert;
                                ppSyncActionModel.target    = PPSyncTarget_Local;
                                
                                break;
                            }
                        }
                    }
                }
                else
                {
                    if(syncRecordModel.remoteSyncCompareModel.deleted==YES)
                    {
                        //本地資料存在,遠端資料不存在,比時間
                        switch([syncRecordModel.localSyncCompareModel.lastModifiedDate compare:syncRecordModel.remoteSyncCompareModel.lastModifiedDate])
                        {
                            case NSOrderedAscending:
                            {
                                //遠端比較新,刪除本地資料
                                ppSyncActionModel.action    = PPSyncAction_Delete;
                                ppSyncActionModel.target    = PPSyncTarget_Local;
                                
                                break;
                            }
                            case NSOrderedDescending:
                            {
                                //本地比較新,新增遠端資料
                                ppSyncActionModel.action    = PPSyncAction_Insert;
                                ppSyncActionModel.target    = PPSyncTarget_Remote;
                                
                                break;
                            }
                            case NSOrderedSame:
                            default:
                            {
                                //保留資料為原則,新增遠端資料
                                ppSyncActionModel.action    = PPSyncAction_Insert;
                                ppSyncActionModel.target    = PPSyncTarget_Remote;
                                
                                break;
                            }
                        }
                    }
                    else
                    {
                        //本地資料存在,遠端資料存在,比時間
                        switch([syncRecordModel.localSyncCompareModel.lastModifiedDate compare:syncRecordModel.remoteSyncCompareModel.lastModifiedDate])
                        {
                            case NSOrderedAscending:
                            {
                                //遠端比較新,更新本地資料
                                ppSyncActionModel.action    = PPSyncAction_Update;
                                ppSyncActionModel.target    = PPSyncTarget_Local;
                                
                                break;
                            }
                            case NSOrderedDescending:
                            {
                                //本地比較新,更新遠端資料
                                ppSyncActionModel.action    = PPSyncAction_Update;
                                ppSyncActionModel.target    = PPSyncTarget_Remote;
                                
                                break;
                            }
                            case NSOrderedSame:
                            default:
                            {
                                //無法區分,沒有任何動作
                                ppSyncActionModel = nil;
                                
                                break;
                            }
                        }
                    }
                }
                
                break;
            }
        }
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return ppSyncActionModel;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPSyncDataControllerDelegate

//================================================================================
//
//================================================================================
- (NSString *)syncDataController:(id)syncDataController copyGroupIDWithOtherSideGroupID:(NSString *)otherSideGroupID error:(NSError **)error
{
    NSString    *groupID        = nil;
    NSError     *originError    = ((error!=nil)?*error:nil);
    NSError     *returnError    = originError;
    
    //////////////////////////////////////////////////
    
    do 
    {
        if(syncDataController==nil || otherSideGroupID==nil)
        {
            returnError = PPErrorParameterInvalidity(returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(syncDataController==self.remoteSyncDataController) 
        {
            PPSyncRecordModel *syncRecordModel = [self groupSyncRecordModelWithUniqueID:otherSideGroupID syncRecordType:PPSyncRecordType_Local error:&returnError];
            if(returnError!=originError)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            if(syncRecordModel!=nil)
            {
                groupID = [syncRecordModel.remoteSyncCompareModel.uniqueID copy];
            }
        }
        else if(syncDataController==self.localSyncDataController)
        {
            PPSyncRecordModel *syncRecordModel = [self groupSyncRecordModelWithUniqueID:otherSideGroupID syncRecordType:PPSyncRecordType_Remote error:&returnError];
            if(returnError!=originError)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            if(syncRecordModel!=nil)
            {
                groupID = [syncRecordModel.localSyncCompareModel.uniqueID copy];
            }
        }
        else
        {
            returnError = PPErrorParameterInvalidity(returnError);
        }
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(returnError!=originError)
    {
        NSString *classString = nil;
        if (syncDataController)
        {
            classString = NSStringFromClass([syncDataController class]);
        }
        [self dumpLogMessageWithFormat:@"%s failed (syncDataController=%@, copyGroupIDWithOtherSideGroupID=%@, returnError=%@)", __func__, classString, otherSideGroupID, returnError];
    }

    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return groupID;
}

//================================================================================
//
//================================================================================
- (NSData *)syncDataController:(id)syncDataController copyOtherSideImageDataWithImageType:(PPSyncCardImageType)imageType error:(NSError **)error
{
    NSData  *imageData      = nil;
    NSError *originError    = ((error!=nil)?*error:nil);
    NSError *returnError    = originError;
    
    //////////////////////////////////////////////////
    
    do
    {
        if(syncDataController==nil)
        {
            returnError = PPErrorParameterInvalidity(returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(syncDataController==self.remoteSyncDataController) 
        {
            //代表是遠方要本地影像
            imageData = [self.localSyncDataController copyImageDataByUniqueID:self.currentSyncActionModel.localSyncCompareModel.uniqueID 
                                                                    imageType:imageType];
            
            //////////////////////////////////////////////////
            
            if(imageData==nil)
            {
                returnError = [[self.localSyncDataController.lastError retain] autorelease];
                if(returnError==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                }
            }
        }
        else if(syncDataController==self.localSyncDataController) 
        {
            //代表是本地要遠方影像
            imageData = [self.remoteSyncDataController copyImageDataByUniqueID:self.currentSyncActionModel.remoteSyncCompareModel.uniqueID 
                                                                     imageType:imageType];
            
            //////////////////////////////////////////////////
            
            if(imageData==nil)
            {
                returnError = [[self.remoteSyncDataController.lastError retain] autorelease];
                if(returnError==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                }
            }
        }
        else
        {
            returnError = PPErrorParameterInvalidity(returnError);
        }
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(returnError!=originError)
    {
        NSString *classString = nil;
        if (syncDataController)
        {
            classString = NSStringFromClass([syncDataController class]);
        }
        [self dumpLogMessageWithFormat:@"%s failed (syncDataController=%@, copyOtherSideImageDataWithImageType=%lu, returnError=%@)", __func__, classString, (unsigned long)imageType, returnError];
    }
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return imageData;
}

//================================================================================
//
//================================================================================
- (PPSyncCardModel *)syncDataController:(id)syncDataController copyOtherSideSyncCardModelWithUniqueID:(NSString *)uniqueID fromVersion:(NSUInteger)fromVersion toVersion:(NSUInteger)toVersion error:(NSError **)error
{
    PPSyncCardModel *syncCardModel  = nil;
    NSError         *originError    = ((error!=nil)?*error:nil);
    NSError         *returnError    = originError;
    
    //////////////////////////////////////////////////
    
    do
    {
        if(syncDataController==nil || uniqueID==nil)
        {
            returnError = PPErrorParameterInvalidity(returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(syncDataController==self.remoteSyncDataController)
        {
            syncCardModel = [self.localSyncDataController copySyncCardModelWithUniqueID:uniqueID fromVersion:fromVersion toVersion:toVersion];
            if(syncCardModel==nil)
            {
                returnError = [[self.localSyncDataController.lastError retain] autorelease];
                if(returnError==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                }
            }
        }
        else if(syncDataController==self.localSyncDataController)
        {            
            syncCardModel = [self.remoteSyncDataController copySyncCardModelWithUniqueID:uniqueID fromVersion:fromVersion toVersion:toVersion];
            if(syncCardModel==nil)
            {
                returnError = [[self.remoteSyncDataController.lastError retain] autorelease];
                if(returnError==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                }
            }
        }
        else
        {
            returnError = PPErrorParameterInvalidity(returnError);
        }
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(returnError!=originError)
    {
        NSString *classString = nil;
        if (syncDataController)
        {
            classString = NSStringFromClass([syncDataController class]);
        }

        [self dumpLogMessageWithFormat:@"%s failed (syncDataController=%@, copyOtherSideSyncCardModelWithUniqueID=%@, fromVersion=%lu, toVersion=%lu, returnError=%@)", __func__, classString, uniqueID, (unsigned long)fromVersion, (unsigned long)toVersion, returnError];
    }
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return syncCardModel;
}

//================================================================================
//
//================================================================================
- (NSString *)syncDataController:(id)syncDataController copyOtherSideUniqueIDWithUniqueID:(NSString *)uniqueID error:(NSError **)error
{
    NSString    *otherSideUniqueID  = nil;
    NSError     *originError        = ((error!=nil)?*error:nil);
    NSError     *returnError        = originError;
    
    //////////////////////////////////////////////////
    
    do
    {
        if(syncDataController==nil || uniqueID==nil)
        {
            returnError = PPErrorParameterInvalidity(returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(syncDataController==self.remoteSyncDataController)
        {
            PPSyncRecordModel *syncRecordModel = [self cardSyncRecordModelWithUniqueID:uniqueID syncRecordType:PPSyncRecordType_Remote error:&returnError];
            if(returnError!=originError)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            if(syncRecordModel!=nil)
            {
                otherSideUniqueID = [syncRecordModel.localSyncCompareModel.uniqueID copy];
            }
        }
        else if(syncDataController==self.localSyncDataController)
        {
            PPSyncRecordModel *syncRecordModel = [self cardSyncRecordModelWithUniqueID:uniqueID syncRecordType:PPSyncRecordType_Local error:&returnError];
            if(returnError!=originError)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            if(syncRecordModel!=nil)
            {
                otherSideUniqueID = [syncRecordModel.remoteSyncCompareModel.uniqueID copy];
            }
        }
        else
        {
            returnError = PPErrorParameterInvalidity(returnError);
        }
        
    }while(0);
    
    //////////////////////////////////////////////////
    
    if(returnError!=originError)
    {
        NSString *classString = nil;
        if (syncDataController)
        {
            classString = NSStringFromClass([syncDataController class]);
        }

        [self dumpLogMessageWithFormat:@"%s failed (syncDataController=%@, copyOtherSideUniqueIDWithUniqueID=%@, returnError=%@)", __func__, classString, uniqueID, returnError];
    }
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return otherSideUniqueID;
}

//================================================================================
//
//================================================================================
- (BOOL)syncDataController:(PPSyncDataController *)syncDataController isCardNotExistError:(NSError *)error
{
    BOOL result = NO;
    
    //////////////////////////////////////////////////
    
    do
    {
        if(syncDataController==nil || error==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(syncDataController==self.remoteSyncDataController)
        {
            result = [self.localSyncDataController isCardNotExistError:error];
        }
        else if(syncDataController==self.localSyncDataController)
        {
            result = [self.remoteSyncDataController isCardNotExistError:error];
        }
        
    }while(0);
    
    //////////////////////////////////////////////////

    NSString *classString = nil;
    if (syncDataController)
    {
        classString = NSStringFromClass([syncDataController class]);
    }

    [self dumpLogMessageWithFormat:@"%s (syncDataController:%@ isCardNotExistError:%@ return:%d)", __func__, classString, error, result];
    
    //////////////////////////////////////////////////
    
    return result;
}

//================================================================================
//
//================================================================================
- (BOOL)syncDataController:(id)syncDataController updateOtherSideImageDictionary:(NSMutableDictionary *)imageDictionary otherSideUniqueID:(NSString *)uniqueID error:(NSError **)error
{
    NSError *originError    = ((error!=nil)?*error:nil);
    NSError *returnError    = originError;
    
    //////////////////////////////////////////////////
    
    do
    {
        if(syncDataController==nil || uniqueID==nil || imageDictionary==nil)
        {
            returnError = PPErrorParameterInvalidity(returnError);
            break;
        }

        //////////////////////////////////////////////////

        if(syncDataController==self.remoteSyncDataController)
        {
            if([self.localSyncDataController updateImageDictionary:imageDictionary uniqueID:uniqueID]==NO)
            {
                returnError = [[self.localSyncDataController.lastError retain] autorelease];
                if(returnError==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                }
            }
        }
        else if(syncDataController==self.localSyncDataController)
        {
            if([self.remoteSyncDataController updateImageDictionary:imageDictionary uniqueID:uniqueID]==NO)
            {
                returnError = [[self.remoteSyncDataController.lastError retain] autorelease];
                if(returnError==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                }
            }
        }
        else
        {
            returnError = PPErrorParameterInvalidity(returnError);
        }

    }while(0);

    //////////////////////////////////////////////////
    
    if(returnError!=originError)
    {
        NSString *classString = nil;
        if (syncDataController)
        {
            classString = NSStringFromClass([syncDataController class]);
        }

        [self dumpLogMessageWithFormat:@"%s failed (syncDataController:%@ updateOtherSideImageDictionary:%@ otherSideUniqueID:%@ returnError:%@)", __func__, classString, imageDictionary, uniqueID, returnError];
    }
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return (returnError==originError);
}

//================================================================================
//
//================================================================================
- (void)syncDataController:(id)syncDataController progressOfCopySyncCardModels:(CGFloat)progress
{
    [self reportProgress:progress];
}

//================================================================================
// 呼叫時group的action已經執行完，所以不用調整self.syncActionModels裡的時間。
//================================================================================
- (void)syncDataController:(id)syncDataController forceUpdateSyncActionGroupModifiedTimeWithDataDict:(NSDictionary *)dataDict
{
    if([dataDict count] == 0)
    {
        return;
    }
    
    //////////////////////////////////////////////////
    // update record
    
    NSMutableArray *uniqueIDs = [NSMutableArray arrayWithArray:[dataDict allKeys]];
    NSMutableArray *modifiedTimes = [NSMutableArray arrayWithArray:[dataDict allValues]];
    PPSyncRecordType syncRecordType = PPSyncRecordType_Local;
    
    if(syncDataController == remoteSyncDataController_)
    {
        syncRecordType = PPSyncRecordType_Remote;
    }
    
    // 這邊因為有record紀錄的才要調整，為了減少時間花費，直接更新不先檢查是否存在。
    // 有錯誤頂多下次同步會變成要更新另一端，不會變成資料不對稱。
    [super beginTransactionWithError:nil];

    for(int i=0; i<[uniqueIDs count]; i++)
    {
        [self updateGroupSyncRecordWithUniqueID:uniqueIDs[i]
                                   modifiedTime:modifiedTimes[i]
                                 syncRecordType:syncRecordType
                                          error:nil];
    }
    
    [super commitTransactionWithError:nil];
}

//================================================================================
// 呼叫時card的action還沒執行，所以self.syncActionModels裡的時間要一起更新。
//================================================================================
- (void)syncDataController:(id)syncDataController forceUpdateSyncActionCardModifiedTimeWithDataDict:(NSDictionary *)dataDict
{
    if([dataDict count] == 0)
    {
        return;
    }
    
    //////////////////////////////////////////////////
    // update record
    
    NSMutableArray *uniqueIDs = [NSMutableArray arrayWithArray:[dataDict allKeys]];
    NSMutableArray *modifiedTimes = [NSMutableArray arrayWithArray:[dataDict allValues]];
    PPSyncRecordType syncRecordType = PPSyncRecordType_Local;
    
    if(syncDataController == remoteSyncDataController_)
    {
        syncRecordType = PPSyncRecordType_Remote;
    }

    // 這邊因為有record紀錄的才要調整，為了減少時間花費，直接更新不先檢查是否存在。
    // 有錯誤頂多下次同步會變成要更新另一端，不會變成資料不對稱。
    [super beginTransactionWithError:nil];
    
    for(int i=0; i<[uniqueIDs count]; i++)
    {
        [super updateCardSyncRecordWithUniqueID:uniqueIDs[i]
                                   modifiedTime:modifiedTimes[i]
                                 syncRecordType:syncRecordType
                                          error:nil];
    }
    
    [super commitTransactionWithError:nil];
    

    //////////////////////////////////////////////////
    // update action
    
    PPSyncTarget syncTarget = PPSyncTarget_Local;
    
    if(syncDataController == remoteSyncDataController_)
    {
        syncTarget = PPSyncTarget_Remote;
    }

    for(PPSyncActionModel *actionModel in self.syncActionModels)
    {
        if(actionModel.type == PPSyncType_Card)
        {
            for(int i=0; i<[uniqueIDs count]; i++)
            {
                NSString *uniqueID = uniqueIDs[i];
                NSDate *newModifiedTime = modifiedTimes[i];
           
                if(syncTarget == PPSyncRecordType_Remote)
                {
                    if([actionModel.remoteSyncCompareModel.uniqueID isEqualToString:uniqueID] == YES)
                    {
                        actionModel.remoteSyncCompareModel.lastModifiedDate = newModifiedTime;
                        
                        // 處理完就移除，減少比對次數。
                        [uniqueIDs removeObjectAtIndex:i];
                        [modifiedTimes removeObjectAtIndex:i];
                        i--;
                    }
                }
                else
                {
                    if([actionModel.localSyncCompareModel.uniqueID isEqualToString:uniqueID] == YES)
                    {
                        actionModel.localSyncCompareModel.lastModifiedDate = newModifiedTime;
                 
                        // 處理完就移除，減少比對次數。
                        [uniqueIDs removeObjectAtIndex:i];
                        [modifiedTimes removeObjectAtIndex:i];
                        i--;
                    }
                }
            }
        }
    }
}


//================================================================================
//
//================================================================================
- (void)syncDataController:(id)syncDataController forceRemoveCardWhenNextSyncWithUniqueID:(NSString *)uniqueID
{
    do
    {
        if([uniqueID length] == 0)
        {
            break;
        }
        
        
        //////////////////////////////////////////////////
        // add record

        PPSyncRecordModel *syncRecordModel = [[[PPSyncRecordModel alloc] init] autorelease];
        
        if(syncRecordModel == nil)
        {
            break;
        }
        
        syncRecordModel.version = 1;
        syncRecordModel.localSyncCompareModel = [[[PPSyncCompareModel alloc] init] autorelease];
        syncRecordModel.remoteSyncCompareModel = [[[PPSyncCompareModel alloc] init] autorelease];
        
        if(syncRecordModel.localSyncCompareModel == nil || syncRecordModel.remoteSyncCompareModel == nil)
        {
            break;
        }        
        
        NSDate *fakeDate = [NSDate date];

        if(syncDataController == remoteSyncDataController_)
        {
            // force remove remote card
            syncRecordModel.localSyncCompareModel.uniqueID = [NSString GUID];
            syncRecordModel.localSyncCompareModel.lastModifiedDate = fakeDate;
            
            syncRecordModel.remoteSyncCompareModel.uniqueID = uniqueID;
            syncRecordModel.remoteSyncCompareModel.lastModifiedDate = fakeDate;
        }
        else
        {
            // force remove local card
            syncRecordModel.remoteSyncCompareModel.uniqueID = [NSString GUID];
            syncRecordModel.remoteSyncCompareModel.lastModifiedDate = fakeDate;
            
            syncRecordModel.localSyncCompareModel.uniqueID = uniqueID;
            syncRecordModel.localSyncCompareModel.lastModifiedDate = fakeDate;
        }

        [super insertCardSyncRecordWithSyncRecordModel:syncRecordModel error:nil];        
    }
    while(0);
}

@end
