//
//  PPSupportController.m
//  
//
//  Created by Mike on 13/5/9.
//  Copyright (c) 2013年 Penpower. All rights reserved.
//

#import "PPSupportController.h"
#import "PPSystemInfoController.h"
#import <sys/sysctl.h>

#if TARGET_OS_IPHONE
#import <Firebase/Firebase.h>
#endif

////////////////////////////////////////////////////////////////////////////////////////////////////

NSString * const PPSupportControllerReportMailTitle               = @"PenPower Technical Support Request(%@)";
NSString * const PPSupportControllerAdditionalInfoKey_PrevMessage = @"PPSupportControllerAdditionalInfoKey_PrevMessage";
NSString * const PPSupportControllerAdditionalInfoKey_PostMessage = @"PPSupportControllerAdditionalInfoKey_PostMessage";

////////////////////////////////////////////////////////////////////////////////////////////////////

NSString * const LicenseString_Active    = @"active";
NSString * const LicenseString_NonActive = @"nonactive";
NSString * const LicenseString_Lite      = @"lite";
NSString * const LicenseString_Paid      = @"paid";
NSString * const LicenseString_Free      = @"free";

////////////////////////////////////////////////////////////////////////////////////////////////////

NSString * const PPSupportController_ZipPassword = @"io9#IH93(mlkas@0";

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPSupportController




#pragma mark - private


//==============================================================================
//
//==============================================================================
+ (NSString *)osType
{
#if TARGET_OS_IPHONE
    
    return @"iOS";
    
#elif TARGET_OS_OSX
    
    return @"macOS";
    
#endif
}


//==============================================================================
//
//==============================================================================
+ (NSString *)deviceModel
{
#if TARGET_OS_IPHONE
    
    return [UIDevice deviceModel];
    
#elif TARGET_OS_OSX
    
    size_t size;
    sysctlbyname("hw.model", NULL, &size, NULL, 0);
    char *machine = malloc(size);
    sysctlbyname("hw.model", machine, &size, NULL, 0);
    NSString *deviceModel = [NSString stringWithCString:machine encoding: NSUTF8StringEncoding];
    free(machine);
    
    return deviceModel;
    
#endif
    
}


//==============================================================================
//
//==============================================================================
+ (NSString *)systemVersion
{
#if TARGET_OS_IPHONE
    
    return [UIDevice currentDevice].systemVersion;
    
#elif TARGET_OS_OSX
    
    NSOperatingSystemVersion systemVersion = [[NSProcessInfo processInfo] operatingSystemVersion];
    return [NSString stringWithFormat:@"%ld.%ld.%ld", systemVersion.majorVersion, systemVersion.minorVersion, systemVersion.patchVersion];
    
#endif
}


//==============================================================================
//
//==============================================================================
+ (NSString *)mailTitle
{
    NSString *emailTitle = [[NSBundle mainBundle] objectForInfoDictionaryKey:PPSupportControllerKey_EmailTitle];
    
    if(emailTitle!=nil &&
       emailTitle.length>0)
    {
        return [NSString stringWithFormat:emailTitle, [[NSDate date] stringWithFormat:NSDateFormat_Second]];
    }
    else
    {
        return [NSString stringWithFormat:PPSupportControllerReportMailTitle, [[NSDate date] stringWithFormat:NSDateFormat_Second]];
    }
}


//==============================================================================
//
//==============================================================================
+ (NSString *)mailBodyWithLicense:(NSString *)license additionalInfos:(NSDictionary*)additionalInfos
{
    ////////////////////////////////////////////////////////////////////////////////////////////////////
    
    //系統資訊
    NSString *osType             = [PPSupportController osType];
    NSString *deviceModel        = [PPSupportController deviceModel];
    NSString *systemVersion      = [PPSupportController systemVersion];
    NSString *localeLanguageCode = [[PPSystemInfoController stringWithLanguage:[PPSystemInfoController language]] lowercaseString];
    NSString *countryCode = [[NSLocale currentLocale] objectForKey:NSLocaleCountryCode];
    if(countryCode==nil)
    {
        countryCode = @"EN";
    }
    
    NSString *localeCountryCode  = [countryCode lowercaseString];
    NSString *userIdentifier     = [PPSupportController systemUUID];
    
    
    //////////////////////////////////////////////////
    
    //程式資訊
    NSString *bundleName = [[[NSBundle mainBundle] objectForInfoDictionaryKey:PPSupportControllerKey_ProductName] lowercaseString];
    
    if(bundleName==nil || bundleName.length<=0)
    {
        bundleName = [[[NSBundle mainBundle] objectForInfoDictionaryKey:@"CFBundleName"] lowercaseString];
    }
    
    NSString *bundleVersion       = [[NSBundle mainBundle] objectForInfoDictionaryKey:@"CFBundleShortVersionString"];
    NSString *buildVersion        = [[NSBundle mainBundle] objectForInfoDictionaryKey:@"CFBundleVersion"];
    NSString *bundleLanguage      = [[NSBundle mainBundle] objectForInfoDictionaryKey:PPSupportControllerKey_AppLang];
    
    //////////////////////////////////////////////////
    // 取得舊版資訊
    
    NSUserDefaults  *userDefaults = [[NSUserDefaults alloc] init];
    NSString *oldBundleVersion    = [userDefaults objectForKey:PPSupportControllerKey_OldAppVersion];
    [userDefaults release];
    
    if(oldBundleVersion!=nil && [oldBundleVersion length]>0)
    {
        bundleVersion = [bundleVersion stringByAppendingFormat:@" (%@)", oldBundleVersion];
    }
    
    ////////////////////////////////////////////////////////////////////////////////////////////////////
    //建立郵件內文
    NSString * mainMessage =[NSString stringWithFormat:@"Your Question: \n\n\n\n---------------------------------------\nOSType : %@\nOSV : %@\nOSLang : %@\nProduct : %@\nAppLicense : %@\nSWV : %@\nBuild : %@\nAppLang : %@\nRegional : %@\nHWV : %@\n",
                             osType,
                             systemVersion,
                             localeLanguageCode,
                             bundleName,
                             license,
                             bundleVersion,
                             buildVersion,
                             bundleLanguage,
                             localeCountryCode,
                             deviceModel];
    
    // !! 如果可以抓到|userIdentifier|才加上去
    if(userIdentifier!=nil && [userIdentifier length]>0)
    {
        mainMessage = [mainMessage stringByAppendingFormat:@"UID : %@\n", userIdentifier];
    }
    
    //////////////////////////////////////////////////
    // prev message
    NSString *prevMessage = @"";
    if(additionalInfos!=nil)
    {
        NSDictionary * prevMessages = [additionalInfos objectForKey:PPSupportControllerAdditionalInfoKey_PrevMessage];
        if([prevMessages count]>0)
        {
            prevMessage = [prevMessage stringByAppendingString:@"---------------------------------------\n"];
            
            for (NSString * key in prevMessages.allKeys)
            {
                prevMessage = [prevMessage stringByAppendingFormat:@"%@ : %@\n", key, [prevMessages objectForKey:key]];
            }
            
            prevMessage = [prevMessage stringByAppendingString:@"---------------------------------------\n\n"];
        }
    }
    
    //////////////////////////////////////////////////
    // post message
    NSString *postMessage = @"";
    if(additionalInfos!=nil)
    {
        NSDictionary * postMessages = [additionalInfos objectForKey:PPSupportControllerAdditionalInfoKey_PostMessage];
        if([postMessages count]>0)
        {
            postMessage = [postMessage stringByAppendingString:@"\n---------------------------------------\n"];
            for (NSString * key in postMessages.allKeys)
            {
                postMessage = [postMessage stringByAppendingFormat:@"%@ : %@\n", key, [postMessages objectForKey:key]];
            }
            
            postMessage = [postMessage stringByAppendingString:@"---------------------------------------\n"];
        }
    }
    
    ////////////////////////////////////////////////////////////////////////////////////////////////////
    
    NSString * fullMessageBody = @"";
    if(prevMessage!=nil && [prevMessage length]>0)
    {
        fullMessageBody = [fullMessageBody stringByAppendingString:prevMessage];
    }
    
    fullMessageBody = [fullMessageBody stringByAppendingString:mainMessage];
    
    
    if(postMessage!=nil && [postMessage length]>0)
    {
        fullMessageBody = [fullMessageBody stringByAppendingString:postMessage];
    }
    
    return fullMessageBody;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Methods


#if TARGET_OS_IPHONE
#pragma mark - for ios

//================================================================================
//
//================================================================================
+ (BOOL)reportMailWithViewController:(UIViewController*)viewController
                             license:(NSString *)license
                          recipients:(NSArray *)recipients
                 attachmentFilePaths:(NSArray *)filePaths
                     additionalInfos:(NSDictionary*)additionalInfos
                            delegate:(id<MFMailComposeViewControllerDelegate>)delegate
                  presetedComplition:(void (^)(void))presetedComplition
{
    BOOL result = NO;
    
    do
    {
        if(viewController==nil || [PPMailComposeViewController canSendMail]==NO)
        {
            break;
        }
        
        ////////////////////////////////////////////////////////////////////////////////////////////////////
        
        PPMailComposeViewController *ppMailComposeViewController = [[PPMailComposeViewController alloc] init];
		if(ppMailComposeViewController!=nil)
        {
            // 如果有delegate的話才設定
            if(delegate!=nil)
            {
                ppMailComposeViewController.mailComposeDelegate  = delegate;
            }

            //設定標題
            [ppMailComposeViewController setSubject:[PPSupportController mailTitle]];
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            [ppMailComposeViewController setMessageBody:[PPSupportController mailBodyWithLicense:license additionalInfos:additionalInfos]
                                                 isHTML:NO];
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            //設定收件人
            NSArray  *supportEmails       = recipients;
            
            // !! 如果沒有傳 recipients ，用Info.Plist中的值
            if ([supportEmails count]==0)
            {
                supportEmails = [[NSBundle mainBundle] objectForInfoDictionaryKey:PPSupportControllerKey_SupportEMails];
            }
            
            [ppMailComposeViewController setToRecipients:supportEmails];
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            //加入附件
            
            for(NSString *filePath in filePaths)
            {
                NSURLRequest *filePathURLRequest = [[NSURLRequest alloc] initWithURL:[NSURL fileURLWithPath:[filePath stringByExpandingTildeInPath]]
                                                                         cachePolicy:NSURLRequestReloadIgnoringCacheData
                                                                     timeoutInterval:5];
                if(filePathURLRequest!=nil)
                {
                    NSURLResponse *filePathURLResponse = nil;
                    NSData *filePathData = [NSURLConnection sendSynchronousRequest:filePathURLRequest
                                                                 returningResponse:&filePathURLResponse
                                                                             error:nil];
                    
                    [ppMailComposeViewController addAttachmentData:filePathData
                                                          mimeType:[filePathURLResponse MIMEType]
                                                          fileName:[filePath lastPathComponent]];
                    
                    [[NSFileManager defaultManager] removeItemAtPath:filePath error:nil];
                    
                    
                    [filePathURLRequest release];
                }
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            [viewController presentViewController:ppMailComposeViewController animated:YES completion:presetedComplition];
			
            result = YES;
            
            [ppMailComposeViewController release];
		}
        
    }while(0);
    
    return result;
}


//================================================================================
//
//================================================================================
+ (BOOL)reportMailWithViewController:(UIViewController*)viewController
                             license:(NSString *)license
                 attachmentFilePaths:(NSArray *)filePaths
                     additionalInfos:(NSDictionary*)additionalInfos
                            delegate:(id<MFMailComposeViewControllerDelegate>)delegate
{
    return [self reportMailWithViewController:viewController
                                      license:license
                                   recipients:nil
                          attachmentFilePaths:filePaths
                              additionalInfos:additionalInfos
                                     delegate:delegate
                           presetedComplition:nil];
}

//================================================================================
//
//================================================================================
+ (BOOL)reportMailWithViewController:(UIViewController*)viewController
                             license:(NSString *)license
                 attachmentFilePaths:(NSArray *)filePaths
                     additionalInfos:(NSDictionary*)additionalInfos
{
    return [self reportMailWithViewController:viewController
                                      license:license
                                   recipients:nil
                          attachmentFilePaths:filePaths
                              additionalInfos:additionalInfos
                                     delegate:nil
                           presetedComplition:nil];
}


//================================================================================
//
//================================================================================
+ (BOOL)reportMailWithViewController:(UIViewController *)viewController
                             license:(NSString *)license
                 attachmentFilePaths:(NSArray *)filePaths
{
    return [self reportMailWithViewController:viewController
                                      license:license
                                   recipients:nil
                          attachmentFilePaths:filePaths
                              additionalInfos:nil
                                     delegate:nil
                           presetedComplition:nil];
}


//==============================================================================
//
//==============================================================================
+ (BOOL)reportMailWithViewController:(UIViewController *)viewController
                             license:(NSString *)license
                          recipients:(NSArray *)recipients
{
    return [self reportMailWithViewController:viewController
                                      license:license
                                   recipients:recipients
                          attachmentFilePaths:nil
                              additionalInfos:nil
                                     delegate:nil
                           presetedComplition:nil];
}


//================================================================================
//
//================================================================================
+ (BOOL)reportMailWithViewController:(UIViewController *)viewController
                             license:(NSString *)license
{
    return [self reportMailWithViewController:viewController
                                      license:license
                                   recipients:nil
                          attachmentFilePaths:nil
                              additionalInfos:nil
                                     delegate:nil
                           presetedComplition:nil];
}


#elif TARGET_OS_OSX
#pragma mark - for OSX

//================================================================================
//
//================================================================================
+ (BOOL)reportMailWithLicense:(NSString *)license
                   recipients:(NSArray *)recipients
          attachmentFilePaths:(NSArray *)filePaths
              additionalInfos:(NSDictionary*)additionalInfos
{
    BOOL result = NO;
    
    do
    {
        if([PPMailComposeWindowController canSendMail]==NO)
        {
            break;
        }
        
        ////////////////////////////////////////////////////////////////////////////////////////////////////
        
        PPMailComposeWindowController *ppMailComposeWindowController = [[PPMailComposeWindowController alloc] init];
        if(ppMailComposeWindowController!=nil)
        {
            //設定標題
            [ppMailComposeWindowController setSubject:[PPSupportController mailTitle]];
          
            ////////////////////////////////////////////////////////////////////////////////////////////////////

            [ppMailComposeWindowController setMessageBody:[PPSupportController mailBodyWithLicense:license additionalInfos:additionalInfos]];
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            //設定收件人
            NSArray  *supportEmails       = recipients;
            
            // !! 如果沒有傳 recipients ，用Info.Plist中的值
            if ([supportEmails count]==0)
            {
                supportEmails = [[NSBundle mainBundle] objectForInfoDictionaryKey:PPSupportControllerKey_SupportEMails];
            }
            
            [ppMailComposeWindowController setToRecipients:supportEmails];
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            //加入附件
            
            for(NSString *filePath in filePaths)
            {
                [ppMailComposeWindowController addAttachmentFileURL:[NSURL fileURLWithPath:[filePath stringByExpandingTildeInPath]]];
            }
            
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            [ppMailComposeWindowController openComposeWindow];
            
            [ppMailComposeWindowController release];
            
            result = YES;
        }
        
    }while(0);
    
    return result;
}


//==============================================================================
//
//==============================================================================
+ (BOOL)reportMailWithLicense:(NSString *)license
                   recipients:(NSArray *)recipients
          attachmentFilePaths:(NSArray *)filePaths
{
    return [PPSupportController reportMailWithLicense:license
                                           recipients:recipients
                                  attachmentFilePaths:filePaths
                                      additionalInfos:nil];
}


//==============================================================================
//
//==============================================================================
+ (BOOL)reportMailWithLicense:(NSString *)license
                   recipients:(NSArray *)recipients
{
    return [PPSupportController reportMailWithLicense:license
                                           recipients:recipients
                                  attachmentFilePaths:nil
                                      additionalInfos:nil];
}


//==============================================================================
//
//==============================================================================
+ (BOOL)reportMailWithLicense:(NSString *)license
{
    return [PPSupportController reportMailWithLicense:license
                                           recipients:nil
                                  attachmentFilePaths:nil
                                      additionalInfos:nil];
}

#endif

#pragma mark - common methods


//==============================================================================
//
//==============================================================================
+ (NSString *)moreAppsURLParamsWithProductName:(NSString *)productName license:(NSString *)license
{
    NSString *countryCode = [[NSLocale currentLocale] objectForKey:NSLocaleCountryCode];
    if(countryCode==nil)
    {
        countryCode = @"EN";
    }
    return [NSString stringWithFormat:@"WebVer=2&OSType=iOS&OSV=%@&OSLang=%@&Product=%@&AppLicense=%@&SWV=%@&AppLang=%@&Regional=%@&HWV=%@",
            [PPSupportController systemVersion],
            [[PPSystemInfoController stringWithLanguage:[PPSystemInfoController language]] lowercaseString],
            [productName lowercaseString],
            license,
            [[[NSBundle mainBundle] objectForInfoDictionaryKey:@"CFBundleShortVersionString"] lowercaseString],
            [[[NSBundle mainBundle] objectForInfoDictionaryKey:PPSupportControllerKey_AppLang] lowercaseString],
            [countryCode lowercaseString],
            [PPSupportController deviceModel]];
}


//================================================================================
//
//================================================================================
+ (NSString *)moreAppsURLParamsWithLicense:(NSString *)license
{
    return [[self class] moreAppsURLParamsWithProductName:[[NSBundle mainBundle] objectForInfoDictionaryKey:@"CFBundleName"] license:license];
}


//================================================================================
//
//================================================================================
+ (NSDictionary *)additionalInfosDictionaryWithPrevDictionary:(NSDictionary*)prevDictionary andPostDictionary:(NSDictionary*)postDictionary
{
    NSDictionary * result = nil;
    NSMutableDictionary * tempResult = [[NSMutableDictionary alloc] init];
    if(tempResult!=nil)
    {
        if (prevDictionary!=nil)
        {
            [tempResult setObject:prevDictionary forKey:PPSupportControllerAdditionalInfoKey_PrevMessage];
        }
        if (postDictionary!=nil)
        {
            [tempResult setObject:postDictionary forKey:PPSupportControllerAdditionalInfoKey_PostMessage];
        }
        
        result = [NSDictionary dictionaryWithDictionary:tempResult];
        [tempResult release];
    }

    return result;
}


//================================================================================
//
//================================================================================
+ (void)checkForUpdateWithHandler:(void(^)(BOOL isUpdateAvailable))updateHandler
{

    NSDictionary *infoDictionary = [[NSBundle mainBundle] infoDictionary];
    
    NSString *appID = infoDictionary[@"CFBundleIdentifier"];
    
    NSURL *url = [NSURL URLWithString:[NSString stringWithFormat:@"http://itunes.apple.com/lookup?bundleId=%@", appID]];
    
    NSURLSession *session = [NSURLSession sharedSession];
    
    NSURLSessionDataTask *theTask = [session dataTaskWithRequest:[NSURLRequest requestWithURL:url]
                                               completionHandler:^(NSData * _Nullable data, NSURLResponse * _Nullable response, NSError * _Nullable error) {
        
        BOOL isUpdateAvailable = NO;
        
        NSDictionary *lookup = [NSJSONSerialization JSONObjectWithData:data options:0 error:nil];
        
        if (lookup && [lookup[@"resultCount"] integerValue] == 1)
        {
            NSString *appStoreVersion = lookup[@"results"][0][@"version"];
            NSString *currentVersion = infoDictionary[@"CFBundleShortVersionString"];
            
            isUpdateAvailable = [appStoreVersion compare:currentVersion options:NSNumericSearch] == NSOrderedDescending;
        }
        
        //////////////////////////////////////////////////
        
        if (updateHandler)
        {
            updateHandler(isUpdateAvailable);
        }
    }];
    
    [theTask resume];
}



////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance System Method

//==============================================================================
//
//==============================================================================
+ (NSString *)systemUUID
{
#if TARGET_OS_IPHONE
    
    UIDevice *currentDevice      = [UIDevice currentDevice];
    if([currentDevice respondsToSelector:@selector(identifierForVendor)])
    {
        return [[[UIDevice currentDevice] identifierForVendor] UUIDString];
    }
    
    return nil;
    
#elif TARGET_OS_OSX
    
    io_service_t platformExpert = IOServiceGetMatchingService(kIOMasterPortDefault,IOServiceMatching("IOPlatformExpertDevice"));
    if (!platformExpert)
        return nil;
    
    CFTypeRef serialNumberAsCFString = IORegistryEntryCreateCFProperty(platformExpert,CFSTR(kIOPlatformUUIDKey),kCFAllocatorDefault, 0);
    IOObjectRelease(platformExpert);
    
    if (!serialNumberAsCFString)
        return nil;
    
    NSString *uuid = [NSString stringWithString:(__bridge NSString *)(serialNumberAsCFString)];
    CFRelease(serialNumberAsCFString);
    return uuid;
    
#endif
    
}


//==============================================================================
//
//==============================================================================
+ (void)initializeCrashlytics
{
#if TARGET_OS_IPHONE
    // MARK: configure只能呼叫一次
    [FIRApp configure];
    
    // 指定UserID
    [[FIRCrashlytics crashlytics] setUserID:[PPSupportController systemUUID]];
#endif    
}


//================================================================================
//
//================================================================================
+ (void)simulateCrash
{
   int *x = NULL; *x = 42;
}


@end
