//
//  PPSplitViewController.m
//  
//
//  Created by Mike on 13/5/13.
//  Copyright (c) 2013年 Penpower. All rights reserved.
//

#import "PPSplitViewController.h"


static BOOL staticEnableMonitorKeyboard = NO;
////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPSplitViewController()

@interface PPSplitViewController ()
@property (nonatomic, retain) UIView *masterContainer;
@property (nonatomic, retain) UIView *detailContainer;
@property (nonatomic, retain) UIView *splitBarView;

@property (nonatomic, retain) UIView *masterMaskView;
@property (nonatomic, retain) UIView *detailMaskView;

@property (nonatomic, retain) NSArray *layoutConstraints;


#pragma mark view controller settings

/// master view controller是否被鎖定
@property (nonatomic, assign) BOOL isMasterViewControllerLocked;

/// detail view controller是否被鎖定
@property (nonatomic, assign) BOOL isDetailViewControllerLocked;


@property (nonatomic, assign, readwrite) PPSplitViewControllerDisplayMode displayMode;

/// 預設的橫式顯示模式, default PPSplitViewControllerDisplayMode_AllVisible
@property (nonatomic, assign) PPSplitViewControllerDisplayMode preferredLandscapeDisplayMode;

/// 預設的直式顯示模式, default PPSplitViewControllerDisplayMode_Detail
@property (nonatomic, assign) PPSplitViewControllerDisplayMode preferredPortraitDisplayMode;

/// 目前的鍵盤高度
@property (nonatomic, assign) CGFloat keyboardHeight;

/// layoutd的鍵盤高度
@property (nonatomic, assign) CGFloat layoutKeyboardHeight;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPSplitViewController





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Deallocating Objects

//==============================================================================
//
//==============================================================================
+ (void)initialize
{
    // do nothing, 元件預設值看起來還是在init做，在initialize做的話，如果子類別沒有實作initialize就會呼叫父類別的initialize造成，appearance重覆設定
}


//==============================================================================
//
//==============================================================================
- (id)init
{
    self = [super init];
	if(self)
	{
        // 預設值要先設定
        [self resetDefaultSettings];
        //////////////////////////////////////////////////
        
        [[PPSplitViewController appearance] applyInvocationTo:self];
        //////////////////////////////////////////////////
        
        if (staticEnableMonitorKeyboard)
        {
            [self registerForKeyboardNotifications];
        }
	}
	
    return self;
}


//==============================================================================
//
//==============================================================================
- (void)dealloc
{
    [self unRegisterForKeyboardNotifications];
    
    [self unlockMasterViewController];
    [self unlockDetailViewController];
    
    [self.masterViewController.view removeFromSuperview];
    self.masterViewController = nil;

    [self.detailViewController.view removeFromSuperview];
    self.detailViewController = nil;
    
    self.defaultBackgroundColor = nil;
    self.defaultSplitBarBackgroundColor = nil;
    self.defaultMasterBackgroundColor = nil;
    self.defaultDetailBackgroundColor = nil;
    
    [self _removeMainUI];
 
    //////////////////////////////////////////////////
	[super dealloc];
}


//==============================================================================
//
//==============================================================================
- (void)resetDefaultSettings
{
    // !! appearance的預設值，要用property設定
    _defaultBackgroundColor = [UIColor clearColor];
    _defaultMasterBackgroundColor = [UIColor clearColor];
    _defaultDetailBackgroundColor = [UIColor clearColor];
    _defaultSplitBarWidth = 1;
    _defaultSplitBarBackgroundColor = [UIColor clearColor];
    
    _preferredLandscapeDisplayMode = PPSplitViewControllerDisplayMode_AllVisible;
    _preferredPortraitDisplayMode = PPSplitViewControllerDisplayMode_Detail;
    
    _displayMode = PPSplitViewControllerDisplayMode_None;
    
    _preferredMasterWidth = 320;
    _layoutMasterBelowTopLayoutGuide = NO;
    _layoutDetailBelowTopLayoutGuide = NO;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - override property


//==============================================================================
//
//==============================================================================
- (void)setDefaultBackgroundColor:(UIColor *)defaultBackgroundColor
{
    [defaultBackgroundColor retain];
    [_defaultBackgroundColor release];
    _defaultBackgroundColor = defaultBackgroundColor;
}







////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - view controller life cycle


//==============================================================================
//
//==============================================================================
- (void)viewWillAppear:(BOOL)animated
{
    @synchronized(self)
    {
        
        [super viewWillAppear:animated];
        
        //////////////////////////////////////////////////
        // view controller setting
        
        [self.view setBackgroundColor:self.defaultBackgroundColor];
        
        //////////////////////////////////////////////////
        [self _prepareSettings];
        [self _prepareMainUI];
        
        [self loadMasterViewController];
        [self loadDetailViewController];
        
        //////////////////////////////////////////////////
        __block typeof(self) blockSelf = self;
        
        dispatch_async(dispatch_get_main_queue(), ^{
            
            if (blockSelf.isMasterViewControllerLocked)
            {
                [blockSelf lockMasterViewController];
            }
            else
            {
                [blockSelf unlockMasterViewController];
            }
            
            if(blockSelf.isDetailViewControllerLocked)
            {
                [blockSelf lockDetailViewController];
            }
            else
            {
                [blockSelf unlockDetailViewController];
            }
        });
    }
}


//==============================================================================
//
//==============================================================================
- (void)viewDidAppear:(BOOL)animated
{
    [super viewDidAppear:animated];
    
    //////////////////////////////////////////////////
    
    self.isViewDidAppear = YES;
}


//==============================================================================
//
//==============================================================================
- (void)viewWillDisappear:(BOOL)animated
{
    for (UIViewController *childViewController in self.childViewControllers)
    {
        [childViewController beginAppearanceTransition:NO animated:animated];
    }
    //////////////////////////////////////////////////
    
    [super viewWillDisappear:animated];
}


//==============================================================================
//
//==============================================================================
- (void)viewDidDisappear:(BOOL)animated
{
    self.isViewDidAppear = NO;
    
    for (UIViewController *childViewController in self.childViewControllers)
    {
        [childViewController endAppearanceTransition];
    }
    
    //////////////////////////////////////////////////
    // 要重設，返回時才會重新layout
    self.displayMode = PPSplitViewControllerDisplayMode_None;
    
    [self unloadViewController:self.masterViewController];
    [self unloadViewController:self.detailViewController];

    [self _removeMainUI];
    
    //////////////////////////////////////////////////

    
    [super viewDidDisappear:animated];
}


//==============================================================================
//
//==============================================================================
- (void)viewDidLayoutSubviews
{
    [super viewDidLayoutSubviews];
    //////////////////////////////////////////////////
    [self _layoutMasterViewController];
    [self _layoutDetailViewController];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - (private) prepare ui


//==============================================================================
//
//==============================================================================
- (void)_prepareSettings
{
    if ([self.delegate respondsToSelector:@selector(preferredDisplayModeForSplitViewController:isPortrait:)])
    {
        self.preferredPortraitDisplayMode = [self.delegate preferredDisplayModeForSplitViewController:self isPortrait:YES];
        self.preferredLandscapeDisplayMode = [self.delegate preferredDisplayModeForSplitViewController:self isPortrait:NO];
    }
}


//==============================================================================
//
//==============================================================================
- (void)_prepareMainUI
{
    @autoreleasepool
    {
        self.masterContainer = [[[UIView alloc] init] autorelease];
        if (self.masterContainer)
        {
            [self.masterContainer setTranslatesAutoresizingMaskIntoConstraints:NO];
            [self.masterContainer setBackgroundColor:self.defaultMasterBackgroundColor];
            
            [self.view addSubview:self.masterContainer];
        }

    
        self.detailContainer = [[[UIView alloc] init] autorelease];
        if (self.detailContainer)
        {
            [self.detailContainer setTranslatesAutoresizingMaskIntoConstraints:NO];
            [self.detailContainer setBackgroundColor:self.defaultDetailBackgroundColor];
            
            [self.view addSubview:self.detailContainer];
        }
        
        self.splitBarView = [[[UIView alloc] init] autorelease];
        if (self.splitBarView)
        {
            [self.splitBarView setTranslatesAutoresizingMaskIntoConstraints:NO];
            [self.splitBarView setBackgroundColor:self.defaultSplitBarBackgroundColor];
            
            [self.view addSubview:self.splitBarView];
        }
        
    }
    
    [self _resetLayoutConstraintsForPortraint:UIInterfaceOrientationIsPortrait(self.interfaceOrientation) keyboardHeight:0];
}


//==============================================================================
//
//==============================================================================
- (void)_removeMainUI
{
    [self _removeLayoutConstraints];
    
    [self.masterMaskView removeFromSuperview];
    self.masterMaskView = nil;
    
    [self.detailMaskView removeFromSuperview];
    self.detailMaskView = nil;
    
    [self.splitBarView removeFromSuperview];
    self.splitBarView = nil;
    
    [self.masterContainer removeFromSuperview];
    self.masterContainer = nil;
    
    [self.detailContainer removeFromSuperview];
    self.detailContainer = nil;
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - layout constraints


//==============================================================================
//
//==============================================================================
- (void)_resetLayoutConstraintsForPortraint:(BOOL)forPortrait keyboardHeight:(CGFloat)keyboardHeight
{
    if (self.masterContainer==nil ||
        self.detailContainer==nil)
    {
        return;
    }

    PPSplitViewControllerDisplayMode currentMode = (forPortrait)?self.preferredPortraitDisplayMode:self.preferredLandscapeDisplayMode;
    
    // 模式有不同才要切換constraint
    if (currentMode == self.displayMode &&
        keyboardHeight==self.layoutKeyboardHeight)
    {
        return ;
    }
    
    self.layoutKeyboardHeight = keyboardHeight;
    self.displayMode = currentMode;
    
    
    [self _removeLayoutConstraints];
    //////////////////////////////////////////////////
    NSDictionary *views = @{@"masterContainer":self.masterContainer,
                            @"detailContainer":self.detailContainer,
                            @"splitBarView":self.splitBarView,
                            @"topLayoutGuide":self.topLayoutGuide};
    NSDictionary *metrics = @{@"preferredMasterWidth":@(self.preferredMasterWidth),
                              @"defaultSplitBarWidth":@(self.defaultSplitBarWidth),
                              @"keyboardHeight":@(keyboardHeight)};
    NSMutableArray *layoutConstraints = [NSMutableArray array];
    
    switch (self.displayMode)
    {
        case PPSplitViewControllerDisplayMode_AllVisible:
        {
            [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[masterContainer(preferredMasterWidth)][splitBarView(defaultSplitBarWidth)][detailContainer]|"
                                                                                           options:NSLayoutFormatDirectionLeftToRight
                                                                                           metrics:metrics
                                                                                             views:views]];
            break;
        }
        case PPSplitViewControllerDisplayMode_Master:
        {
            [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[masterContainer]|"
                                                                                           options:NSLayoutFormatDirectionLeftToRight
                                                                                           metrics:metrics
                                                                                             views:views]];
            [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:[masterContainer][detailContainer]"
                                                                                           options:NSLayoutFormatDirectionLeftToRight
                                                                                           metrics:metrics
                                                                                             views:views]];
            break;
        }
        case PPSplitViewControllerDisplayMode_Detail:
        {
            [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[detailContainer]|"
                                                                                           options:NSLayoutFormatDirectionLeftToRight
                                                                                           metrics:metrics
                                                                                             views:views]];
            [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:[masterContainer][detailContainer]"
                                                                                           options:NSLayoutFormatDirectionLeftToRight
                                                                                           metrics:metrics
                                                                                             views:views]];
            break;
        }
        default:
            break;
    }

    NSString *masterFormatString = [NSString stringWithFormat:@"V:|%@[masterContainer]-(keyboardHeight)-|", (self.layoutMasterBelowTopLayoutGuide?@"[topLayoutGuide]":@"")];
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:masterFormatString
                                                                                   options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                   metrics:metrics
                                                                                     views:views]];

    NSString *detailFormatString = [NSString stringWithFormat:@"V:|%@[detailContainer]-(keyboardHeight)-|", (self.layoutDetailBelowTopLayoutGuide?@"[topLayoutGuide]":@"")];
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:detailFormatString
                                                                                   options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                   metrics:metrics
                                                                                     views:views]];
    
    NSString *splitBarFormatString = [NSString stringWithFormat:@"V:|%@[splitBarView]-(keyboardHeight)-|", (self.layoutMasterBelowTopLayoutGuide?@"[topLayoutGuide]":@"")];
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:splitBarFormatString
                                                                                   options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                   metrics:metrics
                                                                                     views:views]];


   //////////////////////////////////////////////////
    if ([layoutConstraints count])
    {
        self.layoutConstraints = [NSArray arrayWithArray:layoutConstraints];
        [self.view addConstraints:self.layoutConstraints];
        [self.view layoutIfNeeded];
    }
}


//==============================================================================
//
//==============================================================================
- (void)_removeLayoutConstraints
{
    if (self.layoutConstraints)
    {
        [self.view removeConstraints:self.layoutConstraints];
        self.layoutConstraints = nil;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - (Private)load viewController


//==============================================================================
//
//==============================================================================
- (void)loadViewController:(UIViewController *)viewController
                  atContainre:(UIView *)container
                     forceAdd:(BOOL)forceAdd
{
    if (viewController)
    {
        [viewController willMoveToParentViewController:self];
        [self addChildViewController:viewController];
        if (forceAdd)
        {
            [container addSubview:viewController.view];
        }
        [viewController didMoveToParentViewController:self];
    }
}



//==============================================================================
//
//==============================================================================
- (void)unloadViewController:(UIViewController *)viewController
{
    if (viewController)
    {
        [viewController willMoveToParentViewController:nil];
        [viewController.view removeFromSuperview];
        [viewController removeFromParentViewController];
        [viewController didMoveToParentViewController:nil];
    }
}


//==============================================================================
//
//==============================================================================
- (void)loadMasterViewController
{
    [self loadViewController:self.masterViewController atContainre:self.masterContainer forceAdd:YES];
    
    //////////////////////////////////////////////////
    [self _layoutMasterViewController];
}


//==============================================================================
//
//==============================================================================
- (void)loadDetailViewController
{
    [self loadViewController:self.detailViewController atContainre:self.detailContainer forceAdd:YES];
    
    //////////////////////////////////////////////////
    [self _layoutDetailViewController];
}



//==============================================================================
//
//==============================================================================
- (void)_layoutMasterViewController
{
    CGRect frame = self.masterContainer.bounds;
    self.masterViewController.view.frame = frame;
}




//==============================================================================
//
//==============================================================================
- (void)_layoutDetailViewController
{
    CGRect frame = self.detailContainer.bounds;
    self.detailViewController.view.frame = frame;
}







////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Responding to View Rotation Events


//================================================================================
//
//================================================================================
- (void)willAnimateRotationToInterfaceOrientation:(UIInterfaceOrientation)toInterfaceOrientation duration:(NSTimeInterval)duration
{
    [super willAnimateRotationToInterfaceOrientation:toInterfaceOrientation duration:duration];

    ////////////////////////////////////////////////////////////////////////////////////////////////////
    
    // 處理旋轉
    [self _resetLayoutConstraintsForPortraint:UIInterfaceOrientationIsPortrait(toInterfaceOrientation) keyboardHeight:self.keyboardHeight];

}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Override


//==============================================================================
//
//==============================================================================
- (void)setMasterViewController:(UIViewController *)masterViewController
{
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
        
        while (blockSelf.isViewDidAppear==NO)
        {
            [NSThread sleepForTimeInterval:0.01];
        }
        //////////////////////////////////////////////////
        dispatch_async(dispatch_get_main_queue(), ^{
            @synchronized(self)
            {
                if (blockSelf.masterViewController!=masterViewController)
                {
                    [blockSelf unloadViewController:blockSelf.masterViewController];
                    
                    [masterViewController retain];
                    [_masterViewController release];
                    _masterViewController = masterViewController;
                    //////////////////////////////////////////////////
                    
                    [blockSelf loadMasterViewController];
                }
            }
        });
    });
}


//==============================================================================
// viewDidAppear之前才能再設定新的DetailViewController
//==============================================================================
- (void)setDetailViewController:(UIViewController *)detailViewController
{
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
        
        while (blockSelf.isViewDidAppear==NO)
        {
            [NSThread sleepForTimeInterval:0.01];
        }
        //////////////////////////////////////////////////
        dispatch_async(dispatch_get_main_queue(), ^{
            @synchronized(self)
            {
                if (blockSelf.detailViewController!=detailViewController)
                {
                    [blockSelf unloadViewController:blockSelf.detailViewController];
                    
                    [detailViewController retain];
                    [_detailViewController release];
                    _detailViewController = detailViewController;
                    //////////////////////////////////////////////////
                    
                    [blockSelf loadDetailViewController];
                }
            }
        });
    });
}


//==============================================================================
//
//==============================================================================
- (BOOL)isCollapsed
{
    return NO;
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Keyboard show/hide


//================================================================================
//
//================================================================================
- (void)registerForKeyboardNotifications
{
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(recvKeyboardWillShow:)
                                                 name:UIKeyboardWillShowNotification object:nil];
    
    [[NSNotificationCenter defaultCenter] addObserver:self
                                             selector:@selector(recvKeyboardWillHidden:)
                                                 name:UIKeyboardWillHideNotification object:nil];
    
}


//================================================================================
//
//================================================================================
- (void)unRegisterForKeyboardNotifications
{
    
    [[NSNotificationCenter defaultCenter] removeObserver:self name:UIKeyboardWillShowNotification object:nil];
    
    [[NSNotificationCenter defaultCenter] removeObserver:self name:UIKeyboardWillHideNotification object:nil];
    
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - recv Keyboard notifivation


//==============================================================================
//
//==============================================================================
- (void)recvKeyboardWillShow:(NSNotification*)notification
{
    NSValue * rectValue = [notification.userInfo objectForKey:UIKeyboardFrameEndUserInfoKey];
    
    if(rectValue!=nil)
    {
        CGRect keyboardRect = [rectValue CGRectValue];
        self.keyboardHeight = MIN(keyboardRect.size.width,keyboardRect.size.height);
        [self _resetLayoutConstraintsForPortraint:UIInterfaceOrientationIsPortrait(self.interfaceOrientation) keyboardHeight:self.keyboardHeight];
    }
}


//==============================================================================
//
//==============================================================================
- (void)recvKeyboardWillHidden:(NSNotification*)notification
{
    self.keyboardHeight = 0;
    [self _resetLayoutConstraintsForPortraint:UIInterfaceOrientationIsPortrait(self.interfaceOrientation) keyboardHeight:self.keyboardHeight];
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - (Public) methods


//==============================================================================
//
//==============================================================================
- (void)lockMasterViewController
{
    [self.masterMaskView removeFromSuperview];
    self.masterMaskView = nil;
    
    self.masterMaskView = [[[UIView alloc] initWithFrame:self.masterViewController.view.bounds] autorelease];
    self.masterMaskView.backgroundColor = [[UIColor blackColor] colorWithAlphaComponent:0.5];
    
    [self.masterViewController.view addSubview:self.masterMaskView];
    
    self.isMasterViewControllerLocked = YES;
}


//==============================================================================
//
//==============================================================================
- (void)unlockMasterViewController
{
    [self.masterMaskView removeFromSuperview];
    self.masterMaskView = nil;

    self.isMasterViewControllerLocked = NO;
}



//==============================================================================
//
//==============================================================================
- (void)lockDetailViewController
{
    [self.detailMaskView removeFromSuperview];
    self.detailMaskView = nil;
    
    self.detailMaskView = [[[UIView alloc] initWithFrame:self.detailViewController.view.bounds] autorelease];
    self.detailMaskView.backgroundColor = [[UIColor blackColor] colorWithAlphaComponent:0.5];
    
    [self.detailViewController.view addSubview:self.detailMaskView];
    
    self.isDetailViewControllerLocked = YES;
}


//==============================================================================
//
//==============================================================================
- (void)unlockDetailViewController
{
    [self.detailMaskView removeFromSuperview];
    self.detailMaskView = nil;
    
    self.isDetailViewControllerLocked = NO;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class methods


//==============================================================================
//
//==============================================================================
+ (void)enableMonitorKeyboard:(BOOL)enableMonitorKeyboard
{
    staticEnableMonitorKeyboard = enableMonitorKeyboard;
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPAppearance


//==============================================================================
//
//==============================================================================
+ (instancetype)appearance
{
    return [PPAppearance appearanceForClass:[self class]];
}
@end
