//
//  PPSideMenuController.m
//  Pods
//
//  Created by sanhue on 2015/6/11.
//
//

#import "PPSideMenuController.h"

static char PPSideMenuContext;

////////////////////////////////////////////////////////////////////////////////////////////////////
typedef NS_ENUM(NSInteger, PPSideMenuGestureState)
{
    PPSideMenuGestureState_None = 0,
    PPSideMenuGestureState_Left,
    PPSideMenuGestureState_Right
};

////////////////////////////////////////////////////////////////////////////////////////////////////
@interface PPSideMenuController () <UIGestureRecognizerDelegate>

/// 記錄Pan之前的Center container的位置
@property (nonatomic, assign) CGPoint centerContainerLocationBeforePan;

/// 至少要移動|minimumMovePercentage*介面寬|的距離，才算合法移動
@property (nonatomic, assign) CGFloat minimumMovePercentage;

/// 開關drawer最多用多久，用來控制drawer的速度
@property (nonatomic, assign) CGFloat maximumAnimationDuration;

/// drawer的背景透明度
@property (nonatomic, assign) CGFloat drawerBackgroundMaskAlpha;

/// 記錄center container frame的大小
@property (nonatomic, assign) CGRect centerContainerFrame;
/// 記錄left container frame的大小
@property (nonatomic, assign) CGRect leftContainerFrame;
/// 記錄right container frame的大小
@property (nonatomic, assign) CGRect rightContainerFrame;

/// 是否改為left, right panel的frame大小
@property (nonatomic, assign) BOOL shouldResizeRightPanel; // defaults to YES
@property (nonatomic, assign) BOOL shouldResizeLeftPanel;  // defaults to YES

/// the visible width of the Panel
@property (nonatomic, assign) CGFloat leftVisibleWidth;
@property (nonatomic, assign) CGFloat rightVisibleWidth;

@property (nonatomic, assign) PPSideMenuGestureState gestureState;
//////////////////////////////////////////////////
// views
@property (nonatomic, retain) UIView *leftContainer;
@property (nonatomic, retain) UIView *centerContainer;
@property (nonatomic, retain) UIView *rightContainer;
/// 用來處理手勢的view,
@property (nonatomic, retain) UIView *tapView;

//////////////////////////////////////////////////
// property for readonly, set readwrite at .m
// 目前顯示的view controller
@property (nonatomic, assign) UIViewController *visibleViewController;
/// current state
@property(nonatomic) PPSideMenuState state;
/// 回傳用來控制手勢的gesture
@property (nonatomic, retain) UIPanGestureRecognizer *centerPanGestureRecognizer;
@property (nonatomic, retain) UIPanGestureRecognizer *leftPanGestureRecognizer;
@property (nonatomic, retain) UIPanGestureRecognizer *rightPanGestureRecognizer;
@end


////////////////////////////////////////////////////////////////////////////////////////////////////
@implementation PPSideMenuController





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - init/dealloc methods


//==============================================================================
//
//==============================================================================
- (instancetype)init
{
    self = [super init];
    if (self)
    {
        [self resetSettingsToDefault];
        [self preparePanGesture];
    }
    return self;
}


//==============================================================================
//
//==============================================================================
- (void)dealloc
{
    self.delegate = nil;
    
    [self unloadSideViewController];
    
    [self.centerViewController.view removeGestureRecognizer:self.centerPanGestureRecognizer];
    [self unloadCenterViewController];
    
    [self removeContainer];

    self.centerViewController = nil;
    self.leftViewController = nil;
    self.rightViewController = nil;
    self.visibleViewController = nil;
    
    [self releasePanGesture];

    //////////////////////////////////////////////////
    [super dealloc];
}


//==============================================================================
//
//==============================================================================
- (void)resetSettingsToDefault
{
    self.state = PPSideMenuState_CenterVisible;
    self.gestureState = PPSideMenuGestureState_None;
    self.portraitOnly = NO;
    
    self.enableLeftPanToOpen = YES;
    self.enableRightPanToOpen = YES;
    
    self.leftGapPercentage = 0.8f;
    self.rightGapPercentage = 0.8f;
    
    self.minimumMovePercentage = 0.15f;
    self.maximumAnimationDuration = 0.2f;
    self.drawerBackgroundMaskAlpha = 0.5f;
    
    self.shouldResizeLeftPanel = YES;
    self.shouldResizeRightPanel = YES;
    
}







////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - view controller life cycle


//==============================================================================
//
//==============================================================================
- (void)viewWillAppear:(BOOL)animated
{
    [super viewWillAppear:animated];
    
    //////////////////////////////////////////////////
    [self prepareUI];
}


//==============================================================================
//
//==============================================================================
- (void)viewDidAppear:(BOOL)animated
{
    [super viewDidAppear:animated];
    
    //////////////////////////////////////////////////
}


//==============================================================================
//
//==============================================================================
- (void)viewWillDisappear:(BOOL)animated
{
    //////////////////////////////////////////////////
    
    [super viewWillDisappear:animated];
}


//==============================================================================
//
//==============================================================================
- (void)viewDidDisappear:(BOOL)animated
{
    // !! 直接做的話，會造成present其他view controller時，childViewController的ViewDidDisappear收不到
    // !! 用dispatch會造成release時先進dealloc再進dispatch block造成當機，所以先手動送出viewdiddisAppear
    for (UIViewController *viewController in self.childViewControllers)
    {
        if (viewController.isViewLoaded)
        {
            [viewController viewDidDisappear:animated];
        }
    }
    
    [self unloadSideViewController];
    [self unloadCenterViewController];
    
    [self removeContainer];

    
    //////////////////////////////////////////////////
    
    [super viewDidDisappear:animated];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - orietation


//==============================================================================
//
//==============================================================================
- (BOOL)shouldAutorotateToInterfaceOrientation:(UIInterfaceOrientation)toInterfaceOrientation
{
    UIViewController *visiblePanel = self.visibleViewController;
    return [visiblePanel shouldAutorotateToInterfaceOrientation:toInterfaceOrientation];
  
}


//==============================================================================
//
//==============================================================================
- (BOOL)shouldAutorotate
{
    UIViewController *visiblePanel = self.visibleViewController;
    
    if ([visiblePanel respondsToSelector:@selector(shouldAutorotate)])
    {
        return [visiblePanel shouldAutorotate];
    }
    else
    {
        return YES;
    }
}


//==============================================================================
//
//==============================================================================
- (void)willAnimateRotationToInterfaceOrientation:(__unused UIInterfaceOrientation)toInterfaceOrientation duration:(NSTimeInterval)duration
{
    self.centerContainer.frame = [self adjustViewFrame];
    self.leftContainer.frame = [self adjustLeftViewFrame];
    self.rightContainer.frame = [self adjustRightViewFrame];

    [self layoutSideContainers:YES duration:duration];
    [self layoutSideViewControllers];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - private prepare view


//==============================================================================
//
//==============================================================================
- (void)prepareContainerUI
{
    [self adjustSelfViewFrame];
    [self.view setClipsToBounds:YES];
    
    self.centerContainer = [[[UIView alloc] initWithFrame:self.view.bounds] autorelease];
    [self.centerContainer setBackgroundColor:[UIColor whiteColor]];
    
    self.leftContainer = [[[UIView alloc] initWithFrame:self.view.bounds] autorelease];
    [self.leftContainer setBackgroundColor:[UIColor clearColor]];
    self.leftContainer.hidden = YES;
    
    self.rightContainer = [[[UIView alloc] initWithFrame:self.view.bounds] autorelease];
    [self.rightContainer setBackgroundColor:[UIColor clearColor]];
    self.rightContainer.hidden = YES;

    // container config
    self.leftContainer.autoresizingMask = UIViewAutoresizingFlexibleHeight | UIViewAutoresizingFlexibleRightMargin;
    self.rightContainer.autoresizingMask = UIViewAutoresizingFlexibleLeftMargin | UIViewAutoresizingFlexibleHeight;
    self.centerContainer.frame =  self.view.bounds;
    self.centerContainer.autoresizingMask = UIViewAutoresizingFlexibleWidth | UIViewAutoresizingFlexibleHeight;
    
    // addsubview
    [self.view addSubview:self.centerContainer];
    [self.view addSubview:self.leftContainer];
    [self.view addSubview:self.rightContainer];
    
    //////////////////////////////////////////////////
    // 預設是顯示中間
    self.state = PPSideMenuState_CenterVisible;

    [self swapOriginViewController:nil
             withNewViewController:self.centerViewController
                       toContainer:self.centerContainer
                   forceAddSubView:YES];
}


//==============================================================================
//
//==============================================================================
- (void)prepareUI
{
    // prepare containter
    [self prepareContainerUI];
    
    [self layoutSideContainers:YES duration:0.2f];
    [self layoutSideViewControllers];
}


//==============================================================================
//
//==============================================================================
- (void)removeContainer
{
    [self.centerContainer removeFromSuperview];
    self.centerContainer = nil;
    
    [self.leftContainer removeFromSuperview];
    self.leftContainer = nil;
    
    [self.rightContainer removeFromSuperview];
    self.rightContainer = nil;
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - private handle view controller in container


//==============================================================================
//
//==============================================================================
- (void)swapOriginViewController:(UIViewController *)originViewController
           withNewViewController:(UIViewController *)newViewController
                     toContainer:(UIView *)container
                 forceAddSubView:(BOOL)forceAddSubView
{
    if (originViewController)
    {
        [originViewController willMoveToParentViewController:nil];
        [originViewController.view removeFromSuperview];
        [originViewController removeFromParentViewController];
    }
    
    if (newViewController)
    {
        [self addChildViewController:newViewController];
        if (forceAddSubView)
        {
            newViewController.view.frame = container.bounds;
            [container addSubview:newViewController.view];
        }
        [newViewController didMoveToParentViewController:self];
    }
}


//==============================================================================
//
//==============================================================================
- (void)resignObserverFromOldViewController:(UIViewController *)oldViewController toNewViewController:(UIViewController *)newViewController
{
    //移除舊的 observer
    [oldViewController removeObserver:self forKeyPath:@"view"];
    // 設定新的 observer
    [newViewController addObserver:self forKeyPath:@"view" options:NSKeyValueObservingOptionInitial context:&PPSideMenuContext];
}


//==============================================================================
//
//==============================================================================
- (void)toggleScrollsToTopForCenter:(BOOL)center left:(BOOL)left right:(BOOL)right
{
    return ;
    // iPhone only supports 1 active UIScrollViewController at a time
//    if (UI_USER_INTERFACE_IDIOM() == UIUserInterfaceIdiomPhone)
//    {
//        [self toggleScrollsToTop:center forView:self.centerContainer];
//        [self toggleScrollsToTop:left forView:self.leftContainer];
//        [self toggleScrollsToTop:right forView:self.rightContainer];
//    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)toggleScrollsToTop:(BOOL)enabled forView:(UIView *)view
{
    if ([view isKindOfClass:[UIScrollView class]])
    {
        UIScrollView *scrollView = (UIScrollView *)view;
        scrollView.scrollsToTop = enabled;
        return YES;
    }
    else
    {
        for (UIView *subview in view.subviews)
        {
            if([self toggleScrollsToTop:enabled forView:subview])
            {
                return YES;
            }
        }
    }
    return NO;
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - private handling pan guesture


//==============================================================================
//
//==============================================================================
- (void)preparePanGesture
{
    self.centerPanGestureRecognizer = [[[UIPanGestureRecognizer alloc] initWithTarget:self action:@selector(recvPanGesture:)] autorelease];
    self.centerPanGestureRecognizer.delegate = self;
    self.centerPanGestureRecognizer.maximumNumberOfTouches = 1;
    self.centerPanGestureRecognizer.minimumNumberOfTouches = 1;
    
    self.leftPanGestureRecognizer = [[[UIPanGestureRecognizer alloc] initWithTarget:self action:@selector(recvPanGesture:)] autorelease];
    self.leftPanGestureRecognizer.delegate = self;
    self.leftPanGestureRecognizer.maximumNumberOfTouches = 1;
    self.leftPanGestureRecognizer.minimumNumberOfTouches = 1;

    self.rightPanGestureRecognizer = [[[UIPanGestureRecognizer alloc] initWithTarget:self action:@selector(recvPanGesture:)] autorelease];
    self.rightPanGestureRecognizer.delegate = self;
    self.rightPanGestureRecognizer.maximumNumberOfTouches = 1;
    self.rightPanGestureRecognizer.minimumNumberOfTouches = 1;
}


//==============================================================================
//
//==============================================================================
- (void)releasePanGesture
{
    self.centerPanGestureRecognizer = nil;
    self.leftPanGestureRecognizer = nil;
    self.rightPanGestureRecognizer = nil;
}


//==============================================================================
//
//==============================================================================
- (void)addPanGestureToView:(UIView *)view
{
    UIPanGestureRecognizer *panGesture = [[UIPanGestureRecognizer alloc] initWithTarget:self action:@selector(recvPanGesture:)];
    panGesture.delegate = self;
    panGesture.maximumNumberOfTouches = 1;
    panGesture.minimumNumberOfTouches = 1;
    [view addGestureRecognizer:panGesture];
    [panGesture release];
}


//==============================================================================
//
//==============================================================================
- (void)recvPanGesture:(UIGestureRecognizer *)sender
{
    if ([sender isKindOfClass:[UIPanGestureRecognizer class]])
    {
        UIPanGestureRecognizer *pan = (UIPanGestureRecognizer *)sender;
        CGPoint location = [pan locationInView:self.centerContainer];
        CGPoint translate = [pan translationInView:self.centerContainer];
        
        UIView *container = self.centerContainer;
        CGRect originFrame = self.centerContainerFrame;
        
        // !!決定目前是要處理哪一個選單
        if (pan.state == UIGestureRecognizerStateBegan)
        {
            // 如果目前顯示中間，表示要判斷是要開啟左選單或右選單
            if (self.state == PPSideMenuState_CenterVisible)
            {
                if (location.x>=0 &&
                    location.x<self.centerContainer.frame.size.width*self.leftGapPercentage)
                {
                    self.gestureState = PPSideMenuGestureState_Left;
                    container = self.leftContainer;
                    
                }
                else if (location.x<self.centerContainer.frame.size.width &&
                         location.x>=self.centerContainer.frame.size.width*self.rightGapPercentage)
                {
                    self.gestureState = PPSideMenuGestureState_Right;
                    container = self.rightContainer;
                }
            }
            else if (self.state == PPSideMenuState_LeftVisible)
            {
                self.gestureState = PPSideMenuGestureState_Left;
                container = self.leftContainer;
            }
            else if (self.state == PPSideMenuState_RightVisible)
            {
                self.gestureState = PPSideMenuGestureState_Right;
                container = self.rightContainer;
            }
            
            _centerContainerLocationBeforePan = container.frame.origin;
        }

        CGFloat originX = 0;
        // 取得要處理的container及原始的frame
        if (self.gestureState==PPSideMenuGestureState_Left)
        {
            originX = -self.leftVisibleWidth;
            container = self.leftContainer;
            originFrame = self.leftContainerFrame;
        }
        else if (self.gestureState==PPSideMenuGestureState_Right)
        {
            originX = self.rightVisibleWidth;
            container = self.rightContainer;
            originFrame = self.rightContainerFrame;
        }

        // 計算變動後的frame
        CGRect toframe = originFrame;
        toframe.origin.x += roundf([self correctMove:translate.x]);;
        
        // 計算alpha的變化，因為要用到目前的frame，所以在container的setFrame前處理
        // 如果位置不變，就不用改alpha
        if (originFrame.origin.x!=toframe.origin.x)
        {
            CGFloat alpha = [self calculatedAlphaWithTotal:originX originPoint:originFrame.origin toPoint:toframe.origin];
            self.tapView.backgroundColor = [[UIColor blackColor] colorWithAlphaComponent:alpha];
        }

        // 設定container的frame
        container.frame = toframe;

        //////////////////////////////////////////////////
        // if center panel has focus, make sure correct side panel is revealed
        if (self.state == PPSideMenuState_CenterVisible)
        {
            if (self.gestureState==PPSideMenuGestureState_Left)
            {
                [self loadLeftViewController];
            }
            else if(self.gestureState==PPSideMenuGestureState_Right)
            {
                [self loadRightViewController];
            }
        }

        if (sender.state == UIGestureRecognizerStateEnded)
        {
            CGFloat deltaX =  toframe.origin.x - _centerContainerLocationBeforePan.x;
            
            // 如果pan的位移大於某個大小，才能移動
            if ([self isValidateMove:deltaX])
            {
                [self completePan:deltaX];
            }
            else
            {
                // 選原到pan之前的狀態
                [self restorePan];
            }
        }

        else if (sender.state == UIGestureRecognizerStateCancelled)
        {
            // 選原到pan之前的狀態
            [self restorePan];
        }
    }
}


//==============================================================================
// 完成pan的動作
//==============================================================================
- (void)completePan:(CGFloat)deltaX
{
    switch (self.state)
    {
        case PPSideMenuState_CenterVisible:
        {
            if (deltaX > 0.0f)
            {
                // !! 只有從左邊拉出來，最後才會顯示左選單
                if (self.gestureState==PPSideMenuGestureState_Left)
                {
                    [self showLeftContainer:YES];
                }
            }
            else
            {
                // !! 只有從右邊拉出來，最後才會顯示右選單
                if(self.gestureState==PPSideMenuGestureState_Right)
                {
                    [self showRightContainer:YES];
                }
            }
            break;
        }
        case PPSideMenuState_LeftVisible:
        {
            [self showCenterContainer:YES];
            break;
        }
        case PPSideMenuState_RightVisible:
        {
            [self showCenterContainer:YES];
            break;
        }
    }
}


//==============================================================================
//
//==============================================================================
- (void)restorePan
{
    switch (self.state)
    {
        case PPSideMenuState_CenterVisible:
        {
            [self showCenterContainer:YES];
            break;
        }
        case PPSideMenuState_LeftVisible:
        {
            [self showLeftContainer:YES];
            break;
        }
        case PPSideMenuState_RightVisible:
        {
            [self showRightContainer:YES];
        }
    }
}


//==============================================================================
// 修正位移值
//==============================================================================
- (CGFloat)correctMove:(CGFloat)movement
{
    CGRect containerFrame = CGRectZero;
    switch (self.gestureState)
    {
        case PPSideMenuGestureState_Left:
        {
            containerFrame = self.leftContainerFrame;
            CGFloat position = containerFrame.origin.x + movement;

            // 限定左選單的位置
            if (position >= 0)
            {
                return movement-position;
            }
            else if (position<-self.leftVisibleWidth)
            {
                return -self.leftVisibleWidth;
            }
            break;
        }
        case PPSideMenuGestureState_Right:
        {
            containerFrame = self.rightContainerFrame;
            CGFloat position = containerFrame.origin.x + movement;
            
            // 限定右選單的位置
            if (position < self.view.bounds.size.width-self.rightVisibleWidth)
            {
                return (self.view.bounds.size.width-self.rightVisibleWidth)-containerFrame.origin.x;
            }
            else if(position>self.view.bounds.size.width)
            {
                return self.rightVisibleWidth;
            }
            break;
        }
        case PPSideMenuGestureState_None:
        default:
        {
            break;
        }
    }
    return movement;
}


//==============================================================================
//
//==============================================================================
- (BOOL)isValidateMove:(CGFloat)deltaX
{
    CGFloat minimum = floorf(self.view.bounds.size.width * self.minimumMovePercentage);
    switch (self.state) {
        case PPSideMenuState_LeftVisible:
        {
            return deltaX <= -minimum;
        }
        case PPSideMenuState_CenterVisible:
        {
            return ABS(deltaX) >= minimum;
        }
        case PPSideMenuState_RightVisible:
        {
            return deltaX >= minimum;
        }
        default:
        {
            return NO;
        }
    }
    return NO;
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - private handling tap gesture


//==============================================================================
//
//==============================================================================
- (void)addTapGestureToView:(UIView *)view
{
    UITapGestureRecognizer *tapGesture = [[UITapGestureRecognizer alloc] initWithTarget:self action:@selector(centerPanelTapped:)];
    [view addGestureRecognizer:tapGesture];
    [tapGesture release];
}


//==============================================================================
//
//==============================================================================
- (void)centerPanelTapped:(__unused UIGestureRecognizer *)gesture
{
    [self showCenterContainer:YES];
}



////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Animation (reference from JASidePanelController)


//==============================================================================
//
//==============================================================================
- (CGFloat)calculatedAlphaWithTotal:(CGFloat)total originPoint:(CGPoint)originPoint toPoint:(CGPoint)toPoint
{

    //  判斷是要開啟選單還是關閉選單
    CGFloat originMoved = toPoint.x - originPoint.x;
    CGFloat moveRatio = originMoved / total;
    CGFloat absMoveRatio = 0.0;
    
    // 如果originMove與total同樣是正數或同樣是負數, 表示是要關閉
    if (moveRatio<0)
    {
        // 開啟選單
        absMoveRatio = ABS(moveRatio);
    }
    else
    {
        // 關閉選單
        absMoveRatio = (1.0-ABS(moveRatio));
    }

    // 只取到小數點下兩位
    NSString *alphaString = [NSString stringWithFormat:@"%.2f", self.drawerBackgroundMaskAlpha * absMoveRatio];
    CGFloat alpha = [alphaString floatValue];
    return ABS(alpha);
}

//==============================================================================
//
//==============================================================================
- (CGFloat)calculatedDurationWithOriginPoint:(CGPoint)originPoint currentPoint:(CGPoint)currentPoint toPoint:(CGPoint)toPoint
{
    CGFloat remaining = ABS(currentPoint.x - toPoint.x);
    CGFloat max = originPoint.x == toPoint.x ? remaining : ABS(originPoint.x - toPoint.x);
    return max > 0.0f ? self.maximumAnimationDuration * (remaining / max) : self.maximumAnimationDuration;
}


//==============================================================================
//
//==============================================================================
- (void)movePanelwithAnimation:(BOOL)animation completion:(void (^)(BOOL finished))completion
{
    UIView *targetContainer = nil;
    CGRect toContainerFrame = CGRectZero;
    CGRect originContainerFrame = CGRectZero;
    CGRect toCenterContainerFrame = [self adjustViewFrame];
    CGRect toLeftContainerFrame = [self adjustLeftViewFrame];
    CGRect toRightContainerFrame = [self adjustRightViewFrame];
    
    switch (self.state) {
        case PPSideMenuState_CenterVisible:
        {
            targetContainer = self.centerContainer;
            toContainerFrame = toCenterContainerFrame;
            break;
        }
        case PPSideMenuState_LeftVisible:
        {
            targetContainer = self.leftContainer;
            toContainerFrame = toLeftContainerFrame;
            originContainerFrame.origin.x = toLeftContainerFrame.origin.x-toLeftContainerFrame.size.width;
            break;
        }
        case PPSideMenuState_RightVisible:
        {
            targetContainer = self.rightContainer;
            toContainerFrame = toRightContainerFrame;
            originContainerFrame.origin.x = toRightContainerFrame.origin.x-toRightContainerFrame.size.width;

            break;
        }
        default:
            break;
    }
    
    if (targetContainer==nil)
    {
        return ;
    }
    
    CGFloat duration = [self calculatedDurationWithOriginPoint:originContainerFrame.origin currentPoint:targetContainer.frame.origin toPoint:toContainerFrame.origin];
    
    //////////////////////////////////////////////////
    typedef void (^AnimationHandler)(void);
    
    AnimationHandler animationHandler = ^() {

        if (self.state==PPSideMenuState_CenterVisible)
        {
            self.tapView.backgroundColor = [[UIColor blackColor] colorWithAlphaComponent:0.0];
        }
        else
        {
            self.tapView.backgroundColor = [[UIColor blackColor] colorWithAlphaComponent:self.drawerBackgroundMaskAlpha];
        }

        self.centerContainer.frame = toCenterContainerFrame;
        self.leftContainer.frame = toLeftContainerFrame;
        self.rightContainer.frame = toRightContainerFrame;
    };

    //////////////////////////////////////////////////
    typedef void (^CompletionHandler)(BOOL finished);

    CompletionHandler completionHandler = ^(BOOL finished) {
        
        // 如果是PPSideMenuState_CenterVisible，要把tapView清空
        if (self.state==PPSideMenuState_CenterVisible)
        {
            self.tapView = nil;
        }
        
        if (completion)
        {
            completion(finished);
        }
    };
    
    //////////////////////////////////////////////////
    if (animation)
    {
        [UIView animateWithDuration:duration delay:0.0f options:UIViewAnimationOptionCurveLinear|UIViewAnimationOptionLayoutSubviews animations:animationHandler completion:completionHandler];
    }
    else
    {
        animationHandler();
        completionHandler(YES);
    }
    
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Sizing views


//==============================================================================
//
//==============================================================================
- (void)adjustSelfViewFrame
{
    CGSize viewSize = self.view.bounds.size;
    if(self.portraitOnly || UIInterfaceOrientationIsPortrait(self.interfaceOrientation))
    {
        //直式
        self.view.bounds = CGRectMake(0, 0, MIN(viewSize.width, viewSize.height),  MAX(viewSize.width, viewSize.height));
    }
    else
    {
        // 橫式
        self.view.bounds = CGRectMake(0, 0, MAX(viewSize.width, viewSize.height),  MIN(viewSize.width, viewSize.height));
        
    }
}



//==============================================================================
//
//==============================================================================
- (CGRect)adjustViewFrame
{
    CGRect frame = self.view.bounds;
    self.centerContainerFrame = frame;
    return frame;
}



//==============================================================================
//
//==============================================================================
- (CGRect)adjustLeftViewFrame
{
    CGRect frame = self.view.bounds;
    CGFloat width = self.leftVisibleWidth;

    switch (self.state)
    {
        case PPSideMenuState_CenterVisible:
        {
            frame.origin.x = -width;
            break;
        }
        case PPSideMenuState_LeftVisible:
        {
            frame.origin.x = 0.0f;
            break;
        }
        case PPSideMenuState_RightVisible:
        {
            frame.origin.x = -width;
            break;
        }
        default:
        {
            break;
        }
    }
    
    frame.size.width = width;
    self.leftContainerFrame = frame;
    return frame;
}



//==============================================================================
//
//==============================================================================
- (CGRect)adjustRightViewFrame
{
    CGRect frame = self.view.bounds;
    CGFloat width = self.rightVisibleWidth;

    switch (self.state)
    {
        case PPSideMenuState_CenterVisible:
        {
            frame.origin.x = self.view.bounds.size.width;
            break;
        }
        case PPSideMenuState_LeftVisible:
        {
            frame.origin.x = self.view.bounds.size.width;
            break;
        }
        case PPSideMenuState_RightVisible:
        {
            frame.origin.x = ABS(frame.size.width-width);
            break;
        }
        default:
        {
            break;
        }
    }
    frame.size.width = width;
    self.rightContainerFrame = frame;
    return frame;
}

//==============================================================================
//
//==============================================================================
- (CGFloat)leftVisibleWidth
{
    if (self.shouldResizeLeftPanel==NO)
    {
        return self.view.bounds.size.width;
    }
    else
    {
        return floorf(self.view.bounds.size.width * self.leftGapPercentage);
    }
}


//==============================================================================
//
//==============================================================================
- (CGFloat)rightVisibleWidth
{
    if (self.shouldResizeRightPanel==NO)
    {
        return self.view.bounds.size.width;
    }
    else
    {
        return floorf(self.view.bounds.size.width * self.rightGapPercentage);
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Showing containers


//==============================================================================
//
//==============================================================================
- (void)showLeftContainer:(BOOL)animated
{
    self.state = PPSideMenuState_LeftVisible;
    [self loadLeftViewController];
   
    [self movePanelwithAnimation:animated completion:^(BOOL finished) {
        if ([self.delegate respondsToSelector:@selector(sideMenuControllerDidShowLeftPanel:)])
        {
            [self.delegate sideMenuControllerDidShowLeftPanel:self];
        }
    }];
}


//==============================================================================
//
//==============================================================================
- (void)showRightContainer:(BOOL)animated
{
    self.state = PPSideMenuState_RightVisible;
    [self loadRightViewController];
    
    [self movePanelwithAnimation:animated completion:^(BOOL finished) {
        if ([self.delegate respondsToSelector:@selector(sideMenuControllerDidShowRightPanel:)])
        {
            [self.delegate sideMenuControllerDidShowRightPanel:self];
        }
    }];
}


//==============================================================================
//
//==============================================================================
- (void)showCenterContainer:(BOOL)animated
{
    self.state = PPSideMenuState_CenterVisible;
    
    [self adjustViewFrame];
    
    [self movePanelwithAnimation:animated completion:^(BOOL finished) {
        self.leftContainer.hidden = YES;
        self.rightContainer.hidden = YES;
        [self unloadSideViewController];
    }];
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Styling views


//==============================================================================
//
//==============================================================================
//- (void)addMaskViewBelowContainer:(UIView*)container
//{
//    <#Do something here#>
//}
//
//
////==============================================================================
////
////==============================================================================
//- (void)removeMaskView
//{
//    <#Do something here#>
//}


//==============================================================================
//
//==============================================================================
- (void)layoutSideContainers:(BOOL)animate duration:(NSTimeInterval)duration
{
//    CGRect leftFrame = self.view.bounds;
//    CGRect rightFrame = self.view.bounds;
//    
//    leftFrame.size.width = self.leftVisibleWidth;
//    rightFrame.size.width = self.rightVisibleWidth;
    
    self.leftContainer.frame = [self adjustLeftViewFrame];
    self.rightContainer.frame = [self adjustRightViewFrame];
    
    [self styleContainer:self.leftContainer animate:animate duration:duration];
    [self styleContainer:self.rightContainer animate:animate duration:duration];
}

//==============================================================================
//
//==============================================================================
- (void)layoutSideViewControllers
{
    if (self.rightViewController.isViewLoaded)
    {
        self.rightViewController.view.frame = self.rightContainer.bounds;
    }
    
    if (self.leftViewController.isViewLoaded)
    {
        self.leftViewController.view.frame = self.leftContainer.bounds;
    }
}


//==============================================================================
//
//==============================================================================
- (void)styleContainer:(UIView *)container animate:(BOOL)animate duration:(NSTimeInterval)duration
{
    UIBezierPath *shadowPath = [UIBezierPath bezierPathWithRoundedRect:container.bounds cornerRadius:0.0f];
//    if (animate)
//    {
//        CABasicAnimation *animation = [CABasicAnimation animationWithKeyPath:@"shadowPath"];
//        animation.fromValue = (id)container.layer.shadowPath;
//        animation.toValue = (id)shadowPath.CGPath;
//        animation.duration = duration;
//        [container.layer addAnimation:animation forKey:@"shadowPath"];
//    }
    container.layer.shadowPath = shadowPath.CGPath;
    container.layer.shadowColor = [UIColor blackColor].CGColor;
    container.layer.shadowRadius = 10.0f;
    container.layer.shadowOpacity = 0.75f;
    container.clipsToBounds = NO;
}


//==============================================================================
//
//==============================================================================
- (void)styleView:(UIView *)view
{
    view.clipsToBounds = YES;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - load side menu controllers


//==============================================================================
//
//==============================================================================
- (void)loadLeftViewController
{
    self.rightContainer.hidden = YES;
    if (self.leftContainer.hidden && self.leftViewController)
    {
        if (!self.leftViewController.view.superview)
        {
            [self.leftViewController.view addGestureRecognizer:self.leftPanGestureRecognizer];
            
            [self layoutSideViewControllers];
        
            self.leftViewController.view.autoresizingMask = UIViewAutoresizingFlexibleWidth | UIViewAutoresizingFlexibleHeight;
            [self styleView:self.leftViewController.view];
            [self.leftContainer addSubview:self.leftViewController.view];
            
            //////////////////////////////////////////////////
            
            if (self.tapView==nil)
            {
                self.tapView = [[[UIView alloc] init] autorelease];
                self.tapView.backgroundColor = [[UIColor blackColor] colorWithAlphaComponent:0.0];
            }
        }
        
        self.leftContainer.hidden = NO;
    }
}


//==============================================================================
//
//==============================================================================
- (void)loadRightViewController
{
    self.leftContainer.hidden = YES;
    if (self.rightContainer.hidden && self.rightViewController)
    {
        
        if (!self.rightViewController.view.superview)
        {
            [self.rightViewController.view addGestureRecognizer:self.rightPanGestureRecognizer];
            
            [self layoutSideViewControllers];
            
            self.rightViewController.view.autoresizingMask = UIViewAutoresizingFlexibleWidth | UIViewAutoresizingFlexibleHeight;
            [self styleView:self.rightViewController.view];
            [self.rightContainer addSubview:self.rightViewController.view];
            
            
            if (self.tapView==nil)
            {
                self.tapView = [[[UIView alloc] init] autorelease];
                self.tapView.backgroundColor = [[UIColor blackColor] colorWithAlphaComponent:0.0];
            }
        }
        
        self.rightContainer.hidden = NO;
    }
}


//==============================================================================
//
//==============================================================================
- (void)unloadCenterViewController
{
    if (self.centerViewController.isViewLoaded)
    {
        [self.centerViewController.view removeFromSuperview];
    }
}


//==============================================================================
//
//==============================================================================
- (void)unloadSideViewController
{
    if (self.leftViewController.isViewLoaded)
    {
        [self.leftViewController.view removeGestureRecognizer:self.leftPanGestureRecognizer];
        [self.leftViewController.view removeFromSuperview];
    }

    if (self.rightViewController.isViewLoaded)
    {
        [self.rightViewController.view removeGestureRecognizer:self.rightPanGestureRecognizer];
        [self.rightViewController.view removeFromSuperview];
    }
    
    self.tapView = nil;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - property override


//==============================================================================
//
//==============================================================================
- (void)setCenterViewController:(UIViewController *)centerViewController
{
    // 因為之後流程就會把 |_centerViewController| 先release
    // 但release後又會用到 previous, 所以要先retain
    UIViewController *previous = [_centerViewController retain];
    if (centerViewController != _centerViewController)
    {

        [centerViewController retain];
        [_centerViewController release];
        _centerViewController = centerViewController;

        // 重設observer
        [self resignObserverFromOldViewController:previous toNewViewController:centerViewController];


        if (self.state == PPSideMenuState_CenterVisible)
        {
            self.visibleViewController = _centerViewController;
        }
        else
        {
            self.visibleViewController = nil;
        }

    }
    
    if (self.isViewLoaded && self.state == PPSideMenuState_CenterVisible)
    {
        // 如果已load起來，且目前是顯示中間，只要重設view controller就好
        [self swapOriginViewController:previous
                 withNewViewController:_centerViewController
                           toContainer:self.centerContainer
                       forceAddSubView:YES];
    }
    else if (self.isViewLoaded)
    {
        // 如果已load起來，但目前不是顯示中間，要重設為顯示中間頁面
        // update the state immediately to prevent user interaction on the side panels while animating
        self.state = PPSideMenuState_CenterVisible;
        
        [UIView animateWithDuration:0.2f animations:^{
            self.centerContainer.frame = self.centerContainerFrame;
        } completion:^(__unused BOOL finished) {
            [self swapOriginViewController:previous
                     withNewViewController:_centerViewController
                               toContainer:self.centerContainer
                           forceAddSubView:YES];
            
            // 立刻顯示中間頁面
            [self showCenterViewControllerAnimated:YES];
        }];
    }

    [previous release];
}

//==============================================================================
//
//==============================================================================
- (void)setLeftViewController:(UIViewController *)leftViewController
{
    if (_leftViewController != leftViewController)
    {
        [self swapOriginViewController:_leftViewController
                 withNewViewController:leftViewController
                           toContainer:self.leftContainer
                       forceAddSubView:NO];
        
        [leftViewController retain];
        [_leftViewController release];
        _leftViewController = leftViewController;
        
        if (self.state == PPSideMenuState_LeftVisible)
        {
            self.visibleViewController = _leftViewController;
        }
        else
        {
            self.visibleViewController = nil;
        }
    }
}



//==============================================================================
//
//==============================================================================
- (void)setRightViewController:(UIViewController *)rightViewController
{
    if (_rightViewController != rightViewController)
    {
        [self swapOriginViewController:_rightViewController
                 withNewViewController:rightViewController
                           toContainer:self.rightContainer
                       forceAddSubView:NO];
        
        [rightViewController retain];
        [_rightViewController release];
        _rightViewController = rightViewController;
        
        if (self.state == PPSideMenuState_RightVisible)
        {
            self.visibleViewController = _rightViewController;
        }
        else
        {
            self.visibleViewController = nil;
        }
    }
}


//==============================================================================
//
//==============================================================================
- (void)setTapView:(UIView *)tapView
{
    if (tapView != _tapView)
    {
        [tapView retain];
        if (_tapView!=nil)
        {
            [_tapView removeFromSuperview];
            [_tapView release];
        }
        _tapView = tapView;
        
        //////////////////////////////////////////////////
        if (_tapView)
        {
            _tapView.frame = self.centerContainer.bounds;
            _tapView.autoresizingMask = UIViewAutoresizingFlexibleWidth | UIViewAutoresizingFlexibleHeight;
            [self addTapGestureToView:_tapView];
            [self addPanGestureToView:_tapView];
            
            [self.centerContainer addSubview:_tapView];
        }
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - normal usage methods


//==============================================================================
//
//==============================================================================
- (void)showLeftViewControllerAnimated:(BOOL)animated
{
    [self showLeftContainer:animated];
}


//==============================================================================
//
//==============================================================================
- (void)showCenterViewControllerAnimated:(BOOL)animated
{
    [self showCenterContainer:animated];
}


//==============================================================================
//
//==============================================================================
- (void)showRightViewControllerAnimated:(BOOL)animated
{
    [self showRightContainer:animated];
}


//==============================================================================
//
//==============================================================================
- (void)toggleLeftViewController
{
    if (self.state == PPSideMenuState_LeftVisible)
    {
        [self showCenterContainer:YES];
    }
    else if (self.state == PPSideMenuState_CenterVisible)
    {
        [self showLeftContainer:YES];
    }
}


//==============================================================================
//
//==============================================================================
- (void)toggleRightViewController
{
    if (self.state == PPSideMenuState_RightVisible)
    {
        [self showCenterContainer:YES];
    }
    else if (self.state == PPSideMenuState_CenterVisible)
    {
        [self showRightContainer:YES];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - style





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Gesture Recognizer Delegate


//==============================================================================
//
//==============================================================================
- (BOOL)gestureRecognizerShouldBegin:(UIGestureRecognizer *)gestureRecognizer
{
    if (gestureRecognizer.view == self.tapView)
    {
        return YES;
    }
    else if ([gestureRecognizer isKindOfClass:[UIPanGestureRecognizer class]])
    {
        UIPanGestureRecognizer *panGesture = (UIPanGestureRecognizer *)gestureRecognizer;
        if (panGesture.view == self.centerViewController.view)
        {
            CGPoint translate = [panGesture translationInView:self.centerContainer];
            CGPoint locationInView = [panGesture locationInView:self.centerContainer];
            
            // 只有在最左與最右時會觸發
            if ((self.enableLeftPanToOpen && locationInView.x <= self.centerContainer.bounds.size.width *0.2  && self.leftViewController)||
                (self.enableRightPanToOpen && locationInView.x >= self.centerContainer.bounds.size.width *0.8 && self.rightViewController))
            {
                // 如果水平位移量比垂直大，才可以觸發
                BOOL possible = translate.x != 0 && ((ABS(translate.y) / ABS(translate.x)) < 1.0f);
                if (possible &&
                    ((translate.x > 0 && self.leftViewController) ||
                     (translate.x < 0 && self.rightViewController)))
                {
                    return YES;
                }
                
                return NO;
            }
            
        }
        else
        {
            return YES;
        }
        
        // 下面的做法是左邊出現時，只能往左關閉；右邊出現時，只以往右關閉，
        // 但左邊出現時，如果先往左，再往右，還是擋不了，所以這邊先不擋，
        // recvPanGesture的地方，是通用的，所以沒有影響
//        else if (panGesture.view == self.leftViewController.view)
//        {
//            CGPoint translate = [panGesture translationInView:self.leftContainer];
//            
//            // 如果水平位移量比垂直大，才可以觸發
//            BOOL possible = translate.x != 0 && ((ABS(translate.y) / ABS(translate.x)) < 1.0f);
//            if (possible &&
//                (translate.x < 0 && self.leftViewController))
//            {
//                return YES;
//            }
//            
//            return NO;
//        }
//        else if (panGesture.view == self.rightViewController.view)
//        {
//            CGPoint translate = [panGesture translationInView:self.rightContainer];
//            
//            // 如果水平位移量比垂直大，才可以觸發
//            BOOL possible = translate.x != 0 && ((ABS(translate.y) / ABS(translate.x)) < 1.0f);
//            if (possible &&
//                (translate.x > 0 && self.rightViewController))
//            {
//                return YES;
//            }
//            
//            return NO;
//        }
    }
    return NO;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Key Value Observing


//==============================================================================
//
//==============================================================================
- (void)observeValueForKeyPath:(NSString *)keyPath ofObject:(id)object change:(__unused NSDictionary *)change context:(void *)context
{
//    NSLog(@"%s \n===\nkeyPath: %@\nobject: %@\nchange: %@\ncontext: %lx)", __PRETTY_FUNCTION__, keyPath, object, change, context);
    
    if (context == &PPSideMenuContext)
    {
        if ([keyPath isEqualToString:@"view"])
        {
            if (object==self.centerViewController &&
                self.centerViewController.isViewLoaded &&
                self.centerPanGestureRecognizer)
            {
                [self.centerViewController.view addGestureRecognizer:self.centerPanGestureRecognizer];
            }
        }
//        else if ([keyPath isEqualToString:@"viewControllers"] && object == self.centerPanel)
//        {
//            // view controllers have changed, need to replace the button
//            [self _placeButtonForLeftPanel];
//        }
    }
    else
    {
        [super observeValueForKeyPath:keyPath ofObject:object change:change context:context];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - class methods


//==============================================================================
//
//==============================================================================
+ (UIImage *)copyDefaultImage
{
    static UIImage *defaultImage = nil;
//    static dispatch_once_t onceToken;
//    dispatch_once(&onceToken, ^{
        UIGraphicsBeginImageContextWithOptions(CGSizeMake(20.f, 13.f), NO, 0.0f);
        
        [[UIColor blackColor] setFill];
        [[UIBezierPath bezierPathWithRect:CGRectMake(0, 0, 20, 1)] fill];
        [[UIBezierPath bezierPathWithRect:CGRectMake(0, 5, 20, 1)] fill];
        [[UIBezierPath bezierPathWithRect:CGRectMake(0, 10, 20, 1)] fill];
        
        [[UIColor whiteColor] setFill];
        [[UIBezierPath bezierPathWithRect:CGRectMake(0, 1, 20, 2)] fill];
        [[UIBezierPath bezierPathWithRect:CGRectMake(0, 6,  20, 2)] fill];
        [[UIBezierPath bezierPathWithRect:CGRectMake(0, 11, 20, 2)] fill];
        
        defaultImage = UIGraphicsGetImageFromCurrentImageContext();
        UIGraphicsEndImageContext();
        
//    });
    return defaultImage;
}

@end
