//
//  PPScrollPageController.m
//  Pods
//
//  Created by sanhue on 2016/10/18.
//
//

#import "PPScrollPageController.h"
#import "NSTimer+Additions.h"

static CGFloat MinMovePercentage = 0.15f;
static NSInteger PPScrollPageController_PreloadPage = 0;

////////////////////////////////////////////////////////////////////////////////////////////////////
@interface PPScrollPageController () <UIScrollViewDelegate>


////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - views

@property (nonatomic, retain) UIScrollView *pageScrollView;


////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - control setting

/// 至少要移動|minimumMovePercentage*介面寬|的距離，才算合法移動
@property (nonatomic, assign) CGFloat minimumMovePercentage;

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - data

@property (nonatomic, retain) NSArray *layoutConstraints;
@property (nonatomic, retain) UIPanGestureRecognizer *panGestureRecognizer;

@property (nonatomic, assign) NSInteger nextPageIndex;
@property (nonatomic, retain) NSMutableArray *pageViewControllers;

@property (nonatomic, assign) CGFloat oldContentOffset;

@property (nonatomic, retain) NSTimer *clearInvisibleViewTimer;
@end
////////////////////////////////////////////////////////////////////////////////////////////////////
@implementation PPScrollPageController





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - init/dealloc methods


//==============================================================================
//
//==============================================================================
- (instancetype)init
{
    self = [super init];
    if (self)
    {
        self.pageViewControllers = [NSMutableArray array];
        
        //////////////////////////////////////////////////
        self.minimumMovePercentage = MinMovePercentage;

    }
    return self;
}


//==============================================================================
//
//==============================================================================
- (void)dealloc
{
    [self removeClearInvisibleViewTimer];
    //////////////////////////////////////////////////

    // release data
    self.dataSource = nil;
    self.delegate = nil;
    
    self.pageViewControllers = nil;
    
    //////////////////////////////////////////////////
    // release view
    [self removeMainUI];
    
    [self removePanGestureRecognizer];

    //////////////////////////////////////////////////
    [super dealloc];
}







////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - view controller life cycle


//==============================================================================
//
//==============================================================================
- (void)viewWillAppear:(BOOL)animated
{
    [super viewWillAppear:animated];
    
    //////////////////////////////////////////////////
    
    [self prepareMainUI];
    
    //////////////////////////////////////////////////
    
    [self preparePanGestureRecognizer];

    [self loadPageData];
}


//==============================================================================
//
//==============================================================================
- (void)viewDidAppear:(BOOL)animated
{
    [super viewDidAppear:animated];
    
    //////////////////////////////////////////////////
}


//==============================================================================
//
//==============================================================================
- (void)viewWillDisappear:(BOOL)animated
{
    //////////////////////////////////////////////////
    
    [super viewWillDisappear:animated];
}


//==============================================================================
//
//==============================================================================
- (void)viewDidDisappear:(BOOL)animated
{
    [self removeMainUI];
    //////////////////////////////////////////////////
    
    [self removePanGestureRecognizer];
    
    //////////////////////////////////////////////////

    [super viewDidDisappear:animated];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - prepare ui


//==============================================================================
//
//==============================================================================
- (void)prepareMainUI
{
    self.pageScrollView = [[[UIScrollView alloc] init] autorelease];
    if (self.pageScrollView)
    {
        [self.pageScrollView setTranslatesAutoresizingMaskIntoConstraints:NO];
        [self.pageScrollView setDelegate:self];
        [self.pageScrollView setBackgroundColor:[UIColor clearColor]];
        [self.pageScrollView setBounces:NO];
        [self.pageScrollView setShowsVerticalScrollIndicator:NO];
        [self.pageScrollView setShowsHorizontalScrollIndicator:NO];        
        // !! 0026334: [1.0.0(1088)] 通知介面，點進去詳細資訊->左右滑動切換上/下一筆通知時，以雙指滑動的時候會看到畫面整個變藍，參考圖片
        // 下面兩個一定要設為NO
        [self.pageScrollView setScrollEnabled:NO];
        [self.pageScrollView setPagingEnabled:NO];

        [self.view addSubview:self.pageScrollView];
    }
    
    //////////////////////////////////////////////////
    [self resetLayoutConstraints];
}


//==============================================================================
//
//==============================================================================
- (void)removeMainUI
{
    [self removeLayoutConstraints];
    [self clearInvisibleView];
    
    //////////////////////////////////////////////////
    [self.pageScrollView removeFromSuperview];
    self.pageScrollView = nil;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - layout constraints


//==============================================================================
//
//==============================================================================
- (void)resetLayoutConstraints
{
    if(self.pageScrollView==nil)
    {
        return ;
    }
    [self removeLayoutConstraints];
    //////////////////////////////////////////////////
    NSDictionary *views = @{@"pageScrollView":self.pageScrollView,
                            @"topLayoutGuide":self.topLayoutGuide};
    NSDictionary *metrics = nil;
    
    NSMutableArray *layoutConstraints = [NSMutableArray array];
    
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[pageScrollView]|"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:metrics
                                                                                     views:views]];
    
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|[topLayoutGuide][pageScrollView]|"
                                                                                   options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                   metrics:metrics
                                                                                     views:views]];
    
    //////////////////////////////////////////////////
    if ([layoutConstraints count])
    {
        self.layoutConstraints = [NSArray arrayWithArray:layoutConstraints];
        [self.view addConstraints:self.layoutConstraints];
        [self.view layoutIfNeeded];
    }
}


//==============================================================================
//
//==============================================================================
- (void)removeLayoutConstraints
{
    if (self.layoutConstraints)
    {
        [self.view removeConstraints:self.layoutConstraints];
        self.layoutConstraints = nil;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Prepare data


//==============================================================================
//
//==============================================================================
- (void)loadPageData
{
    if([self.dataSource respondsToSelector:@selector(numberOfPageAtController:)] &&
       [self.dataSource respondsToSelector:@selector(scrollPageController:viewControllerAtIndex:)])
    {
        __block typeof(self) blockSelf = self;
        
        dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
            
            [blockSelf.pageViewControllers removeAllObjects];
            
            NSInteger numberOfPage = [blockSelf.dataSource numberOfPageAtController:blockSelf];
            


            //////////////////////////////////////////////////
            // 顯示頁面
            dispatch_async(dispatch_get_main_queue(), ^{
                
                for (NSInteger pageIndex=0; pageIndex<numberOfPage; pageIndex++)
                {
                    UIViewController *pageViewController = [blockSelf.dataSource scrollPageController:blockSelf viewControllerAtIndex:pageIndex];
                    [blockSelf.pageViewControllers addObject:pageViewController];
                }
                
                // 設定contentSize
                [blockSelf.pageScrollView setContentSize:CGSizeMake(blockSelf.view.bounds.size.width * numberOfPage, blockSelf.pageScrollView.bounds.size.height)];

                [blockSelf showPageAtIndex:self.currentPageIndex animated:NO];
            });
        });
    }
}


//==============================================================================
//
//==============================================================================
- (void)showPageAtIndex:(NSInteger)pageIndex animated:(BOOL)animated
{
    if (pageIndex>=[self.pageViewControllers count])
    {
        return ;
    }
    
    //////////////////////////////////////////////////
    // 設定pageScrollview offset
    [self.pageScrollView setContentOffset:CGPointMake(self.pageScrollView.frame.size.width*pageIndex, 0)];
    // 加入新頁面
    [self sendWillShowPageAtIndex:pageIndex];
    
    [self addPageWithPageIndex:pageIndex];
    
    [self sendDidShowPageAtIndex:pageIndex];
    
    // 移除舊頁面
    [self clearInvisibleView];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - clear invisible page timer


//==============================================================================
//
//==============================================================================
- (void)startClearInvisibleViewTimer
{
    __block typeof(self) blockSelf = self;
    
    self.clearInvisibleViewTimer = [NSTimer pp_scheduledTimerWithTimeInterval:0.5
                                                                      repeats:NO
                                                                        block:^(NSTimer * _Nonnull timer) {
                                                                            [blockSelf clearInvisibleView];
                                                                     }];
}


//==============================================================================
//
//==============================================================================
- (void)stopClearInvisibleViewTimer
{
    [self.clearInvisibleViewTimer invalidate];
}


//==============================================================================
//
//==============================================================================
- (void)removeClearInvisibleViewTimer
{
    [self.clearInvisibleViewTimer invalidate];
    self.clearInvisibleViewTimer = nil;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Prepare gesture


//==============================================================================
//
//==============================================================================
- (void)preparePanGestureRecognizer
{
    self.panGestureRecognizer = [[[UIPanGestureRecognizer alloc] initWithTarget:self action:@selector(recvPanGesture:)] autorelease];
    self.panGestureRecognizer.maximumNumberOfTouches = 1;
    self.panGestureRecognizer.minimumNumberOfTouches = 1;
    
    [self.pageScrollView addGestureRecognizer:self.panGestureRecognizer];
}


//==============================================================================
//
//==============================================================================
- (void)removePanGestureRecognizer
{
    [self.pageScrollView removeGestureRecognizer:self.panGestureRecognizer];
    
    self.panGestureRecognizer = nil;
}


//==============================================================================
//
//==============================================================================
- (void)recvPanGesture:(UIGestureRecognizer *)sender
{
//    NSLog(@"%s", __PRETTY_FUNCTION__);

    if ([sender isKindOfClass:[UIPanGestureRecognizer class]])
    {
        UIPanGestureRecognizer *pan = (UIPanGestureRecognizer *)sender;
//        CGPoint location = [pan locationInView:self.view];
        CGPoint translate = [pan translationInView:self.view];
        
//        NSLog(@"state:%ld - %lf - %lf - %@", pan.state, translate.x, location.x, NSStringFromCGPoint(self.pageScrollView.contentOffset));

        
        // 記錄一開始的content offset
        if(pan.state==UIGestureRecognizerStateBegan)
        {
            [self stopClearInvisibleViewTimer];
            
            self.oldContentOffset = self.pageScrollView.contentOffset.x;
//            NSLog(@"start Content offset :%lf", self.oldContentOffset);
        }
        
        if(self.oldContentOffset - translate.x < 0 ||
           self.oldContentOffset - translate.x >= self.pageScrollView.frame.size.width * ([self.pageViewControllers count]-1))
        {
            return;
        }
        
        //////////////////////////////////////////////////
        // 判斷是往左移(下一面)或是往右移(上一頁)
        if (translate.x>0)
        {
//            NSLog(@"右移");
            self.nextPageIndex = self.currentPageIndex-1;
        }
        else
        {
//            NSLog(@"左移");
            self.nextPageIndex = self.currentPageIndex+1;
        }
        
        if (self.nextPageIndex>=(NSInteger)[self.pageViewControllers count])
        {
            self.nextPageIndex = [self.pageViewControllers count]-1;
        }

        if(self.nextPageIndex <0)
        {
            self.nextPageIndex = 0;
        }


        //////////////////////////////////////////////////
        // add subview
        [self addPageWithPageIndex:self.nextPageIndex];

        // 設定scroll的 contentOffset
        [self.pageScrollView setContentOffset:CGPointMake(self.oldContentOffset - translate.x , 0) animated:NO];
        
        
        //////////////////////////////////////////////////
        // 結束的處理
        if(pan.state==UIGestureRecognizerStateEnded)
        {
            BOOL isNeedNotify = NO;
            
            // 如果pan的位移大於某個大小，才能移動
            if([self isValidateMove:translate.x])
            {
                isNeedNotify = YES;
                
                // 調整顯示的位置
                self.currentPageIndex = self.nextPageIndex;
            }

            [self finishPanWithNotify:isNeedNotify];
        }
        
        if (sender.state == UIGestureRecognizerStateCancelled)
        {
            // 選原到pan之前的狀態
            [self finishPanWithNotify:NO];
        }
        
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)isValidateMove:(CGFloat)deltaX
{
    CGFloat minimum = floorf(self.view.bounds.size.width * self.minimumMovePercentage);
    return ABS(deltaX) > ABS(minimum);
}


//==============================================================================
//
//==============================================================================
- (void)finishPanWithNotify:(BOOL)notify
{
    [self.pageScrollView setContentOffset:CGPointMake(self.pageScrollView.frame.size.width * self.currentPageIndex, 0) animated:YES];
    
    self.oldContentOffset = 0;
    
    //////////////////////////////////////////////////
    if (notify==YES)
    {
        [self sendDidShowPageAtIndex:self.currentPageIndex];
    }
}







////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - handle scroll view


//==============================================================================
//
//==============================================================================
- (void)sendWillShowPageAtIndex:(NSInteger)index
{
    if ([self.delegate respondsToSelector:@selector(scrollPageController:willShowPageAtIndex:)])
    {
        [self.delegate scrollPageController:self willShowPageAtIndex:index];
    }
}


//==============================================================================
//
//==============================================================================
- (void)sendDidShowPageAtIndex:(NSInteger)index
{
    if ([self.delegate respondsToSelector:@selector(scrollPageController:didShowPageAtIndex:)])
    {
        [self.delegate scrollPageController:self didShowPageAtIndex:index];
    }
}


//==============================================================================
//
//==============================================================================
- (void)addPageWithPageIndex:(NSInteger)index
{
    UIViewController *viewController = [self.pageViewControllers objectAtIndex:index];
    UIView *pageView = [viewController view];
    
    if ([pageView superview]!=self.pageScrollView)
    {
//        NSLog(@"add %ld", index);
        [self sendWillShowPageAtIndex:index];

        //////////////////////////////////////////////////
        
        [pageView setFrame:CGRectMake(self.pageScrollView.frame.size.width *index, 0, self.pageScrollView.frame.size.width, self.pageScrollView.frame.size.height)];
        
        //////////////////////////////////////////////////
        if([[[UIDevice currentDevice] systemVersion] floatValue]>11.0)
        {
            // 這樣在Asteroom的順序才對
            [viewController willMoveToParentViewController:self];
            [self addChildViewController:viewController];
            [self.pageScrollView addSubview:pageView];
            [viewController didMoveToParentViewController:self];
        }
        else
        {
            // ios 10要這樣的順序才能正常顯示
            [self.pageScrollView addSubview:pageView];

            [viewController willMoveToParentViewController:self];
            [self addChildViewController:viewController];
            [viewController didMoveToParentViewController:self];
        }
    }
}


//==============================================================================
// 移動完後，移除非目前選取頁的畫面
//==============================================================================
- (void)clearInvisibleView
{
    NSInteger index = 0;
    NSInteger minKeepIndex = MAX(0, self.currentPageIndex-PPScrollPageController_PreloadPage);
    NSInteger maxKeepIndex = MIN(self.currentPageIndex+PPScrollPageController_PreloadPage, [self.pageViewControllers count]-1);
    
//    NSLog(@"%s - (%ld , %ld)", __PRETTY_FUNCTION__, minKeepIndex, maxKeepIndex);
    

    for (UIViewController *viewController in self.pageViewControllers)
    {
        if (index<minKeepIndex ||
            index>maxKeepIndex)
        {
            if([viewController.view superview]==self.pageScrollView)
            {
//                NSLog(@"clear:%ld - %@", index, NSStringFromCGPoint(self.pageScrollView.contentOffset));
                [viewController beginAppearanceTransition:NO animated:YES];
                [viewController willMoveToParentViewController:nil];
                
                [viewController.view removeFromSuperview];

                [viewController removeFromParentViewController];
                [viewController didMoveToParentViewController:nil];
                [viewController endAppearanceTransition];
                
            }
        }
        
        index ++;
    }
    
    
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - public


//==============================================================================
//
//==============================================================================
- (UIViewController *)viewControllerAtIndex:(NSInteger)index
{
    if ([self.pageViewControllers count]<=index)
    {
        return nil;
    }
    return  [self.pageViewControllers objectAtIndex:index];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - UIScrollViewDelegate



//==============================================================================
//
//==============================================================================
- (void)scrollViewDidEndScrollingAnimation:(UIScrollView *)scrollView
{
    [self startClearInvisibleViewTimer];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instace Method

//================================================================================
//
//================================================================================
- (void)assignPageToIndex:(NSUInteger)index animated:(BOOL)animated
{
    if(index<self.pageViewControllers.count)
    {
        self.currentPageIndex = index;
        
        [self showPageAtIndex:index animated:animated];
    }
}

@end
