//
//  PPRestObject.m
//  
//
//  Created by Mike on 13/7/19.
//  Copyright (c) 2013年 penpower. All rights reserved.
//

#import "PPRestObject.h"
#import "NSError+Custom.h"

@implementation PPRestObject

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Deallocating Objects

//================================================================================
//
//================================================================================
- (id)initWithJsonString:(NSString *)jsonString error:(NSError **)error
{
    id      object          = nil;
    NSError *originError    = ((error!=nil)?*error:nil);
    NSError *returnError    = originError;
    
    //////////////////////////////////////////////////
    
    if((self=[super init]))
    {
        @autoreleasepool
        {
            do
            {
                if(jsonString==nil || [jsonString length]==0)
                {
                    returnError = PPErrorParameterInvalidity(returnError);
                    break;
                }
                
                //////////////////////////////////////////////////
                
                NSData *jsonData = [jsonString dataUsingEncoding:NSUTF8StringEncoding];
                if(jsonData==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                    break;
                }
                
                //////////////////////////////////////////////////
                
                id jsonObject = [RKMIMETypeSerialization objectFromData:jsonData MIMEType:RKMIMETypeJSON error:&returnError];
                if(jsonObject==nil)
                {
                    break;
                }
                
                //////////////////////////////////////////////////
                
                RKObjectMappingOperationDataSource *dataSource = [[[RKObjectMappingOperationDataSource alloc] init] autorelease];
                if(dataSource==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                    break;
                }
                
                //////////////////////////////////////////////////
                
                RKMappingOperation *mappingOperation = [[[RKMappingOperation alloc] initWithSourceObject:jsonObject
                                                                                       destinationObject:self
                                                                                                 mapping:[[self class] responseMapping]] autorelease];
                if(mappingOperation==nil)
                {
                    returnError = PPErrorOperationFailed(returnError);
                    break;
                }
                
                mappingOperation.dataSource = dataSource;
                
                //////////////////////////////////////////////////
                
                [mappingOperation performMapping:&returnError];
                if(returnError!=originError)
                {
                    break;
                }
                
                //////////////////////////////////////////////////
                
                object = self;
                
            }while(0);
            
            //////////////////////////////////////////////////
            
            [returnError retain];
        }
        
        [returnError autorelease];
    }
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    if(object==nil)
    {
        [self release];
    }
    
    //////////////////////////////////////////////////
    
    return object;
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - NSCopying

//================================================================================
//
//================================================================================
- (id)copyWithZone:(NSZone *)zone
{
    return [[[self class] allocWithZone:zone] init];
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Method

//================================================================================
//
//================================================================================
- (NSString *)jsonStringWithError:(NSError **)error
{
    NSString    *jsonString     = nil;
    NSError     *originError    = ((error!=nil)?*error:nil);
    NSError     *returnError    = originError;
    
    //////////////////////////////////////////////////
    
    @autoreleasepool
    {
        do
        {
            NSMutableDictionary *jsonDictionary = [[[NSMutableDictionary alloc] init] autorelease];
            if(jsonDictionary==nil)
            {
                returnError = PPErrorOperationFailed(returnError);
                break;
            }
            
            //////////////////////////////////////////////////

            RKObjectMappingOperationDataSource *dataSource = [[[RKObjectMappingOperationDataSource alloc] init] autorelease];
            if(dataSource==nil)
            {
                returnError = PPErrorOperationFailed(returnError);
                break;
            }
            
            //////////////////////////////////////////////////
            
            RKMappingOperation *mappingOperation = [[[RKMappingOperation alloc] initWithSourceObject:self
                                                                                   destinationObject:jsonDictionary
                                                                                             mapping:[[self class] requestMapping]] autorelease];
            if(mappingOperation==nil)
            {
                returnError = PPErrorOperationFailed(returnError);
                break;
            }
            
            mappingOperation.dataSource = dataSource;
            
            //////////////////////////////////////////////////
            
            [mappingOperation performMapping:&returnError];
            if(returnError!=originError)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            NSData *jsonData = [RKMIMETypeSerialization dataFromObject:jsonDictionary
                                                              MIMEType:RKMIMETypeJSON
                                                                 error:&returnError];
            if(jsonData==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            jsonString = [[[NSString alloc] initWithData:jsonData encoding:NSUTF8StringEncoding] autorelease];
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        [jsonString retain];
        [returnError retain];
    }

    [jsonString autorelease];
    [returnError autorelease];
    
    //////////////////////////////////////////////////
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    
    return jsonString;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Intended For Subclasses Method

//================================================================================
//
//================================================================================
+ (RKObjectMapping *)requestMapping
{
    return [[self responseMapping] inverseMapping];
}

//================================================================================
//
//================================================================================
+ (RKObjectMapping *)responseMapping
{
    return [RKObjectMapping mappingForClass:[self class]];
}

@end
