//
//  PPQRCodeController.m
//  
//
//  Created by Howard.Lin on 14/2/25.
//  Copyright (c) 2014年 Penpower. All rights reserved.
//

#import "PPQRCodeController.h"

// Define
#import "PPQRCodeController+ParameterDefine.h"

// Operation
#import "PPQRCodeCaptureOperation.h"

// Category
#import "NSError+Custom.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPQRCodeCaptureViewController()

@interface PPQRCodeController () <PPQRCodeCaptureOperationDelegate>

@property (nonatomic,retain) NSOperationQueue *qrCodeImageRecognitionQueue;
@end





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Implementation PPQRCodeController

@implementation PPQRCodeController


////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Dellocating Objects

//================================================================================
//
//================================================================================
- (id)init
{
    if(self=[super init])
    {
        _qrCodeImageRecognitionQueue = [[NSOperationQueue alloc] init];
        
        if(_qrCodeImageRecognitionQueue!=nil)
        {
            [_qrCodeImageRecognitionQueue setMaxConcurrentOperationCount:1];
        }
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    _delegate = nil;
    
    //////////////////////////////////////////////////
    
    [_qrCodeImageRecognitionQueue cancelAllOperations];
    [_qrCodeImageRecognitionQueue release];
    _qrCodeImageRecognitionQueue = nil;
    
    //////////////////////////////////////////////////

    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPQRCodeCaptureOperationDelegate Method

//================================================================================
//
//================================================================================
- (void)ppQRCodeCaptureOperation:(PPQRCodeCaptureOperation *)ppQRCodeCaptureOperation withQRCodeString:(NSString *)string
{
    if([string length] > 0)
    {
        if([self.delegate respondsToSelector:@selector(ppQRCodeViewController:withQRCodeString:)]==YES)
        {
            [self.delegate ppQRCodeViewController:self withQRCodeString:string];
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Method

//================================================================================
//
//================================================================================
- (NSUInteger)qrRecognitionOperationCount
{
    return self.qrCodeImageRecognitionQueue.operationCount;
}


//================================================================================
//
//================================================================================
- (BOOL)startRecognitionForQRCodeImage:(UIImage *)qrCodeImage
                              delegate:(id<PPQRCodeControllerDelegate>)delegate
                                 error:(NSError **)error
{
    BOOL result = NO;
    
    NSError *returnError = nil;
    
    do
    {
        if(qrCodeImage==nil)
        {
            returnError = PPErrorMake(PPQRCodeControllerErrorCode_LoadImageFail, @"QR Code Image is nil", nil);
            
            break;
        }
        else if(self.allowMultiQRCodeImageRecognition==NO &&
                [self qrRecognitionOperationCount]>0)
        {
            returnError = PPErrorMake(PPQRCodeControllerErrorCode_ExceedRecognitionCountLimit, @"Exceed RecognitionCountLimit", nil);

            break;
        }
        
        //////////////////////////////////////////////////

        self.delegate = delegate;
        
        //////////////////////////////////////////////////

        PPQRCodeCaptureOperation *operation = [[PPQRCodeCaptureOperation alloc] init];
        
        if(operation!=nil)
        {
            operation.delegate = self;
            operation.sourceQRCodeImage = qrCodeImage;
            [self.qrCodeImageRecognitionQueue addOperation:operation];
        }
        
        [operation release];
    }
    while (0);
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    return result;
}


//================================================================================
//
//================================================================================
- (void)cancelQRCodeImageOperation
{
    self.delegate = nil;
    
    [self.qrCodeImageRecognitionQueue cancelAllOperations];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class Method

//==============================================================================
//
//==============================================================================
+ (PPQRCodeController *)qrCodeShareInstance
{
    static PPQRCodeController *qrCodeShareInstance = nil;
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^{
        qrCodeShareInstance = [[PPQRCodeController alloc] init];
    });
    
    return qrCodeShareInstance;
}


//================================================================================
// 縮小或放大影像，但不做平滑處理。（避免出現馬賽克效果）
//================================================================================
+ (UIImage *)createNonInterpolatedUIImageFromCIImage:(CIImage *)image withSideLength:(CGFloat)sideLength
{
    UIImage *resultImage = nil;
    
    // Render the CIImage into a CGImage
    CGImageRef cgImage = [[CIContext contextWithOptions:nil] createCGImage:image fromRect:image.extent];
    
    if(cgImage != nil)
    {
        // Now we'll rescale using CoreGraphics
        UIGraphicsBeginImageContext(CGSizeMake(sideLength, sideLength));
        
        CGContextRef context = UIGraphicsGetCurrentContext();
        
        if(context != nil)
        {
            // We don't want to interpolate (since we've got a pixel-correct image)
            CGContextSetInterpolationQuality(context, kCGInterpolationNone);
            CGContextDrawImage(context, CGContextGetClipBoundingBox(context), cgImage);
            
            // Get the image out
            resultImage = UIGraphicsGetImageFromCurrentImageContext();
        }
        
        UIGraphicsEndImageContext();
        CGImageRelease(cgImage);
    }
    
    return resultImage;
}


//================================================================================
//
//================================================================================
+ (UIImage *)QRCodeImageWithContent:(NSString *)content sideLength:(NSInteger)sideLength
{
    // Need to convert the string to a UTF-8 encoded NSData object
    NSData *stringData = [content dataUsingEncoding:NSUTF8StringEncoding];
    
    // Create the filter
    CIFilter *qrFilter = [CIFilter filterWithName:@"CIQRCodeGenerator"];
    
    if(qrFilter != nil)
    {
        // Set the message content and error-correction level
        [qrFilter setValue:stringData forKey:@"inputMessage"];
        [qrFilter setValue:@"H" forKey:@"inputCorrectionLevel"];
        
        // Send the image back
        return [self createNonInterpolatedUIImageFromCIImage:qrFilter.outputImage withSideLength:sideLength];
    }
    
    return nil;
}



@end
