//
//  PPNetworkReachabilityController.m
//  
//
//  Created by Mike on 13/5/8.
//  Copyright (c) 2013年 Penpower. All rights reserved.
//

#import "PPNetworkReachabilityController.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

static void networkReachabilityCallback(SCNetworkReachabilityRef target, SCNetworkReachabilityFlags flags, void* info)
{    
    @autoreleasepool
    {
        [[NSNotificationCenter defaultCenter] postNotificationName:PPNetworkReachabilityControllerChangedNotification
                                                            object:(PPNetworkReachabilityController *)info];
    }
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPNetworkReachabilityController()

@interface PPNetworkReachabilityController()
@property(nonatomic,assign) SCNetworkReachabilityRef	networkReachabilityRef;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPNetworkReachabilityController

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Synthesize

@synthesize networkReachabilityRef  = networkReachabilityRef_;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Deallocating Objects

//================================================================================
//
//================================================================================
- (id)init
{
	if((self = [super init]))
	{	
		networkReachabilityRef_	= NULL;
	}
	
	return self;
}

//================================================================================
//
//================================================================================
- (void)dealloc
{
	[self stopNotifier];

	if(self.networkReachabilityRef!=NULL)
	{
		CFRelease(self.networkReachabilityRef);
		networkReachabilityRef_ = NULL;
	}
	
    //////////////////////////////////////////////////
    
	[super dealloc];
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Methods

//================================================================================
//
//================================================================================
- (BOOL)startNotifier
{
    BOOL result = NO;
    
    do
    {
        if(self.networkReachabilityRef==NULL)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        SCNetworkReachabilityContext context = {0, self, NULL, NULL, NULL};
		
		if(SCNetworkReachabilitySetCallback(self.networkReachabilityRef, networkReachabilityCallback, &context)==false)
		{
			break;
		}
        
        //////////////////////////////////////////////////
        
        result = SCNetworkReachabilityScheduleWithRunLoop(self.networkReachabilityRef, CFRunLoopGetCurrent(), kCFRunLoopDefaultMode);
        
    }while(0);
    
    return result;
}

//================================================================================
//
//================================================================================
- (void)stopNotifier
{
	if(self.networkReachabilityRef!=NULL)
	{
		SCNetworkReachabilityUnscheduleFromRunLoop(self.networkReachabilityRef, CFRunLoopGetCurrent(), kCFRunLoopDefaultMode);
	}
}

//================================================================================
//
//================================================================================
- (PPNetworkReachabilityControllerStatus)status
{
	PPNetworkReachabilityControllerStatus status = PPNetworkReachabilityControllerStatus_ReachableNone;
	
    do
    {
        if(self.networkReachabilityRef==NULL)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        SCNetworkReachabilityFlags networkReachabilityFlags;
        if(SCNetworkReachabilityGetFlags(self.networkReachabilityRef, &networkReachabilityFlags)==NO)
		{
            break;
        }
        
        //////////////////////////////////////////////////
        
        BOOL isReachable = ((networkReachabilityFlags & kSCNetworkReachabilityFlagsReachable) != 0);
        BOOL needsConnection = ((networkReachabilityFlags & kSCNetworkReachabilityFlagsConnectionRequired) != 0);
        
        BOOL canConnectionAutomatically = (((networkReachabilityFlags & kSCNetworkReachabilityFlagsConnectionOnDemand ) != 0) || ((networkReachabilityFlags & kSCNetworkReachabilityFlagsConnectionOnTraffic) != 0));
        
        BOOL canConnectWithoutUserInteraction = (canConnectionAutomatically && (networkReachabilityFlags & kSCNetworkReachabilityFlagsInterventionRequired) == 0);
        
        BOOL isNetworkReachable = (isReachable && (!needsConnection || canConnectWithoutUserInteraction));
        
        //////////////////////////////////////////////////

        if(isNetworkReachable==NO)
        {
            break;
        }

        //////////////////////////////////////////////////

        status = PPNetworkReachabilityControllerStatus_ReachableViaWiFi;
        
        #if	TARGET_OS_IPHONE
        // 判斷是否為3G
        if(networkReachabilityFlags & kSCNetworkReachabilityFlagsIsWWAN)
        {
            status = PPNetworkReachabilityControllerStatus_ReachableViaWWAN;
        }
        #endif
        
    }while(0);
	
	return status;
}

//================================================================================
//
//================================================================================
- (BOOL)connectionRequired
{
    BOOL result = NO;
    
    do
    {
        if(self.networkReachabilityRef==NULL)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        SCNetworkReachabilityFlags networkReachabilityFlags;
        if(SCNetworkReachabilityGetFlags(self.networkReachabilityRef, &networkReachabilityFlags)==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        result = (networkReachabilityFlags&kSCNetworkReachabilityFlagsConnectionRequired);
        
    }while(0);
    
    return result;
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Methods

//================================================================================
//
//================================================================================
+ (PPNetworkReachabilityController *)networkReachabilityControllerWithAddress:(const struct sockaddr_in *)hostAddress
{
    PPNetworkReachabilityController *networkReachabilityController = nil;
    
    do 
    {
        SCNetworkReachabilityRef networkReachabilityRef = SCNetworkReachabilityCreateWithAddress(NULL, (const struct sockaddr *)hostAddress);
        
        if(networkReachabilityRef==NULL) 
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        networkReachabilityController = [[[PPNetworkReachabilityController alloc] init] autorelease];
        
        if(networkReachabilityController!=nil) 
        {
            networkReachabilityController.networkReachabilityRef  = networkReachabilityRef;
        }
        else
        {
            CFRelease(networkReachabilityRef);
        }
        
    }while(0);
	
	return networkReachabilityController;
}

//================================================================================
//
//================================================================================
+ (PPNetworkReachabilityController*)networkReachabilityControllerWithHostName:(NSString*)hostName
{
    PPNetworkReachabilityController *networkReachabilityController = nil;
    
    do 
    {
        if(hostName==nil || [hostName length]==0) 
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        SCNetworkReachabilityRef networkReachabilityRef = SCNetworkReachabilityCreateWithName(NULL, [hostName UTF8String]);
        if(networkReachabilityRef==NULL) 
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        networkReachabilityController = [[[PPNetworkReachabilityController alloc] init] autorelease];
        
        if(networkReachabilityController!=nil) 
        {
            networkReachabilityController.networkReachabilityRef  = networkReachabilityRef;
        }
        else
        {
            CFRelease(networkReachabilityRef);
        }
        
    }while(0);
	
	return networkReachabilityController;
}

//================================================================================
//
//================================================================================
+ (PPNetworkReachabilityController *)networkReachabilityControllerForInternetConnection
{
    struct sockaddr_in zeroAddress;
    
    bzero(&zeroAddress, sizeof(zeroAddress));
    
    zeroAddress.sin_len     = sizeof(zeroAddress);
    zeroAddress.sin_family  = AF_INET;
    
    return [PPNetworkReachabilityController networkReachabilityControllerWithAddress:&zeroAddress];
}

//================================================================================
//
//================================================================================
+ (PPNetworkReachabilityController *)networkReachabilityControllerForLocalWiFi
{
    struct sockaddr_in localWifiAddress;
    
    bzero(&localWifiAddress, sizeof(localWifiAddress));
    
    localWifiAddress.sin_len    = sizeof(localWifiAddress);
    localWifiAddress.sin_family = AF_INET;
    
    localWifiAddress.sin_addr.s_addr = htonl(IN_LINKLOCALNETNUM);   // IN_LINKLOCALNETNUM is defined in <netinet/in.h> as 169.254.0.0
    
    return [PPNetworkReachabilityController networkReachabilityControllerWithAddress:&localWifiAddress];
}

//================================================================================
//
//================================================================================
+ (BOOL)checkWithAddress:(const struct sockaddr_in *)hostAddress
{
    return ([PPNetworkReachabilityController networkReachabilityControllerWithAddress:hostAddress].status!=PPNetworkReachabilityControllerStatus_ReachableNone);
}

+ (BOOL)checkWithHostName:(NSString *)hostName
{
    return ([PPNetworkReachabilityController networkReachabilityControllerWithHostName:hostName].status!=PPNetworkReachabilityControllerStatus_ReachableNone);
}

+ (BOOL)checkForInternetConnection
{
    return ([PPNetworkReachabilityController networkReachabilityControllerForInternetConnection].status!=PPNetworkReachabilityControllerStatus_ReachableNone);
}

+ (BOOL)checkForLocalWiFi
{
	return ([PPNetworkReachabilityController networkReachabilityControllerForLocalWiFi].status!=PPNetworkReachabilityControllerStatus_ReachableNone);
}

@end
