//
//  PPPincodeViewController.m
//  Pods
//
//  Created by sanhue on 2015/9/21.
//
//

#import "PPPincodeViewController.h"
#import "PPPincodeView.h"
#import "NSString+Additions.h"
#import "PPShiftView.h"
#import "PPTouchID.h"
#import "PPNavigationBarView.h"
#import "PPButton+Factory.h"
#import "PPLockScreenController+ResourceDefine.h"


////////////////////////////////////////////////////////////////////////////////////////////////////

typedef NS_ENUM(NSUInteger, PPPincodeViewControllerState)
{
    PPPincodeViewControllerState_None = 0,
    PPPincodeViewControllerState_FirstInputPassword,    // 第一次輸入密碼，後面會出現第二次密碼輸入流程
    PPPincodeViewControllerState_SecendInputPassword,   // 第二次輸入密碼，
    PPPincodeViewControllerState_CheckPassword,         // 檢查密碼並delegate出去的流程
};


////////////////////////////////////////////////////////////////////////////////////////////////////
@interface PPPincodeViewController () <PPButtonDelegate, PPPincodeViewDelegate>
@property (nonatomic, retain) PPNavigationBarView *ppNavigationBarView;
@property (nonatomic, retain) PPButton *ppButtonBack;

@property (nonatomic, retain) PPShiftView *ppShiftView;
@property (nonatomic, retain) PPPincodeView *pincodeView;

@property (nonatomic, retain) NSArray *layoutConstraints;
@property (atomic, assign) PPPincodeViewControllerState state;
@property (atomic, retain) NSString *oldPassword;
@property (atomic, retain) NSString *firstPassword;
@property (atomic, assign) NSUInteger failedCount;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////
@implementation PPPincodeViewController





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - init/dealloc methods


//==============================================================================
//
//==============================================================================
+ (void)initialize
{
    [[PPPincodeViewController appearance] setButtonImageEdgeInsets:UIEdgeInsetsZero];
    [[PPPincodeViewController appearance] setButtonHighlightedBackgroundColor:[UIColor clearColor]];
}


//==============================================================================
//
//==============================================================================
- (instancetype)init
{
    self = [super init];
    if (self)
    {
        [[PPPincodeViewController appearance] applyInvocationTo:self];
        
        //////////////////////////////////////////////////
        self.state = PPPincodeViewControllerState_FirstInputPassword;
        self.failedCount = 0;
        
        // keyboard notifications
        [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(recvKeyboardWillShow:) name:UIKeyboardWillShowNotification object:nil];
        [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(recvKeyboardWillHide:) name:UIKeyboardWillHideNotification object:nil];
    }
    return self;
}


//==============================================================================
//
//==============================================================================
- (void)dealloc
{
    self.delegate = nil;
    
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    
    self.oldPassword = nil;
    self.firstPassword = nil;
    
    self.buttonHighlightedBackgroundColor = nil;
    self.layoutConstraints = nil;
    
    [self removeMainUI];
    //////////////////////////////////////////////////
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - Override Property


//==============================================================================
//
//==============================================================================
- (void)setStyle:(PPPincodeViewControllerStyle)style
{
    if (_style == style)
    {
        return;
    }
    
    _style = style;
    
    switch (style) {
        case PPPincodeViewControllerStyle_CreatePassword:
            self.state = PPPincodeViewControllerState_FirstInputPassword;
            break;
        default:
            self.state = PPPincodeViewControllerState_CheckPassword;
            break;
    }
}


//==============================================================================
//
//==============================================================================
- (void)setButtonImageEdgeInsets:(UIEdgeInsets)buttonImageEdgeInsets
{
    _buttonImageEdgeInsets = buttonImageEdgeInsets;
}


//==============================================================================
//
//==============================================================================
- (void)setButtonHighlightedBackgroundColor:(UIColor *)buttonHighlightedBackgroundColor
{
    [buttonHighlightedBackgroundColor retain];
    [_buttonHighlightedBackgroundColor release];
    _buttonHighlightedBackgroundColor = buttonHighlightedBackgroundColor;
}








////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - view controller life cycle


//==============================================================================
//
//==============================================================================
- (void)viewWillAppear:(BOOL)animated
{
    [super viewWillAppear:animated];
    
    //////////////////////////////////////////////////
    self.view.backgroundColor = [UIColor whiteColor];
    //////////////////////////////////////////////////
    
    [self prepareMainUI];

    //////////////////////////////////////////////////
    //
    [self updateTitleWithStyle:self.style];
    [self updatePincodeViewTitleWithInputView:self.pincodeView];


    [self.pincodeView becomeFirstResponder];
}


//==============================================================================
//
//==============================================================================
- (void)viewDidAppear:(BOOL)animated
{
    [super viewDidAppear:animated];
    
    //////////////////////////////////////////////////
    if (self.enableTouchID&&self.style==PPPincodeViewControllerStyle_CheckPassword)
    {
        [PPTouchID showWithReason:PPLockScreenControllerString_TouchToUnlock
         customEnterPasswordTitle:@""
                       completion:^(PPTouchIDResponse response)
         {
             switch (response)
             {
                 case PPTouchIDResponse_Success:
                 {
                     if ([self.delegate respondsToSelector:@selector(pincodeViewController:didFinishWithPassword:)])
                     {
                         [self.delegate pincodeViewController:self didFinishWithPassword:nil];
                     }
                 }
                 default:
                 {
                     // do nothing
                     
                     break;
                 }
             }
         }];
    }
}


//==============================================================================
//
//==============================================================================
- (void)viewWillDisappear:(BOOL)animated
{
    [self.pincodeView resignFirstResponder];
    //////////////////////////////////////////////////
    
    [super viewWillDisappear:animated];
}


//==============================================================================
//
//==============================================================================
- (void)viewDidDisappear:(BOOL)animated
{
    [self removeMainUI];
    //////////////////////////////////////////////////
    
    [super viewDidDisappear:animated];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - prepare ui


//==============================================================================
//
//==============================================================================
- (void)prepareMainUI
{
    if (self.navigationController)
    {
        _ppNavigationBarView = [[PPNavigationBarView alloc] initWithFrame:self.navigationController.navigationBar.bounds];
        if (self.ppNavigationBarView!=nil)
        {
            self.ppNavigationBarView.style = PPBarViewStyle_CenteredAbsolute;
            self.ppNavigationBarView.titleLabel.text = self.title;
            
            self.ppButtonBack = [PPButton ppButtonWithIconImageName:ImageNamePPButtonIconForSystemBackImitation
                                                              tag:0
                                                         delegate:self
                                            normalBackgroundColor:nil
                                       highlightedBackgroundColor:self.buttonHighlightedBackgroundColor
                                                  imageEdgeInsets:self.buttonImageEdgeInsets];
            if(self.ppButtonBack!=nil)
            {
                //////////////////////////////////////////////////
                
                [self.ppNavigationBarView setViews:@[self.ppButtonBack] forBlockType:PPBarViewBlockType_Left];
            }
            
            self.navigationItem.leftBarButtonItem = [[[UIBarButtonItem alloc] initWithCustomView:self.ppNavigationBarView] autorelease];
        }
    }
    
    //////////////////////////////////////////////////

    
    self.pincodeView = [[[PPPincodeView alloc] init] autorelease];
    if (self.pincodeView)
    {
        [self.pincodeView setTranslatesAutoresizingMaskIntoConstraints:NO];
        [self.pincodeView setDelegate:self];
    }
    
    self.ppShiftView = [[[PPShiftView alloc] init] autorelease];
    if (self.ppShiftView)
    {
        [self.ppShiftView setTranslatesAutoresizingMaskIntoConstraints:NO];
        self.ppShiftView.currentView = self.pincodeView;
        [self.view addSubview:self.ppShiftView];
    }
    
    //////////////////////////////////////////////////
    // 一進入就focus
    [self.pincodeView becomeFirstResponder];
    
    //////////////////////////////////////////////////
    [self resetLayoutConstraintsWithKeyboardHeight:0];
}


//==============================================================================
//
//==============================================================================
- (void)removeMainUI
{
    [self removeLayoutConstraints];
    //////////////////////////////////////////////////
    
    [self.pincodeView removeFromSuperview];
    self.pincodeView = nil;
    
    [self.ppShiftView removeFromSuperview];
    self.ppShiftView = nil;
    
    //////////////////////////////////////////////////
    
    self.navigationItem.leftBarButtonItem = nil;
    
    self.ppNavigationBarView  = nil;
    
    self.ppButtonBack   = nil;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - layout constraints


//==============================================================================
//
//==============================================================================
- (void)resetLayoutConstraintsWithKeyboardHeight:(CGFloat)keybaordHeight
{
    if (self.ppShiftView==nil)
    {
        return ;
    }
    
    [self removeLayoutConstraints];
    //////////////////////////////////////////////////
    NSUInteger topMargin = 0;
    UIApplication *sharedApplication = [UIApplication sharedApplication];
    if(sharedApplication.statusBarHidden==NO)
    {
        topMargin += MIN(sharedApplication.statusBarFrame.size.width, sharedApplication.statusBarFrame.size.height);
    }
    
    if(self.navigationController.navigationBar!=nil &&
       self.navigationController.navigationBarHidden==NO)
    {
        CGRect navigationBarFrame = self.navigationController.navigationBar.frame;
        topMargin += MIN(navigationBarFrame.size.width, navigationBarFrame.size.height);
    }
    
    //////////////////////////////////////////////////
    NSDictionary *views = @{@"ppShiftView":self.ppShiftView};
    NSDictionary *metrics = @{@"topMargin":@(topMargin),
                              @"keyboardHeight":@(keybaordHeight)};
    NSMutableArray *layoutConstraints = [NSMutableArray array];
    
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"|[ppShiftView]|"
                                                                                  options:NSLayoutFormatDirectionLeftToRight
                                                                                  metrics:metrics
                                                                                    views:views]];
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|-(topMargin)-[ppShiftView]-(keyboardHeight)-|"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:metrics
                                                                                     views:views]];
    //////////////////////////////////////////////////
    if ([layoutConstraints count])
    {
        self.layoutConstraints = [NSArray arrayWithArray:layoutConstraints];
        [self.view addConstraints:self.layoutConstraints];
    }
}


//==============================================================================
//
//==============================================================================
- (void)removeLayoutConstraints
{
    if (self.layoutConstraints)
    {
        [self.view removeConstraints:self.layoutConstraints];
        self.layoutConstraints = nil;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - flow control 


//==============================================================================
//
//==============================================================================
- (void)updateTitleWithStyle:(PPPincodeViewControllerStyle)style
{
    NSString *titleString = nil;
    switch (style)
    {
        case PPPincodeViewControllerStyle_CreatePassword:    ///< 建立密碼
        {
            titleString = PPLockScreenControllerString_CreatePassword;
            break;
        }
        case PPPincodeViewControllerStyle_ChangePassword:    ///< 變更密碼
        {
            titleString = PPLockScreenControllerString_ChangePassword;
            break;
        }
        case PPPincodeViewControllerStyle_CheckPassword:     ///< 檢查密碼
        {
            titleString = PPLockScreenControllerString_InputPassword;
            break;
        }
        case PPPincodeViewControllerStyle_ClearPassword:     ///< 清除密碼
        {
            titleString = PPLockScreenControllerString_ClearPassword;
            break;
        }
        default:
            break;
    }
    [self.ppNavigationBarView.titleLabel setText:titleString];
    
}


//==============================================================================
//
//==============================================================================
- (void)updatePincodeViewTitleWithInputView:(PPPincodeView*)inputView
{
    NSString *titleString = nil;

    switch (self.state)
    {
        case PPPincodeViewControllerState_FirstInputPassword:
        {
            if (self.style==PPPincodeViewControllerStyle_ChangePassword)
            {
                titleString = PPLockScreenControllerString_PleaseEnterNewPassword;
            }
            else
            {
                titleString = PPLockScreenControllerString_PleaseEnterPassword;
            }
            break;
        }
        case PPPincodeViewControllerState_SecendInputPassword:
        {
            titleString = PPLockScreenControllerString_PleaseEnterPasswordAgain;
            break;
        }
        case PPPincodeViewControllerState_CheckPassword:
        {
            if (self.style==PPPincodeViewControllerStyle_ChangePassword)
            {
                titleString = PPLockScreenControllerString_PleaseEnterOldPassword;
            }
            else
            {
                titleString = PPLockScreenControllerString_PleaseEnterPassword;
            }
            break;
        }
            
        default:
            break;
    }
    inputView.title = titleString;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - button action


//==============================================================================
//
//==============================================================================
- (void)onClickCancel:(UIBarButtonItem *)sender
{
    [self goBackAnimated:YES];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - recv notification


//==============================================================================
//
//==============================================================================
- (void)recvKeyboardWillShow:(NSNotification *)notification
{
    NSValue * rectValue = [notification.userInfo objectForKey:UIKeyboardFrameEndUserInfoKey];
    if(rectValue!=nil)
    {
        CGRect keyboardRect = [rectValue CGRectValue];
        [self resetLayoutConstraintsWithKeyboardHeight:MIN(keyboardRect.size.width,keyboardRect.size.height)];
    }
}


//==============================================================================
//
//==============================================================================
- (void)recvKeyboardWillHide:(NSNotification *)notification
{
    [self resetLayoutConstraintsWithKeyboardHeight:0];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - PPButtonDelegate


//==============================================================================
//
//==============================================================================
- (void)ppButton:(PPButton *)ppButton controlEvent:(UIControlEvents)controlEvent
{
    [self goBackAnimated:YES];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - PPPincodeViewDelegate


//==============================================================================
//
//==============================================================================
- (void)inputViewDidFinish:(PPPincodeView *)inputView
{
    NSString *password = inputView.pincode;
    
    switch (self.state)
    {
        case PPPincodeViewControllerState_FirstInputPassword:
        {
            if (self.style==PPPincodeViewControllerStyle_ChangePassword && [self.oldPassword isEqualToString:password])
            {
                // 變更密碼，如果新密碼與舊密碼相同，跳出警告
                [inputView shakeAndVibrateCompletion:^{
                    inputView.pincode = nil;
                    inputView.message = PPLockScreenControllerString_CantUseSamePassword;
                }];
            }
            else
            {
                self.firstPassword = password;
                self.state = PPPincodeViewControllerState_SecendInputPassword;
                
                PPPincodeView *confirmInputView = [self.pincodeView copy];

                [self updatePincodeViewTitleWithInputView:confirmInputView];
                [self.ppShiftView showView:confirmInputView withDirection:(PPShiftDirection_Forward)];
                self.pincodeView = confirmInputView;
                [self.pincodeView becomeFirstResponder];
                [confirmInputView release];
            }
            break;
        }
        case PPPincodeViewControllerState_SecendInputPassword:
        {
            if ([password isEqualToString:self.self.firstPassword])
            {
                // 密碼相同
                if ([self.delegate respondsToSelector:@selector(pincodeViewController:didFinishWithPassword:)])
                {
                    [self.delegate pincodeViewController:self didFinishWithPassword:password];
                }
            }
            else
            {
                // 密碼不同
                // 要切回第一次輸入的狀態
                self.state = PPPincodeViewControllerState_FirstInputPassword;
                
                [inputView shakeAndVibrateCompletion:^{
                    PPPincodeView *newPasswordInputView = [self.pincodeView copy];
                    
                    newPasswordInputView.message = PPLockScreenControllerString_PasswordNotMatchAndTryAgain;
                    [self updatePincodeViewTitleWithInputView:newPasswordInputView];
                    
                    [self.ppShiftView showView:newPasswordInputView withDirection:PPShiftDirection_Backward];
                    self.pincodeView = newPasswordInputView;
                    [self.pincodeView becomeFirstResponder];
                    [newPasswordInputView release];
                }];
            }

            break;
        }
        case PPPincodeViewControllerState_CheckPassword:
        {
            BOOL isValidPassword = NO;
            if ([self.delegate respondsToSelector:@selector(pincodeViewController:checkPassword:)])
            {
                isValidPassword = [self.delegate pincodeViewController:self checkPassword:password];
                if (isValidPassword)
                {
                    // 密碼正確
                    if (self.style==PPPincodeViewControllerStyle_ChangePassword)
                    {
                        inputView.message = nil;
                        
                        // 變更密碼，如果正確，下一步是輸入新密碼
                        self.state = PPPincodeViewControllerState_FirstInputPassword;
                        
                        // 輸入正確就重新設定錯誤次數
                        self.failedCount = 0;
                        
                        PPPincodeView *newPasswordInputView = [self.pincodeView copy];
                        
                        [self updatePincodeViewTitleWithInputView:newPasswordInputView];
                        
                        [self.ppShiftView showView:newPasswordInputView withDirection:PPShiftDirection_Forward];
                        self.pincodeView = newPasswordInputView;
                        [self.pincodeView becomeFirstResponder];
                        [newPasswordInputView release];
                    }
                    else
                    {
                        // 通知結束
                        if ([self.delegate respondsToSelector:@selector(pincodeViewController:didFinishWithPassword:)])
                        {
                            [self.delegate pincodeViewController:self didFinishWithPassword:password];
                        }
                    }
                }
                else
                {
                    // 密碼不正確，通知app
                    if ([self.delegate respondsToSelector:@selector(didFailedWithPincodeViewController:)])
                    {
                        [self.delegate didFailedWithPincodeViewController:self];
                    }
                    
                    // 記錄錯誤次數
                    self.failedCount ++;
                    
                    [inputView shakeAndVibrateCompletion:^{
                        inputView.pincode = nil;
                        inputView.message = PPLockScreenControllerString_WrongPasswordAndTryAgain;
                    }];
                }
                
            }
            break;
        }

        default:
            break;
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPAppearance protocol methods

//================================================================================
//
//================================================================================
+ (id)appearance
{
    // MARK: 實作+appearance的方式固定如下
    return [PPAppearance appearanceForClass:[self class]];
}

@end
