//
//  AppleLocation.m
//
//
//  Created by Howard on 2015/12/29.
//
//

#import "AppleLocationManager.h"
#import <CoreLocation/CoreLocation.h>

// Define
#import "PPLocationController+ParameterDefine.h"


// Category
#import "NSError+Custom.h"

typedef void (^VoidCallBack)(void);
typedef void (^ErrorCallBack)(NSError *error);

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Interface AppleLocation

@interface AppleLocationManager() <CLLocationManagerDelegate>

@property (nonatomic,retain) CLLocationManager *locationManager;
@property (nonatomic,retain) CLGeocoder *geoCoder;

@property (nonatomic,copy)   VoidCallBack        updateCurrentLocationSuccessComplete;
@property (nonatomic,copy)   ErrorCallBack       updateCurrentLocationFailComplete;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Implementation AppleLocation

@implementation AppleLocationManager

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Dellocation Object

//================================================================================
//
//================================================================================
- (id)init
{
    if(self=[super init])
    {
        _locationManager = [[CLLocationManager alloc] init];
        
        if(_locationManager!=nil)
        {
            _locationManager.delegate = self;
            _locationManager.distanceFilter = 10.0f; // we don't need to be any more accurate than 10m
        }
        
        //////////////////////////////////////////////////

        _geoCoder = [[CLGeocoder alloc] init];

    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    [_geoCoder cancelGeocode];
    [_geoCoder release];
    _geoCoder = nil;
    
    [_locationManager release];
    _locationManager = nil;
    
    [_updateCurrentLocationFailComplete release];
    _updateCurrentLocationFailComplete = nil;
    
    [_updateCurrentLocationSuccessComplete release];
    _updateCurrentLocationSuccessComplete = nil;
    
    //////////////////////////////////////////////////
    
    [super dealloc];
}




////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Method

//================================================================================
// !! Apple地址如果吧國字轉數字會找不到，先關閉
//================================================================================
- (NSString *)normalizeAddressString:(NSString *)addressString
{
    return addressString;
    
//    NSString *normalizeAddressString = nil;
//
//    do
//    {
//        if(addressString==nil ||
//           addressString.length<=0)
//        {
//            break;
//        }
//
//        //////////////////////////////////////////////////
//
//        normalizeAddressString = [NSString stringWithString:addressString];
//
//        if(normalizeAddressString==nil)
//        {
//            break;
//        }
//
//        //////////////////////////////////////////////////
//
//        NSDictionary *candidatedCharacterDictionary = @{
//                                                        @"1":@"一",
//                                                        @"2":@"二",
//                                                        @"3":@"三",
//                                                        @"4":@"四",
//                                                        @"5":@"五",
//                                                        @"6":@"六",
//                                                        @"7":@"七",
//                                                        @"8":@"八",
//                                                        @"9":@"九",
//                                                        @"10":@"十"
//                                                        };
//
//        if(candidatedCharacterDictionary==nil)
//        {
//            break;
//        }
//
//        //////////////////////////////////////////////////
//
//        for(id key in candidatedCharacterDictionary)
//        {
//            normalizeAddressString = [normalizeAddressString stringByReplacingOccurrencesOfString:[candidatedCharacterDictionary objectForKey:key]
//                                                                                       withString:key];
//        }
//    }
//    while (0);
//
//    return normalizeAddressString;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - CLLocationManagerDelegate Method

//================================================================================
//
//================================================================================
- (void)locationManager:(CLLocationManager *)manager didUpdateToLocation:(CLLocation *)newLocation fromLocation:(CLLocation *)oldLocation
{
    [self stopUpdatingCurrentLocation];
    
    //////////////////////////////////////////////////
    
    [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                        object:self
                                                      userInfo:@{PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetCurrentLocationSuccess)
                                                                 ,PPLocationControllerUserInfoKey_Location:newLocation}];
    
    //////////////////////////////////////////////////
    
    if(self.updateCurrentLocationSuccessComplete!=nil)
    {
        self.updateCurrentLocationSuccessComplete();
        self.updateCurrentLocationSuccessComplete = nil;
    }

    self.updateCurrentLocationFailComplete = nil;
}


//================================================================================
//
//================================================================================
- (void)locationManager:(CLLocationManager *)manager didFailWithError:(NSError *)error
{
    // stop updating
    [self stopUpdatingCurrentLocation];
    
    //////////////////////////////////////////////////
    
    [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                        object:self
                                                      userInfo:@{PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetCurrentLocationFailure),
                                                                 PPLocationControllerUserInfoKey_Error:error}];
    
    //////////////////////////////////////////////////

    if(self.updateCurrentLocationFailComplete!=nil)
    {
        self.updateCurrentLocationFailComplete(error);
        self.updateCurrentLocationFailComplete = nil;
    }

    self.updateCurrentLocationSuccessComplete = nil;
    
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Override Method

//================================================================================
//
//================================================================================
- (LocationBlockOperation *)startUpdatingCurrentLocationBlockOperationWithSuccess:(void (^)(void))success
                                                                          failure:(void (^)(NSError *error))failure
{
    
    __block typeof(self) blockself = self;
    
    __block LocationBlockOperation *blockOperation = [LocationBlockOperation blockOperationWithBlock:^{
        
        do
        {
            // if location services are restricted do nothing
            if ([CLLocationManager authorizationStatus]==kCLAuthorizationStatusDenied ||
                [CLLocationManager authorizationStatus]==kCLAuthorizationStatusRestricted)
            {
                [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                    object:blockself
                                                                  userInfo:@{PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetCurrentLocationFailure),
                                                                             PPLocationControllerUserInfoKey_Error:PPErrorMake(kCLErrorDenied, @"kCLAuthorizationStatusDenied", nil)}];
                break;
            }
            else if(blockOperation.isCancelled==YES)
            {
                break;
            }
            else if([[PPNetworkReachabilityController networkReachabilityControllerForInternetConnection] status]==PPNetworkReachabilityControllerStatus_ReachableNone)
            {
                [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                    object:blockself
                                                                  userInfo:@{PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetCurrentLocationFailure),
                                                                             PPLocationControllerUserInfoKey_Error:PPErrorMake(PPLocationControllerErrorCode_NetworkDisconnected, @"NetworkDisconnected", nil)}];
                
                break;
            }
            
            //////////////////////////////////////////////////
            
            blockself.updateCurrentLocationSuccessComplete = success;
            blockself.updateCurrentLocationFailComplete = failure;
            
            //////////////////////////////////////////////////
            
            if([blockself.locationManager respondsToSelector:@selector(requestWhenInUseAuthorization)]==YES)
            {
                [blockself.locationManager requestWhenInUseAuthorization];
            }
            
            [blockself.locationManager startUpdatingLocation];
            
        }
        while (0);
        
        //operation end
    }];
    
    
    return blockOperation;
}


//================================================================================
//
//================================================================================
- (void)stopUpdatingCurrentLocation
{
    [self.locationManager stopUpdatingLocation];
}


//================================================================================
//
//================================================================================
- (LocationBlockOperation *)fetchLocationFromAddressDictionary:(NSDictionary *)addressDictionary
                                                      identity:(NSString *)identity
                                                       success:(void (^)(void))success
                                                       failure:(void (^)(NSError *error))failure
{
    __block LocationBlockOperation *blockOperation = nil;
    __block NSError *error = nil;
    
    do
    {
        if(addressDictionary==nil ||
           addressDictionary.count<=0)
        {
            error = PPErrorParameterInvalidity(error);
            break;
        }
        
        //////////////////////////////////////////////////
        
        blockOperation = [[LocationBlockOperation alloc] init];
        
        if(blockOperation==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        __block typeof(self) blockself = self;
        
        [blockOperation addExecutionBlock:^{
            
            do
            {
                if(blockOperation.isCancelled==YES)
                {
                    [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                        object:blockself
                                                                      userInfo:@{PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressLocationFailure),
                                                                                 PPLocationControllerUserInfoKey_Error:PPErrorMake(NSErrorCustom_Code_OperationCancel, @"fetchLocationFromCardModel Operatoin Cancel", nil)}];
                    
                    break;
                }
            
                //////////////////////////////////////////////////

                else if([[PPNetworkReachabilityController networkReachabilityControllerForInternetConnection] status]==PPNetworkReachabilityControllerStatus_ReachableNone)
                {
                    [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                        object:blockself
                                                                      userInfo:@{PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressLocationFailure),
                                                                                 PPLocationControllerUserInfoKey_Error:PPErrorMake(PPLocationControllerErrorCode_NetworkDisconnected, @"NetworkDisconnected", nil)}];
                    
                    break;
                }
                
                //////////////////////////////////////////////////
                
                // dictionary 不能塞空值
                __block id blockIdentity = identity;
                
                if(blockIdentity==nil)
                {
                    blockIdentity = [NSNull null];
                }
                
                //////////////////////////////////////////////////
                
                dispatch_group_t group = dispatch_group_create();
                
                if(group==NULL)
                {
                    error = PPErrorOperationFailed(error);
                    break;
                }
                
                dispatch_group_enter(group);
                
                //////////////////////////////////////////////////
                
                [self.geoCoder geocodeAddressDictionary:addressDictionary
                                      completionHandler:^(NSArray *placemarks, NSError *error)
                 {
                     if(error==nil)
                     {
                         [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                             object:blockself
                                                                           userInfo:@{PPLocationControllerUserInfoKey_Method:NSStringFromClass([self class]),
                                                                                      PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressLocationSuccess),
                                                                                      PPLocationControllerUserInfoKey_Identity:blockIdentity,PPLocationControllerUserInfoKey_Placemarks:placemarks}];
                         
                         //////////////////////////////////////////////////
                         
                         success();
                         
                         //////////////////////////////////////////////////
                         
                         dispatch_group_leave(group);
                     }
                     else
                     {
                         NSError *returnError = nil;
                         
                         // request time out
                         if(error.code==kCLErrorNetwork)
                         {
                             returnError = PPErrorMake(PPLocationControllerErrorCode_RequestTimeOut, error.localizedFailureReason, nil);
                         }
                         else
                         {
                             returnError = error;
                         }
                         
                         //////////////////////////////////////////////////
                         
                         [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                             object:blockself
                                                                           userInfo:@{PPLocationControllerUserInfoKey_Method:NSStringFromClass([self class]),
                                                                                      PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressLocationFailure),
                                                                                      PPLocationControllerUserInfoKey_Identity:blockIdentity,PPLocationControllerUserInfoKey_Error:returnError}];
                         
                         //////////////////////////////////////////////////
                         
                         failure(error);
                         
                         //////////////////////////////////////////////////
                         
                         dispatch_group_leave(group);
                     }
                 }];

                //////////////////////////////////////////////////
                
                dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
                
                dispatch_release(group);
                
                // while loop end
            }
            while (0);
            
            //operation end
            
        }];
        // while loop end
    }
    while (0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil && failure!=NULL)
    {
        failure(error);
    }
    
    //////////////////////////////////////////////////
    
    return [blockOperation autorelease];
}





//================================================================================
//
//================================================================================
- (LocationBlockOperation *)fetchLocationFromAddressString:(NSString *)addressString
                                                    region:(CLCircularRegion *)region
                                                  identity:(NSString *)identity
                                                   success:(void (^)(void))success
                                                   failure:(void (^)(NSError *error))failure
{
    __block LocationBlockOperation *blockOperation = nil;
    __block NSError *error = nil;
    
    do
    {
        if(addressString==nil ||
           addressString.length<=0)
        {
            error = PPErrorParameterInvalidity(error);
            break;
        }
        
        //////////////////////////////////////////////////
        
        blockOperation = [[LocationBlockOperation alloc] init];
        
        if(blockOperation==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        __block typeof(self) blockself = self;
        
        [blockOperation addExecutionBlock:^{
            
            do
            {
                if(blockOperation.isCancelled==YES)
                {
                    [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                        object:blockself
                                                                      userInfo:@{PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressLocationFailure),
                                                                                 PPLocationControllerUserInfoKey_Error:PPErrorMake(NSErrorCustom_Code_OperationCancel, @"fetchLocationFromCardModel Operatoin Cancel", nil)}];
                    
                    break;
                }
                else if([[PPNetworkReachabilityController networkReachabilityControllerForInternetConnection] status]==PPNetworkReachabilityControllerStatus_ReachableNone)
                {
                    [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                        object:blockself
                                                                      userInfo:@{PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressLocationFailure),
                                                                                 PPLocationControllerUserInfoKey_Error:PPErrorMake(PPLocationControllerErrorCode_NetworkDisconnected, @"NetworkDisconnected", nil)}];
                    
                    break;
                }
                
                //////////////////////////////////////////////////
                
                // dictionary 不能塞空值
                __block id blockIdentity = identity;
                
                if(blockIdentity==nil)
                {
                    blockIdentity = [NSNull null];
                }
                
                
                //////////////////////////////////////////////////
                
                dispatch_group_t group = dispatch_group_create();
                
                if(group==NULL)
                {
                    error = PPErrorOperationFailed(error);
                    break;
                }
                
                dispatch_group_enter(group);
                
                //////////////////////////////////////////////////
                
                if(region!=nil)
                {
                    [self.geoCoder geocodeAddressString:[self normalizeAddressString:addressString]
                                               inRegion:region
                                      completionHandler:^(NSArray *placemarks, NSError *error)
                     {
                         if(error==nil)
                         {
                             [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                                 object:blockself
                                                                               userInfo:@{PPLocationControllerUserInfoKey_Method:NSStringFromClass([self class]),
                                                                                          PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressLocationSuccess),
                                                                                          PPLocationControllerUserInfoKey_Identity:blockIdentity,PPLocationControllerUserInfoKey_Placemarks:placemarks}];
                             
                             //////////////////////////////////////////////////
                             
                             success();
                             
                             //////////////////////////////////////////////////
                             
                             dispatch_group_leave(group);
                         }
                         else
                         {
                             NSError *returnError = nil;
                             
                             // request time out
                             if(error.code==kCLErrorNetwork)
                             {
                                 returnError = PPErrorMake(PPLocationControllerErrorCode_RequestTimeOut, error.localizedFailureReason, nil);
                             }
                             else
                             {
                                 returnError = error;
                             }
                             
                             //////////////////////////////////////////////////
                             
                             [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                                 object:blockself
                                                                               userInfo:@{PPLocationControllerUserInfoKey_Method:NSStringFromClass([self class]),
                                                                                          PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressLocationFailure),
                                                                                          PPLocationControllerUserInfoKey_Identity:blockIdentity,PPLocationControllerUserInfoKey_Error:returnError,PPLocationControllerUserInfoKey_Address:addressString}];
                             
                             //////////////////////////////////////////////////
                             
                             failure(error);
                             
                             //////////////////////////////////////////////////
                             
                             dispatch_group_leave(group);
                         }
                     }];
                }
                else
                {
                    NSString *normalizeString = [self normalizeAddressString:addressString];
                    
                    [self.geoCoder geocodeAddressString:normalizeString
                                      completionHandler:^(NSArray *placemarks, NSError *error)
                     {
                        // 比對國名會有多語系問題, 所以改到WCMapViewController比對country code
                         // !! 吐回來的值可能是錯的，所以要檢查一下與原地址是不是吻合
                        
//                         if([placemarks count]>0)
//                         {
//                             CLPlacemark *firstPlacemark = [placemarks firstObject];
//                             NSString *country = [PPCountryCodeConvert localizeStringFromCountryCode:[firstPlacemark.ISOcountryCode lowercaseString]];
//
//                             if([addressString containsString:country]==NO)
//                             {
//                                 placemarks = [NSArray array];
//                                 error = PPErrorMake(PPLocationControllerErrorCode_Other, @"return wrong placemark", nil);
//                             }
//                         }
                         
                         if(error==nil)
                         {
                             [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                                 object:blockself
                                                                               userInfo:@{
                                                                                          PPLocationControllerUserInfoKey_Method:NSStringFromClass([self class]),
                                                                                          PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressLocationSuccess),
                                                                                          PPLocationControllerUserInfoKey_Identity:blockIdentity,
                                                                                          PPLocationControllerUserInfoKey_Placemarks:placemarks,
                                                                                          PPLocationControllerUserInfoKey_Address:addressString}];
                             
                             success();
                             
                             //////////////////////////////////////////////////
                             
                             dispatch_group_leave(group);
                         }
                         else
                         {
                             NSError *returnError = nil;
                             
                             // request time out
                             if(error.code==kCLErrorNetwork)
                             {
                                 returnError = PPErrorMake(PPLocationControllerErrorCode_RequestTimeOut, error.localizedFailureReason, nil);
                             }
                             else
                             {
                                 returnError = error;
                             }
                             
                             //////////////////////////////////////////////////

                             [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                                 object:blockself
                                                                               userInfo:@{PPLocationControllerUserInfoKey_Method:NSStringFromClass([self class]),
                                                                                          PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressLocationFailure),
                                                                                          PPLocationControllerUserInfoKey_Identity:blockIdentity,PPLocationControllerUserInfoKey_Error:returnError,PPLocationControllerUserInfoKey_Address:addressString}];
                             
                             failure(error);
                             
                             //////////////////////////////////////////////////
                             
                             dispatch_group_leave(group);
                             
                         }
                     }];
                }
                
                //////////////////////////////////////////////////
                
                dispatch_group_wait(group, DISPATCH_TIME_FOREVER);
                
                dispatch_release(group);
                
                // while loop end
            }
            while (0);
            
            //operation end
            
        }];
        // while loop end
    }
    while (0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil && failure!=NULL)
    {
        failure(error);
    }
    
    //////////////////////////////////////////////////
    
    return [blockOperation autorelease];
}


//================================================================================
//
//================================================================================
- (LocationBlockOperation *)fetchAddressStringFromLocation:(CLLocation *)location
                                                  identity:(NSString *)identity
                                                   success:(void (^)(void))success
                                                   failure:(void (^)(NSError *error))failure
{
    __block LocationBlockOperation *blockOperation = nil;
    __block NSError *error = nil;
    
    do
    {
        if(location==nil)
        {
            error = PPErrorParameterInvalidity(error);
            break;
        }

        //////////////////////////////////////////////////
        
        blockOperation = [[LocationBlockOperation alloc] init];
        
        if(blockOperation==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        __block typeof(self) blockself = self;
        
        [blockOperation addExecutionBlock:^{
            
            do
            {
                if(blockOperation.isCancelled==YES)
                {
                    [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                        object:blockself
                                                                      userInfo:@{PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressStringFailure),
                                                                                 PPLocationControllerUserInfoKey_Error:PPErrorMake(NSErrorCustom_Code_OperationCancel, @"fetchLocationFromCardModel Operatoin Cancel", nil)}];
                    
                    break;
                }
                else if([[PPNetworkReachabilityController networkReachabilityControllerForInternetConnection] status]==PPNetworkReachabilityControllerStatus_ReachableNone)
                {
                    [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                        object:blockself
                                                                      userInfo:@{PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressStringFailure),
                                                                                 PPLocationControllerUserInfoKey_Error:PPErrorMake(PPLocationControllerErrorCode_NetworkDisconnected, @"NetworkDisconnected", nil)}];
                    
                    break;
                }
                
                //////////////////////////////////////////////////
                
                // dictionary 不能塞空值
                __block id blockIdentity = identity;
                
                if(blockIdentity==nil)
                {
                    blockIdentity = [NSNull null];
                }

                //////////////////////////////////////////////////
                
                dispatch_group_t group = dispatch_group_create();
                
                if(group==NULL)
                {
                    error = PPErrorOperationFailed(error);
                    break;
                }
                
                dispatch_group_enter(group);

                //////////////////////////////////////////////////

                [self.geoCoder reverseGeocodeLocation:location completionHandler:^(NSArray *placemarks, NSError *error)
                {
                    if(error!=nil)
                    {
                        NSError *returnError = nil;
                        
                        // request time out
                        if(error.code==kCLErrorNetwork)
                        {
                            returnError = PPErrorMake(PPLocationControllerErrorCode_RequestTimeOut, error.localizedFailureReason, nil);
                        }
                        else
                        {
                            returnError = error;
                        }
                        
                        //////////////////////////////////////////////////

                        [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                            object:blockself
                                                                          userInfo:@{PPLocationControllerUserInfoKey_Method:NSStringFromClass([self class]),
                                                                                     PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressStringFailure),
                                                                                     PPLocationControllerUserInfoKey_Identity:blockIdentity,PPLocationControllerUserInfoKey_Error:returnError,PPLocationControllerUserInfoKey_Location:location}];
                        
                        //////////////////////////////////////////////////

                        failure(error);
                    }
                    else
                    {
                        [[NSNotificationCenter defaultCenter] postNotificationName:PPLocationControllerNotification
                                                                            object:blockself
                                                                          userInfo:@{PPLocationControllerUserInfoKey_Method:NSStringFromClass([self class]),
                                                                                     PPLocationControllerUserInfoKey_AccessResult:@(PPLocationControllerAccessResult_GetAddressStringSuccess),
                                                                                     PPLocationControllerUserInfoKey_Identity:blockIdentity,PPLocationControllerUserInfoKey_Placemarks:placemarks}];
                        
                        success();
                    }

                    //////////////////////////////////////////////////

                    dispatch_group_leave(group);

                }];

                //////////////////////////////////////////////////
                
                dispatch_group_wait(group, DISPATCH_TIME_FOREVER);

                dispatch_release(group);

                // while loop end
            }
            while (0);
            
            //operation end
            
        }];
        // while loop end
    }
    while (0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil && failure!=NULL)
    {
        failure(error);
    }
    
    //////////////////////////////////////////////////
    
    return [blockOperation autorelease];
}


//================================================================================
//
//================================================================================
- (void)cancelGeocode
{
    [self.geoCoder cancelGeocode];
}
@end
