//
//  PPImageScrollCropView.m
//  PPImageScrollCropView
//
//  Created by pp on 2011/6/26.
//  Copyright 2011 Penpower. All rights reserved.
//

#import "PPImageScrollCropView.h"
////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private definition

// crop view min
#define PPISCV_MAX_CROP_IMAGE_LENGTH				1600
#define PPISCV_DEFAULT_CROP_IMAGESIZE				CGSizeMake(PPISCV_MAX_CROP_IMAGE_LENGTH, PPISCV_MAX_CROP_IMAGE_LENGTH)  // Crop image size

// magnifier
#define PPISCV_DEFAULT_MAGNIFIER_RATIO				3
#define PPISCV_DEFAULT_MAGNIFIER_RADIUS				60





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Inerface PPImageScrollCropView ()

@interface PPImageScrollCropView ()





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Property

@property (nonatomic, retain)               PPCropView			*ppCropView;
@property (nonatomic, readwrite, retain)    UIImage				*originalImage;

@end





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Implementation PPImageScrollCropView

@implementation PPImageScrollCropView





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Init & Dealloc

//================================================================================
//
//================================================================================
- (id)initWithFrame:(CGRect)frame {
    
    self = [super initWithFrame:frame];
    if (self)
    {
        self.imageView.userInteractionEnabled = YES;
        _maxEdgeLength                        = PPISCV_MAX_CROP_IMAGE_LENGTH;
        self.zoomEnabled                      = YES;
        _enableResizeCrop                     = YES;
        
		[self addCropView];
        [self setCropViewHidden:NO];
    }
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc 
{
	self.proxy = nil;
	
	[_originalImage release];
	_originalImage =nil;
	
	[self removeCropView];
	
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Override

//================================================================================
//
//================================================================================
- (UIView*)hitTest:(CGPoint)point withEvent:(UIEvent *)event
{
	if (self.ppCropView.hidden==YES)
    {
        
		if (self.zoomEnabled==YES)
        {
			self.scrollEnabled = YES;
		}
		
		return [super hitTest:point withEvent:event];
	}
	
	UIView * testView =[self.ppCropView hitTest:[self convertPoint:point toView:self.ppCropView] withEvent:event];
	if (testView==nil)
    {
        
		if (self.zoomEnabled==YES)
        {
			self.scrollEnabled = YES;
		}

		return self;
	}
	else
    {
        
		if (self.zoomEnabled==YES)
        {
			self.scrollEnabled = NO;
		}
		return testView;
        
	}

	return nil;
}


//================================================================================
//
//================================================================================
- (void)scrollViewDidZoom:(UIScrollView *)scrollView 
{
	if ([self isZooming]==NO) // 在double tap 的放大或縮小時會進來
	{
		
        if (scrollView.zoomScale >=2)
        {
			[self.ppCropView setSmallLineMode:YES];
		}
		else
        {
			[self.ppCropView setSmallLineMode:NO];
		}
        
		[self.ppCropView setNeedsDisplay];
        
	}
	
	// 重新設定目前的放大鏡的倍率
	[self adjustScale];	

	[super scrollViewDidZoom:scrollView];
}


//================================================================================
//
//================================================================================
- (void)scrollViewDidEndZooming:(UIScrollView *)scrollView withView:(UIView *)view atScale:(CGFloat)scale
{
	// zoom完會進來
	if (scrollView.zoomScale >=2)
    {
		[self.ppCropView setSmallLineMode:YES];
	}
	else
    {
		[self.ppCropView setSmallLineMode:NO];
	}
	
	[self.ppCropView setNeedsDisplay];
	
	// 重新設定目前的放大鏡的倍率
	[self adjustScale];

	[super scrollViewDidEndZooming:scrollView withView:view atScale:scale];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Property

//================================================================================
//
//================================================================================
- (void)setImage:(UIImage *)image
{
    // 影像經過scale後做ｃｒｏｐ的效能比較好，所以都先以scale做處理
    UIImage * imageToBeCroping = [image imageRotatedByDegrees:0 scalingMaxLength:self.maxEdgeLength];
    
    // 儲存原始影像
    self.originalImage = image;
    
    [super setImage:imageToBeCroping];
    
    // 重新設定目前的放大鏡的倍率
    [self adjustScale];
    
    // 重設crop的Layout
    [self.ppCropView layoutSubviews];
}


//================================================================================
//
//================================================================================
- (void)setFrame:(CGRect)frame
{
    [super setFrame:frame];
    
    //////////////////////////////////////////////////
    // 設定完Frame後ImageView的Layout要跟著改變，此外也要先記錄下CropPoints然後再設回去，不然CropPoints會跑掉 
    
    PPISCV_CropPoints cropPoints;
    [self getCropPointsOnOriginalImage:&cropPoints];
    
    [self layoutImageView];
    
    [self setInitCropPoints:cropPoints];
    
}


//================================================================================
//
//================================================================================
- (void)setEnableResizeCrop:(BOOL)enableResizeCrop
{
    _enableResizeCrop = enableResizeCrop;
    
    self.ppCropView.touchEnable = enableResizeCrop;
    self.ppCropView.magnifierEnable = enableResizeCrop;	//可以resize時才顯示放大鏡
    //要重設放大鏡的大小
    [self.ppCropView.ppMagnifierView setFrame:CGRectMake(0, 0, PPISCV_DEFAULT_MAGNIFIER_RADIUS*2, PPISCV_DEFAULT_MAGNIFIER_RADIUS*2)];
    
    [self.ppCropView setNeedsDisplay];
}


//================================================================================
//
//================================================================================
- (void)setMoveCropViewEnable:(BOOL)moveEnable
{
    self.ppCropView.moveCropViewEnable = moveEnable;
}


//================================================================================
//
//================================================================================
- (void)setCropPointRadius:(NSInteger)cropPointRadius
{
    [self.ppCropView setCropPointRadius:cropPointRadius];
    [self.ppCropView layoutSubviews];
}


//================================================================================
//
//================================================================================
- (void)setRectCropMode:(BOOL)rectMode
{
    self.ppCropView.rectCropMode = rectMode;
}


//================================================================================
//
//================================================================================
- (void)setMaxCropSize:(CGSize)maxSize
{
    self.ppCropView.maxSize = maxSize;
}


//================================================================================
//
//================================================================================
- (void)setCropViewHidden:(BOOL)hidden
{
    self.ppCropView.hidden = hidden;
    
    if (self.ppCropView.hidden==YES)
    {
        self.edgeInsetsForRecessed = UIEdgeInsetsMake(0, 0, 0, 0);
    }
    else
    {
        NSUInteger cropPointRadius = self.ppCropView.cropPointRadius;
        self.edgeInsetsForRecessed = UIEdgeInsetsMake(cropPointRadius, cropPointRadius, cropPointRadius, cropPointRadius);
    }
    
    [self.ppCropView layoutSubviews];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private - add & remove CropView

//================================================================================
//
//================================================================================
- (void)addCropView
{
	_ppCropView = [[PPCropView alloc] initWithFrame:self.imageView.bounds];
	if (_ppCropView !=nil)
    {
		// 設定放大鏡屬性
		[_ppCropView.ppMagnifierView setMagnifyRatio:PPISCV_DEFAULT_MAGNIFIER_RATIO];
		[_ppCropView.ppMagnifierView setFrame:CGRectMake(0, 0, PPISCV_DEFAULT_MAGNIFIER_RADIUS*2, PPISCV_DEFAULT_MAGNIFIER_RADIUS*2)];
				
		// 是否可以做Crop
		_ppCropView.touchEnable = self.enableResizeCrop;
		[self.imageView addSubview:_ppCropView];
	}
}


//================================================================================
//
//================================================================================
- (void)removeCropView
{
	if (self.ppCropView!=nil)
    {
		[self.ppCropView removeFromSuperview];
		self.ppCropView =nil;
	}
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private - adjust Scale

//================================================================================
//
//================================================================================
- (void)adjustScale
{
	CGFloat newScale = 0.0;
    
	if(self.zoomScale>=self.doubleTapZoomScale)
	{
		newScale = self.zoomScale;
	}
	else
	{
		newScale = self.doubleTapZoomScale;
	}
	
	// 重新設定目前的放大鏡的倍率
	[self.ppCropView.ppMagnifierView setMagnifyRatio:newScale];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private - Others

//================================================================================
//
//================================================================================
- (BOOL)isRectMode
{
    return [self.ppCropView isRectMode];
}


//================================================================================
//
//================================================================================
- (BOOL)isCropViewHidden
{
    return self.ppCropView.hidden;
}

//================================================================================
//
//================================================================================
- (CGSize)maxCropSize
{
    return self.ppCropView.maxSize;
}


//================================================================================
//
//================================================================================
- (BOOL)moveCropViewEnable
{
    return self.ppCropView.moveCropViewEnable;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Method - Crop Point Coordinate Handling

//================================================================================
//
//================================================================================
- (void)setInitCropPoints:(PPISCV_CropPoints)initCropPoints
{
    CGPoint cropPoints[PPCV_SM_Count];
    
    CGAffineTransform t = CGAffineTransformMakeScale(self.imageView.bounds.size.width/self.originalImage.size.width,
                                                     self.imageView.bounds.size.height/self.originalImage.size.height);
    
    cropPoints[PPCV_SM_LeftTop] = CGPointApplyAffineTransform(initCropPoints.ptLeftTop, t);
    cropPoints[PPCV_SM_RightTop] = CGPointApplyAffineTransform(initCropPoints.ptRightTop, t);
    cropPoints[PPCV_SM_LeftBottom] = CGPointApplyAffineTransform(initCropPoints.ptLeftBottom, t);
    cropPoints[PPCV_SM_RightBottom] = CGPointApplyAffineTransform(initCropPoints.ptRightBottom, t);


    [self.ppCropView adjustCropViewFrame];
    [self.ppCropView setCropPointArray:cropPoints];
    [self.ppCropView setNeedsDisplay];
}


//================================================================================
//
//================================================================================
- (BOOL)getCropPointsOnView:(PPISCV_CropPoints*)ppicvCropPoints
{
	if (ppicvCropPoints == NULL)
    {
		return FALSE;
	}
    
	CGPoint cropPoints[PPCV_SM_Count/2];
	BOOL bResult = [self.ppCropView getCropPointArray:cropPoints];
	
	
	ppicvCropPoints->ptLeftTop = cropPoints[PPCV_SM_LeftTop];
	ppicvCropPoints->ptRightTop = cropPoints[PPCV_SM_RightTop];
	ppicvCropPoints->ptLeftBottom = cropPoints[PPCV_SM_LeftBottom];
	ppicvCropPoints->ptRightBottom = cropPoints[PPCV_SM_RightBottom];

	return bResult;
}


//================================================================================
//
//================================================================================
- (BOOL)getCropPointsOnOriginalImage:(PPISCV_CropPoints*)ppicvCropPoints
{	
	if (ppicvCropPoints == NULL)
    {
		return FALSE;
	}
    
	CGPoint cropPoints[PPCV_SM_Count];
	BOOL bResult = [self.ppCropView getCropPointArray:cropPoints];
	
	//依比例算出在原圖中的Crop points座標
	CGAffineTransform t = CGAffineTransformMakeScale(self.originalImage.size.width/self.imageView.bounds.size.width, 
													 self.originalImage.size.height/self.imageView.bounds.size.height);
	
	ppicvCropPoints->ptLeftTop = CGPointApplyAffineTransform(cropPoints[PPCV_SM_LeftTop], t) ;
	ppicvCropPoints->ptRightTop = CGPointApplyAffineTransform(cropPoints[PPCV_SM_RightTop], t) ;
	ppicvCropPoints->ptLeftBottom = CGPointApplyAffineTransform(cropPoints[PPCV_SM_LeftBottom], t) ;
	ppicvCropPoints->ptRightBottom = CGPointApplyAffineTransform(cropPoints[PPCV_SM_RightBottom], t) ;
	
	return bResult;
}


//================================================================================
// 把輸入的rect 轉為對應到imageView的座標
//================================================================================
- (CGRect)convertRectFromOriginalImage:(CGRect)originalRect
{
    CGAffineTransform t = CGAffineTransformMakeScale(self.imageView.bounds.size.width/self.originalImage.size.width,
                                                     self.imageView.bounds.size.height/self.originalImage.size.height);
    
    return CGRectApplyAffineTransform(originalRect, t);
}


//================================================================================
// 把輸入的rect 轉為對應到原始圖片的座標
//================================================================================
- (CGRect)convertRectToOriginalImage:(CGRect)originalRect
{
    CGAffineTransform t = CGAffineTransformMakeScale(self.originalImage.size.width/self.imageView.bounds.size.width,
                                                     self.originalImage.size.height/self.imageView.bounds.size.height);
    
    return CGRectApplyAffineTransform(originalRect, t);
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Method - Others

//================================================================================
//
//================================================================================
- (UIImage*)imageCroped
{
	PPISCV_CropPoints ppCropPoints;
	[self getCropPointsOnOriginalImage:&ppCropPoints];
	
	CGRect cropRect = CGRectMake(MIN(MIN(ppCropPoints.ptLeftTop.x, ppCropPoints.ptLeftBottom.x),MIN(ppCropPoints.ptRightTop.x, ppCropPoints.ptRightBottom.x)), 
								 MIN(MIN(ppCropPoints.ptLeftTop.y, ppCropPoints.ptLeftBottom.y),MIN(ppCropPoints.ptRightTop.y, ppCropPoints.ptRightBottom.y)), 
								 MAX(MAX(ppCropPoints.ptLeftTop.x, ppCropPoints.ptLeftBottom.x),MAX(ppCropPoints.ptRightTop.x, ppCropPoints.ptRightBottom.x))-MIN(MIN(ppCropPoints.ptLeftTop.x, ppCropPoints.ptLeftBottom.x),MIN(ppCropPoints.ptRightTop.x, ppCropPoints.ptRightBottom.x)), 
								 MAX(MAX(ppCropPoints.ptLeftTop.y, ppCropPoints.ptLeftBottom.y),MAX(ppCropPoints.ptRightTop.y, ppCropPoints.ptRightBottom.y))-MIN(MIN(ppCropPoints.ptLeftTop.y, ppCropPoints.ptLeftBottom.y),MIN(ppCropPoints.ptRightTop.y, ppCropPoints.ptRightBottom.y)));

	return [UIImage imageWithCGImage:[self.originalImage imageCroppedByRect:cropRect].CGImage];
}


//================================================================================
//
//================================================================================
- (CGRect)cropMinRect
{
	return [self.ppCropView minimizeRect];
}


@end
