//
//  PPGoogleAuthenticationController.m
//
//  !! 目前的架構只能提供一組服務使用，多組要使用時需調整。
//
#import "PPGoogleAuthenticationController.h"

#import "OIDAuthState+IncrementalAuthorization.h"

#if TARGET_OS_IPHONE
#import "OIDExternalUserAgentIOS.h"
#import "OIDAuthState+IOS.h"
#elif TARGET_OS_MAC
#import "OIDExternalUserAgentMac.h"
#endif

static NSString *PPGoogleAuthenticationController_RedirectURIPosfix = @":/oauthredirect";


////////////////////////////////////////////////////////////////////////////////////////////////////////
@interface PPGoogleAuthenticationController () <OIDAuthStateChangeDelegate>

@property (atomic, assign) id<PPGoogleAuthenticationControllerDelegate> delegate;
    
@property (nonatomic, retain) NSString *keychainName;
@property (nonatomic, retain) NSString *clientID;
@property (nonatomic, retain) NSString *clientSecret;
    

@property (nonatomic, retain) GTMAppAuthFetcherAuthorization *authentication;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPGoogleAuthenticationController

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods

//================================================================================
//
//================================================================================
- (instancetype)init
{
    if(self = [super init])
    {
        // TODO: 檢查更新token是否會收到通知，如果不會，如何得知tokenchange
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    self.authentication = nil;
    self.keychainName = nil;
    self.clientID = nil;
    self.clientSecret = nil;
    
    self.currentAuthorizationFlow = nil;
    
    //////////////////////////////////////////////////
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Recieve Notificaiton Method

#if TARGET_OS_IPHONE
#elif TARGET_OS_MAC

//================================================================================
//
//================================================================================
- (void)handleGetURLEvent:(NSAppleEventDescriptor *)event
           withReplyEvent:(NSAppleEventDescriptor *)replyEvent
{
    NSString *URLString = [[event paramDescriptorForKeyword:keyDirectObject] stringValue];
    NSURL *URL = [NSURL URLWithString:URLString];
    [self.currentAuthorizationFlow resumeExternalUserAgentFlowWithURL:URL];
}
#endif






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods

//================================================================================
//
//================================================================================
- (BOOL)initializeWithKeychainName:(NSString *)keychainName
                          clientID:(NSString *)clientID
                      clientSecret:(NSString *)clientSecret
{
    [self dumpLogMessageWithFormat:@"%s in", __func__];
    
    //////////////////////////////////////////////////
    
    // clientSecret可能是nil
    if([keychainName length]==0 || [clientID length]==0 /*|| [clientSecret length]==0*/)
    {
        [self dumpLogMessageWithFormat:@"%s out (invalid param)", __func__];
        return NO;
    }
    
    self.keychainName = keychainName;
    self.clientID = clientID;
    self.clientSecret = clientSecret;
    
    BOOL result = ([self googleAuthenticationFromKeychain] != nil);
    
    //////////////////////////////////////////////////
    
    [self dumpLogMessageWithFormat:@"%s out (result=%d)", __func__, result];
    
    return result;
}


//================================================================================
//
//================================================================================
- (GTMAppAuthFetcherAuthorization *)googleAuthenticationFromKeychain
{
    [self dumpLogMessageWithFormat:@"%s in", __func__];
    
    @synchronized(self)
    {
        self.authentication =
        [GTMAppAuthFetcherAuthorization authorizationFromKeychainForName:self.keychainName];

        [self dumpLogMessageWithFormat:@"%s out", __func__];
        
        return self.authentication;
    }
}


//================================================================================
//
//================================================================================
- (GTMAppAuthFetcherAuthorization *)lastAuthentication
{
    [self dumpLogMessageWithFormat:@"%s in", __func__];

    @synchronized(self)
    {
        [self dumpLogMessageWithFormat:@"%s out", __func__];
        self.authentication =
        [GTMAppAuthFetcherAuthorization authorizationFromKeychainForName:self.keychainName];

        return self.authentication;
    }
}


//================================================================================
//
//================================================================================
- (BOOL)saveGoogleAuthentication:(GTMAppAuthFetcherAuthorization *)authentication
{
    [self dumpLogMessageWithFormat:@"%s in", __func__];
    
    @synchronized(self)
    {
        BOOL result = NO;
    
        // !! 檢查是不是目前使用的auth
        if(authentication != nil &&
           [self.authentication isEqual:authentication]==NO)
        {
            BOOL isSaveAuthorizatoinSuccess = NO;
            
            // !! 如果authorization不存在，
            // ios的做法|removeAuthorizationFromKeychainForName:|會回傳錯誤, 且saveAuthorization時就會call remove的動作
            // 但macOS內部會當成功，且save時不會call removeAuthorizationFromKeychainForName，所以要在外面呼叫
#if TARGET_OS_IPHONE
            
            isSaveAuthorizatoinSuccess = [GTMAppAuthFetcherAuthorization saveAuthorization:authentication toKeychainForName:self.keychainName];
            
#elif TARGET_OS_MAC
            // 要先remove才能save
            isSaveAuthorizatoinSuccess = ([GTMAppAuthFetcherAuthorization removeAuthorizationFromKeychainForName:self.keychainName] == YES &&
                                          [GTMAppAuthFetcherAuthorization saveAuthorization:authentication toKeychainForName:self.keychainName] == YES);
#endif
			
            if(isSaveAuthorizatoinSuccess)
            {
                self.authentication = authentication;
                result = YES;
            }
        }
    
        [self dumpLogMessageWithFormat:@"%s out (result=%d)", __func__, result];
        
        return result;
    }
}


//================================================================================
//
//================================================================================
- (void)removeGoogleAuthentication
{
    [self dumpLogMessageWithFormat:@"%s in", __func__];
    
    @synchronized(self)
    {
        [GTMAppAuthFetcherAuthorization removeAuthorizationFromKeychainForName:self.keychainName];

        self.authentication = nil;
    
        //////////////////////////////////////////////////
        
        [self dumpLogMessageWithFormat:@"%s out", __func__];
    }
}

    
    
    
////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - login with GTMAppAuth (Private)


//================================================================================
//
//================================================================================
+ (NSString *)reverseClientID:(NSString*)clientID
{
    if ([clientID length]==0)
    {
        return nil;
    }
    
    NSArray *clientIDComponents = [clientID componentsSeparatedByString:@"."];
    NSEnumerator *enumerator = [clientIDComponents reverseObjectEnumerator];
    
    return [[enumerator allObjects] componentsJoinedByString:@"."];
}


//==============================================================================
//
//==============================================================================
- (OIDAuthorizationRequest *)authorizationRequestWithScopes:(NSArray *)scopes
{
    //////////////////////////////////////////////////
    // assert check
    NSAssert(!([self.clientID length]==0),
             @"Initialize with your own client ID first. "
             "Instructions: https://github.com/openid/AppAuth-iOS/blob/master/Example/README.md");
    
    NSString *reversedClientID = [PPGoogleAuthenticationController reverseClientID:self.clientID];
    NSString *redirectURIString = [reversedClientID stringByAppendingString:PPGoogleAuthenticationController_RedirectURIPosfix];
    
    NSAssert(!([redirectURIString length]==0),
             @"Initialize with your own client ID first. "
             "Instructions: https://github.com/openid/AppAuth-iOS/blob/master/Example/README.md");
    
    // verifies that the custom URI scheme has been updated in the Info.plist
    NSArray *urlTypes = [[NSBundle mainBundle] objectForInfoDictionaryKey:@"CFBundleURLTypes"];
    NSAssert(urlTypes.count > 0, @"No custom URI scheme has been configured for the project.");
    NSArray *urlSchemes = ((NSDictionary *)urlTypes.firstObject)[@"CFBundleURLSchemes"];
    NSAssert(urlSchemes.count > 0, @"No custom URI scheme has been configured for the project.");
    
    BOOL isURLSchemeSettingCorrect = NO;
    for (NSDictionary *urlSchemes in urlTypes)
    {
        for (NSString *urlScheme in urlSchemes[@"CFBundleURLSchemes"])
        {
            if([urlScheme isEqualToString:reversedClientID])
            {
                isURLSchemeSettingCorrect = YES;
            }
        }
    }
    
    
    NSAssert((isURLSchemeSettingCorrect==YES),@"URL Scheme: %@ is not one of the url scheme in info.plist. "
             "Full instructions:  https://github.com/openid/AppAuth-iOS/blob/master/Example/README.md", reversedClientID);
    
    //////////////////////////////////////////////////
    OIDAuthorizationRequest *request = nil;
    NSURL *redirectURI = [NSURL URLWithString:redirectURIString];
    
    OIDServiceConfiguration *configuration =
    [GTMAppAuthFetcherAuthorization configurationForGoogle];
    
    if (!configuration) {
        [self dumpLogMessageWithFormat:@"Get configuration failed"];
        [self saveGoogleAuthentication:nil];
        return request;
    }
    
    [self dumpLogMessageWithFormat:@"Got configuration: %@", configuration];
    
    // set login hint
    NSDictionary *additionalParamater = nil;
    if (self.authentication)
    {
        NSString *userEmail = [self.authentication userEmail];
        if ([userEmail length]>0)
        {
            additionalParamater = @{@"login_hint":userEmail};
        }
    }
    //////////////////////////////////////////////////
    // extend scope
    NSMutableArray *adjustScopes = [NSMutableArray arrayWithArray:scopes];
    [adjustScopes addObject:OIDScopeOpenID];
    [adjustScopes addObject:OIDScopeProfile];
    [adjustScopes addObject:OIDScopeEmail];
    // builds authentication request
    request =
    [[[OIDAuthorizationRequest alloc] initWithConfiguration:configuration
                                                   clientId:self.clientID
                                               clientSecret:self.clientSecret
                                                     scopes:adjustScopes
                                                redirectURL:redirectURI
                                               responseType:OIDResponseTypeCode
                                       additionalParameters:additionalParamater] autorelease];
    
    return request;
}


#if  TARGET_OS_IOS

//==============================================================================
//
//==============================================================================
- (void)showSignInWithScopes:(NSArray *)scopes
              viewController:(UIViewController *)viewController
             completeHandler:(GoogleSignInCompleteHandler)handler
{
    // builds authentication request
    OIDAuthorizationRequest *request = [self authorizationRequestWithScopes:scopes];
    // performs authentication request
    
    self.currentAuthorizationFlow =
    [OIDAuthState authStateByPresentingAuthorizationRequest:request
                                   presentingViewController:viewController
                                                   callback:^(OIDAuthState *_Nullable authState,
                                                              NSError *_Nullable error)
     {
         __block typeof(self) blockSelf = self;
         
         dispatch_async(dispatch_get_main_queue(), ^{
             if (authState) {
                 
                 authState.stateChangeDelegate = self;
                 
                 GTMAppAuthFetcherAuthorization *authorization =
                 [[GTMAppAuthFetcherAuthorization alloc] initWithAuthState:authState];
                 
                 [self saveGoogleAuthentication:authorization];
                 
                 if (handler)
                 {
                     handler(authorization, nil);
                 }
                 
                 [authorization release];
                 
                 [blockSelf dumpLogMessageWithFormat:@"Got authorization tokens. Access token: %@",
                  authState.lastTokenResponse.accessToken];
             }
             else
             {
//                 [blockSelf saveGoogleAuthentication:nil];
                 [blockSelf dumpLogMessageWithFormat:@"Authorization error: %@", [error localizedDescription]];
                 
                 if (handler)
                 {
                     handler(nil, error);
                 }
             }
         });
     }];    
}


//================================================================================
//
//================================================================================
- (void)incrementalAuthorizationWithScopes:(NSArray *)scopes
                       viewController:(UIViewController *)viewController
                           completeHandler:(IncrementalAuthorizationCompleteHandler)handler
{
    GTMAppAuthFetcherAuthorization *authorization = [PPGoogleAuthenticationController lastAuthentication];
    
    OIDAuthorizationRequest *request = [authorization.authState incrementalAuthorizationRequestWithScopes:scopes 
                                                                                     additionalParameters:nil];
    
    //////////////////////////////////////////////////
    
    OIDExternalUserAgentIOS *agent = [[[OIDExternalUserAgentIOS alloc] initWithPresentingViewController:viewController] autorelease];
    
    __block typeof(self) blockself = self;
    
    [PPGoogleAuthenticationController sharedInstance].currentAuthorizationFlow =
    
    [authorization.authState presentIncrementalAuthorizationRequest:request externalUserAgent:agent callback:^(BOOL success, NSError * _Nullable error) {
        
        GTMAppAuthFetcherAuthorization *newAuthorization = nil;
        
        if(success==YES)
        {
            authorization.authState.stateChangeDelegate = blockself;
            
            newAuthorization =
            [[GTMAppAuthFetcherAuthorization alloc] initWithAuthState:authorization.authState];
            
            //////////////////////////////////////////////////
            
            [blockself saveGoogleAuthentication:newAuthorization];
            
        }
        
        handler(newAuthorization, error);
        
        [newAuthorization release];
    }];
}


#elif TARGET_OS_MAC


//==============================================================================
//
//==============================================================================
- (void)showSignInWithScopes:(NSArray *)scopes
             completeHandler:(GoogleSignInCompleteHandler)handler
{
    // !! setEventHandler要在開始SignIn前設定，不能在init設定。
    //    因為使用PPCloud的google drive登入時也會做setEventHandler，
    //    會導致這邊收不到handleGetURLEvent。
    
    [[NSAppleEventManager sharedAppleEventManager] setEventHandler:self
                                                       andSelector:@selector(handleGetURLEvent:withReplyEvent:)
                                                     forEventClass:kInternetEventClass
                                                        andEventID:kAEGetURL];
	
    //////////////////////////////////////////////////

    // builds authentication request
    OIDAuthorizationRequest *request = [self authorizationRequestWithScopes:scopes];
    // performs authentication request

	OIDExternalUserAgentMac *externalUserAgent = [[[OIDExternalUserAgentMac alloc] init] autorelease];
	
    self.currentAuthorizationFlow =
	[OIDAuthState authStateByPresentingAuthorizationRequest:request externalUserAgent:externalUserAgent callback:^(OIDAuthState * _Nullable authState, NSError * _Nullable error) {

		__block typeof(self) blockSelf = self;
		
		dispatch_async(dispatch_get_main_queue(), ^{
			if (authState) {
				
				authState.stateChangeDelegate = self;
				
				GTMAppAuthFetcherAuthorization *authorization =
				[[GTMAppAuthFetcherAuthorization alloc] initWithAuthState:authState];
				
				[self saveGoogleAuthentication:authorization];
				
				if (handler)
				{
					handler(authorization, nil);
				}
				
				[authorization release];
				
				[blockSelf dumpLogMessageWithFormat:@"Got authorization tokens. Access token: %@",
				 authState.lastTokenResponse.accessToken];
			}
			else
			{
				[blockSelf dumpLogMessageWithFormat:@"Authorization error: %@", [error localizedDescription]];
				
				if (handler)
				{
					handler(nil, error);
				}
			}
		});
	}];
}


//================================================================================
//
//================================================================================
- (void)incrementalAuthorizationWithScopes:(NSArray *)scopes
                           completeHandler:(IncrementalAuthorizationCompleteHandler)handler
{
    GTMAppAuthFetcherAuthorization *authorization = [PPGoogleAuthenticationController lastAuthentication];
    
    OIDAuthorizationRequest *request = [authorization.authState incrementalAuthorizationRequestWithScopes:scopes additionalParameters:nil];
 
    //////////////////////////////////////////////////

    OIDExternalUserAgentMac *agent = [[[OIDExternalUserAgentMac alloc] init] autorelease];
    
    __block typeof(self) blockself = self;
    
    [PPGoogleAuthenticationController sharedInstance].currentAuthorizationFlow =
    
    [authorization.authState presentIncrementalAuthorizationRequest:request externalUserAgent:agent callback:^(BOOL success, NSError * _Nullable error) {
        
        GTMAppAuthFetcherAuthorization *newAuthorization = nil;
        
        if(success==YES)
        {
            authorization.authState.stateChangeDelegate = blockself;
            
            newAuthorization =
            [[GTMAppAuthFetcherAuthorization alloc] initWithAuthState:authorization.authState];
            
            //////////////////////////////////////////////////
            
            [blockself saveGoogleAuthentication:newAuthorization];
            
        }
        
        handler(newAuthorization, error);
        
        [newAuthorization release];
    }];
}
#endif





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - OIDAuthStateChageDelegate


//==============================================================================
//
//==============================================================================
- (void)didChangeState:(OIDAuthState *)state
{
    [self dumpLogMessageWithFormat:@"%s in (state:%@)", __func__, state];

    GTMAppAuthFetcherAuthorization *authorization =
    [[GTMAppAuthFetcherAuthorization alloc] initWithAuthState:state];
    
    [self saveGoogleAuthentication:authorization];
    
    [authorization release];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - login with GTMAppAuth

#if  TARGET_OS_IOS

//================================================================================
//
//================================================================================
+ (void)showSignInWithScopes:(NSArray *)scopes
              viewController:(UIViewController *)viewController
             completeHandler:(GoogleSignInCompleteHandler)handler
{
    [[PPGoogleAuthenticationController sharedInstance] showSignInWithScopes:scopes
                                                             viewController:viewController
                                                            completeHandler:handler];
}


//================================================================================
//
//================================================================================
+ (void)incrementalAuthorizationWithScopes:(NSArray *)scopes
                            viewController:(UIViewController *)viewController
                           completeHandler:(IncrementalAuthorizationCompleteHandler)handler
{
    [[PPGoogleAuthenticationController sharedInstance] incrementalAuthorizationWithScopes:scopes viewController:viewController completeHandler:^(id<GTMFetcherAuthorizationProtocol> auth, NSError *error) {
        if(handler!=nil)
        {
            handler(auth,error);
        }
    }];
}


#elif TARGET_OS_MAC

//================================================================================
//
//================================================================================
+ (void)showSignInWithScopes:(NSArray *)scopes
             completeHandler:(GoogleSignInCompleteHandler)handler
{
    [[PPGoogleAuthenticationController sharedInstance] showSignInWithScopes:scopes
                                                            completeHandler:handler];
}


//================================================================================
//
//================================================================================
+ (void)incrementalAuthorizationWithScopes:(NSArray *)scopes
                           completeHandler:(IncrementalAuthorizationCompleteHandler)handler
{
    [[PPGoogleAuthenticationController sharedInstance] incrementalAuthorizationWithScopes:scopes completeHandler:^(id<GTMFetcherAuthorizationProtocol> auth, NSError *error) {
        if(handler!=nil)
        {
            handler(auth,error);
        }
    }];
}

#endif





//==============================================================================
//
//==============================================================================
+ (BOOL)handleOpenURL:(NSURL *)url
{
    if([[PPGoogleAuthenticationController sharedInstance].currentAuthorizationFlow resumeExternalUserAgentFlowWithURL:url])
    {
        // 處理後要把currentAuthorizationFlow 清掉，不然下次從Google Drive登入的openURL會call進來，造成當機
        [PPGoogleAuthenticationController sharedInstance].currentAuthorizationFlow = nil;
        return YES;
    }
    return NO;
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Debug message Methods

//==============================================================================
//
//==============================================================================
- (void)dumpLogMessageWithFormat:(NSString *)format, ...
{    
    SEL logSeletor = @selector(logMessage:sender:);
    
    if ([self.delegate respondsToSelector:logSeletor])
    {
        @autoreleasepool
        {
            va_list args;
            va_start(args,format);
            
            NSString *message = [[[NSString alloc] initWithFormat:format arguments:args] autorelease];
            message = [@"\t\t\t##3a## " stringByAppendingString:message];
            [self.delegate performSelector:logSeletor withObject:message withObject:self];
            
            va_end(args);
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class methods

//================================================================================
//
//================================================================================
+ (PPGoogleAuthenticationController *)sharedInstance
{
    static PPGoogleAuthenticationController *sharedInstance = nil;
    static dispatch_once_t onceToken;
    
    dispatch_once(&onceToken, ^{
        sharedInstance = [[PPGoogleAuthenticationController alloc] init];
    });
    
    return sharedInstance;
}


//================================================================================
//
//================================================================================
+ (void)setDelegate:(id<PPGoogleAuthenticationControllerDelegate>)delegate
{
    [self sharedInstance].delegate = delegate;
}


//================================================================================
//
//================================================================================
+ (BOOL)initializeWithKeychainName:(NSString *)keychainName
                          clientID:(NSString *)clientID
                      clientSecret:(NSString *)clientSecret
{
    return [[self sharedInstance] initializeWithKeychainName:keychainName
                                                    clientID:clientID
                                                clientSecret:clientSecret];
}


//================================================================================
//
//================================================================================
+ (GTMAppAuthFetcherAuthorization *)lastAuthentication
{
    return [[self sharedInstance] lastAuthentication];
}


//================================================================================
//
//================================================================================
+ (void)renewGoogleAuthentication
{
    [[self sharedInstance] googleAuthenticationFromKeychain];
}


//================================================================================
//
//================================================================================
+ (BOOL)saveGoogleAuthentication:(GTMAppAuthFetcherAuthorization *)authentication
{
    return [[self sharedInstance] saveGoogleAuthentication:authentication];
}


//================================================================================
//
//================================================================================
+ (void)removeGoogleAuthentication
{
    [[self sharedInstance] removeGoogleAuthentication];
}


//==============================================================================
//
//==============================================================================
+ (void)replaceAppAuthWithNewKeyChainName:(NSString *)newKeyChainName
                      fromOldKeyChainName:(NSString *)oldkeyChainName
                              oldClientID:(NSString *)oldClientID
                             oldSecrectID:(NSString *)oldSecrectID
{
    // Attempt to deserialize from Keychain in GTMAppAuth format.
    id<GTMFetcherAuthorizationProtocol> authorization =
    [GTMAppAuthFetcherAuthorization authorizationFromKeychainForName:newKeyChainName];
    
    // If no data found in the new format, try to deserialize data from GTMOAuth2
    if (!authorization) {
        // Tries to load the data serialized by GTMOAuth2 using old keychain name.
        // If you created a new client id, be sure to use the *previous* client id and secret here.
        authorization =
        [GTMOAuth2KeychainCompatibility authForGoogleFromKeychainForName:oldkeyChainName
                                                                clientID:oldClientID
                                                            clientSecret:oldSecrectID];
        if (authorization) {
            // Remove previously stored GTMOAuth2-formatted data.
            [GTMOAuth2KeychainCompatibility removeAuthFromKeychainForName:oldkeyChainName];
            // Serialize to Keychain in GTMAppAuth format.
            [GTMAppAuthFetcherAuthorization saveAuthorization:(GTMAppAuthFetcherAuthorization *)authorization
                                            toKeychainForName:newKeyChainName];
        }
    }
}


@end
