//
//  PPCropView.m
//  WorldCardContacts
//
//  Created by Eddie on 2009/12/16.
//  Copyright 2009 __MyCompanyName__. All rights reserved.
//

#import "PPCropView.h"
#import "NSTimer+Additions.h"

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - private definition

#define sqr(x) ABS(x)*ABS(x)                                        // 平方計算
#define dist(x,y) sqrtf(sqr(x)+sqr(y))                              // 距離計算

// select mark
#define PPCV_SELECT_MARK_NONE				-1
#define PPCV_SELECT_MARK_MOVE				-2

// magnifier attribute define
#define PPCV_MAGNIFIER_DEFAULT_SHIFT		CGPointMake(0, -20)		// 放大鏡的位移
#define PPCV_MAGNIFIER_RADIUS				40						// 放大鏡的半徑
#define PPCV_MAGNIFIER_DEFAULT_STATE		YES						// 預設開啟放大鏡

// select mark attribute define 
#define PPCV_SELECT_MARK_SIZE				32                      // 圓形選取區域直徑
#define PPCV_EXTRA_TOUCH_MARK_SIZE			0                       // 擴充的選取區可感應的範圍

#define PPCV_CONFIRM_TIMEINTERVAL			0.01                    // 確認點確認秒數
#define PPCV_CONFIRM_DISTANCE				20                      // 'touchEnd時，用來防止滑動

// crop line color
#define PPCV_LINE_COLOR_R                   (240.0/255.0)           // (218, 141,  51) - orange color
#define PPCV_LINE_COLOR_G                   (125.0/255.0)
#define PPCV_LINE_COLOR_B                   (0.0/255.0)

// 被選取的點的顏色
#define PPCV_SEL_POINT_COLOR_R              1                       // (255, 0,  0) - orange color
#define PPCV_SEL_POINT_COLOR_G              0
#define PPCV_SEL_POINT_COLOR_B              0

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - private property declare

@interface PPCropView()
//default 'PPCV_MIN_CROP_SIZE'
@property (nonatomic, assign) CGPoint                   confirmedTouchPoint;
@property (nonatomic, assign) CGPoint                   lastMovePoint;
@property (nonatomic, assign) CGPoint                   moveStartPoint;
@property (nonatomic, assign) CGRect                    imageFrame;
@property (nonatomic, assign) CGSize                    minSize;
@property (nonatomic, retain) CIColor                   *lineColor;
@property (nonatomic, assign) NSInteger                 currentSelectedMark;
@property (nonatomic, retain) NSTimer                   *touchConfirmTimer;
@property (nonatomic, retain) UITouch                   *currentTouchPoint;

@property (nonatomic, readwrite,retain) PPMagnifierView	*ppMagnifierView;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - private method declare

@interface PPCropView (private)

- (void)calculateCropPointsWithNewFrame:(CGRect)newFrame;
- (void)drawSelectRect:(CGRect)markFrame onContext:(CGContextRef)context isSelected:(BOOL)selected;
- (BOOL)getValidPoint:(CGPoint*)curPoint withTouchIndex:(NSInteger)touchIndex;
- (BOOL)isCenterSelectMarkEnableWithStartPoint:(CGPoint)startPoint endPoint:(CGPoint)endPoint;
- (BOOL)isPointInCropView:(CGPoint)point;
- (void)moveCropToPoint:(CGPoint)curPoint;
- (void)setCropNode:(NSInteger)cropNode toPoint:(CGPoint)curPoint;
- (void)setSelectMarkFrame;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - @implementation PPCropView

@implementation PPCropView

@synthesize confirmedTouchPoint = confirmedTouchPoint_;
@synthesize cropPointRadius     = cropPointRadius_;
@synthesize currentTouchPoint   = currentTouchPoint_;
@synthesize currentSelectedMark = currentSelectedMark_;
@synthesize imageFrame          = imageFrame_;
@synthesize lastMovePoint       = lastMovePoint_;
@synthesize lineColor           = lineColor_;
@synthesize magnifierEnable     = magnifierEnable_;
@synthesize maxSize             = maxSize_;
@synthesize minSize             = minSize_;
@synthesize moveCropViewEnable  = moveCropViewEnable_;
@synthesize moveStartPoint      = moveStartPoint_;
@synthesize rectCropMode        = rectCropMode_;
@synthesize smallLineMode       = smallLineMode_;
@synthesize touchConfirmTimer   = touchConfirmTimer_;
@synthesize touchEnable         = touchEnable_;

@synthesize ppMagnifierView     = ppMagnifierView_;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - initial methods

//===============================================================================
//
//===============================================================================
- (id)initWithFrame:(CGRect)frame
{
	self = [super initWithFrame:frame];
    if (self!=nil) 
	{
        self.backgroundColor      = [UIColor clearColor];
        self.multipleTouchEnabled = NO;

        //////////////////////////////////////////////////

        smallLineMode_            = NO;
        touchEnable_              = YES;

        //////////////////////////////////////////////////
		// initial touch state

        currentSelectedMark_      = PPCV_SELECT_MARK_NONE;
        currentTouchPoint_        = nil;

        //////////////////////////////////////////////////
		// initial crop attribute

        cropPointRadius_          = PPCV_SELECT_MARK_SIZE/2;
        moveCropViewEnable_       = NO;
        rectCropMode_             = YES;
        minSize_                  = CGSizeMake(cropPointRadius_, cropPointRadius_);
        maxSize_                  = CGSizeMake(frame.size.width, frame.size.height);
		
        //////////////////////////////////////////////////
		// default enable magnifier
        
		[self setMagnifierEnable:PPCV_MAGNIFIER_DEFAULT_STATE];
        
        lineColor_                = [[CIColor alloc] initWithColor:[UIColor colorWithRed:PPCV_LINE_COLOR_R
                                                                                   green:PPCV_LINE_COLOR_G
                                                                                    blue:PPCV_LINE_COLOR_B
                                                                                   alpha:1]];
    }
    return self;
}


//===============================================================================
//
//===============================================================================
- (void)dealloc 
{
    [lineColor_ release];
    lineColor_ = nil;
    
    [currentTouchPoint_ release];
    currentTouchPoint_ = nil;
    
    [ppMagnifierView_ removeFromSuperview];
    [ppMagnifierView_ release];
    ppMagnifierView_ = nil;
    
    [touchConfirmTimer_ invalidate];
    [touchConfirmTimer_ release];
    touchConfirmTimer_ = nil;
    
    //////////////////////////////////////////////////
	// disable and release magnifier
    
	[self setMagnifierEnable:NO];
	
    //////////////////////////////////////////////////

    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Override

//===============================================================================
//
//===============================================================================
- (void)drawRect:(CGRect)rect 
{
	//------------------------------------------------------
	// begin draw
	//------------------------------------------------------
    
	CGContextRef context = UIGraphicsGetCurrentContext();	
	CGContextSaveGState(context);

	
	//------------------------------------------------------
	// Drawing line
	//------------------------------------------------------
	// 只要把角落的四個點連起來就好
    
	CGContextAddLines(context, _cropPointOfView, PPCV_SM_Count/2);
	CGContextAddLineToPoint(context, _cropPointOfView[0].x, _cropPointOfView[0].y);
	CGFloat fLineWidth = 2.0;
//    for (int i = 0; i < PPCV_SM_Count; i++)
//    {
//        NSLog(@"_cropPointOfView[%d]:%@", i, NSStringFromCGPoint(_cropPointOfView[i]));
//    }

	if (self.smallLineMode == YES)
    {
		fLineWidth/=2;
	}
    
	CGContextSetLineWidth(context, fLineWidth);
	CGContextSetRGBStrokeColor(context, self.lineColor.red, self.lineColor.green, self.lineColor.blue, 0.8);
	CGContextStrokePath(context);
	
    //------------------------------------------------------
	// Drawing select circle
    //------------------------------------------------------

    for(int i=PPCV_SM_Count; i>0; i--)
	{
        CGRect outlineRect = _markFrame[i-1];
        BOOL isSelected    = NO;
		if(self.currentSelectedMark == PPCV_SELECT_MARK_NONE && self.touchEnable==YES)
		{
			switch (i-1) {
				case PPCV_SM_Top:
                    
                    //////////////////////////////////////////////////
					// 上方的移動點
                    
					if ([self isCenterSelectMarkEnableWithStartPoint:_cropPointOfView[PPCV_SM_LeftTop] endPoint:_cropPointOfView[PPCV_SM_RightTop]]==NO)
					{
						CGContextSetRGBFillColor(context, self.lineColor.red, self.lineColor.green, self.lineColor.blue, 1);
						isSelected = YES;
					}
					break;
				case PPCV_SM_Left:
					if ([self isCenterSelectMarkEnableWithStartPoint:_cropPointOfView[PPCV_SM_LeftTop] endPoint:_cropPointOfView[PPCV_SM_LeftBottom]]==NO) 
					{
						CGContextSetRGBFillColor(context, self.lineColor.red, self.lineColor.green, self.lineColor.blue, 1);
						isSelected = YES;
					}
					break;
				case PPCV_SM_Right:
					if ([self isCenterSelectMarkEnableWithStartPoint:_cropPointOfView[PPCV_SM_RightTop] endPoint:_cropPointOfView[PPCV_SM_RightBottom]]==NO) 
					{
						CGContextSetRGBFillColor(context, self.lineColor.red, self.lineColor.green, self.lineColor.blue, 1);
						isSelected = YES;
					}
					break;
				case PPCV_SM_Bottom:
					if ([self isCenterSelectMarkEnableWithStartPoint:_cropPointOfView[PPCV_SM_LeftBottom] endPoint:_cropPointOfView[PPCV_SM_RightBottom]]==NO) 
					{
						CGContextSetRGBFillColor(context, self.lineColor.red, self.lineColor.green, self.lineColor.blue, 1);
						isSelected = YES;
					}
					break;
				default:
                    
                    //////////////////////////////////////////////////
					//沒有選取時畫大的圈
                    
					isSelected = NO;
					break;
			}
			[self drawSelectRect:outlineRect onContext:context isSelected:isSelected];
		}
		else 
		{
            //////////////////////////////////////////////////
			// 選取後畫小點
            
			if (self.currentSelectedMark !=(i-1)) 
				CGContextSetRGBFillColor(context, self.lineColor.red, self.lineColor.green, self.lineColor.blue, 1);
			else 
				CGContextSetRGBFillColor(context, PPCV_SEL_POINT_COLOR_R, PPCV_SEL_POINT_COLOR_G, PPCV_SEL_POINT_COLOR_B, 1);
			
			[self drawSelectRect:_markFrame[i-1] onContext:context isSelected:YES];	
		}
	}
	
    
    //------------------------------------------------------
	// end draw
    //------------------------------------------------------

    CGContextRestoreGState(context);
	UIGraphicsEndImageContext();
	
    //////////////////////////////////////////////////
	//放大鏡在這邊同步更新
    
	if(self.magnifierEnable && self.ppMagnifierView != nil)
    [self.ppMagnifierView setNeedsDisplay];
}


//================================================================================
//
//================================================================================
- (UIView*)hitTest:(CGPoint)point withEvent:(UIEvent *)event
{
	if (self.touchEnable == YES) {

		for (int i=0; i < PPCV_SM_Count; i++)
        {
			if(CGRectContainsPoint(_markFrame[i], point)==YES)
			{
				return self;
			}
		}
		
		if (self.moveCropViewEnable==YES&&[self isPointInCropView:point])
        {
			return self;
		}
	}
	return nil;
}


//================================================================================
//
//================================================================================
- (void)adjustCropViewFrame
{
    // 'calculateCropPointsWithNewFrame'需要用到舊的大小，所以要做完再set新的frame
    
    //        NSLog(@"self.superview.bounds: %@", NSStringFromCGRect(self.superview.bounds));
    if (self.superview!=nil)
    {
        CGRect newFrame = CGRectInset(self.superview.bounds, -1*cropPointRadius_, -1*cropPointRadius_);
        
        if (CGRectEqualToRect(newFrame, self.frame)==NO)
        {
            [self calculateCropPointsWithNewFrame:newFrame];
            
            self.frame =newFrame;
            
            //////////////////////////////////////////////////
            // 計算預設的crop view (與imageView一樣大)
            
            [self setInitialImageFrame:self.superview.bounds];
        }
    }
}


//================================================================================
//
//================================================================================
- (void)layoutSubviews
{
	[super layoutSubviews];
    
    //////////////////////////////////////////////////
	//重算view的大小
    [self adjustCropViewFrame];
}


//===============================================================================
// overwriting to recalculate crop point when frame is changed
//===============================================================================
- (void)setFrame:(CGRect)frame
{
    [super setFrame:frame];
    
    self.maxSize = CGSizeMake(frame.size.width, frame.size.height);
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - instance methods

//===============================================================================
// return NO if not crop
//===============================================================================
- (BOOL)getCropPointArray:(CGPoint*)cropPoints
{
    if (cropPoints == nil) {
        return NO;
    }
    BOOL bIsEqual = YES;
    
    //////////////////////////////////////////////////
    // 只要回傳四個點就好
    
    for(int i=0; i<PPCV_SM_Count/2; i++)
    {
        cropPoints[i].x = _cropPointOfView[i].x;
        cropPoints[i].y = _cropPointOfView[i].y;
        
        if(!CGPointEqualToPoint(_cropPointOfView[i], _imagePointOfView[i]))
        {
            if(bIsEqual == YES)	bIsEqual = NO;
        }
        
        //////////////////////////////////////////////////
        // 轉換為image座標系
        
        cropPoints[i] = [self convertPoint:cropPoints[i] toView:self.superview];
    }
    
    //////////////////////////////////////////////////
    // check if equal to frame size
    
    if(bIsEqual)
    {
        return NO;			
    }
    
    return YES;
}


//================================================================================
//
//================================================================================
- (CGRect)minimizeRect
{
    CGRect minRect = CGRectMake(MIN(MIN(_cropPointOfView[PPCV_SM_LeftTop].x, _cropPointOfView[PPCV_SM_LeftBottom].x),MIN(_cropPointOfView[PPCV_SM_RightTop].x, _cropPointOfView[PPCV_SM_RightBottom].x)),
                                MIN(MIN(_cropPointOfView[PPCV_SM_LeftTop].y, _cropPointOfView[PPCV_SM_LeftBottom].y),MIN(_cropPointOfView[PPCV_SM_RightTop].y, _cropPointOfView[PPCV_SM_RightBottom].y)),
                                MAX(MAX(_cropPointOfView[PPCV_SM_LeftTop].x, _cropPointOfView[PPCV_SM_LeftBottom].x),MAX(_cropPointOfView[PPCV_SM_RightTop].x, _cropPointOfView[PPCV_SM_RightBottom].x))-MIN(MIN(_cropPointOfView[PPCV_SM_LeftTop].x, _cropPointOfView[PPCV_SM_LeftBottom].x),MIN(_cropPointOfView[PPCV_SM_RightTop].x, _cropPointOfView[PPCV_SM_RightBottom].x)),
                                MAX(MAX(_cropPointOfView[PPCV_SM_LeftTop].y, _cropPointOfView[PPCV_SM_LeftBottom].y),MAX(_cropPointOfView[PPCV_SM_RightTop].y, _cropPointOfView[PPCV_SM_RightBottom].y))-MIN(MIN(_cropPointOfView[PPCV_SM_LeftTop].y, _cropPointOfView[PPCV_SM_LeftBottom].y),MIN(_cropPointOfView[PPCV_SM_RightTop].y, _cropPointOfView[PPCV_SM_RightBottom].y)));
    return [self convertRect:minRect toView:self.superview];;
}


//================================================================================
//
//================================================================================
- (void)resetInitImageFrame
{
	self.imageFrame = CGRectZero;
}


//===============================================================================
// return NO if set array failed
//===============================================================================
- (BOOL)setCropPointArray:(CGPoint*)cropPoints
{
    if (cropPoints == nil)
    {
        return NO;
    }
    
    //////////////////////////////////////////////////
    // 傳進來的只有四個點，上下左右的點要再算出來補上
    
    for(int i=0; i<PPCV_SM_Count; i++)
    {
        CGPoint curPoint;
        if (i<PPCV_SM_Count/2)
        {
            //////////////////////////////////////////////////
            // 轉換為cropView座標系
            
            curPoint = [self convertPoint:cropPoints[i] fromView:self.superview];
            
            [self getValidPoint:&curPoint withTouchIndex:self.currentSelectedMark];
            
            [self setCropNode:i toPoint:curPoint];
        }
        else
        {
            switch (i)
            {
                case PPCV_SM_Top:
                {
                    _cropPointOfView[PPCV_SM_Top]    = CGPointMake((_cropPointOfView[PPCV_SM_LeftTop].x+_cropPointOfView[PPCV_SM_RightTop].x)/2, (_cropPointOfView[PPCV_SM_LeftTop].y+_cropPointOfView[PPCV_SM_RightTop].y)/2);
                    break;
                }
                case PPCV_SM_Left:
                {
                    _cropPointOfView[PPCV_SM_Left]   = CGPointMake((_cropPointOfView[PPCV_SM_LeftTop].x+_cropPointOfView[PPCV_SM_LeftBottom].x)/2, (_cropPointOfView[PPCV_SM_LeftTop].y+_cropPointOfView[PPCV_SM_LeftBottom].y)/2);
                    break;
                }
                case PPCV_SM_Right:
                {
                    _cropPointOfView[PPCV_SM_Right]  = CGPointMake((_cropPointOfView[PPCV_SM_RightTop].x+_cropPointOfView[PPCV_SM_RightBottom].x)/2, (_cropPointOfView[PPCV_SM_RightTop].y+_cropPointOfView[PPCV_SM_RightBottom].y)/2);
                    break;
                }
                case PPCV_SM_Bottom:
                {
                    _cropPointOfView[PPCV_SM_Bottom] = CGPointMake((_cropPointOfView[PPCV_SM_LeftBottom].x+_cropPointOfView[PPCV_SM_RightBottom].x)/2, (_cropPointOfView[PPCV_SM_LeftBottom].y+_cropPointOfView[PPCV_SM_RightBottom].y)/2);
                    break;
                }
                default:
                    break;
            }
        }
        
    }
    
    [self setSelectMarkFrame];
    
    return YES;
}


//================================================================================
//
//================================================================================
- (void)setInitialImageFrame:(CGRect)theImageFrame
{
    //////////////////////////////////////////////////
    // 轉換為cropView座標系
    
    theImageFrame.origin = [self convertPoint:theImageFrame.origin fromView:self.superview];
    
    //////////////////////////////////////////////////
    // 傳進來的只要四個點，上下左右的點要再算出來補上
    
    _imagePointOfView[PPCV_SM_LeftTop].x   = theImageFrame.origin.x;
    _imagePointOfView[PPCV_SM_LeftTop].y   = theImageFrame.origin.y;
    _imagePointOfView[PPCV_SM_RightTop]    = CGPointMake(theImageFrame.origin.x+theImageFrame.size.width, theImageFrame.origin.y);
    _imagePointOfView[PPCV_SM_RightBottom] = CGPointMake(theImageFrame.origin.x+theImageFrame.size.width, theImageFrame.origin.y+theImageFrame.size.height);
    _imagePointOfView[PPCV_SM_LeftBottom]  = CGPointMake(theImageFrame.origin.x, theImageFrame.origin.y+theImageFrame.size.height);

    for(int i=0; i<PPCV_SM_Count; i++)
    {
        if (i<PPCV_SM_Count/2)
        {
            _cropPointOfView[i].x = _imagePointOfView[i].x;
            _cropPointOfView[i].y = _imagePointOfView[i].y;
        }
        else
        {
            switch (i)
            {
                case PPCV_SM_Top:
                {
                    _cropPointOfView[PPCV_SM_Top]    = CGPointMake((_imagePointOfView[PPCV_SM_LeftTop].x+_imagePointOfView[PPCV_SM_RightTop].x)/2, (_imagePointOfView[PPCV_SM_LeftTop].y+_imagePointOfView[PPCV_SM_RightTop].y)/2);
                    break;
                }
                case PPCV_SM_Left:
                {
                    _cropPointOfView[PPCV_SM_Left]   = CGPointMake((_imagePointOfView[PPCV_SM_LeftTop].x+_imagePointOfView[PPCV_SM_LeftBottom].x)/2, (_imagePointOfView[PPCV_SM_LeftTop].y+_imagePointOfView[PPCV_SM_LeftBottom].y)/2);
                    break;
                }
                case PPCV_SM_Right:
                {
                    _cropPointOfView[PPCV_SM_Right]  = CGPointMake((_imagePointOfView[PPCV_SM_RightTop].x+_imagePointOfView[PPCV_SM_RightBottom].x)/2, (_imagePointOfView[PPCV_SM_RightTop].y+_imagePointOfView[PPCV_SM_RightBottom].y)/2);
                    break;
                }
                case PPCV_SM_Bottom:
                {
                    _cropPointOfView[PPCV_SM_Bottom] = CGPointMake((_imagePointOfView[PPCV_SM_LeftBottom].x+_imagePointOfView[PPCV_SM_RightBottom].x)/2, (_imagePointOfView[PPCV_SM_LeftBottom].y+_imagePointOfView[PPCV_SM_RightBottom].y)/2);
                    break;
                }
                default:
                    break;
            }
        }
    }
    
    //////////////////////////////////////////////////
    // 計算目前的select mark的frame
    
    [self setSelectMarkFrame];
    
    self.imageFrame = theImageFrame;
    
    [self setNeedsDisplay];
}


//================================================================================
//
//================================================================================
- (void)setLineColor:(CIColor*)lineColor
{
    [lineColor retain];
    [lineColor_ release];
    lineColor_ = lineColor;
}


//================================================================================
//
//================================================================================
- (void)setMagnifierEnable:(BOOL)enableMagnifier
{
    //////////////////////////////////////////////////
	// 這邊如果用'self.magnifierEnable'會變成無窮迴圈，所以改用'magnifierEnable_'
    
	magnifierEnable_ = enableMagnifier;
	
	if (self.magnifierEnable==YES)
    {
		if (self.ppMagnifierView==nil)
        {
            self.ppMagnifierView        = [[[PPMagnifierView alloc] initWithFrame:CGRectMake(0, 0, PPCV_MAGNIFIER_RADIUS*2, PPCV_MAGNIFIER_RADIUS*2)] autorelease];
            self.ppMagnifierView.offset = PPCV_MAGNIFIER_DEFAULT_SHIFT;
		}
	}
	else
    {
		if (self.ppMagnifierView!=nil)
        {
            self.ppMagnifierView.targetView = nil;
            self.ppMagnifierView            = nil;
		}
	}

}


//================================================================================
//
//================================================================================
- (void)setSmallLineMode:(BOOL)smallLineMode
{
    smallLineMode_ = smallLineMode;
    
    [self setSelectMarkFrame];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - private method implementation

//================================================================================
//
//================================================================================
- (void)calculateCropPointsWithNewFrame:(CGRect)newFrame
{
    CGAffineTransform t = CGAffineTransformMakeScale(newFrame.size.width/self.frame.size.width, newFrame.size.height/self.frame.size.height);
    
    for (int i=0 ; i < PPCV_SM_Count; i++)
    {
        
        //////////////////////////////////////////////////
        // '_imagePointOfView' 只有'PPCV_SM_Count'個
        
        if (i<PPCV_SM_Count/2)
        {
            _imagePointOfView[i] = CGPointApplyAffineTransform(_imagePointOfView[i], t);
        }
        
        _cropPointOfView[i] = CGPointApplyAffineTransform(_cropPointOfView[i], t);
    }
    
    [self setSelectMarkFrame];
    
    [self setNeedsDisplay];
}


//================================================================================
//
//================================================================================
- (void)drawSelectRect:(CGRect)markFrame onContext:(CGContextRef)context isSelected:(BOOL)selected
{
    CGRect outlineRect;
    if (selected==NO)
    {
        CGContextSetRGBFillColor(context, self.lineColor.red, self.lineColor.green, self.lineColor.blue, 0.3);
        CGContextFillEllipseInRect(context, markFrame);
        CGContextSetRGBFillColor(context, 1, 1, 1, 0.6);
        outlineRect = CGRectInset(markFrame, 1, 1);
        CGContextFillEllipseInRect(context, outlineRect);
    }
    else
    {
        if (self.smallLineMode == YES)
        {
            //////////////////////////////////////////////////
            //_markFrame變小一半了，所以要縮的值也要小一半
            
            outlineRect = CGRectInset(markFrame, cropPointRadius_/2-1, cropPointRadius_/2-1);
        }
        else
        {
            outlineRect = CGRectInset(markFrame, cropPointRadius_-2, cropPointRadius_-2);
        }
        CGContextFillEllipseInRect(context, outlineRect);
    }
}


//================================================================================
// 修正touch的座標
//================================================================================
- (BOOL)getValidPoint:(CGPoint*)curPoint withTouchIndex:(NSInteger)touchIndex
{
	if (curPoint == NULL) {
		return NO;
	}
	
    //////////////////////////////////////////////////
	// 不會比image大
    
	if(curPoint->x < self.imageFrame.origin.x)
		curPoint->x = self.imageFrame.origin.x;
	else if(curPoint->x > self.imageFrame.origin.x + self.imageFrame.size.width)
		curPoint->x = self.imageFrame.origin.x + self.imageFrame.size.width;
	
	if(curPoint->y < self.imageFrame.origin.y)
		curPoint->y = self.imageFrame.origin.y;
	else if(curPoint->y > self.imageFrame.origin.y + self.imageFrame.size.height)
		curPoint->y = self.imageFrame.origin.y + self.imageFrame.size.height;
	
    //////////////////////////////////////////////////
	// 特殊處理
    
	switch (touchIndex) 
	{
		case PPCV_SM_LeftTop:
		{
            //////////////////////////////////////////////////
			// 判斷最小值
            
			if(_cropPointOfView[PPCV_SM_RightTop].x-curPoint->x < self.minSize.width)
				curPoint->x = _cropPointOfView[PPCV_SM_RightTop].x-self.minSize.width;
						
			if(_cropPointOfView[PPCV_SM_LeftBottom].y-curPoint->y < self.minSize.height)
				curPoint->y = _cropPointOfView[PPCV_SM_LeftBottom].y-self.minSize.height;
			
            //////////////////////////////////////////////////
            //跟右下也要比
            
			if(_cropPointOfView[PPCV_SM_RightBottom].x-curPoint->x < self.minSize.width)
				curPoint->x = _cropPointOfView[PPCV_SM_RightBottom].x-self.minSize.width;	
			if(_cropPointOfView[PPCV_SM_RightBottom].y-curPoint->y < self.minSize.height)
				curPoint->y = _cropPointOfView[PPCV_SM_RightBottom].y-self.minSize.height;
			
            //////////////////////////////////////////////////
			// 判斷最大值
            
			if(_cropPointOfView[PPCV_SM_RightTop].x-curPoint->x > self.maxSize.width)
				curPoint->x = _cropPointOfView[PPCV_SM_RightTop].x-self.maxSize.width;
			
			if(_cropPointOfView[PPCV_SM_LeftBottom].y-curPoint->y > self.maxSize.height)
				curPoint->y = _cropPointOfView[PPCV_SM_LeftBottom].y-self.maxSize.height;
			
            //////////////////////////////////////////////////
			//跟右下也要比
            
			if(_cropPointOfView[PPCV_SM_RightBottom].x-curPoint->x > self.maxSize.width)
				curPoint->x = _cropPointOfView[PPCV_SM_RightBottom].x-self.maxSize.width;	
			if(_cropPointOfView[PPCV_SM_RightBottom].y-curPoint->y > self.maxSize.height)
				curPoint->y = _cropPointOfView[PPCV_SM_RightBottom].y-self.maxSize.height;
			
			break;
		}
			
		case PPCV_SM_RightTop:
		{
            //////////////////////////////////////////////////
			// 判斷最小值
            
			if(curPoint->x - _cropPointOfView[PPCV_SM_LeftTop].x < self.minSize.width)
				curPoint->x = _cropPointOfView[PPCV_SM_LeftTop].x + self.minSize.width;
			
			if(_cropPointOfView[PPCV_SM_RightBottom].y-curPoint->y < self.minSize.height)
				curPoint->y = _cropPointOfView[PPCV_SM_RightBottom].y-self.minSize.height;
			
            //////////////////////////////////////////////////
			//跟左下也要比
            
			if(curPoint->x -_cropPointOfView[PPCV_SM_LeftBottom].x < self.minSize.width)
				curPoint->x = _cropPointOfView[PPCV_SM_LeftBottom].x+self.minSize.width;	
			if(_cropPointOfView[PPCV_SM_LeftBottom].y-curPoint->y < self.minSize.height)
				curPoint->y = _cropPointOfView[PPCV_SM_LeftBottom].y-self.minSize.height;

            //////////////////////////////////////////////////
			// 判斷最大值
            
			if(curPoint->x - _cropPointOfView[PPCV_SM_LeftTop].x > self.maxSize.width)
				curPoint->x = _cropPointOfView[PPCV_SM_LeftTop].x + self.maxSize.width;
			
			if(_cropPointOfView[PPCV_SM_RightBottom].y-curPoint->y > self.maxSize.height)
				curPoint->y = _cropPointOfView[PPCV_SM_RightBottom].y-self.maxSize.height;
			
            //////////////////////////////////////////////////
			//跟左下也要比
            
			if(curPoint->x -_cropPointOfView[PPCV_SM_LeftBottom].x > self.maxSize.width)
				curPoint->x = _cropPointOfView[PPCV_SM_LeftBottom].x+self.maxSize.width;	
			if(_cropPointOfView[PPCV_SM_LeftBottom].y-curPoint->y > self.maxSize.height)
				curPoint->y = _cropPointOfView[PPCV_SM_LeftBottom].y-self.maxSize.height;
			break;
		}
			
		case PPCV_SM_RightBottom:
		{
            //////////////////////////////////////////////////
			// 判斷最小值
            
			if(curPoint->x - _cropPointOfView[PPCV_SM_LeftBottom].x < self.minSize.width)
				curPoint->x = _cropPointOfView[PPCV_SM_LeftBottom].x + self.minSize.width;
			
			if(curPoint->y - _cropPointOfView[PPCV_SM_RightTop].y < self.minSize.height)
				curPoint->y = _cropPointOfView[PPCV_SM_RightTop].y + self.minSize.height;
            
            //////////////////////////////////////////////////
			//跟左上也要比
            
			if(curPoint->x - _cropPointOfView[PPCV_SM_LeftTop].x < self.minSize.width)
				curPoint->x = _cropPointOfView[PPCV_SM_LeftTop].x+self.minSize.width;	
			if(curPoint->y - _cropPointOfView[PPCV_SM_LeftTop].y< self.minSize.height)
				curPoint->y = _cropPointOfView[PPCV_SM_LeftTop].y+self.minSize.height;
			
            //////////////////////////////////////////////////
			// 判斷最大值
            
			if(curPoint->x - _cropPointOfView[PPCV_SM_LeftBottom].x > self.maxSize.width)
				curPoint->x = _cropPointOfView[PPCV_SM_LeftBottom].x + self.maxSize.width;
			
			if(curPoint->y - _cropPointOfView[PPCV_SM_RightTop].y > self.maxSize.height)
				curPoint->y = _cropPointOfView[PPCV_SM_RightTop].y + self.maxSize.height;
            
            //////////////////////////////////////////////////
			//跟左上也要比
            
			if(curPoint->x - _cropPointOfView[PPCV_SM_LeftTop].x > self.maxSize.width)
				curPoint->x = _cropPointOfView[PPCV_SM_LeftTop].x+self.maxSize.width;	
			if(curPoint->y - _cropPointOfView[PPCV_SM_LeftTop].y > self.maxSize.height)
				curPoint->y = _cropPointOfView[PPCV_SM_LeftTop].y+self.maxSize.height;
			break;
		}
			
		case PPCV_SM_LeftBottom:
		{
            //////////////////////////////////////////////////
			// 判斷最小值
            
			if(_cropPointOfView[PPCV_SM_RightBottom].x-curPoint->x < self.minSize.width)
				curPoint->x = _cropPointOfView[PPCV_SM_RightBottom].x-self.minSize.width;
			
			if(curPoint->y - _cropPointOfView[PPCV_SM_LeftTop].y < self.minSize.height)
				curPoint->y = _cropPointOfView[PPCV_SM_LeftTop].y + self.minSize.height;
			
            //////////////////////////////////////////////////
			//跟右上也要比
            
			if(_cropPointOfView[PPCV_SM_RightTop].x-curPoint->x < self.minSize.width)
				curPoint->x = _cropPointOfView[PPCV_SM_RightTop].x-self.minSize.width;	
			if(curPoint->y - _cropPointOfView[PPCV_SM_RightTop].y < self.minSize.height)
				curPoint->y = _cropPointOfView[PPCV_SM_RightTop].y+self.minSize.height;
			
            //////////////////////////////////////////////////
			// 判斷最大值
            
			if(_cropPointOfView[PPCV_SM_RightBottom].x-curPoint->x > self.maxSize.width)
				curPoint->x = _cropPointOfView[PPCV_SM_RightBottom].x-self.maxSize.width;
			
			if(curPoint->y - _cropPointOfView[PPCV_SM_LeftTop].y > self.maxSize.height)
				curPoint->y = _cropPointOfView[PPCV_SM_LeftTop].y + self.maxSize.height;
			
            //////////////////////////////////////////////////
			//跟右上也要比
            
			if(_cropPointOfView[PPCV_SM_RightTop].x-curPoint->x > self.maxSize.width)
				curPoint->x = _cropPointOfView[PPCV_SM_RightTop].x-self.maxSize.width;	
			if(curPoint->y - _cropPointOfView[PPCV_SM_RightTop].y > self.maxSize.height)
				curPoint->y = _cropPointOfView[PPCV_SM_RightTop].y+self.maxSize.height;
			break;
		}	
		case PPCV_SM_Top:
		{
            //////////////////////////////////////////////////
			//左上，右上都不能比邊界小，也不能比左下，右下大
            
            CGFloat shiftY     = curPoint->y-_cropPointOfView[PPCV_SM_Top].y;
            CGFloat minTopY    = MIN(_cropPointOfView[PPCV_SM_LeftTop].y,_cropPointOfView[PPCV_SM_RightTop].y);
            CGFloat maxTopY    = MAX(_cropPointOfView[PPCV_SM_LeftTop].y,_cropPointOfView[PPCV_SM_RightTop].y);
            CGFloat minBottomY = MIN(_cropPointOfView[PPCV_SM_LeftBottom].y,_cropPointOfView[PPCV_SM_RightBottom].y);
            CGFloat maxBottomY = MAX(_cropPointOfView[PPCV_SM_LeftBottom].y,_cropPointOfView[PPCV_SM_RightBottom].y);
		
            //////////////////////////////////////////////////
			// 上邊界
			//最大的上邊界
            
			CGFloat maxMinTopY = MAX(self.imageFrame.origin.y, maxBottomY-self.maxSize.height);
			if(minTopY+shiftY < maxMinTopY)
				curPoint->y =_cropPointOfView[PPCV_SM_Top].y +(maxMinTopY -minTopY);
            
            //////////////////////////////////////////////////
			// 下邊界
            
			if(maxTopY+shiftY >= minBottomY-self.minSize.height)
				curPoint->y = _cropPointOfView[PPCV_SM_Top].y + (minBottomY-self.minSize.height-maxTopY);
			break;
		}
		case PPCV_SM_Left:
		{
            CGFloat shiftX    = curPoint->x-_cropPointOfView[PPCV_SM_Left].x;
            CGFloat maxLeftX  = MAX(_cropPointOfView[PPCV_SM_LeftTop].x,_cropPointOfView[PPCV_SM_LeftBottom].x);
            CGFloat minLeftX  = MIN(_cropPointOfView[PPCV_SM_LeftTop].x,_cropPointOfView[PPCV_SM_LeftBottom].x);
            CGFloat minRightX = MIN(_cropPointOfView[PPCV_SM_RightTop].x,_cropPointOfView[PPCV_SM_RightBottom].x);
            CGFloat maxRightX = MAX(_cropPointOfView[PPCV_SM_RightTop].x,_cropPointOfView[PPCV_SM_RightBottom].x);
			
            //////////////////////////////////////////////////
			// 左邊界
            
			CGFloat maxMinLeftX = MAX(self.imageFrame.origin.x, maxRightX-self.maxSize.width);
			if(minLeftX+shiftX < maxMinLeftX)
				curPoint->x =_cropPointOfView[PPCV_SM_Left].x +(maxMinLeftX-minLeftX);
            
            //////////////////////////////////////////////////
			// 右邊界
            
			if(maxLeftX+shiftX >= minRightX-self.minSize.width)
				curPoint->x = _cropPointOfView[PPCV_SM_Left].x + (minRightX-self.minSize.width-maxLeftX);
			break;
		}
		case PPCV_SM_Right:
		{
            CGFloat shiftX    = curPoint->x-_cropPointOfView[PPCV_SM_Right].x;
            CGFloat maxRightX = MAX(_cropPointOfView[PPCV_SM_RightTop].x,_cropPointOfView[PPCV_SM_RightBottom].x);
            CGFloat minRightX = MIN(_cropPointOfView[PPCV_SM_RightTop].x,_cropPointOfView[PPCV_SM_RightBottom].x);
            CGFloat maxLeftX  = MAX(_cropPointOfView[PPCV_SM_LeftTop].x,_cropPointOfView[PPCV_SM_LeftBottom].x);
            CGFloat minLeftX  = MIN(_cropPointOfView[PPCV_SM_LeftTop].x,_cropPointOfView[PPCV_SM_LeftBottom].x);
			
            //////////////////////////////////////////////////
			// 左邊界
			if(minRightX+shiftX < maxLeftX+self.minSize.width)
				curPoint->x =_cropPointOfView[PPCV_SM_Right].x +(maxLeftX+self.minSize.width-minRightX);
            
            //////////////////////////////////////////////////
			// 右邊界
			// 最小的右邊界最大值
            
			CGFloat minMaxRightX = MIN(self.imageFrame.origin.x+self.imageFrame.size.width, minLeftX+self.maxSize.width);
			if(maxRightX+shiftX >= minMaxRightX)
				curPoint->x = _cropPointOfView[PPCV_SM_Right].x + (minMaxRightX-maxRightX);
			break;
		}
		case PPCV_SM_Bottom:
		{
            //////////////////////////////////////////////////
			//左下，右下都不能比邊界大，也不能比左上，右上小
            
            CGFloat shiftY     = curPoint->y-_cropPointOfView[PPCV_SM_Bottom].y;
            CGFloat maxBottomY = MAX(_cropPointOfView[PPCV_SM_LeftBottom].y,_cropPointOfView[PPCV_SM_RightBottom].y);
            CGFloat minBottomY = MIN(_cropPointOfView[PPCV_SM_LeftBottom].y,_cropPointOfView[PPCV_SM_RightBottom].y);
            CGFloat maxTopY    = MAX(_cropPointOfView[PPCV_SM_LeftTop].y,_cropPointOfView[PPCV_SM_RightTop].y);
            CGFloat minTopY    = MIN(_cropPointOfView[PPCV_SM_LeftTop].y,_cropPointOfView[PPCV_SM_RightTop].y);
			
            //////////////////////////////////////////////////
			// 上邊界
            
			if(minBottomY+shiftY < maxTopY+self.minSize.height)
				curPoint->y =_cropPointOfView[PPCV_SM_Bottom].y +(maxTopY+self.minSize.height-minBottomY);
            
            //////////////////////////////////////////////////
            // 下邊界
			// 最小的下邊界
            
			CGFloat minMaxBottomY = MIN(self.imageFrame.origin.y+self.imageFrame.size.height, minTopY+self.maxSize.height);
			if(maxBottomY+shiftY >= minMaxBottomY)
				curPoint->y = _cropPointOfView[PPCV_SM_Bottom].y + (minMaxBottomY-maxBottomY);
			break;
		}	
		case PPCV_SELECT_MARK_MOVE:
		{
			CGFloat shiftX = curPoint->x-self.moveStartPoint.x;
			CGFloat shiftY = curPoint->y-self.moveStartPoint.y;
			
            //////////////////////////////////////////////////
			// 移動後的crop不能超過可移動的範圍
			// 上邊界
            
			NSInteger minPointTop = MIN(_cropPointOfView[PPCV_SM_LeftTop].y,_cropPointOfView[PPCV_SM_RightTop].y);
			if ((minPointTop+shiftY<self.imageFrame.origin.y)) 
			{
				curPoint->y = self.moveStartPoint.y+(self.imageFrame.origin.y-minPointTop);
			}
            
            //////////////////////////////////////////////////
			// 下邊界
            
			NSInteger maxPointBottom = MAX(_cropPointOfView[PPCV_SM_LeftBottom].y,_cropPointOfView[PPCV_SM_RightBottom].y);
			if ((maxPointBottom+shiftY>self.imageFrame.origin.y+self.imageFrame.size.height)) 
			{
				curPoint->y = self.moveStartPoint.y+(self.imageFrame.origin.y+self.imageFrame.size.height-maxPointBottom);
			}

            //////////////////////////////////////////////////
			// 左邊界
            
			NSInteger minPointLeft = MIN(_cropPointOfView[PPCV_SM_LeftTop].x,_cropPointOfView[PPCV_SM_LeftBottom].x);
			if ((minPointLeft+shiftX<self.imageFrame.origin.x)) 
			{
				curPoint->x = self.moveStartPoint.x+(self.imageFrame.origin.x-minPointLeft);
			}
			
            //////////////////////////////////////////////////
			// 右邊界
            
			NSInteger maxPointRight = MAX(_cropPointOfView[PPCV_SM_RightTop].x,_cropPointOfView[PPCV_SM_RightBottom].x);
			if ((maxPointRight+shiftX>self.imageFrame.origin.x+self.imageFrame.size.width))
			{
				curPoint->x = self.moveStartPoint.x+(self.imageFrame.origin.x+self.imageFrame.size.width-maxPointRight);
			}

			break;
		}
		default:
			break;
	}
	return YES;
}


//================================================================================
//
//================================================================================
- (BOOL)isCenterSelectMarkEnableWithStartPoint:(CGPoint)startPoint endPoint:(CGPoint)endPoint
{
    NSInteger minLength = cropPointRadius_*4;
    
    if (self.smallLineMode == YES)
    {
        minLength = cropPointRadius_*2;
    }
    
    if (dist((startPoint.x-endPoint.x),(startPoint.y-endPoint.y))>=minLength)
    {
        return YES;
    }
    else
    {
        return NO;
    }
    
}


//================================================================================
//
//================================================================================
- (BOOL)isPointInCropView:(CGPoint)point
{
    bool isPointInCropView    = false;
    CGMutablePathRef cropPath = CGPathCreateMutable();
    
    //////////////////////////////////////////////////
    //建立path
    
    CGPathMoveToPoint (cropPath, NULL, _cropPointOfView[0].x, _cropPointOfView[0].y);
    
    for (int i=1; i <PPCV_SM_Count/2; i++)
    {
        CGPathAddLineToPoint (cropPath, NULL, _cropPointOfView[i].x, _cropPointOfView[i].y);
    }
    
    isPointInCropView = CGPathContainsPoint(cropPath, NULL, point, false);
    CGPathRelease(cropPath);
    
    if (isPointInCropView==true)
    {
        return YES;
    }
    else
    {
        return NO;
    }
    
}


//================================================================================
// 移動crop框到point，以'curPoint'到'self.currentTouchPoint'的offset做移動
//================================================================================
- (void)moveCropToPoint:(CGPoint)curPoint
{
	CGFloat shiftX = curPoint.x-self.moveStartPoint.x;
	CGFloat shiftY = curPoint.y-self.moveStartPoint.y;

	for (int i=0; i<PPCV_SM_Count; i++)
    {
		_cropPointOfView[i].x += shiftX;
		_cropPointOfView[i].y += shiftY;
	}
    
	self.moveStartPoint = curPoint;
}


//================================================================================
// 處理點加入時，連動的部份
//================================================================================
- (void)setCropNode:(NSInteger)cropNode toPoint:(CGPoint)curPoint
{
    //////////////////////////////////////////////////
	// 計算offset量
    
	CGFloat shiftX = curPoint.x - _cropPointOfView[cropNode].x;
	CGFloat shiftY = curPoint.y - _cropPointOfView[cropNode].y;
	
	if (cropNode<PPCV_SM_Count/2)
    {
		_cropPointOfView[cropNode] = curPoint;		
	}
	
	if (self.rectCropMode == NO)
	{
        //////////////////////////////////////////////////
		// 同時要修正的點
        
		switch (cropNode)
        {
            //////////////////////////////////////////////////
            // 上下左右要修正的點
                
			case PPCV_SM_Top:
            {
                _cropPointOfView[cropNode].y         += shiftY;
                _cropPointOfView[PPCV_SM_RightTop].y += shiftY;
                _cropPointOfView[PPCV_SM_LeftTop].y  += shiftY;
                break;
            }
			case PPCV_SM_Left:
            {
                _cropPointOfView[cropNode].x           += shiftX;
                _cropPointOfView[PPCV_SM_LeftBottom].x += shiftX;
                _cropPointOfView[PPCV_SM_LeftTop].x    += shiftX;
                break;
            }
			case PPCV_SM_Right:
            {
                _cropPointOfView[cropNode].x            += shiftX;
                _cropPointOfView[PPCV_SM_RightTop].x    += shiftX;
                _cropPointOfView[PPCV_SM_RightBottom].x += shiftX;
                break;
            }
			case PPCV_SM_Bottom:
            {
                _cropPointOfView[cropNode].y            += shiftY;
                _cropPointOfView[PPCV_SM_RightBottom].y += shiftY;
                _cropPointOfView[PPCV_SM_LeftBottom].y  += shiftY;
                break;
            }
			default:
				break;
				
		}	
	}
	else
	{
        //////////////////////////////////////////////////
		// 同時要修正的點
        
		switch (cropNode) 
		{
			case PPCV_SM_LeftTop:
            {
                _cropPointOfView[PPCV_SM_RightTop].y   = curPoint.y;
                _cropPointOfView[PPCV_SM_LeftBottom].x = curPoint.x;
                break;
            }
			case PPCV_SM_RightTop:
            {
                _cropPointOfView[PPCV_SM_LeftTop].y     = curPoint.y;
                _cropPointOfView[PPCV_SM_RightBottom].x = curPoint.x;
                break;
            }
			case PPCV_SM_RightBottom:
            {
                _cropPointOfView[PPCV_SM_LeftBottom].y = curPoint.y;
                _cropPointOfView[PPCV_SM_RightTop].x   = curPoint.x;
                break;
            }
			case PPCV_SM_LeftBottom:
            {
                _cropPointOfView[PPCV_SM_RightBottom].y = curPoint.y;
                _cropPointOfView[PPCV_SM_LeftTop].x     = curPoint.x;
                break;
            }
                
            //////////////////////////////////////////////////
            // 上下左右要修正的點
                
			case PPCV_SM_Top:
            {
                _cropPointOfView[cropNode].y         = curPoint.y;
                _cropPointOfView[PPCV_SM_RightTop].y = curPoint.y;
                _cropPointOfView[PPCV_SM_LeftTop].y  = curPoint.y;
                break;
            }
			case PPCV_SM_Left:
            {
                _cropPointOfView[cropNode].x           = curPoint.x;
                _cropPointOfView[PPCV_SM_LeftBottom].x = curPoint.x;
                _cropPointOfView[PPCV_SM_LeftTop].x    = curPoint.x;
                break;
            }
			case PPCV_SM_Right:
            {
                _cropPointOfView[cropNode].x            = curPoint.x;
                _cropPointOfView[PPCV_SM_RightTop].x    = curPoint.x;
                _cropPointOfView[PPCV_SM_RightBottom].x = curPoint.x;
                break;
            }
			case PPCV_SM_Bottom:
            {
                _cropPointOfView[cropNode].y            = curPoint.y;
                _cropPointOfView[PPCV_SM_RightBottom].y = curPoint.y;
                _cropPointOfView[PPCV_SM_LeftBottom].y  = curPoint.y;
                break;
            }
			default:
				break;
				
		}		
	}
    
    //////////////////////////////////////////////////
	// 重算中心點位置
    
    _cropPointOfView[PPCV_SM_Top]    = CGPointMake((_cropPointOfView[PPCV_SM_LeftTop].x+_cropPointOfView[PPCV_SM_RightTop].x)/2, (_cropPointOfView[PPCV_SM_LeftTop].y+_cropPointOfView[PPCV_SM_RightTop].y)/2);
    _cropPointOfView[PPCV_SM_Left]   = CGPointMake((_cropPointOfView[PPCV_SM_LeftTop].x+_cropPointOfView[PPCV_SM_LeftBottom].x)/2, (_cropPointOfView[PPCV_SM_LeftTop].y+_cropPointOfView[PPCV_SM_LeftBottom].y)/2);
    _cropPointOfView[PPCV_SM_Right]  = CGPointMake((_cropPointOfView[PPCV_SM_RightTop].x+_cropPointOfView[PPCV_SM_RightBottom].x)/2, (_cropPointOfView[PPCV_SM_RightTop].y+_cropPointOfView[PPCV_SM_RightBottom].y)/2);
    _cropPointOfView[PPCV_SM_Bottom] = CGPointMake((_cropPointOfView[PPCV_SM_LeftBottom].x+_cropPointOfView[PPCV_SM_RightBottom].x)/2, (_cropPointOfView[PPCV_SM_LeftBottom].y+_cropPointOfView[PPCV_SM_RightBottom].y)/2);
	
//    for (int i = 0; i < PPCV_SM_Count; i++)
//    {
//        NSLog(@"_cropPointOfView[%d]:%@", i, NSStringFromCGPoint(_cropPointOfView[i]));
//    }
}


//===============================================================================
// set select mark frame for drawing image
//===============================================================================
- (void)setSelectMarkFrame
{
	for(int i=0; i<PPCV_SM_Count; i++)
	{
        CGFloat fRadius = cropPointRadius_;

		if (self.smallLineMode == YES)
        {
        fRadius         = cropPointRadius_/2;
		}
        _markFrame[i].origin.x    = _cropPointOfView[i].x - fRadius;
        _markFrame[i].origin.y    = _cropPointOfView[i].y - fRadius;
        _markFrame[i].size.width  = fRadius*2;
        _markFrame[i].size.height = fRadius*2;
	}
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - touch confirm methods

//===============================================================================
// 
//===============================================================================
- (void)stopConfirmTimer
{
	if(self.touchConfirmTimer)
	{
		[self.touchConfirmTimer invalidate];
        
		self.touchConfirmTimer = nil;
	}	
}


//===============================================================================
// 
//===============================================================================
- (void)startConfirmTimer:(CGPoint)curPoint
{
    self.lastMovePoint     = curPoint;
//	[self stopConfirmTimer];
    __block typeof(self) blockSelf = self;

    self.touchConfirmTimer = [NSTimer pp_scheduledTimerWithTimeInterval:PPCV_CONFIRM_TIMEINTERVAL
                                                                repeats:YES
                                                                  block:^(NSTimer * _Nonnull timer) {
                                                                      [blockSelf timerConfirm:timer];
                                                                  }];
}


//===============================================================================
// timer function to comfirm a moving is valid
//===============================================================================
- (void)timerConfirm:(NSTimer *)theTimer
{
    self.touchConfirmTimer   = nil;
    self.confirmedTouchPoint = self.lastMovePoint;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - touch methods

//===============================================================================
// touch begin
//===============================================================================
- (void)touchesBegan:(NSSet *)touches withEvent:(UIEvent *)event 
{
	if(self.touchEnable==NO) return ;
	
    self.currentSelectedMark = PPCV_SELECT_MARK_NONE;
    self.currentTouchPoint   = nil;
	
	for (UITouch *touch in touches) 
	{		
		CGPoint curPoint     = [touch locationInView:self];
        
        //////////////////////////////////////////////////
		// resize
        
		for(int i=0; i<PPCV_SM_Count; i++)
		{
            //////////////////////////////////////////////////
			//　加大選取區的感應範圍
            
			CGRect outlineRect= CGRectInset(_markFrame[i], -1*PPCV_EXTRA_TOUCH_MARK_SIZE, -1*PPCV_EXTRA_TOUCH_MARK_SIZE);
            
			if (i>=PPCV_SM_Count/2&&i<PPCV_SM_Count)
			{
				if (i==PPCV_SM_Top&&[self isCenterSelectMarkEnableWithStartPoint:_cropPointOfView[PPCV_SM_LeftTop] endPoint:_cropPointOfView[PPCV_SM_RightTop]]==NO) 
				{
                    //////////////////////////////////////////////////
					// 上方的移動點
                    
					outlineRect = CGRectZero;
				}
				
				if (i==PPCV_SM_Bottom&&[self isCenterSelectMarkEnableWithStartPoint:_cropPointOfView[PPCV_SM_LeftBottom] endPoint:_cropPointOfView[PPCV_SM_RightBottom]]==NO) 
				{
                    //////////////////////////////////////////////////
					// 下方的移動點
                    
					outlineRect = CGRectZero;
				}
				if (i==PPCV_SM_Left&&[self isCenterSelectMarkEnableWithStartPoint:_cropPointOfView[PPCV_SM_LeftTop] endPoint:_cropPointOfView[PPCV_SM_LeftBottom]]==NO) 
				{
                    //////////////////////////////////////////////////
					// 右方的移動點
                    
					outlineRect = CGRectZero;
				}
				if (i==PPCV_SM_Right&&[self isCenterSelectMarkEnableWithStartPoint:_cropPointOfView[PPCV_SM_RightTop] endPoint:_cropPointOfView[PPCV_SM_RightBottom]]==NO) 
				{
                    //////////////////////////////////////////////////
					// 左方的移動點
                    
					outlineRect = CGRectZero;
				}
			}

			if(CGRectContainsPoint(outlineRect, curPoint))
			{
                self.currentTouchPoint   = touch;
                self.currentSelectedMark = i;

				[self getValidPoint:&curPoint withTouchIndex:self.currentSelectedMark];
				[self setCropNode:self.currentSelectedMark toPoint:curPoint];
				[self setSelectMarkFrame];
//				[self startConfirmTimer:curPoint];
				
				if(self.magnifierEnable)
				{
					[self setNeedsDisplay];
					
					self.ppMagnifierView.targetView  = self.superview;
					curPoint = [self convertPoint:curPoint toView:self.superview];
					self.ppMagnifierView.targetPoint = curPoint;
				}
				break;
			}
		}
		
		if (self.moveCropViewEnable==YES&&self.currentSelectedMark<0) 
		{
            self.moveStartPoint      = curPoint;
            self.currentTouchPoint   = touch;
            self.currentSelectedMark = PPCV_SELECT_MARK_MOVE;
		}
	}
}


//===============================================================================
// touch move
//===============================================================================
- (void)touchesMoved:(NSSet *)touches withEvent:(UIEvent *)event 
{ 	
	if(self.touchEnable==NO) return ;
	
	BOOL	bNeedRedraw = NO;
	CGPoint	curPoint;
	
	for (UITouch *touch in touches) 
	{
		curPoint = [touch locationInView:self];
		if(self.currentTouchPoint && touch == self.currentTouchPoint)
		{		
			[self getValidPoint:&curPoint withTouchIndex:self.currentSelectedMark];

			if (self.currentSelectedMark>=0) 
			{
				
				[self setCropNode:self.currentSelectedMark toPoint:curPoint];
				[self setSelectMarkFrame];
				
				if(self.magnifierEnable)
				{
					curPoint = [self convertPoint:curPoint toView:self.superview];
					self.ppMagnifierView.targetPoint = curPoint;
				}
                
				bNeedRedraw	= YES;
			}
			else 
			{
				if (self.moveCropViewEnable==YES)
                {
					[self moveCropToPoint:curPoint];
					[self setSelectMarkFrame];	
					
					bNeedRedraw	= YES;
				}
			}
		}
	}	

	if(bNeedRedraw)
    [self setNeedsDisplay];
}


//===============================================================================
// touch end
//===============================================================================
- (void)touchesEnded:(NSSet *)touches withEvent:(UIEvent *)event 
{
	if(self.touchEnable==NO) return ;

//	[self stopConfirmTimer];
//	NSInteger defaultConfirmDistance = PPCV_CONFIRM_DISTANCE;
//	if (self.smallLineMode==YES) {
//		defaultConfirmDistance =PPCV_CONFIRM_DISTANCE/2;
//	}
		
	// !! 校正最後結束的坐標，修正可能出現的滑動現象
//	if(!CGPointEqualToPoint(self.confirmedTouchPoint, CGPointZero))
//	{
//		CGFloat		x = _cropPointOfView[self.currentSelectedMark].x - self.confirmedTouchPoint.x;
//		CGFloat		y = _cropPointOfView[self.currentSelectedMark].y - self.confirmedTouchPoint.y;
//		NSInteger	movedDistance = dist(x, y);
//		
//		if(movedDistance > 0 && movedDistance < defaultConfirmDistance)
//		{
//			// call by referance 無法用'self.confirmedTouchPoint'取，所以還是用'confirmedTouchPoint_'
//			[self getValidPoint:&confirmedTouchPoint_ withTouchIndex:self.currentSelectedMark];
//			
//			[self setCropNode:self.currentSelectedMark toPoint:self.confirmedTouchPoint];
//
//			[self setSelectMarkFrame];
//
//			[self setNeedsDisplay];
//		}
//	}
	
    self.currentSelectedMark = PPCV_SELECT_MARK_NONE;
    self.currentTouchPoint   = nil;

	if(self.magnifierEnable)
	{
		self.ppMagnifierView.targetView = nil;
        
		[self setNeedsDisplay];
	}
}


//===============================================================================
// touch cancel
//===============================================================================
- (void)touchesCancelled:(NSSet *)touches withEvent:(UIEvent *)event
{
	if(self.touchEnable==NO) return ;

    self.currentSelectedMark = PPCV_SELECT_MARK_NONE;
    self.currentTouchPoint   = nil;

	if(self.magnifierEnable)
	{
		self.ppMagnifierView.targetView = nil;
        
		[self setNeedsDisplay];
	}	
}

@end
