//
//  GTLRDriveService+Convert.m
//  
//
//  Created by Howard on 2019/3/12.
//

#import "GTLRDriveService+Convert.h"

// Category
#import "NSError+Custom.h"
#import "NSThread+Additions.h"
#import "GTLRDriveObjects.h"

// Framework
#import <objc/runtime.h>

////////////////////////////////////////////////////////////////////////////////////////////////////

// google drive api取用限制100sec內不能超過1000個request
static CGFloat const GTLRServiceDrive_TimeInterval  = 0.1;
static NSString *  const KGLTRDriveServiceDictionaryKey  = @"KGLTRDriveServiceDictionaryKey";

NSString *PPPCloud_RootPath_GoogleDrive  = @"/";
NSString *GTLDriveFile_MimeType_Folder = @"application/vnd.google-apps.folder";


@implementation GTLRDriveService (Convert)

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Property Method

//================================================================================
//
//================================================================================
- (void)setIdentifierForFolderDictionary:(NSMutableDictionary *)identifierForFolderDictionary
{
    objc_setAssociatedObject(self, (void *)(KGLTRDriveServiceDictionaryKey), identifierForFolderDictionary, OBJC_ASSOCIATION_RETAIN);
}


//================================================================================
//
//================================================================================
- (NSMutableDictionary *)identifierForFolderDictionary
{
    return objc_getAssociatedObject(self, (void *)(KGLTRDriveServiceDictionaryKey));
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Method

//================================================================================
//
//================================================================================
- (NSArray *)filePathesForFileList:(GTLRDriveQuery_FilesList *)fileListQuery
                             error:(NSError **)error
{
    __block NSError *returnError = nil;
    
    __block NSArray *filePaths = nil;
    
    __block BOOL executing = YES;
    
    __block typeof(self) blockself = self;
    
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
        GTLRServiceTicket *fileListTicket = [blockself executeQuery:fileListQuery
                                                  completionHandler:^(GTLRServiceTicket *callbackTicket,
                                                                      GTLRDrive_FileList *fileList,
                                                                      NSError *callbackError) {
            
                                                      if(callbackError!=nil)
                                                      {
                                                          returnError = [callbackError copy];
                                                      }
                                                      else
                                                      {
                                                          filePaths = [fileList.files retain];
                                                      }
                                                       executing = NO;
        }];
        
        if(fileListTicket==nil)
        {
            executing = NO;
        }
    });
    
    ////////////////////////////////////////////////////////////////////////////////////////////////////
    
    while(executing==YES)
    {
        [NSThread waitWithTimeInterval:GTLRServiceDrive_TimeInterval];
    }
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    [returnError autorelease];
    
    return [filePaths autorelease];
}


//================================================================================
//
//================================================================================
- (NSString *)rootFolderIdentifierForFileList:(GTLRDriveQuery_FilesGet *)fileQuery
                                        error:(NSError **)error
{
    __block NSError *returnError = nil;
    
    __block NSString *rootFolderIdentifier = nil;
    
    __block BOOL executing = YES;
    
    __block typeof(self) blockself = self;
    
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
   
        executing = YES;
        fileQuery.fields = @"*";
        
        ////////////////////////////////////////////////////////////////////////////////////////////////////
        GTLRServiceTicket *fileTicket = [blockself executeQuery:fileQuery
                                         completionHandler:^(GTLRServiceTicket *ticket, GTLRDrive_File *file, NSError *fileError)
                                         {
                                             if(fileError!=nil)
                                             {
                                                 returnError = [fileError copy];
                                             }
                                             else if(file!=nil)
                                             {
                                                 rootFolderIdentifier = [file.identifier copy];
                                             }
                                             
                                             executing = NO;
                                         }];
        if(fileTicket==nil)
        {
            executing = NO;
        }
    });
    
    ////////////////////////////////////////////////////////////////////////////////////////////////////
    
    while(executing==YES)
    {
        [NSThread waitWithTimeInterval:GTLRServiceDrive_TimeInterval];
    }
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    [returnError autorelease];
    
    return [rootFolderIdentifier autorelease];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Method

//================================================================================
//
//================================================================================
- (NSArray *)filesForPathComponents:(NSArray *)pathComponents
             currentPathComponentID:(NSString *)currentPathComponentID
                              error:(NSError **)error
{
    NSLog(@"%s, pathComponents:%@, currentPathComponentID:%@",__func__,pathComponents,currentPathComponentID);
    
    //////////////////////////////////////////////////

    NSMutableArray *files = nil;

    do
    {
        if(pathComponents==nil || ([pathComponents count]<1 && currentPathComponentID==nil))
        {
            if(error)
            {
                *error = PPErrorParameterInvalidity(nil);
            }
            break;
        }

        ////////////////////////////////////////////////////////////////////////////////////////////////////

        files = [[[NSMutableArray alloc] init] autorelease];
        if(files==nil)
        {
            if(error)
            {
                *error = PPErrorOperationFailed(nil);
            }
            break;
        }

        ////////////////////////////////////////////////////////////////////////////////////////////////////

        __block BOOL executing = NO;

        if(currentPathComponentID==nil)
        {
            if([PPPCloud_RootPath_GoogleDrive isEqualToString:[pathComponents objectAtIndex:0]]==NO)
            {
                if(error)
                {
                    *error = PPErrorParameterInvalidity(nil);
                }
                break;
            }

            //////////////////////////////////////////////////

            GTLRDriveQuery_FilesGet *rootFolderQuery = [GTLRDriveQuery_FilesGet queryWithFileId:@"root"];
            
            if(rootFolderQuery==nil)
            {
                if(error)
                {
                    *error = PPErrorOperationFailed(nil);
                }
                break;
            }
        
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            executing = YES;
         
            ////////////////////////////////////////////////////////////////////////////////////////////////////
            
            NSString *identifier = [self rootFolderIdentifierForFileList:rootFolderQuery error:error];
            
            if(*error==nil)
            {
                if (identifier==nil)
                {
                    if(error)
                    {
                        *error = PPErrorOperationFailed(nil);
                    }
                }
                else
                {
                    NSArray *filesForPath = [self filesForPathComponents:pathComponents
                                                  currentPathComponentID:identifier
                                                                   error:error];
                    if(*error==nil)
                    {
                        [files addObjectsFromArray:filesForPath];
                    }
                }
            }
        }
        else
        {
            if([pathComponents count]==1)
            {
                GTLRDrive_File *file = [self fileForIdentifier:currentPathComponentID error:error];
                
                if(file==nil)
                {
                    break;
                }
                
                //////////////////////////////////////////////////

                [files addObject:file];
            }
            else
            {
                GTLRDriveQuery_FilesList *fileListQuery = [GTLRDriveQuery_FilesList query];
                
                if(fileListQuery==nil)
                {
                    if(error)
                    {
                        *error = PPErrorOperationFailed(nil);
                    }
                    break;
                }

                //////////////////////////////////////////////////
                // query 語法參考 (https://developers.google.com/drive/v3/web/search-parameters)
                NSString *queryFileName = [pathComponents objectAtIndex:1];
                // !!反斜線(\)要改為(\\), 這個要先做
                queryFileName = [queryFileName stringByReplacingOccurrencesOfString:@"\\" withString:@"\\\\"];
                // !!檔名有單引號(')要改為(\'),
                queryFileName = [queryFileName stringByReplacingOccurrencesOfString:@"'" withString:@"\\'"];

                ////////////////////////////////////////////////////////////////////////////////////////////////////
                fileListQuery.q =  [NSString stringWithFormat:@"trashed=false and '%@' in parents and name='%@'", currentPathComponentID, queryFileName];

                ////////////////////////////////////////////////////////////////////////////////////////////////////
                executing = YES;
                fileListQuery.fields = @"*";

                ////////////////////////////////////////////////////////////////////////////////////////////////////
                
                NSArray *driveFiles = [self filePathesForFileList:fileListQuery error:error];
                
                for(GTLRDrive_File *file in driveFiles)
                {
                    NSMutableArray *remainPathComponents = [[[NSMutableArray alloc] initWithArray:pathComponents] autorelease];
                    if(remainPathComponents==nil)
                    {
                        if(error)
                        {
                            *error = PPErrorOperationFailed(nil);
                        }
                    }
                    
                    [remainPathComponents removeObjectAtIndex:1];
                    
                    ////////////////////////////////////////////////////////////////////////////////////////////////////
                    
                    NSArray *filesForPath = [self filesForPathComponents:remainPathComponents
                                                  currentPathComponentID:file.identifier
                                                                   error:error];
                    if(*error==nil)
                    {
                        [files addObjectsFromArray:filesForPath];
                    }
                }
            }
        }

    }while(0);

    return files;
}


//================================================================================
//
//================================================================================
- (GTLRDrive_File *)fileForIdentifier:(NSString *)identifier
                                error:(NSError **)error
{
    __block BOOL executing = NO;
    __block GTLRDrive_File *targetFile = nil;
    
    do
    {
        if([identifier length]<=0)
        {
            if(error)
            {
                *error = PPErrorParameterInvalidity(nil);
            }
            
            break;
        }
        
        //////////////////////////////////////////////////

        GTLRDriveQuery_FilesGet *fileQuery = [GTLRDriveQuery_FilesGet queryWithFileId:identifier];
        
        if(fileQuery==nil)
        {
            if(error)
            {
                *error = PPErrorOperationFailed(nil);
            }
            break;
        }
        
        //////////////////////////////////////////////////
        
        executing = YES;
        fileQuery.fields = @"*";
        
        ////////////////////////////////////////////////////////////////////////////////////////////////////
        GTLRServiceTicket *fileTicket = [self executeQuery:fileQuery
                                         completionHandler:^(GTLRServiceTicket *ticket, GTLRDrive_File *file, NSError *fileError)
                                         {
                                             if(error)
                                             {
                                                 if(fileError!=nil)
                                                 {
                                                     *error = [fileError copy];
                                                 }
                                                 else if(file!=nil)
                                                 {
                                                     targetFile = [file retain];
                                                 }
                                             }
                                             executing = NO;
                                         }];
        if(fileTicket==nil)
        {
            executing = NO;
            
            break;
        }
        
        ////////////////////////////////////////////////////////////////////////////////////////////////////
        while(executing==YES)
        {
            [NSThread waitWithTimeInterval:GTLRServiceDrive_TimeInterval];
        }
    }
    while (0);
  
    return [targetFile autorelease];
}
@end
