//
//  PPImageLuminance.m
//  PPCameraView
//
//  Created by sanhue cheng on 2020/5/15.
//

#import "PPImageAlignment.h"
#import <Metal/Metal.h>
#import "PPCVKernelTypes.h"
#import "MetalHelper.h"

@interface PPImageAlignment ()

@property (nonatomic, strong) id<MTLDevice> mtlDevice;
@property (nonatomic, strong) id <MTLComputePipelineState> createMappingPipelineState;
@property (nonatomic, strong) id <MTLComputePipelineState> alignPipelineState;
@end


////////////////////////////////////////////////////////////////////////////////////////////////////
@implementation PPImageAlignment





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - init/dealloc methods


//==============================================================================
//
//==============================================================================
- (nullable instancetype)initWithDevice:(nonnull id <MTLDevice>)device
{
    self = [super init];
    if (self)
    {
        self.mtlDevice = device;
        
        NSString *metallibPath = [[NSBundle mainBundle] pathForResource:@"PPHDRGenerator" ofType:@"metallib"];
        id<MTLLibrary> library = [self.mtlDevice newLibraryWithFile:metallibPath error:nil];

        // compute pipline
        id <MTLFunction> createMappingFunction = [library newFunctionWithName:@"createMappingTable"];
        self.createMappingPipelineState = [[self.mtlDevice newComputePipelineStateWithFunction:createMappingFunction error:nil] autorelease];
        [createMappingFunction release];
        [library release];
    }
    return self;
}


//==============================================================================
//
//==============================================================================
- (void)dealloc
{
    self.createMappingPipelineState = nil;
    self.alignPipelineState = nil;
    self.mtlDevice = nil;
    //////////////////////////////////////////////////
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private


//==============================================================================
// 亮度中位數
//==============================================================================
- (LuminanceInfo *)computeMedianLuminanceWithTexture:(id<MTLTexture>)texture rect:(CGRect)rect imageWidth:(NSInteger)imageWidth
{
    if (texture==nil)
    {
        return nil;
    }
    
    int mtb_x = rect.origin.x;
    int mtb_y = rect.origin.y;
    int mtb_width = rect.size.width;
    int mtb_height = rect.size.height;
    
    int n_samples_c = 100;
    int n_w_samples = (int) sqrt(n_samples_c);
    int n_h_samples = n_samples_c / n_w_samples;
    int histo[256] = {0};
    
    int *imageDataBytes = malloc(texture.width*texture.height*sizeof(int));
    
    [texture getBytes:imageDataBytes
          bytesPerRow:texture.width*sizeof(int)
           fromRegion:MTLRegionMake2D(0, 0, texture.width, texture.height)
          mipmapLevel:0];
    
    // 清空
    for (int i = 0; i < 256; i++)
        histo[i] = 0;
    
    int total = 0;
    for (int y = 0; y < n_h_samples; y++)
    {
        double alpha = ((double) y + 1.0) / ((double) n_h_samples + 1.0);
        int y_coord = mtb_y + (int) (alpha * mtb_height);
        
        for (int x = 0; x < n_w_samples; x++)
        {
            double beta = ((double) x + 1.0) / ((double) n_w_samples + 1.0);
            int x_coord = mtb_x + (int) (beta * mtb_width);
            
            NSInteger dataIndex = x_coord*imageWidth+y_coord;
            int color = imageDataBytes[dataIndex];
            
            int r = (color & 0xFF0000) >> 16;
            int g = (color & 0xFF00) >> 8;
            int b = (color & 0xFF);
            int luminance = MAX(r, g);
            luminance = MAX(luminance, b);
            histo[luminance]++;
            total++;
        }
    }
    int middle = total / 2;
    int count = 0;
    BOOL noisy = NO;
    
    for (int i = 0; i < 256; i++)
    {
        count += histo[i];
        
        if (count >= middle)
        {
            int noise_threshold = 4;
            int n_below = 0;
            
            for (int j = 0; j <= i - noise_threshold; j++)
            {
                n_below += histo[j];
            }
            double frac_below = n_below / (double) total;
            if (frac_below < 0.2)
            {
                noisy = true;
            }
            return [LuminanceInfo luminanceInfoWithMedianValue:i noisy:noisy];
        }
    }
    return [LuminanceInfo luminanceInfoWithMedianValue:127 noisy:YES];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Public


//==============================================================================
//
//==============================================================================
-(void)encodeToCommandBuffer:(nonnull id <MTLCommandBuffer>)commandBuffer
              sourceTextures:(nonnull NSArray <id <MTLTexture>>*)sourceTextures
                   imageSize:(CGSize)imageSize
{
    NSLog(@"%s in", __PRETTY_FUNCTION__);
    

    int mtb_width = imageSize.width / 2;
    int mtb_height = imageSize.height / 2;
    int mtb_x = mtb_width / 2;
    int mtb_y = mtb_height / 2;
    
    
    NSMutableArray <LuminanceInfo *>*luminanceInfos = [NSMutableArray array];
    NSMutableArray <id<MTLBuffer>> *mappingTables = [NSMutableArray array];
    //////////////////////////////////////////////////
    id<MTLComputeCommandEncoder> computingCommandEncoder = [commandBuffer computeCommandEncoder]; // 通过渲染描述符构建 encoder
    
    // 建立mapping table
    for (NSInteger i=0; i < [sourceTextures count]; i++)
    {
        // 計算luminance
        id <MTLTexture> sourceTexture = [sourceTextures objectAtIndex:i];
        LuminanceInfo *luminanceInfo = [self computeMedianLuminanceWithTexture:sourceTexture
                                                                          rect:CGRectMake(mtb_x, mtb_y, mtb_width, mtb_height)
                                                                    imageWidth:imageSize.width];
        NSLog(@"luminanceInfo:%@", luminanceInfo);
        [luminanceInfos addObject:luminanceInfo];
        
        //////////////////////////////////////////////////
        if (luminanceInfos[i].noisy) {
//            mtb_allocations[i] = null;
            continue;
        }

        int median_value = luminanceInfos[i].medianValue;
        
        id<MTLBuffer> mappingTable = [self.mtlDevice newBufferWithLength:mtb_width*mtb_height options:MTLResourceStorageModeShared];
        
        if(mappingTable!=nil)
        {
            [mappingTables addObject:mappingTable];
            [mappingTable release];
        }
        
        id<MTLTexture> mappingTableTexture = [MetalHelper textureWithSize:CGSizeMake(mtb_width, mtb_height)
                                                              pixelFormat:sourceTexture.pixelFormat
                                                                   device:self.mtlDevice];

        id<MTLBuffer> creatingMappingParams = [self.mtlDevice newBufferWithLength:sizeof(PPCVCreateMappingParams) options:MTLResourceStorageModeShared];
        PPCVCreateMappingParams *param = (PPCVCreateMappingParams *)creatingMappingParams.contents;
        param->startX = mtb_x;
        param->startY = mtb_y;
        param->medianValue = median_value;

        
        [computingCommandEncoder setComputePipelineState:self.createMappingPipelineState];
//        [computingCommandEncoder setBuffer:creatingMappingParams offset:0 atIndex:0];
        [computingCommandEncoder setTexture:sourceTexture atIndex:0];
        [computingCommandEncoder setTexture:mappingTableTexture atIndex:1];
//        if (@available(iOS 10.0, *)) {
//            [computingCommandEncoder setStageInRegion:MTLRegionMake2D(mtb_x, mtb_y, mtb_width, mtb_height)];
//        } else {
//            // Fallback on earlier versions
//        }
        
        [creatingMappingParams release];
        
    }
    
    //////////////////////////////////////////////////
    // Calculate a threadgroup size.
    NSUInteger w = self.createMappingPipelineState.threadExecutionWidth;
    NSUInteger h = self.createMappingPipelineState.maxTotalThreadsPerThreadgroup / w;
    MTLSize threadsPerThreadgroup = MTLSizeMake(w, h, 1);
    MTLSize threadgroupsPerGrid = MTLSizeMake((imageSize.width + w - 1) / w,
                                              (imageSize.height + h - 1) / h,
                                              1);
    
    // Encode the compute command.
    [computingCommandEncoder dispatchThreadgroups: threadgroupsPerGrid
                            threadsPerThreadgroup: threadsPerThreadgroup];
    
    // End the compute pass.
    [computingCommandEncoder endEncoding];
    
    //////////////////////////////////////////////////
    
    NSLog(@"%s out", __PRETTY_FUNCTION__);
}


@end
