//
//  PPSQLiteController.h
//  
//
//  Created by Mike on 13/5/13.
//  Copyright (c) 2013年 Penpower. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <sqlite3.h>
#import "NSError+Custom.h"

@interface PPSQLiteController : NSObject
{
@private
	NSString    *dbPath_;
	NSString    *plistPath_;
	sqlite3     *dbHandle_;
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Property

@property(nonatomic,readonly,assign) sqlite3 *dbHandle;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Deallocating Objects

/**
 * Initial PPSQLiteController
 *
 * @param fileName for database
 * @param path for database
 * @return PPSQLiteController instance
 */
- (id)initWithFileName:(NSString *)fileName atPath:(NSString *)path error:(NSError **)error;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Open & Close Methods

/**
 * Open database
 *
 * @param error for open database
 * @return Result
 */
- (BOOL)openWithError:(NSError **)error;

/**
 * Close database
 *
 * @param error for close database
 * @return Result
 */
- (BOOL)closeWithError:(NSError **)error;

/**
 * Remove database
 *
 * @param error for remove database
 * @return Result
 */
- (BOOL)removeWithError:(NSError **)error;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - DB Methods

/**
 * Check database already exist
 *
 * @return Result
 */
- (BOOL)isExist;

/**
 * Create database struct command
 *
 * @param commands for create database struct
 * @param error for create database
 * @return Result
 */
- (BOOL)createWithCommands:(NSArray *)commands error:(NSError **)error;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Compact Methods

/**
 * Compact database
 *
 * @param error for create database
 * @return Result
 */
- (BOOL)compactWithError:(NSError **)error;

/**
 * Detect database need to compact
 *
 * @return Result
 */
- (BOOL)needsCompact;

/**
 * When datbase record delete, you should call this method
 *
 */
- (void)incrementDeleteCount;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Status Dictionary Read&Write Operations

/**
 * Status object for database
 *
 * @param key for status object
 * @return status object
 */
- (id)statusObjectForKey:(id)key;

/**
 * Set status object for database
 *
 * @param object for database status
 * @param key for status object
 */
- (void)setStatusObject:(id)object forKey:(id<NSCopying>)key;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Version Control Methods

/**
 * Database version
 *
 * @param major for database version
 * @param minor for database version
 * @param error for set database version
 * @return Result
 */
- (BOOL)setVersionMajor:(int)major minor:(int)minor error:(NSError **)error;

/**
 * Set database version
 *
 * @param major for database version
 * @param minor for database version
 * @param error for get database version
 * @return Result
 */
- (BOOL)getVersionMajor:(int *)major minor:(int *)minor error:(NSError **)error;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Transaction Methods

/**
 * Database begin transaction
 *
 * @param error for database begin transaction
 * @return Result
 */
- (BOOL)beginTransactionWithError:(NSError **)error;

/**
 * Database commit transaction
 *
 * @param error for database commit transaction
 * @return Result
 */
- (BOOL)commitTransactionWithError:(NSError **)error;

/**
 * Database end transaction
 *
 * @param error for database end transaction
 * @return Result
 */
- (BOOL)endTransactionWithError:(NSError **)error;

/**
 * Database rollback transaction
 *
 * @param error for database rollback transaction
 * @return Result
 */
- (BOOL)rollbackTransactionWithError:(NSError **)error;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Methods

/**
 * Execution command for database
 *
 * @param command to execution
 * @param error for execution command
 * @return Result
 */
- (BOOL)runCommand:(NSString *)command error:(NSError **)error;

/**
 * Get integer result by execution command for database
 *
 * @param command to execution
 * @param error for execution command
 * @return Integer result
 */
- (NSInteger)integerResultWithCommand:(NSString *)command error:(NSError **)error;

/**
 * Get count by execution command for database
 *
 * @param command to execution
 * @param error for execution command
 * @return Record count
 */
- (NSInteger)recordCountWithCommand:(NSString *)command error:(NSError **)error;

/**
 * Get count by tableName for database
 *
 * @param tableName for database
 * @param error for counting the tableName
 * @return Record count
 */
- (NSInteger)recordCountWithTableName:(NSString *)tableName error:(NSError **)error;

/**
 * Interrupt A Long-Running Query
 */
- (void)cancel;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Methods

/**
 * Formate command
 *
 * @param string to formate
 * @return Formatted string
 */
+ (NSString *)formattedString:(NSString *)string;

/**
 * Check database with filename already exist
 *
 * @param fileName fileName for database
 * @param atPath path for database
 * @return Result
 */
+ (BOOL)isExistFileName:(NSString *)fileName atPath:(NSString *)atPath;

/**
 * Get integer from statement's column
 *
 * @param statement for database
 * @param column for statement
 * @return Integer
 */
+ (NSInteger)integerFromStatement:(sqlite3_stmt *)statement column:(int)column;

/**
 * Get double from statement's column
 *
 * @param statement for database
 * @param column for statement
 * @return Double
 */
+ (double)doubleFromStatement:(sqlite3_stmt *)statement column:(int)column;

/**
 * Get string from statement's column
 *
 * @param statement for database
 * @param column for statement
 * @return String
 */
+ (NSString *)stringFromStatement:(sqlite3_stmt *)statement column:(int)column;

/**
 * Get data from statement's column
 *
 * @param statement for database
 * @param column for statement
 * @return Data
 */
+ (NSData *)dataFromStatement:(sqlite3_stmt *)statement column:(int)column;

@end
