//
//  PPLocationController.h
//
// important: be sure to include NSLocationWhenInUseUsageDescription along with its
// explanation string in your Info.plist or startUpdatingLocation will not work.
//
//  Created by Howard on 2015/12/29.
//
//

#import <Foundation/Foundation.h>
#import <CoreLocation/CoreLocation.h>


// Define
#import "PPLocationController+ParameterDefine.h"

@interface PPLocationController : NSObject

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Property

//模式選擇 The array of @[[AppleLocationManager class],[BaiduLocationManager class],[GoogleLocationManager class]]],預設全空間 see PPLocationController+ParameterDefine.h
@property (nonatomic,retain) NSArray *locationManagerOrders;

@property (nonatomic,assign) CGFloat operationTimeInterval; // 預設 1秒，太頻繁地址反查Location, apple \ google api 會time out
@property (nonatomic,assign) CGFloat operationTimeOutInterval; // 預設 3秒， apple \ google api 會time out 休息時間

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Method

/**
 *  單例
 *
 *  @return 單例
 */
+ (instancetype)sharedInstance;


/**
 *  位置服務權限判斷
 *
 *  @return YES, 權限開啟
 */
+ (BOOL)checkLocationAccess;


/**
 *  開始更新使用者位置
 *  處理方式:非同步方法，上層要自行處理 UI 行為
 *
 *  @return YES, 開始進行使用者位置更新
 */
+ (BOOL)updatingCurrentLocation;


/**
 *  開始詢問一組地址位置資訊
 *
 *  @param addressDictionary 地址字典
 *  @param identity          額外資訊
 *
 *  @return 一組地址位置資訊
 */
+ (BOOL)fetchLocationFromAddressDictionary:(NSDictionary *)addressDictionary
                                  identity:(NSString *)identity;

/**
 *  開始詢問地址位置資訊
 *
 *  @param addressString 地址
 *  @param region    區域
 *  @param identity  額外資訊
 *  @return YES, 開始詢問地址位置資訊
 */
+ (BOOL)fetchLocationFromAddressString:(NSString *)addressString region:(CLCircularRegion *)region identity:(NSString *)identity;


/**
 *  開始詢問地址
 *
 *  @param location 位置
 *  @param identity 額外資訊
 *
 *  @return YES, 開始詢問地址資訊
 */
+ (BOOL)fetchAddressStringFromLocation:(CLLocation *)location identity:(NSString *)identity;


/**
 *  取消所有操作
 */
+ (void)cancelAllOperation;


/**
 *  等到所有指令結束
 *  處理方式:Location 發出 PPLocationControllerAccessResult_GetAddressLocationEnd
 */
+ (void)waitUntilAllLocationOperationsAreFinished;


/**
 *  目前指令數目
 *
 *  @return 指令數目
 */
+ (NSUInteger)locationOperationCount;


/**
 *  待執行命令數目
 *
 *  @return 待執行命令數目
 */
+ (NSUInteger)readyOperationCount;


/**
 *  計算兩點距離 (meter)
 *
 *  @param location1 點1
 *  @param location2 點2
 *
 *  @return 距離
 */
+ (double)meterDistanceBetweenLocation1:(CLLocation *)location1 location2:(CLLocation *)location2;


/**
 *  CLLocation 轉字串 [NSString stringWithFormat:@"%f,%f", location.coordinate.latitude, location.coordinate.longitude]
 *
 *  @param location 來源Location
 *
 *  @return 字串
 */
+ (NSString *)stringForLocation:(CLLocation *)location;


/**
 *  字串轉 Location
 *
 *  @param GPSString 來源字串
 *
 *  @return Locaiton
 */
+ (CLLocation *)copyLocationFromString:(NSString *)GPSString;


/**
 *  Location 是否有效
 *
 *  @param location 來源Location
 *
 *  @return YES, 有效
 */
+ (BOOL)isValidLocation:(CLLocation *)location;


/**
 *  無效 Location
 *
 *  @return 無效 Location
 */
+ (CLLocation *)invalidLocation;


/**
 *  無效 Location 字串
 *
 *  @return 無效 Location 字串
 */
+ (NSString *)invalidLocationString;

@end
