//
//  WCXFDataController.m
//
//

#import "WCXFDataController.h"
#import "WCXFWrapper.h"
#import "NSError+Custom.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

NSString * const WCXFDataController_FileExt = @"wcxf";

NSString * const WCXFDC_GoogleGroupName_MyContact = @"My Contacts(Gmail)";
NSString * const WCXFDC_GoogleGroupName_Friends = @"Friends(Gmail)";
NSString * const WCXFDC_GoogleGroupName_Family = @"Family(Gmail)";
NSString * const WCXFDC_GoogleGroupName_Coworkers = @"Coworkers(Gmail)";

static WCXFDataController_ProductLine staticProductLine = WCXFDataController_ProductLine_WC;
////////////////////////////////////////////////////////////////////////////////////////////////////

@interface WCXFDataController () <WCXFWrapperDelegate>
@property (nonatomic, assign) id<WCXFDataControllerDelegate> delegate;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation WCXFDataController

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods

//================================================================================
//
//================================================================================
- (instancetype)initWithDelegate:(id<WCXFDataControllerDelegate>)delegate
{
    if(self = [super init])
    {
        self.delegate = delegate;
    }

    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    
    //////////////////////////////////////////////////
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods


//================================================================================
//
//================================================================================
- (NSArray *)copyDefinedGroupNamesWithCardModel:(WCCardModel *)cardModel
{
    NSMutableArray *groupNames = [[NSMutableArray alloc] init];
    
    for(NSString *groupIDString in cardModel.groupIDArray)
    {
        WC_GroupID groupID = [groupIDString integerValue];
        NSString *groupName = nil;
    
        switch (groupID)
        {
            case WC_GID_Google_MyContacts:
            case WC_GID_Google_Friends:
            case WC_GID_Google_Family:
            case WC_GID_Google_Coworkers:
            case WC_GID_Unfiled:
            {
                break; 
            }
            default:
            {
                groupName = [self.delegate wcxfDataContrller:self groupNameToWriteWithGroupID:groupID];
                
                if([groupName length] > 0)
                {
                    [groupNames addObject:groupName];
                }                
                
                break;
            }
        }
        
    }
    
    if([groupNames count] == 0)
    {
        [groupNames release];
        groupNames = nil;
    }
    
    return groupNames;
}


//================================================================================
//
//================================================================================
- (NSArray *)copyDefaultGroupNamesWithCardModel:(WCCardModel *)cardModel
{
    NSMutableArray *groupNames = [[NSMutableArray alloc] init];
    
    for(NSString *groupIDString in cardModel.groupIDArray)
    {
        WC_GroupID groupID = [groupIDString integerValue];
        NSString *groupName = nil;
        
        switch (groupID)
        {
            case WC_GID_Google_MyContacts:
            {
                groupName = WCXFDC_GoogleGroupName_MyContact;
                [groupNames addObject:groupName];
                break;
            }
                
            case WC_GID_Google_Friends:
            {
                groupName = WCXFDC_GoogleGroupName_Friends;
                [groupNames addObject:groupName];
                break;
            }
                
            case WC_GID_Google_Family:
            {
                groupName = WCXFDC_GoogleGroupName_Family;
                [groupNames addObject:groupName];
                break;
            }
                
            case WC_GID_Google_Coworkers:
            {
                groupName = WCXFDC_GoogleGroupName_Coworkers;
                [groupNames addObject:groupName];
                break;
            }
            case WC_GID_Unfiled:
            default:
            {
                break;
            }
        }
        
    }
    
    if([groupNames count] == 0)
    {
        [groupNames release];
        groupNames = nil;
    }
    
    return groupNames;
}


//================================================================================
//
//================================================================================
- (WC_GroupID)groupIDWithName:(NSString *)groupName error:(NSError **)error
{
    WC_GroupID groupID = WC_GID_None;
    
    if([groupName length] > 0)
    {
        // !! 只有WCM才有Google預設群組
        if (staticProductLine==WCXFDataController_ProductLine_WC)
        {
            if([groupName isEqualToString:WCXFDC_GoogleGroupName_MyContact])
            {
                groupID = WC_GID_Google_MyContacts;
            }
            else if([groupName isEqualToString:WCXFDC_GoogleGroupName_Friends])
            {
                groupID = WC_GID_Google_Friends;
            }
            else if([groupName isEqualToString:WCXFDC_GoogleGroupName_Family])
            {
                groupID = WC_GID_Google_Family;
            }
            else if([groupName isEqualToString:WCXFDC_GoogleGroupName_Coworkers])
            {
                groupID = WC_GID_Google_Coworkers;
            }
            else
            {
                groupID = [self.delegate wcxfDataContrller:self groupIDWithReadGroupName:groupName error:error];
            }
        }
        else
        {
            groupID = [self.delegate wcxfDataContrller:self groupIDWithReadGroupName:groupName error:error];
        }

    }
    
    return groupID;
}


//================================================================================
//
//================================================================================
- (NSDictionary *)copyImageDataDictWithCardModel:(WCCardModel *)cardModel
{
    NSMutableDictionary *imageDataDict = [[NSMutableDictionary alloc] init];
    NSData *imageData;
    
    imageData = [self.delegate wcxfDataContrller:self
                      imageDataToWriteWithCardID:cardModel.ID
                                       imageType:WC_IT_FrontSide];
    
    if(imageData != nil)
    {
        [imageDataDict setObject:imageData forKey:@(WC_IT_FrontSide)];
    }

    imageData = [self.delegate wcxfDataContrller:self
                      imageDataToWriteWithCardID:cardModel.ID
                                       imageType:WC_IT_BackSide];
    
    if(imageData != nil)
    {
        [imageDataDict setObject:imageData forKey:@(WC_IT_BackSide)];
    }

    imageData = [self.delegate wcxfDataContrller:self
                      imageDataToWriteWithCardID:cardModel.ID
                                       imageType:WC_IT_IDPhoto];
    
    if(imageData != nil)
    {
        [imageDataDict setObject:imageData forKey:@(WC_IT_IDPhoto)];
    }
    
    //////////////////////////////////////////////////

    if([imageDataDict count] == 0)
    {
        [imageDataDict release];
        imageDataDict = nil;
    }
    
    return imageDataDict;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - WCXFWrapperDelegate

//================================================================================
//
//================================================================================
- (BOOL)wrapper:(WCXFWrapper *)wrapper shouldReadCardWithIndex:(NSInteger)index totalCount:(NSInteger)totalCount
{
    return [self.delegate wcxfDataContrller:self shouldReadCardWithIndex:index totalCount:totalCount];
}


//================================================================================
//
//================================================================================
- (void)wrapper:(WCXFWrapper *)wrapper didReadCardInfo:(WCCardModel *)cardModel
{
    [self.delegate wcxfDataContrller:self readCardModel:cardModel];
}


//================================================================================
//
//================================================================================
- (void)wrapper:(WCXFWrapper *)wrapper didReadImageData:(NSData *)imageData imageType:(WC_ImageType)imageType cardID:(NSString *)cardID
{
    [self.delegate wcxfDataContrller:self
                       readImageData:imageData
                           imageType:imageType
                              cardID:cardID];
}


//================================================================================
//
//================================================================================
- (WC_GroupID)wrapper:(WCXFWrapper *)wrapper groupIDWithGroupName:(NSString *)groupName error:(NSError **)error
{
    return [self groupIDWithName:groupName error:error];
}


//================================================================================
//
//================================================================================
- (void)wrapper:(WCXFWrapper *)wrapper didReadCardWithIndex:(NSInteger)index totalCount:(NSInteger)totalCount
{
    [self.delegate wcxfDataContrller:self didReadCardWithIndex:index totalCount:totalCount];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance methods

//================================================================================
//
//================================================================================
- (WCXFRootInfoModel *)rootInfoOfFile:(NSString *)filePath error:(NSError **)error
{
    NSError *returnError = nil;
    WCXFWrapper *wrapper = nil;
    WCXFRootInfoModel *rootInfoModel = nil;
    
    do
    {
        if((wrapper = [[WCXFWrapper alloc] initWithDelegate:self]) == nil)
        {
            returnError = PPErrorMake(WCXFDataController_Error_FailedToInitObject, @"(WCXFWrapper)", nil);
            break;
        }
        
        if([wrapper openFileForReadingWithPath:filePath] == NO)
        {
            returnError = PPErrorMake(WCXFDataController_Error_FailedToOpenFileForReading, nil, nil);
            break;
        }
        
        //////////////////////////////////////////////////
        
        if((rootInfoModel = [wrapper copyRootInfo]) == nil)
        {
            returnError = PPErrorMake(WCXFDataController_Error_FailedToReadRootInfo, nil, nil);
            break;
        }
    }
    while(0);
    
    [wrapper closeFile];
    [wrapper release];
    
    //////////////////////////////////////////////////
    
    if(error != nil)
    {
        *error = returnError;
    }
    
    return [rootInfoModel autorelease];
}


//================================================================================
//
//================================================================================
- (BOOL)readFileWithPath:(NSString *)filePath error:(NSError **)error
{
    BOOL result = NO;
    NSError *returnError = nil;
    WCXFWrapper *wrapper = nil;
    WCXFRootInfoModel *rootInfoModel = nil;
    
    do
    {
        //////////////////////////////////////////////////
        // 一開始先檢查delegate的目的是減少檢查次數，同時提示此流程會使用的delegate method。
        
        if([self.delegate respondsToSelector:@selector(wcxfDataContrller:shouldReadCardWithIndex:totalCount:)] == NO ||
           [self.delegate respondsToSelector:@selector(wcxfDataContrller:readCardModel:)] == NO ||
           [self.delegate respondsToSelector:@selector(wcxfDataContrller:readImageData:imageType:cardID:)] == NO ||
           [self.delegate respondsToSelector:@selector(wcxfDataContrller:groupIDWithReadGroupName:error:)] == NO ||
           [self.delegate respondsToSelector:@selector(wcxfDataContrller:didReadCardWithIndex:totalCount:)] == NO)
        {
            returnError = PPErrorMake(WCXFDataController_Error_MustHaveDelegateMethod, nil, nil);
            break;
        }
        
        //////////////////////////////////////////////////
        
        if((wrapper = [[WCXFWrapper alloc] initWithDelegate:self]) == nil)
        {
            returnError = PPErrorMake(WCXFDataController_Error_FailedToInitObject, @"(WCXFWrapper)", nil);
            break;
        }
        
        wrapper.productLine = staticProductLine;
        
        if([wrapper openFileForReadingWithPath:filePath] == NO)
        {
            returnError = PPErrorMake(WCXFDataController_Error_FailedToOpenFileForReading, nil, nil);
            break;
        }
        
        //////////////////////////////////////////////////
        
        if((rootInfoModel = [wrapper copyRootInfo]) == nil)
        {
            returnError = PPErrorMake(WCXFDataController_Error_FailedToReadRootInfo, nil, nil);
            break;
        }
        
        //////////////////////////////////////////////////
        // read each card
        
        if([wrapper readCardInfoWithTotalCount:rootInfoModel.cardCount error:&returnError] == NO)
        {
            returnError = PPErrorMake(WCXFDataController_Error_FailedToReadCardInfo, nil, returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        result = YES;
    }
    while(0);
    
    [rootInfoModel release];
    [wrapper closeFile];
    [wrapper release];
    
    //////////////////////////////////////////////////
    
    if(error != nil)
    {
        *error = returnError;
    }
    
    return result;
}



//================================================================================
//
//================================================================================
- (BOOL)writeFileWithPath:(NSString *)filePath cardIDs:(NSArray *)cardIDs error:(NSError **)error
{
    BOOL result = NO;
    NSError *returnError = nil;
    WCXFWrapper *wrapper = nil;
    NSInteger totalCount = [cardIDs count];
    NSInteger writtenCount = 0;
    
    do
    {
        //////////////////////////////////////////////////
        
        if(totalCount == 0)
        {
            returnError = PPErrorMake(WCXFDataController_Error_NoCardToWrite, nil, nil);
            break;
        }
        
        
        //////////////////////////////////////////////////
        // 一開始先檢查delegate的目的是減少檢查次數，同時提示此流程會使用的delegate method。
        
        if([self.delegate respondsToSelector:@selector(wcxfDataContrller:shouldWriteCardWithIndex:totalCount:)] == NO ||
           [self.delegate respondsToSelector:@selector(wcxfDataContrller:cardModelToWriteWithCardID:)] == NO ||
           [self.delegate respondsToSelector:@selector(wcxfDataContrller:groupNameToWriteWithGroupID:)] == NO ||
           [self.delegate respondsToSelector:@selector(wcxfDataContrller:imageDataToWriteWithCardID:imageType:)] == NO ||
           [self.delegate respondsToSelector:@selector(wcxfDataContrller:didWriteCardWithIndex:totalCount:)] == NO)
        {
            returnError = PPErrorMake(WCXFDataController_Error_MustHaveDelegateMethod, nil, nil);
            break;
        }
        
        //////////////////////////////////////////////////

        if((wrapper = [[WCXFWrapper alloc] initWithDelegate:nil]) == nil)
        {
            returnError = PPErrorMake(WCXFDataController_Error_FailedToInitObject, nil, nil);
            break;
        }
        
        wrapper.productLine = staticProductLine;
        
        if([wrapper openFileForWritingWithPath:filePath] == NO)
        {
            returnError = PPErrorMake(WCXFDataController_Error_FailedToOpenFileForWriting, nil, nil);
            break;
        }
        
        //////////////////////////////////////////////////

        if([wrapper writeRootInfoWithFilePath:filePath cardCount:totalCount] == NO)
        {
            returnError = PPErrorMake(WCXFDataController_Error_FailedToWriteRootInfo, nil, nil);
            break;
        }
        
        
        // !! 這邊要多等一秒，是為了確保自己匯出的WCXF可以被自己的WCXF匯入規則給過濾掉
//        [NSThread sleepForTimeInterval:1];
        
        //////////////////////////////////////////////////
        // write each card

        for(int i=0; i<totalCount; i++)
        {
            if([self.delegate wcxfDataContrller:self
                       shouldWriteCardWithIndex:i
                                     totalCount:totalCount]==NO)
            {
                result = NO;
                returnError = PPErrorMake(WCXFDataController_Error_CancelWritingCard, @"cancel writing card", nil);
                break;
            }
            
            //////////////////////////////////////////////////

            @autoreleasepool
            {     
                NSString *cardID = [cardIDs objectAtIndex:i];
                WCCardModel *cardModel = [self.delegate wcxfDataContrller:self cardModelToWriteWithCardID:cardID];
                NSDictionary *imageDataDict = [self copyImageDataDictWithCardModel:cardModel];
                
                // 有資料才能寫入
                if([cardModel hasFieldData] == YES || [imageDataDict count] > 0)
                {
                    NSArray *defaultGroupNames = [self copyDefaultGroupNamesWithCardModel:cardModel];
                    NSArray *definedGroupNames = [self copyDefinedGroupNamesWithCardModel:cardModel];
                    
                    result = [wrapper writeCardInfoWithCardModel:cardModel defaultGroupNames:defaultGroupNames definedGroupNames:definedGroupNames imageDataDict:imageDataDict];
                    
                    [defaultGroupNames release];
                    [definedGroupNames release];
                    [imageDataDict release];
                    
                    if(result == YES)
                    {
                        writtenCount++;
                        [self.delegate wcxfDataContrller:self didWriteCardWithIndex:i totalCount:totalCount];
                    }
                    else
                    {
                        break;
                    }
                }
                else
                {
                    // 寫入時如果已經取不到資料（可能被同步動作刪除），當作成功處理。
                    result = YES;
                    [imageDataDict release];
                }
            }
        }

        if(result == NO &&
           returnError==nil)
        {
            returnError = PPErrorMake(WCXFDataController_Error_FailedToWriteCardInfo, nil, nil);
            break;
        }
    }
    while(0);
    
    [wrapper closeFile];
    [wrapper release];
    
    //////////////////////////////////////////////////
    
    // 失敗時把檔案刪除
    if(result == NO)
    {
       [[NSFileManager defaultManager] removeItemAtPath:filePath error:nil];
    }
    else if(writtenCount == 0)
    {
        [[NSFileManager defaultManager] removeItemAtPath:filePath error:nil];
        result = NO;
        returnError = PPErrorMake(WCXFDataController_Error_NoCardToWrite, nil, nil);
    }

    if(error != nil)
    {
        *error = returnError;
    }
    
    return result;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class methods


//==============================================================================
//
//==============================================================================
+ (void)setProductLine:(WCXFDataController_ProductLine)productLine
{
    staticProductLine = productLine;
}







@end
