//
//  WCTSyncActionController.m
//  Pods
//
//  Created by Eddie Hu on 2016/4/11.
//
//

#import "WCTSyncActionController.h"
#import "CrossPlatformDefine.h"

// define
#import "WCSettingsKey.h"
#import "WCTSettingsKey.h"
#import "WCSyncFlowDefine.h"

// category
#import "NSMutableDictionary+NSInteger.h"
#import "NSData+Base64.h"
#import "NSString+Additions.h"
#import "NSFileManager+Additions.h"
#import "WCTRestClient+ParameterDefine.h"
#import "WCTRestClientController+ErrorCodeDefine.h"
#import "WCTRestClientController+Retry.h"
#import "WCCardModel+WCTRCContactInfo.h"
#import "WCGroupModel+WCTRCSimpleCategoryInfo.h"

// model
#import "WCTRCSyncInfo.h"
#import "WCTRCCategoryCompareAction.h"
#import "WCTRCContactCompareAction.h"
#import "WCTGroupSyncActionModel.h"
#import "WCTCardSyncActionModel.h"

// controller
#import "PPSettingsController.h"
#import "WCTRestClientController.h"
#import "WCTDataController.h"
#import "WCTAccountDataController.h"



////////////////////////////////////////////////////////////////////////////////////////////////////

static NSString * const WCTSyncActionControllerSettingsKey_IsCategoryFirstSync = @"WCTSyncActionControllerSettingsKey_IsCategoryFirstSync";
static NSString * const WCTSyncActionControllerSettingsKey_IsContactFirstSync = @"WCTSyncActionControllerSettingsKey_IsContactFirstSync";
static NSString * const WCTSyncActionControllerSettingsKey_CategoryLastRecTime = @"WCTSyncActionControllerSettingsKey_CategoryLastRecTime";
static NSString * const WCTSyncActionControllerSettingsKey_ContactLastRecTime = @"WCTSyncActionControllerSettingsKey_ContactLastRecTime";
static NSString * const WCTSyncActionControllerSettingsKey_ForceFirstSync = @"WCTSyncActionControllerSettingsKey_ForceFirstSync";
//static NSString * const WCTSyncActionControllerSettingsKey_FirstStartSyncTime = @"WCTSyncActionControllerSettingsKey_FirstStartSyncTime";

////////////////////////////////////////////////////////////////////////////////////////////////////

#define PPSyncRecordTableName   @"SyncRecordTable"

NSInteger const WCTSync_SyncMaxRetryCount = 10;
CGFloat const WCTSync_StartSyncStage1_MaxCount = 10.0;  // start sync stage1是1~10
CGFloat const WCTSync_StartSyncStage2_MaxCount = 70; // start sync stage2最大到11~80

/// local disk最小保留空間
unsigned long long const WCTSync_MinDiskSpace = WCDC_MinDiskSpace;

////////////////////////////////////////////////////////////////////////////////////////////////////

typedef NS_OPTIONS(NSInteger, SyncUpdateAction)
{
    SyncUpdateAction_Unknown,
    SyncUpdateAction_AddServer,
    SyncUpdateAction_AddOrUpdateClient,
    SyncUpdateAction_UpdateServer,
    SyncUpdateAction_UpdateClient,
    SyncUpdateAction_DeleteServer,
    SyncUpdateAction_DeleteClient,
    SyncUpdateAction_Unckeck,
    SyncUpdateAction_DoNothing
};

typedef NS_OPTIONS(NSInteger, SyncUpdateComponent)
{
    SyncUpdateComponent_Unknown,
    SyncUpdateComponent_Group,
    SyncUpdateComponent_Content,
    SyncUpdateComponent_SharedAccount,
    SyncUpdateComponent_FrontSideImage,
    SyncUpdateComponent_BackSideImage,
    SyncUpdateComponent_IDPhoto
};

typedef NS_ENUM(NSUInteger, LogState)
{
    LogState_Normal = 0,
    LogState_MethodIn,
    LogState_MethodOut
};

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface WCTSyncActionController ()
@property (nonatomic, assign) WCTRestClientController *restClientController;
@property (nonatomic, retain) WCTDataController *dataController;

@property (nonatomic, retain) NSMutableDictionary *localGroupIDAndGuidMappingDict;
@property (nonatomic, retain) NSString *favoriteCategoryGuid;
@property (nonatomic, retain) NSString *selfAccountGuid;
@property (nonatomic, retain) NSString *bossAccountGuid;

@property (nonatomic, assign) NSInteger uploadImageCountForLog;
@property (nonatomic, assign) NSInteger downloadImageCountForLog;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation WCTSyncActionController

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods

//================================================================================
//
//================================================================================
- (id)init
{
    if(self = [super init])
    {
        self.restClientController = [WCTRestClientController shareRestClientController];
        self.dataController = [[[WCTDataController alloc] initWithAccessMode:WCDC_AM_All] autorelease];
        
        // default settings
        [PPSettingsController setDefaultIntegerValue:1 withKey:WCTSyncActionControllerSettingsKey_IsCategoryFirstSync];
        [PPSettingsController setDefaultIntegerValue:1 withKey:WCTSyncActionControllerSettingsKey_IsContactFirstSync];
        
        [PPSettingsController setDefaultIntegerValue:0 withKey:WCTSyncActionControllerSettingsKey_ForceFirstSync];

    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    self.restClientController = nil;
    self.dataController = nil;
    
    self.localGroupIDAndGuidMappingDict = nil;
    self.favoriteCategoryGuid = nil;
    self.selfAccountGuid = nil;
    self.bossAccountGuid = nil;
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private - Common Methods

//==============================================================================
//
//==============================================================================
- (void)dumpLogMessageWithState:(LogState)state format:(NSString *)format, ...
{
    @autoreleasepool
    {
        va_list args;
        va_start(args,format);
        
        NSString *message = [[[NSString alloc] initWithFormat:format arguments:args] autorelease];
        
        switch(state)
        {
            case LogState_MethodIn:
                message = [@"##1## ▶▶ " stringByAppendingString:message];
                break;
                
            case LogState_MethodOut:
                message = [@"##1## ◀◀ " stringByAppendingString:message];
                break;
                
            default:
                message = [@"##1##    " stringByAppendingString:message];
                break;
        }
        
        [self logMessage:message sender:self];
        
        va_end(args);
    }
}


//==============================================================================
//
//==============================================================================
- (NSDate*)defaultTestingDate
{
    return [NSDate dateWithTimeIntervalSince1970:11*60*60+22*60+33.999];
}


//================================================================================
//
//================================================================================
- (void)reportProgress:(CGFloat)progress
{
    if([self.delegate respondsToSelector:@selector(ppSyncActionController:progress:)] == YES)
    {
        [self.delegate ppSyncActionController:self progress:progress];
    }
}


//================================================================================
//
//================================================================================
- (void)reportStep:(WCTSyncActionControllerStep)step
{
    if([self.delegate respondsToSelector:@selector(ppSyncActionController:step:)] == YES)
    {
        [self.delegate ppSyncActionController:self step:(PPSyncActionStep)step];
    }
}


//==============================================================================
// 依error判斷要繼續還是中斷
//==============================================================================
- (BOOL)continueSyncWithError:(NSError *)error
{
    // !!網路問題要直接中斷
    if ([error findNetworkError]!=nil)
    {
        return NO;
    }
    else
    {
        // 取得server回傳的status code
        NSInteger statusCode = 0;
        
        statusCode = [WCTRestClientController statusCodeFromAFRKNetworkingError:error];
        
        // 422要繼續
        if (statusCode == WCTServer_Common_ErrorCode_InvalidParameter)
        {
            return YES;
        }
    }
    return NO;
}


//==============================================================================
//
//==============================================================================
- (NSArray *)sortedListWithContactActionList:(NSArray *)contactActionList
{
    // !! action list要先處理
    // 1. 同一名片只保留一個action, 以時間為準，保留新的
    // 2. Del_Server最先
    // 3. uncheck在所有action之後，Add_Server之前
    // 4. Add_Server最後做
    
    NSMutableArray *tempActionList = [[[NSMutableArray alloc] init] autorelease];
    [tempActionList addObjectsFromArray:contactActionList];
    
    //////////////////////////////////////////////////
    // 依contactGuid排序，同guid依lastModifyTime排, 時間新的排前面
    [tempActionList sortUsingComparator:^NSComparisonResult(WCTRCContactCompareAction *obj1, WCTRCContactCompareAction *obj2) {
        
        if ([obj1.contactGuid isEqualToString:obj2.contactGuid]==YES)
        {
            // 沒有lastModifyTime的排前面
            if (obj1.lastModifyTime==nil)
            {
                return NSOrderedDescending;
            }
            
            if (obj2.lastModifyTime==nil)
            {
                return NSOrderedDescending;
            }
            
            // 時間新的排前面
            return [obj2.lastModifyTime compare:obj1.lastModifyTime];
        }
        else
        {
            return [obj1.contactGuid compare:obj2.contactGuid options:NSCaseInsensitiveSearch];
        }
    }];
    
    //////////////////////////////////////////////////
    // 把Add_server, Del_Server獨立出來
    NSMutableArray *containedGuids = [[NSMutableArray alloc] init];
    NSMutableArray *addServerList = [[NSMutableArray alloc] init];
    NSMutableArray *delServerList = [[NSMutableArray alloc] init];
    NSMutableArray *uncheckList = [[NSMutableArray alloc] init];
    NSMutableArray *otherActionList = [[NSMutableArray alloc] init];
    
    for (WCTRCContactCompareAction *contactAction in tempActionList)
    {
        // 如果已經加過就不用再加
        if ([containedGuids containsObject:contactAction.contactGuid]==YES)
        {
            continue;
        }
        
        // 依action分類
        SyncUpdateAction updateAction = [self updateActionWithServerDefinedString:contactAction.updateAction];
        
        switch (updateAction)
        {
                case SyncUpdateAction_AddServer:
            {
                [addServerList addObject:contactAction];
                break;
            }
                case SyncUpdateAction_DeleteServer:
            {
                [delServerList addObject:contactAction];
                break;
            }
                case SyncUpdateAction_Unckeck:
            {
                [uncheckList addObject:contactAction];
                break;
            }
            default:
            {
                [otherActionList addObject:contactAction];
                break;
            }
        }
        
        [containedGuids addObject:contactAction.contactGuid];
    }
    
    //////////////////////////////////////////////////
    // 合併結果
    NSMutableArray *contactCompareActionList = [[[NSMutableArray alloc] init] autorelease];
    [contactCompareActionList addObjectsFromArray:delServerList];
    [contactCompareActionList addObjectsFromArray:otherActionList];
    [contactCompareActionList addObjectsFromArray:uncheckList];
    [contactCompareActionList addObjectsFromArray:addServerList];
    
    
    [containedGuids release];
    [addServerList release];
    [delServerList release];
    [uncheckList release];
    [otherActionList release];
    
    return contactCompareActionList;
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private - Convert Methods

//==============================================================================
//
//==============================================================================
- (NSString *)serverDefinedStringFromUpdateAction:(SyncUpdateAction)updateAction
{
    NSString *definedString = nil;
    switch (updateAction)
    {
            case SyncUpdateAction_AddServer:
        {
            definedString = WCTRC_UpdateAction_AddServer;
            break;
        }
            case SyncUpdateAction_AddOrUpdateClient:
        {
            definedString = WCTRC_UpdateAction_AddOrUpdateClient;
            break;
        }
            case SyncUpdateAction_UpdateServer:
        {
            definedString = WCTRC_UpdateAction_UpdateServer;
            break;
        }
            case SyncUpdateAction_UpdateClient:
        {
            definedString = WCTRC_UpdateAction_UpdateClient;
            break;
        }
            case SyncUpdateAction_DeleteServer:
        {
            definedString = WCTRC_UpdateAction_DeleteServer;
            break;
        }
            case SyncUpdateAction_DeleteClient:
        {
            definedString = WCTRC_UpdateAction_DeleteClient;
            break;
        }
            case SyncUpdateAction_Unckeck:
        {
            definedString = WCTRC_UpdateAction_Unckeck;
            break;
        }
            case SyncUpdateAction_DoNothing:
        {
            definedString = WCTRC_UpdateAction_DoNothing;
            break;
        }
        default:
            break;
    }
    
    return definedString;
}


//==============================================================================
//
//==============================================================================
- (SyncUpdateAction)updateActionWithServerDefinedString:(NSString *)definedString
{
    SyncUpdateAction updateAction = SyncUpdateAction_Unknown;
    
    if([definedString isEqualToString:WCTRC_UpdateAction_AddServer] == YES)
    {
        updateAction = SyncUpdateAction_AddServer;
    }
    else if([definedString isEqualToString:WCTRC_UpdateAction_AddOrUpdateClient] == YES)
    {
        updateAction = SyncUpdateAction_AddOrUpdateClient;
    }
    else if([definedString isEqualToString:WCTRC_UpdateAction_UpdateServer] == YES)
    {
        updateAction = SyncUpdateAction_UpdateServer;
    }
    else if([definedString isEqualToString:WCTRC_UpdateAction_UpdateClient] == YES)
    {
        updateAction = SyncUpdateAction_UpdateClient;
    }
    else if([definedString isEqualToString:WCTRC_UpdateAction_DeleteServer] == YES)
    {
        updateAction = SyncUpdateAction_DeleteServer;
    }
    else if([definedString isEqualToString:WCTRC_UpdateAction_DeleteClient] == YES)
    {
        updateAction = SyncUpdateAction_DeleteClient;
    }
    else if([definedString isEqualToString:WCTRC_UpdateAction_Unckeck] == YES)
    {
        updateAction = SyncUpdateAction_Unckeck;
    }
    else if([definedString isEqualToString:WCTRC_UpdateAction_DoNothing] == YES)
    {
        updateAction = SyncUpdateAction_DoNothing;
    }
    return updateAction;
}


//==============================================================================
//
//==============================================================================
- (SyncUpdateComponent)updateComponentWithServerDefinedString:(NSString *)definedString
{
    SyncUpdateComponent updateComponent = SyncUpdateComponent_Unknown;
    
    if([definedString isEqualToString:WCTRC_UpdateComponent_Category] == YES)
    {
        updateComponent = SyncUpdateComponent_Group;
    }
    else if([definedString isEqualToString:WCTRC_UpdateComponent_Content] == YES)
    {
        updateComponent = SyncUpdateComponent_Content;
    }
    else if([definedString isEqualToString:WCTRC_UpdateComponent_ShareAccount] == YES)
    {
        updateComponent = SyncUpdateComponent_SharedAccount;
    }
    
    else if([definedString isEqualToString:WCTRC_UpdateComponent_FrontImage] == YES)
    {
        updateComponent = SyncUpdateComponent_FrontSideImage;
    }
    else if([definedString isEqualToString:WCTRC_UpdateComponent_BackImage] == YES)
    {
        updateComponent = SyncUpdateComponent_BackSideImage;
    }
    else if([definedString isEqualToString:WCTRC_UpdateComponent_Logo] == YES)
    {
        updateComponent = SyncUpdateComponent_IDPhoto;
    }
    
    return updateComponent;
}


//==============================================================================
//
//==============================================================================
- (WC_ImageType)imageTypeWithServerDefinedString:(NSString *)definedString
{
    WC_ImageType imageType = WC_IT_None;
    
    if([definedString isEqualToString:WCTRC_ImageType_Front] == YES)
    {
        imageType = WC_IT_FrontSide;
    }
    else if([definedString isEqualToString:WCTRC_ImageType_Rear] == YES)
    {
        imageType = WC_IT_BackSide;
    }
    else if([definedString isEqualToString:WCTRC_ImageType_Logo] == YES)
    {
        imageType = WC_IT_IDPhoto;
    }
    
    return imageType;
}


//==============================================================================
//
//==============================================================================
- (NSString *)serverDefinedStringWithSyncActionType:(WCTSyncActionType)type
{
    NSString *definedString = @"";
    
    switch (type)
    {
            case WCTSyncActionType_Add: definedString = WCTRC_Action_Add; break;
            case WCTSyncActionType_Modify: definedString = WCTRC_Action_Modify; break;
            case WCTSyncActionType_Delete: definedString = WCTRC_Action_Delete; break;
        default: break;
    }
    
    return definedString;
}



//==============================================================================
//
//==============================================================================
- (WC_GroupID)groupIDWithMappedGuid:(NSString *)groupGuid
{
    @synchronized (self)
    {
        NSArray *groupGuids = [self.localGroupIDAndGuidMappingDict allValues];
        NSArray *groupIDs = [self.localGroupIDAndGuidMappingDict allKeys];
        
        WC_GroupID groupID = WC_GID_None;
        NSUInteger index = [groupGuids indexOfObject:groupGuid];
        
        if(index != NSNotFound)
        {
            groupID = [[groupIDs objectAtIndex:index] integerValue];
        }
        
        return groupID;
    }
}


//==============================================================================
//
//==============================================================================
- (NSString *)categoryGuidFromGroupID:(WC_GroupID)groupID
{
    @synchronized (self)
    {
        NSString *groupGuid = [self.localGroupIDAndGuidMappingDict objectForKey:@(groupID)];
        
        if([groupGuid length]==0)
        {
            // 如果group存在，要更新mappingDict
            if ([[self.dataController groupNameWithID:[NSString stringWithInteger:groupID]] length]>0)
            {
                self.localGroupIDAndGuidMappingDict =[NSMutableDictionary dictionaryWithDictionary:[self.dataController groupIDAndGuidMappingDict]];
            }
            
            groupGuid = [self.localGroupIDAndGuidMappingDict objectForKey:@(groupID)];
        }
        
        return groupGuid;
    }
}


//==============================================================================
//
//==============================================================================
- (NSMutableArray *)categoryGuidsFromGroupIDs:(NSArray *)groupIDs
{
    @synchronized (self)
    {
        NSMutableArray *groupGuids = [NSMutableArray array];
        
        // !! server不需要的group在這裡濾除
        for(NSString *groupIDString in groupIDs)
        {
            WC_GroupID groupID = [groupIDString integerValue];
            
            if(groupID == WC_GID_All)
            {
                continue;
            }
            
            NSString *groupGuid = [self categoryGuidFromGroupID:groupID];
            
            if([groupGuid length] > 0)
            {
                [groupGuids addObject:groupGuid];
            }
        }
        
        return groupGuids;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private - check should sync action been excutes methods


//==============================================================================
//
//==============================================================================
- (BOOL)contactChangedWithCardID:(NSString *)cardID afterDate:(NSDate *)date
{
    WCTCardSyncActionModel *currentCardSyncActionModel = [[self.dataController copyCardSyncActionModelWithCardID:cardID] autorelease];
    
    if(date !=nil && currentCardSyncActionModel!=nil)
    {
        // !! 如果時間與startSync時不一樣，就不處理這一個動作
        if ([date isEqual:currentCardSyncActionModel.modifiedTime]==NO)
        {
            return YES;
        }
    }
    
    return NO;
}


//==============================================================================
//
//==============================================================================
- (BOOL)shouldExecuteLocalActionWithCardID:(NSString *)cardID actionDate:(NSDate *)actionDate
{
    WCTCardSyncActionModel *currentCardSyncActionModel = [[self.dataController copyCardSyncActionModelWithCardID:cardID] autorelease];
    
    if(actionDate !=nil && currentCardSyncActionModel!=nil)
    {
        // dirty table比較新就不能做
        if ([actionDate compare:currentCardSyncActionModel.modifiedTime]==NSOrderedAscending)
        {
            return NO;
        }
    }
    
    return YES;
}


//==============================================================================
//
//==============================================================================
- (BOOL)shouldExecuteRemoteActionWithCardID:(NSString *)cardID actionDate:(NSDate *)actionDate
{
    WCTCardSyncActionModel *currentCardSyncActionModel = [[self.dataController copyCardSyncActionModelWithCardID:cardID] autorelease];
    
    if(actionDate !=nil && currentCardSyncActionModel!=nil)
    {
        // dirty table比較新就不能做
        if ([actionDate compare:currentCardSyncActionModel.modifiedTime]==NSOrderedDescending)
        {
            return NO;
        }
    }
    
    return YES;
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private - Remote data access methods



//================================================================================
//
//================================================================================
- (WCTRCDetailContactInfo *)remoteContactInfoWithGuid:(NSString *)contactGuid error:(NSError **)error
{
    NSError *returnError = nil;
    WCTRCDetailContactInfo * contactInfo = nil;
    
    do
    {
        if([contactGuid length] == 0)
        {
            returnError = PPErrorMake(WCTSyncActionControllerError_WrongParameter, @"contactGuid is empty", nil);
            break;
        }
        
        //////////////////////////////////////////////////
        
        WCTRCDetailContactInfoResponseResult *response = [self syncGetContactWithContactGuid:contactGuid error:&returnError];
        
        if(returnError != nil)
        {
            returnError = PPErrorMake(WCTSyncActionControllerError_OperationFailed, response.message, returnError);
            break;
        }
        
        contactInfo = response.data;
        
    }
    while(0);
    
    //////////////////////////////////////////////////
    
    if(error != nil)
    {
        *error = returnError;
    }
    
    return contactInfo;
}


//================================================================================
//
//================================================================================
- (CPImage *)remoteContactImageWithGuid:(NSString *)contactGuid imageType:(NSString *)imageType error:(NSError **)error
{
    NSError *returnError = nil;
    
    self.downloadImageCountForLog++;
    
    NSData *imageData = [self privateContactImageDataWithContactGuid:contactGuid
                                                           imageType:imageType
                                                               error:&returnError];
    
    CPImage *image = [[[CPImage alloc] initWithData:imageData] autorelease];
    
    if(error != nil)
    {
        *error = returnError;
    }
    
    // MARK: 用base64 string下載影像(較慢)
    //    WCTRCPrivateContactImageBase64ResponseResult *response = [self.restClientController privateContactImageBase64WithContactGuid:contactGuid
    //                                                                                                                       imageType:imageType
    //
    //                                                                                                                           error:&returnError];
    //    if(error != nil)
    //    {
    //        *error = returnError;
    //    }
    //
    //    CPImage *image = [UIImage imageWithData:[NSData dataWithBase64EncodedStandardString:response.data]];
    
    return image;
}


//================================================================================
//
//================================================================================
- (NSString *)remoteUploadContactImage:(CPImage *)image withGuid:(NSString *)contactGuid error:(NSError **)error
{
    __block NSError *returnError = nil;
    NSString *imageGuid = nil;
    
    self.uploadImageCountForLog++;
    
    __block WCTRCPrivateContactUploadImageResponseResult *response = nil;
    [WCTRestClientController executeProcessHandler:^NSError *(NSInteger retryCount) {
        
        NSError *error = nil;
        
        @autoreleasepool {
            
            [response release];
            response = [[self.restClientController privateContactUploadImage:image
                                                                    fileName:contactGuid
                                                                    mineType:WCTRC_MineType_JPG
                                                                       error:&error] retain];
            [error retain];
            [self dumpLogMessageWithState:LogState_Normal format:@"#### remoteUploadContactImage (try: %ld, error:%@) ", (long)retryCount, error];
        }
        return error;
    }
                                   withShouldRetry:^BOOL(NSInteger retryCount, NSError *error)
     {
         
         // 取消就不retry
         if(cancel_ == YES)
         {
             // 在外部檢查是否cancel再產生error code
             return NO;
         }
         
         //成功就不retry
         if (error==nil)
         {
             return NO;
         }
         else
         {
             // !! 如果是網路問題或timeout才要retry,
             if ([error findNetworkError]!=nil &&
                 retryCount < WCTSync_SyncMaxRetryCount)
             {
                 return YES;
             }
         }
         
         returnError = [error retain];
         return NO;
     }];
    
    [response autorelease];
    
    //////////////////////////////////////////////////
    if(error != nil)
    {
        *error = returnError;
    }
    
    if(returnError == nil)
    {
        imageGuid = response.data;
    }
    
    return imageGuid;
}


//================================================================================
//
//================================================================================
- (BOOL)remoteIsFavoriteWithContactInfo:(WCTRCContactInfo *)contactInfo
{
    for(WCTRCCategoryInfo *categoryInfo in contactInfo.belongCategories)
    {
        if([categoryInfo.guid isEqualToString:self.favoriteCategoryGuid] == YES)
        {
            return YES;
        }
    }
    
    return NO;
}


//==============================================================================
//
//==============================================================================
- (WCTCardSyncActionModel *)remoteCardSyncActionModelWithProfile:(WCTRCContactProfile *)profile
                                                      actionType:(WCTSyncActionType)actionType
{
    WCTCardSyncActionModel *syncActionModel = [[[WCTCardSyncActionModel alloc] init] autorelease];
    
    if(syncActionModel==nil)
    {
        return nil;
    }
    
    syncActionModel.dataGuid = profile.guid;
    syncActionModel.actionType = WCTSyncActionType_Modify;
    syncActionModel.syncState = WCTSyncState_Synced;
    syncActionModel.modifiedTime = profile.syncModifyTime;
    
    syncActionModel.groupSHA1 = profile.categorySha1;
    syncActionModel.sharedAccountSHA1 = profile.sharedAccountSha1;
    syncActionModel.contentSHA1 = profile.textSha1;
    syncActionModel.frontSideSHA1 = profile.frontImageSha1;
    syncActionModel.backSideSHA1 = profile.rearImageSha1;
    syncActionModel.IDPhotoSHA1 = profile.logoImageSha1;
    
    return syncActionModel;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private - Remote sync access methods



//================================================================================
//
//================================================================================
- (NSError *)_remoteAddContactWithAction:(WCTRCContactCompareAction *)action
                             accountGuid:(NSString *)accountGuid
{
    __block NSError *returnError = nil;
    
    do
    {
        if([accountGuid length] == 0)
        {
            returnError = PPErrorMake(WCTSyncActionControllerError_WrongParameter, @"wrong accountGuid", nil);
            break;
        }
        
        // 取得此名片的同步資料
        WCTCardSyncActionModel *cardSyncActionModel = [[self.dataController copyCardSyncActionModelWithCardID:action.contactGuid] autorelease];
        
        if(cardSyncActionModel == nil)
        {
            returnError = PPErrorMake(WCTSyncActionControllerError_InitObjectFailed, @"copyCardSyncActionModelWithCardID failed", self.dataController.lastError);
            break;
        }
        
        // 檢查server上是否已有相同guid的名片
        WCTRCBooleanResponseResult *isExistResult = [self syncIsExistContactWithContactGuid:action.contactGuid
                                                                                accountGuid:accountGuid
                                                                                      error:&returnError];
        
        if(cancel_ == YES)
        {
            // 在autorelease pool外不用在retain
            returnError = PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil);
            break;
        }
        
        if(returnError != nil)
        {
            returnError = PPErrorMake(WCTSyncActionControllerError_OperationFailed, @"syncIsExistContactWithContactGuid failed", returnError);
            break;
        }
        
        // 如果已有當成同步成功，不用做同步動作
        if(isExistResult.data==YES)
        {
            // !! 如果同步成功，同步紀錄改為None
            [self localResetCardSyncActionWithGuid:action.contactGuid checkModifyTime:cardSyncActionModel.modifiedTime];
            [self dumpLogMessageWithState:LogState_Normal format:@"#### syncIsExistContactWithContactGuid (Card exist) "];
            break;
        }
        
        //////////////////////////////////////////////////
        
        WCTRCContactCreationWithImageRequest *request = [[[WCTRCContactCreationWithImageRequest alloc] init] autorelease];
        
        if(request == nil)
        {
            returnError = PPErrorMake(WCTSyncActionControllerError_InitObjectFailed, @"WCTContactCreationWithImageRequest init failed", nil);
            break;
        }
        
        
        
        //////////////////////////////////////////////////
        // 先取得local card資料，避免取得失敗時，上傳的圖片會變無法處理
        
        WCCardModel *localCardModel = [self localCardModelWithCardID:action.contactGuid error:&returnError];
        
        // 取消
        if(cancel_ == YES)
        {
            // 在autorelease pool外不用在retain
            returnError = PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil);
            break;
        }
        
        
        // !! 無法取得時先跳過這一筆，
        if(localCardModel == nil)
        {
            returnError = nil;
            break;
        }
        
        
        //////////////////////////////////////////////////
        // 上傳影像 (正面要排除假名片圖)
        
        CPImage *image = [self localCardImageWithCardID:action.contactGuid imageType:WC_IT_FrontSide];
        
        if(image != nil)
        {
            request.frontImageFileGuid = [self remoteUploadContactImage:image withGuid:action.contactGuid error:&returnError];
            
            // 取消
            if(cancel_ == YES)
            {
                // 在autorelease pool外不用在retain
                returnError = PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil);
                break;
            }
            
            if(returnError != nil)
            {
                returnError = PPErrorMake(WCTSyncActionControllerError_OperationFailed, @"upload front side image failed", returnError);
                break;
            }
        }
        
        
        //////////////////////////////////////////////////
        
        image = [self localCardImageWithCardID:action.contactGuid imageType:WC_IT_BackSide];
        
        if(image != nil)
        {
            request.rearImageFileGuid = [self remoteUploadContactImage:image withGuid:action.contactGuid error:&returnError];
            
            // 取消
            if(cancel_ == YES)
            {
                // 在autorelease pool外不用在retain
                returnError = PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil);
                break;
            }
            
            if(returnError != nil)
            {
                returnError = PPErrorMake(WCTSyncActionControllerError_OperationFailed, @"upload back side image failed", returnError);
                break;
            }
        }
        
        //////////////////////////////////////////////////
        
        image = [self localCardImageWithCardID:action.contactGuid imageType:WC_IT_IDPhoto];
        
        if(image != nil)
        {
            request.logoImageFileGuid = [self remoteUploadContactImage:image withGuid:action.contactGuid error:&returnError];
            
            // 取消
            if(cancel_ == YES)
            {
                // 在autorelease pool外不用在retain
                returnError = PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil);
                break;
            }
            
            if(returnError != nil)
            {
                returnError = PPErrorMake(WCTSyncActionControllerError_OperationFailed, @"upload IDPhoto side image failed", returnError);
                break;
            }
        }
        
        //////////////////////////////////////////////////
        // 取得group資料
        
        if([localCardModel.groupIDArray count] == 0)
        {
            returnError = PPErrorMake(WCTSyncActionControllerError_OperationFailed, @"get local card groups ID failed", nil);
            break;
        }
        
        NSMutableArray *groupGuids = [self categoryGuidsFromGroupIDs:localCardModel.groupIDArray];
        NSMutableArray *simpleCategories =[self localSimpleCategoryInfoWithGuids:groupGuids];
        
        // !! 處理favorite
        if(localCardModel.tagMask & WC_TagMask_Favorite)
        {
            WCTRCSimpleCategoryInfo *favoritesCategoty = [[WCTRCSimpleCategoryInfo alloc] init];
            favoritesCategoty.guid = self.favoriteCategoryGuid;
            [simpleCategories addObject:favoritesCategoty];
            [favoritesCategoty release];
        }
        
        request.belongingCategories = simpleCategories;
        
        // 取消
        if(cancel_ == YES)
        {
            // 在autorelease pool外不用在retain
            returnError = PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil);
            break;
        }
        
        //////////////////////////////////////////////////
        // 轉換contactInfo
        
        WCTRCContactInfo *contactInfo = [localCardModel WCTRCContactInfo];
        
        if(contactInfo == nil)
        {
            returnError = PPErrorMake(WCTSyncActionControllerError_OperationFailed, @"convert contactInfo from local card model failed", nil);
            break;
        }
        
        // !! special case
        request.shareToAccountGuids = localCardModel.sharedAccountGUIDArray;
        
        request.beCorrected = contactInfo.beCorrected;
        request.recogLanguageBack = contactInfo.recogLanguageBack;
        request.recogLanguageFront = contactInfo.recogLanguageFront;
        request.names = contactInfo.names;
        request.jobinfos = contactInfo.jobinfos;
        request.addresses = contactInfo.addresses;
        request.phones = contactInfo.phones;
        request.emails = contactInfo.emails;
        request.urls = contactInfo.urls;
        request.ims = contactInfo.ims;
        request.socials = contactInfo.socials;
        request.dates = contactInfo.dates;
        request.customFields = contactInfo.customFields;
        request.birthday = contactInfo.birthday;
        request.nickname = contactInfo.nickname;
        request.note = contactInfo.note;
        request.uniformNumber = contactInfo.uniformNumber;
        
        
        //////////////////////////////////////////////////
        // 上傳新增名片
        
        request.creatorAccountGuid = accountGuid;
        request.clientCreatedContactGuid = action.contactGuid;
        request.clientCreatedTime = localCardModel.createdTime;
        request.displayModifyTime = localCardModel.modifiedTime;
        
        WCTRCContactProfileResponseResult *response = [self.restClientController syncCreateContactWithRequest:request error:&returnError];
        // 取消
        if(cancel_ == YES)
        {
            // 在autorelease pool外不用在retain
            returnError = PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil);
            break;
        }
        
        if(returnError != nil)
        {
            returnError = PPErrorMake(WCTSyncActionControllerError_OperationFailed, @"syncCreateContactWithRequest failed", returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        // 更新local同步紀錄
        WCTCardSyncActionModel *syncActionModel = [self remoteCardSyncActionModelWithProfile:response.data
                                                                                  actionType:WCTSyncActionType_Add];
        
        
        if(syncActionModel == nil)
        {
            returnError = PPErrorMake(WCTSyncActionControllerError_InitObjectFailed, @"Init WCTCardSyncActionModel failed", nil);
            break;
        }
        
        if([self localUpdateCardSyncAction:syncActionModel withCheckModifyTime:cardSyncActionModel.modifiedTime] == NO)
        {
            returnError = PPErrorMake(WCTSyncActionControllerError_OperationFailed, @"update local sync action failed", nil);
            break;
        }
    }
    while(0);
    
    //////////////////////////////////////////////////
    
    return returnError;
}


//================================================================================
//
//================================================================================
- (NSError *)_remoteUpdateContactWithAction:(WCTRCContactCompareAction *)action accountGuid:(NSString *)accountGuid
{
    NSError *error = nil;
    
    do
    {
        // 取得此名片的同步資料
        WCTCardSyncActionModel *cardSyncActionModel = [[self.dataController copyCardSyncActionModelWithCardID:action.contactGuid] autorelease];
        
        if(cardSyncActionModel == nil)
        {
            error = PPErrorMake(WCTSyncActionControllerError_InitObjectFailed, @"copyCardSyncActionModelWithCardID failed", self.dataController.lastError);
            break;
        }
        
        //////////////////////////////////////////////////
        // 更新前要先取得更新項目
        WCTRCContactUpdateCheckRequest *updateCheckRequest = [[[WCTRCContactUpdateCheckRequest alloc] init] autorelease];
        if(updateCheckRequest == nil)
        {
            error = PPErrorMake(WCTSyncActionControllerError_InitObjectFailed, @"WCTRCContactUpdateCheckRequest init failed", error);
            break;
        }
        
        updateCheckRequest.accountGuid = accountGuid;
        updateCheckRequest.contactGuid = cardSyncActionModel.dataGuid;
        updateCheckRequest.contentSHA1 = cardSyncActionModel.contentSHA1;
        updateCheckRequest.sharedAccountSHA1 = cardSyncActionModel.sharedAccountSHA1;
        updateCheckRequest.categorySHA1 = cardSyncActionModel.groupSHA1;
        updateCheckRequest.frontImageSHA1 = cardSyncActionModel.frontSideSHA1;
        updateCheckRequest.backImageSHA1 = cardSyncActionModel.backSideSHA1;
        updateCheckRequest.logoSHA1 = cardSyncActionModel.IDPhotoSHA1;
        updateCheckRequest.lastModifyTime = cardSyncActionModel.modifiedTime;
        
        //////////////////////////////////////////////////
        
        WCTRCSyncContactUpdateCheckResponseResult *contactUpdateCheckResponseResult = [self syncContactUpdateCheckWithRequest:updateCheckRequest
                                                                                                                        error:&error];
        
        // 取消
        if(cancel_ == YES)
        {
            // 在autorelease pool外不用在retain
            error = PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil);
            break;
        }
        
        if(error != nil)
        {
            error = PPErrorMake(WCTSyncActionControllerError_OperationFailed, @"syncContactUpdateCheckWithRequest failed", error);
            break;
        }
        
        // !! 如果不是 WCTRC_ContactStatus_CLIENT_IS_NEWER, 這一次的動作就不處理
        if([contactUpdateCheckResponseResult.data.contactStatus isEqualToString:WCTRC_ContactStatus_CLIENT_IS_NEWER]==NO)
        {
            [self dumpLogMessageWithState:LogState_Normal format:@"#### syncContactUpdateCheckWithRequest (not Need to Update)"];
            break;
        }
        
        //////////////////////////////////////////////////
        NSMutableArray *updateComponents = [NSMutableArray array];
        
        // !! special case : 當updateCompomentList是空的代表全部更新（import資料後同步會出現此狀況）
        if([contactUpdateCheckResponseResult.data.updateComponents count] > 0)
        {
            [updateComponents addObjectsFromArray:contactUpdateCheckResponseResult.data.updateComponents];
        }
        else
        {
            [updateComponents addObjectsFromArray:@[WCTRC_UpdateComponent_Category,
                                                    WCTRC_UpdateComponent_Content,
                                                    WCTRC_UpdateComponent_ShareAccount,
                                                    WCTRC_UpdateComponent_FrontImage,
                                                    WCTRC_UpdateComponent_BackImage,
                                                    WCTRC_UpdateComponent_Logo]];
        }
        
        
        //////////////////////////////////////////////////
        
        WCTRCContactUpdateInfoRequest *request = [[[WCTRCContactUpdateInfoRequest alloc] init] autorelease];
        
        if(request == nil)
        {
            error = PPErrorMake(WCTSyncActionControllerError_InitObjectFailed, @"WCTRCContactUpdateInfoRequest init failed", nil);
            break;
        }
        
        // !! 一定要寫入
        request.syncModifyTime = cardSyncActionModel.modifiedTime;
        
        // 取消
        if(cancel_ == YES)
        {
            // 在autorelease pool外不用在retain
            error = PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil);
            break;
        }
        
        //////////////////////////////////////////////////
        BOOL isClientModifyTimeNeeded = NO;
        
        for(NSString *updateComponent in updateComponents)
        {
            switch ([self updateComponentWithServerDefinedString:updateComponent])
            {
                    case SyncUpdateComponent_Group:
                {
                    NSMutableArray *groupIDs = [self.dataController copyGroupIDArrayWithCardID:action.contactGuid];
                    NSMutableArray *categoryGuids = [self categoryGuidsFromGroupIDs:groupIDs];
                    NSMutableArray *simpleCategories =[self localSimpleCategoryInfoWithGuids:categoryGuids];
                    
                    // !! 處理favorite
                    if([self.dataController isFavorite:action.contactGuid])
                    {
                        WCTRCSimpleCategoryInfo *favoritesCategoty = [[WCTRCSimpleCategoryInfo alloc] init];
                        favoritesCategoty.guid = self.favoriteCategoryGuid;
                        [simpleCategories addObject:favoritesCategoty];
                        [favoritesCategoty release];
                    }
                    
                    request.belongingCategories = simpleCategories;
                    [groupIDs release];
                    break;
                }
                    
                    case SyncUpdateComponent_Content:
                {
                    WCCardModel *cardModel = [self localCardModelWithCardID:action.contactGuid error:&error];
                    
                    // !! 無法取得時先跳過這一筆，
                    if(error != nil)
                    {
                        error = nil;
                        break;
                    }
                    
                    //////////////////////////////////////////////////
                    
                    WCTRCContactInfo *contactInfo = [cardModel WCTRCContactInfo];
                    
                    request.recogLanguageFront = contactInfo.recogLanguageFront;
                    request.recogLanguageBack = contactInfo.recogLanguageBack;
                    
                    request.names = contactInfo.names;
                    request.jobinfos = contactInfo.jobinfos;
                    request.addresses = contactInfo.addresses;
                    request.phones = contactInfo.phones;
                    request.emails = contactInfo.emails;
                    request.urls = contactInfo.urls;
                    request.ims = contactInfo.ims;
                    request.socials = contactInfo.socials;
                    request.dates = contactInfo.dates;
                    request.customFields = contactInfo.customFields;
                    request.birthday = contactInfo.birthday;
                    request.nickname = contactInfo.nickname;
                    request.note = contactInfo.note;
                    request.uniformNumber = contactInfo.uniformNumber;
                    
                    isClientModifyTimeNeeded = YES;
                    break;
                }
                    case SyncUpdateComponent_SharedAccount:
                {
                    NSArray *sharedAccountGuids = [self localCardSharedAccountGuidsWithCardID:action.contactGuid error:&error];
                    if(error != nil)
                    {
                        error = PPErrorMake(WCTSyncActionControllerError_OperationFailed, @"remoteUpdateContactWithAction get local card shared Account guid failed", error);
                        break;
                    }
                    // !! special case
                    request.shareToAccountGuids = sharedAccountGuids;
                    
                    break;
                }
                    
                    case SyncUpdateComponent_FrontSideImage:
                {
                    CPImage *image = [self localCardImageWithCardID:action.contactGuid imageType:WC_IT_FrontSide];
                    
                    if(image != nil)
                    {
                        request.frontImageFileGuid = [self remoteUploadContactImage:image withGuid:action.contactGuid error:&error];
                    }
                    
                    isClientModifyTimeNeeded = YES;
                    break;
                }
                    
                    case SyncUpdateComponent_BackSideImage:
                {
                    CPImage *image = [self localCardImageWithCardID:action.contactGuid imageType:WC_IT_BackSide];
                    
                    if(image != nil)
                    {
                        request.rearImageFileGuid = [self remoteUploadContactImage:image withGuid:action.contactGuid error:&error];
                    }
                    
                    isClientModifyTimeNeeded = YES;
                    break;
                }
                    
                    case SyncUpdateComponent_IDPhoto:
                {
                    CPImage *image = [self localCardImageWithCardID:action.contactGuid imageType:WC_IT_IDPhoto];
                    
                    if(image != nil)
                    {
                        request.logoImageFileGuid = [self remoteUploadContactImage:image withGuid:action.contactGuid error:&error];
                    }
                    
                    isClientModifyTimeNeeded = YES;
                    break;
                }
                    
                default:
                    break;
            }
            
            if(error != nil)
            {
                break;
            }
        }
        
        // 取消
        if(cancel_ == YES)
        {
            // 在autorelease pool外不用在retain
            error = PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil);
            break;
        }
        
        // !! must have parameter
        request.contactGuid = action.contactGuid;
        request.categoryOwnerAccountGuid = accountGuid;
        request.updateComponents = updateComponents;
        
        // !! 未校正狀態一定要傳
        request.beCorrected = ![self.dataController isUnverifiedWithCardID:action.contactGuid];
        
        // !! 有UC_CONTENT/UC_LOGO/UC_FRONT_IMAGE/UC_BACK_IMAGE就要傳修改時間
        if(isClientModifyTimeNeeded)
        {
            request.displayModifyTime = [self.dataController cardModifiedTimeWithCardID:action.contactGuid];
        }
        
        // 取消
        if(cancel_ == YES)
        {
            // 在autorelease pool外不用在retain
            error = PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil);
            break;
        }
        
        
        //////////////////////////////////////////////////
        // !! 更新前再確認名片是否還在，如果不在就不處理
        WCCardModel *cardModel = [self.dataController copyCardDisplayDataWithCardID:action.contactGuid];
        
        if(cardModel == nil)
        {
            break;
        }
        
        [cardModel release];
        
        //////////////////////////////////////////////////
        WCTRCContactProfileResponseResult *response = [self.restClientController syncUpdateContactWithRequest:request error:&error];
        
        // 取消
        if(cancel_ == YES)
        {
            // 在autorelease pool外不用在retain
            error = PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil);
            break;
        }
        
        if(error != nil)
        {
            if ([WCTRestClientController statusCodeFromAFRKNetworkingError:error]!=WCTServer_Common_ErrorCode_ServerIsNewer)
            {
                error = PPErrorMake(WCTSyncActionControllerError_OperationFailed, @"syncUpdateContactWithRequest failed", error);
                break;
            }
            else
            {
                // !! 如果是462不當error, 但也不更新同步紀錄
                error = nil;
                break;
            }
        }
        
        //////////////////////////////////////////////////
        // 更新local同步紀錄
        WCTCardSyncActionModel *syncActionModel = [self remoteCardSyncActionModelWithProfile:response.data
                                                                                  actionType:WCTSyncActionType_Modify];
        
        
        if(syncActionModel == nil)
        {
            error = PPErrorMake(WCTSyncActionControllerError_InitObjectFailed, @"Init WCTCardSyncActionModel failed", nil);
            break;
        }
        
        // !! 要先檢查與開始時的actionModel是否一樣，不同表示同步開始後有修改過，所以只有相同時才能更新sync action
        if([self localUpdateCardSyncAction:syncActionModel withCheckModifyTime:cardSyncActionModel.modifiedTime] == NO)
        {
            error = PPErrorMake(WCTSyncActionControllerError_OperationFailed, @"update local sync action failed", nil);
            break;
        }
    }
    while(0);
    
    //////////////////////////////////////////////////
    
    return error;
}


//================================================================================
//
//================================================================================
- (NSError *)remoteDeleteContactWithAction:(WCTRCContactCompareAction *)action accountGuid:(NSString *)accountGuid
{
    NSError *error = nil;
    
    WCTRCSyncContactDeleteResponseResult *response = [self syncDeleteContactWithContactGuid:action.contactGuid
                                                                                accountGuid:accountGuid
                                                                                      error:&error];
    
    // 取消
    if(cancel_ == YES)
    {
        // 在autorelease pool外不用在retain
        error = PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil);
        return error;
    }
    
    if(error != nil)
    {
        error = PPErrorMake(WCTSyncActionControllerError_OperationFailed, response.message, error);
    }
    else
    {
        // 刪除動作結束就清除
        [self localDeleteCardSyncActionWithCardID:action.contactGuid];
    }
    
    return error;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private remote sync access (action retry)


//================================================================================
//
//================================================================================
- (NSError *)remoteAddContactWithAction:(WCTRCContactCompareAction *)action
                            accountGuid:(NSString *)accountGuid
{
    BOOL isDuplicateGuid = NO;
    BOOL isGuidChanged = NO;
    __block NSError *returnError = nil;
    
    NSString *oldGuid = action.contactGuid;
    
    do {
        
        [WCTRestClientController executeProcessHandler:^NSError *(NSInteger retryCount) {
            
            NSError *error = nil;
            
            @autoreleasepool {
                
                error = [self _remoteAddContactWithAction:action
                                              accountGuid:accountGuid];
                [error retain];
                
                [self dumpLogMessageWithState:LogState_Normal format:@"^^^^^ remoteAddContactWithAction:accountGuid: (try: %ld, error:%@) ", (long)retryCount, error];
            }
            return [error autorelease];
            
        } withShouldRetry:^BOOL(NSInteger retryCount, NSError *error) {
            
            // 取消就不retry
            if(cancel_ == YES)
            {
                // 在外部檢查是否cancel再產生error code
                return NO;
            }
            
            //成功就不retry
            if (error==nil)
            {
                return NO;
            }
            else
            {
                // !! 如果是網路問題或timeout才要retry,
                if ([error findNetworkError]!=nil &&
                    retryCount < WCTSync_SyncMaxRetryCount)
                {
                    return YES;
                }
            }
            
            returnError = [error retain];
            return NO;
        }];
        
        // 如果是490 guid重復，要用新的guid再上傳一次
        if(returnError!=nil)
        {
            NSInteger statusCode = 0;
            
            statusCode = [WCTRestClientController statusCodeFromAFRKNetworkingError:returnError];
            
            if(statusCode == WCTServer_Common_ErrorCode_GuidHasBeenUsed)
            {
                action.contactGuid = [NSString GUID];
                
                isDuplicateGuid = YES;
                isGuidChanged = YES;
            }
        }
        
    } while (isDuplicateGuid);
    
    if (isGuidChanged) {
        //////////////////////////////////////////////////
        // !! 如果guid不一樣表示有修改guid, 要更新到local database
        [self.dataController replaceGuidFromOld:oldGuid toNew:action.contactGuid];
    }
    
    return [returnError autorelease];
}


//================================================================================
//
//================================================================================
- (NSError *)remoteUpdateContactWithAction:(WCTRCContactCompareAction *)action
                               accountGuid:(NSString *)accountGuid
{
    __block NSError *returnError = nil;
    
    [WCTRestClientController executeProcessHandler:^NSError *(NSInteger retryCount) {
        
        NSError *error = nil;
        
        @autoreleasepool {
            
            error = [self _remoteUpdateContactWithAction:action
                                             accountGuid:accountGuid];
            [error retain];
            
            [self dumpLogMessageWithState:LogState_Normal format:@"^^^^^ remoteUpdateContactWithAction:accountGuid: (try: %ld, error:%@) ", (long)retryCount, error];
        }
        return [error autorelease];
        
    } withShouldRetry:^BOOL(NSInteger retryCount, NSError *error) {
        
        // 取消就不retry
        if(cancel_ == YES)
        {
            // 在外部檢查是否cancel再產生error code
            return NO;
        }
        
        //成功就不retry
        if (error==nil)
        {
            return NO;
        }
        else
        {
            // !! 如果是網路問題或timeout才要retry,
            if ([error findNetworkError]!=nil &&
                retryCount < WCTSync_SyncMaxRetryCount)
            {
                return YES;
            }
        }
        
        returnError = [error retain];
        return NO;
    }];
    
    
    return [returnError autorelease];
}



////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private - remote sync access contact (特殊處理)


//==============================================================================
//
//==============================================================================
- (WCTRCStartSyncResponseResult *)startSyncWithInfo:(WCTRCSyncInfo *)syncInfo
                                              error:(NSError **)error
{
    [self dumpLogMessageWithState:LogState_Normal format:@"#### startSync start"];
    
    __block NSError *returnError = nil;
    WCTRCStartSyncResponseResult *result = nil;
    NSMutableArray *categoryActionResult = [NSMutableArray array];
    NSMutableArray *contactActionResult = [NSMutableArray array];
    
    NSInteger startSyncCount = 0;
    
    // 分段下載參數
    BOOL isContactContinueSeek = NO;
    NSDate *contactContinueSeekTime = nil;
    NSDate *firstSyncTime = nil;
    
    do {
        @autoreleasepool {
            
            if(firstSyncTime)
            {
                syncInfo.firstSyncTime = firstSyncTime;
            }
            syncInfo.isContactContinueSeek = isContactContinueSeek;
            syncInfo.contactContinueSeekTime = contactContinueSeekTime?:[NSDate dateWithTimeIntervalSince1970:0];
            
            __block WCTRCStartSyncResponseResult *response = nil;
            
            // !! 因為ios有可能因為網路問題容易timeout, 所以這邊多做retry
            [WCTRestClientController executeProcessHandler:^NSError *(NSInteger retryCount)
             {
                 
                 NSError *error = nil;
                 
                 @autoreleasepool
                 {
                     [response release];
                     response = [[[WCTRestClientController shareRestClientController] syncStartWithSyncInfo:syncInfo error:&error] retain];
                     [error retain];
                     [self dumpLogMessageWithState:LogState_Normal format:@"#### startSync firstSyncTime:%@, seek:%@, seekTime:%@ ",
                      syncInfo.firstSyncTime, syncInfo.isContactContinueSeek?@"YES":@"NO", syncInfo.contactContinueSeekTime];
                     [self dumpLogMessageWithState:LogState_Normal format:@"#### startSync NO. %ld (try: %ld, action count:%ld error:%@) ",
                      (long)startSyncCount, (long)retryCount, (long)[response.data.contactCompareActionList count], error];
                 }// end of @autoreleasepool
                 
                 return [error autorelease];
             }
                                           withShouldRetry:^BOOL(NSInteger retryCount, NSError *error)
             {
                 
                 // 取消就不retry
                 if(cancel_ == YES)
                 {
                     return NO;
                 }
                 
                 //成功就不retry
                 if (error==nil)
                 {
                     return NO;
                 }
                 else
                 {
                     // !! start sync 不管是不是網路問題都retry
                     if (retryCount < WCTSync_SyncMaxRetryCount)
                     {
                         return YES;
                     }
                 }
                 
                 returnError = [error retain];
                 return NO;
             }];
            
            startSyncCount ++;
            
            if (startSyncCount==WCTSync_StartSyncStage1_MaxCount+1) {
                [self reportStep:WCTSyncActionControllerStep_FetchUpdateActions_Stage2];
            }
            
            if (startSyncCount<=WCTSync_StartSyncStage1_MaxCount)
            {
                [self reportProgress:startSyncCount*0.1];
            }
            else if ((startSyncCount-WCTSync_StartSyncStage1_MaxCount)<WCTSync_StartSyncStage2_MaxCount)
            {
                [self reportProgress:(startSyncCount-WCTSync_StartSyncStage1_MaxCount)/(WCTSync_StartSyncStage2_MaxCount)];
            }
            else
            {
                // 超過的不回報進度
            }
            
            if (returnError!=nil)
            {
                [response release];
                break;
            }
            
            if(cancel_ == YES)
            {
                // 在autorelease pool外不用在retain
                returnError = [PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil) retain];
                [response release];
                break;
            }
            
            // !! 結果，只紀錄一次
            if(result==nil)
            {
                result = [[WCTRCStartSyncResponseResult alloc] init];
                result.data = [[[WCTRCSyncUpdateInfo alloc] init] autorelease];
                result.data.firstSyncTimeResult = response.data.firstSyncTimeResult;
                result.data.categoryLastRecTimeResult = response.data.categoryLastRecTimeResult;
                result.data.firstSyncTimeResult = response.data.firstSyncTimeResult;
                
                // firstSyncTime只有第一次的結果要更新回來
                [firstSyncTime release];
                firstSyncTime = [response.data.firstSyncTimeResult copy];
            }
            
            // 記錄分段下載資訊
            isContactContinueSeek = response.data.isNeedContactContinueSeek;
            
            [contactContinueSeekTime release];
            contactContinueSeekTime = [response.data.contactContinueSeekTimeResult copy];
            
            
            
            //////////////////////////////////////////////////
            // 記錄取得的列表
            if ([response.data.categoryCompareActionList count])
            {
                [categoryActionResult addObjectsFromArray:response.data.categoryCompareActionList];
            }
            
            if ([response.data.contactCompareActionList count])
            {
                [contactActionResult addObjectsFromArray:response.data.contactCompareActionList];
            }
            
            // 記錄contactLastRecTime, contactLastRecTimeResult要取最後一次的傳出去
            result.data.contactLastRecTimeResult = response.data.contactLastRecTimeResult;
#ifdef DEBUG
            NSLog(@"response.data.contactLastRecTimeResult:%@", response.data.contactLastRecTimeResult);
#endif
            [response release];
            response = nil;
        }// end of @autoreleasepool
        
    } while (isContactContinueSeek);
    
    // 釋放分段下載資訊
    [firstSyncTime release];
    [contactContinueSeekTime release];
    
    if (returnError!=nil)
    {
        [returnError autorelease];
        
        [result release];
        result = nil;
        
        if (error!=NULL)
        {
            *error = returnError;
        }
    }
    else
    {
        // 沒有錯誤，寫回action list
        result.data.categoryCompareActionList = [NSArray arrayWithArray:categoryActionResult];
        result.data.contactCompareActionList = [NSArray arrayWithArray:contactActionResult];
    }
    
    
    [self dumpLogMessageWithState:LogState_Normal format:@"#### startSync End (%ld)", (long)startSyncCount];
    
    return [result autorelease];
}


//==============================================================================
// deprecate, 上傳失敗要整個action重做，不然圖片可能會不見，
//==============================================================================
- (WCTRCContactProfileResponseResult *)syncCreateContactWithRequest:(WCTRCContactCreationWithImageRequest *)request
                                                              error:(NSError **)error
{
    BOOL isDuplicateGuid = NO;
    BOOL isGuidChanged = NO;
    
    __block WCTRCContactProfileResponseResult *responseResult = nil;
    __block NSError *returnedError = nil;
    NSString *oldGuid = request.clientCreatedContactGuid;
    
    
    
    do {
        responseResult = nil;
        isDuplicateGuid = NO;
        returnedError = nil;
        
        
        [WCTRestClientController executeProcessHandler:^NSError *(NSInteger retryCount) {
            
            NSError *error = nil;
            
            @autoreleasepool
            {
                [responseResult release];
                responseResult = [[self.restClientController syncCreateContactWithRequest:request error:&error] retain];
                [error retain];
                [self dumpLogMessageWithState:LogState_Normal format:@"#### syncCreateContactWithRequest (try: %ld, error:%@) ", (long)retryCount, error];
            }
            return [error autorelease];
            
        } withShouldRetry:^BOOL(NSInteger retryCount, NSError *error) {
            
            // 取消就不retry
            if(cancel_ == YES)
            {
                // 在外部檢查是否cancel再產生error code
                return NO;
            }
            
            //成功就不retry
            if (error==nil)
            {
                return NO;
            }
            else
            {
                // !! 如果是網路問題或timeout才要retry,
                if ([error findNetworkError]!=nil &&
                    retryCount < WCTSync_SyncMaxRetryCount)
                {
                    return YES;
                }
            }
            
            returnedError = [error retain];
            return NO;
        }];
        
        [responseResult autorelease];
        
        // 如果是490 guid重復，要用新的guid再上傳一次
        if(returnedError!=nil)
        {
            NSInteger statusCode = 0;
            
            statusCode = [WCTRestClientController statusCodeFromAFRKNetworkingError:returnedError];
            
            if(statusCode == WCTServer_Common_ErrorCode_GuidHasBeenUsed)
            {
                request.clientCreatedContactGuid = [NSString GUID];
                
                isDuplicateGuid = YES;
                isGuidChanged = YES;
            }
        }
        
    } while (isDuplicateGuid);
    
    if (isGuidChanged) {
        //////////////////////////////////////////////////
        // !! 如果guid不一樣表示有修改guid, 要更新到local database
        [self.dataController replaceGuidFromOld:oldGuid toNew:request.clientCreatedContactGuid];
    }
    
    if(error!=NULL)
    {
        *error = returnedError;
    }
    
    return responseResult;
}


//==============================================================================
//
//==============================================================================
- (WCTRCBooleanResponseResult *)syncIsExistContactWithContactGuid:(NSString *)contactGuid accountGuid:(NSString *)accountGuid error:(NSError **)error
{
    __block WCTRCBooleanResponseResult *responseResult = nil;
    __block NSError *returnError = nil;
    
    [WCTRestClientController executeProcessHandler:^NSError *(NSInteger retryCount) {
        
        NSError *error = nil;
        @autoreleasepool {
            
            [responseResult release];
            responseResult = [[self.restClientController syncIsExistContactWithContactGuid:contactGuid accountGuid:accountGuid error:&error] retain];
            [error retain];
            
            [self dumpLogMessageWithState:LogState_Normal format:@"#### syncIsExistContactWithContactGuid (try: %ld, error:%@) ", (long)retryCount, error];
        }
        return [error autorelease];
        
    } withShouldRetry:^BOOL(NSInteger retryCount, NSError *error) {
        
        // 取消就不retry
        if(cancel_ == YES)
        {
            // 在外部檢查是否cancel再產生error code
            return NO;
        }
        
        //成功就不retry
        if (error==nil)
        {
            return NO;
        }
        else
        {
            // !! 如果是網路問題或timeout才要retry,
            if ([error findNetworkError]!=nil &&
                retryCount < WCTSync_SyncMaxRetryCount)
            {
                return YES;
            }
        }
        
        returnError = error;
        return NO;
    }];
    
    //////////////////////////////////////////////////
    if (returnError)
    {
        if (error!=NULL)
        {
            *error = returnError;
        }
        
        return nil;
    }
    else
    {
        return [responseResult autorelease];
    }
}


//==============================================================================
//
//==============================================================================
- (WCTRCSyncContactUpdateCheckResponseResult *)syncContactUpdateCheckWithRequest:(WCTRCContactUpdateCheckRequest *)request
                                                                           error:(NSError **)error
{
    __block WCTRCSyncContactUpdateCheckResponseResult *responseResult = nil;
    __block NSError *returnError = nil;
    
    [WCTRestClientController executeProcessHandler:^NSError *(NSInteger retryCount) {
        
        NSError *error = nil;
        
        @autoreleasepool {
            
            [responseResult release];
            responseResult = [[self.restClientController syncContactUpdateCheckWithRequest:request error:&error] retain];
            [error retain];
            [self dumpLogMessageWithState:LogState_Normal format:@"#### syncContactUpdateCheckWithRequest (try: %ld, error:%@) ", (long)retryCount, error];
        }
        
        return [error autorelease];
    } withShouldRetry:^BOOL(NSInteger retryCount, NSError *error) {
        
        // 取消就不retry
        if(cancel_ == YES)
        {
            // 在外部檢查是否cancel再產生error code
            return NO;
        }
        
        //成功就不retry
        if (error==nil)
        {
            return NO;
        }
        else
        {
            // !! 如果是網路問題或timeout才要retry,
            if ([error findNetworkError]!=nil &&
                retryCount < WCTSync_SyncMaxRetryCount)
            {
                return YES;
            }
        }
        
        returnError = error;
        return NO;
    }];
    
    //////////////////////////////////////////////////
    if (returnError)
    {
        if (error!=NULL)
        {
            *error = returnError;
        }
        
        return nil;
    }
    else
    {
        return [responseResult autorelease];
    }
}


//==============================================================================
//
//==============================================================================
- (WCTRCSyncUpdateActionResponseResult *)syncContactCheckStatusWithRequest:(WCTRCSyncContactCheckRequest *)request
                                                                     error:(NSError **)error
{
    __block WCTRCSyncUpdateActionResponseResult *responseResult = nil;
    __block NSError *returnError = nil;
    
    [WCTRestClientController executeProcessHandler:^NSError *(NSInteger retryCount) {
        
        NSError *error = nil;
        
        @autoreleasepool {
            
            [responseResult release];
            responseResult = [[self.restClientController syncContactCheckStatusWithRequest:request
                                                                                     error:&error] retain];
            [error retain];
            
            [self dumpLogMessageWithState:LogState_Normal format:@"#### syncContactCheckStatusWithRequest (try: %ld, error:%@) ", (long)retryCount, error];
        }
        return [error autorelease];
        
    } withShouldRetry:^BOOL(NSInteger retryCount, NSError *error) {
        
        // 取消就不retry
        if(cancel_ == YES)
        {
            // 在外部檢查是否cancel再產生error code
            return NO;
        }
        
        //成功就不retry
        if (error==nil)
        {
            return NO;
        }
        else
        {
            // !! 如果是網路問題或timeout才要retry,
            if ([error findNetworkError]!=nil &&
                retryCount < WCTSync_SyncMaxRetryCount)
            {
                return YES;
            }
        }
        
        returnError = error;
        return NO;
    }];
    
    //////////////////////////////////////////////////
    if (returnError)
    {
        if (error!=NULL)
        {
            *error = returnError;
        }
        
        return nil;
    }
    else
    {
        return [responseResult autorelease];
    }
}


//==============================================================================
// deprecate, 上傳失敗要整個action重做，不然圖片可能會不見，
//==============================================================================
- (WCTRCContactProfileResponseResult *)syncUpdateContactWithRequest:(WCTRCContactUpdateInfoRequest *)request
                                                              error:(NSError **)error
{
    __block WCTRCContactProfileResponseResult *responseResult = nil;
    __block NSError *returnError = nil;
    
    [WCTRestClientController executeProcessHandler:^NSError *(NSInteger retryCount) {
        
        NSError *error = nil;
        
        @autoreleasepool {
            
            [responseResult release];
            responseResult = [[self.restClientController syncUpdateContactWithRequest:request error:&error] retain];
            [error retain];
            
            [self dumpLogMessageWithState:LogState_Normal format:@"#### syncUpdateContactWithRequest (try: %ld, error:%@) ", (long)retryCount, error];
        }
        return [error autorelease];
        
    } withShouldRetry:^BOOL(NSInteger retryCount, NSError *error) {
        
        // 取消就不retry
        if(cancel_ == YES)
        {
            // 在外部檢查是否cancel再產生error code
            return NO;
        }
        
        //成功就不retry
        if (error==nil)
        {
            return NO;
        }
        else
        {
            // !! 如果是網路問題或timeout才要retry,
            if ([error findNetworkError]!=nil &&
                retryCount < WCTSync_SyncMaxRetryCount)
            {
                return YES;
            }
        }
        
        returnError = error;
        return NO;
    }];
    
    //////////////////////////////////////////////////
    if (returnError)
    {
        if (error!=NULL)
        {
            *error = returnError;
        }
        
        return nil;
    }
    else
    {
        return [responseResult autorelease];
    }
}


//==============================================================================
//
//==============================================================================
- (WCTRCSyncContactDeleteResponseResult *)syncDeleteContactWithContactGuid:(NSString *)contactGuid
                                                               accountGuid:(NSString *)accountGuid
                                                                     error:(NSError **)error
{
    __block WCTRCSyncContactDeleteResponseResult *responseResult = nil;
    __block NSError *returnError = nil;
    
    [WCTRestClientController executeProcessHandler:^NSError *(NSInteger retryCount) {
        
        NSError *error = nil;
        
        @autoreleasepool {
            
            [responseResult release];
            responseResult = [[self.restClientController syncDeleteContactWithContactGuid:contactGuid
                                                                              accountGuid:accountGuid
                                                                                    error:&error] retain];
            [error retain];
            
            [self dumpLogMessageWithState:LogState_Normal format:@"#### syncDeleteContactWithContactGuid (try: %ld, error:%@) ", (long)retryCount, error];
        }
        return [error autorelease];
        
    } withShouldRetry:^BOOL(NSInteger retryCount, NSError *error) {
        
        // 取消就不retry
        if(cancel_ == YES)
        {
            // 在外部檢查是否cancel再產生error code
            return NO;
        }
        
        //成功就不retry
        if (error==nil)
        {
            return NO;
        }
        else
        {
            // !! 如果是網路問題或timeout才要retry,
            if ([error findNetworkError]!=nil &&
                retryCount < WCTSync_SyncMaxRetryCount)
            {
                return YES;
            }
        }
        
        returnError = error;
        return NO;
    }];
    
    //////////////////////////////////////////////////
    if (returnError)
    {
        if (error!=NULL)
        {
            *error = returnError;
        }
        
        return nil;
    }
    else
    {
        return [responseResult autorelease];
    }
}


//==============================================================================
//
//==============================================================================
- (WCTRCDetailContactInfoResponseResult *)syncGetContactWithContactGuid:(NSString *)contactGuid
                                                                  error:(NSError **)error

{
    __block WCTRCDetailContactInfoResponseResult *responseResult = nil;
    __block NSError *returnError = nil;
    
    [WCTRestClientController executeProcessHandler:^NSError *(NSInteger retryCount) {
        
        NSError *error = nil;
        
        @autoreleasepool {
            
            [responseResult release];
            responseResult = [[self.restClientController syncGetContactWithContactGuid:contactGuid error:&error] retain];
            [error retain];
            
            [self dumpLogMessageWithState:LogState_Normal format:@"#### syncGetContactWithContactGuid (try: %ld, error:%@) ", (long)retryCount, error];
        }
        
        return [error autorelease];
        
    } withShouldRetry:^BOOL(NSInteger retryCount, NSError *error) {
        
        // 取消就不retry
        if(cancel_ == YES)
        {
            // 在外部檢查是否cancel再產生error code
            return NO;
        }
        
        //成功就不retry
        if (error==nil)
        {
            return NO;
        }
        else
        {
            // !! 如果是網路問題或timeout才要retry,
            if ([error findNetworkError]!=nil &&
                retryCount < WCTSync_SyncMaxRetryCount)
            {
                return YES;
            }
        }
        
        returnError = error;
        return NO;
    }];
    
    //////////////////////////////////////////////////
    if (returnError)
    {
        if (error!=NULL)
        {
            *error = returnError;
        }
        
        return nil;
    }
    else
    {
        return [responseResult autorelease];
    }
}







////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private - remote sync access image (特殊處理)


//==============================================================================
//
//==============================================================================
- (NSData *)privateContactImageDataWithContactGuid:(NSString *)contactGuid
                                         imageType:(NSString *)imageType
                                             error:(NSError **)error
{
    __block NSData *responseResult = nil;
    __block NSError *returnError = nil;
    
    [WCTRestClientController executeProcessHandler:^NSError *(NSInteger retryCount) {
        
        NSError *error = nil;
        @autoreleasepool {
            
            [responseResult release];
            responseResult = [[self.restClientController privateContactImageDataWithContactGuid:contactGuid
                                                                                      imageType:imageType
                                                                                          error:&error] retain];
            [error retain];
            
            [self dumpLogMessageWithState:LogState_Normal format:@"#### privateContactImageDataWithContactGuid (try: %ld, error:%@) ", (long)retryCount, error];
        }
        
        return [error autorelease];
        
    } withShouldRetry:^BOOL(NSInteger retryCount, NSError *error) {
        
        // 取消就不retry
        if(cancel_ == YES)
        {
            // 在外部檢查是否cancel再產生error code
            return NO;
        }
        
        //成功就不retry
        if (error==nil)
        {
            return NO;
        }
        else
        {
            // !! 如果是網路問題或timeout才要retry,
            if ([error findNetworkError]!=nil &&
                retryCount < WCTSync_SyncMaxRetryCount)
            {
                return YES;
            }
        }
        
        returnError = error;
        return NO;
    }];
    
    //////////////////////////////////////////////////
    if (returnError)
    {
        if (error!=NULL)
        {
            *error = returnError;
        }
        
        return nil;
    }
    else
    {
        return [responseResult autorelease];
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private - Local data access methods


//================================================================================
//
//================================================================================
- (void)localGetNecessaryInfo
{
    // 取得目前local group資料
    self.localGroupIDAndGuidMappingDict =[NSMutableDictionary dictionaryWithDictionary:[self.dataController groupIDAndGuidMappingDict]];
    
    // 取得server端favorite group guid
    self.favoriteCategoryGuid = [PPSettingsController stringValueWithKey:WCTSettingsKey_FavoriteCategoryGUID];
    
    // 取得使用者guid
    self.selfAccountGuid = [PPSettingsController stringValueWithKey:WCTSettingsKey_AccountGUID];
    
    // 取得使用者guid
    self.bossAccountGuid = [PPSettingsController stringValueWithKey:WCTSettingsKey_BossAccountGUID];
}


//================================================================================
//
//================================================================================
- (void)localUpdateGroupGuid:(NSString *)groupGuid withID:(WC_GroupID)groupID
{
    if([self.dataController updateGroupGuid:groupGuid withID:groupID] == YES)
    {
        [self.localGroupIDAndGuidMappingDict setObject:groupGuid forKey:@(groupID)];
    }
}


//================================================================================
//
//================================================================================
- (void)localUpdateGroupModifiedTime:(NSDate *)modifiedTime withGuid:(NSString *)guid
{
    [self.dataController updateGroupModifiedTime:modifiedTime withGuid:guid];
}


//================================================================================
//
//================================================================================
- (BOOL)localResetGroupSyncActionWithGuid:(NSString *)guid
{
    return [self.dataController resetGroupSyncActionToNoneWithGuid:guid];
}

//================================================================================
//
//================================================================================
- (WCGroupModel *)localGroupModelWithGroupGuid:(NSString *)groupGuid
{
    return [self.dataController groupWithGuid:groupGuid];
}


//================================================================================
//
//================================================================================
- (NSDate *)localCardSyncActionModifiedTimeWithCardID:(NSString *)cardID
{
    WCTCardSyncActionModel *syncActionModel = [self.dataController copyCardSyncActionModelWithCardID:cardID];
    
    NSDate *modifiedTime = [syncActionModel.modifiedTime retain];
    
    [syncActionModel release];
    
    return [modifiedTime autorelease];
}


//================================================================================
//
//================================================================================
- (WCCardModel *)localCardModelWithCardID:(NSString *)cardID error:(NSError **)error
{
    WCCardModel *cardModel = [self.dataController copyCardFullDataWithCardID:cardID];
    
    
    if(cardModel == nil && error != nil)
    {
        *error = self.dataController.lastError;
    }
    
    return [cardModel autorelease];
}

//================================================================================
//
//================================================================================
- (NSArray *)localCardSharedAccountGuidsWithCardID:(NSString *)cardID error:(NSError **)error
{
    NSArray *sharedAccountGuids = [self.dataController copySharedAccountWithCardID:cardID];
    
    if(sharedAccountGuids == nil && error != nil)
    {
        *error = self.dataController.lastError;
    }
    
    return [sharedAccountGuids autorelease];
}


//================================================================================
//
//================================================================================
- (CPImage *)localCardImageWithCardID:(NSString *)cardID imageType:(WC_ImageType)imageType
{
    CPImage *image = nil;
    
    if([self.dataController hasCardImageWithCardID:cardID type:imageType] == YES)
    {
        image = [self.dataController copyCardImageWithCardID:cardID type:imageType subtype:WC_IST_Browse];
        [image autorelease];
    }
    
    return image;
}


//================================================================================
//
//================================================================================
- (BOOL)localSetCardImage:(CPImage *)image imageSHA1:(NSString *)imageSHA1 withCardID:(NSString *)cardID type:(WC_ImageType)type
{
    return [self.dataController setCardImage:image imageSHA1:imageSHA1 withCardID:cardID type:type];
}


//================================================================================
// return array of WCTRCSimpleCategoryInfo
//================================================================================
- (NSMutableArray *)localSimpleCategoryInfoWithGuids:(NSArray *)guids
{
    NSMutableArray *groupGuidModels = [NSMutableArray array];
    
    for(NSString *categoryGuid in guids)
    {
        WCGroupModel *groupModel = [self.dataController groupWithGuid:categoryGuid];
        
        if(groupGuidModels)
        {
            // 有取到才能加
            if(groupModel)
            {
                [groupGuidModels addObject:[groupModel WCTRCSimpleCategoryInfo]];
            }
        }
    }
    
    // !! 如果處理後沒有類別，要歸到其他聯絡人
    if ([groupGuidModels count]==0)
    {
        NSDictionary *groupMapping = [self.dataController groupIDAndGuidMappingDict];
        NSString *unfiledGuid = [groupMapping objectForKey:@(WC_GID_Unfiled)];
        WCGroupModel *unfiledGroupModel = [self.dataController groupWithGuid:unfiledGuid];
        
        // 有取到才能加
        if(unfiledGroupModel)
        {
            [groupGuidModels addObject:[unfiledGroupModel WCTRCSimpleCategoryInfo]];
        }
    }
    return groupGuidModels;
}


//================================================================================
//
//================================================================================
- (NSError *)localDeleteCardDataWithCardID:(NSString *)cardID
{
    NSError *error = nil;
    
    if([self.dataController removeCardDataForSyncUpdate:cardID] == NO)
    {
        error = self.dataController.lastError;
    }
    
    return error;
}


//================================================================================
//
//================================================================================
- (NSError *)localDeleteCardSyncActionWithCardID:(NSString *)cardID
{
    NSError *error = nil;
    
    if([self.dataController removeCardSyncActionWithCardID:cardID] == NO)
    {
        error = self.dataController.lastError;
    }
    
    return error;
}


//================================================================================
//
//================================================================================
- (void)localUpdateCardSyncActionModifiedTime:(NSDate *)syncActinoModifiedTime withGuid:(NSString *)guid
{
    [self.dataController updateCardSyncActionModifiedTime:syncActinoModifiedTime withCardID:guid];
}


//================================================================================
//
//================================================================================
- (BOOL)localUpdateCardSyncAction:(WCTCardSyncActionModel *)cardSyncAction withCheckModifyTime:(NSDate *)checkModifyTime
{
    return [self.dataController updateCardSyncAction:cardSyncAction withCheckModifyTime:checkModifyTime];
}


//================================================================================
//
//================================================================================
- (BOOL)localResetCardSyncActionWithGuid:(NSString *)guid checkModifyTime:(NSDate *)checkModifyTime
{
    return [self.dataController resetCardSyncActionToNoneWithCardID:guid checkModifyTime:checkModifyTime];
}


//================================================================================
// 濾除自己和老闆
//================================================================================
- (NSArray *)localValidShareAccountGuidsWithAccountLessInfos:(NSArray *)accountLessInfos
{
    NSMutableArray *accountGuids = [NSMutableArray array];
    
    for(WCTRCAccountLessInfo *info in accountLessInfos)
    {
        // !! 記錄有什麼就顯示什麼，不用過濾
        //        if([info.guid isEqualToString:self.selfAccountGuid] == NO &&
        //           [info.guid isEqualToString:self.bossAccountGuid] == NO)
        {
            [accountGuids addObject:info.guid];
        }
    }
    
    return accountGuids;
}


//================================================================================
// 濾除系統group及檢查group是否存在
//================================================================================
- (NSArray *)localValidGroupIDsWithCategoryInfos:(NSArray *)categoryInfos error:(NSError **)error
{
    NSError *returnError = nil;
    NSMutableArray *groupIDArray = [NSMutableArray array];
    
    
    for(WCTRCCategoryInfo *info in categoryInfos)
    {
        if([info.categoryType isEqualToString:WCTRC_CategoryType_Normal] == NO)
        {
            continue;
        }
        
        //////////////////////////////////////////////////
        
        WC_GroupID groupID = [self groupIDWithMappedGuid:info.guid];
        
        // !! 如果目前local沒有這個group的資料，模擬新增流程在local建立。
        if(groupID == WC_GID_None)
        {
            WCTRCCategoryCompareAction *action = [[[WCTRCCategoryCompareAction alloc] init] autorelease];
            
            if(action == nil)
            {
                returnError = PPErrorMake(WCTSyncActionControllerError_InitObjectFailed, @"Init WCTRCCategoryCompareAction failed", nil);
                break;
            }
            
            action.categoryGuid = info.guid;
            action.categoryName = info.name;
            action.categoryType = WCTRC_CategoryType_Normal;
            action.modifyTime = info.updateTime;
            action.updateAction = WCTRC_UpdateAction_AddOrUpdateClient;
            
            WCGroupModel *newGroup = [self localAddCategoryWithAction:action error:&returnError];
            
            if(returnError != nil)
            {
                returnError = PPErrorMake(WCTSyncActionControllerError_OperationFailed, @"Local add group failed", returnError);
                break;
            }
            
            // 設定新的groupID供後續判斷
            groupID = newGroup.ID;
        }
        
        if(groupID >= WC_GID_Unfiled)
        {
            [groupIDArray addObject:[NSString stringWithInteger:groupID]];
        }
    }
    
    if(error != nil)
    {
        *error = returnError;
    }
    
    return groupIDArray;
}


//================================================================================
//
//================================================================================
- (WCCardModel *)localCardModelWithContactInfo:(WCTRCDetailContactInfo *)contactInfo error:(NSError **)error
{
    WCCardModel *cardModel = [WCCardModel modelWithWCTRCContactInfo:contactInfo];
    
    
    //////////////////////////////////////////////////
    // shared account
    
    cardModel.sharedAccountGUIDArray = [self localValidShareAccountGuidsWithAccountLessInfos:contactInfo.beSharedAccounts];
    
    
    //////////////////////////////////////////////////
    // favorite特殊處理
    
    if([self remoteIsFavoriteWithContactInfo:contactInfo] == YES)
    {
        cardModel.tagMask |= WC_TagMask_Favorite;
    }
    
    
    //////////////////////////////////////////////////
    // card groupID
    // !! 如果新增的類別目前的類別mapping沒有，表示是server新增，要新增到local
    // !! 如果新增的類別目前的local沒有，表示local已刪除，在WCTCardDBController, 不會加入此類別
    
    NSArray *groupIDs = [self localValidGroupIDsWithCategoryInfos:contactInfo.belongCategories error:error];
    
    if(*error != nil)
    {
        return nil;
    }
    
    [cardModel setGroupIDArray:groupIDs isInitCard:YES];
    
    return cardModel;
}


//================================================================================
//
//================================================================================
- (WCTCardSyncActionModel *)localCardSyncActionModelWithContactInfo:(WCTRCDetailContactInfo *)contactInfo
                                                         actionType:(WCTSyncActionType)actionType
{
    WCTCardSyncActionModel *cardSyncActionModel = [[[WCTCardSyncActionModel alloc] init] autorelease];
    
    if(cardSyncActionModel != nil)
    {
        cardSyncActionModel.actionType = actionType;
        cardSyncActionModel.syncState = WCTSyncState_Synced;
        cardSyncActionModel.dataGuid = contactInfo.guid;
        cardSyncActionModel.modifiedTime = contactInfo.syncModifyTime;
        
        cardSyncActionModel.contentSHA1 = contactInfo.contentSha1;
        cardSyncActionModel.groupSHA1 = contactInfo.categorySha1;
        cardSyncActionModel.sharedAccountSHA1 = contactInfo.sharedAccountSha1;
        
        // !! 影像的SHA1要和舊紀錄的保持一致
        for (WCTRCContactImageInfo *contactImageInfo in contactInfo.images)
        {
            if([contactImageInfo.type isEqualToString:WCTRC_ImageType_Front])
            {
                cardSyncActionModel.frontSideSHA1 = contactImageInfo.sha1;
            }
            else if([contactImageInfo.type isEqualToString:WCTRC_ImageType_Rear])
            {
                cardSyncActionModel.backSideSHA1 = contactImageInfo.sha1;
            }
            else if([contactImageInfo.type isEqualToString:WCTRC_ImageType_Logo])
            {
                cardSyncActionModel.IDPhotoSHA1 = contactImageInfo.sha1;
            }
        }
    }
    
    return cardSyncActionModel;
}


//==============================================================================
// return array of category guid
//==============================================================================
- (NSArray *)localOrderedCategories
{
    // 取得手機端的類別順序
    NSMutableArray *orderedGuids = [NSMutableArray array];
    
    NSMutableArray *groupModels = [self.dataController copyAllGroupsAndGetCardCount:NO];
    
    for (WCGroupModel *groupModel in groupModels)
    {
        NSString *groupGuid = [self categoryGuidFromGroupID:groupModel.ID];
        
        if(groupGuid)
        {
            [orderedGuids addObject:groupGuid];
        }
    }
    
    [groupModels release];
    
    return [orderedGuids count]>0?orderedGuids:nil;
}


//==============================================================================
// input: array of category guid
//==============================================================================
- (BOOL)localUpdateCategoriesWithOrder:(NSArray *)orderedList modifiedTime:(NSDate *)modifiedTime
{
    // 將新的順序寫入資料庫中
    NSMutableArray *groupIDArray = [NSMutableArray array];
    
    for (NSString *groupGuid in orderedList)
    {
        WC_GroupID groupID = [self groupIDWithMappedGuid:groupGuid];
        if (groupID!=WC_GID_None)
        {
            [groupIDArray addObject:@(groupID)];
        }
    }
    
    if ([groupIDArray count]>0)
    {
        [self.dataController updateOrderWithGroupIDArray:groupIDArray];
        
        // !! updateOrderWithGroupIDArray用local時間，這邊要再寫一次
        [self.dataController setGroupOrderModifiedTime:modifiedTime];
        return YES;
    }
    
    return NO;
}


//==============================================================================
//
//==============================================================================
- (NSDate *)localCategoryOrderModifiedTime
{
    // 取得手機端的類別順序修改時間
    return [self.dataController groupOrderModifiedTime];
}


//==============================================================================
//
//==============================================================================
- (void)localSaveSyncErrorWithContactGuid:(NSString *)contactGuid errorCode:(NSInteger)errorCode startSyncTime:(NSDate *)startSyncTime
{
    // 記錄失敗原因
    [self.dataController updateCardSyncErrorInfoWithCardID:contactGuid errorCode:errorCode startSyncTime:startSyncTime];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private - Local sync access methods

//================================================================================
//
//================================================================================
- (NSMutableArray *)localCategorySyncInfoArrayWithError:(NSError **)error isForceSync:(BOOL)isForceSync
{
    NSMutableArray *infoArray = [NSMutableArray array];
    WCTRCCategorySyncInfo *categorySyncInfo;
    NSMutableArray *actions = nil;
    
    
    //////////////////////////////////////////////////
    //////////////////////////////////////////////////
    if (isForceSync)
    {
        // !! 強制同步要全取
        actions = [self.dataController copyGroupSyncActionsAfterDate:nil];
    }
    else
    {
        actions = [self.dataController copyGroupSyncActions];
    }
    
    //////////////////////////////////////////////////
    
    for(WCTGroupSyncActionModel *actionModel in actions)
    {
        categorySyncInfo = [[WCTRCCategorySyncInfo alloc] init];
        
        if(categorySyncInfo != nil)
        {
            if (isForceSync &&
                actionModel.actionType==WCTSyncActionType_None)
            {
                actionModel.actionType = WCTSyncActionType_Add;
            }
            
            categorySyncInfo.action = [self serverDefinedStringWithSyncActionType:actionModel.actionType];
            categorySyncInfo.categoryGuid = actionModel.dataGuid;
            categorySyncInfo.lastModifyTime = actionModel.modifiedTime;
            
            [infoArray addObject:categorySyncInfo];
            [categorySyncInfo release];
        }
    }
    
    [actions release];
    
    //////////////////////////////////////////////////
    
    return infoArray;
}


//================================================================================
// isForceSync, 不管有沒有action都要回傳，而且action是none的要改為WCTRC_Action_Add
//================================================================================
- (NSMutableArray *)localContactSyncInfoArrayWithError:(NSError **)error isForceSync:(BOOL)isForceSync
{
    NSMutableArray *infoArray = [NSMutableArray array];
    WCTRCContactSyncInfo *contactSyncInfo;
    NSMutableArray *actions = nil;
    
    
    //////////////////////////////////////////////////
    if (isForceSync)
    {
        // !! 強制同步要全取
        actions = [self.dataController copyCardSyncActionsAfterDate:nil];
    }
    else
    {
        actions = [self.dataController copyCardSyncActions];
    }
    
    
    // 給server前先依guid排序
    [actions sortUsingComparator:^NSComparisonResult(WCTCardSyncActionModel *obj1, WCTCardSyncActionModel *obj2) {
        return [obj1.dataGuid compare:obj2.dataGuid options:NSCaseInsensitiveSearch];
    }];
    
    
    //////////////////////////////////////////////////
    
    for(WCTCardSyncActionModel *actionModel in actions)
    {
        contactSyncInfo = [[WCTRCContactSyncInfo alloc] init];
        
        if(contactSyncInfo != nil)
        {
            if (isForceSync &&
                actionModel.actionType==WCTSyncActionType_None)
            {
                actionModel.actionType = WCTSyncActionType_Add;
            }
            
            contactSyncInfo.action = [self serverDefinedStringWithSyncActionType:actionModel.actionType];
            contactSyncInfo.contactGuid = actionModel.dataGuid;
            contactSyncInfo.lastModifyTime = actionModel.modifiedTime;
            
            [infoArray addObject:contactSyncInfo];
            [contactSyncInfo release];
        }
    }
    
    [actions release];
    
    //////////////////////////////////////////////////
    
    return infoArray;
}


//================================================================================
//
//================================================================================
- (NSError *)localAddOrUpdateCategoryWithAction:(WCTRCCategoryCompareAction *)action
                           hasAddOrDeleteClient:(BOOL *)hasAddOrDeleteClient
{
    NSError *error = nil;
    
    do
    {
        WC_GroupID localGroupID = [self.dataController groupIDWithGuid:action.categoryGuid];
        
        if(localGroupID != WC_GID_None) // group exist, update group name.
        {
            error = [self localUpdateCategoryWithAction:action groupID:localGroupID];
        }
        else // group not exist, add new one.
        {
            [self localAddCategoryWithAction:action error:&error];
            if(hasAddOrDeleteClient)
            {
                (*hasAddOrDeleteClient) = YES;
            }
        }
    }
    while(0);
    
    //////////////////////////////////////////////////
    
    return error;
}


//================================================================================
//
//================================================================================
- (WCGroupModel *)localAddCategoryWithAction:(WCTRCCategoryCompareAction *)action error:(NSError **)error
{
    NSError *returnError = nil;
    WCGroupModel *groupModel = nil;
    
    do
    {
        if([action.categoryName length] == 0)
        {
            returnError = PPErrorMake(WCTSyncActionControllerError_WrongParameter, @"Category name is empty", nil);
            break;
        }
        
        WCTGroupSyncActionModel *groupSyncActionModel = [[[WCTGroupSyncActionModel alloc] init] autorelease];
        
        if(groupSyncActionModel != nil)
        {
            groupSyncActionModel.dataGuid = action.categoryGuid;
            groupSyncActionModel.actionType = WCTSyncActionType_Add;
            groupSyncActionModel.modifiedTime = action.modifyTime;
        }
        
        groupModel = [self.dataController newGroupWithNameForSync:action.categoryName
                                                  syncActionModel:groupSyncActionModel];
        
        if(groupModel == nil)
        {
            returnError = self.dataController.lastError;
            break;
        }
        
        //////////////////////////////////////////////////
        
        [self.localGroupIDAndGuidMappingDict setObject:action.categoryGuid forKey:@(groupModel.ID)];
    }
    while(0);
    
    //////////////////////////////////////////////////
    
    if(error != nil)
    {
        *error = returnError;
    }
    
    return [groupModel autorelease];
}


//================================================================================
//
//================================================================================
- (NSError *)localUpdateCategoryWithAction:(WCTRCCategoryCompareAction *)action groupID:(WC_GroupID)groupID
{
    NSError *error = nil;
    WCTGroupSyncActionModel *syncActionModel = nil;
    
    do
    {
        syncActionModel = [[[WCTGroupSyncActionModel alloc] init] autorelease];
        
        if(syncActionModel == nil)
        {
            error = PPErrorMake(WCTSyncActionControllerError_InitObjectFailed, @"Init WCTGroupSyncActionModel failed", nil);
            break;
        }
        
        syncActionModel.dataGuid = [self categoryGuidFromGroupID:groupID];
        syncActionModel.actionType = WCTSyncActionType_Modify;
        syncActionModel.modifiedTime = action.modifyTime;
        
        if([self.dataController updateGroupName:action.categoryName withID:groupID syncActionModel:syncActionModel] == NO)
        {
            error = self.dataController.lastError;
            break;
        }
    }
    while(0);
    
    return error;
}


//================================================================================
//
//================================================================================
- (NSError *)localDeleteCategoryWithAction:(WCTRCCategoryCompareAction *)action
{
    NSError *error = nil;
    
    do
    {
        WC_GroupID groupID = [self.dataController groupIDWithGuid:action.categoryGuid];
        
        if(groupID == WC_GID_None)
        {
            // !! 找不到就當作刪除成功
            break;
        }
        
        if([self.dataController removeGroupWithIDForSync:[NSString stringWithInteger:groupID]] == YES)
        {
            [self.dataController removeGroupSyncActionWithGuid:action.categoryGuid];
        }
        else
        {
            error = PPErrorMake(WCTSyncActionControllerError_OperationFailed, @"Remove local group failed", self.dataController.lastError);
        }
    }
    while(0);
    
    //////////////////////////////////////////////////
    
    return error;
}


//================================================================================
//
//================================================================================
- (NSError *)localAddOrUpdateContactWithAction:(WCTRCContactCompareAction *)action
{
    NSError *error = nil;
    
    do
    {
        BOOL isExist = [self.dataController isCardIDExist:action.contactGuid];
        
        
        if(isExist == YES) // update
        {
            error = [self localUpdateContactWithAction:action];
        }
        else // add
        {
            error = [self localAddContactWithAction:action];
        }
    }
    while(0);
    
    //////////////////////////////////////////////////
    
    return error;
}


//================================================================================
//
//================================================================================
- (NSError *)localAddContactWithAction:(WCTRCContactCompareAction *)action
{
    NSError *error = nil;
    WCCardModel *cardModel = nil;
    WCTRCDetailContactInfo *contactInfo = nil;
    
    [self dumpLogMessageWithState:LogState_Normal format:@"## Local add contact (%@)", action.contactGuid];
    
    
    do
    {
        //////////////////////////////////////////////////
        // 檢查local剩餘磁碟空間
        
        if([NSFileManager hasEnoughDiskSpaceWithSize:WCTSync_MinDiskSpace] == NO)
        {
            error = PPErrorMake(WCTSyncActionControllerError_NotEnoughDiskSpace, @"Not enough disk space", nil);
            break;
        }
        else
        {
            // dump debug log
            NSArray *paths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
            NSDictionary *dictionary = [[NSFileManager defaultManager] attributesOfFileSystemForPath:[paths lastObject] error:&error];
            NSNumber *freeFileSystemSizeInBytes = [dictionary objectForKey:NSFileSystemFreeSize];
            unsigned long long freeSpace = [freeFileSystemSizeInBytes unsignedLongLongValue];
            
            [self dumpLogMessageWithState:LogState_Normal format:@"   rest disk space %.2f MB", freeSpace/1024.0/1024.0];
        }
        
        
        //////////////////////////////////////////////////
        // 取得server端contactInfo
        
        contactInfo = [self remoteContactInfoWithGuid:action.contactGuid error:&error];
        
        // 取消
        if(cancel_ == YES)
        {
            // 在autorelease pool外不用在retain
            error = PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil);
            break;
        }
        
        if(error != nil)
        {
            // !! 如果404(找不到名片資料)就不處理這個動作
            if([WCTRestClientController statusCodeFromAFRKNetworkingError:error]==WCTServer_Common_ErrorCode_DataNotFound)
            {
                error = nil;
            }
            
            break;
        }
        
        
        //////////////////////////////////////////////////
        // 儲存影像
        
        for(WCTRCContactImageInfo *imageInfo in contactInfo.images)
        {
            @autoreleasepool
            {
                // 如果Sha1是EMPTY, 不用下載
                if ([imageInfo.sha1 compare:WCTRC_SHA1_Empty options:NSCaseInsensitiveSearch]==NSOrderedSame)
                {
                    continue;
                }
                
                CPImage *image = [self remoteContactImageWithGuid:contactInfo.guid imageType:imageInfo.type error:&error];
                
                // 取消
                if(cancel_ == YES)
                {
                    // 在autorelease pool外不用在retain
                    error = [PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil) retain];
                    break;
                }
                
                if(error != nil)
                {
                    [error retain];
                    break;
                }
                
                WC_ImageType localImageType = [self imageTypeWithServerDefinedString:imageInfo.type];
                
                BOOL result = [self.dataController setCardImage:image withCardID:contactInfo.guid type:localImageType];
                
                if(result == NO)
                {
                    error = [PPErrorMake(WCTSyncActionControllerError_OperationFailed, @"Local set card image failed", self.dataController.lastError) retain];
                    break;
                }
                
            } // end of @autoreleasepool
        }
        
        if(error != nil)
        {
            [error autorelease];
            break;
        }
        
        
        //////////////////////////////////////////////////
        // 儲存card
        
        // card data
        cardModel = [self localCardModelWithContactInfo:contactInfo error:&error];
        
        if(error != nil)
        {
            break;
        }
        
        
        //////////////////////////////////////////////////
        // 儲存syncAction
        
        // !!下載完成SyncAction改為None
        WCTCardSyncActionModel *cardSyncActionModel = [self localCardSyncActionModelWithContactInfo:contactInfo
                                                                                         actionType:WCTSyncActionType_None];
        
        BOOL result = [self.dataController addCard:cardModel syncActionModel:cardSyncActionModel withSendNotification:YES];
        
        if(result == NO)
        {
            error = PPErrorMake(WCTSyncActionControllerError_OperationFailed, @"Local addCard failed", self.dataController.lastError);
            break;
        }
        
    }
    while(0);
    
    
    //////////////////////////////////////////////////
    
    if(error != nil)
    {
        [self dumpLogMessageWithState:LogState_Normal format:@"   !! Local add contact failed %@", error];
        
        // 失敗就全清掉，重新同步。
        [self localDeleteCardDataWithCardID:action.contactGuid];
        [self localDeleteCardSyncActionWithCardID:action.contactGuid];
    }
    
    return error;
}


//================================================================================
// 
//================================================================================
- (NSError *)localUpdateContactWithAction:(WCTRCContactCompareAction *)action
{
    NSLog(@"localUpdateContactWithAction start:%@", action.contactGuid);
    
    NSError *error = nil;
    NSMutableArray *updateComponents = [NSMutableArray array];
    
    NSString *frontImageSHA1 = nil;
    NSString *backImageSHA1 = nil;
    NSString *logoImageSHA1 = nil;
    
    [self dumpLogMessageWithState:LogState_MethodIn format:@"%s", __func__];
    
    do
    {
        //////////////////////////////////////////////////
        // 檢查local剩餘磁碟空間
        
        if([NSFileManager hasEnoughDiskSpaceWithSize:WCTSync_MinDiskSpace] == NO)
        {
            error = PPErrorMake(WCTSyncActionControllerError_NotEnoughDiskSpace, @"Not enough disk space", nil);
            break;
        }
        
        
        //////////////////////////////////////////////////
        // 產生updateComponents
        // 取得此名片的同步資料
        WCTCardSyncActionModel *cardSyncActionModel = [[self.dataController copyCardSyncActionModelWithCardID:action.contactGuid] autorelease];
        
        if(cardSyncActionModel == nil)
        {
            error = PPErrorMake(WCTSyncActionControllerError_InitObjectFailed, @"copyCardSyncActionModelWithCardID failed", self.dataController.lastError);
            break;
        }
        
        //////////////////////////////////////////////////
        // !! 如果action的lastModifyTime與cardSyncActionModel一樣，就不用再下載聯絡人資料
        if ([cardSyncActionModel.modifiedTime compare:action.lastModifyTime]==NSOrderedSame)
        {
            // 不更新，算成功
            break;
        }
        
        //////////////////////////////////////////////////
        // !! 先取得目前server上的資料
        WCTRCDetailContactInfo *contactInfo = [self remoteContactInfoWithGuid:action.contactGuid error:&error];
        
        
        // 取消
        if(cancel_ == YES)
        {
            // 在autorelease pool外不用在retain
            error = PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil);
            break;
        }
        
        if(error != nil)
        {
            // !! 如果404就不處理這個動作
            if([WCTRestClientController statusCodeFromAFRKNetworkingError:error]==WCTServer_Common_ErrorCode_DataNotFound)
            {
                error = nil;
                break;
            }
            else
            {
                error = PPErrorMake(WCTSyncActionControllerError_InitObjectFailed, @"remoteContactInfoWithGuid failed", error);
                break;
            }
        }
        
        if(cardSyncActionModel != nil)
        {
            if([cardSyncActionModel.groupSHA1 isEqualToString:contactInfo.categorySha1] == NO)
            {
                [updateComponents addObject:WCTRC_UpdateComponent_Category];
            }
            
            if([cardSyncActionModel.contentSHA1 isEqualToString:contactInfo.contentSha1] == NO)
            {
                [updateComponents addObject:WCTRC_UpdateComponent_Content];
            }
            
            if([cardSyncActionModel.sharedAccountSHA1 isEqualToString:contactInfo.sharedAccountSha1] == NO)
            {
                [updateComponents addObject:WCTRC_UpdateComponent_ShareAccount];
            }
            
            for (WCTRCContactImageInfo *contactImageInfo in contactInfo.images)
            {
                // 要判斷是否真的需要更新名片圖，sha1與client不一致才要做
                if([contactImageInfo.type isEqualToString:WCTRC_ImageType_Front])
                {
                    if([self.dataController isImageSha1:contactImageInfo.sha1 equalWithCardID:action.contactGuid type:WC_IT_FrontSide]==NO)
                    {
                        frontImageSHA1 = contactImageInfo.sha1;
                        [updateComponents addObject:WCTRC_UpdateComponent_FrontImage];
                    }
                }
                else if([contactImageInfo.type isEqualToString:WCTRC_ImageType_Rear])
                {
                    if([self.dataController isImageSha1:contactImageInfo.sha1 equalWithCardID:action.contactGuid type:WC_IT_BackSide]==NO)
                    {
                        backImageSHA1 = contactImageInfo.sha1;
                        [updateComponents addObject:WCTRC_UpdateComponent_BackImage];
                    }
                }
                else if([contactImageInfo.type isEqualToString:WCTRC_ImageType_Logo])
                {
                    if([self.dataController isImageSha1:contactImageInfo.sha1 equalWithCardID:action.contactGuid type:WC_IT_IDPhoto]==NO)
                    {
                        logoImageSHA1 = contactImageInfo.sha1;
                        [updateComponents addObject:WCTRC_UpdateComponent_Logo];
                    }
                }
            }
        }
        
        [self dumpLogMessageWithState:LogState_Normal format:@"updateComponents=%@", updateComponents];
        
        //////////////////////////////////////////////////
        // update image if modified
        
        BOOL isUpdateCard = NO;
        BOOL isUpdateImage = NO;
        
        for(NSString *updateComponent in updateComponents)
        {
            switch ([self updateComponentWithServerDefinedString:updateComponent])
            {
                case SyncUpdateComponent_Group:
                {
                    isUpdateCard = YES;
                    break;
                }
                    
                case SyncUpdateComponent_Content:
                {
                    isUpdateCard = YES;
                    break;
                }
                    
                case SyncUpdateComponent_SharedAccount:
                {
                    isUpdateCard = YES;
                    break;
                }
                    
                case SyncUpdateComponent_FrontSideImage:
                {

                    isUpdateImage = YES;
                    
                    CPImage *image = nil;
                    
                    if([frontImageSHA1 compare:WCTRC_SHA1_Empty options:NSCaseInsensitiveSearch] != NSOrderedSame)
                    {
                        image = [self remoteContactImageWithGuid:action.contactGuid
                                                       imageType:WCTRC_ImageType_Front
                                                           error:&error];
                    }
                    
                    
                    // 取消
                    if(cancel_ == YES)
                    {
                        // 在autorelease pool外不用在retain
                        error = PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil);
                        break;
                    }
                    
                    if(error != nil)
                    {
                        break;
                    }
                    
                    if([self localSetCardImage:image
                                     imageSHA1:frontImageSHA1
                                    withCardID:action.contactGuid
                                          type:WC_IT_FrontSide] == NO)
                    {
                        error = PPErrorMake(WCTSyncActionControllerError_OperationFailed, @"Local update front side image failed", self.dataController.lastError);
                    }
                    
                    break;
                }
                    
                    case SyncUpdateComponent_BackSideImage:
                {
                    isUpdateImage = YES;
                    
                    CPImage *image = nil;
                    
                    if([backImageSHA1 compare:WCTRC_SHA1_Empty options:NSCaseInsensitiveSearch] != NSOrderedSame)
                    {
                        image = [self remoteContactImageWithGuid:action.contactGuid
                                                       imageType:WCTRC_ImageType_Rear
                                                           error:&error];
                    }
                    
                    
                    // 取消
                    if(cancel_ == YES)
                    {
                        // 在autorelease pool外不用在retain
                        error = PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil);
                        break;
                    }
                    
                    if(error != nil)
                    {
                        break;
                    }
                    
                    if([self localSetCardImage:image
                                     imageSHA1:backImageSHA1
                                    withCardID:action.contactGuid
                                          type:WC_IT_BackSide] == NO)
                    {
                        error = PPErrorMake(WCTSyncActionControllerError_OperationFailed, @"Local update back side image failed", self.dataController.lastError);
                    }
                    
                    break;
                }
                    
                    case SyncUpdateComponent_IDPhoto:
                {
                    isUpdateImage = YES;
                    
                    CPImage *image = nil;
                    
                    if([logoImageSHA1 compare:WCTRC_SHA1_Empty options:NSCaseInsensitiveSearch] != NSOrderedSame)
                    {
                        image = [self remoteContactImageWithGuid:action.contactGuid
                                                       imageType:WCTRC_ImageType_Logo
                                                           error:&error];
                    }
                    
                    
                    // 取消
                    if(cancel_ == YES)
                    {
                        // 在autorelease pool外不用在retain
                        error = PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil);
                        break;
                    }
                    
                    if(error != nil)
                    {
                        break;
                    }
                    
                    if([self localSetCardImage:image
                                     imageSHA1:logoImageSHA1
                                    withCardID:action.contactGuid
                                          type:WC_IT_IDPhoto] == NO)
                    {
                        error = PPErrorMake(WCTSyncActionControllerError_OperationFailed, @"Local update ID photo failed", self.dataController.lastError);
                    }
                    
                    break;
                }
                    
                default:
                    break;
            }
            
            if(error != nil)
            {
                break;
            }
        }
        
        if(error != nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        // 已校正的狀態，不包含在updateComponent中
        if (contactInfo.beCorrected)
        {
            [self.dataController removeUnverifiedWithCardID:contactInfo.guid withNotification:YES updateCardSyncAction:NO];
        }
        
        //////////////////////////////////////////////////
        // update group/contact if modified
        
        if(isUpdateCard == YES)
        {
            //////////////////////////////////////////////////
            // 儲存card
            
            WCCardModel *cardModel = [self localCardModelWithContactInfo:contactInfo error:&error];
            
            if(error != nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            // 儲存syncAction
            // !!下載完成SyncAction改為None
            
            WCTCardSyncActionModel *cardSyncActionModelBeforUpdate = [[self.dataController copyCardSyncActionModelWithCardID:action.contactGuid] autorelease];
            
            // 如果名片已被修改，就不做更新動作了
            if ([cardSyncActionModelBeforUpdate.modifiedTime isEqual:cardSyncActionModel.modifiedTime]==YES)
            {
                WCTCardSyncActionModel *newCardSyncActionModel = [self localCardSyncActionModelWithContactInfo:contactInfo
                                                                                                    actionType:WCTSyncActionType_None];
                if([self.dataController updateCard:cardModel withSendNotification:YES syncActionModel:newCardSyncActionModel mustExist:NO] == NO)
                {
                    error = PPErrorMake(WCTSyncActionControllerError_OperationFailed, @"Local add card failed", self.dataController.lastError);
                    break;
                }
            }// end of if
        }
        else if(isUpdateImage == YES)
        {
            //////////////////////////////////////////////////
            // 更新圖片也要更新名片修改時間(display)
            [self.dataController updateCardModifiedTime:contactInfo.modifyTimeForDisplay withCardID:contactInfo.guid];
            
            // 更新sync action最後修改時間
            [self.dataController updateCardSyncActionModifiedTime:contactInfo.syncModifyTime withCardID:action.contactGuid];
        }
    }
    while(0);
    
    //////////////////////////////////////////////////
    
    [self dumpLogMessageWithState:LogState_MethodOut format:@"%s error=%@", __func__, error];
    
    NSLog(@"localUpdateContactWithAction end:%@", action.contactGuid);
    return error;
}


//================================================================================
//
//================================================================================
- (NSError *)localDeleteContactWithAction:(WCTRCContactCompareAction *)action
{
    NSError *error = nil;
    error = [self localDeleteCardDataWithCardID:action.contactGuid];
    //////////////////////////////////////////////////
    
    return error;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Overwrite methods

//================================================================================
// !! PPSyncActionController是在local做同步比對動作，但WorldCardTeam中server會做比對動作，所以這裡要覆寫掉。
//================================================================================
- (BOOL)syncWithError:(NSError **)error quickSyncVaildTimeInterval:(NSTimeInterval)quickSyncVaildTimeInterval
{
    WCTRCSyncInfo *syncInfo = [[WCTRCSyncInfo alloc] init];
    BOOL result = NO;
    NSError *returnError = nil;
    NSError *internalError = nil;
    NSTimeInterval syncStartTime = [[NSDate date] timeIntervalSince1970];
    NSDate *startSyncTime = [NSDate date];
    
    //////////////////////////////////////////////////
    
    self.uploadImageCountForLog = 0;
    self.downloadImageCountForLog = 0;
    
    [self dumpLogMessageWithState:LogState_MethodIn format:@"%s", __FUNCTION__];
    
    do
    {
        if(syncInfo == nil)
        {
            returnError = PPErrorMake(WCTSyncActionControllerError_InitObjectFailed, @"WCTSyncInfo init failed", nil);
            break;
        }
        
        
        //////////////////////////////////////////////////
        // MARK: 取得一般同步資訊
        [self reportStep:WCTSyncActionControllerStep_GetNomalSyncInfo];
        [self dumpLogMessageWithState:LogState_Normal format:@"## 取得同步設定"];
        
        [self localGetNecessaryInfo];
        
        syncInfo.accountGuid = self.selfAccountGuid;
        syncInfo.contactLastRecTime = [PPSettingsController dateValueWithKey:WCTSyncActionControllerSettingsKey_ContactLastRecTime];
        syncInfo.isContactFirstSync = [PPSettingsController integerValueWithKey:WCTSyncActionControllerSettingsKey_IsContactFirstSync];
        
        if(syncInfo.contactLastRecTime == nil)
        {
            syncInfo.contactLastRecTime = [NSDate dateWithTimeIntervalSince1970:0];
        }
        
        
        // 這邊的firstSyncTime要先記錄下來，後面流程會用來判斷是否要記錄第一次同步時間
        // 同步完成後就可以清除，所以改與WCTSyncActionControllerSettingsKey_ContactLastRecTime用同一個
        NSDate *firstSyncTime = [PPSettingsController dateValueWithKey:WCTSyncActionControllerSettingsKey_ContactLastRecTime];
        if(firstSyncTime == nil)
        {
            syncInfo.firstSyncTime = [NSDate dateWithTimeIntervalSince1970:0];
        }
        else
        {
            syncInfo.firstSyncTime = firstSyncTime;
        }
        
        //////////////////////////////////////////////////
        // !! 如果離上次同步時間超過30天，就要用重新比對
        BOOL contactFullSync = NO;
        NSDate *contactLastRecTime = [PPSettingsController dateValueWithKey:WCTSyncActionControllerSettingsKey_ContactLastRecTime];
        if(contactLastRecTime!=nil)
        {
            WCTRCSyncNeedForceCompareResponseResult *responseResult = [[WCTRestClientController shareRestClientController] syncNeedForceCompareWithContactLastRecTime:contactLastRecTime
                                                                                                                                                                error:&returnError];
            
            // 錯誤就中斷
            if (returnError)
            {
                break;
            }
            
            if (responseResult.data)
            {
                contactFullSync = YES;
            }
        }
        
        //////////////////////////////////////////////////
        // MARK: 加入強制同步機制
        // 1. lastRecTime要傳1970
        // 2. firstSyncTime要傳1970
        // 3. isFirstSyncTime要傳YES
        // 4. dirty table要全部傳
        // 5. contact的do_nothing不用清除dirty table記錄
        
        //test: 要產生兩邊不mapping的資料
        //[self localResetCardSyncActionWithGuid:@"D07B68D4-E09C-4DF7-B7D8-B5A22A409A2D"];
        
        BOOL isForceFirstSync = (BOOL)[PPSettingsController integerValueWithKey:WCTSyncActionControllerSettingsKey_ForceFirstSync];
        
        if(isForceFirstSync)
        {
            contactFullSync =  YES;
        }
        
        
        //////////////////////////////////////////////////
        // 聯絡人重新比對
        if(contactFullSync)
        {
            syncInfo.contactLastRecTime = [NSDate dateWithTimeIntervalSince1970:0];
            syncInfo.isContactFirstSync = YES;
            
            syncInfo.firstSyncTime = [NSDate dateWithTimeIntervalSince1970:0];
            
            // !! 重新比對時，要把WCTSyncActionControllerSettingsKey_FirstStartSyncTime清掉
            // 這樣 重新比對如果中斷，才會傳正確的firstSyncTime
            [PPSettingsController removeValueWithKey:WCTSyncActionControllerSettingsKey_ContactLastRecTime];
            
            // !! 修正不合法的資料
            [self.dataController fixInvalidRecord];
        }
        
        //////////////////////////////////////////////////
        // dump debug info
        [self dumpLogMessageWithState:LogState_Normal format:@"#########    isForceFistSync : %@, contactFullSync:%@",
         isForceFirstSync?@"YES":@"NO",
         contactFullSync?@"YES":@"NO"];
        [self dumpLogMessageWithState:LogState_Normal format:@"token : %@", [WCTRestClientController curLoginToken]];
        [self dumpLogMessageWithState:LogState_Normal format:@"accountGuid : %@", syncInfo.accountGuid];
        [self dumpLogMessageWithState:LogState_Normal format:@"contactLastRecTime : %@", syncInfo.contactLastRecTime];
        [self dumpLogMessageWithState:LogState_Normal format:@"isContactFirstSync : %d", syncInfo.isContactFirstSync];
        [self dumpLogMessageWithState:LogState_Normal format:@"firstStartSyncTime : %@", syncInfo.firstSyncTime];

        // 回報進度
        [self reportProgress:1.0];
        
        if(returnError != nil)
        {
            break;
        }
        
        if(cancel_ == YES)
        {
            // 在autorelease pool外不用在retain
            returnError = PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil);
            break;
        }

        //////////////////////////////////////////////////
        // MARK: 取得local端card同步資料
        
        [self reportStep:WCTSyncActionControllerStep_GetLocalContactSyncInfo];
        
        syncInfo.contactSyncInfoList = [self localContactSyncInfoArrayWithError:&returnError isForceSync:isForceFirstSync];
        
        [self dumpLogMessageWithState:LogState_Normal format:@"## 取得local端card同步資料 (%ld)", [syncInfo.contactSyncInfoList count]];
        
        // MARK: ####   為了找bug先打開，release前要關掉   ####
        [self dumpLogMessageWithState:LogState_Normal format:@"## contactSyncInfoList: %@", syncInfo.contactSyncInfoList];
        
#ifdef DEBUG
        [self dumpLogMessageWithState:LogState_Normal format:@"## contactSyncInfoList: %@", syncInfo.contactSyncInfoList];
#endif
        // 回報進度
        [self reportProgress:1.0];
        
        if(returnError != nil)
        {
            break;
        }
        
        if(cancel_ == YES)
        {
            // 在autorelease pool外不用在retain
            returnError = PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil);
            break;
        }
        
        //////////////////////////////////////////////////
        // MARK: 從server端取得同步所需更新動作
        
        [self reportStep:WCTSyncActionControllerStep_FetchUpdateActions_Stage1];
        [self dumpLogMessageWithState:LogState_Normal format:@"## 從server端取得同步所需更新動作"];
        
        // !! 分段下載流程在這邊處理
        WCTRCStartSyncResponseResult *response = [self startSyncWithInfo:syncInfo
                                                                   error:&returnError];
        
        if(returnError != nil)
        {
            break;
        }
        
        if(cancel_ == YES)
        {
            // 在autorelease pool外不用在retain
            returnError = PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil);
            break;
        }
        
        // !! 如果之前沒有記錄過第一次同步時間，才會記錄，同步完成後就可以清除，所以改與WCTSyncActionControllerSettingsKey_ContactLastRecTime用同一個
        if(firstSyncTime==nil && response.data.firstSyncTimeResult)
        {
            [PPSettingsController setDateValue:response.data.firstSyncTimeResult withKey:WCTSyncActionControllerSettingsKey_ContactLastRecTime];
        }
        
        //////////////////////////////////////////////////
        // MARK: 排序contact action
        [self dumpLogMessageWithState:LogState_Normal format:@"## 排序contact action"];
        
        NSArray *contactCompareActionList = [self sortedListWithContactActionList:response.data.contactCompareActionList];
        
        [self dumpLogMessageWithState:LogState_Normal format:@"##  排序contact action - Count: %ld -> %ld",
         [response.data.contactCompareActionList count],
         [contactCompareActionList count]];
        
        //        [self reportProgress:0.99];
        
        //////////////////////////////////////////////////
        // MARK: 套用Card同步更新動作
        NSInteger total = [contactCompareActionList count];
        NSInteger index = 0;
        
        [self reportStep:WCTSyncActionControllerStep_ApplyContactUpdateActions];
        
        [self dumpLogMessageWithState:LogState_Normal format:@"## 套用Card同步更新動作 (total:%d)", total];
        
#ifdef DEBUG
        [self dumpLogMessageWithState:LogState_Normal format:@"## syncInfo %@\n\n\n\n", syncInfo];
        
        [self dumpLogMessageWithState:LogState_Normal format:@"## response.data.contactCompareActionList:%@", response.data.contactCompareActionList];
#endif
        // 進行更新
        for(WCTRCContactCompareAction *action in contactCompareActionList)
        {
            [self dumpLogMessageWithState:LogState_Normal format:@"[%d/%d] %@", ++index, total, action];
            
            @autoreleasepool
            {
                if(cancel_ == YES)
                {
                    returnError = [PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil) retain];
                    [self dumpLogMessageWithState:LogState_Normal format:@"Sync cancelled"];
                    break;
                }
                
                SyncUpdateAction updateAction = [self updateActionWithServerDefinedString:action.updateAction];
                
                //////////////////////////////////////////////////
                // !! 如果是UNCHECK, 要先取得真正的action再繼續
                if(updateAction==SyncUpdateAction_Unckeck)
                {
                    // 取得此名片的同步資料
                    WCTCardSyncActionModel *cardSyncActionModel = [[self.dataController copyCardSyncActionModelWithCardID:action.contactGuid] autorelease];
                    
                    if(cardSyncActionModel == nil)
                    {
                        returnError = [PPErrorMake(WCTSyncActionControllerError_InitObjectFailed, @"SyncUpdateAction_Unckeck-copyCardSyncActionModelWithCardID failed", self.dataController.lastError) retain];
                        break;
                    }
                    
                    
                    if (isForceFirstSync &&
                        cardSyncActionModel.actionType==WCTSyncActionType_None)
                    {
                        cardSyncActionModel.actionType = WCTSyncActionType_Add;
                    }
                    // 取得真正的action
                    WCTRCSyncContactCheckRequest *syncContactCheckRequest = [[[WCTRCSyncContactCheckRequest alloc] init] autorelease];
                    syncContactCheckRequest.accountID = syncInfo.accountGuid;
                    syncContactCheckRequest.contactID = action.contactGuid;
                    syncContactCheckRequest.actionType = [self serverDefinedStringWithSyncActionType:cardSyncActionModel.actionType];
                    syncContactCheckRequest.modifyTime = cardSyncActionModel.modifiedTime;
                    
                    [self dumpLogMessageWithState:LogState_Normal format:@"UNCHECK - request: %@", syncContactCheckRequest];
                    
                    WCTRCSyncUpdateActionResponseResult *updateActionResponseResult = [self syncContactCheckStatusWithRequest:syncContactCheckRequest error:&returnError];
                    
                    if(cancel_ == YES)
                    {
                        returnError = [PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil) retain];
                        [self dumpLogMessageWithState:LogState_Normal format:@"Sync cancelled"];
                        break;
                    }
                    
                    if(updateActionResponseResult == nil)
                    {
                        returnError = [PPErrorMake(WCTSyncActionControllerError_InitObjectFailed, @"SyncUpdateAction_Unckeck-updateActionResponseResult failed", returnError) retain];
                        break;
                    }
                    
                    updateAction = [self updateActionWithServerDefinedString:updateActionResponseResult.data];
                    
                    [self dumpLogMessageWithState:LogState_Normal format:@"UNCHECK - updateAction: %@", [self serverDefinedStringFromUpdateAction:updateAction]];
                    
                    //////////////////////////////////////////////////
                    // !! 這邊再檢查一次cotact是否有被修改，避免checkStatus中間的時間差
                    if([self contactChangedWithCardID:action.contactGuid afterDate:cardSyncActionModel.modifiedTime])
                    {
                        [self dumpLogMessageWithState:LogState_Normal format:@"Action Skiped"];
                        continue;
                    }
                }
                
                //////////////////////////////////////////////////
                // 比對是否可以執行，server比較新才執行，
                if ((updateAction==SyncUpdateAction_AddOrUpdateClient||
                     updateAction==SyncUpdateAction_UpdateClient||
                     updateAction==SyncUpdateAction_DeleteClient) &&
                    [self shouldExecuteLocalActionWithCardID:action.contactGuid actionDate:action.lastModifyTime]==NO)
                {
                    [self dumpLogMessageWithState:LogState_Normal format:@"Action Skiped"];
                    continue;
                }
                
                // 比對是否可以執行，client比較新才執行，
                if ((updateAction==SyncUpdateAction_AddServer||
                     updateAction==SyncUpdateAction_UpdateServer||
                     updateAction==SyncUpdateAction_DeleteServer) &&
                    [self shouldExecuteRemoteActionWithCardID:action.contactGuid actionDate:action.lastModifyTime]==NO)
                {
                    [self dumpLogMessageWithState:LogState_Normal format:@"Action Skiped"];
                    continue;
                }
                
                //////////////////////////////////////////////////
                switch (updateAction)
                {
                        case SyncUpdateAction_AddServer:
                    {
                        returnError = [self remoteAddContactWithAction:action
                                                           accountGuid:syncInfo.accountGuid];
                        break;
                    }
                        
                        case SyncUpdateAction_UpdateServer:
                    {
                        returnError = [self remoteUpdateContactWithAction:action
                                                              accountGuid:syncInfo.accountGuid];
                        break;
                    }
                        
                        case SyncUpdateAction_DeleteServer:
                    {
                        returnError = [self remoteDeleteContactWithAction:action
                                                              accountGuid:syncInfo.accountGuid];
                        break;
                    }
                        
                        // !! 因為會出現local delete, 但server回傳update client(server modify)
                        // 所以不論是add or update 或是 update 都用一樣做法
                        case SyncUpdateAction_AddOrUpdateClient:
                        case SyncUpdateAction_UpdateClient:
                    {
                        returnError = [self localAddOrUpdateContactWithAction:action];
                        break;
                    }
                        case SyncUpdateAction_DeleteClient:
                    {
                        returnError = [self localDeleteContactWithAction:action];
                        break;
                    }
                        case SyncUpdateAction_DoNothing:
                    {
                        // 清除dirty table記錄
                        if(!isForceFirstSync)
                        {
                            [self localResetCardSyncActionWithGuid:action.contactGuid checkModifyTime:action.lastModifyTime];
                        }
                        break;
                    }
                        
                    default:
                        break;
                }
                
                [self reportProgress:(CGFloat)index/(CGFloat)total];
                
                //////////////////////////////////////////////////
                //  generate fake error
                //                NSHTTPURLResponse *response = [[NSHTTPURLResponse alloc] initWithURL:@"http://penpower.net"
                //                                                                          statusCode:500
                //                                                                         HTTPVersion:nil
                //                                                                        headerFields:nil];
                //
                //                returnError = [NSError errorWithDomain:@"test" code:-1011 userInfo:@{AFRKNetworkingOperationFailingURLResponseErrorKey:response}];
                
                if(returnError != nil)
                {
                    // !!單筆失敗要看失敗原因決定是否要繼續
                    if ([returnError findNetworkError]!=nil)
                    {
                        // 網路問題就中斷 (retry在前面就做了)
                        [returnError retain];
                        break;
                    }
                    else
                    {
                        // 取得server回傳的status code
                        NSInteger statusCode = 0;
                        
                        statusCode = [WCTRestClientController statusCodeFromAFRKNetworkingError:returnError];
                        
                        // 422要繼續
                        if (statusCode == WCTServer_Common_ErrorCode_InvalidParameter)
                        {
                            // 紀錄到失敗列表
                            [self localSaveSyncErrorWithContactGuid:action.contactGuid errorCode:statusCode startSyncTime:startSyncTime];
                            returnError = nil;
                        }
                        else if (statusCode==WCTServer_Common_ErrorCode_ServerTaskCollision||
                                 statusCode==WCTServer_Common_ErrorCode_RunTimeException)
                        {
                            // !! 500要繼續，但不紀錄到失敗列表，額外紀錄，同步完成時，如果需要時再顯示
                            internalError = [returnError retain];
                            returnError = nil;
                        }
                        else
                        {
                            // 其他錯誤都要中斷
                            [returnError retain];
                            break;
                        }
                    }
                }
                
                if(cancel_ == YES)
                {
                    returnError = [PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil) retain];
                    break;
                }
            } // end of @autoreleasepool
        } //end of for loop

        [self reportProgress:1.0];

        //////////////////////////////////////////////////
        if(returnError != nil)
        {
            [returnError autorelease];
            break;
        }
        
        if(internalError != nil)
        {
            [internalError autorelease];
            break;
        }
        
        if(cancel_ == YES)
        {
            // 在autorelease pool外不用在retain
            returnError = PPErrorMake(WCSyncFlow_Error_UserCancel, @"使用者取消", nil);
            break;
        }
        
        
        //////////////////////////////////////////////////
        // MARK: 同步結束
        [self reportStep:WCTSyncActionControllerStep_SyncDone];

        // 更新群組列表
        [WCTAccountDataController updateCategoriesFromServerWithCompleteHandler:nil];

        // !! 增加同步即時性，可用可不用。
        [self.restClientController syncDoneWithError:nil];
        
        //
        if (isForceFirstSync)
        {
            [PPSettingsController setIntegerValue:0 withKey:WCTSyncActionControllerSettingsKey_ForceFirstSync];
        }
        
        // 紀錄同步參數
        [PPSettingsController setDateValue:response.data.contactLastRecTimeResult withKey:WCTSyncActionControllerSettingsKey_ContactLastRecTime];
        [PPSettingsController setIntegerValue:0 withKey:WCTSyncActionControllerSettingsKey_IsContactFirstSync];

        // !!沒有internal error或其他回傳錯誤才算成功
        if (internalError==nil||
            returnError==nil)
        {
            result = YES;
        }

        [self reportProgress:1.0];
    }
    while (0);
    
    // !! 成功才記錄，且要記錄開始同步的時間
    if (result)
    {
        [PPSettingsController setIntegerValue:[startSyncTime timeIntervalSinceReferenceDate] withKey:WCSC_IV_kLastSyncTime];
    }
    
    NSTimeInterval syncFinishTime = [[NSDate date] timeIntervalSince1970];
    
    [self dumpLogMessageWithState:LogState_MethodOut format:@"%s (cost:%f sec, uploadImageCount:%d, downloadImageCount:%d)", __FUNCTION__, syncFinishTime-syncStartTime, self.uploadImageCountForLog, self.downloadImageCountForLog];
    
    //////////////////////////////////////////////////
    
    [syncInfo release];
    
    if(error != nil)
    {
        // !! 優先顯示其他錯誤，如果都沒有才顯示internal error
        if (returnError)
        {
            *error = returnError;
        }
        else if(internalError)
        {
            *error = internalError;
        }
    }
    
    return result;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class methods

//================================================================================
//
//================================================================================
+ (NSDate *)lastSyncActionSuccessDate
{
    NSDate *date = nil;
    NSTimeInterval timeInterval = [PPSettingsController integerValueWithKey:WCSC_IV_kLastSyncTime];
    
    if(timeInterval > 0)
    {
        date = [NSDate dateWithTimeIntervalSinceReferenceDate:timeInterval];
    }
    
    return date;
}


//================================================================================
//
//================================================================================
+ (void)resetToFirstSync
{
    [PPSettingsController removeValueWithKey:WCTSyncActionControllerSettingsKey_CategoryLastRecTime];
    [PPSettingsController setIntegerValue:1 withKey:WCTSyncActionControllerSettingsKey_IsCategoryFirstSync];
    
    [PPSettingsController removeValueWithKey:WCTSyncActionControllerSettingsKey_ContactLastRecTime];
    [PPSettingsController setIntegerValue:1 withKey:WCTSyncActionControllerSettingsKey_IsContactFirstSync];
    
    // 最後上次更新時間也要修改
    [PPSettingsController removeValueWithKey:WCSC_IV_kLastSyncTime];
    
    // 這邊一定要用remove, 這樣才能判斷要不要記錄
    
}


//==============================================================================
//
//==============================================================================
+ (void)forceRecompareSync
{
    [PPSettingsController setIntegerValue:1 withKey:WCTSyncActionControllerSettingsKey_ForceFirstSync];
}


@end
