//
//  WCTLoginController.m
//  
//
//  Created by Howard on 2017/7/3.
//
//

#import "WCTLoginController.h"

// Define
#import "WCTLoginController+ParameterDefine.h"
#import "WCTLoginController+ResourceDefine.h"
#import "WCTLoginController+SettingsKey.h"
#import "WCTRestClientController+SettingsKey.h"
#import "WCTRestClientController+ErrorCodeDefine.h"
#import "WCTSettingsKey.h"

// Category
#import "NSDate+Format.h"
#import "NSError+Custom.h"
#import "WCTRestClientController+Password.h"
#import "WCTRestClientController+Version.h"

// Controller
#import "PPSettingsController.h"
#import "PPNetworkReachabilityController.h"
#import "WCFormatCheckController.h"
#import "PPLogController.h"
#import "WCToolController.h"


typedef void(^EndLoginCompletion)(WCTRCLoginInfo *loginInfo, WCTLC_ChangePasswordReason changePasswordReason, NSDictionary <NSString *, NSError *> *errorDictionary);

typedef void(^EndSubscriptionURLCompletion)(NSURL *subscriptionURL, NSError *error);


static int g_WCTLoginControllerOfflineLoginTimeoutDays = 30;
static NSString * const WCTLoginController_LogDir = @"WCTLogin";
static BOOL WCTLoginController_EnagleLog = NO;
////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Interface WCTLoginController

@interface WCTLoginController ()

@property (nonatomic,retain) NSOperationQueue *operationQueue;
@property (nonatomic,retain) NSString *accountName;
@property (nonatomic,retain) NSString *accountPassword;
@property (nonatomic,retain) NSString *inputServerString;
@property (nonatomic,assign) BOOL ignoreCheckHttpsPort;
@property (nonatomic,assign) NSUInteger httpsPort;
@property (nonatomic,copy) EndLoginCompletion loginCompletion;
@property (nonatomic,copy) EndSubscriptionURLCompletion subscriptionURLCompletion;

// log
@property (nonatomic,retain) PPLogController *logController;
@end





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Implementation WCTLoginController

@implementation WCTLoginController





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Dellocating Object

//================================================================================
//
//================================================================================
- (id)init
{
    if(self=[super init])
    {
        _operationQueue = [[NSOperationQueue alloc] init];
        
        [_operationQueue setMaxConcurrentOperationCount:1];
        
        //////////////////////////////////////////////////
        [PPSettingsController setDefaultIntegerValue:1 withKey:WCTSettingsKey_ShowExportUI];
        [PPSettingsController setDefaultIntegerValue:1 withKey:WCTSettingsKey_ShowMyCardUI];
        [PPSettingsController setDefaultIntegerValue:1 withKey:WCTSettingsKey_ShowCrmUI];
        [PPSettingsController setDefaultIntegerValue:1 withKey:WCTSettingsKey_ShowContactServerUI];
        [PPSettingsController setDefaultArrayValue:[NSArray array] withKey:WCTLoginControllerSettingKey_DisplayURLAddressList];
        
        //////////////////////////////////////////////////
        // 建立rest client log instance
        
        self.logController = [[[PPLogController alloc] init] autorelease];
        
        if(self.logController != nil)
        {
            NSString *logDirPath = [WCTLoginController logDirPath];
            
            if([[NSFileManager defaultManager] fileExistsAtPath:logDirPath] == NO)
            {
                [[NSFileManager defaultManager] createDirectoryAtPath:logDirPath
                                          withIntermediateDirectories:YES
                                                           attributes:nil
                                                                error:nil];
            }
            
            [self.logController setFileName:WCTLoginController_LogDir atPath:logDirPath];
            [self.logController setMask:PPLogControllerMask_Normal];
            
            NSLog(@"%@",logDirPath);
        }
    }

    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    [_accountName release];
    _accountName = nil;
    
    [_accountPassword release];
    _accountPassword = nil;
    
    [_loginCompletion release];
    _loginCompletion = nil;
    
    [_inputServerString release];
    _inputServerString = nil;
    
    [_operationQueue cancelAllOperations];
    [_operationQueue release];
    _operationQueue = nil;
    
    [_subscriptionURLCompletion release];
    _subscriptionURLCompletion = nil;
    
    self.logController = nil;
    //////////////////////////////////////////////////

    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Format Method

//================================================================================
// MARK:比對伺服器網址是否一樣
//================================================================================
- (BOOL)checkServerIsSameThanLastLoginFromCurLoginUrlString:(NSString *)curLoginUrlString
{
    BOOL result = NO;
    
    do
    {
        if([curLoginUrlString length]<=0)
        {
            break;
        }
        
        
        //////////////////////////////////////////////////
        /// !! 公有雲的link是固定的，如果是公有雲就當成是一樣的
        if([WCTLoginController isCloudServerURLString:curLoginUrlString])
        {
            result = YES;
            break;
        }
        
        
        //////////////////////////////////////////////////
        
        NSString *lastLoginUrlString = [[self recordServerList] firstObject];
        
        if(lastLoginUrlString==nil)
        {
            lastLoginUrlString = [PPSettingsController stringValueWithKey:WCTRestClientController_SettingsKey_Success_URLAddress];
        }
        
        //從來沒登入過
        if([lastLoginUrlString length]<=0)
        {
            result = YES;
            break;
        }
        
        //////////////////////////////////////////////////
        
        result = [curLoginUrlString isEqualToString:lastLoginUrlString];
        
    }
    while (0);
    
    return result;
}


//================================================================================
//
//================================================================================
- (BOOL)checkAccountNameFormatWithError:(NSError **)error
{
    BOOL result = NO;
    NSError *returnError = nil;
    
    do
    {
        self.accountName = [WCTLoginController removeBeforeAndAfterSpaceFromText:self.accountName];
        
        
        if([self.accountName  length]<=0)
        {
            returnError = PPErrorMake(WCTLoginController_ErrorCode_EmailFormatIsIllegal, WCTLC_MLS_EmailFormatIsIllegal, nil);
            
            break;
        }
        
        result = YES;
    }
    while (0);
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    return result;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Check Offline Method


//==============================================================================
// 檢查最後一次登入是否在30天內，如果是，回傳YES，否則回傳NO
//==============================================================================
- (BOOL)checkLastLoginTimeLessThan30Days
{
    [self logMessageWithFormat:@"%s in", __PRETTY_FUNCTION__];
    NSNumber *lastOnlineLoginNumber = [PPSettingsController numberValueWithKey:WCTLoginControllerDictionaryKey_LastOnlineLoginSuccessTime];
    
    // 只有1.2升級的會沒有這個值，先開放可以離線登入
    if(lastOnlineLoginNumber==nil)
    {
        [self logMessageWithFormat:@"%s out(YES)", __PRETTY_FUNCTION__];
        return YES;
    }
    
    // 超過30天，不能用離線登入
    if ([[NSDate date] timeIntervalSince1970]-[lastOnlineLoginNumber doubleValue]>=g_WCTLoginControllerOfflineLoginTimeoutDays*24*60*60)
    {
        [self logMessageWithFormat:@"%s out(NO)", __PRETTY_FUNCTION__];
        return NO;
    }
    
    [self logMessageWithFormat:@"%s out(YES)", __PRETTY_FUNCTION__];
    return YES;
}


//================================================================================
//
//================================================================================
- (BOOL)loginOnOfflineModeWithUrlString:(NSString *)urlString
                      networkConnection:(BOOL)networkConnection
                                  error:(NSError **)error
{
    [self logMessageWithFormat:@"%s in", __PRETTY_FUNCTION__];
    NSError *returnError = nil;
    
    BOOL result = NO;
    
    NSString *errorMessage = WCTLC_MLS_FailedToConnectInternet;
    
    if(networkConnection==YES)
    {
        errorMessage = WCTLC_MLS_FailedToConnectServer;
        
        if (*error && [*error code]==NSURLErrorTimedOut)
        {
            errorMessage = WCTLC_MLS_ServerBusy;
        }
    }
    
    //////////////////////////////////////////////////
    if([self checkLastLoginTimeLessThan30Days]==NO)
    {
        returnError = PPErrorMake(WCTLoginController_ErrorCode_OfflineModeTimeout, errorMessage, nil);
    }
    
    //////////////////////////////////////////////////
    
    else if([self checkServerIsSameThanLastLoginFromCurLoginUrlString:urlString]==NO)
    {
        returnError = PPErrorMake(WCTLoginController_ErrorCode_InvalidServer, errorMessage, nil);
    }
    
    //////////////////////////////////////////////////
    
    else if([self.accountName compare:[PPSettingsController stringValueWithKey:WCTRestClientController_SettingsKey_AccountName]
                                        options:NSCaseInsensitiveSearch]!=NSOrderedSame)
    {
        returnError = PPErrorMake(WCTLoginController_ErrorCode_InvalidUser, errorMessage, nil);
    }
    
    //////////////////////////////////////////////////
    
    else if([self.accountPassword isEqualToString:[WCTRestClientController decryptAccountPassword]]==NO)
    {
        returnError = PPErrorMake(WCTLoginController_ErrorCode_InvalidPassword, errorMessage, nil);
    }
    else
    {
        result = YES;
    }
    
    //////////////////////////////////////////////////

    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////
    [self logMessageWithFormat:@"%s out(%@ - %@)", __PRETTY_FUNCTION__, @(result), returnError];
    return result;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Check Status port method


//================================================================================
//
//================================================================================
- (void)checkStatusPortWithUrlString:(NSString *)urlString
                          completion:(void(^)(NSError *error))completion
{
    @autoreleasepool
    {
        NSError *error = nil;
        
        //////////////////////////////////////////////////
        
        do
        {
            if(self.operationQueue==nil)
            {
                error = PPErrorOperationFailed(error);
                break;
            }
            
            //////////////////////////////////////////////////
            
            __block NSBlockOperation *blockOperation = [[[NSBlockOperation alloc] init] autorelease];
            if(blockOperation==nil)
            {
                error = PPErrorOperationFailed(error);
                break;
            }
            
            //////////////////////////////////////////////////
            
            __block NSError  *blockError = nil;
            
            [blockOperation addExecutionBlock:^{
                
                @autoreleasepool
                {
                    do
                    {
                        [PPSettingsController setStringValue:urlString
                                                     withKey:WCTRestClientController_SettingsKey_URLAddress];
                        
                        if([[WCTRestClientController shareRestClientController] changeURLAddress:urlString withError:&blockError]==NO)
                        {
                            break;
                        }
                        
                        //////////////////////////////////////////////////
                        
                        BOOL hasHttpsPort = NO;
                        
                        //檢查開放的ports
                        WCTRCServiceStatusResponseResult *serviceStatusResponseResult = [[WCTRestClientController shareRestClientController] serverServiceStatusWithError:&blockError];
                        
                        if(serviceStatusResponseResult==nil)
                        {
                            // 檢查離線登入的原因
                            if([self loginOnOfflineModeWithUrlString:self.inputServerString networkConnection:YES error:&blockError]==YES)
                            {
                                blockError = PPErrorMake(WCTLoginController_ErrorCode_WarningOfflineMode,
                                                         WCTLC_MLS_OfflineMode,
                                                         blockError);
                            }
                        }
                        else
                        {
                            for(WCTRCServicePort *servicePort in serviceStatusResponseResult.data.openPortList)
                            {
                                if([servicePort.portName isEqualToString:@"https"]==YES)
                                {
                                    self.httpsPort = servicePort.port;
                                    
                                    hasHttpsPort = YES;
                                    
                                    break;
                                }
                            }
                            
                            //////////////////////////////////////////////////
                            
                            //不開放 9443 port, 詢問是否使用 http 登入
                            
                            if(hasHttpsPort==NO)
                            {
                                blockError = PPErrorMake(WCTLoginController_ErrorCode_WarningSwitchHttp, WCTLC_MLS_AskIfConnectWithHttp, blockError);
                            }
                        }
                        
                        //////////////////////////////////////////////////
                        
                        if(blockOperation.isCancelled==YES && blockError==nil)
                        {
                            blockError = PPErrorOperationCancel(blockError);
                        }
                        
                        //////////////////////////////////////////////////
                        
                        [blockError retain];
                        
                        dispatch_async(dispatch_get_main_queue(), ^{
                            completion(blockError);
                            [blockError release];
                        });
                    }
                    while (0);
                }
            }];
            
            //////////////////////////////////////////////////
            
            @synchronized(self.operationQueue)
            {
                if([self.operationQueue operationCount]>0)
                {
                    [self.operationQueue cancelAllOperations];
                }
                
                [self.operationQueue addOperation:blockOperation];
            }
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil)
        {
            completion(error);
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - ExceedAllowCount


//==============================================================================
//
//==============================================================================
- (NSDictionary *)exceptionDictFromExceptionMessage:(NSString *)exceptionMessage
{
    if ([exceptionMessage length]==0)
    {
        return nil;
    }
    
    if([exceptionMessage hasPrefix:@"MaxFailureCountExceededException:"]==NO)
    {
        return nil;
    }
    
    NSString *jsonString = [exceptionMessage stringByReplacingOccurrencesOfString:@"MaxFailureCountExceededException:" withString:@""];
    
    return [NSJSONSerialization JSONObjectWithData:[jsonString dataUsingEncoding:NSUTF8StringEncoding] options:0 error:nil];
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Mainthread call back method

//================================================================================
//
//================================================================================
- (void)mainThreadCallCompletionWithDictionary:(NSMutableDictionary *)dictionary
{
    if(self.loginCompletion!=nil)
    {
        WCTRCLoginInfo *loginInfo = [dictionary objectForKey:WCTLoginControllerDictionaryKey_LoginInfo];
        
        WCTLC_ChangePasswordReason changePasswordReason = [[dictionary objectForKey:WCTLoginControllerDictionaryKey_NeedChangedPassword] integerValue];

        self.loginCompletion(loginInfo,
                             changePasswordReason,
                             dictionary);
    }
}


//================================================================================
//
//================================================================================
- (void)mainThreadFinishedSubscriptionURLWithDictionary:(NSDictionary *)dictionary
{
    if(self.subscriptionURLCompletion!=nil)
    {
        self.subscriptionURLCompletion([dictionary objectForKey:WCTLoginControllerDictionaryKey_SubscriptionURL],
                                       [dictionary objectForKey:WCTLoginControllerErrorInfoKey_Subscription]);
    }
}




////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Method

//================================================================================
//
//================================================================================
+ (BOOL)validPassword:(NSString *)password withError:(NSError **)error
{
    BOOL result = NO;
    NSError *blockError = nil;
    
    do
    {
        if([password length]<=0)
        {
            BOOL isV6Server = ([WCTRestClientController shareRestClientController].serverVersionInfo.majorVersion>=6.0);
            blockError = PPErrorMake(WCTLoginController_ErrorCode_PassowrdLengthIsIllegal, isV6Server?WCTLC_MLS_PasswordLengthLimit_v6:WCTLC_MLS_PasswordLengthLimit, blockError);
            
            break;
        }
        
        //////////////////////////////////////////////////
        
        result = YES;
    }
    while (0);
    
    if(error!=nil)
    {
        *error = blockError;
    }
    
    return result;
}



//================================================================================
//
//================================================================================
+ (NSString *)removeBeforeAndAfterSpaceFromText:(NSString *)text
{
    NSString *returnString = nil;
    
    do
    {
        if(text==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        returnString = [text stringByTrimmingCharactersInSet:[NSCharacterSet whitespaceCharacterSet]];
    }
    while (0);
    
    return returnString;
}


//================================================================================
//
//================================================================================
+ (NSString *)autoFillServerAddressString:(NSString *)addressString
{
    NSString *autoFillAddressString = [self completeServerUrlHostString:addressString];
    
    do
    {
        if([autoFillAddressString length]<=0)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSURL *url = [NSURL URLWithString:autoFillAddressString];
        
        if(url==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSString *urlPath = [url path];
        
        if([urlPath length]>0 &&
           [urlPath hasSuffix:@"/"]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if([urlPath length]<=0)
        {
            autoFillAddressString = [autoFillAddressString stringByAppendingString:@"/"];
        }
        
        autoFillAddressString = [autoFillAddressString stringByAppendingString:WCTLoginControllerServerName];
    }
    while (0);
    
    return autoFillAddressString;
}


//================================================================================
//
//================================================================================
+ (NSString *)completeServerUrlHostString:(NSString *)urlHostString
{
    NSString *completeHostString = urlHostString;
    
    do
    {
        if([completeHostString length]<=0)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        BOOL addressStringHasInitialProtocol = YES;
        
        if([completeHostString hasPrefix:@"http://"]==NO &&
           [completeHostString hasPrefix:@"https://"]==NO)
        {
            addressStringHasInitialProtocol = NO;
            
            completeHostString = [@"https://" stringByAppendingString:completeHostString];
        }
        
        //////////////////////////////////////////////////
        
        NSURL *url = [NSURL URLWithString:completeHostString];
        
        if(url==nil)
        {
            break;
        }
        
        //有指定 port
        if([url port]!=nil)
        {
            // 原始位址, 沒指定 protocol
            if(addressStringHasInitialProtocol==NO)
            {
                // https port
                if([[url port] integerValue]==WCTLC_HttpsPort)
                {
                    completeHostString = [[[@"https://" stringByAppendingString:[url host]] stringByAppendingString:[NSString stringWithFormat:@":%td",WCTLC_HttpsPort]] stringByAppendingString:[url path]];
                    
                    url = [NSURL URLWithString:completeHostString];
                    
                    if(url==nil)
                    {
                        break;
                    }
                }
                else if([[url port] integerValue]==WCTLC_HttpPort)
                {
                    completeHostString = [[[@"http://" stringByAppendingString:[url host]] stringByAppendingString:[NSString stringWithFormat:@":%td",WCTLC_HttpPort]] stringByAppendingString:[url path]];
                    
                    url = [NSURL URLWithString:completeHostString];
                    
                    if(url==nil)
                    {
                        break;
                    }
                }
            }
        }
        // 沒指定 port
        else
        {
            NSString *newHost = nil;
            
            if([completeHostString hasPrefix:@"http://"]==YES)
            {
                newHost = [[url host] stringByAppendingString:[NSString stringWithFormat:@":%td",WCTLC_HttpPort]];
            }
            else
            {
                newHost = [[url host] stringByAppendingString:[NSString stringWithFormat:@":%td",WCTLC_HttpsPort]];
            }
            
            //////////////////////////////////////////////////

            if(newHost==nil)
            {
                break;
            }
            
            //////////////////////////////////////////////////

            completeHostString = [completeHostString stringByReplacingOccurrencesOfString:[url host] withString:newHost];
            
            //////////////////////////////////////////////////
            
            url = [NSURL URLWithString:completeHostString];
            
            if(url==nil)
            {
                break;
            }
        }
    }
    while (0);
    
    return completeHostString;
}


//================================================================================
//
//================================================================================
+ (NSURLComponents *)urlComponentsFromUrlPath:(NSString *)urlPath
{
    NSURLComponents *components = [[[NSURLComponents alloc] init] autorelease];
    
    do
    {
        if(components==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        NSString *trimmedUrlPath= [self removeBeforeAndAfterSpaceFromText:urlPath];
        
        if([trimmedUrlPath length]==0)
        {
            break;
        }
        
        NSURL *url = [NSURL URLWithString:trimmedUrlPath];
        
        if(url==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        components.scheme = [url scheme];
        components.path = [url path];
        components.host = [url host];
        components.port = [url port];
        if(components.port==nil)
        {
            [WCTLoginController adjustURLComponents:components withHttps:([components.scheme isEqualToString:@"https"])];
        }
    }
    while (0);
    
    return components;
}


//================================================================================
//
//================================================================================
+ (NSString *)urlPathFromUrlComponents:(NSURLComponents *)components
{
    return [[components URL] absoluteString];
}


//==============================================================================
//
//==============================================================================
+ (void)saveLoginSuccessTimeWithOfflineMode:(BOOL)offlineMode
{
    // !! 離線模式只有沒有紀錄時才要記，special case for 1.2 upgrade to 1.3
    // !! online每次都要記
    if(offlineMode==YES)
    {
        __block typeof(self) blockSelf = self;
        
        dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
            [[WCTRestClientController shareRestClientController] offlineLogin];
 
        });


        NSNumber *number = [PPSettingsController numberValueWithKey:WCTLoginControllerDictionaryKey_LastOnlineLoginSuccessTime];
        if (number==nil)
        {
            NSTimeInterval loginTimeInterval = [[NSDate date] timeIntervalSince1970];
            [PPSettingsController setNumberValue:@(loginTimeInterval) withKey:WCTLoginControllerDictionaryKey_LastOnlineLoginSuccessTime];
        }
    }
    else
    {
        NSTimeInterval loginTimeInterval = [[NSDate date] timeIntervalSince1970];
        [PPSettingsController setNumberValue:@(loginTimeInterval) withKey:WCTLoginControllerDictionaryKey_LastOnlineLoginSuccessTime];
    }
    
}



//==============================================================================
//
//==============================================================================
+ (BOOL)isLimitedAccountWithLoginInfo:(WCTRCLoginInfo *)loginInfo returnError:(NSError **)error;
{
    // !! 如果沒有訂閱錯誤，帳號又被暫停使用，要另外顯示訊息
    BOOL limitedAccount = NO;
    // !!如果沒有 accountSubscriptionStatus表示是舊server, 不會有暫停使用的狀態
    if ([loginInfo.accountInfo.accountSubscriptionStatus length]>0)
    {
        limitedAccount = [loginInfo.accountInfo.accountSubscriptionStatus isEqualToString:WCTRC_AccountSubscriptionStatus_TemplateInvalidate];
    }
    
    if (limitedAccount)
    {
        if(error!=NULL)
        {
            BOOL adminAccount = ([loginInfo.accountInfo.role isEqualToString:WCTRC_Role_Admin]==YES);
            
            NSString *errorMessage = WCTLC_MLS_WarningLimitedAccountForAdmin;
            
            if(adminAccount==NO)
            {
                errorMessage = WCTLC_MLS_WarningLimitedAccountForUser;
            }
            
            *error = PPErrorMake(WCTLoginController_ErrorCode_LimitedAccount, errorMessage, nil);
        }
    }
    
    return limitedAccount;
}


//================================================================================
//
//================================================================================
+ (BOOL)publicServerURLComponents:(NSURLComponents *)components
{
    BOOL result = NO;
    
    do
    {
        if(components==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        if([[components.host lowercaseString] hasSuffix:WCTLC_PublicCloudDomain]==NO&&
           [[components.host lowercaseString] isEqualToString:WCTLC_PublicCloudDomain_New]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        result = YES;
    }
    while (0);
    return result;
}


//================================================================================
//
//================================================================================
+ (void)adjustURLComponents:(NSURLComponents *)components
                  withHttps:(BOOL)https
{
    do
    {
        if(components==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if(https==YES)
        {
            components.scheme = @"https";
        }
        else
        {
            components.scheme = @"http";
        }
        
        //////////////////////////////////////////////////
        
        if([WCTLoginController publicServerURLComponents:components]==YES)
        {
            if([components.scheme isEqualToString:@"https"]==YES)
            {
                components.port = @(WCTLC_PublicCloudHttpsPort);
            }
            else
            {
                components.port = @(WCTLC_PublicCloudHttpPort);
            }
        }
        else
        {
            if([components.scheme isEqualToString:@"https"]==YES)
            {
                components.port = @(WCTLC_HttpsPort);
            }
            else
            {
                components.port = @(WCTLC_HttpPort);
            }
        }
    }
    while (0);
}


//================================================================================
//
//================================================================================
+ (BOOL)publicServerURLHost:(NSString *)host
{
    BOOL result = NO;
    
    do
    {
        if([host length]<=0)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        host = [self removeBeforeAndAfterSpaceFromText:host];
        
        //////////////////////////////////////////////////
        
        if([[host lowercaseString] hasSuffix:WCTLC_PublicCloudDomain]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        result = YES;
    }
    while (0);
    
    return result;
}


//================================================================================
//
//================================================================================
+ (NSUInteger)portFromURLHost:(NSString *)host
                    withHttps:(BOOL)https
{
    NSUInteger port = 0;
    
    do
    {
        if([host length]<=0)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        if([WCTLoginController publicServerURLHost:host]==YES)
        {
            if(https==YES)
            {
                port = WCTLC_PublicCloudHttpsPort;
            }
            else
            {
                port = WCTLC_PublicCloudHttpPort;
            }
        }
        else
        {
            if(https==YES)
            {
                port = WCTLC_HttpsPort;
            }
            else
            {
                port = WCTLC_HttpPort;
            }
        }
    }
    while (0);
    
    return port;
}


//==============================================================================
//
//==============================================================================
+ (NSString *)cloudServerURLString
{
    return WCTLC_PublicCloudUrl;
}


//==============================================================================
//
//==============================================================================
+ (NSURL *)cloudRegisterURLWithLongTrialDay:(BOOL)longTrialDay
{
    ////////////////////////////////////////////////////////////////////////////////////////////////////
    NSString *version = [[NSBundle mainBundle] objectForInfoDictionaryKey:@"CFBundleShortVersionString"];
    NSString *platform = nil;
#if TARGET_OS_IPHONE
    platform = @"ios";
#elif TARGET_OS_MAC
    platform = @"mac";
#endif
    NSString *registerURLString = [[WCTLC_PublicCloudRegisterUrl stringByAppendingString:longTrialDay?@"true":@"false"] stringByAppendingFormat:@"&version=%@&platform=%@", version, platform];
    return [NSURL URLWithString:registerURLString];
}


//==============================================================================
//
//==============================================================================
+ (BOOL)isCloudServerURLString:(NSString *)urlString
{
    if([urlString length]==0)
    {
        return NO;
    }
    NSURLComponents *componets = [self urlComponentsFromUrlPath:urlString];
    return ([[componets.host lowercaseString] hasPrefix:WCTLC_PublicCloudDomain_WCT]||
            [[componets.host lowercaseString] hasPrefix:WCTLC_PublicCloudDomain_New]);
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Login Method

//================================================================================
//
//================================================================================
- (void)loginServerByUrlComponents:(NSURLComponents *)urlComponents
                       accountName:(NSString *)accountName
                          password:(NSString *)password
                        completion:(void(^)(WCTRCLoginInfo *loginInfo, WCTLC_ChangePasswordReason changePasswordReason, NSDictionary <NSString *, NSError *> *errorDictionary))completion
{
    [self loginServerByUrlComponents:urlComponents
                         accountName:accountName
                            password:password
                   switchUserHandler:nil
                          completion:completion];
}


//================================================================================
//
//================================================================================
- (void)loginServerByUrlComponents:(NSURLComponents *)urlComponents
                       accountName:(NSString *)accountName
                          password:(NSString *)password
                 switchUserHandler:(void(^)(NSString *preAccountGuid, NSString *curAccountGuid))switchUserHandler
                        completion:(void(^)(WCTRCLoginInfo *loginInfo, WCTLC_ChangePasswordReason changePasswordReason, NSDictionary <NSString *, NSError *> *errorDictionary))completion
{
    [self logMessageWithFormat:@"%s in (%@, %@)", __PRETTY_FUNCTION__, urlComponents, accountName];
    self.inputServerString = [WCTLoginController urlPathFromUrlComponents:urlComponents];

    //////////////////////////////////////////////////
    
    self.loginCompletion = completion;
    
    //////////////////////////////////////////////////
    
    @autoreleasepool
    {
        __block WCTLC_ChangePasswordReason changePasswordReason = WCTLC_ChangePasswordReason_No;
        
        __block WCTRCLoginInfo *loginInfo = nil;
        
        NSError *error = nil;
        
        //////////////////////////////////////////////////
        
        do
        {
            if(self.operationQueue==nil)
            {
                error = PPErrorOperationFailed(error);
                break;
            }
            
            //////////////////////////////////////////////////
            
            __block NSBlockOperation *blockOperation = [[[NSBlockOperation alloc] init] autorelease];
            if(blockOperation==nil)
            {
                error = PPErrorOperationFailed(error);
                break;
            }
            
            //////////////////////////////////////////////////
            
            __block WCTLoginController *blockSelf = self;
            __block NSError            *blockError = nil;
            __block NSError            *subscriptionError = nil; // 訂閱警告，但可以登入的錯誤放這邊
            __block NSError            *duplicatedServerError = nil; // 盜版警告

            [blockOperation addExecutionBlock:^()
             {
                 @autoreleasepool
                 {
                     do
                     {

                         NSDate *clientDatabaseCreateTime = [PPSettingsController dateValueWithKey:WCTRestClientController_SettingsKey_DatabaseCreateTime];
                         [self logMessageWithFormat:@"%s local clientDatabaseCreateTime:%@", __PRETTY_FUNCTION__, clientDatabaseCreateTime];
                         
                         NSString *serverUrlString = [WCTLoginController removeBeforeAndAfterSpaceFromText:[WCTLoginController urlPathFromUrlComponents:urlComponents]];
                         [self logMessageWithFormat:@"%s check serverUrlString:%@ ", __PRETTY_FUNCTION__, serverUrlString];

                         //////////////////////////////////////////////////
                         
                         if([serverUrlString length]==0)
                         {
                             blockError = PPErrorOperationFailed(nil);
                             break;
                         }
                      
                         serverUrlString = [WCTLoginController autoFillServerAddressString:serverUrlString];
                         
                         //////////////////////////////////////////////////
                         
                         self.accountName = [WCTLoginController removeBeforeAndAfterSpaceFromText:accountName];
                         [self logMessageWithFormat:@"%s check accountName:%@ ", __PRETTY_FUNCTION__, self.accountName];

                         if([self checkAccountNameFormatWithError:&blockError]==NO)
                         {
                             break;
                         }
                         
                         //////////////////////////////////////////////////
                         
                         self.accountPassword = [WCTLoginController removeBeforeAndAfterSpaceFromText:password];
                         
                         //////////////////////////////////////////////////
                         
                         [self logMessageWithFormat:@"%s check accountPassword ", __PRETTY_FUNCTION__];
                         if([WCTLoginController validPassword:self.accountPassword withError:&blockError]==NO)
                         {
                             break;
                         }
                         
                         //////////////////////////////////////////////////
                         
                         NSURL *baseURL = [NSURL URLWithString:[serverUrlString lowercaseString]];
                         [self logMessageWithFormat:@"%s check baseURL:%@ ", __PRETTY_FUNCTION__, baseURL];
                         if(baseURL==nil)
                         {
                             NSString *errorReason = [NSString stringWithFormat:@"%@",serverUrlString];
                             
                             blockError = PPErrorMake(NSErrorCustom_Code_ParameterInvalidity, errorReason, blockError);
                             
                             break;
                         }
                         
                         //////////////////////////////////////////////////
                         
                         PPNetworkReachabilityControllerStatus status = [[PPNetworkReachabilityController networkReachabilityControllerForInternetConnection] status];
                         
                         //////////////////////////////////////////////////
                         [self logMessageWithFormat:@"%s check network:%@ ", __PRETTY_FUNCTION__, @(status)];
                         //沒有網路的情況下
                         if(status==PPNetworkReachabilityControllerStatus_ReachableNone)
                         {
                             // 檢查離線登入的原因
                             if([blockSelf loginOnOfflineModeWithUrlString:self.inputServerString networkConnection:NO error:&blockError]==YES)
                             {
                                 blockError = PPErrorMake(WCTLoginController_ErrorCode_WarningOfflineMode,
                                                          WCTLC_MLS_OfflineMode,
                                                          blockError);
                             }
                         }
                         else
                         {
                             [PPSettingsController setStringValue:serverUrlString
                                                          withKey:WCTRestClientController_SettingsKey_URLAddress];
                             
                             [self logMessageWithFormat:@"%s before changeURLAddress", __PRETTY_FUNCTION__];
                             
                             if([[WCTRestClientController shareRestClientController] changeURLAddress:serverUrlString withError:&blockError]==NO)
                             {
                                 break;
                             }
                             
                             //////////////////////////////////////////////////
                             
                             // MARK:版本檢查
                             NSError *versionError = nil;
                             WCTServerCanLoginResult canLoginResult = [[WCTRestClientController shareRestClientController] canLoginWithError:&versionError];
                             
                             [self logMessageWithFormat:@"%s canLoginWithError:%@, versionError:%@", __PRETTY_FUNCTION__, @(canLoginResult), versionError];
                             if (canLoginResult!=WCTServerCanLoginResult_YES||versionError!=nil)
                             {
                                 // !!版本不對也要看能不能離線登入
                                 // 檢查離線登入的原因
                                 if([blockSelf loginOnOfflineModeWithUrlString:self.inputServerString networkConnection:YES error:&blockError]==YES)
                                 {
                                     NSInteger errorCode = WCTLoginController_ErrorCode_WarningOfflineMode;
                                     NSString *message = [NSString stringWithFormat:@"%@(Error: %@)", WCTLC_MLS_OfflineMode, WCTLC_MLS_OfflineModeReason_VersionNotMatch];

                                     // !!如果是維護中要顯示不同的訊息
                                     if(versionError)
                                     {
                                         NSInteger statusCode = [WCTRestClientController statusCodeFromAFRKNetworkingError:versionError];
                                         if(statusCode==WCTServer_Common_ErrorCode_Restoring)
                                         {
                                             message = [NSString stringWithFormat:@"%@(Error: %@)", WCTLC_MLS_OfflineMode, WCTLC_MLS_OfflineModeReason_ServerMaintainance];
                                         }
                                         else
                                         {
                                             message = WCTLC_MLS_OfflineMode;
                                         }
                                     }
                                     else
                                     {
                                         if(canLoginResult==WCTServerCanLoginResult_NO_ServerIsNewer)
                                         {
                                             // !! clien較舊，要提示更新
                                             errorCode = WCTLoginController_ErrorCode_WarningOfflineModeWithUpdateClient;
                                             message = WCTLC_MLS_OfflineModeWithUpdateClient;
                                         }
                                         else if(canLoginResult==WCTServerCanLoginResult_NO_UnsupportedServerVersion)
                                         {
                                             message = [NSString stringWithFormat:@"%@(Error: %@)", WCTLC_MLS_OfflineMode, WCTLC_MLS_ErrorUnsupportedServerVersion];
                                         }
                                     }
                                     
                                     blockError = PPErrorMake(errorCode,
                                                              message,
                                                              blockError);
                                     [self logMessageWithFormat:@"%s 離線登入原因:%@", __PRETTY_FUNCTION__, message];
                                 }
                                 else
                                 {
                                     if(versionError)
                                     {
                                        NSInteger statusCode = [WCTRestClientController statusCodeFromAFRKNetworkingError:versionError];
                                         
                                         if(statusCode==WCTServer_Common_ErrorCode_Restoring)
                                         {
                                             blockError = PPErrorMake(WCTLoginController_ErrorCode_ServerMaintainance,
                                                                      WCTLC_MLS_ServerMaintaining,
                                                                      blockError);
                                         }
                                         else
                                         {
                                             blockError = versionError;
                                         }
                                     }
                                     else
                                     {
                                         if(canLoginResult==WCTServerCanLoginResult_NO_ServerIsNewer)
                                         {
                                             blockError = PPErrorMake(WCTLoginController_ErrorCode_ServerIsNewerThanClient, WCTLC_MLS_ErrorLocalVersionLowerThanServerVersion, blockError);
                                         }
                                         else if(canLoginResult==WCTServerCanLoginResult_NO_UnsupportedServerVersion)
                                         {
                                             blockError = PPErrorMake(WCTLoginController_ErrorCode_UnsupportedServerVersion, WCTLC_MLS_ErrorUnsupportedServerVersion, blockError);
                                         }
                                     }
                                     
                                     [self logMessageWithFormat:@"%s 不能離線登入原因:%@", __PRETTY_FUNCTION__, blockError];
                                 }
                                 break;
                             }
                             
                             [WCTRestClientController setCurLastMininumSupportedVersion:[[WCTRestClientController shareRestClientController] minSupportedVersion]];
                             
                             
                             //////////////////////////////////////////////////
                             // MARK: 公有雲，私有雲判斷
                             //
                             BOOL isMultiCompnay = [[WCTRestClientController shareRestClientController] isMultiCompanyWithError:&blockError];
                             
                             [self logMessageWithFormat:@"%s check isMultiCompnay:%@ error:%@", __PRETTY_FUNCTION__, @(isMultiCompnay), blockError];

                             if(blockError)
                             {
                                 // 檢查離線登入的原因
                                 if([blockSelf loginOnOfflineModeWithUrlString:self.inputServerString networkConnection:YES error:&blockError]==YES)
                                 {
                                     NSString *message = WCTLC_MLS_OfflineMode;
                                     
                                     // !!如果是維護中要顯示不同的訊息
                                     NSInteger statusCode = [WCTRestClientController statusCodeFromAFRKNetworkingError:blockError];
                                     if(statusCode==WCTServer_Common_ErrorCode_Restoring)
                                     {
                                         message = [NSString stringWithFormat:@"%@(Error: %@)", WCTLC_MLS_OfflineMode, WCTLC_MLS_OfflineModeReason_ServerMaintainance];
                                     }
                                     
                                     blockError = PPErrorMake(WCTLoginController_ErrorCode_WarningOfflineMode,
                                                              message,
                                                              blockError);
                                     [self logMessageWithFormat:@"%s isMultiCompanyWithError 離線登入的原因:%@", __PRETTY_FUNCTION__, blockError];

                                 }
                                 // else 直接回傳錯誤
                                 break;
                             }
                             
                             
                             
                             //////////////////////////////////////////////////
                             // MARK: 授權檢查 (私有雲)
                             WCTRCServerLicenseResponseResult *serverLicenseResponseResult = nil;
                             
                             if(isMultiCompnay==NO)
                             {
                                 serverLicenseResponseResult = [[WCTRestClientController shareRestClientController] serverLicenseWithError:&blockError];
                                 
                                 [self logMessageWithFormat:@"%s server license:%@ error:%@", __PRETTY_FUNCTION__,serverLicenseResponseResult, blockError];

                                 if(serverLicenseResponseResult==nil)
                                 {
                                     NSString *message = WCTLC_MLS_OfflineMode;
                                     
                                     // !!如果是維護中要顯示不同的訊息
                                     if(blockError)
                                     {
                                         NSInteger statusCode = [WCTRestClientController statusCodeFromAFRKNetworkingError:blockError];
                                         if(statusCode==WCTServer_Common_ErrorCode_Restoring)
                                         {
                                             message = [NSString stringWithFormat:@"%@(Error: %@)", WCTLC_MLS_OfflineMode, WCTLC_MLS_OfflineModeReason_ServerMaintainance];
                                         }
                                     }
                                     
                                     // 檢查離線登入的原因
                                     if([blockSelf loginOnOfflineModeWithUrlString:self.inputServerString networkConnection:YES error:&blockError]==YES)
                                     {
                                         
                                         blockError = PPErrorMake(WCTLoginController_ErrorCode_WarningOfflineMode,
                                                                  message,
                                                                  blockError);
                                     }
                                     
                                     break;
                                 }
                                 else
                                 {
                                     if([serverLicenseResponseResult.data.licenseMode isEqualToString:WCTRC_LicenseMode_BuyOut])
                                     {
                                         // !! 買斷模式要看serverRegisterStatus
                                         if([serverLicenseResponseResult.data.serverRegisterStatus isEqualToString:WCTRC_ServerNotActived]==YES)
                                         {
                                             blockError = PPErrorMake(WCTLoginController_ErrorCode_ServerIsUnActived,
                                                                      WCTLC_MLS_ServerIsNotActived,
                                                                      blockError);
                                             
                                             break;
                                         }
                                         else if([serverLicenseResponseResult.data.serverRegisterStatus isEqualToString:WCTRC_ServerTrialExpired]==YES)
                                         {
                                             blockError = PPErrorMake(WCTLoginController_ErrorCode_ServerTrialExpired,
                                                                      WCTLC_MLS_ServerTrialExpired,
                                                                      blockError);
                                             break;
                                         }
                                         else if([serverLicenseResponseResult.data.serverRegisterStatus isEqualToString:WCTRC_ServerNeedReinstall]==YES)
                                         {
                                             blockError = PPErrorMake(WCTLoginController_ErrorCode_ServerNeedReIntital,
                                                                      WCTLC_MLS_ServerIsNotActived,
                                                                      blockError);
                                             break;
                                         }
                                     }
                                     else if([serverLicenseResponseResult.data.licenseMode isEqualToString:WCTRC_LicenseMode_None])
                                     {
                                         // !!None，server還沒選擇授權模式
                                         blockError = PPErrorMake(WCTLoginController_ErrorCode_ServerIsUnActived,
                                                                  WCTLC_MLS_ServerIsNotActived,
                                                                  blockError);
                                         break;
                                     }
                                 }
                             }
                             
                             
                             //////////////////////////////////////////////////
                             // MARK: 登入
                             BOOL isServerV6 = ([WCTRestClientController shareRestClientController].serverVersionInfo.majorVersion>=6.0);
                             WCTRCLoginRequestObject *requestObject = [[[WCTRCLoginRequestObject alloc] init] autorelease];
                             
                             if(requestObject==nil)
                             {
                                 [self logMessageWithFormat:@"%s requestObject is nil", __PRETTY_FUNCTION__];
                                 blockError = PPErrorMake(NSErrorCustom_Code_ParameterInvalidity, @"requestObject is nil", blockError);
                                 
                                 break;
                             }
                             // !! 4.0 以上才支援 deviceID/ deviceName 登入
                             if([[WCTRestClientController  curLastMinumSupportedVersion] floatValue]<4.0)
                             {
                                 requestObject.deviceId = nil;
                                 requestObject.deviceName = nil;
                             }
                             
                             requestObject.email = self.accountName;
                             requestObject.password = self.accountPassword;
                             
                             //////////////////////////////////////////////////
                             
                             WCTRCLoginResponseResult *loginResponseResult = [[WCTRestClientController shareRestClientController] loginWithLoginRequest:requestObject withError:&blockError];
                             
                             [self logMessageWithFormat:@"%s login result:%@, versionError:%@", __PRETTY_FUNCTION__, loginResponseResult, blockError];
                             //////////////////////////////////////////////////
                             
                             if(loginResponseResult==nil)
                             {
                                 // MARK: login 失敗後的處理
                                 NSString *message = nil;
                                 
                                 NSUInteger statusCode = [WCTRestClientController statusCodeFromAFRKNetworkingError:blockError];
                                 
                                 switch(statusCode)
                                 {
                                     case WCTServer_Common_ErrorCode_Restoring:
                                     {
                                         // !! 要可以離線登入
                                         message = [NSString stringWithFormat:@"%@(Error: %@)", WCTLC_MLS_OfflineMode, WCTLC_MLS_OfflineModeReason_ServerMaintainance];
                                         break;
                                     }
                                     case WCTServer_Login_ErrorCode_AuthenticationFailed:
                                     {
                                         message = WCTLC_MLS_AccountNameOrPasswordWrongAndTryAgain;
                                         break;
                                     }
                                     case WCTServer_Login_ErrorCode_AccountNotFound:
                                     {
                                         statusCode = WCTLoginController_ErrorCode_AccountNotFound;
                                         message = WCTLC_MLS_AccountNotFound;
                                         break;
                                     }
                                     case WCTServer_Login_ErrorCode_InvalidParameter:
                                     {
                                         message = WCTLC_MLS_ErrorInvalidRequestParameters;
                                         break;
                                     }
                                     case WCTServer_Login_ErrorCode_LoginFailExceedAllowCount:
                                     {
                                         if(isServerV6==YES)
                                         {
                                             NSDictionary *exceptionDict = [self exceptionDictFromExceptionMessage:[blockError alertMessage]];
                                             NSNumber *loginFailureBlockPeriod = exceptionDict[@"loginFailureBlockPeriod"];
                                             NSNumber *loginFailureCountlimit = exceptionDict[@"loginFailureCountlimit"];
                                             message = [NSString stringWithFormat:WCTLC_MLS_ErrorLoginFailExceedAllowCountAndTryAgainLater_V6, loginFailureCountlimit, loginFailureBlockPeriod];
                                         }
                                         else
                                         {
                                             message = WCTLC_MLS_ErrorLoginFailExceedAllowCountAndTryAgainLater;
                                         }
                                         break;
                                     }
                                     case WCTServer_Login_ErrorCode_NotEnoughMemory:
                                     {
                                         message = WCTLC_MLS_ErrorNotEnoughMemory;
                                         
                                         break;
                                     }
                                     case WCTServer_Login_ErrorCode_LicenseNeedReactive:
                                     {
                                         message = WCTLC_MLS_ServerIsNotActived;
                                         break;
                                     }
                                     case WCTServer_Login_ErrorCode_LoginTimeIsLarge:
                                     {
                                         message = WCTLC_MLS_ErrorCurrentClientTimeNeedAdjust;
                                         
                                         //////////////////////////////////////////////////
                                         
                                         do
                                         {
                                             id object = [blockError.userInfo objectForKey:NSErrorCustom_Key_Object];
                                             
                                             if([object isKindOfClass:[NSError class]] == NO)
                                             {
                                                 break;
                                             }
                                             
                                             //////////////////////////////////////////////////
                                             
                                             NSString *localizedDescription = [object localizedDescription];
                                             
                                             if([localizedDescription length]<=0)
                                             {
                                                 break;
                                             }
                                             
                                             //////////////////////////////////////////////////
                                             
                                             NSDictionary *dictionary = [NSJSONSerialization JSONObjectWithData:[localizedDescription dataUsingEncoding:NSUTF8StringEncoding]
                                                                                                        options:NSJSONReadingMutableLeaves
                                                                                                          error:nil];
                                             
                                             NSString *serverReceiveRequestTime = [dictionary objectForKey:@"serverReceiveRequestTime"];
                                             
                                             if([serverReceiveRequestTime length]<=0)
                                             {
                                                 break;
                                             }
                                             
                                             //////////////////////////////////////////////////
                                             
                                             NSDate *serverDate = [NSDate dateFromString:serverReceiveRequestTime
                                                                                  format:@"yyyy-MM-dd HH:mm:ss.SSS"
                                                                                timeZone:[NSTimeZone timeZoneWithAbbreviation:@"UTC"]];
                                             serverReceiveRequestTime = [serverDate stringWithFormat:NSDateFormat_Second];
                                             
                                             message = [WCTLC_MLS_ErrorCurrentClientTimeNeedAdjust stringByReplacingOccurrencesOfString:@"xxxx/xx/xx xx:xx:xx" withString:serverReceiveRequestTime];
                                             
                                             
                                         } while (0);
                                         
                                         break;
                                     }
                                     case WCTServer_Login_ErrorCode_NotAllowedIP:
                                     {
                                         message = WCTLC_MLS_ErrorNotAllowedIP;
                                         
                                         break;
                                     }
                                     case WCTServer_Login_ErrorCode_AD_AuthenticationFailed:
                                     {
                                         message = WCTLC_MLS_AD_LoginAuthenticationFailed;
                                         break;
                                     }
                                     case WCTServer_Login_ErrorCode_AD_FailToConnectServer:
                                     case WCTServer_Login_ErrorCode_AD_NotLoginWithAccuont:
                                     {
                                         message = WCTLC_MLS_AD_FailToConnectServer;
                                         
                                         break;
                                     }
                                     case WCTServer_Login_ErrorCode_AD_InvalidPassword:
                                     {
                                         message = WCTLC_MLS_AD_InvalidPassword;
                                         break;
                                     }
                                     case WCTServer_Login_ErrorCode_AD_InvalidUsername:
                                     {
                                         message = WCTLC_MLS_AD_InvalidUserName;
                                         break;
                                     }
                                     case WCTServer_Login_ErrorCode_AD_ExpiredPassword:
                                     {
                                         message = WCTLC_MLS_AD_ExpiredPassword;
                                         break;
                                     }
                                     case WCTServer_Login_ErrorCode_AD_ExpiredUserAccount:
                                     {
                                         message = WCTLC_MLS_AD_ExpiredUserAccount;
                                         break;
                                     }
                                     case WCTServer_Login_ErrorCode_AD_FirstLoginNeedChangePassword:
                                     {
                                         message = WCTLC_MLS_AD_FirstLoginNeedToChangedPassword;
                                         break;
                                     }
                                     case WCTServer_Login_ErrorCode_AD_ServerBusy:
                                     {
                                         message = WCTLC_MLS_AD_ServerIsBusy;
                                         break;
                                     }
                                     case WCTServer_Login_ErrorCode_AD_CanNotProcessRequest:
                                     {
                                         message = WCTLC_MLS_AD_CanNotProcessRequest;
                                         break;
                                     }
                                     case WCTServer_Login_ErrorCode_UnbindingDevice:
                                     {
                                         message = WCTLC_MLS_ErrorUnbindingDevice;
                                         
                                         break;
                                     }
                                     case WCTServer_Login_ErrorCode_SubscriptionLocked:
                                     {
                                         // !! Admin會登入成功，所以這邊會進來的應該都是一般用戶
                                         NSInteger oriStatusCode = statusCode;
                                         statusCode = WCTLoginController_ErrorCode_SubscriptionLock;
                                         if(isServerV6==YES)
                                         {
                                            message = [NSString stringWithFormat:WCTLC_MLS_ErrorSubscriptionLockForUser_V6, @(oriStatusCode)];
                                         }
                                         else
                                         {
                                             message = WCTLC_MLS_ErrorSubscriptionLockForUser;
                                         }
                                         break;
                                     }
                                     case WCTServer_Login_ErrorCode_LockByLoginFailure:
                                     {
                                         NSInteger oriStatusCode = statusCode;
                                         statusCode = WCTLoginController_ErrorCode_LockByLoginFailure;
                                         message = [NSString stringWithFormat:WCTLC_MLS_ErrorAccountLockByLoginFailureForUser, @(oriStatusCode)];
                                         break;
                                     }
                                     case WCTServer_Login_ErrorCode_LockByAccountActiveExpire:
                                     {
                                         NSInteger oriStatusCode = statusCode;
                                         statusCode = WCTLoginController_ErrorCode_LockByAccountActiveExpired;
                                         message = [NSString stringWithFormat:WCTLC_MLS_ErrorAccountLockByActiveExpiredForUser, @(oriStatusCode)];
                                         break;
                                     }
                                     case WCTServer_Login_ErrorCode_RunTimeException:
                                     {
                                         message = WCTLC_MLS_ErrorRuntimeException;
                                         break;
                                     }

                                     default:
                                     {
                                         // 有可能是系統錯誤
                                         message = blockError.localizedDescription;
                                         
                                         break;
                                     }
                                 }
                                 
                                 [self logMessageWithFormat:@"%s login 失敗(%@) - %@", __PRETTY_FUNCTION__,@(statusCode), message];
                                 //////////////////////////////////////////////////
                                 
                                 // server 返回
                                 if(statusCode>0)
                                 {
                                     if (statusCode==WCTServer_Common_ErrorCode_Restoring)
                                     {
                                         blockError = PPErrorMake(WCTLoginController_ErrorCode_WarningOfflineMode,
                                                                  message,
                                                                  blockError);
                                     }
                                     else
                                     {
                                         blockError = PPErrorMake(statusCode, message, blockError);
                                         
                                     }
                                 }
                                 
                                 //////////////////////////////////////////////////
                                 
                                 //登入失敗要清除密碼，防止點擊說明圖示，viewController又切進來造成自動登入的情況
                                 blockSelf.accountPassword = @"";
                                 [WCTRestClientController encryptAccountPasswordToUserDefault:blockSelf.accountPassword];
                                 
                                 break;
                             } // end of if(loginResponseResult==nil)
                             else
                             {
                                 // MARK: login success後的處理
                                 
                                 //////////////////////////////////////////////////
                                 // lock issue判斷, server 6.0.0以上才需要
                                 if(isServerV6==YES)
                                 {
                                     [self logMessageWithFormat:@"%s accountInfo.lockedIssue: %@", __PRETTY_FUNCTION__, loginResponseResult.data.accountInfo.lockedIssue];
                                     
                                     // !! server 6.0以上不用參考這個值，這邊清空，以避免後面顯示alert時有干擾
                                     loginInfo.accountInfo.accountSubscriptionStatus = nil;
                                     
                                     BOOL adminAccount = NO;
                                     
                                     if([loginResponseResult.data.accountInfo.role isEqualToString:WCTRC_Role_Admin]==YES)
                                     {
                                         adminAccount = YES;
                                     }
                                     
                                     // 管理員要可以進入
                                     if(adminAccount==NO)
                                     {
                                         if([loginResponseResult.data.accountInfo.lockedIssue isEqualToString:WCTRC_LockedIssue_None]==NO)
                                         {
                                             if([loginResponseResult.data.accountInfo.lockedIssue isEqualToString:WCTRC_LockedIssue_LoginFailed])
                                             {
                                                 NSString *errorMessage = [NSString stringWithFormat:WCTLC_MLS_ErrorAccountLockByLoginFailureForAdmin, @(WCTServer_Login_ErrorCode_LockByLoginFailure)];
                                                 blockError = PPErrorMake(WCTLoginController_ErrorCode_LockByLoginFailure, errorMessage, nil);
                                             }
                                             else if([loginResponseResult.data.accountInfo.lockedIssue isEqualToString:WCTRC_LockedIssue_AccountNotActiveExpired])
                                             {
                                                 NSString *errorMessage = [NSString stringWithFormat:WCTLC_MLS_ErrorAccountLockByActiveExpiredForAdmin, @(WCTServer_Login_ErrorCode_LockByAccountActiveExpire)];
                                                 blockError = PPErrorMake(WCTLoginController_ErrorCode_LockByAccountActiveExpired, errorMessage, nil);
                                             }
                                             else
                                             {
                                                 NSString *errorMessage = [NSString stringWithFormat:WCTLC_MLS_ErrorSubscriptionLockForAdmin_V6, @(WCTServer_Login_ErrorCode_SubscriptionLocked)];
                                                 blockError = PPErrorMake(WCTLoginController_ErrorCode_SubscriptionLock, errorMessage, nil);
                                             }
                                             break;
                                         }
                                     }
                                 }
                                 
                                 //////////////////////////////////////////////////
                                 // 檢查server是否合法 (訂閱模式例外)
                                 if([serverLicenseResponseResult.data.licenseMode isEqualToString:WCTRC_LicenseMode_Subscription]==NO &&
                                    [[WCTRestClientController shareRestClientController] checkServerIslegalWithLoginCode:loginResponseResult.loginCode error:&blockError]==NO)
                                 {
                                     break;
                                 }
            
                                 //////////////////////////////////////////////////
                                 // 帳號已被刪除，不應登入
                                 if([loginResponseResult.data.accountInfo.status compare:WCTRC_Status_Deleted]==NSOrderedSame)
                                 {
                                     blockError = PPErrorMake(blockError.code, @"The Account is deleted", blockError);
                                     
                                     break;
                                 }

                                 // 帳號已離職，不應登入
                                 else if([loginResponseResult.data.accountInfo.status compare:WCTRC_Status_Resigned]==NSOrderedSame)
                                 {
                                     blockError = PPErrorMake(blockError.code, @"The Account is resigned", blockError);
                                     break;
                                 }
                                 
                                 //////////////////////////////////////////////////
                                 
                                 // 成功登入，但是該帳號第一次登入需要更新密碼
                                 if([loginResponseResult.data.accountInfo.status compare:WCTRC_Status_InActive]==NSOrderedSame)
                                 {
                                     changePasswordReason = WCTLC_ChangePasswordReason_FirstLogin;
                                 }
                                 
                                 //////////////////////////////////////////////////
                                 
                                 // 成功登入，但是該帳號的密碼超過n天沒有更新，需要更新密碼
                                 if([loginResponseResult.data.accountInfo.status compare:WCTRC_Status_Active]==NSOrderedSame)
                                 {
                                     if (loginResponseResult.data.isNeedChangePassword)
                                     {
                                         changePasswordReason = WCTLC_ChangePasswordReason_PasswordExpired;
                                     }
                                 }
                                 
                                 //////////////////////////////////////////////////
                                 ///
                                 // 只先寫入登入資訊，不能寫設定值
                                 loginInfo = loginResponseResult.data;
                                 
                              
                                 //////////////////////////////////////////////////
                                 // MARK: 判斷訂閱模式
                                 if(loginResponseResult.data.serverStatus!=nil && loginResponseResult.data.serverStatus.subscriptionStatus!=nil)
                                 {
                                     [self logMessageWithFormat:@"%s loginResponseResult.data.serverStatus: %@", __PRETTY_FUNCTION__, loginResponseResult.data.serverStatus];
                                     BOOL adminAccount = NO;
                                     
                                     if([loginResponseResult.data.accountInfo.role isEqualToString:WCTRC_Role_Admin]==YES)
                                     {
                                         adminAccount = YES;
                                     }
                                     
//                                     loginInfo.serverStatus.issueStatus = WCTRC_IssueState_Warning;
//                                     loginResponseResult.data.serverStatus.subscriptionStatus = WCTRC_SubscriptionState_Lock;
                                     //////////////////////////////////////////////////
                                        // 訂閱狀態警告判斷
                                     
                                     if([loginResponseResult.data.serverStatus.subscriptionStatus isEqualToString:WCTRC_SubscriptionState_TrialLeft2Week]==YES||
                                        [loginResponseResult.data.serverStatus.subscriptionStatus isEqualToString:WCTRC_SubscriptionState_SubscriptionLeft2Week]==YES)
                                     {
                                         // !! 這邊改為不秀alert，在主畫面才顯示過期日
                                         // 因為macOS，iOS流程不同，所以不在這邊做了，改為loginFlow自行取loginInfo.serverStatus.expiredDate來處理

                                     }
                                     else if([loginResponseResult.data.serverStatus.subscriptionStatus isEqualToString:WCTRC_SubscriptionState_TrialOutOfDate]==YES)
                                     {
                                         NSString *dateString = [loginInfo.serverStatus.expiredDate stringWithFormat:NSDateFormat_Day];
                                         NSString *errorMessage = [NSString stringWithFormat:WCTLC_MLS_ServerOutOfDateMsgForAdmin, dateString];
                                         
                                         if(adminAccount==NO)
                                         {
                                             errorMessage = [NSString stringWithFormat:WCTLC_MLS_ServerOutOfDateMsgForUser, dateString];
                                         }
                                         
                                         subscriptionError = PPErrorMake(WCTLoginController_ErrorCode_TrialOutOfDate, errorMessage, subscriptionError);
                                     }
                                     else if([loginResponseResult.data.serverStatus.subscriptionStatus isEqualToString:WCTRC_SubscriptionState_SubscriptionOutOfDate]==YES)
                                     {
                                         NSString *dateString = [loginInfo.serverStatus.expiredDate stringWithFormat:NSDateFormat_Day];
                                         NSString *errorMessage = [NSString stringWithFormat:WCTLC_MLS_ServerOutOfDateMsgForAdmin, dateString];
                                         
                                         if(adminAccount==NO)
                                         {
                                             errorMessage = [NSString stringWithFormat:WCTLC_MLS_ServerOutOfDateMsgForUser, dateString];
                                         }

                                         subscriptionError = PPErrorMake(WCTLoginController_ErrorCode_SubscriptionOutOfDate, errorMessage, subscriptionError);
                                     }
                                     else if([loginResponseResult.data.serverStatus.subscriptionStatus isEqualToString:WCTRC_SubscriptionState_Lock]==YES)
                                     {
                                         NSString *errorMessage = nil;
                                         if(adminAccount==NO)
                                         {
                                             if(isServerV6==YES)
                                             {
                                                 errorMessage = [NSString stringWithFormat:WCTLC_MLS_ErrorSubscriptionLockForUser_V6, @(WCTServer_Login_ErrorCode_SubscriptionLocked)];
                                             }
                                             else
                                             {
                                                 errorMessage = WCTLC_MLS_ErrorSubscriptionLockForUser;
                                             }
                                         }
                                         else
                                         {
                                             if(isServerV6==YES)
                                             {
                                                 errorMessage = [NSString stringWithFormat:WCTLC_MLS_ErrorSubscriptionLockForAdmin_V6, @(WCTServer_Login_ErrorCode_SubscriptionLocked)];
                                             }
                                             else
                                             {
                                                 errorMessage = WCTLC_MLS_ErrorSubscriptionLockForAdmin;
                                             }
                                         }
                                         
                                         blockError = PPErrorMake(WCTLoginController_ErrorCode_SubscriptionLock, errorMessage, nil);
                                         
                                         // !! 因為後面還要判斷盜版，所以這邊改為不break, 等盜版訊息判斷完，再一起判斷是否要break
                                         // Admin可以登入，所以不break, 一般User不能登入，所以直接break;
                                         if(adminAccount==NO)
                                         {
                                             break;
                                         }
                                     }
                                 }
                                 
                                 //////////////////////////////////////////////////

                                 // 盜版狀態警告判斷
                                 [self logMessageWithFormat:@"%s loginInfo.serverStatus.issueStatus: %@", __PRETTY_FUNCTION__, loginInfo.serverStatus.issueStatus];
                                 if([loginInfo.serverStatus.issueStatus isEqualToString:WCTRC_IssueState_Locked]==YES)
                                 {
                                     // 重覆伺服器超過期限，帳號鎖定
                                     blockError = PPErrorMake(WCTLoginController_ErrorCode_DuplicatedServerLock, WCTLC_MLS_LockedAccount, nil);
                                     break;
                                 }
                                 else if([loginInfo.serverStatus.issueStatus isEqualToString:WCTRC_IssueState_Warning]==YES)
                                 {
                                     NSString *errorMessage = WCTLC_MLS_WarningDuplicatedServerForUser;
                                     
                                     if([loginInfo.accountInfo.role isEqualToString:WCTRC_Role_Admin]==YES)
                                     {
                                         errorMessage = WCTLC_MLS_WarningDuplicatedServerForAdmin;
                                     }
                                     
                                     //////////////////////////////////////////////////
                                     
                                     duplicatedServerError = PPErrorMake(WCTLoginController_ErrorCode_DuplicatedServerWarning, errorMessage, nil);
                                 }
                                 
                                 //////////////////////////////////////////////////

                                 // 訂閱，或盜版帳號鎖住，直接退出
                                 if(blockError!=nil)
                                 {
                                     break;
                                 }
                          
                                 //////////////////////////////////////////////////
                                 
                                 //檢查ServerGuid是否符合
//                                 
//                                 NSString *guidForServer = [PPSettingsController stringValueWithKey:WCTSettingsKey_ServerGUID];
//                                 
//                                 if(guidForServer!=nil &&
//                                    [guidForServer isGUID]==YES &&
//                                    [guidForServer compare:loginResponseResult.data.serverId options:NSCaseInsensitiveSearch]!=NSOrderedSame)
//                                 {
//                                     blockError = PPErrorMake(WCTLoginController_ErrorCode_WarningSwitchServer, WCTLC_MLS_SwitchUser, blockError);
//                                     
//                                     break;
//                                 }
                                 
                                 //////////////////////////////////////////////////
                                 
                                 //檢查AccountGuid是否符合
                                 
                                 NSString *guidForAccount = [PPSettingsController stringValueWithKey:WCTSettingsKey_AccountGUID];
                                 
                                 if(guidForAccount!=nil &&
                                    [guidForAccount isGUID]==YES &&
                                    [guidForAccount compare:loginResponseResult.data.accountInfo.guid options:NSCaseInsensitiveSearch]!=NSOrderedSame)
                                 {
                                     blockError = PPErrorMake(WCTLoginController_ErrorCode_WarningSwitchUser, WCTLC_MLS_SwitchUser, blockError);
                                     
                                     if(switchUserHandler)
                                     {
                                         switchUserHandler(guidForAccount, loginResponseResult.data.accountInfo.guid);
                                     }
                                 }
                                 
                                 // !!!因為有記錄使用者設定的問題，所以所有設定都要在switch user 之後再寫入
                                 //////////////////////////////////////////////////
                                 // MARK: 檢查是否顯示匯出功能
                                 [WCTLoginController getDisplayOption];
                                 
                                 //////////////////////////////////////////////////

                                 // MARK: 記錄帳號資訊 (匯出，列印，掃描)
                                 
                                 [PPSettingsController setIntegerValue:loginResponseResult.data.accountInfo.exportAbility
                                                               withKey:WCTSettingsKey_ExportAbility];
                                 
                                 [PPSettingsController setIntegerValue:loginResponseResult.data.accountInfo.secretary
                                                               withKey:WCTSettingsKey_HelpScanningAbility];
                                 
                                 [PPSettingsController setIntegerValue:loginResponseResult.data.accountInfo.printAbility
                                                               withKey:WCTSettingsKey_PrintAbility];
                                 
                                 //////////////////////////////////////////////////
                                 // 登入成功要記錄server db build time
                                 NSError *error = nil;
                                 WCTRCDateTimeResponseResult *result = [[WCTRestClientController shareRestClientController] serverDBBuildTimeWithError:&error];
                                 
                                 if (error==nil && result!=nil)
                                 {
                                     [PPSettingsController setDateValue:result.data withKey:WCTSettingsKey_ServerDBBuildTime];
                                 }
                                 
                                 
                         
                             } // end of if(loginResponseResult!=nil)
                         } // end of if(status!=PPNetworkReachabilityControllerStatus_ReachableNone)
                         
                     }while(0);
                     
                     //////////////////////////////////////////////////
                     
                     if(blockOperation.isCancelled==YES && blockError==nil)
                     {
                         blockError = PPErrorOperationCancel(blockError);
                     }
                     
                     //////////////////////////////////////////////////
                     
                     if(completion!=nil)
                     {
                         NSMutableDictionary *dictionary = [NSMutableDictionary dictionaryWithDictionary:@{WCTLoginControllerDictionaryKey_NeedChangedPassword:@(changePasswordReason)}];
                         
                         if(loginInfo!=nil)
                         {
                             [dictionary setObject:loginInfo forKey:WCTLoginControllerDictionaryKey_LoginInfo];
                         }
                         
                         //////////////////////////////////////////////////
                         
                         if(blockError!=nil)
                         {
                             [dictionary setObject:blockError
                                            forKey:WCTLoginControllerErrorInfoKey_Login];
                         }
                         
                         if(subscriptionError!=nil)
                         {
                             [dictionary setObject:subscriptionError
                                            forKey:WCTLoginControllerErrorInfoKey_Subscription];
                         }
                         
                         
                         if(duplicatedServerError!=nil)
                         {
                             [dictionary setObject:duplicatedServerError
                                            forKey:WCTLoginControllerErrorInfoKey_DuplicatedServer];
                         }
                         //////////////////////////////////////////////////
    
                         [self logMessageWithFormat:@"%s login result dictionary: %@", __PRETTY_FUNCTION__, dictionary];
                         [self performSelectorOnMainThread:@selector(mainThreadCallCompletionWithDictionary:)
                                                withObject:dictionary
                                             waitUntilDone:NO];
                     }
                 }
             }];
            
            //////////////////////////////////////////////////
            
            @synchronized(self.operationQueue)
            {
                if([self.operationQueue operationCount]>0)
                {
                    [self.operationQueue cancelAllOperations];
                }
                
                [self.operationQueue addOperation:blockOperation];
            }
            
        }while(0);
        
        //////////////////////////////////////////////////
        
        if(error!=nil &&
           completion!=nil)
        {
            [self logMessageWithFormat:@"%s login error: %@", __PRETTY_FUNCTION__, error];
            [self performSelectorOnMainThread:@selector(mainThreadCallCompletionWithDictionary:)
                                   withObject:[NSMutableDictionary dictionaryWithDictionary:@{WCTLoginControllerDictionaryKey_NeedChangedPassword:@(WCTLC_ChangePasswordReason_No),
                                                                                              WCTLoginControllerErrorInfoKey_Login:error}]
                                waitUntilDone:NO];
        }
    }
    
    [self logMessageWithFormat:@"%s out", __PRETTY_FUNCTION__];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Forget Password Method

//================================================================================
//
//================================================================================
- (NSURL *)forgetPasswordURLWithUrlString:(NSString *)urlString error:(NSError **)error
{
    NSError *returnError = nil;
    
    // MARK: 忘記密碼網頁網址如何取得
    NSURL *forgetPasswordURL = nil;
    
    do
    {
        if([urlString length]<=0)
        {
            returnError = PPErrorMake(WCTLoginController_ErrorCode_ServerAddressEmpty, WCTLC_MLS_InputServerAddress, nil);
            
            break;
        }
        
        //////////////////////////////////////////////////
        
        urlString = [[WCTLoginController completeServerUrlHostString:urlString] stringByAppendingString:@"/#/forgot-password"];
      
        if([urlString length]<=0)
        {
            returnError = PPErrorMake(WCTLoginController_ErrorCode_ServerAddressEmpty, WCTLC_MLS_InputServerAddress, nil);
            
            break;
        }
        
        forgetPasswordURL = [NSURL URLWithString:urlString];
        
    }
    while (0);
    
    if(error!=nil)
    {
        *error = returnError;
    }
    return forgetPasswordURL;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Server Address List Method

//================================================================================
//
//================================================================================
- (NSArray <NSString *> *)recordServerList
{
    return [PPSettingsController arrayValueWithKey:WCTLoginControllerSettingKey_DisplayURLAddressList];
}


//================================================================================
//
//================================================================================
- (void)recordServerListAddServerURL:(NSString *)serverUrl
{
    do
    {
        if([serverUrl length]<=0)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSArray *serverList = [self recordServerList];
        
        if([serverList count]<=0)
        {
            [PPSettingsController setArrayValue:@[serverUrl] withKey:WCTLoginControllerSettingKey_DisplayURLAddressList];
        }
        else
        {
            NSURLComponents *newServerURLComponents = [WCTLoginController urlComponentsFromUrlPath:serverUrl];
            
            NSInteger sameServerUrlIndex = NSNotFound;
            
            // 尋找有無重覆的 URL
            for(NSString *recordServerUrl in serverList)
            {
                NSURLComponents *recordServerURLComponents = [WCTLoginController urlComponentsFromUrlPath:recordServerUrl];
                
                if([[newServerURLComponents host] isEqualToString:[recordServerURLComponents host]]==YES)
                {
                    if([[newServerURLComponents path] length]<=0 || [[newServerURLComponents path] isEqualToString:[recordServerURLComponents path]]==YES)
                    {
                        sameServerUrlIndex = [serverList indexOfObject:recordServerUrl];
                        break;
                    }
                }
            }
           
            
            //////////////////////////////////////////////////
          
            NSMutableArray *newRecordServerList = nil;
            
            if(sameServerUrlIndex!=NSNotFound)
            {
                newRecordServerList = [NSMutableArray arrayWithArray:serverList];
                [newRecordServerList replaceObjectAtIndex:sameServerUrlIndex withObject:serverUrl];
                [newRecordServerList exchangeObjectAtIndex:0 withObjectAtIndex:sameServerUrlIndex];
            }
            else
            {
                newRecordServerList = [NSMutableArray arrayWithObject:serverUrl];
                [newRecordServerList addObject:[serverList firstObject]];
            }
            
            //////////////////////////////////////////////////

            if([newRecordServerList count]>0)
            {
                [PPSettingsController setArrayValue:newRecordServerList withKey:WCTLoginControllerSettingKey_DisplayURLAddressList];
            }
        }
    }
    while (0);
}


//================================================================================
//
//================================================================================
- (void)completedServerListRecord
{
    NSMutableArray *newRecordServerList = [NSMutableArray array];
    
    NSArray *recordServerList = [self recordServerList];
    
    for(NSInteger index=0; index<recordServerList.count; index++)
    {
        NSString *serverPath = [recordServerList objectAtIndex:index];
        
        NSURLComponents *components = [WCTLoginController urlComponentsFromUrlPath:serverPath];
        
        //////////////////////////////////////////////////

        if([[components scheme] length]<=0)
        {
            components.scheme = @"https";
        }
        
        //////////////////////////////////////////////////

        if([[components host] length]<=0)
        {
            components.host = components.path;
            components.path = nil;
        }

        //////////////////////////////////////////////////

        if([components port]==nil)
        {
            if([components.scheme isEqualToString:@"https"]==YES)
            {
                components.port = @(WCTLC_HttpsPort);
            }
            else
            {
                components.port = @(WCTLC_HttpPort);
            }
        }
        
        //////////////////////////////////////////////////

        if([[components host] length]>0 &&
           [newRecordServerList indexOfObject:[components host]]==NSNotFound)
        {
            NSString *completedServerPath = [WCTLoginController urlPathFromUrlComponents:components];
            
            if([completedServerPath length]>0)
            {
                [newRecordServerList addObject:completedServerPath];
            }
        }
    }
    
    //////////////////////////////////////////////////

    if([newRecordServerList count]>0)
    {
        [PPSettingsController setArrayValue:newRecordServerList withKey:WCTLoginControllerSettingKey_DisplayURLAddressList];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Subscription Method

//================================================================================
//
//================================================================================
- (void)subscriptionURLWithCompletion:(void(^)(NSURL *subscriptionURL, NSError *error))completion
{
    NSError *returnError = nil;
    
    do
    {
        self.subscriptionURLCompletion = completion;
        
        if(self.operationQueue==nil)
        {
            returnError = PPErrorOperationFailed(returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        __block NSBlockOperation *blockOperation = [[[NSBlockOperation alloc] init] autorelease];
        if(blockOperation==nil)
        {
            returnError = PPErrorOperationFailed(returnError);
            break;
        }
        
        //////////////////////////////////////////////////
        
        __block NSError *blockError = nil;

        [blockOperation addExecutionBlock:^()
         {
             WCTRCGetSubscriptionURLResponseResult *subscriptionURLResponseResult = [[WCTRestClientController shareRestClientController] subscriptionURLWithError:&blockError];

             //////////////////////////////////////////////////

             NSMutableDictionary *dictionary = [NSMutableDictionary dictionary];
             
             if(blockError!=nil)
             {
                 [dictionary setObject:blockError
                                forKey:WCTLoginControllerErrorInfoKey_Subscription];
             }
             else if([subscriptionURLResponseResult.data length]>0)
             {
                 NSURL *subscriptionURL = [NSURL URLWithString:subscriptionURLResponseResult.data];
                 
                 if(subscriptionURL!=nil)
                 {
                     [dictionary setObject:subscriptionURL
                                    forKey:WCTLoginControllerDictionaryKey_SubscriptionURL];
                 }
             }
             
             //////////////////////////////////////////////////

             [self performSelectorOnMainThread:@selector(mainThreadFinishedSubscriptionURLWithDictionary:)
                                    withObject:dictionary
                                 waitUntilDone:NO];
             
         }];
        
        //////////////////////////////////////////////////
        
        @synchronized(self.operationQueue)
        {
            if([self.operationQueue operationCount]>0)
            {
                [self.operationQueue cancelAllOperations];
            }
            
            [self.operationQueue addOperation:blockOperation];
        }
        
    }while(0);
    
    if(returnError!=nil)
    {
        completion(nil, returnError);
    }
}


//==============================================================================
//
//==============================================================================
+ (void)upgradeFromRetailIfNeededWithCheckHandler:(BOOL(^)(void))checkHandler completeHandler:(void(^)(BOOL needUpgrade, BOOL upgradeSuccess, NSError *error))completeHandler
{
    __block BOOL needUpgrade = NO;
    
    void(^checkProcess)(void) = ^(void) {
        if(checkHandler)
        {
            needUpgrade = checkHandler();
        }
        
        if(needUpgrade==NO)
        {
            if(completeHandler)
            {
                completeHandler(needUpgrade, NO, nil);
            }
            return ;
        }
        
        //////////////////////////////////////////////////
        dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
            
            BOOL success = NO;
            NSError *error = nil;
            
            BOOL hasUpgrade = [[PPSettingsController numberValueWithKey:WCTLoginControllerDictionaryKey_HasUpgradeTrialPeriod] boolValue];
            
            if(hasUpgrade==NO)
            {
                [[WCTRestClientController shareRestClientController] upgradeFromRetailWithError:&error];
                
                NSInteger errorCode = [WCTRestClientController statusCodeFromAFRKNetworkingError:error];
                // 422表示已升級或不符合升級資格
                if(error==nil || errorCode==422)
                {
                    success = YES;
                    [PPSettingsController setNumberValue:@(YES) withKey:WCTLoginControllerDictionaryKey_HasUpgradeTrialPeriod];
                }
            }

            [error retain];
            //////////////////////////////////////////////////
            dispatch_async(dispatch_get_main_queue(), ^{
                if(completeHandler)
                {
                    completeHandler(needUpgrade, success, error);
                }
                [error release];
            });
        });
    };
    
    //////////////////////////////////////////////////
    if([NSThread isMainThread])
    {
        checkProcess();
    }
    else
    {
        dispatch_async(dispatch_get_main_queue(), ^{
            checkProcess();
        });
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class get displayOption


//==============================================================================
// MARK: 要在background thread call
//==============================================================================
+ (void)getDisplayOption
{
    
    NSError *blockError = nil;
    WCTRCUIDisplayOptionResponseResult *displayOptionResponstResult = nil;
    
    displayOptionResponstResult = [[WCTRestClientController shareRestClientController] displayOptionWithError:&blockError];
    
    if(displayOptionResponstResult!=nil)
    {
        [PPSettingsController setIntegerValue:displayOptionResponstResult.data.isHideExportFunction?0:1
                                      withKey:WCTSettingsKey_ShowExportUI];
        [PPSettingsController setIntegerValue:displayOptionResponstResult.data.isHideMycard?0:1
                                      withKey:WCTSettingsKey_ShowMyCardUI];
        [PPSettingsController setIntegerValue:displayOptionResponstResult.data.isHideCrm?0:1
                                      withKey:WCTSettingsKey_ShowCrmUI];
        [PPSettingsController setIntegerValue:displayOptionResponstResult.data.isHideContactServer?0:1
                                      withKey:WCTSettingsKey_ShowContactServerUI];
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - class GetReviewStatus


//==============================================================================
// 升級時要call，要在getReviewStatusWithCompleteHandler之前做
//==============================================================================
+ (void)resetIsAppleReviewing
{
    [PPSettingsController removeValueWithKey:WCTLoginControllerSettingKey_IsReviewing];
}


//==============================================================================
//
//==============================================================================
+ (BOOL)isAppleReviewing
{
    NSNumber *reviewingNumber = [PPSettingsController numberValueWithKey:WCTLoginControllerSettingKey_IsReviewing];
    
    return [reviewingNumber boolValue];
}


//==============================================================================
// 如果還沒有取過review status，再去取，app升級後要重設這個flag
//==============================================================================
+ (void)getReviewStatusWithCompleteHandler:(void(^)(BOOL isReviewing))completeHandler
{
    NSString *currentVersion = [[NSBundle mainBundle] objectForInfoDictionaryKey:@"CFBundleShortVersionString"];
    NSString *oldVersion = [PPSettingsController stringValueWithKey:WCTLoginControllerSettingKey_AppVersion];
    
    if(oldVersion==nil || [oldVersion isEqualToString:currentVersion]==NO)
    {
        [[self class] resetIsAppleReviewing];
        [PPSettingsController setStringValue:currentVersion withKey:WCTLoginControllerSettingKey_AppVersion];
    }
    
    
    //////////////////////////////////////////////////
    NSNumber *reviewingNumber = [PPSettingsController numberValueWithKey:WCTLoginControllerSettingKey_IsReviewing];

    if(reviewingNumber==nil||[[self class] isAppleReviewing]==YES)
    {
        //還沒取過，
        NSString *platform = nil;
#if TARGET_OS_IPHONE
        platform = @"ios";
#elif TARGET_OS_MAC
        platform = @"mac";
#endif
        NSString *propertiesURLString = [NSString stringWithFormat:@"https://subscrib.worldcardteam.com/PRS/automation/wcc_properties?platform=%@&version=%@", platform, currentVersion];
        
        NSURL *propertiesURL = [NSURL URLWithString:propertiesURLString];
        NSURLRequest *urlRequest = [NSURLRequest requestWithURL:propertiesURL cachePolicy:NSURLRequestReloadIgnoringCacheData timeoutInterval:10.0];
        NSURLSession *session = [NSURLSession sharedSession];
        __block NSURLSessionDataTask *task = [session dataTaskWithRequest:urlRequest completionHandler:^(NSData * _Nullable data, NSURLResponse * _Nullable response, NSError * _Nullable error) {
            
            if (error!=nil || [data length]==0)
            {
#ifdef DEBUG
                NSLog(@"error:%@", error);
#endif
                if(completeHandler)
                {
                    completeHandler(NO);
                }
                return;
            }
            
            // format:
//            {
//                "data": {
//                    "availableDaysForTrial": 30,
//                    "availableDaysForRetailUpgrade": 180,
//                    "underReview": true
//                },
//                "message": "success"
//            }
            
            NSError *jsonError = nil;
            NSDictionary *resultDict = [NSJSONSerialization JSONObjectWithData:data options:kNilOptions error:&jsonError];
            if(resultDict==nil)
            {
                completeHandler(NO);
                return ;
            }

            //////////////////////////////////////////////////
            NSDictionary *dataDict = [resultDict objectForKey:@"data"];
            
            if(dataDict==nil)
            {
                completeHandler(NO);
                return ;
            }
            
            //////////////////////////////////////////////////
            NSNumber *reviewNumber = [dataDict objectForKey:@"underReview"];
            if(reviewNumber==nil)
            {
                completeHandler(NO);
                return ;
            }
            [PPSettingsController setNumberValue:reviewNumber withKey:WCTLoginControllerSettingKey_IsReviewing];
            
            if(completeHandler)
            {
                completeHandler([[self class] isAppleReviewing]);
            }
        }];
        
        [task resume];
    }
    else
    {
        // 已經有值，直接回傳值
        if(completeHandler)
        {
            completeHandler([[self class] isAppleReviewing]);
        }
    }
}



//==============================================================================
//
//==============================================================================
+ (void)setOfflineLoginTimeoutDays:(NSInteger)days
{
    g_WCTLoginControllerOfflineLoginTimeoutDays = days;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Logs


//==============================================================================
//
//==============================================================================
+ (NSString *)logDirPath
{
    return [WCToolController baseStorePathWithDirName:WCTLoginController_LogDir isCreatDirPath:NO];
}

//===============================================================================
//
//===============================================================================
- (void)logMessageWithFormat:(NSString *)format, ...
{
    if(WCTLoginController_EnagleLog==NO)
    {
        return ;
    }
    
    va_list arguments;
    va_start(arguments, format);
    [self.logController logWithMask:PPLogControllerMask_Normal format:format arguments:arguments];
    va_end(arguments);
}

//==============================================================================
//
//==============================================================================
+ (void)logEnable:(BOOL)enable
{
    WCTLoginController_EnagleLog = enable;
}

@end
