//
//  WCRecognitionController.m
//
//  Created by Howard on 2015/7/29.
//  Copyright (c) 2015年 Eddie. All rights reserved.
//

#import "WCRecognitionController.h"
#import <Photos/Photos.h>

// define
#import "WCRecognitionController+ParameterDefine.h"
#import "namecard_define.h"
#import "namecard_function.h"

// Controller
#import "WCRecognitionPreferenceController.h"
#import "PPLogController.h"
#import "PPCutPhoneNumberController.h"

// Category
#if TARGET_OS_IPHONE
#import "PHAsset+Image.h"
#import "UIImage+Additions.h"
#elif TARGET_OS_MAC
#import "NSImage+Additions.h"
#endif

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation WCRecognitionController

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying , and Dellocating Object

//================================================================================
//
//================================================================================
- (id)init
{
    if(self=[super init])
    {
        
#if TARGET_OS_IPHONE
        Connect();
#elif TARGET_OS_MAC
        Connect([[[NSBundle mainBundle] resourcePath] UTF8String]);
#endif
        
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    Disconnect();
    
    //////////////////////////////////////////////////
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Method

//===============================================================================
//
//===============================================================================
- (BOOL)isEmptyResult:(unsigned short *)aRecognizeStr
{
    int nGroup = 0;
    
    if (aRecognizeStr == nil)
    {
        return YES;
    }
    
    nGroup = aRecognizeStr[0];
    
    if (nGroup > 0)
    {
        return NO;
    }
    
    return YES;
}


//================================================================================
//
//================================================================================
- (NSError *)lastErrorWithCode:(NSInteger)code description:(NSString *)description
{
    NSMutableDictionary *userInfo = nil;
    
    userInfo = [NSMutableDictionary dictionaryWithObjectsAndKeys:description, NSLocalizedDescriptionKey, nil];
    
    NSError *error = [NSError errorWithDomain:WCRO_ErrorDomain code:code userInfo:userInfo];
    
    PPLogFunction(@"%@",error);
    
    return error;
}


//===============================================================================
//
//===============================================================================
- (void)setCard:(WCRecognitionCardModel *)cardModel withRecogSource:(WCRecogSourceModel *)recogSource
{
    //---------------------------------
    // initialize card model
    //---------------------------------
    cardModel.cardID = recogSource.cardID;
    
    if(recogSource.imageType == WCRecognitionImageType_FrontSide)
    {
        cardModel.frontRecognitionLanguage = recogSource.recogLang;
    }
    else
    {
        cardModel.backRecognitionLanguage = recogSource.recogLang;
    }
    
    [cardModel setSourceImageType:(WCRecognitionImageType)recogSource.imageType];
}


//===============================================================================
// 不同的辨識語系，預設countryCode的值不一樣
//===============================================================================
- (NSString *)defaultCountryCodeWithRecogLang:(NSInteger)recogLang
{
    switch (recogLang)
    {
        case GE_LANG:
        {
            return @"ge";
        }
        case FR_LANG:
        {
            return @"fr";
        }
        case IT_LANG:
        {
            return @"it";
        }
        case NO_LANG:
        {
            return @"no";
        }
        case HU_LANG:
        {
            return @"hu";
        }
        case PL_LANG:
        {
            return @"pl";
        }
        case SE_LANG:
        {
            return @"se";
        }
        case NL_LANG:
        {
            return @"nl";
        }
        case TR_LANG:
        {
            return @"tr";
        }
        case RU_LANG:
        {
            return @"ru";
        }
        case JP_LANG:
        {
            return @"jp";
        }
        case KS_LANG:
        {
            return @"ks";
        }
        case DK_LANG:
        {
            return @"dk";
        }
        case FI_LANG:
        {
            return @"fi";
        }
        case GR_LANG:
        {
            return @"gr";
        }
        case BR_LANG:
        {
            return @"br";
        }
        case MX_LANG:
        {
            return @"mx";
        }
        case CZ_LANG:
        {
            return @"cz";
        }
        case SK_LANG:
        {
            return @"sk";
        }
        case AR_LANG:
        {
            return @"ar";
        }
        case TH_LANG:
        {
            return @"th";
        }
            
        default:
        {
            return [PPCountryCodeConvert defaultCountryCode];
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Image handle Method

//===============================================================================
// !! 寬度必須符合核心alignment需求
//===============================================================================
- (int)imageWidthForRecognition:(int)imageWidth
{
    return ((imageWidth * 8 + 31) / 32) * 4;
}


//===============================================================================
//
//===============================================================================
- (void)removeGrayScaleData:(char **)bmpData
{
    if (*bmpData != nil)
    {
        free(*bmpData);
        *bmpData = nil;
    }
}


//===============================================================================
//
//===============================================================================
- (char *)createGrayScaleData:(CPImage *)image
{
    if(!image)
    {
        return nil;
    }
    
    CGImageRef cgImage = [image CGImage];
    CGSize actualSize = [image actualSize];
    int width = [self imageWidthForRecognition:actualSize.width];
    int height = actualSize.height;
    int dataLength = [self grayScaleDataLengthWithImage:image];
    char *bmpData = (char *)malloc(dataLength);
    
    if (bmpData == nil)
    {
        return nil;
    }
    
    CGColorSpaceRef colorSpace = CGColorSpaceCreateDeviceGray();
    CGContextRef context = CGBitmapContextCreate(bmpData, width, height, 8, width, colorSpace, (CGBitmapInfo)kCGImageAlphaNone);
    
    
    if (context != nil)
    {
        CGContextSetShouldAntialias(context, NO);
        CGContextDrawImage(context, CGRectMake(0, 0, width, height), cgImage);
        CGContextRelease(context);
    }
    
    CGColorSpaceRelease(colorSpace);
    
    return bmpData;
}


//===============================================================================
//
//===============================================================================
- (int)grayScaleDataLengthWithImage:(CPImage *)image
{
    if(!image)
    {
        return 0;
    }
    
    CGSize actualSize = [image actualSize];
    
    return [self imageWidthForRecognition:actualSize.width] * actualSize.height;
}


//===============================================================================
// !! 辨識用的影像統一在此處理
//===============================================================================
- (CPImage *)imageFromRecogSource:(WCRecogSourceModel *)recogSource
{
    __block CPImage *resultImage = nil;
    
    //-------------------------------
    // get image object
    //-------------------------------
#if TARGET_OS_IPHONE
    
    if([recogSource.imageSource isKindOfClass:[UIImage class]])
    {
        resultImage = recogSource.imageSource;
    }
    else if([recogSource.imageSource isKindOfClass:[PHAsset class]])
    {
        PHAsset *asset = (PHAsset *)recogSource.imageSource;
        resultImage = [asset imageWithTagetSize:CGSizeMake(asset.pixelWidth, asset.pixelHeight)];

        // 修正圖片顯析度到1600以下
        if(WC_IMS_Original<MAX(resultImage.size.width, resultImage.size.height) ||
           WC_IMS_Original<MIN(resultImage.size.width, resultImage.size.height))
        {
            resultImage = [resultImage imageScalingMaxLength:WC_IMS_Original];
        }
    }
    else if([recogSource.imageSource isKindOfClass:[NSString class]])
    {
        NSString *filePath = (NSString *)recogSource.imageSource;
        
        NSData *img = [NSData dataWithContentsOfFile:filePath];
        resultImage = [UIImage imageWithData:img];
    }
    
#elif TARGET_OS_MAC
    
    if([recogSource.imageSource isKindOfClass:[NSImage class]])
    {
        resultImage = recogSource.imageSource;
    }
    else if([recogSource.imageSource isKindOfClass:[NSString class]])
    {
        NSString *filePath = (NSString *)recogSource.imageSource;
        
        NSData *img = [NSData dataWithContentsOfFile:filePath];
        resultImage = [NSImage imageWithData:img];
    }
    
#endif
    
    
    CPImage *tempImage = nil;
    
    //-------------------------------
    // rotate/resize image
    //-------------------------------
    if(recogSource.rotateDegree != 255) // 255代表核心自動旋轉
    {
        CGFloat maxLength = MAX([resultImage actualSize].width, [resultImage actualSize].height);
        maxLength = MIN(maxLength, WC_IMS_Original);
        
        tempImage = [resultImage imageRotatedByDegrees:recogSource.rotateDegree scalingMaxLength:maxLength];
        resultImage = tempImage;
        
        recogSource.rotateDegree = 0;
    }
    
    return resultImage;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Field Update Method


//===============================================================================
//
//===============================================================================
- (void)addRecogFieldToCard:(WCRecognitionCardModel *)cardModel
            fromRecogResult:(unsigned short *)recogResult
            withRecogSource:(WCRecogSourceModel *)recogSource
              isReRecognize:(BOOL)isReRecognize
{
    int             i = 0;
    int             j = 0;
    int             nIndex = 0;
    int             nGroup = 0;
    int             nLength = 0;
    int             nType = 0;
    int             nPosLeft = 0;
    int             nPosTop = 0;
    int             nPosRight = 0;
    int             nPosBottom = 0;
    unsigned short  data[WCRO_MaxFieldBuf];
    
    NSString        *separator = @"\n";
    NSInteger        recogResultType;
    NSString        *recogResultString;
    CGRect          recogResultRect;
    
    WCRecognitionFieldModel     *fieldModel = nil;
    WCRecognitionFieldModel     *combineFieldModel = nil;
    
    WCRecognitionFieldSource fieldSource = (recogSource.imageType == WCRecognitionImageType_FrontSide) ? WCRecognitionFieldSource_FrontRecog : WCRecognitionFieldSource_BackRecog;
    
    //--------------------------------------------
    //為了在電話裡面加上國碼而加的流程
    //--------------------------------------------
    NSString       *countryCode=nil;
    
    
    //--------------------------------------------
    // !! special handle for name data
    //--------------------------------------------
    
    WCRecognitionFieldModel *englishNameField = [[WCRecognitionFieldModel alloc] init];
    
    englishNameField.fieldSource = fieldSource;
    englishNameField.fieldMainType = WCRecognitionFieldMainType_Name;
    
    //--------------------------------------------
    // start parsing recogResult
    //--------------------------------------------
    nGroup = recogResult[0];
    nIndex = 1;
    
    
    // 讀取辨識資料
    for(i=0; i <nGroup; i++)
    {
        if(recogSource.hasRecogRect)
        {
            nPosLeft   = recogResult[nIndex++];
            nPosTop    = recogResult[nIndex++];
            nPosRight  = recogResult[nIndex++];
            nPosBottom = recogResult[nIndex++];
        }
        
        nType = recogResult[nIndex++];
        nLength = recogResult[nIndex++];
        
        if(nLength)
        {
            //==================================================
            // copy data from recogResult
            //==================================================
            
            if(nLength > WCRO_MaxFieldBuf)
            {
                nLength = WCRO_MaxFieldBuf;
            }
            
            memset(data, 0, sizeof(data));
            
            for (j=0; j < nLength; j++)
                data[j] = recogResult[nIndex++];
            
            recogResultType = nType;
            recogResultRect = CGRectMake(MIN(nPosLeft, nPosRight), MIN(nPosTop, nPosBottom), abs(nPosRight-nPosLeft), abs(nPosBottom-nPosTop));
            recogResultString = [NSString stringWithCharacters:data length:nLength];
            
            //test
            //            PPLogFunction(@"recog result : %d, %@", recogResultType, recogResultString);
            
            //==================================================
            // convert to WCRecognitionContextModel
            //==================================================
            
            
            switch (recogResultType)
            {
                case ID_BCR_ENG_NAME: // english full name 只取辨識範圍
                {
                    // 手括辨識時，英文名字只會回傳ID_BCR_ENG_NAME
                    NSArray *subType2ValueArray = [recogResultString componentsSeparatedByString:@";"];
                    
                    if([subType2ValueArray count] == 4)
                    {
                        NSString *firstName = [subType2ValueArray objectAtIndex:1];
                        
                        if([firstName length])
                        {
                            WCRecognitionContextModel *firstNameContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:firstName recognitionRect:recogResultRect isCJKStringValue:NO];
                            
                            [englishNameField setWCRecognitionContextModel:firstNameContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_Name_FirstName];
                        }
                        
                        NSString *lastName = [subType2ValueArray objectAtIndex:3];
                        
                        if([lastName length])
                        {
                            WCRecognitionContextModel *lastNameContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:lastName recognitionRect:recogResultRect isCJKStringValue:NO];
                            
                            [englishNameField setWCRecognitionContextModel:lastNameContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_Name_LastName];
                        }
                    }
                    else if([subType2ValueArray count] == 1)
                    {
                        if([englishNameField recognitionContextModelForWCRecognitionFieldSubType:WCRecognitionFieldSubType_Name_FirstName]==nil)
                        {
                            WCRecognitionContextModel *firstNameContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:recogResultString recognitionRect:recogResultRect isCJKStringValue:NO];
                            
                            [englishNameField setWCRecognitionContextModel:firstNameContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_Name_FirstName];
                        }
                    }
                    
                    break;
                }
                    
                case ID_BCR_ENG_1STNAME: // english first name
                {
                    WCRecognitionContextModel *firstNameContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:recogResultString recognitionRect:recogResultRect isCJKStringValue:NO];
                    
                    [englishNameField setWCRecognitionContextModel:firstNameContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_Name_FirstName];
                    
                    break;
                }
                    
                case ID_BCR_ENG_SURNAME: // english last name
                {
                    WCRecognitionContextModel *lastNameContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:recogResultString recognitionRect:recogResultRect isCJKStringValue:NO];
                    
                    [englishNameField setWCRecognitionContextModel:lastNameContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_Name_LastName];
                    
                    break;
                }
                    
                case ID_BCR_ENG_MIDNAME: // english middle name
                {
                    WCRecognitionContextModel *mindleNameContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:recogResultString recognitionRect:recogResultRect isCJKStringValue:NO];
                    
                    [englishNameField setWCRecognitionContextModel:mindleNameContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_Name_MiddleName];
                    
                    break;
                }
                    
                    
                case ID_BCR_NAME: // non-english full name
                {
                    NSString *firstName = nil;
                    NSString *lastName = nil;
                    NSString *firstNamePhonetic = nil;
                    NSString *lastNamePhonetic = nil;
                    
                    // !! 核心如果有切割姓名的話會用"\n"分隔
                    NSArray  *checkNameArray = [recogResultString componentsSeparatedByString:separator];
                    
                    //#ifdef _DUMP_LOG_
                    //                    if([recogSource.imageSource isKindOfClass:[NSString class]])
                    //                        [DumpLog logWithMemSize:NO startTime:0 format:@"%@, name=%@", [recogSource.imageSource lastPathComponent], recogResultString];
                    //#endif
                    
                    if([checkNameArray count] > 1)
                    {
                        lastName = [checkNameArray objectAtIndex:0];
                        firstName = [checkNameArray objectAtIndex:1];
                    }
                    else
                    {
                        if([recogResultString length] > 3)
                        {
                            lastName = [recogResultString substringToIndex:2];
                            firstName = [recogResultString substringFromIndex:2];
                        }
                        else if([recogResultString length] > 0)
                        {
                            lastName = [recogResultString substringToIndex:1];
                            firstName = [recogResultString substringFromIndex:1];
                        }
                    }
                    
                    //////////////////////////////////////////////////
                    
                    PPIndexingStyle ppIndexStyle = PPIndexingStyle_English;
                    
                    // !! 產生讀音
                    if(self.ppIndexStyle==PPIndexingStyle_Hiragana)
                    {
                        ppIndexStyle = self.ppIndexStyle;
                    }
                    else if(recogSource.recogLang==JP_LANG)
                    {
                        ppIndexStyle = PPIndexingStyle_Hiragana;
                    }
                    else if(recogSource.recogLang==KS_LANG)
                    {
                        ppIndexStyle = PPIndexingStyle_Hangul;
                    }
                    
                    //////////////////////////////////////////////////
                    
                    lastNamePhonetic = [PPIndexingController phoneticOfString:lastName forStyle:ppIndexStyle];
                    
                    if(lastName!=nil &&
                       [lastNamePhonetic length] &&
                       [lastNamePhonetic isEqualToString:lastName])
                    {
                        lastNamePhonetic = nil;
                    }
                    
                    //////////////////////////////////////////////////
                    
                    firstNamePhonetic = [PPIndexingController phoneticOfString:firstName forStyle:ppIndexStyle];
                    
                    if(firstName!=nil &&
                       [firstNamePhonetic length] &&
                       [firstNamePhonetic isEqualToString:firstName])
                    {
                        firstNamePhonetic = nil;
                    }
                    
                    
                    //==================================================
                    // set data
                    //==================================================
                    
                    fieldModel = [[WCRecognitionFieldModel alloc] init];
                    fieldModel.fieldSource = fieldSource;
                    fieldModel.fieldMainType = WCRecognitionFieldMainType_Name;
                    
                    WCRecognitionContextModel *firstNameContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:firstName recognitionRect:recogResultRect isCJKStringValue:YES];
                    
                    WCRecognitionContextModel *firstPhoneticNameContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:firstNamePhonetic recognitionRect:recogResultRect isCJKStringValue:YES];
                    
                    WCRecognitionContextModel *lastNameContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:lastName recognitionRect:recogResultRect isCJKStringValue:YES];
                    
                    WCRecognitionContextModel *lastNamePhoneticContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:lastNamePhonetic recognitionRect:recogResultRect isCJKStringValue:YES];
                    
                    [fieldModel setWCRecognitionContextModel:firstNameContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_Name_FirstName];
                    
                    [fieldModel setWCRecognitionContextModel:firstPhoneticNameContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_Name_FirstNamePhonetic];
                    
                    [fieldModel setWCRecognitionContextModel:lastNameContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_Name_LastName];
                    
                    [fieldModel setWCRecognitionContextModel:lastNamePhoneticContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_Name_LastNamePhonetic];
                    
                    break;
                }
                    
                    
                    // !! 一組公司資訊中只能有一個company/department/position，如果有多的要存成另一組公司資訊。
                case ID_BCR_COMPANY:
                case ID_BCR_DEPARTMENT:
                case ID_BCR_POSITION:
                {
                    NSString *companyPhonetic = nil;
                    
                    combineFieldModel = [[WCRecognitionFieldModel alloc] init];
                    combineFieldModel.fieldSource = fieldSource;
                    combineFieldModel.fieldMainType = WCRecognitionFieldMainType_Company;
                    combineFieldModel.fieldLabelType = WCRecognitionFieldLabelType_None;
                    
                    WCRecognitionContextModel *companyContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:recogResultString recognitionRect:recogResultRect isCJKStringValue:YES];
                    
                    switch (recogResultType)
                    {
                        case ID_BCR_COMPANY:
                        {
                            [combineFieldModel setWCRecognitionContextModel:companyContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_Company_Name];
                            
                            //////////////////////////////////////////////////
                            
                            PPIndexingStyle ppIndexStyle = PPIndexingStyle_English;
                            
                            // !! 產生讀音
                            if(self.ppIndexStyle==PPIndexingStyle_Hiragana)
                            {
                                ppIndexStyle = self.ppIndexStyle;
                            }
                            else if(recogSource.recogLang==JP_LANG)
                            {
                                ppIndexStyle = PPIndexingStyle_Hiragana;
                            }
                            
                            
                            //////////////////////////////////////////////////
                            
                            // !! 日、韓文要產生讀音 (產生讀音前要把株式會社等公司名稱濾掉)
                            
                            NSString *temp = [WCRecognitionPreferenceController trimCompanyPrefix:recogResultString];
                            companyPhonetic = [PPIndexingController phoneticOfString:temp forStyle:ppIndexStyle];
                            
                            // 如果讀音和名稱一樣就不用加入
                            if([companyPhonetic length] && [companyPhonetic isEqualToString:recogResultString])
                            {
                                companyPhonetic = nil;
                            }
                            
                            break;
                        }
                            
                        case ID_BCR_DEPARTMENT:
                        {
                            [combineFieldModel setWCRecognitionContextModel:companyContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_Company_Department];
                            
                            break;
                        }
                        case ID_BCR_POSITION:
                        {
                            [combineFieldModel setWCRecognitionContextModel:companyContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_Company_JobTitle];
                            
                            break;
                        }
                    }
                    
                    //////////////////////////////////////////////////
                    
                    if(combineFieldModel!=nil && [companyPhonetic length])
                    {
                        WCRecognitionContextModel *companyPhoneticContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:companyPhonetic recognitionRect:recogResultRect isCJKStringValue:YES];
                        
                        [combineFieldModel setWCRecognitionContextModel:companyPhoneticContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_Company_Phonetic];
                        
                    }
                    
                    [cardModel insertWCRecognitionFieldModel:combineFieldModel];
                    
                    [combineFieldModel release];
                    
                    break;
                }
                    
                    
                    // !! 一組公司資訊中只能有一個company/department/position，如果有多的要存成另一組公司資訊。
                case ID_BCR_ENG_COMPANY:
                case ID_BCR_ENG_DEPARTMENT:
                case ID_BCR_ENG_POSITION:
                {
                    combineFieldModel = [[WCRecognitionFieldModel alloc] init];
                    
                    combineFieldModel.fieldSource    = fieldSource;
                    combineFieldModel.fieldMainType  = WCRecognitionFieldMainType_Company;
                    combineFieldModel.fieldLabelType = WCRecognitionFieldLabelType_None;
                    
                    WCRecognitionContextModel *englishCompanyContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:recogResultString recognitionRect:recogResultRect isCJKStringValue:NO];
                    
                    switch (recogResultType)
                    {
                        case ID_BCR_ENG_COMPANY:
                        {
                            [combineFieldModel setWCRecognitionContextModel:englishCompanyContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_Company_Name];
                            
                            break;
                        }
                        case ID_BCR_ENG_DEPARTMENT:
                        {
                            [combineFieldModel setWCRecognitionContextModel:englishCompanyContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_Company_Department];
                            
                            break;
                        }
                        case ID_BCR_ENG_POSITION:
                        {
                            [combineFieldModel setWCRecognitionContextModel:englishCompanyContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_Company_JobTitle];
                            
                            break;
                        }
                    }
                    
                    [cardModel insertWCRecognitionFieldModel:combineFieldModel];
                    
                    [combineFieldModel release];
                    
                    break;
                }
                    
                    
                case ID_BCR_ADDRESS:
                case ID_BCR_ENG_ADDRESS:
                {
                    // MARK: Howard 這邊有問題，辨識中文名片欄位順序是錯的
                    NSArray *addrFieldArray = [recogResultString componentsSeparatedByString:separator];
                    WCRecognitionFieldSubType addrSubType2[5] = {
                        WCRecognitionFieldSubType_Address_Street,
                        WCRecognitionFieldSubType_Address_City,
                        WCRecognitionFieldSubType_Address_State,
                        WCRecognitionFieldSubType_Address_Zip,
                        WCRecognitionFieldSubType_Address_CountryCode};
                    
                    
                    fieldModel = [[WCRecognitionFieldModel alloc] init];
                    fieldModel.fieldSource    = fieldSource;
                    fieldModel.fieldMainType  = WCRecognitionFieldMainType_Address;
                    fieldModel.fieldLabelType = WCRecognitionFieldLabelType_Address_Work;
                    
                    //////////////////////////////////////////////////
                    
                    WCRecognitionContextModel *addressContextModel = nil;
                    
                    BOOL isCJKStringValue = (recogResultType == ID_BCR_ADDRESS);
                    
                    int i=0;
                    
                    for(NSString *addrString in addrFieldArray)
                    {
                        if(i >= 5)
                            break;
                        
                        switch (addrSubType2[i])
                        {
                            case WCRecognitionFieldSubType_Address_Zip:
                            {
                                // !! eddie : 不是在這裡加〒，要在組合完整地址時才檢查。
                                
                                //                                // 日文郵遞區號前一定要有〒
                                //                                if(recogSource.recogLang == JP_LANG && [[fieldModel recognitionContextModelForWCRecognitionFieldSubType:WCRecognitionFieldSubType_Address_Street].contextString isCJK]&&
                                //                                   [addrString length] && ![addrString hasPrefix:@"〒"])
                                //                                {
                                //                                    addrString = [NSString stringWithFormat:@"〒%@", addrString];
                                //                                }
                                
                                
                                break;
                            }
                            case WCRecognitionFieldSubType_Address_CountryCode:
                            {
                                addrString = [addrString lowercaseString];
                                
                                // !! 沒有country code要依特定規則補上
                                if([addrString length]==0)
                                {
                                    addrString = [[self defaultCountryCodeWithRecogLang:recogSource.recogLang] lowercaseString];
                                }
                                
                                if(countryCode==nil)
                                {
                                    if ([addrString length]>0)
                                    {
                                        countryCode = [addrString retain];
                                    }
                                }
                                break;
                            }
                            default:
                            {
                                break;
                            }
                        }
                        
                        //////////////////////////////////////////////////
                        
                        addressContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:addrString recognitionRect:recogResultRect isCJKStringValue:isCJKStringValue];
                        
                        [fieldModel setWCRecognitionContextModel:addressContextModel forWCRecognitionFieldSubType:addrSubType2[i]];
                        
                        i++;
                    }
                    break;
                }
                    
                    
                case ID_BCR_EMAIL:
                {
                    fieldModel = [[WCRecognitionFieldModel alloc] init];
                    
                    fieldModel.fieldSource    = fieldSource;
                    fieldModel.fieldMainType  = WCRecognitionFieldMainType_Email;
                    fieldModel.fieldLabelType = WCRecognitionFieldLabelType_Email_Work;
                    
                    
                    WCRecognitionContextModel *emailContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:recogResultString recognitionRect:recogResultRect isCJKStringValue:NO];
                    
                    [fieldModel setWCRecognitionContextModel:emailContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_DontCare];
                    
                    break;
                }
                    
                case ID_BCR_HOMEPAGE:
                {
                    fieldModel = [[WCRecognitionFieldModel alloc] init];
                    
                    fieldModel.fieldSource    = fieldSource;
                    fieldModel.fieldMainType  = WCRecognitionFieldMainType_URL;
                    fieldModel.fieldLabelType = WCRecognitionFieldLabelType_URL_HomePage;
                    
                    WCRecognitionContextModel *homePageContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:recogResultString recognitionRect:recogResultRect isCJKStringValue:NO];
                    
                    [fieldModel setWCRecognitionContextModel:homePageContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_DontCare];
                    
                    break;
                }
                    
                case ID_BCR_PHONE:
                {
                    fieldModel = [[WCRecognitionFieldModel alloc] init];
                    
                    fieldModel.fieldSource    = fieldSource;
                    fieldModel.fieldMainType  = WCRecognitionFieldMainType_Phone;
                    fieldModel.fieldLabelType = WCRecognitionFieldLabelType_Phone_Work;
                    
                    // !! WCM7之後，把電話中的"#"改為","
                    recogResultString = [recogResultString stringByReplacingOccurrencesOfString:@"#" withString:@","];
                    
                    WCRecognitionContextModel *phoneContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:recogResultString recognitionRect:recogResultRect isCJKStringValue:NO];
                    
                    [fieldModel setWCRecognitionContextModel:phoneContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_DontCare];
                    
                    break;
                }
                    
                case ID_BCR_MOBILE:
                {
                    fieldModel = [[WCRecognitionFieldModel alloc] init];
                    
                    fieldModel.fieldSource    = fieldSource;
                    fieldModel.fieldMainType  = WCRecognitionFieldMainType_Phone;
                    fieldModel.fieldLabelType = WCRecognitionFieldLabelType_Phone_Mobile;
                    
                    // !! WCM7之後，把電話中的"#"改為","
                    recogResultString = [recogResultString stringByReplacingOccurrencesOfString:@"#" withString:@","];
                    
                    WCRecognitionContextModel *phoneContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:recogResultString recognitionRect:recogResultRect isCJKStringValue:NO];
                    
                    [fieldModel setWCRecognitionContextModel:phoneContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_DontCare];
                    
                    break;
                }
                    
                case ID_BCR_FAX:
                {
                    fieldModel = [[WCRecognitionFieldModel alloc] init];
                    
                    fieldModel.fieldSource    = fieldSource;
                    fieldModel.fieldMainType  = WCRecognitionFieldMainType_Phone;
                    fieldModel.fieldLabelType = WCRecognitionFieldLabelType_Phone_WorkFax;
                    
                    // !! WCM7之後，把電話中的"#"改為","
                    recogResultString = [recogResultString stringByReplacingOccurrencesOfString:@"#" withString:@","];
                    
                    WCRecognitionContextModel *phoneContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:recogResultString recognitionRect:recogResultRect isCJKStringValue:NO];
                    
                    [fieldModel setWCRecognitionContextModel:phoneContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_DontCare];
                    
                    break;
                }
                    
                case ID_BCR_BBCALL:
                {
                    fieldModel = [[WCRecognitionFieldModel alloc] init];
                    
                    fieldModel.fieldSource    = fieldSource;
                    fieldModel.fieldMainType  = WCRecognitionFieldMainType_Phone;
                    fieldModel.fieldLabelType = WCRecognitionFieldLabelType_Phone_Pager;
                    
                    // !! WCM7之後，把電話中的"#"改為","
                    recogResultString = [recogResultString stringByReplacingOccurrencesOfString:@"#" withString:@","];
                    
                    WCRecognitionContextModel *phoneContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:recogResultString recognitionRect:recogResultRect isCJKStringValue:NO];
                    
                    [fieldModel setWCRecognitionContextModel:phoneContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_DontCare];
                    
                    break;
                }
                    
                case ID_BCR_SKYPE:
                {
                    fieldModel = [[WCRecognitionFieldModel alloc] init];
                    
                    fieldModel.fieldSource    = fieldSource;
                    fieldModel.fieldMainType  = WCRecognitionFieldMainType_InstantMessage;
                    fieldModel.fieldLabelType = WCRecognitionFieldLabelType_IM_Skype;
                    
                    WCRecognitionContextModel *instantMessagerContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:recogResultString recognitionRect:recogResultRect isCJKStringValue:NO];
                    
                    [fieldModel setWCRecognitionContextModel:instantMessagerContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_DontCare];
                    
                    break;
                }
                    
                case ID_BCR_MESSENGER:
                {
                    fieldModel = [[WCRecognitionFieldModel alloc] init];
                    
                    fieldModel.fieldSource    = fieldSource;
                    fieldModel.fieldMainType  = WCRecognitionFieldMainType_InstantMessage;
                    fieldModel.fieldLabelType = WCRecognitionFieldLabelType_IM_MSN;
                    
                    WCRecognitionContextModel *instantMessagerContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:recogResultString recognitionRect:recogResultRect isCJKStringValue:NO];
                    
                    [fieldModel setWCRecognitionContextModel:instantMessagerContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_DontCare];
                    
                    
                    break;
                }
                    
                case ID_BCR_COMPANYNO:
                {
                    fieldModel = [[WCRecognitionFieldModel alloc] init];
                    
                    fieldModel.fieldSource    = fieldSource;
                    fieldModel.fieldMainType  = WCRecognitionFieldMainType_UnifiedBusinessNumber;
                    
                    WCRecognitionContextModel *unifiedBusinessNoContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:recogResultString recognitionRect:recogResultRect isCJKStringValue:NO];
                    
                    [fieldModel setWCRecognitionContextModel:unifiedBusinessNoContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_DontCare];
                    
                    break;
                }
                    
                case ID_BCR_TELEX: // 電報,IP連結...
                {
                    fieldModel = [[WCRecognitionFieldModel alloc] init];
                    
                    fieldModel.fieldSource    = fieldSource;
                    fieldModel.fieldMainType  = WCRecognitionFieldMainType_Note;
                    
                    WCRecognitionContextModel *noteContextModel = [WCRecognitionContextModel recognitionContextModelWithContextString:recogResultString recognitionRect:recogResultRect isCJKStringValue:NO];
                    
                    [fieldModel setWCRecognitionContextModel:noteContextModel forWCRecognitionFieldSubType:WCRecognitionFieldSubType_DontCare];
                    
                    break;
                }
            }
            
            // save common field
            if(fieldModel)
            {
                if(fieldModel.fieldMainType == WCRecognitionFieldMainType_Note && [cardModel fieldCountWithWCRecognitionFieldMainType:WCRecognitionFieldMainType_Note]>0)
                {
                    WCRecognitionFieldModel *noteField = [[cardModel recognitionFieldModelArrayWithType:WCRecognitionFieldMainType_Note]objectAtIndex:0];
                    
                    [noteField recognitionContextModelForWCRecognitionFieldSubType:WCRecognitionFieldSubType_DontCare].contextString = [[noteField recognitionContextModelForWCRecognitionFieldSubType:WCRecognitionFieldSubType_DontCare].contextString stringByAppendingFormat:@"\n%@",recogResultString];
                    
                    [fieldModel release];
                    fieldModel = nil;
                    
                    continue;
                }
                
                //////////////////////////////////////////////////
                
                [cardModel addRecognitionFieldModel:fieldModel];
                
                [fieldModel release];
                fieldModel = nil;
            }
        }
    }
    
    //--------------------------------------------
    // save english name field
    //--------------------------------------------
    if([englishNameField validRecognitionFieldModel]==YES)
    {
        [cardModel addRecognitionFieldModel:englishNameField];
    }
    
    [englishNameField release];
    
    //////////////////////////////////////////////////
    
    if(countryCode==nil)
    {
        countryCode = [[PPCountryCodeConvert defaultCountryCode] retain];
    }
    
    //----------------
    //為了將電話加上國碼而增加的流程
    //----------------
    if(self.autoFormatPhone)
    {
        PPCutPhoneNumberController* ppCutPhoneNumberUt;
        ppCutPhoneNumberUt = [[PPCutPhoneNumberController alloc] init];
        
        NSArray *phoneNumberArray=[cardModel recognitionFieldModelArrayWithType:WCRecognitionFieldMainType_Phone];
        
        NSInteger arrayCount=[phoneNumberArray count];
        
        for (int a=0;a<arrayCount;a++)
        {
            WCRecognitionFieldModel *phoneField =[phoneNumberArray objectAtIndex:a];
            
            NSString *phoneString = [phoneField recognitionContextModelForWCRecognitionFieldSubType:WCRecognitionFieldSubType_DontCare].contextString;
            
            NSString *newPhoneString = nil;
            
            newPhoneString = [ppCutPhoneNumberUt copyFormattedPhoneNumber:phoneString withCountryCode:countryCode isFillWithCountryCode:self.addCountryCodeToPhone];
            
            [phoneField recognitionContextModelForWCRecognitionFieldSubType:WCRecognitionFieldSubType_DontCare].contextString = newPhoneString;
            
            [newPhoneString release];
        }
        
        [ppCutPhoneNumberUt release];
    }
    
    [countryCode release];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Method

//================================================================================
//
//================================================================================
- (BOOL)recognizeImage:(CPImage *)image
           toCardModel:(WCRecognitionCardModel *)cardModel
       withRecogSource:(WCRecogSourceModel *)recogSource
                 error:(NSError **)error
{
    BOOL result = NO;
    
    NSError *returnError = nil;
    
    do
    {
        char *bmpData = nil;
        int  resultCode;
        int  tranLang = 0;
        int  direction = 0;
        
        unsigned short resultBuf[WCRO_MaxResultBuf];
        CPImage *recognitionImage = nil;
        
        
        //////////////////////////////////////////////////
        // check image
        
        if(image==nil)
        {
            returnError = [self lastErrorWithCode:WCRO_ErrorCode_NoImage description:nil];
            break;
        }
        
        
        //////////////////////////////////////////////////
        // 配合kernel特性調整影像
        // 1. 最長邊不要超過1600
        // 2. 直式影像辨識時要轉換為橫向，辨識率較好。(hamming說的)
        
        int maxLength = 1600;
        
        @autoreleasepool {
            if(recogSource.rotateDegree == 255)
            {
                recognitionImage = [[image imageRotatedByDegrees:0 scalingMaxLength:maxLength] copy];
                direction = 255;
            }
            else
            {
                if(image.size.width < image.size.height)
                {
                    recognitionImage = [[image imageRotatedByDegrees:90 scalingMaxLength:maxLength] copy];
                    direction = 3;
                }
                else
                {
                    recognitionImage = [[image imageRotatedByDegrees:0 scalingMaxLength:maxLength] copy];
                    direction = 0;
                }
            }
        }

        [recognitionImage autorelease];
        
        //////////////////////////////////////////////////
        // test:看辨識的圖對不對
        
        //        NSArray *paths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
        //        [UIImagePNGRepresentation(recognitionImage) writeToFile:[[paths objectAtIndex:0] stringByAppendingPathComponent:@"recognitionImage.png"] atomically:YES];
        
        
        //////////////////////////////////////////////////
        // prepare card data
        
        [self setCard:cardModel withRecogSource:recogSource];
        
        
        //////////////////////////////////////////////////
        // MARK: Do recognition!
        
#ifdef PPLogMacro
        NSTimeInterval startTime = [[NSDate date] timeIntervalSince1970];
#endif
        
        if((bmpData = [self createGrayScaleData:recognitionImage]))
        {
            PPLogFunction(@"createGrayScaleData cost:(%.6f sec)",[[NSDate date] timeIntervalSince1970]-startTime);
            
            // send to BCR SDK
            CGSize actualSize = [recognitionImage actualSize];
            int width = [self imageWidthForRecognition:actualSize.width];
            int height = actualSize.height;
            int dataLength = [self grayScaleDataLengthWithImage:recognitionImage];
            
            
#ifdef PPLogMacro
            startTime = [[NSDate date] timeIntervalSince1970];
#endif
            PPLogFunction(@"recogSource.recogLang:%d, bmpData, width:%d, height:%d, dataLength:%d",recogSource.recogLang,width,height,dataLength);
            
#if TARGET_OS_IPHONE
            if(CreateBinarizebmp((int)recogSource.recogLang, bmpData, width, height, dataLength) >= 0)
#elif TARGET_OS_MAC
            if(CreateBinarizebmp((int)recogSource.recogLang, bmpData, width, height, 8, dataLength) >= 0)
#endif
            {
                PPLogFunction(@"CreateBinarizebmp cost:(%.6f sec)",[[NSDate date] timeIntervalSince1970]-startTime);
                
                memset(resultBuf, 0, WCRO_MaxResultBuf);
                
                // !! 中文辨識語系要指定輸出繁簡體
                switch (recogSource.recogLang)
                {
                    case B5_LANG:
                    case HK_LANG:
                    case GB_LANG:
                        tranLang = (int)recogSource.outputTranslate;
                        break;
                        
                    default:
                        tranLang = 0;
                        break;
                }
                
                //////////////////////////////////////////////////
#ifdef PPLogMacro
                startTime = [[NSDate date] timeIntervalSince1970];
#endif
                // recognize（回傳值就是旋轉角度）
                resultCode = Understanding((int)recogSource.recogLang, tranLang, direction, (short *)resultBuf);
                
                PPLogFunction(@"Understanding cost:(%.6f sec)",[[NSDate date] timeIntervalSince1970]-startTime);
                
                // parsing result
                if(resultCode >= 0 && resultCode%90 == 0)
                {
                    // !!日、韓文要產生讀音
                    [self preparePhoneticControllerWithRecogLang:recogSource.recogLang];
                    
                    // 轉換辨識結果
                    [self addRecogFieldToCard:cardModel fromRecogResult:resultBuf withRecogSource:recogSource isReRecognize:NO];
                    
                    // 自動旋轉模式下，需要知道影像被核心旋轉了幾度，供後續影像處理使用。
                    if(direction == 255)
                    {
                        recogSource.rotateDegree = resultCode;
                    }
                    else
                    {
                        recogSource.rotateDegree = 0;
                    }
                    
                    result = YES;
                }
                else
                {
                    // 辨識錯誤時旋轉角度要歸零
                    recogSource.rotateDegree = 0;
                    returnError = [self lastErrorWithCode:WCRO_ErrorCode_UnderstandingFailed description:nil];
                }
                
                FreeBinarize();
            }
            else
            {
                returnError = [self lastErrorWithCode:WCRO_ErrorCode_CreateBinarizebmpFailed description:nil];
            }
            
            [self removeGrayScaleData:&bmpData];
        }
        else
        {
            returnError = [self lastErrorWithCode:WCRO_ErrorCode_CreateGrayScaleDataFailed description:nil];
        }
        
    }
    while (0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil &&
       returnError!=nil)
    {
        *error = returnError;
    }
    
    return result;
}


//===============================================================================
//
//===============================================================================
- (void)preparePhoneticControllerWithRecogLang:(NSInteger)recogLang
{
    if(recogLang == JP_LANG)
    {
        self.ppIndexStyle = PPIndexingStyle_Hiragana;
    }
    else
    {
        self.ppIndexStyle = PPIndexingStyle_English;
    }
}


//==============================================================================
//
//==============================================================================
- (BOOL)clipRecogWithRect:(CGRect)clipRect
                recogType:(uint)recogType
              recogSource:(WCRecogSourceModel *)recogSourceModel
                     card:(WCRecognitionCardModel *)cardModel
                    error:(NSError **)error
{
    recogSourceModel.handClipRect = clipRect;
    recogSourceModel.handClipRecogType = recogType;
    return [self clipRecogWithRecogSource:recogSourceModel card:cardModel error:error];
}


//===============================================================================
//
//===============================================================================
- (BOOL)clipRecogWithRecogSource:(WCRecogSourceModel *)recogSourceModel
                            card:(WCRecognitionCardModel *)cardModel
                           error:(NSError **)error
{
    BOOL result = NO;
    NSError *returnError = nil;
    
    do
    {
        //---------------------------------
        // check image
        //---------------------------------
        
        CPImage *image = [[self imageFromRecogSource:recogSourceModel] retain];
        
        if(!image)
        {
            returnError = [self lastErrorWithCode:WCRO_ErrorCode_NoImage description:nil];
            break;
        }
        
        
        
        // test:看裁切的圖對不對
        //        CGImageRef imageRef = CGImageCreateWithImageInRect([image CGImage], recogSourceModel.handClipRect);
        //        CPImage *cropImage = [CPImage imageWithCGImage:imageRef];
        //
        //        CGImageRelease(imageRef);
        //
        //        //////////////////////////////////////////////////
        //
        //        NSArray *paths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
        //        NSString *filePath = [[paths objectAtIndex:0] stringByAppendingPathComponent:@"Image.png"];
        //
        //        [UIImagePNGRepresentation(cropImage) writeToFile:filePath atomically:YES];
        //        [UIImagePNGRepresentation(image) writeToFile:[[paths objectAtIndex:0] stringByAppendingPathComponent:@"sourceImage.png"] atomically:YES];
        
        
        
        
        //---------------------------------
        // start recognition
        //---------------------------------
        char *bmpData = nil;
        int resultCode;
        unsigned short resultBuf[WCRO_MaxResultBuf];
        
        if((bmpData = [self createGrayScaleData:image]))
        {
            // send to BCR SDK
            CGSize actualSize = [image actualSize];
            int    width = [self imageWidthForRecognition:actualSize.width];
            int height = actualSize.height;
            int    dataLength = [self grayScaleDataLengthWithImage:image];
            
            PPLogFunction(@"recogSource.recogLang:%d, width:%d, height:%d, dataLength:%d",recogSourceModel.recogLang,width,height,dataLength);
            
#if TARGET_OS_IPHONE
            if(CreateBinarizebmp((int)recogSourceModel.recogLang, bmpData, width, height, dataLength) >= 0)
#elif TARGET_OS_MAC
            if(CreateBinarizebmp((int)recogSourceModel.recogLang, bmpData, width, height, 8, dataLength) >= 0)
#endif
            {
                memset(resultBuf, 0, WCRO_MaxResultBuf);
                
                PPLogFunction(@"recogSourceModel.recogLang:%d,recogSourceModel.outputTranslate:%d, recogSourceModel.rotateDegree:%d, left:%f, top:%f, right:%f, down:%f, dataLength:%d"
                              ,recogSourceModel.recogLang,recogSourceModel.outputTranslate,recogSourceModel.rotateDegree,CGRectGetMinX(recogSourceModel.handClipRect),CGRectGetMinY(recogSourceModel.handClipRect),CGRectGetMaxX(recogSourceModel.handClipRect),CGRectGetMaxY(recogSourceModel.handClipRect),dataLength);
                
                resultCode = RecognizeClip((int)recogSourceModel.recogLang,
                                           (int)recogSourceModel.outputTranslate,
                                           (int)recogSourceModel.rotateDegree,
                                           (int)recogSourceModel.handClipRecogType,
                                           CGRectGetMinX(recogSourceModel.handClipRect),
                                           CGRectGetMinY(recogSourceModel.handClipRect),
                                           CGRectGetMaxX(recogSourceModel.handClipRect),
                                           CGRectGetMaxY(recogSourceModel.handClipRect),
                                           (short*)resultBuf);
                
                // parsing result
                if(resultCode >= 0 && [self isEmptyResult:resultBuf]==NO)
                {
                    [self setCard:cardModel withRecogSource:recogSourceModel];
                    
                    // !! special case for company
                    switch (recogSourceModel.handClipRecogType)
                    {
                        case ID_BCR_COMPANY:
                        case ID_BCR_ENG_COMPANY:
                        case ID_BCR_DEPARTMENT:
                        case ID_BCR_ENG_DEPARTMENT:
                        case ID_BCR_POSITION:
                        case ID_BCR_ENG_POSITION:
                        {
                            WCRecognitionCardModel *tempCardModel = [[WCRecognitionCardModel alloc] init];
                            
                            [self addRecogFieldToCard:tempCardModel
                                      fromRecogResult:resultBuf
                                      withRecogSource:recogSourceModel
                                        isReRecognize:NO];
                            
                            WCRecognitionFieldModel *newFieldModel = [tempCardModel recognitionFieldModelWithRecognitionFieldMainType:WCRecognitionFieldMainType_Company atIndex:0];
                            
                            [cardModel insertWCRecognitionFieldModel:newFieldModel];
                            
                            [tempCardModel release];
                            
                            break;
                        }
                        default:
                        {
                            [self addRecogFieldToCard:cardModel
                                      fromRecogResult:resultBuf
                                      withRecogSource:recogSourceModel
                                        isReRecognize:NO];
                            break;
                        }
                    }
                    
                    
                    result = YES;
                }
                else
                {
                    returnError = [self lastErrorWithCode:WCRO_ErrorCode_ClipRecognizeFailed description:nil];
                }
                
                FreeBinarize();
            }
            else
            {
                returnError = [self lastErrorWithCode:WCRO_ErrorCode_CreateBinarizebmpFailed description:nil];
            }
            
            [self removeGrayScaleData:&bmpData];
        }
        else
        {
            returnError = [self lastErrorWithCode:WCRO_ErrorCode_CreateGrayScaleDataFailed description:nil];
        }
        
        [image release];
    }
    while (0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil &&
       returnError!=nil)
    {
        *error = returnError;
    }
    
    return result;
}


//================================================================================
//
//================================================================================
- (WCRecognitionCardModel *)cardModelWithSignatureRecogSourceString:(NSString *)sourceString error:(NSError **)error;
{
    NSError *returnError = nil;
    
    WCRecognitionCardModel *cardModel = nil;
    
    do
    {
        NSString *recogString = sourceString;
        
        if(![recogString length])
        {
            returnError = [self lastErrorWithCode:WCRO_ErrorCode_EmptyMainSignatureString description:nil];
            break;
        }
        
        //////////////////////////////////////////////////
        
        unsigned short aResult[2048] = {0};
        unsigned short aResultTmp[2049] = {0};
        unsigned short *pRecogTxt = NULL;
        
        //////////////////////////////////////////////////
        
        pRecogTxt = (unsigned short*)malloc(([recogString length] + 1) * sizeof(unsigned short));
        memset(pRecogTxt, 0, ([recogString length] + 1) * sizeof(unsigned short));
        [recogString getCharacters:pRecogTxt range:NSMakeRange(0, [recogString length])];
        
        int rtn = Parsing(pRecogTxt, (unsigned short*)aResult);
        
        if(rtn >= 0)
        {
            aResultTmp[0]=rtn;
            memcpy(&aResultTmp[1], aResult, 2048*sizeof(unsigned short));
            
            // !! 存到預設group
            WCRecogSourceModel *recogSource = [[WCRecogSourceModel alloc] init];
            
            recogSource.cardID = [[NSString GUID] lowercaseString];
            recogSource.imageType = WCRecognitionImageType_FrontSide;
            recogSource.hasRecogRect = NO;
            recogSource.recogLang = [WCRecognitionPreferenceController systemLanguage];
            
            //////////////////////////////////////////////////
            
            //辨識完畢將結果處理成cardModel
            cardModel = [[WCRecognitionCardModel alloc] init];
            
            //將辨識結果存至cardModel
            [self addRecogFieldToCard:cardModel fromRecogResult:aResultTmp withRecogSource:recogSource isReRecognize:NO];
            
            //////////////////////////////////////////////////
            
            [recogSource release];
        }
        
        //釋放存辨識結果的空間
        free(pRecogTxt);
    }
    while (0);
    
    //////////////////////////////////////////////////
    
    if(error!=nil &&
       returnError!=nil)
    {
        *error = returnError;
    }
    
    return [cardModel autorelease];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - class methods


//==============================================================================
//
//==============================================================================
+ (NSString *)kernelVersion
{
    const char *version = SDKVersion();
    
    if(version==NULL)
    {
        return @"";
    }
    
    NSString *versionString = [NSString stringWithCString:version encoding:NSUTF8StringEncoding];
    
    if ([versionString length]==0)
    {
        return @"";
    }
    
    return versionString;
}


@end
