//
//  WCCardModel.m
//  WorldCard series
//

#import "WCCardModel.h"
#import "PPMutableDictionary+NSInteger.h"
#import "PPCountryCodeConvert.h"
#import "NSString+Additions.h"
#import "WCFieldLabelString.h"
#import "NSDate+Format.h"
#import "NSDate+TimeSection.h"
#import "PPWorldcardAddressController.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

static NSString *NoteAppendingFormat = @"%@: %@\n";

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface WCCardModel ()
@property (nonatomic, retain, readwrite) NSArray *groupIDArray;
@property (nonatomic, retain, readwrite) NSMutableDictionary *fieldArrayDict;
@property (nonatomic, retain) NSMutableArray *searchedFieldIDArray;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation WCCardModel

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods

//===============================================================================
//
//===============================================================================
- (id)init
{
    self = [super init];
    
    if(self)
    {
        self.frontRecogLang = -1;
        self.backRecogLang = -1;
        self.tagMask = WC_TagMask_None;
    }
    
    return self;
}


//===============================================================================
//
//===============================================================================
- (void)dealloc
{
    self.ID = nil;
    self.groupIDArray = nil;
    
    self.displayName = nil;
    self.displayCompany = nil;
    self.displayJobTitle = nil;
    self.displayPhone = nil;
    self.displayAddress = nil;
    self.displayGPS = nil;
    self.sectionTitle = nil;
    self.createdTime = nil;
    self.modifiedTime = nil;
    
    self.fieldArrayDict = nil;
    
    self.creator = nil;
    self.editor = nil;
    self.owner = nil;
    self.sharedAccountGUIDArray = nil;
    
    self.searchedFieldIDArray = nil;
    
    //////////////////////////////////////////////////
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Override methods

//===============================================================================
//
//===============================================================================
- (id)copyWithZone:(NSZone *)zone
{
    WCCardModel *cardModel = [[WCCardModel alloc] init];

    [self copyValueToCardModel:cardModel zone:zone];
   
    return cardModel;
}





//===============================================================================
//
//===============================================================================
- (BOOL)isEqual:(id)object
{
    if(![object isKindOfClass:[WCCardModel class]])
    {
        return NO;
    }
    
    //////////////////////////////////////////////////
    
    BOOL            isEqual = NO;
    WC_FieldType    fieldType;
    NSMutableArray  *srcFieldArray = nil;
    NSMutableArray  *dstFieldArray = nil;
    NSMutableArray  *srcCheckFieldArray = nil;
    NSMutableArray  *dstCheckFieldArray = nil;
    WCFieldModel    *srcFieldModel = nil;
    WCFieldModel    *dstFieldModel = nil;
    WCCardModel     *dstCardModel = (WCCardModel *)object;
    NSArray         *srcFieldTypeArray = [self.fieldArrayDict allKeys];
    NSArray         *dstFieldTypeArray = [dstCardModel.fieldArrayDict allKeys];

    
    //////////////////////////////////////////////////
    // main parameter
    // 動態產生的資料不進行比對 (EX:sectionTitle、displayName等)
    
    // ID
    if([self.ID isEqualToString:dstCardModel.ID] == NO)
    {
//        NSLog(@" --> cards have different ID");
        return NO;
    }
    
    // groupIDArray
    if([self.groupIDArray isEqualToArray:dstCardModel.groupIDArray] == NO)
    {
//        NSLog(@" --> cards have different groupIDArray");
        return NO;
    }

    // frontRecogLang
    if(self.frontRecogLang != dstCardModel.frontRecogLang)
    {
//        NSLog(@" --> cards have different frontRecogLang");
        return NO;
    }
    
    // backRecogLang
    if(self.backRecogLang != dstCardModel.backRecogLang)
    {
//        NSLog(@" --> cards have different backRecogLang");
        return NO;
    }

    // tagMask
    if(self.tagMask != dstCardModel.tagMask)
    {
//        NSLog(@" --> cards have different backRecogLang");
        return NO;
    }

    
    //////////////////////////////////////////////////
    // field
    
    if([srcFieldTypeArray count] != [dstFieldTypeArray count])
    {
//        NSLog(@" --> cards have different field type count");
        return NO;
    }
    
    for(int i=0; i<[srcFieldTypeArray count]; i++)
    {
        fieldType = [[srcFieldTypeArray objectAtIndex:i] intValue];
        srcFieldArray = [self fieldArrayWithType:fieldType];
        dstFieldArray = [dstCardModel fieldArrayWithType:fieldType];
        
        if([srcFieldArray count] != [dstFieldArray count])
        {
//            NSLog(@" --> cards have different field data count");
            goto _EXIT;
        }
        
        if(srcFieldArray)
        {
            srcCheckFieldArray = [[NSMutableArray alloc] initWithArray:srcFieldArray];
        }
        
        if(dstFieldArray)
        {
            dstCheckFieldArray = [[NSMutableArray alloc] initWithArray:dstFieldArray];
        }
        
        for(int j=0; j<[srcCheckFieldArray count]; j++)
        {
            srcFieldModel = [srcCheckFieldArray objectAtIndex:j];
            
            for(int k=0; k<[dstCheckFieldArray count]; k++)
            {
                dstFieldModel = [dstCheckFieldArray objectAtIndex:k];
                
                if([dstFieldModel isEqual:srcFieldModel])
                {
                    // 一樣的就拿掉，不用再比
                    [dstCheckFieldArray removeObjectAtIndex:k];
                    [srcCheckFieldArray removeObjectAtIndex:j];
                    j--;
                    break;
                }
            }
        }
        
        // 還有剩下field代表兩邊不一樣
        if([dstCheckFieldArray count] || [srcCheckFieldArray count])
        {
//            NSLog(@" --> cards have different field data value (src:%@ dst:%@)", srcCheckFieldArray, dstCheckFieldArray);
            goto _EXIT;
        }
        
        [dstCheckFieldArray release];
        dstCheckFieldArray = nil;
        [srcCheckFieldArray release];
        srcCheckFieldArray = nil;
    }
    
    isEqual = YES;
    
    //////////////////////////////////////////////////
    
_EXIT:
    
    [dstCheckFieldArray release];
    [srcCheckFieldArray release];
    
    return isEqual;
}


//===============================================================================
//
//===============================================================================
- (NSString *)description
{
    //#ifdef DEBUG
    
    NSMutableString *msg = [[NSMutableString alloc] initWithFormat:@"\r\n\r\n#################### WCCardModel (%p) ####################\r\n", self];
    
    [msg appendFormat:@"\r\n ID:%@", self.ID];
    [msg appendFormat:@"\r\n groupIDArray:%@", self.groupIDArray];
    [msg appendFormat:@"\r\n displayName:%@", self.displayName];
    [msg appendFormat:@"\r\n displayCompany:%@", self.displayCompany];
    [msg appendFormat:@"\r\n displayJobTitle:%@", self.displayJobTitle];
    [msg appendFormat:@"\r\n displayPhone:%@", self.displayPhone];
    [msg appendFormat:@"\r\n displayAddress:%@", self.displayAddress];
    [msg appendFormat:@"\r\n displayGPS:%@", self.displayGPS];
    [msg appendFormat:@"\r\n sectionTitle:%@", self.sectionTitle];
    [msg appendFormat:@"\r\n createdTime:%@", self.createdTime];
    [msg appendFormat:@"\r\n modifiedTime:%@", self.modifiedTime];
    [msg appendFormat:@"\r\n tagMask:%ld", (long)self.tagMask];
    
    //////////////////////////////////////////////////
    // WCT 新增
    [msg appendFormat:@"\r\n creator:%@", _creator];
    [msg appendFormat:@"\r\n owner:%@", _owner];
    [msg appendFormat:@"\r\n editor:%@", _editor];
    [msg appendFormat:@"\r\n sharedAccountGUID:%@", _sharedAccountGUIDArray];
    
    
    [msg appendString:@"\r\n"];
    
    
    //////////////////////////////////////////////////
    // field
    NSArray *allKeys = [self.fieldArrayDict allKeys];
    
    for(NSString *key in allKeys)
    {
        NSMutableArray *array = [self.fieldArrayDict objectForKey:key];
        
        [msg appendFormat:@"\r\n ##########> <field array> type:%@, count:%d <##########\r\n", WCFieldTypeString([key intValue]), (int)[array count]];
        
        for(WCFieldModel *fieldModel in array)
            [msg appendFormat:@"%@", [fieldModel description]];
    }
    
    [msg appendString:@"\r\n"];
    
    return [msg autorelease];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance methods

//===============================================================================
//
//===============================================================================
- (void)appendSharedAccountGUIDWithGuids:(NSArray *)guids
{
    NSMutableArray *sharedAccountGuids = [NSMutableArray arrayWithArray:self.sharedAccountGUIDArray];
    
    for (NSString *guid in guids)
    {
        if ([sharedAccountGuids containsObject:guid]==NO)
        {
            [sharedAccountGuids addObject:guid];
        }
    }
    
    self.sharedAccountGUIDArray = sharedAccountGuids;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark # card group method

//===============================================================================
//
//===============================================================================
- (void)addGroupID:(WC_GroupID)groupID
{
    NSMutableArray *newGroupIDArray;
    
    if(self.groupIDArray)
        newGroupIDArray = [[NSMutableArray alloc] initWithArray:self.groupIDArray];
    else newGroupIDArray = [[NSMutableArray alloc] init];
    
    [newGroupIDArray addObject:[NSString stringWithInteger:groupID]];
    self.groupIDArray = newGroupIDArray;
    [newGroupIDArray release];
}


//===============================================================================
//
//===============================================================================
- (void)setGroupIDArray:(NSArray *)groupIDArray isInitCard:(BOOL)isInitCard
{
    NSMutableArray *verifiedGroupIDArray = nil;
    
    if([groupIDArray count] == 0)
    {
        verifiedGroupIDArray = [[NSMutableArray alloc] init];
    }
    else
    {
        verifiedGroupIDArray = [[NSMutableArray alloc] initWithArray:groupIDArray];
    }
    
    //////////////////////////////////////////////////

    // !! 檢查是否符合規則
    switch ([verifiedGroupIDArray count])
    {
        case 0: // 至少要有一個group
        {
            [verifiedGroupIDArray addObject:[NSString stringWithInteger:WC_GID_Unfiled]];
            break;
        }
            
        case 1: // 只有一個group時不應該是 WC_GID_All
        {
            WC_GroupID groupID = [[verifiedGroupIDArray objectAtIndex:0] intValue];
            
            switch (groupID)
            {
                case WC_GID_All:
                    [verifiedGroupIDArray removeObjectAtIndex:0];
                    [verifiedGroupIDArray addObject:[NSString stringWithInteger:WC_GID_Unfiled]];
                    break;
                    
                default:
                    break;
            }
            
            break;
        }
            
        default:
            break;
    }
    
    //////////////////////////////////////////////////
    
    if(isInitCard == YES)
    {
        self.groupIDArray = verifiedGroupIDArray;
    }
    else if([self.groupIDArray isEqualToArray:verifiedGroupIDArray] == NO)
    {
        self.groupIDArray = verifiedGroupIDArray;
    }
    
    [verifiedGroupIDArray release];
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark # card field method

//===============================================================================
// get field array by specified type
// retuen : array of WCFieldModel
//===============================================================================
- (NSMutableArray *)fieldArrayWithType:(WC_FieldType)fieldType
{
    if(!self.fieldArrayDict)
        self.fieldArrayDict = [[[NSMutableDictionary alloc] init] autorelease];
    
    NSMutableArray *fieldArray = [self.fieldArrayDict objectForIntKey:fieldType];
    
    if(!fieldArray)
    {
        fieldArray = [[NSMutableArray alloc] init];
        [self.fieldArrayDict setObject:fieldArray forIntKey:fieldType];
        [fieldArray release];
    }
    
    return fieldArray;
}


//===============================================================================
// 新增欄位資料
//===============================================================================
- (void)addField:(WCFieldModel *)fieldModel
{
    NSMutableArray *fieldArray = [self fieldArrayWithType:fieldModel.type];
    
    // append in the tail
    if([fieldArray count])
    {
        WCFieldModel *lastFieldModel = [fieldArray lastObject];
        fieldModel.order = lastFieldModel.order+1;
    }
    else fieldModel.order = 0;
    
    [fieldArray addObject:fieldModel];
}


//===============================================================================
//
//===============================================================================
- (void)insertField:(WCFieldModel *)fieldModel atIndex:(NSInteger)index
{
    NSMutableArray *fieldArray = [self fieldArrayWithType:fieldModel.type];
    
    if(index <= [fieldArray count])
    {
        [fieldArray insertObject:fieldModel atIndex:index];
        
        for(int i=0; i<[fieldArray count]; i++)
        {
            WCFieldModel *oneModel = [fieldArray objectAtIndex:i];
            oneModel.order = i;
        }
    }
}


//===============================================================================
//
//===============================================================================
- (void)removeField:(WCFieldModel *)fieldModel
{
    //-------------------------------------------------
    // hadle field array
    // !! do not reset field order here, it will be done when save to DB.
    //-------------------------------------------------
    // delete from field array
    
    NSMutableArray *fieldArray = [self fieldArrayWithType:fieldModel.type];
    
    // !! 直接用removeObject會把內容相同的都刪除
    [fieldArray removeObject:fieldModel];
}


//===============================================================================
//
//===============================================================================
- (void)removeFieldWithType:(WC_FieldType)fieldType index:(NSInteger)index
{
    NSMutableArray *fieldArray = [self fieldArrayWithType:fieldType];
    
    if(index < [fieldArray count])
    {
        [fieldArray removeObjectAtIndex:index];
    }
    
    if([fieldArray count] == 0)
    {
        [self.fieldArrayDict removeObjectForIntKey:fieldType];
    }
}


//===============================================================================
//
//===============================================================================
- (void)moveFieldToFirstWithType:(WC_FieldType)fieldType index:(NSInteger)index
{
    NSMutableArray *fieldArray = [self fieldArrayWithType:fieldType];
    
    if(index < [fieldArray count])
    {
        [fieldArray exchangeObjectAtIndex:index withObjectAtIndex:0];
        
        // reset order
        for(int i=0; i<[fieldArray count]; i++)
        {
            WCFieldModel *fieldModel = [fieldArray objectAtIndex:i];
            
            fieldModel.order = i;
        }
    }
}


//===============================================================================
//
//===============================================================================
- (void)updateField:(WCFieldModel *)fieldModel
{
    // !! 更新過後的欄位就算是user create
    fieldModel.source = WC_FS_UserCreate;
}


//===============================================================================
// get field model by field ID
//===============================================================================
- (WCFieldModel *)fieldWithType:(WC_FieldType)fieldType ID:(NSInteger)fieldID
{
    NSMutableArray *fieldArray = [self fieldArrayWithType:fieldType];
    
    if(![fieldArray count])
        return nil;
    
    __block WCFieldModel *result = nil;
    
    [fieldArray enumerateObjectsWithOptions:NSEnumerationReverse usingBlock:^(WCFieldModel *fieldModel, NSUInteger idx, BOOL *stop) {
        if(fieldModel.ID == fieldID)
        {
            result = fieldModel;
            *stop = YES;
        }
    }];
    
    return result;
}


//===============================================================================
// get field model by field index
//===============================================================================
- (WCFieldModel *)fieldWithType:(WC_FieldType)fieldType index:(NSInteger)fieldIndex
{
    NSMutableArray *fieldArray = [self fieldArrayWithType:fieldType];
    
    if(fieldIndex >= [fieldArray count])
	{
        return nil;
	}
    
    return [fieldArray objectAtIndex:fieldIndex];
}


//===============================================================================
//
//===============================================================================
- (NSInteger)totalFieldCount
{
    NSArray *allFieldArray = [self.fieldArrayDict allValues];
    NSInteger totalCount = 0;
    
    for(NSArray *fieldArray in allFieldArray)
	{
        totalCount += [fieldArray count];
	}
    
    return totalCount;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark # other method

//===============================================================================
// 找出空的位置塞入subType2資料，如果已經有資料就回傳nil
//===============================================================================
- (WCFieldModel *)insertSubType2Field:(WCFieldModel *)subType2Field
{
    NSArray *existFieldArray = [self fieldArrayWithType:subType2Field.type];
    WCFieldModel *insertedFieldModel = nil;
    
    
    // 先看有沒有空的位置可以塞
    for(WCFieldModel *existFieldModel in existFieldArray)
    {
        // !! 檢查有沒有相同"語言"的資料
        if([existFieldModel hasFieldWithSubType2:subType2Field.subType2] == NO)
        {
            [existFieldModel setSubType2Field:subType2Field];
            insertedFieldModel = existFieldModel;
            break;
        }
        else
        {
            // !! 已經有相同資料就不要塞
            if([[existFieldModel fieldWithSubType2:subType2Field.subType2] isEqual:subType2Field])
                return nil;
        }
    }
    
    // 沒有就要新增一個field儲存
    if(!insertedFieldModel)
    {
        WCFieldModel *newFieldModel = [[WCFieldModel alloc] init];
        
        newFieldModel.isCJKStringValue = subType2Field.isCJKStringValue;
        newFieldModel.source = subType2Field.source;
        newFieldModel.type = subType2Field.type;
        newFieldModel.subType1 = subType2Field.subType1;
        [newFieldModel setSubType2Field:subType2Field];
        
        [self addField:newFieldModel];
        [newFieldModel release];
        
        insertedFieldModel = newFieldModel;
    }
    
    return insertedFieldModel;
}


//===============================================================================
// new card model with display data only
//===============================================================================
- (WCCardModel *)copyCardWithDisplayData
{
    WCCardModel *cardModel = [[WCCardModel alloc] init];
    
    cardModel.ID = self.ID;
    cardModel.sourceID = self.sourceID;
    [cardModel setGroupIDArray:self.groupIDArray isInitCard:YES];
    cardModel.displayName = self.displayName;
    cardModel.displayCompany = self.displayCompany;
    cardModel.displayJobTitle = self.displayJobTitle;
    cardModel.displayPhone = self.displayPhone;
    cardModel.displayAddress = self.displayAddress;
    cardModel.displayGPS = self.displayGPS;
    cardModel.sectionTitle = self.sectionTitle;
    cardModel.tagMask = self.tagMask;
    
    cardModel.createdTime = self.createdTime;
    cardModel.modifiedTime = self.modifiedTime;
    
    // WCT新增
    cardModel.containImageType = self.containImageType;
    cardModel.creator = self.creator;
    cardModel.owner = self.owner;
    cardModel.editor = self.editor;

    if ([self.sharedAccountGUIDArray count]>0)
    {
        cardModel.sharedAccountGUIDArray = [NSArray arrayWithArray:self.sharedAccountGUIDArray];
    }
    return cardModel;
}


//===============================================================================
// add searched field ID
//===============================================================================
- (void)addSeachedFieldID:(NSInteger)fieldID
{
    if(!self.searchedFieldIDArray)
        self.searchedFieldIDArray = [[[NSMutableArray alloc] init] autorelease];
    
    //[searchedFieldIDArray_ addObject:[NSNumber numberWithInt:fieldID]];
    
    BOOL bIsExist = NO;
    
    for(NSNumber *curFieldID in self.searchedFieldIDArray)
    {
        if([curFieldID intValue] == fieldID)
        {
            bIsExist = YES;
            break;
        }
    }
    
    if(!bIsExist)
        [self.searchedFieldIDArray addObject:@(fieldID)];
}


//===============================================================================
// get all searched field ID
// RETURNED: If values in card model is from search result, return FieldID array.
//           Otherwise, return value will be nil.
//===============================================================================
- (NSMutableArray *)searchedFieldIDs
{
    return self.searchedFieldIDArray;
}


//===============================================================================
// check if field value exist
//===============================================================================
- (BOOL)isFieldValueExist:(WCFieldModel *)fieldModel inArray:(NSArray *)existFieldArray
{
    for(WCFieldModel *existField in existFieldArray)
    {
        if([existField hasSameValue:fieldModel])
            return YES;
    }
    
    return NO;
}


//===============================================================================
//
//===============================================================================
- (BOOL)hasFieldData
{
    NSArray *allFieldArray = [self.fieldArrayDict allValues];
    
    for(NSArray *fieldArray in allFieldArray)
    {
        if([fieldArray count])
            return YES;
    }
    
    return NO;
}


//===============================================================================
// remove recogRect in all fields
//===============================================================================
- (void)removeRecogRectInAllFields
{
    NSArray *allFieldArray = [self.fieldArrayDict allValues];
    
    for(NSArray *fieldArray in allFieldArray)
    {
        for(WCFieldModel *fieldModel in fieldArray)
        {
            if(!CGRectIsEmpty(fieldModel.recogRect))
            {
                fieldModel.recogRect = CGRectZero;
            }
            
            if([fieldModel.value isKindOfClass:[NSDictionary class]])
            {
                NSArray *subFieldArray = [(NSDictionary *)fieldModel.value allValues];
                
                for(WCFieldModel *subFieldModel in subFieldArray)
                {
                    if(!CGRectIsEmpty(subFieldModel.recogRect))
                    {
                        subFieldModel.recogRect = CGRectZero;
                    }
                }
            }
        }
    }
}


//===============================================================================
// remove recogRect in all fields
//===============================================================================
- (void)removeRecogRectInAllFieldsWithFieldSource:(WC_FieldSource)fieldSource
{
    NSArray *allFieldArray = [self.fieldArrayDict allValues];
    
    for(NSArray *fieldArray in allFieldArray)
    {
        for(WCFieldModel *fieldModel in fieldArray)
        {
            if(fieldModel.source==fieldSource)
            {
                if(!CGRectIsEmpty(fieldModel.recogRect))
                {
                    fieldModel.recogRect = CGRectZero;
                }
                
                if([fieldModel.value isKindOfClass:[NSDictionary class]])
                {
                    NSArray *subFieldArray = [(NSDictionary *)fieldModel.value allValues];
                    
                    for(WCFieldModel *subFieldModel in subFieldArray)
                    {
                        if(!CGRectIsEmpty(subFieldModel.recogRect))
                        {
                            subFieldModel.recogRect = CGRectZero;
                        }
                    }
                }
            }
        }
    }
}


//===============================================================================
//
//===============================================================================
- (void)removeFieldWithSource:(WC_FieldSource)fieldSource
{
    NSArray *allFieldArray = [self.fieldArrayDict allValues];
    
    for(NSMutableArray *fieldArray in allFieldArray)
    {
        for(int i=0; i<[fieldArray count]; i++)
        {
            WCFieldModel *fieldModel = [fieldArray objectAtIndex:i];
            
            if(fieldModel.source == fieldSource)
            {
                [fieldArray removeObjectAtIndex:i];
                i--;
            }
        }
    }
}


//===============================================================================
// remove empty fields
//===============================================================================
- (void)removeEmptyFields
{
    NSArray *allFieldArray = [self.fieldArrayDict allValues];
    NSArray *allMainTypes = [self.fieldArrayDict allKeys];
    NSMutableArray *removeMainTypes = [[NSMutableArray alloc] init];
    NSCharacterSet *trimCharSet = [NSCharacterSet whitespaceAndNewlineCharacterSet];
    
    
    for(int i=0; i<[allFieldArray count]; i++)
    {
        NSMutableArray *fieldArray = [allFieldArray objectAtIndex:i];
        
        for(int j=0; j<[fieldArray count]; j++)
        {
            WCFieldModel *fieldModel = [fieldArray objectAtIndex:j];
            
            if(!fieldModel.value)
            {
                [fieldArray removeObjectAtIndex:j];
                j--;
            }
            else
            {
                if([fieldModel.value isKindOfClass:[NSDictionary class]])
                {
                    NSMutableDictionary     *dictValue = (NSMutableDictionary *)fieldModel.value;
                    NSMutableArray          *removeKeys = [[NSMutableArray alloc] init];
                    NSArray                 *allKeys = [dictValue allKeys];
                    NSArray                 *allValues = [dictValue allValues];
                    
                    for(int k=0; k<[allValues count]; k++)
                    {
                        WCFieldModel *subType2FieldModel = [allValues objectAtIndex:k];
                        
                        if(!subType2FieldModel.value ||
                           ([subType2FieldModel.value isKindOfClass:[NSString class]] && ![subType2FieldModel.value length]))
                        {
                            [removeKeys addObject:[allKeys objectAtIndex:k]];
                        }
                    }
                    
                    [dictValue removeObjectsForKeys:removeKeys];
                    [removeKeys release];
                    
                    if([dictValue count] == 0)
                    {
                        [fieldArray removeObjectAtIndex:j];
                        j--;
                    }
                }
                else if([fieldModel.value isKindOfClass:[NSString class]])
                {
                    // !! 濾掉全部是空白的資料
                    NSString *valueString = fieldModel.value;
                    valueString = [valueString stringByTrimmingCharactersInSet:trimCharSet];
                    
                    if(![valueString length])
                    {
                        [fieldArray removeObjectAtIndex:j];
                        j--;
                    }
                }
            }
        }
        
        if([fieldArray count] == 0)
            [removeMainTypes addObject:[allMainTypes objectAtIndex:i]];
    }
    
    [self.fieldArrayDict removeObjectsForKeys:removeMainTypes];
    [removeMainTypes release];
}


//===============================================================================
//
//===============================================================================
- (NSInteger)fieldCountWithFieldType:(WC_FieldType)fieldType
{
    NSArray *fieldArray = [self.fieldArrayDict objectForIntKey:fieldType];
    
    return [fieldArray count];
}


//===============================================================================
//
//===============================================================================
- (void)resetAllFieldOrder
{
    NSArray *allFieldArray = [self.fieldArrayDict allValues];
    
    for(NSArray *fieldArray in allFieldArray)
    {
        for(int i=0; i<[fieldArray count]; i++)
        {
            WCFieldModel *fieldModel = [fieldArray objectAtIndex:i];
            fieldModel.order = i;
        }
    }
}


//===============================================================================
// 檢查名片資料是否屬於指定  group
// wcmac中介面上會用WC_GID_Favorite/WC_GID_Unverified，要對應到cardModel的tagMask
//===============================================================================
- (BOOL)isBelongToGroupID:(WC_GroupID)groupID
{
    BOOL result = NO;
    
    switch (groupID)
    {
        case WC_GID_All:
        case WC_GID_AllCardInDb:
        {
            result = YES;
            break;
        }
            
        case WC_GID_Favorite:
        {
            result = (self.tagMask & WC_TagMask_Favorite);
            break;
        }

        case WC_GID_Unverified:
        {
            result = (self.tagMask & WC_TagMask_Unverified);
            break;
        }

        default:
        {
            // one of groupID in cardModle must equal to cache groupID
            for(NSString *groupIDString in self.groupIDArray)
            {
                if([groupIDString intValue] == groupID)
                {
                    result = YES;
                    break;
                }
            }
            
            break;
        }
    }
    
    return result;
}


//===============================================================================
// 檢查名片資料是否屬於指定 source + group
//===============================================================================
- (BOOL)isBelongToSourceID:(WC_SourceID)sourceID groupID:(WC_GroupID)groupID
{
    BOOL result = NO;
    
    if(groupID==WC_GID_AllCardInDb)
    {
        return YES;
    }
    
    
    if(self.sourceID == sourceID)
    {
        if(groupID == WC_GID_All)
        {
            result = YES;
        }
        else
        {   // one of groupID in cardModle must equal to cache groupID
            for(NSString *groupIDString in self.groupIDArray)
            {
                if([groupIDString intValue] == groupID)
                {
                    result = YES;
                    break;
                }
            }
        }
    }
    
    return result;
}


//===============================================================================
// 取得辨識語系對應的語系代碼
//===============================================================================
- (NSString *)languageCodeWithFieldSource:(WC_FieldSource)fieldSource
{
    NSString *langCode = nil;
    
    switch (fieldSource)
    {
        case WC_FS_FrontRecog:
            langCode = [PPCountryCodeConvert languageCodeWithRecogLang:self.frontRecogLang];
            break;
            
        case WC_FS_BackRecog:
            langCode = [PPCountryCodeConvert languageCodeWithRecogLang:self.backRecogLang];
            break;
            
        default:
            break;
    }
    
    return langCode;
}


//===============================================================================
// 取得辨識語系, 手動時回傳-1
//===============================================================================
- (NSInteger)recognitionlanguageWithFieldSource:(WC_FieldSource)fieldSource
{
    switch (fieldSource)
    {
        case WC_FS_FrontRecog:
            return self.frontRecogLang;
            
        case WC_FS_BackRecog:
            return self.backRecogLang;
            
        default:
            return -1;
    }
}


//===============================================================================
//
//===============================================================================
- (BOOL)isSameGroupIDArray:(NSArray *)groupIDArray
{
    NSArray *srcArray = [self.groupIDArray sortedArrayUsingSelector:@selector(caseInsensitiveCompare:)];
    NSArray *dstArray = [groupIDArray sortedArrayUsingSelector:@selector(caseInsensitiveCompare:)];
    
    return [srcArray isEqualToArray:dstArray];
}


//==============================================================================
//
//==============================================================================
- (void)clearAllAddressGPS
{
    NSMutableArray *addressArray = [self fieldArrayWithType:WC_FT_Address];

    for (WCFieldModel *fieldModel in addressArray)
    {
        [fieldModel removeFieldWithWithSubType2:WC_FST2_Address_Location];
    }
}


//==============================================================================
//
//==============================================================================
- (NSString *)displaySectionTitleWithSortByField:(WC_SortedByField)sortByField indexingMode:(PPSIC_Mode)indexingMode
{
    switch (sortByField)
    {
        case WC_SBF_Name:
        case WC_SBF_Company:
        {
            NSArray *sortingFieldTitles = [self.sectionTitle componentsSeparatedByString:@";"];
            if([sortingFieldTitles count]>1)
            {
                NSArray *sectionModeTitles = [sortingFieldTitles[sortByField-1] componentsSeparatedByString:@"|"];
                if([sectionModeTitles count]>0)
                {
                    return sectionModeTitles[indexingMode-1];
                }
            }
            else
            {
                // 如果無法切開，直接回傳
                return self.sectionTitle;
            }
            return @"";
        }
        case WC_SBF_CreateTime:
        {
            return [self.createdTime sectionTitle];
        }
        case WC_SBF_ModifiedTime:
        {
            return [self.modifiedTime sectionTitle];
        }
        default:
            return @"";
    }
}


//================================================================================
//
//================================================================================
- (void)copyValueToCardModel:(WCCardModel *)cardModel zone:(NSZone *)zone
{
    if(cardModel != nil)
    {
        cardModel.ID = [[self.ID copyWithZone:zone] autorelease];
        cardModel.sourceID = self.sourceID;
        [cardModel setGroupIDArray:self.groupIDArray isInitCard:YES];
        
        cardModel.displayName = [[self.displayName copyWithZone:zone] autorelease];
        cardModel.displayCompany = [[self.displayCompany copyWithZone:zone] autorelease];
        cardModel.displayJobTitle = [[self.displayJobTitle copyWithZone:zone] autorelease];
        cardModel.displayPhone = [[self.displayPhone copyWithZone:zone] autorelease];
        cardModel.displayAddress = [[self.displayAddress copyWithZone:zone] autorelease];
        cardModel.displayGPS = [[self.displayGPS copyWithZone:zone] autorelease];
        cardModel.sectionTitle = [[self.sectionTitle copyWithZone:zone] autorelease];
        cardModel.createdTime = [[self.createdTime copyWithZone:zone] autorelease];
        cardModel.modifiedTime = [[self.modifiedTime copyWithZone:zone] autorelease];
        cardModel.frontRecogLang = self.frontRecogLang;
        cardModel.backRecogLang = self.backRecogLang;
        cardModel.isSelected = self.isSelected;
        cardModel.tagMask = self.tagMask;
        
        //////////////////////////////////////////////////
        
        NSArray *allKeys = [self.fieldArrayDict allKeys];
        
        for(NSString *key in allKeys)
        {
            NSMutableArray *array = [self.fieldArrayDict objectForKey:key];
            
            for(WCFieldModel *fieldModel in array)
            {
                WCFieldModel *newFieldModel = [fieldModel copy];
                [cardModel addField:newFieldModel];
                [newFieldModel release];
            }
        }
        
        //////////////////////////////////////////////////
        // WCMAC 新增
        
        cardModel.containImageType = self.containImageType;
        
        //////////////////////////////////////////////////
        // WCT 新增
        
        cardModel.creator = [[self.creator copyWithZone:zone] autorelease];
        cardModel.owner = [[self.owner copyWithZone:zone] autorelease];
        cardModel.editor = [[self.editor copyWithZone:zone] autorelease];
        cardModel.sharedAccountGUIDArray = [[self.sharedAccountGUIDArray copyWithZone:zone] autorelease];
    }
}


//================================================================================
// 將特定欄位的首字進行大寫處理
//================================================================================
- (void)capitalizedString
{
    NSArray *fieldArray = nil;
    
    
    //////////////////////////////////////////////////
    // Name
    
    fieldArray = [self fieldArrayWithType:WC_FT_Name];
    
    for(WCFieldModel *srcFieldModel in fieldArray)
    {
        NSDictionary *srcValuDict = srcFieldModel.value;
        NSArray *allValues = srcValuDict.allValues;
        
        for(WCFieldModel *srcSubFieldModel in allValues)
        {
            if(srcSubFieldModel.subType2 == WC_FST2_Name_First ||
               srcSubFieldModel.subType2 == WC_FST2_Name_Last  ||
               srcSubFieldModel.subType2 == WC_FST2_Name_Middle ||
               srcSubFieldModel.subType2 == WC_FST2_Name_Prefix ||
               srcSubFieldModel.subType2 == WC_FST2_Name_Suffix )
            {
                NSString *stringValue = (NSString *)srcSubFieldModel.value;
                
                // Chung, 首字母大寫, 是將全部變小寫, 僅首字母為大寫
                srcSubFieldModel.value = [[stringValue lowercaseString] capitalizedString];
            }
        }
    }
    
    
    //////////////////////////////////////////////////
    // Company
    
    fieldArray = [self fieldArrayWithType:WC_FT_Company];
    
    for(WCFieldModel *srcFieldModel in fieldArray)
    {
        NSDictionary *srcValuDict = srcFieldModel.value;
        NSArray *allValues = srcValuDict.allValues;
        
        for(WCFieldModel *srcSubFieldModel in allValues)
        {
            if(srcSubFieldModel.subType2 == WC_FST2_Company_Name ||
               srcSubFieldModel.subType2 == WC_FST2_Company_Department  ||
               srcSubFieldModel.subType2 == WC_FST2_Company_JobTitle)
            {
                NSString *stringValue = (NSString *)srcSubFieldModel.value;
                
                // Chung, 首字母大寫, 是將全部變小寫, 僅首字母為大寫
                srcSubFieldModel.value = [[stringValue lowercaseString] capitalizedString];
            }
        }
    }
}


//================================================================================
//
//================================================================================
- (NSString *)simpleDescription
{
    WCFieldModel *nameField = [self fieldWithType:WC_FT_Name index:0];
    NSDictionary *subFieldDict = [nameField value];
    NSMutableArray *nameValues = [NSMutableArray array];
    
    for(WCFieldModel *subField in [subFieldDict allValues])
    {
        if([subField.value length] > 0)
        {
            [nameValues addObject:subField.value];
        }
    }
    
    return [NSString stringWithFormat:@"{name=%@, groupIDArray=%@}",
            [nameValues componentsJoinedByString:@","],
            [self.groupIDArray componentsJoinedByString:@","]];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - combind methods

//==============================================================================
//
//==============================================================================
- (NSUInteger)fieldMaxCountWithType:(WC_FieldType)fieldType
{
    switch (fieldType)
    {
        case WC_FT_Name:
        {
            return WC_FMC_Name;
        }
        case WC_FT_Company:
        {
            return WC_FMC_Company;
        }
        case WC_FT_Address:
        {
            return WC_FMC_Address;
        }
        case WC_FT_Phone:
        {
            return WC_FMC_Phone;
        }
        case WC_FT_Email:
        {
            return WC_FMC_Email;
        }
        case WC_FT_URL:
        {
            return WC_FMC_URL;
        }
        case WC_FT_InstantMessage:
        {
            return WC_FMC_InstantMessage;
        }
        case WC_FT_Date:
        {
            return WC_FMC_Date;
        }
        case WC_FT_Note:
        {
            return WC_FMC_Note;
        }
        case WC_FT_Nickname:
        {
            return WC_FMC_Nickname;
        }
        case WC_FT_UnifiedBusinessNo:
        {
            return WC_FMC_UnifiedBusinessNo;
        }
        case WC_FT_SocialNetwork:
        {
            return WC_FMC_SocialNetWork;
        }
        case WC_FT_UserDefine:
        {
            return WC_FMC_UserDefine;
        }
        default:
            return 0;
    }
}


//==============================================================================
// 目前的field type能不能再加新欄位，如果 currentCount>=fieldMaxCount就不能加了
//==============================================================================
- (BOOL)canAddFieldWithType:(WC_FieldType)fieldType
{
    NSUInteger currentCount = [[self fieldArrayWithType:fieldType] count];
    NSUInteger fieldMaxCount = [self fieldMaxCountWithType:fieldType];
    
    return (currentCount<fieldMaxCount);
}


//==============================================================================
//
//==============================================================================
- (BOOL)canAddBirthdayField
{
    NSMutableArray *dateArray = [self fieldArrayWithType:WC_FT_Date];
    
    for (WCFieldModel *fieldModel in dateArray)
    {
        if (fieldModel.subType1==WC_FST1_Date_Birthday)
        {
            return NO;
        }
    }
    
    return YES;
}


//==============================================================================
// 將fieldModel轉為可加到note的字串
//==============================================================================
- (NSString *)noteStringWithFieldModel:(WCFieldModel *)fieldModel
{
    NSMutableString *noteString = [NSMutableString string];
    switch (fieldModel.type)
    {
        case WC_FT_Name:
        {
            for (WC_FieldSubType2 subType2 = WC_FST2_Name_First; subType2 <=WC_FST2_Name_Suffix; subType2++)
            {
                NSString *label = [WCFieldLabelString labelStringWithType:subType2];
                NSString *value = (NSString *)[fieldModel valueWithSubType2:subType2];
                if ([value length]>0)
                {
                    [noteString appendFormat:NoteAppendingFormat,label, value];
                }
            }
            break;
        }
        case WC_FT_Company:
        {
            for (WC_FieldSubType2 subType2 = WC_FST2_Company_Name; subType2 <=WC_FST2_Company_JobTitle; subType2++)
            {
                NSString *label = [WCFieldLabelString labelStringWithType:subType2];
                NSString *value = (NSString *)[fieldModel valueWithSubType2:subType2];
                if ([value length]>0)
                {
                    [noteString appendFormat:NoteAppendingFormat,label, value];
                }
            }
            break;
        }
        case WC_FT_Address:
        {
            NSString *label = [WCFieldLabelString labelStringWithType:fieldModel.subType1];
            
            //////////////////////////////////////////////////
            // !! 這邊的code是從PPWorldcardAddressController+WCFieldModel來的
            // 因為PPWorldcardAddressController+WCFieldModel會dependency WCDataModel/Model，所以不能直接使用
            // 取得完整地址
            NSInteger bcrLang = -1;
            
            switch (fieldModel.source)
            {
                case WC_FS_FrontRecog:
                {
                    bcrLang = self.frontRecogLang;
                    break;
                }
                case WC_FS_BackRecog:
                {
                    bcrLang = self.backRecogLang;
                    break;
                }
                default:
                    break;
            }
            
            NSString *countryCode = [fieldModel valueWithSubType2:WC_FST2_Address_CountryCode];
            NSString *country = [fieldModel valueWithSubType2:WC_FST2_Address_Country];
            NSString *street = [fieldModel valueWithSubType2:WC_FST2_Address_Street];
            NSString *state = [fieldModel valueWithSubType2:WC_FST2_Address_State];
            NSString *city = [fieldModel valueWithSubType2:WC_FST2_Address_City];
            NSString *zip = [fieldModel valueWithSubType2:WC_FST2_Address_ZIP];
            
            NSMutableString *returnedCountry = [NSMutableString string];
            NSMutableString *returnedCountryCode = [NSMutableString string];
            
            // 有country code要設定到returnedCountryCode
            if ([countryCode length]>0)
            {
                [returnedCountryCode setString:countryCode];
            }
            
            // 有country code要設定到returnedCountry
            if ([country length]>0)
            {
                [returnedCountry setString:country];
            }
            
            // 沒有country code也沒有country，country code用default
            if([returnedCountry length]==0 && [returnedCountryCode length]==0)
            {
                [returnedCountryCode setString:[PPCountryCodeConvert defaultCountryCode]];
            }
            
            //////////////////////////////////////////////////
            NSInteger format = [PPWorldcardAddressController formatFromStreet:street
                                                                         city:city
                                                                     province:state
                                                                   postalCode:zip
                                                                  countryName:returnedCountry
                                                                       format:-1
                                                                  countryCode:returnedCountryCode
                                                                  bcrLanguage:bcrLang
                                                               systemLanguage:[PPWorldcardAddressController addressSystemLanguage]];
            
            //////////////////////////////////////////////////
            
            NSString *fullAddress = [PPWorldcardAddressController addressFromStreet:street city:city province:state postalCode:zip countryName:returnedCountry format:format];
            
            if ([fullAddress length]>0)
            {
                [noteString appendFormat:NoteAppendingFormat,label, fullAddress];
            }
            break;
        }
        case WC_FT_Phone:
        case WC_FT_Email:
        case WC_FT_URL:
        case WC_FT_InstantMessage:
        case WC_FT_SocialNetwork:
        {
            NSString *label = [WCFieldLabelString labelStringWithType:fieldModel.subType1];
            NSString *value = fieldModel.value;
            if ([value length]>0)
            {
                [noteString appendFormat:NoteAppendingFormat,label, value];
            }
            break;
        }
        case WC_FT_Nickname:
        case WC_FT_UnifiedBusinessNo:
        {
            NSString *label = [WCFieldLabelString labelStringWithType:fieldModel.type];
            NSString *value = fieldModel.value;
            if ([value length]>0)
            {
                [noteString appendFormat:NoteAppendingFormat,label, value];
            }
            break;
        }
        case WC_FT_Date:
        {
            NSString *label = [WCFieldLabelString labelStringWithType:fieldModel.type];
            NSString *value = [(NSDate *)fieldModel.value stringWithFormat:NSDateFormat_Day timeZone:[NSTimeZone timeZoneForSecondsFromGMT:0]];
            if ([value length]>0)
            {
                [noteString appendFormat:NoteAppendingFormat,label, value];
            }
            break;
        }
        case WC_FT_Note:
        {
            // 另外處理，這邊不會進來
            break;
        }
            
        default:
            break;
    }
    return ([noteString length]>0)?[NSString stringWithString:noteString]:nil;
}


//===============================================================================
// combine other card
//===============================================================================
- (void)combineCard:(WCCardModel *)cardModel
{
    NSArray         *sourceFieldTypeArray =[ [cardModel.fieldArrayDict allKeys] sortedArrayUsingSelector:@selector(caseInsensitiveCompare:)];
    NSUInteger      sourceFieldTypeCount = [sourceFieldTypeArray count];
    NSArray         *sourceFieldArray    = nil;
    NSArray         *targetFieldArray    = nil;
    WC_FieldType    fieldType            = WC_FT_None;
    NSString        *noteAppendString    = @"";
    
    //////////////////////////////////////////////////
    for(NSUInteger i=0; i<sourceFieldTypeCount; i++)
    {
        fieldType = [[sourceFieldTypeArray objectAtIndex:i] intValue];
        
        // !!note要直接合併到目前的note裡
        if (fieldType==WC_FT_Note)
        {
            sourceFieldArray = [cardModel fieldArrayWithType:fieldType];
            WCFieldModel *sourceNoteFieldModel = [sourceFieldArray firstObject];
            NSString *sourceNote = sourceNoteFieldModel.value;
            
            targetFieldArray = [self fieldArrayWithType:fieldType];
            WCFieldModel *targetNoteFieldModel = [targetFieldArray firstObject];
            if (targetNoteFieldModel==nil)
            {
                targetNoteFieldModel = [WCFieldModel newFieldModel];
                targetNoteFieldModel.type = WC_FT_Note;
                [self addField:targetNoteFieldModel];
                [targetNoteFieldModel release];
            }
            
            NSString *targetNote = targetNoteFieldModel.value;
            
            if ([targetNote length]>0 &&
                [targetNoteFieldModel.value isEqualToString:sourceNote]==NO)
            {
                targetNoteFieldModel.value  = [targetNote stringByAppendingFormat:@"\n%@",sourceNote];
            }
            else
            {
                targetNoteFieldModel.value = sourceNote;
            }
            continue;
        }
        

        //////////////////////////////////////////////////
        sourceFieldArray = [cardModel fieldArrayWithType:fieldType];
        targetFieldArray = [self fieldArrayWithType:fieldType];
        
        // 一般欄位合併
        for(WCFieldModel *fieldModel in sourceFieldArray)
        {
            // test
//            NSString *noteString = [self noteStringWithFieldModel:fieldModel];
//            if (noteString)
//            {
//                noteAppendString = [noteAppendString stringByAppendingString:noteString];
//            }
            
            

            if(![self isFieldValueExist:fieldModel inArray:targetFieldArray])
            {
                // Birthday要另外處理，上限1，不包含在date中
                if (fieldModel.type==WC_FT_Date && fieldModel.subType1==WC_FST1_Date_Birthday)
                {
                    if ([self canAddBirthdayField]==NO)
                    {
                        NSString *noteString = [self noteStringWithFieldModel:fieldModel];
                        if (noteString)
                        {
                            noteAppendString = [noteAppendString stringByAppendingString:noteString];
                        }
                    }
                    else
                    {
                        [self addField:fieldModel];
                    }
                }
                else
                {
                    // !! 如果目前名片的欄位數大於限制，要加到note
                    if ([self canAddFieldWithType:fieldType]==NO)
                    {
                        NSString *noteString = [self noteStringWithFieldModel:fieldModel];
                        if (noteString)
                        {
                            noteAppendString = [noteAppendString stringByAppendingString:noteString];
                        }
                    }
                    else
                    {
                        [self addField:fieldModel];
                    }
                }

            }
        }
    }
    
    //////////////////////////////////////////////////
    // 合併超過的字串到note中
    if ([noteAppendString length]>0)
    {
        NSString *dateString = [[NSDate date] stringWithFormat:NSDateFormat_Day];
        NSString *mergedTitle = [NSString stringWithFormat:[@"MLS_MergeTitleFormat" localizedFromTable:@"WCFieldLabelString"], dateString];
        
        targetFieldArray = [self fieldArrayWithType:WC_FT_Note];
        WCFieldModel *targetNoteFieldModel = [targetFieldArray firstObject];
        if (targetNoteFieldModel==nil)
        {
            targetNoteFieldModel = [WCFieldModel newFieldModel];
            targetNoteFieldModel.type = WC_FT_Note;
            [self addField:targetNoteFieldModel];
            [targetNoteFieldModel release];
        }
        
        NSString *targetNote = targetNoteFieldModel.value;
        if ([targetNote length]>0)
        {
            targetNoteFieldModel.value  = [targetNote stringByAppendingFormat:@"\n%@\n%@",mergedTitle, noteAppendString];
        }
        else
        {
            targetNoteFieldModel.value = [NSString stringWithFormat:@"%@\n%@",mergedTitle, noteAppendString];
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - 


//==============================================================================
//
//==============================================================================
- (void)fixFieldCount
{
    // MARK: 修正欄位數量，多放到note
    // 自訂欄位正常應該不會超過，所以不在這邊處理
    NSMutableString *additionalNote = [NSMutableString string];
    for (WC_FieldType type=WC_FT_Name; type<WC_FT_UserDefine; type++)
    {
        NSInteger maxFieldCount = [self fieldMaxCountWithType:type];
        NSMutableArray *fieldArray = [self fieldArrayWithType:type];
       
        //////////////////////////////////////////////////

        if(type==WC_FT_Date)
        {
            if([fieldArray count]>maxFieldCount &&
               maxFieldCount>0)
            {
                NSUInteger totalDateCount = 0;
                NSUInteger birthDateCount = 0;
                
                //沒有超過就不處理
                for (NSInteger index=0; index<fieldArray.count; index++)
                {
                    // 超過的要放到note
                    WCFieldModel *fieldModel = [fieldArray objectAtIndex:index];
                 
                    if(fieldModel.subType1==WC_FST1_Date_Birthday)
                    {
                        birthDateCount++;
                        
                        if(birthDateCount<=WC_FMC_BirthDay)
                        {
                            continue;
                        }
                    }
                    else
                    {
                        totalDateCount++;
                        
                        if(birthDateCount<=maxFieldCount)
                        {
                            continue;
                        }
                    }
                    
                    //////////////////////////////////////////////////

                    //將多餘的欄位移至 Note
                    NSString *fieldString = [self noteStringWithFieldModel:fieldModel];
                    [fieldArray removeObject:fieldModel];
                    
                    //////////////////////////////////////////////////
                    if ([fieldString length]>0)
                    {
                        [additionalNote appendString:fieldString];
                    }
                   
                }
            }
        }
        //非日期欄位，且超過欄位上限
        else if([fieldArray count]>maxFieldCount &&
                maxFieldCount>0)
        {
            NSMutableArray *fieldModelToBeRemove = [NSMutableArray array];

            //沒有超過就不處理
            for (NSInteger index=[fieldArray count]-1; index>maxFieldCount-1; index--)
            {
                // 超過的要放到note
                WCFieldModel *fieldModel = [fieldArray objectAtIndex:index];

                NSString *fieldString = [self noteStringWithFieldModel:fieldModel];

                // 紀錄下來，後面再一次remove
                [fieldModelToBeRemove addObject:fieldModel];

                //////////////////////////////////////////////////
                if ([fieldString length]>0)
                {
                    [additionalNote appendString:fieldString];
                }
            }

            if ([fieldModelToBeRemove count]>0)
            {
                [fieldArray removeObjectsInArray:fieldModelToBeRemove];
            }
        }
    }
    
    //////////////////////////////////////////////////
    // 加到note
    if ([additionalNote length]>0)
    {
        WCFieldModel *noteField = [self fieldWithType:WC_FT_Note index:0];
        if (noteField==nil)
        {
            noteField = [WCFieldModel newFieldModel];
            noteField.type = WC_FT_Note;
            [self addField:noteField];
            [noteField release];
        }
        
        NSString *targetNote = noteField.value;
        if ([targetNote length]>0)
        {
            noteField.value  = [targetNote stringByAppendingFormat:@"\nOver max field count:\n%@", additionalNote];
        }
        else
        {
            noteField.value = [NSString stringWithFormat:@"%@", additionalNote];
        }
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - sort

//==============================================================================
//
//==============================================================================
- (void)sortNameByFrontCJKFirst
{
    NSMutableArray *fields = [self fieldArrayWithType:WC_FT_Name];
    // 沒有欄位或個數只有1個，就不用sort
    if (!fields || [fields count]<2)
    {
        return ;
    }
    
    int fieldIndex = 0;
    for (WCFieldModel *fieldModel in fields)
    {
        // 不是正面就不處理
        if (fieldModel.source!=WC_FS_FrontRecog)
        {
            fieldIndex++;
            continue;
        }
        
        // 如果第一筆就是CJK，就不用處理
        if (fieldIndex==0)
        {
            if ([(NSString *)[fieldModel valueWithSubType2:WC_FST2_Name_First] isCJK]||
                [(NSString *)[fieldModel valueWithSubType2:WC_FST2_Name_Last] isCJK])
            {
                break;
            }
        }
        
        // 找到第一筆是CJK的紀錄就交換順序
        if ([(NSString *)[fieldModel valueWithSubType2:WC_FST2_Name_First] isCJK]||
            [(NSString *)[fieldModel valueWithSubType2:WC_FST2_Name_Last] isCJK])
        {
            [fieldModel retain];
            [fields removeObject:fieldModel];
            [fields insertObject:fieldModel atIndex:0];
            [fieldModel release];
            break;
        }
        fieldIndex++;
    }
}


//==============================================================================
//
//==============================================================================
- (void)sortCompanyByFrontCJKFirst
{
    NSMutableArray *fields = [self fieldArrayWithType:WC_FT_Company];
    // 沒有欄位或個數只有1個，就不用sort
    if (!fields || [fields count]<2)
    {
        return ;
    }
    
    int fieldIndex = 0;
    for (WCFieldModel *fieldModel in fields)
    {
        // 不是正面就不處理
        if (fieldModel.source!=WC_FS_FrontRecog)
        {
            fieldIndex++;
            continue;
        }
        
        // 如果第一筆就是CJK，就不用處理
        if (fieldIndex==0)
        {
            if ([(NSString *)[fieldModel valueWithSubType2:WC_FST2_Company_Name] isCJK]||
                [(NSString *)[fieldModel valueWithSubType2:WC_FST2_Company_Department] isCJK]||
                [(NSString *)[fieldModel valueWithSubType2:WC_FST2_Company_JobTitle] isCJK])
            {
                break;
            }
        }
        
        // 找到第一筆是CJK的紀錄就交換順序
        if ([(NSString *)[fieldModel valueWithSubType2:WC_FST2_Company_Name] isCJK]||
            [(NSString *)[fieldModel valueWithSubType2:WC_FST2_Company_Department] isCJK]||
            [(NSString *)[fieldModel valueWithSubType2:WC_FST2_Company_JobTitle] isCJK])
        {
            [fieldModel retain];
            [fields removeObject:fieldModel];
            [fields insertObject:fieldModel atIndex:0];
            [fieldModel release];
            break;
        }
        fieldIndex++;
    }
}

//===============================================================================
//
//===============================================================================
- (void)sortPhoneFields
{
    NSMutableArray *phoneArray = [self fieldArrayWithType:WC_FT_Phone];
    
    __block NSDictionary *orderDict = [NSDictionary dictionaryWithObjectsAndKeys:
                                       @"1", [NSNumber numberWithInt:WC_FST1_Phone_Mobile],
                                       @"2", [NSNumber numberWithInt:WC_FST1_Phone_Work],
                                       @"3", [NSNumber numberWithInt:WC_FST1_Phone_Home],
                                       @"4", [NSNumber numberWithInt:WC_FST1_Phone_Main],
                                       @"5", [NSNumber numberWithInt:WC_FST1_Phone_iPhone],
                                       @"6", [NSNumber numberWithInt:WC_FST1_Phone_WorkFax],
                                       @"7", [NSNumber numberWithInt:WC_FST1_Phone_HomeFax],
                                       @"8", [NSNumber numberWithInt:WC_FST1_Phone_Pager],
                                       @"9", [NSNumber numberWithInt:WC_FST1_Phone_Other],
                                       @"10", [NSNumber numberWithInt:WC_FST1_Phone_GoogleVoice],
                                       nil];
    
    
    [phoneArray sortUsingComparator:^(WCFieldModel *field1, WCFieldModel *field2)
     {
         NSString *orderStr1 = [orderDict objectForKey:[NSNumber numberWithInt:field1.subType1]];
         NSString *orderStr2 = [orderDict objectForKey:[NSNumber numberWithInt:field2.subType1]];
         NSInteger order1 = 99;
         NSInteger order2 = 99;
         NSComparisonResult result = NSOrderedSame;
         
         if([orderStr1 length])
             order1 = [orderStr1 integerValue];
         
         if([orderStr2 length])
             order2 = [orderStr2 integerValue];
         
         if(order1 < order2)
             result = NSOrderedAscending;
         else if(order1 > order2)
             result = NSOrderedDescending;
         
         return result;
     }];
}

@end
