//
//  WCCacheModel.m
//  WorldCardMobile
//
//  Created by  Eddie on 12/4/13.
//  Copyright (c) 2012年 __MyCompanyName__. All rights reserved.
//
//

#import "WCCacheModel.h"
#import "PPSectionIndexDefine.h"
#import "WCCompareTool.h"
#import "NSDate+TimeSection.h"

#define WCCM_MaxCacheSize_ThumbImage    2*1024*1024
#define WCCM_MaxCacheSize_IDPhotoImage  2*1024*1024
#define WCCM_MaxCacheSize_Location      1*1024*1024


static NSString *const favoriteSectionTitle = @"Favorite";

#pragma - NSString+CacheModel
////////////////////////////////////////////////////////////////////////////////////////////////////
@interface NSString (WCCacheModel)
- (NSString *)thumbImageIdentifier;
- (NSString *)idPhotoImageIdentifier;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////
@implementation NSString (WCCacheModel)


//==============================================================================
//
//==============================================================================
- (NSString *)thumbImageIdentifier
{
    return [NSString stringWithFormat:@"%@%@", self, @"_thumb"];
}


//==============================================================================
//
//==============================================================================
- (NSString *)idPhotoImageIdentifier
{
    return [NSString stringWithFormat:@"%@%@", self, @"_idPhoto"];
}
@end




#pragma - NSArray+WCCardSectionModel
////////////////////////////////////////////////////////////////////////////////////////////////////
@interface NSArray (WCCardSectionModel)
- (WCCardSectionModel *)sectionModelWithTitle:(NSString *)aTitle;
@end

@implementation NSArray (WCCardSectionModel)


//==============================================================================
//
//==============================================================================
- (WCCardSectionModel *)sectionModelWithTitle:(NSString *)aTitle
{
    for (WCCardSectionModel *cardSectionModel in self) {
        if ([cardSectionModel.title isEqualToString:aTitle]) {
            return cardSectionModel;
        }
    }
    
    return nil;
}
@end


////////////////////////////////////////////////////////////////////////////////////////////////////
@interface WCCacheModel ()
@property (atomic, assign              ) dispatch_queue_t    cacheModelQueue;           // 所有動作都要在queue中處理

@property (nonatomic, retain           ) NSCache             *cache;                    // cache, 存放名片縮圖，大頭貼，地址

@property (nonatomic, readwrite, retain) NSMutableDictionary *allCardDict;              // 所有card

@property (atomic, readwrite, retain   ) NSMutableArray      *allCardSectionArray;      // ios目前的cardSectionArray
@property (atomic, readwrite, retain   ) NSMutableDictionary *allCardSectionDict;       // mac的cardSection

@property (nonatomic, readwrite, retain) NSMutableDictionary *favoriteCardDict;         // 所有我的最愛名片
@property (atomic, readwrite, retain   ) NSMutableArray *favoriteCardsArray;            // 我的最愛的cardArray
@end


////////////////////////////////////////////////////////////////////////////////////////////////////
@implementation WCCacheModel





#pragma - Life cycle methods


//==============================================================================
//
//==============================================================================
- (id)init
{
    self = [super init];
    
    if (self)
    {
        self.sourceID = WC_SID_None;
        self.groupID = WC_GID_None;
        
        self.cacheModelQueue = dispatch_queue_create("com.penpower.cachemodel", NULL);
        
        NSUInteger totalMaxCacheSize = WCCM_MaxCacheSize_Location + WCCM_MaxCacheSize_ThumbImage + WCCM_MaxCacheSize_IDPhotoImage;
        self.cache = [[[NSCache alloc] init] autorelease];
        [self.cache setTotalCostLimit:totalMaxCacheSize];
        
        _favoriteCardsArray = [[NSMutableArray alloc] init];
    }
    
    return self;
}


//==============================================================================
//
//==============================================================================
- (void)dealloc
{
    [self clearCachedData];
    [self clearFavoriteCachedData];
    
    self.allCardSectionDict = nil;
    self.cache = nil;
    
    // !!release queue
    dispatch_release(self.cacheModelQueue);
    self.cacheModelQueue = nil;
    
    //////////////////////////////////////////////////
    [super dealloc];
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Setter / Getter


//==============================================================================
//
//==============================================================================
- (NSArray *)allCards
{
    __block NSArray *array = nil;
    dispatch_sync(self.cacheModelQueue, ^{
        array = [self.allCardDict allValues];
    });
    return array;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Thumb image cache methods


//==============================================================================
//
//==============================================================================
- (CPImage *)thumbImageWithCardID:(NSString *)cardID
{
    __block CPImage *image = nil;
    
    // !! 傳進來的cardID可能被release, 所以這邊要先自已copy下來
    __block NSString *tempCardID = [cardID copy];
    dispatch_sync(self.cacheModelQueue, ^{
        image = [self.cache objectForKey:[cardID thumbImageIdentifier]];
        [tempCardID release];
        tempCardID = nil;
    });
    return image;
}


//==============================================================================
//
//==============================================================================
- (void)addThumbImage:(CPImage *)image withCardID:(NSString *)cardID
{
    if (image == nil) return;
    
    // !! 傳進來的cardID可能被release, 所以這邊要先自已copy下來
    __block NSString *tempCardID = [cardID copy];
    dispatch_sync(self.cacheModelQueue, ^{
        [self.cache setObject:image forKey:[tempCardID thumbImageIdentifier]];
        [tempCardID release];
        tempCardID = nil;
    });
}


//==============================================================================
//
//==============================================================================
- (void)removeThumbImageWithCardID:(NSString *)cardID
{
    // !! 傳進來的cardID可能被release, 所以這邊要先自已copy下來
    __block NSString *tempCardID = [cardID copy];
    dispatch_sync(self.cacheModelQueue, ^{
        if ([cardID length])
        {
            [self.cache removeObjectForKey:[cardID thumbImageIdentifier]];
            [tempCardID release];
            tempCardID = nil;
        }
    });
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - IDPhoto image cache methods


//==============================================================================
//
//==============================================================================
- (CPImage *)idPhotoWithCardID:(NSString *)cardID
{
    __block CPImage *image = nil;
    
    // !! 傳進來的cardID可能被release, 所以這邊要先自已copy下來
    __block NSString *tempCardID = [cardID copy];
    dispatch_sync(self.cacheModelQueue, ^{
        image = [self.cache objectForKey:[cardID idPhotoImageIdentifier]];
        [tempCardID release];
        tempCardID = nil;
    });
    return image;
}


//==============================================================================
//
//==============================================================================
- (void)addIDPhoto:(CPImage *)image withCardID:(NSString *)cardID;
{
    if (image == nil) return;
    
    // !! 傳進來的cardID可能被release, 所以這邊要先自已copy下來
    __block NSString *tempCardID = [cardID copy];
    dispatch_sync(self.cacheModelQueue, ^{
        [self.cache setObject:image forKey:[tempCardID idPhotoImageIdentifier]];
        [tempCardID release];
        tempCardID = nil;
    });
}


//==============================================================================
//
//==============================================================================
- (void)removeIDPhotoWithCardID:(NSString *)cardID;
{
    // !! 傳進來的cardID可能被release, 所以這邊要先自已copy下來
    __block NSString *tempCardID = [cardID copy];
    dispatch_sync(self.cacheModelQueue, ^{
        if ([cardID length])
        {
            [self.cache removeObjectForKey:[cardID idPhotoImageIdentifier]];
            [tempCardID release];
            tempCardID = nil;
        }
    });
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Location cache methods

//===============================================================================
// get location from cache
//===============================================================================
- (CLLocation *)locationWithAddress:(NSString *)address
{
    __block CLLocation *location = nil;
    dispatch_sync(self.cacheModelQueue, ^{
        location = [self.cache objectForKey:address];
    });
    return location;
}


//===============================================================================
// add location to cache
//===============================================================================
- (void)addLocation:(CLLocation *)location withAddress:(NSString *)address
{
    if (location == nil) return;
    
    dispatch_sync(self.cacheModelQueue, ^{
        [self.cache setObject:location forKey:address];
    });
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - General methods


//==============================================================================
//
//==============================================================================
- (void)clearCachedData
{
    dispatch_sync(self.cacheModelQueue, ^{
        self.sourceID = WC_SID_None;
        self.groupID  = WC_GID_None;
        
        // !! don't use self.allCardSectionArray, this is on same queue
        @synchronized (self.allCardSectionArray)
        {
            self.allCardSectionArray = nil;
        }
        
        @synchronized (self.allCardSectionDict)
        {
            self.allCardSectionDict = nil;
        }
        
        @synchronized (self.searchResultSectionArray)
        {
            self.searchResultSectionArray = nil;
        }
        
        self.allCardDict = nil;
                
        [self.cache removeAllObjects];
    });
}


//==============================================================================
// 設定所有名片的cache資料
//==============================================================================
- (void)updateFromAllCardSections:(NSMutableArray *)allCardSections
{
    // 更新資料
    dispatch_sync(self.cacheModelQueue, ^{
        
        @synchronized (self.allCardSectionArray)
        {
            // MARK: allCardSectionArray/allCardSectionDict只會有一個存在
            self.allCardSectionDict = nil;
            self.allCardSectionArray = allCardSections;
            
            if (!self.allCardSectionArray)
            {
                return;
            }
            
            //////////////////////////////////////////////////
            
            self.allCardDict = [NSMutableDictionary dictionary];
            
            for (WCCardSectionModel *cardSectionModel in self.allCardSectionArray)
            {
                for (WCCardModel *cardModel in cardSectionModel.cardArray)
                {
                    [self.allCardDict setObject:cardModel forKey:cardModel.ID];
                }
            }
        }
    });
}


//==============================================================================
// 設定所有名片的cache資料  (for WCMAC/WCTMAC)
//==============================================================================
- (void)updateFromAllCardSectionDict:(NSMutableDictionary *)allCardSectionDict
{
    // 更新資料
    dispatch_sync(self.cacheModelQueue, ^{
        
        @synchronized (self.allCardSectionDict)
        {
            // MARK: allCardSectionArray/allCardSectionDict只會有一個存在
            if(allCardSectionDict == nil)
            {
                return;
            }
            
            self.allCardSectionArray = nil;
            self.allCardSectionDict = allCardSectionDict;
            
            //////////////////////////////////////////////////
            
            self.allCardDict = [NSMutableDictionary dictionary];
            
            NSArray *allCards = [self.allCardSectionDict objectForKey:PPSIC_Title_All];
            
            for (WCCardModel *cardModel in allCards)
            {
                [self.allCardDict setObject:cardModel forKey:cardModel.ID];
            }
        }
    });
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods

//==============================================================================
// 這是內部使用，不用包到queue中，外部使用要call addCardToCache
//==============================================================================
- (void)addCardModel:(WCCardModel *)aCardModel
{
    if (!self.allCardDict)
    {
        self.allCardDict = [NSMutableDictionary dictionary];
    }
    
    //////////////////////////////////////////////////
    // 加到allCardDict
    
    // !! 注意後續的處理都是用"addCardModel"，不可使用輸入的"aCardModel"。
    WCCardModel *addCardModel = [aCardModel copyCardWithDisplayData];
    
    [self.allCardDict setObject:addCardModel forKey:addCardModel.ID];
    
    //////////////////////////////////////////////////
#if TARGET_OS_IPHONE
    
    // WCM/ WCT
    //////////////////////////////////////////////////
    // 加到allCardSectionArray
    
    @synchronized (self.allCardSectionArray)
    {
        WCCardSectionModel *cardSectionModel = [self.allCardSectionArray sectionModelWithTitle:addCardModel.sectionTitle];
        
        if (cardSectionModel==nil)
        {
            // 如果本來的section不在，產生新的 cardSectionModel
            cardSectionModel                = [[[WCCardSectionModel alloc] init] autorelease];
            cardSectionModel.title          = addCardModel.sectionTitle;
            
            cardSectionModel.cardArray = [[[NSMutableArray alloc] init] autorelease];
            
            if (!self.allCardSectionArray)
            {
                self.allCardSectionArray = [NSMutableArray array];
            }
            
            [self.allCardSectionArray addObject:cardSectionModel];
        }
        
        [cardSectionModel.cardArray addObject:addCardModel];
        cardSectionModel.needResortCard = YES;
        
    } // end of synchronize
    
#elif TARGET_OS_MAC
    
    //////////////////////////////////////////////////
    // 加到allCardSectionDict (WCMAC/WCTMAC)
    // 同WCTCardDBController裡的copyCardDisplayDataDictWithGroupID
    
    @synchronized (self.allCardSectionDict)
    {
        // !! special case : 當key為WC_SBF_None時，value就是allCardArray，沒有再分indexMode。
        NSMutableArray *allCardArray = [self.allCardSectionDict objectForKey:PPSIC_Title_All];
        
        if(allCardArray == nil)
        {
            allCardArray = [NSMutableArray array];
            [self.allCardSectionDict setObject:allCardArray forKey:PPSIC_Title_All];
        }
        
        [allCardArray addObject:addCardModel];
        
        
        //////////////////////////////////////////////////
        // WCTMAC儲存的sectionTitle結構是 ”姓名的所有模式的索引值;公司名的所有模式的索引值"
        // 要轉換成cardArrayDict儲存，結構是{SortingField:{IndexMode:{SectionTitle:CardArray}}}
        
        WC_SortedByField sortingFields[2] = {WC_SBF_Name, WC_SBF_Company};
        PPSIC_Mode sectionModes[6] = {PPSIC_M_English, PPSIC_M_Stroke, PPSIC_M_Hanpin, PPSIC_M_Zuyin, PPSIC_M_Hiragana, PPSIC_M_Hangul};
        
        for(int i=0; i<2; i++)
        {
            for(int j=0; j<6; j++)
            {
                NSMutableDictionary *sortingFieldDict = [self.allCardSectionDict objectForKey:@(sortingFields[i])];
                
                if(sortingFieldDict == nil)
                {
                    sortingFieldDict = [NSMutableDictionary dictionary];
                    [self.allCardSectionDict setObject:sortingFieldDict forKey:@(sortingFields[i])];
                }
                
                NSMutableDictionary *indexModeDict = [sortingFieldDict objectForKey:@(sectionModes[j])];
                
                if(indexModeDict == nil)
                {
                    indexModeDict = [NSMutableDictionary dictionary];
                    [sortingFieldDict setObject:indexModeDict forKey:@(sectionModes[j])];
                }
                
                NSString *sectionTitle = [addCardModel displaySectionTitleWithSortByField:sortingFields[i] indexingMode:sectionModes[j]];
                NSMutableArray *cardArray;
                
                if((cardArray = [indexModeDict objectForKey:sectionTitle]) == nil)
                {
                    cardArray = [NSMutableArray array];
                    [indexModeDict setObject:cardArray forKey:sectionTitle];
                }
                
                [cardArray addObject:addCardModel];
            }
        }
    } // end of synchronize
    
#endif
    
    
    // !! 統一在這邊release
    [addCardModel release];
}


//==============================================================================
// 這是內部使用，不用包到queue中，外部使用要call updateCardInCache
//==============================================================================
- (void)updateCardModel:(WCCardModel *)aCardModel
{
#if TARGET_OS_IPHONE
    
    // WCM
    WCCardModel *currentCardModel = [self.allCardDict objectForKey:aCardModel.ID];
    
    if ([aCardModel.sectionTitle isEqualToString:currentCardModel.sectionTitle])
    {
        // 在同一section
        [currentCardModel setGroupIDArray:aCardModel.groupIDArray isInitCard:NO];
        
        currentCardModel.displayName        = aCardModel.displayName;
        currentCardModel.displayCompany     = aCardModel.displayCompany;
        currentCardModel.displayPhone       = aCardModel.displayPhone;
        currentCardModel.displayJobTitle    = aCardModel.displayJobTitle;
        currentCardModel.displayAddress     = aCardModel.displayAddress;
        currentCardModel.displayGPS         = aCardModel.displayGPS;
        currentCardModel.tagMask            = aCardModel.tagMask;
        
        currentCardModel.createdTime        = aCardModel.createdTime;
        currentCardModel.modifiedTime       = aCardModel.modifiedTime;
        
        //////////////////////////////////////////////////
        // for WCT
        currentCardModel.creator        = aCardModel.creator;
        currentCardModel.owner          = aCardModel.owner;
        currentCardModel.editor         = aCardModel.editor;
        
        currentCardModel.sharedAccountGUIDArray = [NSArray arrayWithArray:aCardModel.sharedAccountGUIDArray];
        
        //////////////////////////////////////////////////
        
        @synchronized (self.allCardSectionArray)
        {
            // same section, only resort card array in section
            WCCardSectionModel *cardSectionModel =
            [self.allCardSectionArray sectionModelWithTitle:currentCardModel.sectionTitle];
            
            if (cardSectionModel)
            {
                cardSectionModel.needResortCard = YES;
            }
        } // end of synchronize
    }
    else
    {
        // 在不同section
        [self removeCardModelWithCardID:currentCardModel.ID];
        [self addCardModel:aCardModel];
    }
    
#elif TARGET_OS_MAC
    
    [aCardModel retain];
    [self removeCardModelWithCardID:aCardModel.ID];
    [self addCardModel:aCardModel];
    [aCardModel release];
    
#endif
}


//==============================================================================
// 這是內部使用，不用包到queue中，外部使用要call removeCardFromCacheWithCardID
//==============================================================================
- (void)removeCardModelWithCardID:(NSString *)cardID
{
    if (![cardID length])
    {
        return;
    }
    
    //////////////////////////////////////////////////
    
    WCCardModel *delCardModel = [self.allCardDict objectForKey:cardID];
    
    if (delCardModel==nil)
    {
        return ;
    }
#if TARGET_OS_IPHONE
    
    // WCM/WCT
    WCCardSectionModel *cardSectionModel = nil;
    
    @synchronized (self.allCardSectionArray)
    {
        if ((cardSectionModel = [self.allCardSectionArray sectionModelWithTitle:delCardModel.sectionTitle]))
        {
            if([cardSectionModel removeCardWithCardID:cardID] && ![cardSectionModel.cardArray count])
            {
                [self.allCardSectionArray removeObject:cardSectionModel];
            }
        }
    }  // end of synchronize
    
    @synchronized (self.searchResultSectionArray)
    {
        if ((cardSectionModel = [self.searchResultSectionArray sectionModelWithTitle:delCardModel.sectionTitle]))
        {
            if([cardSectionModel removeCardWithCardID:cardID] && ![cardSectionModel.cardArray count])
            {
                [self.searchResultSectionArray removeObject:cardSectionModel];
            }
        }
    } // end of synchronize
    
#elif TARGET_OS_MAC
    
    // WCMAC/WCTMAC
    @synchronized (self.allCardSectionDict)
    {
        //////////////////////////////////////////////////
        // !! special case : 當key為WC_SBF_None時，value就是allCardArray，沒有再分indexMode。
        NSMutableArray *allCardArray = [self.allCardSectionDict objectForKey:PPSIC_Title_All];
        [allCardArray removeObject:delCardModel];
        
        //////////////////////////////////////////////////
        WC_SortedByField sortingFields[2] = {WC_SBF_Name, WC_SBF_Company};
        PPSIC_Mode sectionModes[6] = {PPSIC_M_English, PPSIC_M_Stroke, PPSIC_M_Hanpin, PPSIC_M_Zuyin, PPSIC_M_Hiragana, PPSIC_M_Hangul};
        
        for(int i=0; i<2; i++)
        {
            NSMutableDictionary *sortingFieldDict = [self.allCardSectionDict objectForKey:@(sortingFields[i])];
            
            if(sortingFieldDict == nil)
            {
                continue;
            }
            
            for(int j=0; j<6; j++)
            {
                NSMutableDictionary *indexModeDict = [sortingFieldDict objectForKey:@(sectionModes[j])];
                
                if(indexModeDict == nil)
                {
                    continue;
                }
                
                NSArray *allKeys = [indexModeDict allKeys];
                
                for(int i=0; i<[allKeys count]; i++)
                {
                    NSString *key = [allKeys objectAtIndex:i];
                    NSMutableArray *cardArray = [indexModeDict objectForKey:key];
                    
                    [cardArray removeObject:delCardModel];
                    
                    // !! 沒有資料的section要移除
                    if([cardArray count] == 0)
                    {
                        [indexModeDict removeObjectForKey:key];
                    }
                }
            }
        }
    }
    
#endif
    
    //////////////////////////////////////////////////
    [self.allCardDict removeObjectForKey:cardID];
    
    // !! 其他cache中的也要移除
    [self.cache removeObjectForKey:[cardID thumbImageIdentifier]];
    [self.cache removeObjectForKey:[cardID idPhotoImageIdentifier]];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Card cache methods

//===============================================================================
// add card to cached sections
//===============================================================================
- (void)addCardToCache:(WCCardModel *)cardModel
{
    dispatch_sync(self.cacheModelQueue, ^{
        [self addCardModel:cardModel];
    });
}


//===============================================================================
// del card from cached sections
//===============================================================================
- (void)removeCardFromCacheWithCardID:(NSString *)cardID
{
    dispatch_sync(self.cacheModelQueue, ^{
        [self removeCardModelWithCardID:cardID];
    });
}


//===============================================================================
// update card in cached sections
//===============================================================================
- (void)updateCardInCache:(WCCardModel *)cardModel
{
    dispatch_sync(self.cacheModelQueue, ^{
        [self updateCardModel:cardModel];
    });
}


//===============================================================================
// get card model from cache
//===============================================================================
- (WCCardModel *)cardFromCacheWithCardID:(NSString *)cardID
{
    __block WCCardModel *cardModel = nil;
    dispatch_sync(self.cacheModelQueue, ^{
        cardModel = [self.allCardDict objectForKey:cardID];
    });
    return cardModel;
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods for favorites


//==============================================================================
// 這是內部使用，不用包到queue中，外部使用要call addFavoriteCardToCache
//==============================================================================
- (void)addFavoriteCardModelToCache:(WCCardModel *)aCardModel
{
    //////////////////////////////////////////////////
    // 加到allCardDict
    WCCardModel *addCardModel = [aCardModel copyCardWithDisplayData];
    
    if (!self.favoriteCardDict)
    {
        self.favoriteCardDict = [NSMutableDictionary dictionary];
    }
    [self.favoriteCardDict setObject:addCardModel forKey:addCardModel.ID];
    
    //////////////////////////////////////////////////
    // 加到favoriteCardsArray
    
    @synchronized (self.favoriteCardsArray)
    {
        WCCardSectionModel *cardSectionModel = [self.favoriteCardsArray sectionModelWithTitle:favoriteSectionTitle];
        
        if (cardSectionModel==nil)
        {
            // 如果本來的section不在，產生新的 cardSectionModel
            cardSectionModel                = [[WCCardSectionModel alloc] init];
            cardSectionModel.title          = favoriteSectionTitle;
            
            cardSectionModel.cardArray = [[[NSMutableArray alloc] init] autorelease];
            
            if (!self.favoriteCardsArray)
            {
                self.favoriteCardsArray = [NSMutableArray array];
            }
            
            [self.favoriteCardsArray addObject:cardSectionModel];
            [cardSectionModel release];
        }
        
        cardSectionModel.needResortCard = YES;
        
        [cardSectionModel.cardArray addObject:addCardModel];
        
        [addCardModel release];
    }
}


//==============================================================================
// 這是內部使用，不用包到queue中，外部使用要call updateFavoriteCardInCache
//==============================================================================
- (void)updateFavoriteCardModelInSectionModel:(WCCardModel *)aCardModel
{
    WCCardModel *currentCardModel = [self.favoriteCardDict objectForKey:aCardModel.ID];
    
    if (currentCardModel)
    {
        // 在同一section
        [currentCardModel setGroupIDArray:aCardModel.groupIDArray isInitCard:NO];
        
        currentCardModel.displayName        = aCardModel.displayName;
        currentCardModel.displayCompany     = aCardModel.displayCompany;
        currentCardModel.displayPhone       = aCardModel.displayPhone;
        currentCardModel.displayJobTitle    = aCardModel.displayJobTitle;
        currentCardModel.displayAddress     = aCardModel.displayAddress;
        currentCardModel.displayGPS         = aCardModel.displayGPS;
        currentCardModel.tagMask            = aCardModel.tagMask;
        
        currentCardModel.createdTime        = aCardModel.createdTime;
        currentCardModel.modifiedTime       = aCardModel.modifiedTime;
        
        //////////////////////////////////////////////////
        // for WCT
        currentCardModel.creator        = aCardModel.creator;
        currentCardModel.owner          = aCardModel.owner;
        currentCardModel.editor         = aCardModel.editor;
        
        currentCardModel.sharedAccountGUIDArray = [NSArray arrayWithArray:aCardModel.sharedAccountGUIDArray];
        
        //////////////////////////////////////////////////
        // same section, only resort card array in section
        @synchronized (self.favoriteCardsArray)
        {
            WCCardSectionModel *cardSectionModel =
            [self.favoriteCardsArray sectionModelWithTitle:aCardModel.sectionTitle];
            
            if (cardSectionModel)
            {
                cardSectionModel.needResortCard = YES;
            }
        }
    }
    else
    {
        [self removeFavoriteCardModelFromSectionModelWithCardID:currentCardModel.ID];
        [self addFavoriteCardModelToCache:aCardModel];
    }
    
}


//==============================================================================
//
//==============================================================================
- (void)moveFavoriteCardID:(NSString *)cardID toOrder:(NSInteger)toOrder
{
    if (![cardID length]) return;
    
    WCCardModel *cardModel = [[self.favoriteCardDict objectForKey:cardID] retain];
    WCCardSectionModel *cardSectionModel = nil;
    
    @synchronized (self.favoriteCardsArray)
    {
        if ((cardSectionModel = [self.favoriteCardsArray sectionModelWithTitle:favoriteSectionTitle]))
        {
            // 從原本的array中移除
            if([cardSectionModel removeCardWithCardID:cardID])
            {
                // 加到新的位置
                [cardSectionModel.cardArray insertObject:cardModel atIndex:toOrder];
            }
        }
    }
    
    [cardModel release];
}


//==============================================================================
// 這是內部使用，不用包到queue中，外部使用要call removeFavoriteCardFromCacheWithCardID
//==============================================================================
- (void)removeFavoriteCardModelFromSectionModelWithCardID:(NSString *)cardID
{
    if (![cardID length]) return;
    
    WCCardModel *delCardModel = [self.favoriteCardDict objectForKey:cardID];
    WCCardSectionModel *cardSectionModel = nil;
    
    @synchronized (self.favoriteCardsArray)
    {
        if ((cardSectionModel = [self.favoriteCardsArray sectionModelWithTitle:favoriteSectionTitle]))
        {
            if([cardSectionModel removeCardWithCardID:cardID] && ![cardSectionModel.cardArray count])
            {
                [self.favoriteCardsArray removeObject:cardSectionModel];
            }
        }
    }
    
    @synchronized (self.searchFavoriteCardsArray)
    {
        if ((cardSectionModel = [self.searchFavoriteCardsArray sectionModelWithTitle:delCardModel.sectionTitle]))
        {
            if([cardSectionModel removeCardWithCardID:cardID] && ![cardSectionModel.cardArray count])
            {
                [self.searchFavoriteCardsArray removeObject:cardSectionModel];
            }
        }
    }
    
    [self.favoriteCardDict removeObjectForKey:cardID];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark -Favorite Card cache methods

//===============================================================================
// add card to cached sections
//===============================================================================
- (void)addFavoriteCardToCache:(WCCardModel *)cardModel
{
    dispatch_sync(self.cacheModelQueue, ^{
        [self addFavoriteCardModelToCache:cardModel];
    });
}


//===============================================================================
// del card from cached sections
//===============================================================================
- (void)removeFavoriteCardFromCacheWithCardID:(NSString *)cardID
{
    dispatch_sync(self.cacheModelQueue, ^{
        [self removeFavoriteCardModelFromSectionModelWithCardID:cardID];
    });
}


//===============================================================================
// update card in cached sections
//===============================================================================
- (void)updateFavoriteCardInCache:(WCCardModel *)cardModel
{
    dispatch_sync(self.cacheModelQueue, ^{
        [self updateFavoriteCardModelInSectionModel:cardModel];
    });
}


//===============================================================================
// move card from cached sections
//===============================================================================
- (void)moveFavoriteCardToOrder:(NSInteger)toOrder withCardID:(NSString *)cardID
{
    dispatch_sync(self.cacheModelQueue, ^{
        [self moveFavoriteCardID:cardID toOrder:toOrder];
    });
}



//===============================================================================
// get card model from cache
//===============================================================================
- (WCCardModel *)favoriteCardFromCacheWithCardID:(NSString *)cardID
{
    __block WCCardModel *cardModel = nil;
    dispatch_sync(self.cacheModelQueue, ^{
        cardModel = [self.favoriteCardDict objectForKey:cardID];
    });
    return cardModel;
}


//==============================================================================
//
//==============================================================================
- (void)clearFavoriteCachedData
{
    [self.favoriteCardDict removeAllObjects];
    self.favoriteCardDict = nil;
    
    @synchronized (self.favoriteCardsArray)
    {
        [self.favoriteCardsArray removeAllObjects];
        self.favoriteCardsArray = nil;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - section array methods

//==============================================================================
//
//==============================================================================
- (NSMutableArray *)copyCardSectionArray:(NSArray *)srcCardSectionArray
{
    if (![srcCardSectionArray count]) return nil;
    
    __block NSMutableArray *dstCardSectionArray = nil;
    dispatch_sync(self.cacheModelQueue, ^{
        dstCardSectionArray = [[NSMutableArray alloc] init];
        
        for (WCCardSectionModel *secCardSectionModel in srcCardSectionArray)
        {
            WCCardSectionModel *dstCardSectionModel = [secCardSectionModel copy];
            [dstCardSectionArray addObject:dstCardSectionModel];
            [dstCardSectionModel release];
        }
    });
    return dstCardSectionArray;
}


//==============================================================================
//
//==============================================================================
- (NSMutableArray *)copyCardSectionArrayFromCache
{
    if (![self.allCardSectionArray count]) return nil;
    
    __block NSMutableArray *dstCardSectionArray = nil;
    dispatch_sync(self.cacheModelQueue, ^{
        dstCardSectionArray = [[NSMutableArray alloc] init];
        
        for (WCCardSectionModel *secCardSectionModel in self.allCardSectionArray)
        {
            WCCardSectionModel *dstCardSectionModel = [secCardSectionModel copy];
            [dstCardSectionArray addObject:dstCardSectionModel];
            [dstCardSectionModel release];
        }
    });
    return dstCardSectionArray;
}


//==============================================================================
//
//==============================================================================
- (NSMutableArray *)copyFavoriteCardSectionArrayFromCache
{
    if (![self.favoriteCardsArray count]) return nil;
    
    __block NSMutableArray *dstCardSectionArray = nil;
    dispatch_sync(self.cacheModelQueue, ^{
        dstCardSectionArray = [[NSMutableArray alloc] init];
        
        for (WCCardSectionModel *secCardSectionModel in self.favoriteCardsArray)
        {
            WCCardSectionModel *dstCardSectionModel = [secCardSectionModel copy];
            [dstCardSectionArray addObject:dstCardSectionModel];
            [dstCardSectionModel release];
        }
    });
    return dstCardSectionArray;
}


//==============================================================================
//
//==============================================================================
- (void)sectionArray:(NSMutableArray *)sectionArray sortByField:(WC_SortedByField)sortByField
{
    dispatch_sync(self.cacheModelQueue, ^{
        // sort cards
        [WCCompareTool cardSectionArray:sectionArray sortByField:sortByField];
    });
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - section dict methods (macOS)

//================================================================================
//
//================================================================================
- (NSMutableArray *)copyCards:(NSArray *)srcCards
{
    if ([srcCards count] == 0)
    {
        return nil;
    }
    
    __block NSMutableArray *dstCards = nil;
    
    dispatch_sync(self.cacheModelQueue, ^{
        
        dstCards = [[NSMutableArray alloc] init];
        
        for (WCCardModel *srcCardModel in srcCards)
        {
            WCCardModel *dstCardModel = [srcCardModel copy];
            
            if(dstCardModel != nil)
            {
                [dstCards addObject:dstCardModel];
                [dstCardModel release];
            }
        }
    });
    
    return dstCards;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class methods


//==============================================================================
//
//==============================================================================
+ (WCCacheModel *)defaultCache
{
    static WCCacheModel *cacheModel = nil;
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^{
        cacheModel = [[WCCacheModel alloc] init];
    });
    
    return cacheModel;
}


//==============================================================================
//
//==============================================================================
+ (NSMutableArray *)copyCardSectionArray:(NSArray *)srcCardSectionArray
{
    return [[WCCacheModel defaultCache] copyCardSectionArray:srcCardSectionArray];
}


//==============================================================================
//
//==============================================================================
+ (NSMutableArray *)copyCardSectionArrayFromCache
{
    return [[WCCacheModel defaultCache] copyCardSectionArrayFromCache];
}


//==============================================================================
//
//==============================================================================
+ (NSMutableArray *)copyFavoriteCardSectionArrayFromCache;
{
    return [[WCCacheModel defaultCache] copyFavoriteCardSectionArrayFromCache];
}


//==============================================================================
//
//==============================================================================
+ (void)sectionArray:(NSMutableArray *)sectionArray sortByField:(WC_SortedByField)sortByField
{
    [[WCCacheModel defaultCache] sectionArray:sectionArray sortByField:sortByField];
}


//================================================================================
//
//================================================================================
+ (NSMutableArray *)copyCards:(NSArray *)srcCards
{
    return [[WCCacheModel defaultCache] copyCards:srcCards];
}

@end
