//
//  WCCoverFlowViewController.m
//  Pods
//
//  Created by duke on 2016/5/30.
//
//


#import "WCCoverFlowViewController.h"

//define
#import "WCCoverFlowViewController+ResourceDefine.h"
#import "WCDataControllerProtocol.h"
#import "WCSettingsKey.h"

// model
#import "WCCardModel.h"

// category
#import "NSObject+PPBusyView.h"
#import "UIDevice+DeviceModel.h"

// view
#import "CardImageReflectionView.h"
#import "ReflectionView.h"

// controllers
#import "PPBarView.h"
#import "PPIndexSectionController.h"
#import "PPLogControllerExtension.h"
#import "PPSettingsController.h"
#import "PPWorldcardAddressController.h"
#import "WCToastController.h"



NSInteger const WCCoverFlowViewController_ActionButtonCount = 3;
CGFloat const WCCoverFlowViewController_ActionBarButtonWidth = 64;
CGFloat const WCCoverFlowViewController_ContentBarButtonWidth = 48;

CGFloat const WCCoverFlowViewController_SectionIndexViewHeight = 30;
CGFloat const WCCoverFlowViewController_CardDisplayNameLabelHeight = 56;
CGFloat const WCCoverFlowViewController_CategoryNameLabelHeight = WCCoverFlowViewController_CardDisplayNameLabelHeight;
CGFloat const WCCoverFlowViewController_CategoryLabelFontSize = 18;
CGFloat const WCCoverFlowViewController_NameLabelFontSize = 24;

CGFloat const WCCoverFlowViewController_ContentBarTitleMaxFontSize= 20;
CGFloat const WCCoverFlowViewController_ContentBarTitleMinFontSize = 16;

CGFloat const WCCoverFlowViewController_ActionBarButtonWidth_ForHeight320 = 48;
CGFloat const WCCoverFlowViewController_ContentBarButtonWidth_ForHeight320 = 34;

CGFloat const WCCoverFlowViewController_SectionIndexViewHeight_ForHeight320 = 20;
CGFloat const WCCoverFlowViewController_CardDisplayNameLabelHeight_ForHeight320 = 34;
CGFloat const WCCoverFlowViewController_CategoryNameLabelHeight_ForHeight320 = WCCoverFlowViewController_CardDisplayNameLabelHeight_ForHeight320;
CGFloat const WCCoverFlowViewController_CategoryLabelFontSize_ForHeight320 = 12;
CGFloat const WCCoverFlowViewController_NameLabelFontSize_ForHeight320 = 18;

CGFloat const WCCoverFlowViewController_ContentBarTitleMaxFontSize_ForHeight320 = 18;
CGFloat const WCCoverFlowViewController_ContentBarTitleMinFontSize_ForHeight320 = 14;

// 如果設為0，|carouselDidEndScrollingAnimation:|這個就不會進去，造成無法load大圖，所以改為0.005
CGFloat const WCCoverFlowViewController_ScrollDuration = 0.005;

/// 橫式名片的高度佔cover flow 高的多少
CGFloat const WCCoverFlowViewController_HorzCardHeightRatio = 0.9;

// 沒有名片的顯示大小
CGFloat const WCCoverFlowViewController_NoCardLabelFontSize = 35;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Enum





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - WCCoverFlowControler()

@interface WCCoverFlowViewController() <iCarouselDelegate, iCarouselDataSource, PPSectionIndexViewDelegate, WCCoverFlowActionBarDelegate>





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Property

@property (nonatomic, retain) CAGradientLayer *gradientLayer;
@property (nonatomic ,retain, readwrite) iCarousel *coverFlowView;

@property (nonatomic ,retain, readwrite) PPSectionIndexView *sectionIndexView;

@property (nonatomic ,retain) WCCoverFlowActionBar *toolbar;

@property (nonatomic ,retain) UILabel *noCardLabel;
@property (nonatomic ,retain) UILabel *categoryNameLabel;
@property (nonatomic ,retain) UILabel *currentCardDisplayNameLabel;

@property (nonatomic, retain) NSMutableArray *layoutConstraints;

@property (nonatomic, retain) PPIndexSectionController *cardSectionController;

@property (nonatomic ,assign) Class itemViewClass;






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - data

/// 目前的顯示模式
//@property (nonatomic, assign, readwrite) WCCoverFlowViewControllerMode mode;

@property (nonatomic, assign) NSTimeInterval lastIndexingTime;

/// cover flow did load
@property (atomic, assign) BOOL coverFlowDidLoaded;

/// background queue
@property (nonatomic, retain) NSOperationQueue *loadingQueue;

/// current load image operation
@property (atomic, retain) NSOperation *currentImageLoadOperation;

@property (atomic, retain) WCCardModel *currentCardModel;


@end

@implementation WCCoverFlowViewController






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Init & dealloc

//================================================================================
// Initial Parameter
//================================================================================
-(instancetype)initWithMode:(WCCoverFlowViewControllerMode)mode;
{
    if (self = [super init])
    {
        // !! 目前都先不放倒影，不然還是太慢，如果比較快的倒影的做法再考慮要不要開
        // !!  較舊的機器不用倒影，不然會太慢
        // iphone 4以下，ipod 5G以下，ipad 2 以下
//        PPDeviceModelID deviceModelID = [UIDevice deviceModelID];
//        BOOL oldDevice = (deviceModelID>PPDeviceModelID_iPhone && deviceModelID<=PPDeviceModelID_iPhone_4S) ||
//        (deviceModelID>PPDeviceModelID_iPod && deviceModelID<=PPDeviceModelID_iPod_5G) ||
//        (deviceModelID>PPDeviceModelID_iPad && deviceModelID<=PPDeviceModelID_iPad_2);
//        
//        // 舊機器使用UIImageView當icarousel的item view
//        self.itemViewClass = [CardImageReflectionView class];
//        if (oldDevice)
        {
            // 不使用倒影
            self.itemViewClass = [UIImageView class];
        }

        //////////////////////////////////////////////////
        self.loadingQueue = [[[NSOperationQueue alloc] init] autorelease];
        [self.loadingQueue setMaxConcurrentOperationCount:1];
        
        self.cardSectionController = [[[PPIndexSectionController alloc] init] autorelease];
        self.cardSectionController.style = PPIndexSectionControllerStyle_StrokeAndEnglish;
        self.mode = mode;
    }
    else
    {
        PPLogWarn(@"Fail to init from super!");
    }
    return self;
}


//================================================================================
// release object
//================================================================================
- (void)dealloc
{
    [self.loadingQueue cancelAllOperations];
    self.loadingQueue = nil;
    
    [self removeMainUI];
   
    self.cardSectionController = nil;
    self.categoryName = nil;
    self.currentCardModel = nil;
    self.currentImageLoadOperation = nil;
    self.noCardLabel = nil;
    self.noCardString = nil;
    self.noNameString = nil;
    
    //////////////////////////////////////////////////
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Responding to View Events

//================================================================================
// initializing views & components
//================================================================================
- (void)viewWillAppear:(BOOL)animated
{
    PPLogViewEventIn();
    
    [super viewWillAppear:animated];
    
    //////////////////////////////////////////////////
    
    PPLogViewEventOut();
}


//================================================================================
//
//================================================================================
- (void)viewDidAppear:(BOOL)animated
{
    PPLogViewEventIn();
    
    [super viewDidAppear:animated];
    
    //////////////////////////////////////////////////
    
    PPLogViewEventOut();
}


//================================================================================
//
//================================================================================
- (void)viewWillDisappear:(BOOL)animated
{
    PPLogViewEventIn();

    //////////////////////////////////////////////////
    
    [super viewWillDisappear:animated];
    
    PPLogViewEventOut();
}


//================================================================================
//
//================================================================================
- (void)viewDidDisappear:(BOOL)animated
{
    PPLogViewEventIn();

     //////////////////////////////////////////////////
    [super viewDidDisappear:animated];
    
    PPLogViewEventOut();
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - prepare ui


//==============================================================================
//
//==============================================================================
-(void)prepareMainUI
{
    do
    {
        //////////////////////////////////////////////////
        //!! 若該View的大小為CGRectZero，則該View的later會是nil
        
        if(self.view.layer!=nil)
        {
            if (self.gradientLayer==nil)
            {
                self.gradientLayer        = [CAGradientLayer layer];
                self.gradientLayer.colors = [NSArray arrayWithObjects:(id)[WCCoverFlowViewController_TopBackgroundColor CGColor],
                                             (id)[WCCoverFlowViewController_BottomBackgroundColor CGColor], nil]; // 由上到下的漸層顏色
                
                [self.view.layer insertSublayer:self.gradientLayer atIndex:0];
            }
            
            CGRect frame = self.view.bounds;
            // !! iphone X 橫式沒有status bar 所以在iphone X上不用+20
            if ([UIApplication sharedApplication].isStatusBarHidden==NO)
            {
                frame.origin.y += 20;
            }
            self.gradientLayer.frame = frame;
        }
        
        _coverFlowView = [[iCarousel alloc] init];
        
        if (self.coverFlowView == nil)
        {
            break;
        }
        
        [self.coverFlowView setTranslatesAutoresizingMaskIntoConstraints:NO];
        self.coverFlowView.delegate = self;
        self.coverFlowView.dataSource = self;
        self.coverFlowView.type = iCarouselTypeCoverFlow;
        [self.view addSubview:self.coverFlowView];
        
        //////////////////////////////////////////////////
        
        _sectionIndexView = [[PPSectionIndexView alloc] init];
        
        if (self.sectionIndexView == nil)
        {
            break;
        }
        
        self.sectionIndexView.style = PPSectionIndexViewStyleHorizontal;
        self.sectionIndexView.delegate = self;
        self.sectionIndexView.titleNormalColor = [UIColor whiteColor];
        self.sectionIndexView.titleHighlightColor = [UIColor blueColor];
        
        [self.sectionIndexView setDisplayIndexTitles:[self.cardSectionController sectionIndexTitles]
                                     fullIndexTitles:[self.cardSectionController indexs]
                                      withExtraTitle:PPSectionIndexViewExtraTitleNone];
        
        
        [self.sectionIndexView setTranslatesAutoresizingMaskIntoConstraints:NO];
        
        [self.view addSubview:self.sectionIndexView];
        
        //////////////////////////////////////////////////
        
        _currentCardDisplayNameLabel = [[UILabel alloc] init];
        
        if (self.currentCardDisplayNameLabel == nil)
        {
            break;
        }
        
        [self.currentCardDisplayNameLabel setTranslatesAutoresizingMaskIntoConstraints:NO];
        [self.currentCardDisplayNameLabel setTextColor:[UIColor whiteColor]];
        [self.currentCardDisplayNameLabel setBackgroundColor:[UIColor clearColor]];
        [self.currentCardDisplayNameLabel setTextAlignment:NSTextAlignmentCenter];
        [self.currentCardDisplayNameLabel setFont:[UIFont systemFontOfSize:WCCoverFlowViewController_NameLabelFontSize]];
        [self.currentCardDisplayNameLabel setNumberOfLines:2];
        
        [self.view addSubview:self.currentCardDisplayNameLabel];
     
        //////////////////////////////////////////////////
        _categoryNameLabel = [[UILabel alloc] init];
        
        [self.categoryNameLabel setTranslatesAutoresizingMaskIntoConstraints:NO];
        [self.categoryNameLabel setTextColor:[UIColor whiteColor]];
        [self.categoryNameLabel setBackgroundColor:[UIColor clearColor]];
        [self.categoryNameLabel setTextAlignment:NSTextAlignmentLeft];
        [self.categoryNameLabel setFont:[UIFont systemFontOfSize:WCCoverFlowViewController_CategoryLabelFontSize]];
        [self.categoryNameLabel setNumberOfLines:2];
        
        self.categoryNameLabel.text = self.categoryName;
        
        [self.view addSubview:self.categoryNameLabel];

        //////////////////////////////////////////////////
        _noCardLabel = [[UILabel alloc] init];
        
        [self.noCardLabel setTranslatesAutoresizingMaskIntoConstraints:NO];
        [self.noCardLabel setTextColor:[UIColor whiteColor]];
        [self.noCardLabel setBackgroundColor:[UIColor clearColor]];
        [self.noCardLabel setTextAlignment:NSTextAlignmentCenter];
        [self.noCardLabel setFont:[UIFont systemFontOfSize:WCCoverFlowViewController_NoCardLabelFontSize]];
        
        [self.noCardLabel setHidden:YES];
        self.noCardLabel.text = self.noCardString;
        
        [self.view addSubview:self.noCardLabel];
        
        //////////////////////////////////////////////////
        // 功能選單
        _toolbar = [[WCCoverFlowActionBar alloc] init];
        
        [self.toolbar setTranslatesAutoresizingMaskIntoConstraints:NO];
        [self.toolbar setDelegate:self];
        [self.view addSubview:self.toolbar];
        
        //////////////////////////////////////////////////
        // 調整toolbar layout
        [self.toolbar setContentBarBackgroundColor:WCCoverFlowViewController_BarBackgroundColor];
        
        UIEdgeInsets contentBarButtonEdgeInsets = WCCoverFlowContentBarButtonEdgeInsets;
        UIEdgeInsets actionBarButtonEdgeInsets = WCCoverFlowBarButtonEdgeInsets;
        CGFloat contentBarButtonWidth = WCCoverFlowViewController_ContentBarButtonWidth;
        CGFloat contentBarMaxFontSize = WCCoverFlowViewController_ContentBarTitleMaxFontSize;
        CGFloat contentBarMixFontSize = WCCoverFlowViewController_ContentBarTitleMinFontSize;
        
        CGSize screenSize = [[UIScreen mainScreen] bounds].size;
        
        // 小尺寸的layout
        CGFloat minWidth = fminf(screenSize.width, screenSize.height);
        if (minWidth<=320.0)
        {
            contentBarButtonEdgeInsets = WCCoverFlowContentBarButtonEdgeInsets_ForHeight320;
            actionBarButtonEdgeInsets = WCCoverFlowBarButtonEdgeInsets_ForHeight320;
            contentBarButtonWidth = WCCoverFlowViewController_ContentBarButtonWidth_ForHeight320;
            contentBarMaxFontSize = WCCoverFlowViewController_ContentBarTitleMaxFontSize_ForHeight320;
            contentBarMixFontSize = WCCoverFlowViewController_ContentBarTitleMinFontSize_ForHeight320;
        }

        [self.toolbar setContentBarButtonEdgeInsets:contentBarButtonEdgeInsets];
        [self.toolbar setActionBarButtonEdgeInsets:actionBarButtonEdgeInsets];
        [self.toolbar setContentBarButtonWidth:contentBarButtonWidth];
        [self.toolbar setMaxTitleFontSize:contentBarMaxFontSize];
        [self.toolbar setMinTitleFontSize:contentBarMixFontSize];

    }
    while (NO);

    //////////////////////////////////////////////////
    // debug
//    [self.sectionIndexView setBackgroundColor:[[UIColor redColor] colorWithAlphaComponent:0.5]];
//    [self.coverFlowView setBackgroundColor:[[UIColor blueColor] colorWithAlphaComponent:0.5]];
//    [self.noCardLabel setBackgroundColor:[UIColor magentaColor]];
//    [self.currentCardDisplayNameLabel setBackgroundColor:[[UIColor greenColor] colorWithAlphaComponent:0.5]];
//    [self.categoryNameLabel setBackgroundColor:[[UIColor purpleColor] colorWithAlphaComponent:0.5]];
    //////////////////////////////////////////////////
    // layout
    [self setupLayoutConstraints];
}



//==============================================================================
//
//==============================================================================
- (void)removeMainUI
{
    [self.gradientLayer removeFromSuperlayer];
    self.gradientLayer = nil;
    
    self.toolbar.delegate = nil;
    [self.toolbar removeFromSuperview];
    self.toolbar = nil;
    
    [self.noCardLabel removeFromSuperview];
    self.noCardLabel = nil;
    
    [self.categoryNameLabel removeFromSuperview];
    self.categoryNameLabel = nil;
    
    [self.currentCardDisplayNameLabel removeFromSuperview];
    self.currentCardDisplayNameLabel = nil;
    
    [self.sectionIndexView removeFromSuperview];
    self.sectionIndexView = nil;
    
    [self.coverFlowView removeFromSuperview];
    self.coverFlowView.dataSource = nil;
    self.coverFlowView.delegate = nil;
    self.coverFlowView = nil;
 
    //////////////////////////////////////////////////
    [self removeLayoutConstraints];
}


//==============================================================================
//
//==============================================================================
- (void)updateSectionIndexView
{
    WC_SortedByField sortedByField = (WC_SortedByField)[PPSettingsController integerValueWithKey:WCSC_IV_kSortingByField];
    
    if (sortedByField==WC_SBF_CreateTime||
        sortedByField==WC_SBF_ModifiedTime)
    {
        [self setSectionIndexStyle:PPSIC_M_None];
    }
    else
    {
        [self setSectionIndexStyle:[PPSettingsController integerValueWithKey:WCSC_IV_kSectionIndexMode]];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - layout constaints

//================================================================================
//
//================================================================================
- (void)setupLayoutConstraints
{
    [self removeLayoutConstraints];
    //////////////////////////////////////////////////
    // Setup Layout Constraints
    CGFloat statusBarHeight= [UIApplication sharedApplication].isStatusBarHidden?0.0:20.0;
    
    CGFloat sectionIndexHeight = WCCoverFlowViewController_SectionIndexViewHeight;
    CGFloat nameLabelHeight = WCCoverFlowViewController_CardDisplayNameLabelHeight;
    CGFloat categoryLabelHeight = WCCoverFlowViewController_CategoryNameLabelHeight;
    CGFloat nameLabelFontSize = WCCoverFlowViewController_NameLabelFontSize;
    CGFloat toolbarButtonWidth = WCCoverFlowViewController_ActionBarButtonWidth;
    //////////////////////////////////////////////////
    // 判斷layout 樣式
    CGSize screenSize = [[UIScreen mainScreen] bounds].size;

    // 小尺寸的layout
    CGFloat minWidth = fminf(screenSize.width, screenSize.height);
    if (minWidth<=320.0)
    {
        sectionIndexHeight = WCCoverFlowViewController_SectionIndexViewHeight_ForHeight320;
        nameLabelHeight = WCCoverFlowViewController_CardDisplayNameLabelHeight_ForHeight320;
        categoryLabelHeight = WCCoverFlowViewController_CategoryNameLabelHeight_ForHeight320;
        
        nameLabelFontSize = WCCoverFlowViewController_NameLabelFontSize_ForHeight320;
        toolbarButtonWidth = WCCoverFlowViewController_ActionBarButtonWidth_ForHeight320;
    }

    if (self.mode==WCCoverFlowViewControllerMode_Favorite)
    {
        sectionIndexHeight = 0;
    }
    
    [self.currentCardDisplayNameLabel setFont:[UIFont systemFontOfSize:nameLabelFontSize]];

    CGFloat toolbarHeight = nameLabelHeight;
    CGFloat toolbarWidth = toolbarButtonWidth * WCCoverFlowViewController_ActionButtonCount;
    CGFloat categoryWitdh = toolbarWidth - 10;
    
    // MARK:!! ios7的topLayoutGuide在時旋轉有問題，可能會拿到0，所以改用hard code的高度
    // ios8之後就正常了，等之後支援ios8以上就可以改掉
    NSDictionary *views   = @{@"topGuide":[self topLayoutGuide],
                              @"CoverFlow":self.coverFlowView,
                              @"sectionIndex":self.sectionIndexView,
                              @"nameLabel":self.currentCardDisplayNameLabel,
                              @"categoryNameLabel":self.categoryNameLabel,
                              @"noCardLabel":self.noCardLabel,
                              @"toolbar":self.toolbar};
    
    
    NSDictionary *metrics = @{@"sectionIndexHeight":@(sectionIndexHeight),
                              @"nameLabelHeight":@(nameLabelHeight),
                              @"categoryLabelHeight":@(categoryLabelHeight),
                              @"statusBarHeight":@(statusBarHeight),
                              @"toolbarHeight":@(toolbarHeight),
                              @"toolbarWidth":@(toolbarWidth),
                              @"categoryWitdh":@(categoryWitdh)};
    
    NSMutableArray *layoutConstraints = [NSMutableArray array];

    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|-(statusBarHeight)-[sectionIndex(sectionIndexHeight)][CoverFlow][nameLabel(nameLabelHeight)]|"
                                                                                   options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                   metrics:metrics
                                                                                     views:views]];
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|-(statusBarHeight)-[sectionIndex(sectionIndexHeight)][CoverFlow][categoryNameLabel(categoryLabelHeight)]|"
                                                                                   options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                   metrics:metrics
                                                                                     views:views]];
    
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[CoverFlow]|"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:nil
                                                                                     views:views]];
    
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[sectionIndex]|"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:nil
                                                                                     views:views]];

    // no card label constraints
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|-(statusBarHeight)-[sectionIndex(sectionIndexHeight)][noCardLabel][nameLabel(nameLabelHeight)]|"
                                                                                   options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                   metrics:metrics
                                                                                     views:views]];
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[noCardLabel]|"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:nil
                                                                                     views:views]];

    // toolbar constraints
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|-(statusBarHeight)-[sectionIndex(sectionIndexHeight)][CoverFlow][toolbar(categoryLabelHeight)]|"
                                                                                   options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                   metrics:metrics
                                                                                     views:views]];
    
    
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|-(5)-[categoryNameLabel(<=categoryWitdh@751)]-(5)-[nameLabel(>=50@750)][toolbar(==toolbarWidth)]|"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:metrics
                                                                                     views:views]];
    
    if ([layoutConstraints count] > 0)
    {
        [self.view addConstraints:layoutConstraints];
        self.layoutConstraints = layoutConstraints;
        
        [self.toolbar layoutIfNeeded];
    }
    
}


//==============================================================================
//
//==============================================================================
- (void)removeLayoutConstraints
{
    if (self.layoutConstraints != nil)
    {
        [self.view removeConstraints:self.layoutConstraints];
        self.layoutConstraints = nil;
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - prepare data


//==============================================================================
//
//==============================================================================
- (void)reloadData
{
    [self reloadDataWithBusy:YES];
}


//==============================================================================
//
//==============================================================================
- (void)reloadDataWithBusy:(BOOL)showBusy
{
    if ([self.dataSource respondsToSelector:@selector(cardSectionArrayWithCoverFlowViewController:)])
    {
        __block typeof(self) blockSelf = self;
        if(showBusy)
        {
            [self setBusy:@(YES) withView:self.view];
        }
        
        __block NSBlockOperation *blockOperation = [NSBlockOperation blockOperationWithBlock:^{
            
            blockSelf.coverFlowDidLoaded = NO;
            NSArray * cardSectionArray = [blockSelf.dataSource cardSectionArrayWithCoverFlowViewController:blockSelf];
            if(blockOperation.isCancelled==YES)
            {
                if(showBusy)
                {
                    dispatch_async(dispatch_get_main_queue(), ^{
                        [blockSelf setBusy:@(NO) withView:blockSelf.view];
                    });
                }
                return ;
            }
            
            //////////////////////////////////////////////////
            dispatch_sync(dispatch_get_main_queue(), ^{
                /// 重設currentItemIndex
                if(blockSelf.coverFlowView.currentItemIndex==-1)
                {
                    blockSelf.coverFlowView.currentItemIndex = 0;
                }
                
                // 取得目前的cardID, 可能會被替換掉，所以要先retain
                NSString *currentCardID = [[self cardIDFromItemIndex:self.currentItemIndex] retain];
                
                // 替換sectionModel
                [blockSelf.cardSectionController replaceSectionModels:cardSectionArray forSearching:NO];
                
                //////////////////////////////////////////////////
                // !!這裡不能被cancel不然下次的index會跳到別張名片上
                // 先計算原本的cardID, 在reload後的index,
                NSInteger itemIndexAfterReload = [self itemIndexFromCardID:currentCardID];
                
                [currentCardID release];
                
                // 找不到的話表示被刪掉了，所以往前一個
                if(itemIndexAfterReload==-1)
                {
                    itemIndexAfterReload = blockSelf.currentItemIndex-1;
                    
                    // 減完如果是-1，表示前面沒有了停在
                    if(itemIndexAfterReload<0)
                    {
                        itemIndexAfterReload = blockSelf.currentItemIndex;
                    }
                }
                
                blockSelf.currentItemIndex = itemIndexAfterReload;

                //////////////////////////////////////////////////
                if(blockOperation.isCancelled==YES)
                {
                    if(showBusy)
                    {
                        dispatch_async(dispatch_get_main_queue(), ^{
                            [blockSelf setBusy:@(NO) withView:blockSelf.view];
                        });
                    }
                    return ;
                }
                
                [blockSelf.coverFlowView reloadData];
                if(blockOperation.isCancelled==YES)
                {
                    if(showBusy)
                    {
                        dispatch_async(dispatch_get_main_queue(), ^{
                            [blockSelf setBusy:@(NO) withView:blockSelf.view];
                        });
                    }
                    return ;
                }
                
                [blockSelf.coverFlowView setCurrentItemIndex:blockSelf.currentItemIndex];
                
                blockSelf.coverFlowDidLoaded = YES;

                dispatch_async(dispatch_get_main_queue(), ^{
                    [self refreshCurrentCardDisplayName];
                    
                    if(showBusy)
                    {
                        [blockSelf setBusy:@(NO) withView:blockSelf.view];
                    }
                });
            });
        }];
        
        // 重load時取消原本的動作
        [self.loadingQueue cancelAllOperations];
        
        // 加入目前動作
        [self.loadingQueue addOperation:blockOperation];
        
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Property


//==============================================================================
//
//==============================================================================
- (void)setCategoryName:(NSString *)categoryName
{
    [categoryName retain];
    [_categoryName release];
    _categoryName = categoryName;
    
    //////////////////////////////////////////////////
    
    if(self.categoryNameLabel)
    {
        self.categoryNameLabel.text = self.categoryName;
    }
}


//==============================================================================
//
//==============================================================================
- (void)setNoCardString:(NSString *)noCardString
{
    [noCardString retain];
    [_noCardString release];
    _noCardString = noCardString;
    
    //////////////////////////////////////////////////
    
    if(self.noCardLabel)
    {
        self.noCardLabel.text = self.noCardString;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - private


//================================================================================
//
//================================================================================
- (void)refreshCurrentCardDisplayName
{
    PPRowModel *rowModel =  [self.cardSectionController rowModelAtIndex:self.coverFlowView.currentItemIndex forSearching:NO];
    if (rowModel)
    {
        WCCardModel *cardModel = (WCCardModel *)rowModel.object;
        [self.currentCardDisplayNameLabel setText:[cardModel.displayName length] ? cardModel.displayName : self.noNameString];
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - (Private) section title methods



//===============================================================================
// return -1 if not found
//===============================================================================
- (NSInteger)sectionIndexByTitle:(NSString *)title
{
    NSArray *sectionModelsForDefault = [self.cardSectionController sectionModelsForDefault];
    
    PPSectionModel *sectionModel = nil;
    NSInteger sectionCount = [sectionModelsForDefault count];
    NSInteger	i, j, k;
    
    
    //---------------------------------------------------
    // special conditions
    //---------------------------------------------------
    // direct return if only one section
    if(sectionCount == 1)
        return 0;
    
    // '#' is always put as last section
    if([title isEqualToString:@"#"]) // "#"
        return sectionCount - 1;
    else if([title isEqualToString:@"•"])	// "•"
        return -1;
    
    
    //---------------------------------------------------
    // find section with index title
    //---------------------------------------------------
    // get current index title order
    NSArray *allIndexArray = [self.cardSectionController indexs];
    
    for(i=0; i<[allIndexArray count]; i++)
    {
        if([title isEqualToString:[allIndexArray objectAtIndex:i]])
            break;
    }
    
    if(i >= [allIndexArray count])
        i = [allIndexArray count]-1;
    
    // reverse search from current order
    for(j=i; j>=0; j--)
    {
        // check if index title in section index
        for(k=0; k<sectionCount; k++)
        {
            sectionModel = [sectionModelsForDefault objectAtIndex:k];
            
            // input letter is in section title
            if([sectionModel.title isEqualToString:[allIndexArray objectAtIndex:j]])
                return k;
        }
    }
    
    
    // return -1 if not found
    return -1;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Notification receiver

//===============================================================================
//
//===============================================================================
- (void)recvLoadCardSectionsResult:(NSNotification *)notification
{
    PPLogFunctionIn(@"%@", notification);
    BOOL result = [[notification.userInfo objectForKey:WCDC_NOTIFY_UserInfo_kResult] boolValue];
    
    if(result)
    {
        [self reloadData];
    }
    
    PPLogFunctionOut();
}


//===============================================================================
//
//===============================================================================
- (void)recvDisplayDataChanged:(NSNotification *)notification
{
    PPLogFunctionIn(@"%@", notification);
    
    [self updateSectionIndexView];
    [self reloadDataWithBusy:NO];
    
    PPLogFunctionOut();
}


//===============================================================================
//
//===============================================================================
- (void)recvLoadFavoritesResult:(NSNotification *)notification
{
    PPLogFunctionIn(@"%@", notification);
    BOOL result = [[notification.userInfo objectForKey:WCDC_NOTIFY_UserInfo_kResult] boolValue];
    
    if(result)
    {
        [self reloadData];
    }
    
    PPLogFunctionOut();
}


//===============================================================================
//
//===============================================================================
- (void)recvFavoriteChanged:(NSNotification *)notification
{
    PPLogFunctionIn(@"%@", notification);
    
    [self updateSectionIndexView];
    [self reloadDataWithBusy:NO];
    
    PPLogFunctionOut();
}




////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Action

//================================================================================
//
//================================================================================
//- (<#return#>)<#method description#>:(<#model name#> *)<#param name#>
//{
//}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Style

//================================================================================
//
//================================================================================
- (void)setSectionIndexStyle:(PPSIC_Mode)sectionIndexStyle
{
    PPIndexSectionControllerStyle style = PPIndexSectionControllerStyle_None;
    
    switch (sectionIndexStyle)
    {
        case PPSIC_M_Stroke:
        {
            style = PPIndexSectionControllerStyle_StrokeAndEnglish;
            break;
        }
        case PPSIC_M_Hanpin:
        {
            style = PPIndexSectionControllerStyle_Hanpin;
            break;
        }
        case PPSIC_M_Zuyin:
        {
            style = PPIndexSectionControllerStyle_Zhuyin;
            break;
        }
        case PPSIC_M_Hiragana:
        {
            style = PPIndexSectionControllerStyle_HiraganaAndEnglish;
            break;
        }
        case PPSIC_M_Hangul:
        {
            style = PPIndexSectionControllerStyle_HangulAndEnglish;
            break;
        }
        case PPSIC_M_Thai:
        {
            style = PPIndexSectionControllerStyle_ThaiAndEnglish;
            break;
        }
        case PPSIC_M_Swedish:
        {
            style = PPIndexSectionControllerStyle_Swedish;
            break;
        }
        case PPSIC_M_English:
        case PPSIC_M_None:
        {
            style = PPIndexSectionControllerStyle_English;
            break;
        }

    }
    self.cardSectionController.style = style;
    
    [self.sectionIndexView setDisplayIndexTitles:[self.cardSectionController sectionIndexTitles]
                                 fullIndexTitles:[self.cardSectionController indexs]
                                  withExtraTitle:PPSectionIndexViewExtraTitleNone];
    
    [self.sectionIndexView setHidden:(sectionIndexStyle==PPSIC_M_None)];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - child view controller methods


//==============================================================================
//
//==============================================================================
- (void)addOnViewController:(UIViewController *)viewController
                   withMode:(WCCoverFlowViewControllerMode)mode
{
    PPLogFuncIn();
    if(viewController==nil)
    {
        NSAssert(viewController==nil, @"viewController不應該是空的");
        return ;
    }
    
    //////////////////////////////////////////////////
    // set mode
    self.mode = mode;
    
    //////////////////////////////////////////////////
    // style setting
    
    //////////////////////////////////////////////////
    // adjust horzViewController frame

    CGRect viewFrame = self.view.frame;
    if (UIInterfaceOrientationIsLandscape(self.interfaceOrientation))
    {
        viewFrame = CGRectMake(0, 0, MAX(viewFrame.size.width, viewFrame.size.height), MIN(viewFrame.size.width, viewFrame.size.height));
    }
    else
    {
        viewFrame = CGRectMake(0, 0, MIN(viewFrame.size.width, viewFrame.size.height), MAX(viewFrame.size.width, viewFrame.size.height));
    }
    
    self.view.frame = viewFrame;
    self.view.center = CGPointMake(self.view.frame.size.width/2, self.view.frame.size.height/2);

    //////////////////////////////////////////////////
    // 動畫
    [self coverFlowViewControllerWillAppear];
    
    self.view.alpha = 0.0;
    
    [viewController addChildViewController:self];
    [self didMoveToParentViewController:viewController];
    
    // add subview
    [viewController.view addSubview:self.view];
    
    //////////////////////////////////////////////////
    // animation for display horzViewController
    
    [UIView animateWithDuration:0.3
                          delay:0.0
                        options:UIViewAnimationOptionTransitionCrossDissolve
                     animations:^{
                         
                         self.view.alpha = 1.0;
                         
                     } completion:^(BOOL finished) {
                     }];
    PPLogFuncOut();
}


//==============================================================================
//
//==============================================================================
- (void)removeFromParent
{
    if(self.parentViewController==nil)
    {
        return ;
    }
    
    self.view.alpha = 1.0;
    
    [UIView animateWithDuration:0.3
                          delay:0.0
                        options:UIViewAnimationOptionTransitionCrossDissolve
                     animations:^{
                         self.view.alpha = 0.0;
                     } completion:^(BOOL finished) {
                         
                         [self willMoveToParentViewController:nil];
                         [self.view removeFromSuperview];
                         [self removeFromParentViewController];
                         
                         [self coverFlowViewControllerDidDisappear];
                     }];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - view life cycle


//==============================================================================
//
//==============================================================================
- (void)coverFlowViewControllerWillAppear
{
    if (self.coverFlowDidLoaded)
    {
        return;
    }

    PPLogFuncIn();
    
    [self prepareMainUI];
    //////////////////////////////////////////////////
    // notifications
    NSNotificationCenter *nc = [NSNotificationCenter defaultCenter];
    
    if (self.mode==WCCoverFlowViewControllerMode_CardHolder)
    {
        // cardholder通知
        [nc addObserver:self selector:@selector(recvLoadCardSectionsResult:) name:WCDC_NOTIFY_LoadCardSectionsResult object:nil];
        [nc addObserver:self selector:@selector(recvDisplayDataChanged:) name:WCDC_NOTIFY_DisplayDataChanged object:nil];
    }
    else
    {
        // 我的最愛通知
        [nc addObserver:self selector:@selector(recvLoadFavoritesResult:) name:WCDC_NOTIFY_LoadFavoritesResult object:nil];
        [nc addObserver:self selector:@selector(recvFavoriteChanged:) name:WCDC_NOTIFY_FavoriteChanged object:nil];
    }
    
    
    
    // reload data
    [self reloadData];
    
    self.coverFlowDidLoaded = YES;
    
    PPLogFuncOut();
}


//==============================================================================
//
//==============================================================================
- (void)coverFlowViewControllerDidDisappear
{
    PPLogFuncIn();
    
    [self.loadingQueue cancelAllOperations];
    
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    
    [self removeMainUI];
    
    self.coverFlowDidLoaded = NO;

    PPLogFuncOut();
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance Methods


//==============================================================================
//
//==============================================================================
- (NSString *)cardIDFromItemIndex:(NSInteger)itemIndex
{
    if(itemIndex==-1)
    {
        return nil;
    }
    
    PPRowModel *rowModel = [self.cardSectionController rowModelAtIndex:itemIndex forSearching:NO];
    return [(WCCardModel *)[rowModel object] ID];
}


//==============================================================================
//
//==============================================================================
- (NSInteger)itemIndexFromCardID:(NSString *)cardID
{
    PPLogFuncIn(@"-%@", cardID);
    
    NSInteger itemIndex = -1;
    
    if ([cardID length]==0)
    {
        return itemIndex;
    }
    
    //////////////////////////////////////////////////
    BOOL match = NO;
    
    for (PPSectionModel *sectionModel in self.cardSectionController.sectionModelsForDefault)
    {
        for (PPRowModel *rowModel in sectionModel.rowModels)
        {
            itemIndex++;
            WCCardModel *cardModel = rowModel.object;
            PPLogFuncInfo(@"-%@:%@", cardModel.ID, cardID);
            if ([cardModel.ID isEqualToString:cardID])
            {
                return itemIndex;
            }
        }
    }
    
    return match?itemIndex:-1;
}


//==============================================================================
//
//==============================================================================
- (NSIndexPath *)indexPathForCurrentItemIndex
{
    PPRowModel *rowModel = [self.cardSectionController rowModelAtIndex:self.currentItemIndex forSearching:NO];
    return [self.cardSectionController indexPathForRowModel:rowModel forSearching:NO];
}


//==============================================================================
// 如果不符合就回傳-1
//==============================================================================
- (NSInteger)itemIndexFromIndexPath:(NSIndexPath *)indexPath
{
    NSInteger currentSection = 0;
    NSInteger itemIndex = -1;

    for (PPSectionModel *sectionModel in self.cardSectionController.sectionModelsForDefault)
    {
        if(itemIndex==-1)
        {
            itemIndex = 0;
        }
        
        if (currentSection == indexPath.section)
        {
            itemIndex += indexPath.row;
            break;
        }
        
        itemIndex += [sectionModel.rowModels count];
        currentSection++;
    }
    
    return itemIndex;
}


//==============================================================================
//
//==============================================================================
- (NSArray *)allCardIDArray
{
    NSMutableArray *tempResultArray = nil;
    NSMutableArray *sectionModels = [self.cardSectionController sectionModelsForSearching:NO];
    
    for (PPSectionModel *sectionModel in sectionModels)
    {
        for (PPRowModel *rowModel in sectionModel.rowModels)
        {
            WCCardModel *cardModel = [rowModel object];

            if(tempResultArray==nil)
            {
                tempResultArray = [NSMutableArray array];
            }
            
            [tempResultArray addObject:cardModel.ID];
        }
    }
    
    if(tempResultArray)
    {
        return [NSArray arrayWithArray:tempResultArray];
    }
    
    return nil;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - load image methods

//===============================================================================
//
//===============================================================================
- (void)startLoadCardImageWithCardID:(NSString *)cardID
{
    [self stopCurrentImageLoadOperation];
    
    self.currentImageLoadOperation = [[[NSInvocationOperation alloc] initWithTarget:self
                                                                           selector:@selector(loadAndShowCardImageWithCardID:)
                                                                             object:cardID] autorelease];
    
    [self.loadingQueue addOperation:self.currentImageLoadOperation];
}


//===============================================================================
//
//===============================================================================
- (void)stopCurrentImageLoadOperation
{
    [self.currentImageLoadOperation cancel];
    self.currentImageLoadOperation = nil;
}


//===============================================================================
//
//===============================================================================
- (void)loadAndShowCardImageWithCardID:(NSString *)cardID
{
    PPLogFuncIn(@" %@", cardID);
    
    do {

        NSOperation *curOperation = nil;
        UIImage *browseImage = nil;

        UIView *currentView = self.coverFlowView.currentItemView;
        if (!currentView)
        {
            break;
        }
        
        // check if canceled
        if([curOperation isCancelled])
        {
            break;
        }
        
        //////////////////////////////////////////////////
        // get current operation
        curOperation = [[[self.loadingQueue operations] firstObject] retain];
        
        //////////////////////////////////////////////////
        // check if canceled
        if([curOperation isCancelled])
        {
            [curOperation release];
            break;
        }
        
        //////////////////////////////////////////////////
        // load image
        if ([self.dataSource respondsToSelector:@selector(coverFlowViewController:cardImageWithCardID:)])
        {
            browseImage = [self.dataSource coverFlowViewController:self cardImageWithCardID:cardID];
        }
        
        if(!browseImage)
        {
            [curOperation release];
            break;
        }
        
        //////////////////////////////////////////////////
        // show image
        if ([currentView respondsToSelector:@selector(setImage:)])
        {
            dispatch_sync(dispatch_get_main_queue(), ^{
                [currentView performSelector:@selector(setImage:) withObject:browseImage];
            });
        }
        
        [curOperation release];

   } while (NO);
    PPLogFuncOut();
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - iCarousel DataSource

//================================================================================
//
//================================================================================
- (NSInteger)numberOfItemsInCarousel:(__unused iCarousel *)carousel
{
    NSInteger numberOfItem = [self.cardSectionController rowObjectsCountForSearching:NO];
    [self.noCardLabel setHidden:(numberOfItem!=0)];
    if(numberOfItem==0)
    {
        self.currentCardModel = nil;
        [self.currentCardDisplayNameLabel setText:nil];
        [self.toolbar hideContentBar];
    }
    return numberOfItem;
}


//================================================================================
//
//================================================================================
- (UIView *)carousel:(__unused iCarousel *)carousel viewForItemAtIndex:(NSInteger)index reusingView:(UIView *)view
{
    PPLogDelegateIn();
    PPRowModel *rowModel =  [self.cardSectionController rowModelAtIndex:index forSearching:NO];
    if (rowModel)
    {
        WCCardModel *cardModel = (WCCardModel *)rowModel.object;
        UIImage *cardImage = nil;
        
        if ([self.dataSource respondsToSelector:@selector(coverFlowViewController:thumbnailCardImageWithCardID:)])
        {
            cardImage = [self.dataSource coverFlowViewController:self thumbnailCardImageWithCardID:cardModel.ID];
        }
        
        //////////////////////////////////////////////////
        //create new view if no view is available for recycling
        if (view == nil)
        {
            view = [[[self.itemViewClass alloc] initWithFrame:CGRectZero] autorelease];
        }
        
        // 名片圖的大小計算
        CGFloat cardImageWidth = cardImage.size.width;
        CGFloat cardImageHeight = cardImage.size.height;
        CGFloat maxHeight = 1.0;
        
        if (cardImageWidth==0 ||
            cardImageHeight==0)
        {
            return view;
        }
        
        if (cardImageWidth<cardImageHeight)
        {
            maxHeight = carousel.contentView.bounds.size.height;
        }
        else
        {
            if(cardImageWidth/cardImageHeight>carousel.contentView.bounds.size.width/carousel.contentView.bounds.size.height)
            {
                maxHeight = carousel.contentView.bounds.size.width*WCCoverFlowViewController_HorzCardHeightRatio*cardImageHeight/cardImageWidth;
            }
            else
            {
                maxHeight = carousel.contentView.bounds.size.height*WCCoverFlowViewController_HorzCardHeightRatio;
            }
        }
        
        //////////////////////////////////////////////////
        // 重設view 的屬性
        // common property
        view.clipsToBounds = NO;
        
        if ([view respondsToSelector:@selector(setImage:)])
        {
            // 使用performSelector避免warning
            [view performSelector:@selector(setImage:) withObject:nil];
            view.frame = CGRectMake(0.0,
                                    0.0,
                                    cardImageWidth * maxHeight/cardImageHeight,
                                    cardImageHeight * maxHeight/cardImageHeight);
            [view performSelector:@selector(setImage:) withObject:cardImage];
        }
    }
    PPLogDelegateIn();
    return view;
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - iCarouselDelegate


//==============================================================================
//
//==============================================================================
- (CGFloat)carouselItemWidth:(iCarousel *)carousel
{
    return carousel.contentView.bounds.size.width*0.7;
}


//================================================================================
//
//================================================================================
- (CATransform3D)carousel:(__unused iCarousel *)carousel itemTransformForOffset:(CGFloat)offset baseTransform:(CATransform3D)transform
{
    //implement 'flip3D' style carousel
    transform = CATransform3DRotate(transform, M_PI / 8.0f, 0.0f, 1.0f, 0.0f);
    return CATransform3DTranslate(transform, 0.0f, 0.0f, offset * self.coverFlowView.itemWidth);
}


//================================================================================
//
//================================================================================
- (CGFloat)carousel:(__unused iCarousel *)carousel valueForOption:(iCarouselOption)option withDefault:(CGFloat)value
{
    //customize carousel display
    switch (option)
    {
        case iCarouselOptionWrap:
        {
            return NO;
        }
        case iCarouselOptionSpacing:
        {
            return 0.5;
        }
        case iCarouselOptionFadeMax:
        case iCarouselOptionShowBackfaces:
        case iCarouselOptionRadius:
        case iCarouselOptionAngle:
        case iCarouselOptionArc:
        case iCarouselOptionTilt:
        case iCarouselOptionCount:
        case iCarouselOptionFadeMin:
        case iCarouselOptionFadeMinAlpha:
        case iCarouselOptionFadeRange:
        case iCarouselOptionOffsetMultiplier:
        case iCarouselOptionVisibleItems:
        {
            return value;
        }
    }
}

//================================================================================
//
//================================================================================
- (void)carousel:(__unused iCarousel *)carousel didSelectItemAtIndex:(NSInteger)index
{
    PPRowModel *rowModel =  [self.cardSectionController rowModelAtIndex:index forSearching:NO];
    if (rowModel)
    {
        WCCardModel *cardModel = (WCCardModel *)rowModel.object;
        if ([self.delegate respondsToSelector:@selector(coverFlowViewController:didSelectCardWithCardID:)])
        {
            [self.delegate coverFlowViewController:self didSelectCardWithCardID:cardModel.ID];
        }
    }
}


//================================================================================
//
//================================================================================
- (void)carouselCurrentItemIndexDidChange:(__unused iCarousel *)carousel
{
    [self refreshCurrentCardDisplayName];
}


//==============================================================================
//
//==============================================================================
- (void)carouselDidScroll:(iCarousel *)carousel
{
    // coverflow load完才可以收這個，不然會讓預設要顯示的名片跑走
    if(self.coverFlowDidLoaded)
    {
        NSInteger oldIndex = carousel.currentItemIndex;
        
        if (self.currentItemIndex != oldIndex)
        {
            // 關閉actoin bar
            [self.toolbar hideContentBar];
            self.currentItemIndex = oldIndex;
        }
    }
}


//==============================================================================
//
//==============================================================================
- (void)carouselWillBeginDragging:(iCarousel *)carousel
{
    // 關閉actoin bar
    [self.toolbar setUserInteractionEnabled:NO];
}

//==============================================================================
//
//==============================================================================
- (void)carouselWillBeginScrollingAnimation:(iCarousel *)carousel
{
    // 一開始換就停止load圖
   [self stopCurrentImageLoadOperation];
}



//==============================================================================
//
//==============================================================================
- (void)carouselDidEndScrollingAnimation:(iCarousel *)carousel
{
    [self.toolbar setUserInteractionEnabled:YES];

    // load大圖
    PPRowModel *rowModel =  [self.cardSectionController rowModelAtIndex:self.coverFlowView.currentItemIndex forSearching:NO];
    if (rowModel)
    {
        WCCardModel *cardModel = (WCCardModel *)rowModel.object;
        [self startLoadCardImageWithCardID:cardModel.ID];
        self.currentCardModel = cardModel;
        
        // 取得目前的名片資料
        if ([self.dataSource respondsToSelector:@selector(coverFlowViewController:requestFullCardModelWithCardID:)])
        {
            __block typeof(self) blockSelf = self;
            
            dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
                blockSelf.currentCardModel = [blockSelf.dataSource coverFlowViewController:self requestFullCardModelWithCardID:cardModel.ID];
            });
        }
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPSectionIndexViewDelegate

//================================================================================
//
//================================================================================
- (void)sectionIndexView:(PPSectionIndexView *)sectionIndexView didSelectedIndexTitle:(NSString *)indexTitle;
{
    // 過濾掉不必要的動作
    if([[NSDate date] timeIntervalSince1970]-self.lastIndexingTime<=WCCoverFlowViewController_ScrollDuration)
    {
        return ;
    }
    
    [WCToastController showSectionIndexToastFromSuperView:self.view withTitle:indexTitle];
   
    //////////////////////////////////////////////////
    // 跳到指定的section title的第一張
    NSInteger section = [self sectionIndexByTitle:indexTitle];
    
    if(section<0)
    {
        section = 0;
    }

    NSInteger scrollToIndex = [self itemIndexFromIndexPath:[NSIndexPath indexPathForRow:0 inSection:section]];
    
    [self.coverFlowView scrollToItemAtIndex:scrollToIndex duration:WCCoverFlowViewController_ScrollDuration];
    self.lastIndexingTime = [[NSDate date] timeIntervalSince1970];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - WCCoverFlowActionBarDelegate


//==============================================================================
//
//==============================================================================
- (NSString *)contentStringAtIndex:(NSInteger)index withAction:(NSInteger)action
{
    if(self.currentCardModel==nil)
    {
        return nil;
    }
    
    WC_FieldType fieldType = WC_FT_None;
    switch (action)
    {
        case WCCoverFlowBarAction_Tel:
        {
            fieldType = WC_FT_Phone;
            break;
        }
        case WCCoverFlowBarAction_Map:
        {
            fieldType = WC_FT_Address;
            break;
        }
        case WCCoverFlowBarAction_Mail:
        {
            fieldType = WC_FT_Email;
            break;
        }
        default:
            break;
    }
    
    if (fieldType==WC_FT_None)
    {
        return nil;
    }
    
    NSString *content = nil;
    NSMutableArray *fieldArray = [self.currentCardModel fieldArrayWithType:fieldType];
    if([fieldArray count]<=0)
    {
        return nil;
    }
    
    // 調整index最大值
    if ([fieldArray count]<=index)
    {
        index = [fieldArray count] -1;
    }
    
    WCFieldModel * fieldModel = [fieldArray objectAtIndex:index];
    if (fieldModel==nil)
    {
        return nil;
    }
    
    if (fieldType==WC_FT_Address)
    {
        NSInteger bcrLang = [self.currentCardModel recognitionlanguageWithFieldSource:fieldModel.source];
        content = [fieldModel stringDisplayAddressWithBCRLanguage:bcrLang];
    }
    else
    {
        content = fieldModel.value;
    }
    
    return content;
}


//==============================================================================
//
//==============================================================================
- (BOOL)hasPrevWithAction:(NSInteger)action index:(NSInteger)index;
{
    if(self.currentCardModel==nil)
    {
        return NO;
    }
    
    WC_FieldType fieldType = WC_FT_None;
    switch (action)
    {
        case WCCoverFlowBarAction_Tel:
        {
            fieldType = WC_FT_Phone;
            break;
        }
        case WCCoverFlowBarAction_Map:
        {
            fieldType = WC_FT_Address;
            break;
        }
        case WCCoverFlowBarAction_Mail:
        {
            fieldType = WC_FT_Email;
            break;
        }
        default:
            break;
    }
    
    if (fieldType==WC_FT_None)
    {
        return NO;
    }
    
    NSInteger prevIndex = index-1;
    
    NSInteger fieldCount = [self.currentCardModel fieldCountWithFieldType:fieldType];
    BOOL result = (prevIndex>=0 && prevIndex < fieldCount)?YES:NO;
    return result;
}


//==============================================================================
//
//==============================================================================
- (BOOL)hasNextWithAction:(NSInteger)action index:(NSInteger)index;
{
    if(self.currentCardModel==nil)
    {
        return NO;
    }
    
    WC_FieldType fieldType = WC_FT_None;
    switch (action)
    {
        case WCCoverFlowBarAction_Tel:
        {
            fieldType = WC_FT_Phone;
            break;
        }
        case WCCoverFlowBarAction_Map:
        {
            fieldType = WC_FT_Address;
            break;
        }
        case WCCoverFlowBarAction_Mail:
        {
            fieldType = WC_FT_Email;
            break;
        }
        default:
            break;
    }
    
    if (fieldType==WC_FT_None)
    {
        return NO;
    }
    
    NSInteger nextIndex = index+1;
    
    NSInteger fieldCount = [self.currentCardModel fieldCountWithFieldType:fieldType];
    BOOL result = (nextIndex>=0 && nextIndex < fieldCount)?YES:NO;
    return result;
}


//==============================================================================
//
//==============================================================================
- (BOOL)shouldOpenContentBarWithAction:(NSInteger)action
{
    if(self.currentCardModel==nil)
    {
        return NO;
    }
    
    NSString *toastMessage = nil;
    WC_FieldType fieldType = WC_FT_None;
    
    switch (action)
    {
        case WCCoverFlowBarAction_Tel:
        {
            fieldType = WC_FT_Phone;
            toastMessage = WCCoverFlowViewControllerString_NoPhoneData;
            break;
        }
        case WCCoverFlowBarAction_Map:
        {
            fieldType = WC_FT_Address;
            toastMessage = WCCoverFlowViewControllerString_NoAddressData;
            break;
        }
        case WCCoverFlowBarAction_Mail:
        {
            fieldType = WC_FT_Email;
            toastMessage = WCCoverFlowViewControllerString_NoEMailData;
            break;
        }
        default:
            break;
    }
    
    if (fieldType==WC_FT_None)
    {
        return NO;
    }
    

    NSInteger fieldCount = [self.currentCardModel fieldCountWithFieldType:fieldType];

    BOOL result = (fieldCount>0)?YES:NO;
    if (result==NO)
    {
        [WCToastController showMessageToastFromSuperView:self.view
                                             withMessage:toastMessage
                                                position:PPToastPositionCenter];
    }
    return result;
}


//==============================================================================
//
//==============================================================================
- (void)actionBar:(WCCoverFlowActionBar *)actionBar didClickAction:(NSInteger)action withContent:(NSString *)content index:(NSInteger)index;
{
    if ([self.delegate respondsToSelector:@selector(coverFlowViewController:actionType:withActionContent:displayString:cardID:)])
    {
        if(self.currentCardModel==nil)
        {
            return ;
        }
        
        PPURLControllerType actionType = PPURLControllerType_None;
        WC_FieldType fieldType = WC_FT_None;
        switch (action)
        {
            case WCCoverFlowBarAction_Tel:
            {
                fieldType = WC_FT_Phone;
                actionType = PPURLControllerType_Tel;
                break;
            }
            case WCCoverFlowBarAction_Map:
            {
                fieldType = WC_FT_Address;
                actionType = PPURLControllerType_GoogleMap;
                break;
            }
            case WCCoverFlowBarAction_Mail:
            {
                fieldType = WC_FT_Email;
                actionType = PPURLControllerType_Email;
                break;
            }
            default:
                break;
        }
        
        if (fieldType==WC_FT_None)
        {
            return ;
        }
        
        NSString *actionContent = nil;
        NSString *cardID = self.currentCardModel.ID;
        NSMutableArray *fieldArray = [self.currentCardModel fieldArrayWithType:fieldType];
        if([fieldArray count]<=0)
        {
            return ;
        }

        WCFieldModel * fieldModel = [fieldArray objectAtIndex:index];
        if (fieldModel==nil)
        {
            return ;
        }
        
        // !! 地址的type會不同
        if (fieldType==WC_FT_Address)
        {
            NSInteger bcrLang = [self.currentCardModel recognitionlanguageWithFieldSource:fieldModel.source];
            actionType = [fieldModel addressURLTypeWithBCRLanguage:bcrLang];

            if(actionType==PPURLControllerType_GoogleMap)
            {
                actionContent = [fieldModel stringGoogleAddressWithBCRLanguage:bcrLang];
                
            }
            else
            {
                actionContent = [fieldModel stringDisplayAddressWithBCRLanguage:bcrLang];
                
            }
        }
        else
        {
            actionContent = content;
        }
        
        [self.delegate coverFlowViewController:self actionType:actionType withActionContent:actionContent displayString:content cardID:cardID];
    }
    
    [actionBar hideContentBar];
}


@end
