//
//  ContactDatePickerTableViewCell.m
//
//
//  Created by Howard on 2017/2/9.
//
//

#import "ContactDatePickerTableViewCell.h"

// Define
#import "ContactApperanceDefine.h"
#import "WCAppearanceDefine.h"
#import "WCContactViewController+ResourceDefine.h"

// Controller
#import "PPDateSectionController.h"


// View
#import "PPTableViewCellLineView.h"

// Category
#import "UIColor+Contact.h"
#import "NSDate+Format.h"

@interface ContactDatePickerTableViewCell ()
<
UIPickerViewDataSource,
UIPickerViewDelegate
>

@property (nonatomic,retain) PPTableViewCellLineView *contactTableViewCellLineView;
@property (nonatomic,retain) PPDateSectionController *dateSectionController;

@end


@implementation ContactDatePickerTableViewCell

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Dellocating Object

//================================================================================
//
//================================================================================
- (id)initWithStyle:(UITableViewCellStyle)style reuseIdentifier:(NSString *)reuseIdentifier
{
    self = [super initWithStyle:style reuseIdentifier:reuseIdentifier];
    
    if(self!=nil)
    {
        self.pickerView.dataSource = self;
        self.pickerView.delegate = self;
        
        //////////////////////////////////////////////////
        
        [self.textLabel setAdjustsFontSizeToFitWidth:YES];
        
        //////////////////////////////////////////////////
        
        [self.textField setTextColor:[UIColor contactEditViewInputFieldTextColor]];
        
        //////////////////////////////////////////////////
        
        _contactTableViewCellLineView = [[PPTableViewCellLineView alloc] init];
        
        if(_contactTableViewCellLineView!=nil)
        {
            _contactTableViewCellLineView.lineColor = WCAppearanceDefine_SeparatorColor;
            
            [self.contentView addSubview:_contactTableViewCellLineView];
            [self.contentView sendSubviewToBack:_contactTableViewCellLineView];
        }
        
        //////////////////////////////////////////////////
        
        _dateSectionController = [[PPDateSectionController alloc] init];
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    _contactKeyboardToolBarViewDelegate = nil;
    
    //////////////////////////////////////////////////
    
    [_dateSectionController release];
    _dateSectionController = nil;
    
    //////////////////////////////////////////////////
    
    [_contactTableViewCellLineView removeFromSuperview];
    [_contactTableViewCellLineView release];
    _contactTableViewCellLineView = nil;
    
    //////////////////////////////////////////////////
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Layout Of View

//================================================================================
//
//================================================================================
- (void)layoutSubviews
{
    [super layoutSubviews];
    
    //////////////////////////////////////////////////
    
    CGRect textLabelFrame = self.textLabel.frame;
    
    textLabelFrame.size.width = ContactTableViewCellLabelWidth;
    
    self.textLabel.frame = textLabelFrame;
    
    //////////////////////////////////////////////////
    
    self.contactTableViewCellLineView.frame = self.contentView.bounds;
    self.contactTableViewCellLineView.leftMargin = textLabelFrame.origin.x + textLabelFrame.size.width +ContactTableViewCellContentEdgeInset.left;
    [self.contactTableViewCellLineView performSelectorOnMainThread:@selector(setNeedsDisplay) withObject:nil waitUntilDone:NO];
    
    //////////////////////////////////////////////////
    
    CGRect textFieldFrame = self.textField.frame;
    textFieldFrame.origin.x   = self.contactTableViewCellLineView.leftMargin + ContactTableViewCellContentEdgeInset.left;
    textFieldFrame.size.width = self.contentView.bounds.size.width - ContactTableViewCellContentEdgeInset.right - textFieldFrame.origin.x;
    self.textField.frame = textFieldFrame;
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Property Method

//================================================================================
//
//================================================================================
- (void)setContactKeyboardToolBarViewDelegate:(id<ContactKeyboardToolBarViewDelegate>)contactKeyboardToolBarViewDelegate
{
    if(_contactKeyboardToolBarViewDelegate==nil)
    {
        _contactKeyboardToolBarViewDelegate = contactKeyboardToolBarViewDelegate;
        
        //////////////////////////////////////////////////
        
        ContactKeyboardToolBarView *toolBarView = [[ContactKeyboardToolBarView alloc] init];
        
        if(toolBarView!=nil)
        {
            self.textField.inputAccessoryView = toolBarView;
            toolBarView.contactKeyboardToolBarViewDelegate = self.contactKeyboardToolBarViewDelegate;
        }
        
        [toolBarView release];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Current Date IndexPath Method

//================================================================================
//
//================================================================================
- (NSIndexPath *)selectedIndexPathForSectionTitle:(NSString *)title
{
    NSIndexPath *indexPath = nil;
    
    
    do
    {
        if([title length]<=0)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSInteger selectedRow = NSNotFound;
        
        NSInteger section = [self.dateSectionController sectionForSectionIndexTitle:title                                                                           forSearching:NO];
        
        if(section==PPSectionNULL)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        selectedRow = [self.pickerView selectedRowInComponent:section];
        
        indexPath = [NSIndexPath indexPathForRow:selectedRow inSection:section];
    }
    while (0);
    
    
    return indexPath;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - CurrentInputViewProtocol Method

//================================================================================
//
//================================================================================
- (UITextField *)currentFocusTextField
{
    return self.textField;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UIPickerViewDataSource Method

//================================================================================
//
//================================================================================
- (NSInteger)numberOfComponentsInPickerView:(UIPickerView *)pickerView
{
    return [self.dateSectionController numberOfSectionsForSearching:NO];
}


//================================================================================
//
//================================================================================
- (NSInteger)pickerView:(UIPickerView *)pickerView numberOfRowsInComponent:(NSInteger)component
{
    NSUInteger count = 0;
    
    do
    {
        if([self.dateSectionController.sectionModelsForDefault count]<=component)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        PPSectionModel *sectionModel = [[self.dateSectionController sectionModelsForDefault] objectAtIndex:component];
        
        count = sectionModel.rowModels.count;
    }
    while (0);
    
    return count;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - UIPickerDelegate Method

//================================================================================
//
//================================================================================
- (UIView *)pickerView:(UIPickerView *)pickerView viewForRow:(NSInteger)row forComponent:(NSInteger)component reusingView:(UIView *)view {
    
    UILabel *label = (UILabel *) view;
    
    if (nil == label)
    {
        label = [[[UILabel alloc] init] autorelease];
    }
    
    [label setAutoresizingMask:(UIViewAutoresizingFlexibleWidth|UIViewAutoresizingFlexibleHeight)];
    
    //////////////////////////////////////////////////
    
    [label setFont:[UIFont systemFontOfSize:14.0f]];
    
    // 年月日
    if([self.dateSectionController numberOfSectionsForSearching:NO]==3)
    {
        if(component==0)
        {
            [label setTextAlignment:NSTextAlignmentRight];
        }
        else if(component==1)
        {
            [label setTextAlignment:NSTextAlignmentCenter];
        }
        else
        {
            [label setTextAlignment:NSTextAlignmentLeft];
        }
    }
    // 月日
    else if([self.dateSectionController numberOfSectionsForSearching:NO]==2)
    {
        if(component==0)
        {
            [label setTextAlignment:NSTextAlignmentRight];
        }
        else
        {
            [label setTextAlignment:NSTextAlignmentCenter];
        }
    }
    
    //////////////////////////////////////////////////
    
    PPRowModel *rowModel = [self.dateSectionController rowModelAtIndexPath:[NSIndexPath indexPathForRow:row
                                                                                              inSection:component]
                                                              forSearching:NO];
    
    if(rowModel!=nil)
    {
        [label setText:rowModel.text];
    }
    
    //////////////////////////////////////////////////
    
    PPSectionModel *sectionModel = [[self.dateSectionController sectionModelsForSearching:NO] objectAtIndex:component];
    
    // 日期要做 gray
    if([[sectionModel title] compare:[NSString stringWithInteger:PPDateSectionControllerComponent_Day]]==NSOrderedSame)
    {
        NSIndexPath *yearIndexPath = [self selectedIndexPathForSectionTitle:[NSString stringWithInteger:PPDateSectionControllerComponent_Year]];
        NSIndexPath *monthIndexPath = [self selectedIndexPathForSectionTitle:[NSString stringWithInteger:PPDateSectionControllerComponent_Month]];
        
        NSIndexPath *selectedDayIndexPath = [self selectedIndexPathForSectionTitle:[NSString stringWithInteger:PPDateSectionControllerComponent_Day]];
        
        
        //////////////////////////////////////////////////
        
        if([self.dateSectionController validDayWithYearIndexPath:yearIndexPath
                                                  monthIndexPath:monthIndexPath
                                                    dayIndexPath:[NSIndexPath indexPathForRow:row inSection:selectedDayIndexPath.section]
                                                           error:nil]==YES)
        {
            [label setTextColor:[UIColor blackColor]];
        }
        else
        {
            [label setTextColor:[UIColor lightGrayColor]];
        }
    }
    
    //////////////////////////////////////////////////
    
    
    return label;
}


//================================================================================
//
//================================================================================
- (void)pickerView:(UIPickerView *)pickerView
      didSelectRow:(NSInteger)row
       inComponent:(NSInteger)component
{
    // 尋找有無日期欄位
    for(NSUInteger index=0; index<self.dateSectionController.sectionModelsForDefault.count; index++)
    {
        PPSectionModel *sectionModel = [self.dateSectionController.sectionModelsForDefault objectAtIndex:index];
        
        // 日期要做 gray
        if([[sectionModel title] compare:[NSString stringWithInteger:PPDateSectionControllerComponent_Day]]==NSOrderedSame)
        {
            [self.pickerView reloadComponent:index];
            break;
        }
    }
    
    //////////////////////////////////////////////////
    
    NSIndexPath *yearIndexPath = [self selectedIndexPathForSectionTitle:[NSString stringWithInteger:PPDateSectionControllerComponent_Year]];
    NSIndexPath *monthIndexPath = [self selectedIndexPathForSectionTitle:[NSString stringWithInteger:PPDateSectionControllerComponent_Month]];
    NSIndexPath *dayIndexPath = [self selectedIndexPathForSectionTitle:[NSString stringWithInteger:PPDateSectionControllerComponent_Day]];
    
    //////////////////////////////////////////////////
    
    __block typeof(self) blockself = self;
    
    dispatch_async(dispatch_get_main_queue(), ^{
        
        NSIndexPath *approachDayIndexPath = [blockself.dateSectionController approachValidDayIndexPathWithYearIndexPath:yearIndexPath
                                                                                                         monthIndexPath:monthIndexPath
                                                                                                           dayIndexPath:dayIndexPath];
        
        
        //不允許點選此欄位
        if([approachDayIndexPath section]!=[dayIndexPath section] ||
           [approachDayIndexPath row]!=[dayIndexPath row])
        {
            [blockself.pickerView selectRow:[approachDayIndexPath row]
                                inComponent:[approachDayIndexPath section]
                                   animated:YES];
            
            [blockself.pickerView reloadComponent:[approachDayIndexPath section]];
        }
        
        //////////////////////////////////////////////////
        
        NSString *yearText = [self.dateSectionController rowModelAtIndexPath:yearIndexPath                                                                forSearching:NO].text;
        
        NSString *monthText = [self.dateSectionController rowModelAtIndexPath:monthIndexPath                                                                 forSearching:NO].text;
        
        NSString *dayText = [self.dateSectionController rowModelAtIndexPath:approachDayIndexPath                                                               forSearching:NO].text;
        
        //轉換失敗使用生日格式
        if([yearText integerValue]==0)
        {
            self.textField.text = [monthText stringByAppendingString:[NSString stringWithFormat:@"/%@",dayText]];
        }
        else
        {
            self.textField.text = [yearText stringByAppendingString:[NSString stringWithFormat:@"/%@/%@",monthText,dayText]];
        }
        
        //////////////////////////////////////////////////
        
        [blockself textFieldDidChange];
    });
    
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Method

//================================================================================
//
//================================================================================
- (void)refreshDisplayWithDateComponents:(NSArray <NSNumber *> *)components
{
    [self.dateSectionController generateDataForComponents:components];
}


//================================================================================
//
//================================================================================
- (void)updateDefaultDisplayRowContent
{
    // 月年日
    //預設欄位點擊
    NSInteger yearRow = NSNotFound;
    NSInteger monthRow = NSNotFound;
    NSInteger dayRow = NSNotFound;
    
    //////////////////////////////////////////////////

    NSInteger yearSection = [self.dateSectionController sectionForSectionIndexTitle:[NSString stringWithInteger:PPDateSectionControllerComponent_Year]
                                                             forSearching:NO];
    
    
    
    //////////////////////////////////////////////////
    
    NSInteger monthSection = [self.dateSectionController sectionForSectionIndexTitle:[NSString stringWithInteger:PPDateSectionControllerComponent_Month]
                                                              forSearching:NO];
    
    
    
    
    //////////////////////////////////////////////////
    
    NSInteger daySection = [self.dateSectionController sectionForSectionIndexTitle:[NSString stringWithInteger:PPDateSectionControllerComponent_Day]
                                                            forSearching:NO];
    
    
    //////////////////////////////////////////////////
    
    do
    {
        if([self.textField.text length]<=0)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        //轉不出來代表字串異常
        if([NSDate dateFromString:self.textField.text format:NSDateFormat_Year]==nil &&
           [NSDate dateFromString:self.textField.text format:NSDateFormat_MonthAndDay]==nil &&
           [NSDate dateFromString:self.textField.text format:NSDateFormat_Day]==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        NSArray *dateComponents = [self.textField.text componentsSeparatedByString:@"/"];
        
        //////////////////////////////////////////////////
        
        if(dateComponents.count>=3)
        {
            yearRow = [self.dateSectionController rowForRowModelText:[dateComponents firstObject] atSection:yearSection];
            
            //////////////////////////////////////////////////
            
            dayRow = [self.dateSectionController rowForRowModelText:[dateComponents lastObject] atSection:daySection];
            
            //////////////////////////////////////////////////
            
            monthRow = [self.dateSectionController rowForRowModelText:[dateComponents objectAtIndex:1] atSection:monthSection];
        }
        else
        {
            NSString *yearString = [[NSDate date] stringWithFormat:NSDateFormat_Year];
            
            yearRow = [self.dateSectionController rowForRowModelText:yearString atSection:daySection];
            
            dayRow = [self.dateSectionController rowForRowModelText:[dateComponents lastObject] atSection:daySection];
            
            //////////////////////////////////////////////////
            
            monthRow = [self.dateSectionController rowForRowModelText:[dateComponents firstObject] atSection:monthSection];
        }
    }
    while (0);
    
    //////////////////////////////////////////////////
    
    NSDateComponents *components = [[NSCalendar currentCalendar] components:NSCalendarUnitDay | NSCalendarUnitMonth | NSCalendarUnitYear fromDate:[NSDate date]];

    //////////////////////////////////////////////////

    NSUInteger selectedRow = 0;
    
    if(yearRow==NSNotFound)
    {
        selectedRow = [self.dateSectionController numberOfRowsInSection:yearSection forSearching:NO]-1;
        
        [self.pickerView selectRow:selectedRow
                       inComponent:yearSection animated:NO];
    }
    else
    {
        [self.pickerView selectRow:yearRow inComponent:yearSection animated:NO];
    }
    
    //////////////////////////////////////////////////
    
    if(monthRow==NSNotFound)
    {
        selectedRow = [self.dateSectionController rowForRowModelText:[NSString stringWithFormat:@"%02td",[components month]]
                                                           atSection:monthSection];
        
        [self.pickerView selectRow:selectedRow
                       inComponent:monthSection animated:NO];
    }
    else
    {
        [self.pickerView selectRow:monthRow inComponent:monthSection animated:NO];
    }
    
    //////////////////////////////////////////////////
    
    if(dayRow==NSNotFound)
    {
        selectedRow = [self.dateSectionController rowForRowModelText:[NSString stringWithFormat:@"%02td",[components day]]
                                                           atSection:daySection];
        
        [self.pickerView selectRow:selectedRow
                       inComponent:daySection animated:NO];
    }
    else
    {
        [self.pickerView selectRow:dayRow inComponent:daySection animated:NO];
    }
    
}

@end
