//
//  PPScrollBarView.m
//  Pods
//
//  Created by Duke on 2015/4/7.
//
//

#import "CVCScrollMenu.h"
#import "WCCaptureViewController+ResourceDefine.h"
#import "PPButton.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

static const CGFloat CVCScrollMenu_MaxItemWidth = 150.0;
static const CGFloat CVCScrollMenu_ButtonHorzMargin = 15.0;
static const CGFloat CVCScrollMenu_ButtomMargin = 10.0;
static const CGFloat CVCScrollMenu_DotRadius = 3.0;

#define CVCScrollMenu_ActiveTextColor WCCVC_LD_LightBlueColor
#define CVCScrollMenu_InactiveTextColor [UIColor whiteColor]
#define CVCScrollMenu_TextFont [UIFont boldSystemFontOfSize:15]

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface CVCScrollMenu() <PPButtonDelegate, UIScrollViewDelegate>

@property (nonatomic, retain) NSMutableArray *itemButtons;
@property (nonatomic ,assign) BOOL isAnimating;
@property (nonatomic ,retain) UIScrollView *scrollView;
@property (nonatomic ,retain) NSArray *layoutContraints;

@end





////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation CVCScrollMenu

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Init & Dealloc

//================================================================================
//
//================================================================================
- (id)init
{
    if (self = [super init])
    {
        self.backgroundColor = WCCVC_LD_DarkBlueColor;
        
        _scrollView = [[UIScrollView alloc] init];
        
        if(self.scrollView != nil)
        {
            self.scrollView.backgroundColor = [UIColor clearColor];
            self.scrollView.delegate = self;
            self.scrollView.showsHorizontalScrollIndicator = NO;
            self.scrollView.scrollEnabled = NO;
            [self addSubview:self.scrollView];
            
            self.scrollView.translatesAutoresizingMaskIntoConstraints = NO;
            [self setupLayoutConstraint];

        }
        
        self.isAnimating = NO;
        self.currentIndex = 0;
    }
                
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    if (self.layoutContraints)
    {
        [self removeConstraints:self.layoutContraints];
        self.layoutContraints = nil;
    }
    
    [self.scrollView removeFromSuperview];
    self.scrollView = nil;
    
    self.proxy = nil;
    self.itemTitles = nil;
    self.itemButtons = nil;
    
    
    //////////////////////////////////////////////////

    [super dealloc];

}




////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Override methods


//================================================================================
//
//================================================================================
- (void)drawRect:(CGRect)rect
{
    CGContextRef context = UIGraphicsGetCurrentContext();

    if(context != nil)
    {
        CGContextSaveGState(context);
        CGContextSetFillColorWithColor(context, WCCVC_LD_LightBlueColor.CGColor);
        
        CGPoint dotCenter = CGPointMake(CGRectGetMidX(rect),
                                        rect.size.height - CVCScrollMenu_DotRadius - 1);
        
        CGRect dotRect = CGRectMake(dotCenter.x-CVCScrollMenu_DotRadius,
                                    dotCenter.y-CVCScrollMenu_DotRadius,
                                    CVCScrollMenu_DotRadius*2,
                                    CVCScrollMenu_DotRadius*2);
                                    
        CGContextFillEllipseInRect(context, dotRect);
        
        CGContextRestoreGState(context);

    }
}


//================================================================================
//
//================================================================================
- (void)setHidden:(BOOL)hidden
{
    BOOL needUpdateConstraints = (hidden != self.isHidden);
    
    [super setHidden:hidden];
    
    if(needUpdateConstraints == YES)
    {
        [self setupLayoutConstraint];
    }
}


//================================================================================
//
//================================================================================
- (void)setItemTitles:(NSMutableArray *)itemTitles
{
    if(_itemTitles != itemTitles)
    {
        [itemTitles retain];
        [_itemTitles release];
        _itemTitles = itemTitles;
        
        if (self.itemTitles!=nil)
        {
            self.itemButtons = [NSMutableArray array];
            
            if (self.itemButtons != nil && itemTitles != nil)
            {
                for (NSInteger i = 0; i < [itemTitles count] ; i++)
                {
                    PPButton *button = [[PPButton alloc] initWithFrame:CGRectZero];
                    
                    button.delegate = self;
                    button.titleLabel.font = CVCScrollMenu_TextFont;
                    [button setTitle:[itemTitles objectAtIndex:i] forState:UIControlStateNormal];
                    [button setTitleColor:CVCScrollMenu_InactiveTextColor forState:UIControlStateNormal];
                    [button setBackgroundColor:[UIColor clearColor]];
                    [button addControlEvents:UIControlEventTouchUpInside];
                    button.tag = i;
                    
                    [self.scrollView addSubview:button];
                    [self.itemButtons addObject:button];
                    
                    [button release];
                }
            }    

        }
    }
}

//================================================================================
//
//================================================================================
- (void)layoutSubviews
{
    [super layoutSubviews];

    //////////////////////////////////////////////////
    
    // 讓scrollBar在捲動時不會執行到排版流程
    if (self.isAnimating)
    {
        return;
    }
    
    //////////////////////////////////////////////////

    if (self.itemButtons != nil && [self.itemButtons count] != 0)
    {
        CGSize defaultSize = CGSizeMake(CVCScrollMenu_MaxItemWidth, self.bounds.size.height);
        CGFloat originX = 0.0;

        for (NSInteger i = 0; i < [self.itemButtons count] ; i++)
        {
            PPButton *button = [self.itemButtons objectAtIndex:i];
            CGSize size = [button sizeThatFits:defaultSize];

            size.width += 2*CVCScrollMenu_ButtonHorzMargin;
            size.height = defaultSize.height;
            [button setFrame:CGRectMake(originX, 0, size.width, size.height)];
            originX += size.width;
        }
    }
    
    PPButton *button = [self.itemButtons objectAtIndex:self.currentIndex];
    [button setTitleColor:CVCScrollMenu_ActiveTextColor forState:UIControlStateNormal];
    [self.scrollView setContentOffset:CGPointMake(button.center.x-self.bounds.size.width/2, 0.0)];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPButtonDelegate methods

//================================================================================
//
//================================================================================
- (void)ppButton:(PPButton *)ppButton controlEvent:(UIControlEvents)controlEvent
{
    [self willScrollToNewIndex:ppButton.tag];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods

//================================================================================
//
//================================================================================
- (void)willScrollToNewIndex:(NSInteger)newIndex
{
    if(self.currentIndex == newIndex)
    {
        return;
    }
    
    //////////////////////////////////////////////////
    // 送出將要捲動的Delegate
    
    if ([self.proxy respondsToSelector:@selector(scrollMenu:willScrollToIndex:)])
    {
        [self.proxy scrollMenu:self willScrollToIndex:newIndex];
    }
}


//================================================================================
//
//================================================================================
- (void)setupLayoutConstraint
{
    if (self.layoutContraints)
    {
        [self removeConstraints:self.layoutContraints];
        self.layoutContraints = nil;
    }
    
    NSDictionary *views = @{@"scrollView":self.scrollView};
    NSDictionary *metrics = @{@"bottomMargin": (self.isHidden == YES) ? @(0) : @(CVCScrollMenu_ButtomMargin)};
    NSMutableArray *layoutConstraints = [NSMutableArray array];
    
    
    // 水平佈局
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"H:|[scrollView]|"
                                                                                   options:NSLayoutFormatDirectionLeftToRight
                                                                                   metrics:metrics
                                                                                     views:views]];
    
    // 垂直佈局
    [layoutConstraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:@"V:|[scrollView]-(bottomMargin)-|"
                                                                                   options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                   metrics:metrics
                                                                                     views:views]];
    
    
    if ([layoutConstraints count] > 0)
    {
        [self addConstraints:layoutConstraints];
        self.layoutContraints = layoutConstraints;
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance methods

//================================================================================
//
//================================================================================
- (void)scrollToRightSideItem
{
    if([self.itemButtons count] == 0)
    {
        return;
    }
    
    NSInteger newIndex = MIN([self.itemButtons count]-1, self.currentIndex+1);
    
    [self willScrollToNewIndex:newIndex];
}


//================================================================================
//
//================================================================================
- (void)scrollToLeftSideItem
{
    if([self.itemButtons count] == 0)
    {
        return;
    }
    
    NSInteger newIndex = MAX(0, self.currentIndex-1);
    
    [self willScrollToNewIndex:newIndex];
}


//================================================================================
//
//================================================================================
- (void)moveToItemIndex:(NSInteger)index
{
    //////////////////////////////////////////////////
    // 設定textLabel色彩
    
    PPButton *button = [self.itemButtons objectAtIndex:self.currentIndex];
    [button setTitleColor:CVCScrollMenu_InactiveTextColor forState:UIControlStateNormal];
    
    button = [self.itemButtons objectAtIndex:index];
    [button setTitleColor:CVCScrollMenu_ActiveTextColor forState:UIControlStateNormal];

    //////////////////////////////////////////////////
    // 移動UI
    
    [self.scrollView setContentOffset:CGPointMake(button.center.x-self.bounds.size.width/2, 0.0)];

    //////////////////////////////////////////////////

    self.currentIndex = index;
}



@end
