//
//  WCCSVDataController.m
//

#import "WCCSVDataController.h"

#import "NSError+Custom.h"
#import "PPCSVWriter.h"
#import "PPCSVFileHeader.h"
#import "PPCSVReader.h"


////////////////////////////////////////////////////////////////////////////////////////////////////

NSString * const WCCSVDC_GoogleGroupName_MyContact = @"My Contacts(Gmail)";
NSString * const WCCSVDC_GoogleGroupName_Friends = @"Friends(Gmail)";
NSString * const WCCSVDC_GoogleGroupName_Family = @"Family(Gmail)";
NSString * const WCCSVDC_GoogleGroupName_Coworkers = @"Coworkers(Gmail)";

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface WCCSVDataController ()
@property (nonatomic, assign) id<WCCSVDataControllerDelegate> delegate;
@property (nonatomic, assign) WCCSVDataController_Format foramt;
@property (nonatomic, assign) NSInteger writtenCount;
@property (nonatomic, retain) NSArray *writeCardIDs;
@property (nonatomic, retain) NSArray *csvDatas;
@property (nonatomic, retain) NSArray *readFilePaths;
@property (nonatomic, retain) PPCSVFileHeader *csvFileHeader;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation WCCSVDataController

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods

//================================================================================
//
//================================================================================
- (instancetype)initWithDelegate:(id<WCCSVDataControllerDelegate>)delegate
{
    if(self = [super init])
    {
        self.delegate = delegate;
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    self.delegate = nil;
    self.writeCardIDs = nil;
    self.csvDatas = nil;
    self.csvFileHeader = nil;
    self.readFilePaths = nil;
    
    //////////////////////////////////////////////////
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPCSVWriterDelegate methods

//================================================================================
//
//================================================================================
- (PPCSVWriterRet)writerQueryHeader:(NSObject*)aDelegateUserData
                   outputFileHeader:(PPCSVFileHeader**)aCSVFileHeader
{
    PPCSVWriterRet retValue = KPPCSVWriterRet_Failed;
    
    NSInteger iContactCount = [self.writeCardIDs count];
    
    self.csvFileHeader = nil;
    _csvFileHeader = [[PPCSVFileHeader alloc] initWithContactCount:iContactCount];
    
    if(self.csvFileHeader != nil)
    {
        *aCSVFileHeader = self.csvFileHeader;
        retValue = KPPCSVWriterRet_Success;
    }
    
    return retValue;
}


//================================================================================
//
//================================================================================
- (PPCSVWriterRet)writerQueryTitleDataArray:(NSObject*)aDelegateUserData
                          outputContactData:(NSArray**)aTitleDataArray
                         addDoubleQuotation:(BOOL*)bAddDoubleQuotationPtr
{
    self.csvDatas = nil;
    
    if(aTitleDataArray)
    {
        *aTitleDataArray = nil;
    }
    
    if(bAddDoubleQuotationPtr)
    {
        *bAddDoubleQuotationPtr = NO;
    }

    //////////////////////////////////////////////////

    PPCSVData* csvData = [[PPCSVData alloc]initWithContactTemplateModel];
    
    if(csvData)
    {
        //#if defined (FOR_JAPANESE_VERSION) || defined (FOR_KOREAN_VERSION)
        [csvData enablePhoneticNameField:YES];
        //#else
        //[csvData enablePhoneticNameField:NO];
        //#endif
        
        NSArray* titleArrayForWriter = nil;

        switch (self.foramt)
        {
            case WCCSVDataController_Format_Gmail:
            {
                titleArrayForWriter = [csvData newGmailCSVTitleArrayForWriter];
                
                if(bAddDoubleQuotationPtr)
                {
                    *bAddDoubleQuotationPtr = NO;
                }
                
                break;
            }
                
            case WCCSVDataController_Format_Outlook:
            {
                titleArrayForWriter = [csvData newOutlookCSVTitleArrayForWriter];
                
                if(bAddDoubleQuotationPtr)
                {
                    *bAddDoubleQuotationPtr = YES;
                }
                
                break;
            }
                
            case WCCSVDataController_Format_UTF8:
            case WCCSVDataController_Format_UTF16:
            case WCCSVDataController_Format_iWorks:
            case WCCSVDataController_Format_WindowsExcel:
            {
                titleArrayForWriter = [csvData newCSVTitleArrayForWriter];
                
                if(bAddDoubleQuotationPtr)
                {
                    *bAddDoubleQuotationPtr = YES;
                }
                
                break;
            }
                
            default:
                break;
        }
        
        if(titleArrayForWriter)
        {
            self.csvDatas = titleArrayForWriter;
            
            if(aTitleDataArray)
            {
                *aTitleDataArray = self.csvDatas;
            }
            
            [titleArrayForWriter release];
            titleArrayForWriter = nil;
        }
        
        [csvData release];
        csvData = nil;
    }
    
    return KPPCSVWriterRet_Success;
}


//================================================================================
//
//================================================================================
- (PPCSVWriterRet)writerQueryItemDataArray:(NSObject*)aDelegateUserData
                              contactIndex:(NSInteger)aContactIndex
                         outputContactData:(NSArray**)aItemDataArray
{
    PPCSVWriterRet retValue = KPPCSVWriterRet_Failed;
    
    do
    {
        if([self.delegate csvDataContrller:self
                  shouldWriteCardWithIndex:aContactIndex
                                totalCount:self.writeCardIDs.count]==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        self.csvDatas = nil;
        
        if(aItemDataArray)
        {
            *aItemDataArray = nil;
        }
        
        NSInteger iTotalUidCountToExport = [self.writeCardIDs count];
        if(aContactIndex < iTotalUidCountToExport)
        {
            NSString* nsUid = nil;
            nsUid = (NSString*)[self.writeCardIDs objectAtIndex:aContactIndex];
            if(nsUid)
            {
                PPCSVData* csvData = [self newCSVDataForCardID:nsUid];
                
                if(csvData)
                {
                    [csvData enablePhoneticNameField:YES];
                    
                    NSArray* dataArray = nil;
                    
                    switch (self.foramt)
                    {
                        case WCCSVDataController_Format_Outlook:
                            dataArray = [csvData newOutlookCSVDataArrayForWriter];
                            break;
                            
                        case WCCSVDataController_Format_Gmail:
                            dataArray = [csvData newGmailCSVDataArrayForWriter];
                            break;
                            
                        case WCCSVDataController_Format_WindowsExcel:
                            dataArray = [csvData newCSVDataArrayForWriter:YES];
                            break;
                            
                        case WCCSVDataController_Format_UTF8:
                        case WCCSVDataController_Format_UTF16:
                        case WCCSVDataController_Format_iWorks:
                            dataArray = [csvData newCSVDataArrayForWriter:NO];
                            break;
                            
                        default:
                            break;
                    }
                    
                    if(dataArray)
                    {
                        self.csvDatas = dataArray;
                        
                        if(aItemDataArray)
                        {
                            *aItemDataArray = self.csvDatas;
                            self.writtenCount++;
                        }
                        
                        [dataArray release];
                        dataArray = nil;
                    }
                    
                    [csvData release];
                    csvData = nil;
                }
            }
            
            retValue = KPPCSVWriterRet_Success;
        }
        
        if(retValue == KPPCSVWriterRet_Success)
        {
            [self.delegate csvDataContrller:self didWriteCardWithIndex:aContactIndex totalCount:iTotalUidCountToExport];
        }
    }
    while (0);
   
    return retValue;
}


//================================================================================
//
//================================================================================
- (void)writerDidFinishWriting:(NSObject*)aDelegateUserData
{
    self.csvFileHeader = nil;
    self.csvDatas = nil;
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods

//================================================================================
//
//================================================================================
- (NSArray *)copyGroupNamesWithCardModel:(WCCardModel *)cardModel
{
    NSMutableArray *groupNames = [[NSMutableArray alloc] init];
    
    for(NSString *groupIDString in cardModel.groupIDArray)
    {
        WC_GroupID groupID = [groupIDString integerValue];
        NSString *groupName = nil;
        
        switch (groupID)
        {
            case WC_GID_Unfiled:
                // 未分類不處理
                break;
                
            case WC_GID_Google_MyContacts:
            {
                groupName = WCCSVDC_GoogleGroupName_MyContact;
                [groupNames addObject:groupName];
                break;
            }
                
            case WC_GID_Google_Friends:
            {
                groupName = WCCSVDC_GoogleGroupName_Friends;
                [groupNames addObject:groupName];
                break;
            }
                
            case WC_GID_Google_Family:
            {
                groupName = WCCSVDC_GoogleGroupName_Family;
                [groupNames addObject:groupName];
                break;
            }
                
            case WC_GID_Google_Coworkers:
            {
                groupName = WCCSVDC_GoogleGroupName_Coworkers;
                [groupNames addObject:groupName];
                break;
            }
                
            default:
            {
                groupName = [self.delegate csvDataContrller:self groupNameToWriteWithGroupID:groupID];
                
                if([groupName length] > 0)
                {
                    [groupNames addObject:groupName];
                }
                
                break;
            }
        }
    }
    
    if([groupNames count] == 0)
    {
        [groupNames release];
        groupNames = nil;
    }
    
    return groupNames;
}


//================================================================================
//
//================================================================================
- (PPCSVData*)newCSVDataForCardID:(NSString *)cardID
{
    PPCSVData* resultCSVData = [[PPCSVData alloc] initWithContactTemplateModel];
    
    if(resultCSVData == nil)
    {
        return nil;
    }
    
    //////////////////////////////////////////////////

    WCCardModel *aCardModel = [self.delegate csvDataContrller:self cardModelToWriteWithCardID:cardID];

    if(aCardModel != nil)
    {
        // group
        NSArray *groupNames = [self copyGroupNamesWithCardModel:aCardModel];
        
        for(NSString *groupName in groupNames)
        {
            [resultCSVData addOneItem:groupName forKey:KPPCSV_CDL1_Group];
        }
        
        [groupNames release];
        
        //////////////////////////////////////////////////
        
        //name
        NSMutableArray *nameArray = [aCardModel fieldArrayWithType:WC_FT_Name];
        for(int i=0;i<[nameArray count];i++)
        {
            NSString *firstName = @"", *lastName = @"", *firstNamePhonetic = @"", *lastNamePhonetic = @"", *middleName = @"", *prefix = @"", *suffix = @"";
            if([((WCFieldModel *)[nameArray objectAtIndex:i]) hasFieldWithSubType2:WC_FST2_Name_First])
                firstName = [[NSString alloc] initWithString:[((WCFieldModel *)[nameArray objectAtIndex:i]) fieldWithSubType2:WC_FST2_Name_First].value];
            if([((WCFieldModel *)[nameArray objectAtIndex:i]) hasFieldWithSubType2:WC_FST2_Name_Last])
                lastName = [[NSString alloc] initWithString:[((WCFieldModel *)[nameArray objectAtIndex:i]) fieldWithSubType2:WC_FST2_Name_Last].value];
            if([((WCFieldModel *)[nameArray objectAtIndex:i]) hasFieldWithSubType2:WC_FST2_Name_FirstPhonetic])
                firstNamePhonetic = [[NSString alloc] initWithString:[((WCFieldModel *)[nameArray objectAtIndex:i]) fieldWithSubType2:WC_FST2_Name_FirstPhonetic].value];
            if([((WCFieldModel *)[nameArray objectAtIndex:i]) hasFieldWithSubType2:WC_FST2_Name_LastPhonetic])
                lastNamePhonetic = [[NSString alloc] initWithString:[((WCFieldModel *)[nameArray objectAtIndex:i]) fieldWithSubType2:WC_FST2_Name_LastPhonetic].value];
            if([((WCFieldModel *)[nameArray objectAtIndex:i]) hasFieldWithSubType2:WC_FST2_Name_Middle])
                middleName = [[NSString alloc] initWithString:[((WCFieldModel *)[nameArray objectAtIndex:i]) fieldWithSubType2:WC_FST2_Name_Middle].value];
            if([((WCFieldModel *)[nameArray objectAtIndex:i]) hasFieldWithSubType2:WC_FST2_Name_Prefix])
                prefix = [[NSString alloc] initWithString:[((WCFieldModel *)[nameArray objectAtIndex:i]) fieldWithSubType2:WC_FST2_Name_Prefix].value];
            if([((WCFieldModel *)[nameArray objectAtIndex:i]) hasFieldWithSubType2:WC_FST2_Name_Suffix])
                suffix = [[NSString alloc] initWithString:[((WCFieldModel *)[nameArray objectAtIndex:i]) fieldWithSubType2:WC_FST2_Name_Suffix].value];
            
            [resultCSVData addOneItem:firstName forKey:KPPCSV_CDL1_FirstName];
            [resultCSVData addOneItem:firstNamePhonetic forKey:KPPCSV_CDL1_PhoneticFirstName];
            [resultCSVData addOneItem:lastName forKey:KPPCSV_CDL1_LastName];
            [resultCSVData addOneItem:lastNamePhonetic forKey:KPPCSV_CDL1_PhoneticLastName];
            [resultCSVData addOneItem:middleName forKey:KPPCSV_CDL1_MiddleName];
            [resultCSVData addOneItem:prefix forKey:KPPCSV_CDL1_Prefix];
            [resultCSVData addOneItem:suffix forKey:KPPCSV_CDL1_Suffix];
            [firstName release];
            [lastName release];
            [firstNamePhonetic release];
            [lastNamePhonetic release];
            [middleName release];
            [prefix release];
            [suffix release];
        }
        //company
        NSMutableArray *companyArray = [aCardModel fieldArrayWithType:WC_FT_Company];
        for(int i=0;i<[companyArray count];i++)
        {
            NSString *companyName = @"", *companyPhonetic = @"", *jobTitle = @"", *department = @"";
            if([((WCFieldModel *)[companyArray objectAtIndex:i]) hasFieldWithSubType2:WC_FST2_Company_Name])
                companyName = [[NSString alloc] initWithString:[((WCFieldModel *)[companyArray objectAtIndex:i]) fieldWithSubType2:WC_FST2_Company_Name].value];
            if([((WCFieldModel *)[companyArray objectAtIndex:i]) hasFieldWithSubType2:WC_FST2_Company_Phonetic])
                companyPhonetic = [[NSString alloc] initWithString:[((WCFieldModel *)[companyArray objectAtIndex:i]) fieldWithSubType2:WC_FST2_Company_Phonetic].value];
            if([((WCFieldModel *)[companyArray objectAtIndex:i]) hasFieldWithSubType2:WC_FST2_Company_JobTitle])
                jobTitle = [[NSString alloc] initWithString:[((WCFieldModel *)[companyArray objectAtIndex:i]) fieldWithSubType2:WC_FST2_Company_JobTitle].value];
            if([((WCFieldModel *)[companyArray objectAtIndex:i]) hasFieldWithSubType2:WC_FST2_Company_Department])
                department = [[NSString alloc] initWithString:[((WCFieldModel *)[companyArray objectAtIndex:i]) fieldWithSubType2:WC_FST2_Company_Department].value];
            
            [resultCSVData addOneItem:companyName forKey:KPPCSV_CDL1_Company];
            [resultCSVData addOneItem:companyPhonetic forKey:KPPCSV_CDL1_PhoneticCompany];
            [resultCSVData addOneItem:jobTitle forKey:KPPCSV_CDL1_JobTitle];
            [resultCSVData addOneItem:department forKey:KPPCSV_CDL1_Department];
            [companyName release];
            [companyPhonetic release];
            [jobTitle release];
            [department release];
        }
        //address
        NSMutableArray *addressArray = [aCardModel fieldArrayWithType:WC_FT_Address];
        for(int i=0;i<[addressArray count];i++)
        {
            WCFieldModel *fieldModel = [addressArray objectAtIndex:i];
            
            NSString *street = (NSString *)[fieldModel valueWithSubType2:WC_FST2_Address_Street];
            NSString *city = (NSString *)[fieldModel valueWithSubType2:WC_FST2_Address_City];
            NSString *state = (NSString *)[fieldModel valueWithSubType2:WC_FST2_Address_State];
            NSString *ZIP = (NSString *)[fieldModel valueWithSubType2:WC_FST2_Address_ZIP];

            //////////////////////////////////////////////////
            // 取得完整地址
            NSString *fullAddress = nil;
            
            NSInteger bcrLang = [aCardModel recognitionlanguageWithFieldSource:fieldModel.source];
            fullAddress = [fieldModel stringDisplayAddressWithBCRLanguage:bcrLang];
            
            // !! special case for country string
            NSString *country = [fieldModel valueWithSubType2:WC_FST2_Address_Country];
            
            if([country length]==0)
            {
                country = [fieldModel stringDisplayCountryWithBCRLanguage:bcrLang];
            }
            

            // 欄位要對應，所以不能輸入nil。
            street = (street == nil) ? @"" : street;
            city = (city == nil) ? @"" : city;
            state = (state == nil) ? @"" : state;
            ZIP = (ZIP == nil) ? @"" : ZIP;
            country = (country == nil) ? @"" : country;
            
            switch (fieldModel.subType1)
            {
                case WC_FST1_Address_Work:
                {
                    // full address
                    [resultCSVData addOneItem:fullAddress forKey:KPPCSV_CDL1_Address_Work];

                    // separate address
                    [resultCSVData addOneItem:street forKey:KPPCSV_CDL1_Address_Work_Street];
                    [resultCSVData addOneItem:city forKey:KPPCSV_CDL1_Address_Work_City];
                    [resultCSVData addOneItem:state forKey:KPPCSV_CDL1_Address_Work_State];
                    [resultCSVData addOneItem:ZIP forKey:KPPCSV_CDL1_Address_Work_ZIP];
                    [resultCSVData addOneItem:country forKey:KPPCSV_CDL1_Address_Work_Country];
                    break;
                }
                    
                case WC_FST1_Address_Home:
                {
                    // full address
                    [resultCSVData addOneItem:fullAddress forKey:KPPCSV_CDL1_Address_Home];
                    
                    // separate address
                    [resultCSVData addOneItem:street forKey:KPPCSV_CDL1_Address_Home_Street];
                    [resultCSVData addOneItem:city forKey:KPPCSV_CDL1_Address_Home_City];
                    [resultCSVData addOneItem:state forKey:KPPCSV_CDL1_Address_Home_State];
                    [resultCSVData addOneItem:ZIP forKey:KPPCSV_CDL1_Address_Home_ZIP];
                    [resultCSVData addOneItem:country forKey:KPPCSV_CDL1_Address_Home_Country];
                    break;
                    break;
                }
                    
                //case WC_FST1_Address_Other:
                default:
                {
                    // full address
                    [resultCSVData addOneItem:fullAddress forKey:KPPCSV_CDL1_Address_Other];
                    
                    // separate address
                    [resultCSVData addOneItem:street forKey:KPPCSV_CDL1_Address_Other_Street];
                    [resultCSVData addOneItem:city forKey:KPPCSV_CDL1_Address_Other_City];
                    [resultCSVData addOneItem:state forKey:KPPCSV_CDL1_Address_Other_State];
                    [resultCSVData addOneItem:ZIP forKey:KPPCSV_CDL1_Address_Other_ZIP];
                    [resultCSVData addOneItem:country forKey:KPPCSV_CDL1_Address_Other_Country];
                    break;
                }
            };
        }
        //phone
        NSMutableArray *phoneArray = [aCardModel fieldArrayWithType:WC_FT_Phone];
        for(int i=0;i<[phoneArray count];i++)
        {
            NSString *phone = @"";
            phone = [[NSString alloc] initWithString:((WCFieldModel *)[phoneArray objectAtIndex:i]).value];
            NSString *nType;
            switch ([((WCFieldModel *)[phoneArray objectAtIndex:i]) subType1]) {
                case WC_FST1_Phone_Work:
                    nType = KPPCSV_CDL1_Phone_Work;
                    break;
                case WC_FST1_Phone_Home:
                    nType = KPPCSV_CDL1_Phone_Home;
                    break;
                case WC_FST1_Phone_Main:
                    nType = KPPCSV_CDL1_Phone_Main;
                    break;
                case WC_FST1_Phone_GoogleVoice:
                    nType = KPPCSV_CDL1_Phone_GoogleVoice;
                    break;
                case WC_FST1_Phone_HomeFax:
                    nType = KPPCSV_CDL1_Phone_HomeFax;
                    break;
                case WC_FST1_Phone_iPhone:
                    nType = KPPCSV_CDL1_Phone_iPhone;
                    break;
                case WC_FST1_Phone_Mobile:
                    nType = KPPCSV_CDL1_Phone_Mobile;
                    break;
                case WC_FST1_Phone_Pager:
                    nType = KPPCSV_CDL1_Phone_Pager;
                    break;
                case WC_FST1_Phone_WorkFax:
                    nType = KPPCSV_CDL1_Phone_WorkFax;
                    break;
                case WC_FST1_Phone_Other:
                    nType = KPPCSV_CDL1_Phone_Other;
                    break;
                default:
                    nType = KPPCSV_CDL1_Phone_Other;
                    break;
            };
            
            [resultCSVData addOneItem:phone forKey:nType];
            [phone release];
        }
        //email
        NSMutableArray *emailArray = [aCardModel fieldArrayWithType:WC_FT_Email];
        for(int i=0;i<[emailArray count];i++)
        {
            NSString *email = @"";
            email = [[NSString alloc] initWithString:((WCFieldModel *)[emailArray objectAtIndex:i]).value];
            NSString *nType;
            switch ([((WCFieldModel *)[emailArray objectAtIndex:i]) subType1]) {
                case WC_FST1_Email_Work:
                    nType = KPPCSV_CDL1_Email_Work;
                    break;
                case WC_FST1_Email_Home:
                    nType = KPPCSV_CDL1_Email_Home;
                    break;
                case WC_FST1_Email_Other:
                    nType = KPPCSV_CDL1_Email_Other;
                    break;
                default:
                    nType = KPPCSV_CDL1_Email_Other;
                    break;
            };
            
            [resultCSVData addOneItem:email forKey:nType];
            [email release];
        }
        //url
        NSMutableArray *urlArray = [aCardModel fieldArrayWithType:WC_FT_URL];
        for(int i=0;i<[urlArray count];i++)
        {
            NSString *url = @"";
            url = [[NSString alloc] initWithString:((WCFieldModel *)[urlArray objectAtIndex:i]).value];
            NSString *nType;
            switch ([((WCFieldModel *)[urlArray objectAtIndex:i]) subType1]) {
                case WC_FST1_URL_Blog:
                    nType = KPPCSV_CDL1_URL_Blog;
                    break;
                case WC_FST1_URL_Home:
                    nType = KPPCSV_CDL1_URL_Home;
                    break;
                case WC_FST1_URL_HomePage:
                    nType = KPPCSV_CDL1_URL_Homepage;
                    break;
                case WC_FST1_URL_Profile:
                    nType = KPPCSV_CDL1_URL_Profile;
                    break;
                case WC_FST1_URL_Work:
                    nType = KPPCSV_CDL1_URL_Work;
                    break;
                case WC_FST1_URL_Other:
                    nType = KPPCSV_CDL1_URL_Other;
                    break;
                default:
                    nType = KPPCSV_CDL1_URL_Other;
                    break;
            };
            
            [resultCSVData addOneItem:url forKey:nType];
            [url release];
        }
        //Date
        NSMutableArray *dateArray = [aCardModel fieldArrayWithType:WC_FT_Date];
        for(int i=0;i<[dateArray count];i++)
        {
            NSString *date = @"";
            date = [[NSString alloc] initWithString:[self convertDateToDateString:((WCFieldModel *)[dateArray objectAtIndex:i]).value]];
            NSString *nType;
            switch ([((WCFieldModel *)[dateArray objectAtIndex:i]) subType1]) {
                case WC_FST1_Date_Anniversary:
                    nType = KPPCSV_CDL1_Date_Anniversary;
                    break;
                case WC_FST1_Date_Birthday:
                    nType = KPPCSV_CDL1_Date_Birthday;
                    break;
                case WC_FST1_Date_Other:
                    nType = KPPCSV_CDL1_Date_Other;
                    break;
                default:
                    nType = KPPCSV_CDL1_Date_Other;
                    break;
            };
            
            [resultCSVData addOneItem:date forKey:nType];
            [date release];
        }
        //InstantMessage
        NSMutableArray *instantMessageArray = [aCardModel fieldArrayWithType:WC_FT_InstantMessage];
        for(int i=0;i<[instantMessageArray count];i++)
        {
            NSString *instantMessage = @"";
            instantMessage = [[NSString alloc] initWithString:((WCFieldModel *)[instantMessageArray objectAtIndex:i]).value];
            NSString *nType;
            switch ([((WCFieldModel *)[instantMessageArray objectAtIndex:i]) subType1])
            {
                case WC_FST1_IM_MSN:
                    nType = KPPCSV_CDL1_IM_MSN;
                    break;
                case WC_FST1_IM_AIM:
                    nType = KPPCSV_CDL1_IM_AIM;
                    break;
                case WC_FST1_IM_GoogleTalk:
                    nType = KPPCSV_CDL1_IM_GoogleTalk;
                    break;
                case WC_FST1_IM_ICQ:
                    nType = KPPCSV_CDL1_IM_ICQ;
                    break;
                case WC_FST1_IM_Jabber:
                    nType = KPPCSV_CDL1_IM_Jabber;
                    break;
                case WC_FST1_IM_QQ:
                    nType = KPPCSV_CDL1_IM_QQ;
                    break;
                case WC_FST1_IM_Skype:
                    nType = KPPCSV_CDL1_IM_Skype;
                    break;
                case WC_FST1_IM_Yahoo:
                    nType = KPPCSV_CDL1_IM_Yahoo;
                    break;
                case WC_FST1_IM_GaduGadu:
                    nType = KPPCSV_CDL1_IM_GaduGadu;
                    break;
                case WC_FST1_IM_Other:
                    nType = KPPCSV_CDL1_IM_Other;
                    break;
                default:
                    nType = KPPCSV_CDL1_IM_AIM;
                    break;
            };
            
            
            [resultCSVData addOneItem:instantMessage forKey:nType];
            [instantMessage release];
        }
        //nickname
        NSMutableArray *nickNameArray  = [aCardModel fieldArrayWithType:WC_FT_Nickname];
        for(int i=0;i<[nickNameArray count];i++)
        {
            NSString *nickName = @"";
            nickName = [[NSString alloc] initWithString:((WCFieldModel *)[nickNameArray objectAtIndex:i]).value];
            
            [resultCSVData addOneItem:nickName forKey:KPPCSV_CDL1_NickName];
            [nickName release];
        }
        //unfiedbusinessno
        NSMutableArray *unfiedBusinessNoArray  = [aCardModel fieldArrayWithType:WC_FT_UnifiedBusinessNo];
        for(int i=0;i<[unfiedBusinessNoArray count];i++)
        {
            NSString *unfiedBusinessNo = @"";
            unfiedBusinessNo = [[NSString alloc] initWithString:((WCFieldModel *)[unfiedBusinessNoArray objectAtIndex:i]).value];
            
            [resultCSVData addOneItem:unfiedBusinessNo forKey:KPPCSV_CDL1_UnifiedBusinessNo];
            [unfiedBusinessNo release];
        }
        //note
        NSMutableArray *noteArray  = [aCardModel fieldArrayWithType:WC_FT_Note];
        for(int i=0;i<[noteArray count];i++)
        {
            NSString *note = @"";
            note = [[NSString alloc] initWithString:((WCFieldModel *)[noteArray objectAtIndex:i]).value];
            
            [resultCSVData addOneItem:note forKey:KPPCSV_CDL1_Note];
            [note release];
        }

    }
    
    return resultCSVData;
}


//================================================================================
//
//================================================================================
- (NSString *)convertDateToDateString:(NSDate *)date
{
    NSDateFormatter *dateFormatter = [[NSDateFormatter alloc] init];
    [dateFormatter setDateFormat:@"yyyy-MM-dd"];
    NSString *dateString = [dateFormatter stringFromDate:date];
    [dateFormatter release];
    return dateString;
}


//================================================================================
//
//================================================================================
- (NSDate *)convertDateFromDateString:(NSString *)dateString
{
    NSDateFormatter *dateFormatter = [[NSDateFormatter alloc] init];
    [dateFormatter setDateFormat:@"yyyy-MM-dd"];
    NSDate *date = [dateFormatter dateFromString:dateString];
    [dateFormatter release];
    return date;
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private CSV to WCCardModel Method

//================================================================================
//
//================================================================================
- (WC_FieldType)fieldTypeFromSourceType:(NSInteger)sourceType
{
    WC_FieldType fieldType = sourceType;
    
    if(sourceType>=WC_FST2_Name_First &&
       sourceType<=WC_FST2_Name_Suffix)
    {
        fieldType = WC_FT_Name;
    }
    else if(sourceType>=WC_FST2_Company_Name &&
            sourceType<=WC_FST2_Company_JobTitle)
    {
        fieldType = WC_FT_Company;
    }
    else if(sourceType>=WC_FST2_Address_Street &&
            sourceType<=WC_FST2_Address_Country)
    {
        fieldType = WC_FT_Address;
    }
    else if(sourceType>=WC_FST1_Phone_Work &&
            sourceType<=WC_FST1_Phone_OtherFax)
    {
        fieldType = WC_FT_Phone;
    }
    else if(sourceType>=WC_FST1_Email_Work &&
            sourceType<=WC_FST1_Email_Other)
    {
        fieldType = WC_FT_Email;
    }
    else if(sourceType>=WC_FST1_URL_HomePage &&
            sourceType<=WC_FST1_URL_Other)
    {
        fieldType = WC_FT_URL;
    }
    else if(sourceType>=WC_FST1_IM_MSN &&
            sourceType<=WC_FST1_IM_GaduGadu)
    {
        fieldType = WC_FT_InstantMessage;
    }
    else if(sourceType>=WC_FST1_Date_Birthday &&
            sourceType<=WC_FST1_Date_Other)
    {
        fieldType = WC_FT_Date;
    }
    else if(sourceType>=WC_FST1_SN_Twitter &&
            sourceType<=WC_FST1_SN_Other)
    {
        fieldType = WC_FT_SocialNetwork;
    }
    
    return fieldType;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance methods

//================================================================================
//
//================================================================================
- (BOOL)writeFileWithPath:(NSString *)filePath cardIDs:(NSArray *)cardIDs format:(WCCSVDataController_Format)format willSendByEmail:(BOOL)willSendByEmail error:(NSError **)error
{
    BOOL result = NO;
    NSError *returnError = nil;
    NSInteger cardCount = [cardIDs count];
    PPCSVWriter *csvWriter = nil;
    
    do
    {
        //////////////////////////////////////////////////
        
        if(cardCount == 0)
        {
            returnError = PPErrorMake(WCCSVDataController_Error_NoCardToWrite, nil, nil);
            break;
        }

        //////////////////////////////////////////////////
        
        self.writeCardIDs = cardIDs;
        self.foramt = format;
        self.writtenCount = 0;
        
        //////////////////////////////////////////////////
        // 一開始先檢查delegate的目的是減少檢查次數，同時提示此流程會使用的delegate method。
        
        if([self.delegate respondsToSelector:@selector(csvDataContrller:cardModelToWriteWithCardID:)] == NO ||
           [self.delegate respondsToSelector:@selector(csvDataContrller:groupNameToWriteWithGroupID:)] == NO ||
           [self.delegate respondsToSelector:@selector(csvDataContrller:shouldWriteCardWithIndex:totalCount:)] == NO ||
           [self.delegate respondsToSelector:@selector(csvDataContrller:didWriteCardWithIndex:totalCount:)] == NO)
        {
            returnError = PPErrorMake(WCCSVDataController_Error_MustHaveDelegateMethod, nil, nil);
            break;
        }
        
        //////////////////////////////////////////////////
        
        PPCSVWriterEncoding writerEncoding = KPPCSVWriterEncoding_UTF8;
        
        switch(format)
        {
                // MARK: 發送mail時0x0A會因不明原因被改為0x0D0A的處理
                // gmail原來用UTF16，改為UTF8。
            case WCCSVDataController_Format_Gmail:
                writerEncoding = KPPCSVWriterEncoding_UTF8;
                break;
                
            case WCCSVDataController_Format_UTF8:
            case WCCSVDataController_Format_iWorks:
                writerEncoding = KPPCSVWriterEncoding_UTF8;
                break;
                
            case WCCSVDataController_Format_UTF16:
                writerEncoding = KPPCSVWriterEncoding_UTF16;
                break;
                
            case WCCSVDataController_Format_Outlook:
            case WCCSVDataController_Format_WindowsExcel:
                writerEncoding = KPPCSVWriterEncoding_ANSI;
                break;
                
            default:
                break;
        }
        
        csvWriter = [[PPCSVWriter alloc] initWithDelegate:self
                                      andDelegateUserData:nil
                                              andFilePath:filePath
                                              andEncoding:writerEncoding
                                            willSendByEmail:willSendByEmail];
        
        if(csvWriter == nil)
        {
            returnError = PPErrorMake(WCCSVDataController_Error_FailedToInitObject, @"(PPCSVWriter)", nil);
            break;
        }
        
        result = [csvWriter startWriting];
        
        if(result == NO)
        {
            returnError = PPErrorMake(WCCSVDataController_Error_FailedToWriteCardInfo, nil, nil);
        }
    }
    while(0);
    
    [csvWriter release];
    
    //////////////////////////////////////////////////
    
    // 失敗時把檔案刪除
    if(result == NO)
    {
        [[NSFileManager defaultManager] removeItemAtPath:filePath error:nil];
    }
    else if(self.writtenCount == 0)
    {
        [[NSFileManager defaultManager] removeItemAtPath:filePath error:nil];
        result = NO;
        returnError = PPErrorMake(WCCSVDataController_Error_NoCardToWrite, nil, nil);
    }
    
    if(error != nil)
    {
        *error = returnError;
    }
    
    return result;
}


//================================================================================
//
//================================================================================
- (BOOL)readFileWithPaths:(NSArray *)filePaths
                    error:(NSError **)error
{
    BOOL result = NO;
    NSError *returnError = nil;
    
    do
    {
        NSUInteger totalCount = [filePaths count];
        
        if(totalCount<=0)
        {
            returnError = PPErrorMake(WCCSVDataController_Error_NoFileToRead, @"(PPCSVWriter)", nil);
            break;
        }
        
        //////////////////////////////////////////////////
        
        // 一開始先檢查delegate的目的是減少檢查次數，同時提示此流程會使用的delegate method。
        
        if([self.delegate respondsToSelector:@selector(csvDataContrller:sourceRowEntryDataArray:csvDataArray:withPath:)] == NO ||
           [self.delegate respondsToSelector:@selector(csvDataContrller:didReadFileWithIndex:totalCount:)] == NO)
        {
            returnError = PPErrorMake(WCCSVDataController_Error_MustHaveDelegateMethod, nil, nil);
            break;
        }
        
        //////////////////////////////////////////////////

        self.readFilePaths = filePaths;
        
        //////////////////////////////////////////////////
        
        //給定可以接受的格式
        NSArray *encodings = @[@(NSUTF8StringEncoding),
                               @([NSString defaultCStringEncoding]),
                               @(NSUTF16StringEncoding),
                               @(NSShiftJISStringEncoding),
                               ];
  
        for(NSString *filePath in self.readFilePaths)
        {
            NSData *fileData = nil;
   
            if([filePath hasPrefix:@"file://"]==YES)
            {
                if([[NSURL URLWithString:filePath] checkResourceIsReachableAndReturnError:&returnError]==NO)
                {
                    returnError = PPErrorMake(WCCSVDataController_Error_NoFileToRead, nil, nil);
                    break;
                }
                
                fileData = [[NSData alloc] initWithContentsOfURL:[NSURL URLWithString:filePath]];
            }
            else
            {
                if([[NSFileManager defaultManager] fileExistsAtPath:filePath]==NO)
                {
                    returnError = PPErrorMake(WCCSVDataController_Error_NoFileToRead, nil, nil);
                    break;
                }
                
                fileData = [[NSData alloc] initWithContentsOfFile:filePath];
            }
            
            //////////////////////////////////////////////////
           
            if(fileData==nil)
            {
                returnError = PPErrorMake(WCCSVDataController_Error_FailedToReadFileInfo, nil, nil);
                break;
            }
            
            //////////////////////////////////////////////////

            NSString *allPersonString = nil;
          
            for(NSNumber *encodingNumber in encodings)
            {
                //無法預期是何種格式的 CSV，所以先給能接收的編碼，在一種一種編號檢查
                allPersonString = [[[NSString alloc] initWithData:fileData
                                                         encoding:[encodingNumber integerValue]] autorelease];
                
                if(allPersonString!=nil)
                {
                    break;
                }
            }
            
            [fileData release];
        
            //////////////////////////////////////////////////

            //!! Howard replace all /" string, 否則讀取Outlook資料，會解析錯誤
            allPersonString = [allPersonString stringByReplacingOccurrencesOfString:@"\"" withString:@""];
            
            //////////////////////////////////////////////////

            //!! 開始解析 CSV 檔案字串格式
            NSArray *separatePersons = [allPersonString componentsSeparatedByString:@"\n"];
            
            // !! 有些資料是用 \r 來分開
            if([separatePersons count]==1)
            {
                separatePersons = [allPersonString componentsSeparatedByString:@"\r"];
            }
  
            if([separatePersons count]<=0)
            {
                returnError = PPErrorMake(WCCSVDataController_Error_FailedToParseFileInfo, nil, nil);
                break;
            }
            
            //////////////////////////////////////////////////
            
            NSMutableArray *eveyrPersonRowArray = [PPCSVReader curCSVHandleStringsBasedOnEnterChar:separatePersons];

            if([eveyrPersonRowArray count]<=0)
            {
                returnError = PPErrorMake(WCCSVDataController_Error_FailedToParseFileInfo, nil, nil);
                break;
            }
            
            //////////////////////////////////////////////////

            NSMutableArray *decodingResultArray = [NSMutableArray array];
            
            if(decodingResultArray==nil)
            {
                returnError = PPErrorMake(WCCSVDataController_Error_FailedToInitObject, nil, nil);
                break;
            }
    
            //////////////////////////////////////////////////
            
            for(NSUInteger index=0; index<[eveyrPersonRowArray count]; index++)
            {
                NSString *onePeopleRowString = [eveyrPersonRowArray objectAtIndex:index];
                
                if(index==[eveyrPersonRowArray count]-1 &&
                   [onePeopleRowString length]==0)
                {
                    continue;
                }
                
                NSArray *fieldStrings = [onePeopleRowString componentsSeparatedByString:@","];
                
                NSMutableArray *oneFieldString = [PPCSVReader curCSVHandleStringsBasedOnCommaChar:fieldStrings];
                
                [decodingResultArray addObject:oneFieldString];
            }
            
            //////////////////////////////////////////////////

            [self.delegate csvDataContrller:self
                    sourceRowEntryDataArray:eveyrPersonRowArray
                               csvDataArray:decodingResultArray
                                   withPath:filePath];
            
            //////////////////////////////////////////////////

            [self.delegate csvDataContrller:self
                       didReadFileWithIndex:[self.readFilePaths indexOfObject:filePath]
                                 totalCount:self.readFilePaths.count];
        }

        //////////////////////////////////////////////////

        if(returnError==nil)
        {
            result = YES;
        }
    }
    while (0);

    //////////////////////////////////////////////////

    if(error!=nil)
    {
        *error = returnError;
    }

    //////////////////////////////////////////////////

    return result;
}


//================================================================================
// 
//================================================================================
- (WCCardModel *)cardModelFromTypesArray:(NSArray *)typesArray
                        valueStringArray:(NSArray *)valueStringArray
                                   error:(NSError **)error
{
    NSError *returnError = nil;
    
    WCCardModel *cardModel = nil;
    
    do
    {
        if([typesArray count]<=0 ||
           [valueStringArray count]<=0 ||
           [typesArray count]!=[valueStringArray count])
        {
            returnError = PPErrorMake(WCCSVDataController_Error_FailedToTransformCardModel, nil, nil);
            
            break;
        }
    
        //////////////////////////////////////////////////

        cardModel = [[WCCardModel alloc] init];
        
        if(cardModel==nil)
        {
            returnError = PPErrorMake(WCCSVDataController_Error_FailedToInitObject, nil, nil);
            
            break;
        }
        
        //////////////////////////////////////////////////

        for(NSNumber *typeNumber in typesArray)
        {
            WC_FieldType fieldType = [self fieldTypeFromSourceType:[typeNumber integerValue]];
            
            if(fieldType==WC_FT_None)
            {
                continue;
            }
            
            //////////////////////////////////////////////////

            NSString *valueString = [valueStringArray objectAtIndex:[typesArray indexOfObject:typeNumber]];
            
            if([valueString length]<=0)
            {
                continue;
            }
            
            //////////////////////////////////////////////////

            if(fieldType==WC_FT_Name ||
               fieldType==WC_FT_Company ||
               fieldType==WC_FT_Address)
            {
                WC_FieldSubType2 subType2 = [typeNumber integerValue];
                
                //////////////////////////////////////////////////

                //如果沒有指定subType2，則給預設
                if([typeNumber integerValue]==WC_FST2_None)
                {
                    switch (fieldType)
                    {
                        case WC_FT_Name:
                        {
                            subType2 = WC_FST2_Name_First;
                            break;
                        }
                        case WC_FT_Address:
                        {
                            subType2 = WC_FST2_Address_Street;
                            break;
                        }
                        case WC_FT_Company:
                        {
                            subType2 = WC_FST2_Company_Name;
                            break;
                        }
                        default:
                        {
                            break;
                        }
                    }
                }
                
                //////////////////////////////////////////////////

                WCFieldModel *fieldModel = [[cardModel fieldArrayWithType:fieldType] lastObject];
                
                if(fieldModel==nil ||
                   [fieldModel hasFieldWithSubType2:subType2]==YES)
                {
                    fieldModel = [[[WCFieldModel alloc] init] autorelease];
                    
                    if(fieldModel==nil)
                    {
                        returnError = PPErrorMake(WCCSVDataController_Error_FailedToInitObject, nil, nil);
                        break;
                    }

                    fieldModel.type = fieldType;
       
                    [cardModel addField:fieldModel];
                }

                //////////////////////////////////////////////////
                
                [fieldModel setSubType2FieldWithValue:valueString
                                            recogRect:CGRectZero
                                             subType2:subType2];
            }
            else if([typeNumber integerValue]==WCCSV_FieldType_CateogryID)
            {
                // TODO: CategoryID 匯入
//                NSMutableArray *groupIDArray = [NSMutableArray array];
//                
//                if(groupIDArray==nil)
//                {
//                    returnError = PPErrorMake(WCCSVDataController_Error_FailedToInitObject, nil, nil);
//                    break;
//                }
//                
//                //////////////////////////////////////////////////
//
//                if([cardModel.groupIDArray count]>0)
//                {
//                    [groupIDArray addObjectsFromArray:cardModel.groupIDArray];
//                }
            }
            else if([typeNumber integerValue]==WCCSV_FieldType_CreateTime)
            {
                cardModel.createdTime = [self convertDateFromDateString:valueString];
            }
            else if([typeNumber integerValue]==WCCSV_FieldType_ModifyTime)
            {
                 cardModel.modifiedTime = [self convertDateFromDateString:valueString];
            }
            else if([typeNumber integerValue]==WC_FST1_Date_Other ||
                    [typeNumber integerValue]==WC_FST1_Date_Anniversary ||
                    [typeNumber integerValue]==WC_FST1_Date_Birthday)
            {
                WCFieldModel *fieldModel = [[WCFieldModel alloc] init];
                
                if(fieldModel==nil)
                {
                    returnError = PPErrorMake(WCCSVDataController_Error_FailedToInitObject, nil, nil);
                    break;
                }
                
                fieldModel.type = fieldType;
                fieldModel.subType1 = [typeNumber integerValue];
                fieldModel.value = [self convertDateFromDateString:valueString];
                
                [cardModel addField:fieldModel];
                
                [fieldModel release];
            }
            else
            {
                WCFieldModel *fieldModel = [[WCFieldModel alloc] init];
                
                if(fieldModel==nil)
                {
                    returnError = PPErrorMake(WCCSVDataController_Error_FailedToInitObject, nil, nil);
                    break;
                }
                
                fieldModel.type = fieldType;
                fieldModel.subType1 = [typeNumber integerValue];
                fieldModel.value = valueString;
                
                [cardModel addField:fieldModel];
                
                [fieldModel release];
            }
    
        }
    }
    while (0);
    
    if(error!=nil)
    {
        *error = returnError;
    }
    
    //////////////////////////////////////////////////

    return [cardModel autorelease];
}
@end
