//
//  WCBackupRestoreFlowController.m
//  
//

#import "WCBackupRestoreFlowController.h"
#import "WCBackupRestoreFlowController+ResourceDefine.h"
#import "WCBackupRestoreFlowController+SettingsKey.h"
#import "WCToolController.h"
#import "BRFCMainViewController.h"
#import "PPStorageSpaceSelectViewController.h"
#import "PPStorageSpaceSelectViewControllerDelegate.h"
#import "PPStorageFileSelectViewController.h"
#import "PPNavigationController.h"
#import "PPAlertView.h"
#import "PPBusyView.h"
#import "PPRowModel.h"
#import "PPSettingsController.h"
#import "PPZipController.h"
#import "NSError+Custom.h"
#import "PPBackgroundTaskController.h"
#import "WCAppearanceDefine.h"
#import "UIApplication+Idle.h"
#import "NSObject+PPBusyView.h"

#import "WCCloudFileDefine.h"
#import "PPCloudController.h"
#import "PPCloud_Dropbox.h"
#import "PPCloud_GoogleDrive.h"
#import "PPCloud_iCloud.h"

#import "WCXFDataController.h"

#if defined (PRODUCTLINE_WCT)

#import "WCTDataController.h"

#define DataController WCTDataController

#elif defined (PRODUCTLINE_WCM)

#import "WCDataController.h"

#define DataController WCDataController

#endif

////////////////////////////////////////////////////////////////////////////////////////////////////

WCBackupRestoreFlowController *g_backupRestoreFlowController = nil;

////////////////////////////////////////////////////////////////////////////////////////////////////

typedef NS_ENUM(NSInteger, WCBackupRestoreFlowControllerTag)
{
    WCBackupRestoreFlowControllerTag_FinishWithError = 0,
    WCBackupRestoreFlowControllerTag_FinishFlow,
    WCBackupRestoreFlowControllerTag_SelectBackupType,
    WCBackupRestoreFlowControllerTag_SelectRestoreType,
};

double const WCBackupRestoreFlowController_NotAvailable = -1;
NSString * const WCBackupRestoreFlowController_TempFilePath = @"WCBackupRestoreFlowController_TempFilePath";


////////////////////////////////////////////////////////////////////////////////////////////////////

@interface WCBackupRestoreFlowController () <PPStorageSpaceSelectViewControllerDelegate,
                                             PPStorageFileSelectViewControllerDelegate,
                                             UIAlertViewDelegate,
                                             BRFCMainViewControllerDelegate,
                                             PPCloudDelegate_LoadAccountInfo,
                                             PPCloudDelegate_CreateFolder,
                                             PPCloudDelegate_UploadFile,
                                             PPCloudDelegate_LoadFile,
                                             WCXFDataControllerDelegate>

@property (nonatomic, retain) PPNavigationController *navigationController;
@property (nonatomic, retain) BRFCMainViewController *brfcMainViewController;
@property (nonatomic, retain) NSDictionary *cloudKeyDict;
@property (nonatomic, retain) NSString *baseStoreDirPath;
@property (nonatomic, retain) NSArray *excludeDirPatterns;
@property (nonatomic, copy) BRFCHandleDataBlock handleRestoreDataBlock;
@property (nonatomic, copy) BRFCCompleteBlock flowCompleteBlock;
@property (nonatomic, assign) PPStorageSpaceSelectControllerSpaceType storageSpaceType;

@property (nonatomic, assign) WCBackupRestoreFlowController_Action action;
@property (atomic, retain) NSNumber *requestResponse;
@property (nonatomic, retain) NSError *cloudError;
@property (nonatomic, retain) NSError *lastError;
@property (nonatomic, assign) BOOL showSpaceInfo;
@property (nonatomic, assign) double spaceTotalBytes;
@property (nonatomic, assign) double spaceFreeBytes;

#if defined (PRODUCTLINE_WCT) || defined (PRODUCTLINE_WCM)
@property (nonatomic, retain) DataController *dataController;
#endif

//test
//@property (nonatomic, retain) NSString *testDownloadPath;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation WCBackupRestoreFlowController

////////////////////////////////////////////////////////////////////////////////////////////////////


//================================================================================
//
//================================================================================
- (instancetype)init
{
    if(self = [super init])
    {
        self.spaceTotalBytes = WCBackupRestoreFlowController_NotAvailable;
        self.spaceFreeBytes = WCBackupRestoreFlowController_NotAvailable;
        self.showSpaceInfo = YES;
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    self.navigationController = nil;
    self.brfcMainViewController = nil;
    self.cloudKeyDict = nil;
    self.baseStoreDirPath = nil;
    self.excludeDirPatterns = nil;
    self.handleRestoreDataBlock = nil;
    self.flowCompleteBlock = nil;
    
    self.requestResponse = nil;
    
    self.cloudError = nil;
    self.lastError = nil;

#if defined (PRODUCTLINE_WCT) || defined (PRODUCTLINE_WCM)
    self.dataController = nil;
#endif
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - private methods


//==============================================================================
//
//==============================================================================
- (void)showBackupTypeSelectAlertWithViewCotnroller:(BRFCMainViewController *)viewController
{
    self.brfcMainViewController = viewController;
    
    [PPAlertView showWithStyle:UIAlertViewStyleDefault
                         title:@""
                       message:WCBRFC_MLS_SelectBackupFile
                      delegate:self
                           tag:WCBackupRestoreFlowControllerTag_SelectBackupType
             cancelButtonTitle:WCBRFC_MLS_Cancel
             otherButtonTitles:WCBRFC_MLS_FileType_IWC, WCBRFC_MLS_FileType_WCBK, nil];
}


//==============================================================================
//
//==============================================================================
- (void)showRestoreTypeSelectAlertWithViewCotnroller:(BRFCMainViewController *)viewController
{
    self.brfcMainViewController = viewController;
    
    [PPAlertView showWithStyle:UIAlertViewStyleDefault
                         title:@""
                       message:WCBRFC_MLS_SelectBackupFile
                      delegate:self
                           tag:WCBackupRestoreFlowControllerTag_SelectRestoreType
             cancelButtonTitle:WCBRFC_MLS_Cancel
             otherButtonTitles:WCBRFC_MLS_FileType_IWC, WCBRFC_MLS_FileType_WCBK, nil];
}


//==============================================================================
//
//==============================================================================
- (void)showRestoreFileSelectWithFileType:(NSString *)fileType
{
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_main_queue(), ^{
        
        PPStorageFileSelectViewControllerSpaceType storageFileSpaceType;
        NSString *backupDirPath = [self iwcBackupRestoreDir];
        
#if defined (PRODUCTLINE_WCT) || defined (PRODUCTLINE_WCM)
        if ([fileType isEqualToString:WCBRFC_MLS_FileType_WCBK])
        {
            backupDirPath = [self wcbkBackupRestoreDir];;
        }
#endif
        switch (self.storageSpaceType)
        {
                case PPStorageSpaceSelectControllerSpaceType_iTune:
                storageFileSpaceType = PPStorageFileSelectViewControllerSpaceType_iTune;
                
                // !! iTune直接放到documents目錄下
                backupDirPath = @"";
                break;
                
                case PPStorageSpaceSelectControllerSpaceType_iCloud:
                storageFileSpaceType = PPStorageFileSelectViewControllerSpaceType_iCloud;
                break;
                
                case PPStorageSpaceSelectControllerSpaceType_Dropbox:
                storageFileSpaceType = PPStorageFileSelectViewControllerSpaceType_Dropbox;
                break;
                
                case PPStorageSpaceSelectControllerSpaceType_GoogleDrive:
                storageFileSpaceType = PPStorageFileSelectViewControllerSpaceType_GoogleDrive;
                break;
                
                case PPStorageSpaceSelectControllerSpaceType_OneDrive:
                storageFileSpaceType = PPStorageFileSelectViewControllerSpaceType_OneDrive;
                break;
                
                case PPStorageSpaceSelectControllerSpaceType_Huawei:
                storageFileSpaceType = PPStorageFileSelectViewControllerSpaceType_Huawei;
                
                // !! 華為拿掉"WorldCard"
                NSMutableArray *components = [NSMutableArray arrayWithArray:[backupDirPath pathComponents]];
                [components removeObjectAtIndex:1];
                backupDirPath = [NSString pathWithComponents:components];
                break;
                
                case PPStorageSpaceSelectControllerSpaceType_Baidu:
                storageFileSpaceType = PPStorageFileSelectViewControllerSpaceType_Baidu;
                break;
                
            default:
                return;
        }
        
        //////////////////////////////////////////////////
        
        PPStorageFileSelectViewController *storageFileSelectViewController = [[PPStorageFileSelectViewController alloc] init];
        
        if(storageFileSelectViewController!=nil)
        {
            storageFileSelectViewController.delegate = self;
            storageFileSelectViewController.storageFileSelectViewControllerSpaceType = storageFileSpaceType;
            

#if defined (PRODUCTLINE_WCT) || defined (PRODUCTLINE_WCM)
            if ([fileType isEqualToString:WCBRFC_MLS_FileType_IWC])
            {
                storageFileSelectViewController.extensionFileNameArray = @[WCCloudFileDefine_iPhoneBackupFileExt];
                storageFileSelectViewController.sectionTitleDictionary = @{WCCloudFileDefine_iPhoneBackupFileExt:WCBRFC_MLS_FileType_IWC};
            }
            else
            {
                storageFileSelectViewController.extensionFileNameArray = @[WCCloudFileDefine_GeneralBackupFileExt];
                storageFileSelectViewController.sectionTitleDictionary = @{WCCloudFileDefine_GeneralBackupFileExt:WCBRFC_MLS_FileType_WCBK};
            }
#else
            storageFileSelectViewController.extensionFileNameArray = @[WCCloudFileDefine_iPhoneBackupFileExt];
            
#endif
            storageFileSelectViewController.defaultRootFilePath = backupDirPath;
            storageFileSelectViewController.controllerTitle = WCBRFC_MLS_BackupRestore;
            storageFileSelectViewController.headerTitleLabelText = WCBRFC_MLS_SelectBackupFile;
            storageFileSelectViewController.multiSelection = NO;
            storageFileSelectViewController.hidenBackButton = NO;
            storageFileSelectViewController.autoDownload = NO;
            storageFileSelectViewController.navigationViewButtonHighlightedBackgroundColor = WCAppearanceDefine_ButtonBackgroundColor;
            storageFileSelectViewController.navigationViewButtonImageEdgeInsets = WCAppearanceDefine_ButtonImageEdgeInset;
            
            [blockSelf.navigationController pushViewController:storageFileSelectViewController animated:YES];
            [storageFileSelectViewController release];
        }
    });
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - UIAlertViewDelegate methods

//================================================================================
//
//================================================================================
- (void)alertView:(UIAlertView *)alertView didDismissWithButtonIndex:(NSInteger)buttonIndex
{
    switch (alertView.tag)
    {
        case WCBackupRestoreFlowControllerTag_FinishFlow:
        {
            [self finishFlowWithAction:self.action error:self.lastError];
            break;
        }
        case WCBackupRestoreFlowControllerTag_SelectRestoreType:
        {
            if(buttonIndex!=alertView.cancelButtonIndex)
            {
                NSString *buttonTitle = [alertView buttonTitleAtIndex:buttonIndex];
                [self showRestoreFileSelectWithFileType:buttonTitle];
            }
            break;
        }
        case WCBackupRestoreFlowControllerTag_SelectBackupType:
        {
            NSString *buttonTitle = [alertView buttonTitleAtIndex:buttonIndex];
            
            if([buttonTitle isEqualToString:WCBRFC_MLS_FileType_IWC])
            {
                [self backupIWCWithMainViewController:self.brfcMainViewController];
            }
            else if([buttonTitle isEqualToString:WCBRFC_MLS_FileType_WCBK])
            {
#if defined (PRODUCTLINE_WCT) || defined (PRODUCTLINE_WCM)
                [self backupWCBKWithMainViewController:self.brfcMainViewController];
#endif
            }
 
            break;
        }
        default:
            break;
    }
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - BRFCMainViewControllerDelegate methods

//================================================================================
//
//================================================================================
- (void)mainViewControllerRequestBackupInfo:(BRFCMainViewController *)mainViewController
{
    NSString *backupInfo = [WCBackupRestoreFlowController readBackupInfoFromSettingsWithSapceType:self.storageSpaceType];
    
    [mainViewController updateBackupInfo:backupInfo];
}


//================================================================================
//
//================================================================================
- (void)mainViewControllerRequestSapceInfo:(BRFCMainViewController *)mainViewController
{
    [self requestSpaceInfoWithMainViewController:mainViewController];
}


//================================================================================
//
//================================================================================
- (void)mainViewControllerDidClickBackup:(BRFCMainViewController *)mainViewController
{
    self.action = WCBackupRestoreFlowController_Action_Backup;
    self.lastError = nil;
    
    //////////////////////////////////////////////////
#if defined (PRODUCTLINE_WCT) || defined (PRODUCTLINE_WCM)
    [self showBackupTypeSelectAlertWithViewCotnroller:mainViewController];

    // NEXT:alertView:didDismissWithButtonIndex:
    
#else

    [self backupIWCWithMainViewController:mainViewController];

#endif
}


//================================================================================
//
//================================================================================
- (void)mainViewControllerDidClickRestore:(BRFCMainViewController *)mainViewController
{
    self.action = WCBackupRestoreFlowController_Action_Restore;
    self.lastError = nil;
    
    //////////////////////////////////////////////////
#if defined (PRODUCTLINE_WCT) || defined (PRODUCTLINE_WCM)
    [self showRestoreTypeSelectAlertWithViewCotnroller:mainViewController];
    
    // NEXT:alertView:didDismissWithButtonIndex:
    
#else
    
    [self backupIWCWithMainViewController:mainViewController];
    
#endif

}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPStorageFileSelectViewControllerDelegate methods

//================================================================================
//
//================================================================================
- (void)storageFileSelectViewController:(PPStorageFileSelectViewController *)storageFileSelectViewController
                   didSelectedFileArray:(NSArray *)selectedFileArray
{   
    PPStorageFielSelectRowModel *rowModel = [selectedFileArray lastObject];
    
    
    // 還原時會有兩種檔案格式
    NSString *srcFilePath = rowModel.object;
    
    if ([[srcFilePath pathExtension] compare:WCCloudFileDefine_iPhoneBackupFileExt options:NSCaseInsensitiveSearch]==NSOrderedSame)
    {
        [self restoreIWCWithSrcFilePath:srcFilePath];
    }
#if defined (PRODUCTLINE_WCT) || defined (PRODUCTLINE_WCM)
    else if ([[srcFilePath pathExtension] compare:WCCloudFileDefine_GeneralBackupFileExt options:NSCaseInsensitiveSearch]==NSOrderedSame)
    {
        [self restoreWCBKWithSrcFilePath:srcFilePath];
    }
#endif
    
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPStorageSpaceSelectViewControllerDelegate methods

//================================================================================
//
//================================================================================
- (void)storageSpaceSelectViewController:(PPStorageSpaceSelectViewController *)storageSpaceSelectViewController
didStorageSpaceSelectControllerSpaceType:(PPStorageSpaceSelectControllerSpaceType)storageSpaceSelectControllerSpaceType
{
    self.storageSpaceType = storageSpaceSelectControllerSpaceType;
    
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_main_queue(), ^{
        
        BRFCMainViewController *mainViewController = [[BRFCMainViewController alloc] initWithDelegate:blockSelf
                                                                                            spaceType:storageSpaceSelectControllerSpaceType];
        
        if(mainViewController != nil)
        {
            [blockSelf.navigationController pushViewController:mainViewController animated:YES];
            [mainViewController release];
        }
    });
}


//================================================================================
//
//================================================================================
- (void)storageSpaceSelectViewControllerDidCancel:(PPStorageSpaceSelectViewController *)storageSpaceSelectViewController
{
    [self finishFlowWithAction:WCBackupRestoreFlowController_Action_None error:self.lastError];
}


//================================================================================
//
//================================================================================
- (BOOL)storageSpaceSelectViewControllerShowAlertWithError:(NSError *)error
{
    if(error.code == PPCloudCommonError_CancelLogin)
    {
        return NO;
    }
    
    return YES;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPCloudDelegate_LoadAccountInfo methods

//================================================================================
//
//================================================================================
- (void)ppCloud:(PPCloud *)ppCloud loadAccountInfoFailedWithError:(NSError *)error userInfo:(id)userInfo
{
    self.cloudError = error;
    self.requestResponse = @(NO);
}


//================================================================================
//
//================================================================================
- (void)ppCloud:(PPCloud *)ppCloud loadedAccountInfo:(PPCloudAccountInfo *)accountInfo userInfo:(id)userInfo
{
    if (accountInfo.totalBytes!=nil)
    {
        self.spaceTotalBytes = [accountInfo.totalBytes doubleValue];
        self.spaceFreeBytes = self.spaceTotalBytes - [accountInfo.usedBytes doubleValue];
    }
    else
    {
        self.spaceTotalBytes = WCBackupRestoreFlowController_NotAvailable;
        self.spaceFreeBytes = WCBackupRestoreFlowController_NotAvailable;
        self.showSpaceInfo = NO;
    }
        
    self.cloudError = nil;
    self.requestResponse = @(YES);
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPCloudDelegate_CreateFolder methods

//================================================================================
//
//================================================================================
- (void)ppCloud:(PPCloud *)ppCloud createFolderFailedWithError:(NSError *)error userInfo:(id)userInfo
{
    // !! 因為folder存在也會當作錯誤，且真的無法建立時下一步一樣會回傳錯誤，所以這裡都當作成功處理。
    self.cloudError = nil;
    self.requestResponse = @(YES);
}


//================================================================================
//
//================================================================================
- (void)ppCloud:(PPCloud *)ppCloud createdFolder:(PPCloudMetadata *)folder userInfo:(id)userInfo
{
    self.cloudError = nil;
    self.requestResponse = @(YES);
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPCloudDelegate_UploadFile methods

//================================================================================
//
//================================================================================
- (void)ppCloud:(PPCloud *)ppCloud uploadFileFailedWithError:(NSError *)error userInfo:(id)userInfo
{
    self.cloudError = error;
    self.requestResponse = @(NO);
}


//================================================================================
//
//================================================================================
- (void)ppCloud:(PPCloud *)ppCloud uploadedFile:(NSString *)destPath from:(NSString *)srcPath metadata:(PPCloudMetadata *)metadata userInfo:(id)userInfo
{
    self.cloudError = nil;
    self.requestResponse = @(YES);
}


//================================================================================
//
//================================================================================
- (void)ppCloud:(PPCloud *)ppCloud uploadProgress:(CGFloat)progress forFile:(NSString *)destPath from:(NSString *)srcPath userInfo:(id)userInfo
{
    NSString *message = [NSString stringWithFormat:@"%@ %d%%", WCBRFC_MLS_Uploading, (int)(progress*100)];
    [PPBusyView postMessage:message];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - PPCloudDelegate_LoadFile methods

//================================================================================
//
//================================================================================
- (void)ppCloud:(PPCloud *)ppCloud loadFileFailedWithError:(NSError *)error userInfo:(id)userInfo
{
    self.cloudError = error;
    self.requestResponse = @(NO);
}


//================================================================================
//
//================================================================================
- (void)ppCloud:(PPCloud *)ppCloud loadedFile:(NSString *)destPath userInfo:(id)userInfo
{
    self.cloudError = nil;
    self.requestResponse = @(YES);
}


//================================================================================
//
//================================================================================
- (void)ppCloud:(PPCloud *)ppCloud loadProgress:(CGFloat)progress forFile:(NSString *)destPath userInfo:(id)userInfo
{
    NSString *message = [NSString stringWithFormat:@"%@ %d%%", WCBRFC_MLS_Downloading, (int)(progress*100)];
    [PPBusyView postMessage:message];
}





#if defined (PRODUCTLINE_WCT) || defined (PRODUCTLINE_WCM)





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - WCXFDataControllerDelegate (Read)


//==============================================================================
//
//==============================================================================
- (BOOL)wcxfDataContrller:(WCXFDataController *)wcxfDataController
  shouldReadCardWithIndex:(NSInteger)index
               totalCount:(NSInteger)totalCount
{
    if(self.lastError!=nil)
    {
        return NO;
    }
    else
    {
        return YES;
    }
}


//==============================================================================
//
//==============================================================================
- (void)wcxfDataContrller:(WCXFDataController *)wcxfDataController
            readCardModel:(WCCardModel *)cardModel
{
    [self.dataController addCard:cardModel withSendNotification:NO];
}


//==============================================================================
//
//==============================================================================
- (void)wcxfDataContrller:(WCXFDataController *)wcxfDataController
            readImageData:(NSData *)imageData
                imageType:(WC_ImageType)imageType
                   cardID:(NSString *)cardID
{
    [self.dataController setCardImageData:imageData withCardID:cardID type:imageType isImport:NO];
}


//==============================================================================
//
//==============================================================================
- (WC_GroupID)wcxfDataContrller:(WCXFDataController *)wcxfDataController
       groupIDWithReadGroupName:(NSString *)groupName
                          error:(NSError **)error
{
    
    WC_GroupID groupID = WC_GID_None;
    
    if([self.dataController isGroupNameExist:groupName]==YES)
    {
        NSString *groupIDString = [self.dataController groupIDWithName:groupName];
        if([groupIDString length]>0)
        {
            groupID = [groupIDString integerValue];
        }
    }
    else
    {
        WCGroupModel *groupModel = nil;
        groupModel = [self.dataController newGroupWithName:groupName];
        
        if(groupModel!=nil)
        {
            groupID = groupModel.ID;
        }
        
        [groupModel release];
    }
    //////////////////////////////////////////////////
    
    return groupID;
}


//==============================================================================
//
//==============================================================================
- (void)wcxfDataContrller:(WCXFDataController *)wcxfDataController
     didReadCardWithIndex:(NSInteger)index
               totalCount:(NSInteger)totalCount
{
    NSString *message = [NSString stringWithFormat:@"%@ %d%%", WCBRFC_MLS_GeneratingData, (int)(((CGFloat)index/(CGFloat)totalCount)*100)];
    [PPBusyView postMessage:message];
}







////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - WCXFDataControllerDelegate (Write)


//==============================================================================
//
//==============================================================================
- (BOOL)wcxfDataContrller:(WCXFDataController *)wcxfDataController
 shouldWriteCardWithIndex:(NSInteger)index
               totalCount:(NSInteger)totalCount
{
    return YES;
}


//==============================================================================
//
//==============================================================================
- (WCCardModel *)wcxfDataContrller:(WCXFDataController *)wcxfDataController
        cardModelToWriteWithCardID:(NSString *)cardID
{
    WCCardModel *cardModel = [self.dataController copyCardFullDataWithCardID:cardID];
    return [cardModel autorelease];
}


//==============================================================================
//
//==============================================================================
- (NSData *)wcxfDataContrller:(WCXFDataController *)wcxfDataController
   imageDataToWriteWithCardID:(NSString *)cardID
                    imageType:(WC_ImageType)imageType
{
    NSData *imageData = [self.dataController copyCardImageDataWithCardID:cardID type:imageType subtype:WC_IST_Original];
    return [imageData autorelease];
}


//==============================================================================
//
//==============================================================================
- (NSString *)wcxfDataContrller:(WCXFDataController *)wcxfDataController
    groupNameToWriteWithGroupID:(WC_GroupID)groupID
{
    return [self.dataController groupNameWithID:[NSString stringWithInteger:groupID]];
}


//==============================================================================
//
//==============================================================================
- (void)wcxfDataContrller:(WCXFDataController *)wcxfDataController
    didWriteCardWithIndex:(NSInteger)index
               totalCount:(NSInteger)totalCount
{
    NSString *message = [NSString stringWithFormat:@"%@ %d%%", WCBRFC_MLS_GeneratingData, (int)(((CGFloat)index/(CGFloat)totalCount)*100)];
    [PPBusyView postMessage:message];
}

#endif






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private Class methods (Settings)

//================================================================================
//
//================================================================================
+ (NSString *)readBackupInfoFromSettingsWithSapceType:(PPStorageSpaceSelectControllerSpaceType)spaceType
{
    NSString *backupInfo = nil;
    NSData *data = [PPSettingsController dataValueWithKey:WCBackupRestoreFlowController_SettingsKey_LastBackupInfo];
    
    if(data != nil)
    {
        NSDictionary *dict = [NSPropertyListSerialization propertyListWithData:data
                                                                       options:kNilOptions
                                                                        format:NULL
                                                                         error:NULL];
        
        backupInfo = [dict objectForKey:[NSString stringWithFormat:@"%d", (int)spaceType]];
    }
    
    return backupInfo;
}


//================================================================================
//
//================================================================================
+ (void)writeBackupInfoToSettings:(NSString *)backupInfo withSapceType:(PPStorageSpaceSelectControllerSpaceType)spaceType
{
    if([backupInfo length] == 0)
    {
        return;
    }
    
    NSData *data = [PPSettingsController dataValueWithKey:WCBackupRestoreFlowController_SettingsKey_LastBackupInfo];
    NSMutableDictionary *dict = nil;
    
    if(data != nil)
    {
        dict = [NSPropertyListSerialization propertyListWithData:data
                                                         options:kNilOptions
                                                          format:NULL
                                                           error:NULL];
    }
    
    if(dict == nil)
    {
        dict = [NSMutableDictionary dictionary];
    }
    
    [dict setObject:backupInfo forKey:[NSString stringWithFormat:@"%d", (int)spaceType]];
    
    data = [NSPropertyListSerialization dataWithPropertyList:dict
                                                      format:NSPropertyListXMLFormat_v1_0
                                                     options:kNilOptions
                                                       error:NULL];

    [PPSettingsController setDataValue:data withKey:WCBackupRestoreFlowController_SettingsKey_LastBackupInfo];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods (Format)

//================================================================================
//
//================================================================================
- (NSString *)sizeTextWithSize:(double)size
{
    if(size == 0)
    {
        return nil;
    }
    
    CGFloat divisor = 1024.0;
    NSString *unit = @"bytes";
    
    if(size/divisor > 1.0)
    {
        size /= divisor;
        unit = @"KB";
        
        if(size/divisor > 1.0)
        {
            size /= divisor;
            unit = @"MB";
            
            if(size/divisor > 1.0)
            {
                size /= divisor;
                unit = @"GB";
            }
        }
    }
    
    return [NSString stringWithFormat:@"%.2f %@", size, unit];
}


//===============================================================================
//
//===============================================================================
- (NSString *)tempDirPath
{
    NSString *tempDir = [NSString stringWithFormat:@"%@/backup", [WCToolController tempFolderPath]];
    
    [WCToolController createDirPath:tempDir];
    
    return tempDir;
}


//===============================================================================
//
//===============================================================================
- (NSString *)backupFileNameWithPathExtension:(NSString *)pathExtension
{
    NSString *dateString = nil;
    NSDateFormatter* dateFormatter = [[NSDateFormatter alloc] init];
    [dateFormatter setDateFormat:WCCloudFileDefine_DateTimeFormat];
    dateString = [dateFormatter stringFromDate:[NSDate date]];
    [dateFormatter release];
    
    // !! 裝置名稱不可有空白
    NSString *deviceName = [[UIDevice currentDevice].name stringByReplacingOccurrencesOfString:@" " withString:@"_"];
    NSString *fileName = [NSString stringWithFormat:WCCloudFileDefine_BackupFileNameFormat, deviceName, dateString, pathExtension];
    
    return fileName;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods (Flow control)

//================================================================================
//
//================================================================================
- (void)finishFlowWithAction:(WCBackupRestoreFlowController_Action)action error:(NSError *)error
{
    if(self.navigationController != nil)
    {
        __block typeof(self) blockSelf = self;
        
        [self.navigationController dismissViewControllerAnimated:YES completion:^{
            
            blockSelf.flowCompleteBlock(action, error);
        }];
    }
    else
    {
        self.flowCompleteBlock(action, error);
    }
}


//================================================================================
//
//================================================================================
- (void)showAlertWithTitle:(NSString *)title message:(NSString *)message
{
    dispatch_async(dispatch_get_main_queue(), ^{
        
        [PPAlertView showWithStyle:UIAlertViewStyleDefault
                             title:title
                           message:message
                 cancelButtonTitle:WCBRFC_MLS_OK
                 otherButtonTitles:nil];
    });
}


//================================================================================
//
//================================================================================
- (void)requestSpaceInfoWithMainViewController:(BRFCMainViewController *)mainViewController
{
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
        
        __block PPBusyView *busyView = nil;
        
        dispatch_sync(dispatch_get_main_queue(), ^{
            
            busyView = [[PPBusyView alloc] initWithSuperView:blockSelf.navigationController.view];
        });
        

        //////////////////////////////////////////////////
        
        blockSelf.requestResponse = nil;
        blockSelf.cloudError = nil;
        blockSelf.spaceTotalBytes = 0;
        blockSelf.spaceFreeBytes = 0;
        
        //////////////////////////////////////////////////

        switch (blockSelf.storageSpaceType)
        {
            case PPStorageSpaceSelectControllerSpaceType_iTune:
            {
                NSArray *paths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
                NSDictionary *dictionary = [[NSFileManager defaultManager] attributesOfFileSystemForPath:[paths lastObject] error:nil];
                
                blockSelf.spaceTotalBytes = [[dictionary objectForKey:NSFileSystemSize] doubleValue];
                blockSelf.spaceFreeBytes = [[dictionary objectForKey:NSFileSystemFreeSize] doubleValue];
                
                blockSelf.requestResponse = @(YES);
                break;
            }
                
            case PPStorageSpaceSelectControllerSpaceType_iCloud:
            {
                // not available
                blockSelf.spaceTotalBytes = WCBackupRestoreFlowController_NotAvailable;
                blockSelf.spaceFreeBytes = WCBackupRestoreFlowController_NotAvailable;
                blockSelf.requestResponse = @(YES);
                break;
            }
                
            case PPStorageSpaceSelectControllerSpaceType_Dropbox:
            {
                [PPCloudController loadAccountInfoWithDelegate:blockSelf
                                                      userInfo:nil
                                                    cloudClass:[PPCloud_Dropbox class]];
                break;
            }
                
            case PPStorageSpaceSelectControllerSpaceType_GoogleDrive:
            {
                [PPCloudController loadAccountInfoWithDelegate:blockSelf
                                                      userInfo:nil
                                                    cloudClass:[PPCloud_GoogleDrive class]];
                break;
            }
              
            case PPStorageSpaceSelectControllerSpaceType_OneDrive:
            {
                [PPCloudController loadAccountInfoWithDelegate:blockSelf
                                                      userInfo:nil
                                                    cloudClass:[PPCloud_OneDrive class]];
                break;
            }
                
            case PPStorageSpaceSelectControllerSpaceType_Huawei:
            {
                // !! 2015/10/01 以後取值有問題先關閉
//                [PPCloudController loadAccountInfoWithDelegate:blockSelf
//                                                      userInfo:nil
//                                                    cloudClass:[PPCloud_Huawei class]];
                
                blockSelf.spaceTotalBytes = WCBackupRestoreFlowController_NotAvailable;
                blockSelf.spaceFreeBytes = WCBackupRestoreFlowController_NotAvailable;
                blockSelf.requestResponse = @(YES);
                
                break;
            }
                
            default:
                break;
        }
        
        //////////////////////////////////////////////////

        while (blockSelf.requestResponse == nil)
        {
            [NSThread sleepForTimeInterval:0.1];
        }
        
        if (blockSelf.showSpaceInfo==NO)
        {
            [mainViewController hideSpaceInfo];
        }
        
        if([blockSelf.requestResponse boolValue] == YES &&
           blockSelf.spaceTotalBytes != WCBackupRestoreFlowController_NotAvailable)
        {
            NSString *totalBytesText = [blockSelf sizeTextWithSize:blockSelf.spaceTotalBytes];
            NSString *freeBytesText = [blockSelf sizeTextWithSize:blockSelf.spaceFreeBytes];
            
            [mainViewController updateSpaceInfoWithTotalBytesText:totalBytesText freeBytesText:freeBytesText];
        }
        
        //////////////////////////////////////////////////

        dispatch_sync(dispatch_get_main_queue(), ^{
            
            [busyView removeFromSuperview];
            [busyView release];
        });
    });
}


//================================================================================
//
//================================================================================
- (NSString *)requestCreateBackupDir:(NSString *)backupDirPath
{
    NSString *dirPath = backupDirPath;
    
    self.requestResponse = nil;
    self.cloudError = nil;

    //////////////////////////////////////////////////
    
    switch (self.storageSpaceType)
    {
        case PPStorageSpaceSelectControllerSpaceType_iTune:
        {
            // !! iTune直接放到documents目錄下
            dirPath = [WCToolController shareFolderPath];
            
            [[NSFileManager defaultManager] createDirectoryAtPath:dirPath withIntermediateDirectories:YES attributes:nil error:nil];
            
            self.requestResponse = @(YES);
            
            break;
        }
            
        case PPStorageSpaceSelectControllerSpaceType_iCloud:
        {
            [PPCloudController createFolder:dirPath
                                   delegate:self
                                   userInfo:nil
                                 cloudClass:[PPCloud_iCloud class]];
            break;
        }
            
        case PPStorageSpaceSelectControllerSpaceType_Dropbox:
        {
            [PPCloudController createFolder:dirPath
                                   delegate:self
                                   userInfo:nil
                                 cloudClass:[PPCloud_Dropbox class]];
            break;
        }
            
        case PPStorageSpaceSelectControllerSpaceType_GoogleDrive:
        {
            [PPCloudController createFolder:dirPath
                                   delegate:self
                                   userInfo:nil
                                 cloudClass:[PPCloud_GoogleDrive class]];
            break;
        }
            
        case PPStorageSpaceSelectControllerSpaceType_OneDrive:
        {
            [PPCloudController createFolder:dirPath
                                   delegate:self
                                   userInfo:nil
                                 cloudClass:[PPCloud_OneDrive class]];
            break;
        }

//        case PPStorageSpaceSelectControllerSpaceType_Huawei:
//        {
//            // !! 華為拿掉"WorldCard"
//            NSMutableArray *components = [NSMutableArray arrayWithArray:[dirPath pathComponents]];
//            [components removeObjectAtIndex:1];
//            dirPath = [NSString pathWithComponents:components];
//
//            [PPCloudController createFolder:dirPath
//                                   delegate:self
//                                   userInfo:nil
//                                 cloudClass:[PPCloud_Huawei class]];
//            break;
//        }
//
        default:
            break;
    }
    
    //////////////////////////////////////////////////
    
    while (self.requestResponse == nil)
    {
        [NSThread sleepForTimeInterval:0.1];
    }
    
    return dirPath;
}


//================================================================================
//
//================================================================================
- (BOOL)requestUploadFile:(NSString *)srcFilePath toDstDirPath:(NSString *)dstDirPath dstFileName:(NSString *)dstFileName error:(NSError **)error
{
    self.requestResponse = nil;
    self.cloudError = nil;
    
    //////////////////////////////////////////////////
    
    switch (self.storageSpaceType)
    {
        case PPStorageSpaceSelectControllerSpaceType_iTune:
        {
            NSString *dstFilePath = [NSString stringWithFormat:@"%@/%@", dstDirPath, dstFileName];
            
            if([[NSFileManager defaultManager] moveItemAtPath:srcFilePath toPath:dstFilePath error:&_cloudError] == YES)
            {
                self.requestResponse = @(YES);
            }
            else
            {
                self.requestResponse = @(NO);
            }
            
            break;
        }
            
        case PPStorageSpaceSelectControllerSpaceType_iCloud:
        {
            [PPCloudController uploadFile:dstFileName
                                   toPath:dstDirPath
                                 fromPath:srcFilePath
                                 delegate:self
                                 userInfo:nil
                               cloudClass:[PPCloud_iCloud class]];
            
            break;
        }
            
        case PPStorageSpaceSelectControllerSpaceType_Dropbox:
        {
            [PPCloudController uploadFile:dstFileName
                                   toPath:dstDirPath
                                 fromPath:srcFilePath
                                 delegate:self
                                 userInfo:nil
                               cloudClass:[PPCloud_Dropbox class]];
            
            break;
        }
            
        case PPStorageSpaceSelectControllerSpaceType_GoogleDrive:
        {
            [PPCloudController uploadFile:dstFileName
                                   toPath:dstDirPath
                                 fromPath:srcFilePath
                                 delegate:self
                                 userInfo:nil
                               cloudClass:[PPCloud_GoogleDrive class]];
            
            break;
        }
            
        case PPStorageSpaceSelectControllerSpaceType_OneDrive:
        {
            [PPCloudController uploadFile:dstFileName
                                   toPath:dstDirPath
                                 fromPath:srcFilePath
                                 delegate:self
                                 userInfo:nil
                               cloudClass:[PPCloud_OneDrive class]];
            
            break;
        }
            
//        case PPStorageSpaceSelectControllerSpaceType_Huawei:
//        {
//            [PPCloudController uploadFile:dstFileName
//                                   toPath:dstDirPath
//                                 fromPath:srcFilePath
//                                 delegate:self
//                                 userInfo:nil
//                               cloudClass:[PPCloud_Huawei class]];
//            
//            break;
//        }
            
        default:
            break;
    }
    
    //////////////////////////////////////////////////
    
    while (self.requestResponse == nil)
    {
        [NSThread sleepForTimeInterval:0.1];
    }
    
    if(error != nil)
    {
        *error = [[self.cloudError copy] autorelease];
    }
    
    return [self.requestResponse boolValue];
}


//================================================================================
//
//================================================================================
- (NSString *)requestDownloadFile:(NSString *)srcFilePath needRemoveTempFile:(BOOL *)needRemoveTempFile error:(NSError **)error
{
    self.requestResponse = nil;
    self.cloudError = nil;
    
    //////////////////////////////////////////////////
    
    NSString *tempDirPath = [self tempDirPath];
    NSString *tempFilePath = [NSString stringWithFormat:@"%@/%@", tempDirPath, [srcFilePath lastPathComponent]];
    
    *needRemoveTempFile = YES;

    switch (self.storageSpaceType)
    {
        case PPStorageSpaceSelectControllerSpaceType_iTune:
        {
            tempFilePath = srcFilePath;
            
            // iTune沒有另外下載，所以不需要刪除。
            *needRemoveTempFile = NO;
            
            self.requestResponse = @(YES);
            break;
        }
            
        case PPStorageSpaceSelectControllerSpaceType_iCloud:
        {
            [PPCloudController loadFileFromPath:srcFilePath
                                       orFileID:nil
                                       intoPath:tempFilePath
                                       delegate:self
                                       userInfo:nil
                                     cloudClass:[PPCloud_iCloud class]];
            
            break;
        }
            
        case PPStorageSpaceSelectControllerSpaceType_Dropbox:
        {
            [PPCloudController loadFileFromPath:srcFilePath
                                       orFileID:nil
                                       intoPath:tempFilePath
                                       delegate:self
                                       userInfo:nil
                                     cloudClass:[PPCloud_Dropbox class]];
            
            break;
        }
            
        case PPStorageSpaceSelectControllerSpaceType_GoogleDrive:
        {
            [PPCloudController loadFileFromPath:srcFilePath
                                       orFileID:nil
                                       intoPath:tempFilePath
                                       delegate:self
                                       userInfo:nil
                                     cloudClass:[PPCloud_GoogleDrive class]];
            
            break;
        }
            
        case PPStorageSpaceSelectControllerSpaceType_OneDrive:
        {
            [PPCloudController loadFileFromPath:srcFilePath
                                       orFileID:nil
                                       intoPath:tempFilePath
                                       delegate:self
                                       userInfo:nil
                                     cloudClass:[PPCloud_OneDrive class]];
            
            break;
        }
            
//        case PPStorageSpaceSelectControllerSpaceType_Huawei:
//        {
//            [PPCloudController loadFileFromPath:srcFilePath
//                                       orFileID:nil
//                                       intoPath:tempFilePath
//                                       delegate:self
//                                       userInfo:nil
//                                     cloudClass:[PPCloud_Huawei class]];
//            
//            break;
//        }
            
        default:
            break;
    }
    
    //////////////////////////////////////////////////
    
    while (self.requestResponse == nil)
    {
        [NSThread sleepForTimeInterval:0.1];
    }
    
    if(error != nil)
    {
        *error = [[self.cloudError copy] autorelease];
    }
    
    return [self.requestResponse boolValue] ? tempFilePath : nil;
}


//================================================================================
//
//================================================================================
- (void)startWithSuperViewController:(UIViewController *)superViewController
                 cloudSpaceTypeArray:(NSArray *)cloudSpaceTypeArray
                          completion:(void (^)(WCBackupRestoreFlowController_Action action, NSError *error))completion
{
    self.flowCompleteBlock  = completion;
    
    //////////////////////////////////////////////////
    
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_main_queue(), ^{
        
        PPStorageSpaceSelectViewController *storageSpaceSelectViewController = [[PPStorageSpaceSelectViewController alloc] init];
        
        if(storageSpaceSelectViewController != nil)
        {
            storageSpaceSelectViewController.delegate = blockSelf;
            storageSpaceSelectViewController.controllerTitle = WCBRFC_MLS_BackupRestore;
            storageSpaceSelectViewController.navigationViewButtonHighlightedBackgroundColor = WCAppearanceDefine_ButtonBackgroundColor;
            storageSpaceSelectViewController.navigationBarButtonImageInsets = WCAppearanceDefine_ButtonImageEdgeInset;            
            storageSpaceSelectViewController.cloudKeyDictionary = self.cloudKeyDict;
            storageSpaceSelectViewController.spaceTypeSelectArray = cloudSpaceTypeArray;
            
            blockSelf.navigationController = [[[PPNavigationController alloc] initWithRootViewController:storageSpaceSelectViewController] autorelease];
            [superViewController presentViewController:blockSelf.navigationController animated:YES completion:nil];
            [storageSpaceSelectViewController release];
        }
    });
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - backup & restore process


//================================================================================
//
//================================================================================
- (void)backupWithFileName:(nonnull NSString *)fileName
             backupDirPath:(NSString *)backupDirPath
               contentSize:(double)contentSize
        mainViewController:(BRFCMainViewController *)mainViewController
           porocessHandler:(nonnull NSError *(^)(NSString *tempFilePath))porocessHandler
{
    NSString *bgTaskKey = [NSString stringWithFormat:@"%s", __func__];
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_main_queue(), ^{
        
        //////////////////////////////////////////////////
        // show busy view
        
        __block PPBusyView *busyView = [[PPBusyView alloc] initWithSuperView:self.navigationController.view];
        
        [[UIApplication sharedApplication] enableIdle:NO];

        //////////////////////////////////////////////////
        // start backup steps
        
        dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
            
            NSError *error = nil;
            
            //////////////////////////////////////////////////
            // 開始長時間運算
            
            [PPBackgroundTaskController addTaskWithKey:bgTaskKey terminate:nil];
            
            do
            {
                NSString *tempDirPath = [blockSelf tempDirPath];
                NSString *tempFilePath = [NSString stringWithFormat:@"%@/%@", tempDirPath, fileName];

                //////////////////////////////////////////////////
                // 檢查網路空間是否足夠 (壓縮檔會比原始檔案小，所以用原始大小當作需要的最小空間)
                
                if(blockSelf.storageSpaceType != PPStorageFileSelectViewControllerSpaceType_iTune &&
                   blockSelf.spaceFreeBytes != WCBackupRestoreFlowController_NotAvailable &&
                   blockSelf.spaceFreeBytes < contentSize)
                {
                    error = PPErrorMake(WCBackupRestoreFlowController_Error_NotEnoughStorage,
                                        WCBRFC_MLS_NotEnoughCloudStorage,
                                        nil);
                    break;
                }
                
                
                //////////////////////////////////////////////////
                // 檢查手機空間是否足夠 (壓縮檔會比原始檔案小，所以用原始大小當作需要的最小空間)
                
                NSArray *paths = NSSearchPathForDirectoriesInDomains(NSDocumentDirectory, NSUserDomainMask, YES);
                NSDictionary *dictionary = [[NSFileManager defaultManager] attributesOfFileSystemForPath:[paths lastObject] error:nil];
                
                if([[dictionary objectForKey:NSFileSystemFreeSize] doubleValue] < contentSize)
                {
                    error = PPErrorMake(WCBackupRestoreFlowController_Error_NotEnoughStorage,
                                        WCBRFC_MLS_NotEnoughDeviceStorage,
                                        nil);
                    break;
                }
                
                
                //////////////////////////////////////////////////
                // TODO: 備份動作
                if(porocessHandler)
                {
                    error = porocessHandler(tempFilePath);
                    
                    if(error)
                    {
                        break;
                    }
                }
                
                //////////////////////////////////////////////////
                // prepare backup info
                
                NSDictionary *fileAttrDict = [[NSFileManager defaultManager] attributesOfItemAtPath:tempFilePath error:nil];
                NSDate *createDate = [fileAttrDict fileCreationDate];
                NSDateFormatter *dateFormatter = [[NSDateFormatter alloc] init];
                [dateFormatter setDateFormat:@"yyyy/MM/dd HH:mm:ss"];
                NSString *dateString = [dateFormatter stringFromDate:createDate];
                [dateFormatter release];
                
                NSString *backupInfo = [NSString stringWithFormat:@"%@ : %@", WCBRFC_MLS_LastBackupInfo, dateString];
                
                
                //////////////////////////////////////////////////
                // upload
                
                [PPBusyView postMessage:WCBRFC_MLS_Uploading];
                
                // prepare destination folder
                NSString *dstDirPath = [blockSelf requestCreateBackupDir:backupDirPath];
                
                // upload to destination
                if([blockSelf requestUploadFile:tempFilePath toDstDirPath:dstDirPath dstFileName:fileName error:&error] == NO)
                {
                    break;
                }
                
                // update backup info after upload success
                if(mainViewController != nil)
                {
                    [mainViewController updateBackupInfo:backupInfo];
                    [WCBackupRestoreFlowController writeBackupInfoToSettings:backupInfo withSapceType:blockSelf.storageSpaceType];
                    
                    // 紀錄最後備份時間
                    [PPSettingsController setDateValue:createDate withKey:WCBackupRestoreFlowController_SettingsKey_LastBackupDate];
                }
                
                //test
                //                self.testDownloadPath = [NSString stringWithFormat:@"%@/%@", dstDirPath, backupFileName];
                
                
                
                //////////////////////////////////////////////////
                // clear temp file
                
                [[NSFileManager defaultManager] removeItemAtPath:tempFilePath error:nil];
                [[NSFileManager defaultManager] removeItemAtPath:tempDirPath error:nil];
                

            }
            while(0);
            
            //////////////////////////////////////////////////
            // 結束長時間運算
            
            [PPBackgroundTaskController removeTaskWithKey:bgTaskKey];
            
            
            //////////////////////////////////////////////////
            // keep last error
            
            blockSelf.lastError = error;
            
            
            //////////////////////////////////////////////////
            // hide busy view
            
            dispatch_sync(dispatch_get_main_queue(), ^{
                
                [[UIApplication sharedApplication] enableIdle:YES];

                [busyView removeFromSuperview];
                [busyView release];
                
                
                //////////////////////////////////////////////////
                // show alert
                
                NSString *title = @"";
                NSString *message = nil;
                NSInteger tag = 0;
                
                if(error == nil)
                {
                    message = WCBRFC_MLS_BackupFinish;
                    tag = WCBackupRestoreFlowControllerTag_FinishFlow;
                }
                else
                {
                    switch (error.code)
                    {
                        case WCBackupRestoreFlowController_Error_FailedToMakeBackupFile:
                        case WCBackupRestoreFlowController_Error_NotEnoughStorage:
                            message = error.localizedFailureReason;
                            break;
                            
                        default:
                            title = WCBRFC_MLS_FailedToUpload;
                            message = [error alertMessage];
                            break;
                    }
                }
                
                [PPAlertView showWithStyle:UIAlertViewStyleDefault
                                     title:title
                                   message:message
                                  delegate:self
                                       tag:tag
                         cancelButtonTitle:WCBRFC_MLS_OK
                         otherButtonTitles:nil];
            });
        });
    });

}



//================================================================================
//
//================================================================================
- (void)restoreWithSrcFilePath:(nonnull NSString *)srcFilePath
                processHandler:(NSError *(^)(NSString *tempFilePath, NSString *tempDstDir))processHandler
{
    //test
    //    srcFilePath = self.testDownloadPath;
    
    if([srcFilePath length] == 0)
    {
        return;
    }
    
    NSString *bgTaskKey = [NSString stringWithFormat:@"%s", __func__];
    __block typeof(self) blockSelf = self;
    
    dispatch_async(dispatch_get_main_queue(), ^{
        
        //////////////////////////////////////////////////
        // show busy view
        [[UIApplication sharedApplication] enableIdle:NO];
        
        __block PPBusyView *busyView = [[PPBusyView alloc] initWithSuperView:self.navigationController.view];
        
        
        //////////////////////////////////////////////////
        // start restore steps
        
        dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
            
            NSError     *error = nil;
            NSString    *tempFilePath = nil;
            BOOL        needRemoveTempFile = YES;
            
            
            //////////////////////////////////////////////////
            // 開始長時間運算
            
            [PPBackgroundTaskController addTaskWithKey:bgTaskKey terminate:nil];
            
            do
            {
                //////////////////////////////////////////////////
                // download from source
                
                [PPBusyView postMessage:WCBRFC_MLS_Downloading];
                tempFilePath = [blockSelf requestDownloadFile:srcFilePath needRemoveTempFile:&needRemoveTempFile error:&error];
                
                // check result
                if(tempFilePath == nil)
                {
                    error = PPErrorMake(WCBackupRestoreFlowController_Error_FailedToDownloadFile,
                                        nil,
                                        error);
                    break;
                }
                
                // prepare tempDir
                NSString *tempDstDir = [NSString stringWithFormat:@"%@/temp", [self tempDirPath]];
                [[NSFileManager defaultManager] removeItemAtPath:tempDstDir error:nil];
                [WCToolController createDirPath:tempDstDir];
                
                //////////////////////////////////////////////////
                // restore process
                if(processHandler)
                {
                    error = processHandler(tempFilePath, tempDstDir);
                    if (error)
                    {
                        break;
                    }
                }
                
                
                //////////////////////////////////////////////////
                // 處理檔案（檢查格式、移除不需要的資料）
                
                if(self.handleRestoreDataBlock != nil)
                {
                    if(self.handleRestoreDataBlock(tempDstDir) == NO)
                    {
                        error = PPErrorMake(WCBackupRestoreFlowController_Error_FailedToParseBackupFile,
                                            WCBRFC_MLS_WrongBackupDataFormat,
                                            nil);
                        break;
                    }
                }
                
                //////////////////////////////////////////////////
                // move to destination
                
                [[NSFileManager defaultManager] removeItemAtPath:blockSelf.baseStoreDirPath error:nil];
                [[NSFileManager defaultManager] moveItemAtPath:tempDstDir toPath:self.baseStoreDirPath error:&error];
                
                if(error != nil)
                {
                    error = PPErrorMake(WCBackupRestoreFlowController_Error_FailedToMoveFile,
                                        WCBRFC_MLS_FailedToRestore,
                                        error);
                }
            }
            while(0);
            
            
            //////////////////////////////////////////////////
            // clear temp file
            
            if(needRemoveTempFile == YES && [tempFilePath length]>0)
            {
                [[NSFileManager defaultManager] removeItemAtPath:tempFilePath error:nil];
            }
            
            [[NSFileManager defaultManager] removeItemAtPath:[self tempDirPath] error:nil];
            
            
            //////////////////////////////////////////////////
            // 結束長時間運算
            
            [PPBackgroundTaskController removeTaskWithKey:bgTaskKey];
            
            
            //////////////////////////////////////////////////
            // keep last error
            
            blockSelf.lastError = error;
            
            
            //////////////////////////////////////////////////
            // hide busy view
            
            dispatch_sync(dispatch_get_main_queue(), ^{
                
                [[UIApplication sharedApplication] enableIdle:YES];

                [busyView removeFromSuperview];
                [busyView release];
                
                //////////////////////////////////////////////////
                // show alert
                
                NSString *title = @"";
                NSString *message = nil;
                NSInteger tag = 0;
                
                if(error == nil)
                {
                    message = WCBRFC_MLS_RestoreFinish;
                    tag = WCBackupRestoreFlowControllerTag_FinishFlow;
                }
                else
                {
                    switch (error.code)
                    {
                        case WCBackupRestoreFlowController_Error_FailedToDownloadFile:
                        {
                            title = WCBRFC_MLS_FailedToDownload;
                            message = [error alertMessage];
                            break;
                        }
                            
                        case WCBackupRestoreFlowController_Error_FailedToDecryptBackupFile:
                        case WCBackupRestoreFlowController_Error_FailedToParseBackupFile:
                        case WCBackupRestoreFlowController_Error_FailedToMoveFile:
                            message = error.localizedFailureReason;
                            break;
                            
                        default:
                            message = [error alertMessage];
                            break;
                    }
                }
                
                [PPAlertView showWithStyle:UIAlertViewStyleDefault
                                     title:title
                                   message:message
                                  delegate:self
                                       tag:tag
                         cancelButtonTitle:WCBRFC_MLS_OK
                         otherButtonTitles:nil];
            });
        });
    });
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - IWC backup



//==============================================================================
//
//==============================================================================
- (NSString *)iwcBackupFileName
{
    return [self backupFileNameWithPathExtension:WCCloudFileDefine_iPhoneBackupFileExt];
}


//==============================================================================
//
//==============================================================================
- (NSString *)iwcBackupRestoreDir
{
    return WCCloudFileDefine_BackupDirPath;
}


//================================================================================
//
//================================================================================
- (void)backupIWCWithMainViewController:(BRFCMainViewController *)mainViewController
{
    __block typeof(self) blockSelf = self;
    
    [self setBusy:@(YES)];

    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
        double contentSize = [PPZipController contentSizeWithSrcDir:blockSelf.baseStoreDirPath excludeDirPatterns:blockSelf.excludeDirPatterns];
        NSString *backupFileName = [self iwcBackupFileName];
        
        dispatch_async(dispatch_get_main_queue(), ^{
            [self setBusy:@(NO)];
            //////////////////////////////////////////////////
            [self backupWithFileName:backupFileName
                       backupDirPath:[self iwcBackupRestoreDir]
                         contentSize:contentSize
                  mainViewController:mainViewController
                     porocessHandler:^NSError *(NSString *tempFilePath) {
                         
                         [PPBusyView postMessage:WCBRFC_MLS_GeneratingData];
                         
                         NSError *error = nil;
                         
                         if([PPZipController zipWithSrcDir:blockSelf.baseStoreDirPath
                                        excludeDirPatterns:blockSelf.excludeDirPatterns
                                                   dstFile:tempFilePath
                                                  password:WCCloudFileDefine_ZipPassword
                                           progressHandler:^(CGFloat progress){
                                               
                                               NSString *message = [NSString stringWithFormat:@"%@ %d%%", WCBRFC_MLS_GeneratingData, (int)(progress*100)];
                                               [PPBusyView postMessage:message];
                                               
                                           }] == NO)
                         {
                             error = PPErrorMake(WCBackupRestoreFlowController_Error_FailedToMakeBackupFile,
                                                 WCBRFC_MLS_FailedToMakeBackupFile,
                                                 nil);
                         }
                         
                         return error;
                     }];
        });
    });
    
    
    
    
}


//================================================================================
//
//================================================================================
- (void)restoreIWCWithSrcFilePath:(NSString *)srcFilePath
{
    [self restoreWithSrcFilePath:srcFilePath
                  processHandler:^NSError *(NSString *tempFilePath, NSString *tempDstDir) {
                      
                      NSError *error =  [PPZipController unzipWithSrcFile:tempFilePath
                                                        dstDir:tempDstDir
                                                      password:WCCloudFileDefine_ZipPassword
                                               progressHandler:^(CGFloat progress) {
                                                   
                                                   NSString *message = [NSString stringWithFormat:@"%@ %d%%", WCBRFC_MLS_ParsingData, (int)(progress*100)];
                                                   [PPBusyView postMessage:message];
                                                   
                                               }];

                      return error;
                  }];
}







////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - WCBK backup (WCT or WCM only)

#if defined (PRODUCTLINE_WCT) || defined (PRODUCTLINE_WCM)


//==============================================================================
//
//==============================================================================
- (NSString *)wcbkBackupFileName
{
    return [self backupFileNameWithPathExtension:WCCloudFileDefine_GeneralBackupFileExt];
}


//==============================================================================
//
//==============================================================================
- (NSString *)wcbkBackupRestoreDir
{
    return WCCloudFileDefine_BackupDirPath_WCBK;
}


//================================================================================
//
//================================================================================
- (void)backupWCBKWithMainViewController:(BRFCMainViewController *)mainViewController
{
    // WCBK備份
    __block typeof(self) blockSelf = self;
    
    [self setBusy:@(YES)];
    
    dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
        double contentSize = [PPZipController contentSizeWithSrcDir:blockSelf.baseStoreDirPath excludeDirPatterns:blockSelf.excludeDirPatterns];
        NSString *backupFileName = [self wcbkBackupFileName];
        
        dispatch_async(dispatch_get_main_queue(), ^{
            [self setBusy:@(NO)];
            //////////////////////////////////////////////////
            [self backupWithFileName:backupFileName
                       backupDirPath:[self wcbkBackupRestoreDir]
                         contentSize:contentSize
                  mainViewController:mainViewController
                     porocessHandler:^NSError *(NSString *tempFilePath) {
                         
                         [PPBusyView postMessage:WCBRFC_MLS_GeneratingData];
                         
                         NSError *error = nil;
                         NSError *returnError = nil;
                         
                         
                         blockSelf.dataController = [[[DataController alloc] initWithAccessMode:WCDC_AM_All] autorelease];
                         NSArray *allCardIDs = [blockSelf.dataController copyCardIDArrayWithGroupID:[NSString stringWithInteger:WC_GID_All]];
                         
                         // 匯出
                         //                 NSString *wcxfPath = [tempFilePath stringByAppendingPathComponent:backupFileName];
                         WCXFDataController *wcxfDataController = [[WCXFDataController alloc] initWithDelegate:self];
                         if([wcxfDataController writeFileWithPath:tempFilePath cardIDs:allCardIDs error:&returnError]==NO)
                         {
                             error = PPErrorMake(WCBackupRestoreFlowController_Error_FailedToMakeBackupFile,
                                                 WCBRFC_MLS_FailedToMakeBackupFile,
                                                 returnError);
                         }
                         [allCardIDs release];
                         
                         [wcxfDataController release];
                         blockSelf.dataController = nil;
                         
                         return error;
                     }];
        });
    });
    
}


//================================================================================
//
//================================================================================
- (void)restoreWCBKWithSrcFilePath:(NSString *)srcFilePath
{
    // TODO: wcbk還原
    __block typeof(self) blockSelf = self;

    [self restoreWithSrcFilePath:srcFilePath
                  processHandler:^NSError *(NSString *tempFilePath, NSString *tempDstDir) {
                      
                      NSError *error = nil;
                      NSError *returnError = nil;

                      NSString *dbPath = [tempDstDir stringByAppendingString:@"/WorldCard"];
                      [WCToolController createDirPath:dbPath];
                      blockSelf.dataController = [[[DataController alloc] initWithAccessMode:WCDC_AM_All
                                                                                baseDirPath:dbPath
                                                                        useSharedDBInstance:NO
                                                                               isExistingDB:NO] autorelease];

                      // 讀取wcxf
                      WCXFDataController *wcxfDataController = [[WCXFDataController alloc] initWithDelegate:self];

                      if([wcxfDataController readFileWithPath:tempFilePath error:&returnError]==NO)
                      {
                          error = PPErrorMake(WCBackupRestoreFlowController_Error_FailedToDecryptBackupFile,
                                              WCBRFC_MLS_FailedToRestore,
                                              returnError);
                      }
                      
                      blockSelf.dataController = nil;
                      return error;
                  }];
}

#endif



////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class methods

//================================================================================
//
//================================================================================
+ (void)startWithSuperViewController:(UIViewController *)superViewController
                  cloudKeyDictionary:(NSDictionary *)cloudKeyDictionary
                 cloudSpaceTypeArray:(NSArray *)cloudSpaceTypeArray
                    baseStoreDirPath:(NSString *)baseStoreDirPath
                  excludeDirPatterns:(NSArray *)excludeDirPatterns
                   handleRestoreData:(BOOL (^)(NSString *restoreTempDir))handleRestoreData
                          completion:(void (^)(WCBackupRestoreFlowController_Action action, NSError *error))completion
{
    [g_backupRestoreFlowController release];
    g_backupRestoreFlowController = [[WCBackupRestoreFlowController alloc] init];
    
    if(g_backupRestoreFlowController != nil)
    {
        g_backupRestoreFlowController.cloudKeyDict = cloudKeyDictionary;
        g_backupRestoreFlowController.baseStoreDirPath = baseStoreDirPath;
        g_backupRestoreFlowController.excludeDirPatterns = excludeDirPatterns;
        g_backupRestoreFlowController.handleRestoreDataBlock = handleRestoreData;
        
        [g_backupRestoreFlowController startWithSuperViewController:superViewController
                                                cloudSpaceTypeArray:cloudSpaceTypeArray
                                                         completion:^(WCBackupRestoreFlowController_Action action, NSError *error){
                                                             
                                                             completion(action, error);
                                                             
                                                             [g_backupRestoreFlowController autorelease];
                                                             g_backupRestoreFlowController = nil;
                                                         }];
        
    }
}

/*
//================================================================================
//
//================================================================================
+ (void)localBackupWithSuperView:(UIView *)superView
                baseStoreDirPath:(NSString *)baseStoreDirPath
              excludeDirPatterns:(NSArray *)excludeDirPatterns
                      completion:(void (^)(NSError *error))completion
{
    [g_backupRestoreFlowController release];
    g_backupRestoreFlowController = [[WCBackupRestoreFlowController alloc] init];
    
    if(g_backupRestoreFlowController != nil)
    {
        BRFCCompleteBlock localBackupComplete = ^(NSError *error){
        
            completion(error);
            
            [g_backupRestoreFlowController autorelease];
            g_backupRestoreFlowController = nil;
        };
        
        //////////////////////////////////////////////////

        g_backupRestoreFlowController.baseStoreDirPath = baseStoreDirPath;
        g_backupRestoreFlowController.excludeDirPatterns = excludeDirPatterns;
        g_backupRestoreFlowController.flowCompleteBlock = localBackupComplete;
        g_backupRestoreFlowController.storageSpaceType = PPStorageSpaceSelectControllerSpaceType_iTune;
        [g_backupRestoreFlowController backupWithMainViewController:nil];
    }
}
*/

+ (NSDate *)lastBackupDate
{
    return  [PPSettingsController dateValueWithKey:WCBackupRestoreFlowController_SettingsKey_LastBackupDate];
}

@end
