//
//  WCABDataController.m
//
//

#import "WCABDataController.h"

// Define
#import "WCABDataController+ParameterDefine.h"
#import "WCABDataController+ResourceDefine.h"
#import "WCFieldDefine.h"

// Controller
#import "PPCountryCodeConvert.h"
#import "ABDCSourceController.h"

// Model
#import "PPProgressModel.h"

// Category
#import "NSDictionary+Additions.h"
#import "NSError+Custom.h"
#import "UIImage+Additions.h"
#import "NSDate+Format.h"
#import "WCABCardModel+CNContact.h"

static BOOL g_noteFieldWithTransform = YES;

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface WCABDataController ()

@property (nonatomic, retain) CNContactStore *contactStore;
@property (nonatomic, retain) ABDCSourceController *sourceController;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation WCABDataController

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class methods

//================================================================================
//
//================================================================================
+ (void)checkAccessPermissionWithCompletion:(void (^)(BOOL authorized))completion
{
//    typedef void(^CheckComplete)(BOOL authorized);
//    CheckComplete checkComplete = [completion copy];
    
    CNAuthorizationStatus status = [CNContactStore authorizationStatusForEntityType:CNEntityTypeContacts];
    switch (status)
    {
        case CNAuthorizationStatusAuthorized:
        {
            completion(YES);
            break;
        }
            
        case CNAuthorizationStatusNotDetermined:
        {
            CNContactStore *contatStore = [[[CNContactStore alloc] init] autorelease];
            
            [contatStore requestAccessForEntityType:CNEntityTypeContacts completionHandler:^(BOOL granted, NSError * _Nullable error) {
                
                dispatch_async(dispatch_get_main_queue(), ^{
                    completion(granted);
                });
            }];
            
            break;
        }
            
        default:
        {
            completion(NO);
            break;
        }
    }
}


//================================================================================
//
//================================================================================
+ (void)setNoteFieldTransform:(BOOL)noteFieldTransform
{
    g_noteFieldWithTransform = noteFieldTransform;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods

//================================================================================
//
//================================================================================
- (id)initWithError:(NSError **)error
{
    NSError *returnError = nil;
    id returnInstance = nil;
    
    do
    {
        if([CNContactStore authorizationStatusForEntityType:CNEntityTypeContacts]!=CNAuthorizationStatusAuthorized)
        {
            returnError = PPErrorMake(WCABDataController_Error_NoAccessPermission, nil, nil);
            break;
        }
        
        //////////////////////////////////////////////////
        
        if (self = [super init])
        {
            self.contactStore = [[[CNContactStore alloc] init] autorelease];
            
            if(self.contactStore == nil)
            {
                [self dealloc];
                self = nil;

                returnError = PPErrorMake(WCABDataController_Error_FailedToCreateContactStore, @"WCABDataController_Error_FailedToCreateContactStore", nil);
                break;
            }
            
            //////////////////////////////////////////////////

            _sourceController = [[ABDCSourceController alloc] init];
            
            if(self.sourceController == nil)
            {
                [self dealloc];
                self = nil;
                
                returnError = PPErrorMake(WCABDataController_Error_FailedToGetSourceController, @"WCABDataController_Error_FailedToGetSourceController", nil);
                break;
            
            }
            
            self.sourceController.contactStore = self.contactStore;

            //////////////////////////////////////////////////
            
            returnInstance = self;
        }
    }
    while(0);
    
    //////////////////////////////////////////////////
    
    if(error != nil)
    {
        *error = returnError;
    }
    
    return returnInstance;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    self.contactStore = nil;
    self.sourceController = nil;
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods

//==============================================================================
//
//==============================================================================
- (CNContainerType)containerTypeWithSourceID:(NSString *)sourceID
{
    if ([sourceID length]==0)
    {
        return CNContainerTypeLocal;
    }
    
    
    NSPredicate *predicate = [CNContainer predicateForContainersWithIdentifiers:@[sourceID]];
    NSArray *containers = [self.contactStore containersMatchingPredicate:predicate error:nil];
    CNContainer *container = [containers firstObject];
    
    if(container==nil)
    {
        return 0;
    }

    return container.type;
}

//================================================================================
//
//================================================================================
- (NSData *)imageDataForSaveWithImage:(UIImage *)image
{
    UIImage *accessImage = [image imageByAdjustOrientationWithMaxLength:WCABDC_MaxPhotoLength];
    
    return UIImageJPEGRepresentation(accessImage, 0.3);
}


//==============================================================================
//
//==============================================================================
- (void)convertDataWithMode:(WCABDC_ConvertMode)convertMode
                  cardModel:(WCABCardModel *)cardModel
                    contact:(CNContact *)contact
{
    // !! 如果是要轉到系統聯絡人，而且source是exchange，要特殊處理
    // 只有forExchange才要有convertFieldCount
    NSMutableDictionary *convertFieldCount = nil;
    if (convertMode==ConvertMode_WC_to_AB)
    {
        BOOL forExchange = ([self containerTypeWithSourceID:cardModel.abSourceID]==CNContainerTypeExchange);

        if(forExchange)
        {
            convertFieldCount = [[[NSMutableDictionary alloc] init] autorelease];
        }
    }

    //////////////////////////////////////////////////

    [cardModel convertDataWithMode:convertMode
                         cardModel:cardModel
                 convertFieldCount:convertFieldCount
                           contact:contact
                       convertNote:g_noteFieldWithTransform];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Contacts help methods


//==============================================================================
//
//==============================================================================
- (NSArray *)keyToFetch
{
    NSMutableArray *keyToFetch = [NSMutableArray arrayWithArray:
                                  @[CNContactGivenNameKey, CNContactMiddleNameKey, CNContactFamilyNameKey, CNContactNamePrefixKey, CNContactNameSuffixKey,CNContactPhoneticGivenNameKey,CNContactPhoneticFamilyNameKey,
                                    CNContactOrganizationNameKey, CNContactDepartmentNameKey, CNContactJobTitleKey,
                                    CNContactPostalAddressesKey,
                                    CNContactBirthdayKey, CNContactDatesKey,
                                    CNContactImageDataKey,
                                    CNContactPhoneNumbersKey,
                                    CNContactEmailAddressesKey,
                                    CNContactUrlAddressesKey,
                                    CNContactInstantMessageAddressesKey,
                                    CNContactSocialProfilesKey,
                                    CNContactNicknameKey]];
 
    if(g_noteFieldWithTransform==YES)
    {
        [keyToFetch addObject:CNContactNoteKey];
    }
    
    // !! CNContactPhoneticOrganizationNameKey只有10以上才可以用
    if([[[UIDevice currentDevice] systemVersion] floatValue]>=10.0)
    {
        [keyToFetch addObject:CNContactPhoneticOrganizationNameKey];
    }
    return [NSArray arrayWithArray:keyToFetch];
}


//==============================================================================
//
//==============================================================================
- (NSArray *)contactsWithSourceID:(NSString *)sourceID groupID:(NSString *)groupID error:(NSError **)error
{
    NSArray *keyToFetch = [self keyToFetch];
    NSPredicate *contactPredicate = [CNContact predicateForContactsInGroupWithIdentifier:groupID];
    if ([groupID length]==0)
    {
        contactPredicate = [CNContact predicateForContactsInContainerWithIdentifier:sourceID];
    }
    
    return [self.contactStore unifiedContactsMatchingPredicate:contactPredicate keysToFetch:keyToFetch error:error];
}


//==============================================================================
//
//==============================================================================
- (CNContact *)contactWithName:(NSString *)name  error:(NSError **)error
{
    if ([name length]==0)
    {
        if(error)
        {
            *error = PPErrorMake(WCABDataController_Error_InvalidParameters,
                                 @"[name length]==0",
                                 nil);
        }
        return nil;
    }
    NSArray *keyToFetch = [self keyToFetch];
    NSPredicate *predicate = [CNContact predicateForContactsMatchingName:name];
    return [[self.contactStore unifiedContactsMatchingPredicate:predicate keysToFetch:keyToFetch error:error] firstObject];
}


//==============================================================================
//
//==============================================================================
- (CNContact *)contactWithPersonID:(NSString *)personID  error:(NSError **)error
{
    if ([personID length]==0)
    {
        if(error)
        {
            *error = PPErrorMake(WCABDataController_Error_InvalidParameters,
                                 @"[personID length]==0",
                                 nil);
        }
        return nil;
    }
    NSArray *keyToFetch = [self keyToFetch];
    
    return [self.contactStore unifiedContactWithIdentifier:personID keysToFetch:keyToFetch error:error];
}


//==============================================================================
//
//==============================================================================
- (CNGroup *)groupWithGroupID:(NSString *)groupID error:(NSError **)error
{
    if (groupID==nil)
    {
        return nil;
    }
    
    NSPredicate *contactPredicate = [CNGroup predicateForGroupsWithIdentifiers:@[groupID]];
    NSArray *groups = [self.contactStore groupsMatchingPredicate:contactPredicate error:error];
    
    return [groups firstObject];
}



//==============================================================================
// newGroupID 如果沒有變，就回傳nil
//==============================================================================
- (BOOL)addContact:(CNContact *)contact toSourceID:(NSString *)sourceID groupID:(NSString *)groupID newGroupID:(NSString **)newGroupID error:(NSError **)error
{
    if ([sourceID length]==0)
    {
        if(error)
        {
            *error = PPErrorMake(WCABDataController_Error_InvalidParameters,
                                 @"[sourceID length]==0",
                                 nil);
        }
        return NO;
    }
    
    NSError *returnError = nil;
    NSArray *groups = nil;
    
    if ([groupID length]!=0)
    {
        // 加到group
        NSPredicate *predicate = [CNGroup predicateForGroupsWithIdentifiers:@[groupID]];
        groups = [self.contactStore groupsMatchingPredicate:predicate error:&returnError];
        
        if (returnError)
        {
            groups = nil;
            
            if(newGroupID)
            {
                *newGroupID = groupID;
            }
            // !! 這個取不到或有錯沒關係，後面會直接加到container上
            returnError = nil;
        }
    }

    //////////////////////////////////////////////////
    // saveRequest
    CNSaveRequest *saveRequest = [[CNSaveRequest alloc] init];
    
    // 加到container
    CNMutableContact *mutableContact = [contact mutableCopy];
    [saveRequest addContact:mutableContact toContainerWithIdentifier:sourceID];
    [mutableContact release];
    
    // 加到group
    if ([groups count]>0)
    {
        [saveRequest addMember:contact toGroup:[groups firstObject]];
    }
    
    [self.contactStore executeSaveRequest:saveRequest error:&returnError];
    [saveRequest release];
    
    //////////////////////////////////////////////////
    if (error)
    {
        *error = returnError;
    }
    
    return (returnError==nil);
}



//==============================================================================
// 更新聯絡人內容
//==============================================================================
- (BOOL)updateContact:(CNMutableContact *)contact error:(NSError **)error
{
    if (contact==nil)
    {
        if(error)
        {
            *error = PPErrorMake(WCABDataController_Error_InvalidParameters,
                                 @"contact==nil",
                                 nil);
        }
        return NO;
    }
    
    NSError *returnError = nil;

    //////////////////////////////////////////////////
    // saveRequest
    CNSaveRequest *saveRequest = [[CNSaveRequest alloc] init];
    [saveRequest updateContact:contact];

    [self.contactStore executeSaveRequest:saveRequest error:&returnError];
    [saveRequest release];
    
    //////////////////////////////////////////////////
    if (error)
    {
        *error = returnError;
    }
    
    return (returnError==nil);
}


//==============================================================================
//
//==============================================================================
- (BOOL)moveContact:(CNContact *)contact fromGroupID:(NSString *)groupID toNewGroupID:(NSString *)newGroupID error:(NSError **)error
{
    if (contact==nil)
    {
        if(error)
        {
            *error = PPErrorMake(WCABDataController_Error_InvalidParameters,
                                 @"contact==nil",
                                 nil);
        }
        
        return NO;
    }

    //////////////////////////////////////////////////
    NSError *returnError = nil;
    
    do {
        //////////////////////////////////////////////////
        // get currentGroup
        CNGroup *currentGroup = nil;
        
        if ([groupID length]>0)
        {
            currentGroup = [self groupWithGroupID:groupID error:&returnError];
            
            if (returnError)
            {
                break;
            }
            
        }

        CNGroup *newGroup = nil;
        
        if (newGroupID)
        {
            newGroup = [self groupWithGroupID:newGroupID error:&returnError];
            
            if (returnError)
            {
                break;
            }
        }


        //////////////////////////////////////////////////
        // saveRequest
        CNSaveRequest *saveRequest = [[CNSaveRequest alloc] init];
        
        // 從舊的移除
        if (currentGroup)
        {
            [saveRequest removeMember:contact fromGroup:currentGroup];
        }
        
        // 加到新的
        if (newGroup)
        {
            [saveRequest addMember:contact toGroup:newGroup];
        }
        
        if (currentGroup!=nil ||
            newGroup!=nil)
        {
            [self.contactStore executeSaveRequest:saveRequest error:&returnError];
        }
        [saveRequest release];
        
    } while (0);
    
    //////////////////////////////////////////////////
    if (error)
    {
        *error = returnError;
    }
    
    if (returnError)
    {
        return NO;
    }
    
    return YES;
}


//==============================================================================
//
//==============================================================================
- (BOOL)removeWithPersonID:(NSString *)personID error:(NSError **)error
{
    if ([personID length]==0)
    {
        if(error)
        {
            *error = PPErrorMake(WCABDataController_Error_InvalidParameters,
                                 @"[personID length]==0",
                                 nil);
        }
        return NO;
    }
    
    NSError *returnError = nil;

    do {
        //////////////////////////////////////////////////
        // get contact
        CNContact *contact = [self contactWithPersonID:personID error:&returnError];
        
        if (returnError)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        // saveRequest
        CNSaveRequest *saveRequest = [[CNSaveRequest alloc] init];
        
        CNMutableContact *mutableContact = [contact mutableCopy];
        [saveRequest deleteContact:mutableContact];
        [mutableContact release];

        [self.contactStore executeSaveRequest:saveRequest error:&returnError];
        [saveRequest release];
        
    } while (0);

    //////////////////////////////////////////////////
    if (error)
    {
        *error = returnError;
    }
    
    return (returnError==nil);
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Group methods (Group)

//================================================================================
//
//================================================================================
- (NSString *)addGroupWithSourceID:(NSString *)sourceID name:(NSString *)name
{
    NSError *error = nil;
    NSString *groupID = nil;
    CNSaveRequest *saveRequest = [[CNSaveRequest alloc] init];
    
    CNMutableGroup *group = [[CNMutableGroup alloc] init];
    group.name = name;
    groupID = group.identifier;
    [saveRequest addGroup:group toContainerWithIdentifier:sourceID];
    [group release];
    
    [self.contactStore executeSaveRequest:saveRequest error:&error];
    [saveRequest release];
    
    return groupID;
}


//==============================================================================
//
//==============================================================================
- (void)deleteGroupWithGroupID:(NSString *)groupID
{
    NSError *error = nil;
    
    CNGroup *currentGroup = [self groupWithGroupID:groupID error:&error];
    
    if (error)
    {
        return ;
    }
    
    CNSaveRequest *saveRequest = [[CNSaveRequest alloc] init];
    
    CNMutableGroup *group = [currentGroup mutableCopy];
    [saveRequest deleteGroup:group];
    [group release];
    
    [self.contactStore executeSaveRequest:saveRequest error:&error];
    [saveRequest release];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance methods


//==============================================================================
//
//==============================================================================
- (NSString *)sourceIDOfPersonWithPersonID:(NSString *)personID
{
    NSError *error = nil;
    NSPredicate *predicate = [CNContainer predicateForContainerOfContactWithIdentifier:personID];
    NSArray *containers = [self.contactStore containersMatchingPredicate:predicate error:&error];
    
    return [[containers firstObject] identifier];
}


//==============================================================================
//
//==============================================================================
- (NSString *)groupIDOfPersonWithPersonID:(NSString *)personID sourceID:(NSString *)sourceID
{
    if ([sourceID length]==0)
    {
        return nil;
    }
    
    __block NSString *groupID = nil;
    NSError *error = nil;
    NSPredicate *predicate = [CNGroup predicateForGroupsInContainerWithIdentifier:sourceID];
    NSArray *groups = [self.contactStore groupsMatchingPredicate:predicate error:&error];

    for (CNGroup *group in groups)
    {
        CNContactFetchRequest *request = [[CNContactFetchRequest alloc] initWithKeysToFetch:[self keyToFetch]];
        request.predicate = [CNContact predicateForContactsInGroupWithIdentifier:group.identifier];
        [self.contactStore enumerateContactsWithFetchRequest:request error:&error usingBlock:^(CNContact * _Nonnull contact, BOOL * _Nonnull stop) {
            
            if ([contact.identifier isEqualToString:personID])
            {
                groupID = group.identifier;
                *stop = YES;
            }
        }];
        
        [request release];
        
        //////////////////////////////////////////////////
        // 找到後就停止
        if ([groupID length]!=0)
        {
            break;
        }
    }

    return groupID;
}



//================================================================================
// 取得聯絡人列表資料
//================================================================================
- (NSArray *)copyCardModelsForListingWithSourceID:(NSString *)sourceID groupID:(NSString *)groupID error:(NSError **)error
{
    NSError *returnError = nil;
    NSArray *contacts = nil;
    
    NSMutableArray *cardModels = [[NSMutableArray alloc] init];
    CGFloat updateProgress = 0;

    do
    {
        PPProgressModel *progressModel = [[[PPProgressModel alloc] init] autorelease];
        
        if(progressModel==nil)
        {
            break;
        }
        
        //////////////////////////////////////////////////

        if(cardModels == nil)
        {
            returnError = PPErrorMake(WCABDataController_Error_FailedToInitObject, @"cardModels == nil", nil);
            break;
        }
        
        if([sourceID length]==0)
        {
            returnError = PPErrorMake(WCABDataController_Error_InvalidParameters,
                                      @"[sourceID length]==0",
                                      nil);
            break;
        }
        
        //////////////////////////////////////////////////

        contacts = [self contactsWithSourceID:sourceID groupID:groupID error:&returnError];
        
        //////////////////////////////////////////////////

        if(returnError!=nil)
        {
            break;
        }
        
        progressModel.denominator = [contacts count];
        
        //////////////////////////////////////////////////

        [[NSNotificationCenter defaultCenter] postNotificationName:WCABDC_FetchCardModelsProgress
                                                            object:@"0%"];
        
        //////////////////////////////////////////////////
        
        for(CNContact *contact in contacts)
        {
            WCABCardModel *cardModel = [[WCABCardModel alloc] init];
            cardModel.abSourceID = sourceID;
            cardModel.abGroupID = groupID;
            cardModel.abPersonID = contact.identifier;
            
            //!! Howard 配合 spec 調整無姓名欄位的資料取法
            NSArray *fieldTypes = @[@(WC_FT_Name),@(WC_FT_Company),@(WC_FT_Phone),@(WC_FT_Email),@(WC_FT_InstantMessage),@(WC_FT_Address),@(WC_FT_URL)];
            
            //////////////////////////////////////////////////
            
            @autoreleasepool
            {
                for(id typeNumber in fieldTypes)
                {
                    if([typeNumber integerValue]==WC_FT_Name ||
                       [typeNumber integerValue]==WC_FT_Company)
                    {
                        if([cardModel convertType2FieldWithCardModel:cardModel
                                                      contact:contact
                                                      fieldType:[typeNumber integerValue]
                                                    convertMode:ConvertMode_AB_to_WC
                                              convertFieldCount:nil
                                                          error:nil] &&
                           [cardModel fieldArrayDict].count>0)
                        {
                            [cardModels addObject:cardModel];
                            
                            break;
                        }
                    }
                    else if([typeNumber integerValue]==WC_FT_Email ||
                            [typeNumber integerValue]==WC_FT_URL ||
                            [typeNumber integerValue]==WC_FT_Phone)
                    {
                        if([cardModel convertType4FieldWithCardModel:cardModel
                                                        contact:contact
                                                      fieldType:[typeNumber integerValue]
                                                    convertMode:ConvertMode_AB_to_WC
                                              convertFieldCount:nil
                                                          error:nil] &&
                           [cardModel fieldArrayDict].count>0)
                        {
                            [cardModels addObject:cardModel];
                            
                            break;
                        }
                    }
                    else if([typeNumber integerValue]==WC_FT_Address)
                    {
                        if([cardModel convertType3FieldWithCardModel:cardModel
                                                        contact:contact
                                                      fieldType:[typeNumber integerValue]
                                                    convertMode:ConvertMode_AB_to_WC
                                              convertFieldCount:nil
                                                          error:nil] &&
                           [cardModel fieldArrayDict].count>0)
                        {
                            [cardModels addObject:cardModel];
                            
                            break;
                        }
                    }
                    else if([typeNumber integerValue]==WC_FT_InstantMessage)
                    {
                        if([cardModel convertType5FieldWithCardModel:cardModel
                                                        contact:contact
                                                      fieldType:[typeNumber integerValue]
                                                    convertMode:ConvertMode_AB_to_WC
                                              convertFieldCount:nil
                                                          error:nil]  &&
                           [cardModel fieldArrayDict].count>0)
                        {
                            [cardModels addObject:cardModel];
                            
                            break;
                        }
                    }
                    
                    // for loop end
                }
                
                //////////////////////////////////////////////////
                
                // 顯示沒有姓名
                if([cardModels indexOfObject:cardModel]==NSNotFound)
                {
                    [cardModels addObject:cardModel];
                }
                
                //////////////////////////////////////////////////
                
                progressModel.numerator++;
                
                //不要每次一直丟
                if([progressModel progress]>(updateProgress+0.01) ||progressModel.progress==1)
                {
                    updateProgress = [progressModel progress];
                 
                    
                    [[NSNotificationCenter defaultCenter] postNotificationName:WCABDC_FetchCardModelsProgress
                                                                        object:[NSString stringWithFormat:@"%d%%", (int)(progressModel.progress*100)]];
                }
                
                //////////////////////////////////////////////////

                // autorelease pool
            }
            
            //////////////////////////////////////////////////
            
            [cardModel release];
        }
    }
    while(0);
    
    //////////////////////////////////////////////////

    if([cardModels count] == 0)
    {
        [cardModels release];
        cardModels = nil;
    }
    
    if(error != nil)
    {
        *error = returnError;
    }
    
    return (NSArray *)cardModels;
}


//==============================================================================
//
//==============================================================================
- (WCABCardModel *)copyCardModelWithName:(NSString *)name error:(NSError **)error
{
    NSError *returnError = nil;
    WCABCardModel *cardModel = nil;
    
    do
    {
        CNContact *contact = [self contactWithName:name error:&returnError];
        
        if(contact == nil)
        {
            returnError = PPErrorMake(WCABDataController_Error_FailedToGetContact, @"無法取得contact", returnError);
            break;
        }
        
        cardModel = [[WCABCardModel alloc] init];
        
        if(cardModel == nil)
        {
            returnError = PPErrorMake(WCABDataController_Error_FailedToInitObject, @"cardModel == nil", nil);
            break;
        }
        
        //////////////////////////////////////////////////
        
        UIImage *image = [self copyPhotoImageWithPersonID:contact.identifier error:&returnError];
        
        if(returnError != nil)
        {
            [image release];
            break;
        }
        
        cardModel.abPersonID = contact.identifier;
        cardModel.abSourceID = [self sourceIDOfPersonWithPersonID:cardModel.abPersonID];
        cardModel.abGroupID = [self groupIDOfPersonWithPersonID:cardModel.abPersonID sourceID:cardModel.abSourceID];
        cardModel.abPhotoImage = image;
        [image release];
        
        //////////////////////////////////////////////////
        
        [self convertDataWithMode:ConvertMode_AB_to_WC
                        cardModel:cardModel
                          contact:contact];
    }
    while(0);
    
    //////////////////////////////////////////////////
    
    if(error != nil)
    {
        *error = returnError;
    }
    
    return cardModel;
}


//================================================================================
// 取得AddressBook聯絡人
//================================================================================
- (WCABCardModel *)copyCardModelWithPersonID:(NSString *)personID error:(NSError **)error
{
    NSError *returnError = nil;
    WCABCardModel *cardModel = nil;
    
    do
    {
        CNContact *contact = [self contactWithPersonID:personID error:error];

        if(contact == nil)
        {
            returnError = PPErrorMake(WCABDataController_Error_FailedToGetContact, @"無法取得contact", *error);
            break;
        }
        
        cardModel = [[WCABCardModel alloc] init];
        
        if(cardModel == nil)
        {
            returnError = PPErrorMake(WCABDataController_Error_FailedToInitObject, @"cardModel == nil", nil);
            break;
        }
        
        //////////////////////////////////////////////////
       
        UIImage *image = [self copyPhotoImageWithPersonID:personID error:&returnError];
        
        if(returnError != nil)
        {
            [image release];
            break;
        }
        
        cardModel.abPersonID = contact.identifier;
        cardModel.abSourceID = [self sourceIDOfPersonWithPersonID:cardModel.abPersonID];
        cardModel.abGroupID = [self groupIDOfPersonWithPersonID:cardModel.abPersonID sourceID:cardModel.abSourceID];
        cardModel.abPhotoImage = image;
        [image release];
            
        //////////////////////////////////////////////////
            
        [self convertDataWithMode:ConvertMode_AB_to_WC
                        cardModel:cardModel
                          contact:contact];
    }
    while(0);

    //////////////////////////////////////////////////

    if(error != nil)
    {
        *error = returnError;
    }
    
    return cardModel;
}


//================================================================================
// 新增AddressBook聯絡人
//================================================================================
- (BOOL)addPersonWithCardModel:(WCABCardModel *)cardModel error:(NSError **)error
{
    BOOL result = NO;
    NSError *returnError = nil;
    
    @autoreleasepool
    {
        
        NSString * sourceID = cardModel.abSourceID;
        NSString * groupID = cardModel.abGroupID;
        
        do
        {
            if([sourceID length]==0)
            {
                returnError = PPErrorMake(WCABDataController_Error_InvalidParameters,
                                          @"[sourceID length]==0",
                                          nil);
                break;
            }
            
            CNContact *contact = [[[CNContact alloc] init] autorelease];
            if(contact==nil)
            {
                returnError = PPErrorMake(WCABDataController_Error_FailedToCreatePerson,
                                          @"contact==nil",
                                          nil);
                break;
            }
            
            //////////////////////////////////////////////////
            // set ID photo
            
            if(cardModel.abPhotoImage != nil)
            {
                NSData *photoData = photoData = [self imageDataForSaveWithImage:cardModel.abPhotoImage];

                if(photoData==nil)
                {
                    returnError = PPErrorMake(WCABDataController_Error_FailedToConvertImageData, nil, nil);
                    break;
                }
                
                [contact setValue:photoData forKey:CNContactImageDataKey];
            }
            
            //////////////////////////////////////////////////
            // covert data
            [self convertDataWithMode:ConvertMode_WC_to_AB
                            cardModel:cardModel
                              contact:contact];
            
            // add to Contacts
            NSString *newGroupID = nil;
            [self addContact:contact toSourceID:sourceID groupID:groupID newGroupID:&newGroupID error:&returnError];
            
            //////////////////////////////////////////////////
            if ([newGroupID length]>0)
            {
                cardModel.abGroupID = newGroupID;
            }
            
            cardModel.abPersonID = contact.identifier;

            if (returnError==nil)
            {
                result = YES;
            }            
        }
        while (0);

        //////////////////////////////////////////////////

        if(returnError != nil)
        {
            [returnError retain];
        }
        
    } // end of @autoreleasepool

    //////////////////////////////////////////////////
    [returnError autorelease];
    
    if(error != nil)
    {
        *error = returnError;
    }
    
    return result;
}


//================================================================================
// 更新AddressBook聯絡人
//================================================================================
- (BOOL)updatePersonWithCardModel:(WCABCardModel *)cardModel
                      newSourceID:(NSString *)newSourceID
                       newGroupID:(NSString *)newGroupID
                            error:(NSError **)error
{
    BOOL result = NO;
    NSError *returnError = nil;
    
    @autoreleasepool
    {
        do
        {
            if([newSourceID length]==0)
            {
                returnError = PPErrorMake(WCABDataController_Error_InvalidParameters,
                                          @"[newSourceID length]==0",
                                          nil);
                break;
            }
            
            // 先取得目前聯絡人
            CNContact *currentContact = [self contactWithPersonID:cardModel.abPersonID error:&returnError];
            
            if(currentContact==nil)
            {
                returnError = PPErrorMake(WCABDataController_Error_FailedToGetContact, @"無法取得聯絡人", returnError);
                break;
            }
            
            //////////////////////////////////////////////////
            // get current source ID
            NSString *currentSourceID = [self sourceIDOfPersonWithPersonID:currentContact.identifier];
            
            if(currentSourceID == nil)
            {
                returnError = PPErrorMake(WCABDataController_Error_FailedToGetSourceID, nil, nil);
                break;
            }
            
            //////////////////////////////////////////////////
            
            // 新增到新的source
            NSString * preSourceID = cardModel.abSourceID;
            NSString * preGroupID = cardModel.abGroupID;
            NSString * prePersonID = cardModel.abPersonID;
            
            cardModel.abSourceID = newSourceID;
            cardModel.abGroupID = newGroupID;
            
            // --- 移到不同source ---
            if([currentSourceID isEqualToString:newSourceID]==NO)
            {
                // 刪除在舊source中的資料
                if([self removePersonWithPersonID:prePersonID error:&returnError] == NO)
                {
                    break;
                }
                
                // 加入資料到新的source
                if([self addPersonWithCardModel:cardModel error:&returnError] == NO)
                {
                    // 新增失敗要還原回原來的狀態
                    cardModel.abSourceID = preSourceID;
                    cardModel.abGroupID = preGroupID;
                    cardModel.abPersonID = prePersonID;
                    break;
                }

            }
            else // --- 同一個source ---
            {
                //////////////////////////////////////////////////
                // --- 移到不同group ---
                
                if([preGroupID isEqualToString:cardModel.abGroupID]==NO)
                {
                    if([self moveContact:currentContact fromGroupID:preGroupID toNewGroupID:cardModel.abGroupID error:&returnError]==NO)
                    {
                        // 移動失敗要還原回原來的狀態
                        cardModel.abGroupID = preGroupID;
                        cardModel.abPersonID = prePersonID;
                        break;
                    }
                }
                //////////////////////////////////////////////////
                CNMutableContact *mutableContact = [currentContact mutableCopy];
                [self convertDataWithMode:ConvertMode_WC_to_AB
                                cardModel:cardModel
                                  contact:mutableContact];
                
                [self updateContact:mutableContact error:&returnError];
                
                [mutableContact release];
                
                if (returnError)
                {
                    break;
                }
            }
            

            //////////////////////////////////////////////////
            
            result = YES;
            
        }
        while(0);
        
        //////////////////////////////////////////////////
        
        if(error != nil)
        {
            *error = [returnError retain];
        }
        
    } // end of @autoreleasepool

    //////////////////////////////////////////////////

    if(error != nil)
    {
        [returnError autorelease];
    }
    
    return result;
}


//================================================================================
// 刪除AddressBook聯絡人
//================================================================================
- (BOOL)removePersonWithPersonID:(NSString *)personID error:(NSError **)error
{
    BOOL result = NO;
    NSError *returnError = nil;
    
    do
    {
        [self removeWithPersonID:personID error:&returnError];
        
        result = YES;
    }
    while(0);
 
    //////////////////////////////////////////////////
    
    if(error != nil)
    {
        *error = returnError;
    }

    
    return result;
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance methods (OpenIn)

//================================================================================
//
//================================================================================
- (BOOL)updatePersonPhotoImage:(UIImage *)photoImage withPersonID:(NSString *)personID error:(NSError **)error
{
    BOOL result = NO;
    NSError *returnError = nil;
    
    @autoreleasepool
    {
        do
        {
            // 先取得目前聯絡人
            CNContact *currentContact = [self contactWithPersonID:personID error:&returnError];
            
            if(currentContact==nil)
            {
                returnError = PPErrorMake(WCABDataController_Error_FailedToGetContact, @"無法取得聯絡人", returnError);
                break;
            }
            
            NSData *photoImageData = nil;
            
            if(photoImage)
            {
                photoImageData = [self imageDataForSaveWithImage:photoImage];
            }

            CNSaveRequest *saveRequest = [[CNSaveRequest alloc] init];
            CNMutableContact *mutableContact = [currentContact mutableCopy];
            mutableContact.imageData = photoImageData;
            [saveRequest updateContact:mutableContact];
            [mutableContact release];
            
            [self.contactStore executeSaveRequest:saveRequest error:&returnError];
            [saveRequest release];
            
            result = YES;
        }
        while(0);
        
        //////////////////////////////////////////////////
        
        if(error != nil)
        {
            *error = [returnError retain];
        }
        
    } // end of @autoreleasepool
    
    //////////////////////////////////////////////////

    if(error != nil)
    {
        [returnError autorelease];
    }
    
    return result;
}


//================================================================================
// 回傳nil為無大頭貼
//================================================================================
- (UIImage *)copyPhotoImageWithPersonID:(NSString *)personID error:(NSError **)error
{
    UIImage *photoImage = nil;
    NSError *returnError = nil;
    
    do
    {
        CNContact *contact = [self contactWithPersonID:personID error:&returnError];
        
        if(contact==nil)
        {
            returnError = PPErrorMake(WCABDataController_Error_FailedToGetContact, @"無法取得contact", returnError);
            break;
        }
        
        NSData *imageData = contact.imageData;
        
        if (imageData!=nil)
        {
            photoImage = [[UIImage alloc] initWithData:imageData];
        }
    }
    while(0);
    
    //////////////////////////////////////////////////
    
    if(error != nil)
    {
        *error = returnError;
    }

    return photoImage;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance methods (source)

//================================================================================
//
//================================================================================
- (NSString *)defaultSourceIDWithExcludedGmailAccount:(NSString *)excludedGmailAccount
{
    if ([excludedGmailAccount length]==0)
    {
        return [self.sourceController defaultSourceID];
    }
    else
    {
        NSArray *sourceModels = [self.sourceController copyAllSourcesAndGroups];
        NSString * newSouceID = nil;
        for (WCABSourceModel *sourceModel in sourceModels)
        {
            if ([sourceModel.name compare:@"Gmail" options:NSCaseInsensitiveSearch]!=NSOrderedSame)
            {
                newSouceID = sourceModel.abSourceID;
                break;
            }
        }
        [sourceModels release];
        
        return newSouceID;
    }
}


//================================================================================
//
//================================================================================
- (BOOL)isSourceExist:(NSString *)sourceID
{
    return [self.sourceController isSourceExist:sourceID];
}


//================================================================================
//
//================================================================================
- (NSMutableArray *)copyAllSourcesAndGroups
{
    return [self.sourceController copyAllSourcesAndGroups];
}


//================================================================================
//
//================================================================================
- (NSString *)displayNameWithSourceID:(NSString *)sourceID
                              groupID:(NSString *)groupID
{
    return [self.sourceController displayNameWithSourceID:sourceID
                                                  groupID:groupID];
}


//================================================================================
//
//================================================================================
- (WCABSourceModel *)sourceModelWithSourceID:(NSString *)sourceID
{
    return [self.sourceController sourceModelWithSourceID:sourceID];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance methods (vCard)

//================================================================================
// 目前不讀取影像
//================================================================================
- (WCABCardModel *)cardModelWithVCardData:(NSData *)vCardData error:(NSError **)error
{
    WCABCardModel *cardModel = nil;
    NSError *returnError = nil;

    @autoreleasepool
    {
        CFArrayRef personsRef = nil;
        
        do
        {
            if([vCardData bytes] == 0)
            {
                break;
            }

            
            //////////////////////////////////////////////////
            // !! special case
            //    UTF8資料VERSION要3.0以上，AddressBook轉換中文才不會變亂碼；Unicode資料無此問題。
            //    所以這裡檢查如果是UTF8資料，轉換成Unicode處理。
            
            NSString *utf8Header = @"BEGIN:VCARD";
            NSData *headerData = [NSData dataWithBytes:[vCardData bytes] length:[utf8Header length]];
            NSString *checkHeader = [[[NSString alloc] initWithData:headerData encoding:NSUTF8StringEncoding] autorelease];
            
            // 檢查是否是UTF8格式
            if([checkHeader length] > 0 && [utf8Header isEqualToString:checkHeader] == YES)
            {
                NSString *vCardString = [[NSString alloc] initWithData:vCardData encoding:NSUTF8StringEncoding];
                
                if(vCardString != nil)
                {
                    vCardData = [vCardString dataUsingEncoding:NSUnicodeStringEncoding];
                    [vCardString release];
                }
            }
            
            
            //////////////////////////////////////////////////
            NSError *error = nil;
            NSArray *contacts = [CNContactVCardSerialization contactsWithData:vCardData error:&error];
            
//            personsRef = ABPersonCreatePeopleInSourceWithVCardRepresentation(NULL, (CFDataRef)vCardData);
//            
//            if(personsRef == nil || CFArrayGetCount(personsRef) == 0)
//            {
//                returnError = PPErrorMake(WCABDataController_Error_FailedToConvertVCard, @"ABPersonCreatePeopleInSourceWithVCardRepresentation failed", nil);
//                break;
//            }
//            
//            //////////////////////////////////////////////////
//            
//            ABRecordRef personRef = CFArrayGetValueAtIndex(personsRef, 0);
//            
//            if(personRef == nil)
//            {
//                returnError = PPErrorMake(WCABDataController_Error_FailedToConvertVCard, @"CFArrayGetValueAtIndex failed", nil);
//                break;
//            }
            
            //////////////////////////////////////////////////
            
            // 目前不讀取影像
            //        UIImage *image = [self copyPhotoImageWithPersonID:personID error:&returnError];
            //
            //        if(returnError != nil)
            //        {
            //            break;
            //        }
            //
            //        cardModel.abPersonID = personID;
            //        cardModel.abPhotoImage = image;
            //        [image release];
            
            //////////////////////////////////////////////////
            
            if ([contacts count]>0)
            {
                cardModel = [[WCABCardModel alloc] init];
                
                if(cardModel == nil)
                {
                    returnError = PPErrorMake(WCABDataController_Error_FailedToInitObject, @"WCABCardModel alloc failed", nil);
                    break;
                }
                
                [self convertDataWithMode:ConvertMode_AB_to_WC
                                cardModel:cardModel
                                  contact:[contacts firstObject]];
            }
            
        }
        while (0);
        
        //////////////////////////////////////////////////
        
        if(personsRef != nil)
        {
            CFRelease(personsRef);
        }
        
        //////////////////////////////////////////////////
        
        if(error != nil)
        {
            *error = [returnError retain];
        }
        
    } // end of @autoreleasepool
    
    //////////////////////////////////////////////////
    
    if(error != nil)
    {
        [returnError autorelease];
    }
    
    return [cardModel autorelease];
}


//================================================================================
// 目前不寫入Anniversary/Other Date
// 影像由外部傳入的值決定是否要寫入
//================================================================================
- (NSData *)vCardDataWithCardModel:(WCABCardModel *)cardModel error:(NSError **)error
{
    NSData *vCardData = nil;
    NSError *returnError = nil;
    
    @autoreleasepool
    {
        CNContact *contact = nil;

        do
        {
            if(cardModel == nil)
            {
                returnError = PPErrorMake(WCABDataController_Error_InvalidParameters, @"cardModel is nil", nil);
                break;
            }
            
            //////////////////////////////////////////////////
            
            contact = [[[CNContact alloc] init] autorelease];
            
            if(contact == nil)
            {
                returnError = PPErrorMake(WCABDataController_Error_FailedToInitObject, @"CNContact init failed", nil);
                break;
            }
            
            //////////////////////////////////////////////////
            // Anniversary/Other date轉vcard有問題，先移掉。
            NSMutableArray *fieldModels = [cardModel fieldArrayWithType:WC_FT_Date];

            for(int i=0; i<[fieldModels count]; i++)
            {
                WCFieldModel *fieldModel = [fieldModels objectAtIndex:i];

                if (fieldModel.subType1 == WC_FST1_Date_Anniversary ||
                    fieldModel.subType1 == WC_FST1_Date_Other)
                {
                    [fieldModels removeObjectAtIndex:i];
                    i--;
                }
            }
            
            // 開始轉換
            [self convertDataWithMode:ConvertMode_WC_to_AB
                                        cardModel:cardModel
                                          contact:contact];
            

            
            vCardData = [CNContactVCardSerialization dataWithContacts:@[contact] error:&returnError];
            
            //////////////////////////////////////////////////
            // vcf 大頭貼要另外處理
            if(cardModel.abPhotoImage)
            {
                NSData *photoData = [self imageDataForSaveWithImage:cardModel.abPhotoImage];
                
                NSString* base64ImageString = [photoData base64EncodedStringWithOptions:NSDataBase64Encoding64CharacterLineLength];
                // !!要@"\n "mac上才可以顯示，不知道為何
                base64ImageString = [base64ImageString stringByReplacingOccurrencesOfString:@"\n" withString:@"\n "];
                
                NSString* vcardImageString = [NSString stringWithFormat:@"PHOTO;ENCODING=b;TYPE=JPEG:%@\n", base64ImageString];
                
                // 把照片置換到字串中
                NSString* vcString = [[[NSString alloc] initWithData:vCardData encoding:NSUTF8StringEncoding] autorelease];
                vcString = [vcString stringByReplacingOccurrencesOfString:@"END:VCARD" withString:[vcardImageString stringByAppendingString:@"END:VCARD"]];
                vCardData = [vcString dataUsingEncoding:NSUTF8StringEncoding];
            }
            
            [vCardData retain];
        }
        while (0);

        //////////////////////////////////////////////////
        
        if(error != nil)
        {
            *error = [returnError retain];
        }
        
    } // end of @autoreleasepool
    
    //////////////////////////////////////////////////
    
    if(error != nil)
    {
        [returnError autorelease];
    }
    
    return [vCardData autorelease];
}



@end
