//
//  PPToastController.m
//  WorldCam
//
//  Created by AndrewLai on 2018/10/04.
//  Copyright © 2018年 AndrewLai. All rights reserved.
//

#import "PPToastController.h"

//Extension
#import "UIView+Appearance.h"
#import "UIView+Toast.h"

//View
#import "PPAutoResizedLabel+Factory.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

static UIView *lastToastView = nil;

#define PPToastBGColor [[UIColor lightGrayColor] colorWithAlphaComponent:0.85]
#define PPToastTitleColor [UIColor whiteColor]
#define PPToastTitleFontSize 16

#define PPToastMessageShortShowDuration 2.0
#define PPToastMessageLongShowDuration  3.5

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPToastController Implementation

@implementation PPToastController





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - init/dealloc methods


//==============================================================================
//
//==============================================================================
+ (void)initialize
{
    [[PPToastController appearance] setTextColor:PPToastTitleColor];
    [[PPToastController appearance] setTextFontSize:PPToastTitleFontSize];
    [[PPToastController appearance] setBackgroundColor:PPToastBGColor];
    [[PPToastController appearance] setTextAlignment:NSTextAlignmentCenter];
    [[PPToastController appearance] setDuration:0.0];
    
}


//==============================================================================
//
//==============================================================================
- (instancetype)init
{
    self = [super init];
    if (self)
    {
        [[PPToastController appearance] applyInvocationTo:self];
        
        //////////////////////////////////////////////////
//        _textFontSize = PPToastTitleFontSize;
//        _textAlignment = NSTextAlignmentCenter;
//        _duration = 0.0;
//        _textColor = [PPToastTitleColor retain];
//        _backgroundColor = [PPToastBGColor retain];
    }
    return self;
}


//==============================================================================
//
//==============================================================================
- (void)dealloc
{
    [_textColor release];
    _textColor = nil;
    
    [_backgroundColor release];
    _backgroundColor = nil;
    
    //////////////////////////////////////////////////
    [super dealloc];
}






////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Private Methods


//================================================================================
//
//================================================================================
- (void)showToastFromSuperView:(UIView *)superView
                withCustomView:(UIView *)customView
                      duration:(NSTimeInterval)duration
                      position:(PPToastPosition)position
{
    if (superView==nil)
    {
        return ;
    }
    
    //////////////////////////////////////////////////
    NSString *positionString = (NSString *)CSToastPositionBottom;
    if (position==PPToastPositionTop)
    {
        positionString = (NSString *)CSToastPositionTop;
    }
    else if (position==PPToastPositionCenter)
    {
        positionString = (NSString *)CSToastPositionCenter;
    }
    

    //////////////////////////////////////////////////
    // 先關閉舊的
    [self hideMessageToast];
    
    [superView showToast:customView
                duration:duration
                position:positionString
              completion:^(BOOL didTap) {
                  
              }];
}

//================================================================================
//
//================================================================================
- (void)showMessageToastFromSuperView:(UIView *)superView
                          withMessage:(NSString*)message
                             fontSize:(CGFloat)fontSize
                            textColor:(UIColor *)textColor
                      backgroundColor:(UIColor *)backgroundColor
                        textAlignment:(NSTextAlignment)textAlignment
                             duration:(NSTimeInterval)duration
                             position:(PPToastPosition)position
{
    PPAutoResizedLabel *toastLabel = [PPAutoResizedLabel counterStyleLableWithFont:[UIFont systemFontOfSize:fontSize]
                                                                         textColor:textColor
                                                                   backgroundColor:backgroundColor];
    if (toastLabel)
    {
//        [toastLabel setFrame:CGRectMake(0, 0, superView.bounds.size.width-20, 0)];
        [toastLabel setCornerRadius:5];
        [toastLabel setContentInsets:UIEdgeInsetsMake(7, 7, 7, 7)];
        toastLabel.textAlignment = textAlignment;
        toastLabel.lineBreakMode = NSLineBreakByWordWrapping;
        toastLabel.numberOfLines = 0;
        [toastLabel setAdjustsFontSizeToFitWidth:YES];
        [toastLabel setMaxSize:CGSizeMake(superView.bounds.size.width*0.8, superView.bounds.size.width*0.8)];
        toastLabel.numberOfLines = 10;

        //////////////////////////////////////////////////
        // 計算一行高度
        toastLabel.text = @"a";
        CGFloat oneLineHeight = toastLabel.bounds.size.height-(toastLabel.contentInsets.top+toastLabel.contentInsets.bottom);
        
        toastLabel.text = message;
        [toastLabel sizeToFit];

        // !! 如果duration是<=0，那就依文字長度計算時間
        if (duration<=0.0)
        {
            // 預設2.0
            duration = PPToastMessageShortShowDuration;
            
            CGFloat lineHeight = toastLabel.bounds.size.height - (toastLabel.contentInsets.top+toastLabel.contentInsets.bottom);
            NSInteger numberOfLine = lineHeight/oneLineHeight + 0.5;
            
            // 兩行以上就用3.5秒
            if (numberOfLine>2)
            {
                duration = PPToastMessageLongShowDuration;
            }
        }

        [self showToastFromSuperView:superView
                      withCustomView:toastLabel
                            duration:duration
                            position:position];
        
        lastToastView = [toastLabel retain];

    }
}

//================================================================================
//
//================================================================================
- (void)showMessageToastFromSuperView:(UIView *)superView
                          withMessage:(NSString*)message
                             position:(PPToastPosition)position
{
    [self showMessageToastFromSuperView:superView
                            withMessage:message
                               fontSize:self.textFontSize
                              textColor:self.textColor
                        backgroundColor:self.backgroundColor
                          textAlignment:self.textAlignment
                               duration:self.duration
                               position:position];
}


//================================================================================
//
//================================================================================
- (void)hideMessageToast
{
    [self hideToast:lastToastView];
    [lastToastView release];
    lastToastView = nil;
}


//================================================================================
//
//================================================================================
- (void)hideToast:(UIView *)toast
{
    [UIView animateWithDuration:0.2
                          delay:0.0
                        options:(UIViewAnimationOptionCurveEaseIn | UIViewAnimationOptionBeginFromCurrentState)
                     animations:^{
                         toast.alpha = 0.0;
                     } completion:^(BOOL finished) {
                         [toast removeFromSuperview];
                     }];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPAppearance protocol methods

//================================================================================
//
//================================================================================
+ (id)appearance
{
    // MARK: 實作+appearance的方式固定如下
    return [PPAppearance appearanceForClass:[self class]];
}





////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Appearance Property Method


//================================================================================
//
//================================================================================
- (void)setTextFontSize:(CGFloat)textFontSize
{
    _textFontSize = textFontSize;
}


//================================================================================
//
//================================================================================
- (void)setTextColor:(UIColor *)textColor
{
    [textColor retain];
    [_textColor release];
    _textColor = textColor;
}


//================================================================================
//
//================================================================================
- (void)setBackgroundColor:(UIColor *)backgroundColor
{
    [backgroundColor retain];
    [_backgroundColor release];
    _backgroundColor = backgroundColor;
}


//================================================================================
//
//================================================================================
- (void)setTextAlignment:(NSTextAlignment)textAlignment
{
    _textAlignment = textAlignment;
}


//================================================================================
//
//================================================================================
- (void)setDuration:(NSTimeInterval)duration
{
    _duration = duration;
}



////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Singleton Creating Object

//================================================================================
// 初始化或方便設定，單一物件
//================================================================================
+ (instancetype)sharedInstance
{
    static PPToastController *sharedInstance = nil;
    static dispatch_once_t onceToken;
    
    dispatch_once(&onceToken, ^{
        sharedInstance = [[self alloc] init];
    });
    
    return sharedInstance;
}




////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Class Methods


//==============================================================================
//
//==============================================================================
+ (void)showToastFromSuperView:(UIView *)superView
                withCustomView:(UIView *)customView
                      duration:(NSTimeInterval)duration
                      position:(PPToastPosition)position
{
    [[PPToastController sharedInstance] showToastFromSuperView:superView
                                                withCustomView:customView
                                                      duration:duration
                                                      position:position];
}


//================================================================================
//
//================================================================================
+ (void)showMessageToastFromSuperView:(UIView *)superView
                          withMessage:(NSString*)message
                             fontSize:(CGFloat)fontSize
                            textColor:(UIColor *)textColor
                      backgroundColor:(UIColor *)backgroundColor
                        textAlignment:(NSTextAlignment)textAlignment
                             duration:(NSTimeInterval)duration
                             position:(PPToastPosition)position
{
    [[PPToastController sharedInstance] showMessageToastFromSuperView:superView
                                                          withMessage:message
                                                             fontSize:fontSize
                                                            textColor:textColor
                                                      backgroundColor:backgroundColor
                                                        textAlignment:textAlignment
                                                             duration:duration
                                                             position:position];
}


//================================================================================
//
//================================================================================
+ (void)showMessageToastFromSuperView:(UIView *)superView
                          withMessage:(NSString*)message
                             position:(PPToastPosition)position
{
    [[PPToastController sharedInstance] showMessageToastFromSuperView:superView
                                                          withMessage:message
                                                             fontSize:[PPToastController sharedInstance].textFontSize
                                                            textColor:[PPToastController sharedInstance].textColor
                                                      backgroundColor:[PPToastController sharedInstance].backgroundColor
                                                        textAlignment:[PPToastController sharedInstance].textAlignment
                                                             duration:[PPToastController sharedInstance].duration
                                                             position:position];
}


//================================================================================
//
//================================================================================
+ (void)hideMessageToast
{
    [[PPToastController sharedInstance] hideMessageToast];
}

@end
