//
//  PPSettingsController.m
//

#import "PPSettingsController.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface PPSettingsController ()
@property (atomic, retain) NSUserDefaults *settings;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPSettingsController
@synthesize settings = _settings;

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods

//================================================================================
//
//================================================================================
+ (instancetype)sharedInstance
{
    static id sharedInstance = nil;
    static dispatch_once_t onceToken;
    
    dispatch_once(&onceToken, ^{
        sharedInstance = [[self alloc] init];
    });
    
    return sharedInstance;
}


//================================================================================
//
//================================================================================
- (id)initWithSuiteName:(NSString*)suiteName
{
    self = [super init];
    
    if (self)
    {
        if (@available(iOS 7.0, *)) {
            _settings = [[NSUserDefaults alloc] initWithSuiteName:suiteName];
        } else {
            // Fallback on earlier versions
            _settings = [[NSUserDefaults alloc] init];
        }
    }
    
    return self;
}



//================================================================================
//
//================================================================================
- (id)init
{
    self = [super init];
    
    if (self)
    {
        _settings = [[NSUserDefaults alloc] init];
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    self.settings = nil;
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods

//================================================================================
// !! 設定預設值，只有沒有設定值時才會寫入。
//================================================================================
- (void)setDefaultArrayValue:(NSArray *)arrayValue withKey:(NSString *)key
{
    if([key length] && arrayValue != nil)
    {
        [self.settings registerDefaults:@{key:arrayValue}];
    }
}


//================================================================================
//
//================================================================================
- (void)setArrayValue:(NSArray *)arrayValue withKey:(NSString *)key
{
    if([key length] && arrayValue != nil)
    {
        [self.settings setObject:arrayValue forKey:key];
        [self.settings synchronize];
    }
}


//================================================================================
//
//================================================================================
- (NSArray *)arrayValueWithKey:(NSString *)key
{
    NSArray *value = nil;
    
    if([key length])
    {
        value = [self.settings objectForKey:key];
    }
    
    return value;
}


//================================================================================
// !! 設定預設值，只有沒有設定值時才會寫入。
//================================================================================
- (void)setDefaultDictionaryValue:(NSDictionary *)dictionaryValue withKey:(NSString *)key
{
    if([key length] && dictionaryValue != nil)
    {
        [self.settings registerDefaults:@{key:dictionaryValue}];
    }
}


//================================================================================
//
//================================================================================
- (void)setDictionaryValue:(NSDictionary *)dictionaryValue withKey:(NSString *)key
{
    if([key length] && dictionaryValue != nil)
    {
        [self.settings setObject:dictionaryValue forKey:key];
        [self.settings synchronize];
    }
}


//================================================================================
//
//================================================================================
- (NSDictionary *)dictionaryValueWithKey:(NSString *)key
{
    NSDictionary *value = nil;
    
    if([key length])
    {
        value = [self.settings objectForKey:key];
    }
    
    return value;
}


//================================================================================
// !! 設定預設值，只有沒有設定值時才會寫入。
//================================================================================
- (void)setDefaultDataValue:(NSData *)dataValue withKey:(NSString *)key
{
    if([key length] && dataValue != nil)
    {
        [self.settings registerDefaults:@{key:dataValue}];
    }
}


//================================================================================
//
//================================================================================
- (void)setDataValue:(NSData *)dataValue withKey:(NSString *)key
{
    if([key length] && dataValue != nil)
    {
        [self.settings setObject:dataValue forKey:key];
        [self.settings synchronize];
    }
}


//================================================================================
//
//================================================================================
- (NSData *)dataValueWithKey:(NSString *)key
{
    NSData *value = nil;
    
    if([key length])
    {
        value = [self.settings dataForKey:key];
    }
    
    return value;
}


//================================================================================
// !! 設定預設值，只有沒有設定值時才會寫入。
//================================================================================
- (void)setDefaultIntegerValue:(NSInteger)integerValue withKey:(NSString *)key
{
    if([key length])
    {
        [self.settings registerDefaults:@{key:[NSNumber numberWithInteger:integerValue]}];
    }
}


//================================================================================
//
//================================================================================
- (void)setIntegerValue:(NSInteger)integerValue withKey:(NSString *)key
{
    if([key length] && integerValue != PPSettingsController_UnsetIntValue)
    {
        NSNumber *intNumber = [NSNumber numberWithInteger:integerValue];
        
        [self.settings setObject:intNumber forKey:key];
        [self.settings synchronize];    
    }
}


//================================================================================
//
//================================================================================
- (NSInteger)integerValueWithKey:(NSString *)key
{
    NSInteger intValue = PPSettingsController_UnsetIntValue;
    
    if([key length])
    {
        id object = [self.settings objectForKey:key];
        
        if(object)
        {
            intValue = [object integerValue];
        }
    }

    return intValue;
}


//================================================================================
// !! 設定預設值，只有沒有設定值時才會寫入。
//================================================================================
- (void)setDefaultStringValue:(NSString *)stringValue withKey:(NSString *)key
{
    if([key length] && stringValue != nil)
    {
        [self.settings registerDefaults:@{key:stringValue}];
    }
}


//================================================================================
//
//================================================================================
- (void)setStringValue:(NSString *)stringValue withKey:(NSString *)key
{
    if([key length] && stringValue != nil)
    {
        [self.settings setObject:stringValue forKey:key];
        [self.settings synchronize];
    }
}


//================================================================================
//
//================================================================================
- (NSString *)stringValueWithKey:(NSString *)key
{
    NSString *value = nil;
    
    if([key length])
    {
        value = [self.settings stringForKey:key];
    }
    
    return value;
}


//================================================================================
//
//================================================================================
- (void)setDefaultDateValue:(NSDate *)dateValue withKey:(NSString *)key
{
    if([key length] && dateValue != nil)
    {
        [self.settings registerDefaults:@{key:dateValue}];
    }
}


//================================================================================
//
//================================================================================
- (void)setDateValue:(NSDate *)dateValue withKey:(NSString *)key
{
    if([key length] && dateValue != nil)
    {
        [self.settings setObject:dateValue forKey:key];
        [self.settings synchronize];
    }
}


//================================================================================
//
//================================================================================
- (NSDate *)dateValueWithKey:(NSString *)key;
{
    NSDate *value = nil;
    
    if([key length])
    {
        value = [self.settings objectForKey:key];
    }
    
    return value;
}


//================================================================================
//
//================================================================================
- (void)setDefaultNumberValue:(NSNumber *)numberValue withKey:(NSString *)key
{
    if([key length] && numberValue != nil)
    {
        [self.settings registerDefaults:@{key:numberValue}];
    }
}


//================================================================================
//
//================================================================================
- (void)setNumberValue:(NSNumber *)numberValue withKey:(NSString *)key
{
    if([key length] && numberValue != nil)
    {
        [self.settings setObject:numberValue forKey:key];
        [self.settings synchronize];
    }
}


//================================================================================
//
//================================================================================
- (NSNumber *)numberValueWithKey:(NSString *)key
{
    NSNumber *value = nil;
    
    if([key length])
    {
        value = [self.settings objectForKey:key];
    }
    
    return value;
}


//================================================================================
//
//================================================================================
- (void)removeValueWithKey:(NSString *)key
{
    if([key length])
    {
        [self.settings removeObjectForKey:key];
        [self.settings synchronize];
    }
}


//================================================================================
//
//================================================================================
- (NSDictionary *)allSettingsDict
{
    NSString *domainName = [[NSBundle mainBundle] bundleIdentifier];
    
    return [self.settings persistentDomainForName:domainName];
}


//================================================================================
//
//================================================================================
- (void)setAllSettingsWithDict:(NSDictionary *)dict
{
    NSString *domainName = [[NSBundle mainBundle] bundleIdentifier];
    
    [self.settings setPersistentDomain:dict forName:domainName];
}


//================================================================================
//
//================================================================================
- (void)removeAllSettings
{
    NSString *domainName = [[NSBundle mainBundle] bundleIdentifier];

    [self.settings removePersistentDomainForName:domainName];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class methods

//================================================================================
//
//================================================================================
+ (void)setDefaultArrayValue:(NSArray *)arrayValue withKey:(NSString *)key
{
    [[PPSettingsController sharedInstance] setDefaultArrayValue:arrayValue withKey:key];
}


//================================================================================
//
//================================================================================
+ (void)setArrayValue:(NSArray *)arrayValue withKey:(NSString *)key
{
    [[PPSettingsController sharedInstance] setArrayValue:arrayValue withKey:key];
}


//================================================================================
//
//================================================================================
+ (NSArray *)arrayValueWithKey:(NSString *)key
{
    return [[PPSettingsController sharedInstance] arrayValueWithKey:key];
}


//================================================================================
//
//================================================================================
+ (void)setDefaultDictionaryValue:(NSDictionary *)dictionaryValue withKey:(NSString *)key
{
    [[PPSettingsController sharedInstance] setDefaultDictionaryValue:dictionaryValue withKey:key];
}


//================================================================================
//
//================================================================================
+ (void)setDictionaryValue:(NSDictionary *)dictionaryValue withKey:(NSString *)key
{
    [[PPSettingsController sharedInstance] setDictionaryValue:dictionaryValue withKey:key];
}


//================================================================================
//
//================================================================================
+ (NSDictionary *)dictionaryValueWithKey:(NSString *)key
{
    return [[PPSettingsController sharedInstance] dictionaryValueWithKey:key];
}


//================================================================================
//
//================================================================================
+ (void)setDefaultDataValue:(NSData *)dataValue withKey:(NSString *)key
{
    [[PPSettingsController sharedInstance] setDefaultDataValue:dataValue withKey:key];
}


//================================================================================
//
//================================================================================
+ (void)setDataValue:(NSData *)dataValue withKey:(NSString *)key
{
    [[PPSettingsController sharedInstance] setDataValue:dataValue withKey:key];
}


//================================================================================
//
//================================================================================
+ (NSData *)dataValueWithKey:(NSString *)key
{
    return [[PPSettingsController sharedInstance] dataValueWithKey:key];
}


//================================================================================
//
//================================================================================
+ (void)setDefaultIntegerValue:(NSInteger)integerValue withKey:(NSString *)key
{
    [[PPSettingsController sharedInstance] setDefaultIntegerValue:integerValue withKey:key];
}


//================================================================================
//
//================================================================================
+ (void)setIntegerValue:(NSInteger)integerValue withKey:(NSString *)key
{
    [[PPSettingsController sharedInstance] setIntegerValue:integerValue withKey:key];
}


//================================================================================
//
//================================================================================
+ (NSInteger)integerValueWithKey:(NSString *)key
{
    return [[PPSettingsController sharedInstance] integerValueWithKey:key];
}


//================================================================================
//
//================================================================================
+ (void)setDefaultStringValue:(NSString *)stringValue withKey:(NSString *)key
{
    [[PPSettingsController sharedInstance] setDefaultStringValue:stringValue withKey:key];
}


//================================================================================
//
//================================================================================
+ (void)setStringValue:(NSString *)stringValue withKey:(NSString *)key
{
    [[PPSettingsController sharedInstance] setStringValue:stringValue withKey:key];
}


//================================================================================
//
//================================================================================
+ (NSString *)stringValueWithKey:(NSString *)key
{
    return [[PPSettingsController sharedInstance] stringValueWithKey:key];
}


//================================================================================
//
//================================================================================
+ (void)setDefaultDateValue:(NSDate *)dateValue withKey:(NSString *)key
{
    [[PPSettingsController sharedInstance] setDefaultDateValue:dateValue withKey:key];
}


//================================================================================
//
//================================================================================
+ (void)setDateValue:(NSDate *)dateValue withKey:(NSString *)key
{
    [[PPSettingsController sharedInstance] setDateValue:dateValue withKey:key];
}


//================================================================================
//
//================================================================================
+ (NSDate *)dateValueWithKey:(NSString *)key
{
    return [[PPSettingsController sharedInstance] dateValueWithKey:key];
}


//================================================================================
//
//================================================================================
+ (void)setDefaultNumberValue:(NSNumber *)numberValue withKey:(NSString *)key
{
    [[PPSettingsController sharedInstance] setDefaultNumberValue:numberValue withKey:key];
}


//================================================================================
//
//================================================================================
+ (void)setNumberValue:(NSNumber *)numberValue withKey:(NSString *)key
{
    [[PPSettingsController sharedInstance] setNumberValue:numberValue withKey:key];
}


//================================================================================
//
//================================================================================
+ (NSNumber *)numberValueWithKey:(NSString *)key
{
    return [[PPSettingsController sharedInstance] numberValueWithKey:key];
}


//================================================================================
//
//================================================================================
+ (void)removeValueWithKey:(NSString *)key
{
    [[PPSettingsController sharedInstance] removeValueWithKey:key];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Class methods - All settings

//================================================================================
//
//================================================================================
+ (NSDictionary *)allSettingsDict
{
    return [[PPSettingsController sharedInstance] allSettingsDict];
}


//================================================================================
//
//================================================================================
+ (void)setAllSettingsWithDict:(NSDictionary *)dict
{
    [[PPSettingsController sharedInstance] setAllSettingsWithDict:dict];
}


//================================================================================
//
//================================================================================
+ (void)removeAllSettings
{
    [[PPSettingsController sharedInstance] removeAllSettings];
}

@end
