//
//  PPSQLiteDBController.m
//

#import "PPSQLiteDBController.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

#define PPSQLDBC_MaxBusyRetry   10      // 遇到資料庫忙碌狀態時最多重試次數
#define PPSQLDBC_RetryInterval  0.1     // 重試的等待時間

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface PPSQLiteDBController ()

@property (atomic, retain, readwrite) NSString *dbPath;
@property (atomic, assign) NSInteger connectErrorCode;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPSQLiteDBController

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma - Life cycle methods

//===============================================================================
// 
//===============================================================================
- (id)initWithPath:(NSString *)dbPath
{
	if([dbPath length] == 0)
    {
		return nil;
    }

	if(self = [super init])
	{
		self.dbPath = dbPath;
	}
	
	return self;
}


//===============================================================================
// 
//===============================================================================
- (void)dealloc 
{
	[self disconnectDB];
    
    self.dbPath = nil;
    self.lastError = nil;
    
	[super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance methods (資料庫)

//===============================================================================
//
//===============================================================================
- (BOOL)isDBExist
{
    return [[NSFileManager defaultManager] fileExistsAtPath:self.dbPath];
}


//===============================================================================
//
//===============================================================================
- (void)resetDBPath:(NSString *)dbPath
{
    if([dbPath length] > 0 && [self.dbPath isEqualToString:dbPath] == NO)
    {
        // 路徑變動，要中斷目前資料庫重新連線。
        [self disconnectDB];
        
        self.dbPath = dbPath;
    }
}


//===============================================================================
// connect database
//===============================================================================
- (BOOL)connectDB
{
    if(self.dbHandle != nil)
    {
        return YES;
    }
    
    NSInteger result = sqlite3_open([self.dbPath UTF8String], &_dbHandle);
    
    if(result != SQLITE_OK)
    {
        self.connectErrorCode = result;
    }
    
    return (result == SQLITE_OK);
}


//===============================================================================
// disconnect database
//===============================================================================
- (void)disconnectDB
{
    if(self.dbHandle)
    {
        sqlite3_close(self.dbHandle);
        self.dbHandle = nil;
    }	
}


//===============================================================================
// compact database
// 移除database/table fragmentation，保持資料庫效能。
// !! 一定要找機會壓縮資料庫，不然會越長越大。
//===============================================================================
- (void)compactDB
{
    [self runSqlCommand:@"VACUUM;"];
}


//===============================================================================
//
//===============================================================================
- (BOOL)isTableExist:(NSString *)tableName
{
    NSString *sqlCommand = [NSString stringWithFormat:@"SELECT COUNT(*) FROM sqlite_master WHERE type='table' AND name='%@';", tableName];
    NSInteger result = [self intResultWithSelectCommand:sqlCommand];
    
    return (result > 0);
}


//===============================================================================
//
//===============================================================================
- (BOOL)isIndexExist:(NSString *)indexName
{
    NSString *sqlCommand = [NSString stringWithFormat:@"SELECT COUNT(*) FROM sqlite_master WHERE type='index' AND name='%@';", indexName];
    NSInteger result = [self intResultWithSelectCommand:sqlCommand];
    
    return (result > 0);
}



//==============================================================================
//
//==============================================================================
- (BOOL)isColumn:(NSString *)columnName inTable:(NSString *)tableName
{
    if ([columnName length]==0 ||
        [tableName length]==0)
    {
        return NO;
    }
    
    BOOL result = NO;
    self.lastError = nil;
    
    NSString *sqlCommand = [NSString stringWithFormat:@"PRAGMA table_info(%@);", tableName];    
    sqlite3_stmt *stmt;
    
    //////////////////////////////////////////////////
    
    if(sqlite3_prepare_v2(self.dbHandle, [sqlCommand UTF8String], -1, &stmt, NULL) == SQLITE_OK)
    {
        while (sqlite3_step(stmt) == SQLITE_ROW)
        {
            //-------------------------------------------------
            // handel result
            //-------------------------------------------------
            // name是在column 1
            NSString *name = [self stringFromStmt:stmt column:1];

            if ([name isEqualToString:columnName])
            {
                result = YES;
                break;
            }
        }
        
        sqlite3_finalize(stmt);
    }
    else
    {
        [self dbErrorMessage];
        return result;
    }
    
    return result;
}

//===============================================================================
//
//===============================================================================
- (void)beginTransaction
{
    [self runSqlCommand:@"BEGIN;"];
}


//===============================================================================
//
//===============================================================================
- (void)endTransaction
{
    [self runSqlCommand:@"END;"];
}


//===============================================================================
//
//===============================================================================
- (BOOL)commitTransaction
{
    return [self runSqlCommand:@"COMMIT;"];
}


//===============================================================================
//
//===============================================================================
- (void)rollbackTransaction
{
    [self runSqlCommand:@"ROLLBACK;"];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance methods (指令操作)

//===============================================================================
//
//===============================================================================
- (BOOL)createDBWithCommands:(NSArray *)commandArray
{
    BOOL result = YES;
    
    // create DB and tables if not exist
    if([self connectDB])
    {
        for(NSString *tableCreate in commandArray)
        {
            if(![self runSqlCommand:tableCreate])
            {
                result = NO;
                break;
            }
        }
        
        // disconnect database
        [self disconnectDB];
        
        // delete database file if create failed
        if(!result)
        {
            [[NSFileManager defaultManager] removeItemAtPath:self.dbPath error:nil];
        }
    }
    else
    {
        result = NO;
    }
    
    return result;
}


//===============================================================================
// RETURN: -1代表資料庫未連接
//===============================================================================
- (NSInteger)dbErrorCode
{
    NSInteger errcode = -1;
    
    if(self.dbHandle != nil)
    {
        errcode = sqlite3_errcode(self.dbHandle);
    }
    
    return errcode;
}


//===============================================================================
//
//===============================================================================
- (NSString *)dbErrorMessage
{
    NSInteger   errcode = 0;
    NSString    *errmsg = @"";
    NSString    *message = @"";    
    
    if(self.dbHandle != nil)
    {
        errcode = sqlite3_errcode(self.dbHandle);
        errmsg = [NSString stringWithUTF8String:(char*)sqlite3_errmsg(self.dbHandle)];
        message = [NSString stringWithFormat:@"%@ (%ld)", errmsg, (long)errcode];
    }
    else
    {
        errcode = -1;
        message = [NSString stringWithFormat:@"Database not connected (error code : %ld)", (long)self.connectErrorCode];
    }
    

#ifdef DEBUG
    
    if(errcode != 0)
    {
        NSLog(@"=== SQLite error ===\n(%d) %@\n", (int)errcode, message);
    }
    
#endif    
    
    return message;
}


//===============================================================================
//
//===============================================================================
- (NSString *)sqlString:(NSString *)input
{
    //////////////////////////////////////////////////
    // 舊作法
    
//	if([input length] > 0)
//	{
//		NSString *result = [input stringByReplacingOccurrencesOfString:@"'" withString: @"''"];
//        result = [result stringByReplacingOccurrencesOfString:@"\0"
//                                                   withString:@""
//                                                      options:NSCaseInsensitiveSearch
//                                                        range:NSMakeRange(0, [result length])];
//		return result;
//	}
//	else
//    {
//        return @"";
//    }

    
    //////////////////////////////////////////////////
    // 新作法
    
    NSString *formattedString = @"";
    
    if([input length] > 0)
    {
        char *cString = sqlite3_mprintf("%q", [input UTF8String]);
        
        if(cString!=NULL)
        {
            formattedString = [NSString stringWithCString:cString encoding:NSUTF8StringEncoding];
            sqlite3_free(cString);
        }
    }
    
    return formattedString;
}


//===============================================================================
// 執行 sql command
//===============================================================================
- (BOOL)runSqlCommand:(NSString *)sqlCommand
{
	sqlite3_stmt	*stmt;
	char			*utf8Command = nil;
    int             result;
	
	if(!self.dbHandle || !(utf8Command = (char*)[sqlCommand UTF8String]))
    {
		return NO;
    }
	
    for(int i=0; i<PPSQLDBC_MaxBusyRetry; i++)
    {
        // !!用sqlite3_exec會有leak
        result = sqlite3_prepare_v2(self.dbHandle, utf8Command, -1, &stmt, NULL);
        
        if(result == SQLITE_OK)
        {
            result = sqlite3_step(stmt);
            
            switch (result)
            {
                case SQLITE_BUSY: // try again
                {
                    sqlite3_finalize(stmt);
                    
                    @autoreleasepool
                    {
                        [[NSRunLoop currentRunLoop] runUntilDate:[NSDate dateWithTimeIntervalSinceNow:PPSQLDBC_RetryInterval]];
                    }

                    break;
                }
                    
                case SQLITE_DONE: // success
                {
                    sqlite3_finalize(stmt);
                    return YES;
                }
                    
                default: // failed
                {
                    
#ifdef DEBUG
                    NSLog(@"==> runSqlCommand:%@\n sqlite3_step failed (%d)", sqlCommand, result);
                    [self dbErrorMessage];
#endif                    
                    
                    sqlite3_finalize(stmt);
                    return NO;
                }
            }
        }
        else
        {
            
#ifdef DEBUG
            NSLog(@"==> runSqlCommand:%@\n sqlite3_prepare_v2 failed (%d)", sqlCommand, result);
            [self dbErrorMessage];
#endif
            
            break;
        }
    }
    
	return NO;	
}


//===============================================================================
// 執行 sql command 取得單一integer回傳值
// return -1 if failed
//===============================================================================
- (NSInteger)intResultWithSelectCommand:(NSString *)sqlCommand
{
	sqlite3_stmt	*stmt;
	char			*utf8Command = nil;
	int				result = -1;
	
	
	if(!self.dbHandle || !(utf8Command = (char*)[sqlCommand UTF8String]))
    {
		return NO;
    }
	
    for(int i=0; i<PPSQLDBC_MaxBusyRetry; i++)
    {
        if(sqlite3_prepare_v2(self.dbHandle, utf8Command, -1, &stmt, NULL) == SQLITE_OK)
        {
            switch (sqlite3_step(stmt)) 
            {
                case SQLITE_BUSY: // try again
                {
                    sqlite3_finalize(stmt);
                    
                    @autoreleasepool
                    {
                        [[NSRunLoop currentRunLoop] runUntilDate:[NSDate dateWithTimeIntervalSinceNow:PPSQLDBC_RetryInterval]];
                    }
                    
                    break;
                }
                    
                case SQLITE_ROW: // success
                {
                    result = sqlite3_column_int(stmt, 0);
                    sqlite3_finalize(stmt);
                    return result;
                }
                    
                default: // failed
                {
                    sqlite3_finalize(stmt);
                    return result;
                }
            }
        }
        else
        {
            
#ifdef DEBUG
            NSLog(@"==> intResultWithSelectCommand:%@\n sqlite3_prepare_v2 failed", sqlCommand);
            [self dbErrorMessage];
#endif
            
            break;
        }
    }

	return result;	
}


//===============================================================================
// 執行 sql command 取得單一string回傳值
// return nil if failed or not exist
//===============================================================================
- (NSString *)stringResultWithSelectCommand:(NSString *)sqlCommand
{
    sqlite3_stmt *stmt;
    char *utf8Command = nil;
    NSString *result = nil;
    
    
    if(!self.dbHandle || !(utf8Command = (char*)[sqlCommand UTF8String]))
        return nil;
    
    for(int i=0; i<PPSQLDBC_MaxBusyRetry; i++)
    {
        if(sqlite3_prepare_v2(self.dbHandle, utf8Command, -1, &stmt, NULL) == SQLITE_OK)
        {
            switch (sqlite3_step(stmt))
            {
                case SQLITE_BUSY: // try again
                {
                    sqlite3_finalize(stmt);

                    @autoreleasepool
                    {
                        [[NSRunLoop currentRunLoop] runUntilDate:[NSDate dateWithTimeIntervalSinceNow:PPSQLDBC_RetryInterval]];
                    }

                    break;
                }
                    
                case SQLITE_ROW: // success
                {
                    result = [self stringFromStmt:stmt column:0];
                    sqlite3_finalize(stmt);
                    return result;
                }
                    
                default: // failed
                {
                    sqlite3_finalize(stmt);
                    return result;
                }
            }
        }
        else
        {
            
#ifdef DEBUG
            NSLog(@"==> stringResultWithSelectCommand:%@\n sqlite3_prepare_v2 failed", sqlCommand);
            [self dbErrorMessage];
#endif
            
            break;
        }
    }
    
    return result;	
}


//===============================================================================
//
//===============================================================================
- (BOOL)hasRecord:(NSString *)sqlCommand
{
    NSInteger result = [self intResultWithSelectCommand:sqlCommand];
    
    return (result > 0);
}


//===============================================================================
// get record count in db table
//===============================================================================
- (NSInteger)recordCountInTable:(NSString*)tableName
{	
    NSString *sqlCommand = [NSString stringWithFormat:@"SELECT COUNT(*) FROM %@", tableName];
    
    return [self intResultWithSelectCommand:sqlCommand];
}


//===============================================================================
// 
//===============================================================================
- (NSInteger)integerFromStmt:(sqlite3_stmt *)stmt column:(int)column
{
    return sqlite3_column_int(stmt, column);
}


//===============================================================================
// 
//===============================================================================
- (NSString *)stringFromStmt:(sqlite3_stmt *)stmt column:(int)column
{
    const char *utf8String = (const char *)sqlite3_column_text(stmt, column);
    
    return (utf8String ? [NSString stringWithUTF8String:utf8String] : nil);
}


//==============================================================================
//
//==============================================================================
- (NSDate *)dateFromStmt:(sqlite3_stmt *)stmt column:(int)column
{
    double timeIntervalSecond = sqlite3_column_double(stmt, column);
    return [NSDate dateWithTimeIntervalSince1970:timeIntervalSecond];
}


@end
