//
//  PPRefreshControl.m
//  
//
//  Created by Mike.Shih on 13/9/16.
//  Copyright (c) 2013年 Mike.Shih. All rights reserved.
//

#import "PPRefreshControl.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

#define PPRefreshControl_DefaultAnimationDuration   0.3
#define PPRefreshControl_DefaultHeight              44
#define PPRefreshControl_DefaultRadius              6
#define PPRefreshControl_DefaultLength              3
#define PPRefreshControl_DefaultWidth               2
#define PPRefreshControl_WhiteLargeRadius           10
#define PPRefreshControl_WhiteLargeLength           6
#define PPRefreshControl_WhiteLargeWidth            3

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - PPRefreshControl()

@interface PPRefreshControl ()
@property(nonatomic,assign) BOOL    canRefresh;
@property(nonatomic,assign) BOOL    modifyContentInset;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPRefreshControl

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Synthesize

@synthesize
insetHeight             = insetHeight_,
activityIndicatorView   = activityIndicatorView_;

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Creating, Copying, and Deallocating Objects

//================================================================================
//
//================================================================================
- (id)init
{
    if((self=[super init]))
    {
        self.enabled = NO;
        
        //////////////////////////////////////////////////
        
        self.insetHeight = PPRefreshControl_DefaultHeight;
        
        //////////////////////////////////////////////////
        
        activityIndicatorView_ = [[UIActivityIndicatorView alloc] initWithActivityIndicatorStyle:UIActivityIndicatorViewStyleWhiteLarge];
        if(self.activityIndicatorView!=nil)
        {
            [self addSubview:self.activityIndicatorView];
        }
    }
	
    return self;
}

//================================================================================
//
//================================================================================
- (void)dealloc
{
    if(self.activityIndicatorView!=nil)
    {
        [activityIndicatorView_ removeFromSuperview];
        [activityIndicatorView_ release];
        activityIndicatorView_ = nil;
	}
    
    //////////////////////////////////////////////////
    
    [super dealloc];
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Laying out Subviews

//================================================================================
//
//================================================================================
- (void)layoutSubviews
{
    [super layoutSubviews];
    
    //////////////////////////////////////////////////
    
    if(self.superview!=nil && [self.superview isKindOfClass:[UIScrollView class]]==YES && self.activityIndicatorView!=nil)
    {
        UIScrollView *scrollView = (UIScrollView *)self.superview;
        
        self.activityIndicatorView.center = CGPointMake(self.bounds.size.width/2,
                                                        scrollView.contentOffset.y+scrollView.contentInset.top+self.frame.size.height+(self.insetHeight/2));
    }
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Managing the View Hierarchy

//================================================================================
//
//================================================================================
- (void)willMoveToSuperview:(UIView *)superview
{
    [super willMoveToSuperview:superview];
    
    //////////////////////////////////////////////////
    
    do
    {
        if(self.superview==nil || [self.superview isKindOfClass:[UIScrollView class]]==NO)
        {
            break;
        }
        
        UIScrollView *scrollView = (UIScrollView *)self.superview;
        
        //////////////////////////////////////////////////
        
        [self endRefreshing];
        
        //////////////////////////////////////////////////
        
        [scrollView removeObserver:self forKeyPath:@"contentInset"];
        [scrollView removeObserver:self forKeyPath:@"contentOffset"];
        [scrollView removeObserver:self forKeyPath:@"contentSize"];
        [scrollView removeObserver:self forKeyPath:@"frame"];
        
    }while(0);
}

//================================================================================
//
//================================================================================
- (void)didMoveToSuperview
{
    [super didMoveToSuperview];
    
    //////////////////////////////////////////////////
    
    do
    {
        if(self.superview==nil || [self.superview isKindOfClass:[UIScrollView class]]==NO)
        {
            break;
        }
        
        UIScrollView *scrollView = (UIScrollView *)self.superview;
        
        //////////////////////////////////////////////////
        
        self.frame = CGRectMake(0,
                                -(scrollView.contentInset.top+MAX(scrollView.bounds.size.height, self.insetHeight)-(self.modifyContentInset==YES?self.insetHeight:0)),
                                scrollView.bounds.size.width,
                                MAX(scrollView.bounds.size.height, self.insetHeight));
        
        self.canRefresh = YES;
        
        //////////////////////////////////////////////////
        
        [scrollView addObserver:self forKeyPath:@"contentInset"     options:NSKeyValueObservingOptionNew context:NULL];
        [scrollView addObserver:self forKeyPath:@"contentOffset"    options:NSKeyValueObservingOptionNew context:NULL];
        [scrollView addObserver:self forKeyPath:@"contentSize"      options:NSKeyValueObservingOptionNew context:NULL];
        [scrollView addObserver:self forKeyPath:@"frame"            options:NSKeyValueObservingOptionNew context:NULL];
        
    }while(0);
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Drawing and Updating the View

//================================================================================
//
//================================================================================
- (void)drawRect:(CGRect)rect
{
    [super drawRect:rect];
    
    //////////////////////////////////////////////////
    
    do
    {
        if(self.superview==nil                                      ||
           [self.superview isKindOfClass:[UIScrollView class]]==NO  ||
           CGRectIsEmpty(self.bounds)==YES                          ||
           self.activityIndicatorView==nil                          ||
           self.canRefresh==NO                                      ||
           [self isRefreshing]==YES                                  )
        {
            break;
        }
        
        UIScrollView *scrollView = (UIScrollView *)self.superview;
        
        //////////////////////////////////////////////////
        
        if(scrollView.isDragging==NO || scrollView.isTracking==NO)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        CGContextRef context = UIGraphicsGetCurrentContext();
        if(context==NULL)
        {
            break;
        }
        
        //////////////////////////////////////////////////
        
        CGFloat length  = 0;
        CGFloat radius  = 0;
        CGFloat width   = 0;
        
        switch(self.activityIndicatorView.activityIndicatorViewStyle)
        {
            case UIActivityIndicatorViewStyleWhiteLarge:
            {
                length  = PPRefreshControl_WhiteLargeLength;
                radius  = PPRefreshControl_WhiteLargeRadius;
                width   = PPRefreshControl_WhiteLargeWidth;
                
                break;
            }
            case UIActivityIndicatorViewStyleWhite:
            case UIActivityIndicatorViewStyleGray:
            default:
            {
                length  = PPRefreshControl_DefaultLength;
                radius  = PPRefreshControl_DefaultRadius;
                width   = PPRefreshControl_DefaultWidth;
                
                break;
            }
        }
        
        //////////////////////////////////////////////////
        
        NSUInteger progress = MIN(MAX(3*(self.frame.size.height-self.activityIndicatorView.center.y+radius)/(radius+length), 0), 12);
        if(progress!=0)
        {
            CGContextSaveGState(context);
            CGContextTranslateCTM(context, self.activityIndicatorView.center.x, self.activityIndicatorView.center.y);
            CGContextScaleCTM(context, 1, -1);
            CGContextSetLineCap(context, kCGLineCapRound);
            CGContextSetLineWidth(context, width);
            
            if(self.activityIndicatorView.activityIndicatorViewStyle==UIActivityIndicatorViewStyleGray)
            {
                CGContextSetStrokeColorWithColor(context, [UIColor grayColor].CGColor);
            }
            else
            {
                CGContextSetStrokeColorWithColor(context, [UIColor whiteColor].CGColor);
            }
            
            for(NSUInteger index=0; index<progress; index++)
            {
                CGContextMoveToPoint(context, 0, radius);
                CGContextAddLineToPoint(context, 0, radius+length);
                CGContextRotateCTM(context, -30*M_PI/180);
            }
            
            CGContextStrokePath(context);
            CGContextRestoreGState(context);
            
            //////////////////////////////////////////////////
            
            if(progress==12)
            {
                [self sendActionsForControlEvents:UIControlEventValueChanged];
            }
        }
        
    }while(0);
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - NSKeyValueObserving

//================================================================================
//
//================================================================================
- (void)observeValueForKeyPath:(NSString *)keyPath ofObject:(id)object change:(NSDictionary *)change context:(void *)context
{
    do
    {
        if(self.superview==nil || [self.superview isKindOfClass:[UIScrollView class]]==NO || self.activityIndicatorView==nil)
        {
            break;
        }
        
        __block UIScrollView *blockScrollView = (UIScrollView *)self.superview;
        
        //////////////////////////////////////////////////
        
        if([keyPath isEqualToString:@"contentInset"]==YES || [keyPath isEqualToString:@"contentSize"]==YES || [keyPath isEqualToString:@"frame"]==YES)
        {
            self.frame = CGRectMake(0,
                                    -(blockScrollView.contentInset.top+MAX(blockScrollView.bounds.size.height, self.insetHeight))+(self.modifyContentInset==YES?self.insetHeight:0),
                                    blockScrollView.bounds.size.width,
                                    MAX(blockScrollView.bounds.size.height, self.insetHeight));
        }
        else if([keyPath isEqualToString:@"contentOffset"]==YES)
        {
            if(self.enabled==NO)
            {
                break;
            }
            
            //////////////////////////////////////////////////
            
            self.activityIndicatorView.center = CGPointMake(self.bounds.size.width/2,
                                                            blockScrollView.contentOffset.y+blockScrollView.contentInset.top+self.frame.size.height+(self.insetHeight/2)-(self.modifyContentInset==YES?self.insetHeight:0));
            
            //////////////////////////////////////////////////
            
            if([self isRefreshing]==NO && self.canRefresh==NO && blockScrollView.isDragging==NO && self.frame.origin.y+self.frame.size.height<=blockScrollView.contentOffset.y)
            {
                self.canRefresh = YES;
            }
            
            //////////////////////////////////////////////////
            
            if(blockScrollView.isTracking==NO && [self isRefreshing]==YES && self.modifyContentInset==NO)
            {
                self.modifyContentInset = YES;
                
                [UIView animateWithDuration:PPRefreshControl_DefaultAnimationDuration
                                 animations:^
                 {
                     blockScrollView.contentInset = UIEdgeInsetsMake(blockScrollView.contentInset.top+self.insetHeight,
                                                                     blockScrollView.contentInset.left,
                                                                     blockScrollView.contentInset.bottom,
                                                                     blockScrollView.contentInset.right);
                     
                     blockScrollView.contentOffset = CGPointMake(0, -blockScrollView.contentInset.top);
                 }];
                
                break;
            }
            
            //////////////////////////////////////////////////
            
            [self setNeedsDisplay];
        }
        
    }while(0);
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Property Methods

//================================================================================
//
//================================================================================
- (void)setEnabled:(BOOL)enabled
{ 
    [super setEnabled:enabled];
    
    //////////////////////////////////////////////////

    self.canRefresh = self.enabled;
}

////////////////////////////////////////////////////////////////////////////////////////////////////

#pragma mark - Instance Methods

//================================================================================
//
//================================================================================
- (BOOL)isRefreshing
{
    return self.activityIndicatorView.isAnimating;
}

//================================================================================
//
//================================================================================
- (void)beginRefreshing
{
    do
    {
        if(self.superview==nil || [self.superview isKindOfClass:[UIScrollView class]]==NO || [self isRefreshing]==YES)
        {
            break;
        }
        
        __block UIScrollView *blockScrollView = (UIScrollView *)self.superview;
        
        //////////////////////////////////////////////////
        
        [self.activityIndicatorView startAnimating];
        
        //////////////////////////////////////////////////
        
        self.canRefresh = NO;
        
        //////////////////////////////////////////////////
        
        if(blockScrollView.isTracking==NO && self.modifyContentInset==NO)
        {
            self.modifyContentInset = YES;
            
            [UIView animateWithDuration:PPRefreshControl_DefaultAnimationDuration
                             animations:^
             {
                 blockScrollView.contentInset = UIEdgeInsetsMake(blockScrollView.contentInset.top+self.insetHeight,
                                                                 blockScrollView.contentInset.left,
                                                                 blockScrollView.contentInset.bottom,
                                                                 blockScrollView.contentInset.right);
                 
                 blockScrollView.contentOffset = CGPointMake(0, -blockScrollView.contentInset.top);
             }];
        }
        
    }while(0);
}

//================================================================================
//
//================================================================================
- (void)endRefreshing
{
    do
    {
        if(self.superview==nil || [self.superview isKindOfClass:[UIScrollView class]]==NO || [self isRefreshing]==NO)
        {
            break;
        }
        
        __block UIScrollView *blockScrollView = (UIScrollView *)self.superview;
        
        //////////////////////////////////////////////////
        
        [self.activityIndicatorView stopAnimating];
        
        //////////////////////////////////////////////////
        
        if(self.modifyContentInset==YES)
        {
            self.modifyContentInset = NO;
            
            [UIView animateWithDuration:PPRefreshControl_DefaultAnimationDuration
                             animations:^
             {
                 blockScrollView.contentInset = UIEdgeInsetsMake(blockScrollView.contentInset.top-self.insetHeight,
                                                                 blockScrollView.contentInset.left,
                                                                 blockScrollView.contentInset.bottom,
                                                                 blockScrollView.contentInset.right);
             }];
        }
        
    }while(0);
}

@end
