//
//  BKPasscodeField.m
//  BKPasscodeViewDemo
//
//  Created by Byungkook Jang on 2014. 4. 20..
//  Copyright (c) 2014년 Byungkook Jang. All rights reserved.
//

#import "PPDigitalPinCodeField.h"

@interface PPDigitalPinCodeField ()

/// 輸入點的大小
@property (nonatomic) CGSize                dotSize;

/// 未輸入的線粗
@property (nonatomic) CGFloat           	lineHeight;

/// 字元間距
@property (nonatomic) CGFloat           	dotSpacing;

/// 點的顏色
@property (nonatomic, retain) UIColor       *dotColor;


@property (nonatomic, retain) NSMutableString       *mutablePincode;
@property (nonatomic, retain) NSRegularExpression   *nonDigitRegularExpression;

@end

////////////////////////////////////////////////////////////////////////////////////////////////////
@implementation PPDigitalPinCodeField





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - init/dealloc


//==============================================================================
//
//==============================================================================
- (id)initWithFrame:(CGRect)frame
{
    self = [super initWithFrame:frame];
    if (self)
    {
        [self initialize];
    }
    return self;
}


//==============================================================================
//
//==============================================================================
- (id)initWithCoder:(NSCoder *)aDecoder
{
    self = [super initWithCoder:aDecoder];
    if (self)
    {
        [self initialize];
    }
    return self;
}


//==============================================================================
//
//==============================================================================
- (id)init
{
    self = [super init];
    if (self)
    {
        [self initialize];
    }
    return self;
}


//==============================================================================
//
//==============================================================================
- (void)initialize
{
    _maximumLength = 4;
    _dotSize = CGSizeMake(18.0f, 19.0f);
    _dotSpacing = 25.0f;
    _lineHeight = 3.0f;
    _dotColor = [UIColor blackColor];
    
    self.backgroundColor = [UIColor clearColor];
    
    _mutablePincode = [[NSMutableString alloc] initWithCapacity:self.maximumLength];
    
    [self addTarget:self action:@selector(onTouchInside:) forControlEvents:UIControlEventTouchUpInside];
}


//==============================================================================
//
//==============================================================================
- (void)dealloc
{
    self.delegate = nil;
    self.imageSource = nil;
    
    [_dotColor release];
    _dotColor = nil;
    
    [_mutablePincode release];
    _mutablePincode = nil;
    
    [_nonDigitRegularExpression release];
    _nonDigitRegularExpression = nil;
    //////////////////////////////////////////////////
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - property


//==============================================================================
//
//==============================================================================
- (NSRegularExpression *)nonDigitRegularExpression
{
    if (nil == _nonDigitRegularExpression) {
        _nonDigitRegularExpression = [[NSRegularExpression alloc] initWithPattern:@"[^0-9]+" options:0 error:nil];
    }
    return _nonDigitRegularExpression;
}


//==============================================================================
//
//==============================================================================
- (NSString *)pincode
{
    return self.mutablePincode;
}


//==============================================================================
//
//==============================================================================
- (void)setPincode:(NSString *)pincode
{
    if (pincode)
    {
        if ([pincode length] > self.maximumLength)
        {
            pincode = [pincode substringWithRange:NSMakeRange(0, self.maximumLength)];
        }
        self.mutablePincode = [NSMutableString stringWithString:pincode];
    }
    else
    {
        self.mutablePincode = [NSMutableString string];
    }
    
    [self setNeedsDisplay];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - UIKeyInput protocol


//==============================================================================
//
//==============================================================================
- (BOOL)hasText
{
    return ([self.mutablePincode length] > 0);
}


//==============================================================================
//
//==============================================================================
- (void)insertText:(NSString *)text
{
    // 是否有enable
    if (self.enabled == NO) {
        return;
    }
    
    // 如果是數字鍵盤，要把非數字過濾
    if (self.keyboardType == UIKeyboardTypeNumberPad)
    {
        text = [self.nonDigitRegularExpression stringByReplacingMatchesInString:text
                                                                        options:0
                                                                          range:NSMakeRange(0, text.length)
                                                                   withTemplate:@""];
    }
    
    if (text.length == 0)
    {
        return;
    }
    
    NSInteger newLength = [self.mutablePincode length] + [text length];
    
    // 如果太長就不處理
    if (newLength > self.maximumLength)
    {
        return;
    }
    
    // 是否可以輸入
    if ([self.delegate respondsToSelector:@selector(pincodeField:shouldInsertText:)])
    {
        BOOL shouldInsert = [self.delegate pincodeField:self shouldInsertText:text];
        if (shouldInsert==NO)
        {
            return;
        }
    }
    
    [self.mutablePincode appendString:text];

    [self setNeedsDisplay];
    // 設定為變更
    [self sendActionsForControlEvents:UIControlEventEditingChanged];
}


//==============================================================================
//
//==============================================================================
- (void)deleteBackward
{
    if (self.enabled == NO)
    {
        return;
    }
    
    if ([self.delegate respondsToSelector:@selector(pincodeFieldShouldDeleteBackward:)])
    {
        BOOL shouldDelete = [self.delegate pincodeFieldShouldDeleteBackward:self];
        if (shouldDelete==NO)
        {
            return;
        }
    }
    
    if ([self.mutablePincode length] == 0) {
        return;
    }
    
    [self.mutablePincode deleteCharactersInRange:NSMakeRange(self.mutablePincode.length - 1, 1)];
    
    [self setNeedsDisplay];
    // 設定為變更
    [self sendActionsForControlEvents:UIControlEventEditingChanged];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - UITextInputTraits protocol


//==============================================================================
//
//==============================================================================
- (UITextAutocapitalizationType)autocapitalizationType
{
    return UITextAutocapitalizationTypeNone;
}


//==============================================================================
//
//==============================================================================
- (UITextAutocorrectionType)autocorrectionType
{
    return UITextAutocorrectionTypeNo;
}


//==============================================================================
//
//==============================================================================
- (UITextSpellCheckingType)spellCheckingType
{
    return UITextSpellCheckingTypeNo;
}


//==============================================================================
//
//==============================================================================
- (BOOL)enablesReturnKeyAutomatically
{
    return YES;
}


//==============================================================================
//
//==============================================================================
- (UIKeyboardAppearance)keyboardAppearance
{
    return UIKeyboardAppearanceDefault;
}


//==============================================================================
//
//==============================================================================
- (UIReturnKeyType)returnKeyType
{
    return UIReturnKeyDone;
}


//==============================================================================
//
//==============================================================================
- (BOOL)isSecureTextEntry
{
    return YES;
}

#pragma mark - UIView


//==============================================================================
//
//==============================================================================
- (CGSize)contentSize
{
    return CGSizeMake(self.maximumLength * _dotSize.width + (self.maximumLength - 1) * _dotSpacing,
                      _dotSize.height);
}


//==============================================================================
//
//==============================================================================
- (void)setFrame:(CGRect)frame
{
    [super setFrame:frame];
    [self setNeedsDisplay];
}


//==============================================================================
//
//==============================================================================
- (void)drawRect:(CGRect)rect
{
    CGSize contentSize = [self contentSize];
    
    CGPoint origin = CGPointMake(floorf((self.frame.size.width - contentSize.width) * 0.5f),
                                 floorf((self.frame.size.height - contentSize.height) * 0.5f));
    
    // 使用客制化圖片
    if ([self.imageSource respondsToSelector:@selector(pincodeField:dotImageAtIndex:filled:)]) {
        
        for (NSUInteger i = 0; i < self.maximumLength; i++)
        {
            
            UIImage *image = nil;
            
            if (i < self.mutablePincode.length)
            {
                // draw filled image
                image = [self.imageSource pincodeField:self dotImageAtIndex:i filled:YES];
            }
            else
            {
                // draw blank image
                image = [self.imageSource pincodeField:self dotImageAtIndex:i filled:NO];
            }
            
            if (image)
            {
                CGRect imageFrame = CGRectMake(origin.x, origin.y, self.dotSize.width, self.dotSize.height);
                [image drawInRect:imageFrame];
            }
            
            origin.x += (self.dotSize.width + self.dotSpacing);
        }
        
    }
    else
    {
        
        CGContextRef context = UIGraphicsGetCurrentContext();
        CGContextSetFillColorWithColor(context, self.dotColor.CGColor);
        
        for (NSUInteger i = 0; i < self.maximumLength; i++)
        {
            
            if (i < self.mutablePincode.length)
            {
                // draw circle
                CGRect circleFrame = CGRectMake(origin.x, origin.y, self.dotSize.width, self.dotSize.height);
                CGContextFillEllipseInRect(context, circleFrame);
            }
            else
            {
                // draw line
                CGRect lineFrame = CGRectMake(origin.x, origin.y + floorf((self.dotSize.height - self.lineHeight) * 0.5f),
                                              self.dotSize.width, self.lineHeight);
                CGContextFillRect(context, lineFrame);
            }
            
            origin.x += (self.dotSize.width + self.dotSpacing);
        }
    }
}


//==============================================================================
// for sizeToFit
//==============================================================================
- (CGSize)sizeThatFits:(CGSize)size
{
    return [self contentSize];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - UIResponder


//==============================================================================
//
//==============================================================================
- (BOOL)canBecomeFirstResponder
{
    return YES;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - Actions


//==============================================================================
//
//==============================================================================
- (void)onTouchInside:(id)sender
{
    [self becomeFirstResponder];
}

@end
