//
//  PPLockScreenController.m
//  Pods
//
//  Created by sanhue on 2015/9/21.
//
//

#import "PPLockScreenController.h"
#import "PPKeychainController.h"
#import "PPEncodeController.h"
#import "PPNavigationController.h"
#import "PPTouchID.h"
#import "NSString+Additions.h"
#import "PPLockScreenController+SettingsKey.h"
#import "UIViewController+ShareApplication.h"


#define PPLockScreenEncodeKey			"PPLockScreenEncodeKey"


////////////////////////////////////////////////////////////////////////////////////////////////////
@interface PPLockScreenController () <PPPincodeViewControllerDelegate>
@property (atomic, retain) NSString *keyName;
@property (atomic, assign) NSUInteger lockDelayTime;
@property (nonatomic, assign) BOOL saveToKeychain;

// 鎖定時間控制
/// 是否有進入背景
@property (nonatomic, assign) BOOL hasEnterBackground;

/// 是否在willEnterForground已經秀過
@property (nonatomic, assign) BOOL isShowFromBackground;
@property (atomic, assign) NSTimeInterval enterBackgroundTimeInterval;

/// 是否已經顯示lock screen
@property (atomic, assign) BOOL hasShowLockScreen;

@property (atomic, retain) PPViewController *oldRootViewController;

@end



////////////////////////////////////////////////////////////////////////////////////////////////////
@implementation PPLockScreenController




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - initial


//==============================================================================
//
//==============================================================================
- (instancetype)init
{
    self = [super init];
    if (self)
    {
        _hasShowLockScreen = NO;
        _hasEnterBackground = YES;
        _saveToKeychain = NO;
        _isShowFromBackground = NO;
        
        NSNotificationCenter *notificationCenter = [NSNotificationCenter defaultCenter];
        [notificationCenter addObserver:self selector:@selector(recvDidEnterBackground:) name:UIApplicationDidEnterBackgroundNotification object:nil];
        [notificationCenter addObserver:self selector:@selector(recvWillEnterForeground:) name:UIApplicationWillEnterForegroundNotification object:nil];
        [notificationCenter addObserver:self selector:@selector(recvDidBecomeActive:) name:UIApplicationDidBecomeActiveNotification object:nil];
    }
    return self;
}


//==============================================================================
//
//==============================================================================
- (void)dealloc
{
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    
    self.keyName = nil;
    self.checkCompletion = nil;
    self.oldRootViewController = nil;
    //////////////////////////////////////////////////
    [super dealloc];
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - (private) encode/decode password data


//==============================================================================
//
//==============================================================================
- (NSString *)passwordFromKaychainName:(NSString *)keychainName
{
    NSData *passwordData = nil;

    if (self.saveToKeychain)
    {
        [PPKeychainController getData:&passwordData withKeyTag:keychainName];
    }
    else
    {
        NSUserDefaults *userDefaults = [NSUserDefaults standardUserDefaults];

        passwordData = [userDefaults objectForKey:keychainName];
    }
    
    
    
    if ([passwordData length]>0)
    {
        NSString *result = nil;
        NSData *decodeResultData = nil;
        int  retLen = 0;
        Byte *keyBytes = (Byte *)PPLockScreenEncodeKey;
        Byte *sourceBytes = (Byte *)[passwordData bytes];
        Byte *encodeBytes = [PPEncodeController allocDecodeDataWithSource:sourceBytes
                                                                sourceLen:(int)[passwordData length]
                                                                  keyByte:keyBytes
                                                                   keyLen:(int)strlen((const char*)keyBytes)
                                                                   retLen:&retLen];
        decodeResultData = [NSData dataWithBytes:encodeBytes length:retLen];
        if ([decodeResultData length]>0)
        {
            result = [[[NSString alloc] initWithData:decodeResultData encoding:NSUTF8StringEncoding] autorelease];
        }
        
        free(encodeBytes);
        
        return result;
    }
    return nil;
}


//==============================================================================
//
//==============================================================================
- (void)savePassword:(NSString *)password withKeychainName:(NSString *)keychainName
{
    if ([password length]==0)
    {
        return ;
    }
    
    NSData *result=nil;
    int  retLen = 0;
    Byte *keyBytes = (Byte *)PPLockScreenEncodeKey;
    Byte *sourceBytes = (Byte *)[password cStringUsingEncoding:NSUTF8StringEncoding];
    Byte *encodeBytes = [PPEncodeController allocEncodeDataWithSource:sourceBytes
                                                            sourceLen:(int)strlen((const char*)sourceBytes)
                                                              keyByte:keyBytes
                                                               keyLen:(int)strlen((const char*)keyBytes)
                                                               retLen:&retLen];
    result = [NSData dataWithBytes:encodeBytes length:retLen];
    
    if ([result length]>0)
    {
        if (self.saveToKeychain)
        {
            // 刪除存在setting中舊的key
            [[NSUserDefaults standardUserDefaults] removeObjectForKey:self.keyName];
            
            [PPKeychainController addData:result withKeyTag:keychainName];
        }
        else
        {
            // 刪除存在keychain中舊的key
            [PPKeychainController removeKeyTag:keychainName];
            
            NSUserDefaults *userDefaults = [NSUserDefaults standardUserDefaults];
            [userDefaults setObject:result forKey:keychainName];
            [userDefaults synchronize];
        }
    }

    free(encodeBytes);
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - usage methods


//==============================================================================
//
//==============================================================================
- (void)setStoreKeyName:(NSString *)storeKeyName
          lockDelayTime:(NSInteger)secend
{
    // 加入PPLockScreenController postfix 以避免重復
    self.keyName = [storeKeyName stringByAppendingFormat:@"-%@",NSStringFromClass([PPLockScreenController class])];
    self.lockDelayTime = secend;
}


//==============================================================================
//
//==============================================================================
- (void)setSaveToKeychain:(BOOL)saveToKeychain
{
    _saveToKeychain = saveToKeychain;
}


//==============================================================================
//
//==============================================================================
- (void)lockScreen
{
    NSLog(@"%s", __PRETTY_FUNCTION__);
    
    // 如果還沒設定密碼，就不鎖定
    if (![self hasSetPassword])
    {
        return ;
    }
    
    
    //////////////////////////////////////////////////
    // 呈現鎖定畫面
    // !! 要dispatch出去才不會造成view controller flow異常
    dispatch_async(dispatch_get_main_queue(), ^{
        UIViewController *topController = [UIApplication sharedApplication].keyWindow.rootViewController;
        [self presentFromSuperViewController:[topController topMostViewController] withStyle:PPPincodeViewControllerStyle_CheckPassword];
    });
}


//==============================================================================
//
//==============================================================================
- (BOOL)isLockScreenDisplaying
{
    return self.hasShowLockScreen;
}




////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - touchID setting


//==============================================================================
//
//==============================================================================
- (void)setUseTouchIDIfPossible:(BOOL)useTouchIDIfPossible
{
    [[NSUserDefaults standardUserDefaults] setObject:@(useTouchIDIfPossible) forKey:PPLockScreenController_EnableTouchID];
    [[NSUserDefaults standardUserDefaults] synchronize];
}


//==============================================================================
//
//==============================================================================
- (BOOL)useTouchIDIfPossible
{
    NSNumber *useTouchIDIfPossible = [[NSUserDefaults standardUserDefaults] objectForKey:PPLockScreenController_EnableTouchID];
    return [useTouchIDIfPossible boolValue];
}


//==============================================================================
//
//==============================================================================
- (BOOL)canUseTouchID
{
    return [PPTouchID canUseTouchID];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - password methods


//==============================================================================
// 查詢是否設定密碼
//==============================================================================
- (BOOL)hasSetPassword
{
    return ([[self currentPassword] length]>0)?YES:NO;
}


//==============================================================================
// 回傳目前設定的密碼
//==============================================================================
- (NSString *)currentPassword
{
    return [self passwordFromKaychainName:self.keyName];
}


//==============================================================================
/**
 * 檢查密碼是否正確
 * @param password 要用來檢查的密碼
 * @return BOOL YES: 正確；NO:不正確
 */
//==============================================================================
- (BOOL)isValidPassword:(NSString *)password
{
    return [password isEqualToString:[self currentPassword]];
}


//==============================================================================
/**
 * 設定密碼
 * @param password 設定解鎖用的密碼
 */
//==============================================================================
- (void)setPassword:(NSString *)password
{
    [self savePassword:password withKeychainName:self.keyName];
}


//==============================================================================
//
//==============================================================================
- (void)deletePassword
{
    if (self.saveToKeychain)
    {
        [PPKeychainController removeKeyTag:self.keyName];
    }
    else
    {
        [[NSUserDefaults standardUserDefaults] removeObjectForKey:self.keyName];
    }
}



////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - recv Notification


//==============================================================================
//
//==============================================================================
- (void)recvDidEnterBackground:(NSNotification *)notification
{
    // 進背景大於lockDelayTime才顯示
    self.enterBackgroundTimeInterval = [NSDate timeIntervalSinceReferenceDate];
    self.hasEnterBackground = YES;
}


//==============================================================================
//
//==============================================================================
- (void)recvWillEnterForeground:(NSNotification *)notification
{
    if (self.hasEnterBackground &&
        self.enterBackgroundTimeInterval + self.lockDelayTime <= [NSDate timeIntervalSinceReferenceDate])
    {
        self.isShowFromBackground = YES;
        [self lockScreen];
    }
}


//==============================================================================
//
//==============================================================================
- (void)recvDidBecomeActive:(NSNotification *)notification
{
    // !! 如果是重新launch會跑這一段，但如果enterForeground已秀過，就不會再秀
    if (self.hasEnterBackground==YES &&
        self.isShowFromBackground == NO &&
        self.enterBackgroundTimeInterval + self.lockDelayTime <= [NSDate timeIntervalSinceReferenceDate])
    {
        self.isShowFromBackground = NO;
        [self lockScreen];
    }
    self.hasEnterBackground = NO;
}



////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - PPPincodeViewControllerDelegate


//==============================================================================
//
//==============================================================================
- (BOOL)pincodeViewController:(PPPincodeViewController *)pincodeViewController checkPassword:(NSString *)password
{
    return [self isValidPassword:password];
}


//==============================================================================
//
//==============================================================================
- (void)pincodeViewController:(PPPincodeViewController *)pincodeViewController didFinishWithPassword:(NSString *)password
{
    switch (pincodeViewController.style)
    {
        case PPPincodeViewControllerStyle_CreatePassword:    ///< 建立密碼
        case PPPincodeViewControllerStyle_ChangePassword:    ///< 變更密碼
        {
            // 如果是變更密碼或是建立密碼，完成時要儲存下來
            [self setPassword:password];
            break;
        }
        case PPPincodeViewControllerStyle_CheckPassword:     ///< 檢查密碼
        {
            // 通過驗証，關閉viewcontroller
            self.hasShowLockScreen = NO;
            break;
        }
        case PPPincodeViewControllerStyle_ClearPassword:     ///< 清除密碼
        {
            // 清除密碼
            [self deletePassword];
            self.hasShowLockScreen = NO;
            break;
        }
        default:
            break;
    }

    // !!檢查密碼，與其他style的關閉方式不同
    if (self.oldRootViewController)
    {
        [[UIApplication sharedApplication].keyWindow setRootViewController:self.oldRootViewController];
        [[UIApplication sharedApplication].keyWindow makeKeyAndVisible];

        self.oldRootViewController = nil;
        
        //////////////////////////////////////////////////
        __block typeof(self) blockSelf = self;
        
        dispatch_async(dispatch_get_main_queue(), ^{
            
            if(blockSelf.checkCompletion)
            {
                blockSelf.checkCompletion();
            }
        });
    }
    else
    {
        [pincodeViewController dismissAnimated:YES completion:^{
            
            if (pincodeViewController.style==PPPincodeViewControllerStyle_CheckPassword)
            {
                //////////////////////////////////////////////////
                __block typeof(self) blockSelf = self;
                
                dispatch_async(dispatch_get_main_queue(), ^{
                    
                    if(blockSelf.checkCompletion)
                    {
                        blockSelf.checkCompletion();
                    }
                });
            }
        }];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - presenting viewcontroller methods


//==============================================================================
//
//==============================================================================
- (void)presentFromSuperViewController:(UIViewController *)viewController withStyle:(PPPincodeViewControllerStyle)style
{
    if ([PPLockScreenController shareInstance].hasShowLockScreen==YES)
    {
        return ;
    }
    
    PPPincodeViewController *pincodeViewController = [[PPPincodeViewController alloc] init];
    
    if (pincodeViewController)
    {
        
        pincodeViewController.delegate = self;
        pincodeViewController.style = style;
        pincodeViewController.enableTouchID = [self useTouchIDIfPossible];
        pincodeViewController.view.backgroundColor = [UIColor lightGrayColor];
        
        // 如果是check password, 不用navigation bar
        if (style==PPPincodeViewControllerStyle_CheckPassword)
        {
            [PPLockScreenController shareInstance].hasShowLockScreen = YES;
            // 目前的所有rootViewController都是用PPViewController建立
            
            dispatch_async(dispatch_get_global_queue(DISPATCH_QUEUE_PRIORITY_DEFAULT, 0), ^{
                
                PPViewController *topMostViewController = (PPViewController *)viewController;
                if([topMostViewController respondsToSelector:@selector(isViewDidAppear)])
                {
                    while ([topMostViewController isViewDidAppear]==NO)
                    {
                        [NSThread sleepForTimeInterval:0.01];
                    }
                }
                
                dispatch_async(dispatch_get_main_queue(), ^{
                    [viewController presentViewController:pincodeViewController animated:YES completion:nil];
//                    [[UIApplication sharedApplication].keyWindow setRootViewController:pincodeViewController];
//                    [[UIApplication sharedApplication].keyWindow makeKeyAndVisible];
                });
            });
        }
        else
        {
            //////////////////////////////////////////////////
            PPNavigationController *ppNavigationController = [[PPNavigationController alloc] initWithRootViewController:pincodeViewController];
            [viewController presentViewController:ppNavigationController animated:YES completion:nil];
            [ppNavigationController release];
        }
        
        
        [pincodeViewController release];
    }
    
    
}







////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark
#pragma mark - Class methods


//==============================================================================
//
//==============================================================================
+ (PPLockScreenController *)shareInstance
{
    static PPLockScreenController *shareInstance = nil;
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^{
        shareInstance = [[PPLockScreenController alloc] init];
    });
    
    return shareInstance;
}



@end
