//
//  PPFloatingMenuItemView.m
//
//  Created by eddie on 2015/6/4.
//  Copyright (c) 2015年 Eddie. All rights reserved.
//

#import "PPFloatingMenuItemView.h"
#import "PPFloatingMenuItemViewLabel.h"

////////////////////////////////////////////////////////////////////////////////////////////////////

// 預設text color
#define PPFloatingMenuItemView_TextColor [UIColor whiteColor]

// 預設text background color
#define PPFloatingMenuItemView_TextBackgroundColor [UIColor grayColor]

// text內縮距離
UIEdgeInsets const PPFloatingMenuItemView_TextInset = {2, 10, 2, 10};

// 陰影偏移
CGSize const PPFloatingMenuItemView_ShadowOffset = {-3, 5};

// 陰影圓角程度
CGFloat const PPFloatingMenuItemView_ShadowRadius = 5.0;

// 陰影透明度
CGFloat const PPFloatingMenuItemView_ShadowOpacity = 0.4;

// icon和text間的水平距離
NSInteger const PPFloatingMenuItemView_Gap = 10;

// text背景圓角程度
CGFloat const PPFloatingMenuItemView_CornerRadius = 3.0;

////////////////////////////////////////////////////////////////////////////////////////////////////

@interface PPFloatingMenuItemView()
@property (nonatomic, assign) id<PPFloatingMenuItemViewDelegate> delegate;
@property (nonatomic, assign) NSInteger identifier;
@property (nonatomic, assign) PPFloatingMenuItemView_LayoutStyle layoutStyle;
@property (nonatomic, retain) UIImageView *iconImageView;
@property (nonatomic, retain) PPFloatingMenuItemViewLabel *textLabel;
@property (nonatomic, retain) NSMutableArray *constraints;
@property (nonatomic, retain) UITapGestureRecognizer *tapRecognizer;
@end

////////////////////////////////////////////////////////////////////////////////////////////////////

@implementation PPFloatingMenuItemView

////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Life cycle methods

//================================================================================
//
//================================================================================
- (id)initWithIdentifier:(NSInteger)identifier
                    icon:(UIImage *)icon
                    text:(NSString *)text
                delegate:(id<PPFloatingMenuItemViewDelegate>) delegate;
{
    if(self = [super init])
    {
        self.backgroundColor = [UIColor clearColor];
        self.autoresizesSubviews = NO;
        self.layer.shadowOffset = PPFloatingMenuItemView_ShadowOffset;
        self.layer.shadowRadius = PPFloatingMenuItemView_ShadowRadius;
        self.layer.shadowOpacity = PPFloatingMenuItemView_ShadowOpacity;
        
        self.identifier = identifier;
        self.delegate = delegate;
        
        _iconImageView = [[UIImageView alloc] initWithImage:icon];
        self.iconImageView.translatesAutoresizingMaskIntoConstraints = NO;
        [self addSubview:self.iconImageView];
        
        _textLabel = [[PPFloatingMenuItemViewLabel alloc] initWithFrame:CGRectZero];
        self.textLabel.translatesAutoresizingMaskIntoConstraints = NO;
        self.textLabel.autoresizingMask = UIViewAutoresizingNone;
        self.textLabel.text = text;
        self.textLabel.textColor = PPFloatingMenuItemView_TextColor;
        self.textLabel.numberOfLines = 1;
        self.textLabel.backgroundColor = PPFloatingMenuItemView_TextBackgroundColor;

        self.textLabel.contentInsets = PPFloatingMenuItemView_TextInset;
        [self.textLabel sizeToFit];
        
        [self.textLabel.layer setCornerRadius:PPFloatingMenuItemView_CornerRadius];
        [self.textLabel.layer setMasksToBounds:YES];
        
        [self addSubview:self.textLabel];
        
        //////////////////////////////////////////////////

        _tapRecognizer = [[UITapGestureRecognizer alloc] initWithTarget:self action:@selector(onTapItem)];
        [self.tapRecognizer setNumberOfTouchesRequired:1];
        [self addGestureRecognizer:self.tapRecognizer];
    }
    
    return self;
}


//================================================================================
//
//================================================================================
- (void)dealloc
{
    self.delegate = nil;
    self.menuItemTextFont = nil;
    self.menuItemTextColor = nil;
    self.menuItemTextBackgroundColor = nil;

    //////////////////////////////////////////////////
    
    [self removeGestureRecognizer:self.tapRecognizer];
    self.tapRecognizer = nil;
    
    //////////////////////////////////////////////////

    [self.iconImageView setImage:nil];
    [self.iconImageView removeFromSuperview];
    self.iconImageView = nil;

    [self.textLabel removeFromSuperview];
    self.textLabel = nil;
    
    self.constraints = nil;
    
    [super dealloc];
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Overwrite methods

//================================================================================
//
//================================================================================
- (void)layoutSubviews
{
//    NSLog(@"%s1 %@ %@", __func__, NSStringFromCGRect(self.textLabel.frame), self.textLabel.text);
    
//    if(self.constraints != nil)
//    {
//        [self removeConstraints:self.constraints];
//        self.constraints = nil;
//    }
    
    if(self.constraints == nil)
    {
        NSMutableArray *constraints = [NSMutableArray array];
        NSString *format;
        CGSize iconSize = [self iconSize];
        CGFloat textLabelWidth = self.textLabel.frame.size.width;
        
        
        // !! 調整寬度不超過範圍
        if(textLabelWidth+PPFloatingMenuItemView_Gap+iconSize.width > self.frame.size.width)
        {
            textLabelWidth = self.frame.size.width-(PPFloatingMenuItemView_Gap+iconSize.width);
            textLabelWidth = [self.textLabel adjustsFontWithWidth:textLabelWidth];
        }

        NSDictionary *views = @{@"iconImageView":self.iconImageView,
                                @"textLabel":self.textLabel};
        NSDictionary *metrics = @{@"horzGap":@(PPFloatingMenuItemView_Gap),
                                  @"iconVertMargin":@((self.frame.size.height-iconSize.height)/2),
                                  @"textVertMargin":@((self.frame.size.height-self.textLabel.frame.size.height)/2),
                                  @"textWidth":@(textLabelWidth)};
        
        
        // !! textLabel寬度一定要指定，否則呼叫[super layoutSubviews]後又會被設定為沒有加上inset的寬度。
        // !! 但是textLabel高度不會有此現象，很奇怪。
        if(self.layoutStyle == PPFloatingMenuItemView_LayoutStyle_TextIcon)
        {
            format = @"H:|[textLabel(textWidth)]-(horzGap)-[iconImageView]|";
        }
        else
        {
            format = @"H:|[iconImageView]-(horzGap)-[textLabel(textWidth)]|";
        }
        
        [constraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:format
                                                                                 options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                 metrics:metrics
                                                                                   views:views]];
        
        // !! 為了避免旋轉時強制關閉會出現conflict
        for(NSLayoutConstraint *constraint in constraints)
        {
            constraint.priority = UILayoutPriorityRequired-1;
        }
        
        format = @"V:|-(iconVertMargin)-[iconImageView]-(iconVertMargin)-|";
        [constraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:format
                                                                                 options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                 metrics:metrics
                                                                                   views:views]];
        
        format = @"V:|-(textVertMargin)-[textLabel]-(textVertMargin)-|";
        [constraints addObjectsFromArray:[NSLayoutConstraint constraintsWithVisualFormat:format
                                                                                 options:NSLayoutFormatDirectionLeadingToTrailing
                                                                                 metrics:metrics
                                                                                   views:views]];
        
        [self addConstraints:constraints];
        self.constraints = constraints;
    }
    
    
    [super layoutSubviews];
    
//    NSLog(@"%s2 %@ %@", __func__, NSStringFromCGSize(self.frame.size), NSStringFromCGSize(self.textLabel.frame.size));
}


//================================================================================
//
//================================================================================
- (void)setMenuItemTextFont:(UIFont *)menuItemTextFont
{
    if(menuItemTextFont != _menuItemTextFont)
    {
        [_menuItemTextFont release];
        _menuItemTextFont = [menuItemTextFont retain];
        self.textLabel.font = menuItemTextFont;
        [self.textLabel sizeToFit];
    }
}


//================================================================================
//
//================================================================================
- (void)setMenuItemTextColor:(UIColor *)menuItemTextColor
{
    if(menuItemTextColor != _menuItemTextColor)
    {
        [_menuItemTextColor release];
        _menuItemTextColor = [menuItemTextColor retain];
        self.textLabel.textColor = menuItemTextColor;
    }
}


//================================================================================
//
//================================================================================
- (void)setMenuItemTextBackgroundColor:(UIColor *)menuItemTextBackgroundColor
{
    if(menuItemTextBackgroundColor != _menuItemTextBackgroundColor)
    {
        [_menuItemTextBackgroundColor release];
        _menuItemTextBackgroundColor = [menuItemTextBackgroundColor retain];
        self.textLabel.backgroundColor = menuItemTextBackgroundColor;
    }
}






////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - UITapGestureRecognizer action

//================================================================================
//
//================================================================================
- (void)onTapItem
{
    if([self.delegate respondsToSelector:@selector(floatingMenuItemView:didTapItemWithIdentifier:)])
    {
        [self.delegate floatingMenuItemView:self didTapItemWithIdentifier:self.identifier];
    }
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Private methods

//================================================================================
// 取得目前的icon大小，以外部設定的menuItemIconSize優先，沒有的話就取影像大小。
//================================================================================
- (CGSize)iconSize
{
    CGSize size = self.menuItemIconSize;
    
    if(CGSizeEqualToSize(size, CGSizeZero) == YES)
    {
        size = self.iconImageView.image.size;
    }
    
    return size;
}





////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma mark - Instance methods

//================================================================================
//
//================================================================================
- (void)setLayoutStyle:(PPFloatingMenuItemView_LayoutStyle)style
{
    _layoutStyle = style;
}


//================================================================================
//
//================================================================================
- (CGSize)estimatedViewSize
{
    CGSize viewSize;
    CGSize iconSize = [self iconSize];
    
    viewSize.width = iconSize.width + PPFloatingMenuItemView_Gap + self.textLabel.frame.size.width;
    viewSize.height = fmaxf(iconSize.height, self.textLabel.frame.size.height);

    return viewSize;
}


//================================================================================
//
//================================================================================
- (CGRect)estimatedIconFrame
{
    CGSize iconSize = [self iconSize];
    CGSize viewSize = [self estimatedViewSize];
    CGRect frame = CGRectZero;
    
    if(self.layoutStyle == PPFloatingMenuItemView_LayoutStyle_TextIcon)
    {
        frame.size = iconSize;
        frame.origin.x = self.textLabel.frame.size.width + PPFloatingMenuItemView_Gap;
        frame.origin.y = (viewSize.height - frame.size.height)/2;
    }
    else
    {
        frame.size = iconSize;
        frame.origin.x = 0;
        frame.origin.y = (viewSize.height - frame.size.height)/2;
    }

    return frame;
}

@end
